/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("tsickle/src/tsickle", ["require", "exports", "typescript", "tsickle/src/cli_support", "tsickle/src/decorator_downlevel_transformer", "tsickle/src/decorators", "tsickle/src/enum_transformer", "tsickle/src/externs", "tsickle/src/fileoverview_comment_transformer", "tsickle/src/googmodule", "tsickle/src/jsdoc_transformer", "tsickle/src/modules_manifest", "tsickle/src/transformer_util", "tsickle/src/cli_support", "tsickle/src/externs", "tsickle/src/modules_manifest"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    const ts = require("typescript");
    const cli_support_1 = require("tsickle/src/cli_support");
    const decorator_downlevel_transformer_1 = require("tsickle/src/decorator_downlevel_transformer");
    const decorators_1 = require("tsickle/src/decorators");
    const enum_transformer_1 = require("tsickle/src/enum_transformer");
    const externs_1 = require("tsickle/src/externs");
    const fileoverview_comment_transformer_1 = require("tsickle/src/fileoverview_comment_transformer");
    const googmodule = require("tsickle/src/googmodule");
    const jsdoc_transformer_1 = require("tsickle/src/jsdoc_transformer");
    const modules_manifest_1 = require("tsickle/src/modules_manifest");
    const transformer_util_1 = require("tsickle/src/transformer_util");
    // Exported for users as a default impl of pathToModuleName.
    var cli_support_2 = require("tsickle/src/cli_support");
    exports.pathToModuleName = cli_support_2.pathToModuleName;
    // Retained here for API compatibility.
    var externs_2 = require("tsickle/src/externs");
    exports.getGeneratedExterns = externs_2.getGeneratedExterns;
    var modules_manifest_2 = require("tsickle/src/modules_manifest");
    exports.ModulesManifest = modules_manifest_2.ModulesManifest;
    function mergeEmitResults(emitResults) {
        const diagnostics = [];
        let emitSkipped = true;
        const emittedFiles = [];
        const externs = {};
        const modulesManifest = new modules_manifest_1.ModulesManifest();
        for (const er of emitResults) {
            diagnostics.push(...er.diagnostics);
            emitSkipped = emitSkipped || er.emitSkipped;
            if (er.emittedFiles) {
                emittedFiles.push(...er.emittedFiles);
            }
            Object.assign(externs, er.externs);
            modulesManifest.addManifest(er.modulesManifest);
        }
        return { diagnostics, emitSkipped, emittedFiles, externs, modulesManifest };
    }
    exports.mergeEmitResults = mergeEmitResults;
    /** @deprecated Exposed for backward compat with Angular.  Use emit() instead. */
    function emitWithTsickle(program, host, tsHost, tsOptions, targetSourceFile, writeFile, cancellationToken, emitOnlyDtsFiles, customTransformers = {}) {
        return emit(program, host, writeFile || tsHost.writeFile.bind(tsHost), targetSourceFile, cancellationToken, emitOnlyDtsFiles, customTransformers);
    }
    exports.emitWithTsickle = emitWithTsickle;
    function emit(program, host, writeFile, targetSourceFile, cancellationToken, emitOnlyDtsFiles, customTransformers = {}) {
        for (const sf of program.getSourceFiles()) {
            cli_support_1.assertAbsolute(sf.fileName);
        }
        let tsickleDiagnostics = [];
        const typeChecker = program.getTypeChecker();
        const tsOptions = program.getCompilerOptions();
        const tsickleSourceTransformers = [];
        if (host.transformTypesToClosure) {
            // Only add @suppress {checkTypes} comments when also adding type annotations.
            tsickleSourceTransformers.push(fileoverview_comment_transformer_1.transformFileoverviewCommentFactory(tsOptions, tsickleDiagnostics));
            tsickleSourceTransformers.push(jsdoc_transformer_1.jsdocTransformer(host, tsOptions, typeChecker, tsickleDiagnostics));
            tsickleSourceTransformers.push(enum_transformer_1.enumTransformer(typeChecker, tsickleDiagnostics));
        }
        if (host.transformDecorators) {
            tsickleSourceTransformers.push(decorator_downlevel_transformer_1.decoratorDownlevelTransformer(typeChecker, tsickleDiagnostics));
        }
        const modulesManifest = new modules_manifest_1.ModulesManifest();
        const tsTransformers = {
            before: [
                ...(customTransformers.beforeTsickle || []),
                ...(tsickleSourceTransformers || []).map(tf => skipTransformForSourceFileIfNeeded(host, tf)),
                ...(customTransformers.beforeTs || []),
            ],
            after: [
                ...(customTransformers.afterTs || []),
            ],
            afterDeclarations: customTransformers.afterDeclarations,
        };
        if (host.transformTypesToClosure) {
            // See comment on remoteTypeAssertions.
            tsTransformers.before.push(jsdoc_transformer_1.removeTypeAssertions());
        }
        if (host.googmodule) {
            tsTransformers.after.push(googmodule.commonJsToGoogmoduleTransformer(host, modulesManifest, typeChecker, tsickleDiagnostics));
            tsTransformers.after.push(decorators_1.transformDecoratorsOutputForClosurePropertyRenaming(tsickleDiagnostics));
        }
        // Wrap the writeFile callback to hook writing of the dts file.
        const writeFileImpl = (fileName, content, writeByteOrderMark, onError, sourceFiles) => {
            cli_support_1.assertAbsolute(fileName);
            if (host.addDtsClutzAliases && transformer_util_1.isDtsFileName(fileName) && sourceFiles) {
                // Only bundle emits pass more than one source file for .d.ts writes. Bundle emits however
                // are not supported by tsickle, as we cannot annotate them for Closure in any meaningful
                // way anyway.
                if (!sourceFiles || sourceFiles.length > 1) {
                    throw new Error(`expected exactly one source file for .d.ts emit, got ${sourceFiles.map(sf => sf.fileName)}`);
                }
                const originalSource = sourceFiles[0];
                content = addClutzAliases(content, originalSource, typeChecker, host, tsOptions);
            }
            writeFile(fileName, content, writeByteOrderMark, onError, sourceFiles);
        };
        const { diagnostics: tsDiagnostics, emitSkipped, emittedFiles } = program.emit(targetSourceFile, writeFileImpl, cancellationToken, emitOnlyDtsFiles, tsTransformers);
        const externs = {};
        if (host.transformTypesToClosure) {
            const sourceFiles = targetSourceFile ? [targetSourceFile] : program.getSourceFiles();
            for (const sourceFile of sourceFiles) {
                const isDts = transformer_util_1.isDtsFileName(sourceFile.fileName);
                if (isDts && host.shouldSkipTsickleProcessing(sourceFile.fileName)) {
                    continue;
                }
                const { output, diagnostics } = externs_1.generateExterns(typeChecker, sourceFile, host, host.moduleResolutionHost, program.getCompilerOptions());
                if (output) {
                    externs[sourceFile.fileName] = output;
                }
                if (diagnostics) {
                    tsickleDiagnostics.push(...diagnostics);
                }
            }
        }
        // All diagnostics (including warnings) are treated as errors.
        // If the host decides to ignore warnings, just discard them.
        // Warnings include stuff like "don't use @type in your jsdoc"; tsickle
        // warns and then fixes up the code to be Closure-compatible anyway.
        tsickleDiagnostics = tsickleDiagnostics.filter(d => d.category === ts.DiagnosticCategory.Error ||
            !host.shouldIgnoreWarningsForPath(d.file.fileName));
        return {
            modulesManifest,
            emitSkipped,
            emittedFiles: emittedFiles || [],
            diagnostics: [...tsDiagnostics, ...tsickleDiagnostics],
            externs
        };
    }
    exports.emit = emit;
    /** Compares two strings and returns a number suitable for use in sort(). */
    function stringCompare(a, b) {
        if (a < b)
            return -1;
        if (a > b)
            return 1;
        return 0;
    }
    /**
     * A tsickle produced declaration file might be consumed be referenced by Clutz
     * produced .d.ts files, which use symbol names based on Closure's internal
     * naming conventions, so we need to provide aliases for all the exported symbols
     * in the Clutz naming convention.
     */
    function addClutzAliases(dtsFileContent, sourceFile, typeChecker, host, options) {
        const moduleSymbol = typeChecker.getSymbolAtLocation(sourceFile);
        const moduleExports = moduleSymbol && typeChecker.getExportsOfModule(moduleSymbol);
        if (!moduleExports)
            return dtsFileContent;
        // .d.ts files can be transformed, too, so we need to compare the original node below.
        const origSourceFile = ts.getOriginalNode(sourceFile);
        // In order to write aliases, the exported symbols need to be available in the
        // the module scope. That is not always the case:
        //
        // export
        // 1) export const X;           // works
        //
        // reexport
        // 2) export {X} from './foo';  // doesn't
        //
        // imported reexport
        // 3) import {X} from './foo';  // works
        //    export {X} from './foo';
        //
        // getExportsOfModule returns all three types, but we need to separate 2).
        // For now we 'fix' 2) by simply not emitting a clutz alias, since clutz
        // interop is used in minority of scenarios.
        //
        // TODO(radokirov): attempt to add appropriate imports for 2) so that
        // currently finding out local appears even harder than fixing exports.
        const localExports = moduleExports.filter(e => {
            // If there are no declarations, be conservative and don't emit the aliases.
            // I don't know how can this happen, we have no tests that excercise it.
            if (!e.declarations)
                return false;
            // Skip default exports, they are not currently supported.
            // default is a keyword in typescript, so the name of the export being
            // default means that it's a default export.
            if (e.name === 'default')
                return false;
            // Use the declaration location to determine separate cases above.
            for (const d of e.declarations) {
                // This is a special case for export *. Technically, it is outside the
                // three cases outlined, but at this point we have rewritten it to a
                // reexport or an imported reexport. However, it appears that the
                // rewriting also has made it behave different from explicit named export
                // in the sense that the declaration appears to point at the original
                // location not the reexport location.  Since we can't figure out whether
                // there is a local import here, we err on the side of less emit.
                if (d.getSourceFile() !== origSourceFile) {
                    return false;
                }
                // @internal marked APIs are not exported, so must not get aliases.
                // This uses an internal TS API, assuming that accessing this will be more stable compared to
                // implementing our own version.
                // tslint:disable-next-line: no-any
                if (options.stripInternal && ts['isInternalDeclaration'](d, origSourceFile)) {
                    return false;
                }
                if (!ts.isExportSpecifier(d)) {
                    // we have a pure export (case 1) thus safe to emit clutz alias.
                    return true;
                }
                // The declaration d is useless to separate reexport and import-reexport
                // because they both point to the reexporting file and not to the original
                // one.  However, there is another ts API that can do a deeper resolution.
                const localSymbol = typeChecker.getExportSpecifierLocalTargetSymbol(d);
                // I don't know how can this happen, but err on the side of less emit.
                if (!localSymbol)
                    return false;
                // `declarations` is undefined for builtin symbols, such as `unknown`.
                if (!localSymbol.declarations)
                    return false;
                // In case of no import we ended up in a declaration in foo.ts, while in
                // case of having an import localD is still in the reexporing file.
                for (const localD of localSymbol.declarations) {
                    if (localD.getSourceFile() !== origSourceFile) {
                        return false;
                    }
                }
            }
            return true;
        });
        if (!localExports.length)
            return dtsFileContent;
        // TypeScript 2.8 and TypeScript 2.9 differ on the order in which the
        // module symbols come out, so sort here to make the tests stable.
        localExports.sort((a, b) => stringCompare(a.name, b.name));
        const moduleName = host.pathToModuleName('', sourceFile.fileName);
        const clutzModuleName = moduleName.replace(/\./g, '$');
        // Clutz might refer to the name in two different forms (stemming from goog.provide and
        // goog.module respectively).
        // 1) global in clutz:   ಠ_ಠ.clutz.module$contents$path$to$module_Symbol...
        // 2) local in a module: ಠ_ಠ.clutz.module$exports$path$to$module.Symbol ..
        // See examples at:
        // https://github.com/angular/clutz/tree/master/src/test/java/com/google/javascript/clutz
        // Case (1) from above.
        let globalSymbols = '';
        // Case (2) from above.
        let nestedSymbols = '';
        for (const symbol of localExports) {
            let localName = symbol.name;
            const declaration = symbol.declarations.find(d => d.getSourceFile() === origSourceFile);
            if (declaration && ts.isExportSpecifier(declaration) && declaration.propertyName) {
                // If declared in an "export {X as Y};" export specifier, then X (stored in propertyName) is
                // the local name that resolves within the module, whereas Y is only available on the exports,
                // i.e. the name used to address the symbol from outside the module.
                // Use the localName for the export then, but publish under the external name.
                localName = declaration.propertyName.text;
            }
            const mangledName = `module$contents$${clutzModuleName}_${symbol.name}`;
            globalSymbols += `\t\texport {${localName} as ${mangledName}}\n`;
            // TODO(mprobst): Once tsickle is on TS3.7, the two lines below can be replaced with
            // "export {localName};". However in TS3.5, localName resolves within the module, so
            // exporting {localName} causes a circular definition error. The workaround is to import the
            // mangled name.
            nestedSymbols += `\t\timport ${localName}$clutz = ಠ_ಠ.clutz.${mangledName};\n`;
            nestedSymbols += `\t\texport {${localName}$clutz as ${symbol.name}};\n`;
        }
        dtsFileContent += 'declare global {\n';
        dtsFileContent += `\tnamespace ಠ_ಠ.clutz {\n`;
        dtsFileContent += globalSymbols;
        dtsFileContent += `\t}\n`;
        dtsFileContent += `\tnamespace ಠ_ಠ.clutz.module$exports$${clutzModuleName} {\n`;
        // TODO(martinprobst): See https://github.com/Microsoft/TypeScript/issues/35385, remove once fixed
        dtsFileContent += `\t\tconst clutz$workaround$tissue$35385: number;\n`;
        dtsFileContent += nestedSymbols;
        dtsFileContent += `\t}\n`;
        dtsFileContent += '}\n';
        return dtsFileContent;
    }
    function skipTransformForSourceFileIfNeeded(host, delegateFactory) {
        return (context) => {
            const delegate = delegateFactory(context);
            return (sourceFile) => {
                if (host.shouldSkipTsickleProcessing(sourceFile.fileName)) {
                    return sourceFile;
                }
                return delegate(sourceFile);
            };
        };
    }
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidHNpY2tsZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uL3NyYy90c2lja2xlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBOzs7Ozs7R0FNRzs7Ozs7Ozs7Ozs7O0lBRUgsaUNBQWlDO0lBR2pDLHlEQUE2QztJQUM3QyxpR0FBZ0Y7SUFDaEYsdURBQWlGO0lBQ2pGLG1FQUFtRDtJQUNuRCxpREFBMEM7SUFDMUMsbUdBQXVGO0lBQ3ZGLHFEQUEyQztJQUMzQyxxRUFBMkU7SUFDM0UsbUVBQW1EO0lBQ25ELG1FQUFpRDtJQUVqRCw0REFBNEQ7SUFDNUQsdURBQStDO0lBQXZDLHlDQUFBLGdCQUFnQixDQUFBO0lBQ3hCLHVDQUF1QztJQUN2QywrQ0FBOEM7SUFBdEMsd0NBQUEsbUJBQW1CLENBQUE7SUFDM0IsaUVBQTREO0lBQTNDLDZDQUFBLGVBQWUsQ0FBQTtJQW1DaEMsU0FBZ0IsZ0JBQWdCLENBQUMsV0FBeUI7UUFDeEQsTUFBTSxXQUFXLEdBQW9CLEVBQUUsQ0FBQztRQUN4QyxJQUFJLFdBQVcsR0FBRyxJQUFJLENBQUM7UUFDdkIsTUFBTSxZQUFZLEdBQWEsRUFBRSxDQUFDO1FBQ2xDLE1BQU0sT0FBTyxHQUFpQyxFQUFFLENBQUM7UUFDakQsTUFBTSxlQUFlLEdBQUcsSUFBSSxrQ0FBZSxFQUFFLENBQUM7UUFDOUMsS0FBSyxNQUFNLEVBQUUsSUFBSSxXQUFXLEVBQUU7WUFDNUIsV0FBVyxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUNwQyxXQUFXLEdBQUcsV0FBVyxJQUFJLEVBQUUsQ0FBQyxXQUFXLENBQUM7WUFDNUMsSUFBSSxFQUFFLENBQUMsWUFBWSxFQUFFO2dCQUNuQixZQUFZLENBQUMsSUFBSSxDQUFDLEdBQUcsRUFBRSxDQUFDLFlBQVksQ0FBQyxDQUFDO2FBQ3ZDO1lBQ0QsTUFBTSxDQUFDLE1BQU0sQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBQ25DLGVBQWUsQ0FBQyxXQUFXLENBQUMsRUFBRSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1NBQ2pEO1FBQ0QsT0FBTyxFQUFDLFdBQVcsRUFBRSxXQUFXLEVBQUUsWUFBWSxFQUFFLE9BQU8sRUFBRSxlQUFlLEVBQUMsQ0FBQztJQUM1RSxDQUFDO0lBaEJELDRDQWdCQztJQXdCRCxpRkFBaUY7SUFDakYsU0FBZ0IsZUFBZSxDQUMzQixPQUFtQixFQUFFLElBQWlCLEVBQUUsTUFBdUIsRUFBRSxTQUE2QixFQUM5RixnQkFBZ0MsRUFBRSxTQUFnQyxFQUNsRSxpQkFBd0MsRUFBRSxnQkFBMEIsRUFDcEUscUJBQXVDLEVBQUU7UUFDM0MsT0FBTyxJQUFJLENBQ1AsT0FBTyxFQUFFLElBQUksRUFBRSxTQUFTLElBQUksTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEVBQUUsZ0JBQWdCLEVBQzNFLGlCQUFpQixFQUFFLGdCQUFnQixFQUFFLGtCQUFrQixDQUFDLENBQUM7SUFDL0QsQ0FBQztJQVJELDBDQVFDO0lBRUQsU0FBZ0IsSUFBSSxDQUNoQixPQUFtQixFQUFFLElBQWlCLEVBQUUsU0FBK0IsRUFDdkUsZ0JBQWdDLEVBQUUsaUJBQXdDLEVBQzFFLGdCQUEwQixFQUFFLHFCQUF1QyxFQUFFO1FBQ3ZFLEtBQUssTUFBTSxFQUFFLElBQUksT0FBTyxDQUFDLGNBQWMsRUFBRSxFQUFFO1lBQ3pDLDRCQUFjLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQzdCO1FBRUQsSUFBSSxrQkFBa0IsR0FBb0IsRUFBRSxDQUFDO1FBQzdDLE1BQU0sV0FBVyxHQUFHLE9BQU8sQ0FBQyxjQUFjLEVBQUUsQ0FBQztRQUM3QyxNQUFNLFNBQVMsR0FBRyxPQUFPLENBQUMsa0JBQWtCLEVBQUUsQ0FBQztRQUMvQyxNQUFNLHlCQUF5QixHQUFnRCxFQUFFLENBQUM7UUFDbEYsSUFBSSxJQUFJLENBQUMsdUJBQXVCLEVBQUU7WUFDaEMsOEVBQThFO1lBQzlFLHlCQUF5QixDQUFDLElBQUksQ0FDMUIsc0VBQW1DLENBQUMsU0FBUyxFQUFFLGtCQUFrQixDQUFDLENBQUMsQ0FBQztZQUN4RSx5QkFBeUIsQ0FBQyxJQUFJLENBQzFCLG9DQUFnQixDQUFDLElBQUksRUFBRSxTQUFTLEVBQUUsV0FBVyxFQUFFLGtCQUFrQixDQUFDLENBQUMsQ0FBQztZQUN4RSx5QkFBeUIsQ0FBQyxJQUFJLENBQUMsa0NBQWUsQ0FBQyxXQUFXLEVBQUUsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO1NBQ2xGO1FBQ0QsSUFBSSxJQUFJLENBQUMsbUJBQW1CLEVBQUU7WUFDNUIseUJBQXlCLENBQUMsSUFBSSxDQUFDLCtEQUE2QixDQUFDLFdBQVcsRUFBRSxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7U0FDaEc7UUFDRCxNQUFNLGVBQWUsR0FBRyxJQUFJLGtDQUFlLEVBQUUsQ0FBQztRQUM5QyxNQUFNLGNBQWMsR0FBMEI7WUFDNUMsTUFBTSxFQUFFO2dCQUNOLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxhQUFhLElBQUksRUFBRSxDQUFDO2dCQUMzQyxHQUFHLENBQUMseUJBQXlCLElBQUksRUFBRSxDQUFDLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsa0NBQWtDLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxDQUFDO2dCQUM1RixHQUFHLENBQUMsa0JBQWtCLENBQUMsUUFBUSxJQUFJLEVBQUUsQ0FBQzthQUN2QztZQUNELEtBQUssRUFBRTtnQkFDTCxHQUFHLENBQUMsa0JBQWtCLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQzthQUN0QztZQUNELGlCQUFpQixFQUFFLGtCQUFrQixDQUFDLGlCQUFpQjtTQUN4RCxDQUFDO1FBQ0YsSUFBSSxJQUFJLENBQUMsdUJBQXVCLEVBQUU7WUFDaEMsdUNBQXVDO1lBQ3ZDLGNBQWMsQ0FBQyxNQUFPLENBQUMsSUFBSSxDQUFDLHdDQUFvQixFQUFFLENBQUMsQ0FBQztTQUNyRDtRQUNELElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNuQixjQUFjLENBQUMsS0FBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsK0JBQStCLENBQ2pFLElBQUksRUFBRSxlQUFlLEVBQUUsV0FBVyxFQUFFLGtCQUFrQixDQUFDLENBQUMsQ0FBQztZQUM3RCxjQUFjLENBQUMsS0FBTSxDQUFDLElBQUksQ0FDdEIsZ0VBQW1ELENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO1NBQzlFO1FBRUQsK0RBQStEO1FBQy9ELE1BQU0sYUFBYSxHQUNmLENBQUMsUUFBUSxFQUFFLE9BQU8sRUFBRSxrQkFBa0IsRUFBRSxPQUFPLEVBQUUsV0FBVyxFQUFFLEVBQUU7WUFDOUQsNEJBQWMsQ0FBQyxRQUFRLENBQUMsQ0FBQztZQUN6QixJQUFJLElBQUksQ0FBQyxrQkFBa0IsSUFBSSxnQ0FBYSxDQUFDLFFBQVEsQ0FBQyxJQUFJLFdBQVcsRUFBRTtnQkFDckUsMEZBQTBGO2dCQUMxRix5RkFBeUY7Z0JBQ3pGLGNBQWM7Z0JBQ2QsSUFBSSxDQUFDLFdBQVcsSUFBSSxXQUFXLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtvQkFDMUMsTUFBTSxJQUFJLEtBQUssQ0FBQyx3REFDWixXQUFXLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQztpQkFDM0M7Z0JBQ0QsTUFBTSxjQUFjLEdBQUcsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUN0QyxPQUFPLEdBQUcsZUFBZSxDQUFDLE9BQU8sRUFBRSxjQUFjLEVBQUUsV0FBVyxFQUFFLElBQUksRUFBRSxTQUFTLENBQUMsQ0FBQzthQUNsRjtZQUNELFNBQVMsQ0FBQyxRQUFRLEVBQUUsT0FBTyxFQUFFLGtCQUFrQixFQUFFLE9BQU8sRUFBRSxXQUFXLENBQUMsQ0FBQztRQUN6RSxDQUFDLENBQUM7UUFFTixNQUFNLEVBQUMsV0FBVyxFQUFFLGFBQWEsRUFBRSxXQUFXLEVBQUUsWUFBWSxFQUFDLEdBQUcsT0FBTyxDQUFDLElBQUksQ0FDeEUsZ0JBQWdCLEVBQUUsYUFBYSxFQUFFLGlCQUFpQixFQUFFLGdCQUFnQixFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBRTFGLE1BQU0sT0FBTyxHQUFpQyxFQUFFLENBQUM7UUFDakQsSUFBSSxJQUFJLENBQUMsdUJBQXVCLEVBQUU7WUFDaEMsTUFBTSxXQUFXLEdBQUcsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLGNBQWMsRUFBRSxDQUFDO1lBQ3JGLEtBQUssTUFBTSxVQUFVLElBQUksV0FBVyxFQUFFO2dCQUNwQyxNQUFNLEtBQUssR0FBRyxnQ0FBYSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQztnQkFDakQsSUFBSSxLQUFLLElBQUksSUFBSSxDQUFDLDJCQUEyQixDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsRUFBRTtvQkFDbEUsU0FBUztpQkFDVjtnQkFDRCxNQUFNLEVBQUMsTUFBTSxFQUFFLFdBQVcsRUFBQyxHQUFHLHlCQUFlLENBQ3pDLFdBQVcsRUFBRSxVQUFVLEVBQUUsSUFBSSxFQUFFLElBQUksQ0FBQyxvQkFBb0IsRUFBRSxPQUFPLENBQUMsa0JBQWtCLEVBQUUsQ0FBQyxDQUFDO2dCQUM1RixJQUFJLE1BQU0sRUFBRTtvQkFDVixPQUFPLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxHQUFHLE1BQU0sQ0FBQztpQkFDdkM7Z0JBQ0QsSUFBSSxXQUFXLEVBQUU7b0JBQ2Ysa0JBQWtCLENBQUMsSUFBSSxDQUFDLEdBQUcsV0FBVyxDQUFDLENBQUM7aUJBQ3pDO2FBQ0Y7U0FDRjtRQUNELDhEQUE4RDtRQUM5RCw2REFBNkQ7UUFDN0QsdUVBQXVFO1FBQ3ZFLG9FQUFvRTtRQUNwRSxrQkFBa0IsR0FBRyxrQkFBa0IsQ0FBQyxNQUFNLENBQzFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFFBQVEsS0FBSyxFQUFFLENBQUMsa0JBQWtCLENBQUMsS0FBSztZQUMzQyxDQUFDLElBQUksQ0FBQywyQkFBMkIsQ0FBQyxDQUFDLENBQUMsSUFBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7UUFFN0QsT0FBTztZQUNMLGVBQWU7WUFDZixXQUFXO1lBQ1gsWUFBWSxFQUFFLFlBQVksSUFBSSxFQUFFO1lBQ2hDLFdBQVcsRUFBRSxDQUFDLEdBQUcsYUFBYSxFQUFFLEdBQUcsa0JBQWtCLENBQUM7WUFDdEQsT0FBTztTQUNSLENBQUM7SUFDSixDQUFDO0lBcEdELG9CQW9HQztJQUVELDRFQUE0RTtJQUM1RSxTQUFTLGFBQWEsQ0FBQyxDQUFTLEVBQUUsQ0FBUztRQUN6QyxJQUFJLENBQUMsR0FBRyxDQUFDO1lBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQztRQUNyQixJQUFJLENBQUMsR0FBRyxDQUFDO1lBQUUsT0FBTyxDQUFDLENBQUM7UUFDcEIsT0FBTyxDQUFDLENBQUM7SUFDWCxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSCxTQUFTLGVBQWUsQ0FDcEIsY0FBc0IsRUFBRSxVQUF5QixFQUFFLFdBQTJCLEVBQzlFLElBQWlCLEVBQUUsT0FBMkI7UUFDaEQsTUFBTSxZQUFZLEdBQUcsV0FBVyxDQUFDLG1CQUFtQixDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ2pFLE1BQU0sYUFBYSxHQUFHLFlBQVksSUFBSSxXQUFXLENBQUMsa0JBQWtCLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDbkYsSUFBSSxDQUFDLGFBQWE7WUFBRSxPQUFPLGNBQWMsQ0FBQztRQUUxQyxzRkFBc0Y7UUFDdEYsTUFBTSxjQUFjLEdBQUcsRUFBRSxDQUFDLGVBQWUsQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUN0RCw4RUFBOEU7UUFDOUUsaURBQWlEO1FBQ2pELEVBQUU7UUFDRixTQUFTO1FBQ1Qsd0NBQXdDO1FBQ3hDLEVBQUU7UUFDRixXQUFXO1FBQ1gsMENBQTBDO1FBQzFDLEVBQUU7UUFDRixvQkFBb0I7UUFDcEIsd0NBQXdDO1FBQ3hDLDhCQUE4QjtRQUM5QixFQUFFO1FBQ0YsMEVBQTBFO1FBQzFFLHdFQUF3RTtRQUN4RSw0Q0FBNEM7UUFDNUMsRUFBRTtRQUNGLHFFQUFxRTtRQUNyRSx1RUFBdUU7UUFDdkUsTUFBTSxZQUFZLEdBQUcsYUFBYSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUM1Qyw0RUFBNEU7WUFDNUUsd0VBQXdFO1lBQ3hFLElBQUksQ0FBQyxDQUFDLENBQUMsWUFBWTtnQkFBRSxPQUFPLEtBQUssQ0FBQztZQUVsQywwREFBMEQ7WUFDMUQsc0VBQXNFO1lBQ3RFLDRDQUE0QztZQUM1QyxJQUFJLENBQUMsQ0FBQyxJQUFJLEtBQUssU0FBUztnQkFBRSxPQUFPLEtBQUssQ0FBQztZQUV2QyxrRUFBa0U7WUFDbEUsS0FBSyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsWUFBWSxFQUFFO2dCQUM5QixzRUFBc0U7Z0JBQ3RFLG9FQUFvRTtnQkFDcEUsaUVBQWlFO2dCQUNqRSx5RUFBeUU7Z0JBQ3pFLHFFQUFxRTtnQkFDckUseUVBQXlFO2dCQUN6RSxpRUFBaUU7Z0JBQ2pFLElBQUksQ0FBQyxDQUFDLGFBQWEsRUFBRSxLQUFLLGNBQWMsRUFBRTtvQkFDeEMsT0FBTyxLQUFLLENBQUM7aUJBQ2Q7Z0JBRUQsbUVBQW1FO2dCQUNuRSw2RkFBNkY7Z0JBQzdGLGdDQUFnQztnQkFDaEMsbUNBQW1DO2dCQUNuQyxJQUFJLE9BQU8sQ0FBQyxhQUFhLElBQUssRUFBVSxDQUFDLHVCQUF1QixDQUFDLENBQUMsQ0FBQyxFQUFFLGNBQWMsQ0FBQyxFQUFFO29CQUNwRixPQUFPLEtBQUssQ0FBQztpQkFDZDtnQkFFRCxJQUFJLENBQUMsRUFBRSxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxFQUFFO29CQUM1QixnRUFBZ0U7b0JBQ2hFLE9BQU8sSUFBSSxDQUFDO2lCQUNiO2dCQUVELHdFQUF3RTtnQkFDeEUsMEVBQTBFO2dCQUMxRSwwRUFBMEU7Z0JBQzFFLE1BQU0sV0FBVyxHQUFHLFdBQVcsQ0FBQyxtQ0FBbUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDdkUsc0VBQXNFO2dCQUN0RSxJQUFJLENBQUMsV0FBVztvQkFBRSxPQUFPLEtBQUssQ0FBQztnQkFDL0Isc0VBQXNFO2dCQUN0RSxJQUFJLENBQUMsV0FBVyxDQUFDLFlBQVk7b0JBQUUsT0FBTyxLQUFLLENBQUM7Z0JBRTVDLHdFQUF3RTtnQkFDeEUsbUVBQW1FO2dCQUNuRSxLQUFLLE1BQU0sTUFBTSxJQUFJLFdBQVcsQ0FBQyxZQUFZLEVBQUU7b0JBQzdDLElBQUksTUFBTSxDQUFDLGFBQWEsRUFBRSxLQUFLLGNBQWMsRUFBRTt3QkFDN0MsT0FBTyxLQUFLLENBQUM7cUJBQ2Q7aUJBQ0Y7YUFDRjtZQUNELE9BQU8sSUFBSSxDQUFDO1FBQ2QsQ0FBQyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsWUFBWSxDQUFDLE1BQU07WUFBRSxPQUFPLGNBQWMsQ0FBQztRQUVoRCxxRUFBcUU7UUFDckUsa0VBQWtFO1FBQ2xFLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLElBQUksRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUUzRCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsRUFBRSxFQUFFLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUNsRSxNQUFNLGVBQWUsR0FBRyxVQUFVLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQztRQUV2RCx1RkFBdUY7UUFDdkYsNkJBQTZCO1FBQzdCLDJFQUEyRTtRQUMzRSwwRUFBMEU7UUFDMUUsbUJBQW1CO1FBQ25CLHlGQUF5RjtRQUV6Rix1QkFBdUI7UUFDdkIsSUFBSSxhQUFhLEdBQUcsRUFBRSxDQUFDO1FBQ3ZCLHVCQUF1QjtRQUN2QixJQUFJLGFBQWEsR0FBRyxFQUFFLENBQUM7UUFDdkIsS0FBSyxNQUFNLE1BQU0sSUFBSSxZQUFZLEVBQUU7WUFDakMsSUFBSSxTQUFTLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQztZQUM1QixNQUFNLFdBQVcsR0FBRyxNQUFNLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxhQUFhLEVBQUUsS0FBSyxjQUFjLENBQUMsQ0FBQztZQUN4RixJQUFJLFdBQVcsSUFBSSxFQUFFLENBQUMsaUJBQWlCLENBQUMsV0FBVyxDQUFDLElBQUksV0FBVyxDQUFDLFlBQVksRUFBRTtnQkFDaEYsNEZBQTRGO2dCQUM1Riw4RkFBOEY7Z0JBQzlGLG9FQUFvRTtnQkFDcEUsOEVBQThFO2dCQUM5RSxTQUFTLEdBQUcsV0FBVyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUM7YUFDM0M7WUFDRCxNQUFNLFdBQVcsR0FBRyxtQkFBbUIsZUFBZSxJQUFJLE1BQU0sQ0FBQyxJQUFJLEVBQUUsQ0FBQztZQUN4RSxhQUFhLElBQUksZUFBZSxTQUFTLE9BQU8sV0FBVyxLQUFLLENBQUM7WUFDakUsb0ZBQW9GO1lBQ3BGLG9GQUFvRjtZQUNwRiw0RkFBNEY7WUFDNUYsZ0JBQWdCO1lBQ2hCLGFBQWEsSUFBSSxjQUFjLFNBQVMsc0JBQXNCLFdBQVcsS0FBSyxDQUFDO1lBQy9FLGFBQWEsSUFBSSxlQUFlLFNBQVMsYUFBYSxNQUFNLENBQUMsSUFBSSxNQUFNLENBQUM7U0FDekU7UUFFRCxjQUFjLElBQUksb0JBQW9CLENBQUM7UUFDdkMsY0FBYyxJQUFJLDJCQUEyQixDQUFDO1FBQzlDLGNBQWMsSUFBSSxhQUFhLENBQUM7UUFDaEMsY0FBYyxJQUFJLE9BQU8sQ0FBQztRQUMxQixjQUFjLElBQUksd0NBQXdDLGVBQWUsTUFBTSxDQUFDO1FBQ2hGLGtHQUFrRztRQUNsRyxjQUFjLElBQUksb0RBQW9ELENBQUM7UUFDdkUsY0FBYyxJQUFJLGFBQWEsQ0FBQztRQUNoQyxjQUFjLElBQUksT0FBTyxDQUFDO1FBQzFCLGNBQWMsSUFBSSxLQUFLLENBQUM7UUFFeEIsT0FBTyxjQUFjLENBQUM7SUFDeEIsQ0FBQztJQUVELFNBQVMsa0NBQWtDLENBQ3ZDLElBQWlCLEVBQ2pCLGVBQXFEO1FBQ3ZELE9BQU8sQ0FBQyxPQUFpQyxFQUFFLEVBQUU7WUFDM0MsTUFBTSxRQUFRLEdBQUcsZUFBZSxDQUFDLE9BQU8sQ0FBQyxDQUFDO1lBQzFDLE9BQU8sQ0FBQyxVQUF5QixFQUFFLEVBQUU7Z0JBQ25DLElBQUksSUFBSSxDQUFDLDJCQUEyQixDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsRUFBRTtvQkFDekQsT0FBTyxVQUFVLENBQUM7aUJBQ25CO2dCQUNELE9BQU8sUUFBUSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBQzlCLENBQUMsQ0FBQztRQUNKLENBQUMsQ0FBQztJQUNKLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIEBsaWNlbnNlXG4gKiBDb3B5cmlnaHQgR29vZ2xlIEluYy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiBVc2Ugb2YgdGhpcyBzb3VyY2UgY29kZSBpcyBnb3Zlcm5lZCBieSBhbiBNSVQtc3R5bGUgbGljZW5zZSB0aGF0IGNhbiBiZVxuICogZm91bmQgaW4gdGhlIExJQ0VOU0UgZmlsZSBhdCBodHRwczovL2FuZ3VsYXIuaW8vbGljZW5zZVxuICovXG5cbmltcG9ydCAqIGFzIHRzIGZyb20gJ3R5cGVzY3JpcHQnO1xuXG5pbXBvcnQge0Fubm90YXRvckhvc3R9IGZyb20gJy4vYW5ub3RhdG9yX2hvc3QnO1xuaW1wb3J0IHthc3NlcnRBYnNvbHV0ZX0gZnJvbSAnLi9jbGlfc3VwcG9ydCc7XG5pbXBvcnQge2RlY29yYXRvckRvd25sZXZlbFRyYW5zZm9ybWVyfSBmcm9tICcuL2RlY29yYXRvcl9kb3dubGV2ZWxfdHJhbnNmb3JtZXInO1xuaW1wb3J0IHt0cmFuc2Zvcm1EZWNvcmF0b3JzT3V0cHV0Rm9yQ2xvc3VyZVByb3BlcnR5UmVuYW1pbmd9IGZyb20gJy4vZGVjb3JhdG9ycyc7XG5pbXBvcnQge2VudW1UcmFuc2Zvcm1lcn0gZnJvbSAnLi9lbnVtX3RyYW5zZm9ybWVyJztcbmltcG9ydCB7Z2VuZXJhdGVFeHRlcm5zfSBmcm9tICcuL2V4dGVybnMnO1xuaW1wb3J0IHt0cmFuc2Zvcm1GaWxlb3ZlcnZpZXdDb21tZW50RmFjdG9yeX0gZnJvbSAnLi9maWxlb3ZlcnZpZXdfY29tbWVudF90cmFuc2Zvcm1lcic7XG5pbXBvcnQgKiBhcyBnb29nbW9kdWxlIGZyb20gJy4vZ29vZ21vZHVsZSc7XG5pbXBvcnQge2pzZG9jVHJhbnNmb3JtZXIsIHJlbW92ZVR5cGVBc3NlcnRpb25zfSBmcm9tICcuL2pzZG9jX3RyYW5zZm9ybWVyJztcbmltcG9ydCB7TW9kdWxlc01hbmlmZXN0fSBmcm9tICcuL21vZHVsZXNfbWFuaWZlc3QnO1xuaW1wb3J0IHtpc0R0c0ZpbGVOYW1lfSBmcm9tICcuL3RyYW5zZm9ybWVyX3V0aWwnO1xuXG4vLyBFeHBvcnRlZCBmb3IgdXNlcnMgYXMgYSBkZWZhdWx0IGltcGwgb2YgcGF0aFRvTW9kdWxlTmFtZS5cbmV4cG9ydCB7cGF0aFRvTW9kdWxlTmFtZX0gZnJvbSAnLi9jbGlfc3VwcG9ydCc7XG4vLyBSZXRhaW5lZCBoZXJlIGZvciBBUEkgY29tcGF0aWJpbGl0eS5cbmV4cG9ydCB7Z2V0R2VuZXJhdGVkRXh0ZXJuc30gZnJvbSAnLi9leHRlcm5zJztcbmV4cG9ydCB7RmlsZU1hcCwgTW9kdWxlc01hbmlmZXN0fSBmcm9tICcuL21vZHVsZXNfbWFuaWZlc3QnO1xuXG5leHBvcnQgaW50ZXJmYWNlIFRzaWNrbGVIb3N0IGV4dGVuZHMgZ29vZ21vZHVsZS5Hb29nTW9kdWxlUHJvY2Vzc29ySG9zdCwgQW5ub3RhdG9ySG9zdCB7XG4gIC8qKlxuICAgKiBXaGV0aGVyIHRvIGRvd25sZXZlbCBkZWNvcmF0b3JzXG4gICAqL1xuICB0cmFuc2Zvcm1EZWNvcmF0b3JzPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogV2hldGhlciB0byBjb252ZXJzIHR5cGVzIHRvIGNsb3N1cmVcbiAgICovXG4gIHRyYW5zZm9ybVR5cGVzVG9DbG9zdXJlPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogV2hldGhlciB0byBhZGQgYWxpYXNlcyB0byB0aGUgLmQudHMgZmlsZXMgdG8gYWRkIHRoZSBleHBvcnRzIHRvIHRoZVxuICAgKiDgsqBf4LKgLmNsdXR6IG5hbWVzcGFjZS5cbiAgICovXG4gIGFkZER0c0NsdXR6QWxpYXNlcz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIElmIHRydWUsIHRzaWNrbGUgYW5kIGRlY29yYXRvciBkb3dubGV2ZWwgcHJvY2Vzc2luZyB3aWxsIGJlIHNraXBwZWQgZm9yXG4gICAqIHRoYXQgZmlsZS5cbiAgICovXG4gIHNob3VsZFNraXBUc2lja2xlUHJvY2Vzc2luZyhmaWxlTmFtZTogc3RyaW5nKTogYm9vbGVhbjtcblxuICAvKipcbiAgICogVHNpY2tsZSB0cmVhdHMgd2FybmluZ3MgYXMgZXJyb3JzLCBpZiB0cnVlLCBpZ25vcmUgd2FybmluZ3MuICBUaGlzIG1pZ2h0IGJlXG4gICAqIHVzZWZ1bCBmb3IgZS5nLiB0aGlyZCBwYXJ0eSBjb2RlLlxuICAgKi9cbiAgc2hvdWxkSWdub3JlV2FybmluZ3NGb3JQYXRoKGZpbGVQYXRoOiBzdHJpbmcpOiBib29sZWFuO1xuXG4gIC8qKiBXaGV0aGVyIHRvIGNvbnZlcnQgQ29tbW9uSlMgcmVxdWlyZSgpIGltcG9ydHMgdG8gZ29vZy5tb2R1bGUoKSBhbmQgZ29vZy5yZXF1aXJlKCkgY2FsbHMuICovXG4gIGdvb2dtb2R1bGU6IGJvb2xlYW47XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBtZXJnZUVtaXRSZXN1bHRzKGVtaXRSZXN1bHRzOiBFbWl0UmVzdWx0W10pOiBFbWl0UmVzdWx0IHtcbiAgY29uc3QgZGlhZ25vc3RpY3M6IHRzLkRpYWdub3N0aWNbXSA9IFtdO1xuICBsZXQgZW1pdFNraXBwZWQgPSB0cnVlO1xuICBjb25zdCBlbWl0dGVkRmlsZXM6IHN0cmluZ1tdID0gW107XG4gIGNvbnN0IGV4dGVybnM6IHtbZmlsZU5hbWU6IHN0cmluZ106IHN0cmluZ30gPSB7fTtcbiAgY29uc3QgbW9kdWxlc01hbmlmZXN0ID0gbmV3IE1vZHVsZXNNYW5pZmVzdCgpO1xuICBmb3IgKGNvbnN0IGVyIG9mIGVtaXRSZXN1bHRzKSB7XG4gICAgZGlhZ25vc3RpY3MucHVzaCguLi5lci5kaWFnbm9zdGljcyk7XG4gICAgZW1pdFNraXBwZWQgPSBlbWl0U2tpcHBlZCB8fCBlci5lbWl0U2tpcHBlZDtcbiAgICBpZiAoZXIuZW1pdHRlZEZpbGVzKSB7XG4gICAgICBlbWl0dGVkRmlsZXMucHVzaCguLi5lci5lbWl0dGVkRmlsZXMpO1xuICAgIH1cbiAgICBPYmplY3QuYXNzaWduKGV4dGVybnMsIGVyLmV4dGVybnMpO1xuICAgIG1vZHVsZXNNYW5pZmVzdC5hZGRNYW5pZmVzdChlci5tb2R1bGVzTWFuaWZlc3QpO1xuICB9XG4gIHJldHVybiB7ZGlhZ25vc3RpY3MsIGVtaXRTa2lwcGVkLCBlbWl0dGVkRmlsZXMsIGV4dGVybnMsIG1vZHVsZXNNYW5pZmVzdH07XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgRW1pdFJlc3VsdCBleHRlbmRzIHRzLkVtaXRSZXN1bHQge1xuICAvLyBUaGUgbWFuaWZlc3Qgb2YgSlMgbW9kdWxlcyBvdXRwdXQgYnkgdGhlIGNvbXBpbGVyLlxuICBtb2R1bGVzTWFuaWZlc3Q6IE1vZHVsZXNNYW5pZmVzdDtcbiAgLyoqXG4gICAqIGV4dGVybnMuanMgZmlsZXMgcHJvZHVjZWQgYnkgdHNpY2tsZSwgaWYgYW55LiBtb2R1bGUgSURzIGFyZSByZWxhdGl2ZSBwYXRocyBmcm9tXG4gICAqIGZpbGVOYW1lVG9Nb2R1bGVJZC5cbiAgICovXG4gIGV4dGVybnM6IHtbbW9kdWxlSWQ6IHN0cmluZ106IHN0cmluZ307XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgRW1pdFRyYW5zZm9ybWVycyB7XG4gIC8qKiBDdXN0b20gdHJhbnNmb3JtZXJzIHRvIGV2YWx1YXRlIGJlZm9yZSBUc2lja2xlIC5qcyB0cmFuc2Zvcm1hdGlvbnMuICovXG4gIGJlZm9yZVRzaWNrbGU/OiB0cy5DdXN0b21UcmFuc2Zvcm1lcnNbJ2JlZm9yZSddO1xuICAvKiogQ3VzdG9tIHRyYW5zZm9ybWVycyB0byBldmFsdWF0ZSBiZWZvcmUgYnVpbHQtaW4gLmpzIHRyYW5zZm9ybWF0aW9ucy4gKi9cbiAgYmVmb3JlVHM/OiB0cy5DdXN0b21UcmFuc2Zvcm1lcnNbJ2JlZm9yZSddO1xuICAvKiogQ3VzdG9tIHRyYW5zZm9ybWVycyB0byBldmFsdWF0ZSBhZnRlciBidWlsdC1pbiAuanMgdHJhbnNmb3JtYXRpb25zLiAqL1xuICBhZnRlclRzPzogdHMuQ3VzdG9tVHJhbnNmb3JtZXJzWydhZnRlciddO1xuICAvKiogQ3VzdG9tIHRyYW5zZm9ybWVycyB0byBldmFsdWF0ZSBhZnRlciBidWlsdC1pbiAuZC50cyB0cmFuc2Zvcm1hdGlvbnMuICovXG4gIGFmdGVyRGVjbGFyYXRpb25zPzogdHMuQ3VzdG9tVHJhbnNmb3JtZXJzWydhZnRlckRlY2xhcmF0aW9ucyddO1xufVxuXG5cbi8qKiBAZGVwcmVjYXRlZCBFeHBvc2VkIGZvciBiYWNrd2FyZCBjb21wYXQgd2l0aCBBbmd1bGFyLiAgVXNlIGVtaXQoKSBpbnN0ZWFkLiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGVtaXRXaXRoVHNpY2tsZShcbiAgICBwcm9ncmFtOiB0cy5Qcm9ncmFtLCBob3N0OiBUc2lja2xlSG9zdCwgdHNIb3N0OiB0cy5Db21waWxlckhvc3QsIHRzT3B0aW9uczogdHMuQ29tcGlsZXJPcHRpb25zLFxuICAgIHRhcmdldFNvdXJjZUZpbGU/OiB0cy5Tb3VyY2VGaWxlLCB3cml0ZUZpbGU/OiB0cy5Xcml0ZUZpbGVDYWxsYmFjayxcbiAgICBjYW5jZWxsYXRpb25Ub2tlbj86IHRzLkNhbmNlbGxhdGlvblRva2VuLCBlbWl0T25seUR0c0ZpbGVzPzogYm9vbGVhbixcbiAgICBjdXN0b21UcmFuc2Zvcm1lcnM6IEVtaXRUcmFuc2Zvcm1lcnMgPSB7fSk6IEVtaXRSZXN1bHQge1xuICByZXR1cm4gZW1pdChcbiAgICAgIHByb2dyYW0sIGhvc3QsIHdyaXRlRmlsZSB8fCB0c0hvc3Qud3JpdGVGaWxlLmJpbmQodHNIb3N0KSwgdGFyZ2V0U291cmNlRmlsZSxcbiAgICAgIGNhbmNlbGxhdGlvblRva2VuLCBlbWl0T25seUR0c0ZpbGVzLCBjdXN0b21UcmFuc2Zvcm1lcnMpO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gZW1pdChcbiAgICBwcm9ncmFtOiB0cy5Qcm9ncmFtLCBob3N0OiBUc2lja2xlSG9zdCwgd3JpdGVGaWxlOiB0cy5Xcml0ZUZpbGVDYWxsYmFjayxcbiAgICB0YXJnZXRTb3VyY2VGaWxlPzogdHMuU291cmNlRmlsZSwgY2FuY2VsbGF0aW9uVG9rZW4/OiB0cy5DYW5jZWxsYXRpb25Ub2tlbixcbiAgICBlbWl0T25seUR0c0ZpbGVzPzogYm9vbGVhbiwgY3VzdG9tVHJhbnNmb3JtZXJzOiBFbWl0VHJhbnNmb3JtZXJzID0ge30pOiBFbWl0UmVzdWx0IHtcbiAgZm9yIChjb25zdCBzZiBvZiBwcm9ncmFtLmdldFNvdXJjZUZpbGVzKCkpIHtcbiAgICBhc3NlcnRBYnNvbHV0ZShzZi5maWxlTmFtZSk7XG4gIH1cblxuICBsZXQgdHNpY2tsZURpYWdub3N0aWNzOiB0cy5EaWFnbm9zdGljW10gPSBbXTtcbiAgY29uc3QgdHlwZUNoZWNrZXIgPSBwcm9ncmFtLmdldFR5cGVDaGVja2VyKCk7XG4gIGNvbnN0IHRzT3B0aW9ucyA9IHByb2dyYW0uZ2V0Q29tcGlsZXJPcHRpb25zKCk7XG4gIGNvbnN0IHRzaWNrbGVTb3VyY2VUcmFuc2Zvcm1lcnM6IEFycmF5PHRzLlRyYW5zZm9ybWVyRmFjdG9yeTx0cy5Tb3VyY2VGaWxlPj4gPSBbXTtcbiAgaWYgKGhvc3QudHJhbnNmb3JtVHlwZXNUb0Nsb3N1cmUpIHtcbiAgICAvLyBPbmx5IGFkZCBAc3VwcHJlc3Mge2NoZWNrVHlwZXN9IGNvbW1lbnRzIHdoZW4gYWxzbyBhZGRpbmcgdHlwZSBhbm5vdGF0aW9ucy5cbiAgICB0c2lja2xlU291cmNlVHJhbnNmb3JtZXJzLnB1c2goXG4gICAgICAgIHRyYW5zZm9ybUZpbGVvdmVydmlld0NvbW1lbnRGYWN0b3J5KHRzT3B0aW9ucywgdHNpY2tsZURpYWdub3N0aWNzKSk7XG4gICAgdHNpY2tsZVNvdXJjZVRyYW5zZm9ybWVycy5wdXNoKFxuICAgICAgICBqc2RvY1RyYW5zZm9ybWVyKGhvc3QsIHRzT3B0aW9ucywgdHlwZUNoZWNrZXIsIHRzaWNrbGVEaWFnbm9zdGljcykpO1xuICAgIHRzaWNrbGVTb3VyY2VUcmFuc2Zvcm1lcnMucHVzaChlbnVtVHJhbnNmb3JtZXIodHlwZUNoZWNrZXIsIHRzaWNrbGVEaWFnbm9zdGljcykpO1xuICB9XG4gIGlmIChob3N0LnRyYW5zZm9ybURlY29yYXRvcnMpIHtcbiAgICB0c2lja2xlU291cmNlVHJhbnNmb3JtZXJzLnB1c2goZGVjb3JhdG9yRG93bmxldmVsVHJhbnNmb3JtZXIodHlwZUNoZWNrZXIsIHRzaWNrbGVEaWFnbm9zdGljcykpO1xuICB9XG4gIGNvbnN0IG1vZHVsZXNNYW5pZmVzdCA9IG5ldyBNb2R1bGVzTWFuaWZlc3QoKTtcbiAgY29uc3QgdHNUcmFuc2Zvcm1lcnM6IHRzLkN1c3RvbVRyYW5zZm9ybWVycyA9IHtcbiAgICBiZWZvcmU6IFtcbiAgICAgIC4uLihjdXN0b21UcmFuc2Zvcm1lcnMuYmVmb3JlVHNpY2tsZSB8fCBbXSksXG4gICAgICAuLi4odHNpY2tsZVNvdXJjZVRyYW5zZm9ybWVycyB8fCBbXSkubWFwKHRmID0+IHNraXBUcmFuc2Zvcm1Gb3JTb3VyY2VGaWxlSWZOZWVkZWQoaG9zdCwgdGYpKSxcbiAgICAgIC4uLihjdXN0b21UcmFuc2Zvcm1lcnMuYmVmb3JlVHMgfHwgW10pLFxuICAgIF0sXG4gICAgYWZ0ZXI6IFtcbiAgICAgIC4uLihjdXN0b21UcmFuc2Zvcm1lcnMuYWZ0ZXJUcyB8fCBbXSksXG4gICAgXSxcbiAgICBhZnRlckRlY2xhcmF0aW9uczogY3VzdG9tVHJhbnNmb3JtZXJzLmFmdGVyRGVjbGFyYXRpb25zLFxuICB9O1xuICBpZiAoaG9zdC50cmFuc2Zvcm1UeXBlc1RvQ2xvc3VyZSkge1xuICAgIC8vIFNlZSBjb21tZW50IG9uIHJlbW90ZVR5cGVBc3NlcnRpb25zLlxuICAgIHRzVHJhbnNmb3JtZXJzLmJlZm9yZSEucHVzaChyZW1vdmVUeXBlQXNzZXJ0aW9ucygpKTtcbiAgfVxuICBpZiAoaG9zdC5nb29nbW9kdWxlKSB7XG4gICAgdHNUcmFuc2Zvcm1lcnMuYWZ0ZXIhLnB1c2goZ29vZ21vZHVsZS5jb21tb25Kc1RvR29vZ21vZHVsZVRyYW5zZm9ybWVyKFxuICAgICAgICBob3N0LCBtb2R1bGVzTWFuaWZlc3QsIHR5cGVDaGVja2VyLCB0c2lja2xlRGlhZ25vc3RpY3MpKTtcbiAgICB0c1RyYW5zZm9ybWVycy5hZnRlciEucHVzaChcbiAgICAgICAgdHJhbnNmb3JtRGVjb3JhdG9yc091dHB1dEZvckNsb3N1cmVQcm9wZXJ0eVJlbmFtaW5nKHRzaWNrbGVEaWFnbm9zdGljcykpO1xuICB9XG5cbiAgLy8gV3JhcCB0aGUgd3JpdGVGaWxlIGNhbGxiYWNrIHRvIGhvb2sgd3JpdGluZyBvZiB0aGUgZHRzIGZpbGUuXG4gIGNvbnN0IHdyaXRlRmlsZUltcGw6IHRzLldyaXRlRmlsZUNhbGxiYWNrID1cbiAgICAgIChmaWxlTmFtZSwgY29udGVudCwgd3JpdGVCeXRlT3JkZXJNYXJrLCBvbkVycm9yLCBzb3VyY2VGaWxlcykgPT4ge1xuICAgICAgICBhc3NlcnRBYnNvbHV0ZShmaWxlTmFtZSk7XG4gICAgICAgIGlmIChob3N0LmFkZER0c0NsdXR6QWxpYXNlcyAmJiBpc0R0c0ZpbGVOYW1lKGZpbGVOYW1lKSAmJiBzb3VyY2VGaWxlcykge1xuICAgICAgICAgIC8vIE9ubHkgYnVuZGxlIGVtaXRzIHBhc3MgbW9yZSB0aGFuIG9uZSBzb3VyY2UgZmlsZSBmb3IgLmQudHMgd3JpdGVzLiBCdW5kbGUgZW1pdHMgaG93ZXZlclxuICAgICAgICAgIC8vIGFyZSBub3Qgc3VwcG9ydGVkIGJ5IHRzaWNrbGUsIGFzIHdlIGNhbm5vdCBhbm5vdGF0ZSB0aGVtIGZvciBDbG9zdXJlIGluIGFueSBtZWFuaW5nZnVsXG4gICAgICAgICAgLy8gd2F5IGFueXdheS5cbiAgICAgICAgICBpZiAoIXNvdXJjZUZpbGVzIHx8IHNvdXJjZUZpbGVzLmxlbmd0aCA+IDEpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgZXhwZWN0ZWQgZXhhY3RseSBvbmUgc291cmNlIGZpbGUgZm9yIC5kLnRzIGVtaXQsIGdvdCAke1xuICAgICAgICAgICAgICAgIHNvdXJjZUZpbGVzLm1hcChzZiA9PiBzZi5maWxlTmFtZSl9YCk7XG4gICAgICAgICAgfVxuICAgICAgICAgIGNvbnN0IG9yaWdpbmFsU291cmNlID0gc291cmNlRmlsZXNbMF07XG4gICAgICAgICAgY29udGVudCA9IGFkZENsdXR6QWxpYXNlcyhjb250ZW50LCBvcmlnaW5hbFNvdXJjZSwgdHlwZUNoZWNrZXIsIGhvc3QsIHRzT3B0aW9ucyk7XG4gICAgICAgIH1cbiAgICAgICAgd3JpdGVGaWxlKGZpbGVOYW1lLCBjb250ZW50LCB3cml0ZUJ5dGVPcmRlck1hcmssIG9uRXJyb3IsIHNvdXJjZUZpbGVzKTtcbiAgICAgIH07XG5cbiAgY29uc3Qge2RpYWdub3N0aWNzOiB0c0RpYWdub3N0aWNzLCBlbWl0U2tpcHBlZCwgZW1pdHRlZEZpbGVzfSA9IHByb2dyYW0uZW1pdChcbiAgICAgIHRhcmdldFNvdXJjZUZpbGUsIHdyaXRlRmlsZUltcGwsIGNhbmNlbGxhdGlvblRva2VuLCBlbWl0T25seUR0c0ZpbGVzLCB0c1RyYW5zZm9ybWVycyk7XG5cbiAgY29uc3QgZXh0ZXJuczoge1tmaWxlTmFtZTogc3RyaW5nXTogc3RyaW5nfSA9IHt9O1xuICBpZiAoaG9zdC50cmFuc2Zvcm1UeXBlc1RvQ2xvc3VyZSkge1xuICAgIGNvbnN0IHNvdXJjZUZpbGVzID0gdGFyZ2V0U291cmNlRmlsZSA/IFt0YXJnZXRTb3VyY2VGaWxlXSA6IHByb2dyYW0uZ2V0U291cmNlRmlsZXMoKTtcbiAgICBmb3IgKGNvbnN0IHNvdXJjZUZpbGUgb2Ygc291cmNlRmlsZXMpIHtcbiAgICAgIGNvbnN0IGlzRHRzID0gaXNEdHNGaWxlTmFtZShzb3VyY2VGaWxlLmZpbGVOYW1lKTtcbiAgICAgIGlmIChpc0R0cyAmJiBob3N0LnNob3VsZFNraXBUc2lja2xlUHJvY2Vzc2luZyhzb3VyY2VGaWxlLmZpbGVOYW1lKSkge1xuICAgICAgICBjb250aW51ZTtcbiAgICAgIH1cbiAgICAgIGNvbnN0IHtvdXRwdXQsIGRpYWdub3N0aWNzfSA9IGdlbmVyYXRlRXh0ZXJucyhcbiAgICAgICAgICB0eXBlQ2hlY2tlciwgc291cmNlRmlsZSwgaG9zdCwgaG9zdC5tb2R1bGVSZXNvbHV0aW9uSG9zdCwgcHJvZ3JhbS5nZXRDb21waWxlck9wdGlvbnMoKSk7XG4gICAgICBpZiAob3V0cHV0KSB7XG4gICAgICAgIGV4dGVybnNbc291cmNlRmlsZS5maWxlTmFtZV0gPSBvdXRwdXQ7XG4gICAgICB9XG4gICAgICBpZiAoZGlhZ25vc3RpY3MpIHtcbiAgICAgICAgdHNpY2tsZURpYWdub3N0aWNzLnB1c2goLi4uZGlhZ25vc3RpY3MpO1xuICAgICAgfVxuICAgIH1cbiAgfVxuICAvLyBBbGwgZGlhZ25vc3RpY3MgKGluY2x1ZGluZyB3YXJuaW5ncykgYXJlIHRyZWF0ZWQgYXMgZXJyb3JzLlxuICAvLyBJZiB0aGUgaG9zdCBkZWNpZGVzIHRvIGlnbm9yZSB3YXJuaW5ncywganVzdCBkaXNjYXJkIHRoZW0uXG4gIC8vIFdhcm5pbmdzIGluY2x1ZGUgc3R1ZmYgbGlrZSBcImRvbid0IHVzZSBAdHlwZSBpbiB5b3VyIGpzZG9jXCI7IHRzaWNrbGVcbiAgLy8gd2FybnMgYW5kIHRoZW4gZml4ZXMgdXAgdGhlIGNvZGUgdG8gYmUgQ2xvc3VyZS1jb21wYXRpYmxlIGFueXdheS5cbiAgdHNpY2tsZURpYWdub3N0aWNzID0gdHNpY2tsZURpYWdub3N0aWNzLmZpbHRlcihcbiAgICAgIGQgPT4gZC5jYXRlZ29yeSA9PT0gdHMuRGlhZ25vc3RpY0NhdGVnb3J5LkVycm9yIHx8XG4gICAgICAgICAgIWhvc3Quc2hvdWxkSWdub3JlV2FybmluZ3NGb3JQYXRoKGQuZmlsZSEuZmlsZU5hbWUpKTtcblxuICByZXR1cm4ge1xuICAgIG1vZHVsZXNNYW5pZmVzdCxcbiAgICBlbWl0U2tpcHBlZCxcbiAgICBlbWl0dGVkRmlsZXM6IGVtaXR0ZWRGaWxlcyB8fCBbXSxcbiAgICBkaWFnbm9zdGljczogWy4uLnRzRGlhZ25vc3RpY3MsIC4uLnRzaWNrbGVEaWFnbm9zdGljc10sXG4gICAgZXh0ZXJuc1xuICB9O1xufVxuXG4vKiogQ29tcGFyZXMgdHdvIHN0cmluZ3MgYW5kIHJldHVybnMgYSBudW1iZXIgc3VpdGFibGUgZm9yIHVzZSBpbiBzb3J0KCkuICovXG5mdW5jdGlvbiBzdHJpbmdDb21wYXJlKGE6IHN0cmluZywgYjogc3RyaW5nKTogbnVtYmVyIHtcbiAgaWYgKGEgPCBiKSByZXR1cm4gLTE7XG4gIGlmIChhID4gYikgcmV0dXJuIDE7XG4gIHJldHVybiAwO1xufVxuXG4vKipcbiAqIEEgdHNpY2tsZSBwcm9kdWNlZCBkZWNsYXJhdGlvbiBmaWxlIG1pZ2h0IGJlIGNvbnN1bWVkIGJlIHJlZmVyZW5jZWQgYnkgQ2x1dHpcbiAqIHByb2R1Y2VkIC5kLnRzIGZpbGVzLCB3aGljaCB1c2Ugc3ltYm9sIG5hbWVzIGJhc2VkIG9uIENsb3N1cmUncyBpbnRlcm5hbFxuICogbmFtaW5nIGNvbnZlbnRpb25zLCBzbyB3ZSBuZWVkIHRvIHByb3ZpZGUgYWxpYXNlcyBmb3IgYWxsIHRoZSBleHBvcnRlZCBzeW1ib2xzXG4gKiBpbiB0aGUgQ2x1dHogbmFtaW5nIGNvbnZlbnRpb24uXG4gKi9cbmZ1bmN0aW9uIGFkZENsdXR6QWxpYXNlcyhcbiAgICBkdHNGaWxlQ29udGVudDogc3RyaW5nLCBzb3VyY2VGaWxlOiB0cy5Tb3VyY2VGaWxlLCB0eXBlQ2hlY2tlcjogdHMuVHlwZUNoZWNrZXIsXG4gICAgaG9zdDogVHNpY2tsZUhvc3QsIG9wdGlvbnM6IHRzLkNvbXBpbGVyT3B0aW9ucyk6IHN0cmluZyB7XG4gIGNvbnN0IG1vZHVsZVN5bWJvbCA9IHR5cGVDaGVja2VyLmdldFN5bWJvbEF0TG9jYXRpb24oc291cmNlRmlsZSk7XG4gIGNvbnN0IG1vZHVsZUV4cG9ydHMgPSBtb2R1bGVTeW1ib2wgJiYgdHlwZUNoZWNrZXIuZ2V0RXhwb3J0c09mTW9kdWxlKG1vZHVsZVN5bWJvbCk7XG4gIGlmICghbW9kdWxlRXhwb3J0cykgcmV0dXJuIGR0c0ZpbGVDb250ZW50O1xuXG4gIC8vIC5kLnRzIGZpbGVzIGNhbiBiZSB0cmFuc2Zvcm1lZCwgdG9vLCBzbyB3ZSBuZWVkIHRvIGNvbXBhcmUgdGhlIG9yaWdpbmFsIG5vZGUgYmVsb3cuXG4gIGNvbnN0IG9yaWdTb3VyY2VGaWxlID0gdHMuZ2V0T3JpZ2luYWxOb2RlKHNvdXJjZUZpbGUpO1xuICAvLyBJbiBvcmRlciB0byB3cml0ZSBhbGlhc2VzLCB0aGUgZXhwb3J0ZWQgc3ltYm9scyBuZWVkIHRvIGJlIGF2YWlsYWJsZSBpbiB0aGVcbiAgLy8gdGhlIG1vZHVsZSBzY29wZS4gVGhhdCBpcyBub3QgYWx3YXlzIHRoZSBjYXNlOlxuICAvL1xuICAvLyBleHBvcnRcbiAgLy8gMSkgZXhwb3J0IGNvbnN0IFg7ICAgICAgICAgICAvLyB3b3Jrc1xuICAvL1xuICAvLyByZWV4cG9ydFxuICAvLyAyKSBleHBvcnQge1h9IGZyb20gJy4vZm9vJzsgIC8vIGRvZXNuJ3RcbiAgLy9cbiAgLy8gaW1wb3J0ZWQgcmVleHBvcnRcbiAgLy8gMykgaW1wb3J0IHtYfSBmcm9tICcuL2Zvbyc7ICAvLyB3b3Jrc1xuICAvLyAgICBleHBvcnQge1h9IGZyb20gJy4vZm9vJztcbiAgLy9cbiAgLy8gZ2V0RXhwb3J0c09mTW9kdWxlIHJldHVybnMgYWxsIHRocmVlIHR5cGVzLCBidXQgd2UgbmVlZCB0byBzZXBhcmF0ZSAyKS5cbiAgLy8gRm9yIG5vdyB3ZSAnZml4JyAyKSBieSBzaW1wbHkgbm90IGVtaXR0aW5nIGEgY2x1dHogYWxpYXMsIHNpbmNlIGNsdXR6XG4gIC8vIGludGVyb3AgaXMgdXNlZCBpbiBtaW5vcml0eSBvZiBzY2VuYXJpb3MuXG4gIC8vXG4gIC8vIFRPRE8ocmFkb2tpcm92KTogYXR0ZW1wdCB0byBhZGQgYXBwcm9wcmlhdGUgaW1wb3J0cyBmb3IgMikgc28gdGhhdFxuICAvLyBjdXJyZW50bHkgZmluZGluZyBvdXQgbG9jYWwgYXBwZWFycyBldmVuIGhhcmRlciB0aGFuIGZpeGluZyBleHBvcnRzLlxuICBjb25zdCBsb2NhbEV4cG9ydHMgPSBtb2R1bGVFeHBvcnRzLmZpbHRlcihlID0+IHtcbiAgICAvLyBJZiB0aGVyZSBhcmUgbm8gZGVjbGFyYXRpb25zLCBiZSBjb25zZXJ2YXRpdmUgYW5kIGRvbid0IGVtaXQgdGhlIGFsaWFzZXMuXG4gICAgLy8gSSBkb24ndCBrbm93IGhvdyBjYW4gdGhpcyBoYXBwZW4sIHdlIGhhdmUgbm8gdGVzdHMgdGhhdCBleGNlcmNpc2UgaXQuXG4gICAgaWYgKCFlLmRlY2xhcmF0aW9ucykgcmV0dXJuIGZhbHNlO1xuXG4gICAgLy8gU2tpcCBkZWZhdWx0IGV4cG9ydHMsIHRoZXkgYXJlIG5vdCBjdXJyZW50bHkgc3VwcG9ydGVkLlxuICAgIC8vIGRlZmF1bHQgaXMgYSBrZXl3b3JkIGluIHR5cGVzY3JpcHQsIHNvIHRoZSBuYW1lIG9mIHRoZSBleHBvcnQgYmVpbmdcbiAgICAvLyBkZWZhdWx0IG1lYW5zIHRoYXQgaXQncyBhIGRlZmF1bHQgZXhwb3J0LlxuICAgIGlmIChlLm5hbWUgPT09ICdkZWZhdWx0JykgcmV0dXJuIGZhbHNlO1xuXG4gICAgLy8gVXNlIHRoZSBkZWNsYXJhdGlvbiBsb2NhdGlvbiB0byBkZXRlcm1pbmUgc2VwYXJhdGUgY2FzZXMgYWJvdmUuXG4gICAgZm9yIChjb25zdCBkIG9mIGUuZGVjbGFyYXRpb25zKSB7XG4gICAgICAvLyBUaGlzIGlzIGEgc3BlY2lhbCBjYXNlIGZvciBleHBvcnQgKi4gVGVjaG5pY2FsbHksIGl0IGlzIG91dHNpZGUgdGhlXG4gICAgICAvLyB0aHJlZSBjYXNlcyBvdXRsaW5lZCwgYnV0IGF0IHRoaXMgcG9pbnQgd2UgaGF2ZSByZXdyaXR0ZW4gaXQgdG8gYVxuICAgICAgLy8gcmVleHBvcnQgb3IgYW4gaW1wb3J0ZWQgcmVleHBvcnQuIEhvd2V2ZXIsIGl0IGFwcGVhcnMgdGhhdCB0aGVcbiAgICAgIC8vIHJld3JpdGluZyBhbHNvIGhhcyBtYWRlIGl0IGJlaGF2ZSBkaWZmZXJlbnQgZnJvbSBleHBsaWNpdCBuYW1lZCBleHBvcnRcbiAgICAgIC8vIGluIHRoZSBzZW5zZSB0aGF0IHRoZSBkZWNsYXJhdGlvbiBhcHBlYXJzIHRvIHBvaW50IGF0IHRoZSBvcmlnaW5hbFxuICAgICAgLy8gbG9jYXRpb24gbm90IHRoZSByZWV4cG9ydCBsb2NhdGlvbi4gIFNpbmNlIHdlIGNhbid0IGZpZ3VyZSBvdXQgd2hldGhlclxuICAgICAgLy8gdGhlcmUgaXMgYSBsb2NhbCBpbXBvcnQgaGVyZSwgd2UgZXJyIG9uIHRoZSBzaWRlIG9mIGxlc3MgZW1pdC5cbiAgICAgIGlmIChkLmdldFNvdXJjZUZpbGUoKSAhPT0gb3JpZ1NvdXJjZUZpbGUpIHtcbiAgICAgICAgcmV0dXJuIGZhbHNlO1xuICAgICAgfVxuXG4gICAgICAvLyBAaW50ZXJuYWwgbWFya2VkIEFQSXMgYXJlIG5vdCBleHBvcnRlZCwgc28gbXVzdCBub3QgZ2V0IGFsaWFzZXMuXG4gICAgICAvLyBUaGlzIHVzZXMgYW4gaW50ZXJuYWwgVFMgQVBJLCBhc3N1bWluZyB0aGF0IGFjY2Vzc2luZyB0aGlzIHdpbGwgYmUgbW9yZSBzdGFibGUgY29tcGFyZWQgdG9cbiAgICAgIC8vIGltcGxlbWVudGluZyBvdXIgb3duIHZlcnNpb24uXG4gICAgICAvLyB0c2xpbnQ6ZGlzYWJsZS1uZXh0LWxpbmU6IG5vLWFueVxuICAgICAgaWYgKG9wdGlvbnMuc3RyaXBJbnRlcm5hbCAmJiAodHMgYXMgYW55KVsnaXNJbnRlcm5hbERlY2xhcmF0aW9uJ10oZCwgb3JpZ1NvdXJjZUZpbGUpKSB7XG4gICAgICAgIHJldHVybiBmYWxzZTtcbiAgICAgIH1cblxuICAgICAgaWYgKCF0cy5pc0V4cG9ydFNwZWNpZmllcihkKSkge1xuICAgICAgICAvLyB3ZSBoYXZlIGEgcHVyZSBleHBvcnQgKGNhc2UgMSkgdGh1cyBzYWZlIHRvIGVtaXQgY2x1dHogYWxpYXMuXG4gICAgICAgIHJldHVybiB0cnVlO1xuICAgICAgfVxuXG4gICAgICAvLyBUaGUgZGVjbGFyYXRpb24gZCBpcyB1c2VsZXNzIHRvIHNlcGFyYXRlIHJlZXhwb3J0IGFuZCBpbXBvcnQtcmVleHBvcnRcbiAgICAgIC8vIGJlY2F1c2UgdGhleSBib3RoIHBvaW50IHRvIHRoZSByZWV4cG9ydGluZyBmaWxlIGFuZCBub3QgdG8gdGhlIG9yaWdpbmFsXG4gICAgICAvLyBvbmUuICBIb3dldmVyLCB0aGVyZSBpcyBhbm90aGVyIHRzIEFQSSB0aGF0IGNhbiBkbyBhIGRlZXBlciByZXNvbHV0aW9uLlxuICAgICAgY29uc3QgbG9jYWxTeW1ib2wgPSB0eXBlQ2hlY2tlci5nZXRFeHBvcnRTcGVjaWZpZXJMb2NhbFRhcmdldFN5bWJvbChkKTtcbiAgICAgIC8vIEkgZG9uJ3Qga25vdyBob3cgY2FuIHRoaXMgaGFwcGVuLCBidXQgZXJyIG9uIHRoZSBzaWRlIG9mIGxlc3MgZW1pdC5cbiAgICAgIGlmICghbG9jYWxTeW1ib2wpIHJldHVybiBmYWxzZTtcbiAgICAgIC8vIGBkZWNsYXJhdGlvbnNgIGlzIHVuZGVmaW5lZCBmb3IgYnVpbHRpbiBzeW1ib2xzLCBzdWNoIGFzIGB1bmtub3duYC5cbiAgICAgIGlmICghbG9jYWxTeW1ib2wuZGVjbGFyYXRpb25zKSByZXR1cm4gZmFsc2U7XG5cbiAgICAgIC8vIEluIGNhc2Ugb2Ygbm8gaW1wb3J0IHdlIGVuZGVkIHVwIGluIGEgZGVjbGFyYXRpb24gaW4gZm9vLnRzLCB3aGlsZSBpblxuICAgICAgLy8gY2FzZSBvZiBoYXZpbmcgYW4gaW1wb3J0IGxvY2FsRCBpcyBzdGlsbCBpbiB0aGUgcmVleHBvcmluZyBmaWxlLlxuICAgICAgZm9yIChjb25zdCBsb2NhbEQgb2YgbG9jYWxTeW1ib2wuZGVjbGFyYXRpb25zKSB7XG4gICAgICAgIGlmIChsb2NhbEQuZ2V0U291cmNlRmlsZSgpICE9PSBvcmlnU291cmNlRmlsZSkge1xuICAgICAgICAgIHJldHVybiBmYWxzZTtcbiAgICAgICAgfVxuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gdHJ1ZTtcbiAgfSk7XG4gIGlmICghbG9jYWxFeHBvcnRzLmxlbmd0aCkgcmV0dXJuIGR0c0ZpbGVDb250ZW50O1xuXG4gIC8vIFR5cGVTY3JpcHQgMi44IGFuZCBUeXBlU2NyaXB0IDIuOSBkaWZmZXIgb24gdGhlIG9yZGVyIGluIHdoaWNoIHRoZVxuICAvLyBtb2R1bGUgc3ltYm9scyBjb21lIG91dCwgc28gc29ydCBoZXJlIHRvIG1ha2UgdGhlIHRlc3RzIHN0YWJsZS5cbiAgbG9jYWxFeHBvcnRzLnNvcnQoKGEsIGIpID0+IHN0cmluZ0NvbXBhcmUoYS5uYW1lLCBiLm5hbWUpKTtcblxuICBjb25zdCBtb2R1bGVOYW1lID0gaG9zdC5wYXRoVG9Nb2R1bGVOYW1lKCcnLCBzb3VyY2VGaWxlLmZpbGVOYW1lKTtcbiAgY29uc3QgY2x1dHpNb2R1bGVOYW1lID0gbW9kdWxlTmFtZS5yZXBsYWNlKC9cXC4vZywgJyQnKTtcblxuICAvLyBDbHV0eiBtaWdodCByZWZlciB0byB0aGUgbmFtZSBpbiB0d28gZGlmZmVyZW50IGZvcm1zIChzdGVtbWluZyBmcm9tIGdvb2cucHJvdmlkZSBhbmRcbiAgLy8gZ29vZy5tb2R1bGUgcmVzcGVjdGl2ZWx5KS5cbiAgLy8gMSkgZ2xvYmFsIGluIGNsdXR6OiAgIOCyoF/gsqAuY2x1dHoubW9kdWxlJGNvbnRlbnRzJHBhdGgkdG8kbW9kdWxlX1N5bWJvbC4uLlxuICAvLyAyKSBsb2NhbCBpbiBhIG1vZHVsZTog4LKgX+CyoC5jbHV0ei5tb2R1bGUkZXhwb3J0cyRwYXRoJHRvJG1vZHVsZS5TeW1ib2wgLi5cbiAgLy8gU2VlIGV4YW1wbGVzIGF0OlxuICAvLyBodHRwczovL2dpdGh1Yi5jb20vYW5ndWxhci9jbHV0ei90cmVlL21hc3Rlci9zcmMvdGVzdC9qYXZhL2NvbS9nb29nbGUvamF2YXNjcmlwdC9jbHV0elxuXG4gIC8vIENhc2UgKDEpIGZyb20gYWJvdmUuXG4gIGxldCBnbG9iYWxTeW1ib2xzID0gJyc7XG4gIC8vIENhc2UgKDIpIGZyb20gYWJvdmUuXG4gIGxldCBuZXN0ZWRTeW1ib2xzID0gJyc7XG4gIGZvciAoY29uc3Qgc3ltYm9sIG9mIGxvY2FsRXhwb3J0cykge1xuICAgIGxldCBsb2NhbE5hbWUgPSBzeW1ib2wubmFtZTtcbiAgICBjb25zdCBkZWNsYXJhdGlvbiA9IHN5bWJvbC5kZWNsYXJhdGlvbnMuZmluZChkID0+IGQuZ2V0U291cmNlRmlsZSgpID09PSBvcmlnU291cmNlRmlsZSk7XG4gICAgaWYgKGRlY2xhcmF0aW9uICYmIHRzLmlzRXhwb3J0U3BlY2lmaWVyKGRlY2xhcmF0aW9uKSAmJiBkZWNsYXJhdGlvbi5wcm9wZXJ0eU5hbWUpIHtcbiAgICAgIC8vIElmIGRlY2xhcmVkIGluIGFuIFwiZXhwb3J0IHtYIGFzIFl9O1wiIGV4cG9ydCBzcGVjaWZpZXIsIHRoZW4gWCAoc3RvcmVkIGluIHByb3BlcnR5TmFtZSkgaXNcbiAgICAgIC8vIHRoZSBsb2NhbCBuYW1lIHRoYXQgcmVzb2x2ZXMgd2l0aGluIHRoZSBtb2R1bGUsIHdoZXJlYXMgWSBpcyBvbmx5IGF2YWlsYWJsZSBvbiB0aGUgZXhwb3J0cyxcbiAgICAgIC8vIGkuZS4gdGhlIG5hbWUgdXNlZCB0byBhZGRyZXNzIHRoZSBzeW1ib2wgZnJvbSBvdXRzaWRlIHRoZSBtb2R1bGUuXG4gICAgICAvLyBVc2UgdGhlIGxvY2FsTmFtZSBmb3IgdGhlIGV4cG9ydCB0aGVuLCBidXQgcHVibGlzaCB1bmRlciB0aGUgZXh0ZXJuYWwgbmFtZS5cbiAgICAgIGxvY2FsTmFtZSA9IGRlY2xhcmF0aW9uLnByb3BlcnR5TmFtZS50ZXh0O1xuICAgIH1cbiAgICBjb25zdCBtYW5nbGVkTmFtZSA9IGBtb2R1bGUkY29udGVudHMkJHtjbHV0ek1vZHVsZU5hbWV9XyR7c3ltYm9sLm5hbWV9YDtcbiAgICBnbG9iYWxTeW1ib2xzICs9IGBcXHRcXHRleHBvcnQgeyR7bG9jYWxOYW1lfSBhcyAke21hbmdsZWROYW1lfX1cXG5gO1xuICAgIC8vIFRPRE8obXByb2JzdCk6IE9uY2UgdHNpY2tsZSBpcyBvbiBUUzMuNywgdGhlIHR3byBsaW5lcyBiZWxvdyBjYW4gYmUgcmVwbGFjZWQgd2l0aFxuICAgIC8vIFwiZXhwb3J0IHtsb2NhbE5hbWV9O1wiLiBIb3dldmVyIGluIFRTMy41LCBsb2NhbE5hbWUgcmVzb2x2ZXMgd2l0aGluIHRoZSBtb2R1bGUsIHNvXG4gICAgLy8gZXhwb3J0aW5nIHtsb2NhbE5hbWV9IGNhdXNlcyBhIGNpcmN1bGFyIGRlZmluaXRpb24gZXJyb3IuIFRoZSB3b3JrYXJvdW5kIGlzIHRvIGltcG9ydCB0aGVcbiAgICAvLyBtYW5nbGVkIG5hbWUuXG4gICAgbmVzdGVkU3ltYm9scyArPSBgXFx0XFx0aW1wb3J0ICR7bG9jYWxOYW1lfSRjbHV0eiA9IOCyoF/gsqAuY2x1dHouJHttYW5nbGVkTmFtZX07XFxuYDtcbiAgICBuZXN0ZWRTeW1ib2xzICs9IGBcXHRcXHRleHBvcnQgeyR7bG9jYWxOYW1lfSRjbHV0eiBhcyAke3N5bWJvbC5uYW1lfX07XFxuYDtcbiAgfVxuXG4gIGR0c0ZpbGVDb250ZW50ICs9ICdkZWNsYXJlIGdsb2JhbCB7XFxuJztcbiAgZHRzRmlsZUNvbnRlbnQgKz0gYFxcdG5hbWVzcGFjZSDgsqBf4LKgLmNsdXR6IHtcXG5gO1xuICBkdHNGaWxlQ29udGVudCArPSBnbG9iYWxTeW1ib2xzO1xuICBkdHNGaWxlQ29udGVudCArPSBgXFx0fVxcbmA7XG4gIGR0c0ZpbGVDb250ZW50ICs9IGBcXHRuYW1lc3BhY2Ug4LKgX+CyoC5jbHV0ei5tb2R1bGUkZXhwb3J0cyQke2NsdXR6TW9kdWxlTmFtZX0ge1xcbmA7XG4gIC8vIFRPRE8obWFydGlucHJvYnN0KTogU2VlIGh0dHBzOi8vZ2l0aHViLmNvbS9NaWNyb3NvZnQvVHlwZVNjcmlwdC9pc3N1ZXMvMzUzODUsIHJlbW92ZSBvbmNlIGZpeGVkXG4gIGR0c0ZpbGVDb250ZW50ICs9IGBcXHRcXHRjb25zdCBjbHV0eiR3b3JrYXJvdW5kJHRpc3N1ZSQzNTM4NTogbnVtYmVyO1xcbmA7XG4gIGR0c0ZpbGVDb250ZW50ICs9IG5lc3RlZFN5bWJvbHM7XG4gIGR0c0ZpbGVDb250ZW50ICs9IGBcXHR9XFxuYDtcbiAgZHRzRmlsZUNvbnRlbnQgKz0gJ31cXG4nO1xuXG4gIHJldHVybiBkdHNGaWxlQ29udGVudDtcbn1cblxuZnVuY3Rpb24gc2tpcFRyYW5zZm9ybUZvclNvdXJjZUZpbGVJZk5lZWRlZChcbiAgICBob3N0OiBUc2lja2xlSG9zdCxcbiAgICBkZWxlZ2F0ZUZhY3Rvcnk6IHRzLlRyYW5zZm9ybWVyRmFjdG9yeTx0cy5Tb3VyY2VGaWxlPik6IHRzLlRyYW5zZm9ybWVyRmFjdG9yeTx0cy5Tb3VyY2VGaWxlPiB7XG4gIHJldHVybiAoY29udGV4dDogdHMuVHJhbnNmb3JtYXRpb25Db250ZXh0KSA9PiB7XG4gICAgY29uc3QgZGVsZWdhdGUgPSBkZWxlZ2F0ZUZhY3RvcnkoY29udGV4dCk7XG4gICAgcmV0dXJuIChzb3VyY2VGaWxlOiB0cy5Tb3VyY2VGaWxlKSA9PiB7XG4gICAgICBpZiAoaG9zdC5zaG91bGRTa2lwVHNpY2tsZVByb2Nlc3Npbmcoc291cmNlRmlsZS5maWxlTmFtZSkpIHtcbiAgICAgICAgcmV0dXJuIHNvdXJjZUZpbGU7XG4gICAgICB9XG4gICAgICByZXR1cm4gZGVsZWdhdGUoc291cmNlRmlsZSk7XG4gICAgfTtcbiAgfTtcbn1cbiJdfQ==