/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("tsickle/src/type_translator", ["require", "exports", "typescript", "tsickle/src/annotator_host", "tsickle/src/path", "tsickle/src/transformer_util"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    const ts = require("typescript");
    const annotator_host_1 = require("tsickle/src/annotator_host");
    const path = require("tsickle/src/path");
    const transformer_util_1 = require("tsickle/src/transformer_util");
    /**
     * TypeScript allows you to write identifiers quoted, like:
     *   interface Foo {
     *     'bar': string;
     *     'complex name': string;
     *   }
     *   Foo.bar;  // ok
     *   Foo['bar']  // ok
     *   Foo['complex name']  // ok
     *
     * In Closure-land, we want identify that the legal name 'bar' can become an
     * ordinary field, but we need to skip strings like 'complex name'.
     */
    function isValidClosurePropertyName(name) {
        // In local experimentation, it appears that reserved words like 'var' and
        // 'if' are legal JS and still accepted by Closure.
        return /^[a-zA-Z_][a-zA-Z0-9_]*$/.test(name);
    }
    exports.isValidClosurePropertyName = isValidClosurePropertyName;
    /**
     * Determines if fileName refers to a builtin lib.d.ts file.
     * This is a terrible hack but it mirrors a similar thing done in Clutz.
     */
    function isBuiltinLibDTS(fileName) {
        return fileName.match(/\blib\.(?:[^/]+\.)?d\.ts$/) != null;
    }
    exports.isBuiltinLibDTS = isBuiltinLibDTS;
    /**
     * Returns true if the source file is generated by Clutz, i.e. has the magic
     * Clutz header.
     */
    function isClutzDts(sourceFile) {
        return sourceFile.text.startsWith('//!! generated by clutz.');
    }
    /**
     * typeValueConflictHandled returns true for symbols whose type/value conflict is handled outside of
     * tsickle.
     *
     * This covers two cases:
     *
     * - symbols provided by Clutz. Given that Closure has a merged type/value namespace, apparent
     *   type/value conflicts on the TypeScript level are actually fine.
     * - builtin lib*.d.ts symbols, such as "Array", which are considered Closure-compatible. Note that
     *   we don't actually enforce that the types are actually compatible, but mostly just hope that
     *   they are due to being derived from the same HTML specs.
     */
    function typeValueConflictHandled(symbol) {
        // TODO(#1072): if the symbol comes from a tsickle-transpiled file, either .ts or .d.ts with
        // externs generation? then maybe we can emit it with name mangling.
        return symbol.declarations != null &&
            symbol.declarations.some(n => isBuiltinLibDTS(n.getSourceFile().fileName) || isClutzDts(n.getSourceFile()));
    }
    exports.typeValueConflictHandled = typeValueConflictHandled;
    function typeToDebugString(type) {
        let debugString = `flags:0x${type.flags.toString(16)}`;
        if (type.aliasSymbol) {
            debugString += ` alias:${symbolToDebugString(type.aliasSymbol)}`;
        }
        if (type.aliasTypeArguments) {
            debugString += ` aliasArgs:<${type.aliasTypeArguments.map(typeToDebugString).join(',')}>`;
        }
        // Just the unique flags (powers of two). Declared in src/compiler/types.ts.
        const basicTypes = [
            ts.TypeFlags.Any, ts.TypeFlags.String, ts.TypeFlags.Number,
            ts.TypeFlags.Boolean, ts.TypeFlags.Enum, ts.TypeFlags.StringLiteral,
            ts.TypeFlags.NumberLiteral, ts.TypeFlags.BooleanLiteral, ts.TypeFlags.EnumLiteral,
            ts.TypeFlags.BigIntLiteral, ts.TypeFlags.ESSymbol, ts.TypeFlags.UniqueESSymbol,
            ts.TypeFlags.Void, ts.TypeFlags.Undefined, ts.TypeFlags.Null,
            ts.TypeFlags.Never, ts.TypeFlags.TypeParameter, ts.TypeFlags.Object,
            ts.TypeFlags.Union, ts.TypeFlags.Intersection, ts.TypeFlags.Index,
            ts.TypeFlags.IndexedAccess, ts.TypeFlags.Conditional, ts.TypeFlags.Substitution,
        ];
        for (const flag of basicTypes) {
            if ((type.flags & flag) !== 0) {
                debugString += ` ${ts.TypeFlags[flag]}`;
            }
        }
        if (type.flags === ts.TypeFlags.Object) {
            const objType = type;
            debugString += ` objectFlags:0x${objType.objectFlags}`;
            // Just the unique flags (powers of two). Declared in src/compiler/types.ts.
            const objectFlags = [
                ts.ObjectFlags.Class,
                ts.ObjectFlags.Interface,
                ts.ObjectFlags.Reference,
                ts.ObjectFlags.Tuple,
                ts.ObjectFlags.Anonymous,
                ts.ObjectFlags.Mapped,
                ts.ObjectFlags.Instantiated,
                ts.ObjectFlags.ObjectLiteral,
                ts.ObjectFlags.EvolvingArray,
                ts.ObjectFlags.ObjectLiteralPatternWithComputedProperties,
            ];
            for (const flag of objectFlags) {
                if ((objType.objectFlags & flag) !== 0) {
                    debugString += ` object:${ts.ObjectFlags[flag]}`;
                }
            }
        }
        if (type.symbol && type.symbol.name !== '__type') {
            debugString += ` symbol.name:${JSON.stringify(type.symbol.name)}`;
        }
        if (type.pattern) {
            debugString += ` destructuring:true`;
        }
        return `{type ${debugString}}`;
    }
    exports.typeToDebugString = typeToDebugString;
    function symbolToDebugString(sym) {
        let debugString = `${JSON.stringify(sym.name)} flags:0x${sym.flags.toString(16)}`;
        // Just the unique flags (powers of two). Declared in src/compiler/types.ts.
        const symbolFlags = [
            ts.SymbolFlags.FunctionScopedVariable,
            ts.SymbolFlags.BlockScopedVariable,
            ts.SymbolFlags.Property,
            ts.SymbolFlags.EnumMember,
            ts.SymbolFlags.Function,
            ts.SymbolFlags.Class,
            ts.SymbolFlags.Interface,
            ts.SymbolFlags.ConstEnum,
            ts.SymbolFlags.RegularEnum,
            ts.SymbolFlags.ValueModule,
            ts.SymbolFlags.NamespaceModule,
            ts.SymbolFlags.TypeLiteral,
            ts.SymbolFlags.ObjectLiteral,
            ts.SymbolFlags.Method,
            ts.SymbolFlags.Constructor,
            ts.SymbolFlags.GetAccessor,
            ts.SymbolFlags.SetAccessor,
            ts.SymbolFlags.Signature,
            ts.SymbolFlags.TypeParameter,
            ts.SymbolFlags.TypeAlias,
            ts.SymbolFlags.ExportValue,
            ts.SymbolFlags.Alias,
            ts.SymbolFlags.Prototype,
            ts.SymbolFlags.ExportStar,
            ts.SymbolFlags.Optional,
            ts.SymbolFlags.Transient,
        ];
        for (const flag of symbolFlags) {
            if ((sym.flags & flag) !== 0) {
                debugString += ` ${ts.SymbolFlags[flag]}`;
            }
        }
        return debugString;
    }
    exports.symbolToDebugString = symbolToDebugString;
    /**
     * Searches for an ambient module declaration in the ancestors of declarations, depth first, and
     * returns the first or null if none found.
     */
    function getContainingAmbientModuleDeclaration(declarations) {
        for (const declaration of declarations) {
            let parent = declaration.parent;
            while (parent) {
                if (ts.isModuleDeclaration(parent) && ts.isStringLiteral(parent.name)) {
                    return parent;
                }
                parent = parent.parent;
            }
        }
        return null;
    }
    /** Returns true if any of declarations is a top level declaration in an external module. */
    function isTopLevelExternal(declarations) {
        for (const declaration of declarations) {
            if (declaration.parent === undefined)
                continue;
            if (ts.isSourceFile(declaration.parent) && ts.isExternalModule(declaration.parent))
                return true;
        }
        return false;
    }
    /**
     * Returns true if a and b are (or were originally before transformation) nodes of the same source
     * file.
     */
    function isDeclaredInSameFile(a, b) {
        return ts.getOriginalNode(a).getSourceFile() === ts.getOriginalNode(b).getSourceFile();
    }
    /**
     * TypeTranslator translates TypeScript types to Closure types. It keeps state per type, so each
     * translation operation has to create a new instance.
     */
    class TypeTranslator {
        /**
         * @param node is the source AST ts.Node the type comes from.  This is used
         *     in some cases (e.g. anonymous types) for looking up field names.
         * @param pathBlackList is a set of paths that should never get typed;
         *     any reference to symbols defined in these paths should by typed
         *     as {?}.
         * @param symbolsToAliasedNames a mapping from symbols (`Foo`) to a name in scope they should be
         *     emitted as (e.g. `tsickle_reqType_1.Foo`). Can be augmented during type translation, e.g.
         *     to blacklist a symbol.
         */
        constructor(host, typeChecker, node, pathBlackList, symbolsToAliasedNames, symbolToNameCache, ensureSymbolDeclared = () => { }) {
            this.host = host;
            this.typeChecker = typeChecker;
            this.node = node;
            this.pathBlackList = pathBlackList;
            this.symbolsToAliasedNames = symbolsToAliasedNames;
            this.symbolToNameCache = symbolToNameCache;
            this.ensureSymbolDeclared = ensureSymbolDeclared;
            /**
             * A list of type literals we've encountered while emitting; used to avoid getting stuck in
             * recursive types.
             */
            this.seenAnonymousTypes = new Set();
            /**
             * Whether to write types suitable for an #externs file. Externs types must not refer to
             * non-externs types (i.e. non ambient types) and need to use fully qualified names.
             */
            this.isForExterns = false;
            // Normalize paths to not break checks on Windows.
            this.pathBlackList =
                new Set(Array.from(this.pathBlackList.values()).map(p => path.normalize(p)));
        }
        /**
         * Converts a ts.Symbol to a string, applying aliases and ensuring symbols are imported.
         * @return a string representation of the symbol as a valid Closure type name, or `undefined` if
         *     the type cannot be expressed (e.g. for anonymous types).
         */
        symbolToString(sym) {
            // symbolToEntityName can be relatively expensive (40 ms calls with symbols in large namespaces
            // with many declarations, i.e. Clutz). symbolToString is idempotent per symbol and file, thus
            // we cache the entire operation to avoid the hit.
            const cachedName = this.symbolToNameCache.get(sym);
            if (cachedName)
                return cachedName;
            // TypeScript resolves e.g. union types to their members, which can include symbols not declared
            // in the current scope. Ensure that all symbols found this way are actually declared.
            // This must happen before the alias check below, it might introduce a new alias for the symbol.
            if (!this.isForExterns && (sym.flags & ts.SymbolFlags.TypeParameter) === 0) {
                this.ensureSymbolDeclared(sym);
            }
            const name = this.typeChecker.symbolToEntityName(sym, ts.SymbolFlags.Type, this.node, ts.NodeBuilderFlags.UseFullyQualifiedType);
            // name might be undefined, e.g. for anonymous classes.
            if (!name)
                return undefined;
            let str = '';
            /** Recursively visits components of entity name and writes them to `str` above. */
            const writeEntityWithSymbols = (name) => {
                let identifier;
                if (ts.isQualifiedName(name)) {
                    writeEntityWithSymbols(name.left);
                    str += '.';
                    identifier = name.right;
                }
                else {
                    identifier = name;
                }
                let symbol = identifier.symbol;
                // When writing a symbol, check if there is an alias for it in the current scope that should
                // take precedence, e.g. from a goog.requireType.
                if (symbol.flags & ts.SymbolFlags.Alias) {
                    symbol = this.typeChecker.getAliasedSymbol(symbol);
                }
                const alias = this.symbolsToAliasedNames.get(symbol);
                if (alias) {
                    // If so, discard the entire current text and only use the alias - otherwise if a symbol has
                    // a local alias but appears in a dotted type path (e.g. when it's imported using import *
                    // as foo), str would contain both the prefx *and* the full alias (foo.alias.name).
                    str = alias;
                    return;
                }
                let text = transformer_util_1.getIdentifierText(identifier);
                if (str.length === 0) {
                    const mangledPrefix = this.maybeGetMangledNamePrefix(symbol);
                    text = mangledPrefix + text;
                }
                str += text;
            };
            writeEntityWithSymbols(name);
            str = this.stripClutzNamespace(str);
            this.symbolToNameCache.set(sym, str);
            return str;
        }
        /**
         * Returns the mangled name prefix for symbol, or an empty string if not applicable.
         *
         * Type names are emitted with a mangled prefix if they are top level symbols declared in an
         * external module (.d.ts or .ts), and are ambient declarations ("declare ..."). This is because
         * their declarations get moved to externs files (to make external names visible to Closure and
         * prevent renaming), which only use global names. This means the names must be mangled to prevent
         * collisions and allow referencing them uniquely.
         *
         * This method also handles the special case of symbols declared in an ambient external module
         * context.
         *
         * Symbols declared in a global block, e.g. "declare global { type X; }", are handled implicitly:
         * when referenced, they are written as just "X", which is not a top level declaration, so the
         * code below ignores them.
         */
        maybeGetMangledNamePrefix(symbol) {
            if (!symbol.declarations)
                return '';
            const declarations = symbol.declarations;
            let ambientModuleDeclaration = null;
            // If the symbol is neither a top level declaration in an external module nor in an ambient
            // block, tsickle should not emit a prefix: it's either not an external symbol, or it's an
            // external symbol nested in a module, so it will need to be qualified, and the mangling prefix
            // goes on the qualifier.
            if (!isTopLevelExternal(declarations)) {
                ambientModuleDeclaration = getContainingAmbientModuleDeclaration(declarations);
                if (!ambientModuleDeclaration)
                    return '';
            }
            // At this point, the declaration is from an external module (possibly ambient).
            // These declarations must be prefixed if either:
            // (a) tsickle is emitting an externs file, so all symbols are qualified within it
            // (b) or the declaration must be an exported ambient declaration from the local file.
            // Ambient external declarations from other files are imported, so there's a local alias for the
            // module and no mangling is needed.
            if (!this.isForExterns &&
                !declarations.every(d => isDeclaredInSameFile(this.node, d) && transformer_util_1.isAmbient(d) &&
                    transformer_util_1.hasModifierFlag(d, ts.ModifierFlags.Export))) {
                return '';
            }
            // If from an ambient declaration, use and resolve the name from that. Otherwise, use the file
            // name from the (arbitrary) first declaration to mangle.
            const fileName = ambientModuleDeclaration ?
                ambientModuleDeclaration.name.text :
                ts.getOriginalNode(declarations[0]).getSourceFile().fileName;
            const mangled = annotator_host_1.moduleNameAsIdentifier(this.host, fileName);
            return mangled + '.';
        }
        // Clutz (https://github.com/angular/clutz) emits global type symbols hidden in a special
        // ಠ_ಠ.clutz namespace. While most code seen by Tsickle will only ever see local aliases, Clutz
        // symbols can be written by users directly in code, and they can appear by dereferencing
        // TypeAliases. The code below simply strips the prefix, the remaining type name then matches
        // Closure's type.
        stripClutzNamespace(name) {
            if (name.startsWith('ಠ_ಠ.clutz.'))
                return name.substring('ಠ_ಠ.clutz.'.length);
            return name;
        }
        translate(type) {
            // NOTE: Though type.flags has the name "flags", it usually can only be one
            // of the enum options at a time (except for unions of literal types, e.g. unions of boolean
            // values, string values, enum values). This switch handles all the cases in the ts.TypeFlags
            // enum in the order they occur.
            // NOTE: Some TypeFlags are marked "internal" in the d.ts but still show up in the value of
            // type.flags. This mask limits the flag checks to the ones in the public API. "lastFlag" here
            // is the last flag handled in this switch statement, and should be kept in sync with
            // typescript.d.ts.
            // NonPrimitive occurs on its own on the lower case "object" type. Special case to "!Object".
            if (type.flags === ts.TypeFlags.NonPrimitive)
                return '!Object';
            // Avoid infinite loops on recursive type literals.
            // It would be nice to just emit the name of the recursive type here (in type.aliasSymbol
            // below), but Closure Compiler does not allow recursive type definitions.
            if (this.seenAnonymousTypes.has(type))
                return '?';
            let isAmbient = false;
            let isInNamespace = false;
            let isModule = false;
            if (type.symbol) {
                for (const decl of type.symbol.declarations || []) {
                    if (ts.isExternalModule(decl.getSourceFile()))
                        isModule = true;
                    if (decl.getSourceFile().isDeclarationFile)
                        isAmbient = true;
                    let current = decl;
                    while (current) {
                        if (ts.getCombinedModifierFlags(current) & ts.ModifierFlags.Ambient)
                            isAmbient = true;
                        if (current.kind === ts.SyntaxKind.ModuleDeclaration)
                            isInNamespace = true;
                        current = current.parent;
                    }
                }
            }
            // tsickle cannot generate types for non-ambient namespaces nor any symbols contained in them.
            if (isInNamespace && !isAmbient)
                return '?';
            // Types in externs cannot reference types from external modules.
            // However ambient types in modules get moved to externs, too, so type references work and we
            // can emit a precise type.
            if (this.isForExterns && isModule && !isAmbient)
                return '?';
            const lastFlag = ts.TypeFlags.Substitution;
            const mask = (lastFlag << 1) - 1;
            switch (type.flags & mask) {
                case ts.TypeFlags.Any:
                    return '?';
                case ts.TypeFlags.Unknown:
                    return '*';
                case ts.TypeFlags.String:
                case ts.TypeFlags.StringLiteral:
                    return 'string';
                case ts.TypeFlags.Number:
                case ts.TypeFlags.NumberLiteral:
                    return 'number';
                case ts.TypeFlags.Boolean:
                case ts.TypeFlags.BooleanLiteral:
                    // See the note in translateUnion about booleans.
                    return 'boolean';
                case ts.TypeFlags.Enum:
                    if (!type.symbol) {
                        this.warn(`EnumType without a symbol`);
                        return '?';
                    }
                    return this.symbolToString(type.symbol) || '?';
                case ts.TypeFlags.ESSymbol:
                case ts.TypeFlags.UniqueESSymbol:
                    // ESSymbol indicates something typed symbol.
                    // UniqueESSymbol indicates a specific unique symbol, used e.g. to index into an object.
                    // Closure does not have this distinction, so tsickle emits both as 'symbol'.
                    return 'symbol';
                case ts.TypeFlags.Void:
                    return 'void';
                case ts.TypeFlags.Undefined:
                    return 'undefined';
                case ts.TypeFlags.BigInt:
                case ts.TypeFlags.BigIntLiteral:
                    return 'bigintPlaceholder';
                case ts.TypeFlags.Null:
                    return 'null';
                case ts.TypeFlags.Never:
                    this.warn(`should not emit a 'never' type`);
                    return '?';
                case ts.TypeFlags.TypeParameter:
                    // This is e.g. the T in a type like Foo<T>.
                    if (!type.symbol) {
                        this.warn(`TypeParameter without a symbol`); // should not happen (tm)
                        return '?';
                    }
                    // In Closure, type parameters ("<T>") are non-nullable by default, unlike references to
                    // classes or interfaces. However this code path can be reached by bound type parameters,
                    // where the type parameter's symbol references a plain class or interface. In this case,
                    // add `!` to avoid emitting a nullable type.
                    let prefix = '';
                    if ((type.symbol.flags & ts.SymbolFlags.TypeParameter) === 0) {
                        prefix = '!';
                    }
                    const name = this.symbolToString(type.symbol);
                    if (!name)
                        return '?';
                    return prefix + name;
                case ts.TypeFlags.Object:
                    return this.translateObject(type);
                case ts.TypeFlags.Union:
                    return this.translateUnion(type);
                case ts.TypeFlags.Conditional:
                case ts.TypeFlags.Substitution:
                    this.warn(`emitting ? for conditional/substitution type`);
                    return '?';
                case ts.TypeFlags.Intersection:
                case ts.TypeFlags.Index:
                case ts.TypeFlags.IndexedAccess:
                    // TODO(ts2.1): handle these special types.
                    this.warn(`unhandled type flags: ${ts.TypeFlags[type.flags]}`);
                    return '?';
                default:
                    // Handle cases where multiple flags are set.
                    // Types with literal members are represented as
                    //   ts.TypeFlags.Union | [literal member]
                    // E.g. an enum typed value is a union type with the enum's members as its members. A
                    // boolean type is a union type with 'true' and 'false' as its members.
                    // Note also that in a more complex union, e.g. boolean|number, then it's a union of three
                    // things (true|false|number) and ts.TypeFlags.Boolean doesn't show up at all.
                    if (type.flags & ts.TypeFlags.Union) {
                        return this.translateUnion(type);
                    }
                    if (type.flags & ts.TypeFlags.EnumLiteral) {
                        return this.translateEnumLiteral(type);
                    }
                    // The switch statement should have been exhaustive.
                    throw new Error(`unknown type flags ${type.flags} on ${typeToDebugString(type)}`);
            }
        }
        translateUnion(type) {
            // Union types that include literals (e.g. boolean, enum) can end up repeating the same Closure
            // type. For example: true | boolean will be translated to boolean | boolean.
            // Remove duplicates to produce types that read better.
            const parts = new Set(type.types.map(t => this.translate(t)));
            // If it's a single element set, return the single member.
            if (parts.size === 1)
                return parts.values().next().value;
            return `(${Array.from(parts.values()).join('|')})`;
        }
        translateEnumLiteral(type) {
            // Suppose you had:
            //   enum EnumType { MEMBER }
            // then the type of "EnumType.MEMBER" is an enum literal (the thing passed to this function)
            // and it has type flags that include
            //   ts.TypeFlags.NumberLiteral | ts.TypeFlags.EnumLiteral
            //
            // Closure Compiler doesn't support literals in types, so this code must not emit
            // "EnumType.MEMBER", but rather "EnumType".
            const enumLiteralBaseType = this.typeChecker.getBaseTypeOfLiteralType(type);
            if (!enumLiteralBaseType.symbol) {
                this.warn(`EnumLiteralType without a symbol`);
                return '?';
            }
            let symbol = enumLiteralBaseType.symbol;
            if (enumLiteralBaseType === type) {
                // TypeScript's API will return the same EnumLiteral type if the enum only has a single member
                // value. See https://github.com/Microsoft/TypeScript/issues/28869.
                // In that case, take the parent symbol of the enum member, which should be the enum
                // declaration.
                // tslint:disable-next-line:no-any working around a TS API deficiency.
                const parent = symbol['parent'];
                if (!parent)
                    return '?';
                symbol = parent;
            }
            const name = this.symbolToString(symbol);
            if (!name)
                return '?';
            // In Closure, enum types are non-null by default, so we wouldn't need to emit the `!` here.
            // However that's confusing to users, to the point that style guides and linters require to
            // *always* specify the nullability modifier. To be consistent with that style, include it here
            // as well.
            return '!' + name;
        }
        // translateObject translates a ts.ObjectType, which is the type of all
        // object-like things in TS, such as classes and interfaces.
        translateObject(type) {
            if (type.symbol && this.isBlackListed(type.symbol))
                return '?';
            // NOTE: objectFlags is an enum, but a given type can have multiple flags.
            // Array<string> is both ts.ObjectFlags.Reference and ts.ObjectFlags.Interface.
            if (type.objectFlags & ts.ObjectFlags.Class) {
                if (!type.symbol) {
                    this.warn('class has no symbol');
                    return '?';
                }
                const name = this.symbolToString(type.symbol);
                if (!name) {
                    // An anonymous type. Make sure not to emit '!?', as that is a syntax error in Closure
                    // Compiler.
                    return '?';
                }
                return '!' + name;
            }
            else if (type.objectFlags & ts.ObjectFlags.Interface) {
                // Note: ts.InterfaceType has a typeParameters field, but that
                // specifies the parameters that the interface type *expects*
                // when it's used, and should not be transformed to the output.
                // E.g. a type like Array<number> is a TypeReference to the
                // InterfaceType "Array", but the "number" type parameter is
                // part of the outer TypeReference, not a typeParameter on
                // the InterfaceType.
                if (!type.symbol) {
                    this.warn('interface has no symbol');
                    return '?';
                }
                if (type.symbol.flags & ts.SymbolFlags.Value) {
                    // The symbol is both a type and a value.
                    // For user-defined types in this state, we may not have a Closure name
                    // for the type.  See the type_and_value test.
                    if (!typeValueConflictHandled(type.symbol)) {
                        this.warn(`type/symbol conflict for ${type.symbol.name}, using {?} for now`);
                        return '?';
                    }
                }
                return '!' + this.symbolToString(type.symbol);
            }
            else if (type.objectFlags & ts.ObjectFlags.Reference) {
                // A reference to another type, e.g. Array<number> refers to Array.
                // Emit the referenced type and any type arguments.
                const referenceType = type;
                // A tuple is a ReferenceType where the target is flagged Tuple and the
                // typeArguments are the tuple arguments.  Just treat it as a mystery
                // array, because Closure doesn't understand tuples.
                if (referenceType.target.objectFlags & ts.ObjectFlags.Tuple) {
                    return '!Array<?>';
                }
                let typeStr = '';
                if (referenceType.target === referenceType) {
                    // We get into an infinite loop here if the inner reference is
                    // the same as the outer; this can occur when this function
                    // fails to translate a more specific type before getting to
                    // this point.
                    throw new Error(`reference loop in ${typeToDebugString(referenceType)} ${referenceType.flags}`);
                }
                typeStr += this.translate(referenceType.target);
                // Translate can return '?' for a number of situations, e.g. type/value conflicts.
                // `?<?>` is illegal syntax in Closure Compiler, so just return `?` here.
                if (typeStr === '?')
                    return '?';
                const typeArgs = this.typeChecker.getTypeArguments(referenceType);
                if (typeArgs) {
                    const params = typeArgs.map(t => this.translate(t));
                    typeStr += `<${params.join(', ')}>`;
                }
                return typeStr;
            }
            else if (type.objectFlags & ts.ObjectFlags.Anonymous) {
                return this.translateAnonymousType(type);
            }
            /*
            TODO(ts2.1): more unhandled object type flags:
              Tuple
              Mapped
              Instantiated
              ObjectLiteral
              EvolvingArray
              ObjectLiteralPatternWithComputedProperties
            */
            this.warn(`unhandled type ${typeToDebugString(type)}`);
            return '?';
        }
        /**
         * translateAnonymousType translates a ts.TypeFlags.ObjectType that is also
         * ts.ObjectFlags.Anonymous. That is, this type's symbol does not have a name. This is the
         * anonymous type encountered in e.g.
         *     let x: {a: number};
         * But also the inferred type in:
         *     let x = {a: 1};  // type of x is {a: number}, as above
         */
        translateAnonymousType(type) {
            this.seenAnonymousTypes.add(type);
            if (!type.symbol) {
                // This comes up when generating code for an arrow function as passed
                // to a generic function.  The passed-in type is tagged as anonymous
                // and has no properties so it's hard to figure out what to generate.
                // Just avoid it for now so we don't crash.
                this.warn('anonymous type has no symbol');
                return '?';
            }
            if (type.symbol.flags & ts.SymbolFlags.Function || type.symbol.flags & ts.SymbolFlags.Method) {
                const sigs = this.typeChecker.getSignaturesOfType(type, ts.SignatureKind.Call);
                if (sigs.length === 1) {
                    return this.signatureToClosure(sigs[0]);
                }
                this.warn('unhandled anonymous type with multiple call signatures');
                return '?';
            }
            // Gather up all the named fields and whether the object is also callable.
            let callable = false;
            let indexable = false;
            const fields = [];
            if (!type.symbol.members) {
                this.warn('anonymous type has no symbol');
                return '?';
            }
            // special-case construct signatures.
            const ctors = type.getConstructSignatures();
            if (ctors.length) {
                // TODO(martinprobst): this does not support additional properties defined on constructors
                // (not expressible in Closure), nor multiple constructors (same).
                const decl = ctors[0].declaration;
                if (!decl) {
                    this.warn('unhandled anonymous type with constructor signature but no declaration');
                    return '?';
                }
                if (decl.kind === ts.SyntaxKind.JSDocSignature) {
                    this.warn('unhandled JSDoc based constructor signature');
                    return '?';
                }
                // new <T>(tee: T) is not supported by Closure, blacklist as ?.
                this.blacklistTypeParameters(this.symbolsToAliasedNames, decl.typeParameters);
                const params = this.convertParams(ctors[0], decl.parameters);
                const paramsStr = params.length ? (', ' + params.join(', ')) : '';
                const constructedType = this.translate(ctors[0].getReturnType());
                const constructedTypeStr = constructedType[0] === '!' ? constructedType.substring(1) : constructedType;
                // In the specific case of the "new" in a function, the correct Closure
                // type is:
                //
                //   function(new:Bar, ...args)
                //
                // Including the nullability annotation can cause the Closure compiler to
                // no longer recognize the function as a constructor type in externs.
                return `function(new:${constructedTypeStr}${paramsStr})`;
            }
            // members is an ES6 map, but the .d.ts defining it defined their own map
            // type, so typescript doesn't believe that .keys() is iterable
            // tslint:disable-next-line:no-any
            for (const field of type.symbol.members.keys()) {
                switch (field) {
                    case '__call':
                        callable = true;
                        break;
                    case '__index':
                        indexable = true;
                        break;
                    default:
                        if (!isValidClosurePropertyName(field)) {
                            this.warn(`omitting inexpressible property name: ${field}`);
                            continue;
                        }
                        const member = type.symbol.members.get(field);
                        // optional members are handled by the type including |undefined in a union type.
                        const memberType = this.translate(this.typeChecker.getTypeOfSymbolAtLocation(member, this.node));
                        fields.push(`${field}: ${memberType}`);
                        break;
                }
            }
            // Try to special-case plain key-value objects and functions.
            if (fields.length === 0) {
                if (callable && !indexable) {
                    // A function type.
                    const sigs = this.typeChecker.getSignaturesOfType(type, ts.SignatureKind.Call);
                    if (sigs.length === 1) {
                        return this.signatureToClosure(sigs[0]);
                    }
                }
                else if (indexable && !callable) {
                    // A plain key-value map type.
                    let keyType = 'string';
                    let valType = this.typeChecker.getIndexTypeOfType(type, ts.IndexKind.String);
                    if (!valType) {
                        keyType = 'number';
                        valType = this.typeChecker.getIndexTypeOfType(type, ts.IndexKind.Number);
                    }
                    if (!valType) {
                        this.warn('unknown index key type');
                        return `!Object<?,?>`;
                    }
                    return `!Object<${keyType},${this.translate(valType)}>`;
                }
                else if (!callable && !indexable) {
                    // The object has no members.  This is the TS type '{}',
                    // which means "any value other than null or undefined".
                    // What is this in Closure's type system?
                    //
                    // First, {!Object} is wrong because it is not a supertype of
                    // {string} or {number}.  This would mean you cannot assign a
                    // number to a variable of TS type {}.
                    //
                    // We get closer with {*}, aka the ALL type.  This one better
                    // captures the typical use of the TS {}, which users use for
                    // "I don't care".
                    //
                    // {*} unfortunately does include null/undefined, so it's a closer
                    // match for TS 3.0's 'unknown'.
                    return '*';
                }
            }
            if (!callable && !indexable) {
                // Not callable, not indexable; implies a plain object with fields in it.
                return `{${fields.join(', ')}}`;
            }
            this.warn('unhandled anonymous type');
            return '?';
        }
        /** Converts a ts.Signature (function signature) to a Closure function type. */
        signatureToClosure(sig) {
            // TODO(martinprobst): Consider harmonizing some overlap with emitFunctionType in tsickle.ts.
            if (!sig.declaration) {
                this.warn('signature without declaration');
                return 'Function';
            }
            if (sig.declaration.kind === ts.SyntaxKind.JSDocSignature) {
                this.warn('signature with JSDoc declaration');
                return 'Function';
            }
            this.blacklistTypeParameters(this.symbolsToAliasedNames, sig.declaration.typeParameters);
            let typeStr = `function(`;
            let paramDecls = sig.declaration.parameters || [];
            const maybeThisParam = paramDecls[0];
            // Oddly, the this type shows up in paramDecls, but not in the type's parameters.
            // Handle it here and then pass paramDecls down without its first element.
            if (maybeThisParam && maybeThisParam.name.getText() === 'this') {
                if (maybeThisParam.type) {
                    const thisType = this.typeChecker.getTypeAtLocation(maybeThisParam.type);
                    typeStr += `this: (${this.translate(thisType)})`;
                    if (paramDecls.length > 1)
                        typeStr += ', ';
                }
                else {
                    this.warn('this type without type');
                }
                paramDecls = paramDecls.slice(1);
            }
            const params = this.convertParams(sig, paramDecls);
            typeStr += `${params.join(', ')})`;
            const retType = this.translate(this.typeChecker.getReturnTypeOfSignature(sig));
            if (retType) {
                typeStr += `: ${retType}`;
            }
            return typeStr;
        }
        /**
         * Converts parameters for the given signature. Takes parameter declarations as those might not
         * match the signature parameters (e.g. there might be an additional this parameter). This
         * difference is handled by the caller, as is converting the "this" parameter.
         */
        convertParams(sig, paramDecls) {
            const paramTypes = [];
            for (let i = 0; i < sig.parameters.length; i++) {
                const param = sig.parameters[i];
                const paramDecl = paramDecls[i];
                const optional = !!paramDecl.questionToken;
                const varArgs = !!paramDecl.dotDotDotToken;
                let paramType = this.typeChecker.getTypeOfSymbolAtLocation(param, this.node);
                if (varArgs) {
                    if ((paramType.flags & ts.TypeFlags.Object) === 0) {
                        this.warn('var args type is not an object type');
                        paramTypes.push('!Array<?>');
                        continue;
                    }
                    if ((paramType.objectFlags & ts.ObjectFlags.Reference) === 0) {
                        this.warn('unsupported var args type (not an array reference)');
                        paramTypes.push('!Array<?>');
                        continue;
                    }
                    const typeArgs = this.typeChecker.getTypeArguments(paramType);
                    if (typeArgs.length === 0) {
                        // When a rest argument resolves empty, i.e. the concrete instantiation does not take any
                        // arguments, the type arguments are empty. Emit a function type that takes no arg in this
                        // position then.
                        continue;
                    }
                    paramType = typeArgs[0];
                }
                let typeStr = this.translate(paramType);
                if (varArgs)
                    typeStr = '...' + typeStr;
                if (optional)
                    typeStr = typeStr + '=';
                paramTypes.push(typeStr);
            }
            return paramTypes;
        }
        warn(msg) {
            // By default, warn() does nothing.  The caller will overwrite this
            // if it wants different behavior.
        }
        /** @return true if sym should always have type {?}. */
        isBlackListed(symbol) {
            return isBlacklisted(this.pathBlackList, symbol);
        }
        /**
         * Closure doesn not support type parameters for function types, i.e. generic function types.
         * Blacklist the symbols declared by them and emit a ? for the types.
         *
         * This mutates the given blacklist map. The map's scope is one file, and symbols are
         * unique objects, so this should neither lead to excessive memory consumption nor introduce
         * errors.
         *
         * @param blacklist a map to store the blacklisted symbols in, with a value of '?'. In practice,
         *     this is always === this.symbolsToAliasedNames, but we're passing it explicitly to make it
         *    clear that the map is mutated (in particular when used from outside the class).
         * @param decls the declarations whose symbols should be blacklisted.
         */
        blacklistTypeParameters(blacklist, decls) {
            if (!decls || !decls.length)
                return;
            for (const tpd of decls) {
                const sym = this.typeChecker.getSymbolAtLocation(tpd.name);
                if (!sym) {
                    this.warn(`type parameter with no symbol`);
                    continue;
                }
                blacklist.set(sym, '?');
            }
        }
    }
    exports.TypeTranslator = TypeTranslator;
    /** @return true if sym should always have type {?}. */
    function isBlacklisted(pathBlackList, symbol) {
        if (pathBlackList === undefined)
            return false;
        // Some builtin types, such as {}, get represented by a symbol that has no declarations.
        if (symbol.declarations === undefined)
            return false;
        return symbol.declarations.every(n => {
            const fileName = path.normalize(n.getSourceFile().fileName);
            return pathBlackList.has(fileName);
        });
    }
    exports.isBlacklisted = isBlacklisted;
});
//# sourceMappingURL=data:application/json;base64,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