"use strict";
const fs = require('fs');
const path = require('path');
const rules_1 = require('./rules');
function formatUsage(usage) {
    return usage.replace(/~~~/g, '```').replace(/(^[ \t]*\n)/gm, '\n').replace(/^    /mg, '');
}
exports.formatUsage = formatUsage;
function createRuleTable() {
    const buffer = [];
    let category = null;
    rules_1.rules.forEach((rule) => {
        if (category !== rule.category) {
            category = rule.category;
            buffer.push(`\n### ${category}\n\n`);
            buffer.push(`${rules_1.categories[category]}\n\n`);
            buffer.push('| :grey_question: | ESLint | TSLint | Description |\n');
            buffer.push('| :---            | :---:  | :---:  | :---        |\n');
        }
        let available;
        if (rule.available) {
            available = rule.provider === 'native' ? ':ballot_box_with_check:' : ':white_check_mark:';
        }
        else {
            available = rule.tslintRule === 'Not applicable' ? ':no_entry_sign:' : ':x:';
        }
        const tsRuleName = rule.tslintUrl ? `[${rule.tslintRule}](${rule.tslintUrl})` : rule.tslintRule;
        const tsRule = rule.tslintRule === 'Not applicable' ? 'Not applicable' : tsRuleName;
        buffer.push('|');
        buffer.push(`${available}|`);
        buffer.push(`[${rule.eslintRule}](${rule.eslintUrl})|`);
        buffer.push(`${tsRule}|`);
        buffer.push(`${rule.description}|`);
        buffer.push('\n');
    });
    return buffer.join('');
}
function updateReadme(cb) {
    fs.readFile('README.md', 'utf8', (readErr, data) => {
        if (readErr) {
            return console.error(readErr);
        }
        let content = data.replace(/^<!-- Start:AutoTable((.*?(\n))+.*?)End:AutoTable -->$/gm, '<!-- Start:AutoTable:: Modify `src/readme/rules.ts` and run `gulp readme` to update block -->\n' +
            createRuleTable() +
            '<!-- End:AutoTable -->');
        fs.writeFile('README.md', content, 'utf8', (writeErr) => {
            if (writeErr) {
                return console.error(writeErr);
            }
            console.log('[DONE] updating README.md ...');
            cb();
        });
    });
}
exports.updateReadme = updateReadme;
function createRuleContent(rule) {
    const usage = rule.usage ? `\n\n### Usage\n\n${formatUsage(rule.usage)}` : '';
    const note = rule.note ? `\n\n### Note\n\n${rule.note}\n` : '';
    return `## ${rule.tslintRule} (ESLint: [${rule.eslintRule}](${rule.eslintUrl}))

${rule.description}${usage}${note}
`;
}
function updateRuleFile(name, rule) {
    const baseUrl = 'https://github.com/buzinas/tslint-eslint-rules/blob/master';
    const docFileName = `src/docs/rules/${name}Rule.md`;
    return new Promise((fulfill, reject) => {
        fs.readFile(docFileName, 'utf8', (readErr, data) => {
            rule.tslintUrl = rule.tslintUrl || `${baseUrl}/${docFileName}`;
            let content = readErr || !data ? '<!-- Start:AutoDoc\n End:AutoDoc -->' : data;
            content = content.replace(/^<!-- Start:AutoDoc((.*?(\n))+.*?)End:AutoDoc -->$/gm, [
                '<!-- Start:AutoDoc:: Modify `src/readme/rules.ts` and run `gulp readme` to update block -->\n',
                createRuleContent(rule),
                '\n<!-- End:AutoDoc -->' + (readErr ? '\n' : '')
            ].join(''));
            fs.writeFile(docFileName, content, 'utf8', (writeErr) => {
                if (writeErr) {
                    return reject(writeErr);
                }
                console.log(` - ${name}`);
                fulfill();
            });
        });
    });
}
function updateRuleFiles(cb) {
    const ruleDir = 'src/rules/';
    const allFiles = fs.readdirSync(ruleDir).filter(file => fs.lstatSync(path.join(ruleDir, file)).isFile());
    const ruleNames = allFiles
        .filter(name => name.endsWith('.ts'))
        .map(name => name.substr(0, name.length - 7));
    const allPromises = [];
    ruleNames.forEach((name) => {
        allPromises.push(updateRuleFile(name, rules_1.ruleTSMap[name]));
    });
    Promise.all(allPromises).then(() => {
        console.log('[DONE] processing rule files ...');
        cb();
    });
}
exports.updateRuleFiles = updateRuleFiles;

//# sourceMappingURL=data:application/json;base64,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
