"use strict";
const ts = require('typescript');
const Lint = require('tslint/lib/lint');
class Rule extends Lint.Rules.AbstractRule {
    apply(sourceFile) {
        const walker = new NoInnerDeclarationsWalker(sourceFile, this.getOptions());
        return this.applyWithWalker(walker);
    }
}
exports.Rule = Rule;
class NoInnerDeclarationsWalker extends Lint.RuleWalker {
    constructor() {
        super(...arguments);
        this.VALID_PARENT_TYPES = [
            ts.SyntaxKind.SourceFile,
            ts.SyntaxKind.FunctionDeclaration,
            ts.SyntaxKind.FunctionExpression,
            ts.SyntaxKind.ArrowFunction,
            ts.SyntaxKind.MethodDeclaration,
            ts.SyntaxKind.ModuleDeclaration,
            ts.SyntaxKind.Constructor
        ];
    }
    visitFunctionDeclaration(node) {
        this.validateInnerDeclaration(node);
        super.visitFunctionDeclaration(node);
    }
    visitVariableStatement(node) {
        if (this.hasOption('both') && node.declarationList.getFirstToken().kind === ts.SyntaxKind.VarKeyword) {
            this.validateInnerDeclaration(node);
        }
        super.visitVariableStatement(node);
    }
    validateInnerDeclaration(node) {
        const body = this.nearestBody(node);
        const isValid = (body.isSourceFile && body.distance === 1) || body.distance === 2;
        if (!isValid) {
            const decl = node.kind === ts.SyntaxKind.FunctionDeclaration ? 'function' : 'variable';
            const root = body.isSourceFile ? 'program' : 'function body';
            this.addFailure(this.createFailure(node.getStart(), node.getWidth(), `move ${decl} declaration to ${root} root`));
        }
    }
    nearestBody(node) {
        let ancestor = node.parent;
        let generation = 1;
        while (ancestor && this.VALID_PARENT_TYPES.indexOf(ancestor.kind) === -1) {
            generation++;
            ancestor = ancestor.parent;
        }
        return {
            isSourceFile: (ancestor && ancestor.kind === ts.SyntaxKind.SourceFile) || !ancestor,
            distance: generation
        };
    }
}

//# sourceMappingURL=data:application/json;base64,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
