"use strict";
var es6_promise_1 = require('es6-promise');
var fs = require('fs');
var path = require('path');
var rules_1 = require('./rules');
function formatUsage(usage) {
    return usage.replace(/~~~/g, '```').replace(/(^[ \t]*\n)/gm, '\n').replace(/^    /mg, '');
}
exports.formatUsage = formatUsage;
function createRuleTable() {
    var buffer = [];
    var category = null;
    rules_1.rules.forEach(function (rule) {
        if (category !== rule.category) {
            category = rule.category;
            buffer.push("\n### " + category + "\n\n");
            buffer.push(rules_1.categories[category] + "\n\n");
            buffer.push('| :grey_question: | ESLint | TSLint | Description |\n');
            buffer.push('| :---            | :---:  | :---:  | :---        |\n');
        }
        var available;
        if (rule.available) {
            available = rule.provider === 'native' ? ':ballot_box_with_check:' : ':white_check_mark:';
        }
        else {
            available = rule.tslintRule === 'Not applicable' ? ':no_entry_sign:' : ':x:';
        }
        var tsRuleName = rule.tslintUrl ? "[" + rule.tslintRule + "](" + rule.tslintUrl + ")" : rule.tslintRule;
        var tsRule = rule.tslintRule === 'Not applicable' ? 'Not applicable' : tsRuleName;
        buffer.push('|');
        buffer.push(available + "|");
        buffer.push("[" + rule.eslintRule + "](" + rule.eslintUrl + ")|");
        buffer.push(tsRule + "|");
        buffer.push(rule.description + "|");
        buffer.push('\n');
    });
    return buffer.join('');
}
function updateReadme(cb) {
    fs.readFile('README.md', 'utf8', function (readErr, data) {
        if (readErr) {
            return console.error(readErr);
        }
        var content = data.replace(/^<!-- Start:AutoTable((.*?(\n))+.*?)End:AutoTable -->$/gm, '<!-- Start:AutoTable:: Modify `src/readme/rules.ts` and run `gulp readme` to update block -->\n' +
            createRuleTable() +
            '<!-- End:AutoTable -->');
        fs.writeFile('README.md', content, 'utf8', function (writeErr) {
            if (writeErr) {
                return console.error(writeErr);
            }
            console.log('[DONE] updating README.md ...');
            cb();
        });
    });
}
exports.updateReadme = updateReadme;
function createRuleContent(rule) {
    var moduleName = "../rules/" + rules_1.toCamelCase(rule.tslintRule) + "Rule.js";
    var module = require(moduleName);
    var metaData = module.Rule.metadata;
    if (metaData) {
        if (metaData.ruleName !== rule.tslintRule) {
            console.warn('[WARNING]: metadata.ruleName !== rule.tslintRule');
            console.warn("           " + metaData.ruleName + " !== " + rule.tslintRule);
        }
        if (metaData.description !== rule.description) {
            console.warn('[WARNING]: metadata.description !== rule.description');
            console.warn("           " + metaData.description + " !== " + rule.description);
        }
        var examples = metaData.optionExamples.map(function (x) { return ['```json', x, '```'].join(''); }).join('\n\n');
        var schema = [
            '```json',
            JSON.stringify(metaData.options, null, 2),
            '```'
        ].join('\n');
        return [
            ("## " + rule.tslintRule + " (ESLint: [" + rule.eslintRule + "](" + rule.eslintUrl + "))\n"),
            (rule.description + "\n"),
            ("#### Rationale\n" + metaData.rationale),
            ("### Config\n" + metaData.optionsDescription),
            ("#### Examples\n\n" + examples),
            ("#### Schema\n\n" + schema)
        ].join('\n');
    }
    var usage = rule.usage ? "\n\n### Usage\n\n" + formatUsage(rule.usage) : '';
    var note = rule.note ? "\n\n### Note\n\n" + rule.note + "\n" : '';
    return "## " + rule.tslintRule + " (ESLint: [" + rule.eslintRule + "](" + rule.eslintUrl + "))\n\n" + rule.description + usage + note + "\n";
}
function updateRuleFile(name, rule) {
    var baseUrl = 'https://github.com/buzinas/tslint-eslint-rules/blob/master';
    var docFileName = "src/docs/rules/" + name + "Rule.md";
    return new es6_promise_1.Promise(function (fulfill, reject) {
        fs.readFile(docFileName, 'utf8', function (readErr, data) {
            rule.tslintUrl = rule.tslintUrl || baseUrl + "/" + docFileName;
            var content = readErr || !data ? '<!-- Start:AutoDoc\n End:AutoDoc -->' : data;
            content = content.replace(/^<!-- Start:AutoDoc((.*?(\n))+.*?)End:AutoDoc -->$/gm, [
                '<!-- Start:AutoDoc:: Modify `src/readme/rules.ts` and run `gulp readme` to update block -->\n',
                createRuleContent(rule),
                '\n<!-- End:AutoDoc -->' + (readErr ? '\n' : '')
            ].join(''));
            fs.writeFile(docFileName, content, 'utf8', function (writeErr) {
                if (writeErr) {
                    return reject(writeErr);
                }
                console.log(" - " + name);
                fulfill();
            });
        });
    });
}
function updateRuleFiles(cb) {
    var ruleDir = 'src/rules/';
    var allFiles = fs.readdirSync(ruleDir).filter(function (file) { return fs.lstatSync(path.join(ruleDir, file)).isFile(); });
    var ruleNames = allFiles
        .filter(function (name) { return /\.ts$/.test(name); })
        .map(function (name) { return name.substr(0, name.length - 7); });
    var allPromises = [];
    ruleNames.forEach(function (name) {
        allPromises.push(updateRuleFile(name, rules_1.ruleTSMap[name]));
    });
    es6_promise_1.Promise.all(allPromises).then(function () {
        console.log('[DONE] processing rule files ...');
        cb();
    });
}
exports.updateRuleFiles = updateRuleFiles;

//# sourceMappingURL=data:application/json;charset=utf8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInJlYWRtZS9pbmRleC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsNEJBQXdCLGFBQWEsQ0FBQyxDQUFBO0FBQ3RDLElBQVksRUFBRSxXQUFNLElBQUksQ0FBQyxDQUFBO0FBQ3pCLElBQVksSUFBSSxXQUFNLE1BQU0sQ0FBQyxDQUFBO0FBQzdCLHNCQUFpRSxTQUFTLENBQUMsQ0FBQTtBQUUzRSxxQkFBcUIsS0FBSztJQUN4QixNQUFNLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLENBQUMsT0FBTyxDQUFDLGVBQWUsRUFBRSxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0FBQzVGLENBQUM7QUE0SUMsbUJBQVcsZUE1SVo7QUFFRDtJQUNFLElBQU0sTUFBTSxHQUFHLEVBQUUsQ0FBQztJQUNsQixJQUFJLFFBQVEsR0FBRyxJQUFJLENBQUM7SUFDcEIsYUFBSyxDQUFDLE9BQU8sQ0FBQyxVQUFDLElBQUk7UUFDakIsRUFBRSxDQUFDLENBQUMsUUFBUSxLQUFLLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO1lBQy9CLFFBQVEsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDO1lBQ3pCLE1BQU0sQ0FBQyxJQUFJLENBQUMsV0FBUyxRQUFRLFNBQU0sQ0FBQyxDQUFDO1lBQ3JDLE1BQU0sQ0FBQyxJQUFJLENBQUksa0JBQVUsQ0FBQyxRQUFRLENBQUMsU0FBTSxDQUFDLENBQUM7WUFDM0MsTUFBTSxDQUFDLElBQUksQ0FBQyx1REFBdUQsQ0FBQyxDQUFDO1lBQ3JFLE1BQU0sQ0FBQyxJQUFJLENBQUMsdURBQXVELENBQUMsQ0FBQztRQUN2RSxDQUFDO1FBRUQsSUFBSSxTQUFTLENBQUM7UUFDZCxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQztZQUNuQixTQUFTLEdBQUcsSUFBSSxDQUFDLFFBQVEsS0FBSyxRQUFRLEdBQUcseUJBQXlCLEdBQUcsb0JBQW9CLENBQUM7UUFDNUYsQ0FBQztRQUFDLElBQUksQ0FBQyxDQUFDO1lBQ04sU0FBUyxHQUFHLElBQUksQ0FBQyxVQUFVLEtBQUssZ0JBQWdCLEdBQUcsaUJBQWlCLEdBQUcsS0FBSyxDQUFDO1FBQy9FLENBQUM7UUFDRCxJQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsU0FBUyxHQUFHLE1BQUksSUFBSSxDQUFDLFVBQVUsVUFBSyxJQUFJLENBQUMsU0FBUyxNQUFHLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQztRQUNoRyxJQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsVUFBVSxLQUFLLGdCQUFnQixHQUFHLGdCQUFnQixHQUFHLFVBQVUsQ0FBQztRQUNwRixNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2pCLE1BQU0sQ0FBQyxJQUFJLENBQUksU0FBUyxNQUFHLENBQUMsQ0FBQztRQUM3QixNQUFNLENBQUMsSUFBSSxDQUFDLE1BQUksSUFBSSxDQUFDLFVBQVUsVUFBSyxJQUFJLENBQUMsU0FBUyxPQUFJLENBQUMsQ0FBQztRQUN4RCxNQUFNLENBQUMsSUFBSSxDQUFJLE1BQU0sTUFBRyxDQUFDLENBQUM7UUFDMUIsTUFBTSxDQUFDLElBQUksQ0FBSSxJQUFJLENBQUMsV0FBVyxNQUFHLENBQUMsQ0FBQztRQUNwQyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3BCLENBQUMsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7QUFDekIsQ0FBQztBQUVELHNCQUFzQixFQUFZO0lBQ2hDLEVBQUUsQ0FBQyxRQUFRLENBQUMsV0FBVyxFQUFFLE1BQU0sRUFBRSxVQUFDLE9BQU8sRUFBRSxJQUFJO1FBQzdDLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDWixNQUFNLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNoQyxDQUFDO1FBQ0QsSUFBSSxPQUFPLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FDeEIsMERBQTBELEVBQzFELGlHQUFpRztZQUNqRyxlQUFlLEVBQUU7WUFDakIsd0JBQXdCLENBQ3pCLENBQUM7UUFDRixFQUFFLENBQUMsU0FBUyxDQUFDLFdBQVcsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFLFVBQUMsUUFBUTtZQUNsRCxFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDO2dCQUNiLE1BQU0sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1lBQ2pDLENBQUM7WUFDRCxPQUFPLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7WUFDN0MsRUFBRSxFQUFFLENBQUM7UUFDUCxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQztBQTBGQyxvQkFBWSxnQkExRmI7QUFFRCwyQkFBMkIsSUFBVztJQUNwQyxJQUFNLFVBQVUsR0FBRyxjQUFZLG1CQUFXLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxZQUFTLENBQUM7SUFDckUsSUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQ25DLElBQU0sUUFBUSxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDO0lBQ3RDLEVBQUUsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7UUFFYixFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsUUFBUSxLQUFLLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1lBQzFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsa0RBQWtELENBQUMsQ0FBQztZQUNqRSxPQUFPLENBQUMsSUFBSSxDQUFDLGdCQUFjLFFBQVEsQ0FBQyxRQUFRLGFBQVEsSUFBSSxDQUFDLFVBQVksQ0FBQyxDQUFDO1FBQ3pFLENBQUM7UUFDRCxFQUFFLENBQUMsQ0FBQyxRQUFRLENBQUMsV0FBVyxLQUFLLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDO1lBQzlDLE9BQU8sQ0FBQyxJQUFJLENBQUMsc0RBQXNELENBQUMsQ0FBQztZQUNyRSxPQUFPLENBQUMsSUFBSSxDQUFDLGdCQUFjLFFBQVEsQ0FBQyxXQUFXLGFBQVEsSUFBSSxDQUFDLFdBQWEsQ0FBQyxDQUFDO1FBQzdFLENBQUM7UUFDRCxJQUFNLFFBQVEsR0FBRyxRQUFRLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FDMUMsVUFBQSxDQUFDLElBQUksT0FBQSxDQUFDLFNBQVMsRUFBRSxDQUFDLEVBQUUsS0FBSyxDQUFDLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxFQUE5QixDQUE4QixDQUNwQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNmLElBQU0sTUFBTSxHQUFHO1lBQ2IsU0FBUztZQUNULElBQUksQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLE9BQU8sRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1lBQ3pDLEtBQUs7U0FDTixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNiLE1BQU0sQ0FBQztZQUNMLFNBQU0sSUFBSSxDQUFDLFVBQVUsbUJBQWMsSUFBSSxDQUFDLFVBQVUsVUFBSyxJQUFJLENBQUMsU0FBUyxVQUFNO1lBQzNFLENBQUcsSUFBSSxDQUFDLFdBQVcsUUFBSTtZQUN2QixzQkFBbUIsUUFBUSxDQUFDLFNBQVMsQ0FBRTtZQUN2QyxrQkFBZSxRQUFRLENBQUMsa0JBQWtCLENBQUU7WUFDNUMsdUJBQW9CLFFBQVEsQ0FBRTtZQUM5QixxQkFBa0IsTUFBTSxDQUFFO1NBQzNCLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ2YsQ0FBQztJQUdELElBQU0sS0FBSyxHQUFHLElBQUksQ0FBQyxLQUFLLEdBQUcsc0JBQW9CLFdBQVcsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFHLEdBQUcsRUFBRSxDQUFDO0lBQzlFLElBQU0sSUFBSSxHQUFHLElBQUksQ0FBQyxJQUFJLEdBQUcscUJBQW1CLElBQUksQ0FBQyxJQUFJLE9BQUksR0FBRyxFQUFFLENBQUM7SUFDL0QsTUFBTSxDQUFDLFFBQU0sSUFBSSxDQUFDLFVBQVUsbUJBQWMsSUFBSSxDQUFDLFVBQVUsVUFBSyxJQUFJLENBQUMsU0FBUyxjQUU1RSxJQUFJLENBQUMsV0FBVyxHQUFHLEtBQUssR0FBRyxJQUFJLE9BQ2hDLENBQUM7QUFDRixDQUFDO0FBRUQsd0JBQXdCLElBQVksRUFBRSxJQUFXO0lBQy9DLElBQU0sT0FBTyxHQUFHLDREQUE0RCxDQUFDO0lBQzdFLElBQU0sV0FBVyxHQUFHLG9CQUFrQixJQUFJLFlBQVMsQ0FBQztJQUNwRCxNQUFNLENBQUMsSUFBSSxxQkFBTyxDQUFDLFVBQUMsT0FBTyxFQUFFLE1BQU07UUFDakMsRUFBRSxDQUFDLFFBQVEsQ0FBQyxXQUFXLEVBQUUsTUFBTSxFQUFFLFVBQUMsT0FBTyxFQUFFLElBQUk7WUFDN0MsSUFBSSxDQUFDLFNBQVMsR0FBRyxJQUFJLENBQUMsU0FBUyxJQUFPLE9BQU8sU0FBSSxXQUFhLENBQUM7WUFDL0QsSUFBSSxPQUFPLEdBQUcsT0FBTyxJQUFJLENBQUMsSUFBSSxHQUFHLHNDQUFzQyxHQUFHLElBQUksQ0FBQztZQUMvRSxPQUFPLEdBQUcsT0FBTyxDQUFDLE9BQU8sQ0FDdkIsc0RBQXNELEVBQ3REO2dCQUNFLCtGQUErRjtnQkFDL0YsaUJBQWlCLENBQUMsSUFBSSxDQUFDO2dCQUN2Qix3QkFBd0IsR0FBRyxDQUFDLE9BQU8sR0FBRyxJQUFJLEdBQUcsRUFBRSxDQUFDO2FBQ2pELENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUNYLENBQUM7WUFDRixFQUFFLENBQUMsU0FBUyxDQUFDLFdBQVcsRUFBRSxPQUFPLEVBQUUsTUFBTSxFQUFFLFVBQUMsUUFBUTtnQkFDbEQsRUFBRSxDQUFDLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQztvQkFDYixNQUFNLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDO2dCQUMxQixDQUFDO2dCQUNELE9BQU8sQ0FBQyxHQUFHLENBQUMsUUFBTSxJQUFNLENBQUMsQ0FBQztnQkFDMUIsT0FBTyxFQUFFLENBQUM7WUFDWixDQUFDLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDO0FBRUQseUJBQXlCLEVBQVk7SUFDbkMsSUFBTSxPQUFPLEdBQUcsWUFBWSxDQUFDO0lBQzdCLElBQU0sUUFBUSxHQUFHLEVBQUUsQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDLENBQUMsTUFBTSxDQUM3QyxVQUFBLElBQUksSUFBSSxPQUFBLEVBQUUsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBL0MsQ0FBK0MsQ0FDeEQsQ0FBQztJQUNGLElBQU0sU0FBUyxHQUFHLFFBQVE7U0FDdkIsTUFBTSxDQUFDLFVBQUEsSUFBSSxJQUFJLE9BQUEsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBbEIsQ0FBa0IsQ0FBQztTQUNsQyxHQUFHLENBQUMsVUFBQSxJQUFJLElBQUksT0FBQSxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsRUFBRSxJQUFJLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxFQUEvQixDQUErQixDQUFDLENBQUM7SUFDaEQsSUFBTSxXQUFXLEdBQUcsRUFBRSxDQUFDO0lBQ3ZCLFNBQVMsQ0FBQyxPQUFPLENBQUMsVUFBQyxJQUFJO1FBQ3JCLFdBQVcsQ0FBQyxJQUFJLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxpQkFBUyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUMxRCxDQUFDLENBQUMsQ0FBQztJQUVILHFCQUFPLENBQUMsR0FBRyxDQUFDLFdBQVcsQ0FBQyxDQUFDLElBQUksQ0FBQztRQUM1QixPQUFPLENBQUMsR0FBRyxDQUFDLGtDQUFrQyxDQUFDLENBQUM7UUFDaEQsRUFBRSxFQUFFLENBQUM7SUFDUCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUM7QUFLQyx1QkFBZSxtQkFMaEI7QUFNQyIsImZpbGUiOiJyZWFkbWUvaW5kZXguanMiLCJzb3VyY2VSb290IjoiL1VzZXJzL2ptbG9wZXovV29ya3NwYWNlL3RzbGludC1lc2xpbnQtcnVsZXMvc3JjIn0=
