"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var ts = require('typescript');
var Lint = require('tslint');
var RULE_NAME = 'ter-max-len';
var CODE = 'code';
var COMMENTS = 'comments';
var TAB_WIDTH = 'tabWidth';
var IGNORE_PATTERN = 'ignorePattern';
var IGNORE_COMMENTS = 'ignoreComments';
var IGNORE_STRINGS = 'ignoreStrings';
var IGNORE_URLS = 'ignoreUrls';
var IGNORE_TEMPLATE_LITERALS = 'ignoreTemplateLiterals';
var IGNORE_REG_EXP_LITERALS = 'ignoreRegExpLiterals';
var IGNORE_TRAILING_COMMENTS = 'ignoreTrailingComments';
var IGNORE_IMPORTS = 'ignoreImports';
function computeLineLength(line, tabWidth) {
    var extraCharacterCount = 0;
    line.replace(/\t/g, function (_, offset) {
        var totalOffset = offset + extraCharacterCount;
        var previousTabStopOffset = tabWidth ? totalOffset % tabWidth : 0;
        var spaceCount = tabWidth - previousTabStopOffset;
        extraCharacterCount += spaceCount - 1;
        return '\t';
    });
    return line.length + extraCharacterCount;
}
function isFullLineComment(line, lineNumber, comment) {
    var start = comment.start;
    var end = comment.end;
    var isFirstTokenOnLine = !line.slice(0, start[1]).trim();
    return comment &&
        (start[0] < lineNumber || (start[0] === lineNumber && isFirstTokenOnLine)) &&
        (end[0] > lineNumber || (end[0] === lineNumber && end[1] === line.length));
}
function isTrailingComment(line, lineNumber, comment) {
    return comment &&
        (comment.start[0] === lineNumber && lineNumber <= comment.end[0]) &&
        (comment.end[0] > lineNumber || comment.end[1] === line.length);
}
function stripTrailingComment(line, comment) {
    return line.slice(0, comment.start[1]).replace(/\s+$/, '');
}
function groupByLineNumber(acc, node) {
    var startLoc = node.start;
    var endLoc = node.end;
    for (var i = startLoc[0]; i <= endLoc[0]; ++i) {
        if (!Array.isArray(acc[i])) {
            acc[i] = [];
        }
        acc[i].push(node);
    }
    return acc;
}
var Rule = (function (_super) {
    __extends(Rule, _super);
    function Rule() {
        _super.apply(this, arguments);
    }
    Rule.mergeOptions = function (options) {
        var optionsObj = {};
        var obj = options[0];
        if (typeof obj === 'number') {
            optionsObj[CODE] = obj || 80;
            obj = options[1];
        }
        if (typeof obj === 'number') {
            optionsObj[TAB_WIDTH] = obj || 4;
            obj = options[2];
        }
        if (typeof obj === 'object' && !Array.isArray(obj)) {
            Object.keys(obj).forEach(function (key) {
                optionsObj[key] = obj[key];
            });
        }
        optionsObj[CODE] = optionsObj[CODE] || 80;
        optionsObj[TAB_WIDTH] = optionsObj[TAB_WIDTH] || 4;
        return optionsObj;
    };
    Rule.prototype.isEnabled = function () {
        if (_super.prototype.isEnabled.call(this)) {
            var options = this.getOptions().ruleArguments;
            var option = options[0];
            if (typeof option === 'number' && option > 0) {
                return true;
            }
            var optionsObj = Rule.mergeOptions(options);
            if (optionsObj[CODE]) {
                return true;
            }
        }
        return false;
    };
    Rule.prototype.apply = function (sourceFile) {
        return this.applyWithWalker(new MaxLenWalker(sourceFile, this.getOptions()));
    };
    Rule.metadata = {
        ruleName: RULE_NAME,
        description: 'enforce a maximum line length',
        rationale: (_a = ["\n      Limiting the length of a line of code improves code readability.\n      It also makes comparing code side-by-side easier and improves compatibility with\n      various editors, IDEs, and diff viewers.\n      "], _a.raw = ["\n      Limiting the length of a line of code improves code readability.\n      It also makes comparing code side-by-side easier and improves compatibility with\n      various editors, IDEs, and diff viewers.\n      "], Lint.Utils.dedent(_a)),
        optionsDescription: (_b = ["\n      An integer indicating the maximum length of lines followed by an optional integer specifying\n      the character width for tab characters.\n\n      An optional object may be provided to fine tune the rule:\n\n      * `\"", "\"`: (default 80) enforces a maximum line length\n      * `\"", "\"`: (default 4) specifies the character width for tab characters\n      * `\"", "\"`: enforces a maximum line length for comments; defaults to value of code\n      * `\"", "\"`: ignores lines matching a regular expression; can only match a single\n                                 line and need to be double escaped when written in JSON\n      * `\"", "\"`: true ignores all trailing comments and comments on their own line\n      * `\"", "\"`: true ignores only trailing comments\n      * `\"", "\"`: true ignores lines that contain a URL\n      * `\"", "\"`: true ignores lines that contain a double-quoted or single-quoted string\n      * `\"", "\"`: true ignores lines that contain a template literal\n      * `\"", "\"`: true ignores lines that contain a RegExp literal\n      * `\"", "\"`: true ignores lines that contain an import module specifier\n      "], _b.raw = ["\n      An integer indicating the maximum length of lines followed by an optional integer specifying\n      the character width for tab characters.\n\n      An optional object may be provided to fine tune the rule:\n\n      * \\`\"", "\"\\`: (default 80) enforces a maximum line length\n      * \\`\"", "\"\\`: (default 4) specifies the character width for tab characters\n      * \\`\"", "\"\\`: enforces a maximum line length for comments; defaults to value of code\n      * \\`\"", "\"\\`: ignores lines matching a regular expression; can only match a single\n                                 line and need to be double escaped when written in JSON\n      * \\`\"", "\"\\`: true ignores all trailing comments and comments on their own line\n      * \\`\"", "\"\\`: true ignores only trailing comments\n      * \\`\"", "\"\\`: true ignores lines that contain a URL\n      * \\`\"", "\"\\`: true ignores lines that contain a double-quoted or single-quoted string\n      * \\`\"", "\"\\`: true ignores lines that contain a template literal\n      * \\`\"", "\"\\`: true ignores lines that contain a RegExp literal\n      * \\`\"", "\"\\`: true ignores lines that contain an import module specifier\n      "], Lint.Utils.dedent(_b, CODE, TAB_WIDTH, COMMENTS, IGNORE_PATTERN, IGNORE_COMMENTS, IGNORE_TRAILING_COMMENTS, IGNORE_URLS, IGNORE_STRINGS, IGNORE_TEMPLATE_LITERALS, IGNORE_REG_EXP_LITERALS, IGNORE_IMPORTS)),
        options: {
            type: 'array',
            items: [{
                    type: 'number',
                    minimum: '0'
                }, {
                    type: 'object',
                    properties: (_c = {},
                        _c[CODE] = {
                            type: 'number',
                            minumum: '1'
                        },
                        _c[COMMENTS] = {
                            type: 'number',
                            minumum: '1'
                        },
                        _c[TAB_WIDTH] = {
                            type: 'number',
                            minumum: '1'
                        },
                        _c[IGNORE_PATTERN] = {
                            type: 'string'
                        },
                        _c[IGNORE_COMMENTS] = {
                            type: 'boolean'
                        },
                        _c[IGNORE_STRINGS] = {
                            type: 'boolean'
                        },
                        _c[IGNORE_URLS] = {
                            type: 'boolean'
                        },
                        _c[IGNORE_TEMPLATE_LITERALS] = {
                            type: 'boolean'
                        },
                        _c[IGNORE_REG_EXP_LITERALS] = {
                            type: 'boolean'
                        },
                        _c[IGNORE_TRAILING_COMMENTS] = {
                            type: 'boolean'
                        },
                        _c[IGNORE_IMPORTS] = {
                            type: 'boolean'
                        },
                        _c
                    ),
                    additionalProperties: false
                }],
            minLength: 1,
            maxLength: 3
        },
        optionExamples: [
            (_d = ["\n        \"", "\": [true, 100]\n        "], _d.raw = ["\n        \"", "\": [true, 100]\n        "], Lint.Utils.dedent(_d, RULE_NAME)),
            (_e = ["\n        \"", "\": [\n          true,\n          100,\n          2,\n          {\n            \"", "\": true,\n            \"", "\": \"^\\\\s*(let|const)\\\\s.+=\\\\s*require\\\\s*\\\\(\"\n          }\n        ]\n        "], _e.raw = ["\n        \"", "\": [\n          true,\n          100,\n          2,\n          {\n            \"", "\": true,\n            \"", "\": \"^\\\\\\\\s*(let|const)\\\\\\\\s.+=\\\\\\\\s*require\\\\\\\\s*\\\\\\\\(\"\n          }\n        ]\n        "], Lint.Utils.dedent(_e, RULE_NAME, IGNORE_URLS, IGNORE_PATTERN)),
            (_f = ["\n        \"", "\": [\n          true,\n          {\n            \"", "\": 100,\n            \"", "\": 2,\n            \"", "\": true,\n            \"", "\": true,\n            \"", "\": \"^\\\\s*(let|const)\\\\s.+=\\\\s*require\\\\s*\\\\(\"\n          }\n        ]\n        "], _f.raw = ["\n        \"", "\": [\n          true,\n          {\n            \"", "\": 100,\n            \"", "\": 2,\n            \"", "\": true,\n            \"", "\": true,\n            \"", "\": \"^\\\\\\\\s*(let|const)\\\\\\\\s.+=\\\\\\\\s*require\\\\\\\\s*\\\\\\\\(\"\n          }\n        ]\n        "], Lint.Utils.dedent(_f, RULE_NAME, CODE, TAB_WIDTH, IGNORE_IMPORTS, IGNORE_URLS, IGNORE_PATTERN))
        ],
        typescriptOnly: false,
        type: 'style'
    };
    Rule.URL_REGEXP = /[^:/?#]:\/\/[^?#]/;
    return Rule;
    var _a, _b, _c, _d, _e, _f;
}(Lint.Rules.AbstractRule));
exports.Rule = Rule;
var MaxLenWalker = (function (_super) {
    __extends(MaxLenWalker, _super);
    function MaxLenWalker(sourceFile, options) {
        _super.call(this, sourceFile, options);
        this.ignoredIntervals = [];
        this.optionsObj = {};
        this.comments = [];
        this.strings = [];
        this.templates = [];
        this.regExp = [];
        this.optionsObj = Rule.mergeOptions(this.getOptions());
    }
    MaxLenWalker.prototype.hasOption = function (option) {
        if (this.optionsObj[option] && this.optionsObj[option]) {
            return true;
        }
        return false;
    };
    MaxLenWalker.prototype.getOption = function (option) {
        return this.optionsObj[option];
    };
    MaxLenWalker.prototype.visitStringLiteral = function (node) {
        this.strings.push(this.getINode(node.kind, node.getText(), node.getStart()));
        _super.prototype.visitStringLiteral.call(this, node);
    };
    MaxLenWalker.prototype.visitRegularExpressionLiteral = function (node) {
        this.regExp.push(this.getINode(node.kind, node.getText(), node.getStart()));
        _super.prototype.visitRegularExpressionLiteral.call(this, node);
    };
    MaxLenWalker.prototype.getINode = function (kind, text, startPos) {
        var width = text.length;
        var src = this.getSourceFile();
        var startLoc = src.getLineAndCharacterOfPosition(startPos);
        var endLoc = src.getLineAndCharacterOfPosition(startPos + width);
        return {
            kind: kind,
            text: text,
            startPosition: startPos,
            endPosition: startPos + width,
            start: [startLoc.line, startLoc.character],
            end: [endLoc.line, endLoc.character]
        };
    };
    MaxLenWalker.prototype.visitSourceFile = function (node) {
        var _this = this;
        _super.prototype.visitSourceFile.call(this, node);
        Lint.scanAllTokens(ts.createScanner(ts.ScriptTarget.ES5, false, ts.LanguageVariant.Standard, node.text), function (scanner) {
            var token = scanner.getToken();
            var startPos = scanner.getStartPos();
            if (_this.tokensToSkipStartEndMap[startPos]) {
                scanner.setTextPos(_this.tokensToSkipStartEndMap[startPos]);
                return;
            }
            if (token === ts.SyntaxKind.SingleLineCommentTrivia ||
                token === ts.SyntaxKind.MultiLineCommentTrivia) {
                _this.comments.push(_this.getINode(token, scanner.getTokenText(), startPos));
            }
            else if (token === ts.SyntaxKind.FirstTemplateToken) {
                _this.templates.push(_this.getINode(token, scanner.getTokenText(), startPos));
            }
        });
        this.findFailures(node);
    };
    MaxLenWalker.prototype.visitImportDeclaration = function (node) {
        _super.prototype.visitImportDeclaration.call(this, node);
        var startPos = node.moduleSpecifier.getStart();
        var text = node.moduleSpecifier.getText();
        var width = text.length;
        if (this.hasOption(IGNORE_IMPORTS)) {
            this.ignoredIntervals.push({
                endPosition: startPos + width,
                startPosition: startPos
            });
        }
    };
    MaxLenWalker.prototype.findFailures = function (sourceFile) {
        var lineStarts = sourceFile.getLineStarts();
        var source = sourceFile.getFullText();
        var lineLimit = this.getOption(CODE) || 80;
        var ignoreTrailingComments = this.getOption(IGNORE_TRAILING_COMMENTS) ||
            this.getOption(IGNORE_COMMENTS) ||
            false;
        var ignoreComments = this.getOption(IGNORE_COMMENTS) || false;
        var ignoreStrings = this.getOption(IGNORE_STRINGS) || false;
        var ignoreTemplateLiterals = this.getOption(IGNORE_TEMPLATE_LITERALS) || false;
        var ignoreUrls = this.getOption(IGNORE_URLS) || false;
        var ignoreRexExpLiterals = this.getOption(IGNORE_REG_EXP_LITERALS) || false;
        var pattern = this.getOption(IGNORE_PATTERN) || null;
        var tabWidth = this.getOption(TAB_WIDTH) || 4;
        var maxCommentLength = this.getOption(COMMENTS);
        var comments = ignoreComments || maxCommentLength || ignoreTrailingComments ? this.comments : [];
        var commentsIndex = 0;
        var stringsByLine = this.strings.reduce(groupByLineNumber, {});
        var templatesByLine = this.templates.reduce(groupByLineNumber, {});
        var regExpByLine = this.regExp.reduce(groupByLineNumber, {});
        var totalLines = lineStarts.length;
        for (var i = 0; i < totalLines; ++i) {
            var from = lineStarts[i];
            var to = lineStarts[i + 1];
            var line = source.substring(from, i === totalLines - 1 ? to : to - 1);
            var lineIsComment = false;
            if (commentsIndex < comments.length) {
                var comment = null;
                do {
                    comment = comments[++commentsIndex];
                } while (comment && comment.start[0] <= i);
                comment = comments[--commentsIndex];
                if (isFullLineComment(line, i, comment)) {
                    lineIsComment = true;
                }
                else if (ignoreTrailingComments && isTrailingComment(line, i, comment)) {
                    line = stripTrailingComment(line, comment);
                }
            }
            if (ignoreUrls && Rule.URL_REGEXP.test(line) ||
                pattern && new RegExp(pattern).test(line) ||
                ignoreStrings && stringsByLine[i] ||
                ignoreTemplateLiterals && templatesByLine[i] ||
                ignoreRexExpLiterals && regExpByLine[i]) {
                continue;
            }
            var lineLength = computeLineLength(line, tabWidth);
            if (lineIsComment && ignoreComments) {
                continue;
            }
            var ruleFailure = void 0;
            if (lineIsComment && exceedLineLimit(lineLength, maxCommentLength, source[to - 2])) {
                ruleFailure = new Lint.RuleFailure(sourceFile, from, to - 1, "Line " + (i + 1) + " exceeds the maximum comment line length of " + maxCommentLength + ".", RULE_NAME);
            }
            else if (exceedLineLimit(lineLength, lineLimit, source[to - 2])) {
                ruleFailure = new Lint.RuleFailure(sourceFile, from, to - 1, "Line " + (i + 1) + " exceeds the maximum line length of " + lineLimit + ".", RULE_NAME);
            }
            if (ruleFailure && !Lint.doesIntersect(ruleFailure, this.ignoredIntervals)) {
                this.addFailure(ruleFailure);
            }
        }
    };
    return MaxLenWalker;
}(Lint.SkippableTokenAwareRuleWalker));
function exceedLineLimit(lineLength, lineLimit, secondToLast) {
    return lineLength > lineLimit && !((lineLength - 1) === lineLimit && secondToLast === '\r');
}

//# sourceMappingURL=data:application/json;charset=utf8;base64,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
