"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var ts = require('typescript');
var Lint = require('tslint');
var RULE_NAME = 'ter-prefer-arrow-callback';
var OPTIONS;
var Rule = (function (_super) {
    __extends(Rule, _super);
    function Rule() {
        _super.apply(this, arguments);
    }
    Rule.prototype.apply = function (sourceFile) {
        var walker = new RuleWalker(sourceFile, this.getOptions());
        return this.applyWithWalker(walker);
    };
    Rule.metadata = {
        ruleName: RULE_NAME,
        description: 'require arrow functions as callbacks',
        rationale: (_a = ["\n      Arrow functions are suited to callbacks, because:\n      \n      * `this` keywords in arrow functions bind to the upper scope\u2019s.\n      * The notation of the arrow function is shorter than function expression\u2019s.\n      "], _a.raw = ["\n      Arrow functions are suited to callbacks, because:\n      \n      * \\`this\\` keywords in arrow functions bind to the upper scope’s.\n      * The notation of the arrow function is shorter than function expression’s.\n      "], Lint.Utils.dedent(_a)),
        optionsDescription: (_b = ["\n      This rule takes one optional argument, an object which is an options object. This object\n      may specify the following properties:\n      \n      * `\"allowNamedFunctions\"` (default false) When set to `true`, the rule doesn't warn on\n                                  named functions used as callback.\n      * `\"allowUnboundThis\"` (default true) When set to `false`, this option allows the use of\n                               `this` without restriction and checks for dynamically assigned\n                               `this` values such as when using `Array.prototype.map` with a\n                               `context` argument. Normally, the rule will flag the use of this\n                               whenever a function does not use `bind()` to specify the value of\n                               `this` constantly.\n      "], _b.raw = ["\n      This rule takes one optional argument, an object which is an options object. This object\n      may specify the following properties:\n      \n      * \\`\"allowNamedFunctions\"\\` (default false) When set to \\`true\\`, the rule doesn't warn on\n                                  named functions used as callback.\n      * \\`\"allowUnboundThis\"\\` (default true) When set to \\`false\\`, this option allows the use of\n                               \\`this\\` without restriction and checks for dynamically assigned\n                               \\`this\\` values such as when using \\`Array.prototype.map\\` with a\n                               \\`context\\` argument. Normally, the rule will flag the use of this\n                               whenever a function does not use \\`bind()\\` to specify the value of\n                               \\`this\\` constantly.\n      "], Lint.Utils.dedent(_b)),
        options: {
            type: 'array',
            items: [{
                    type: 'object',
                    properties: {
                        allowNamedFunctions: {
                            type: 'boolean'
                        },
                        allowUnboundThis: {
                            type: 'boolean'
                        }
                    },
                    additionalProperties: false
                }],
            maxLength: 1
        },
        optionExamples: [
            (_c = ["\n        \"", "\": [true]\n        "], _c.raw = ["\n        \"", "\": [true]\n        "], Lint.Utils.dedent(_c, RULE_NAME)),
            (_d = ["\n        \"", "\": [true, {\n          \"allowNamedFunctions\": true\n        }]\n        "], _d.raw = ["\n        \"", "\": [true, {\n          \"allowNamedFunctions\": true\n        }]\n        "], Lint.Utils.dedent(_d, RULE_NAME)),
            (_e = ["\n        \"", "\": [true, {\n          \"allowUnboundThis\": false\n        }]\n        "], _e.raw = ["\n        \"", "\": [true, {\n          \"allowUnboundThis\": false\n        }]\n        "], Lint.Utils.dedent(_e, RULE_NAME)),
            (_f = ["\n        \"", "\": [true, {\n          \"allowNamedFunctions\": true,\n          \"allowUnboundThis\": false\n        }]\n        "], _f.raw = ["\n        \"", "\": [true, {\n          \"allowNamedFunctions\": true,\n          \"allowUnboundThis\": false\n        }]\n        "], Lint.Utils.dedent(_f, RULE_NAME))
        ],
        typescriptOnly: false,
        type: 'typescript'
    };
    return Rule;
    var _a, _b, _c, _d, _e, _f;
}(Lint.Rules.AbstractRule));
exports.Rule = Rule;
function checkMetaProperty(node, name, prop) {
    return node.parent.getFirstToken().getText() === name && node.name.text === prop;
}
function getCallbackInfo(func) {
    var retv = { isCallback: false, isLexicalThis: false };
    var node = func;
    var parent = node.parent;
    while (node) {
        switch (parent.kind) {
            case ts.SyntaxKind.BinaryExpression:
            case ts.SyntaxKind.ConditionalExpression:
                break;
            case ts.SyntaxKind.PropertyAccessExpression:
                if (parent.name.kind === ts.SyntaxKind.Identifier &&
                    parent.name.text === 'bind' &&
                    parent.parent.kind === ts.SyntaxKind.CallExpression &&
                    parent.parent.expression === parent) {
                    retv.isLexicalThis = (parent.parent.arguments.length === 1 &&
                        parent.parent.arguments[0].kind === ts.SyntaxKind.ThisKeyword);
                    node = parent;
                    parent = parent.parent;
                }
                else {
                    return retv;
                }
                break;
            case ts.SyntaxKind.CallExpression:
            case ts.SyntaxKind.NewExpression:
                if (parent.expression !== node) {
                    retv.isCallback = true;
                }
                return retv;
            default:
                return retv;
        }
        node = parent;
        parent = parent.parent;
    }
    throw new Error('unreachable');
}
var RuleWalker = (function (_super) {
    __extends(RuleWalker, _super);
    function RuleWalker(sourceFile, options) {
        _super.call(this, sourceFile, options);
        this.stack = [];
        OPTIONS = {
            allowUnboundThis: true,
            allowNamedFunctions: null
        };
        var userOptions = this.getOptions()[0];
        if (userOptions) {
            OPTIONS.allowUnboundThis = userOptions.allowUnboundThis !== false;
            OPTIONS.allowNamedFunctions = userOptions.allowNamedFunctions;
        }
        this.srcFile = sourceFile;
        this.srcText = sourceFile.getFullText();
    }
    RuleWalker.prototype.enterScope = function (functionName) {
        this.stack.push({
            functionName: functionName,
            isRecursive: false,
            hasThis: false,
            hasSuper: false,
            hasMeta: false,
            hasArguments: false
        });
    };
    RuleWalker.prototype.exitScope = function () {
        return this.stack.pop();
    };
    RuleWalker.prototype.exitFunctionExpression = function (node) {
        var scopeInfo = this.exitScope();
        if (node.asteriskToken) {
            return;
        }
        if (node.name && node.name.text) {
            if (OPTIONS.allowNamedFunctions || scopeInfo.isRecursive) {
                return;
            }
        }
        var params = node.parameters.map(function (x) { return x.name.getText(); });
        var argumentsIsParam = params.indexOf('arguments') !== -1;
        if (!argumentsIsParam && scopeInfo.hasArguments) {
            return;
        }
        var callbackInfo = getCallbackInfo(node);
        if (callbackInfo.isCallback &&
            (!OPTIONS.allowUnboundThis || !scopeInfo.hasThis || callbackInfo.isLexicalThis) &&
            !scopeInfo.hasSuper &&
            !scopeInfo.hasMeta) {
            var failure = this.createFailure(node.getStart(), node.getWidth(), 'Unexpected function expression.');
            this.addFailure(failure);
        }
    };
    RuleWalker.prototype.visitSourceFile = function (node) {
        this.stack = [];
        _super.prototype.visitSourceFile.call(this, node);
    };
    RuleWalker.prototype.visitFunctionDeclaration = function (node) {
        this.enterScope();
        _super.prototype.visitFunctionDeclaration.call(this, node);
        this.exitScope();
    };
    RuleWalker.prototype.visitFunctionExpression = function (node) {
        this.enterScope(node.name ? node.name.text : undefined);
        _super.prototype.visitFunctionExpression.call(this, node);
        this.exitFunctionExpression(node);
    };
    RuleWalker.prototype.visitNode = function (node) {
        var info = this.stack[this.stack.length - 1];
        if (info && node.parent.kind !== ts.SyntaxKind.FunctionExpression) {
            if (node.kind === ts.SyntaxKind.ThisKeyword) {
                info.hasThis = true;
            }
            else if (node.kind === ts.SyntaxKind.SuperKeyword) {
                info.hasSuper = true;
            }
            else if (node.kind === ts.SyntaxKind.Identifier) {
                var text = node.text;
                if (text === 'arguments') {
                    info.hasArguments = true;
                }
                else if (text === info.functionName) {
                    info.isRecursive = true;
                }
            }
            else if (node.kind === ts.SyntaxKind.PropertyAccessExpression &&
                checkMetaProperty(node, 'new', 'target')) {
                info.hasMeta = true;
            }
        }
        _super.prototype.visitNode.call(this, node);
    };
    return RuleWalker;
}(Lint.RuleWalker));

//# sourceMappingURL=data:application/json;charset=utf8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInJ1bGVzL3RlclByZWZlckFycm93Q2FsbGJhY2tSdWxlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7OztBQU1BLElBQVksRUFBRSxXQUFNLFlBQVksQ0FBQyxDQUFBO0FBQ2pDLElBQVksSUFBSSxXQUFNLFFBQVEsQ0FBQyxDQUFBO0FBRS9CLElBQU0sU0FBUyxHQUFHLDJCQUEyQixDQUFDO0FBQzlDLElBQUksT0FBWSxDQUFDO0FBRWpCO0lBQTBCLHdCQUF1QjtJQUFqRDtRQUEwQiw4QkFBdUI7SUFvRWpELENBQUM7SUFKUSxvQkFBSyxHQUFaLFVBQWEsVUFBeUI7UUFDcEMsSUFBTSxNQUFNLEdBQUcsSUFBSSxVQUFVLENBQUMsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDO1FBQzdELE1BQU0sQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFsRWEsYUFBUSxHQUF1QjtRQUMzQyxRQUFRLEVBQUUsU0FBUztRQUNuQixXQUFXLEVBQUUsc0NBQXNDO1FBQ25ELFNBQVMsRUFBRSxPQUFpQiwrT0FLekIseVBBTFEsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLEtBS3pCO1FBQ0gsa0JBQWtCLEVBQUUsT0FBaUIseTFCQVlsQyxpNUJBWmlCLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxLQVlsQztRQUNILE9BQU8sRUFBRTtZQUNQLElBQUksRUFBRSxPQUFPO1lBQ2IsS0FBSyxFQUFFLENBQUM7b0JBQ04sSUFBSSxFQUFFLFFBQVE7b0JBQ2QsVUFBVSxFQUFFO3dCQUNWLG1CQUFtQixFQUFFOzRCQUNuQixJQUFJLEVBQUUsU0FBUzt5QkFDaEI7d0JBQ0QsZ0JBQWdCLEVBQUU7NEJBQ2hCLElBQUksRUFBRSxTQUFTO3lCQUNoQjtxQkFDRjtvQkFDRCxvQkFBb0IsRUFBRSxLQUFLO2lCQUM1QixDQUFDO1lBQ0YsU0FBUyxFQUFFLENBQUM7U0FDYjtRQUNELGNBQWMsRUFBRTtZQUNkLE9BQWlCLGNBQ1osRUFBUyxzQkFDWCxzREFGSCxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sS0FDWixTQUFTLEVBQ1g7WUFDSCxPQUFpQixjQUNaLEVBQVMsNkVBR1gsNkdBSkgsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLEtBQ1osU0FBUyxFQUdYO1lBQ0gsT0FBaUIsY0FDWixFQUFTLDJFQUdYLDJHQUpILElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxLQUNaLFNBQVMsRUFHWDtZQUNILE9BQWlCLGNBQ1osRUFBUyxxSEFJWCxxSkFMSCxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sS0FDWixTQUFTLEVBSVg7U0FDSjtRQUNELGNBQWMsRUFBRSxLQUFLO1FBQ3JCLElBQUksRUFBRSxZQUFZO0tBQ25CLENBQUM7SUFNSixXQUFDOztBQUFELENBcEVBLEFBb0VDLENBcEV5QixJQUFJLENBQUMsS0FBSyxDQUFDLFlBQVksR0FvRWhEO0FBcEVZLFlBQUksT0FvRWhCLENBQUE7QUFFRCwyQkFBMkIsSUFBaUMsRUFBRSxJQUFZLEVBQUUsSUFBWTtJQUN0RixNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxhQUFhLEVBQUUsQ0FBQyxPQUFPLEVBQUUsS0FBSyxJQUFJLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEtBQUssSUFBSSxDQUFDO0FBQ25GLENBQUM7QUFPRCx5QkFBeUIsSUFBMkI7SUFDbEQsSUFBTSxJQUFJLEdBQUcsRUFBRSxVQUFVLEVBQUUsS0FBSyxFQUFFLGFBQWEsRUFBRSxLQUFLLEVBQUUsQ0FBQztJQUN6RCxJQUFJLElBQUksR0FBRyxJQUFlLENBQUM7SUFDM0IsSUFBSSxNQUFNLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQztJQUV6QixPQUFPLElBQUksRUFBRSxDQUFDO1FBQ1osTUFBTSxDQUFDLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7WUFDcEIsS0FBSyxFQUFFLENBQUMsVUFBVSxDQUFDLGdCQUFnQixDQUFDO1lBQ3BDLEtBQUssRUFBRSxDQUFDLFVBQVUsQ0FBQyxxQkFBcUI7Z0JBQ3RDLEtBQUssQ0FBQztZQUNSLEtBQUssRUFBRSxDQUFDLFVBQVUsQ0FBQyx3QkFBd0I7Z0JBQ3pDLEVBQUUsQ0FBQyxDQUNBLE1BQXNDLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsVUFBVSxDQUFDLFVBQVU7b0JBQzdFLE1BQXNDLENBQUMsSUFBSSxDQUFDLElBQUksS0FBSyxNQUFNO29CQUM1RCxNQUFNLENBQUMsTUFBTSxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsVUFBVSxDQUFDLGNBQWM7b0JBQ2xELE1BQU0sQ0FBQyxNQUE0QixDQUFDLFVBQVUsS0FBSyxNQUN0RCxDQUFDLENBQUMsQ0FBQztvQkFDRCxJQUFJLENBQUMsYUFBYSxHQUFHLENBQ2xCLE1BQU0sQ0FBQyxNQUE0QixDQUFDLFNBQVMsQ0FBQyxNQUFNLEtBQUssQ0FBQzt3QkFDMUQsTUFBTSxDQUFDLE1BQTRCLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FDckYsQ0FBQztvQkFDRixJQUFJLEdBQUcsTUFBTSxDQUFDO29CQUNkLE1BQU0sR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDO2dCQUN6QixDQUFDO2dCQUFDLElBQUksQ0FBQyxDQUFDO29CQUNOLE1BQU0sQ0FBQyxJQUFJLENBQUM7Z0JBQ2QsQ0FBQztnQkFDRCxLQUFLLENBQUM7WUFDUixLQUFLLEVBQUUsQ0FBQyxVQUFVLENBQUMsY0FBYyxDQUFDO1lBQ2xDLEtBQUssRUFBRSxDQUFDLFVBQVUsQ0FBQyxhQUFhO2dCQUM5QixFQUFFLENBQUMsQ0FBRSxNQUE0QixDQUFDLFVBQVUsS0FBSyxJQUFJLENBQUMsQ0FBQyxDQUFDO29CQUN0RCxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksQ0FBQztnQkFDekIsQ0FBQztnQkFDRCxNQUFNLENBQUMsSUFBSSxDQUFDO1lBQ2Q7Z0JBQ0UsTUFBTSxDQUFDLElBQUksQ0FBQztRQUNoQixDQUFDO1FBRUQsSUFBSSxHQUFHLE1BQU0sQ0FBQztRQUNkLE1BQU0sR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDO0lBQ3pCLENBQUM7SUFFRCxNQUFNLElBQUksS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDO0FBQ2pDLENBQUM7QUFXRDtJQUF5Qiw4QkFBZTtJQUt0QyxvQkFBWSxVQUF5QixFQUFFLE9BQXNCO1FBQzNELGtCQUFNLFVBQVUsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUhyQixVQUFLLEdBQXFCLEVBQUUsQ0FBQztRQUluQyxPQUFPLEdBQUc7WUFDUixnQkFBZ0IsRUFBRSxJQUFJO1lBQ3RCLG1CQUFtQixFQUFFLElBQUk7U0FDMUIsQ0FBQztRQUNGLElBQU0sV0FBVyxHQUFHLElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN6QyxFQUFFLENBQUMsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDO1lBQ2hCLE9BQU8sQ0FBQyxnQkFBZ0IsR0FBRyxXQUFXLENBQUMsZ0JBQWdCLEtBQUssS0FBSyxDQUFDO1lBQ2xFLE9BQU8sQ0FBQyxtQkFBbUIsR0FBRyxXQUFXLENBQUMsbUJBQW1CLENBQUM7UUFDaEUsQ0FBQztRQUNELElBQUksQ0FBQyxPQUFPLEdBQUcsVUFBVSxDQUFDO1FBQzFCLElBQUksQ0FBQyxPQUFPLEdBQUcsVUFBVSxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQzFDLENBQUM7SUFLTywrQkFBVSxHQUFsQixVQUFtQixZQUFxQjtRQUN0QyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQztZQUNkLDBCQUFZO1lBQ1osV0FBVyxFQUFFLEtBQUs7WUFDbEIsT0FBTyxFQUFFLEtBQUs7WUFDZCxRQUFRLEVBQUUsS0FBSztZQUNmLE9BQU8sRUFBRSxLQUFLO1lBQ2QsWUFBWSxFQUFFLEtBQUs7U0FDcEIsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUtPLDhCQUFTLEdBQWpCO1FBQ0UsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxFQUFFLENBQUM7SUFDMUIsQ0FBQztJQUVPLDJDQUFzQixHQUE5QixVQUErQixJQUEyQjtRQUN4RCxJQUFNLFNBQVMsR0FBRyxJQUFJLENBQUMsU0FBUyxFQUFFLENBQUM7UUFHbkMsRUFBRSxDQUFDLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUM7WUFDdkIsTUFBTSxDQUFDO1FBQ1QsQ0FBQztRQUdELEVBQUUsQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1lBQ2hDLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxtQkFBbUIsSUFBSSxTQUFTLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQztnQkFDekQsTUFBTSxDQUFDO1lBQ1QsQ0FBQztRQUNILENBQUM7UUFHRCxJQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxVQUFBLENBQUMsSUFBSSxPQUFBLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxFQUFFLEVBQWhCLENBQWdCLENBQUMsQ0FBQztRQUMxRCxJQUFNLGdCQUFnQixHQUFHLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFDNUQsRUFBRSxDQUFDLENBQUMsQ0FBQyxnQkFBZ0IsSUFBSSxTQUFTLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQztZQUNoRCxNQUFNLENBQUM7UUFDVCxDQUFDO1FBRUQsSUFBTSxZQUFZLEdBQUcsZUFBZSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzNDLEVBQUUsQ0FBQyxDQUNELFlBQVksQ0FBQyxVQUFVO1lBQ3ZCLENBQUMsQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLElBQUksQ0FBQyxTQUFTLENBQUMsT0FBTyxJQUFJLFlBQVksQ0FBQyxhQUFhLENBQUM7WUFDL0UsQ0FBQyxTQUFTLENBQUMsUUFBUTtZQUNuQixDQUFDLFNBQVMsQ0FBQyxPQUNiLENBQUMsQ0FBQyxDQUFDO1lBQ0QsSUFBTSxPQUFPLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FDaEMsSUFBSSxDQUFDLFFBQVEsRUFBRSxFQUNmLElBQUksQ0FBQyxRQUFRLEVBQUUsRUFDZixpQ0FBaUMsQ0FDbEMsQ0FBQztZQUNGLElBQUksQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDM0IsQ0FBQztJQUNILENBQUM7SUFFUyxvQ0FBZSxHQUF6QixVQUEwQixJQUFtQjtRQUUzQyxJQUFJLENBQUMsS0FBSyxHQUFHLEVBQUUsQ0FBQztRQUNoQixnQkFBSyxDQUFDLGVBQWUsWUFBQyxJQUFJLENBQUMsQ0FBQztJQUM5QixDQUFDO0lBRVMsNkNBQXdCLEdBQWxDLFVBQW1DLElBQTRCO1FBQzdELElBQUksQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUNsQixnQkFBSyxDQUFDLHdCQUF3QixZQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3JDLElBQUksQ0FBQyxTQUFTLEVBQUUsQ0FBQztJQUNuQixDQUFDO0lBRVMsNENBQXVCLEdBQWpDLFVBQWtDLElBQTJCO1FBQzNELElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxTQUFTLENBQUMsQ0FBQztRQUN4RCxnQkFBSyxDQUFDLHVCQUF1QixZQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3BDLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUNwQyxDQUFDO0lBRVMsOEJBQVMsR0FBbkIsVUFBb0IsSUFBYTtRQUMvQixJQUFNLElBQUksR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQyxDQUFDO1FBRS9DLEVBQUUsQ0FBQyxDQUFDLElBQUksSUFBSSxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsVUFBVSxDQUFDLGtCQUFrQixDQUFDLENBQUMsQ0FBQztZQUNsRSxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLEVBQUUsQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQztnQkFDNUMsSUFBSSxDQUFDLE9BQU8sR0FBRyxJQUFJLENBQUM7WUFDdEIsQ0FBQztZQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLEVBQUUsQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQztnQkFDcEQsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLENBQUM7WUFDdkIsQ0FBQztZQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxLQUFLLEVBQUUsQ0FBQyxVQUFVLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztnQkFDbEQsSUFBTSxJQUFJLEdBQUksSUFBc0IsQ0FBQyxJQUFJLENBQUM7Z0JBQzFDLEVBQUUsQ0FBQyxDQUFDLElBQUksS0FBSyxXQUFXLENBQUMsQ0FBQyxDQUFDO29CQUN6QixJQUFJLENBQUMsWUFBWSxHQUFHLElBQUksQ0FBQztnQkFDM0IsQ0FBQztnQkFBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUMsSUFBSSxLQUFLLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDO29CQUN0QyxJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksQ0FBQztnQkFDMUIsQ0FBQztZQUNILENBQUM7WUFBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQ1IsSUFBSSxDQUFDLElBQUksS0FBSyxFQUFFLENBQUMsVUFBVSxDQUFDLHdCQUF3QjtnQkFDcEQsaUJBQWlCLENBQUMsSUFBbUMsRUFBRSxLQUFLLEVBQUUsUUFBUSxDQUN4RSxDQUFDLENBQUMsQ0FBQztnQkFDRCxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksQ0FBQztZQUN0QixDQUFDO1FBQ0gsQ0FBQztRQUNELGdCQUFLLENBQUMsU0FBUyxZQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3hCLENBQUM7SUFDSCxpQkFBQztBQUFELENBekhBLEFBeUhDLENBekh3QixJQUFJLENBQUMsVUFBVSxHQXlIdkMiLCJmaWxlIjoicnVsZXMvdGVyUHJlZmVyQXJyb3dDYWxsYmFja1J1bGUuanMiLCJzb3VyY2VSb290IjoiL1VzZXJzL2ptbG9wZXovV29ya3NwYWNlL3RzbGludC1lc2xpbnQtcnVsZXMvc3JjIn0=
