"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const sweepline_intersections_1 = tslib_1.__importDefault(require("sweepline-intersections"));
const helpers_1 = require("@turf/helpers");
/**
 * Takes a {@link LineString|linestring}, {@link MultiLineString|multi-linestring},
 * {@link MultiPolygon|multi-polygon} or {@link Polygon|polygon} and
 * returns {@link Point|points} at all self-intersections.
 *
 * @name kinks
 * @param {Feature<LineString|MultiLineString|MultiPolygon|Polygon>} featureIn input feature
 * @returns {FeatureCollection<Point>} self-intersections
 * @example
 * var poly = turf.polygon([[
 *   [-12.034835, 8.901183],
 *   [-12.060413, 8.899826],
 *   [-12.03638, 8.873199],
 *   [-12.059383, 8.871418],
 *   [-12.034835, 8.901183]
 * ]]);
 *
 * var kinks = turf.kinks(poly);
 *
 * //addToMap
 * var addToMap = [poly, kinks]
 */
function kinks(featureIn) {
    const results = {
        type: "FeatureCollection",
        features: [],
    };
    if (featureIn.type === "Feature" &&
        (featureIn.geometry.type === "Point" ||
            featureIn.geometry.type === "MultiPoint")) {
        throw new Error("Input must be a LineString, MultiLineString, " +
            "Polygon, or MultiPolygon Feature or Geometry");
    }
    const intersections = sweepline_intersections_1.default(featureIn, false);
    for (let i = 0; i < intersections.length; ++i) {
        const intersection = intersections[i];
        results.features.push(helpers_1.point([intersection[0], intersection[1]]));
    }
    return results;
}
exports.default = kinks;
