/**
 * Earth Radius used with the Harvesine formula and approximates using a spherical (non-ellipsoid) Earth.
 */
var earthRadius = 6371008.8;

/**
 * Unit of measurement factors using a spherical (non-ellipsoid) earth radius.
 */
var factors = {
    meters: earthRadius,
    metres: earthRadius,
    millimeters: earthRadius * 1000,
    millimetres: earthRadius * 1000,
    centimeters: earthRadius * 100,
    centimetres: earthRadius * 100,
    kilometers: earthRadius / 1000,
    kilometres: earthRadius / 1000,
    miles: earthRadius / 1609.344,
    nauticalmiles: earthRadius / 1852,
    inches: earthRadius * 39.370,
    yards: earthRadius / 1.0936,
    feet: earthRadius * 3.28084,
    radians: 1,
    degrees: earthRadius / 111325,
};

/**
 * Units of measurement factors based on 1 meter.
 */
var unitsFactors = {
    meters: 1,
    metres: 1,
    millimeters: 1000,
    millimetres: 1000,
    centimeters: 100,
    centimetres: 100,
    kilometers: 1 / 1000,
    kilometres: 1 / 1000,
    miles: 1 / 1609.344,
    nauticalmiles: 1 / 1852,
    inches: 39.370,
    yards: 1 / 1.0936,
    feet: 3.28084,
    radians: 1 / earthRadius,
    degrees: 1 / 111325,
};

/**
 * Area of measurement factors based on 1 square meter.
 */
var areaFactors = {
    meters: 1,
    metres: 1,
    millimeters: 1000000,
    millimetres: 1000000,
    centimeters: 10000,
    centimetres: 10000,
    kilometers: 0.000001,
    kilometres: 0.000001,
    acres: 0.000247105,
    miles: 3.86e-7,
    yards: 1.195990046,
    feet: 10.763910417,
    inches: 1550.003100006
};

/**
 * Wraps a GeoJSON {@link Geometry} in a GeoJSON {@link Feature}.
 *
 * @name feature
 * @param {Geometry} geometry input geometry
 * @param {Object} [properties={}] an Object of key-value pairs to add as properties
 * @param {Object} [options={}] Optional Parameters
 * @param {Array<number>} [options.bbox] Bounding Box Array [west, south, east, north] associated with the Feature
 * @param {string|number} [options.id] Identifier associated with the Feature
 * @returns {Feature} a GeoJSON Feature
 * @example
 * var geometry = {
 *   "type": "Point",
 *   "coordinates": [110, 50]
 * };
 *
 * var feature = turf.feature(geometry);
 *
 * //=feature
 */
function feature(geometry, properties, options) {
    // Optional Parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var bbox = options.bbox;
    var id = options.id;

    // Validation
    if (geometry === undefined) throw new Error('geometry is required');
    if (properties && properties.constructor !== Object) throw new Error('properties must be an Object');
    if (bbox) validateBBox(bbox);
    if (id) validateId(id);

    // Main
    var feat = {type: 'Feature'};
    if (id) feat.id = id;
    if (bbox) feat.bbox = bbox;
    feat.properties = properties || {};
    feat.geometry = geometry;
    return feat;
}

/**
 * Creates a GeoJSON {@link Geometry} from a Geometry string type & coordinates.
 * For GeometryCollection type use `helpers.geometryCollection`
 *
 * @name geometry
 * @param {string} type Geometry Type
 * @param {Array<number>} coordinates Coordinates
 * @param {Object} [options={}] Optional Parameters
 * @param {Array<number>} [options.bbox] Bounding Box Array [west, south, east, north] associated with the Geometry
 * @returns {Geometry} a GeoJSON Geometry
 * @example
 * var type = 'Point';
 * var coordinates = [110, 50];
 *
 * var geometry = turf.geometry(type, coordinates);
 *
 * //=geometry
 */
function geometry(type, coordinates, options) {
    // Optional Parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var bbox = options.bbox;

    // Validation
    if (!type) throw new Error('type is required');
    if (!coordinates) throw new Error('coordinates is required');
    if (!Array.isArray(coordinates)) throw new Error('coordinates must be an Array');
    if (bbox) validateBBox(bbox);

    // Main
    var geom;
    switch (type) {
    case 'Point': geom = point(coordinates).geometry; break;
    case 'LineString': geom = lineString(coordinates).geometry; break;
    case 'Polygon': geom = polygon(coordinates).geometry; break;
    case 'MultiPoint': geom = multiPoint(coordinates).geometry; break;
    case 'MultiLineString': geom = multiLineString(coordinates).geometry; break;
    case 'MultiPolygon': geom = multiPolygon(coordinates).geometry; break;
    default: throw new Error(type + ' is invalid');
    }
    if (bbox) geom.bbox = bbox;
    return geom;
}

/**
 * Creates a {@link Point} {@link Feature} from a Position.
 *
 * @name point
 * @param {Array<number>} coordinates longitude, latitude position (each in decimal degrees)
 * @param {Object} [properties={}] an Object of key-value pairs to add as properties
 * @param {Object} [options={}] Optional Parameters
 * @param {Array<number>} [options.bbox] Bounding Box Array [west, south, east, north] associated with the Feature
 * @param {string|number} [options.id] Identifier associated with the Feature
 * @returns {Feature<Point>} a Point feature
 * @example
 * var point = turf.point([-75.343, 39.984]);
 *
 * //=point
 */
function point(coordinates, properties, options) {
    if (!coordinates) throw new Error('coordinates is required');
    if (!Array.isArray(coordinates)) throw new Error('coordinates must be an Array');
    if (coordinates.length < 2) throw new Error('coordinates must be at least 2 numbers long');
    if (!isNumber(coordinates[0]) || !isNumber(coordinates[1])) throw new Error('coordinates must contain numbers');

    return feature({
        type: 'Point',
        coordinates: coordinates
    }, properties, options);
}

/**
 * Creates a {@link Point} {@link FeatureCollection} from an Array of Point coordinates.
 *
 * @name points
 * @param {Array<Array<number>>} coordinates an array of Points
 * @param {Object} [properties={}] Translate these properties to each Feature
 * @param {Object} [options={}] Optional Parameters
 * @param {Array<number>} [options.bbox] Bounding Box Array [west, south, east, north] associated with the FeatureCollection
 * @param {string|number} [options.id] Identifier associated with the FeatureCollection
 * @returns {FeatureCollection<Point>} Point Feature
 * @example
 * var points = turf.points([
 *   [-75, 39],
 *   [-80, 45],
 *   [-78, 50]
 * ]);
 *
 * //=points
 */
function points(coordinates, properties, options) {
    if (!coordinates) throw new Error('coordinates is required');
    if (!Array.isArray(coordinates)) throw new Error('coordinates must be an Array');

    return featureCollection(coordinates.map(function (coords) {
        return point(coords, properties);
    }), options);
}

/**
 * Creates a {@link Polygon} {@link Feature} from an Array of LinearRings.
 *
 * @name polygon
 * @param {Array<Array<Array<number>>>} coordinates an array of LinearRings
 * @param {Object} [properties={}] an Object of key-value pairs to add as properties
 * @param {Object} [options={}] Optional Parameters
 * @param {Array<number>} [options.bbox] Bounding Box Array [west, south, east, north] associated with the Feature
 * @param {string|number} [options.id] Identifier associated with the Feature
 * @returns {Feature<Polygon>} Polygon Feature
 * @example
 * var polygon = turf.polygon([[[-5, 52], [-4, 56], [-2, 51], [-7, 54], [-5, 52]]], { name: 'poly1' });
 *
 * //=polygon
 */
function polygon(coordinates, properties, options) {
    if (!coordinates) throw new Error('coordinates is required');

    for (var i = 0; i < coordinates.length; i++) {
        var ring = coordinates[i];
        if (ring.length < 4) {
            throw new Error('Each LinearRing of a Polygon must have 4 or more Positions.');
        }
        for (var j = 0; j < ring[ring.length - 1].length; j++) {
            // Check if first point of Polygon contains two numbers
            if (i === 0 && j === 0 && !isNumber(ring[0][0]) || !isNumber(ring[0][1])) throw new Error('coordinates must contain numbers');
            if (ring[ring.length - 1][j] !== ring[0][j]) {
                throw new Error('First and last Position are not equivalent.');
            }
        }
    }

    return feature({
        type: 'Polygon',
        coordinates: coordinates
    }, properties, options);
}

/**
 * Creates a {@link Polygon} {@link FeatureCollection} from an Array of Polygon coordinates.
 *
 * @name polygons
 * @param {Array<Array<Array<Array<number>>>>} coordinates an array of Polygon coordinates
 * @param {Object} [properties={}] an Object of key-value pairs to add as properties
 * @param {Object} [options={}] Optional Parameters
 * @param {Array<number>} [options.bbox] Bounding Box Array [west, south, east, north] associated with the Feature
 * @param {string|number} [options.id] Identifier associated with the FeatureCollection
 * @returns {FeatureCollection<Polygon>} Polygon FeatureCollection
 * @example
 * var polygons = turf.polygons([
 *   [[[-5, 52], [-4, 56], [-2, 51], [-7, 54], [-5, 52]]],
 *   [[[-15, 42], [-14, 46], [-12, 41], [-17, 44], [-15, 42]]],
 * ]);
 *
 * //=polygons
 */
function polygons(coordinates, properties, options) {
    if (!coordinates) throw new Error('coordinates is required');
    if (!Array.isArray(coordinates)) throw new Error('coordinates must be an Array');

    return featureCollection(coordinates.map(function (coords) {
        return polygon(coords, properties);
    }), options);
}

/**
 * Creates a {@link LineString} {@link Feature} from an Array of Positions.
 *
 * @name lineString
 * @param {Array<Array<number>>} coordinates an array of Positions
 * @param {Object} [properties={}] an Object of key-value pairs to add as properties
 * @param {Object} [options={}] Optional Parameters
 * @param {Array<number>} [options.bbox] Bounding Box Array [west, south, east, north] associated with the Feature
 * @param {string|number} [options.id] Identifier associated with the Feature
 * @returns {Feature<LineString>} LineString Feature
 * @example
 * var linestring1 = turf.lineString([[-24, 63], [-23, 60], [-25, 65], [-20, 69]], {name: 'line 1'});
 * var linestring2 = turf.lineString([[-14, 43], [-13, 40], [-15, 45], [-10, 49]], {name: 'line 2'});
 *
 * //=linestring1
 * //=linestring2
 */
function lineString(coordinates, properties, options) {
    if (!coordinates) throw new Error('coordinates is required');
    if (coordinates.length < 2) throw new Error('coordinates must be an array of two or more positions');
    // Check if first point of LineString contains two numbers
    if (!isNumber(coordinates[0][1]) || !isNumber(coordinates[0][1])) throw new Error('coordinates must contain numbers');

    return feature({
        type: 'LineString',
        coordinates: coordinates
    }, properties, options);
}

/**
 * Creates a {@link LineString} {@link FeatureCollection} from an Array of LineString coordinates.
 *
 * @name lineStrings
 * @param {Array<Array<number>>} coordinates an array of LinearRings
 * @param {Object} [properties={}] an Object of key-value pairs to add as properties
 * @param {Object} [options={}] Optional Parameters
 * @param {Array<number>} [options.bbox] Bounding Box Array [west, south, east, north] associated with the FeatureCollection
 * @param {string|number} [options.id] Identifier associated with the FeatureCollection
 * @returns {FeatureCollection<LineString>} LineString FeatureCollection
 * @example
 * var linestrings = turf.lineStrings([
 *   [[-24, 63], [-23, 60], [-25, 65], [-20, 69]],
 *   [[-14, 43], [-13, 40], [-15, 45], [-10, 49]]
 * ]);
 *
 * //=linestrings
 */
function lineStrings(coordinates, properties, options) {
    if (!coordinates) throw new Error('coordinates is required');
    if (!Array.isArray(coordinates)) throw new Error('coordinates must be an Array');

    return featureCollection(coordinates.map(function (coords) {
        return lineString(coords, properties);
    }), options);
}

/**
 * Takes one or more {@link Feature|Features} and creates a {@link FeatureCollection}.
 *
 * @name featureCollection
 * @param {Feature[]} features input features
 * @param {Object} [options={}] Optional Parameters
 * @param {Array<number>} [options.bbox] Bounding Box Array [west, south, east, north] associated with the Feature
 * @param {string|number} [options.id] Identifier associated with the Feature
 * @returns {FeatureCollection} FeatureCollection of Features
 * @example
 * var locationA = turf.point([-75.343, 39.984], {name: 'Location A'});
 * var locationB = turf.point([-75.833, 39.284], {name: 'Location B'});
 * var locationC = turf.point([-75.534, 39.123], {name: 'Location C'});
 *
 * var collection = turf.featureCollection([
 *   locationA,
 *   locationB,
 *   locationC
 * ]);
 *
 * //=collection
 */
function featureCollection(features, options) {
    // Optional Parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var bbox = options.bbox;
    var id = options.id;

    // Validation
    if (!features) throw new Error('No features passed');
    if (!Array.isArray(features)) throw new Error('features must be an Array');
    if (bbox) validateBBox(bbox);
    if (id) validateId(id);

    // Main
    var fc = {type: 'FeatureCollection'};
    if (id) fc.id = id;
    if (bbox) fc.bbox = bbox;
    fc.features = features;
    return fc;
}

/**
 * Creates a {@link Feature<MultiLineString>} based on a
 * coordinate array. Properties can be added optionally.
 *
 * @name multiLineString
 * @param {Array<Array<Array<number>>>} coordinates an array of LineStrings
 * @param {Object} [properties={}] an Object of key-value pairs to add as properties
 * @param {Object} [options={}] Optional Parameters
 * @param {Array<number>} [options.bbox] Bounding Box Array [west, south, east, north] associated with the Feature
 * @param {string|number} [options.id] Identifier associated with the Feature
 * @returns {Feature<MultiLineString>} a MultiLineString feature
 * @throws {Error} if no coordinates are passed
 * @example
 * var multiLine = turf.multiLineString([[[0,0],[10,10]]]);
 *
 * //=multiLine
 */
function multiLineString(coordinates, properties, options) {
    if (!coordinates) throw new Error('coordinates is required');

    return feature({
        type: 'MultiLineString',
        coordinates: coordinates
    }, properties, options);
}

/**
 * Creates a {@link Feature<MultiPoint>} based on a
 * coordinate array. Properties can be added optionally.
 *
 * @name multiPoint
 * @param {Array<Array<number>>} coordinates an array of Positions
 * @param {Object} [properties={}] an Object of key-value pairs to add as properties
 * @param {Object} [options={}] Optional Parameters
 * @param {Array<number>} [options.bbox] Bounding Box Array [west, south, east, north] associated with the Feature
 * @param {string|number} [options.id] Identifier associated with the Feature
 * @returns {Feature<MultiPoint>} a MultiPoint feature
 * @throws {Error} if no coordinates are passed
 * @example
 * var multiPt = turf.multiPoint([[0,0],[10,10]]);
 *
 * //=multiPt
 */
function multiPoint(coordinates, properties, options) {
    if (!coordinates) throw new Error('coordinates is required');

    return feature({
        type: 'MultiPoint',
        coordinates: coordinates
    }, properties, options);
}

/**
 * Creates a {@link Feature<MultiPolygon>} based on a
 * coordinate array. Properties can be added optionally.
 *
 * @name multiPolygon
 * @param {Array<Array<Array<Array<number>>>>} coordinates an array of Polygons
 * @param {Object} [properties={}] an Object of key-value pairs to add as properties
 * @param {Object} [options={}] Optional Parameters
 * @param {Array<number>} [options.bbox] Bounding Box Array [west, south, east, north] associated with the Feature
 * @param {string|number} [options.id] Identifier associated with the Feature
 * @returns {Feature<MultiPolygon>} a multipolygon feature
 * @throws {Error} if no coordinates are passed
 * @example
 * var multiPoly = turf.multiPolygon([[[[0,0],[0,10],[10,10],[10,0],[0,0]]]]);
 *
 * //=multiPoly
 *
 */
function multiPolygon(coordinates, properties, options) {
    if (!coordinates) throw new Error('coordinates is required');

    return feature({
        type: 'MultiPolygon',
        coordinates: coordinates
    }, properties, options);
}

/**
 * Creates a {@link Feature<GeometryCollection>} based on a
 * coordinate array. Properties can be added optionally.
 *
 * @name geometryCollection
 * @param {Array<Geometry>} geometries an array of GeoJSON Geometries
 * @param {Object} [properties={}] an Object of key-value pairs to add as properties
 * @param {Object} [options={}] Optional Parameters
 * @param {Array<number>} [options.bbox] Bounding Box Array [west, south, east, north] associated with the Feature
 * @param {string|number} [options.id] Identifier associated with the Feature
 * @returns {Feature<GeometryCollection>} a GeoJSON GeometryCollection Feature
 * @example
 * var pt = {
 *     "type": "Point",
 *       "coordinates": [100, 0]
 *     };
 * var line = {
 *     "type": "LineString",
 *     "coordinates": [ [101, 0], [102, 1] ]
 *   };
 * var collection = turf.geometryCollection([pt, line]);
 *
 * //=collection
 */
function geometryCollection(geometries, properties, options) {
    if (!geometries) throw new Error('geometries is required');
    if (!Array.isArray(geometries)) throw new Error('geometries must be an Array');

    return feature({
        type: 'GeometryCollection',
        geometries: geometries
    }, properties, options);
}

/**
 * Round number to precision
 *
 * @param {number} num Number
 * @param {number} [precision=0] Precision
 * @returns {number} rounded number
 * @example
 * turf.round(120.4321)
 * //=120
 *
 * turf.round(120.4321, 2)
 * //=120.43
 */
function round(num, precision) {
    if (num === undefined || num === null || isNaN(num)) throw new Error('num is required');
    if (precision && !(precision >= 0)) throw new Error('precision must be a positive number');
    var multiplier = Math.pow(10, precision || 0);
    return Math.round(num * multiplier) / multiplier;
}

/**
 * Convert a distance measurement (assuming a spherical Earth) from radians to a more friendly unit.
 * Valid units: miles, nauticalmiles, inches, yards, meters, metres, kilometers, centimeters, feet
 *
 * @name radiansToLength
 * @param {number} radians in radians across the sphere
 * @param {string} [units='kilometers'] can be degrees, radians, miles, or kilometers inches, yards, metres, meters, kilometres, kilometers.
 * @returns {number} distance
 */
function radiansToLength(radians, units) {
    if (radians === undefined || radians === null) throw new Error('radians is required');

    if (units && typeof units !== 'string') throw new Error('units must be a string');
    var factor = factors[units || 'kilometers'];
    if (!factor) throw new Error(units + ' units is invalid');
    return radians * factor;
}

/**
 * Convert a distance measurement (assuming a spherical Earth) from a real-world unit into radians
 * Valid units: miles, nauticalmiles, inches, yards, meters, metres, kilometers, centimeters, feet
 *
 * @name lengthToRadians
 * @param {number} distance in real units
 * @param {string} [units='kilometers'] can be degrees, radians, miles, or kilometers inches, yards, metres, meters, kilometres, kilometers.
 * @returns {number} radians
 */
function lengthToRadians(distance, units) {
    if (distance === undefined || distance === null) throw new Error('distance is required');

    if (units && typeof units !== 'string') throw new Error('units must be a string');
    var factor = factors[units || 'kilometers'];
    if (!factor) throw new Error(units + ' units is invalid');
    return distance / factor;
}

/**
 * Convert a distance measurement (assuming a spherical Earth) from a real-world unit into degrees
 * Valid units: miles, nauticalmiles, inches, yards, meters, metres, centimeters, kilometres, feet
 *
 * @name lengthToDegrees
 * @param {number} distance in real units
 * @param {string} [units='kilometers'] can be degrees, radians, miles, or kilometers inches, yards, metres, meters, kilometres, kilometers.
 * @returns {number} degrees
 */
function lengthToDegrees(distance, units) {
    return radiansToDegrees(lengthToRadians(distance, units));
}

/**
 * Converts any bearing angle from the north line direction (positive clockwise)
 * and returns an angle between 0-360 degrees (positive clockwise), 0 being the north line
 *
 * @name bearingToAzimuth
 * @param {number} bearing angle, between -180 and +180 degrees
 * @returns {number} angle between 0 and 360 degrees
 */
function bearingToAzimuth(bearing) {
    if (bearing === null || bearing === undefined) throw new Error('bearing is required');

    var angle = bearing % 360;
    if (angle < 0) angle += 360;
    return angle;
}

/**
 * Converts an angle in radians to degrees
 *
 * @name radiansToDegrees
 * @param {number} radians angle in radians
 * @returns {number} degrees between 0 and 360 degrees
 */
function radiansToDegrees(radians) {
    if (radians === null || radians === undefined) throw new Error('radians is required');

    var degrees = radians % (2 * Math.PI);
    return degrees * 180 / Math.PI;
}

/**
 * Converts an angle in degrees to radians
 *
 * @name degreesToRadians
 * @param {number} degrees angle between 0 and 360 degrees
 * @returns {number} angle in radians
 */
function degreesToRadians(degrees) {
    if (degrees === null || degrees === undefined) throw new Error('degrees is required');

    var radians = degrees % 360;
    return radians * Math.PI / 180;
}

/**
 * Converts a length to the requested unit.
 * Valid units: miles, nauticalmiles, inches, yards, meters, metres, kilometers, centimeters, feet
 *
 * @param {number} length to be converted
 * @param {string} originalUnit of the length
 * @param {string} [finalUnit='kilometers'] returned unit
 * @returns {number} the converted length
 */
function convertLength(length, originalUnit, finalUnit) {
    if (length === null || length === undefined) throw new Error('length is required');
    if (!(length >= 0)) throw new Error('length must be a positive number');

    return radiansToLength(lengthToRadians(length, originalUnit), finalUnit || 'kilometers');
}

/**
 * Converts a area to the requested unit.
 * Valid units: kilometers, kilometres, meters, metres, centimetres, millimeters, acres, miles, yards, feet, inches
 * @param {number} area to be converted
 * @param {string} [originalUnit='meters'] of the distance
 * @param {string} [finalUnit='kilometers'] returned unit
 * @returns {number} the converted distance
 */
function convertArea(area, originalUnit, finalUnit) {
    if (area === null || area === undefined) throw new Error('area is required');
    if (!(area >= 0)) throw new Error('area must be a positive number');

    var startFactor = areaFactors[originalUnit || 'meters'];
    if (!startFactor) throw new Error('invalid original units');

    var finalFactor = areaFactors[finalUnit || 'kilometers'];
    if (!finalFactor) throw new Error('invalid final units');

    return (area / startFactor) * finalFactor;
}

/**
 * isNumber
 *
 * @param {*} num Number to validate
 * @returns {boolean} true/false
 * @example
 * turf.isNumber(123)
 * //=true
 * turf.isNumber('foo')
 * //=false
 */
function isNumber(num) {
    return !isNaN(num) && num !== null && !Array.isArray(num);
}

/**
 * isObject
 *
 * @param {*} input variable to validate
 * @returns {boolean} true/false
 * @example
 * turf.isObject({elevation: 10})
 * //=true
 * turf.isObject('foo')
 * //=false
 */
function isObject(input) {
    return (!!input) && (input.constructor === Object);
}

/**
 * Validate BBox
 *
 * @private
 * @param {Array<number>} bbox BBox to validate
 * @returns {void}
 * @throws Error if BBox is not valid
 * @example
 * validateBBox([-180, -40, 110, 50])
 * //=OK
 * validateBBox([-180, -40])
 * //=Error
 * validateBBox('Foo')
 * //=Error
 * validateBBox(5)
 * //=Error
 * validateBBox(null)
 * //=Error
 * validateBBox(undefined)
 * //=Error
 */
function validateBBox(bbox) {
    if (!bbox) throw new Error('bbox is required');
    if (!Array.isArray(bbox)) throw new Error('bbox must be an Array');
    if (bbox.length !== 4 && bbox.length !== 6) throw new Error('bbox must be an Array of 4 or 6 numbers');
    bbox.forEach(function (num) {
        if (!isNumber(num)) throw new Error('bbox must only contain numbers');
    });
}

/**
 * Validate Id
 *
 * @private
 * @param {string|number} id Id to validate
 * @returns {void}
 * @throws Error if Id is not valid
 * @example
 * validateId([-180, -40, 110, 50])
 * //=Error
 * validateId([-180, -40])
 * //=Error
 * validateId('Foo')
 * //=OK
 * validateId(5)
 * //=OK
 * validateId(null)
 * //=Error
 * validateId(undefined)
 * //=Error
 */
function validateId(id) {
    if (!id) throw new Error('id is required');
    if (['string', 'number'].indexOf(typeof id) === -1) throw new Error('id must be a number or a string');
}

// Deprecated methods
function radians2degrees() {
    throw new Error('method has been renamed to `radiansToDegrees`');
}

function degrees2radians() {
    throw new Error('method has been renamed to `degreesToRadians`');
}

function distanceToDegrees() {
    throw new Error('method has been renamed to `lengthToDegrees`');
}

function distanceToRadians() {
    throw new Error('method has been renamed to `lengthToRadians`');
}

function radiansToDistance() {
    throw new Error('method has been renamed to `radiansToLength`');
}

function bearingToAngle() {
    throw new Error('method has been renamed to `bearingToAzimuth`');
}

function convertDistance() {
    throw new Error('method has been renamed to `convertLength`');
}




var main_es$1 = Object.freeze({
	earthRadius: earthRadius,
	factors: factors,
	unitsFactors: unitsFactors,
	areaFactors: areaFactors,
	feature: feature,
	geometry: geometry,
	point: point,
	points: points,
	polygon: polygon,
	polygons: polygons,
	lineString: lineString,
	lineStrings: lineStrings,
	featureCollection: featureCollection,
	multiLineString: multiLineString,
	multiPoint: multiPoint,
	multiPolygon: multiPolygon,
	geometryCollection: geometryCollection,
	round: round,
	radiansToLength: radiansToLength,
	lengthToRadians: lengthToRadians,
	lengthToDegrees: lengthToDegrees,
	bearingToAzimuth: bearingToAzimuth,
	radiansToDegrees: radiansToDegrees,
	degreesToRadians: degreesToRadians,
	convertLength: convertLength,
	convertArea: convertArea,
	isNumber: isNumber,
	isObject: isObject,
	validateBBox: validateBBox,
	validateId: validateId,
	radians2degrees: radians2degrees,
	degrees2radians: degrees2radians,
	distanceToDegrees: distanceToDegrees,
	distanceToRadians: distanceToRadians,
	radiansToDistance: radiansToDistance,
	bearingToAngle: bearingToAngle,
	convertDistance: convertDistance
});

/**
 * Callback for coordEach
 *
 * @callback coordEachCallback
 * @param {Array<number>} currentCoord The current coordinate being processed.
 * @param {number} coordIndex The current index of the coordinate being processed.
 * @param {number} featureIndex The current index of the Feature being processed.
 * @param {number} multiFeatureIndex The current index of the Multi-Feature being processed.
 * @param {number} geometryIndex The current index of the Geometry being processed.
 */

/**
 * Iterate over coordinates in any GeoJSON object, similar to Array.forEach()
 *
 * @name coordEach
 * @param {FeatureCollection|Feature|Geometry} geojson any GeoJSON object
 * @param {Function} callback a method that takes (currentCoord, coordIndex, featureIndex, multiFeatureIndex)
 * @param {boolean} [excludeWrapCoord=false] whether or not to include the final coordinate of LinearRings that wraps the ring in its iteration.
 * @example
 * var features = turf.featureCollection([
 *   turf.point([26, 37], {"foo": "bar"}),
 *   turf.point([36, 53], {"hello": "world"})
 * ]);
 *
 * turf.coordEach(features, function (currentCoord, coordIndex, featureIndex, multiFeatureIndex, geometryIndex) {
 *   //=currentCoord
 *   //=coordIndex
 *   //=featureIndex
 *   //=multiFeatureIndex
 *   //=geometryIndex
 * });
 */
function coordEach(geojson, callback, excludeWrapCoord) {
    // Handles null Geometry -- Skips this GeoJSON
    if (geojson === null) return;
    var j, k, l, geometry$$1, stopG, coords,
        geometryMaybeCollection,
        wrapShrink = 0,
        coordIndex = 0,
        isGeometryCollection,
        type = geojson.type,
        isFeatureCollection = type === 'FeatureCollection',
        isFeature = type === 'Feature',
        stop = isFeatureCollection ? geojson.features.length : 1;

    // This logic may look a little weird. The reason why it is that way
    // is because it's trying to be fast. GeoJSON supports multiple kinds
    // of objects at its root: FeatureCollection, Features, Geometries.
    // This function has the responsibility of handling all of them, and that
    // means that some of the `for` loops you see below actually just don't apply
    // to certain inputs. For instance, if you give this just a
    // Point geometry, then both loops are short-circuited and all we do
    // is gradually rename the input until it's called 'geometry'.
    //
    // This also aims to allocate as few resources as possible: just a
    // few numbers and booleans, rather than any temporary arrays as would
    // be required with the normalization approach.
    for (var featureIndex = 0; featureIndex < stop; featureIndex++) {
        geometryMaybeCollection = (isFeatureCollection ? geojson.features[featureIndex].geometry :
            (isFeature ? geojson.geometry : geojson));
        isGeometryCollection = (geometryMaybeCollection) ? geometryMaybeCollection.type === 'GeometryCollection' : false;
        stopG = isGeometryCollection ? geometryMaybeCollection.geometries.length : 1;

        for (var geomIndex = 0; geomIndex < stopG; geomIndex++) {
            var multiFeatureIndex = 0;
            var geometryIndex = 0;
            geometry$$1 = isGeometryCollection ?
                geometryMaybeCollection.geometries[geomIndex] : geometryMaybeCollection;

            // Handles null Geometry -- Skips this geometry
            if (geometry$$1 === null) continue;
            coords = geometry$$1.coordinates;
            var geomType = geometry$$1.type;

            wrapShrink = (excludeWrapCoord && (geomType === 'Polygon' || geomType === 'MultiPolygon')) ? 1 : 0;

            switch (geomType) {
            case null:
                break;
            case 'Point':
                callback(coords, coordIndex, featureIndex, multiFeatureIndex, geometryIndex);
                coordIndex++;
                multiFeatureIndex++;
                break;
            case 'LineString':
            case 'MultiPoint':
                for (j = 0; j < coords.length; j++) {
                    callback(coords[j], coordIndex, featureIndex, multiFeatureIndex, geometryIndex);
                    coordIndex++;
                    if (geomType === 'MultiPoint') multiFeatureIndex++;
                }
                if (geomType === 'LineString') multiFeatureIndex++;
                break;
            case 'Polygon':
            case 'MultiLineString':
                for (j = 0; j < coords.length; j++) {
                    for (k = 0; k < coords[j].length - wrapShrink; k++) {
                        callback(coords[j][k], coordIndex, featureIndex, multiFeatureIndex, geometryIndex);
                        coordIndex++;
                    }
                    if (geomType === 'MultiLineString') multiFeatureIndex++;
                    if (geomType === 'Polygon') geometryIndex++;
                }
                if (geomType === 'Polygon') multiFeatureIndex++;
                break;
            case 'MultiPolygon':
                for (j = 0; j < coords.length; j++) {
                    if (geomType === 'MultiPolygon') geometryIndex = 0;
                    for (k = 0; k < coords[j].length; k++) {
                        for (l = 0; l < coords[j][k].length - wrapShrink; l++) {
                            callback(coords[j][k][l], coordIndex, featureIndex, multiFeatureIndex, geometryIndex);
                            coordIndex++;
                        }
                        geometryIndex++;
                    }
                    multiFeatureIndex++;
                }
                break;
            case 'GeometryCollection':
                for (j = 0; j < geometry$$1.geometries.length; j++)
                    coordEach(geometry$$1.geometries[j], callback, excludeWrapCoord);
                break;
            default:
                throw new Error('Unknown Geometry Type');
            }
        }
    }
}

/**
 * Callback for coordReduce
 *
 * The first time the callback function is called, the values provided as arguments depend
 * on whether the reduce method has an initialValue argument.
 *
 * If an initialValue is provided to the reduce method:
 *  - The previousValue argument is initialValue.
 *  - The currentValue argument is the value of the first element present in the array.
 *
 * If an initialValue is not provided:
 *  - The previousValue argument is the value of the first element present in the array.
 *  - The currentValue argument is the value of the second element present in the array.
 *
 * @callback coordReduceCallback
 * @param {*} previousValue The accumulated value previously returned in the last invocation
 * of the callback, or initialValue, if supplied.
 * @param {Array<number>} currentCoord The current coordinate being processed.
 * @param {number} coordIndex The current index of the coordinate being processed.
 * Starts at index 0, if an initialValue is provided, and at index 1 otherwise.
 * @param {number} featureIndex The current index of the Feature being processed.
 * @param {number} multiFeatureIndex The current index of the Multi-Feature being processed.
 * @param {number} geometryIndex The current index of the Geometry being processed.
 */

/**
 * Reduce coordinates in any GeoJSON object, similar to Array.reduce()
 *
 * @name coordReduce
 * @param {FeatureCollection|Geometry|Feature} geojson any GeoJSON object
 * @param {Function} callback a method that takes (previousValue, currentCoord, coordIndex)
 * @param {*} [initialValue] Value to use as the first argument to the first call of the callback.
 * @param {boolean} [excludeWrapCoord=false] whether or not to include the final coordinate of LinearRings that wraps the ring in its iteration.
 * @returns {*} The value that results from the reduction.
 * @example
 * var features = turf.featureCollection([
 *   turf.point([26, 37], {"foo": "bar"}),
 *   turf.point([36, 53], {"hello": "world"})
 * ]);
 *
 * turf.coordReduce(features, function (previousValue, currentCoord, coordIndex, featureIndex, multiFeatureIndex, geometryIndex) {
 *   //=previousValue
 *   //=currentCoord
 *   //=coordIndex
 *   //=featureIndex
 *   //=multiFeatureIndex
 *   //=geometryIndex
 *   return currentCoord;
 * });
 */
function coordReduce(geojson, callback, initialValue, excludeWrapCoord) {
    var previousValue = initialValue;
    coordEach(geojson, function (currentCoord, coordIndex, featureIndex, multiFeatureIndex, geometryIndex) {
        if (coordIndex === 0 && initialValue === undefined) previousValue = currentCoord;
        else previousValue = callback(previousValue, currentCoord, coordIndex, featureIndex, multiFeatureIndex, geometryIndex);
    }, excludeWrapCoord);
    return previousValue;
}

/**
 * Callback for propEach
 *
 * @callback propEachCallback
 * @param {Object} currentProperties The current Properties being processed.
 * @param {number} featureIndex The current index of the Feature being processed.
 */

/**
 * Iterate over properties in any GeoJSON object, similar to Array.forEach()
 *
 * @name propEach
 * @param {FeatureCollection|Feature} geojson any GeoJSON object
 * @param {Function} callback a method that takes (currentProperties, featureIndex)
 * @example
 * var features = turf.featureCollection([
 *     turf.point([26, 37], {foo: 'bar'}),
 *     turf.point([36, 53], {hello: 'world'})
 * ]);
 *
 * turf.propEach(features, function (currentProperties, featureIndex) {
 *   //=currentProperties
 *   //=featureIndex
 * });
 */
function propEach(geojson, callback) {
    var i;
    switch (geojson.type) {
    case 'FeatureCollection':
        for (i = 0; i < geojson.features.length; i++) {
            callback(geojson.features[i].properties, i);
        }
        break;
    case 'Feature':
        callback(geojson.properties, 0);
        break;
    }
}


/**
 * Callback for propReduce
 *
 * The first time the callback function is called, the values provided as arguments depend
 * on whether the reduce method has an initialValue argument.
 *
 * If an initialValue is provided to the reduce method:
 *  - The previousValue argument is initialValue.
 *  - The currentValue argument is the value of the first element present in the array.
 *
 * If an initialValue is not provided:
 *  - The previousValue argument is the value of the first element present in the array.
 *  - The currentValue argument is the value of the second element present in the array.
 *
 * @callback propReduceCallback
 * @param {*} previousValue The accumulated value previously returned in the last invocation
 * of the callback, or initialValue, if supplied.
 * @param {*} currentProperties The current Properties being processed.
 * @param {number} featureIndex The current index of the Feature being processed.
 */

/**
 * Reduce properties in any GeoJSON object into a single value,
 * similar to how Array.reduce works. However, in this case we lazily run
 * the reduction, so an array of all properties is unnecessary.
 *
 * @name propReduce
 * @param {FeatureCollection|Feature} geojson any GeoJSON object
 * @param {Function} callback a method that takes (previousValue, currentProperties, featureIndex)
 * @param {*} [initialValue] Value to use as the first argument to the first call of the callback.
 * @returns {*} The value that results from the reduction.
 * @example
 * var features = turf.featureCollection([
 *     turf.point([26, 37], {foo: 'bar'}),
 *     turf.point([36, 53], {hello: 'world'})
 * ]);
 *
 * turf.propReduce(features, function (previousValue, currentProperties, featureIndex) {
 *   //=previousValue
 *   //=currentProperties
 *   //=featureIndex
 *   return currentProperties
 * });
 */
function propReduce(geojson, callback, initialValue) {
    var previousValue = initialValue;
    propEach(geojson, function (currentProperties, featureIndex) {
        if (featureIndex === 0 && initialValue === undefined) previousValue = currentProperties;
        else previousValue = callback(previousValue, currentProperties, featureIndex);
    });
    return previousValue;
}

/**
 * Callback for featureEach
 *
 * @callback featureEachCallback
 * @param {Feature<any>} currentFeature The current Feature being processed.
 * @param {number} featureIndex The current index of the Feature being processed.
 */

/**
 * Iterate over features in any GeoJSON object, similar to
 * Array.forEach.
 *
 * @name featureEach
 * @param {FeatureCollection|Feature|Geometry} geojson any GeoJSON object
 * @param {Function} callback a method that takes (currentFeature, featureIndex)
 * @example
 * var features = turf.featureCollection([
 *   turf.point([26, 37], {foo: 'bar'}),
 *   turf.point([36, 53], {hello: 'world'})
 * ]);
 *
 * turf.featureEach(features, function (currentFeature, featureIndex) {
 *   //=currentFeature
 *   //=featureIndex
 * });
 */
function featureEach(geojson, callback) {
    if (geojson.type === 'Feature') {
        callback(geojson, 0);
    } else if (geojson.type === 'FeatureCollection') {
        for (var i = 0; i < geojson.features.length; i++) {
            callback(geojson.features[i], i);
        }
    }
}

/**
 * Callback for featureReduce
 *
 * The first time the callback function is called, the values provided as arguments depend
 * on whether the reduce method has an initialValue argument.
 *
 * If an initialValue is provided to the reduce method:
 *  - The previousValue argument is initialValue.
 *  - The currentValue argument is the value of the first element present in the array.
 *
 * If an initialValue is not provided:
 *  - The previousValue argument is the value of the first element present in the array.
 *  - The currentValue argument is the value of the second element present in the array.
 *
 * @callback featureReduceCallback
 * @param {*} previousValue The accumulated value previously returned in the last invocation
 * of the callback, or initialValue, if supplied.
 * @param {Feature} currentFeature The current Feature being processed.
 * @param {number} featureIndex The current index of the Feature being processed.
 */

/**
 * Reduce features in any GeoJSON object, similar to Array.reduce().
 *
 * @name featureReduce
 * @param {FeatureCollection|Feature|Geometry} geojson any GeoJSON object
 * @param {Function} callback a method that takes (previousValue, currentFeature, featureIndex)
 * @param {*} [initialValue] Value to use as the first argument to the first call of the callback.
 * @returns {*} The value that results from the reduction.
 * @example
 * var features = turf.featureCollection([
 *   turf.point([26, 37], {"foo": "bar"}),
 *   turf.point([36, 53], {"hello": "world"})
 * ]);
 *
 * turf.featureReduce(features, function (previousValue, currentFeature, featureIndex) {
 *   //=previousValue
 *   //=currentFeature
 *   //=featureIndex
 *   return currentFeature
 * });
 */
function featureReduce(geojson, callback, initialValue) {
    var previousValue = initialValue;
    featureEach(geojson, function (currentFeature, featureIndex) {
        if (featureIndex === 0 && initialValue === undefined) previousValue = currentFeature;
        else previousValue = callback(previousValue, currentFeature, featureIndex);
    });
    return previousValue;
}

/**
 * Get all coordinates from any GeoJSON object.
 *
 * @name coordAll
 * @param {FeatureCollection|Feature|Geometry} geojson any GeoJSON object
 * @returns {Array<Array<number>>} coordinate position array
 * @example
 * var features = turf.featureCollection([
 *   turf.point([26, 37], {foo: 'bar'}),
 *   turf.point([36, 53], {hello: 'world'})
 * ]);
 *
 * var coords = turf.coordAll(features);
 * //= [[26, 37], [36, 53]]
 */
function coordAll(geojson) {
    var coords = [];
    coordEach(geojson, function (coord) {
        coords.push(coord);
    });
    return coords;
}

/**
 * Callback for geomEach
 *
 * @callback geomEachCallback
 * @param {Geometry} currentGeometry The current Geometry being processed.
 * @param {number} featureIndex The current index of the Feature being processed.
 * @param {Object} featureProperties The current Feature Properties being processed.
 * @param {Array<number>} featureBBox The current Feature BBox being processed.
 * @param {number|string} featureId The current Feature Id being processed.
 */

/**
 * Iterate over each geometry in any GeoJSON object, similar to Array.forEach()
 *
 * @name geomEach
 * @param {FeatureCollection|Feature|Geometry} geojson any GeoJSON object
 * @param {Function} callback a method that takes (currentGeometry, featureIndex, featureProperties, featureBBox, featureId)
 * @example
 * var features = turf.featureCollection([
 *     turf.point([26, 37], {foo: 'bar'}),
 *     turf.point([36, 53], {hello: 'world'})
 * ]);
 *
 * turf.geomEach(features, function (currentGeometry, featureIndex, featureProperties, featureBBox, featureId) {
 *   //=currentGeometry
 *   //=featureIndex
 *   //=featureProperties
 *   //=featureBBox
 *   //=featureId
 * });
 */
function geomEach(geojson, callback) {
    var i, j, g, geometry$$1, stopG,
        geometryMaybeCollection,
        isGeometryCollection,
        featureProperties,
        featureBBox,
        featureId,
        featureIndex = 0,
        isFeatureCollection = geojson.type === 'FeatureCollection',
        isFeature = geojson.type === 'Feature',
        stop = isFeatureCollection ? geojson.features.length : 1;

    // This logic may look a little weird. The reason why it is that way
    // is because it's trying to be fast. GeoJSON supports multiple kinds
    // of objects at its root: FeatureCollection, Features, Geometries.
    // This function has the responsibility of handling all of them, and that
    // means that some of the `for` loops you see below actually just don't apply
    // to certain inputs. For instance, if you give this just a
    // Point geometry, then both loops are short-circuited and all we do
    // is gradually rename the input until it's called 'geometry'.
    //
    // This also aims to allocate as few resources as possible: just a
    // few numbers and booleans, rather than any temporary arrays as would
    // be required with the normalization approach.
    for (i = 0; i < stop; i++) {

        geometryMaybeCollection = (isFeatureCollection ? geojson.features[i].geometry :
            (isFeature ? geojson.geometry : geojson));
        featureProperties = (isFeatureCollection ? geojson.features[i].properties :
            (isFeature ? geojson.properties : {}));
        featureBBox = (isFeatureCollection ? geojson.features[i].bbox :
            (isFeature ? geojson.bbox : undefined));
        featureId = (isFeatureCollection ? geojson.features[i].id :
            (isFeature ? geojson.id : undefined));
        isGeometryCollection = (geometryMaybeCollection) ? geometryMaybeCollection.type === 'GeometryCollection' : false;
        stopG = isGeometryCollection ? geometryMaybeCollection.geometries.length : 1;

        for (g = 0; g < stopG; g++) {
            geometry$$1 = isGeometryCollection ?
                geometryMaybeCollection.geometries[g] : geometryMaybeCollection;

            // Handle null Geometry
            if (geometry$$1 === null) {
                callback(null, featureIndex, featureProperties, featureBBox, featureId);
                continue;
            }
            switch (geometry$$1.type) {
            case 'Point':
            case 'LineString':
            case 'MultiPoint':
            case 'Polygon':
            case 'MultiLineString':
            case 'MultiPolygon': {
                callback(geometry$$1, featureIndex, featureProperties, featureBBox, featureId);
                break;
            }
            case 'GeometryCollection': {
                for (j = 0; j < geometry$$1.geometries.length; j++) {
                    callback(geometry$$1.geometries[j], featureIndex, featureProperties, featureBBox, featureId);
                }
                break;
            }
            default:
                throw new Error('Unknown Geometry Type');
            }
        }
        // Only increase `featureIndex` per each feature
        featureIndex++;
    }
}

/**
 * Callback for geomReduce
 *
 * The first time the callback function is called, the values provided as arguments depend
 * on whether the reduce method has an initialValue argument.
 *
 * If an initialValue is provided to the reduce method:
 *  - The previousValue argument is initialValue.
 *  - The currentValue argument is the value of the first element present in the array.
 *
 * If an initialValue is not provided:
 *  - The previousValue argument is the value of the first element present in the array.
 *  - The currentValue argument is the value of the second element present in the array.
 *
 * @callback geomReduceCallback
 * @param {*} previousValue The accumulated value previously returned in the last invocation
 * of the callback, or initialValue, if supplied.
 * @param {Geometry} currentGeometry The current Geometry being processed.
 * @param {number} featureIndex The current index of the Feature being processed.
 * @param {Object} featureProperties The current Feature Properties being processed.
 * @param {Array<number>} featureBBox The current Feature BBox being processed.
 * @param {number|string} featureId The current Feature Id being processed.
 */

/**
 * Reduce geometry in any GeoJSON object, similar to Array.reduce().
 *
 * @name geomReduce
 * @param {FeatureCollection|Feature|Geometry} geojson any GeoJSON object
 * @param {Function} callback a method that takes (previousValue, currentGeometry, featureIndex, featureProperties, featureBBox, featureId)
 * @param {*} [initialValue] Value to use as the first argument to the first call of the callback.
 * @returns {*} The value that results from the reduction.
 * @example
 * var features = turf.featureCollection([
 *     turf.point([26, 37], {foo: 'bar'}),
 *     turf.point([36, 53], {hello: 'world'})
 * ]);
 *
 * turf.geomReduce(features, function (previousValue, currentGeometry, featureIndex, featureProperties, featureBBox, featureId) {
 *   //=previousValue
 *   //=currentGeometry
 *   //=featureIndex
 *   //=featureProperties
 *   //=featureBBox
 *   //=featureId
 *   return currentGeometry
 * });
 */
function geomReduce(geojson, callback, initialValue) {
    var previousValue = initialValue;
    geomEach(geojson, function (currentGeometry, featureIndex, featureProperties, featureBBox, featureId) {
        if (featureIndex === 0 && initialValue === undefined) previousValue = currentGeometry;
        else previousValue = callback(previousValue, currentGeometry, featureIndex, featureProperties, featureBBox, featureId);
    });
    return previousValue;
}

/**
 * Callback for flattenEach
 *
 * @callback flattenEachCallback
 * @param {Feature} currentFeature The current flattened feature being processed.
 * @param {number} featureIndex The current index of the Feature being processed.
 * @param {number} multiFeatureIndex The current index of the Multi-Feature being processed.
 */

/**
 * Iterate over flattened features in any GeoJSON object, similar to
 * Array.forEach.
 *
 * @name flattenEach
 * @param {FeatureCollection|Feature|Geometry} geojson any GeoJSON object
 * @param {Function} callback a method that takes (currentFeature, featureIndex, multiFeatureIndex)
 * @example
 * var features = turf.featureCollection([
 *     turf.point([26, 37], {foo: 'bar'}),
 *     turf.multiPoint([[40, 30], [36, 53]], {hello: 'world'})
 * ]);
 *
 * turf.flattenEach(features, function (currentFeature, featureIndex, multiFeatureIndex) {
 *   //=currentFeature
 *   //=featureIndex
 *   //=multiFeatureIndex
 * });
 */
function flattenEach(geojson, callback) {
    geomEach(geojson, function (geometry$$1, featureIndex, properties, bbox, id) {
        // Callback for single geometry
        var type = (geometry$$1 === null) ? null : geometry$$1.type;
        switch (type) {
        case null:
        case 'Point':
        case 'LineString':
        case 'Polygon':
            callback(feature(geometry$$1, properties, {bbox: bbox, id: id}), featureIndex, 0);
            return;
        }

        var geomType;

        // Callback for multi-geometry
        switch (type) {
        case 'MultiPoint':
            geomType = 'Point';
            break;
        case 'MultiLineString':
            geomType = 'LineString';
            break;
        case 'MultiPolygon':
            geomType = 'Polygon';
            break;
        }

        geometry$$1.coordinates.forEach(function (coordinate, multiFeatureIndex) {
            var geom = {
                type: geomType,
                coordinates: coordinate
            };
            callback(feature(geom, properties), featureIndex, multiFeatureIndex);
        });

    });
}

/**
 * Callback for flattenReduce
 *
 * The first time the callback function is called, the values provided as arguments depend
 * on whether the reduce method has an initialValue argument.
 *
 * If an initialValue is provided to the reduce method:
 *  - The previousValue argument is initialValue.
 *  - The currentValue argument is the value of the first element present in the array.
 *
 * If an initialValue is not provided:
 *  - The previousValue argument is the value of the first element present in the array.
 *  - The currentValue argument is the value of the second element present in the array.
 *
 * @callback flattenReduceCallback
 * @param {*} previousValue The accumulated value previously returned in the last invocation
 * of the callback, or initialValue, if supplied.
 * @param {Feature} currentFeature The current Feature being processed.
 * @param {number} featureIndex The current index of the Feature being processed.
 * @param {number} multiFeatureIndex The current index of the Multi-Feature being processed.
 */

/**
 * Reduce flattened features in any GeoJSON object, similar to Array.reduce().
 *
 * @name flattenReduce
 * @param {FeatureCollection|Feature|Geometry} geojson any GeoJSON object
 * @param {Function} callback a method that takes (previousValue, currentFeature, featureIndex, multiFeatureIndex)
 * @param {*} [initialValue] Value to use as the first argument to the first call of the callback.
 * @returns {*} The value that results from the reduction.
 * @example
 * var features = turf.featureCollection([
 *     turf.point([26, 37], {foo: 'bar'}),
 *     turf.multiPoint([[40, 30], [36, 53]], {hello: 'world'})
 * ]);
 *
 * turf.flattenReduce(features, function (previousValue, currentFeature, featureIndex, multiFeatureIndex) {
 *   //=previousValue
 *   //=currentFeature
 *   //=featureIndex
 *   //=multiFeatureIndex
 *   return currentFeature
 * });
 */
function flattenReduce(geojson, callback, initialValue) {
    var previousValue = initialValue;
    flattenEach(geojson, function (currentFeature, featureIndex, multiFeatureIndex) {
        if (featureIndex === 0 && multiFeatureIndex === 0 && initialValue === undefined) previousValue = currentFeature;
        else previousValue = callback(previousValue, currentFeature, featureIndex, multiFeatureIndex);
    });
    return previousValue;
}

/**
 * Callback for segmentEach
 *
 * @callback segmentEachCallback
 * @param {Feature<LineString>} currentSegment The current Segment being processed.
 * @param {number} featureIndex The current index of the Feature being processed.
 * @param {number} multiFeatureIndex The current index of the Multi-Feature being processed.
 * @param {number} geometryIndex The current index of the Geometry being processed.
 * @param {number} segmentIndex The current index of the Segment being processed.
 * @returns {void}
 */

/**
 * Iterate over 2-vertex line segment in any GeoJSON object, similar to Array.forEach()
 * (Multi)Point geometries do not contain segments therefore they are ignored during this operation.
 *
 * @param {FeatureCollection|Feature|Geometry} geojson any GeoJSON
 * @param {Function} callback a method that takes (currentSegment, featureIndex, multiFeatureIndex, geometryIndex, segmentIndex)
 * @returns {void}
 * @example
 * var polygon = turf.polygon([[[-50, 5], [-40, -10], [-50, -10], [-40, 5], [-50, 5]]]);
 *
 * // Iterate over GeoJSON by 2-vertex segments
 * turf.segmentEach(polygon, function (currentSegment, featureIndex, multiFeatureIndex, geometryIndex, segmentIndex) {
 *   //=currentSegment
 *   //=featureIndex
 *   //=multiFeatureIndex
 *   //=geometryIndex
 *   //=segmentIndex
 * });
 *
 * // Calculate the total number of segments
 * var total = 0;
 * turf.segmentEach(polygon, function () {
 *     total++;
 * });
 */
function segmentEach(geojson, callback) {
    flattenEach(geojson, function (feature$$1, featureIndex, multiFeatureIndex) {
        var segmentIndex = 0;

        // Exclude null Geometries
        if (!feature$$1.geometry) return;
        // (Multi)Point geometries do not contain segments therefore they are ignored during this operation.
        var type = feature$$1.geometry.type;
        if (type === 'Point' || type === 'MultiPoint') return;

        // Generate 2-vertex line segments
        coordReduce(feature$$1, function (previousCoords, currentCoord, coordIndex, featureIndexCoord, mutliPartIndexCoord, geometryIndex) {
            var currentSegment = lineString([previousCoords, currentCoord], feature$$1.properties);
            callback(currentSegment, featureIndex, multiFeatureIndex, geometryIndex, segmentIndex);
            segmentIndex++;
            return currentCoord;
        });
    });
}

/**
 * Callback for segmentReduce
 *
 * The first time the callback function is called, the values provided as arguments depend
 * on whether the reduce method has an initialValue argument.
 *
 * If an initialValue is provided to the reduce method:
 *  - The previousValue argument is initialValue.
 *  - The currentValue argument is the value of the first element present in the array.
 *
 * If an initialValue is not provided:
 *  - The previousValue argument is the value of the first element present in the array.
 *  - The currentValue argument is the value of the second element present in the array.
 *
 * @callback segmentReduceCallback
 * @param {*} previousValue The accumulated value previously returned in the last invocation
 * of the callback, or initialValue, if supplied.
 * @param {Feature<LineString>} currentSegment The current Segment being processed.
 * @param {number} featureIndex The current index of the Feature being processed.
 * @param {number} multiFeatureIndex The current index of the Multi-Feature being processed.
 * @param {number} geometryIndex The current index of the Geometry being processed.
 * @param {number} segmentIndex The current index of the Segment being processed.
 */

/**
 * Reduce 2-vertex line segment in any GeoJSON object, similar to Array.reduce()
 * (Multi)Point geometries do not contain segments therefore they are ignored during this operation.
 *
 * @param {FeatureCollection|Feature|Geometry} geojson any GeoJSON
 * @param {Function} callback a method that takes (previousValue, currentSegment, currentIndex)
 * @param {*} [initialValue] Value to use as the first argument to the first call of the callback.
 * @returns {void}
 * @example
 * var polygon = turf.polygon([[[-50, 5], [-40, -10], [-50, -10], [-40, 5], [-50, 5]]]);
 *
 * // Iterate over GeoJSON by 2-vertex segments
 * turf.segmentReduce(polygon, function (previousSegment, currentSegment, featureIndex, multiFeatureIndex, geometryIndex, segmentIndex) {
 *   //= previousSegment
 *   //= currentSegment
 *   //= featureIndex
 *   //= multiFeatureIndex
 *   //= geometryIndex
 *   //= segmentInex
 *   return currentSegment
 * });
 *
 * // Calculate the total number of segments
 * var initialValue = 0
 * var total = turf.segmentReduce(polygon, function (previousValue) {
 *     previousValue++;
 *     return previousValue;
 * }, initialValue);
 */
function segmentReduce(geojson, callback, initialValue) {
    var previousValue = initialValue;
    var started = false;
    segmentEach(geojson, function (currentSegment, featureIndex, multiFeatureIndex, geometryIndex, segmentIndex) {
        if (started === false && initialValue === undefined) previousValue = currentSegment;
        else previousValue = callback(previousValue, currentSegment, featureIndex, multiFeatureIndex, geometryIndex, segmentIndex);
        started = true;
    });
    return previousValue;
}

/**
 * Callback for lineEach
 *
 * @callback lineEachCallback
 * @param {Feature<LineString>} currentLine The current LineString|LinearRing being processed
 * @param {number} featureIndex The current index of the Feature being processed
 * @param {number} multiFeatureIndex The current index of the Multi-Feature being processed
 * @param {number} geometryIndex The current index of the Geometry being processed
 */

/**
 * Iterate over line or ring coordinates in LineString, Polygon, MultiLineString, MultiPolygon Features or Geometries,
 * similar to Array.forEach.
 *
 * @name lineEach
 * @param {Geometry|Feature<LineString|Polygon|MultiLineString|MultiPolygon>} geojson object
 * @param {Function} callback a method that takes (currentLine, featureIndex, multiFeatureIndex, geometryIndex)
 * @example
 * var multiLine = turf.multiLineString([
 *   [[26, 37], [35, 45]],
 *   [[36, 53], [38, 50], [41, 55]]
 * ]);
 *
 * turf.lineEach(multiLine, function (currentLine, featureIndex, multiFeatureIndex, geometryIndex) {
 *   //=currentLine
 *   //=featureIndex
 *   //=multiFeatureIndex
 *   //=geometryIndex
 * });
 */
function lineEach(geojson, callback) {
    // validation
    if (!geojson) throw new Error('geojson is required');

    flattenEach(geojson, function (feature$$1, featureIndex, multiFeatureIndex) {
        if (feature$$1.geometry === null) return;
        var type = feature$$1.geometry.type;
        var coords = feature$$1.geometry.coordinates;
        switch (type) {
        case 'LineString':
            callback(feature$$1, featureIndex, multiFeatureIndex, 0, 0);
            break;
        case 'Polygon':
            for (var geometryIndex = 0; geometryIndex < coords.length; geometryIndex++) {
                callback(lineString(coords[geometryIndex], feature$$1.properties), featureIndex, multiFeatureIndex, geometryIndex);
            }
            break;
        }
    });
}

/**
 * Callback for lineReduce
 *
 * The first time the callback function is called, the values provided as arguments depend
 * on whether the reduce method has an initialValue argument.
 *
 * If an initialValue is provided to the reduce method:
 *  - The previousValue argument is initialValue.
 *  - The currentValue argument is the value of the first element present in the array.
 *
 * If an initialValue is not provided:
 *  - The previousValue argument is the value of the first element present in the array.
 *  - The currentValue argument is the value of the second element present in the array.
 *
 * @callback lineReduceCallback
 * @param {*} previousValue The accumulated value previously returned in the last invocation
 * of the callback, or initialValue, if supplied.
 * @param {Feature<LineString>} currentLine The current LineString|LinearRing being processed.
 * @param {number} featureIndex The current index of the Feature being processed
 * @param {number} multiFeatureIndex The current index of the Multi-Feature being processed
 * @param {number} geometryIndex The current index of the Geometry being processed
 */

/**
 * Reduce features in any GeoJSON object, similar to Array.reduce().
 *
 * @name lineReduce
 * @param {Geometry|Feature<LineString|Polygon|MultiLineString|MultiPolygon>} geojson object
 * @param {Function} callback a method that takes (previousValue, currentLine, featureIndex, multiFeatureIndex, geometryIndex)
 * @param {*} [initialValue] Value to use as the first argument to the first call of the callback.
 * @returns {*} The value that results from the reduction.
 * @example
 * var multiPoly = turf.multiPolygon([
 *   turf.polygon([[[12,48],[2,41],[24,38],[12,48]], [[9,44],[13,41],[13,45],[9,44]]]),
 *   turf.polygon([[[5, 5], [0, 0], [2, 2], [4, 4], [5, 5]]])
 * ]);
 *
 * turf.lineReduce(multiPoly, function (previousValue, currentLine, featureIndex, multiFeatureIndex, geometryIndex) {
 *   //=previousValue
 *   //=currentLine
 *   //=featureIndex
 *   //=multiFeatureIndex
 *   //=geometryIndex
 *   return currentLine
 * });
 */
function lineReduce(geojson, callback, initialValue) {
    var previousValue = initialValue;
    lineEach(geojson, function (currentLine, featureIndex, multiFeatureIndex, geometryIndex) {
        if (featureIndex === 0 && initialValue === undefined) previousValue = currentLine;
        else previousValue = callback(previousValue, currentLine, featureIndex, multiFeatureIndex, geometryIndex);
    });
    return previousValue;
}




var main_es = Object.freeze({
	coordEach: coordEach,
	coordReduce: coordReduce,
	propEach: propEach,
	propReduce: propReduce,
	featureEach: featureEach,
	featureReduce: featureReduce,
	coordAll: coordAll,
	geomEach: geomEach,
	geomReduce: geomReduce,
	flattenEach: flattenEach,
	flattenReduce: flattenReduce,
	segmentEach: segmentEach,
	segmentReduce: segmentReduce,
	lineEach: lineEach,
	lineReduce: lineReduce
});

/**
 * Takes a set of features, calculates the bbox of all input features, and returns a bounding box.
 *
 * @name bbox
 * @param {GeoJSON} geojson any GeoJSON object
 * @returns {BBox} bbox extent in [minX, minY, maxX, maxY] order
 * @example
 * var line = turf.lineString([[-74, 40], [-78, 42], [-82, 35]]);
 * var bbox = turf.bbox(line);
 * var bboxPolygon = turf.bboxPolygon(bbox);
 *
 * //addToMap
 * var addToMap = [line, bboxPolygon]
 */
function bbox(geojson) {
    var BBox = [Infinity, Infinity, -Infinity, -Infinity];
    coordEach(geojson, function (coord) {
        if (BBox[0] > coord[0]) BBox[0] = coord[0];
        if (BBox[1] > coord[1]) BBox[1] = coord[1];
        if (BBox[2] < coord[0]) BBox[2] = coord[0];
        if (BBox[3] < coord[1]) BBox[3] = coord[1];
    });
    return BBox;
}

/**
 * Unwrap a coordinate from a Point Feature, Geometry or a single coordinate.
 *
 * @name getCoord
 * @param {Array<number>|Geometry<Point>|Feature<Point>} obj Object
 * @returns {Array<number>} coordinates
 * @example
 * var pt = turf.point([10, 10]);
 *
 * var coord = turf.getCoord(pt);
 * //= [10, 10]
 */
function getCoord(obj) {
    if (!obj) throw new Error('obj is required');

    var coordinates = getCoords(obj);

    // getCoord() must contain at least two numbers (Point)
    if (coordinates.length > 1 && isNumber(coordinates[0]) && isNumber(coordinates[1])) {
        return coordinates;
    } else {
        throw new Error('Coordinate is not a valid Point');
    }
}

/**
 * Unwrap coordinates from a Feature, Geometry Object or an Array of numbers
 *
 * @name getCoords
 * @param {Array<number>|Geometry|Feature} obj Object
 * @returns {Array<number>} coordinates
 * @example
 * var poly = turf.polygon([[[119.32, -8.7], [119.55, -8.69], [119.51, -8.54], [119.32, -8.7]]]);
 *
 * var coord = turf.getCoords(poly);
 * //= [[[119.32, -8.7], [119.55, -8.69], [119.51, -8.54], [119.32, -8.7]]]
 */
function getCoords(obj) {
    if (!obj) throw new Error('obj is required');
    var coordinates;

    // Array of numbers
    if (obj.length) {
        coordinates = obj;

    // Geometry Object
    } else if (obj.coordinates) {
        coordinates = obj.coordinates;

    // Feature
    } else if (obj.geometry && obj.geometry.coordinates) {
        coordinates = obj.geometry.coordinates;
    }
    // Checks if coordinates contains a number
    if (coordinates) {
        containsNumber(coordinates);
        return coordinates;
    }
    throw new Error('No valid coordinates');
}

/**
 * Checks if coordinates contains a number
 *
 * @name containsNumber
 * @param {Array<any>} coordinates GeoJSON Coordinates
 * @returns {boolean} true if Array contains a number
 */
function containsNumber(coordinates) {
    if (coordinates.length > 1 && isNumber(coordinates[0]) && isNumber(coordinates[1])) {
        return true;
    }

    if (Array.isArray(coordinates[0]) && coordinates[0].length) {
        return containsNumber(coordinates[0]);
    }
    throw new Error('coordinates must only contain numbers');
}

/**
 * Enforce expectations about types of GeoJSON objects for Turf.
 *
 * @name geojsonType
 * @param {GeoJSON} value any GeoJSON object
 * @param {string} type expected GeoJSON type
 * @param {string} name name of calling function
 * @throws {Error} if value is not the expected type.
 */
function geojsonType(value, type, name) {
    if (!type || !name) throw new Error('type and name required');

    if (!value || value.type !== type) {
        throw new Error('Invalid input to ' + name + ': must be a ' + type + ', given ' + value.type);
    }
}

/**
 * Enforce expectations about types of {@link Feature} inputs for Turf.
 * Internally this uses {@link geojsonType} to judge geometry types.
 *
 * @name featureOf
 * @param {Feature} feature a feature with an expected geometry type
 * @param {string} type expected GeoJSON type
 * @param {string} name name of calling function
 * @throws {Error} error if value is not the expected type.
 */
function featureOf(feature$$1, type, name) {
    if (!feature$$1) throw new Error('No feature passed');
    if (!name) throw new Error('.featureOf() requires a name');
    if (!feature$$1 || feature$$1.type !== 'Feature' || !feature$$1.geometry) {
        throw new Error('Invalid input to ' + name + ', Feature with geometry required');
    }
    if (!feature$$1.geometry || feature$$1.geometry.type !== type) {
        throw new Error('Invalid input to ' + name + ': must be a ' + type + ', given ' + feature$$1.geometry.type);
    }
}

/**
 * Enforce expectations about types of {@link FeatureCollection} inputs for Turf.
 * Internally this uses {@link geojsonType} to judge geometry types.
 *
 * @name collectionOf
 * @param {FeatureCollection} featureCollection a FeatureCollection for which features will be judged
 * @param {string} type expected GeoJSON type
 * @param {string} name name of calling function
 * @throws {Error} if value is not the expected type.
 */
function collectionOf(featureCollection$$1, type, name) {
    if (!featureCollection$$1) throw new Error('No featureCollection passed');
    if (!name) throw new Error('.collectionOf() requires a name');
    if (!featureCollection$$1 || featureCollection$$1.type !== 'FeatureCollection') {
        throw new Error('Invalid input to ' + name + ', FeatureCollection required');
    }
    for (var i = 0; i < featureCollection$$1.features.length; i++) {
        var feature$$1 = featureCollection$$1.features[i];
        if (!feature$$1 || feature$$1.type !== 'Feature' || !feature$$1.geometry) {
            throw new Error('Invalid input to ' + name + ', Feature with geometry required');
        }
        if (!feature$$1.geometry || feature$$1.geometry.type !== type) {
            throw new Error('Invalid input to ' + name + ': must be a ' + type + ', given ' + feature$$1.geometry.type);
        }
    }
}

/**
 * Get Geometry from Feature or Geometry Object
 *
 * @param {Feature|Geometry} geojson GeoJSON Feature or Geometry Object
 * @returns {Geometry|null} GeoJSON Geometry Object
 * @throws {Error} if geojson is not a Feature or Geometry Object
 * @example
 * var point = {
 *   "type": "Feature",
 *   "properties": {},
 *   "geometry": {
 *     "type": "Point",
 *     "coordinates": [110, 40]
 *   }
 * }
 * var geom = turf.getGeom(point)
 * //={"type": "Point", "coordinates": [110, 40]}
 */
function getGeom(geojson) {
    if (!geojson) throw new Error('geojson is required');
    if (geojson.geometry !== undefined) return geojson.geometry;
    if (geojson.coordinates || geojson.geometries) return geojson;
    throw new Error('geojson must be a valid Feature or Geometry Object');
}

/**
 * Get Geometry Type from Feature or Geometry Object
 *
 * @throws {Error} **DEPRECATED** in v5.0.0 in favor of getType
 */
function getGeomType() {
    throw new Error('invariant.getGeomType has been deprecated in v5.0 in favor of invariant.getType');
}

/**
 * Get GeoJSON object's type, Geometry type is prioritize.
 *
 * @param {GeoJSON} geojson GeoJSON object
 * @param {string} [name] name of the variable to display in error message
 * @returns {string} GeoJSON type
 * @example
 * var point = {
 *   "type": "Feature",
 *   "properties": {},
 *   "geometry": {
 *     "type": "Point",
 *     "coordinates": [110, 40]
 *   }
 * }
 * var geom = turf.getType(point)
 * //="Point"
 */
function getType(geojson, name) {
    if (!geojson) throw new Error((name || 'geojson') + ' is required');
    // GeoJSON Feature & GeometryCollection
    if (geojson.geometry && geojson.geometry.type) return geojson.geometry.type;
    // GeoJSON Geometry & FeatureCollection
    if (geojson.type) return geojson.type;
    throw new Error((name || 'geojson') + ' is invalid');
}




var main_es$2 = Object.freeze({
	getCoord: getCoord,
	getCoords: getCoords,
	containsNumber: containsNumber,
	geojsonType: geojsonType,
	featureOf: featureOf,
	collectionOf: collectionOf,
	getGeom: getGeom,
	getGeomType: getGeomType,
	getType: getType
});

/**
 * @license GNU Affero General Public License.
 * Copyright (c) 2015, 2015 Ronny Lorenz <ronny@tbi.univie.ac.at>
 * v. 1.2.0
 * https://github.com/RaumZeit/MarchingSquares.js
 */

/**
 * Compute the isocontour(s) of a scalar 2D field given
 * a certain threshold by applying the Marching Squares
 * Algorithm. The function returns a list of path coordinates
 */
var defaultSettings = {
    successCallback: null,
    verbose: false
};

var settings = {};

function isoContours(data, threshold, options) {
    /* process options */
    options = options ? options : {};

    var optionKeys = Object.keys(defaultSettings);

    for (var i = 0; i < optionKeys.length; i++) {
        var key = optionKeys[i];
        var val = options[key];
        val = ((typeof val !== 'undefined') && (val !== null)) ? val : defaultSettings[key];

        settings[key] = val;
    }

    if (settings.verbose)
        console.log('MarchingSquaresJS-isoContours: computing isocontour for ' + threshold);

    var ret = contourGrid2Paths(computeContourGrid(data, threshold));

    if (typeof settings.successCallback === 'function')
        settings.successCallback(ret);

    return ret;
}

/*
  Thats all for the public interface, below follows the actual
  implementation
*/

/*
################################
Isocontour implementation below
################################
*/

/* assume that x1 == 1 &&  x0 == 0 */
function interpolateX(y, y0, y1) {
    return (y - y0) / (y1 - y0);
}

/* compute the isocontour 4-bit grid */
function computeContourGrid(data, threshold) {
    var rows = data.length - 1;
    var cols = data[0].length - 1;
    var ContourGrid = { rows: rows, cols: cols, cells: [] };

    for (var j = 0; j < rows; ++j) {
        ContourGrid.cells[j] = [];
        for (var i = 0; i < cols; ++i) {
            /* compose the 4-bit corner representation */
            var cval = 0;

            var tl = data[j + 1][i];
            var tr = data[j + 1][i + 1];
            var br = data[j][i + 1];
            var bl = data[j][i];

            if (isNaN(tl) || isNaN(tr) || isNaN(br) || isNaN(bl)) {
                continue;
            }
            cval |= ((tl >= threshold) ? 8 : 0);
            cval |= ((tr >= threshold) ? 4 : 0);
            cval |= ((br >= threshold) ? 2 : 0);
            cval |= ((bl >= threshold) ? 1 : 0);

            /* resolve ambiguity for cval == 5 || 10 via averaging */
            var flipped = false;
            if (cval === 5 || cval === 10) {
                var average = (tl + tr + br + bl) / 4;
                if (cval === 5 && (average < threshold)) {
                    cval = 10;
                    flipped = true;
                } else if (cval === 10 && (average < threshold)) {
                    cval = 5;
                    flipped = true;
                }
            }

            /* add cell to ContourGrid if it contains edges */
            if (cval !== 0 && cval !== 15) {
                var top, bottom, left, right;
                top = bottom = left = right = 0.5;
                /* interpolate edges of cell */
                if (cval === 1) {
                    left    = 1 - interpolateX(threshold, tl, bl);
                    bottom  = 1 - interpolateX(threshold, br, bl);
                } else if (cval === 2) {
                    bottom  = interpolateX(threshold, bl, br);
                    right   = 1 - interpolateX(threshold, tr, br);
                } else if (cval === 3) {
                    left    = 1 - interpolateX(threshold, tl, bl);
                    right   = 1 - interpolateX(threshold, tr, br);
                } else if (cval === 4) {
                    top     = interpolateX(threshold, tl, tr);
                    right   = interpolateX(threshold, br, tr);
                } else if (cval === 5) {
                    top     = interpolateX(threshold, tl, tr);
                    right   = interpolateX(threshold, br, tr);
                    bottom  = 1 - interpolateX(threshold, br, bl);
                    left    = 1 - interpolateX(threshold, tl, bl);
                } else if (cval === 6) {
                    bottom  = interpolateX(threshold, bl, br);
                    top     = interpolateX(threshold, tl, tr);
                } else if (cval === 7) {
                    left    = 1 - interpolateX(threshold, tl, bl);
                    top     = interpolateX(threshold, tl, tr);
                } else if (cval === 8) {
                    left    = interpolateX(threshold, bl, tl);
                    top     = 1 - interpolateX(threshold, tr, tl);
                } else if (cval === 9) {
                    bottom  = 1 - interpolateX(threshold, br, bl);
                    top     = 1 - interpolateX(threshold, tr, tl);
                } else if (cval === 10) {
                    top     = 1 - interpolateX(threshold, tr, tl);
                    right   = 1 - interpolateX(threshold, tr, br);
                    bottom  = interpolateX(threshold, bl, br);
                    left    = interpolateX(threshold, bl, tl);
                } else if (cval === 11) {
                    top     = 1 - interpolateX(threshold, tr, tl);
                    right   = 1 - interpolateX(threshold, tr, br);
                } else if (cval === 12) {
                    left    = interpolateX(threshold, bl, tl);
                    right   = interpolateX(threshold, br, tr);
                } else if (cval === 13) {
                    bottom  = 1 - interpolateX(threshold, br, bl);
                    right   = interpolateX(threshold, br, tr);
                } else if (cval === 14) {
                    left    = interpolateX(threshold, bl, tl);
                    bottom  = interpolateX(threshold, bl, br);
                } else {
                    console.log('MarchingSquaresJS-isoContours: Illegal cval detected: ' + cval);
                }
                ContourGrid.cells[j][i] = {
                    cval: cval,
                    flipped: flipped,
                    top: top,
                    right: right,
                    bottom: bottom,
                    left: left
                };
            }

        }
    }

    return ContourGrid;
}

function isSaddle(cell) {
    return cell.cval === 5 || cell.cval === 10;
}

function isTrivial(cell) {
    return cell.cval === 0 || cell.cval === 15;
}

function clearCell(cell) {
    if ((!isTrivial(cell)) && (cell.cval !== 5) && (cell.cval !== 10)) {
        cell.cval = 15;
    }
}

function getXY(cell, edge) {
    if (edge === 'top') {
        return [cell.top, 1.0];
    } else if (edge === 'bottom') {
        return [cell.bottom, 0.0];
    } else if (edge === 'right') {
        return [1.0, cell.right];
    } else if (edge === 'left') {
        return [0.0, cell.left];
    }
}

function contourGrid2Paths(grid) {
    var paths = [];
    var path_idx = 0;
    var rows = grid.rows;
    var cols = grid.cols;
    var epsilon = 1e-7;

    grid.cells.forEach(function (g, j) {
        g.forEach(function (gg, i) {
            if ((typeof gg !== 'undefined') && (!isSaddle(gg)) && (!isTrivial(gg))) {
                var p = tracePath(grid.cells, j, i);
                var merged = false;
                /* we may try to merge paths at this point */
                if (p.info === 'mergeable') {
                    /*
            search backwards through the path array to find an entry
            that starts with where the current path ends...
          */
                    var x = p.path[p.path.length - 1][0],
                        y = p.path[p.path.length - 1][1];

                    for (var k = path_idx - 1; k >= 0; k--) {
                        if ((Math.abs(paths[k][0][0] - x) <= epsilon) && (Math.abs(paths[k][0][1] - y) <= epsilon)) {
                            for (var l = p.path.length - 2; l >= 0; --l) {
                                paths[k].unshift(p.path[l]);
                            }
                            merged = true;
                            break;
                        }
                    }
                }
                if (!merged)
                    paths[path_idx++] = p.path;
            }
        });
    });

    return paths;
}

/*
  construct consecutive line segments from starting cell by
  walking arround the enclosed area clock-wise
  */
function tracePath(grid, j, i) {
    var maxj = grid.length;
    var p = [];
    var dxContour = [0, 0, 1, 1, 0, 0, 0, 0, -1, 0, 1, 1, -1, 0, -1, 0];
    var dyContour = [0, -1, 0, 0, 1, 1, 1, 1, 0, -1, 0, 0, 0, -1, 0, 0];
    var dx, dy;
    var startEdge = ['none', 'left', 'bottom', 'left', 'right', 'none', 'bottom', 'left', 'top', 'top', 'none', 'top', 'right', 'right', 'bottom', 'none'];
    var nextEdge  = ['none', 'bottom', 'right', 'right', 'top', 'top', 'top', 'top', 'left', 'bottom', 'right', 'right', 'left', 'bottom', 'left', 'none'];
    var edge;

    var startCell   = grid[j][i];
    var currentCell = grid[j][i];

    var cval = currentCell.cval;
    var edge = startEdge[cval];

    var pt = getXY(currentCell, edge);

    /* push initial segment */
    p.push([i + pt[0], j + pt[1]]);
    edge = nextEdge[cval];
    pt = getXY(currentCell, edge);
    p.push([i + pt[0], j + pt[1]]);
    clearCell(currentCell);

    /* now walk arround the enclosed area in clockwise-direction */
    var k = i + dxContour[cval];
    var l = j + dyContour[cval];
    var prev_cval = cval;

    while ((k >= 0) && (l >= 0) && (l < maxj) && ((k != i) || (l != j))) {
        currentCell = grid[l][k];
        if (typeof currentCell === 'undefined') { /* path ends here */
            //console.log(k + " " + l + " is undefined, stopping path!");
            break;
        }
        cval = currentCell.cval;
        if ((cval === 0) || (cval === 15)) {
            return { path: p, info: 'mergeable' };
        }
        edge  = nextEdge[cval];
        dx    = dxContour[cval];
        dy    = dyContour[cval];
        if ((cval === 5) || (cval === 10)) {
            /* select upper or lower band, depending on previous cells cval */
            if (cval === 5) {
                if (currentCell.flipped) { /* this is actually a flipped case 10 */
                    if (dyContour[prev_cval] === -1) {
                        edge  = 'left';
                        dx    = -1;
                        dy    = 0;
                    } else {
                        edge  = 'right';
                        dx    = 1;
                        dy    = 0;
                    }
                } else { /* real case 5 */
                    if (dxContour[prev_cval] === -1) {
                        edge  = 'bottom';
                        dx    = 0;
                        dy    = -1;
                    }
                }
            } else if (cval === 10) {
                if (currentCell.flipped) { /* this is actually a flipped case 5 */
                    if (dxContour[prev_cval] === -1) {
                        edge  = 'top';
                        dx    = 0;
                        dy    = 1;
                    } else {
                        edge  = 'bottom';
                        dx    = 0;
                        dy    = -1;
                    }
                } else {  /* real case 10 */
                    if (dyContour[prev_cval] === 1) {
                        edge  = 'left';
                        dx    = -1;
                        dy    = 0;
                    }
                }
            }
        }
        pt = getXY(currentCell, edge);
        p.push([k + pt[0], l + pt[1]]);
        clearCell(currentCell);
        k += dx;
        l += dy;
        prev_cval = cval;
    }

    return { path: p, info: 'closed' };
}

/**
 * Takes a {@link Point} grid and returns a correspondent matrix {Array<Array<number>>}
 * of the 'property' values
 *
 * @name gridToMatrix
 * @param {FeatureCollection<Point>} grid of points
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.zProperty='elevation'] the property name in `points` from which z-values will be pulled
 * @param {boolean} [options.flip=false] returns the matrix upside-down
 * @param {boolean} [options.flags=false] flags, adding a `matrixPosition` array field ([row, column]) to its properties,
 * the grid points with coordinates on the matrix
 * @returns {Array<Array<number>>} matrix of property values
 * @example
 *   var extent = [-70.823364, -33.553984, -70.473175, -33.302986];
 *   var cellSize = 3;
 *   var grid = turf.pointGrid(extent, cellSize);
 *   // add a random property to each point between 0 and 60
 *   for (var i = 0; i < grid.features.length; i++) {
 *     grid.features[i].properties.elevation = (Math.random() * 60);
 *   }
 *   gridToMatrix(grid);
 *   //= [
 *     [ 1, 13, 10,  9, 10, 13, 18],
 *     [34,  8,  5,  4,  5,  8, 13],
 *     [10,  5,  2,  1,  2,  5,  4],
 *     [ 0,  4, 56, 19,  1,  4,  9],
 *     [10,  5,  2,  1,  2,  5, 10],
 *     [57,  8,  5,  4,  5,  0, 57],
 *     [ 3, 13, 10,  9,  5, 13, 18],
 *     [18, 13, 10,  9, 78, 13, 18]
 *   ]
 */
function gridToMatrix(grid, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var zProperty = options.zProperty || 'elevation';
    var flip = options.flip;
    var flags = options.flags;

    // validation
    collectionOf(grid, 'Point', 'input must contain Points');

    var pointsMatrix = sortPointsByLatLng(grid, flip);

    var matrix = [];
    // create property matrix from sorted points
    // looping order matters here
    for (var r = 0; r < pointsMatrix.length; r++) {
        var pointRow = pointsMatrix[r];
        var row = [];
        for (var c = 0; c < pointRow.length; c++) {
            var point$$1 = pointRow[c];
            // Check if zProperty exist
            if (point$$1.properties[zProperty]) row.push(point$$1.properties[zProperty]);
            else row.push(0);
            // add flags
            if (flags === true) point$$1.properties.matrixPosition = [r, c];
        }
        matrix.push(row);
    }

    return matrix;
}

/**
 * Sorts points by latitude and longitude, creating a 2-dimensional array of points
 *
 * @private
 * @param {FeatureCollection<Point>} points GeoJSON Point features
 * @param {boolean} [flip=false] returns the matrix upside-down
 * @returns {Array<Array<Point>>} points ordered by latitude and longitude
 */
function sortPointsByLatLng(points$$1, flip) {
    var pointsByLatitude = {};

    // divide points by rows with the same latitude
    featureEach(points$$1, function (point$$1) {
        var lat = getCoords(point$$1)[1];
        if (!pointsByLatitude[lat]) pointsByLatitude[lat] = [];
        pointsByLatitude[lat].push(point$$1);
    });

    // sort points (with the same latitude) by longitude
    var orderedRowsByLatitude = Object.keys(pointsByLatitude).map(function (lat) {
        var row = pointsByLatitude[lat];
        var rowOrderedByLongitude = row.sort(function (a, b) {
            return getCoords(a)[0] - getCoords(b)[0];
        });
        return rowOrderedByLongitude;
    });

    // sort rows (of points with the same latitude) by latitude
    var pointMatrix = orderedRowsByLatitude.sort(function (a, b) {
        if (flip) return getCoords(a[0])[1] - getCoords(b[0])[1];
        else return getCoords(b[0])[1] - getCoords(a[0])[1];
    });

    return pointMatrix;
}

/**
 * Takes a grid {@link FeatureCollection} of {@link Point} features with z-values and an array of
 * value breaks and generates [isolines](http://en.wikipedia.org/wiki/Isoline).
 *
 * @name isolines
 * @param {FeatureCollection<Point>} pointGrid input points
 * @param {Array<number>} breaks values of `zProperty` where to draw isolines
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.zProperty='elevation'] the property name in `points` from which z-values will be pulled
 * @param {Object} [options.commonProperties={}] GeoJSON properties passed to ALL isolines
 * @param {Array<Object>} [options.breaksProperties=[]] GeoJSON properties passed, in order, to the correspondent isoline;
 * the breaks array will define the order in which the isolines are created
 * @returns {FeatureCollection<MultiLineString>} a FeatureCollection of {@link MultiLineString} features representing isolines
 * @example
 * // create a grid of points with random z-values in their properties
 * var extent = [0, 30, 20, 50];
 * var cellWidth = 100;
 * var pointGrid = turf.pointGrid(extent, cellWidth, {units: 'miles'});
 *
 * for (var i = 0; i < pointGrid.features.length; i++) {
 *     pointGrid.features[i].properties.temperature = Math.random() * 10;
 * }
 * var breaks = [0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10];
 *
 * var lines = turf.isolines(pointGrid, breaks, {zProperty: 'temperature'});
 *
 * //addToMap
 * var addToMap = [lines];
 */
function isolines(pointGrid, breaks, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var zProperty = options.zProperty || 'elevation';
    var commonProperties = options.commonProperties || {};
    var breaksProperties = options.breaksProperties || [];

    // Input validation
    collectionOf(pointGrid, 'Point', 'Input must contain Points');
    if (!breaks) throw new Error('breaks is required');
    if (!Array.isArray(breaks)) throw new Error('breaks must be an Array');
    if (!isObject(commonProperties)) throw new Error('commonProperties must be an Object');
    if (!Array.isArray(breaksProperties)) throw new Error('breaksProperties must be an Array');

    // Isoline methods
    var matrix = gridToMatrix(pointGrid, {zProperty: zProperty, flip: true});
    var createdIsoLines = createIsoLines(matrix, breaks, zProperty, commonProperties, breaksProperties);
    var scaledIsolines = rescaleIsolines(createdIsoLines, matrix, pointGrid);

    return featureCollection(scaledIsolines);
}

/**
 * Creates the isolines lines (featuresCollection of MultiLineString features) from the 2D data grid
 *
 * Marchingsquares process the grid data as a 3D representation of a function on a 2D plane, therefore it
 * assumes the points (x-y coordinates) are one 'unit' distance. The result of the isolines function needs to be
 * rescaled, with turfjs, to the original area and proportions on the map
 *
 * @private
 * @param {Array<Array<number>>} matrix Grid Data
 * @param {Array<number>} breaks Breaks
 * @param {string} zProperty name of the z-values property
 * @param {Object} [commonProperties={}] GeoJSON properties passed to ALL isolines
 * @param {Object} [breaksProperties=[]] GeoJSON properties passed to the correspondent isoline
 * @returns {Array<MultiLineString>} isolines
 */
function createIsoLines(matrix, breaks, zProperty, commonProperties, breaksProperties) {
    var results = [];
    for (var i = 1; i < breaks.length; i++) {
        var threshold = +breaks[i]; // make sure it's a number

        var properties = Object.assign(
            {},
            commonProperties,
            breaksProperties[i]
        );
        properties[zProperty] = threshold;
        var isoline = multiLineString(isoContours(matrix, threshold), properties);

        results.push(isoline);
    }
    return results;
}

/**
 * Translates and scales isolines
 *
 * @private
 * @param {Array<MultiLineString>} createdIsoLines to be rescaled
 * @param {Array<Array<number>>} matrix Grid Data
 * @param {Object} points Points by Latitude
 * @returns {Array<MultiLineString>} isolines
 */
function rescaleIsolines(createdIsoLines, matrix, points$$1) {

    // get dimensions (on the map) of the original grid
    var gridBbox = bbox(points$$1); // [ minX, minY, maxX, maxY ]
    var originalWidth = gridBbox[2] - gridBbox[0];
    var originalHeigth = gridBbox[3] - gridBbox[1];

    // get origin, which is the first point of the last row on the rectangular data on the map
    var x0 = gridBbox[0];
    var y0 = gridBbox[1];

    // get number of cells per side
    var matrixWidth = matrix[0].length - 1;
    var matrixHeight = matrix.length - 1;

    // calculate the scaling factor between matrix and rectangular grid on the map
    var scaleX = originalWidth / matrixWidth;
    var scaleY = originalHeigth / matrixHeight;

    var resize = function (point$$1) {
        point$$1[0] = point$$1[0] * scaleX + x0;
        point$$1[1] = point$$1[1] * scaleY + y0;
    };

    // resize and shift each point/line of the createdIsoLines
    createdIsoLines.forEach(function (isoline) {
        coordEach(isoline, resize);
    });
    return createdIsoLines;
}

var quickselect = partialSort;

// Floyd-Rivest selection algorithm:
// Rearrange items so that all items in the [left, k] range are smaller than all items in (k, right];
// The k-th element will have the (k - left + 1)th smallest value in [left, right]

function partialSort(arr, k, left, right, compare) {
    left = left || 0;
    right = right || (arr.length - 1);
    compare = compare || defaultCompare;

    while (right > left) {
        if (right - left > 600) {
            var n = right - left + 1;
            var m = k - left + 1;
            var z = Math.log(n);
            var s = 0.5 * Math.exp(2 * z / 3);
            var sd = 0.5 * Math.sqrt(z * s * (n - s) / n) * (m - n / 2 < 0 ? -1 : 1);
            var newLeft = Math.max(left, Math.floor(k - m * s / n + sd));
            var newRight = Math.min(right, Math.floor(k + (n - m) * s / n + sd));
            partialSort(arr, k, newLeft, newRight, compare);
        }

        var t = arr[k];
        var i = left;
        var j = right;

        swap(arr, left, k);
        if (compare(arr[right], t) > 0) swap(arr, left, right);

        while (i < j) {
            swap(arr, i, j);
            i++;
            j--;
            while (compare(arr[i], t) < 0) i++;
            while (compare(arr[j], t) > 0) j--;
        }

        if (compare(arr[left], t) === 0) swap(arr, left, j);
        else {
            j++;
            swap(arr, j, right);
        }

        if (j <= k) left = j + 1;
        if (k <= j) right = j - 1;
    }
}

function swap(arr, i, j) {
    var tmp = arr[i];
    arr[i] = arr[j];
    arr[j] = tmp;
}

function defaultCompare(a, b) {
    return a < b ? -1 : a > b ? 1 : 0;
}

var rbush_1 = rbush;



function rbush(maxEntries, format) {
    if (!(this instanceof rbush)) return new rbush(maxEntries, format);

    // max entries in a node is 9 by default; min node fill is 40% for best performance
    this._maxEntries = Math.max(4, maxEntries || 9);
    this._minEntries = Math.max(2, Math.ceil(this._maxEntries * 0.4));

    if (format) {
        this._initFormat(format);
    }

    this.clear();
}

rbush.prototype = {

    all: function () {
        return this._all(this.data, []);
    },

    search: function (bbox) {

        var node = this.data,
            result = [],
            toBBox = this.toBBox;

        if (!intersects$1(bbox, node)) return result;

        var nodesToSearch = [],
            i, len, child, childBBox;

        while (node) {
            for (i = 0, len = node.children.length; i < len; i++) {

                child = node.children[i];
                childBBox = node.leaf ? toBBox(child) : child;

                if (intersects$1(bbox, childBBox)) {
                    if (node.leaf) result.push(child);
                    else if (contains(bbox, childBBox)) this._all(child, result);
                    else nodesToSearch.push(child);
                }
            }
            node = nodesToSearch.pop();
        }

        return result;
    },

    collides: function (bbox) {

        var node = this.data,
            toBBox = this.toBBox;

        if (!intersects$1(bbox, node)) return false;

        var nodesToSearch = [],
            i, len, child, childBBox;

        while (node) {
            for (i = 0, len = node.children.length; i < len; i++) {

                child = node.children[i];
                childBBox = node.leaf ? toBBox(child) : child;

                if (intersects$1(bbox, childBBox)) {
                    if (node.leaf || contains(bbox, childBBox)) return true;
                    nodesToSearch.push(child);
                }
            }
            node = nodesToSearch.pop();
        }

        return false;
    },

    load: function (data) {
        if (!(data && data.length)) return this;

        if (data.length < this._minEntries) {
            for (var i = 0, len = data.length; i < len; i++) {
                this.insert(data[i]);
            }
            return this;
        }

        // recursively build the tree with the given data from stratch using OMT algorithm
        var node = this._build(data.slice(), 0, data.length - 1, 0);

        if (!this.data.children.length) {
            // save as is if tree is empty
            this.data = node;

        } else if (this.data.height === node.height) {
            // split root if trees have the same height
            this._splitRoot(this.data, node);

        } else {
            if (this.data.height < node.height) {
                // swap trees if inserted one is bigger
                var tmpNode = this.data;
                this.data = node;
                node = tmpNode;
            }

            // insert the small tree into the large tree at appropriate level
            this._insert(node, this.data.height - node.height - 1, true);
        }

        return this;
    },

    insert: function (item) {
        if (item) this._insert(item, this.data.height - 1);
        return this;
    },

    clear: function () {
        this.data = createNode([]);
        return this;
    },

    remove: function (item, equalsFn) {
        if (!item) return this;

        var node = this.data,
            bbox = this.toBBox(item),
            path = [],
            indexes = [],
            i, parent, index, goingUp;

        // depth-first iterative tree traversal
        while (node || path.length) {

            if (!node) { // go up
                node = path.pop();
                parent = path[path.length - 1];
                i = indexes.pop();
                goingUp = true;
            }

            if (node.leaf) { // check current node
                index = findItem(item, node.children, equalsFn);

                if (index !== -1) {
                    // item found, remove the item and condense tree upwards
                    node.children.splice(index, 1);
                    path.push(node);
                    this._condense(path);
                    return this;
                }
            }

            if (!goingUp && !node.leaf && contains(node, bbox)) { // go down
                path.push(node);
                indexes.push(i);
                i = 0;
                parent = node;
                node = node.children[0];

            } else if (parent) { // go right
                i++;
                node = parent.children[i];
                goingUp = false;

            } else node = null; // nothing found
        }

        return this;
    },

    toBBox: function (item) { return item; },

    compareMinX: compareNodeMinX,
    compareMinY: compareNodeMinY,

    toJSON: function () { return this.data; },

    fromJSON: function (data) {
        this.data = data;
        return this;
    },

    _all: function (node, result) {
        var nodesToSearch = [];
        while (node) {
            if (node.leaf) result.push.apply(result, node.children);
            else nodesToSearch.push.apply(nodesToSearch, node.children);

            node = nodesToSearch.pop();
        }
        return result;
    },

    _build: function (items, left, right, height) {

        var N = right - left + 1,
            M = this._maxEntries,
            node;

        if (N <= M) {
            // reached leaf level; return leaf
            node = createNode(items.slice(left, right + 1));
            calcBBox(node, this.toBBox);
            return node;
        }

        if (!height) {
            // target height of the bulk-loaded tree
            height = Math.ceil(Math.log(N) / Math.log(M));

            // target number of root entries to maximize storage utilization
            M = Math.ceil(N / Math.pow(M, height - 1));
        }

        node = createNode([]);
        node.leaf = false;
        node.height = height;

        // split the items into M mostly square tiles

        var N2 = Math.ceil(N / M),
            N1 = N2 * Math.ceil(Math.sqrt(M)),
            i, j, right2, right3;

        multiSelect(items, left, right, N1, this.compareMinX);

        for (i = left; i <= right; i += N1) {

            right2 = Math.min(i + N1 - 1, right);

            multiSelect(items, i, right2, N2, this.compareMinY);

            for (j = i; j <= right2; j += N2) {

                right3 = Math.min(j + N2 - 1, right2);

                // pack each entry recursively
                node.children.push(this._build(items, j, right3, height - 1));
            }
        }

        calcBBox(node, this.toBBox);

        return node;
    },

    _chooseSubtree: function (bbox, node, level, path) {

        var i, len, child, targetNode, area, enlargement, minArea, minEnlargement;

        while (true) {
            path.push(node);

            if (node.leaf || path.length - 1 === level) break;

            minArea = minEnlargement = Infinity;

            for (i = 0, len = node.children.length; i < len; i++) {
                child = node.children[i];
                area = bboxArea(child);
                enlargement = enlargedArea(bbox, child) - area;

                // choose entry with the least area enlargement
                if (enlargement < minEnlargement) {
                    minEnlargement = enlargement;
                    minArea = area < minArea ? area : minArea;
                    targetNode = child;

                } else if (enlargement === minEnlargement) {
                    // otherwise choose one with the smallest area
                    if (area < minArea) {
                        minArea = area;
                        targetNode = child;
                    }
                }
            }

            node = targetNode || node.children[0];
        }

        return node;
    },

    _insert: function (item, level, isNode) {

        var toBBox = this.toBBox,
            bbox = isNode ? item : toBBox(item),
            insertPath = [];

        // find the best node for accommodating the item, saving all nodes along the path too
        var node = this._chooseSubtree(bbox, this.data, level, insertPath);

        // put the item into the node
        node.children.push(item);
        extend(node, bbox);

        // split on node overflow; propagate upwards if necessary
        while (level >= 0) {
            if (insertPath[level].children.length > this._maxEntries) {
                this._split(insertPath, level);
                level--;
            } else break;
        }

        // adjust bboxes along the insertion path
        this._adjustParentBBoxes(bbox, insertPath, level);
    },

    // split overflowed node into two
    _split: function (insertPath, level) {

        var node = insertPath[level],
            M = node.children.length,
            m = this._minEntries;

        this._chooseSplitAxis(node, m, M);

        var splitIndex = this._chooseSplitIndex(node, m, M);

        var newNode = createNode(node.children.splice(splitIndex, node.children.length - splitIndex));
        newNode.height = node.height;
        newNode.leaf = node.leaf;

        calcBBox(node, this.toBBox);
        calcBBox(newNode, this.toBBox);

        if (level) insertPath[level - 1].children.push(newNode);
        else this._splitRoot(node, newNode);
    },

    _splitRoot: function (node, newNode) {
        // split root node
        this.data = createNode([node, newNode]);
        this.data.height = node.height + 1;
        this.data.leaf = false;
        calcBBox(this.data, this.toBBox);
    },

    _chooseSplitIndex: function (node, m, M) {

        var i, bbox1, bbox2, overlap, area, minOverlap, minArea, index;

        minOverlap = minArea = Infinity;

        for (i = m; i <= M - m; i++) {
            bbox1 = distBBox(node, 0, i, this.toBBox);
            bbox2 = distBBox(node, i, M, this.toBBox);

            overlap = intersectionArea(bbox1, bbox2);
            area = bboxArea(bbox1) + bboxArea(bbox2);

            // choose distribution with minimum overlap
            if (overlap < minOverlap) {
                minOverlap = overlap;
                index = i;

                minArea = area < minArea ? area : minArea;

            } else if (overlap === minOverlap) {
                // otherwise choose distribution with minimum area
                if (area < minArea) {
                    minArea = area;
                    index = i;
                }
            }
        }

        return index;
    },

    // sorts node children by the best axis for split
    _chooseSplitAxis: function (node, m, M) {

        var compareMinX = node.leaf ? this.compareMinX : compareNodeMinX,
            compareMinY = node.leaf ? this.compareMinY : compareNodeMinY,
            xMargin = this._allDistMargin(node, m, M, compareMinX),
            yMargin = this._allDistMargin(node, m, M, compareMinY);

        // if total distributions margin value is minimal for x, sort by minX,
        // otherwise it's already sorted by minY
        if (xMargin < yMargin) node.children.sort(compareMinX);
    },

    // total margin of all possible split distributions where each node is at least m full
    _allDistMargin: function (node, m, M, compare) {

        node.children.sort(compare);

        var toBBox = this.toBBox,
            leftBBox = distBBox(node, 0, m, toBBox),
            rightBBox = distBBox(node, M - m, M, toBBox),
            margin = bboxMargin(leftBBox) + bboxMargin(rightBBox),
            i, child;

        for (i = m; i < M - m; i++) {
            child = node.children[i];
            extend(leftBBox, node.leaf ? toBBox(child) : child);
            margin += bboxMargin(leftBBox);
        }

        for (i = M - m - 1; i >= m; i--) {
            child = node.children[i];
            extend(rightBBox, node.leaf ? toBBox(child) : child);
            margin += bboxMargin(rightBBox);
        }

        return margin;
    },

    _adjustParentBBoxes: function (bbox, path, level) {
        // adjust bboxes along the given tree path
        for (var i = level; i >= 0; i--) {
            extend(path[i], bbox);
        }
    },

    _condense: function (path) {
        // go through the path, removing empty nodes and updating bboxes
        for (var i = path.length - 1, siblings; i >= 0; i--) {
            if (path[i].children.length === 0) {
                if (i > 0) {
                    siblings = path[i - 1].children;
                    siblings.splice(siblings.indexOf(path[i]), 1);

                } else this.clear();

            } else calcBBox(path[i], this.toBBox);
        }
    },

    _initFormat: function (format) {
        // data format (minX, minY, maxX, maxY accessors)

        // uses eval-type function compilation instead of just accepting a toBBox function
        // because the algorithms are very sensitive to sorting functions performance,
        // so they should be dead simple and without inner calls

        var compareArr = ['return a', ' - b', ';'];

        this.compareMinX = new Function('a', 'b', compareArr.join(format[0]));
        this.compareMinY = new Function('a', 'b', compareArr.join(format[1]));

        this.toBBox = new Function('a',
            'return {minX: a' + format[0] +
            ', minY: a' + format[1] +
            ', maxX: a' + format[2] +
            ', maxY: a' + format[3] + '};');
    }
};

function findItem(item, items, equalsFn) {
    if (!equalsFn) return items.indexOf(item);

    for (var i = 0; i < items.length; i++) {
        if (equalsFn(item, items[i])) return i;
    }
    return -1;
}

// calculate node's bbox from bboxes of its children
function calcBBox(node, toBBox) {
    distBBox(node, 0, node.children.length, toBBox, node);
}

// min bounding rectangle of node children from k to p-1
function distBBox(node, k, p, toBBox, destNode) {
    if (!destNode) destNode = createNode(null);
    destNode.minX = Infinity;
    destNode.minY = Infinity;
    destNode.maxX = -Infinity;
    destNode.maxY = -Infinity;

    for (var i = k, child; i < p; i++) {
        child = node.children[i];
        extend(destNode, node.leaf ? toBBox(child) : child);
    }

    return destNode;
}

function extend(a, b) {
    a.minX = Math.min(a.minX, b.minX);
    a.minY = Math.min(a.minY, b.minY);
    a.maxX = Math.max(a.maxX, b.maxX);
    a.maxY = Math.max(a.maxY, b.maxY);
    return a;
}

function compareNodeMinX(a, b) { return a.minX - b.minX; }
function compareNodeMinY(a, b) { return a.minY - b.minY; }

function bboxArea(a)   { return (a.maxX - a.minX) * (a.maxY - a.minY); }
function bboxMargin(a) { return (a.maxX - a.minX) + (a.maxY - a.minY); }

function enlargedArea(a, b) {
    return (Math.max(b.maxX, a.maxX) - Math.min(b.minX, a.minX)) *
           (Math.max(b.maxY, a.maxY) - Math.min(b.minY, a.minY));
}

function intersectionArea(a, b) {
    var minX = Math.max(a.minX, b.minX),
        minY = Math.max(a.minY, b.minY),
        maxX = Math.min(a.maxX, b.maxX),
        maxY = Math.min(a.maxY, b.maxY);

    return Math.max(0, maxX - minX) *
           Math.max(0, maxY - minY);
}

function contains(a, b) {
    return a.minX <= b.minX &&
           a.minY <= b.minY &&
           b.maxX <= a.maxX &&
           b.maxY <= a.maxY;
}

function intersects$1(a, b) {
    return b.minX <= a.maxX &&
           b.minY <= a.maxY &&
           b.maxX >= a.minX &&
           b.maxY >= a.minY;
}

function createNode(children) {
    return {
        children: children,
        height: 1,
        leaf: true,
        minX: Infinity,
        minY: Infinity,
        maxX: -Infinity,
        maxY: -Infinity
    };
}

// sort an array so that items come in groups of n unsorted items, with groups sorted between each other;
// combines selection algorithm with binary divide & conquer approach

function multiSelect(arr, left, right, n, compare) {
    var stack = [left, right],
        mid;

    while (stack.length) {
        right = stack.pop();
        left = stack.pop();

        if (right - left <= n) continue;

        mid = left + Math.ceil((right - left) / n / 2) * n;
        quickselect(arr, mid, left, right, compare);

        stack.push(left, mid, mid, right);
    }
}

function createCommonjsModule(fn, module) {
	return module = { exports: {} }, fn(module, module.exports), module.exports;
}

var twoProduct_1 = twoProduct;

var SPLITTER = +(Math.pow(2, 27) + 1.0);

function twoProduct(a, b, result) {
  var x = a * b;

  var c = SPLITTER * a;
  var abig = c - a;
  var ahi = c - abig;
  var alo = a - ahi;

  var d = SPLITTER * b;
  var bbig = d - b;
  var bhi = d - bbig;
  var blo = b - bhi;

  var err1 = x - (ahi * bhi);
  var err2 = err1 - (alo * bhi);
  var err3 = err2 - (ahi * blo);

  var y = alo * blo - err3;

  if(result) {
    result[0] = y;
    result[1] = x;
    return result
  }

  return [ y, x ]
}

var robustSum = linearExpansionSum;

//Easy case: Add two scalars
function scalarScalar(a, b) {
  var x = a + b;
  var bv = x - a;
  var av = x - bv;
  var br = b - bv;
  var ar = a - av;
  var y = ar + br;
  if(y) {
    return [y, x]
  }
  return [x]
}

function linearExpansionSum(e, f) {
  var ne = e.length|0;
  var nf = f.length|0;
  if(ne === 1 && nf === 1) {
    return scalarScalar(e[0], f[0])
  }
  var n = ne + nf;
  var g = new Array(n);
  var count = 0;
  var eptr = 0;
  var fptr = 0;
  var abs = Math.abs;
  var ei = e[eptr];
  var ea = abs(ei);
  var fi = f[fptr];
  var fa = abs(fi);
  var a, b;
  if(ea < fa) {
    b = ei;
    eptr += 1;
    if(eptr < ne) {
      ei = e[eptr];
      ea = abs(ei);
    }
  } else {
    b = fi;
    fptr += 1;
    if(fptr < nf) {
      fi = f[fptr];
      fa = abs(fi);
    }
  }
  if((eptr < ne && ea < fa) || (fptr >= nf)) {
    a = ei;
    eptr += 1;
    if(eptr < ne) {
      ei = e[eptr];
      ea = abs(ei);
    }
  } else {
    a = fi;
    fptr += 1;
    if(fptr < nf) {
      fi = f[fptr];
      fa = abs(fi);
    }
  }
  var x = a + b;
  var bv = x - a;
  var y = b - bv;
  var q0 = y;
  var q1 = x;
  var _x, _bv, _av, _br, _ar;
  while(eptr < ne && fptr < nf) {
    if(ea < fa) {
      a = ei;
      eptr += 1;
      if(eptr < ne) {
        ei = e[eptr];
        ea = abs(ei);
      }
    } else {
      a = fi;
      fptr += 1;
      if(fptr < nf) {
        fi = f[fptr];
        fa = abs(fi);
      }
    }
    b = q0;
    x = a + b;
    bv = x - a;
    y = b - bv;
    if(y) {
      g[count++] = y;
    }
    _x = q1 + x;
    _bv = _x - q1;
    _av = _x - _bv;
    _br = x - _bv;
    _ar = q1 - _av;
    q0 = _ar + _br;
    q1 = _x;
  }
  while(eptr < ne) {
    a = ei;
    b = q0;
    x = a + b;
    bv = x - a;
    y = b - bv;
    if(y) {
      g[count++] = y;
    }
    _x = q1 + x;
    _bv = _x - q1;
    _av = _x - _bv;
    _br = x - _bv;
    _ar = q1 - _av;
    q0 = _ar + _br;
    q1 = _x;
    eptr += 1;
    if(eptr < ne) {
      ei = e[eptr];
    }
  }
  while(fptr < nf) {
    a = fi;
    b = q0;
    x = a + b;
    bv = x - a;
    y = b - bv;
    if(y) {
      g[count++] = y;
    } 
    _x = q1 + x;
    _bv = _x - q1;
    _av = _x - _bv;
    _br = x - _bv;
    _ar = q1 - _av;
    q0 = _ar + _br;
    q1 = _x;
    fptr += 1;
    if(fptr < nf) {
      fi = f[fptr];
    }
  }
  if(q0) {
    g[count++] = q0;
  }
  if(q1) {
    g[count++] = q1;
  }
  if(!count) {
    g[count++] = 0.0;  
  }
  g.length = count;
  return g
}

var twoSum = fastTwoSum;

function fastTwoSum(a, b, result) {
	var x = a + b;
	var bv = x - a;
	var av = x - bv;
	var br = b - bv;
	var ar = a - av;
	if(result) {
		result[0] = ar + br;
		result[1] = x;
		return result
	}
	return [ar+br, x]
}

var robustScale = scaleLinearExpansion;

function scaleLinearExpansion(e, scale) {
  var n = e.length;
  if(n === 1) {
    var ts = twoProduct_1(e[0], scale);
    if(ts[0]) {
      return ts
    }
    return [ ts[1] ]
  }
  var g = new Array(2 * n);
  var q = [0.1, 0.1];
  var t = [0.1, 0.1];
  var count = 0;
  twoProduct_1(e[0], scale, q);
  if(q[0]) {
    g[count++] = q[0];
  }
  for(var i=1; i<n; ++i) {
    twoProduct_1(e[i], scale, t);
    var pq = q[1];
    twoSum(pq, t[0], q);
    if(q[0]) {
      g[count++] = q[0];
    }
    var a = t[1];
    var b = q[1];
    var x = a + b;
    var bv = x - a;
    var y = b - bv;
    q[1] = x;
    if(y) {
      g[count++] = y;
    }
  }
  if(q[1]) {
    g[count++] = q[1];
  }
  if(count === 0) {
    g[count++] = 0.0;
  }
  g.length = count;
  return g
}

var robustDiff = robustSubtract;

//Easy case: Add two scalars
function scalarScalar$1(a, b) {
  var x = a + b;
  var bv = x - a;
  var av = x - bv;
  var br = b - bv;
  var ar = a - av;
  var y = ar + br;
  if(y) {
    return [y, x]
  }
  return [x]
}

function robustSubtract(e, f) {
  var ne = e.length|0;
  var nf = f.length|0;
  if(ne === 1 && nf === 1) {
    return scalarScalar$1(e[0], -f[0])
  }
  var n = ne + nf;
  var g = new Array(n);
  var count = 0;
  var eptr = 0;
  var fptr = 0;
  var abs = Math.abs;
  var ei = e[eptr];
  var ea = abs(ei);
  var fi = -f[fptr];
  var fa = abs(fi);
  var a, b;
  if(ea < fa) {
    b = ei;
    eptr += 1;
    if(eptr < ne) {
      ei = e[eptr];
      ea = abs(ei);
    }
  } else {
    b = fi;
    fptr += 1;
    if(fptr < nf) {
      fi = -f[fptr];
      fa = abs(fi);
    }
  }
  if((eptr < ne && ea < fa) || (fptr >= nf)) {
    a = ei;
    eptr += 1;
    if(eptr < ne) {
      ei = e[eptr];
      ea = abs(ei);
    }
  } else {
    a = fi;
    fptr += 1;
    if(fptr < nf) {
      fi = -f[fptr];
      fa = abs(fi);
    }
  }
  var x = a + b;
  var bv = x - a;
  var y = b - bv;
  var q0 = y;
  var q1 = x;
  var _x, _bv, _av, _br, _ar;
  while(eptr < ne && fptr < nf) {
    if(ea < fa) {
      a = ei;
      eptr += 1;
      if(eptr < ne) {
        ei = e[eptr];
        ea = abs(ei);
      }
    } else {
      a = fi;
      fptr += 1;
      if(fptr < nf) {
        fi = -f[fptr];
        fa = abs(fi);
      }
    }
    b = q0;
    x = a + b;
    bv = x - a;
    y = b - bv;
    if(y) {
      g[count++] = y;
    }
    _x = q1 + x;
    _bv = _x - q1;
    _av = _x - _bv;
    _br = x - _bv;
    _ar = q1 - _av;
    q0 = _ar + _br;
    q1 = _x;
  }
  while(eptr < ne) {
    a = ei;
    b = q0;
    x = a + b;
    bv = x - a;
    y = b - bv;
    if(y) {
      g[count++] = y;
    }
    _x = q1 + x;
    _bv = _x - q1;
    _av = _x - _bv;
    _br = x - _bv;
    _ar = q1 - _av;
    q0 = _ar + _br;
    q1 = _x;
    eptr += 1;
    if(eptr < ne) {
      ei = e[eptr];
    }
  }
  while(fptr < nf) {
    a = fi;
    b = q0;
    x = a + b;
    bv = x - a;
    y = b - bv;
    if(y) {
      g[count++] = y;
    } 
    _x = q1 + x;
    _bv = _x - q1;
    _av = _x - _bv;
    _br = x - _bv;
    _ar = q1 - _av;
    q0 = _ar + _br;
    q1 = _x;
    fptr += 1;
    if(fptr < nf) {
      fi = -f[fptr];
    }
  }
  if(q0) {
    g[count++] = q0;
  }
  if(q1) {
    g[count++] = q1;
  }
  if(!count) {
    g[count++] = 0.0;  
  }
  g.length = count;
  return g
}

var orientation_1 = createCommonjsModule(function (module) {
var NUM_EXPAND = 5;

var EPSILON     = 1.1102230246251565e-16;
var ERRBOUND3   = (3.0 + 16.0 * EPSILON) * EPSILON;
var ERRBOUND4   = (7.0 + 56.0 * EPSILON) * EPSILON;

function cofactor(m, c) {
  var result = new Array(m.length-1);
  for(var i=1; i<m.length; ++i) {
    var r = result[i-1] = new Array(m.length-1);
    for(var j=0,k=0; j<m.length; ++j) {
      if(j === c) {
        continue
      }
      r[k++] = m[i][j];
    }
  }
  return result
}

function matrix(n) {
  var result = new Array(n);
  for(var i=0; i<n; ++i) {
    result[i] = new Array(n);
    for(var j=0; j<n; ++j) {
      result[i][j] = ["m", j, "[", (n-i-1), "]"].join("");
    }
  }
  return result
}

function sign(n) {
  if(n & 1) {
    return "-"
  }
  return ""
}

function generateSum(expr) {
  if(expr.length === 1) {
    return expr[0]
  } else if(expr.length === 2) {
    return ["sum(", expr[0], ",", expr[1], ")"].join("")
  } else {
    var m = expr.length>>1;
    return ["sum(", generateSum(expr.slice(0, m)), ",", generateSum(expr.slice(m)), ")"].join("")
  }
}

function determinant(m) {
  if(m.length === 2) {
    return [["sum(prod(", m[0][0], ",", m[1][1], "),prod(-", m[0][1], ",", m[1][0], "))"].join("")]
  } else {
    var expr = [];
    for(var i=0; i<m.length; ++i) {
      expr.push(["scale(", generateSum(determinant(cofactor(m, i))), ",", sign(i), m[0][i], ")"].join(""));
    }
    return expr
  }
}

function orientation(n) {
  var pos = [];
  var neg = [];
  var m = matrix(n);
  var args = [];
  for(var i=0; i<n; ++i) {
    if((i&1)===0) {
      pos.push.apply(pos, determinant(cofactor(m, i)));
    } else {
      neg.push.apply(neg, determinant(cofactor(m, i)));
    }
    args.push("m" + i);
  }
  var posExpr = generateSum(pos);
  var negExpr = generateSum(neg);
  var funcName = "orientation" + n + "Exact";
  var code = ["function ", funcName, "(", args.join(), "){var p=", posExpr, ",n=", negExpr, ",d=sub(p,n);\
return d[d.length-1];};return ", funcName].join("");
  var proc = new Function("sum", "prod", "scale", "sub", code);
  return proc(robustSum, twoProduct_1, robustScale, robustDiff)
}

var orientation3Exact = orientation(3);
var orientation4Exact = orientation(4);

var CACHED = [
  function orientation0() { return 0 },
  function orientation1() { return 0 },
  function orientation2(a, b) { 
    return b[0] - a[0]
  },
  function orientation3(a, b, c) {
    var l = (a[1] - c[1]) * (b[0] - c[0]);
    var r = (a[0] - c[0]) * (b[1] - c[1]);
    var det = l - r;
    var s;
    if(l > 0) {
      if(r <= 0) {
        return det
      } else {
        s = l + r;
      }
    } else if(l < 0) {
      if(r >= 0) {
        return det
      } else {
        s = -(l + r);
      }
    } else {
      return det
    }
    var tol = ERRBOUND3 * s;
    if(det >= tol || det <= -tol) {
      return det
    }
    return orientation3Exact(a, b, c)
  },
  function orientation4(a,b,c,d) {
    var adx = a[0] - d[0];
    var bdx = b[0] - d[0];
    var cdx = c[0] - d[0];
    var ady = a[1] - d[1];
    var bdy = b[1] - d[1];
    var cdy = c[1] - d[1];
    var adz = a[2] - d[2];
    var bdz = b[2] - d[2];
    var cdz = c[2] - d[2];
    var bdxcdy = bdx * cdy;
    var cdxbdy = cdx * bdy;
    var cdxady = cdx * ady;
    var adxcdy = adx * cdy;
    var adxbdy = adx * bdy;
    var bdxady = bdx * ady;
    var det = adz * (bdxcdy - cdxbdy) 
            + bdz * (cdxady - adxcdy)
            + cdz * (adxbdy - bdxady);
    var permanent = (Math.abs(bdxcdy) + Math.abs(cdxbdy)) * Math.abs(adz)
                  + (Math.abs(cdxady) + Math.abs(adxcdy)) * Math.abs(bdz)
                  + (Math.abs(adxbdy) + Math.abs(bdxady)) * Math.abs(cdz);
    var tol = ERRBOUND4 * permanent;
    if ((det > tol) || (-det > tol)) {
      return det
    }
    return orientation4Exact(a,b,c,d)
  }
];

function slowOrient(args) {
  var proc = CACHED[args.length];
  if(!proc) {
    proc = CACHED[args.length] = orientation(args.length);
  }
  return proc.apply(undefined, args)
}

function generateOrientationProc() {
  while(CACHED.length <= NUM_EXPAND) {
    CACHED.push(orientation(CACHED.length));
  }
  var args = [];
  var procArgs = ["slow"];
  for(var i=0; i<=NUM_EXPAND; ++i) {
    args.push("a" + i);
    procArgs.push("o" + i);
  }
  var code = [
    "function getOrientation(", args.join(), "){switch(arguments.length){case 0:case 1:return 0;"
  ];
  for(var i=2; i<=NUM_EXPAND; ++i) {
    code.push("case ", i, ":return o", i, "(", args.slice(0, i).join(), ");");
  }
  code.push("}var s=new Array(arguments.length);for(var i=0;i<arguments.length;++i){s[i]=arguments[i]};return slow(s);}return getOrientation");
  procArgs.push(code.join(""));

  var proc = Function.apply(undefined, procArgs);
  module.exports = proc.apply(undefined, [slowOrient].concat(CACHED));
  for(var i=0; i<=NUM_EXPAND; ++i) {
    module.exports[i] = CACHED[i];
  }
}

generateOrientationProc();
});

var monotoneConvexHull2d = monotoneConvexHull2D;

var orient$1 = orientation_1[3];

function monotoneConvexHull2D(points) {
  var n = points.length;

  if(n < 3) {
    var result = new Array(n);
    for(var i=0; i<n; ++i) {
      result[i] = i;
    }

    if(n === 2 &&
       points[0][0] === points[1][0] &&
       points[0][1] === points[1][1]) {
      return [0]
    }

    return result
  }

  //Sort point indices along x-axis
  var sorted = new Array(n);
  for(var i=0; i<n; ++i) {
    sorted[i] = i;
  }
  sorted.sort(function(a,b) {
    var d = points[a][0]-points[b][0];
    if(d) {
      return d
    }
    return points[a][1] - points[b][1]
  });

  //Construct upper and lower hulls
  var lower = [sorted[0], sorted[1]];
  var upper = [sorted[0], sorted[1]];

  for(var i=2; i<n; ++i) {
    var idx = sorted[i];
    var p   = points[idx];

    //Insert into lower list
    var m = lower.length;
    while(m > 1 && orient$1(
        points[lower[m-2]], 
        points[lower[m-1]], 
        p) <= 0) {
      m -= 1;
      lower.pop();
    }
    lower.push(idx);

    //Insert into upper list
    m = upper.length;
    while(m > 1 && orient$1(
        points[upper[m-2]], 
        points[upper[m-1]], 
        p) >= 0) {
      m -= 1;
      upper.pop();
    }
    upper.push(idx);
  }

  //Merge lists together
  var result = new Array(upper.length + lower.length - 2);
  var ptr    = 0;
  for(var i=0, nl=lower.length; i<nl; ++i) {
    result[ptr++] = lower[i];
  }
  for(var j=upper.length-2; j>0; --j) {
    result[ptr++] = upper[j];
  }

  //Return result
  return result
}

var tinyqueue = TinyQueue;
var default_1$1 = TinyQueue;

function TinyQueue(data, compare) {
    if (!(this instanceof TinyQueue)) return new TinyQueue(data, compare);

    this.data = data || [];
    this.length = this.data.length;
    this.compare = compare || defaultCompare$1;

    if (this.length > 0) {
        for (var i = (this.length >> 1) - 1; i >= 0; i--) this._down(i);
    }
}

function defaultCompare$1(a, b) {
    return a < b ? -1 : a > b ? 1 : 0;
}

TinyQueue.prototype = {

    push: function (item) {
        this.data.push(item);
        this.length++;
        this._up(this.length - 1);
    },

    pop: function () {
        if (this.length === 0) return undefined;

        var top = this.data[0];
        this.length--;

        if (this.length > 0) {
            this.data[0] = this.data[this.length];
            this._down(0);
        }
        this.data.pop();

        return top;
    },

    peek: function () {
        return this.data[0];
    },

    _up: function (pos) {
        var data = this.data;
        var compare = this.compare;
        var item = data[pos];

        while (pos > 0) {
            var parent = (pos - 1) >> 1;
            var current = data[parent];
            if (compare(item, current) >= 0) break;
            data[pos] = current;
            pos = parent;
        }

        data[pos] = item;
    },

    _down: function (pos) {
        var data = this.data;
        var compare = this.compare;
        var halfLength = this.length >> 1;
        var item = data[pos];

        while (pos < halfLength) {
            var left = (pos << 1) + 1;
            var right = left + 1;
            var best = data[left];

            if (right < this.length && compare(data[right], best) < 0) {
                left = right;
                best = data[right];
            }
            if (compare(best, item) >= 0) break;

            data[pos] = best;
            pos = left;
        }

        data[pos] = item;
    }
};

tinyqueue.default = default_1$1;

var pointInPolygon = function (point, vs) {
    // ray-casting algorithm based on
    // http://www.ecse.rpi.edu/Homepages/wrf/Research/Short_Notes/pnpoly.html
    
    var x = point[0], y = point[1];
    
    var inside = false;
    for (var i = 0, j = vs.length - 1; i < vs.length; j = i++) {
        var xi = vs[i][0], yi = vs[i][1];
        var xj = vs[j][0], yj = vs[j][1];
        
        var intersect = ((yi > y) != (yj > y))
            && (x < (xj - xi) * (y - yi) / (yj - yi) + xi);
        if (intersect) inside = !inside;
    }
    
    return inside;
};

var orient = orientation_1[3];

var concaveman_1 = concaveman;
var default_1 = concaveman;

function concaveman(points, concavity, lengthThreshold) {
    // a relative measure of concavity; higher value means simpler hull
    concavity = Math.max(0, concavity === undefined ? 2 : concavity);

    // when a segment goes below this length threshold, it won't be drilled down further
    lengthThreshold = lengthThreshold || 0;

    // start with a convex hull of the points
    var hull = fastConvexHull(points);

    // index the points with an R-tree
    var tree = rbush_1(16, ['[0]', '[1]', '[0]', '[1]']).load(points);

    // turn the convex hull into a linked list and populate the initial edge queue with the nodes
    var queue = [];
    for (var i = 0, last; i < hull.length; i++) {
        var p = hull[i];
        tree.remove(p);
        last = insertNode(p, last);
        queue.push(last);
    }

    // index the segments with an R-tree (for intersection checks)
    var segTree = rbush_1(16);
    for (i = 0; i < queue.length; i++) segTree.insert(updateBBox(queue[i]));

    var sqConcavity = concavity * concavity;
    var sqLenThreshold = lengthThreshold * lengthThreshold;

    // process edges one by one
    while (queue.length) {
        var node = queue.shift();
        var a = node.p;
        var b = node.next.p;

        // skip the edge if it's already short enough
        var sqLen = getSqDist(a, b);
        if (sqLen < sqLenThreshold) continue;

        var maxSqLen = sqLen / sqConcavity;

        // find the best connection point for the current edge to flex inward to
        p = findCandidate(tree, node.prev.p, a, b, node.next.next.p, maxSqLen, segTree);

        // if we found a connection and it satisfies our concavity measure
        if (p && Math.min(getSqDist(p, a), getSqDist(p, b)) <= maxSqLen) {
            // connect the edge endpoints through this point and add 2 new edges to the queue
            queue.push(node);
            queue.push(insertNode(p, node));

            // update point and segment indexes
            tree.remove(p);
            segTree.remove(node);
            segTree.insert(updateBBox(node));
            segTree.insert(updateBBox(node.next));
        }
    }

    // convert the resulting hull linked list to an array of points
    node = last;
    var concave = [];
    do {
        concave.push(node.p);
        node = node.next;
    } while (node !== last);

    concave.push(node.p);

    return concave;
}

function findCandidate(tree, a, b, c, d, maxDist, segTree) {
    var queue = new tinyqueue(null, compareDist);
    var node = tree.data;

    // search through the point R-tree with a depth-first search using a priority queue
    // in the order of distance to the edge (b, c)
    while (node) {
        for (var i = 0; i < node.children.length; i++) {
            var child = node.children[i];

            var dist = node.leaf ? sqSegDist(child, b, c) : sqSegBoxDist(b, c, child);
            if (dist > maxDist) continue; // skip the node if it's farther than we ever need

            queue.push({
                node: child,
                dist: dist
            });
        }

        while (queue.length && !queue.peek().node.children) {
            var item = queue.pop();
            var p = item.node;

            // skip all points that are as close to adjacent edges (a,b) and (c,d),
            // and points that would introduce self-intersections when connected
            var d0 = sqSegDist(p, a, b);
            var d1 = sqSegDist(p, c, d);
            if (item.dist < d0 && item.dist < d1 &&
                noIntersections(b, p, segTree) &&
                noIntersections(c, p, segTree)) return p;
        }

        node = queue.pop();
        if (node) node = node.node;
    }

    return null;
}

function compareDist(a, b) {
    return a.dist - b.dist;
}

// square distance from a segment bounding box to the given one
function sqSegBoxDist(a, b, bbox) {
    if (inside(a, bbox) || inside(b, bbox)) return 0;
    var d1 = sqSegSegDist(a[0], a[1], b[0], b[1], bbox.minX, bbox.minY, bbox.maxX, bbox.minY);
    if (d1 === 0) return 0;
    var d2 = sqSegSegDist(a[0], a[1], b[0], b[1], bbox.minX, bbox.minY, bbox.minX, bbox.maxY);
    if (d2 === 0) return 0;
    var d3 = sqSegSegDist(a[0], a[1], b[0], b[1], bbox.maxX, bbox.minY, bbox.maxX, bbox.maxY);
    if (d3 === 0) return 0;
    var d4 = sqSegSegDist(a[0], a[1], b[0], b[1], bbox.minX, bbox.maxY, bbox.maxX, bbox.maxY);
    if (d4 === 0) return 0;
    return Math.min(d1, d2, d3, d4);
}

function inside(a, bbox) {
    return a[0] >= bbox.minX &&
           a[0] <= bbox.maxX &&
           a[1] >= bbox.minY &&
           a[1] <= bbox.maxY;
}

// check if the edge (a,b) doesn't intersect any other edges
function noIntersections(a, b, segTree) {
    var minX = Math.min(a[0], b[0]);
    var minY = Math.min(a[1], b[1]);
    var maxX = Math.max(a[0], b[0]);
    var maxY = Math.max(a[1], b[1]);

    var edges = segTree.search({minX: minX, minY: minY, maxX: maxX, maxY: maxY});
    for (var i = 0; i < edges.length; i++) {
        if (intersects(edges[i].p, edges[i].next.p, a, b)) return false;
    }
    return true;
}

// check if the edges (p1,q1) and (p2,q2) intersect
function intersects(p1, q1, p2, q2) {
    return p1 !== q2 && q1 !== p2 &&
        orient(p1, q1, p2) > 0 !== orient(p1, q1, q2) > 0 &&
        orient(p2, q2, p1) > 0 !== orient(p2, q2, q1) > 0;
}

// update the bounding box of a node's edge
function updateBBox(node) {
    var p1 = node.p;
    var p2 = node.next.p;
    node.minX = Math.min(p1[0], p2[0]);
    node.minY = Math.min(p1[1], p2[1]);
    node.maxX = Math.max(p1[0], p2[0]);
    node.maxY = Math.max(p1[1], p2[1]);
    return node;
}

// speed up convex hull by filtering out points inside quadrilateral formed by 4 extreme points
function fastConvexHull(points) {
    var left = points[0];
    var top = points[0];
    var right = points[0];
    var bottom = points[0];

    // find the leftmost, rightmost, topmost and bottommost points
    for (var i = 0; i < points.length; i++) {
        var p = points[i];
        if (p[0] < left[0]) left = p;
        if (p[0] > right[0]) right = p;
        if (p[1] < top[1]) top = p;
        if (p[1] > bottom[1]) bottom = p;
    }

    // filter out points that are inside the resulting quadrilateral
    var cull = [left, top, right, bottom];
    var filtered = cull.slice();
    for (i = 0; i < points.length; i++) {
        if (!pointInPolygon(points[i], cull)) filtered.push(points[i]);
    }

    // get convex hull around the filtered points
    var indices = monotoneConvexHull2d(filtered);

    // return the hull as array of points (rather than indices)
    var hull = [];
    for (i = 0; i < indices.length; i++) hull.push(filtered[indices[i]]);
    return hull;
}

// create a new node in a doubly linked list
function insertNode(p, prev) {
    var node = {
        p: p,
        prev: null,
        next: null,
        minX: 0,
        minY: 0,
        maxX: 0,
        maxY: 0
    };

    if (!prev) {
        node.prev = node;
        node.next = node;

    } else {
        node.next = prev.next;
        node.prev = prev;
        prev.next.prev = node;
        prev.next = node;
    }
    return node;
}

// square distance between 2 points
function getSqDist(p1, p2) {

    var dx = p1[0] - p2[0],
        dy = p1[1] - p2[1];

    return dx * dx + dy * dy;
}

// square distance from a point to a segment
function sqSegDist(p, p1, p2) {

    var x = p1[0],
        y = p1[1],
        dx = p2[0] - x,
        dy = p2[1] - y;

    if (dx !== 0 || dy !== 0) {

        var t = ((p[0] - x) * dx + (p[1] - y) * dy) / (dx * dx + dy * dy);

        if (t > 1) {
            x = p2[0];
            y = p2[1];

        } else if (t > 0) {
            x += dx * t;
            y += dy * t;
        }
    }

    dx = p[0] - x;
    dy = p[1] - y;

    return dx * dx + dy * dy;
}

// segment to segment distance, ported from http://geomalgorithms.com/a07-_distance.html by Dan Sunday
function sqSegSegDist(x0, y0, x1, y1, x2, y2, x3, y3) {
    var ux = x1 - x0;
    var uy = y1 - y0;
    var vx = x3 - x2;
    var vy = y3 - y2;
    var wx = x0 - x2;
    var wy = y0 - y2;
    var a = ux * ux + uy * uy;
    var b = ux * vx + uy * vy;
    var c = vx * vx + vy * vy;
    var d = ux * wx + uy * wy;
    var e = vx * wx + vy * wy;
    var D = a * c - b * b;

    var sc, sN, tc, tN;
    var sD = D;
    var tD = D;

    if (D === 0) {
        sN = 0;
        sD = 1;
        tN = e;
        tD = c;
    } else {
        sN = b * e - c * d;
        tN = a * e - b * d;
        if (sN < 0) {
            sN = 0;
            tN = e;
            tD = c;
        } else if (sN > sD) {
            sN = sD;
            tN = e + b;
            tD = c;
        }
    }

    if (tN < 0.0) {
        tN = 0.0;
        if (-d < 0.0) sN = 0.0;
        else if (-d > a) sN = sD;
        else {
            sN = -d;
            sD = a;
        }
    } else if (tN > tD) {
        tN = tD;
        if ((-d + b) < 0.0) sN = 0;
        else if (-d + b > a) sN = sD;
        else {
            sN = -d + b;
            sD = a;
        }
    }

    sc = sN === 0 ? 0 : sN / sD;
    tc = tN === 0 ? 0 : tN / tD;

    var cx = (1 - sc) * x0 + sc * x1;
    var cy = (1 - sc) * y0 + sc * y1;
    var cx2 = (1 - tc) * x2 + tc * x3;
    var cy2 = (1 - tc) * y2 + tc * y3;
    var dx = cx2 - cx;
    var dy = cy2 - cy;

    return dx * dx + dy * dy;
}

concaveman_1.default = default_1;

/**
 * Takes a {@link Feature} or a {@link FeatureCollection} and returns a convex hull {@link Polygon}.
 *
 * Internally this uses
 * the [convex-hull](https://github.com/mikolalysenko/convex-hull) module that
 * implements a [monotone chain hull](http://en.wikibooks.org/wiki/Algorithm_Implementation/Geometry/Convex_hull/Monotone_chain).
 *
 * @name convex
 * @param {GeoJSON} geojson input Feature or FeatureCollection
 * @param {Object} [options={}] Optional parameters
 * @param {number} [options.concavity=Infinity] 1 - thin shape. Infinity - convex hull.
 * @returns {Feature<Polygon>} a convex hull
 * @example
 * var points = turf.featureCollection([
 *   turf.point([10.195312, 43.755225]),
 *   turf.point([10.404052, 43.8424511]),
 *   turf.point([10.579833, 43.659924]),
 *   turf.point([10.360107, 43.516688]),
 *   turf.point([10.14038, 43.588348]),
 *   turf.point([10.195312, 43.755225])
 * ]);
 *
 * var hull = turf.convex(points);
 *
 * //addToMap
 * var addToMap = [points, hull]
 */
function convex(geojson, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var concavity = options.concavity || Infinity;
    var points$$1 = [];

    // Convert all points to flat 2D coordinate Array
    coordEach(geojson, function (coord) {
        points$$1.push([coord[0], coord[1]]);
    });
    if (!points$$1.length) return null;

    var convexHull = concaveman_1(points$$1, concavity);

    // Convex hull should have at least 3 different vertices in order to create a valid polygon
    if (convexHull.length > 3) {
        return polygon([convexHull]);
    }
    return null;
}

// http://en.wikipedia.org/wiki/Even%E2%80%93odd_rule
// modified from: https://github.com/substack/point-in-polygon/blob/master/index.js
// which was modified from http://www.ecse.rpi.edu/Homepages/wrf/Research/Short_Notes/pnpoly.html

/**
 * Takes a {@link Point} and a {@link Polygon} or {@link MultiPolygon} and determines if the point resides inside the polygon. The polygon can
 * be convex or concave. The function accounts for holes.
 *
 * @name booleanPointInPolygon
 * @param {Coord} point input point
 * @param {Feature<Polygon|MultiPolygon>} polygon input polygon or multipolygon
 * @param {Object} [options={}] Optional parameters
 * @param {boolean} [options.ignoreBoundary=false] True if polygon boundary should be ignored when determining if the point is inside the polygon otherwise false.
 * @returns {boolean} `true` if the Point is inside the Polygon; `false` if the Point is not inside the Polygon
 * @example
 * var pt = turf.point([-77, 44]);
 * var poly = turf.polygon([[
 *   [-81, 41],
 *   [-81, 47],
 *   [-72, 47],
 *   [-72, 41],
 *   [-81, 41]
 * ]]);
 *
 * turf.booleanPointInPolygon(pt, poly);
 * //= true
 */
function booleanPointInPolygon(point, polygon, options) {
    // Optional parameters
    options = options || {};
    if (typeof options !== 'object') throw new Error('options is invalid');
    var ignoreBoundary = options.ignoreBoundary;

    // validation
    if (!point) throw new Error('point is required');
    if (!polygon) throw new Error('polygon is required');

    var pt = getCoord(point);
    var polys = getCoords(polygon);
    var type = (polygon.geometry) ? polygon.geometry.type : polygon.type;
    var bbox = polygon.bbox;

    // Quick elimination if point is not inside bbox
    if (bbox && inBBox(pt, bbox) === false) return false;

    // normalize to multipolygon
    if (type === 'Polygon') polys = [polys];

    for (var i = 0, insidePoly = false; i < polys.length && !insidePoly; i++) {
        // check if it is in the outer ring first
        if (inRing(pt, polys[i][0], ignoreBoundary)) {
            var inHole = false;
            var k = 1;
            // check for the point in any of the holes
            while (k < polys[i].length && !inHole) {
                if (inRing(pt, polys[i][k], !ignoreBoundary)) {
                    inHole = true;
                }
                k++;
            }
            if (!inHole) insidePoly = true;
        }
    }
    return insidePoly;
}

/**
 * inRing
 *
 * @private
 * @param {Array<number>} pt [x,y]
 * @param {Array<Array<number>>} ring [[x,y], [x,y],..]
 * @param {boolean} ignoreBoundary ignoreBoundary
 * @returns {boolean} inRing
 */
function inRing(pt, ring, ignoreBoundary) {
    var isInside = false;
    if (ring[0][0] === ring[ring.length - 1][0] && ring[0][1] === ring[ring.length - 1][1]) ring = ring.slice(0, ring.length - 1);

    for (var i = 0, j = ring.length - 1; i < ring.length; j = i++) {
        var xi = ring[i][0], yi = ring[i][1];
        var xj = ring[j][0], yj = ring[j][1];
        var onBoundary = (pt[1] * (xi - xj) + yi * (xj - pt[0]) + yj * (pt[0] - xi) === 0) &&
            ((xi - pt[0]) * (xj - pt[0]) <= 0) && ((yi - pt[1]) * (yj - pt[1]) <= 0);
        if (onBoundary) return !ignoreBoundary;
        var intersect = ((yi > pt[1]) !== (yj > pt[1])) &&
        (pt[0] < (xj - xi) * (pt[1] - yi) / (yj - yi) + xi);
        if (intersect) isInside = !isInside;
    }
    return isInside;
}

/**
 * inBBox
 *
 * @private
 * @param {Position} pt point [x,y]
 * @param {BBox} bbox BBox [west, south, east, north]
 * @returns {boolean} true/false if point is inside BBox
 */
function inBBox(pt, bbox) {
    return bbox[0] <= pt[0] &&
           bbox[1] <= pt[1] &&
           bbox[2] >= pt[0] &&
           bbox[3] >= pt[1];
}

/**
 * Finds {@link Points} that fall within {@link (Multi)Polygon(s)}.
 *
 * @name pointsWithinPolygon
 * @param {Feauture|FeatureCollection<Point>} points Points as input search
 * @param {FeatureCollection|Geoemtry|Feature<Polygon|MultiPolygon>} polygons Points must be within these (Multi)Polygon(s)
 * @returns {FeatureCollection<Point>} points that land within at least one polygon
 * @example
 * var points = turf.points([
 *     [-46.6318, -23.5523],
 *     [-46.6246, -23.5325],
 *     [-46.6062, -23.5513],
 *     [-46.663, -23.554],
 *     [-46.643, -23.557]
 * ]);
 *
 * var searchWithin = turf.polygon([[
 *     [-46.653,-23.543],
 *     [-46.634,-23.5346],
 *     [-46.613,-23.543],
 *     [-46.614,-23.559],
 *     [-46.631,-23.567],
 *     [-46.653,-23.560],
 *     [-46.653,-23.543]
 * ]]);
 *
 * var ptsWithin = turf.pointsWithinPolygon(points, searchWithin);
 *
 * //addToMap
 * var addToMap = [points, searchWithin, ptsWithin]
 * turf.featureEach(ptsWithin, function (currentFeature) {
 *   currentFeature.properties['marker-size'] = 'large';
 *   currentFeature.properties['marker-color'] = '#000';
 * });
 */
function pointsWithinPolygon(points$$1, polygons$$1) {
    var results = [];
    geomEach(polygons$$1, function (polygon$$1) {
        featureEach(points$$1, function (point$$1) {
            if (booleanPointInPolygon(point$$1, polygon$$1)) results.push(point$$1);
        });
    });
    return featureCollection(results);
}

//http://en.wikipedia.org/wiki/Delaunay_triangulation
//https://github.com/ironwallaby/delaunay
/**
 * Takes a set of {@link Point|points} and creates a
 * [Triangulated Irregular Network](http://en.wikipedia.org/wiki/Triangulated_irregular_network),
 * or a TIN for short, returned as a collection of Polygons. These are often used
 * for developing elevation contour maps or stepped heat visualizations.
 *
 * If an optional z-value property is provided then it is added as properties called `a`, `b`,
 * and `c` representing its value at each of the points that represent the corners of the
 * triangle.
 *
 * @name tin
 * @param {FeatureCollection<Point>} points input points
 * @param {String} [z] name of the property from which to pull z values
 * This is optional: if not given, then there will be no extra data added to the derived triangles.
 * @returns {FeatureCollection<Polygon>} TIN output
 * @example
 * // generate some random point data
 * var points = turf.randomPoint(30, {bbox: [50, 30, 70, 50]});
 *
 * // add a random property to each point between 0 and 9
 * for (var i = 0; i < points.features.length; i++) {
 *   points.features[i].properties.z = ~~(Math.random() * 9);
 * }
 * var tin = turf.tin(points, 'z');
 *
 * //addToMap
 * var addToMap = [tin, points]
 * for (var i = 0; i < tin.features.length; i++) {
 *   var properties  = tin.features[i].properties;
 *   properties.fill = '#' + properties.a + properties.b + properties.c;
 * }
 */
function tin(points$$1, z) {
    if (points$$1.type !== 'FeatureCollection') throw new Error('points must be a FeatureCollection');
    //break down points
    var isPointZ = false;
    return featureCollection(triangulate(points$$1.features.map(function (p) {
        var point$$1 = {
            x: p.geometry.coordinates[0],
            y: p.geometry.coordinates[1]
        };
        if (z) {
            point$$1.z = p.properties[z];
        } else if (p.geometry.coordinates.length === 3) {
            isPointZ = true;
            point$$1.z = p.geometry.coordinates[2];
        }
        return point$$1;
    })).map(function (triangle) {

        var a = [triangle.a.x, triangle.a.y];
        var b = [triangle.b.x, triangle.b.y];
        var c = [triangle.c.x, triangle.c.y];
        var properties = {};

        // Add z coordinates to triangle points if user passed
        // them in that way otherwise add it as a property.
        if (isPointZ) {
            a.push(triangle.a.z);
            b.push(triangle.b.z);
            c.push(triangle.c.z);
        } else {
            properties = {
                a: triangle.a.z,
                b: triangle.b.z,
                c: triangle.c.z
            };
        }

        return polygon([[a, b, c, a]], properties);

    }));
}

function Triangle(a, b, c) {
    this.a = a;
    this.b = b;
    this.c = c;

    var A = b.x - a.x,
        B = b.y - a.y,
        C = c.x - a.x,
        D = c.y - a.y,
        E = A * (a.x + b.x) + B * (a.y + b.y),
        F = C * (a.x + c.x) + D * (a.y + c.y),
        G = 2 * (A * (c.y - b.y) - B * (c.x - b.x)),
        dx, dy;

    // If the points of the triangle are collinear, then just find the
    // extremes and use the midpoint as the center of the circumcircle.
    this.x = (D * E - B * F) / G;
    this.y = (A * F - C * E) / G;
    dx = this.x - a.x;
    dy = this.y - a.y;
    this.r = dx * dx + dy * dy;
}

function byX(a, b) {
    return b.x - a.x;
}

function dedup(edges) {
    var j = edges.length,
        a, b, i, m, n;

    outer:
    while (j) {
        b = edges[--j];
        a = edges[--j];
        i = j;
        while (i) {
            n = edges[--i];
            m = edges[--i];
            if ((a === m && b === n) || (a === n && b === m)) {
                edges.splice(j, 2);
                edges.splice(i, 2);
                j -= 2;
                continue outer;
            }
        }
    }
}

function triangulate(vertices) {
    // Bail if there aren't enough vertices to form any triangles.
    if (vertices.length < 3)
        return [];

    // Ensure the vertex array is in order of descending X coordinate
    // (which is needed to ensure a subquadratic runtime), and then find
    // the bounding box around the points.
    vertices.sort(byX);

    var i = vertices.length - 1,
        xmin = vertices[i].x,
        xmax = vertices[0].x,
        ymin = vertices[i].y,
        ymax = ymin,
        epsilon = 1e-12;

    var a,
        b,
        c,
        A,
        B,
        G;

    while (i--) {
        if (vertices[i].y < ymin)
            ymin = vertices[i].y;
        if (vertices[i].y > ymax)
            ymax = vertices[i].y;
    }

    //Find a supertriangle, which is a triangle that surrounds all the
    //vertices. This is used like something of a sentinel value to remove
    //cases in the main algorithm, and is removed before we return any
    // results.

    // Once found, put it in the "open" list. (The "open" list is for
    // triangles who may still need to be considered; the "closed" list is
    // for triangles which do not.)
    var dx = xmax - xmin,
        dy = ymax - ymin,
        dmax = (dx > dy) ? dx : dy,
        xmid = (xmax + xmin) * 0.5,
        ymid = (ymax + ymin) * 0.5,
        open = [
            new Triangle({
                x: xmid - 20 * dmax,
                y: ymid - dmax,
                __sentinel: true
            }, {
                x: xmid,
                y: ymid + 20 * dmax,
                __sentinel: true
            }, {
                x: xmid + 20 * dmax,
                y: ymid - dmax,
                __sentinel: true
            }
            )],
        closed = [],
        edges = [],
        j;

    // Incrementally add each vertex to the mesh.
    i = vertices.length;
    while (i--) {
        // For each open triangle, check to see if the current point is
        // inside it's circumcircle. If it is, remove the triangle and add
        // it's edges to an edge list.
        edges.length = 0;
        j = open.length;
        while (j--) {
            // If this point is to the right of this triangle's circumcircle,
            // then this triangle should never get checked again. Remove it
            // from the open list, add it to the closed list, and skip.
            dx = vertices[i].x - open[j].x;
            if (dx > 0 && dx * dx > open[j].r) {
                closed.push(open[j]);
                open.splice(j, 1);
                continue;
            }

            // If not, skip this triangle.
            dy = vertices[i].y - open[j].y;
            if (dx * dx + dy * dy > open[j].r)
                continue;

            // Remove the triangle and add it's edges to the edge list.
            edges.push(
                open[j].a, open[j].b,
                open[j].b, open[j].c,
                open[j].c, open[j].a
            );
            open.splice(j, 1);
        }

        // Remove any doubled edges.
        dedup(edges);

        // Add a new triangle for each edge.
        j = edges.length;
        while (j) {
            b = edges[--j];
            a = edges[--j];
            c = vertices[i];
            // Avoid adding colinear triangles (which have error-prone
            // circumcircles)
            A = b.x - a.x;
            B = b.y - a.y;
            G = 2 * (A * (c.y - b.y) - B * (c.x - b.x));
            if (Math.abs(G) > epsilon) {
                open.push(new Triangle(a, b, c));
            }
        }
    }

    // Copy any remaining open triangles to the closed list, and then
    // remove any triangles that share a vertex with the supertriangle.
    Array.prototype.push.apply(closed, open);

    i = closed.length;
    while (i--)
        if (closed[i].a.__sentinel ||
            closed[i].b.__sentinel ||
            closed[i].c.__sentinel)
            closed.splice(i, 1);

    return closed;
}

//http://en.wikipedia.org/wiki/Haversine_formula
//http://www.movable-type.co.uk/scripts/latlong.html

/**
 * Calculates the distance between two {@link Point|points} in degrees, radians,
 * miles, or kilometers. This uses the
 * [Haversine formula](http://en.wikipedia.org/wiki/Haversine_formula)
 * to account for global curvature.
 *
 * @name distance
 * @param {Coord} from origin point
 * @param {Coord} to destination point
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units='kilometers'] can be degrees, radians, miles, or kilometers
 * @returns {number} distance between the two points
 * @example
 * var from = turf.point([-75.343, 39.984]);
 * var to = turf.point([-75.534, 39.123]);
 * var options = {units: 'miles'};
 *
 * var distance = turf.distance(from, to, options);
 *
 * //addToMap
 * var addToMap = [from, to];
 * from.properties.distance = distance;
 * to.properties.distance = distance;
 */
function distance(from, to, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var units = options.units;

    var coordinates1 = getCoord(from);
    var coordinates2 = getCoord(to);
    var dLat = degreesToRadians((coordinates2[1] - coordinates1[1]));
    var dLon = degreesToRadians((coordinates2[0] - coordinates1[0]));
    var lat1 = degreesToRadians(coordinates1[1]);
    var lat2 = degreesToRadians(coordinates2[1]);

    var a = Math.pow(Math.sin(dLat / 2), 2) +
          Math.pow(Math.sin(dLon / 2), 2) * Math.cos(lat1) * Math.cos(lat2);

    return radiansToLength(2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a)), units);
}

/**
 * Returns a cloned copy of the passed GeoJSON Object, including possible 'Foreign Members'.
 * ~3-5x faster than the common JSON.parse + JSON.stringify combo method.
 *
 * @name clone
 * @param {GeoJSON} geojson GeoJSON Object
 * @returns {GeoJSON} cloned GeoJSON Object
 * @example
 * var line = turf.lineString([[-74, 40], [-78, 42], [-82, 35]], {color: 'red'});
 *
 * var lineCloned = turf.clone(line);
 */
function clone(geojson) {
    if (!geojson) throw new Error('geojson is required');

    switch (geojson.type) {
    case 'Feature':
        return cloneFeature(geojson);
    case 'FeatureCollection':
        return cloneFeatureCollection(geojson);
    case 'Point':
    case 'LineString':
    case 'Polygon':
    case 'MultiPoint':
    case 'MultiLineString':
    case 'MultiPolygon':
    case 'GeometryCollection':
        return cloneGeometry(geojson);
    default:
        throw new Error('unknown GeoJSON type');
    }
}

/**
 * Clone Feature
 *
 * @private
 * @param {Feature<any>} geojson GeoJSON Feature
 * @returns {Feature<any>} cloned Feature
 */
function cloneFeature(geojson) {
    var cloned = {type: 'Feature'};
    // Preserve Foreign Members
    Object.keys(geojson).forEach(function (key) {
        switch (key) {
        case 'type':
        case 'properties':
        case 'geometry':
            return;
        default:
            cloned[key] = geojson[key];
        }
    });
    // Add properties & geometry last
    cloned.properties = cloneProperties(geojson.properties);
    cloned.geometry = cloneGeometry(geojson.geometry);
    return cloned;
}

/**
 * Clone Properties
 *
 * @private
 * @param {Object} properties GeoJSON Properties
 * @returns {Object} cloned Properties
 */
function cloneProperties(properties) {
    var cloned = {};
    if (!properties) return cloned;
    Object.keys(properties).forEach(function (key) {
        var value = properties[key];
        if (typeof value === 'object') {
            if (value === null) {
                // handle null
                cloned[key] = null;
            } else if (value.length) {
                // handle Array
                cloned[key] = value.map(function (item) {
                    return item;
                });
            } else {
                // handle generic Object
                cloned[key] = cloneProperties(value);
            }
        } else cloned[key] = value;
    });
    return cloned;
}

/**
 * Clone Feature Collection
 *
 * @private
 * @param {FeatureCollection<any>} geojson GeoJSON Feature Collection
 * @returns {FeatureCollection<any>} cloned Feature Collection
 */
function cloneFeatureCollection(geojson) {
    var cloned = {type: 'FeatureCollection'};

    // Preserve Foreign Members
    Object.keys(geojson).forEach(function (key) {
        switch (key) {
        case 'type':
        case 'features':
            return;
        default:
            cloned[key] = geojson[key];
        }
    });
    // Add features
    cloned.features = geojson.features.map(function (feature) {
        return cloneFeature(feature);
    });
    return cloned;
}

/**
 * Clone Geometry
 *
 * @private
 * @param {Geometry<any>} geometry GeoJSON Geometry
 * @returns {Geometry<any>} cloned Geometry
 */
function cloneGeometry(geometry) {
    var geom = {type: geometry.type};
    if (geometry.bbox) geom.bbox = geometry.bbox;

    if (geometry.type === 'GeometryCollection') {
        geom.geometries = geometry.geometries.map(function (geom) {
            return cloneGeometry(geom);
        });
        return geom;
    }
    geom.coordinates = deepSlice(geometry.coordinates);
    return geom;
}

/**
 * Deep Slice coordinates
 *
 * @private
 * @param {Coordinates} coords Coordinates
 * @returns {Coordinates} all coordinates sliced
 */
function deepSlice(coords) {
    if (typeof coords[0] !== 'object') { return coords.slice(); }
    return coords.map(function (coord) {
        return deepSlice(coord);
    });
}

var identity = function(x) {
  return x;
};

var transform = function(transform) {
  if (transform == null) return identity;
  var x0,
      y0,
      kx = transform.scale[0],
      ky = transform.scale[1],
      dx = transform.translate[0],
      dy = transform.translate[1];
  return function(input, i) {
    if (!i) x0 = y0 = 0;
    var j = 2, n = input.length, output = new Array(n);
    output[0] = (x0 += input[0]) * kx + dx;
    output[1] = (y0 += input[1]) * ky + dy;
    while (j < n) output[j] = input[j], ++j;
    return output;
  };
};

var reverse = function(array, n) {
  var t, j = array.length, i = j - n;
  while (i < --j) t = array[i], array[i++] = array[j], array[j] = t;
};

function object(topology, o) {
  var transformPoint = transform(topology.transform),
      arcs = topology.arcs;

  function arc(i, points) {
    if (points.length) points.pop();
    for (var a = arcs[i < 0 ? ~i : i], k = 0, n = a.length; k < n; ++k) {
      points.push(transformPoint(a[k], k));
    }
    if (i < 0) reverse(points, n);
  }

  function point(p) {
    return transformPoint(p);
  }

  function line(arcs) {
    var points = [];
    for (var i = 0, n = arcs.length; i < n; ++i) arc(arcs[i], points);
    if (points.length < 2) points.push(points[0]); // This should never happen per the specification.
    return points;
  }

  function ring(arcs) {
    var points = line(arcs);
    while (points.length < 4) points.push(points[0]); // This may happen if an arc has only two points.
    return points;
  }

  function polygon(arcs) {
    return arcs.map(ring);
  }

  function geometry(o) {
    var type = o.type, coordinates;
    switch (type) {
      case "GeometryCollection": return {type: type, geometries: o.geometries.map(geometry)};
      case "Point": coordinates = point(o.coordinates); break;
      case "MultiPoint": coordinates = o.coordinates.map(point); break;
      case "LineString": coordinates = line(o.arcs); break;
      case "MultiLineString": coordinates = o.arcs.map(line); break;
      case "Polygon": coordinates = polygon(o.arcs); break;
      case "MultiPolygon": coordinates = o.arcs.map(polygon); break;
      default: return null;
    }
    return {type: type, coordinates: coordinates};
  }

  return geometry(o);
}

var stitch = function(topology, arcs) {
  var stitchedArcs = {},
      fragmentByStart = {},
      fragmentByEnd = {},
      fragments = [],
      emptyIndex = -1;

  // Stitch empty arcs first, since they may be subsumed by other arcs.
  arcs.forEach(function(i, j) {
    var arc = topology.arcs[i < 0 ? ~i : i], t;
    if (arc.length < 3 && !arc[1][0] && !arc[1][1]) {
      t = arcs[++emptyIndex], arcs[emptyIndex] = i, arcs[j] = t;
    }
  });

  arcs.forEach(function(i) {
    var e = ends(i),
        start = e[0],
        end = e[1],
        f, g;

    if (f = fragmentByEnd[start]) {
      delete fragmentByEnd[f.end];
      f.push(i);
      f.end = end;
      if (g = fragmentByStart[end]) {
        delete fragmentByStart[g.start];
        var fg = g === f ? f : f.concat(g);
        fragmentByStart[fg.start = f.start] = fragmentByEnd[fg.end = g.end] = fg;
      } else {
        fragmentByStart[f.start] = fragmentByEnd[f.end] = f;
      }
    } else if (f = fragmentByStart[end]) {
      delete fragmentByStart[f.start];
      f.unshift(i);
      f.start = start;
      if (g = fragmentByEnd[start]) {
        delete fragmentByEnd[g.end];
        var gf = g === f ? f : g.concat(f);
        fragmentByStart[gf.start = g.start] = fragmentByEnd[gf.end = f.end] = gf;
      } else {
        fragmentByStart[f.start] = fragmentByEnd[f.end] = f;
      }
    } else {
      f = [i];
      fragmentByStart[f.start = start] = fragmentByEnd[f.end = end] = f;
    }
  });

  function ends(i) {
    var arc = topology.arcs[i < 0 ? ~i : i], p0 = arc[0], p1;
    if (topology.transform) p1 = [0, 0], arc.forEach(function(dp) { p1[0] += dp[0], p1[1] += dp[1]; });
    else p1 = arc[arc.length - 1];
    return i < 0 ? [p1, p0] : [p0, p1];
  }

  function flush(fragmentByEnd, fragmentByStart) {
    for (var k in fragmentByEnd) {
      var f = fragmentByEnd[k];
      delete fragmentByStart[f.start];
      delete f.start;
      delete f.end;
      f.forEach(function(i) { stitchedArcs[i < 0 ? ~i : i] = 1; });
      fragments.push(f);
    }
  }

  flush(fragmentByEnd, fragmentByStart);
  flush(fragmentByStart, fragmentByEnd);
  arcs.forEach(function(i) { if (!stitchedArcs[i < 0 ? ~i : i]) fragments.push([i]); });

  return fragments;
};

function planarRingArea(ring) {
  var i = -1, n = ring.length, a, b = ring[n - 1], area = 0;
  while (++i < n) a = b, b = ring[i], area += a[0] * b[1] - a[1] * b[0];
  return Math.abs(area); // Note: doubled area!
}

var merge = function(topology) {
  return object(topology, mergeArcs.apply(this, arguments));
};

function mergeArcs(topology, objects) {
  var polygonsByArc = {},
      polygons = [],
      groups = [];

  objects.forEach(geometry);

  function geometry(o) {
    switch (o.type) {
      case "GeometryCollection": o.geometries.forEach(geometry); break;
      case "Polygon": extract(o.arcs); break;
      case "MultiPolygon": o.arcs.forEach(extract); break;
    }
  }

  function extract(polygon) {
    polygon.forEach(function(ring) {
      ring.forEach(function(arc) {
        (polygonsByArc[arc = arc < 0 ? ~arc : arc] || (polygonsByArc[arc] = [])).push(polygon);
      });
    });
    polygons.push(polygon);
  }

  function area(ring) {
    return planarRingArea(object(topology, {type: "Polygon", arcs: [ring]}).coordinates[0]);
  }

  polygons.forEach(function(polygon) {
    if (!polygon._) {
      var group = [],
          neighbors = [polygon];
      polygon._ = 1;
      groups.push(group);
      while (polygon = neighbors.pop()) {
        group.push(polygon);
        polygon.forEach(function(ring) {
          ring.forEach(function(arc) {
            polygonsByArc[arc < 0 ? ~arc : arc].forEach(function(polygon) {
              if (!polygon._) {
                polygon._ = 1;
                neighbors.push(polygon);
              }
            });
          });
        });
      }
    }
  });

  polygons.forEach(function(polygon) {
    delete polygon._;
  });

  return {
    type: "MultiPolygon",
    arcs: groups.map(function(polygons) {
      var arcs = [], n;

      // Extract the exterior (unique) arcs.
      polygons.forEach(function(polygon) {
        polygon.forEach(function(ring) {
          ring.forEach(function(arc) {
            if (polygonsByArc[arc < 0 ? ~arc : arc].length < 2) {
              arcs.push(arc);
            }
          });
        });
      });

      // Stitch the arcs into one or more rings.
      arcs = stitch(topology, arcs);

      // If more than one ring is returned,
      // at most one of these rings can be the exterior;
      // choose the one with the greatest absolute area.
      if ((n = arcs.length) > 1) {
        for (var i = 1, k = area(arcs[0]), ki, t; i < n; ++i) {
          if ((ki = area(arcs[i])) > k) {
            t = arcs[0], arcs[0] = arcs[i], arcs[i] = t, k = ki;
          }
        }
      }

      return arcs;
    })
  };
}

// Computes the bounding box of the specified hash of GeoJSON objects.
var bounds = function(objects) {
  var x0 = Infinity,
      y0 = Infinity,
      x1 = -Infinity,
      y1 = -Infinity;

  function boundGeometry(geometry) {
    if (geometry != null && boundGeometryType.hasOwnProperty(geometry.type)) boundGeometryType[geometry.type](geometry);
  }

  var boundGeometryType = {
    GeometryCollection: function(o) { o.geometries.forEach(boundGeometry); },
    Point: function(o) { boundPoint(o.coordinates); },
    MultiPoint: function(o) { o.coordinates.forEach(boundPoint); },
    LineString: function(o) { boundLine(o.arcs); },
    MultiLineString: function(o) { o.arcs.forEach(boundLine); },
    Polygon: function(o) { o.arcs.forEach(boundLine); },
    MultiPolygon: function(o) { o.arcs.forEach(boundMultiLine); }
  };

  function boundPoint(coordinates) {
    var x = coordinates[0],
        y = coordinates[1];
    if (x < x0) x0 = x;
    if (x > x1) x1 = x;
    if (y < y0) y0 = y;
    if (y > y1) y1 = y;
  }

  function boundLine(coordinates) {
    coordinates.forEach(boundPoint);
  }

  function boundMultiLine(coordinates) {
    coordinates.forEach(boundLine);
  }

  for (var key in objects) {
    boundGeometry(objects[key]);
  }

  return x1 >= x0 && y1 >= y0 ? [x0, y0, x1, y1] : undefined;
};

var hashset = function(size, hash, equal, type, empty) {
  if (arguments.length === 3) {
    type = Array;
    empty = null;
  }

  var store = new type(size = 1 << Math.max(4, Math.ceil(Math.log(size) / Math.LN2))),
      mask = size - 1;

  for (var i = 0; i < size; ++i) {
    store[i] = empty;
  }

  function add(value) {
    var index = hash(value) & mask,
        match = store[index],
        collisions = 0;
    while (match != empty) {
      if (equal(match, value)) return true;
      if (++collisions >= size) throw new Error("full hashset");
      match = store[index = (index + 1) & mask];
    }
    store[index] = value;
    return true;
  }

  function has(value) {
    var index = hash(value) & mask,
        match = store[index],
        collisions = 0;
    while (match != empty) {
      if (equal(match, value)) return true;
      if (++collisions >= size) break;
      match = store[index = (index + 1) & mask];
    }
    return false;
  }

  function values() {
    var values = [];
    for (var i = 0, n = store.length; i < n; ++i) {
      var match = store[i];
      if (match != empty) values.push(match);
    }
    return values;
  }

  return {
    add: add,
    has: has,
    values: values
  };
};

var hashmap = function(size, hash, equal, keyType, keyEmpty, valueType) {
  if (arguments.length === 3) {
    keyType = valueType = Array;
    keyEmpty = null;
  }

  var keystore = new keyType(size = 1 << Math.max(4, Math.ceil(Math.log(size) / Math.LN2))),
      valstore = new valueType(size),
      mask = size - 1;

  for (var i = 0; i < size; ++i) {
    keystore[i] = keyEmpty;
  }

  function set(key, value) {
    var index = hash(key) & mask,
        matchKey = keystore[index],
        collisions = 0;
    while (matchKey != keyEmpty) {
      if (equal(matchKey, key)) return valstore[index] = value;
      if (++collisions >= size) throw new Error("full hashmap");
      matchKey = keystore[index = (index + 1) & mask];
    }
    keystore[index] = key;
    valstore[index] = value;
    return value;
  }

  function maybeSet(key, value) {
    var index = hash(key) & mask,
        matchKey = keystore[index],
        collisions = 0;
    while (matchKey != keyEmpty) {
      if (equal(matchKey, key)) return valstore[index];
      if (++collisions >= size) throw new Error("full hashmap");
      matchKey = keystore[index = (index + 1) & mask];
    }
    keystore[index] = key;
    valstore[index] = value;
    return value;
  }

  function get(key, missingValue) {
    var index = hash(key) & mask,
        matchKey = keystore[index],
        collisions = 0;
    while (matchKey != keyEmpty) {
      if (equal(matchKey, key)) return valstore[index];
      if (++collisions >= size) break;
      matchKey = keystore[index = (index + 1) & mask];
    }
    return missingValue;
  }

  function keys() {
    var keys = [];
    for (var i = 0, n = keystore.length; i < n; ++i) {
      var matchKey = keystore[i];
      if (matchKey != keyEmpty) keys.push(matchKey);
    }
    return keys;
  }

  return {
    set: set,
    maybeSet: maybeSet, // set if unset
    get: get,
    keys: keys
  };
};

var equalPoint = function(pointA, pointB) {
  return pointA[0] === pointB[0] && pointA[1] === pointB[1];
};

// TODO if quantized, use simpler Int32 hashing?

var buffer = new ArrayBuffer(16);
var floats = new Float64Array(buffer);
var uints = new Uint32Array(buffer);

var hashPoint = function(point) {
  floats[0] = point[0];
  floats[1] = point[1];
  var hash = uints[0] ^ uints[1];
  hash = hash << 5 ^ hash >> 7 ^ uints[2] ^ uints[3];
  return hash & 0x7fffffff;
};

// Given an extracted (pre-)topology, identifies all of the junctions. These are
// the points at which arcs (lines or rings) will need to be cut so that each
// arc is represented uniquely.
//
// A junction is a point where at least one arc deviates from another arc going
// through the same point. For example, consider the point B. If there is a arc
// through ABC and another arc through CBA, then B is not a junction because in
// both cases the adjacent point pairs are {A,C}. However, if there is an
// additional arc ABD, then {A,D} != {A,C}, and thus B becomes a junction.
//
// For a closed ring ABCA, the first point A’s adjacent points are the second
// and last point {B,C}. For a line, the first and last point are always
// considered junctions, even if the line is closed; this ensures that a closed
// line is never rotated.
var join = function(topology) {
  var coordinates = topology.coordinates,
      lines = topology.lines,
      rings = topology.rings,
      indexes = index(),
      visitedByIndex = new Int32Array(coordinates.length),
      leftByIndex = new Int32Array(coordinates.length),
      rightByIndex = new Int32Array(coordinates.length),
      junctionByIndex = new Int8Array(coordinates.length),
      junctionCount = 0, // upper bound on number of junctions
      i, n,
      previousIndex,
      currentIndex,
      nextIndex;

  for (i = 0, n = coordinates.length; i < n; ++i) {
    visitedByIndex[i] = leftByIndex[i] = rightByIndex[i] = -1;
  }

  for (i = 0, n = lines.length; i < n; ++i) {
    var line = lines[i],
        lineStart = line[0],
        lineEnd = line[1];
    currentIndex = indexes[lineStart];
    nextIndex = indexes[++lineStart];
    ++junctionCount, junctionByIndex[currentIndex] = 1; // start
    while (++lineStart <= lineEnd) {
      sequence(i, previousIndex = currentIndex, currentIndex = nextIndex, nextIndex = indexes[lineStart]);
    }
    ++junctionCount, junctionByIndex[nextIndex] = 1; // end
  }

  for (i = 0, n = coordinates.length; i < n; ++i) {
    visitedByIndex[i] = -1;
  }

  for (i = 0, n = rings.length; i < n; ++i) {
    var ring = rings[i],
        ringStart = ring[0] + 1,
        ringEnd = ring[1];
    previousIndex = indexes[ringEnd - 1];
    currentIndex = indexes[ringStart - 1];
    nextIndex = indexes[ringStart];
    sequence(i, previousIndex, currentIndex, nextIndex);
    while (++ringStart <= ringEnd) {
      sequence(i, previousIndex = currentIndex, currentIndex = nextIndex, nextIndex = indexes[ringStart]);
    }
  }

  function sequence(i, previousIndex, currentIndex, nextIndex) {
    if (visitedByIndex[currentIndex] === i) return; // ignore self-intersection
    visitedByIndex[currentIndex] = i;
    var leftIndex = leftByIndex[currentIndex];
    if (leftIndex >= 0) {
      var rightIndex = rightByIndex[currentIndex];
      if ((leftIndex !== previousIndex || rightIndex !== nextIndex)
        && (leftIndex !== nextIndex || rightIndex !== previousIndex)) {
        ++junctionCount, junctionByIndex[currentIndex] = 1;
      }
    } else {
      leftByIndex[currentIndex] = previousIndex;
      rightByIndex[currentIndex] = nextIndex;
    }
  }

  function index() {
    var indexByPoint = hashmap(coordinates.length * 1.4, hashIndex, equalIndex, Int32Array, -1, Int32Array),
        indexes = new Int32Array(coordinates.length);

    for (var i = 0, n = coordinates.length; i < n; ++i) {
      indexes[i] = indexByPoint.maybeSet(i, i);
    }

    return indexes;
  }

  function hashIndex(i) {
    return hashPoint(coordinates[i]);
  }

  function equalIndex(i, j) {
    return equalPoint(coordinates[i], coordinates[j]);
  }

  visitedByIndex = leftByIndex = rightByIndex = null;

  var junctionByPoint = hashset(junctionCount * 1.4, hashPoint, equalPoint), j;

  // Convert back to a standard hashset by point for caller convenience.
  for (i = 0, n = coordinates.length; i < n; ++i) {
    if (junctionByIndex[j = indexes[i]]) {
      junctionByPoint.add(coordinates[j]);
    }
  }

  return junctionByPoint;
};

// Given an extracted (pre-)topology, cuts (or rotates) arcs so that all shared
// point sequences are identified. The topology can then be subsequently deduped
// to remove exact duplicate arcs.
var cut = function(topology) {
  var junctions = join(topology),
      coordinates = topology.coordinates,
      lines = topology.lines,
      rings = topology.rings,
      next,
      i, n;

  for (i = 0, n = lines.length; i < n; ++i) {
    var line = lines[i],
        lineMid = line[0],
        lineEnd = line[1];
    while (++lineMid < lineEnd) {
      if (junctions.has(coordinates[lineMid])) {
        next = {0: lineMid, 1: line[1]};
        line[1] = lineMid;
        line = line.next = next;
      }
    }
  }

  for (i = 0, n = rings.length; i < n; ++i) {
    var ring = rings[i],
        ringStart = ring[0],
        ringMid = ringStart,
        ringEnd = ring[1],
        ringFixed = junctions.has(coordinates[ringStart]);
    while (++ringMid < ringEnd) {
      if (junctions.has(coordinates[ringMid])) {
        if (ringFixed) {
          next = {0: ringMid, 1: ring[1]};
          ring[1] = ringMid;
          ring = ring.next = next;
        } else { // For the first junction, we can rotate rather than cut.
          rotateArray(coordinates, ringStart, ringEnd, ringEnd - ringMid);
          coordinates[ringEnd] = coordinates[ringStart];
          ringFixed = true;
          ringMid = ringStart; // restart; we may have skipped junctions
        }
      }
    }
  }

  return topology;
};

function rotateArray(array, start, end, offset) {
  reverse$1(array, start, end);
  reverse$1(array, start, start + offset);
  reverse$1(array, start + offset, end);
}

function reverse$1(array, start, end) {
  for (var mid = start + ((end-- - start) >> 1), t; start < mid; ++start, --end) {
    t = array[start], array[start] = array[end], array[end] = t;
  }
}

// Given a cut topology, combines duplicate arcs.
var dedup$1 = function(topology) {
  var coordinates = topology.coordinates,
      lines = topology.lines, line,
      rings = topology.rings, ring,
      arcCount = lines.length + rings.length,
      i, n;

  delete topology.lines;
  delete topology.rings;

  // Count the number of (non-unique) arcs to initialize the hashmap safely.
  for (i = 0, n = lines.length; i < n; ++i) {
    line = lines[i]; while (line = line.next) ++arcCount;
  }
  for (i = 0, n = rings.length; i < n; ++i) {
    ring = rings[i]; while (ring = ring.next) ++arcCount;
  }

  var arcsByEnd = hashmap(arcCount * 2 * 1.4, hashPoint, equalPoint),
      arcs = topology.arcs = [];

  for (i = 0, n = lines.length; i < n; ++i) {
    line = lines[i];
    do {
      dedupLine(line);
    } while (line = line.next);
  }

  for (i = 0, n = rings.length; i < n; ++i) {
    ring = rings[i];
    if (ring.next) { // arc is no longer closed
      do {
        dedupLine(ring);
      } while (ring = ring.next);
    } else {
      dedupRing(ring);
    }
  }

  function dedupLine(arc) {
    var startPoint,
        endPoint,
        startArcs, startArc,
        endArcs, endArc,
        i, n;

    // Does this arc match an existing arc in order?
    if (startArcs = arcsByEnd.get(startPoint = coordinates[arc[0]])) {
      for (i = 0, n = startArcs.length; i < n; ++i) {
        startArc = startArcs[i];
        if (equalLine(startArc, arc)) {
          arc[0] = startArc[0];
          arc[1] = startArc[1];
          return;
        }
      }
    }

    // Does this arc match an existing arc in reverse order?
    if (endArcs = arcsByEnd.get(endPoint = coordinates[arc[1]])) {
      for (i = 0, n = endArcs.length; i < n; ++i) {
        endArc = endArcs[i];
        if (reverseEqualLine(endArc, arc)) {
          arc[1] = endArc[0];
          arc[0] = endArc[1];
          return;
        }
      }
    }

    if (startArcs) startArcs.push(arc); else arcsByEnd.set(startPoint, [arc]);
    if (endArcs) endArcs.push(arc); else arcsByEnd.set(endPoint, [arc]);
    arcs.push(arc);
  }

  function dedupRing(arc) {
    var endPoint,
        endArcs,
        endArc,
        i, n;

    // Does this arc match an existing line in order, or reverse order?
    // Rings are closed, so their start point and end point is the same.
    if (endArcs = arcsByEnd.get(endPoint = coordinates[arc[0]])) {
      for (i = 0, n = endArcs.length; i < n; ++i) {
        endArc = endArcs[i];
        if (equalRing(endArc, arc)) {
          arc[0] = endArc[0];
          arc[1] = endArc[1];
          return;
        }
        if (reverseEqualRing(endArc, arc)) {
          arc[0] = endArc[1];
          arc[1] = endArc[0];
          return;
        }
      }
    }

    // Otherwise, does this arc match an existing ring in order, or reverse order?
    if (endArcs = arcsByEnd.get(endPoint = coordinates[arc[0] + findMinimumOffset(arc)])) {
      for (i = 0, n = endArcs.length; i < n; ++i) {
        endArc = endArcs[i];
        if (equalRing(endArc, arc)) {
          arc[0] = endArc[0];
          arc[1] = endArc[1];
          return;
        }
        if (reverseEqualRing(endArc, arc)) {
          arc[0] = endArc[1];
          arc[1] = endArc[0];
          return;
        }
      }
    }

    if (endArcs) endArcs.push(arc); else arcsByEnd.set(endPoint, [arc]);
    arcs.push(arc);
  }

  function equalLine(arcA, arcB) {
    var ia = arcA[0], ib = arcB[0],
        ja = arcA[1], jb = arcB[1];
    if (ia - ja !== ib - jb) return false;
    for (; ia <= ja; ++ia, ++ib) if (!equalPoint(coordinates[ia], coordinates[ib])) return false;
    return true;
  }

  function reverseEqualLine(arcA, arcB) {
    var ia = arcA[0], ib = arcB[0],
        ja = arcA[1], jb = arcB[1];
    if (ia - ja !== ib - jb) return false;
    for (; ia <= ja; ++ia, --jb) if (!equalPoint(coordinates[ia], coordinates[jb])) return false;
    return true;
  }

  function equalRing(arcA, arcB) {
    var ia = arcA[0], ib = arcB[0],
        ja = arcA[1], jb = arcB[1],
        n = ja - ia;
    if (n !== jb - ib) return false;
    var ka = findMinimumOffset(arcA),
        kb = findMinimumOffset(arcB);
    for (var i = 0; i < n; ++i) {
      if (!equalPoint(coordinates[ia + (i + ka) % n], coordinates[ib + (i + kb) % n])) return false;
    }
    return true;
  }

  function reverseEqualRing(arcA, arcB) {
    var ia = arcA[0], ib = arcB[0],
        ja = arcA[1], jb = arcB[1],
        n = ja - ia;
    if (n !== jb - ib) return false;
    var ka = findMinimumOffset(arcA),
        kb = n - findMinimumOffset(arcB);
    for (var i = 0; i < n; ++i) {
      if (!equalPoint(coordinates[ia + (i + ka) % n], coordinates[jb - (i + kb) % n])) return false;
    }
    return true;
  }

  // Rings are rotated to a consistent, but arbitrary, start point.
  // This is necessary to detect when a ring and a rotated copy are dupes.
  function findMinimumOffset(arc) {
    var start = arc[0],
        end = arc[1],
        mid = start,
        minimum = mid,
        minimumPoint = coordinates[mid];
    while (++mid < end) {
      var point = coordinates[mid];
      if (point[0] < minimumPoint[0] || point[0] === minimumPoint[0] && point[1] < minimumPoint[1]) {
        minimum = mid;
        minimumPoint = point;
      }
    }
    return minimum - start;
  }

  return topology;
};

// Given an array of arcs in absolute (but already quantized!) coordinates,
// converts to fixed-point delta encoding.
// This is a destructive operation that modifies the given arcs!
var delta = function(arcs) {
  var i = -1,
      n = arcs.length;

  while (++i < n) {
    var arc = arcs[i],
        j = 0,
        k = 1,
        m = arc.length,
        point = arc[0],
        x0 = point[0],
        y0 = point[1],
        x1,
        y1;

    while (++j < m) {
      point = arc[j], x1 = point[0], y1 = point[1];
      if (x1 !== x0 || y1 !== y0) arc[k++] = [x1 - x0, y1 - y0], x0 = x1, y0 = y1;
    }

    if (k === 1) arc[k++] = [0, 0]; // Each arc must be an array of two or more positions.

    arc.length = k;
  }

  return arcs;
};

// Extracts the lines and rings from the specified hash of geometry objects.
//
// Returns an object with three properties:
//
// * coordinates - shared buffer of [x, y] coordinates
// * lines - lines extracted from the hash, of the form [start, end]
// * rings - rings extracted from the hash, of the form [start, end]
//
// For each ring or line, start and end represent inclusive indexes into the
// coordinates buffer. For rings (and closed lines), coordinates[start] equals
// coordinates[end].
//
// For each line or polygon geometry in the input hash, including nested
// geometries as in geometry collections, the `coordinates` array is replaced
// with an equivalent `arcs` array that, for each line (for line string
// geometries) or ring (for polygon geometries), points to one of the above
// lines or rings.
var extract = function(objects) {
  var index = -1,
      lines = [],
      rings = [],
      coordinates = [];

  function extractGeometry(geometry) {
    if (geometry && extractGeometryType.hasOwnProperty(geometry.type)) extractGeometryType[geometry.type](geometry);
  }

  var extractGeometryType = {
    GeometryCollection: function(o) { o.geometries.forEach(extractGeometry); },
    LineString: function(o) { o.arcs = extractLine(o.arcs); },
    MultiLineString: function(o) { o.arcs = o.arcs.map(extractLine); },
    Polygon: function(o) { o.arcs = o.arcs.map(extractRing); },
    MultiPolygon: function(o) { o.arcs = o.arcs.map(extractMultiRing); }
  };

  function extractLine(line) {
    for (var i = 0, n = line.length; i < n; ++i) coordinates[++index] = line[i];
    var arc = {0: index - n + 1, 1: index};
    lines.push(arc);
    return arc;
  }

  function extractRing(ring) {
    for (var i = 0, n = ring.length; i < n; ++i) coordinates[++index] = ring[i];
    var arc = {0: index - n + 1, 1: index};
    rings.push(arc);
    return arc;
  }

  function extractMultiRing(rings) {
    return rings.map(extractRing);
  }

  for (var key in objects) {
    extractGeometry(objects[key]);
  }

  return {
    type: "Topology",
    coordinates: coordinates,
    lines: lines,
    rings: rings,
    objects: objects
  };
};

// Given a hash of GeoJSON objects, returns a hash of GeoJSON geometry objects.
// Any null input geometry objects are represented as {type: null} in the output.
// Any feature.{id,properties,bbox} are transferred to the output geometry object.
// Each output geometry object is a shallow copy of the input (e.g., properties, coordinates)!
var geometry$1 = function(inputs) {
  var outputs = {}, key;
  for (key in inputs) outputs[key] = geomifyObject(inputs[key]);
  return outputs;
};

function geomifyObject(input) {
  return input == null ? {type: null}
      : (input.type === "FeatureCollection" ? geomifyFeatureCollection
      : input.type === "Feature" ? geomifyFeature
      : geomifyGeometry)(input);
}

function geomifyFeatureCollection(input) {
  var output = {type: "GeometryCollection", geometries: input.features.map(geomifyFeature)};
  if (input.bbox != null) output.bbox = input.bbox;
  return output;
}

function geomifyFeature(input) {
  var output = geomifyGeometry(input.geometry), key; // eslint-disable-line no-unused-vars
  if (input.id != null) output.id = input.id;
  if (input.bbox != null) output.bbox = input.bbox;
  for (key in input.properties) { output.properties = input.properties; break; }
  return output;
}

function geomifyGeometry(input) {
  if (input == null) return {type: null};
  var output = input.type === "GeometryCollection" ? {type: "GeometryCollection", geometries: input.geometries.map(geomifyGeometry)}
      : input.type === "Point" || input.type === "MultiPoint" ? {type: input.type, coordinates: input.coordinates}
      : {type: input.type, arcs: input.coordinates}; // TODO Check for unknown types?
  if (input.bbox != null) output.bbox = input.bbox;
  return output;
}

var prequantize = function(objects, bbox, n) {
  var x0 = bbox[0],
      y0 = bbox[1],
      x1 = bbox[2],
      y1 = bbox[3],
      kx = x1 - x0 ? (n - 1) / (x1 - x0) : 1,
      ky = y1 - y0 ? (n - 1) / (y1 - y0) : 1;

  function quantizePoint(input) {
    return [Math.round((input[0] - x0) * kx), Math.round((input[1] - y0) * ky)];
  }

  function quantizePoints(input, m) {
    var i = -1,
        j = 0,
        n = input.length,
        output = new Array(n), // pessimistic
        pi,
        px,
        py,
        x,
        y;

    while (++i < n) {
      pi = input[i];
      x = Math.round((pi[0] - x0) * kx);
      y = Math.round((pi[1] - y0) * ky);
      if (x !== px || y !== py) output[j++] = [px = x, py = y]; // non-coincident points
    }

    output.length = j;
    while (j < m) j = output.push([output[0][0], output[0][1]]);
    return output;
  }

  function quantizeLine(input) {
    return quantizePoints(input, 2);
  }

  function quantizeRing(input) {
    return quantizePoints(input, 4);
  }

  function quantizePolygon(input) {
    return input.map(quantizeRing);
  }

  function quantizeGeometry(o) {
    if (o != null && quantizeGeometryType.hasOwnProperty(o.type)) quantizeGeometryType[o.type](o);
  }

  var quantizeGeometryType = {
    GeometryCollection: function(o) { o.geometries.forEach(quantizeGeometry); },
    Point: function(o) { o.coordinates = quantizePoint(o.coordinates); },
    MultiPoint: function(o) { o.coordinates = o.coordinates.map(quantizePoint); },
    LineString: function(o) { o.arcs = quantizeLine(o.arcs); },
    MultiLineString: function(o) { o.arcs = o.arcs.map(quantizeLine); },
    Polygon: function(o) { o.arcs = quantizePolygon(o.arcs); },
    MultiPolygon: function(o) { o.arcs = o.arcs.map(quantizePolygon); }
  };

  for (var key in objects) {
    quantizeGeometry(objects[key]);
  }

  return {
    scale: [1 / kx, 1 / ky],
    translate: [x0, y0]
  };
};

// Constructs the TopoJSON Topology for the specified hash of features.
// Each object in the specified hash must be a GeoJSON object,
// meaning FeatureCollection, a Feature or a geometry object.
var topology = function(objects, quantization) {
  var bbox = bounds(objects = geometry$1(objects)),
      transform = quantization > 0 && bbox && prequantize(objects, bbox, quantization),
      topology = dedup$1(cut(extract(objects))),
      coordinates = topology.coordinates,
      indexByArc = hashmap(topology.arcs.length * 1.4, hashArc, equalArc);

  objects = topology.objects; // for garbage collection
  topology.bbox = bbox;
  topology.arcs = topology.arcs.map(function(arc, i) {
    indexByArc.set(arc, i);
    return coordinates.slice(arc[0], arc[1] + 1);
  });

  delete topology.coordinates;
  coordinates = null;

  function indexGeometry(geometry) {
    if (geometry && indexGeometryType.hasOwnProperty(geometry.type)) indexGeometryType[geometry.type](geometry);
  }

  var indexGeometryType = {
    GeometryCollection: function(o) { o.geometries.forEach(indexGeometry); },
    LineString: function(o) { o.arcs = indexArcs(o.arcs); },
    MultiLineString: function(o) { o.arcs = o.arcs.map(indexArcs); },
    Polygon: function(o) { o.arcs = o.arcs.map(indexArcs); },
    MultiPolygon: function(o) { o.arcs = o.arcs.map(indexMultiArcs); }
  };

  function indexArcs(arc) {
    var indexes = [];
    do {
      var index = indexByArc.get(arc);
      indexes.push(arc[0] < arc[1] ? index : ~index);
    } while (arc = arc.next);
    return indexes;
  }

  function indexMultiArcs(arcs) {
    return arcs.map(indexArcs);
  }

  for (var key in objects) {
    indexGeometry(objects[key]);
  }

  if (transform) {
    topology.transform = transform;
    topology.arcs = delta(topology.arcs);
  }

  return topology;
};

function hashArc(arc) {
  var i = arc[0], j = arc[1], t;
  if (j < i) t = i, i = j, j = t;
  return i + 31 * j;
}

function equalArc(arcA, arcB) {
  var ia = arcA[0], ja = arcA[1],
      ib = arcB[0], jb = arcB[1], t;
  if (ja < ia) t = ia, ia = ja, ja = t;
  if (jb < ib) t = ib, ib = jb, jb = t;
  return ia === ib && ja === jb;
}

/**
 * Merges all connected (non-forking, non-junctioning) line strings into single lineStrings.
 * [LineString] -> LineString|MultiLineString
 *
 * @param {FeatureCollection<LineString|MultiLineString>} geojson Lines to dissolve
 * @param {Object} [options={}] Optional parameters
 * @param {boolean} [options.mutate=false] Prevent input mutation
 * @returns {Feature<LineString|MultiLineString>} Dissolved lines
 */
function lineDissolve(geojson, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var mutate = options.mutate;

    // Validation
    if (getType(geojson) !== 'FeatureCollection') throw new Error('geojson must be a FeatureCollection');
    if (!geojson.features.length) throw new Error('geojson is empty');

    // Clone geojson to avoid side effects
    if (mutate === false || mutate === undefined) geojson = clone(geojson);

    var result = [];
    var lastLine = lineReduce(geojson, function (previousLine, currentLine) {
        // Attempt to merge this LineString with the other LineStrings, updating
        // the reference as it is merged with others and grows.
        var merged = mergeLineStrings(previousLine, currentLine);

        // Accumulate the merged LineString
        if (merged) return merged;

        // Put the unmerged LineString back into the list
        else {
            result.push(previousLine);
            return currentLine;
        }
    });
    // Append the last line
    if (lastLine) result.push(lastLine);

    // Return null if no lines were dissolved
    if (!result.length) return null;
    // Return LineString if only 1 line was dissolved
    else if (result.length === 1) return result[0];
    // Return MultiLineString if multiple lines were dissolved with gaps
    else return multiLineString(result.map(function (line) { return line.coordinates; }));
}

// [Number, Number] -> String
function coordId(coord) {
    return coord[0].toString() + ',' + coord[1].toString();
}

/**
 * LineString, LineString -> LineString
 *
 * @private
 * @param {Feature<LineString>} a line1
 * @param {Feature<LineString>} b line2
 * @returns {Feature<LineString>|null} Merged LineString
 */
function mergeLineStrings(a, b) {
    var coords1 = a.geometry.coordinates;
    var coords2 = b.geometry.coordinates;

    var s1 = coordId(coords1[0]);
    var e1 = coordId(coords1[coords1.length - 1]);
    var s2 = coordId(coords2[0]);
    var e2 = coordId(coords2[coords2.length - 1]);

    // TODO: handle case where more than one of these is true!
    var coords;
    if (s1 === e2) coords = coords2.concat(coords1.slice(1));
    else if (s2 === e1) coords = coords1.concat(coords2.slice(1));
    else if (s1 === s2) coords = coords1.slice(1).reverse().concat(coords2);
    else if (e1 === e2) coords = coords1.concat(coords2.reverse().slice(1));
    else return null;

    return lineString(coords);
}

/**
 * Dissolves all overlapping (Multi)Polygon
 *
 * @param {FeatureCollection<Polygon|MultiPolygon>} geojson Polygons to dissolve
 * @param {Object} [options={}] Optional parameters
 * @param {boolean} [options.mutate=false] Prevent input mutation
 * @returns {Feature<Polygon|MultiPolygon>} Dissolved Polygons
 */
function polygonDissolve(geojson, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var mutate = options.mutate;

    // Validation
    if (getType(geojson) !== 'FeatureCollection') throw new Error('geojson must be a FeatureCollection');
    if (!geojson.features.length) throw new Error('geojson is empty');

    // Clone geojson to avoid side effects
    // Topojson modifies in place, so we need to deep clone first
    if (mutate === false || mutate === undefined) geojson = clone(geojson);

    var geoms = [];
    flattenEach(geojson, function (feature$$1) {
        geoms.push(feature$$1.geometry);
    });
    var topo = topology({geoms: geometryCollection(geoms).geometry});
    return merge(topo, topo.objects.geoms.geometries);
}

/**
 * Transform function: attempts to dissolve geojson objects where possible
 * [GeoJSON] -> GeoJSON geometry
 *
 * @private
 * @param {FeatureCollection<LineString|MultiLineString|Polygon|MultiPolygon>} geojson Features to dissolved
 * @param {Object} [options={}] Optional parameters
 * @param {boolean} [options.mutate=false] Prevent input mutation
 * @returns {Feature<MultiLineString|MultiPolygon>} Dissolved Features
 */
function dissolve(geojson, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var mutate = options.mutate;

    // Validation
    if (getType(geojson) !== 'FeatureCollection') throw new Error('geojson must be a FeatureCollection');
    if (!geojson.features.length) throw new Error('geojson is empty');

    // Clone geojson to avoid side effects
    // Topojson modifies in place, so we need to deep clone first
    if (mutate === false || mutate === undefined) geojson = clone(geojson);

    // Assert homogenity
    var type = getHomogenousType(geojson);
    if (!type) throw new Error('geojson must be homogenous');

    switch (type) {
    case 'LineString':
        return lineDissolve(geojson, options);
    case 'Polygon':
        return polygonDissolve(geojson, options);
    default:
        throw new Error(type + ' is not supported');
    }
}

/**
 * Checks if GeoJSON is Homogenous
 *
 * @private
 * @param {GeoJSON} geojson GeoJSON
 * @returns {string|null} Homogenous type or null if multiple types
 */
function getHomogenousType(geojson) {
    var types = {};
    flattenEach(geojson, function (feature$$1) {
        types[feature$$1.geometry.type] = true;
    });
    var keys = Object.keys(types);
    if (keys.length === 1) return keys[0];
    return null;
}

/**
 * Takes a set of {@link Point|points} and returns a concave hull Polygon or MultiPolygon.
 * Internally, this uses [turf-tin](https://github.com/Turfjs/turf-tin) to generate geometries.
 *
 * @name concave
 * @param {FeatureCollection<Point>} points input points
 * @param {Object} [options={}] Optional parameters
 * @param {number} [options.maxEdge=Infinity] the length (in 'units') of an edge necessary for part of the hull to become concave.
 * @param {string} [options.units='kilometers'] can be degrees, radians, miles, or kilometers
 * @returns {Feature<(Polygon|MultiPolygon)>|null} a concave hull (null value is returned if unable to compute hull)
 * @example
 * var points = turf.featureCollection([
 *   turf.point([-63.601226, 44.642643]),
 *   turf.point([-63.591442, 44.651436]),
 *   turf.point([-63.580799, 44.648749]),
 *   turf.point([-63.573589, 44.641788]),
 *   turf.point([-63.587665, 44.64533]),
 *   turf.point([-63.595218, 44.64765])
 * ]);
 * var options = {units: 'miles', maxEdge: 1};
 *
 * var hull = turf.concave(points, options);
 *
 * //addToMap
 * var addToMap = [points, hull]
 */
function concave(points$$1, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');

    // validation
    if (!points$$1) throw new Error('points is required');
    var maxEdge = options.maxEdge || Infinity;
    if (!isNumber(maxEdge)) throw new Error('maxEdge is invalid');

    var cleaned = removeDuplicates(points$$1);

    var tinPolys = tin(cleaned);
    // calculate length of all edges and area of all triangles
    // and remove triangles that fail the max length test
    tinPolys.features = tinPolys.features.filter(function (triangle) {
        var pt1 = triangle.geometry.coordinates[0][0];
        var pt2 = triangle.geometry.coordinates[0][1];
        var pt3 = triangle.geometry.coordinates[0][2];
        var dist1 = distance(pt1, pt2, options);
        var dist2 = distance(pt2, pt3, options);
        var dist3 = distance(pt1, pt3, options);
        return (dist1 <= maxEdge && dist2 <= maxEdge && dist3 <= maxEdge);
    });

    if (tinPolys.features.length < 1) return null;

    // merge the adjacent triangles
    var dissolved = dissolve(tinPolys, options);

    // geojson-dissolve always returns a MultiPolygon
    if (dissolved.coordinates.length === 1) {
        dissolved.coordinates = dissolved.coordinates[0];
        dissolved.type = 'Polygon';
    }
    return feature(dissolved);
}

/**
 * Removes duplicated points in a collection returning a new collection
 *
 * @private
 * @param {FeatureCollection<Point>} points to be cleaned
 * @returns {FeatureCollection<Point>} cleaned set of points
 */
function removeDuplicates(points$$1) {
    var cleaned = [];
    var existing = {};

    featureEach(points$$1, function (pt) {
        if (!pt.geometry) return;
        var key = pt.geometry.coordinates.join('-');
        if (!existing.hasOwnProperty(key)) {
            cleaned.push(pt);
            existing[key] = true;
        }
    });
    return featureCollection(cleaned);
}

/**
 * Merges a specified property from a FeatureCollection of points into a
 * FeatureCollection of polygons. Given an `inProperty` on points and an `outProperty`
 * for polygons, this finds every point that lies within each polygon, collects the
 * `inProperty` values from those points, and adds them as an array to `outProperty`
 * on the polygon.
 *
 * @name collect
 * @param {FeatureCollection<Polygon>} polygons polygons with values on which to aggregate
 * @param {FeatureCollection<Point>} points points to be aggregated
 * @param {string} inProperty property to be nested from
 * @param {string} outProperty property to be nested into
 * @returns {FeatureCollection<Polygon>} polygons with properties listed based on `outField`
 * @example
 * var poly1 = turf.polygon([[[0,0],[10,0],[10,10],[0,10],[0,0]]]);
 * var poly2 = turf.polygon([[[10,0],[20,10],[20,20],[20,0],[10,0]]]);
 * var polyFC = turf.featureCollection([poly1, poly2]);
 * var pt1 = turf.point([5,5], {population: 200});
 * var pt2 = turf.point([1,3], {population: 600});
 * var pt3 = turf.point([14,2], {population: 100});
 * var pt4 = turf.point([13,1], {population: 200});
 * var pt5 = turf.point([19,7], {population: 300});
 * var pointFC = turf.featureCollection([pt1, pt2, pt3, pt4, pt5]);
 * var collected = turf.collect(polyFC, pointFC, 'population', 'values');
 * var values = collected.features[0].properties.values
 * //=values => [200, 600]
 *
 * //addToMap
 * var addToMap = [pointFC, collected]
 */
function collect(polygons, points, inProperty, outProperty) {
    var rtree = rbush_1(6);

    var treeItems = points.features.map(function (item) {
        return {
            minX: item.geometry.coordinates[0],
            minY: item.geometry.coordinates[1],
            maxX: item.geometry.coordinates[0],
            maxY: item.geometry.coordinates[1],
            property: item.properties[inProperty]
        };
    });

    rtree.load(treeItems);
    polygons.features.forEach(function (poly) {

        if (!poly.properties) {
            poly.properties = {};
        }
        var bbox$$1 = bbox(poly);
        var potentialPoints = rtree.search({minX: bbox$$1[0], minY: bbox$$1[1], maxX: bbox$$1[2], maxY: bbox$$1[3]});
        var values = [];
        potentialPoints.forEach(function (pt) {
            if (booleanPointInPolygon([pt.minX, pt.minY], poly)) {
                values.push(pt.property);
            }
        });

        poly.properties[outProperty] = values;
    });

    return polygons;
}

/**
 * Takes input features and flips all of their coordinates from `[x, y]` to `[y, x]`.
 *
 * @name flip
 * @param {GeoJSON} geojson input features
 * @param {Object} [options={}] Optional parameters
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated (significant performance increase if true)
 * @returns {GeoJSON} a feature or set of features of the same type as `input` with flipped coordinates
 * @example
 * var serbia = turf.point([20.566406, 43.421008]);
 *
 * var saudiArabia = turf.flip(serbia);
 *
 * //addToMap
 * var addToMap = [serbia, saudiArabia];
 */
function flip(geojson, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var mutate = options.mutate;

    if (!geojson) throw new Error('geojson is required');
    // ensure that we don't modify features in-place and changes to the
    // output do not change the previous feature, including changes to nested
    // properties.
    if (mutate === false || mutate === undefined) geojson = clone(geojson);

    coordEach(geojson, function (coord) {
        var x = coord[0];
        var y = coord[1];
        coord[0] = y;
        coord[1] = x;
    });
    return geojson;
}

/**
 * Removes redundant coordinates from any GeoJSON Geometry.
 *
 * @name cleanCoords
 * @param {Geometry|Feature} geojson Feature or Geometry
 * @param {Object} [options={}] Optional parameters
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated
 * @returns {Geometry|Feature} the cleaned input Feature/Geometry
 * @example
 * var line = turf.lineString([[0, 0], [0, 2], [0, 5], [0, 8], [0, 8], [0, 10]]);
 * var multiPoint = turf.multiPoint([[0, 0], [0, 0], [2, 2]]);
 *
 * turf.cleanCoords(line).geometry.coordinates;
 * //= [[0, 0], [0, 10]]
 *
 * turf.cleanCoords(multiPoint).geometry.coordinates;
 * //= [[0, 0], [2, 2]]
 */
function cleanCoords(geojson, options) {
    // Backwards compatible with v4.0
    var mutate = (typeof options === 'object') ? options.mutate : options;
    if (!geojson) throw new Error('geojson is required');
    var type = getType(geojson);

    // Store new "clean" points in this Array
    var newCoords = [];

    switch (type) {
    case 'LineString':
        newCoords = cleanLine(geojson);
        break;
    case 'MultiLineString':
    case 'Polygon':
        getCoords(geojson).forEach(function (line) {
            newCoords.push(cleanLine(line));
        });
        break;
    case 'MultiPolygon':
        getCoords(geojson).forEach(function (polygons$$1) {
            var polyPoints = [];
            polygons$$1.forEach(function (ring) {
                polyPoints.push(cleanLine(ring));
            });
            newCoords.push(polyPoints);
        });
        break;
    case 'Point':
        return geojson;
    case 'MultiPoint':
        var existing = {};
        getCoords(geojson).forEach(function (coord) {
            var key = coord.join('-');
            if (!existing.hasOwnProperty(key)) {
                newCoords.push(coord);
                existing[key] = true;
            }
        });
        break;
    default:
        throw new Error(type + ' geometry not supported');
    }

    // Support input mutation
    if (geojson.coordinates) {
        if (mutate === true) {
            geojson.coordinates = newCoords;
            return geojson;
        }
        return {type: type, coordinates: newCoords};
    } else {
        if (mutate === true) {
            geojson.geometry.coordinates = newCoords;
            return geojson;
        }
        return feature({type: type, coordinates: newCoords}, geojson.properties, geojson.bbox, geojson.id);
    }
}

/**
 * Clean Coords
 *
 * @private
 * @param {Array<number>|LineString} line Line
 * @returns {Array<number>} Cleaned coordinates
 */
function cleanLine(line) {
    var points$$1 = getCoords(line);
    // handle "clean" segment
    if (points$$1.length === 2 && !equals(points$$1[0], points$$1[1])) return points$$1;

    var prevPoint, point$$1, nextPoint;
    var newPoints = [];
    var secondToLast = points$$1.length - 1;

    newPoints.push(points$$1[0]);
    for (var i = 1; i < secondToLast; i++) {
        prevPoint = points$$1[i - 1];
        point$$1 = points$$1[i];
        nextPoint = points$$1[i + 1];

        if (!isPointOnLineSegment(prevPoint, nextPoint, point$$1)) {
            newPoints.push(point$$1);
        }
    }
    newPoints.push(nextPoint);
    return newPoints;
}

/**
 * Compares two points and returns if they are equals
 *
 * @private
 * @param {Position} pt1 point
 * @param {Position} pt2 point
 * @returns {boolean} true if they are equals
 */
function equals(pt1, pt2) {
    return pt1[0] === pt2[0] && pt1[1] === pt2[1];
}

/**
 * Returns if `point` is on the segment between `start` and `end`.
 * Borrowed from `@turf/boolean-point-on-line` to speed up the evaluation (instead of using the module as dependency)
 *
 * @private
 * @param {Position} start coord pair of start of line
 * @param {Position} end coord pair of end of line
 * @param {Position} point coord pair of point to check
 * @returns {boolean} true/false
 */
function isPointOnLineSegment(start, end, point$$1) {
    var x = point$$1[0], y = point$$1[1];
    var startX = start[0], startY = start[1];
    var endX = end[0], endY = end[1];

    var dxc = x - startX;
    var dyc = y - startY;
    var dxl = endX - startX;
    var dyl = endY - startY;
    var cross = dxc * dyl - dyc * dxl;

    if (cross !== 0) return false;
    else if (Math.abs(dxl) >= Math.abs(dyl)) return dxl > 0 ? startX <= x && x <= endX : endX <= x && x <= startX;
    else return dyl > 0 ? startY <= y && y <= endY : endY <= y && y <= startY;
}

/*
 (c) 2013, Vladimir Agafonkin
 Simplify.js, a high-performance JS polyline simplification library
 mourner.github.io/simplify-js
*/

// to suit your point format, run search/replace for '.x' and '.y';
// for 3D version, see 3d branch (configurability would draw significant performance overhead)

// square distance between 2 points
function getSqDist$1(p1, p2) {

    var dx = p1.x - p2.x,
        dy = p1.y - p2.y;

    return dx * dx + dy * dy;
}

// square distance from a point to a segment
function getSqSegDist(p, p1, p2) {

    var x = p1.x,
        y = p1.y,
        dx = p2.x - x,
        dy = p2.y - y;

    if (dx !== 0 || dy !== 0) {

        var t = ((p.x - x) * dx + (p.y - y) * dy) / (dx * dx + dy * dy);

        if (t > 1) {
            x = p2.x;
            y = p2.y;

        } else if (t > 0) {
            x += dx * t;
            y += dy * t;
        }
    }

    dx = p.x - x;
    dy = p.y - y;

    return dx * dx + dy * dy;
}
// rest of the code doesn't care about point format

// basic distance-based simplification
function simplifyRadialDist(points$$1, sqTolerance) {

    var prevPoint = points$$1[0],
        newPoints = [prevPoint],
        point$$1;

    for (var i = 1, len = points$$1.length; i < len; i++) {
        point$$1 = points$$1[i];

        if (getSqDist$1(point$$1, prevPoint) > sqTolerance) {
            newPoints.push(point$$1);
            prevPoint = point$$1;
        }
    }

    if (prevPoint !== point$$1) newPoints.push(point$$1);

    return newPoints;
}

function simplifyDPStep(points$$1, first, last, sqTolerance, simplified) {
    var maxSqDist = sqTolerance,
        index;

    for (var i = first + 1; i < last; i++) {
        var sqDist = getSqSegDist(points$$1[i], points$$1[first], points$$1[last]);

        if (sqDist > maxSqDist) {
            index = i;
            maxSqDist = sqDist;
        }
    }

    if (maxSqDist > sqTolerance) {
        if (index - first > 1) simplifyDPStep(points$$1, first, index, sqTolerance, simplified);
        simplified.push(points$$1[index]);
        if (last - index > 1) simplifyDPStep(points$$1, index, last, sqTolerance, simplified);
    }
}

// simplification using Ramer-Douglas-Peucker algorithm
function simplifyDouglasPeucker(points$$1, sqTolerance) {
    var last = points$$1.length - 1;

    var simplified = [points$$1[0]];
    simplifyDPStep(points$$1, 0, last, sqTolerance, simplified);
    simplified.push(points$$1[last]);

    return simplified;
}

// both algorithms combined for awesome performance
function simplify$2(points$$1, tolerance, highestQuality) {

    if (points$$1.length <= 2) return points$$1;

    var sqTolerance = tolerance !== undefined ? tolerance * tolerance : 1;

    points$$1 = highestQuality ? points$$1 : simplifyRadialDist(points$$1, sqTolerance);
    points$$1 = simplifyDouglasPeucker(points$$1, sqTolerance);

    return points$$1;
}

/**
 * Takes a {@link GeoJSON} object and returns a simplified version. Internally uses
 * [simplify-js](http://mourner.github.io/simplify-js/) to perform simplification using the Ramer-Douglas-Peucker algorithm.
 *
 * @name simplify
 * @param {GeoJSON} geojson object to be simplified
 * @param {Object} [options={}] Optional parameters
 * @param {number} [options.tolerance=1] simplification tolerance
 * @param {boolean} [options.highQuality=false] whether or not to spend more time to create a higher-quality simplification with a different algorithm
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated (significant performance increase if true)
 * @returns {GeoJSON} a simplified GeoJSON
 * @example
 * var geojson = turf.polygon([[
 *   [-70.603637, -33.399918],
 *   [-70.614624, -33.395332],
 *   [-70.639343, -33.392466],
 *   [-70.659942, -33.394759],
 *   [-70.683975, -33.404504],
 *   [-70.697021, -33.419406],
 *   [-70.701141, -33.434306],
 *   [-70.700454, -33.446339],
 *   [-70.694274, -33.458369],
 *   [-70.682601, -33.465816],
 *   [-70.668869, -33.472117],
 *   [-70.646209, -33.473835],
 *   [-70.624923, -33.472117],
 *   [-70.609817, -33.468107],
 *   [-70.595397, -33.458369],
 *   [-70.587158, -33.442901],
 *   [-70.587158, -33.426283],
 *   [-70.590591, -33.414248],
 *   [-70.594711, -33.406224],
 *   [-70.603637, -33.399918]
 * ]]);
 * var options = {tolerance: 0.01, highQuality: false};
 * var simplified = turf.simplify(geojson, options);
 *
 * //addToMap
 * var addToMap = [geojson, simplified]
 */
function simplify(geojson, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var tolerance = options.tolerance !== undefined ? options.tolerance : 1;
    var highQuality = options.highQuality || false;
    var mutate = options.mutate || false;

    if (!geojson) throw new Error('geojson is required');
    if (tolerance && tolerance < 0) throw new Error('invalid tolerance');

    // Clone geojson to avoid side effects
    if (mutate !== true) geojson = clone(geojson);

    geomEach(geojson, function (geom) {
        simplifyGeom(geom, tolerance, highQuality);
    });
    return geojson;
}

/**
 * Simplifies a feature's coordinates
 *
 * @private
 * @param {Geometry} geometry to be simplified
 * @param {number} [tolerance=1] simplification tolerance
 * @param {boolean} [highQuality=false] whether or not to spend more time to create a higher-quality simplification with a different algorithm
 * @returns {Geometry} output
 */
function simplifyGeom(geometry$$1, tolerance, highQuality) {
    var type = geometry$$1.type;

    // "unsimplyfiable" geometry types
    if (type === 'Point' || type === 'MultiPoint') return geometry$$1;

    // Remove any extra coordinates
    cleanCoords(geometry$$1, true);

    var coordinates = geometry$$1.coordinates;
    switch (type) {
    case 'LineString':
        geometry$$1['coordinates'] = simplifyLine(coordinates, tolerance, highQuality);
        break;
    case 'MultiLineString':
        geometry$$1['coordinates'] = coordinates.map(function (lines) {
            return simplifyLine(lines, tolerance, highQuality);
        });
        break;
    case 'Polygon':
        geometry$$1['coordinates'] = simplifyPolygon(coordinates, tolerance, highQuality);
        break;
    case 'MultiPolygon':
        geometry$$1['coordinates'] = coordinates.map(function (rings) {
            return simplifyPolygon(rings, tolerance, highQuality);
        });
    }
    return geometry$$1;
}


/**
 * Simplifies the coordinates of a LineString with simplify-js
 *
 * @private
 * @param {Array<number>} coordinates to be processed
 * @param {number} tolerance simplification tolerance
 * @param {boolean} highQuality whether or not to spend more time to create a higher-quality
 * @returns {Array<Array<number>>} simplified coords
 */
function simplifyLine(coordinates, tolerance, highQuality) {
    return simplify$2(coordinates.map(function (coord) {
        return {x: coord[0], y: coord[1], z: coord[2]};
    }), tolerance, highQuality).map(function (coords) {
        return (coords.z) ? [coords.x, coords.y, coords.z] : [coords.x, coords.y];
    });
}


/**
 * Simplifies the coordinates of a Polygon with simplify-js
 *
 * @private
 * @param {Array<number>} coordinates to be processed
 * @param {number} tolerance simplification tolerance
 * @param {boolean} highQuality whether or not to spend more time to create a higher-quality
 * @returns {Array<Array<Array<number>>>} simplified coords
 */
function simplifyPolygon(coordinates, tolerance, highQuality) {
    return coordinates.map(function (ring) {
        var pts = ring.map(function (coord) {
            return {x: coord[0], y: coord[1]};
        });
        if (pts.length < 4) {
            throw new Error('invalid polygon');
        }
        var simpleRing = simplify$2(pts, tolerance, highQuality).map(function (coords) {
            return [coords.x, coords.y];
        });
        //remove 1 percent of tolerance until enough points to make a triangle
        while (!checkValidity(simpleRing)) {
            tolerance -= tolerance * 0.01;
            simpleRing = simplify$2(pts, tolerance, highQuality).map(function (coords) {
                return [coords.x, coords.y];
            });
        }
        if (
            (simpleRing[simpleRing.length - 1][0] !== simpleRing[0][0]) ||
            (simpleRing[simpleRing.length - 1][1] !== simpleRing[0][1])) {
            simpleRing.push(simpleRing[0]);
        }
        return simpleRing;
    });
}


/**
 * Returns true if ring has at least 3 coordinates and its first coordinate is the same as its last
 *
 * @private
 * @param {Array<number>} ring coordinates to be checked
 * @returns {boolean} true if valid
 */
function checkValidity(ring) {
    if (ring.length < 3) return false;
    //if the last point is the same as the first, it's not a triangle
    return !(ring.length === 3 && ((ring[2][0] === ring[0][0]) && (ring[2][1] === ring[0][1])));
}

/* eslint-disable */

 /**
   * BezierSpline
   * https://github.com/leszekr/bezier-spline-js
   *
   * @private
   * @copyright
   * Copyright (c) 2013 Leszek Rybicki
   *
   * Permission is hereby granted, free of charge, to any person obtaining a copy
   * of this software and associated documentation files (the "Software"), to deal
   * in the Software without restriction, including without limitation the rights
   * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
   * copies of the Software, and to permit persons to whom the Software is
   * furnished to do so, subject to the following conditions:
   *
   * The above copyright notice and this permission notice shall be included in all
   * copies or substantial portions of the Software.
   *
   * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
   * SOFTWARE.
   */
var Spline = function (options) {
    this.points = options.points || [];
    this.duration = options.duration || 10000;
    this.sharpness = options.sharpness || 0.85;
    this.centers = [];
    this.controls = [];
    this.stepLength = options.stepLength || 60;
    this.length = this.points.length;
    this.delay = 0;
    // this is to ensure compatibility with the 2d version
    for (var i = 0; i < this.length; i++) this.points[i].z = this.points[i].z || 0;
    for (var i = 0; i < this.length - 1; i++) {
        var p1 = this.points[i];
        var p2 = this.points[i + 1];
        this.centers.push({
            x: (p1.x + p2.x) / 2,
            y: (p1.y + p2.y) / 2,
            z: (p1.z + p2.z) / 2
        });
    }
    this.controls.push([this.points[0], this.points[0]]);
    for (var i = 0; i < this.centers.length - 1; i++) {
        var p1 = this.centers[i];
        var p2 = this.centers[i + 1];
        var dx = this.points[i + 1].x - (this.centers[i].x + this.centers[i + 1].x) / 2;
        var dy = this.points[i + 1].y - (this.centers[i].y + this.centers[i + 1].y) / 2;
        var dz = this.points[i + 1].z - (this.centers[i].y + this.centers[i + 1].z) / 2;
        this.controls.push([{
            x: (1.0 - this.sharpness) * this.points[i + 1].x + this.sharpness * (this.centers[i].x + dx),
            y: (1.0 - this.sharpness) * this.points[i + 1].y + this.sharpness * (this.centers[i].y + dy),
            z: (1.0 - this.sharpness) * this.points[i + 1].z + this.sharpness * (this.centers[i].z + dz)},
            {
                x: (1.0 - this.sharpness) * this.points[i + 1].x + this.sharpness * (this.centers[i + 1].x + dx),
                y: (1.0 - this.sharpness) * this.points[i + 1].y + this.sharpness * (this.centers[i + 1].y + dy),
                z: (1.0 - this.sharpness) * this.points[i + 1].z + this.sharpness * (this.centers[i + 1].z + dz)}]);
    }
    this.controls.push([this.points[this.length - 1], this.points[this.length - 1]]);
    this.steps = this.cacheSteps(this.stepLength);
    return this;
};

  /*
    Caches an array of equidistant (more or less) points on the curve.
  */
Spline.prototype.cacheSteps = function (mindist) {
    var steps = [];
    var laststep = this.pos(0);
    steps.push(0);
    for (var t = 0; t < this.duration; t += 10) {
        var step = this.pos(t);
        var dist = Math.sqrt((step.x - laststep.x) * (step.x - laststep.x) + (step.y - laststep.y) * (step.y - laststep.y) + (step.z - laststep.z) * (step.z - laststep.z));
        if (dist > mindist) {
            steps.push(t);
            laststep = step;
        }
    }
    return steps;
};

  /*
    returns angle and speed in the given point in the curve
  */
Spline.prototype.vector = function (t) {
    var p1 = this.pos(t + 10);
    var p2 = this.pos(t - 10);
    return {
        angle:180 * Math.atan2(p1.y - p2.y, p1.x - p2.x) / 3.14,
        speed:Math.sqrt((p2.x - p1.x) * (p2.x - p1.x) + (p2.y - p1.y) * (p2.y - p1.y) + (p2.z - p1.z) * (p2.z - p1.z))
    };
};

  /*
    Gets the position of the point, given time.

    WARNING: The speed is not constant. The time it takes between control points is constant.

    For constant speed, use Spline.steps[i];
  */
Spline.prototype.pos = function (time) {

    function bezier(t, p1, c1, c2, p2) {
        var B = function (t) {
            var t2 = t * t, t3 = t2 * t;
            return [(t3), (3 * t2 * (1 - t)), (3 * t * (1 - t) * (1 - t)), ((1 - t) * (1 - t) * (1 - t))];
        };
        var b = B(t);
        var pos = {
            x : p2.x * b[0] + c2.x * b[1] + c1.x * b[2] + p1.x * b[3],
            y : p2.y * b[0] + c2.y * b[1] + c1.y * b[2] + p1.y * b[3],
            z : p2.z * b[0] + c2.z * b[1] + c1.z * b[2] + p1.z * b[3]
        };
        return pos;
    }
    var t = time - this.delay;
    if (t < 0) t = 0;
    if (t > this.duration) t = this.duration - 1;
    //t = t-this.delay;
    var t2 = (t) / this.duration;
    if (t2 >= 1) return this.points[this.length - 1];

    var n = Math.floor((this.points.length - 1) * t2);
    var t1 = (this.length - 1) * t2 - n;
    return bezier(t1, this.points[n], this.controls[n][1], this.controls[n + 1][0], this.points[n + 1]);
};

/**
 * Takes a {@link LineString|line} and returns a curved version
 * by applying a [Bezier spline](http://en.wikipedia.org/wiki/B%C3%A9zier_spline)
 * algorithm.
 *
 * The bezier spline implementation is by [Leszek Rybicki](http://leszek.rybicki.cc/).
 *
 * @name bezierSpline
 * @param {Feature<LineString>} line input LineString
 * @param {Object} [options={}] Optional parameters
 * @param {number} [options.resolution=10000] time in milliseconds between points
 * @param {number} [options.sharpness=0.85] a measure of how curvy the path should be between splines
 * @returns {Feature<LineString>} curved line
 * @example
 * var line = turf.lineString([
 *   [-76.091308, 18.427501],
 *   [-76.695556, 18.729501],
 *   [-76.552734, 19.40443],
 *   [-74.61914, 19.134789],
 *   [-73.652343, 20.07657],
 *   [-73.157958, 20.210656]
 * ]);
 *
 * var curved = turf.bezierSpline(line);
 *
 * //addToMap
 * var addToMap = [line, curved]
 * curved.properties = { stroke: '#0F0' };
 */
function bezier(line, options) {
    // Optional params
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var resolution = options.resolution || 10000;
    var sharpness = options.sharpness || 0.85;

    // validation
    if (!line) throw new Error('line is required');
    if (!isNumber(resolution)) throw new Error('resolution must be an number');
    if (!isNumber(sharpness)) throw new Error('sharpness must be an number');

    var coords = [];
    var spline = new Spline({
        points: getGeom(line).coordinates.map(function (pt) {
            return {x: pt[0], y: pt[1]};
        }),
        duration: resolution,
        sharpness: sharpness
    });

    for (var i = 0; i < spline.duration; i += 10) {
        var pos = spline.pos(i);
        if (Math.floor(i / 100) % 2 === 0) {
            coords.push([pos.x, pos.y]);
        }
    }

    return lineString(coords, line.properties);
}

/**
 * Takes a set of {@link Point|points} and a set of {@link Polygon|polygons} and performs a spatial join.
 *
 * @name tag
 * @param {FeatureCollection<Point>} points input points
 * @param {FeatureCollection<Polygon>} polygons input polygons
 * @param {string} field property in `polygons` to add to joined {<Point>} features
 * @param {string} outField property in `points` in which to store joined property from `polygons`
 * @returns {FeatureCollection<Point>} points with `containingPolyId` property containing values from `polyId`
 * @example
 * var pt1 = turf.point([-77, 44]);
 * var pt2 = turf.point([-77, 38]);
 * var poly1 = turf.polygon([[
 *   [-81, 41],
 *   [-81, 47],
 *   [-72, 47],
 *   [-72, 41],
 *   [-81, 41]
 * ]], {pop: 3000});
 * var poly2 = turf.polygon([[
 *   [-81, 35],
 *   [-81, 41],
 *   [-72, 41],
 *   [-72, 35],
 *   [-81, 35]
 * ]], {pop: 1000});
 *
 * var points = turf.featureCollection([pt1, pt2]);
 * var polygons = turf.featureCollection([poly1, poly2]);
 *
 * var tagged = turf.tag(points, polygons, 'pop', 'population');
 *
 * //addToMap
 * var addToMap = [tagged, polygons]
 */
function tag(points, polygons, field, outField) {
    // prevent mutations
    points = clone(points);
    polygons = clone(polygons);
    featureEach(points, function (pt) {
        if (!pt.properties) pt.properties = {};
        featureEach(polygons, function (poly) {
            if (pt.properties[outField] === undefined) {
                if (booleanPointInPolygon(pt, poly)) pt.properties[outField] = poly.properties[field];
            }
        });
    });
    return points;
}

// http://stackoverflow.com/questions/11935175/sampling-a-random-subset-from-an-array
/**
 * Takes a {@link FeatureCollection} and returns a FeatureCollection with given number of {@link Feature|features} at random.
 *
 * @name sample
 * @param {FeatureCollection} featurecollection set of input features
 * @param {number} num number of features to select
 * @returns {FeatureCollection} a FeatureCollection with `n` features
 * @example
 * var points = turf.randomPoint(100, {bbox: [-80, 30, -60, 60]});
 *
 * var sample = turf.sample(points, 5);
 *
 * //addToMap
 * var addToMap = [points, sample]
 * turf.featureEach(sample, function (currentFeature) {
 *   currentFeature.properties['marker-size'] = 'large';
 *   currentFeature.properties['marker-color'] = '#000';
 * });
 */
function sample(featurecollection, num) {
    if (!featurecollection) throw new Error('featurecollection is required');
    if (num === null || num === undefined) throw new Error('num is required');
    if (typeof num !== 'number') throw new Error('num must be a number');

    var outFC = featureCollection(getRandomSubarray(featurecollection.features, num));
    return outFC;
}

function getRandomSubarray(arr, size) {
    var shuffled = arr.slice(0), i = arr.length, min = i - size, temp, index;
    while (i-- > min) {
        index = Math.floor((i + 1) * Math.random());
        temp = shuffled[index];
        shuffled[index] = shuffled[i];
        shuffled[i] = temp;
    }
    return shuffled.slice(min);
}

/**
 * Takes a bbox and returns an equivalent {@link Polygon|polygon}.
 *
 * @name bboxPolygon
 * @param {BBox} bbox extent in [minX, minY, maxX, maxY] order
 * @returns {Feature<Polygon>} a Polygon representation of the bounding box
 * @example
 * var bbox = [0, 0, 10, 10];
 *
 * var poly = turf.bboxPolygon(bbox);
 *
 * //addToMap
 * var addToMap = [poly]
 */
function bboxPolygon(bbox) {
    validateBBox(bbox);
    // Convert BBox positions to Numbers
    // No performance loss for including Number()
    // https://github.com/Turfjs/turf/issues/1119
    var west = Number(bbox[0]);
    var south = Number(bbox[1]);
    var east = Number(bbox[2]);
    var north = Number(bbox[3]);

    if (bbox.length === 6) throw new Error('@turf/bbox-polygon does not support BBox with 6 positions');

    var lowLeft = [west, south];
    var topLeft = [west, north];
    var topRight = [east, north];
    var lowRight = [east, south];

    return polygon([[
        lowLeft,
        lowRight,
        topRight,
        topLeft,
        lowLeft
    ]]);
}

/**
 * Takes any number of features and returns a rectangular {@link Polygon} that encompasses all vertices.
 *
 * @name envelope
 * @param {GeoJSON} geojson input features
 * @returns {Feature<Polygon>} a rectangular Polygon feature that encompasses all vertices
 * @example
 * var features = turf.featureCollection([
 *   turf.point([-75.343, 39.984], {"name": "Location A"}),
 *   turf.point([-75.833, 39.284], {"name": "Location B"}),
 *   turf.point([-75.534, 39.123], {"name": "Location C"})
 * ]);
 *
 * var enveloped = turf.envelope(features);
 *
 * //addToMap
 * var addToMap = [features, enveloped];
 */
function envelope(geojson) {
    return bboxPolygon(bbox(geojson));
}

/**
 * Takes a bounding box and calculates the minimum square bounding box that
 * would contain the input.
 *
 * @name square
 * @param {BBox} bbox extent in [west, south, east, north] order
 * @returns {BBox} a square surrounding `bbox`
 * @example
 * var bbox = [-20, -20, -15, 0];
 * var squared = turf.square(bbox);
 *
 * //addToMap
 * var addToMap = [turf.bboxPolygon(bbox), turf.bboxPolygon(squared)]
 */
function square(bbox) {
    var west = bbox[0];
    var south = bbox[1];
    var east = bbox[2];
    var north = bbox[3];

    var horizontalDistance = distance(bbox.slice(0, 2), [east, south]);
    var verticalDistance = distance(bbox.slice(0, 2), [west, north]);
    if (horizontalDistance >= verticalDistance) {
        var verticalMidpoint = (south + north) / 2;
        return [
            west,
            verticalMidpoint - ((east - west) / 2),
            east,
            verticalMidpoint + ((east - west) / 2)
        ];
    } else {
        var horizontalMidpoint = (west + east) / 2;
        return [
            horizontalMidpoint - ((north - south) / 2),
            south,
            horizontalMidpoint + ((north - south) / 2),
            north
        ];
    }
}

//http://en.wikipedia.org/wiki/Haversine_formula
//http://www.movable-type.co.uk/scripts/latlong.html
/**
 * Takes a {@link Point} and calculates the location of a destination point given a distance in degrees, radians, miles, or kilometers; and bearing in degrees. This uses the [Haversine formula](http://en.wikipedia.org/wiki/Haversine_formula) to account for global curvature.
 *
 * @name destination
 * @param {Coord} origin starting point
 * @param {number} distance distance from the origin point
 * @param {number} bearing ranging from -180 to 180
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units='kilometers'] miles, kilometers, degrees, or radians
 * @param {Object} [options.properties={}] Translate properties to Point
 * @returns {Feature<Point>} destination point
 * @example
 * var point = turf.point([-75.343, 39.984]);
 * var distance = 50;
 * var bearing = 90;
 * var options = {units: 'miles'};
 *
 * var destination = turf.destination(point, distance, bearing, options);
 *
 * //addToMap
 * var addToMap = [point, destination]
 * destination.properties['marker-color'] = '#f00';
 * point.properties['marker-color'] = '#0f0';
 */
function destination(origin, distance, bearing, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var units = options.units;
    var properties = options.properties;

    // Handle input
    var coordinates1 = getCoord(origin);
    var longitude1 = degreesToRadians(coordinates1[0]);
    var latitude1 = degreesToRadians(coordinates1[1]);
    var bearing_rad = degreesToRadians(bearing);
    var radians = lengthToRadians(distance, units);

    // Main
    var latitude2 = Math.asin(Math.sin(latitude1) * Math.cos(radians) +
        Math.cos(latitude1) * Math.sin(radians) * Math.cos(bearing_rad));
    var longitude2 = longitude1 + Math.atan2(Math.sin(bearing_rad) * Math.sin(radians) * Math.cos(latitude1),
        Math.cos(radians) - Math.sin(latitude1) * Math.sin(latitude2));
    var lng = radiansToDegrees(longitude2);
    var lat = radiansToDegrees(latitude2);

    return point([lng, lat], properties);
}

/**
 * Takes a {@link Point} and calculates the circle polygon given a radius in degrees, radians, miles, or kilometers; and steps for precision.
 *
 * @name circle
 * @param {Feature<Point>|number[]} center center point
 * @param {number} radius radius of the circle
 * @param {Object} [options={}] Optional parameters
 * @param {number} [options.steps=64] number of steps
 * @param {string} [options.units='kilometers'] miles, kilometers, degrees, or radians
 * @param {Object} [options.properties={}] properties
 * @returns {Feature<Polygon>} circle polygon
 * @example
 * var center = [-75.343, 39.984];
 * var radius = 5;
 * var options = {steps: 10, units: 'kilometers', properties: {foo: 'bar'}};
 * var circle = turf.circle(center, radius, options);
 *
 * //addToMap
 * var addToMap = [turf.point(center), circle]
 */
function circle(center, radius, options) {
    // Optional params
    options = options || {};
    var steps = options.steps || 64;
    var properties = options.properties;

    // validation
    if (!center) throw new Error('center is required');
    if (!radius) throw new Error('radius is required');
    if (typeof options !== 'object') throw new Error('options must be an object');
    if (typeof steps !== 'number') throw new Error('steps must be a number');

    // default params
    steps = steps || 64;
    properties = properties || center.properties || {};

    var coordinates = [];
    for (var i = 0; i < steps; i++) {
        coordinates.push(destination(center, radius, i * -360 / steps, options).geometry.coordinates);
    }
    coordinates.push(coordinates[0]);

    return polygon([coordinates], properties);
}

//http://en.wikipedia.org/wiki/Haversine_formula
//http://www.movable-type.co.uk/scripts/latlong.html

/**
 * Takes two {@link Point|points} and finds the geographic bearing between them,
 * i.e. the angle measured in degrees from the north line (0 degrees)
 *
 * @name bearing
 * @param {Coord} start starting Point
 * @param {Coord} end ending Point
 * @param {Object} [options={}] Optional parameters
 * @param {boolean} [options.final=false] calculates the final bearing if true
 * @returns {number} bearing in decimal degrees, between -180 and 180 degrees (positive clockwise)
 * @example
 * var point1 = turf.point([-75.343, 39.984]);
 * var point2 = turf.point([-75.534, 39.123]);
 *
 * var bearing = turf.bearing(point1, point2);
 *
 * //addToMap
 * var addToMap = [point1, point2]
 * point1.properties['marker-color'] = '#f00'
 * point2.properties['marker-color'] = '#0f0'
 * point1.properties.bearing = bearing
 */
function bearing(start, end, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var final = options.final;

    // Reverse calculation
    if (final === true) return calculateFinalBearing(start, end);

    var coordinates1 = getCoord(start);
    var coordinates2 = getCoord(end);

    var lon1 = degreesToRadians(coordinates1[0]);
    var lon2 = degreesToRadians(coordinates2[0]);
    var lat1 = degreesToRadians(coordinates1[1]);
    var lat2 = degreesToRadians(coordinates2[1]);
    var a = Math.sin(lon2 - lon1) * Math.cos(lat2);
    var b = Math.cos(lat1) * Math.sin(lat2) -
        Math.sin(lat1) * Math.cos(lat2) * Math.cos(lon2 - lon1);

    return radiansToDegrees(Math.atan2(a, b));
}

/**
 * Calculates Final Bearing
 *
 * @private
 * @param {Coord} start starting Point
 * @param {Coord} end ending Point
 * @returns {number} bearing
 */
function calculateFinalBearing(start, end) {
    // Swap start & end
    var bear = bearing(end, start);
    bear = (bear + 180) % 360;
    return bear;
}

/**
 * Takes two {@link Point|points} and returns a point midway between them.
 * The midpoint is calculated geodesically, meaning the curvature of the earth is taken into account.
 *
 * @name midpoint
 * @param {Coord} point1 first point
 * @param {Coord} point2 second point
 * @returns {Feature<Point>} a point midway between `pt1` and `pt2`
 * @example
 * var point1 = turf.point([144.834823, -37.771257]);
 * var point2 = turf.point([145.14244, -37.830937]);
 *
 * var midpoint = turf.midpoint(point1, point2);
 *
 * //addToMap
 * var addToMap = [point1, point2, midpoint];
 * midpoint.properties['marker-color'] = '#f00';
 */
function midpoint(point1, point2) {
    var dist = distance(point1, point2);
    var heading = bearing(point1, point2);
    var midpoint = destination(point1, dist / 2, heading);

    return midpoint;
}

/**
 * Takes a {@link Feature} or {@link FeatureCollection} and returns the absolute center point of all features.
 *
 * @name center
 * @param {GeoJSON} geojson GeoJSON to be centered
 * @param {Object} [options={}] Optional parameters
 * @param {Object} [options.properties={}] an Object that is used as the {@link Feature}'s properties
 * @returns {Feature<Point>} a Point feature at the absolute center point of all input features
 * @example
 * var features = turf.featureCollection([
 *   turf.point( [-97.522259, 35.4691]),
 *   turf.point( [-97.502754, 35.463455]),
 *   turf.point( [-97.508269, 35.463245])
 * ]);
 *
 * var center = turf.center(features);
 *
 * //addToMap
 * var addToMap = [features, center]
 * center.properties['marker-size'] = 'large';
 * center.properties['marker-color'] = '#000';
 */
function center(geojson, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var properties = options.properties;

    // Input validation
    if (!geojson) throw new Error('geojson is required');

    var ext = bbox(geojson);
    var x = (ext[0] + ext[2]) / 2;
    var y = (ext[1] + ext[3]) / 2;
    return point([x, y], properties);
}

/**
 * Takes one or more features and calculates the centroid using the mean of all vertices.
 * This lessens the effect of small islands and artifacts when calculating the centroid of a set of polygons.
 *
 * @name centroid
 * @param {GeoJSON} geojson GeoJSON to be centered
 * @param {Object} [properties={}] an Object that is used as the {@link Feature}'s properties
 * @returns {Feature<Point>} the centroid of the input features
 * @example
 * var polygon = turf.polygon([[[-81, 41], [-88, 36], [-84, 31], [-80, 33], [-77, 39], [-81, 41]]]);
 *
 * var centroid = turf.centroid(polygon);
 *
 * //addToMap
 * var addToMap = [polygon, centroid]
 */
function centroid(geojson, properties) {
    var xSum = 0;
    var ySum = 0;
    var len = 0;
    coordEach(geojson, function (coord) {
        xSum += coord[0];
        ySum += coord[1];
        len++;
    }, true);
    return point([xSum / len, ySum / len], properties);
}

/**
 * Takes any {@link Feature} or a {@link FeatureCollection} and returns its [center of mass](https://en.wikipedia.org/wiki/Center_of_mass) using this formula: [Centroid of Polygon](https://en.wikipedia.org/wiki/Centroid#Centroid_of_polygon).
 *
 * @name centerOfMass
 * @param {GeoJSON} geojson GeoJSON to be centered
 * @param {Object} [properties={}] an Object that is used as the {@link Feature}'s properties
 * @returns {Feature<Point>} the center of mass
 * @example
 * var polygon = turf.polygon([[[-81, 41], [-88, 36], [-84, 31], [-80, 33], [-77, 39], [-81, 41]]]);
 *
 * var center = turf.centerOfMass(polygon);
 *
 * //addToMap
 * var addToMap = [polygon, center]
 */
function centerOfMass(geojson, properties) {
    switch (getType(geojson)) {
    case 'Point':
        return geojson;
    case 'Polygon':
        var coords = [];
        coordEach(geojson, function (coord) {
            coords.push(coord);
        });

        // First, we neutralize the feature (set it around coordinates [0,0]) to prevent rounding errors
        // We take any point to translate all the points around 0
        var centre = centroid(geojson, properties);
        var translation = centre.geometry.coordinates;
        var sx = 0;
        var sy = 0;
        var sArea = 0;
        var i, pi, pj, xi, xj, yi, yj, a;

        var neutralizedPoints = coords.map(function (point$$1) {
            return [
                point$$1[0] - translation[0],
                point$$1[1] - translation[1]
            ];
        });

        for (i = 0; i < coords.length - 1; i++) {
            // pi is the current point
            pi = neutralizedPoints[i];
            xi = pi[0];
            yi = pi[1];

            // pj is the next point (pi+1)
            pj = neutralizedPoints[i + 1];
            xj = pj[0];
            yj = pj[1];

            // a is the common factor to compute the signed area and the final coordinates
            a = xi * yj - xj * yi;

            // sArea is the sum used to compute the signed area
            sArea += a;

            // sx and sy are the sums used to compute the final coordinates
            sx += (xi + xj) * a;
            sy += (yi + yj) * a;
        }

        // Shape has no area: fallback on turf.centroid
        if (sArea === 0) {
            return centre;
        } else {
            // Compute the signed area, and factorize 1/6A
            var area = sArea * 0.5;
            var areaFactor = 1 / (6 * area);

            // Compute the final coordinates, adding back the values that have been neutralized
            return point([
                translation[0] + areaFactor * sx,
                translation[1] + areaFactor * sy
            ], properties);
        }
    default:
        // Not a polygon: Compute the convex hull and work with that
        var hull = convex(geojson);

        if (hull) return centerOfMass(hull, properties);
        // Hull is empty: fallback on the centroid
        else return centroid(geojson, properties);
    }
}

/**
 * Combines a {@link FeatureCollection} of {@link Point}, {@link LineString}, or {@link Polygon} features
 * into {@link MultiPoint}, {@link MultiLineString}, or {@link MultiPolygon} features.
 *
 * @name combine
 * @param {FeatureCollection<Point|LineString|Polygon>} fc a FeatureCollection of any type
 * @returns {FeatureCollection<MultiPoint|MultiLineString|MultiPolygon>} a FeatureCollection of corresponding type to input
 * @example
 * var fc = turf.featureCollection([
 *   turf.point([19.026432, 47.49134]),
 *   turf.point([19.074497, 47.509548])
 * ]);
 *
 * var combined = turf.combine(fc);
 *
 * //addToMap
 * var addToMap = [combined]
 */
function combine(fc) {
    var groups = {
        MultiPoint: {coordinates: [], properties: []},
        MultiLineString: {coordinates: [], properties: []},
        MultiPolygon: {coordinates: [], properties: []}
    };

    var multiMapping = Object.keys(groups).reduce(function (memo, item) {
        memo[item.replace('Multi', '')] = item;
        return memo;
    }, {});

    function addToGroup(feature$$1, key, multi) {
        if (!multi) {
            groups[key].coordinates.push(feature$$1.geometry.coordinates);
        } else {
            groups[key].coordinates = groups[key].coordinates.concat(feature$$1.geometry.coordinates);
        }
        groups[key].properties.push(feature$$1.properties);
    }

    featureEach(fc, function (feature$$1) {
        if (!feature$$1.geometry) return;
        if (groups[feature$$1.geometry.type]) {
            addToGroup(feature$$1, feature$$1.geometry.type, true);
        } else if (multiMapping[feature$$1.geometry.type]) {
            addToGroup(feature$$1, multiMapping[feature$$1.geometry.type], false);
        }
    });

    return featureCollection(Object.keys(groups)
        .filter(function (key) {
            return groups[key].coordinates.length;
        })
        .sort()
        .map(function (key) {
            var geometry$$1 = { type: key, coordinates: groups[key].coordinates };
            var properties = { collectedProperties: groups[key].properties };
            return feature(geometry$$1, properties);
        }));
}

/**
 * Takes a feature or set of features and returns all positions as {@link Point|points}.
 *
 * @name explode
 * @param {GeoJSON} geojson input features
 * @returns {FeatureCollection<point>} points representing the exploded input features
 * @throws {Error} if it encounters an unknown geometry type
 * @example
 * var polygon = turf.polygon([[[-81, 41], [-88, 36], [-84, 31], [-80, 33], [-77, 39], [-81, 41]]]);
 *
 * var explode = turf.explode(polygon);
 *
 * //addToMap
 * var addToMap = [polygon, explode]
 */
function explode(geojson) {
    var points$$1 = [];
    if (geojson.type === 'FeatureCollection') {
        featureEach(geojson, function (feature$$1) {
            coordEach(feature$$1, function (coord) {
                points$$1.push(point(coord, feature$$1.properties));
            });
        });
    } else {
        coordEach(geojson, function (coord) {
            points$$1.push(point(coord, geojson.properties));
        });
    }
    return featureCollection(points$$1);
}

var earcut_1 = earcut;
var default_1$2 = earcut;

function earcut(data, holeIndices, dim) {

    dim = dim || 2;

    var hasHoles = holeIndices && holeIndices.length,
        outerLen = hasHoles ? holeIndices[0] * dim : data.length,
        outerNode = linkedList(data, 0, outerLen, dim, true),
        triangles = [];

    if (!outerNode) return triangles;

    var minX, minY, maxX, maxY, x, y, invSize;

    if (hasHoles) outerNode = eliminateHoles(data, holeIndices, outerNode, dim);

    // if the shape is not too simple, we'll use z-order curve hash later; calculate polygon bbox
    if (data.length > 80 * dim) {
        minX = maxX = data[0];
        minY = maxY = data[1];

        for (var i = dim; i < outerLen; i += dim) {
            x = data[i];
            y = data[i + 1];
            if (x < minX) minX = x;
            if (y < minY) minY = y;
            if (x > maxX) maxX = x;
            if (y > maxY) maxY = y;
        }

        // minX, minY and invSize are later used to transform coords into integers for z-order calculation
        invSize = Math.max(maxX - minX, maxY - minY);
        invSize = invSize !== 0 ? 1 / invSize : 0;
    }

    earcutLinked(outerNode, triangles, dim, minX, minY, invSize);

    return triangles;
}

// create a circular doubly linked list from polygon points in the specified winding order
function linkedList(data, start, end, dim, clockwise) {
    var i, last;

    if (clockwise === (signedArea(data, start, end, dim) > 0)) {
        for (i = start; i < end; i += dim) last = insertNode$1(i, data[i], data[i + 1], last);
    } else {
        for (i = end - dim; i >= start; i -= dim) last = insertNode$1(i, data[i], data[i + 1], last);
    }

    if (last && equals$1(last, last.next)) {
        removeNode(last);
        last = last.next;
    }

    return last;
}

// eliminate colinear or duplicate points
function filterPoints(start, end) {
    if (!start) return start;
    if (!end) end = start;

    var p = start,
        again;
    do {
        again = false;

        if (!p.steiner && (equals$1(p, p.next) || area(p.prev, p, p.next) === 0)) {
            removeNode(p);
            p = end = p.prev;
            if (p === p.next) break;
            again = true;

        } else {
            p = p.next;
        }
    } while (again || p !== end);

    return end;
}

// main ear slicing loop which triangulates a polygon (given as a linked list)
function earcutLinked(ear, triangles, dim, minX, minY, invSize, pass) {
    if (!ear) return;

    // interlink polygon nodes in z-order
    if (!pass && invSize) indexCurve(ear, minX, minY, invSize);

    var stop = ear,
        prev, next;

    // iterate through ears, slicing them one by one
    while (ear.prev !== ear.next) {
        prev = ear.prev;
        next = ear.next;

        if (invSize ? isEarHashed(ear, minX, minY, invSize) : isEar(ear)) {
            // cut off the triangle
            triangles.push(prev.i / dim);
            triangles.push(ear.i / dim);
            triangles.push(next.i / dim);

            removeNode(ear);

            // skipping the next vertice leads to less sliver triangles
            ear = next.next;
            stop = next.next;

            continue;
        }

        ear = next;

        // if we looped through the whole remaining polygon and can't find any more ears
        if (ear === stop) {
            // try filtering points and slicing again
            if (!pass) {
                earcutLinked(filterPoints(ear), triangles, dim, minX, minY, invSize, 1);

            // if this didn't work, try curing all small self-intersections locally
            } else if (pass === 1) {
                ear = cureLocalIntersections(ear, triangles, dim);
                earcutLinked(ear, triangles, dim, minX, minY, invSize, 2);

            // as a last resort, try splitting the remaining polygon into two
            } else if (pass === 2) {
                splitEarcut(ear, triangles, dim, minX, minY, invSize);
            }

            break;
        }
    }
}

// check whether a polygon node forms a valid ear with adjacent nodes
function isEar(ear) {
    var a = ear.prev,
        b = ear,
        c = ear.next;

    if (area(a, b, c) >= 0) return false; // reflex, can't be an ear

    // now make sure we don't have other points inside the potential ear
    var p = ear.next.next;

    while (p !== ear.prev) {
        if (pointInTriangle(a.x, a.y, b.x, b.y, c.x, c.y, p.x, p.y) &&
            area(p.prev, p, p.next) >= 0) return false;
        p = p.next;
    }

    return true;
}

function isEarHashed(ear, minX, minY, invSize) {
    var a = ear.prev,
        b = ear,
        c = ear.next;

    if (area(a, b, c) >= 0) return false; // reflex, can't be an ear

    // triangle bbox; min & max are calculated like this for speed
    var minTX = a.x < b.x ? (a.x < c.x ? a.x : c.x) : (b.x < c.x ? b.x : c.x),
        minTY = a.y < b.y ? (a.y < c.y ? a.y : c.y) : (b.y < c.y ? b.y : c.y),
        maxTX = a.x > b.x ? (a.x > c.x ? a.x : c.x) : (b.x > c.x ? b.x : c.x),
        maxTY = a.y > b.y ? (a.y > c.y ? a.y : c.y) : (b.y > c.y ? b.y : c.y);

    // z-order range for the current triangle bbox;
    var minZ = zOrder(minTX, minTY, minX, minY, invSize),
        maxZ = zOrder(maxTX, maxTY, minX, minY, invSize);

    // first look for points inside the triangle in increasing z-order
    var p = ear.nextZ;

    while (p && p.z <= maxZ) {
        if (p !== ear.prev && p !== ear.next &&
            pointInTriangle(a.x, a.y, b.x, b.y, c.x, c.y, p.x, p.y) &&
            area(p.prev, p, p.next) >= 0) return false;
        p = p.nextZ;
    }

    // then look for points in decreasing z-order
    p = ear.prevZ;

    while (p && p.z >= minZ) {
        if (p !== ear.prev && p !== ear.next &&
            pointInTriangle(a.x, a.y, b.x, b.y, c.x, c.y, p.x, p.y) &&
            area(p.prev, p, p.next) >= 0) return false;
        p = p.prevZ;
    }

    return true;
}

// go through all polygon nodes and cure small local self-intersections
function cureLocalIntersections(start, triangles, dim) {
    var p = start;
    do {
        var a = p.prev,
            b = p.next.next;

        if (!equals$1(a, b) && intersects$2(a, p, p.next, b) && locallyInside(a, b) && locallyInside(b, a)) {

            triangles.push(a.i / dim);
            triangles.push(p.i / dim);
            triangles.push(b.i / dim);

            // remove two nodes involved
            removeNode(p);
            removeNode(p.next);

            p = start = b;
        }
        p = p.next;
    } while (p !== start);

    return p;
}

// try splitting polygon into two and triangulate them independently
function splitEarcut(start, triangles, dim, minX, minY, invSize) {
    // look for a valid diagonal that divides the polygon into two
    var a = start;
    do {
        var b = a.next.next;
        while (b !== a.prev) {
            if (a.i !== b.i && isValidDiagonal(a, b)) {
                // split the polygon in two by the diagonal
                var c = splitPolygon(a, b);

                // filter colinear points around the cuts
                a = filterPoints(a, a.next);
                c = filterPoints(c, c.next);

                // run earcut on each half
                earcutLinked(a, triangles, dim, minX, minY, invSize);
                earcutLinked(c, triangles, dim, minX, minY, invSize);
                return;
            }
            b = b.next;
        }
        a = a.next;
    } while (a !== start);
}

// link every hole into the outer loop, producing a single-ring polygon without holes
function eliminateHoles(data, holeIndices, outerNode, dim) {
    var queue = [],
        i, len, start, end, list;

    for (i = 0, len = holeIndices.length; i < len; i++) {
        start = holeIndices[i] * dim;
        end = i < len - 1 ? holeIndices[i + 1] * dim : data.length;
        list = linkedList(data, start, end, dim, false);
        if (list === list.next) list.steiner = true;
        queue.push(getLeftmost(list));
    }

    queue.sort(compareX);

    // process holes from left to right
    for (i = 0; i < queue.length; i++) {
        eliminateHole(queue[i], outerNode);
        outerNode = filterPoints(outerNode, outerNode.next);
    }

    return outerNode;
}

function compareX(a, b) {
    return a.x - b.x;
}

// find a bridge between vertices that connects hole with an outer ring and and link it
function eliminateHole(hole, outerNode) {
    outerNode = findHoleBridge(hole, outerNode);
    if (outerNode) {
        var b = splitPolygon(outerNode, hole);
        filterPoints(b, b.next);
    }
}

// David Eberly's algorithm for finding a bridge between hole and outer polygon
function findHoleBridge(hole, outerNode) {
    var p = outerNode,
        hx = hole.x,
        hy = hole.y,
        qx = -Infinity,
        m;

    // find a segment intersected by a ray from the hole's leftmost point to the left;
    // segment's endpoint with lesser x will be potential connection point
    do {
        if (hy <= p.y && hy >= p.next.y && p.next.y !== p.y) {
            var x = p.x + (hy - p.y) * (p.next.x - p.x) / (p.next.y - p.y);
            if (x <= hx && x > qx) {
                qx = x;
                if (x === hx) {
                    if (hy === p.y) return p;
                    if (hy === p.next.y) return p.next;
                }
                m = p.x < p.next.x ? p : p.next;
            }
        }
        p = p.next;
    } while (p !== outerNode);

    if (!m) return null;

    if (hx === qx) return m.prev; // hole touches outer segment; pick lower endpoint

    // look for points inside the triangle of hole point, segment intersection and endpoint;
    // if there are no points found, we have a valid connection;
    // otherwise choose the point of the minimum angle with the ray as connection point

    var stop = m,
        mx = m.x,
        my = m.y,
        tanMin = Infinity,
        tan;

    p = m.next;

    while (p !== stop) {
        if (hx >= p.x && p.x >= mx && hx !== p.x &&
                pointInTriangle(hy < my ? hx : qx, hy, mx, my, hy < my ? qx : hx, hy, p.x, p.y)) {

            tan = Math.abs(hy - p.y) / (hx - p.x); // tangential

            if ((tan < tanMin || (tan === tanMin && p.x > m.x)) && locallyInside(p, hole)) {
                m = p;
                tanMin = tan;
            }
        }

        p = p.next;
    }

    return m;
}

// interlink polygon nodes in z-order
function indexCurve(start, minX, minY, invSize) {
    var p = start;
    do {
        if (p.z === null) p.z = zOrder(p.x, p.y, minX, minY, invSize);
        p.prevZ = p.prev;
        p.nextZ = p.next;
        p = p.next;
    } while (p !== start);

    p.prevZ.nextZ = null;
    p.prevZ = null;

    sortLinked(p);
}

// Simon Tatham's linked list merge sort algorithm
// http://www.chiark.greenend.org.uk/~sgtatham/algorithms/listsort.html
function sortLinked(list) {
    var i, p, q, e, tail, numMerges, pSize, qSize,
        inSize = 1;

    do {
        p = list;
        list = null;
        tail = null;
        numMerges = 0;

        while (p) {
            numMerges++;
            q = p;
            pSize = 0;
            for (i = 0; i < inSize; i++) {
                pSize++;
                q = q.nextZ;
                if (!q) break;
            }
            qSize = inSize;

            while (pSize > 0 || (qSize > 0 && q)) {

                if (pSize !== 0 && (qSize === 0 || !q || p.z <= q.z)) {
                    e = p;
                    p = p.nextZ;
                    pSize--;
                } else {
                    e = q;
                    q = q.nextZ;
                    qSize--;
                }

                if (tail) tail.nextZ = e;
                else list = e;

                e.prevZ = tail;
                tail = e;
            }

            p = q;
        }

        tail.nextZ = null;
        inSize *= 2;

    } while (numMerges > 1);

    return list;
}

// z-order of a point given coords and inverse of the longer side of data bbox
function zOrder(x, y, minX, minY, invSize) {
    // coords are transformed into non-negative 15-bit integer range
    x = 32767 * (x - minX) * invSize;
    y = 32767 * (y - minY) * invSize;

    x = (x | (x << 8)) & 0x00FF00FF;
    x = (x | (x << 4)) & 0x0F0F0F0F;
    x = (x | (x << 2)) & 0x33333333;
    x = (x | (x << 1)) & 0x55555555;

    y = (y | (y << 8)) & 0x00FF00FF;
    y = (y | (y << 4)) & 0x0F0F0F0F;
    y = (y | (y << 2)) & 0x33333333;
    y = (y | (y << 1)) & 0x55555555;

    return x | (y << 1);
}

// find the leftmost node of a polygon ring
function getLeftmost(start) {
    var p = start,
        leftmost = start;
    do {
        if (p.x < leftmost.x) leftmost = p;
        p = p.next;
    } while (p !== start);

    return leftmost;
}

// check if a point lies within a convex triangle
function pointInTriangle(ax, ay, bx, by, cx, cy, px, py) {
    return (cx - px) * (ay - py) - (ax - px) * (cy - py) >= 0 &&
           (ax - px) * (by - py) - (bx - px) * (ay - py) >= 0 &&
           (bx - px) * (cy - py) - (cx - px) * (by - py) >= 0;
}

// check if a diagonal between two polygon nodes is valid (lies in polygon interior)
function isValidDiagonal(a, b) {
    return a.next.i !== b.i && a.prev.i !== b.i && !intersectsPolygon(a, b) &&
           locallyInside(a, b) && locallyInside(b, a) && middleInside(a, b);
}

// signed area of a triangle
function area(p, q, r) {
    return (q.y - p.y) * (r.x - q.x) - (q.x - p.x) * (r.y - q.y);
}

// check if two points are equal
function equals$1(p1, p2) {
    return p1.x === p2.x && p1.y === p2.y;
}

// check if two segments intersect
function intersects$2(p1, q1, p2, q2) {
    if ((equals$1(p1, q1) && equals$1(p2, q2)) ||
        (equals$1(p1, q2) && equals$1(p2, q1))) return true;
    return area(p1, q1, p2) > 0 !== area(p1, q1, q2) > 0 &&
           area(p2, q2, p1) > 0 !== area(p2, q2, q1) > 0;
}

// check if a polygon diagonal intersects any polygon segments
function intersectsPolygon(a, b) {
    var p = a;
    do {
        if (p.i !== a.i && p.next.i !== a.i && p.i !== b.i && p.next.i !== b.i &&
                intersects$2(p, p.next, a, b)) return true;
        p = p.next;
    } while (p !== a);

    return false;
}

// check if a polygon diagonal is locally inside the polygon
function locallyInside(a, b) {
    return area(a.prev, a, a.next) < 0 ?
        area(a, b, a.next) >= 0 && area(a, a.prev, b) >= 0 :
        area(a, b, a.prev) < 0 || area(a, a.next, b) < 0;
}

// check if the middle point of a polygon diagonal is inside the polygon
function middleInside(a, b) {
    var p = a,
        inside = false,
        px = (a.x + b.x) / 2,
        py = (a.y + b.y) / 2;
    do {
        if (((p.y > py) !== (p.next.y > py)) && p.next.y !== p.y &&
                (px < (p.next.x - p.x) * (py - p.y) / (p.next.y - p.y) + p.x))
            inside = !inside;
        p = p.next;
    } while (p !== a);

    return inside;
}

// link two polygon vertices with a bridge; if the vertices belong to the same ring, it splits polygon into two;
// if one belongs to the outer ring and another to a hole, it merges it into a single ring
function splitPolygon(a, b) {
    var a2 = new Node(a.i, a.x, a.y),
        b2 = new Node(b.i, b.x, b.y),
        an = a.next,
        bp = b.prev;

    a.next = b;
    b.prev = a;

    a2.next = an;
    an.prev = a2;

    b2.next = a2;
    a2.prev = b2;

    bp.next = b2;
    b2.prev = bp;

    return b2;
}

// create a node and optionally link it with previous one (in a circular doubly linked list)
function insertNode$1(i, x, y, last) {
    var p = new Node(i, x, y);

    if (!last) {
        p.prev = p;
        p.next = p;

    } else {
        p.next = last.next;
        p.prev = last;
        last.next.prev = p;
        last.next = p;
    }
    return p;
}

function removeNode(p) {
    p.next.prev = p.prev;
    p.prev.next = p.next;

    if (p.prevZ) p.prevZ.nextZ = p.nextZ;
    if (p.nextZ) p.nextZ.prevZ = p.prevZ;
}

function Node(i, x, y) {
    // vertice index in coordinates array
    this.i = i;

    // vertex coordinates
    this.x = x;
    this.y = y;

    // previous and next vertice nodes in a polygon ring
    this.prev = null;
    this.next = null;

    // z-order curve value
    this.z = null;

    // previous and next nodes in z-order
    this.prevZ = null;
    this.nextZ = null;

    // indicates whether this is a steiner point
    this.steiner = false;
}

// return a percentage difference between the polygon area and its triangulation area;
// used to verify correctness of triangulation
earcut.deviation = function (data, holeIndices, dim, triangles) {
    var hasHoles = holeIndices && holeIndices.length;
    var outerLen = hasHoles ? holeIndices[0] * dim : data.length;

    var polygonArea = Math.abs(signedArea(data, 0, outerLen, dim));
    if (hasHoles) {
        for (var i = 0, len = holeIndices.length; i < len; i++) {
            var start = holeIndices[i] * dim;
            var end = i < len - 1 ? holeIndices[i + 1] * dim : data.length;
            polygonArea -= Math.abs(signedArea(data, start, end, dim));
        }
    }

    var trianglesArea = 0;
    for (i = 0; i < triangles.length; i += 3) {
        var a = triangles[i] * dim;
        var b = triangles[i + 1] * dim;
        var c = triangles[i + 2] * dim;
        trianglesArea += Math.abs(
            (data[a] - data[c]) * (data[b + 1] - data[a + 1]) -
            (data[a] - data[b]) * (data[c + 1] - data[a + 1]));
    }

    return polygonArea === 0 && trianglesArea === 0 ? 0 :
        Math.abs((trianglesArea - polygonArea) / polygonArea);
};

function signedArea(data, start, end, dim) {
    var sum = 0;
    for (var i = start, j = end - dim; i < end; i += dim) {
        sum += (data[j] - data[i]) * (data[i + 1] + data[j + 1]);
        j = i;
    }
    return sum;
}

// turn a polygon in a multi-dimensional array form (e.g. as in GeoJSON) into a form Earcut accepts
earcut.flatten = function (data) {
    var dim = data[0][0].length,
        result = {vertices: [], holes: [], dimensions: dim},
        holeIndex = 0;

    for (var i = 0; i < data.length; i++) {
        for (var j = 0; j < data[i].length; j++) {
            for (var d = 0; d < dim; d++) result.vertices.push(data[i][j][d]);
        }
        if (i > 0) {
            holeIndex += data[i - 1].length;
            result.holes.push(holeIndex);
        }
    }
    return result;
};

earcut_1.default = default_1$2;

/**
 * Tesselates a {@link Feature<Polygon>} into a {@link FeatureCollection<Polygon>} of triangles
 * using [earcut](https://github.com/mapbox/earcut).
 *
 * @name tesselate
 * @param {Feature<Polygon>} poly the polygon to tesselate
 * @returns {FeatureCollection<Polygon>} a geometrycollection feature
 * @example
 * var poly = turf.polygon([[[11, 0], [22, 4], [31, 0], [31, 11], [21, 15], [11, 11], [11, 0]]]);
 * var triangles = turf.tesselate(poly);
 *
 * //addToMap
 * var addToMap = [poly, triangles]
 */
function tesselate(poly) {
    if (!poly.geometry || (poly.geometry.type !== 'Polygon' && poly.geometry.type !== 'MultiPolygon')) {
        throw new Error('input must be a Polygon or MultiPolygon');
    }

    var fc = {type: 'FeatureCollection', features: []};

    if (poly.geometry.type === 'Polygon') {
        fc.features = processPolygon(poly.geometry.coordinates);
    } else {
        poly.geometry.coordinates.forEach(function (coordinates) {
            fc.features = fc.features.concat(processPolygon(coordinates));
        });
    }

    return fc;
}

function processPolygon(coordinates) {
    var data = flattenCoords(coordinates);
    var dim = 2;
    var result = earcut_1(data.vertices, data.holes, dim);

    var features = [];
    var vertices = [];

    result.forEach(function (vert, i) {
        var index = result[i];
        vertices.push([data.vertices[index * dim], data.vertices[index * dim + 1]]);
    });

    for (var i = 0; i < vertices.length; i += 3) {
        var coords = vertices.slice(i, i + 3);
        coords.push(vertices[i]);
        features.push(polygon([coords]));
    }

    return features;
}

function flattenCoords(data) {
    var dim = data[0][0].length,
        result = {vertices: [], holes: [], dimensions: dim},
        holeIndex = 0;

    for (var i = 0; i < data.length; i++) {
        for (var j = 0; j < data[i].length; j++) {
            for (var d = 0; d < dim; d++) result.vertices.push(data[i][j][d]);
        }
        if (i > 0) {
            holeIndex += data[i - 1].length;
            result.holes.push(holeIndex);
        }
    }

    return result;
}

/**
 * Takes a reference {@link Point|point} and a FeatureCollection of Features
 * with Point geometries and returns the
 * point from the FeatureCollection closest to the reference. This calculation
 * is geodesic.
 *
 * @name nearestPoint
 * @param {Coord} targetPoint the reference point
 * @param {FeatureCollection<Point>} points against input point set
 * @returns {Feature<Point>} the closest point in the set to the reference point
 * @example
 * var targetPoint = turf.point([28.965797, 41.010086], {"marker-color": "#0F0"});
 * var points = turf.featureCollection([
 *     turf.point([28.973865, 41.011122]),
 *     turf.point([28.948459, 41.024204]),
 *     turf.point([28.938674, 41.013324])
 * ]);
 *
 * var nearest = turf.nearestPoint(targetPoint, points);
 *
 * //addToMap
 * var addToMap = [targetPoint, points, nearest];
 * nearest.properties['marker-color'] = '#F00';
 */
function nearestPoint(targetPoint, points) {
    // Input validation
    if (!targetPoint) throw new Error('targetPoint is required');
    if (!points) throw new Error('points is required');

    var nearest;
    var minDist = Infinity;
    featureEach(points, function (pt, featureIndex) {
        var distanceToPoint = distance(targetPoint, pt);
        if (distanceToPoint < minDist) {
            nearest = clone(pt);
            nearest.properties.featureIndex = featureIndex;
            nearest.properties.distanceToPoint = distanceToPoint;
            minDist = distanceToPoint;
        }

    });
    return nearest;
}

function quickselect$3(arr, k, left, right, compare) {
    quickselectStep(arr, k, left || 0, right || (arr.length - 1), compare || defaultCompare$2);
}

function quickselectStep(arr, k, left, right, compare) {

    while (right > left) {
        if (right - left > 600) {
            var n = right - left + 1;
            var m = k - left + 1;
            var z = Math.log(n);
            var s = 0.5 * Math.exp(2 * z / 3);
            var sd = 0.5 * Math.sqrt(z * s * (n - s) / n) * (m - n / 2 < 0 ? -1 : 1);
            var newLeft = Math.max(left, Math.floor(k - m * s / n + sd));
            var newRight = Math.min(right, Math.floor(k + (n - m) * s / n + sd));
            quickselectStep(arr, k, newLeft, newRight, compare);
        }

        var t = arr[k];
        var i = left;
        var j = right;

        swap$1(arr, left, k);
        if (compare(arr[right], t) > 0) swap$1(arr, left, right);

        while (i < j) {
            swap$1(arr, i, j);
            i++;
            j--;
            while (compare(arr[i], t) < 0) i++;
            while (compare(arr[j], t) > 0) j--;
        }

        if (compare(arr[left], t) === 0) swap$1(arr, left, j);
        else {
            j++;
            swap$1(arr, j, right);
        }

        if (j <= k) left = j + 1;
        if (k <= j) right = j - 1;
    }
}

function swap$1(arr, i, j) {
    var tmp = arr[i];
    arr[i] = arr[j];
    arr[j] = tmp;
}

function defaultCompare$2(a, b) {
    return a < b ? -1 : a > b ? 1 : 0;
}

function rbush$4(maxEntries, format) {
    if (!(this instanceof rbush$4)) return new rbush$4(maxEntries, format);

    // max entries in a node is 9 by default; min node fill is 40% for best performance
    this._maxEntries = Math.max(4, maxEntries || 9);
    this._minEntries = Math.max(2, Math.ceil(this._maxEntries * 0.4));

    if (format) {
        this._initFormat(format);
    }

    this.clear();
}

rbush$4.prototype = {

    all: function () {
        return this._all(this.data, []);
    },

    search: function (bbox) {

        var node = this.data,
            result = [],
            toBBox = this.toBBox;

        if (!intersects$4(bbox, node)) return result;

        var nodesToSearch = [],
            i, len, child, childBBox;

        while (node) {
            for (i = 0, len = node.children.length; i < len; i++) {

                child = node.children[i];
                childBBox = node.leaf ? toBBox(child) : child;

                if (intersects$4(bbox, childBBox)) {
                    if (node.leaf) result.push(child);
                    else if (contains$1(bbox, childBBox)) this._all(child, result);
                    else nodesToSearch.push(child);
                }
            }
            node = nodesToSearch.pop();
        }

        return result;
    },

    collides: function (bbox) {

        var node = this.data,
            toBBox = this.toBBox;

        if (!intersects$4(bbox, node)) return false;

        var nodesToSearch = [],
            i, len, child, childBBox;

        while (node) {
            for (i = 0, len = node.children.length; i < len; i++) {

                child = node.children[i];
                childBBox = node.leaf ? toBBox(child) : child;

                if (intersects$4(bbox, childBBox)) {
                    if (node.leaf || contains$1(bbox, childBBox)) return true;
                    nodesToSearch.push(child);
                }
            }
            node = nodesToSearch.pop();
        }

        return false;
    },

    load: function (data) {
        if (!(data && data.length)) return this;

        if (data.length < this._minEntries) {
            for (var i = 0, len = data.length; i < len; i++) {
                this.insert(data[i]);
            }
            return this;
        }

        // recursively build the tree with the given data from scratch using OMT algorithm
        var node = this._build(data.slice(), 0, data.length - 1, 0);

        if (!this.data.children.length) {
            // save as is if tree is empty
            this.data = node;

        } else if (this.data.height === node.height) {
            // split root if trees have the same height
            this._splitRoot(this.data, node);

        } else {
            if (this.data.height < node.height) {
                // swap trees if inserted one is bigger
                var tmpNode = this.data;
                this.data = node;
                node = tmpNode;
            }

            // insert the small tree into the large tree at appropriate level
            this._insert(node, this.data.height - node.height - 1, true);
        }

        return this;
    },

    insert: function (item) {
        if (item) this._insert(item, this.data.height - 1);
        return this;
    },

    clear: function () {
        this.data = createNode$1([]);
        return this;
    },

    remove: function (item, equalsFn) {
        if (!item) return this;

        var node = this.data,
            bbox = this.toBBox(item),
            path = [],
            indexes = [],
            i, parent, index, goingUp;

        // depth-first iterative tree traversal
        while (node || path.length) {

            if (!node) { // go up
                node = path.pop();
                parent = path[path.length - 1];
                i = indexes.pop();
                goingUp = true;
            }

            if (node.leaf) { // check current node
                index = findItem$1(item, node.children, equalsFn);

                if (index !== -1) {
                    // item found, remove the item and condense tree upwards
                    node.children.splice(index, 1);
                    path.push(node);
                    this._condense(path);
                    return this;
                }
            }

            if (!goingUp && !node.leaf && contains$1(node, bbox)) { // go down
                path.push(node);
                indexes.push(i);
                i = 0;
                parent = node;
                node = node.children[0];

            } else if (parent) { // go right
                i++;
                node = parent.children[i];
                goingUp = false;

            } else node = null; // nothing found
        }

        return this;
    },

    toBBox: function (item) { return item; },

    compareMinX: compareNodeMinX$1,
    compareMinY: compareNodeMinY$1,

    toJSON: function () { return this.data; },

    fromJSON: function (data) {
        this.data = data;
        return this;
    },

    _all: function (node, result) {
        var nodesToSearch = [];
        while (node) {
            if (node.leaf) result.push.apply(result, node.children);
            else nodesToSearch.push.apply(nodesToSearch, node.children);

            node = nodesToSearch.pop();
        }
        return result;
    },

    _build: function (items, left, right, height) {

        var N = right - left + 1,
            M = this._maxEntries,
            node;

        if (N <= M) {
            // reached leaf level; return leaf
            node = createNode$1(items.slice(left, right + 1));
            calcBBox$1(node, this.toBBox);
            return node;
        }

        if (!height) {
            // target height of the bulk-loaded tree
            height = Math.ceil(Math.log(N) / Math.log(M));

            // target number of root entries to maximize storage utilization
            M = Math.ceil(N / Math.pow(M, height - 1));
        }

        node = createNode$1([]);
        node.leaf = false;
        node.height = height;

        // split the items into M mostly square tiles

        var N2 = Math.ceil(N / M),
            N1 = N2 * Math.ceil(Math.sqrt(M)),
            i, j, right2, right3;

        multiSelect$1(items, left, right, N1, this.compareMinX);

        for (i = left; i <= right; i += N1) {

            right2 = Math.min(i + N1 - 1, right);

            multiSelect$1(items, i, right2, N2, this.compareMinY);

            for (j = i; j <= right2; j += N2) {

                right3 = Math.min(j + N2 - 1, right2);

                // pack each entry recursively
                node.children.push(this._build(items, j, right3, height - 1));
            }
        }

        calcBBox$1(node, this.toBBox);

        return node;
    },

    _chooseSubtree: function (bbox, node, level, path) {

        var i, len, child, targetNode, area, enlargement, minArea, minEnlargement;

        while (true) {
            path.push(node);

            if (node.leaf || path.length - 1 === level) break;

            minArea = minEnlargement = Infinity;

            for (i = 0, len = node.children.length; i < len; i++) {
                child = node.children[i];
                area = bboxArea$1(child);
                enlargement = enlargedArea$1(bbox, child) - area;

                // choose entry with the least area enlargement
                if (enlargement < minEnlargement) {
                    minEnlargement = enlargement;
                    minArea = area < minArea ? area : minArea;
                    targetNode = child;

                } else if (enlargement === minEnlargement) {
                    // otherwise choose one with the smallest area
                    if (area < minArea) {
                        minArea = area;
                        targetNode = child;
                    }
                }
            }

            node = targetNode || node.children[0];
        }

        return node;
    },

    _insert: function (item, level, isNode) {

        var toBBox = this.toBBox,
            bbox = isNode ? item : toBBox(item),
            insertPath = [];

        // find the best node for accommodating the item, saving all nodes along the path too
        var node = this._chooseSubtree(bbox, this.data, level, insertPath);

        // put the item into the node
        node.children.push(item);
        extend$1(node, bbox);

        // split on node overflow; propagate upwards if necessary
        while (level >= 0) {
            if (insertPath[level].children.length > this._maxEntries) {
                this._split(insertPath, level);
                level--;
            } else break;
        }

        // adjust bboxes along the insertion path
        this._adjustParentBBoxes(bbox, insertPath, level);
    },

    // split overflowed node into two
    _split: function (insertPath, level) {

        var node = insertPath[level],
            M = node.children.length,
            m = this._minEntries;

        this._chooseSplitAxis(node, m, M);

        var splitIndex = this._chooseSplitIndex(node, m, M);

        var newNode = createNode$1(node.children.splice(splitIndex, node.children.length - splitIndex));
        newNode.height = node.height;
        newNode.leaf = node.leaf;

        calcBBox$1(node, this.toBBox);
        calcBBox$1(newNode, this.toBBox);

        if (level) insertPath[level - 1].children.push(newNode);
        else this._splitRoot(node, newNode);
    },

    _splitRoot: function (node, newNode) {
        // split root node
        this.data = createNode$1([node, newNode]);
        this.data.height = node.height + 1;
        this.data.leaf = false;
        calcBBox$1(this.data, this.toBBox);
    },

    _chooseSplitIndex: function (node, m, M) {

        var i, bbox1, bbox2, overlap, area, minOverlap, minArea, index;

        minOverlap = minArea = Infinity;

        for (i = m; i <= M - m; i++) {
            bbox1 = distBBox$1(node, 0, i, this.toBBox);
            bbox2 = distBBox$1(node, i, M, this.toBBox);

            overlap = intersectionArea$1(bbox1, bbox2);
            area = bboxArea$1(bbox1) + bboxArea$1(bbox2);

            // choose distribution with minimum overlap
            if (overlap < minOverlap) {
                minOverlap = overlap;
                index = i;

                minArea = area < minArea ? area : minArea;

            } else if (overlap === minOverlap) {
                // otherwise choose distribution with minimum area
                if (area < minArea) {
                    minArea = area;
                    index = i;
                }
            }
        }

        return index;
    },

    // sorts node children by the best axis for split
    _chooseSplitAxis: function (node, m, M) {

        var compareMinX = node.leaf ? this.compareMinX : compareNodeMinX$1,
            compareMinY = node.leaf ? this.compareMinY : compareNodeMinY$1,
            xMargin = this._allDistMargin(node, m, M, compareMinX),
            yMargin = this._allDistMargin(node, m, M, compareMinY);

        // if total distributions margin value is minimal for x, sort by minX,
        // otherwise it's already sorted by minY
        if (xMargin < yMargin) node.children.sort(compareMinX);
    },

    // total margin of all possible split distributions where each node is at least m full
    _allDistMargin: function (node, m, M, compare) {

        node.children.sort(compare);

        var toBBox = this.toBBox,
            leftBBox = distBBox$1(node, 0, m, toBBox),
            rightBBox = distBBox$1(node, M - m, M, toBBox),
            margin = bboxMargin$1(leftBBox) + bboxMargin$1(rightBBox),
            i, child;

        for (i = m; i < M - m; i++) {
            child = node.children[i];
            extend$1(leftBBox, node.leaf ? toBBox(child) : child);
            margin += bboxMargin$1(leftBBox);
        }

        for (i = M - m - 1; i >= m; i--) {
            child = node.children[i];
            extend$1(rightBBox, node.leaf ? toBBox(child) : child);
            margin += bboxMargin$1(rightBBox);
        }

        return margin;
    },

    _adjustParentBBoxes: function (bbox, path, level) {
        // adjust bboxes along the given tree path
        for (var i = level; i >= 0; i--) {
            extend$1(path[i], bbox);
        }
    },

    _condense: function (path) {
        // go through the path, removing empty nodes and updating bboxes
        for (var i = path.length - 1, siblings; i >= 0; i--) {
            if (path[i].children.length === 0) {
                if (i > 0) {
                    siblings = path[i - 1].children;
                    siblings.splice(siblings.indexOf(path[i]), 1);

                } else this.clear();

            } else calcBBox$1(path[i], this.toBBox);
        }
    },

    _initFormat: function (format) {
        // data format (minX, minY, maxX, maxY accessors)

        // uses eval-type function compilation instead of just accepting a toBBox function
        // because the algorithms are very sensitive to sorting functions performance,
        // so they should be dead simple and without inner calls

        var compareArr = ['return a', ' - b', ';'];

        this.compareMinX = new Function('a', 'b', compareArr.join(format[0]));
        this.compareMinY = new Function('a', 'b', compareArr.join(format[1]));

        this.toBBox = new Function('a',
            'return {minX: a' + format[0] +
            ', minY: a' + format[1] +
            ', maxX: a' + format[2] +
            ', maxY: a' + format[3] + '};');
    }
};

function findItem$1(item, items, equalsFn) {
    if (!equalsFn) return items.indexOf(item);

    for (var i = 0; i < items.length; i++) {
        if (equalsFn(item, items[i])) return i;
    }
    return -1;
}

// calculate node's bbox from bboxes of its children
function calcBBox$1(node, toBBox) {
    distBBox$1(node, 0, node.children.length, toBBox, node);
}

// min bounding rectangle of node children from k to p-1
function distBBox$1(node, k, p, toBBox, destNode) {
    if (!destNode) destNode = createNode$1(null);
    destNode.minX = Infinity;
    destNode.minY = Infinity;
    destNode.maxX = -Infinity;
    destNode.maxY = -Infinity;

    for (var i = k, child; i < p; i++) {
        child = node.children[i];
        extend$1(destNode, node.leaf ? toBBox(child) : child);
    }

    return destNode;
}

function extend$1(a, b) {
    a.minX = Math.min(a.minX, b.minX);
    a.minY = Math.min(a.minY, b.minY);
    a.maxX = Math.max(a.maxX, b.maxX);
    a.maxY = Math.max(a.maxY, b.maxY);
    return a;
}

function compareNodeMinX$1(a, b) { return a.minX - b.minX; }
function compareNodeMinY$1(a, b) { return a.minY - b.minY; }

function bboxArea$1(a)   { return (a.maxX - a.minX) * (a.maxY - a.minY); }
function bboxMargin$1(a) { return (a.maxX - a.minX) + (a.maxY - a.minY); }

function enlargedArea$1(a, b) {
    return (Math.max(b.maxX, a.maxX) - Math.min(b.minX, a.minX)) *
           (Math.max(b.maxY, a.maxY) - Math.min(b.minY, a.minY));
}

function intersectionArea$1(a, b) {
    var minX = Math.max(a.minX, b.minX),
        minY = Math.max(a.minY, b.minY),
        maxX = Math.min(a.maxX, b.maxX),
        maxY = Math.min(a.maxY, b.maxY);

    return Math.max(0, maxX - minX) *
           Math.max(0, maxY - minY);
}

function contains$1(a, b) {
    return a.minX <= b.minX &&
           a.minY <= b.minY &&
           b.maxX <= a.maxX &&
           b.maxY <= a.maxY;
}

function intersects$4(a, b) {
    return b.minX <= a.maxX &&
           b.minY <= a.maxY &&
           b.maxX >= a.minX &&
           b.maxY >= a.minY;
}

function createNode$1(children) {
    return {
        children: children,
        height: 1,
        leaf: true,
        minX: Infinity,
        minY: Infinity,
        maxX: -Infinity,
        maxY: -Infinity
    };
}

// sort an array so that items come in groups of n unsorted items, with groups sorted between each other;
// combines selection algorithm with binary divide & conquer approach

function multiSelect$1(arr, left, right, n, compare) {
    var stack = [left, right],
        mid;

    while (stack.length) {
        right = stack.pop();
        left = stack.pop();

        if (right - left <= n) continue;

        mid = left + Math.ceil((right - left) / n / 2) * n;
        quickselect$3(arr, mid, left, right, compare);

        stack.push(left, mid, mid, right);
    }
}

/**
 * GeoJSON implementation of [RBush](https://github.com/mourner/rbush#rbush) spatial index.
 *
 * @name rbush
 * @param {number} [maxEntries=9] defines the maximum number of entries in a tree node. 9 (used by default) is a
 * reasonable choice for most applications. Higher value means faster insertion and slower search, and vice versa.
 * @returns {RBush} GeoJSON RBush
 * @example
 * import geojsonRbush from 'geojson-rbush';
 * var tree = geojsonRbush();
 */
function geojsonRbush(maxEntries) {
    var tree = rbush$4(maxEntries);
    /**
     * [insert](https://github.com/mourner/rbush#data-format)
     *
     * @param {Feature<any>} feature insert single GeoJSON Feature
     * @returns {RBush} GeoJSON RBush
     * @example
     * var polygon = {
     *   "type": "Feature",
     *   "properties": {},
     *   "geometry": {
     *     "type": "Polygon",
     *     "coordinates": [[[-78, 41], [-67, 41], [-67, 48], [-78, 48], [-78, 41]]]
     *   }
     * }
     * tree.insert(polygon)
     */
    tree.insert = function (feature) {
        if (Array.isArray(feature)) {
            var bbox = feature;
            feature = bboxPolygon$2(bbox);
            feature.bbox = bbox;
        } else {
            feature.bbox = feature.bbox ? feature.bbox : turfBBox(feature);
        }
        return rbush$4.prototype.insert.call(this, feature);
    };

    /**
     * [load](https://github.com/mourner/rbush#bulk-inserting-data)
     *
     * @param {BBox[]|FeatureCollection<any>} features load entire GeoJSON FeatureCollection
     * @returns {RBush} GeoJSON RBush
     * @example
     * var polygons = {
     *   "type": "FeatureCollection",
     *   "features": [
     *     {
     *       "type": "Feature",
     *       "properties": {},
     *       "geometry": {
     *         "type": "Polygon",
     *         "coordinates": [[[-78, 41], [-67, 41], [-67, 48], [-78, 48], [-78, 41]]]
     *       }
     *     },
     *     {
     *       "type": "Feature",
     *       "properties": {},
     *       "geometry": {
     *         "type": "Polygon",
     *         "coordinates": [[[-93, 32], [-83, 32], [-83, 39], [-93, 39], [-93, 32]]]
     *       }
     *     }
     *   ]
     * }
     * tree.load(polygons)
     */
    tree.load = function (features) {
        var load = [];
        // Load an Array of BBox
        if (Array.isArray(features)) {
            features.forEach(function (bbox) {
                var feature = bboxPolygon$2(bbox);
                feature.bbox = bbox;
                load.push(feature);
            });
        } else {
            // Load FeatureCollection
            featureEach(features, function (feature) {
                feature.bbox = feature.bbox ? feature.bbox : turfBBox(feature);
                load.push(feature);
            });
        }
        return rbush$4.prototype.load.call(this, load);
    };

    /**
     * [remove](https://github.com/mourner/rbush#removing-data)
     *
     * @param {BBox|Feature<any>} feature remove single GeoJSON Feature
     * @returns {RBush} GeoJSON RBush
     * @example
     * var polygon = {
     *   "type": "Feature",
     *   "properties": {},
     *   "geometry": {
     *     "type": "Polygon",
     *     "coordinates": [[[-78, 41], [-67, 41], [-67, 48], [-78, 48], [-78, 41]]]
     *   }
     * }
     * tree.remove(polygon)
     */
    tree.remove = function (feature) {
        if (Array.isArray(feature)) {
            var bbox = feature;
            feature = bboxPolygon$2(bbox);
            feature.bbox = bbox;
        }
        return rbush$4.prototype.remove.call(this, feature);
    };

    /**
     * [clear](https://github.com/mourner/rbush#removing-data)
     *
     * @returns {RBush} GeoJSON Rbush
     * @example
     * tree.clear()
     */
    tree.clear = function () {
        return rbush$4.prototype.clear.call(this);
    };

    /**
     * [search](https://github.com/mourner/rbush#search)
     *
     * @param {BBox|FeatureCollection|Feature<any>} geojson search with GeoJSON
     * @returns {FeatureCollection<any>} all features that intersects with the given GeoJSON.
     * @example
     * var polygon = {
     *   "type": "Feature",
     *   "properties": {},
     *   "geometry": {
     *     "type": "Polygon",
     *     "coordinates": [[[-78, 41], [-67, 41], [-67, 48], [-78, 48], [-78, 41]]]
     *   }
     * }
     * tree.search(polygon)
     */
    tree.search = function (geojson) {
        var features = rbush$4.prototype.search.call(this, this.toBBox(geojson));
        return {
            type: 'FeatureCollection',
            features: features
        };
    };

    /**
     * [collides](https://github.com/mourner/rbush#collisions)
     *
     * @param {BBox|FeatureCollection|Feature<any>} geojson collides with GeoJSON
     * @returns {boolean} true if there are any items intersecting the given GeoJSON, otherwise false.
     * @example
     * var polygon = {
     *   "type": "Feature",
     *   "properties": {},
     *   "geometry": {
     *     "type": "Polygon",
     *     "coordinates": [[[-78, 41], [-67, 41], [-67, 48], [-78, 48], [-78, 41]]]
     *   }
     * }
     * tree.collides(polygon)
     */
    tree.collides = function (geojson) {
        return rbush$4.prototype.collides.call(this, this.toBBox(geojson));
    };

    /**
     * [all](https://github.com/mourner/rbush#search)
     *
     * @returns {FeatureCollection<any>} all the features in RBush
     * @example
     * tree.all()
     * //=FeatureCollection
     */
    tree.all = function () {
        var features = rbush$4.prototype.all.call(this);
        return {
            type: 'FeatureCollection',
            features: features
        };
    };

    /**
     * [toJSON](https://github.com/mourner/rbush#export-and-import)
     *
     * @returns {any} export data as JSON object
     * @example
     * var exported = tree.toJSON()
     * //=JSON object
     */
    tree.toJSON = function () {
        return rbush$4.prototype.toJSON.call(this);
    };

    /**
     * [fromJSON](https://github.com/mourner/rbush#export-and-import)
     *
     * @param {any} json import previously exported data
     * @returns {RBush} GeoJSON RBush
     * @example
     * var exported = {
     *   "children": [
     *     {
     *       "type": "Feature",
     *       "geometry": {
     *         "type": "Point",
     *         "coordinates": [110, 50]
     *       },
     *       "properties": {},
     *       "bbox": [110, 50, 110, 50]
     *     }
     *   ],
     *   "height": 1,
     *   "leaf": true,
     *   "minX": 110,
     *   "minY": 50,
     *   "maxX": 110,
     *   "maxY": 50
     * }
     * tree.fromJSON(exported)
     */
    tree.fromJSON = function (json) {
        return rbush$4.prototype.fromJSON.call(this, json);
    };

    /**
     * Converts GeoJSON to {minX, minY, maxX, maxY} schema
     *
     * @private
     * @param {BBox|FeatureCollectio|Feature<any>} geojson feature(s) to retrieve BBox from
     * @returns {Object} converted to {minX, minY, maxX, maxY}
     */
    tree.toBBox = function (geojson) {
        var bbox;
        if (geojson.bbox) bbox = geojson.bbox;
        else if (Array.isArray(geojson) && geojson.length === 4) bbox = geojson;
        else bbox = turfBBox(geojson);

        return {
            minX: bbox[0],
            minY: bbox[1],
            maxX: bbox[2],
            maxY: bbox[3]
        };
    };
    return tree;
}

/**
 * Takes a bbox and returns an equivalent {@link Polygon|polygon}.
 *
 * @private
 * @name bboxPolygon
 * @param {Array<number>} bbox extent in [minX, minY, maxX, maxY] order
 * @returns {Feature<Polygon>} a Polygon representation of the bounding box
 * @example
 * var bbox = [0, 0, 10, 10];
 *
 * var poly = turf.bboxPolygon(bbox);
 *
 * //addToMap
 * var addToMap = [poly]
 */
function bboxPolygon$2(bbox) {
    var lowLeft = [bbox[0], bbox[1]];
    var topLeft = [bbox[0], bbox[3]];
    var topRight = [bbox[2], bbox[3]];
    var lowRight = [bbox[2], bbox[1]];
    var coordinates = [[lowLeft, lowRight, topRight, topLeft, lowLeft]];

    return {
        type: 'Feature',
        bbox: bbox,
        properties: {},
        geometry: {
            type: 'Polygon',
            coordinates: coordinates
        }
    };
}

/**
 * Takes a set of features, calculates the bbox of all input features, and returns a bounding box.
 *
 * @private
 * @name bbox
 * @param {FeatureCollection|Feature<any>} geojson input features
 * @returns {Array<number>} bbox extent in [minX, minY, maxX, maxY] order
 * @example
 * var line = turf.lineString([[-74, 40], [-78, 42], [-82, 35]]);
 * var bbox = turf.bbox(line);
 * var bboxPolygon = turf.bboxPolygon(bbox);
 *
 * //addToMap
 * var addToMap = [line, bboxPolygon]
 */
function turfBBox(geojson) {
    var bbox = [Infinity, Infinity, -Infinity, -Infinity];
    coordEach(geojson, function (coord) {
        if (bbox[0] > coord[0]) bbox[0] = coord[0];
        if (bbox[1] > coord[1]) bbox[1] = coord[1];
        if (bbox[2] < coord[0]) bbox[2] = coord[0];
        if (bbox[3] < coord[1]) bbox[3] = coord[1];
    });
    return bbox;
}

/**
 * Creates a {@link FeatureCollection} of 2-vertex {@link LineString} segments from a {@link LineString|(Multi)LineString} or {@link Polygon|(Multi)Polygon}.
 *
 * @name lineSegment
 * @param {Geometry|FeatureCollection|Feature<LineString|MultiLineString|MultiPolygon|Polygon>} geojson GeoJSON Polygon or LineString
 * @returns {FeatureCollection<LineString>} 2-vertex line segments
 * @example
 * var polygon = turf.polygon([[[-50, 5], [-40, -10], [-50, -10], [-40, 5], [-50, 5]]]);
 * var segments = turf.lineSegment(polygon);
 *
 * //addToMap
 * var addToMap = [polygon, segments]
 */
function lineSegment(geojson) {
    if (!geojson) throw new Error('geojson is required');

    var results = [];
    flattenEach(geojson, function (feature$$1) {
        lineSegmentFeature(feature$$1, results);
    });
    return featureCollection(results);
}

/**
 * Line Segment
 *
 * @private
 * @param {Feature<LineString|Polygon>} geojson Line or polygon feature
 * @param {Array} results push to results
 * @returns {void}
 */
function lineSegmentFeature(geojson, results) {
    var coords = [];
    var geometry$$1 = geojson.geometry;
    switch (geometry$$1.type) {
    case 'Polygon':
        coords = getCoords(geometry$$1);
        break;
    case 'LineString':
        coords = [getCoords(geometry$$1)];
    }
    coords.forEach(function (coord) {
        var segments = createSegments(coord, geojson.properties);
        segments.forEach(function (segment) {
            segment.id = results.length;
            results.push(segment);
        });
    });
}

/**
 * Create Segments from LineString coordinates
 *
 * @private
 * @param {LineString} coords LineString coordinates
 * @param {*} properties GeoJSON properties
 * @returns {Array<Feature<LineString>>} line segments
 */
function createSegments(coords, properties) {
    var segments = [];
    coords.reduce(function (previousCoords, currentCoords) {
        var segment = lineString([previousCoords, currentCoords], properties);
        segment.bbox = bbox$3(previousCoords, currentCoords);
        segments.push(segment);
        return currentCoords;
    });
    return segments;
}

/**
 * Create BBox between two coordinates (faster than @turf/bbox)
 *
 * @private
 * @param {Array<number>} coords1 Point coordinate
 * @param {Array<number>} coords2 Point coordinate
 * @returns {BBox} [west, south, east, north]
 */
function bbox$3(coords1, coords2) {
    var x1 = coords1[0];
    var y1 = coords1[1];
    var x2 = coords2[0];
    var y2 = coords2[1];
    var west = (x1 < x2) ? x1 : x2;
    var south = (y1 < y2) ? y1 : y2;
    var east = (x1 > x2) ? x1 : x2;
    var north = (y1 > y2) ? y1 : y2;
    return [west, south, east, north];
}

/**
 * Takes any LineString or Polygon GeoJSON and returns the intersecting point(s).
 *
 * @name lineIntersect
 * @param {Geometry|FeatureCollection|Feature<LineString|MultiLineString|Polygon|MultiPolygon>} line1 any LineString or Polygon
 * @param {Geometry|FeatureCollection|Feature<LineString|MultiLineString|Polygon|MultiPolygon>} line2 any LineString or Polygon
 * @returns {FeatureCollection<Point>} point(s) that intersect both
 * @example
 * var line1 = turf.lineString([[126, -11], [129, -21]]);
 * var line2 = turf.lineString([[123, -18], [131, -14]]);
 * var intersects = turf.lineIntersect(line1, line2);
 *
 * //addToMap
 * var addToMap = [line1, line2, intersects]
 */
function lineIntersect(line1, line2) {
    var unique = {};
    var results = [];

    // First, normalize geometries to features
    // Then, handle simple 2-vertex segments
    if (line1.type === 'LineString') line1 = feature(line1);
    if (line2.type === 'LineString') line2 = feature(line2);
    if (line1.type === 'Feature' &&
        line2.type === 'Feature' &&
        line1.geometry.type === 'LineString' &&
        line2.geometry.type === 'LineString' &&
        line1.geometry.coordinates.length === 2 &&
        line2.geometry.coordinates.length === 2) {
        var intersect = intersects$3(line1, line2);
        if (intersect) results.push(intersect);
        return featureCollection(results);
    }

    // Handles complex GeoJSON Geometries
    var tree = geojsonRbush();
    tree.load(lineSegment(line2));
    featureEach(lineSegment(line1), function (segment) {
        featureEach(tree.search(segment), function (match) {
            var intersect = intersects$3(segment, match);
            if (intersect) {
                // prevent duplicate points https://github.com/Turfjs/turf/issues/688
                var key = getCoords(intersect).join(',');
                if (!unique[key]) {
                    unique[key] = true;
                    results.push(intersect);
                }
            }
        });
    });
    return featureCollection(results);
}

/**
 * Find a point that intersects LineStrings with two coordinates each
 *
 * @private
 * @param {Feature<LineString>} line1 GeoJSON LineString (Must only contain 2 coordinates)
 * @param {Feature<LineString>} line2 GeoJSON LineString (Must only contain 2 coordinates)
 * @returns {Feature<Point>} intersecting GeoJSON Point
 */
function intersects$3(line1, line2) {
    var coords1 = getCoords(line1);
    var coords2 = getCoords(line2);
    if (coords1.length !== 2) {
        throw new Error('<intersects> line1 must only contain 2 coordinates');
    }
    if (coords2.length !== 2) {
        throw new Error('<intersects> line2 must only contain 2 coordinates');
    }
    var x1 = coords1[0][0];
    var y1 = coords1[0][1];
    var x2 = coords1[1][0];
    var y2 = coords1[1][1];
    var x3 = coords2[0][0];
    var y3 = coords2[0][1];
    var x4 = coords2[1][0];
    var y4 = coords2[1][1];
    var denom = ((y4 - y3) * (x2 - x1)) - ((x4 - x3) * (y2 - y1));
    var numeA = ((x4 - x3) * (y1 - y3)) - ((y4 - y3) * (x1 - x3));
    var numeB = ((x2 - x1) * (y1 - y3)) - ((y2 - y1) * (x1 - x3));

    if (denom === 0) {
        if (numeA === 0 && numeB === 0) {
            return null;
        }
        return null;
    }

    var uA = numeA / denom;
    var uB = numeB / denom;

    if (uA >= 0 && uA <= 1 && uB >= 0 && uB <= 1) {
        var x = x1 + (uA * (x2 - x1));
        var y = y1 + (uA * (y2 - y1));
        return point([x, y]);
    }
    return null;
}

/**
 * Takes a {@link Point} and a {@link LineString} and calculates the closest Point on the (Multi)LineString.
 *
 * @name nearestPointOnLine
 * @param {Geometry|Feature<LineString|MultiLineString>} lines lines to snap to
 * @param {Geometry|Feature<Point>|number[]} pt point to snap from
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units='kilometers'] can be degrees, radians, miles, or kilometers
 * @returns {Feature<Point>} closest point on the `line` to `point`. The properties object will contain three values: `index`: closest point was found on nth line part, `dist`: distance between pt and the closest point, `location`: distance along the line between start and the closest point.
 * @example
 * var line = turf.lineString([
 *     [-77.031669, 38.878605],
 *     [-77.029609, 38.881946],
 *     [-77.020339, 38.884084],
 *     [-77.025661, 38.885821],
 *     [-77.021884, 38.889563],
 *     [-77.019824, 38.892368]
 * ]);
 * var pt = turf.point([-77.037076, 38.884017]);
 *
 * var snapped = turf.nearestPointOnLine(line, pt, {units: 'miles'});
 *
 * //addToMap
 * var addToMap = [line, pt, snapped];
 * snapped.properties['marker-color'] = '#00f';
 */
function nearestPointOnLine(lines, pt, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');

    // validation
    var type = (lines.geometry) ? lines.geometry.type : lines.type;
    if (type !== 'LineString' && type !== 'MultiLineString') {
        throw new Error('lines must be LineString or MultiLineString');
    }

    var closestPt = point([Infinity, Infinity], {
        dist: Infinity
    });

    var length = 0.0;
    flattenEach(lines, function (line) {
        var coords = getCoords(line);

        for (var i = 0; i < coords.length - 1; i++) {
            //start
            var start = point(coords[i]);
            start.properties.dist = distance(pt, start, options);
            //stop
            var stop = point(coords[i + 1]);
            stop.properties.dist = distance(pt, stop, options);
            // sectionLength
            var sectionLength = distance(start, stop, options);
            //perpendicular
            var heightDistance = Math.max(start.properties.dist, stop.properties.dist);
            var direction = bearing(start, stop);
            var perpendicularPt1 = destination(pt, heightDistance, direction + 90, options);
            var perpendicularPt2 = destination(pt, heightDistance, direction - 90, options);
            var intersect = lineIntersect(
                lineString([perpendicularPt1.geometry.coordinates, perpendicularPt2.geometry.coordinates]),
                lineString([start.geometry.coordinates, stop.geometry.coordinates])
            );
            var intersectPt = null;
            if (intersect.features.length > 0) {
                intersectPt = intersect.features[0];
                intersectPt.properties.dist = distance(pt, intersectPt, options);
                intersectPt.properties.location = length + distance(start, intersectPt, options);
            }

            if (start.properties.dist < closestPt.properties.dist) {
                closestPt = start;
                closestPt.properties.index = i;
                closestPt.properties.location = length;
            }
            if (stop.properties.dist < closestPt.properties.dist) {
                closestPt = stop;
                closestPt.properties.index = i + 1;
                closestPt.properties.location = length + sectionLength;
            }
            if (intersectPt && intersectPt.properties.dist < closestPt.properties.dist) {
                closestPt = intersectPt;
                closestPt.properties.index = i;
            }
            // update length
            length += sectionLength;
        }

    });

    return closestPt;
}

// https://en.wikipedia.org/wiki/Rhumb_line
/**
 * Takes two {@link Point|points} and finds the bearing angle between them along a Rhumb line
 * i.e. the angle measured in degrees start the north line (0 degrees)
 *
 * @name rhumbBearing
 * @param {Coord} start starting Point
 * @param {Coord} end ending Point
 * @param {Object} [options] Optional parameters
 * @param {boolean} [options.final=false] calculates the final bearing if true
 * @returns {number} bearing from north in decimal degrees, between -180 and 180 degrees (positive clockwise)
 * @example
 * var point1 = turf.point([-75.343, 39.984], {"marker-color": "#F00"});
 * var point2 = turf.point([-75.534, 39.123], {"marker-color": "#00F"});
 *
 * var bearing = turf.rhumbBearing(point1, point2);
 *
 * //addToMap
 * var addToMap = [point1, point2];
 * point1.properties.bearing = bearing;
 * point2.properties.bearing = bearing;
 */
function rhumbBearing(start, end, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var final = options.final;

    // validation
    if (!start) throw new Error('start point is required');
    if (!end) throw new Error('end point is required');

    var bear360;

    if (final) bear360 = calculateRhumbBearing(getCoord(end), getCoord(start));
    else bear360 = calculateRhumbBearing(getCoord(start), getCoord(end));

    var bear180 = (bear360 > 180) ? -(360 - bear360) : bear360;

    return bear180;
}

/**
 * Returns the bearing from ‘this’ point to destination point along a rhumb line.
 * Adapted from Geodesy: https://github.com/chrisveness/geodesy/blob/master/latlon-spherical.js
 *
 * @private
 * @param   {Array<number>} from - origin point.
 * @param   {Array<number>} to - destination point.
 * @returns {number} Bearing in degrees from north.
 * @example
 * var p1 = new LatLon(51.127, 1.338);
 * var p2 = new LatLon(50.964, 1.853);
 * var d = p1.rhumbBearingTo(p2); // 116.7 m
 */
function calculateRhumbBearing(from, to) {
    // φ => phi
    // Δλ => deltaLambda
    // Δψ => deltaPsi
    // θ => theta
    var phi1 = degreesToRadians(from[1]);
    var phi2 = degreesToRadians(to[1]);
    var deltaLambda = degreesToRadians((to[0] - from[0]));
    // if deltaLambdaon over 180° take shorter rhumb line across the anti-meridian:
    if (deltaLambda > Math.PI) deltaLambda -= 2 * Math.PI;
    if (deltaLambda < -Math.PI) deltaLambda += 2 * Math.PI;

    var deltaPsi = Math.log(Math.tan(phi2 / 2 + Math.PI / 4) / Math.tan(phi1 / 2 + Math.PI / 4));

    var theta = Math.atan2(deltaLambda, deltaPsi);

    return (radiansToDegrees(theta) + 360) % 360;
}

// https://en.wikipedia.org/wiki/Rhumb_line
/**
 * Calculates the distance along a rhumb line between two {@link Point|points} in degrees, radians,
 * miles, or kilometers.
 *
 * @name rhumbDistance
 * @param {Coord} from origin point
 * @param {Coord} to destination point
 * @param {Object} [options] Optional parameters
 * @param {string} [options.units="kilometers"] can be degrees, radians, miles, or kilometers
 * @returns {number} distance between the two points
 * @example
 * var from = turf.point([-75.343, 39.984]);
 * var to = turf.point([-75.534, 39.123]);
 * var options = {units: 'miles'};
 *
 * var distance = turf.rhumbDistance(from, to, options);
 *
 * //addToMap
 * var addToMap = [from, to];
 * from.properties.distance = distance;
 * to.properties.distance = distance;
 */
function rhumbDistance(from, to, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var units = options.units;

    // validation
    if (!from) throw new Error('from point is required');
    if (!to) throw new Error('to point is required');

    var origin = getCoord(from);
    var destination = getCoord(to);

    // compensate the crossing of the 180th meridian (https://macwright.org/2016/09/26/the-180th-meridian.html)
    // solution from https://github.com/mapbox/mapbox-gl-js/issues/3250#issuecomment-294887678
    destination[0] += (destination[0] - origin[0] > 180) ? -360 : (origin[0] - destination[0] > 180) ? 360 : 0;
    var distanceInMeters = calculateRhumbDistance(origin, destination);
    var distance = convertLength(distanceInMeters, 'meters', units);
    return distance;
}

/**
 * Returns the distance travelling from ‘this’ point to destination point along a rhumb line.
 * Adapted from Geodesy: https://github.com/chrisveness/geodesy/blob/master/latlon-spherical.js
 *
 * @private
 * @param   {Array<number>} origin point.
 * @param   {Array<number>} destination point.
 * @param   {number} [radius=6371e3] - (Mean) radius of earth (defaults to radius in metres).
 * @returns {number} Distance in km between this point and destination point (same units as radius).
 *
 * @example
 *     var p1 = new LatLon(51.127, 1.338);
 *     var p2 = new LatLon(50.964, 1.853);
 *     var d = p1.distanceTo(p2); // 40.31 km
 */
function calculateRhumbDistance(origin, destination, radius) {
    // φ => phi
    // λ => lambda
    // ψ => psi
    // Δ => Delta
    // δ => delta
    // θ => theta

    radius = (radius === undefined) ? earthRadius : Number(radius);
    // see www.edwilliams.org/avform.htm#Rhumb

    var R = radius;
    var phi1 = origin[1] * Math.PI / 180;
    var phi2 = destination[1] * Math.PI / 180;
    var DeltaPhi = phi2 - phi1;
    var DeltaLambda = Math.abs(destination[0] - origin[0]) * Math.PI / 180;
    // if dLon over 180° take shorter rhumb line across the anti-meridian:
    if (DeltaLambda > Math.PI) DeltaLambda -= 2 * Math.PI;

    // on Mercator projection, longitude distances shrink by latitude; q is the 'stretch factor'
    // q becomes ill-conditioned along E-W line (0/0); use empirical tolerance to avoid it
    var DeltaPsi = Math.log(Math.tan(phi2 / 2 + Math.PI / 4) / Math.tan(phi1 / 2 + Math.PI / 4));
    var q = Math.abs(DeltaPsi) > 10e-12 ? DeltaPhi / DeltaPsi : Math.cos(phi1);

    // distance is pythagoras on 'stretched' Mercator projection
    var delta = Math.sqrt(DeltaPhi * DeltaPhi + q * q * DeltaLambda * DeltaLambda); // angular distance in radians
    var dist = delta * R;

    return dist;
}

/**
 * Converts a WGS84 GeoJSON object into Mercator (EPSG:900913) projection
 *
 * @name toMercator
 * @param {GeoJSON|Position} geojson WGS84 GeoJSON object
 * @param {Object} [options] Optional parameters
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated (significant performance increase if true)
 * @returns {GeoJSON} true/false
 * @example
 * var pt = turf.point([-71,41]);
 * var converted = turf.toMercator(pt);
 *
 * //addToMap
 * var addToMap = [pt, converted];
 */
function toMercator(geojson, options) {
    return convert(geojson, 'mercator', options);
}

/**
 * Converts a Mercator (EPSG:900913) GeoJSON object into WGS84 projection
 *
 * @name toWgs84
 * @param {GeoJSON|Position} geojson Mercator GeoJSON object
 * @param {Object} [options] Optional parameters
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated (significant performance increase if true)
 * @returns {GeoJSON} true/false
 * @example
 * var pt = turf.point([-7903683.846322424, 5012341.663847514]);
 * var converted = turf.toWgs84(pt);
 *
 * //addToMap
 * var addToMap = [pt, converted];
 */
function toWgs84(geojson, options) {
    return convert(geojson, 'wgs84', options);
}


/**
 * Converts a GeoJSON coordinates to the defined `projection`
 *
 * @private
 * @param {GeoJSON} geojson GeoJSON Feature or Geometry
 * @param {string} projection defines the projection system to convert the coordinates to
 * @param {Object} [options] Optional parameters
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated (significant performance increase if true)
 * @returns {GeoJSON} true/false
 */
function convert(geojson, projection, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var mutate = options.mutate;

    // Validation
    if (!geojson) throw new Error('geojson is required');

    // Handle Position
    if (Array.isArray(geojson) && isNumber(geojson[0])) geojson = (projection === 'mercator') ? convertToMercator(geojson) : convertToWgs84(geojson);

    // Handle GeoJSON
    else {
        // Handle possible data mutation
        if (mutate !== true) geojson = clone(geojson);

        coordEach(geojson, function (coord) {
            var newCoord = (projection === 'mercator') ? convertToMercator(coord) : convertToWgs84(coord);
            coord[0] = newCoord[0];
            coord[1] = newCoord[1];
        });
    }
    return geojson;
}

/**
 * Convert lon/lat values to 900913 x/y.
 * (from https://github.com/mapbox/sphericalmercator)
 *
 * @private
 * @param {Array<number>} lonLat WGS84 point
 * @returns {Array<number>} Mercator [x, y] point
 */
function convertToMercator(lonLat) {
    var D2R = Math.PI / 180,
        // 900913 properties
        A = 6378137.0,
        MAXEXTENT = 20037508.342789244;

    // compensate longitudes passing the 180th meridian
    // from https://github.com/proj4js/proj4js/blob/master/lib/common/adjust_lon.js
    var adjusted = (Math.abs(lonLat[0]) <= 180) ? lonLat[0] : (lonLat[0] - (sign(lonLat[0]) * 360));
    var xy = [
        A * adjusted * D2R,
        A * Math.log(Math.tan((Math.PI * 0.25) + (0.5 * lonLat[1] * D2R)))
    ];

    // if xy value is beyond maxextent (e.g. poles), return maxextent
    if (xy[0] > MAXEXTENT) xy[0] = MAXEXTENT;
    if (xy[0] < -MAXEXTENT) xy[0] = -MAXEXTENT;
    if (xy[1] > MAXEXTENT) xy[1] = MAXEXTENT;
    if (xy[1] < -MAXEXTENT) xy[1] = -MAXEXTENT;

    return xy;
}

/**
 * Convert 900913 x/y values to lon/lat.
 * (from https://github.com/mapbox/sphericalmercator)
 *
 * @private
 * @param {Array<number>} xy Mercator [x, y] point
 * @returns {Array<number>} WGS84 [lon, lat] point
 */
function convertToWgs84(xy) {
    // 900913 properties.
    var R2D = 180 / Math.PI;
    var A = 6378137.0;

    return [
        (xy[0] * R2D / A),
        ((Math.PI * 0.5) - 2.0 * Math.atan(Math.exp(-xy[1] / A))) * R2D
    ];
}

/**
 * Returns the sign of the input, or zero
 *
 * @private
 * @param {number} x input
 * @returns {number} -1|0|1 output
 */
function sign(x) {
    return (x < 0) ? -1 : (x > 0) ? 1 : 0;
}




var main_es$3 = Object.freeze({
	toMercator: toMercator,
	toWgs84: toWgs84
});

// (logic of computation inspired by:
// https://stackoverflow.com/questions/32771458/distance-from-lat-lng-point-to-minor-arc-segment)

/**
 * Returns the minimum distance between a {@link Point} and a {@link LineString}, being the distance from a line the
 * minimum distance between the point and any segment of the `LineString`.
 *
 * @name pointToLineDistance
 * @param {Coord} pt Feature or Geometry
 * @param {Feature<LineString>} line GeoJSON Feature or Geometry
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units='kilometers'] can be degrees, radians, miles, or kilometers
 * @param {boolean} [options.mercator=false] if distance should be on Mercator or WGS84 projection
 * @returns {number} distance between point and line
 * @example
 * var pt = turf.point([0, 0]);
 * var line = turf.lineString([[1, 1],[-1, 1]]);
 *
 * var distance = turf.pointToLineDistance(pt, line, {units: 'miles'});
 * //=69.11854715938406
 */
function pointToLineDistance(pt, line, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');

    // validation
    if (!pt) throw new Error('pt is required');
    if (Array.isArray(pt)) pt = point(pt);
    else if (pt.type === 'Point') pt = feature(pt);
    else featureOf(pt, 'Point', 'point');

    if (!line) throw new Error('line is required');
    if (Array.isArray(line)) line = lineString(line);
    else if (line.type === 'LineString') line = feature(line);
    else featureOf(line, 'LineString', 'line');

    var distance$$1 = Infinity;
    var p = pt.geometry.coordinates;
    segmentEach(line, function (segment) {
        var a = segment.geometry.coordinates[0];
        var b = segment.geometry.coordinates[1];
        var d = distanceToSegment(p, a, b, options);
        if (distance$$1 > d) distance$$1 = d;
    });
    return distance$$1;
}


/**
 * Returns the distance between a point P on a segment AB.
 *
 * @private
 * @param {Array<number>} p external point
 * @param {Array<number>} a first segment point
 * @param {Array<number>} b second segment point
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units='kilometers'] can be degrees, radians, miles, or kilometers
 * @param {boolean} [options.mercator=false] if distance should be on Mercator or WGS84 projection
 * @returns {number} distance
 */
function distanceToSegment(p, a, b, options) {
    var mercator = options.mercator;
    var distanceAP = (mercator !== true) ? distance(a, p, options) : euclideanDistance(a, p, options);
    var azimuthAP = bearingToAzimuth((mercator !== true) ? bearing(a, p) : rhumbBearing(a, p));
    var azimuthAB = bearingToAzimuth((mercator !== true) ? bearing(a, b) : rhumbBearing(a, b));
    var angleA = Math.abs(azimuthAP - azimuthAB);
    // if (angleA > 180) angleA = Math.abs(angleA - 360);
    // if the angle PAB is obtuse its projection on the line extending the segment falls outside the segment
    // thus return distance between P and the start point A
    /*
        P__
        |\ \____
        | \     \____
        |  \         \____
        |   \_____________\
        H    A             B
     */
    if (angleA > 90) return distanceAP;

    var azimuthBA = (azimuthAB + 180) % 360;
    var azimuthBP = bearingToAzimuth((mercator !== true) ? bearing(b, p) : rhumbBearing(b, p));
    var angleB = Math.abs(azimuthBP - azimuthBA);
    if (angleB > 180) angleB = Math.abs(angleB - 360);
    // also if the angle ABP is acute the projection of P falls outside the segment, on the other side
    // so return the distance between P and the end point B
    /*
                        ____P
                   ____/   /|
              ____/       / |
         ____/           /  |
        /______________/    |
       A               B    H
    */
    if (angleB > 90) return (mercator !== true) ? distance(p, b, options) : euclideanDistance(p, b, options);
    // finally if the projection falls inside the segment
    // return the distance between P and the segment
    /*
                     P
                  __/|\
               __/   | \
            __/      |  \
         __/         |   \
        /____________|____\
       A             H     B
    */
    if (mercator !== true) return distanceAP * Math.sin(degreesToRadians(angleA));
    return mercatorPH(a, b, p, options);
}

/**
 * Returns the distance between a point P on a segment AB, on Mercator projection
 *
 * @private
 * @param {Array<number>} a first segment point
 * @param {Array<number>} b second segment point
 * @param {Array<number>} p external point
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units='kilometers'] can be degrees, radians, miles, or kilometers
 * @returns {number} distance
 */
function mercatorPH(a, b, p, options) {
    var delta = 0;
    // translate points if any is crossing the 180th meridian
    if (Math.abs(a[0]) >= 180 || Math.abs(b[0]) >= 180 || Math.abs(p[0]) >= 180) {
        delta = (a[0] > 0 || b[0] > 0 || p[0] > 0) ? -180 : 180;
    }

    var origin = point(p);
    var A = toMercator([a[0] + delta, a[1]]);
    var B = toMercator([b[0] + delta, b[1]]);
    var P = toMercator([p[0] + delta, p[1]]);
    var h = toWgs84(euclideanIntersection(A, B, P));

    if (delta !== 0) h[0] -= delta; // translate back to original position
    var distancePH = rhumbDistance(origin, h, options);
    return distancePH;
}

/**
 * Returns the point H projection of a point P on a segment AB, on the euclidean plain
 * from https://stackoverflow.com/questions/10301001/perpendicular-on-a-line-segment-from-a-given-point#answer-12499474
 *            P
 *           |
 *           |
 *  _________|____
 * A          H   B
 *
 * @private
 * @param {Array<number>} a first segment point
 * @param {Array<number>} b second segment point
 * @param {Array<number>} p external point
 * @returns {Array<number>} projected point
 */
function euclideanIntersection(a, b, p) {
    var x1 = a[0], y1 = a[1],
        x2 = b[0], y2 = b[1],
        x3 = p[0], y3 = p[1];
    var px = x2 - x1, py = y2 - y1;
    var dab = px * px + py * py;
    var u = ((x3 - x1) * px + (y3 - y1) * py) / dab;
    var x = x1 + u * px, y = y1 + u * py;
    return [x, y]; // H
}

/**
 * Returns euclidean distance between two points
 *
 * @private
 * @param {Object} from first point
 * @param {Object} to second point
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units='kilometers'] can be degrees, radians, miles, or kilometers
 * @returns {number} squared distance
 */
function euclideanDistance(from, to, options) {
    var units = options.units;
    // translate points if any is crossing the 180th meridian
    var delta = 0;
    if (Math.abs(from[0]) >= 180) {
        delta = (from[0] > 0) ? -180 : 180;
    }
    if (Math.abs(to[0]) >= 180) {
        delta = (to[0] > 0) ? -180 : 180;
    }
    var p1 = toMercator([from[0] + delta, from[1]]);
    var p2 = toMercator([to[0] + delta, to[1]]);

    var sqr = function (n) { return n * n; };
    var squareD = sqr(p1[0] - p2[0]) + sqr(p1[1] - p2[1]);
    var d = Math.sqrt(squareD);
    return convertLength(d, 'meters', units);
}

/**
 * Returns the closest {@link Point|point}, of a {@link FeatureCollection|collection} of points, to a {@link LineString|line}.
 * The returned point has a `dist` property indicating its distance to the line.
 *
 * @name nearestPointToLine
 * @param {FeatureCollection|GeometryCollection<Point>} points Point Collection
 * @param {Feature|Geometry<LineString>} line Line Feature
 * @param {Object} [options] Optional parameters
 * @param {string} [options.units='kilometers'] unit of the output distance property, can be degrees, radians, miles, or kilometers
 * @param {Object} [options.properties={}] Translate Properties to Point
 * @returns {Feature<Point>} the closest point
 * @example
 * var pt1 = turf.point([0, 0]);
 * var pt2 = turf.point([0.5, 0.5]);
 * var points = turf.featureCollection([pt1, pt2]);
 * var line = turf.lineString([[1,1], [-1,1]]);
 *
 * var nearest = turf.nearestPointToLine(points, line);
 *
 * //addToMap
 * var addToMap = [nearest, line];
 */
function nearestPointToLine(points$$1, line, options) {
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var units = options.units;
    var properties = options.properties || {};

    // validation
    if (!points$$1) throw new Error('points is required');
    points$$1 = normalize(points$$1);
    if (!points$$1.features.length) throw new Error('points must contain features');

    if (!line) throw new Error('line is required');
    if (getType(line) !== 'LineString') throw new Error('line must be a LineString');

    var dist = Infinity;
    var pt = null;

    featureEach(points$$1, function (point$$1) {
        var d = pointToLineDistance(point$$1, line, { units: units });
        if (d < dist) {
            dist = d;
            pt = point$$1;
        }
    });
    /**
     * Translate Properties to final Point, priorities:
     * 1. options.properties
     * 2. inherent Point properties
     * 3. dist custom properties created by NearestPointToLine
     */
    if (pt) pt.properties = Object.assign({dist: dist}, pt.properties, properties);
    return pt;
}

/**
 * Convert Collection to FeatureCollection
 *
 * @private
 * @param {FeatureCollection|GeometryCollection<Point>} points Points
 * @returns {FeatureCollection<Point>} points
 */
function normalize(points$$1) {
    var features = [];
    var type = points$$1.geometry ? points$$1.geometry.type : points$$1.type;
    switch (type) {
    case 'GeometryCollection':
        geomEach(points$$1, function (geom) {
            if (geom.type === 'Point') features.push({type: 'Feature', properties: {}, geometry: geom});
        });
        return {type: 'FeatureCollection', features: features};
    case 'FeatureCollection':
        points$$1.features = points$$1.features.filter(function (feature$$1) {
            return feature$$1.geometry.type === 'Point';
        });
        return points$$1;
    default:
        throw new Error('points must be a Point Collection');
    }
}

/**
 * Takes a triangular plane as a {@link Polygon}
 * and a {@link Point} within that triangle and returns the z-value
 * at that point. The Polygon should have properties `a`, `b`, and `c`
 * that define the values at its three corners. Alternatively, the z-values
 * of each triangle point can be provided by their respective 3rd coordinate
 * if their values are not provided as properties.
 *
 * @name planepoint
 * @param {Coord} point the Point for which a z-value will be calculated
 * @param {Feature<Polygon>} triangle a Polygon feature with three vertices
 * @returns {number} the z-value for `interpolatedPoint`
 * @example
 * var point = turf.point([-75.3221, 39.529]);
 * // "a", "b", and "c" values represent the values of the coordinates in order.
 * var triangle = turf.polygon([[
 *   [-75.1221, 39.57],
 *   [-75.58, 39.18],
 *   [-75.97, 39.86],
 *   [-75.1221, 39.57]
 * ]], {
 *   "a": 11,
 *   "b": 122,
 *   "c": 44
 * });
 *
 * var zValue = turf.planepoint(point, triangle);
 * point.properties.zValue = zValue;
 *
 * //addToMap
 * var addToMap = [triangle, point];
 */
function planepoint(point, triangle) {
    // Normalize input
    var coord = getCoord(point);
    var geom = getGeom(triangle);
    var coords = geom.coordinates;
    var outer = coords[0];
    if (outer.length < 4) throw new Error('OuterRing of a Polygon must have 4 or more Positions.');
    var properties = triangle.properties || {};
    var a = properties.a;
    var b = properties.b;
    var c = properties.c;

    // Planepoint
    var x = coord[0];
    var y = coord[1];
    var x1 = outer[0][0];
    var y1 = outer[0][1];
    var z1 = (a !== undefined ? a : outer[0][2]);
    var x2 = outer[1][0];
    var y2 = outer[1][1];
    var z2 = (b !== undefined ? b : outer[1][2]);
    var x3 = outer[2][0];
    var y3 = outer[2][1];
    var z3 = (c !== undefined ? c : outer[2][2]);
    var z = (z3 * (x - x1) * (y - y2) + z1 * (x - x2) * (y - y3) + z2 * (x - x3) * (y - y1) -
             z2 * (x - x1) * (y - y3) - z3 * (x - x2) * (y - y1) - z1 * (x - x3) * (y - y2)) /
           ((x - x1) * (y - y2) + (x - x2) * (y - y3) + (x - x3) * (y - y1) -
            (x - x1) * (y - y3) - (x - x2) * (y - y1) - (x - x3) * (y - y2));

    return z;
}

/**
 * Takes a {@link LineString|linestring}, {@link MultiLineString|multi-linestring}, {@link MultiPolygon|multi-polygon}, or {@link Polygon|polygon} and returns {@link Point|points} at all self-intersections.
 *
 * @name kinks
 * @param {Feature<LineString|MultiLineString|MultiPolygon|Polygon>} featureIn input feature
 * @returns {FeatureCollection<Point>} self-intersections
 * @example
 * var poly = turf.polygon([[
 *   [-12.034835, 8.901183],
 *   [-12.060413, 8.899826],
 *   [-12.03638, 8.873199],
 *   [-12.059383, 8.871418],
 *   [-12.034835, 8.901183]
 * ]]);
 *
 * var kinks = turf.kinks(poly);
 *
 * //addToMap
 * var addToMap = [poly, kinks]
 */
function kinks(featureIn) {
    var coordinates;
    var feature$$1;
    var results = {
        type: 'FeatureCollection',
        features: []
    };
    if (featureIn.type === 'Feature') {
        feature$$1 = featureIn.geometry;
    } else {
        feature$$1 = featureIn;
    }
    if (feature$$1.type === 'LineString') {
        coordinates = [feature$$1.coordinates];
    } else if (feature$$1.type === 'MultiLineString') {
        coordinates = feature$$1.coordinates;
    } else if (feature$$1.type === 'MultiPolygon') {
        coordinates = [].concat.apply([], feature$$1.coordinates);
    } else if (feature$$1.type === 'Polygon') {
        coordinates = feature$$1.coordinates;
    } else {
        throw new Error('Input must be a LineString, MultiLineString, ' +
            'Polygon, or MultiPolygon Feature or Geometry');
    }
    coordinates.forEach(function (line1) {
        coordinates.forEach(function (line2) {
            for (var i = 0; i < line1.length - 1; i++) {
                // start iteration at i, intersections for k < i have already been checked in previous outer loop iterations
                for (var k = i; k < line2.length - 1; k++) {
                    if (line1 === line2) {
                        // segments are adjacent and always share a vertex, not a kink
                        if (Math.abs(i - k) === 1) {
                            continue;
                        }
                        // first and last segment in a closed lineString or ring always share a vertex, not a kink
                        if (
                            // segments are first and last segment of lineString
                            i === 0 &&
                            k === line1.length - 2 &&
                            // lineString is closed
                            line1[i][0] === line1[line1.length - 1][0] &&
                            line1[i][1] === line1[line1.length - 1][1]
                        ) {
                            continue;
                        }
                    }

                    var intersection = lineIntersects(line1[i][0], line1[i][1], line1[i + 1][0], line1[i + 1][1],
                        line2[k][0], line2[k][1], line2[k + 1][0], line2[k + 1][1]);
                    if (intersection) {
                        results.features.push(point([intersection[0], intersection[1]]));
                    }
                }
            }
        });
    });
    return results;
}


// modified from http://jsfiddle.net/justin_c_rounds/Gd2S2/light/
function lineIntersects(line1StartX, line1StartY, line1EndX, line1EndY, line2StartX, line2StartY, line2EndX, line2EndY) {
    // if the lines intersect, the result contains the x and y of the intersection (treating the lines as infinite) and booleans for whether line segment 1 or line segment 2 contain the point
    var denominator, a, b, numerator1, numerator2,
        result = {
            x: null,
            y: null,
            onLine1: false,
            onLine2: false
        };
    denominator = ((line2EndY - line2StartY) * (line1EndX - line1StartX)) - ((line2EndX - line2StartX) * (line1EndY - line1StartY));
    if (denominator === 0) {
        if (result.x !== null && result.y !== null) {
            return result;
        } else {
            return false;
        }
    }
    a = line1StartY - line2StartY;
    b = line1StartX - line2StartX;
    numerator1 = ((line2EndX - line2StartX) * a) - ((line2EndY - line2StartY) * b);
    numerator2 = ((line1EndX - line1StartX) * a) - ((line1EndY - line1StartY) * b);
    a = numerator1 / denominator;
    b = numerator2 / denominator;

    // if we cast these lines infinitely in both directions, they intersect here:
    result.x = line1StartX + (a * (line1EndX - line1StartX));
    result.y = line1StartY + (a * (line1EndY - line1StartY));

    // if line1 is a segment and line2 is infinite, they intersect if:
    if (a >= 0 && a <= 1) {
        result.onLine1 = true;
    }
    // if line2 is a segment and line1 is infinite, they intersect if:
    if (b >= 0 && b <= 1) {
        result.onLine2 = true;
    }
    // if line1 and line2 are segments, they intersect if both of the above are true
    if (result.onLine1 && result.onLine2) {
        return [result.x, result.y];
    } else {
        return false;
    }
}

/**
 * Takes a Feature or FeatureCollection and returns a {@link Point} guaranteed to be on the surface of the feature.
 *
 * * Given a {@link Polygon}, the point will be in the area of the polygon
 * * Given a {@link LineString}, the point will be along the string
 * * Given a {@link Point}, the point will the same as the input
 *
 * @name pointOnFeature
 * @param {GeoJSON} geojson any Feature or FeatureCollection
 * @returns {Feature<Point>} a point on the surface of `input`
 * @example
 * var polygon = turf.polygon([[
 *   [116, -36],
 *   [131, -32],
 *   [146, -43],
 *   [155, -25],
 *   [133, -9],
 *   [111, -22],
 *   [116, -36]
 * ]]);
 *
 * var pointOnPolygon = turf.pointOnFeature(polygon);
 *
 * //addToMap
 * var addToMap = [polygon, pointOnPolygon];
 */
function pointOnFeature(geojson) {
    // normalize
    var fc = normalize$1(geojson);

    // get centroid
    var cent = center(fc);

    // check to see if centroid is on surface
    var onSurface = false;
    var i = 0;
    while (!onSurface && i < fc.features.length) {
        var geom = fc.features[i].geometry;
        var x, y, x1, y1, x2, y2, k;
        var onLine = false;
        if (geom.type === 'Point') {
            if (cent.geometry.coordinates[0] === geom.coordinates[0] &&
                cent.geometry.coordinates[1] === geom.coordinates[1]) {
                onSurface = true;
            }
        } else if (geom.type === 'MultiPoint') {
            var onMultiPoint = false;
            k = 0;
            while (!onMultiPoint && k < geom.coordinates.length) {
                if (cent.geometry.coordinates[0] === geom.coordinates[k][0] &&
          cent.geometry.coordinates[1] === geom.coordinates[k][1]) {
                    onSurface = true;
                    onMultiPoint = true;
                }
                k++;
            }
        } else if (geom.type === 'LineString') {
            k = 0;
            while (!onLine && k < geom.coordinates.length - 1) {
                x = cent.geometry.coordinates[0];
                y = cent.geometry.coordinates[1];
                x1 = geom.coordinates[k][0];
                y1 = geom.coordinates[k][1];
                x2 = geom.coordinates[k + 1][0];
                y2 = geom.coordinates[k + 1][1];
                if (pointOnSegment(x, y, x1, y1, x2, y2)) {
                    onLine = true;
                    onSurface = true;
                }
                k++;
            }
        } else if (geom.type === 'MultiLineString') {
            var j = 0;
            while (j < geom.coordinates.length) {
                onLine = false;
                k = 0;
                var line = geom.coordinates[j];
                while (!onLine && k < line.length - 1) {
                    x = cent.geometry.coordinates[0];
                    y = cent.geometry.coordinates[1];
                    x1 = line[k][0];
                    y1 = line[k][1];
                    x2 = line[k + 1][0];
                    y2 = line[k + 1][1];
                    if (pointOnSegment(x, y, x1, y1, x2, y2)) {
                        onLine = true;
                        onSurface = true;
                    }
                    k++;
                }
                j++;
            }
        } else if (geom.type === 'Polygon' || geom.type === 'MultiPolygon') {
            if (booleanPointInPolygon(cent, geom)) {
                onSurface = true;
            }
        }
        i++;
    }
    if (onSurface) {
        return cent;
    } else {
        var vertices = featureCollection([]);
        for (i = 0; i < fc.features.length; i++) {
            vertices.features = vertices.features.concat(explode(fc.features[i]).features);
        }
        // Remove distanceToPoint properties from nearestPoint()
        return point(nearestPoint(cent, vertices).geometry.coordinates);
    }
}

/**
 * Normalizes any GeoJSON to a FeatureCollection
 *
 * @private
 * @name normalize
 * @param {GeoJSON} geojson Any GeoJSON
 * @returns {FeatureCollection} FeatureCollection
 */
function normalize$1(geojson) {
    if (geojson.type !== 'FeatureCollection') {
        if (geojson.type !== 'Feature') {
            return featureCollection([feature(geojson)]);
        }
        return featureCollection([geojson]);
    }
    return geojson;
}

function pointOnSegment(x, y, x1, y1, x2, y2) {
    var ab = Math.sqrt((x2 - x1) * (x2 - x1) + (y2 - y1) * (y2 - y1));
    var ap = Math.sqrt((x - x1) * (x - x1) + (y - y1) * (y - y1));
    var pb = Math.sqrt((x2 - x) * (x2 - x) + (y2 - y) * (y2 - y));
    return ab === ap + pb;
}

/**
 * Takes one or more features and returns their area in square meters.
 *
 * @name area
 * @param {GeoJSON} geojson input GeoJSON feature(s)
 * @returns {number} area in square meters
 * @example
 * var polygon = turf.polygon([[[125, -15], [113, -22], [154, -27], [144, -15], [125, -15]]]);
 *
 * var area = turf.area(polygon);
 *
 * //addToMap
 * var addToMap = [polygon]
 * polygon.properties.area = area
 */
function area$1(geojson) {
    return geomReduce(geojson, function (value, geom) {
        return value + calculateArea(geom);
    }, 0);
}

var RADIUS = 6378137;
// var FLATTENING_DENOM = 298.257223563;
// var FLATTENING = 1 / FLATTENING_DENOM;
// var POLAR_RADIUS = RADIUS * (1 - FLATTENING);

/**
 * Calculate Area
 *
 * @private
 * @param {GeoJSON} geojson GeoJSON
 * @returns {number} area
 */
function calculateArea(geojson) {
    var area = 0, i;
    switch (geojson.type) {
    case 'Polygon':
        return polygonArea(geojson.coordinates);
    case 'MultiPolygon':
        for (i = 0; i < geojson.coordinates.length; i++) {
            area += polygonArea(geojson.coordinates[i]);
        }
        return area;
    case 'Point':
    case 'MultiPoint':
    case 'LineString':
    case 'MultiLineString':
        return 0;
    case 'GeometryCollection':
        for (i = 0; i < geojson.geometries.length; i++) {
            area += calculateArea(geojson.geometries[i]);
        }
        return area;
    }
}

function polygonArea(coords) {
    var area = 0;
    if (coords && coords.length > 0) {
        area += Math.abs(ringArea(coords[0]));
        for (var i = 1; i < coords.length; i++) {
            area -= Math.abs(ringArea(coords[i]));
        }
    }
    return area;
}

/**
 * @private
 * Calculate the approximate area of the polygon were it projected onto the earth.
 * Note that this area will be positive if ring is oriented clockwise, otherwise it will be negative.
 *
 * Reference:
 * Robert. G. Chamberlain and William H. Duquette, "Some Algorithms for Polygons on a Sphere", JPL Publication 07-03, Jet Propulsion
 * Laboratory, Pasadena, CA, June 2007 http://trs-new.jpl.nasa.gov/dspace/handle/2014/40409
 *
 * @param {Array<Array<number>>} coords Ring Coordinates
 * @returns {number} The approximate signed geodesic area of the polygon in square meters.
 */
function ringArea(coords) {
    var p1;
    var p2;
    var p3;
    var lowerIndex;
    var middleIndex;
    var upperIndex;
    var i;
    var area = 0;
    var coordsLength = coords.length;

    if (coordsLength > 2) {
        for (i = 0; i < coordsLength; i++) {
            if (i === coordsLength - 2) { // i = N-2
                lowerIndex = coordsLength - 2;
                middleIndex = coordsLength - 1;
                upperIndex = 0;
            } else if (i === coordsLength - 1) { // i = N-1
                lowerIndex = coordsLength - 1;
                middleIndex = 0;
                upperIndex = 1;
            } else { // i = 0 to N-3
                lowerIndex = i;
                middleIndex = i + 1;
                upperIndex = i + 2;
            }
            p1 = coords[lowerIndex];
            p2 = coords[middleIndex];
            p3 = coords[upperIndex];
            area += (rad(p3[0]) - rad(p1[0])) * Math.sin(rad(p2[1]));
        }

        area = area * RADIUS * RADIUS / 2;
    }

    return area;
}

function rad(_) {
    return _ * Math.PI / 180;
}

/**
 * Takes a {@link LineString} and returns a {@link Point} at a specified distance along the line.
 *
 * @name along
 * @param {Feature<LineString>} line input line
 * @param {number} distance distance along the line
 * @param {Object} [options] Optional parameters
 * @param {string} [options.units="kilometers"] can be degrees, radians, miles, or kilometers
 * @returns {Feature<Point>} Point `distance` `units` along the line
 * @example
 * var line = turf.lineString([[-83, 30], [-84, 36], [-78, 41]]);
 * var options = {units: 'miles'};
 *
 * var along = turf.along(line, 200, options);
 *
 * //addToMap
 * var addToMap = [along, line]
 */
function along(line, distance$$1, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');

    // Validation
    var coords;
    if (line.type === 'Feature') coords = line.geometry.coordinates;
    else if (line.type === 'LineString') coords = line.coordinates;
    else throw new Error('input must be a LineString Feature or Geometry');
    if (!isNumber(distance$$1)) throw new Error('distance must be a number');

    var travelled = 0;
    for (var i = 0; i < coords.length; i++) {
        if (distance$$1 >= travelled && i === coords.length - 1) break;
        else if (travelled >= distance$$1) {
            var overshot = distance$$1 - travelled;
            if (!overshot) return point(coords[i]);
            else {
                var direction = bearing(coords[i], coords[i - 1]) - 180;
                var interpolated = destination(coords[i], overshot, direction, options);
                return interpolated;
            }
        } else {
            travelled += distance(coords[i], coords[i + 1], options);
        }
    }
    return point(coords[coords.length - 1]);
}

/**
 * Takes a {@link GeoJSON} and measures its length in the specified units, {@link (Multi)Point}'s distance are ignored.
 *
 * @name length
 * @param {GeoJSON} geojson GeoJSON to measure
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units=kilometers] can be degrees, radians, miles, or kilometers
 * @returns {number} length of GeoJSON
 * @example
 * var line = turf.lineString([[115, -32], [131, -22], [143, -25], [150, -34]]);
 * var length = turf.length(line, {units: 'miles'});
 *
 * //addToMap
 * var addToMap = [line];
 * line.properties.distance = length;
 */
function length(geojson, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');

    // Input Validation
    if (!geojson) throw new Error('geojson is required');

    // Calculate distance from 2-vertex line segements
    return segmentReduce(geojson, function (previousValue, segment) {
        var coords = segment.geometry.coordinates;
        return previousValue + distance(coords[0], coords[1], options);
    }, 0);
}

/**
 * Takes a {@link LineString|line}, a start {@link Point}, and a stop point
 * and returns a subsection of the line in-between those points.
 * The start & stop points don't need to fall exactly on the line.
 *
 * This can be useful for extracting only the part of a route between waypoints.
 *
 * @name lineSlice
 * @param {Coord} startPt starting point
 * @param {Coord} stopPt stopping point
 * @param {Feature<LineString>|LineString} line line to slice
 * @returns {Feature<LineString>} sliced line
 * @example
 * var line = turf.lineString([
 *     [-77.031669, 38.878605],
 *     [-77.029609, 38.881946],
 *     [-77.020339, 38.884084],
 *     [-77.025661, 38.885821],
 *     [-77.021884, 38.889563],
 *     [-77.019824, 38.892368]
 * ]);
 * var start = turf.point([-77.029609, 38.881946]);
 * var stop = turf.point([-77.021884, 38.889563]);
 *
 * var sliced = turf.lineSlice(start, stop, line);
 *
 * //addToMap
 * var addToMap = [start, stop, line]
 */
function lineSlice(startPt, stopPt, line) {
    // Validation
    var coords = getCoords(line);
    if (getType(line) !== 'LineString') throw new Error('line must be a LineString');

    var startVertex = nearestPointOnLine(line, startPt);
    var stopVertex = nearestPointOnLine(line, stopPt);
    var ends;
    if (startVertex.properties.index <= stopVertex.properties.index) {
        ends = [startVertex, stopVertex];
    } else {
        ends = [stopVertex, startVertex];
    }
    var clipCoords = [ends[0].geometry.coordinates];
    for (var i = ends[0].properties.index + 1; i < ends[1].properties.index + 1; i++) {
        clipCoords.push(coords[i]);
    }
    clipCoords.push(ends[1].geometry.coordinates);
    return lineString(clipCoords, line.properties);
}

/**
 * Takes a {@link LineString|line}, a specified distance along the line to a start {@link Point},
 * and a specified  distance along the line to a stop point
 * and returns a subsection of the line in-between those points.
 *
 * This can be useful for extracting only the part of a route between two distances.
 *
 * @name lineSliceAlong
 * @param {Feature<LineString>|LineString} line input line
 * @param {number} startDist distance along the line to starting point
 * @param {number} stopDist distance along the line to ending point
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units='kilometers'] can be degrees, radians, miles, or kilometers
 * @returns {Feature<LineString>} sliced line
 * @example
 * var line = turf.lineString([[7, 45], [9, 45], [14, 40], [14, 41]]);
 * var start = 12.5;
 * var stop = 25;
 * var sliced = turf.lineSliceAlong(line, start, stop, {units: 'miles'});
 *
 * //addToMap
 * var addToMap = [line, start, stop, sliced]
 */
function lineSliceAlong(line, startDist, stopDist, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');

    var coords;
    var slice = [];

    // Validation
    if (line.type === 'Feature') coords = line.geometry.coordinates;
    else if (line.type === 'LineString') coords = line.coordinates;
    else throw new Error('input must be a LineString Feature or Geometry');

    var travelled = 0;
    var overshot, direction, interpolated;
    for (var i = 0; i < coords.length; i++) {
        if (startDist >= travelled && i === coords.length - 1) break;
        else if (travelled > startDist && slice.length === 0) {
            overshot = startDist - travelled;
            if (!overshot) {
                slice.push(coords[i]);
                return lineString(slice);
            }
            direction = bearing(coords[i], coords[i - 1]) - 180;
            interpolated = destination(coords[i], overshot, direction, options);
            slice.push(interpolated.geometry.coordinates);
        }

        if (travelled >= stopDist) {
            overshot = stopDist - travelled;
            if (!overshot) {
                slice.push(coords[i]);
                return lineString(slice);
            }
            direction = bearing(coords[i], coords[i - 1]) - 180;
            interpolated = destination(coords[i], overshot, direction, options);
            slice.push(interpolated.geometry.coordinates);
            return lineString(slice);
        }

        if (travelled >= startDist) {
            slice.push(coords[i]);
        }

        if (i === coords.length - 1) {
            return lineString(slice);
        }

        travelled += distance(coords[i], coords[i + 1], options);
    }
    return lineString(coords[coords.length - 1]);
}

/**
 * Returns true if a point is on a line. Accepts a optional parameter to ignore the start and end vertices of the linestring.
 *
 * @name booleanPointOnLine
 * @param {Coord} pt GeoJSON Point
 * @param {Feature<LineString>} line GeoJSON LineString
 * @param {Object} [options={}] Optional parameters
 * @param {boolean} [options.ignoreEndVertices=false] whether to ignore the start and end vertices.
 * @returns {boolean} true/false
 * @example
 * var pt = turf.point([0, 0]);
 * var line = turf.lineString([[-1, -1],[1, 1],[1.5, 2.2]]);
 * var isPointOnLine = turf.booleanPointOnLine(pt, line);
 * //=true
 */
function booleanPointOnLine(pt, line, options) {
    // Optional parameters
    options = options || {};
    var ignoreEndVertices = options.ignoreEndVertices;
    if (!isObject(options)) throw new Error('invalid options');

    // Validate input
    if (!pt) throw new Error('pt is required');
    if (!line) throw new Error('line is required');

    // Normalize inputs
    var ptCoords = getCoord(pt);
    var lineCoords = getCoords(line);

    // Main
    for (var i = 0; i < lineCoords.length - 1; i++) {
        var ignoreBoundary = false;
        if (ignoreEndVertices) {
            if (i === 0) ignoreBoundary = 'start';
            if (i === lineCoords.length - 2) ignoreBoundary = 'end';
            if (i === 0 && i + 1 === lineCoords.length - 1) ignoreBoundary = 'both';
        }
        if (isPointOnLineSegment$1(lineCoords[i], lineCoords[i + 1], ptCoords, ignoreBoundary)) return true;
    }
    return false;
}

// See http://stackoverflow.com/a/4833823/1979085
/**
 * @private
 * @param {Position} lineSegmentStart coord pair of start of line
 * @param {Position} lineSegmentEnd coord pair of end of line
 * @param {Position} pt coord pair of point to check
 * @param {boolean|string} excludeBoundary whether the point is allowed to fall on the line ends. If true which end to ignore.
 * @returns {boolean} true/false
 */
function isPointOnLineSegment$1(lineSegmentStart, lineSegmentEnd, pt, excludeBoundary) {
    var x = pt[0];
    var y = pt[1];
    var x1 = lineSegmentStart[0];
    var y1 = lineSegmentStart[1];
    var x2 = lineSegmentEnd[0];
    var y2 = lineSegmentEnd[1];
    var dxc = pt[0] - x1;
    var dyc = pt[1] - y1;
    var dxl = x2 - x1;
    var dyl = y2 - y1;
    var cross = dxc * dyl - dyc * dxl;
    if (cross !== 0) {
        return false;
    }
    if (!excludeBoundary) {
        if (Math.abs(dxl) >= Math.abs(dyl)) {
            return dxl > 0 ? x1 <= x && x <= x2 : x2 <= x && x <= x1;
        }
        return dyl > 0 ? y1 <= y && y <= y2 : y2 <= y && y <= y1;
    } else if (excludeBoundary === 'start') {
        if (Math.abs(dxl) >= Math.abs(dyl)) {
            return dxl > 0 ? x1 < x && x <= x2 : x2 <= x && x < x1;
        }
        return dyl > 0 ? y1 < y && y <= y2 : y2 <= y && y < y1;
    } else if (excludeBoundary === 'end') {
        if (Math.abs(dxl) >= Math.abs(dyl)) {
            return dxl > 0 ? x1 <= x && x < x2 : x2 < x && x <= x1;
        }
        return dyl > 0 ? y1 <= y && y < y2 : y2 < y && y <= y1;
    } else if (excludeBoundary === 'both') {
        if (Math.abs(dxl) >= Math.abs(dyl)) {
            return dxl > 0 ? x1 < x && x < x2 : x2 < x && x < x1;
        }
        return dyl > 0 ? y1 < y && y < y2 : y2 < y && y < y1;
    }
}

/**
 * Boolean-within returns true if the first geometry is completely within the second geometry.
 * The interiors of both geometries must intersect and, the interior and boundary of the primary (geometry a)
 * must not intersect the exterior of the secondary (geometry b).
 * Boolean-within returns the exact opposite result of the `@turf/boolean-contains`.
 *
 * @name booleanWithin
 * @param {Geometry|Feature<any>} feature1 GeoJSON Feature or Geometry
 * @param {Geometry|Feature<any>} feature2 GeoJSON Feature or Geometry
 * @returns {boolean} true/false
 * @example
 * var line = turf.lineString([[1, 1], [1, 2], [1, 3], [1, 4]]);
 * var point = turf.point([1, 2]);
 *
 * turf.booleanWithin(point, line);
 * //=true
 */
function booleanWithin(feature1, feature2) {
    var type1 = getType(feature1);
    var type2 = getType(feature2);
    var geom1 = getGeom(feature1);
    var geom2 = getGeom(feature2);

    switch (type1) {
    case 'Point':
        switch (type2) {
        case 'MultiPoint':
            return isPointInMultiPoint(geom1, geom2);
        case 'LineString':
            return booleanPointOnLine(geom1, geom2, {ignoreEndVertices: true});
        case 'Polygon':
            return booleanPointInPolygon(geom1, geom2, {ignoreBoundary: true});
        default:
            throw new Error('feature2 ' + type2 + ' geometry not supported');
        }
    case 'MultiPoint':
        switch (type2) {
        case 'MultiPoint':
            return isMultiPointInMultiPoint(geom1, geom2);
        case 'LineString':
            return isMultiPointOnLine(geom1, geom2);
        case 'Polygon':
            return isMultiPointInPoly(geom1, geom2);
        default:
            throw new Error('feature2 ' + type2 + ' geometry not supported');
        }
    case 'LineString':
        switch (type2) {
        case 'LineString':
            return isLineOnLine(geom1, geom2);
        case 'Polygon':
            return isLineInPoly(geom1, geom2);
        default:
            throw new Error('feature2 ' + type2 + ' geometry not supported');
        }
    case 'Polygon':
        switch (type2) {
        case 'Polygon':
            return isPolyInPoly(geom1, geom2);
        default:
            throw new Error('feature2 ' + type2 + ' geometry not supported');
        }
    default:
        throw new Error('feature1 ' + type1 + ' geometry not supported');
    }
}

function isPointInMultiPoint(point, multiPoint) {
    var i;
    var output = false;
    for (i = 0; i < multiPoint.coordinates.length; i++) {
        if (compareCoords(multiPoint.coordinates[i], point.coordinates)) {
            output = true;
            break;
        }
    }
    return output;
}

function isMultiPointInMultiPoint(multiPoint1, multiPoint2) {
    for (var i = 0; i < multiPoint1.coordinates.length; i++) {
        var anyMatch = false;
        for (var i2 = 0; i2 < multiPoint2.coordinates.length; i2++) {
            if (compareCoords(multiPoint1.coordinates[i], multiPoint2.coordinates[i2])) {
                anyMatch = true;
            }
        }
        if (!anyMatch) {
            return false;
        }
    }
    return true;
}

function isMultiPointOnLine(multiPoint, lineString) {
    var foundInsidePoint = false;

    for (var i = 0; i < multiPoint.coordinates.length; i++) {
        if (!booleanPointOnLine(multiPoint.coordinates[i], lineString)) {
            return false;
        }
        if (!foundInsidePoint) {
            foundInsidePoint = booleanPointOnLine(multiPoint.coordinates[i], lineString, {ignoreEndVertices: true});
        }
    }
    return foundInsidePoint;
}

function isMultiPointInPoly(multiPoint, polygon) {
    var output = true;
    var oneInside = false;
    for (var i = 0; i < multiPoint.coordinates.length; i++) {
        var isInside = booleanPointInPolygon(multiPoint.coordinates[1], polygon);
        if (!isInside) {
            output = false;
            break;
        }
        if (!oneInside) {
            isInside = booleanPointInPolygon(multiPoint.coordinates[1], polygon, {ignoreBoundary: true});
        }
    }
    return output && isInside;
}

function isLineOnLine(lineString1, lineString2) {
    for (var i = 0; i < lineString1.coordinates.length; i++) {
        if (!booleanPointOnLine(lineString1.coordinates[i], lineString2)) {
            return false;
        }
    }
    return true;
}

function isLineInPoly(linestring, polygon) {
    var polyBbox = bbox(polygon);
    var lineBbox = bbox(linestring);
    if (!doBBoxOverlap(polyBbox, lineBbox)) {
        return false;
    }
    var foundInsidePoint = false;

    for (var i = 0; i < linestring.coordinates.length - 1; i++) {
        if (!booleanPointInPolygon(linestring.coordinates[i], polygon)) {
            return false;
        }
        if (!foundInsidePoint) {
            foundInsidePoint = booleanPointInPolygon(linestring.coordinates[i], polygon, {ignoreBoundary: true});
        }
        if (!foundInsidePoint) {
            var midpoint = getMidpoint(linestring.coordinates[i], linestring.coordinates[i + 1]);
            foundInsidePoint = booleanPointInPolygon(midpoint, polygon, {ignoreBoundary: true});

        }
    }
    return foundInsidePoint;
}

/**
 * Is Polygon2 in Polygon1
 * Only takes into account outer rings
 *
 * @private
 * @param {Geometry|Feature<Polygon>} feature1 Polygon1
 * @param {Geometry|Feature<Polygon>} feature2 Polygon2
 * @returns {boolean} true/false
 */
function isPolyInPoly(feature1, feature2) {
    var poly1Bbox = bbox(feature1);
    var poly2Bbox = bbox(feature2);
    if (!doBBoxOverlap(poly2Bbox, poly1Bbox)) {
        return false;
    }
    for (var i = 0; i < feature1.coordinates[0].length; i++) {
        if (!booleanPointInPolygon(feature1.coordinates[0][i], feature2)) {
            return false;
        }
    }
    return true;
}

function doBBoxOverlap(bbox1, bbox2) {
    if (bbox1[0] > bbox2[0]) return false;
    if (bbox1[2] < bbox2[2]) return false;
    if (bbox1[1] > bbox2[1]) return false;
    if (bbox1[3] < bbox2[3]) return false;
    return true;
}

/**
 * compareCoords
 *
 * @private
 * @param {Position} pair1 point [x,y]
 * @param {Position} pair2 point [x,y]
 * @returns {boolean} true/false if coord pairs match
 */
function compareCoords(pair1, pair2) {
    return pair1[0] === pair2[0] && pair1[1] === pair2[1];
}

/**
 * getMidpoint
 *
 * @private
 * @param {Position} pair1 point [x,y]
 * @param {Position} pair2 point [x,y]
 * @returns {Position} midpoint of pair1 and pair2
 */
function getMidpoint(pair1, pair2) {
    return [(pair1[0] + pair2[0]) / 2, (pair1[1] + pair2[1]) / 2];
}

/**
 * Creates a {@link Point} grid from a bounding box, {@link FeatureCollection} or {@link Feature}.
 *
 * @name pointGrid
 * @param {Array<number>} bbox extent in [minX, minY, maxX, maxY] order
 * @param {number} cellSide the distance between points, in units
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units='kilometers'] used in calculating cellSide, can be degrees, radians, miles, or kilometers
 * @param {Feature<Polygon|MultiPolygon>} [options.mask] if passed a Polygon or MultiPolygon, the grid Points will be created only inside it
 * @param {Object} [options.properties={}] passed to each point of the grid
 * @returns {FeatureCollection<Point>} grid of points
 * @example
 * var extent = [-70.823364, -33.553984, -70.473175, -33.302986];
 * var cellSide = 3;
 * var options = {units: 'miles'};
 *
 * var grid = turf.pointGrid(extent, cellSide, options);
 *
 * //addToMap
 * var addToMap = [grid];
 */
function pointGrid(bbox, cellSide, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    // var units = options.units;
    var mask = options.mask;
    var properties = options.properties;

    // Containers
    var results = [];

    // Input Validation
    if (cellSide === null || cellSide === undefined) throw new Error('cellSide is required');
    if (!isNumber(cellSide)) throw new Error('cellSide is invalid');
    if (!bbox) throw new Error('bbox is required');
    if (!Array.isArray(bbox)) throw new Error('bbox must be array');
    if (bbox.length !== 4) throw new Error('bbox must contain 4 numbers');
    if (mask && ['Polygon', 'MultiPolygon'].indexOf(getType(mask)) === -1) throw new Error('options.mask must be a (Multi)Polygon');

    var west = bbox[0];
    var south = bbox[1];
    var east = bbox[2];
    var north = bbox[3];

    var xFraction = cellSide / (distance([west, south], [east, south], options));
    var cellWidth = xFraction * (east - west);
    var yFraction = cellSide / (distance([west, south], [west, north], options));
    var cellHeight = yFraction * (north - south);

    var bboxWidth = (east - west);
    var bboxHeight = (north - south);
    var columns = Math.floor(bboxWidth / cellWidth);
    var rows = Math.floor(bboxHeight / cellHeight);
    // adjust origin of the grid
    var deltaX = (bboxWidth - columns * cellWidth) / 2;
    var deltaY = (bboxHeight - rows * cellHeight) / 2;

    var currentX = west + deltaX;
    while (currentX <= east) {
        var currentY = south + deltaY;
        while (currentY <= north) {
            var cellPt = point([currentX, currentY], properties);
            if (mask) {
                if (booleanWithin(cellPt, mask)) results.push(cellPt);
            } else {
                results.push(cellPt);
            }
            currentY += cellHeight;
        }
        currentX += cellWidth;
    }

    return featureCollection(results);
}

/**
 * Takes a GeoJSON Feature or FeatureCollection and truncates the precision of the geometry.
 *
 * @name truncate
 * @param {GeoJSON} geojson any GeoJSON Feature, FeatureCollection, Geometry or GeometryCollection.
 * @param {Object} [options={}] Optional parameters
 * @param {number} [options.precision=6] coordinate decimal precision
 * @param {number} [options.coordinates=3] maximum number of coordinates (primarly used to remove z coordinates)
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated (significant performance increase if true)
 * @returns {GeoJSON} layer with truncated geometry
 * @example
 * var point = turf.point([
 *     70.46923055566859,
 *     58.11088890802906,
 *     1508
 * ]);
 * var options = {precision: 3, coordinates: 2};
 * var truncated = turf.truncate(point, options);
 * //=truncated.geometry.coordinates => [70.469, 58.111]
 *
 * //addToMap
 * var addToMap = [truncated];
 */
function truncate(geojson, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var precision = options.precision;
    var coordinates = options.coordinates;
    var mutate = options.mutate;

    // default params
    precision = (precision === undefined || precision === null || isNaN(precision)) ? 6 : precision;
    coordinates = (coordinates === undefined || coordinates === null || isNaN(coordinates)) ? 3 : coordinates;

    // validation
    if (!geojson) throw new Error('<geojson> is required');
    if (typeof precision !== 'number') throw new Error('<precision> must be a number');
    if (typeof coordinates !== 'number') throw new Error('<coordinates> must be a number');

    // prevent input mutation
    if (mutate === false || mutate === undefined) geojson = JSON.parse(JSON.stringify(geojson));

    var factor = Math.pow(10, precision);

    // Truncate Coordinates
    coordEach(geojson, function (coords) {
        truncateCoords(coords, factor, coordinates);
    });
    return geojson;
}

/**
 * Truncate Coordinates - Mutates coordinates in place
 *
 * @private
 * @param {Array<any>} coords Geometry Coordinates
 * @param {number} factor rounding factor for coordinate decimal precision
 * @param {number} coordinates maximum number of coordinates (primarly used to remove z coordinates)
 * @returns {Array<any>} mutated coordinates
 */
function truncateCoords(coords, factor, coordinates) {
    // Remove extra coordinates (usually elevation coordinates and more)
    if (coords.length > coordinates) coords.splice(coordinates, coords.length);

    // Truncate coordinate decimals
    for (var i = 0; i < coords.length; i++) {
        coords[i] = Math.round(coords[i] * factor) / factor;
    }
    return coords;
}

/**
 * Flattens any {@link GeoJSON} to a {@link FeatureCollection} inspired by [geojson-flatten](https://github.com/tmcw/geojson-flatten).
 *
 * @name flatten
 * @param {GeoJSON} geojson any valid GeoJSON Object
 * @returns {FeatureCollection<any>} all Multi-Geometries are flattened into single Features
 * @example
 * var multiGeometry = turf.multiPolygon([
 *   [[[102.0, 2.0], [103.0, 2.0], [103.0, 3.0], [102.0, 3.0], [102.0, 2.0]]],
 *   [[[100.0, 0.0], [101.0, 0.0], [101.0, 1.0], [100.0, 1.0], [100.0, 0.0]],
 *   [[100.2, 0.2], [100.8, 0.2], [100.8, 0.8], [100.2, 0.8], [100.2, 0.2]]]
 * ]);
 *
 * var flatten = turf.flatten(multiGeometry);
 *
 * //addToMap
 * var addToMap = [flatten]
 */
function flatten(geojson) {
    if (!geojson) throw new Error('geojson is required');

    var results = [];
    flattenEach(geojson, function (feature$$1) {
        results.push(feature$$1);
    });
    return featureCollection(results);
}

/**
 * Divides a {@link LineString} into chunks of a specified length.
 * If the line is shorter than the segment length then the original line is returned.
 *
 * @name lineChunk
 * @param {FeatureCollection|Geometry|Feature<LineString|MultiLineString>} geojson the lines to split
 * @param {number} segmentLength how long to make each segment
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units='kilometers'] units can be degrees, radians, miles, or kilometers
 * @param {boolean} [options.reverse=false] reverses coordinates to start the first chunked segment at the end
 * @returns {FeatureCollection<LineString>} collection of line segments
 * @example
 * var line = turf.lineString([[-95, 40], [-93, 45], [-85, 50]]);
 *
 * var chunk = turf.lineChunk(line, 15, {units: 'miles'});
 *
 * //addToMap
 * var addToMap = [chunk];
 */
function lineChunk(geojson, segmentLength, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var units = options.units;
    var reverse = options.reverse;

    // Validation
    if (!geojson) throw new Error('geojson is required');
    if (segmentLength <= 0) throw new Error('segmentLength must be greater than 0');

    // Container
    var results = [];

    // Flatten each feature to simple LineString
    flattenEach(geojson, function (feature$$1) {
        // reverses coordinates to start the first chunked segment at the end
        if (reverse) feature$$1.geometry.coordinates = feature$$1.geometry.coordinates.reverse();

        sliceLineSegments(feature$$1, segmentLength, units, function (segment) {
            results.push(segment);
        });
    });
    return featureCollection(results);
}

/**
 * Slice Line Segments
 *
 * @private
 * @param {Feature<LineString>} line GeoJSON LineString
 * @param {number} segmentLength how long to make each segment
 * @param {string}[units='kilometers'] units can be degrees, radians, miles, or kilometers
 * @param {Function} callback iterate over sliced line segments
 * @returns {void}
 */
function sliceLineSegments(line, segmentLength, units, callback) {
    var lineLength = length(line, {units: units});

    // If the line is shorter than the segment length then the orginal line is returned.
    if (lineLength <= segmentLength) return callback(line);

    var numberOfSegments = lineLength / segmentLength;

    // If numberOfSegments is integer, no need to plus 1
    if (!Number.isInteger(numberOfSegments)) {
        numberOfSegments = Math.floor(numberOfSegments) + 1;
    }

    for (var i = 0; i < numberOfSegments; i++) {
        var outline = lineSliceAlong(line, segmentLength * i, segmentLength * (i + 1), {units: units});
        callback(outline, i);
    }
}

// Find self-intersections in geojson polygon (possibly with interior rings)
var isects = function (feature$$1, filterFn, useSpatialIndex) {
    if (feature$$1.geometry.type !== 'Polygon') throw new Error('The input feature must be a Polygon');
    if (useSpatialIndex === undefined) useSpatialIndex = 1;

    var coord = feature$$1.geometry.coordinates;

    var output = [];
    var seen = {};

    if (useSpatialIndex) {
        var allEdgesAsRbushTreeItems = [];
        for (var ring0 = 0; ring0 < coord.length; ring0++) {
            for (var edge0 = 0; edge0 < coord[ring0].length - 1; edge0++) {
                allEdgesAsRbushTreeItems.push(rbushTreeItem(ring0, edge0));
            }
        }
        var tree = rbush_1();
        tree.load(allEdgesAsRbushTreeItems);
    }

    for (var ringA = 0; ringA < coord.length; ringA++) {
        for (var edgeA = 0; edgeA < coord[ringA].length - 1; edgeA++) {
            if (useSpatialIndex) {
                var bboxOverlaps = tree.search(rbushTreeItem(ringA, edgeA));
                bboxOverlaps.forEach(function (bboxIsect) {
                    var ring1 = bboxIsect.ring;
                    var edge1 = bboxIsect.edge;
                    ifIsectAddToOutput(ringA, edgeA, ring1, edge1);
                });
            } else {
                for (var ring1 = 0; ring1 < coord.length; ring1++) {
                    for (var edge1 = 0; edge1 < coord[ring1].length - 1; edge1++) {
                        // TODO: speedup possible if only interested in unique: start last two loops at ringA and edgeA+1
                        ifIsectAddToOutput(ringA, edgeA, ring1, edge1);
                    }
                }
            }
        }
    }

    if (!filterFn) output = {type: 'Feature', geometry: {type: 'MultiPoint', coordinates: output}};
    return output;

    // Function to check if two edges intersect and add the intersection to the output
    function ifIsectAddToOutput(ring0, edge0, ring1, edge1) {
        var start0 = coord[ring0][edge0];
        var end0 = coord[ring0][edge0 + 1];
        var start1 = coord[ring1][edge1];
        var end1 = coord[ring1][edge1 + 1];

        var isect = intersect(start0, end0, start1, end1);

        if (isect === null) return; // discard parallels and coincidence
        var frac0;
        var frac1;
        if (end0[0] !== start0[0]) {
            frac0 = (isect[0] - start0[0]) / (end0[0] - start0[0]);
        } else {
            frac0 = (isect[1] - start0[1]) / (end0[1] - start0[1]);
        }
        if (end1[0] !== start1[0]) {
            frac1 = (isect[0] - start1[0]) / (end1[0] - start1[0]);
        } else {
            frac1 = (isect[1] - start1[1]) / (end1[1] - start1[1]);
        }
        if (frac0 >= 1 || frac0 <= 0 || frac1 >= 1 || frac1 <= 0) return; // require segment intersection

        var key = isect;
        var unique = !seen[key];
        if (unique) {
            seen[key] = true;
        }

        if (filterFn) {
            output.push(filterFn(isect, ring0, edge0, start0, end0, frac0, ring1, edge1, start1, end1, frac1, unique));
        } else {
            output.push(isect);
        }
    }

    // Function to return a rbush tree item given an ring and edge number
    function rbushTreeItem(ring, edge) {

        var start = coord[ring][edge];
        var end = coord[ring][edge + 1];
        var minX;
        var maxX;
        var minY;
        var maxY;
        if (start[0] < end[0]) {
            minX = start[0];
            maxX = end[0];
        } else {
            minX = end[0];
            maxX = start[0];
        }
        if (start[1] < end[1]) {
            minY = start[1];
            maxY = end[1];
        } else {
            minY = end[1];
            maxY = start[1];
        }
        return {minX: minX, minY: minY, maxX: maxX, maxY: maxY, ring: ring, edge: edge};
    }
};

// Function to compute where two lines (not segments) intersect. From https://en.wikipedia.org/wiki/Line%E2%80%93line_intersection
function intersect(start0, end0, start1, end1) {
    if (equalArrays$1(start0, start1) || equalArrays$1(start0, end1) || equalArrays$1(end0, start1) || equalArrays$1(end1, start1)) return null;
    var x0 = start0[0],
        y0 = start0[1],
        x1 = end0[0],
        y1 = end0[1],
        x2 = start1[0],
        y2 = start1[1],
        x3 = end1[0],
        y3 = end1[1];
    var denom = (x0 - x1) * (y2 - y3) - (y0 - y1) * (x2 - x3);
    if (denom === 0) return null;
    var x4 = ((x0 * y1 - y0 * x1) * (x2 - x3) - (x0 - x1) * (x2 * y3 - y2 * x3)) / denom;
    var y4 = ((x0 * y1 - y0 * x1) * (y2 - y3) - (y0 - y1) * (x2 * y3 - y2 * x3)) / denom;
    return [x4, y4];
}

// Function to compare Arrays of numbers. From http://stackoverflow.com/questions/7837456/how-to-compare-arrays-in-javascript
function equalArrays$1(array1, array2) {
    // if the other array is a falsy value, return
    if (!array1 || !array2)
        return false;

    // compare lengths - can save a lot of time
    if (array1.length !== array2.length)
        return false;

    for (var i = 0, l = array1.length; i < l; i++) {
        // Check if we have nested arrays
        if (array1[i] instanceof Array && array2[i] instanceof Array) {
            // recurse into the nested arrays
            if (!equalArrays$1(array1[i], array2[i]))
                return false;
        } else if (array1[i] !== array2[i]) {
            // Warning - two different object instances will never be equal: {x:20} !== {x:20}
            return false;
        }
    }
    return true;
}

/**
 * Takes a complex (i.e. self-intersecting) geojson polygon, and breaks it down into its composite simple, non-self-intersecting one-ring polygons.
 *
 * @module simplepolygon
 * @param {Feature} feature Input polygon. This polygon may be unconform the {@link https://en.wikipedia.org/wiki/Simple_Features|Simple Features standard} in the sense that it's inner and outer rings may cross-intersect or self-intersect, that the outer ring must not contain the optional inner rings and that the winding number must not be positive for the outer and negative for the inner rings.
 * @return {FeatureCollection} Feature collection containing the simple, non-self-intersecting one-ring polygon features that the complex polygon is composed of. These simple polygons have properties such as their parent polygon, winding number and net winding number.
 *
 * @example
 * var poly = {
 *   "type": "Feature",
 *   "geometry": {
 *     "type": "Polygon",
 *     "coordinates": [[[0,0],[2,0],[0,2],[2,2],[0,0]]]
 *   }
 * };
 *
 * var result = simplepolygon(poly);
 *
 * // =result
 * // which will be a featureCollection of two polygons, one with coordinates [[[0,0],[2,0],[1,1],[0,0]]], parent -1, winding 1 and net winding 1, and one with coordinates [[[1,1],[0,2],[2,2],[1,1]]], parent -1, winding -1 and net winding -1
 */
var simplepolygon = function (feature$$1) {
    // Check input
    if (feature$$1.type != 'Feature') throw new Error('The input must a geojson object of type Feature');
    if ((feature$$1.geometry === undefined) || (feature$$1.geometry == null)) throw new Error('The input must a geojson object with a non-empty geometry');
    if (feature$$1.geometry.type != 'Polygon') throw new Error('The input must be a geojson Polygon');

    // Process input
    var numRings = feature$$1.geometry.coordinates.length;
    var vertices = [];
    for (var i = 0; i < numRings; i++) {
        var ring = feature$$1.geometry.coordinates[i];
        if (!equalArrays(ring[0], ring[ring.length - 1])) {
            ring.push(ring[0]); // Close input ring if it is not
        }
        vertices.push.apply(vertices, ring.slice(0, ring.length - 1));
    }
    if (!isUnique(vertices)) throw new Error('The input polygon may not have duplicate vertices (except for the first and last vertex of each ring)');
    var numvertices = vertices.length; // number of input ring vertices, with the last closing vertices not counted

    // Compute self-intersections
    var selfIsectsData = isects(feature$$1, function filterFn(isect, ring0, edge0, start0, end0, frac0, ring1, edge1, start1, end1, frac1, unique) {
        return [isect, ring0, edge0, start0, end0, frac0, ring1, edge1, start1, end1, frac1, unique];
    });
    var numSelfIsect = selfIsectsData.length;

    // If no self-intersections are found, the input rings are the output rings. Hence, we must only compute their winding numbers, net winding numbers and (since ohers rings could lie outside the first ring) parents.
    if (numSelfIsect == 0) {
        var outputFeatureArray = [];
        for (var i = 0; i < numRings; i++) {
            outputFeatureArray.push(polygon([feature$$1.geometry.coordinates[i]], {parent: -1, winding: windingOfRing(feature$$1.geometry.coordinates[i])}));
        }
        var output = featureCollection(outputFeatureArray);
        determineParents();
        setNetWinding();

        return output;
    }

    // If self-intersections are found, we will compute the output rings with the help of two intermediate variables
    // First, we build the pseudo vertex list and intersection list
    // The Pseudo vertex list is an array with for each ring an array with for each edge an array containing the pseudo-vertices (as made by their constructor) that have this ring and edge as ringAndEdgeIn, sorted for each edge by their fractional distance on this edge. It's length hence equals numRings.
    var pseudoVtxListByRingAndEdge = [];
    // The intersection list is an array containing intersections (as made by their constructor). First all numvertices ring-vertex-intersections, then all self-intersections (intra- and inter-ring). The order of the latter is not important but is permanent once given.
    var isectList = [];
    // Adding ring-pseudo-vertices to pseudoVtxListByRingAndEdge and ring-vertex-intersections to isectList
    for (var i = 0; i < numRings; i++) {
        pseudoVtxListByRingAndEdge.push([]);
        for (var j = 0; j < feature$$1.geometry.coordinates[i].length - 1; j++) {
            // Each edge will feature one ring-pseudo-vertex in its array, on the last position. i.e. edge j features the ring-pseudo-vertex of the ring vertex j+1, which has ringAndEdgeIn = [i,j], on the last position.
            pseudoVtxListByRingAndEdge[i].push([new PseudoVtx(feature$$1.geometry.coordinates[i][(j + 1).modulo(feature$$1.geometry.coordinates[i].length - 1)], 1, [i, j], [i, (j + 1).modulo(feature$$1.geometry.coordinates[i].length - 1)], undefined)]);
            // The first numvertices elements in isectList correspond to the ring-vertex-intersections
            isectList.push(new Isect(feature$$1.geometry.coordinates[i][j], [i, (j - 1).modulo(feature$$1.geometry.coordinates[i].length - 1)], [i, j], undefined, undefined, false, true));
        }
    }
    // Adding intersection-pseudo-vertices to pseudoVtxListByRingAndEdge and self-intersections to isectList
    for (var i = 0; i < numSelfIsect; i++) {
    // Adding intersection-pseudo-vertices made using selfIsectsData to pseudoVtxListByRingAndEdge's array corresponding to the incomming ring and edge
        pseudoVtxListByRingAndEdge[selfIsectsData[i][1]][selfIsectsData[i][2]].push(new PseudoVtx(selfIsectsData[i][0], selfIsectsData[i][5], [selfIsectsData[i][1], selfIsectsData[i][2]], [selfIsectsData[i][6], selfIsectsData[i][7]], undefined));
        // selfIsectsData contains double mentions of each intersection, but we only want to add them once to isectList
        if (selfIsectsData[i][11]) isectList.push(new Isect(selfIsectsData[i][0], [selfIsectsData[i][1], selfIsectsData[i][2]], [selfIsectsData[i][6], selfIsectsData[i][7]], undefined, undefined, true, true));
    }
    var numIsect = isectList.length;
    // Sort edge arrays of pseudoVtxListByRingAndEdge by the fractional distance 'param'
    for (var i = 0; i < pseudoVtxListByRingAndEdge.length; i++) {
        for (var j = 0; j < pseudoVtxListByRingAndEdge[i].length; j++) {
            pseudoVtxListByRingAndEdge[i][j].sort(function (a, b) { return (a.param < b.param) ? -1 : 1; });
        }
    }

    // Make a spatial index of intersections, in preperation for the following two steps
    var allIsectsAsIsectRbushTreeItem = [];
    for (var i = 0; i < numIsect; i++) {
        allIsectsAsIsectRbushTreeItem.push({minX: isectList[i].coord[0], minY: isectList[i].coord[1], maxX: isectList[i].coord[0], maxY: isectList[i].coord[1], index: i}); // could pass isect: isectList[i], but not necessary
    }
    var isectRbushTree = rbush_1();
    isectRbushTree.load(allIsectsAsIsectRbushTreeItem);

    // Now we will teach each intersection in isectList which is the next intersection along both it's [ring, edge]'s, in two steps.
    // First, we find the next intersection for each pseudo-vertex in pseudoVtxListByRingAndEdge:
    // For each pseudovertex in pseudoVtxListByRingAndEdge (3 loops) look at the next pseudovertex on that edge and find the corresponding intersection by comparing coordinates
    for (var i = 0; i < pseudoVtxListByRingAndEdge.length; i++) {
        for (var j = 0; j < pseudoVtxListByRingAndEdge[i].length; j++) {
            for (var k = 0; k < pseudoVtxListByRingAndEdge[i][j].length; k++) {
                var coordToFind;
                if (k == pseudoVtxListByRingAndEdge[i][j].length - 1) { // If it's the last pseudoVertex on that edge, then the next pseudoVertex is the first one on the next edge of that ring.
                    coordToFind = pseudoVtxListByRingAndEdge[i][(j + 1).modulo(feature$$1.geometry.coordinates[i].length - 1)][0].coord;
                } else {
                    coordToFind = pseudoVtxListByRingAndEdge[i][j][k + 1].coord;
                }
                var IsectRbushTreeItemFound = isectRbushTree.search({minX: coordToFind[0], minY: coordToFind[1], maxX: coordToFind[0], maxY: coordToFind[1]})[0]; // We can take [0] of the result, because there is only one isect correponding to a pseudo-vertex
                pseudoVtxListByRingAndEdge[i][j][k].nxtIsectAlongEdgeIn = IsectRbushTreeItemFound.index;
            }
        }
    }

    // Second, we port this knowledge of the next intersection over to the intersections in isectList, by finding the intersection corresponding to each pseudo-vertex and copying the pseudo-vertex' knownledge of the next-intersection over to the intersection
    for (var i = 0; i < pseudoVtxListByRingAndEdge.length; i++) {
        for (var j = 0; j < pseudoVtxListByRingAndEdge[i].length; j++) {
            for (var k = 0; k < pseudoVtxListByRingAndEdge[i][j].length; k++) {
                var coordToFind = pseudoVtxListByRingAndEdge[i][j][k].coord;
                var IsectRbushTreeItemFound = isectRbushTree.search({minX: coordToFind[0], minY: coordToFind[1], maxX: coordToFind[0], maxY: coordToFind[1]})[0]; // We can take [0] of the result, because there is only one isect correponding to a pseudo-vertex
                var l = IsectRbushTreeItemFound.index;
                if (l < numvertices) { // Special treatment at ring-vertices: we correct the misnaming that happened in the previous block, since ringAndEdgeOut = ringAndEdge2 for ring vertices.
                    isectList[l].nxtIsectAlongRingAndEdge2 = pseudoVtxListByRingAndEdge[i][j][k].nxtIsectAlongEdgeIn;
                } else { // Port the knowledge of the next intersection from the pseudo-vertices to the intersections, depending on how the edges are labeled in the pseudo-vertex and intersection.
                    if (equalArrays(isectList[l].ringAndEdge1, pseudoVtxListByRingAndEdge[i][j][k].ringAndEdgeIn)) {
                        isectList[l].nxtIsectAlongRingAndEdge1 = pseudoVtxListByRingAndEdge[i][j][k].nxtIsectAlongEdgeIn;
                    } else {
                        isectList[l].nxtIsectAlongRingAndEdge2 = pseudoVtxListByRingAndEdge[i][j][k].nxtIsectAlongEdgeIn;
                    }
                }
            }
        }
    }
    // This explains why, eventhough when we will walk away from an intersection, we will walk way from the corresponding pseudo-vertex along edgeOut, pseudo-vertices have the property 'nxtIsectAlongEdgeIn' in stead of some propery 'nxtPseudoVtxAlongEdgeOut'. This is because this property (which is easy to find out) is used in the above for nxtIsectAlongRingAndEdge1 and nxtIsectAlongRingAndEdge2!

    // Before we start walking over the intersections to build the output rings, we prepare a queue that stores information on intersections we still have to deal with, and put at least one intersection in it.
    // This queue will contain information on intersections where we can start walking from once the current walk is finished, and its parent output ring (the smallest output ring it lies within, -1 if no parent or parent unknown yet) and its winding number (which we can already determine).
    var queue = [];
    // For each output ring, add the ring-vertex-intersection with the smalles x-value (i.e. the left-most) as a start intersection. By choosing such an extremal intersections, we are sure to start at an intersection that is a convex vertex of its output ring. By adding them all to the queue, we are sure that no rings will be forgotten. If due to ring-intersections such an intersection will be encountered while walking, it will be removed from the queue.
    var i = 0;
    for (var j = 0; j < numRings; j++) {
        var leftIsect = i;
        for (var k = 0; k < feature$$1.geometry.coordinates[j].length - 1; k++) {
            if (isectList[i].coord[0] < isectList[leftIsect].coord[0]) {
                leftIsect = i;
            }
            i++;
        }
        // Compute winding at this left-most ring-vertex-intersection. We thus this by using our knowledge that this extremal vertex must be a convex vertex.
        // We first find the intersection before and after it, and then use them to determine the winding number of the corresponding output ring, since we know that an extremal vertex of a simple, non-self-intersecting ring is always convex, so the only reason it would not be is because the winding number we use to compute it is wrong
        var isectAfterLeftIsect = isectList[leftIsect].nxtIsectAlongRingAndEdge2;
        for (var k = 0; k < isectList.length; k++) {
            if ((isectList[k].nxtIsectAlongRingAndEdge1 == leftIsect) || (isectList[k].nxtIsectAlongRingAndEdge2 == leftIsect)) {
                var isectBeforeLeftIsect = k;
                break;
            }
        }
        var windingAtIsect = isConvex([isectList[isectBeforeLeftIsect].coord, isectList[leftIsect].coord, isectList[isectAfterLeftIsect].coord], true) ? 1 : -1;

        queue.push({isect: leftIsect, parent: -1, winding: windingAtIsect});
    }
    // Sort the queue by the same criterion used to find the leftIsect: the left-most leftIsect must be last in the queue, such that it will be popped first, such that we will work from out to in regarding input rings. This assumtion is used when predicting the winding number and parent of a new queue member.
    queue.sort(function (a, b) { return (isectList[a.isect].coord > isectList[b.isect].coord) ? -1 : 1; });

    // Initialise output
    var outputFeatureArray = [];

    // While the queue is not empty, take the last object (i.e. its intersection) out and start making an output ring by walking in the direction that has not been walked away over yet.
    while (queue.length > 0) {
    // Get the last object out of the queue
        var popped = queue.pop();
        var startIsect = popped.isect;
        var currentOutputRingParent = popped.parent;
        var currentOutputRingWinding = popped.winding;
        // Make new output ring and add vertex from starting intersection
        var currentOutputRing = outputFeatureArray.length;
        var currentOutputRingCoords = [isectList[startIsect].coord];
        // Set up the variables used while walking over intersections: 'currentIsect', 'nxtIsect' and 'walkingRingAndEdge'
        var currentIsect = startIsect;
        if (isectList[startIsect].ringAndEdge1Walkable) {
            var walkingRingAndEdge = isectList[startIsect].ringAndEdge1;
            var nxtIsect = isectList[startIsect].nxtIsectAlongRingAndEdge1;
        } else {
            var walkingRingAndEdge = isectList[startIsect].ringAndEdge2;
            var nxtIsect = isectList[startIsect].nxtIsectAlongRingAndEdge2;
        }
        // While we have not arrived back at the same intersection, keep walking
        while (!equalArrays(isectList[startIsect].coord, isectList[nxtIsect].coord)) {
            currentOutputRingCoords.push(isectList[nxtIsect].coord);
            // If the next intersection is queued, we can remove it, because we will go there now.
            var nxtIsectInQueue = undefined;
            for (var i = 0; i < queue.length; i++) { if (queue[i].isect == nxtIsect) { nxtIsectInQueue = i; break; } }
            if (nxtIsectInQueue != undefined) {
                queue.splice(nxtIsectInQueue, 1);
            }
            // Arriving at this new intersection, we know which will be our next walking ring and edge (if we came from 1 we will walk away from 2 and vice versa),
            // So we can set it as our new walking ring and intersection and remember that we (will) have walked over it
            // If we have never walked away from this new intersection along the other ring and edge then we will soon do, add the intersection (and the parent wand winding number) to the queue
            // (We can predict the winding number and parent as follows: if the edge is convex, the other output ring started from there will have the alternate winding and lie outside of the current one, and thus have the same parent ring as the current ring. Otherwise, it will have the same winding number and lie inside of the current ring. We are, however, only sure of this of an output ring started from there does not enclose the current ring. This is why the initial queue's intersections must be sorted such that outer ones come out first.)
            // We then update the other two walking variables.
            if (equalArrays(walkingRingAndEdge, isectList[nxtIsect].ringAndEdge1)) {
                walkingRingAndEdge = isectList[nxtIsect].ringAndEdge2;
                isectList[nxtIsect].ringAndEdge2Walkable = false;
                if (isectList[nxtIsect].ringAndEdge1Walkable) {
                    var pushing = {isect: nxtIsect};
                    if (isConvex([isectList[currentIsect].coord, isectList[nxtIsect].coord, isectList[isectList[nxtIsect].nxtIsectAlongRingAndEdge2].coord], currentOutputRingWinding == 1)) {
                        pushing.parent = currentOutputRingParent;
                        pushing.winding = -currentOutputRingWinding;
                    } else {
                        pushing.parent = currentOutputRing;
                        pushing.winding = currentOutputRingWinding;
                    }
                    queue.push(pushing);
                }
                currentIsect = nxtIsect;
                nxtIsect = isectList[nxtIsect].nxtIsectAlongRingAndEdge2;
            } else {
                walkingRingAndEdge = isectList[nxtIsect].ringAndEdge1;
                isectList[nxtIsect].ringAndEdge1Walkable = false;
                if (isectList[nxtIsect].ringAndEdge2Walkable) {
                    var pushing = {isect: nxtIsect};
                    if (isConvex([isectList[currentIsect].coord, isectList[nxtIsect].coord, isectList[isectList[nxtIsect].nxtIsectAlongRingAndEdge1].coord], currentOutputRingWinding == 1)) {
                        pushing.parent = currentOutputRingParent;
                        pushing.winding = -currentOutputRingWinding;
                    } else {
                        pushing.parent = currentOutputRing;
                        pushing.winding = currentOutputRingWinding;
                    }
                    queue.push(pushing);
                }
                currentIsect = nxtIsect;
                nxtIsect = isectList[nxtIsect].nxtIsectAlongRingAndEdge1;
            }
        }
        // Close output ring
        currentOutputRingCoords.push(isectList[nxtIsect].coord);
        // Push output ring to output
        outputFeatureArray.push(polygon([currentOutputRingCoords], {index: currentOutputRing, parent: currentOutputRingParent, winding: currentOutputRingWinding, netWinding: undefined}));
    }

    var output = featureCollection(outputFeatureArray);

    determineParents();

    setNetWinding();

    // These functions are also used if no intersections are found
    function determineParents() {
        var featuresWithoutParent = [];
        for (var i = 0; i < output.features.length; i++) {
            if (output.features[i].properties.parent == -1) featuresWithoutParent.push(i);
        }
        if (featuresWithoutParent.length > 1) {
            for (var i = 0; i < featuresWithoutParent.length; i++) {
                var parent = -1;
                var parentArea = Infinity;
                for (var j = 0; j < output.features.length; j++) {
                    if (featuresWithoutParent[i] == j) continue;
                    if (booleanPointInPolygon(output.features[featuresWithoutParent[i]].geometry.coordinates[0][0], output.features[j], {ignoreBoundary: true})) {
                        if (area$1(output.features[j]) < parentArea) {
                            parent = j;
                        }
                    }
                }
                output.features[featuresWithoutParent[i]].properties.parent = parent;
            }
        }
    }

    function setNetWinding() {
        for (var i = 0; i < output.features.length; i++) {
            if (output.features[i].properties.parent == -1) {
                var netWinding = output.features[i].properties.winding;
                output.features[i].properties.netWinding = netWinding;
                setNetWindingOfChildren(i, netWinding);
            }
        }
    }

    function setNetWindingOfChildren(parent, ParentNetWinding) {
        for (var i = 0; i < output.features.length; i++) {
            if (output.features[i].properties.parent == parent) {
                var netWinding = ParentNetWinding + output.features[i].properties.winding;
                output.features[i].properties.netWinding = netWinding;
                setNetWindingOfChildren(i, netWinding);
            }
        }
    }


    return output;
};



// Constructor for (ring- or intersection-) pseudo-vertices.
var PseudoVtx = function (coord, param, ringAndEdgeIn, ringAndEdgeOut, nxtIsectAlongEdgeIn) {
    this.coord = coord; // [x,y] of this pseudo-vertex
    this.param = param; // fractional distance of this intersection on incomming edge
    this.ringAndEdgeIn = ringAndEdgeIn; // [ring index, edge index] of incomming edge
    this.ringAndEdgeOut = ringAndEdgeOut; // [ring index, edge index] of outgoing edge
    this.nxtIsectAlongEdgeIn = nxtIsectAlongEdgeIn; // The next intersection when following the incomming edge (so not when following ringAndEdgeOut!)
};

// Constructor for an intersection. There are two intersection-pseudo-vertices per self-intersection and one ring-pseudo-vertex per ring-vertex-intersection. Their labels 1 and 2 are not assigned a particular meaning but are permanent once given.
var Isect = function (coord, ringAndEdge1, ringAndEdge2, nxtIsectAlongRingAndEdge1, nxtIsectAlongRingAndEdge2, ringAndEdge1Walkable, ringAndEdge2Walkable) {
    this.coord = coord; // [x,y] of this intersection
    this.ringAndEdge1 = ringAndEdge1; // first edge of this intersection
    this.ringAndEdge2 = ringAndEdge2; // second edge of this intersection
    this.nxtIsectAlongRingAndEdge1 = nxtIsectAlongRingAndEdge1; // the next intersection when following ringAndEdge1
    this.nxtIsectAlongRingAndEdge2 = nxtIsectAlongRingAndEdge2; // the next intersection when following ringAndEdge2
    this.ringAndEdge1Walkable = ringAndEdge1Walkable; // May we (still) walk away from this intersection over ringAndEdge1?
    this.ringAndEdge2Walkable = ringAndEdge2Walkable; // May we (still) walk away from this intersection over ringAndEdge2?
};

// Function to determine if three consecutive points of a simple, non-self-intersecting ring make up a convex vertex, assuming the ring is right- or lefthanded
function isConvex(pts, righthanded) {
    // 'pts' is an [x,y] pair
    // 'righthanded' is a boolean
    if (typeof (righthanded) === 'undefined') righthanded = true;
    if (pts.length != 3) throw new Error('This function requires an array of three points [x,y]');
    var d = (pts[1][0] - pts[0][0]) * (pts[2][1] - pts[0][1]) - (pts[1][1] - pts[0][1]) * (pts[2][0] - pts[0][0]);
    return (d >= 0) == righthanded;
}

// Function to compute winding of simple, non-self-intersecting ring
function windingOfRing(ring) {
    // 'ring' is an array of [x,y] pairs with the last equal to the first
    // Compute the winding number based on the vertex with the smallest x-value, it precessor and successor. An extremal vertex of a simple, non-self-intersecting ring is always convex, so the only reason it is not is because the winding number we use to compute it is wrong
    var leftVtx = 0;
    for (var i = 0; i < ring.length - 1; i++) { if (ring[i][0] < ring[leftVtx][0]) leftVtx = i; }
    if (isConvex([ring[(leftVtx - 1).modulo(ring.length - 1)], ring[leftVtx], ring[(leftVtx + 1).modulo(ring.length - 1)]], true)) {
        var winding = 1;
    } else {
        var winding = -1;
    }
    return winding;
}

// Function to compare Arrays of numbers. From http://stackoverflow.com/questions/7837456/how-to-compare-arrays-in-javascript
function equalArrays(array1, array2) {
    // if the other array is a falsy value, return
    if (!array1 || !array2)
        return false;

    // compare lengths - can save a lot of time
    if (array1.length != array2.length)
        return false;

    for (var i = 0, l = array1.length; i < l; i++) {
        // Check if we have nested arrays
        if (array1[i] instanceof Array && array2[i] instanceof Array) {
            // recurse into the nested arrays
            if (!equalArrays(array1[i], array2[i]))
                return false;
        } else if (array1[i] != array2[i]) {
            // Warning - two different object instances will never be equal: {x:20} != {x:20}
            return false;
        }
    }
    return true;
}

// Fix Javascript modulo for negative number. From http://stackoverflow.com/questions/4467539/javascript-modulo-not-behaving
Number.prototype.modulo = function (n) {
    return ((this % n) + n) % n;
};

// Function to check if array is unique (i.e. all unique elements, i.e. no duplicate elements)
function isUnique(array) {
    var u = {};
    var isUnique = 1;
    for (var i = 0, l = array.length; i < l; ++i) {
        if (u.hasOwnProperty(array[i])) {
            isUnique = 0;
            break;
        }
        u[array[i]] = 1;
    }
    return isUnique;
}

/**
 * Takes a kinked polygon and returns a feature collection of polygons that have no kinks.
 * Uses [simplepolygon](https://github.com/mclaeysb/simplepolygon) internally.
 *
 * @name unkinkPolygon
 * @param {FeatureCollection|Feature<Polygon|MultiPolygon>} geojson GeoJSON Polygon or MultiPolygon
 * @returns {FeatureCollection<Polygon>} Unkinked polygons
 * @example
 * var poly = turf.polygon([[[0, 0], [2, 0], [0, 2], [2, 2], [0, 0]]]);
 *
 * var result = turf.unkinkPolygon(poly);
 *
 * //addToMap
 * var addToMap = [poly, result]
 */
function unkinkPolygon(geojson) {
    var features = [];
    flattenEach(geojson, function (feature$$1) {
        if (feature$$1.geometry.type !== 'Polygon') return;
        featureEach(simplepolygon(feature$$1), function (poly) {
            features.push(polygon(poly.geometry.coordinates, feature$$1.properties));
        });
    });
    return featureCollection(features);
}

var D2R = Math.PI / 180;
var R2D = 180 / Math.PI;

var Coord = function (lon, lat) {
    this.lon = lon;
    this.lat = lat;
    this.x = D2R * lon;
    this.y = D2R * lat;
};

Coord.prototype.view = function () {
    return String(this.lon).slice(0, 4) + ',' + String(this.lat).slice(0, 4);
};

Coord.prototype.antipode = function () {
    var anti_lat = -1 * this.lat;
    var anti_lon = (this.lon < 0) ? 180 + this.lon : (180 - this.lon) * -1;
    return new Coord(anti_lon, anti_lat);
};

var LineString = function () {
    this.coords = [];
    this.length = 0;
};

LineString.prototype.move_to = function (coord) {
    this.length++;
    this.coords.push(coord);
};

var Arc = function (properties) {
    this.properties = properties || {};
    this.geometries = [];
};

Arc.prototype.json = function () {
    if (this.geometries.length <= 0) {
        return {'geometry': { 'type': 'LineString', 'coordinates': null },
            'type': 'Feature', 'properties': this.properties
        };
    } else if (this.geometries.length === 1) {
        return {'geometry': { 'type': 'LineString', 'coordinates': this.geometries[0].coords },
            'type': 'Feature', 'properties': this.properties
        };
    } else {
        var multiline = [];
        for (var i = 0; i < this.geometries.length; i++) {
            multiline.push(this.geometries[i].coords);
        }
        return {'geometry': { 'type': 'MultiLineString', 'coordinates': multiline },
            'type': 'Feature', 'properties': this.properties
        };
    }
};

// TODO - output proper multilinestring
Arc.prototype.wkt = function () {
    var wkt_string = '';
    var wkt = 'LINESTRING(';
    var collect = function (c) { wkt += c[0] + ' ' + c[1] + ','; };
    for (var i = 0; i < this.geometries.length; i++) {
        if (this.geometries[i].coords.length === 0) {
            return 'LINESTRING(empty)';
        } else {
            var coords = this.geometries[i].coords;
            coords.forEach(collect);
            wkt_string += wkt.substring(0, wkt.length - 1) + ')';
        }
    }
    return wkt_string;
};

/*
 * http://en.wikipedia.org/wiki/Great-circle_distance
 *
 */
var GreatCircle = function (start, end, properties) {
    if (!start || start.x === undefined || start.y === undefined) {
        throw new Error('GreatCircle constructor expects two args: start and end objects with x and y properties');
    }
    if (!end || end.x === undefined || end.y === undefined) {
        throw new Error('GreatCircle constructor expects two args: start and end objects with x and y properties');
    }
    this.start = new Coord(start.x, start.y);
    this.end = new Coord(end.x, end.y);
    this.properties = properties || {};

    var w = this.start.x - this.end.x;
    var h = this.start.y - this.end.y;
    var z = Math.pow(Math.sin(h / 2.0), 2) +
                Math.cos(this.start.y) *
                   Math.cos(this.end.y) *
                     Math.pow(Math.sin(w / 2.0), 2);
    this.g = 2.0 * Math.asin(Math.sqrt(z));

    if (this.g === Math.PI) {
        throw new Error('it appears ' + start.view() + ' and ' + end.view() + ' are \'antipodal\', e.g diametrically opposite, thus there is no single route but rather infinite');
    } else if (isNaN(this.g)) {
        throw new Error('could not calculate great circle between ' + start + ' and ' + end);
    }
};

/*
 * http://williams.best.vwh.net/avform.htm#Intermediate
 */
GreatCircle.prototype.interpolate = function (f) {
    var A = Math.sin((1 - f) * this.g) / Math.sin(this.g);
    var B = Math.sin(f * this.g) / Math.sin(this.g);
    var x = A * Math.cos(this.start.y) * Math.cos(this.start.x) + B * Math.cos(this.end.y) * Math.cos(this.end.x);
    var y = A * Math.cos(this.start.y) * Math.sin(this.start.x) + B * Math.cos(this.end.y) * Math.sin(this.end.x);
    var z = A * Math.sin(this.start.y) + B * Math.sin(this.end.y);
    var lat = R2D * Math.atan2(z, Math.sqrt(Math.pow(x, 2) + Math.pow(y, 2)));
    var lon = R2D * Math.atan2(y, x);
    return [lon, lat];
};



/*
 * Generate points along the great circle
 */
GreatCircle.prototype.Arc = function (npoints, options) {
    var first_pass = [];
    if (!npoints || npoints <= 2) {
        first_pass.push([this.start.lon, this.start.lat]);
        first_pass.push([this.end.lon, this.end.lat]);
    } else {
        var delta = 1.0 / (npoints - 1);
        for (var i = 0; i < npoints; ++i) {
            var step = delta * i;
            var pair = this.interpolate(step);
            first_pass.push(pair);
        }
    }
    /* partial port of dateline handling from:
      gdal/ogr/ogrgeometryfactory.cpp

      TODO - does not handle all wrapping scenarios yet
    */
    var bHasBigDiff = false;
    var dfMaxSmallDiffLong = 0;
    // from http://www.gdal.org/ogr2ogr.html
    // -datelineoffset:
    // (starting with GDAL 1.10) offset from dateline in degrees (default long. = +/- 10deg, geometries within 170deg to -170deg will be splited)
    var dfDateLineOffset = options && options.offset ? options.offset : 10;
    var dfLeftBorderX = 180 - dfDateLineOffset;
    var dfRightBorderX = -180 + dfDateLineOffset;
    var dfDiffSpace = 360 - dfDateLineOffset;

    // https://github.com/OSGeo/gdal/blob/7bfb9c452a59aac958bff0c8386b891edf8154ca/gdal/ogr/ogrgeometryfactory.cpp#L2342
    for (var j = 1; j < first_pass.length; ++j) {
        var dfPrevX = first_pass[j - 1][0];
        var dfX = first_pass[j][0];
        var dfDiffLong = Math.abs(dfX - dfPrevX);
        if (dfDiffLong > dfDiffSpace &&
            ((dfX > dfLeftBorderX && dfPrevX < dfRightBorderX) || (dfPrevX > dfLeftBorderX && dfX < dfRightBorderX))) {
            bHasBigDiff = true;
        } else if (dfDiffLong > dfMaxSmallDiffLong) {
            dfMaxSmallDiffLong = dfDiffLong;
        }
    }

    var poMulti = [];
    if (bHasBigDiff && dfMaxSmallDiffLong < dfDateLineOffset) {
        var poNewLS = [];
        poMulti.push(poNewLS);
        for (var k = 0; k < first_pass.length; ++k) {
            var dfX0 = parseFloat(first_pass[k][0]);
            if (k > 0 &&  Math.abs(dfX0 - first_pass[k - 1][0]) > dfDiffSpace) {
                var dfX1 = parseFloat(first_pass[k - 1][0]);
                var dfY1 = parseFloat(first_pass[k - 1][1]);
                var dfX2 = parseFloat(first_pass[k][0]);
                var dfY2 = parseFloat(first_pass[k][1]);
                if (dfX1 > -180 && dfX1 < dfRightBorderX && dfX2 === 180 &&
                    k + 1 < first_pass.length &&
                   first_pass[k - 1][0] > -180 && first_pass[k - 1][0] < dfRightBorderX) {
                    poNewLS.push([-180, first_pass[k][1]]);
                    k++;
                    poNewLS.push([first_pass[k][0], first_pass[k][1]]);
                    continue;
                } else if (dfX1 > dfLeftBorderX && dfX1 < 180 && dfX2 === -180 &&
                     k + 1 < first_pass.length &&
                     first_pass[k - 1][0] > dfLeftBorderX && first_pass[k - 1][0] < 180) {
                    poNewLS.push([180, first_pass[k][1]]);
                    k++;
                    poNewLS.push([first_pass[k][0], first_pass[k][1]]);
                    continue;
                }

                if (dfX1 < dfRightBorderX && dfX2 > dfLeftBorderX) {
                    // swap dfX1, dfX2
                    var tmpX = dfX1;
                    dfX1 = dfX2;
                    dfX2 = tmpX;
                    // swap dfY1, dfY2
                    var tmpY = dfY1;
                    dfY1 = dfY2;
                    dfY2 = tmpY;
                }
                if (dfX1 > dfLeftBorderX && dfX2 < dfRightBorderX) {
                    dfX2 += 360;
                }

                if (dfX1 <= 180 && dfX2 >= 180 && dfX1 < dfX2) {
                    var dfRatio = (180 - dfX1) / (dfX2 - dfX1);
                    var dfY = dfRatio * dfY2 + (1 - dfRatio) * dfY1;
                    poNewLS.push([first_pass[k - 1][0] > dfLeftBorderX ? 180 : -180, dfY]);
                    poNewLS = [];
                    poNewLS.push([first_pass[k - 1][0] > dfLeftBorderX ? -180 : 180, dfY]);
                    poMulti.push(poNewLS);
                } else {
                    poNewLS = [];
                    poMulti.push(poNewLS);
                }
                poNewLS.push([dfX0, first_pass[k][1]]);
            } else {
                poNewLS.push([first_pass[k][0], first_pass[k][1]]);
            }
        }
    } else {
        // add normally
        var poNewLS0 = [];
        poMulti.push(poNewLS0);
        for (var l = 0; l < first_pass.length; ++l) {
            poNewLS0.push([first_pass[l][0], first_pass[l][1]]);
        }
    }

    var arc = new Arc(this.properties);
    for (var m = 0; m < poMulti.length; ++m) {
        var line = new LineString();
        arc.geometries.push(line);
        var points = poMulti[m];
        for (var j0 = 0; j0 < points.length; ++j0) {
            line.move_to(points[j0]);
        }
    }
    return arc;
};

/**
 * Calculate great circles routes as {@link LineString}
 *
 * @name greatCircle
 * @param {Coord} start source point feature
 * @param {Coord} end destination point feature
 * @param {Object} [options={}] Optional parameters
 * @param {Object} [options.properties={}] line feature properties
 * @param {number} [options.npoints=100] number of points
 * @param {number} [options.offset=10] offset controls the likelyhood that lines will
 * be split which cross the dateline. The higher the number the more likely.
 * @returns {Feature<LineString>} great circle line feature
 * @example
 * var start = turf.point([-122, 48]);
 * var end = turf.point([-77, 39]);
 *
 * var greatCircle = turf.greatCircle(start, end, {'name': 'Seattle to DC'});
 *
 * //addToMap
 * var addToMap = [start, end, greatCircle]
 */
function greatCircle(start, end, options) {
    // Optional parameters
    options = options || {};
    if (typeof options !== 'object') throw new Error('options is invalid');
    var properties = options.properties;
    var npoints = options.npoints;
    var offset = options.offset;

    start = getCoord(start);
    end = getCoord(end);
    properties = properties || {};
    npoints = npoints || 100;
    offset = offset || 10;

    var generator = new GreatCircle({x: start[0], y: start[1]}, {x: end[0], y: end[1]}, properties);

    /* eslint-disable */
    var line = generator.Arc(npoints, {offset: offset});
    /* eslint-enable */

    return line.json();
}

/**
 * Split a LineString by another GeoJSON Feature.
 *
 * @name lineSplit
 * @param {Feature<LineString>} line LineString Feature to split
 * @param {Feature<any>} splitter Feature used to split line
 * @returns {FeatureCollection<LineString>} Split LineStrings
 * @example
 * var line = turf.lineString([[120, -25], [145, -25]]);
 * var splitter = turf.lineString([[130, -15], [130, -35]]);
 *
 * var split = turf.lineSplit(line, splitter);
 *
 * //addToMap
 * var addToMap = [line, splitter]
 */
function lineSplit(line, splitter) {
    if (!line) throw new Error('line is required');
    if (!splitter) throw new Error('splitter is required');

    var lineType = getType(line);
    var splitterType = getType(splitter);

    if (lineType !== 'LineString') throw new Error('line must be LineString');
    if (splitterType === 'FeatureCollection') throw new Error('splitter cannot be a FeatureCollection');
    if (splitterType === 'GeometryCollection') throw new Error('splitter cannot be a GeometryCollection');

    // remove excessive decimals from splitter
    // to avoid possible approximation issues in rbush
    var truncatedSplitter = truncate(splitter, {precision: 7});

    switch (splitterType) {
    case 'Point':
        return splitLineWithPoint(line, truncatedSplitter);
    case 'MultiPoint':
        return splitLineWithPoints(line, truncatedSplitter);
    case 'LineString':
    case 'MultiLineString':
    case 'Polygon':
    case 'MultiPolygon':
        return splitLineWithPoints(line, lineIntersect(line, truncatedSplitter));
    }
}

/**
 * Split LineString with MultiPoint
 *
 * @private
 * @param {Feature<LineString>} line LineString
 * @param {FeatureCollection<Point>} splitter Point
 * @returns {FeatureCollection<LineString>} split LineStrings
 */
function splitLineWithPoints(line, splitter) {
    var results = [];
    var tree = geojsonRbush();

    flattenEach(splitter, function (point$$1) {
        // Add index/id to features (needed for filter)
        results.forEach(function (feature$$1, index) {
            feature$$1.id = index;
        });
        // First Point - doesn't need to handle any previous line results
        if (!results.length) {
            results = splitLineWithPoint(line, point$$1).features;

            // Add Square BBox to each feature for GeoJSON-RBush
            results.forEach(function (feature$$1) {
                if (!feature$$1.bbox) feature$$1.bbox = square(bbox(feature$$1));
            });
            tree.load(featureCollection(results));
        // Split with remaining points - lines might needed to be split multiple times
        } else {
            // Find all lines that are within the splitter's bbox
            var search = tree.search(point$$1);

            if (search.features.length) {
                // RBush might return multiple lines - only process the closest line to splitter
                var closestLine = findClosestFeature(point$$1, search);

                // Remove closest line from results since this will be split into two lines
                // This removes any duplicates inside the results & index
                results = results.filter(function (feature$$1) { return feature$$1.id !== closestLine.id; });
                tree.remove(closestLine);

                // Append the two newly split lines into the results
                featureEach(splitLineWithPoint(closestLine, point$$1), function (line) {
                    results.push(line);
                    tree.insert(line);
                });
            }
        }
    });
    return featureCollection(results);
}

/**
 * Split LineString with Point
 *
 * @private
 * @param {Feature<LineString>} line LineString
 * @param {Feature<Point>} splitter Point
 * @returns {FeatureCollection<LineString>} split LineStrings
 */
function splitLineWithPoint(line, splitter) {
    var results = [];

    // handle endpoints
    var startPoint = getCoords(line)[0];
    var endPoint = getCoords(line)[line.geometry.coordinates.length - 1];
    if (pointsEquals(startPoint, getCoord(splitter)) ||
        pointsEquals(endPoint, getCoord(splitter))) return featureCollection([line]);

    // Create spatial index
    var tree = geojsonRbush();
    var segments = lineSegment(line);
    tree.load(segments);

    // Find all segments that are within bbox of splitter
    var search = tree.search(splitter);

    // Return itself if point is not within spatial index
    if (!search.features.length) return featureCollection([line]);

    // RBush might return multiple lines - only process the closest line to splitter
    var closestSegment = findClosestFeature(splitter, search);

    // Initial value is the first point of the first segments (beginning of line)
    var initialValue = [startPoint];
    var lastCoords = featureReduce(segments, function (previous, current, index) {
        var currentCoords = getCoords(current)[1];
        var splitterCoords = getCoord(splitter);

        // Location where segment intersects with line
        if (index === closestSegment.id) {
            previous.push(splitterCoords);
            results.push(lineString(previous));
            // Don't duplicate splitter coordinate (Issue #688)
            if (pointsEquals(splitterCoords, currentCoords)) return [splitterCoords];
            return [splitterCoords, currentCoords];

        // Keep iterating over coords until finished or intersection is found
        } else {
            previous.push(currentCoords);
            return previous;
        }
    }, initialValue);
    // Append last line to final split results
    if (lastCoords.length > 1) {
        results.push(lineString(lastCoords));
    }
    return featureCollection(results);
}


/**
 * Find Closest Feature
 *
 * @private
 * @param {Feature<Point>} point Feature must be closest to this point
 * @param {FeatureCollection<LineString>} lines Collection of Features
 * @returns {Feature<LineString>} closest LineString
 */
function findClosestFeature(point$$1, lines) {
    if (!lines.features.length) throw new Error('lines must contain features');
    // Filter to one segment that is the closest to the line
    if (lines.features.length === 1) return lines.features[0];

    var closestFeature;
    var closestDistance = Infinity;
    featureEach(lines, function (segment) {
        var pt = nearestPointOnLine(segment, point$$1);
        var dist = pt.properties.dist;
        if (dist < closestDistance) {
            closestFeature = segment;
            closestDistance = dist;
        }
    });
    return closestFeature;
}

/**
 * Compares two points and returns if they are equals
 *
 * @private
 * @param {Array<number>} pt1 point
 * @param {Array<number>} pt2 point
 * @returns {boolean} true if they are equals
 */
function pointsEquals(pt1, pt2) {
    return pt1[0] === pt2[0] && pt1[1] === pt2[1];
}

/**
 * Creates a circular arc, of a circle of the given radius and center point, between bearing1 and bearing2;
 * 0 bearing is North of center point, positive clockwise.
 *
 * @name lineArc
 * @param {Coord} center center point
 * @param {number} radius radius of the circle
 * @param {number} bearing1 angle, in decimal degrees, of the first radius of the arc
 * @param {number} bearing2 angle, in decimal degrees, of the second radius of the arc
 * @param {Object} [options={}] Optional parameters
 * @param {number} [options.steps=64] number of steps
 * @param {string} [options.units='kilometers'] miles, kilometers, degrees, or radians
 * @returns {Feature<LineString>} line arc
 * @example
 * var center = turf.point([-75, 40]);
 * var radius = 5;
 * var bearing1 = 25;
 * var bearing2 = 47;
 *
 * var arc = turf.lineArc(center, radius, bearing1, bearing2);
 *
 * //addToMap
 * var addToMap = [center, arc]
 */
function lineArc(center, radius, bearing1, bearing2, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var steps = options.steps;
    var units = options.units;

    // validation
    if (!center) throw new Error('center is required');
    if (!radius) throw new Error('radius is required');
    if (bearing1 === undefined || bearing1 === null) throw new Error('bearing1 is required');
    if (bearing2 === undefined || bearing2 === null) throw new Error('bearing2 is required');
    if (typeof options !== 'object') throw new Error('options must be an object');

    // default params
    steps = steps || 64;

    var angle1 = convertAngleTo360(bearing1);
    var angle2 = convertAngleTo360(bearing2);
    var properties = center.properties;

    // handle angle parameters
    if (angle1 === angle2) {
        return lineString(circle(center, radius, options).geometry.coordinates[0], properties);
    }
    var arcStartDegree = angle1;
    var arcEndDegree = (angle1 < angle2) ? angle2 : angle2 + 360;

    var alfa = arcStartDegree;
    var coordinates = [];
    var i = 0;

    while (alfa < arcEndDegree) {
        coordinates.push(destination(center, radius, alfa, units).geometry.coordinates);
        i++;
        alfa = arcStartDegree + i * 360 / steps;
    }
    if (alfa > arcEndDegree) {
        coordinates.push(destination(center, radius, arcEndDegree, units).geometry.coordinates);
    }
    return lineString(coordinates, properties);
}


/**
 * Takes any angle in  degrees
 * and returns a valid angle between 0-360 degrees
 *
 * @private
 * @param {number} alfa angle between -180-180 degrees
 * @returns {number} angle between 0-360 degrees
 */
function convertAngleTo360(alfa) {
    var beta = alfa % 360;
    if (beta < 0) {
        beta += 360;
    }
    return beta;
}

/**
 * Converts a {@link Polygon} to {@link LineString|(Multi)LineString} or {@link MultiPolygon} to a {@link FeatureCollection} of {@link LineString|(Multi)LineString}.
 *
 * @name polygonToLine
 * @param {Feature<Polygon|MultiPolygon>} polygon Feature to convert
 * @param {Object} [options={}] Optional parameters
 * @param {Object} [options.properties={}] translates GeoJSON properties to Feature
 * @returns {FeatureCollection|Feature<LineString|MultiLinestring>} converted (Multi)Polygon to (Multi)LineString
 * @example
 * var poly = turf.polygon([[[125, -30], [145, -30], [145, -20], [125, -20], [125, -30]]]);
 *
 * var line = turf.polygonToLine(poly);
 *
 * //addToMap
 * var addToMap = [line];
 */
function polygonToLine(polygon$$1, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var properties = options.properties;

    // Variables
    var geom = getType(polygon$$1);
    var coords = getCoords(polygon$$1);
    properties = properties || polygon$$1.properties || {};

    if (!coords.length) throw new Error('polygon must contain coordinates');

    switch (geom) {
    case 'Polygon':
        return coordsToLine(coords, properties);
    case 'MultiPolygon':
        var lines = [];
        coords.forEach(function (coord) {
            lines.push(coordsToLine(coord, properties));
        });
        return featureCollection(lines);
    default:
        throw new Error('geom ' + geom + ' not supported');
    }
}

function coordsToLine(coords, properties) {
    if (coords.length > 1) return multiLineString(coords, properties);
    return lineString(coords[0], properties);
}

/**
 * Converts (Multi)LineString(s) to Polygon(s).
 *
 * @name lineToPolygon
 * @param {FeatureCollection|Feature<LineString|MultiLineString>} lines Features to convert
 * @param {Object} [options={}] Optional parameters
 * @param {Object} [options.properties={}] translates GeoJSON properties to Feature
 * @param {boolean} [options.autoComplete=true] auto complete linestrings (matches first & last coordinates)
 * @param {boolean} [options.orderCoords=true] sorts linestrings to place outer ring at the first position of the coordinates
 * @returns {Feature<Polygon|MultiPolygon>} converted to Polygons
 * @example
 * var line = turf.lineString([[125, -30], [145, -30], [145, -20], [125, -20], [125, -30]]);
 *
 * var polygon = turf.lineToPolygon(line);
 *
 * //addToMap
 * var addToMap = [polygon];
 */
function lineToPolygon(lines, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var properties = options.properties;
    var autoComplete = options.autoComplete;
    var orderCoords = options.orderCoords;

    // validation
    if (!lines) throw new Error('lines is required');

    // default params
    autoComplete = (autoComplete !== undefined) ? autoComplete : true;
    orderCoords = (orderCoords !== undefined) ? orderCoords : true;
    var type = getType(lines);

    switch (type) {
    case 'FeatureCollection':
    case 'GeometryCollection':
        var coords = [];
        var features = (lines.features) ? lines.features : lines.geometries;
        features.forEach(function (line) {
            coords.push(getCoords(lineStringToPolygon(line, {}, autoComplete, orderCoords)));
        });
        return multiPolygon(coords, properties);
    }
    return lineStringToPolygon(lines, properties, autoComplete, orderCoords);
}

/**
 * LineString to Polygon
 *
 * @private
 * @param {Feature<LineString|MultiLineString>} line line
 * @param {Object} [properties] translates GeoJSON properties to Feature
 * @param {boolean} [autoComplete=true] auto complete linestrings
 * @param {boolean} [orderCoords=true] sorts linestrings to place outer ring at the first position of the coordinates
 * @returns {Feature<Polygon>} line converted to Polygon
 */
function lineStringToPolygon(line, properties, autoComplete, orderCoords) {
    properties = properties || line.properties || {};
    var coords = getCoords(line);
    var type = getType(line);

    if (!coords.length) throw new Error('line must contain coordinates');

    switch (type) {
    case 'LineString':
        if (autoComplete) coords = autoCompleteCoords(coords);
        return polygon([coords], properties);
    case 'MultiLineString':
        var multiCoords = [];
        var largestArea = 0;

        coords.forEach(function (coord) {
            if (autoComplete) coord = autoCompleteCoords(coord);

            // Largest LineString to be placed in the first position of the coordinates array
            if (orderCoords) {
                var area = calculateArea$1(bbox(lineString(coord)));
                if (area > largestArea) {
                    multiCoords.unshift(coord);
                    largestArea = area;
                } else multiCoords.push(coord);
            } else {
                multiCoords.push(coord);
            }
        });
        return polygon(multiCoords, properties);
    default:
        throw new Error('geometry type ' + type + ' is not supported');
    }
}

/**
 * Auto Complete Coords - matches first & last coordinates
 *
 * @private
 * @param {Array<Array<number>>} coords Coordinates
 * @returns {Array<Array<number>>} auto completed coordinates
 */
function autoCompleteCoords(coords) {
    var first = coords[0];
    var x1 = first[0];
    var y1 = first[1];
    var last = coords[coords.length - 1];
    var x2 = last[0];
    var y2 = last[1];
    if (x1 !== x2 || y1 !== y2) {
        coords.push(first);
    }
    return coords;
}

/**
 * area - quick approximate area calculation (used to sort)
 *
 * @private
 * @param {Array<number>} bbox BBox [west, south, east, north]
 * @returns {number} very quick area calculation
 */
function calculateArea$1(bbox$$1) {
    var west = bbox$$1[0];
    var south = bbox$$1[1];
    var east = bbox$$1[2];
    var north = bbox$$1[3];
    return Math.abs(west - east) * Math.abs(south - north);
}

var lineclip_1 = lineclip;

lineclip.polyline = lineclip;
lineclip.polygon = polygonclip;


// Cohen-Sutherland line clippign algorithm, adapted to efficiently
// handle polylines rather than just segments

function lineclip(points, bbox, result) {

    var len = points.length,
        codeA = bitCode(points[0], bbox),
        part = [],
        i, a, b, codeB, lastCode;

    if (!result) result = [];

    for (i = 1; i < len; i++) {
        a = points[i - 1];
        b = points[i];
        codeB = lastCode = bitCode(b, bbox);

        while (true) {

            if (!(codeA | codeB)) { // accept
                part.push(a);

                if (codeB !== lastCode) { // segment went outside
                    part.push(b);

                    if (i < len - 1) { // start a new line
                        result.push(part);
                        part = [];
                    }
                } else if (i === len - 1) {
                    part.push(b);
                }
                break;

            } else if (codeA & codeB) { // trivial reject
                break;

            } else if (codeA) { // a outside, intersect with clip edge
                a = intersect$1(a, b, codeA, bbox);
                codeA = bitCode(a, bbox);

            } else { // b outside
                b = intersect$1(a, b, codeB, bbox);
                codeB = bitCode(b, bbox);
            }
        }

        codeA = lastCode;
    }

    if (part.length) result.push(part);

    return result;
}

// Sutherland-Hodgeman polygon clipping algorithm

function polygonclip(points, bbox) {

    var result, edge, prev, prevInside, i, p, inside;

    // clip against each side of the clip rectangle
    for (edge = 1; edge <= 8; edge *= 2) {
        result = [];
        prev = points[points.length - 1];
        prevInside = !(bitCode(prev, bbox) & edge);

        for (i = 0; i < points.length; i++) {
            p = points[i];
            inside = !(bitCode(p, bbox) & edge);

            // if segment goes through the clip window, add an intersection
            if (inside !== prevInside) result.push(intersect$1(prev, p, edge, bbox));

            if (inside) result.push(p); // add a point if it's inside

            prev = p;
            prevInside = inside;
        }

        points = result;

        if (!points.length) break;
    }

    return result;
}

// intersect a segment against one of the 4 lines that make up the bbox

function intersect$1(a, b, edge, bbox) {
    return edge & 8 ? [a[0] + (b[0] - a[0]) * (bbox[3] - a[1]) / (b[1] - a[1]), bbox[3]] : // top
           edge & 4 ? [a[0] + (b[0] - a[0]) * (bbox[1] - a[1]) / (b[1] - a[1]), bbox[1]] : // bottom
           edge & 2 ? [bbox[2], a[1] + (b[1] - a[1]) * (bbox[2] - a[0]) / (b[0] - a[0])] : // right
           edge & 1 ? [bbox[0], a[1] + (b[1] - a[1]) * (bbox[0] - a[0]) / (b[0] - a[0])] : // left
           null;
}

// bit code reflects the point position relative to the bbox:

//         left  mid  right
//    top  1001  1000  1010
//    mid  0001  0000  0010
// bottom  0101  0100  0110

function bitCode(p, bbox) {
    var code = 0;

    if (p[0] < bbox[0]) code |= 1; // left
    else if (p[0] > bbox[2]) code |= 2; // right

    if (p[1] < bbox[1]) code |= 4; // bottom
    else if (p[1] > bbox[3]) code |= 8; // top

    return code;
}

/**
 * Takes a {@link Feature} and a bbox and clips the feature to the bbox using [lineclip](https://github.com/mapbox/lineclip).
 * May result in degenerate edges when clipping Polygons.
 *
 * @name bboxClip
 * @param {Feature<LineString|MultiLineString|Polygon|MultiPolygon>} feature feature to clip to the bbox
 * @param {BBox} bbox extent in [minX, minY, maxX, maxY] order
 * @returns {Feature<LineString|MultiLineString|Polygon|MultiPolygon>} clipped Feature
 * @example
 * var bbox = [0, 0, 10, 10];
 * var poly = turf.polygon([[[2, 2], [8, 4], [12, 8], [3, 7], [2, 2]]]);
 *
 * var clipped = turf.bboxClip(poly, bbox);
 *
 * //addToMap
 * var addToMap = [bbox, poly, clipped]
 */
function bboxClip(feature$$1, bbox) {
    var geom = getGeom$1(feature$$1);
    var coords = getCoords(feature$$1);
    var properties = feature$$1.properties;

    switch (geom) {
    case 'LineString':
    case 'MultiLineString':
        var lines = [];
        if (geom === 'LineString') coords = [coords];
        coords.forEach(function (line) {
            lineclip_1(line, bbox, lines);
        });
        if (lines.length === 1) return lineString(lines[0], properties);
        return multiLineString(lines, properties);
    case 'Polygon':
        return polygon(clipPolygon(coords, bbox), properties);
    case 'MultiPolygon':
        return multiPolygon(coords.map(function (polygon$$1) {
            return clipPolygon(polygon$$1, bbox);
        }), properties);
    default:
        throw new Error('geometry ' + geom + ' not supported');
    }
}

function clipPolygon(rings, bbox) {
    var outRings = [];
    for (var i = 0; i < rings.length; i++) {
        var clipped = lineclip_1.polygon(rings[i], bbox);
        if (clipped.length > 0) {
            if (clipped[0][0] !== clipped[clipped.length - 1][0] || clipped[0][1] !== clipped[clipped.length - 1][1]) {
                clipped.push(clipped[0]);
            }
            if (clipped.length >= 4) {
                outRings.push(clipped);
            }
        }
    }
    return outRings;
}

function getGeom$1(feature$$1) {
    return (feature$$1.geometry) ? feature$$1.geometry.type : feature$$1.type;
}

var pSlice = Array.prototype.slice;

function isArguments(object) {
    return Object.prototype.toString.call(object) === '[object Arguments]';
}

function deepEqual(actual, expected, opts) {
    if (!opts) opts = {};
    // 7.1. All identical values are equivalent, as determined by ===.
    if (actual === expected) {
        return true;

    } else if (actual instanceof Date && expected instanceof Date) {
        return actual.getTime() === expected.getTime();

        // 7.3. Other pairs that do not both pass typeof value == 'object',
        // equivalence is determined by ==.
    } else if (!actual || !expected || typeof actual != 'object' && typeof expected != 'object') {
        return opts.strict ? actual === expected : actual === expected;

        // 7.4. For all other Object pairs, including Array objects, equivalence is
        // determined by having the same number of owned properties (as verified
        // with Object.prototype.hasOwnProperty.call), the same set of keys
        // (although not necessarily the same order), equivalent values for every
        // corresponding key, and an identical 'prototype' property. Note: this
        // accounts for both named and indexed properties on Arrays.
    } else {
        return objEquiv(actual, expected, opts);
    }
}

function isUndefinedOrNull(value) {
    return value === null || value === undefined;
}

function isBuffer(x) {
    if (!x || typeof x !== 'object' || typeof x.length !== 'number') return false;
    if (typeof x.copy !== 'function' || typeof x.slice !== 'function') {
        return false;
    }
    if (x.length > 0 && typeof x[0] !== 'number') return false;
    return true;
}

function objEquiv(a, b, opts) {
    var i, key;
    if (isUndefinedOrNull(a) || isUndefinedOrNull(b))
        return false;
    // an identical 'prototype' property.
    if (a.prototype !== b.prototype) return false;
    //~~~I've managed to break Object.keys through screwy arguments passing.
    //   Converting to array solves the problem.
    if (isArguments(a)) {
        if (!isArguments(b)) {
            return false;
        }
        a = pSlice.call(a);
        b = pSlice.call(b);
        return deepEqual(a, b, opts);
    }
    if (isBuffer(a)) {
        if (!isBuffer(b)) {
            return false;
        }
        if (a.length !== b.length) return false;
        for (i = 0; i < a.length; i++) {
            if (a[i] !== b[i]) return false;
        }
        return true;
    }
    try {
        var ka = Object.keys(a),
            kb = Object.keys(b);
    } catch (e) { //happens when one is a string literal and the other isn't
        return false;
    }
    // having the same number of owned properties (keys incorporates
    // hasOwnProperty)
    if (ka.length !== kb.length)
        return false;
    //the same set of keys (although not necessarily the same order),
    ka.sort();
    kb.sort();
    //~~~cheap key test
    for (i = ka.length - 1; i >= 0; i--) {
        if (ka[i] !== kb[i])
            return false;
    }
    //equivalent values for every corresponding key, and
    //~~~possibly expensive deep test
    for (i = ka.length - 1; i >= 0; i--) {
        key = ka[i];
        if (!deepEqual(a[key], b[key], opts)) return false;
    }
    return typeof a === typeof b;
}

/**
 * Takes any LineString or Polygon and returns the overlapping lines between both features.
 *
 * @name lineOverlap
 * @param {Geometry|Feature<LineString|MultiLineString|Polygon|MultiPolygon>} line1 any LineString or Polygon
 * @param {Geometry|Feature<LineString|MultiLineString|Polygon|MultiPolygon>} line2 any LineString or Polygon
 * @param {Object} [options={}] Optional parameters
 * @param {number} [options.tolerance=0] Tolerance distance to match overlapping line segments (in kilometers)
 * @returns {FeatureCollection<LineString>} lines(s) that are overlapping between both features
 * @example
 * var line1 = turf.lineString([[115, -35], [125, -30], [135, -30], [145, -35]]);
 * var line2 = turf.lineString([[115, -25], [125, -30], [135, -30], [145, -25]]);
 *
 * var overlapping = turf.lineOverlap(line1, line2);
 *
 * //addToMap
 * var addToMap = [line1, line2, overlapping]
 */
function lineOverlap(line1, line2, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var tolerance = options.tolerance || 0;

    // Containers
    var features = [];

    // Create Spatial Index
    var tree = geojsonRbush();
    tree.load(lineSegment(line1));
    var overlapSegment;

    // Line Intersection

    // Iterate over line segments
    segmentEach(line2, function (segment) {
        var doesOverlaps = false;

        // Iterate over each segments which falls within the same bounds
        featureEach(tree.search(segment), function (match) {
            if (doesOverlaps === false) {
                var coordsSegment = getCoords(segment).sort();
                var coordsMatch = getCoords(match).sort();

                // Segment overlaps feature
                if (deepEqual(coordsSegment, coordsMatch)) {
                    doesOverlaps = true;
                    // Overlaps already exists - only append last coordinate of segment
                    if (overlapSegment) overlapSegment = concatSegment(overlapSegment, segment);
                    else overlapSegment = segment;
                // Match segments which don't share nodes (Issue #901)
                } else if (
                    (tolerance === 0) ?
                        booleanPointOnLine(coordsSegment[0], match) && booleanPointOnLine(coordsSegment[1], match) :
                        nearestPointOnLine(match, coordsSegment[0]).properties.dist <= tolerance &&
                        nearestPointOnLine(match, coordsSegment[1]).properties.dist <= tolerance) {
                    doesOverlaps = true;
                    if (overlapSegment) overlapSegment = concatSegment(overlapSegment, segment);
                    else overlapSegment = segment;
                } else if (
                    (tolerance === 0) ?
                        booleanPointOnLine(coordsMatch[0], segment) && booleanPointOnLine(coordsMatch[1], segment) :
                        nearestPointOnLine(segment, coordsMatch[0]).properties.dist <= tolerance &&
                        nearestPointOnLine(segment, coordsMatch[1]).properties.dist <= tolerance) {
                    // Do not define (doesOverlap = true) since more matches can occur within the same segment
                    // doesOverlaps = true;
                    if (overlapSegment) overlapSegment = concatSegment(overlapSegment, match);
                    else overlapSegment = match;
                }
            }
        });

        // Segment doesn't overlap - add overlaps to results & reset
        if (doesOverlaps === false && overlapSegment) {
            features.push(overlapSegment);
            overlapSegment = undefined;
        }
    });
    // Add last segment if exists
    if (overlapSegment) features.push(overlapSegment);

    return featureCollection(features);
}


/**
 * Concat Segment
 *
 * @private
 * @param {Feature<LineString>} line LineString
 * @param {Feature<LineString>} segment 2-vertex LineString
 * @returns {Feature<LineString>} concat linestring
 */
function concatSegment(line, segment) {
    var coords = getCoords(segment);
    var lineCoords = getCoords(line);
    var start = lineCoords[0];
    var end = lineCoords[lineCoords.length - 1];
    var geom = line.geometry.coordinates;

    if (deepEqual(coords[0], start)) geom.unshift(coords[1]);
    else if (deepEqual(coords[0], end)) geom.push(coords[1]);
    else if (deepEqual(coords[1], start)) geom.unshift(coords[0]);
    else if (deepEqual(coords[1], end)) geom.push(coords[0]);
    return line;
}

/**
 * Creates a circular sector of a circle of given radius and center {@link Point},
 * between (clockwise) bearing1 and bearing2; 0 bearing is North of center point, positive clockwise.
 *
 * @name sector
 * @param {Coord} center center point
 * @param {number} radius radius of the circle
 * @param {number} bearing1 angle, in decimal degrees, of the first radius of the sector
 * @param {number} bearing2 angle, in decimal degrees, of the second radius of the sector
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units='kilometers'] miles, kilometers, degrees, or radians
 * @param {number} [options.steps=64] number of steps
 * @returns {Feature<Polygon>} sector polygon
 * @example
 * var center = turf.point([-75, 40]);
 * var radius = 5;
 * var bearing1 = 25;
 * var bearing2 = 45;
 *
 * var sector = turf.sector(center, radius, bearing1, bearing2);
 *
 * //addToMap
 * var addToMap = [center, sector];
 */
function sector(center, radius, bearing1, bearing2, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');

    // validation
    if (!center) throw new Error('center is required');
    if (bearing1 === undefined || bearing1 === null) throw new Error('bearing1 is required');
    if (bearing2 === undefined || bearing2 === null) throw new Error('bearing2 is required');
    if (!radius) throw new Error('radius is required');
    if (typeof options !== 'object') throw new Error('options must be an object');

    if (convertAngleTo360$1(bearing1) === convertAngleTo360$1(bearing2)) {
        return circle(center, radius, options);
    }
    var coords = getCoords(center);
    var arc = lineArc(center, radius, bearing1, bearing2, options);
    var sliceCoords = [[coords]];
    coordEach(arc, function (currentCoords) {
        sliceCoords[0].push(currentCoords);
    });
    sliceCoords[0].push(coords);

    return polygon(sliceCoords);
}

/**
 * Takes any angle in degrees
 * and returns a valid angle between 0-360 degrees
 *
 * @private
 * @param {number} alfa angle between -180-180 degrees
 * @returns {number} angle between 0-360 degrees
 */
function convertAngleTo360$1(alfa) {
    var beta = alfa % 360;
    if (beta < 0) beta += 360;
    return beta;
}

// https://en.wikipedia.org/wiki/Rhumb_line
/**
 * Returns the destination {@link Point} having travelled the given distance along a Rhumb line from the
 * origin Point with the (varant) given bearing.
 *
 * @name rhumbDestination
 * @param {Coord} origin starting point
 * @param {number} distance distance from the starting point
 * @param {number} bearing varant bearing angle ranging from -180 to 180 degrees from north
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units='kilometers'] can be degrees, radians, miles, or kilometers
 * @param {Object} [options.properties={}] translate properties to destination point
 * @returns {Feature<Point>} Destination point.
 * @example
 * var pt = turf.point([-75.343, 39.984], {"marker-color": "F00"});
 * var distance = 50;
 * var bearing = 90;
 * var options = {units: 'miles'};
 *
 * var destination = turf.rhumbDestination(pt, distance, bearing, options);
 *
 * //addToMap
 * var addToMap = [pt, destination]
 * destination.properties['marker-color'] = '#00F';
 */
function rhumbDestination(origin, distance, bearing, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var units = options.units;
    var properties = options.properties;

    // validation
    if (!origin) throw new Error('origin is required');
    if (distance === undefined || distance === null) throw new Error('distance is required');
    if (bearing === undefined || bearing === null) throw new Error('bearing is required');
    if (!(distance >= 0)) throw new Error('distance must be greater than 0');

    var distanceInMeters = convertLength(distance, units, 'meters');
    var coords = getCoord(origin);
    var destination = calculateRhumbDestination(coords, distanceInMeters, bearing);

    // compensate the crossing of the 180th meridian (https://macwright.org/2016/09/26/the-180th-meridian.html)
    // solution from https://github.com/mapbox/mapbox-gl-js/issues/3250#issuecomment-294887678
    destination[0] += (destination[0] - coords[0] > 180) ? -360 : (coords[0] - destination[0] > 180) ? 360 : 0;
    return point(destination, properties);
}

/**
 * Returns the destination point having travelled along a rhumb line from origin point the given
 * distance on the  given bearing.
 * Adapted from Geodesy: http://www.movable-type.co.uk/scripts/latlong.html#rhumblines
 *
 * @private
 * @param   {Array<number>} origin - point
 * @param   {number} distance - Distance travelled, in same units as earth radius (default: metres).
 * @param   {number} bearing - Bearing in degrees from north.
 * @param   {number} [radius=6371e3] - (Mean) radius of earth (defaults to radius in metres).
 * @returns {Array<number>} Destination point.
 */
function calculateRhumbDestination(origin, distance, bearing, radius) {
    // φ => phi
    // λ => lambda
    // ψ => psi
    // Δ => Delta
    // δ => delta
    // θ => theta

    radius = (radius === undefined) ? earthRadius : Number(radius);

    var delta = distance / radius; // angular distance in radians
    var lambda1 = origin[0] * Math.PI / 180; // to radians, but without normalize to 𝜋
    var phi1 = degreesToRadians(origin[1]);
    var theta = degreesToRadians(bearing);

    var DeltaPhi = delta * Math.cos(theta);
    var phi2 = phi1 + DeltaPhi;

    // check for some daft bugger going past the pole, normalise latitude if so
    if (Math.abs(phi2) > Math.PI / 2) phi2 = phi2 > 0 ? Math.PI - phi2 : -Math.PI - phi2;

    var DeltaPsi = Math.log(Math.tan(phi2 / 2 + Math.PI / 4) / Math.tan(phi1 / 2 + Math.PI / 4));
    var q = Math.abs(DeltaPsi) > 10e-12 ? DeltaPhi / DeltaPsi : Math.cos(phi1); // E-W course becomes ill-conditioned with 0/0

    var DeltaLambda = delta * Math.sin(theta) / q;
    var lambda2 = lambda1 + DeltaLambda;

    return [((lambda2 * 180 / Math.PI) + 540) % 360 - 180, phi2 * 180 / Math.PI]; // normalise to −180..+180°
}

/**
 * Finds the tangents of a {@link Polygon|(Multi)Polygon} from a {@link Point}.
 *
 * @name polygonTangents
 * @param {Coord} pt to calculate the tangent points from
 * @param {Feature<Polygon|MultiPolygon>} polygon to get tangents from
 * @returns {FeatureCollection<Point>} Feature Collection containing the two tangent points
 * @example
 * var polygon = turf.polygon([[[11, 0], [22, 4], [31, 0], [31, 11], [21, 15], [11, 11], [11, 0]]]);
 * var point = turf.point([61, 5]);
 *
 * var tangents = turf.polygonTangents(point, polygon)
 *
 * //addToMap
 * var addToMap = [tangents, point, polygon];
 */
function polygonTangents(pt, polygon$$1) {
    var pointCoords = getCoords(pt);
    var polyCoords = getCoords(polygon$$1);

    var rtan;
    var ltan;
    var enext;
    var eprev;

    var type = getType(polygon$$1);
    switch (type) {
    case 'Polygon':
        rtan = polyCoords[0][0];
        ltan = polyCoords[0][0];
        eprev = isLeft(polyCoords[0][0], polyCoords[0][polyCoords[0].length - 1], pointCoords);
        var out = processPolygon$1(polyCoords[0], pointCoords, eprev, enext, rtan, ltan);
        rtan = out[0];
        ltan = out[1];
        break;
    case 'MultiPolygon':
        rtan = polyCoords[0][0][0];
        ltan = polyCoords[0][0][0];
        eprev = isLeft(polyCoords[0][0][0], polyCoords[0][0][polyCoords[0][0].length - 1], pointCoords);
        polyCoords.forEach(function (ring) {
            var out = processPolygon$1(ring[0], pointCoords, eprev, enext, rtan, ltan);
            rtan = out[0];
            ltan = out[1];
        });
        break;
    }
    return featureCollection([point(rtan), point(ltan)]);
}

function processPolygon$1(polygonCoords, ptCoords, eprev, enext, rtan, ltan) {
    for (var i = 0; i < polygonCoords.length; i++) {
        var currentCoords = polygonCoords[i];
        var nextCoordPair = polygonCoords[i + 1];
        if (i === polygonCoords.length - 1) {
            nextCoordPair = polygonCoords[0];
        }
        enext = isLeft(currentCoords, nextCoordPair, ptCoords);
        if (eprev <= 0 && enext > 0) {
            if (!isBelow(ptCoords, currentCoords, rtan)) {
                rtan = currentCoords;
            }
        } else if (eprev > 0 && enext <= 0) {
            if (!isAbove(ptCoords, currentCoords, ltan)) {
                ltan = currentCoords;
            }
        }
        eprev = enext;
    }
    return [rtan, ltan];
}

function isAbove(point1, point2, point3) {
    return isLeft(point1, point2, point3) > 0;
}

function isBelow(point1, point2, point3) {
    return isLeft(point1, point2, point3) < 0;
}

function isLeft(point1, point2, point3) {
    return (point2[0] - point1[0]) * (point3[1] - point1[1]) - (point3[0] - point1[0]) * (point2[1] - point1[1]);
}

/**
 * Takes a ring and return true or false whether or not the ring is clockwise or counter-clockwise.
 *
 * @name booleanClockwise
 * @param {Feature<LineString>} line to be evaluated
 * @returns {boolean} true/false
 * @example
 * var clockwiseRing = turf.lineString([[0,0],[1,1],[1,0],[0,0]]);
 * var counterClockwiseRing = turf.lineString([[0,0],[1,0],[1,1],[0,0]]);
 *
 * turf.booleanClockwise(clockwiseRing)
 * //=true
 * turf.booleanClockwise(counterClockwiseRing)
 * //=false
 */
function booleanClockwise(line) {
    // validation
    if (!line) throw new Error('line is required');
    var type = (line.geometry) ? line.geometry.type : line.type;
    if (!Array.isArray(line) && type !== 'LineString') throw new Error('geometry must be a LineString');

    var ring = getCoords(line);
    var sum = 0;
    var i = 1;
    var prev, cur;
    while (i < ring.length) {
        prev = cur || ring[0];
        cur = ring[i];
        sum += ((cur[0] - prev[0]) * (cur[1] + prev[1]));
        i++;
    }
    return sum > 0;
}

/**
 * Rewind {@link LineString|(Multi)LineString} or {@link Polygon|(Multi)Polygon} outer ring counterclockwise and inner rings clockwise (Uses {@link http://en.wikipedia.org/wiki/Shoelace_formula|Shoelace Formula}).
 *
 * @name rewind
 * @param {GeoJSON} geojson input GeoJSON Polygon
 * @param {Object} [options={}] Optional parameters
 * @param {boolean} [options.reverse=false] enable reverse winding
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated (significant performance increase if true)
 * @returns {GeoJSON} rewind Polygon
 * @example
 * var polygon = turf.polygon([[[121, -29], [138, -29], [138, -18], [121, -18], [121, -29]]]);
 *
 * var rewind = turf.rewind(polygon);
 *
 * //addToMap
 * var addToMap = [rewind];
 */
function rewind(geojson, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var reverse = options.reverse || false;
    var mutate = options.mutate || false;

    // validation
    if (!geojson) throw new Error('<geojson> is required');
    if (typeof reverse !== 'boolean') throw new Error('<reverse> must be a boolean');
    if (typeof mutate !== 'boolean') throw new Error('<mutate> must be a boolean');

    // prevent input mutation
    if (mutate === false) geojson = clone(geojson);

    // Support Feature Collection or Geometry Collection
    var results = [];
    switch (geojson.type) {
    case 'GeometryCollection':
        geomEach(geojson, function (geometry$$1) {
            rewindFeature(geometry$$1, reverse);
        });
        return geojson;
    case 'FeatureCollection':
        featureEach(geojson, function (feature$$1) {
            featureEach(rewindFeature(feature$$1, reverse), function (result) {
                results.push(result);
            });
        });
        return featureCollection(results);
    }
    // Support Feature or Geometry Objects
    return rewindFeature(geojson, reverse);
}

/**
 * Rewind
 *
 * @private
 * @param {Geometry|Feature<any>} geojson Geometry or Feature
 * @param {Boolean} [reverse=false] enable reverse winding
 * @returns {Geometry|Feature<any>} rewind Geometry or Feature
 */
function rewindFeature(geojson, reverse) {
    var type = (geojson.type === 'Feature') ? geojson.geometry.type : geojson.type;

    // Support all GeoJSON Geometry Objects
    switch (type) {
    case 'GeometryCollection':
        geomEach(geojson, function (geometry$$1) {
            rewindFeature(geometry$$1, reverse);
        });
        return geojson;
    case 'LineString':
        rewindLineString(getCoords(geojson), reverse);
        return geojson;
    case 'Polygon':
        rewindPolygon(getCoords(geojson), reverse);
        return geojson;
    case 'MultiLineString':
        getCoords(geojson).forEach(function (lineCoords) {
            rewindLineString(lineCoords, reverse);
        });
        return geojson;
    case 'MultiPolygon':
        getCoords(geojson).forEach(function (lineCoords) {
            rewindPolygon(lineCoords, reverse);
        });
        return geojson;
    case 'Point':
    case 'MultiPoint':
        return geojson;
    }
}

/**
 * Rewind LineString - outer ring clockwise
 *
 * @private
 * @param {Array<Array<number>>} coords GeoJSON LineString geometry coordinates
 * @param {Boolean} [reverse=false] enable reverse winding
 * @returns {void} mutates coordinates
 */
function rewindLineString(coords, reverse) {
    if (booleanClockwise(coords) === reverse) coords.reverse();
}

/**
 * Rewind Polygon - outer ring counterclockwise and inner rings clockwise.
 *
 * @private
 * @param {Array<Array<Array<number>>>} coords GeoJSON Polygon geometry coordinates
 * @param {Boolean} [reverse=false] enable reverse winding
 * @returns {void} mutates coordinates
 */
function rewindPolygon(coords, reverse) {
    // outer ring
    if (booleanClockwise(coords[0]) !== reverse) {
        coords[0].reverse();
    }
    // inner rings
    for (var i = 1; i < coords.length; i++) {
        if (booleanClockwise(coords[i]) === reverse) {
            coords[i].reverse();
        }
    }
}

/**
 * Takes a {@link Point} grid and returns a correspondent matrix {Array<Array<number>>}
 * of the 'property' values
 *
 * @name gridToMatrix
 * @param {FeatureCollection<Point>} grid of points
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.zProperty='elevation'] the property name in `points` from which z-values will be pulled
 * @param {boolean} [options.flip=false] returns the matrix upside-down
 * @param {boolean} [options.flags=false] flags, adding a `matrixPosition` array field ([row, column]) to its properties,
 * the grid points with coordinates on the matrix
 * @returns {Array<Array<number>>} matrix of property values
 * @example
 *   var extent = [-70.823364, -33.553984, -70.473175, -33.302986];
 *   var cellSize = 3;
 *   var grid = turf.pointGrid(extent, cellSize);
 *   // add a random property to each point between 0 and 60
 *   for (var i = 0; i < grid.features.length; i++) {
 *     grid.features[i].properties.elevation = (Math.random() * 60);
 *   }
 *   gridToMatrix(grid);
 *   //= [
 *     [ 1, 13, 10,  9, 10, 13, 18],
 *     [34,  8,  5,  4,  5,  8, 13],
 *     [10,  5,  2,  1,  2,  5,  4],
 *     [ 0,  4, 56, 19,  1,  4,  9],
 *     [10,  5,  2,  1,  2,  5, 10],
 *     [57,  8,  5,  4,  5,  0, 57],
 *     [ 3, 13, 10,  9,  5, 13, 18],
 *     [18, 13, 10,  9, 78, 13, 18]
 *   ]
 */
function gridToMatrix$1(grid, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var zProperty = options.zProperty || 'elevation';
    var flip = options.flip;
    var flags = options.flags;

    // validation
    collectionOf(grid, 'Point', 'input must contain Points');

    var pointsMatrix = sortPointsByLatLng$1(grid, flip);

    var matrix = [];
    // create property matrix from sorted points
    // looping order matters here
    for (var r = 0; r < pointsMatrix.length; r++) {
        var pointRow = pointsMatrix[r];
        var row = [];
        for (var c = 0; c < pointRow.length; c++) {
            var point$$1 = pointRow[c];
            // Check if zProperty exist
            if (point$$1.properties[zProperty]) row.push(point$$1.properties[zProperty]);
            else row.push(0);
            // add flags
            if (flags === true) point$$1.properties.matrixPosition = [r, c];
        }
        matrix.push(row);
    }

    return matrix;
}

/**
 * Sorts points by latitude and longitude, creating a 2-dimensional array of points
 *
 * @private
 * @param {FeatureCollection<Point>} points GeoJSON Point features
 * @param {boolean} [flip=false] returns the matrix upside-down
 * @returns {Array<Array<Point>>} points ordered by latitude and longitude
 */
function sortPointsByLatLng$1(points$$1, flip) {
    var pointsByLatitude = {};

    // divide points by rows with the same latitude
    featureEach(points$$1, function (point$$1) {
        var lat = getCoords(point$$1)[1];
        if (!pointsByLatitude[lat]) pointsByLatitude[lat] = [];
        pointsByLatitude[lat].push(point$$1);
    });

    // sort points (with the same latitude) by longitude
    var orderedRowsByLatitude = Object.keys(pointsByLatitude).map(function (lat) {
        var row = pointsByLatitude[lat];
        var rowOrderedByLongitude = row.sort(function (a, b) {
            return getCoords(a)[0] - getCoords(b)[0];
        });
        return rowOrderedByLongitude;
    });

    // sort rows (of points with the same latitude) by latitude
    var pointMatrix = orderedRowsByLatitude.sort(function (a, b) {
        if (flip) return getCoords(a[0])[1] - getCoords(b[0])[1];
        else return getCoords(b[0])[1] - getCoords(a[0])[1];
    });

    return pointMatrix;
}

/*!
* @license GNU Affero General Public License.
* Copyright (c) 2015, 2015 Ronny Lorenz <ronny@tbi.univie.ac.at>
* v. 1.2.0
* https://github.com/RaumZeit/MarchingSquares.js
*/

var defaultSettings$1 = {
    successCallback: null,
    verbose: false,
    polygons: false
};

var settings$1 = {};

/*
  Compute isobands(s) of a scalar 2D field given a certain
  threshold and a bandwidth by applying the Marching Squares
  Algorithm. The function returns a list of path coordinates
  either for individual polygons within each grid cell, or the
  outline of connected polygons.
*/
function isoBands(data, minV, bandwidth, options) {
    /* process options */
    options = options ? options : {};

    var optionKeys = Object.keys(defaultSettings$1);

    for (var i = 0; i < optionKeys.length; i++) {
        var key = optionKeys[i];
        var val = options[key];
        val = ((typeof val !== 'undefined') && (val !== null)) ? val : defaultSettings$1[key];

        settings$1[key] = val;
    }

    if (settings$1.verbose)
        console.log('MarchingSquaresJS-isoBands: computing isobands for [' + minV + ':' + (minV + bandwidth) + ']');

    var grid = computeBandGrid(data, minV, bandwidth);

    var ret;
    if (settings$1.polygons) {
        if (settings$1.verbose)
            console.log('MarchingSquaresJS-isoBands: returning single polygons for each grid cell');
        ret = BandGrid2Areas(grid);
    } else {
        if (settings$1.verbose)
            console.log('MarchingSquaresJS-isoBands: returning polygon paths for entire data grid');
        ret = BandGrid2AreaPaths(grid);
    }

    if (typeof settings$1.successCallback === 'function')
        settings$1.successCallback(ret);

    return ret;
}

/*
  Thats all for the public interface, below follows the actual
  implementation
*/

/* Some private variables */
var Node0 = 64;
var Node1 = 16;
var Node2 = 4;
var Node3 = 1;

/*
  The look-up tables for tracing back the contour path
  of isoBands
*/

var isoBandNextXTL = [];
var isoBandNextYTL = [];
var isoBandNextOTL = [];

var isoBandNextXTR = [];
var isoBandNextYTR = [];
var isoBandNextOTR = [];

var isoBandNextXRT = [];
var isoBandNextYRT = [];
var isoBandNextORT = [];

var isoBandNextXRB = [];
var isoBandNextYRB = [];
var isoBandNextORB = [];

var isoBandNextXBL = [];
var isoBandNextYBL = [];
var isoBandNextOBL = [];

var isoBandNextXBR = [];
var isoBandNextYBR = [];
var isoBandNextOBR = [];

var isoBandNextXLT = [];
var isoBandNextYLT = [];
var isoBandNextOLT = [];

var isoBandNextXLB = [];
var isoBandNextYLB = [];
var isoBandNextOLB = [];

isoBandNextXRT[85] = isoBandNextXRB[85] = -1;
isoBandNextYRT[85] = isoBandNextYRB[85] = 0;
isoBandNextORT[85] = isoBandNextORB[85] = 1;
isoBandNextXLT[85] = isoBandNextXLB[85] = 1;
isoBandNextYLT[85] = isoBandNextYLB[85] = 0;
isoBandNextOLT[85] = isoBandNextOLB[85] = 1;

isoBandNextXTL[85] = isoBandNextXTR[85] = 0;
isoBandNextYTL[85] = isoBandNextYTR[85] = -1;
isoBandNextOTL[85] = isoBandNextOBL[85] = 0;
isoBandNextXBR[85] = isoBandNextXBL[85] = 0;
isoBandNextYBR[85] = isoBandNextYBL[85] = 1;
isoBandNextOTR[85] = isoBandNextOBR[85] = 1;


/* triangle cases */
isoBandNextXLB[1] = isoBandNextXLB[169] = 0;
isoBandNextYLB[1] = isoBandNextYLB[169] = -1;
isoBandNextOLB[1] = isoBandNextOLB[169] = 0;
isoBandNextXBL[1] = isoBandNextXBL[169] = -1;
isoBandNextYBL[1] = isoBandNextYBL[169] = 0;
isoBandNextOBL[1] = isoBandNextOBL[169] = 0;

isoBandNextXRB[4] = isoBandNextXRB[166] = 0;
isoBandNextYRB[4] = isoBandNextYRB[166] = -1;
isoBandNextORB[4] = isoBandNextORB[166] = 1;
isoBandNextXBR[4] = isoBandNextXBR[166] = 1;
isoBandNextYBR[4] = isoBandNextYBR[166] = 0;
isoBandNextOBR[4] = isoBandNextOBR[166] = 0;

isoBandNextXRT[16] = isoBandNextXRT[154] = 0;
isoBandNextYRT[16] = isoBandNextYRT[154] = 1;
isoBandNextORT[16] = isoBandNextORT[154] = 1;
isoBandNextXTR[16] = isoBandNextXTR[154] = 1;
isoBandNextYTR[16] = isoBandNextYTR[154] = 0;
isoBandNextOTR[16] = isoBandNextOTR[154] = 1;

isoBandNextXLT[64] = isoBandNextXLT[106] = 0;
isoBandNextYLT[64] = isoBandNextYLT[106] = 1;
isoBandNextOLT[64] = isoBandNextOLT[106] = 0;
isoBandNextXTL[64] = isoBandNextXTL[106] = -1;
isoBandNextYTL[64] = isoBandNextYTL[106] = 0;
isoBandNextOTL[64] = isoBandNextOTL[106] = 1;

/* single trapezoid cases */
isoBandNextXLT[2] = isoBandNextXLT[168] = 0;
isoBandNextYLT[2] = isoBandNextYLT[168] = -1;
isoBandNextOLT[2] = isoBandNextOLT[168] = 1;
isoBandNextXLB[2] = isoBandNextXLB[168] = 0;
isoBandNextYLB[2] = isoBandNextYLB[168] = -1;
isoBandNextOLB[2] = isoBandNextOLB[168] = 0;
isoBandNextXBL[2] = isoBandNextXBL[168] = -1;
isoBandNextYBL[2] = isoBandNextYBL[168] = 0;
isoBandNextOBL[2] = isoBandNextOBL[168] = 0;
isoBandNextXBR[2] = isoBandNextXBR[168] = -1;
isoBandNextYBR[2] = isoBandNextYBR[168] = 0;
isoBandNextOBR[2] = isoBandNextOBR[168] = 1;

isoBandNextXRT[8] = isoBandNextXRT[162] = 0;
isoBandNextYRT[8] = isoBandNextYRT[162] = -1;
isoBandNextORT[8] = isoBandNextORT[162] = 0;
isoBandNextXRB[8] = isoBandNextXRB[162] = 0;
isoBandNextYRB[8] = isoBandNextYRB[162] = -1;
isoBandNextORB[8] = isoBandNextORB[162] = 1;
isoBandNextXBL[8] = isoBandNextXBL[162] = 1;
isoBandNextYBL[8] = isoBandNextYBL[162] = 0;
isoBandNextOBL[8] = isoBandNextOBL[162] = 1;
isoBandNextXBR[8] = isoBandNextXBR[162] = 1;
isoBandNextYBR[8] = isoBandNextYBR[162] = 0;
isoBandNextOBR[8] = isoBandNextOBR[162] = 0;

isoBandNextXRT[32] = isoBandNextXRT[138] = 0;
isoBandNextYRT[32] = isoBandNextYRT[138] = 1;
isoBandNextORT[32] = isoBandNextORT[138] = 1;
isoBandNextXRB[32] = isoBandNextXRB[138] = 0;
isoBandNextYRB[32] = isoBandNextYRB[138] = 1;
isoBandNextORB[32] = isoBandNextORB[138] = 0;
isoBandNextXTL[32] = isoBandNextXTL[138] = 1;
isoBandNextYTL[32] = isoBandNextYTL[138] = 0;
isoBandNextOTL[32] = isoBandNextOTL[138] = 0;
isoBandNextXTR[32] = isoBandNextXTR[138] = 1;
isoBandNextYTR[32] = isoBandNextYTR[138] = 0;
isoBandNextOTR[32] = isoBandNextOTR[138] = 1;

isoBandNextXLB[128] = isoBandNextXLB[42] = 0;
isoBandNextYLB[128] = isoBandNextYLB[42] = 1;
isoBandNextOLB[128] = isoBandNextOLB[42] = 1;
isoBandNextXLT[128] = isoBandNextXLT[42] = 0;
isoBandNextYLT[128] = isoBandNextYLT[42] = 1;
isoBandNextOLT[128] = isoBandNextOLT[42] = 0;
isoBandNextXTL[128] = isoBandNextXTL[42] = -1;
isoBandNextYTL[128] = isoBandNextYTL[42] = 0;
isoBandNextOTL[128] = isoBandNextOTL[42] = 1;
isoBandNextXTR[128] = isoBandNextXTR[42] = -1;
isoBandNextYTR[128] = isoBandNextYTR[42] = 0;
isoBandNextOTR[128] = isoBandNextOTR[42] = 0;

/* single rectangle cases */
isoBandNextXRB[5] = isoBandNextXRB[165] = -1;
isoBandNextYRB[5] = isoBandNextYRB[165] = 0;
isoBandNextORB[5] = isoBandNextORB[165] = 0;
isoBandNextXLB[5] = isoBandNextXLB[165] = 1;
isoBandNextYLB[5] = isoBandNextYLB[165] = 0;
isoBandNextOLB[5] = isoBandNextOLB[165] = 0;

isoBandNextXBR[20] = isoBandNextXBR[150] = 0;
isoBandNextYBR[20] = isoBandNextYBR[150] = 1;
isoBandNextOBR[20] = isoBandNextOBR[150] = 1;
isoBandNextXTR[20] = isoBandNextXTR[150] = 0;
isoBandNextYTR[20] = isoBandNextYTR[150] = -1;
isoBandNextOTR[20] = isoBandNextOTR[150] = 1;

isoBandNextXRT[80] = isoBandNextXRT[90] = -1;
isoBandNextYRT[80] = isoBandNextYRT[90] = 0;
isoBandNextORT[80] = isoBandNextORT[90] = 1;
isoBandNextXLT[80] = isoBandNextXLT[90] = 1;
isoBandNextYLT[80] = isoBandNextYLT[90] = 0;
isoBandNextOLT[80] = isoBandNextOLT[90] = 1;

isoBandNextXBL[65] = isoBandNextXBL[105] = 0;
isoBandNextYBL[65] = isoBandNextYBL[105] = 1;
isoBandNextOBL[65] = isoBandNextOBL[105] = 0;
isoBandNextXTL[65] = isoBandNextXTL[105] = 0;
isoBandNextYTL[65] = isoBandNextYTL[105] = -1;
isoBandNextOTL[65] = isoBandNextOTL[105] = 0;

isoBandNextXRT[160] = isoBandNextXRT[10] = -1;
isoBandNextYRT[160] = isoBandNextYRT[10] = 0;
isoBandNextORT[160] = isoBandNextORT[10] = 1;
isoBandNextXRB[160] = isoBandNextXRB[10] = -1;
isoBandNextYRB[160] = isoBandNextYRB[10] = 0;
isoBandNextORB[160] = isoBandNextORB[10] = 0;
isoBandNextXLB[160] = isoBandNextXLB[10] = 1;
isoBandNextYLB[160] = isoBandNextYLB[10] = 0;
isoBandNextOLB[160] = isoBandNextOLB[10] = 0;
isoBandNextXLT[160] = isoBandNextXLT[10] = 1;
isoBandNextYLT[160] = isoBandNextYLT[10] = 0;
isoBandNextOLT[160] = isoBandNextOLT[10] = 1;

isoBandNextXBR[130] = isoBandNextXBR[40] = 0;
isoBandNextYBR[130] = isoBandNextYBR[40] = 1;
isoBandNextOBR[130] = isoBandNextOBR[40] = 1;
isoBandNextXBL[130] = isoBandNextXBL[40] = 0;
isoBandNextYBL[130] = isoBandNextYBL[40] = 1;
isoBandNextOBL[130] = isoBandNextOBL[40] = 0;
isoBandNextXTL[130] = isoBandNextXTL[40] = 0;
isoBandNextYTL[130] = isoBandNextYTL[40] = -1;
isoBandNextOTL[130] = isoBandNextOTL[40] = 0;
isoBandNextXTR[130] = isoBandNextXTR[40] = 0;
isoBandNextYTR[130] = isoBandNextYTR[40] = -1;
isoBandNextOTR[130] = isoBandNextOTR[40] = 1;

/* single hexagon cases */
isoBandNextXRB[37] = isoBandNextXRB[133] = 0;
isoBandNextYRB[37] = isoBandNextYRB[133] = 1;
isoBandNextORB[37] = isoBandNextORB[133] = 1;
isoBandNextXLB[37] = isoBandNextXLB[133] = 0;
isoBandNextYLB[37] = isoBandNextYLB[133] = 1;
isoBandNextOLB[37] = isoBandNextOLB[133] = 0;
isoBandNextXTL[37] = isoBandNextXTL[133] = -1;
isoBandNextYTL[37] = isoBandNextYTL[133] = 0;
isoBandNextOTL[37] = isoBandNextOTL[133] = 0;
isoBandNextXTR[37] = isoBandNextXTR[133] = 1;
isoBandNextYTR[37] = isoBandNextYTR[133] = 0;
isoBandNextOTR[37] = isoBandNextOTR[133] = 0;

isoBandNextXBR[148] = isoBandNextXBR[22] = -1;
isoBandNextYBR[148] = isoBandNextYBR[22] = 0;
isoBandNextOBR[148] = isoBandNextOBR[22] = 0;
isoBandNextXLB[148] = isoBandNextXLB[22] = 0;
isoBandNextYLB[148] = isoBandNextYLB[22] = -1;
isoBandNextOLB[148] = isoBandNextOLB[22] = 1;
isoBandNextXLT[148] = isoBandNextXLT[22] = 0;
isoBandNextYLT[148] = isoBandNextYLT[22] = 1;
isoBandNextOLT[148] = isoBandNextOLT[22] = 1;
isoBandNextXTR[148] = isoBandNextXTR[22] = -1;
isoBandNextYTR[148] = isoBandNextYTR[22] = 0;
isoBandNextOTR[148] = isoBandNextOTR[22] = 1;

isoBandNextXRT[82] = isoBandNextXRT[88] = 0;
isoBandNextYRT[82] = isoBandNextYRT[88] = -1;
isoBandNextORT[82] = isoBandNextORT[88] = 1;
isoBandNextXBR[82] = isoBandNextXBR[88] = 1;
isoBandNextYBR[82] = isoBandNextYBR[88] = 0;
isoBandNextOBR[82] = isoBandNextOBR[88] = 1;
isoBandNextXBL[82] = isoBandNextXBL[88] = -1;
isoBandNextYBL[82] = isoBandNextYBL[88] = 0;
isoBandNextOBL[82] = isoBandNextOBL[88] = 1;
isoBandNextXLT[82] = isoBandNextXLT[88] = 0;
isoBandNextYLT[82] = isoBandNextYLT[88] = -1;
isoBandNextOLT[82] = isoBandNextOLT[88] = 0;

isoBandNextXRT[73] = isoBandNextXRT[97] = 0;
isoBandNextYRT[73] = isoBandNextYRT[97] = 1;
isoBandNextORT[73] = isoBandNextORT[97] = 0;
isoBandNextXRB[73] = isoBandNextXRB[97] = 0;
isoBandNextYRB[73] = isoBandNextYRB[97] = -1;
isoBandNextORB[73] = isoBandNextORB[97] = 0;
isoBandNextXBL[73] = isoBandNextXBL[97] = 1;
isoBandNextYBL[73] = isoBandNextYBL[97] = 0;
isoBandNextOBL[73] = isoBandNextOBL[97] = 0;
isoBandNextXTL[73] = isoBandNextXTL[97] = 1;
isoBandNextYTL[73] = isoBandNextYTL[97] = 0;
isoBandNextOTL[73] = isoBandNextOTL[97] = 1;

isoBandNextXRT[145] = isoBandNextXRT[25] = 0;
isoBandNextYRT[145] = isoBandNextYRT[25] = -1;
isoBandNextORT[145] = isoBandNextORT[25] = 0;
isoBandNextXBL[145] = isoBandNextXBL[25] = 1;
isoBandNextYBL[145] = isoBandNextYBL[25] = 0;
isoBandNextOBL[145] = isoBandNextOBL[25] = 1;
isoBandNextXLB[145] = isoBandNextXLB[25] = 0;
isoBandNextYLB[145] = isoBandNextYLB[25] = 1;
isoBandNextOLB[145] = isoBandNextOLB[25] = 1;
isoBandNextXTR[145] = isoBandNextXTR[25] = -1;
isoBandNextYTR[145] = isoBandNextYTR[25] = 0;
isoBandNextOTR[145] = isoBandNextOTR[25] = 0;

isoBandNextXRB[70] = isoBandNextXRB[100] = 0;
isoBandNextYRB[70] = isoBandNextYRB[100] = 1;
isoBandNextORB[70] = isoBandNextORB[100] = 0;
isoBandNextXBR[70] = isoBandNextXBR[100] = -1;
isoBandNextYBR[70] = isoBandNextYBR[100] = 0;
isoBandNextOBR[70] = isoBandNextOBR[100] = 1;
isoBandNextXLT[70] = isoBandNextXLT[100] = 0;
isoBandNextYLT[70] = isoBandNextYLT[100] = -1;
isoBandNextOLT[70] = isoBandNextOLT[100] = 1;
isoBandNextXTL[70] = isoBandNextXTL[100] = 1;
isoBandNextYTL[70] = isoBandNextYTL[100] = 0;
isoBandNextOTL[70] = isoBandNextOTL[100] = 0;

/* single pentagon cases */
isoBandNextXRB[101] = isoBandNextXRB[69] = 0;
isoBandNextYRB[101] = isoBandNextYRB[69] = 1;
isoBandNextORB[101] = isoBandNextORB[69] = 0;
isoBandNextXTL[101] = isoBandNextXTL[69] = 1;
isoBandNextYTL[101] = isoBandNextYTL[69] = 0;
isoBandNextOTL[101] = isoBandNextOTL[69] = 0;

isoBandNextXLB[149] = isoBandNextXLB[21] = 0;
isoBandNextYLB[149] = isoBandNextYLB[21] = 1;
isoBandNextOLB[149] = isoBandNextOLB[21] = 1;
isoBandNextXTR[149] = isoBandNextXTR[21] = -1;
isoBandNextYTR[149] = isoBandNextYTR[21] = 0;
isoBandNextOTR[149] = isoBandNextOTR[21] = 0;

isoBandNextXBR[86] = isoBandNextXBR[84] = -1;
isoBandNextYBR[86] = isoBandNextYBR[84] = 0;
isoBandNextOBR[86] = isoBandNextOBR[84] = 1;
isoBandNextXLT[86] = isoBandNextXLT[84] = 0;
isoBandNextYLT[86] = isoBandNextYLT[84] = -1;
isoBandNextOLT[86] = isoBandNextOLT[84] = 1;

isoBandNextXRT[89] = isoBandNextXRT[81] = 0;
isoBandNextYRT[89] = isoBandNextYRT[81] = -1;
isoBandNextORT[89] = isoBandNextORT[81] = 0;
isoBandNextXBL[89] = isoBandNextXBL[81] = 1;
isoBandNextYBL[89] = isoBandNextYBL[81] = 0;
isoBandNextOBL[89] = isoBandNextOBL[81] = 1;

isoBandNextXRT[96] = isoBandNextXRT[74] = 0;
isoBandNextYRT[96] = isoBandNextYRT[74] = 1;
isoBandNextORT[96] = isoBandNextORT[74] = 0;
isoBandNextXRB[96] = isoBandNextXRB[74] = -1;
isoBandNextYRB[96] = isoBandNextYRB[74] = 0;
isoBandNextORB[96] = isoBandNextORB[74] = 1;
isoBandNextXLT[96] = isoBandNextXLT[74] = 1;
isoBandNextYLT[96] = isoBandNextYLT[74] = 0;
isoBandNextOLT[96] = isoBandNextOLT[74] = 0;
isoBandNextXTL[96] = isoBandNextXTL[74] = 1;
isoBandNextYTL[96] = isoBandNextYTL[74] = 0;
isoBandNextOTL[96] = isoBandNextOTL[74] = 1;

isoBandNextXRT[24] = isoBandNextXRT[146] = 0;
isoBandNextYRT[24] = isoBandNextYRT[146] = -1;
isoBandNextORT[24] = isoBandNextORT[146] = 1;
isoBandNextXBR[24] = isoBandNextXBR[146] = 1;
isoBandNextYBR[24] = isoBandNextYBR[146] = 0;
isoBandNextOBR[24] = isoBandNextOBR[146] = 1;
isoBandNextXBL[24] = isoBandNextXBL[146] = 0;
isoBandNextYBL[24] = isoBandNextYBL[146] = 1;
isoBandNextOBL[24] = isoBandNextOBL[146] = 1;
isoBandNextXTR[24] = isoBandNextXTR[146] = 0;
isoBandNextYTR[24] = isoBandNextYTR[146] = -1;
isoBandNextOTR[24] = isoBandNextOTR[146] = 0;

isoBandNextXRB[6] = isoBandNextXRB[164] = -1;
isoBandNextYRB[6] = isoBandNextYRB[164] = 0;
isoBandNextORB[6] = isoBandNextORB[164] = 1;
isoBandNextXBR[6] = isoBandNextXBR[164] = -1;
isoBandNextYBR[6] = isoBandNextYBR[164] = 0;
isoBandNextOBR[6] = isoBandNextOBR[164] = 0;
isoBandNextXLB[6] = isoBandNextXLB[164] = 0;
isoBandNextYLB[6] = isoBandNextYLB[164] = -1;
isoBandNextOLB[6] = isoBandNextOLB[164] = 1;
isoBandNextXLT[6] = isoBandNextXLT[164] = 1;
isoBandNextYLT[6] = isoBandNextYLT[164] = 0;
isoBandNextOLT[6] = isoBandNextOLT[164] = 0;

isoBandNextXBL[129] = isoBandNextXBL[41] = 0;
isoBandNextYBL[129] = isoBandNextYBL[41] = 1;
isoBandNextOBL[129] = isoBandNextOBL[41] = 1;
isoBandNextXLB[129] = isoBandNextXLB[41] = 0;
isoBandNextYLB[129] = isoBandNextYLB[41] = 1;
isoBandNextOLB[129] = isoBandNextOLB[41] = 0;
isoBandNextXTL[129] = isoBandNextXTL[41] = -1;
isoBandNextYTL[129] = isoBandNextYTL[41] = 0;
isoBandNextOTL[129] = isoBandNextOTL[41] = 0;
isoBandNextXTR[129] = isoBandNextXTR[41] = 0;
isoBandNextYTR[129] = isoBandNextYTR[41] = -1;
isoBandNextOTR[129] = isoBandNextOTR[41] = 0;

isoBandNextXBR[66] = isoBandNextXBR[104] = 0;
isoBandNextYBR[66] = isoBandNextYBR[104] = 1;
isoBandNextOBR[66] = isoBandNextOBR[104] = 0;
isoBandNextXBL[66] = isoBandNextXBL[104] = -1;
isoBandNextYBL[66] = isoBandNextYBL[104] = 0;
isoBandNextOBL[66] = isoBandNextOBL[104] = 1;
isoBandNextXLT[66] = isoBandNextXLT[104] = 0;
isoBandNextYLT[66] = isoBandNextYLT[104] = -1;
isoBandNextOLT[66] = isoBandNextOLT[104] = 0;
isoBandNextXTL[66] = isoBandNextXTL[104] = 0;
isoBandNextYTL[66] = isoBandNextYTL[104] = -1;
isoBandNextOTL[66] = isoBandNextOTL[104] = 1;

isoBandNextXRT[144] = isoBandNextXRT[26] = -1;
isoBandNextYRT[144] = isoBandNextYRT[26] = 0;
isoBandNextORT[144] = isoBandNextORT[26] = 0;
isoBandNextXLB[144] = isoBandNextXLB[26] = 1;
isoBandNextYLB[144] = isoBandNextYLB[26] = 0;
isoBandNextOLB[144] = isoBandNextOLB[26] = 1;
isoBandNextXLT[144] = isoBandNextXLT[26] = 0;
isoBandNextYLT[144] = isoBandNextYLT[26] = 1;
isoBandNextOLT[144] = isoBandNextOLT[26] = 1;
isoBandNextXTR[144] = isoBandNextXTR[26] = -1;
isoBandNextYTR[144] = isoBandNextYTR[26] = 0;
isoBandNextOTR[144] = isoBandNextOTR[26] = 1;

isoBandNextXRB[36] = isoBandNextXRB[134] = 0;
isoBandNextYRB[36] = isoBandNextYRB[134] = 1;
isoBandNextORB[36] = isoBandNextORB[134] = 1;
isoBandNextXBR[36] = isoBandNextXBR[134] = 0;
isoBandNextYBR[36] = isoBandNextYBR[134] = 1;
isoBandNextOBR[36] = isoBandNextOBR[134] = 0;
isoBandNextXTL[36] = isoBandNextXTL[134] = 0;
isoBandNextYTL[36] = isoBandNextYTL[134] = -1;
isoBandNextOTL[36] = isoBandNextOTL[134] = 1;
isoBandNextXTR[36] = isoBandNextXTR[134] = 1;
isoBandNextYTR[36] = isoBandNextYTR[134] = 0;
isoBandNextOTR[36] = isoBandNextOTR[134] = 0;

isoBandNextXRT[9] = isoBandNextXRT[161] = -1;
isoBandNextYRT[9] = isoBandNextYRT[161] = 0;
isoBandNextORT[9] = isoBandNextORT[161] = 0;
isoBandNextXRB[9] = isoBandNextXRB[161] = 0;
isoBandNextYRB[9] = isoBandNextYRB[161] = -1;
isoBandNextORB[9] = isoBandNextORB[161] = 0;
isoBandNextXBL[9] = isoBandNextXBL[161] = 1;
isoBandNextYBL[9] = isoBandNextYBL[161] = 0;
isoBandNextOBL[9] = isoBandNextOBL[161] = 0;
isoBandNextXLB[9] = isoBandNextXLB[161] = 1;
isoBandNextYLB[9] = isoBandNextYLB[161] = 0;
isoBandNextOLB[9] = isoBandNextOLB[161] = 1;

/* 8-sided cases */
isoBandNextXRT[136] = 0;
isoBandNextYRT[136] = 1;
isoBandNextORT[136] = 1;
isoBandNextXRB[136] = 0;
isoBandNextYRB[136] = 1;
isoBandNextORB[136] = 0;
isoBandNextXBR[136] = -1;
isoBandNextYBR[136] = 0;
isoBandNextOBR[136] = 1;
isoBandNextXBL[136] = -1;
isoBandNextYBL[136] = 0;
isoBandNextOBL[136] = 0;
isoBandNextXLB[136] = 0;
isoBandNextYLB[136] = -1;
isoBandNextOLB[136] = 0;
isoBandNextXLT[136] = 0;
isoBandNextYLT[136] = -1;
isoBandNextOLT[136] = 1;
isoBandNextXTL[136] = 1;
isoBandNextYTL[136] = 0;
isoBandNextOTL[136] = 0;
isoBandNextXTR[136] = 1;
isoBandNextYTR[136] = 0;
isoBandNextOTR[136] = 1;

isoBandNextXRT[34] = 0;
isoBandNextYRT[34] = -1;
isoBandNextORT[34] = 0;
isoBandNextXRB[34] = 0;
isoBandNextYRB[34] = -1;
isoBandNextORB[34] = 1;
isoBandNextXBR[34] = 1;
isoBandNextYBR[34] = 0;
isoBandNextOBR[34] = 0;
isoBandNextXBL[34] = 1;
isoBandNextYBL[34] = 0;
isoBandNextOBL[34] = 1;
isoBandNextXLB[34] = 0;
isoBandNextYLB[34] = 1;
isoBandNextOLB[34] = 1;
isoBandNextXLT[34] = 0;
isoBandNextYLT[34] = 1;
isoBandNextOLT[34] = 0;
isoBandNextXTL[34] = -1;
isoBandNextYTL[34] = 0;
isoBandNextOTL[34] = 1;
isoBandNextXTR[34] = -1;
isoBandNextYTR[34] = 0;
isoBandNextOTR[34] = 0;

isoBandNextXRT[35] = 0;
isoBandNextYRT[35] = 1;
isoBandNextORT[35] = 1;
isoBandNextXRB[35] = 0;
isoBandNextYRB[35] = -1;
isoBandNextORB[35] = 1;
isoBandNextXBR[35] = 1;
isoBandNextYBR[35] = 0;
isoBandNextOBR[35] = 0;
isoBandNextXBL[35] = -1;
isoBandNextYBL[35] = 0;
isoBandNextOBL[35] = 0;
isoBandNextXLB[35] = 0;
isoBandNextYLB[35] = -1;
isoBandNextOLB[35] = 0;
isoBandNextXLT[35] = 0;
isoBandNextYLT[35] = 1;
isoBandNextOLT[35] = 0;
isoBandNextXTL[35] = -1;
isoBandNextYTL[35] = 0;
isoBandNextOTL[35] = 1;
isoBandNextXTR[35] = 1;
isoBandNextYTR[35] = 0;
isoBandNextOTR[35] = 1;

/* 6-sided cases */
isoBandNextXRT[153] = 0;
isoBandNextYRT[153] = 1;
isoBandNextORT[153] = 1;
isoBandNextXBL[153] = -1;
isoBandNextYBL[153] = 0;
isoBandNextOBL[153] = 0;
isoBandNextXLB[153] = 0;
isoBandNextYLB[153] = -1;
isoBandNextOLB[153] = 0;
isoBandNextXTR[153] = 1;
isoBandNextYTR[153] = 0;
isoBandNextOTR[153] = 1;

isoBandNextXRB[102] = 0;
isoBandNextYRB[102] = -1;
isoBandNextORB[102] = 1;
isoBandNextXBR[102] = 1;
isoBandNextYBR[102] = 0;
isoBandNextOBR[102] = 0;
isoBandNextXLT[102] = 0;
isoBandNextYLT[102] = 1;
isoBandNextOLT[102] = 0;
isoBandNextXTL[102] = -1;
isoBandNextYTL[102] = 0;
isoBandNextOTL[102] = 1;

isoBandNextXRT[155] = 0;
isoBandNextYRT[155] = -1;
isoBandNextORT[155] = 0;
isoBandNextXBL[155] = 1;
isoBandNextYBL[155] = 0;
isoBandNextOBL[155] = 1;
isoBandNextXLB[155] = 0;
isoBandNextYLB[155] = 1;
isoBandNextOLB[155] = 1;
isoBandNextXTR[155] = -1;
isoBandNextYTR[155] = 0;
isoBandNextOTR[155] = 0;

isoBandNextXRB[103] = 0;
isoBandNextYRB[103] = 1;
isoBandNextORB[103] = 0;
isoBandNextXBR[103] = -1;
isoBandNextYBR[103] = 0;
isoBandNextOBR[103] = 1;
isoBandNextXLT[103] = 0;
isoBandNextYLT[103] = -1;
isoBandNextOLT[103] = 1;
isoBandNextXTL[103] = 1;
isoBandNextYTL[103] = 0;
isoBandNextOTL[103] = 0;

/* 7-sided cases */
isoBandNextXRT[152] = 0;
isoBandNextYRT[152] = 1;
isoBandNextORT[152] = 1;
isoBandNextXBR[152] = -1;
isoBandNextYBR[152] = 0;
isoBandNextOBR[152] = 1;
isoBandNextXBL[152] = -1;
isoBandNextYBL[152] = 0;
isoBandNextOBL[152] = 0;
isoBandNextXLB[152] = 0;
isoBandNextYLB[152] = -1;
isoBandNextOLB[152] = 0;
isoBandNextXLT[152] = 0;
isoBandNextYLT[152] = -1;
isoBandNextOLT[152] = 1;
isoBandNextXTR[152] = 1;
isoBandNextYTR[152] = 0;
isoBandNextOTR[152] = 1;

isoBandNextXRT[156] = 0;
isoBandNextYRT[156] = -1;
isoBandNextORT[156] = 1;
isoBandNextXBR[156] = 1;
isoBandNextYBR[156] = 0;
isoBandNextOBR[156] = 1;
isoBandNextXBL[156] = -1;
isoBandNextYBL[156] = 0;
isoBandNextOBL[156] = 0;
isoBandNextXLB[156] = 0;
isoBandNextYLB[156] = -1;
isoBandNextOLB[156] = 0;
isoBandNextXLT[156] = 0;
isoBandNextYLT[156] = 1;
isoBandNextOLT[156] = 1;
isoBandNextXTR[156] = -1;
isoBandNextYTR[156] = 0;
isoBandNextOTR[156] = 1;

isoBandNextXRT[137] = 0;
isoBandNextYRT[137] = 1;
isoBandNextORT[137] = 1;
isoBandNextXRB[137] = 0;
isoBandNextYRB[137] = 1;
isoBandNextORB[137] = 0;
isoBandNextXBL[137] = -1;
isoBandNextYBL[137] = 0;
isoBandNextOBL[137] = 0;
isoBandNextXLB[137] = 0;
isoBandNextYLB[137] = -1;
isoBandNextOLB[137] = 0;
isoBandNextXTL[137] = 1;
isoBandNextYTL[137] = 0;
isoBandNextOTL[137] = 0;
isoBandNextXTR[137] = 1;
isoBandNextYTR[137] = 0;
isoBandNextOTR[137] = 1;

isoBandNextXRT[139] = 0;
isoBandNextYRT[139] = 1;
isoBandNextORT[139] = 1;
isoBandNextXRB[139] = 0;
isoBandNextYRB[139] = -1;
isoBandNextORB[139] = 0;
isoBandNextXBL[139] = 1;
isoBandNextYBL[139] = 0;
isoBandNextOBL[139] = 0;
isoBandNextXLB[139] = 0;
isoBandNextYLB[139] = 1;
isoBandNextOLB[139] = 0;
isoBandNextXTL[139] = -1;
isoBandNextYTL[139] = 0;
isoBandNextOTL[139] = 0;
isoBandNextXTR[139] = 1;
isoBandNextYTR[139] = 0;
isoBandNextOTR[139] = 1;

isoBandNextXRT[98] = 0;
isoBandNextYRT[98] = -1;
isoBandNextORT[98] = 0;
isoBandNextXRB[98] = 0;
isoBandNextYRB[98] = -1;
isoBandNextORB[98] = 1;
isoBandNextXBR[98] = 1;
isoBandNextYBR[98] = 0;
isoBandNextOBR[98] = 0;
isoBandNextXBL[98] = 1;
isoBandNextYBL[98] = 0;
isoBandNextOBL[98] = 1;
isoBandNextXLT[98] = 0;
isoBandNextYLT[98] = 1;
isoBandNextOLT[98] = 0;
isoBandNextXTL[98] = -1;
isoBandNextYTL[98] = 0;
isoBandNextOTL[98] = 1;

isoBandNextXRT[99] = 0;
isoBandNextYRT[99] = 1;
isoBandNextORT[99] = 0;
isoBandNextXRB[99] = 0;
isoBandNextYRB[99] = -1;
isoBandNextORB[99] = 1;
isoBandNextXBR[99] = 1;
isoBandNextYBR[99] = 0;
isoBandNextOBR[99] = 0;
isoBandNextXBL[99] = -1;
isoBandNextYBL[99] = 0;
isoBandNextOBL[99] = 1;
isoBandNextXLT[99] = 0;
isoBandNextYLT[99] = -1;
isoBandNextOLT[99] = 0;
isoBandNextXTL[99] = 1;
isoBandNextYTL[99] = 0;
isoBandNextOTL[99] = 1;

isoBandNextXRB[38] = 0;
isoBandNextYRB[38] = -1;
isoBandNextORB[38] = 1;
isoBandNextXBR[38] = 1;
isoBandNextYBR[38] = 0;
isoBandNextOBR[38] = 0;
isoBandNextXLB[38] = 0;
isoBandNextYLB[38] = 1;
isoBandNextOLB[38] = 1;
isoBandNextXLT[38] = 0;
isoBandNextYLT[38] = 1;
isoBandNextOLT[38] = 0;
isoBandNextXTL[38] = -1;
isoBandNextYTL[38] = 0;
isoBandNextOTL[38] = 1;
isoBandNextXTR[38] = -1;
isoBandNextYTR[38] = 0;
isoBandNextOTR[38] = 0;

isoBandNextXRB[39] = 0;
isoBandNextYRB[39] = 1;
isoBandNextORB[39] = 1;
isoBandNextXBR[39] = -1;
isoBandNextYBR[39] = 0;
isoBandNextOBR[39] = 0;
isoBandNextXLB[39] = 0;
isoBandNextYLB[39] = -1;
isoBandNextOLB[39] = 1;
isoBandNextXLT[39] = 0;
isoBandNextYLT[39] = 1;
isoBandNextOLT[39] = 0;
isoBandNextXTL[39] = -1;
isoBandNextYTL[39] = 0;
isoBandNextOTL[39] = 1;
isoBandNextXTR[39] = 1;
isoBandNextYTR[39] = 0;
isoBandNextOTR[39] = 0;


/*
  Define helper functions for the polygon_table
  */

/* triangle cases */
var p00 = function (cell) {
    return [[cell.bottomleft, 0], [0, 0], [0, cell.leftbottom]];
};
var p01 = function (cell) {
    return [[1, cell.rightbottom], [1, 0], [cell.bottomright, 0]];
};
var p02 = function (cell) {
    return [[cell.topright, 1], [1, 1], [1, cell.righttop]];
};
var p03 = function (cell) {
    return [[0, cell.lefttop], [0, 1], [cell.topleft, 1]];
};
/* trapezoid cases */
var p04 = function (cell) {
    return [[cell.bottomright, 0], [cell.bottomleft, 0], [0, cell.leftbottom], [0, cell.lefttop]];
};
var p05 = function (cell) {
    return [[cell.bottomright, 0], [cell.bottomleft, 0], [1, cell.righttop], [1, cell.rightbottom]];
};
var p06 = function (cell) {
    return [[1, cell.righttop], [1, cell.rightbottom], [cell.topleft, 1], [cell.topright, 1]];
};
var p07 = function (cell) {
    return [[0, cell.leftbottom], [0, cell.lefttop], [cell.topleft, 1], [cell.topright, 1]];
};
/* rectangle cases */
var p08 = function (cell) {
    return [[0, 0], [0, cell.leftbottom], [1, cell.rightbottom], [1, 0]];
};
var p09 = function (cell) {
    return [[1, 0], [cell.bottomright, 0], [cell.topright, 1], [1, 1]];
};
var p10 = function (cell) {
    return [[1, 1], [1, cell.righttop], [0, cell.lefttop], [0, 1]];
};
var p11 = function (cell) {
    return [[cell.bottomleft, 0], [0, 0], [0, 1], [cell.topleft, 1]];
};
var p12 = function (cell) {
    return [[1, cell.righttop], [1, cell.rightbottom], [0, cell.leftbottom], [0, cell.lefttop]];
};
var p13 = function (cell) {
    return [[cell.topleft, 1], [cell.topright, 1], [cell.bottomright, 0], [cell.bottomleft, 0]];
};
/* square case */
var p14 = function () {
    return [[0, 0], [0, 1], [1, 1], [1, 0]];
};
/* pentagon cases */
var p15 = function (cell) {
    return [[1, cell.rightbottom], [1, 0], [0, 0], [0, 1], [cell.topleft, 1]];
};
/* 1211 || 1011 */
var p16 = function (cell) {
    return [[cell.topright, 1], [1, 1], [1, 0], [0, 0], [0, cell.leftbottom]];
};
/* 2111 || 0111 */
var p17 = function (cell) {
    return [[1, 0], [cell.bottomright, 0], [0, cell.lefttop], [0, 1], [1, 1]];
};
/* 1112 || 1110 */
var p18 = function (cell) {
    return [[1, 1], [1, cell.righttop], [cell.bottomleft, 0], [0, 0], [0, 1]];
};
/* 1121 || 1101 */
var p19 = function (cell) {
    return [[1, cell.righttop], [1, cell.rightbottom], [0, cell.lefttop], [0, 1], [cell.topleft, 1]];
};
/* 1200 || 1022 */
var p20 = function (cell) {
    return [[1, 1], [1, cell.righttop], [cell.bottomright, 0], [cell.bottomleft, 0], [cell.topright, 1]];
};
/* 0120 || 2102 */
var p21 = function (cell) {
    return [[1, cell.rightbottom], [1, 0], [cell.bottomright, 0], [0, cell.leftbottom], [0, cell.lefttop]];
};
/* 0012 || 2210 */
var p22 = function (cell) {
    return [[cell.topright, 1], [cell.bottomleft, 0], [0, 0], [0, cell.leftbottom], [cell.topleft, 1]];
};
/* 2001 || 0221 */
var p23 = function (cell) {
    return [[cell.bottomright, 0], [cell.bottomleft, 0], [0, cell.lefttop], [0, 1], [cell.topleft, 1]];
};
/* 1002 || 1220 */
var p24 = function (cell) {
    return [[1, 1], [1, cell.righttop], [0, cell.leftbottom], [0, cell.lefttop], [cell.topright, 1]];
};
/* 2100 || 0122 */
var p25 = function (cell) {
    return [[1, cell.rightbottom], [1, 0], [cell.bottomright, 0], [cell.topleft, 1], [cell.topright, 1]];
};
/* 0210 || 2012 */
var p26 = function (cell) {
    return [[1, cell.righttop], [1, cell.rightbottom], [cell.bottomleft, 0], [0, 0], [0, cell.leftbottom]];
};
/* 0021 || 2201 */
/*hexagon cases */
var p27 = function (cell) {
    return [[1, cell.rightbottom], [1, 0], [0, 0], [0, cell.leftbottom], [cell.topleft, 1], [cell.topright, 1]];
};
/* 0211 || 2011 */
var p28 = function (cell) {
    return [[1, 1], [1, 0], [cell.bottomright, 0], [0, cell.leftbottom], [0, cell.lefttop], [cell.topright, 1]];
};
/* 2110 || 0112 */
var p29 = function (cell) {
    return [[1, 1], [1, cell.righttop], [cell.bottomright, 0], [cell.bottomleft, 0], [0, cell.lefttop], [0, 1]];
};
/* 1102 || 1120 */
var p30 = function (cell) {
    return [[1, cell.righttop], [1, cell.rightbottom], [cell.bottomleft, 0], [0, 0], [0, 1], [cell.topleft, 1]];
};
/* 1021 || 1201 */
var p31 = function (cell) {
    return [[1, 1], [1, cell.righttop], [cell.bottomleft, 0], [0, 0], [0, cell.leftbottom], [cell.topright, 1]];
};
/* 2101 || 0121 */
var p32 = function (cell) {
    return [[1, cell.rightbottom], [1, 0], [cell.bottomright, 0], [0, cell.lefttop], [0, 1], [cell.topleft, 1]];
};
/* 1012 || 1210 */
/* 8-sided cases */
var p33 = function (cell) {
    return [[1, cell.righttop], [1, cell.rightbottom], [cell.bottomright, 0], [cell.bottomleft, 0], [0, cell.leftbottom], [0, cell.lefttop], [cell.topleft, 1], [cell.topright, 1]];
};
/* flipped == 1 state for 0202 and 2020 */
/* 6-sided cases */
var p34 = function (cell) {
    return [[1, 1], [1, cell.righttop], [cell.bottomleft, 0], [0, 0], [0, cell.leftbottom], [cell.topright, 1]];
};
/* 0101 with flipped == 1 || 2121 with flipped == 1 */
var p35 = function (cell) {
    return [[1, cell.rightbottom], [1, 0], [cell.bottomright, 0], [0, cell.lefttop], [0, 1], [cell.topleft, 1]];
};
/* 1010 with flipped == 1 || 1212 with flipped == 1 */
/* 7-sided cases */
var p36 = function (cell) {
    return [[1, 1], [1, cell.righttop], [cell.bottomright, 0], [cell.bottomleft, 0], [0, cell.leftbottom], [0, cell.lefttop], [cell.topright, 1]];
};
/* 2120 with flipped == 1 || 0102 with flipped == 1 */
var p37 = function (cell) {
    return [[1, cell.righttop], [1, cell.rightbottom], [cell.bottomleft, 0], [0, 0], [0, cell.leftbottom], [cell.topleft, 1], [cell.topright, 1]];
};
/* 2021 with flipped == 1 || 0201 with flipped == 1 */
var p38 = function (cell) {
    return [[1, cell.righttop], [1, cell.rightbottom], [cell.bottomright, 0], [cell.bottomleft, 0], [0, cell.lefttop], [0, 1], [cell.topleft, 1]];
};
/* 1202 with flipped == 1 || 1020 with flipped == 1 */
var p39 = function (cell) {
    return [[1, cell.rightbottom], [1, 0], [cell.bottomright, 0], [0, cell.leftbottom], [0, cell.lefttop], [cell.topleft, 1], [cell.topright, 1]];
};
/* 0212 with flipped == 1 || 2010 with flipped == 1 */



/*
  The lookup tables for edge number given the polygon
  is entered at a specific location
*/

var isoBandEdgeRT = [];
var isoBandEdgeRB = [];
var isoBandEdgeBR = [];
var isoBandEdgeBL = [];
var isoBandEdgeLB = [];
var isoBandEdgeLT = [];
var isoBandEdgeTL = [];
var isoBandEdgeTR = [];

/* triangle cases */
isoBandEdgeBL[1]    = isoBandEdgeLB[1]    = 18;
isoBandEdgeBL[169]  = isoBandEdgeLB[169]  = 18;
isoBandEdgeBR[4]    = isoBandEdgeRB[4]    = 12;
isoBandEdgeBR[166]  = isoBandEdgeRB[166]  = 12;
isoBandEdgeRT[16]   = isoBandEdgeTR[16]   = 4;
isoBandEdgeRT[154]  = isoBandEdgeTR[154]  = 4;
isoBandEdgeLT[64]   = isoBandEdgeTL[64]   = 22;
isoBandEdgeLT[106]  = isoBandEdgeTL[106]  = 22;

/* trapezoid cases */
isoBandEdgeBR[2]    = isoBandEdgeLT[2]    = 17;
isoBandEdgeBL[2]    = isoBandEdgeLB[2]    = 18;
isoBandEdgeBR[168]  = isoBandEdgeLT[168]  = 17;
isoBandEdgeBL[168]  = isoBandEdgeLB[168]  = 18;
isoBandEdgeRT[8]    = isoBandEdgeBL[8]    = 9;
isoBandEdgeRB[8]    = isoBandEdgeBR[8]    = 12;
isoBandEdgeRT[162]  = isoBandEdgeBL[162]  = 9;
isoBandEdgeRB[162]  = isoBandEdgeBR[162]  = 12;
isoBandEdgeRT[32]   = isoBandEdgeTR[32]   = 4;
isoBandEdgeRB[32]   = isoBandEdgeTL[32]   = 1;
isoBandEdgeRT[138]  = isoBandEdgeTR[138]  = 4;
isoBandEdgeRB[138]  = isoBandEdgeTL[138]  = 1;
isoBandEdgeLB[128]  = isoBandEdgeTR[128]  = 21;
isoBandEdgeLT[128]  = isoBandEdgeTL[128]  = 22;
isoBandEdgeLB[42]   = isoBandEdgeTR[42]   = 21;
isoBandEdgeLT[42]   = isoBandEdgeTL[42]   = 22;

/* rectangle cases */
isoBandEdgeRB[5] = isoBandEdgeLB[5] = 14;
isoBandEdgeRB[165] = isoBandEdgeLB[165] = 14;
isoBandEdgeBR[20] = isoBandEdgeTR[20] = 6;
isoBandEdgeBR[150] = isoBandEdgeTR[150] = 6;
isoBandEdgeRT[80] = isoBandEdgeLT[80] = 11;
isoBandEdgeRT[90] = isoBandEdgeLT[90] = 11;
isoBandEdgeBL[65] = isoBandEdgeTL[65] = 3;
isoBandEdgeBL[105] = isoBandEdgeTL[105] = 3;
isoBandEdgeRT[160] = isoBandEdgeLT[160] = 11;
isoBandEdgeRB[160] = isoBandEdgeLB[160] = 14;
isoBandEdgeRT[10] = isoBandEdgeLT[10] = 11;
isoBandEdgeRB[10] = isoBandEdgeLB[10] = 14;
isoBandEdgeBR[130] = isoBandEdgeTR[130] = 6;
isoBandEdgeBL[130] = isoBandEdgeTL[130] = 3;
isoBandEdgeBR[40] = isoBandEdgeTR[40] = 6;
isoBandEdgeBL[40] = isoBandEdgeTL[40] = 3;

/* pentagon cases */
isoBandEdgeRB[101] = isoBandEdgeTL[101] = 1;
isoBandEdgeRB[69] = isoBandEdgeTL[69] = 1;
isoBandEdgeLB[149] = isoBandEdgeTR[149] = 21;
isoBandEdgeLB[21] = isoBandEdgeTR[21] = 21;
isoBandEdgeBR[86] = isoBandEdgeLT[86] = 17;
isoBandEdgeBR[84] = isoBandEdgeLT[84] = 17;
isoBandEdgeRT[89] = isoBandEdgeBL[89] = 9;
isoBandEdgeRT[81] = isoBandEdgeBL[81] = 9;
isoBandEdgeRT[96] = isoBandEdgeTL[96] = 0;
isoBandEdgeRB[96] = isoBandEdgeLT[96] = 15;
isoBandEdgeRT[74] = isoBandEdgeTL[74] = 0;
isoBandEdgeRB[74] = isoBandEdgeLT[74] = 15;
isoBandEdgeRT[24] = isoBandEdgeBR[24] = 8;
isoBandEdgeBL[24] = isoBandEdgeTR[24] = 7;
isoBandEdgeRT[146] = isoBandEdgeBR[146] = 8;
isoBandEdgeBL[146] = isoBandEdgeTR[146] = 7;
isoBandEdgeRB[6] = isoBandEdgeLT[6] = 15;
isoBandEdgeBR[6] = isoBandEdgeLB[6] = 16;
isoBandEdgeRB[164] = isoBandEdgeLT[164] = 15;
isoBandEdgeBR[164] = isoBandEdgeLB[164] = 16;
isoBandEdgeBL[129] = isoBandEdgeTR[129] = 7;
isoBandEdgeLB[129] = isoBandEdgeTL[129] = 20;
isoBandEdgeBL[41] = isoBandEdgeTR[41] = 7;
isoBandEdgeLB[41] = isoBandEdgeTL[41] = 20;
isoBandEdgeBR[66] = isoBandEdgeTL[66] = 2;
isoBandEdgeBL[66] = isoBandEdgeLT[66] = 19;
isoBandEdgeBR[104] = isoBandEdgeTL[104] = 2;
isoBandEdgeBL[104] = isoBandEdgeLT[104] = 19;
isoBandEdgeRT[144] = isoBandEdgeLB[144] = 10;
isoBandEdgeLT[144] = isoBandEdgeTR[144] = 23;
isoBandEdgeRT[26] = isoBandEdgeLB[26] = 10;
isoBandEdgeLT[26] = isoBandEdgeTR[26] = 23;
isoBandEdgeRB[36] = isoBandEdgeTR[36] = 5;
isoBandEdgeBR[36] = isoBandEdgeTL[36] = 2;
isoBandEdgeRB[134] = isoBandEdgeTR[134] = 5;
isoBandEdgeBR[134] = isoBandEdgeTL[134] = 2;
isoBandEdgeRT[9] = isoBandEdgeLB[9] = 10;
isoBandEdgeRB[9] = isoBandEdgeBL[9] = 13;
isoBandEdgeRT[161] = isoBandEdgeLB[161] = 10;
isoBandEdgeRB[161] = isoBandEdgeBL[161] = 13;

/* hexagon cases */
isoBandEdgeRB[37] = isoBandEdgeTR[37] = 5;
isoBandEdgeLB[37] = isoBandEdgeTL[37] = 20;
isoBandEdgeRB[133] = isoBandEdgeTR[133] = 5;
isoBandEdgeLB[133] = isoBandEdgeTL[133] = 20;
isoBandEdgeBR[148] = isoBandEdgeLB[148] = 16;
isoBandEdgeLT[148] = isoBandEdgeTR[148] = 23;
isoBandEdgeBR[22] = isoBandEdgeLB[22] = 16;
isoBandEdgeLT[22] = isoBandEdgeTR[22] = 23;
isoBandEdgeRT[82] = isoBandEdgeBR[82] = 8;
isoBandEdgeBL[82] = isoBandEdgeLT[82] = 19;
isoBandEdgeRT[88] = isoBandEdgeBR[88] = 8;
isoBandEdgeBL[88] = isoBandEdgeLT[88] = 19;
isoBandEdgeRT[73] = isoBandEdgeTL[73] = 0;
isoBandEdgeRB[73] = isoBandEdgeBL[73] = 13;
isoBandEdgeRT[97] = isoBandEdgeTL[97] = 0;
isoBandEdgeRB[97] = isoBandEdgeBL[97] = 13;
isoBandEdgeRT[145] = isoBandEdgeBL[145] = 9;
isoBandEdgeLB[145] = isoBandEdgeTR[145] = 21;
isoBandEdgeRT[25] = isoBandEdgeBL[25] = 9;
isoBandEdgeLB[25] = isoBandEdgeTR[25] = 21;
isoBandEdgeRB[70] = isoBandEdgeTL[70] = 1;
isoBandEdgeBR[70] = isoBandEdgeLT[70] = 17;
isoBandEdgeRB[100] = isoBandEdgeTL[100] = 1;
isoBandEdgeBR[100] = isoBandEdgeLT[100] = 17;

/* 8-sided cases */
isoBandEdgeRT[34] = isoBandEdgeBL[34] = 9;
isoBandEdgeRB[34] = isoBandEdgeBR[34] = 12;
isoBandEdgeLB[34] = isoBandEdgeTR[34] = 21;
isoBandEdgeLT[34] = isoBandEdgeTL[34] = 22;
isoBandEdgeRT[136] = isoBandEdgeTR[136] = 4;
isoBandEdgeRB[136] = isoBandEdgeTL[136] = 1;
isoBandEdgeBR[136] = isoBandEdgeLT[136] = 17;
isoBandEdgeBL[136] = isoBandEdgeLB[136] = 18;
isoBandEdgeRT[35] = isoBandEdgeTR[35] = 4;
isoBandEdgeRB[35] = isoBandEdgeBR[35] = 12;
isoBandEdgeBL[35] = isoBandEdgeLB[35] = 18;
isoBandEdgeLT[35] = isoBandEdgeTL[35] = 22;

/* 6-sided cases */
isoBandEdgeRT[153] = isoBandEdgeTR[153] = 4;
isoBandEdgeBL[153] = isoBandEdgeLB[153] = 18;
isoBandEdgeRB[102] = isoBandEdgeBR[102] = 12;
isoBandEdgeLT[102] = isoBandEdgeTL[102] = 22;
isoBandEdgeRT[155] = isoBandEdgeBL[155] = 9;
isoBandEdgeLB[155] = isoBandEdgeTR[155] = 23;
isoBandEdgeRB[103] = isoBandEdgeTL[103] = 1;
isoBandEdgeBR[103] = isoBandEdgeLT[103] = 17;

/* 7-sided cases */
isoBandEdgeRT[152] = isoBandEdgeTR[152] = 4;
isoBandEdgeBR[152] = isoBandEdgeLT[152] = 17;
isoBandEdgeBL[152] = isoBandEdgeLB[152] = 18;
isoBandEdgeRT[156] = isoBandEdgeBR[156] = 8;
isoBandEdgeBL[156] = isoBandEdgeLB[156] = 18;
isoBandEdgeLT[156] = isoBandEdgeTR[156] = 23;
isoBandEdgeRT[137] = isoBandEdgeTR[137] = 4;
isoBandEdgeRB[137] = isoBandEdgeTL[137] = 1;
isoBandEdgeBL[137] = isoBandEdgeLB[137] = 18;
isoBandEdgeRT[139] = isoBandEdgeTR[139] = 4;
isoBandEdgeRB[139] = isoBandEdgeBL[139] = 13;
isoBandEdgeLB[139] = isoBandEdgeTL[139] = 20;
isoBandEdgeRT[98] = isoBandEdgeBL[98] = 9;
isoBandEdgeRB[98] = isoBandEdgeBR[98] = 12;
isoBandEdgeLT[98] = isoBandEdgeTL[98] = 22;
isoBandEdgeRT[99] = isoBandEdgeTL[99] = 0;
isoBandEdgeRB[99] = isoBandEdgeBR[99] = 12;
isoBandEdgeBL[99] = isoBandEdgeLT[99] = 19;
isoBandEdgeRB[38] = isoBandEdgeBR[38] = 12;
isoBandEdgeLB[38] = isoBandEdgeTR[38] = 21;
isoBandEdgeLT[38] = isoBandEdgeTL[38] = 22;
isoBandEdgeRB[39] = isoBandEdgeTR[39] = 5;
isoBandEdgeBR[39] = isoBandEdgeLB[39] = 16;
isoBandEdgeLT[39] = isoBandEdgeTL[39] = 22;

/*
  The lookup tables for all different polygons that
  may appear within a grid cell
*/

var polygon_table = [];

/* triangle cases */
polygon_table[1] = polygon_table[169] = p00; /* 2221 || 0001 */
polygon_table[4] = polygon_table[166] = p01; /* 2212 || 0010 */
polygon_table[16] = polygon_table[154] = p02; /* 2122 || 0100 */
polygon_table[64] = polygon_table[106] = p03; /* 1222 || 1000 */

/* trapezoid cases */
polygon_table[168] = polygon_table[2] = p04; /* 2220 || 0002 */
polygon_table[162] = polygon_table[8] = p05; /* 2202 || 0020 */
polygon_table[138] = polygon_table[32] = p06; /* 2022 || 0200 */
polygon_table[42] = polygon_table[128] = p07; /* 0222 || 2000 */

/* rectangle cases */
polygon_table[5] = polygon_table[165] = p08; /* 0011 || 2211 */
polygon_table[20] = polygon_table[150] = p09; /* 0110 || 2112 */
polygon_table[80] = polygon_table[90] = p10; /* 1100 || 1122 */
polygon_table[65] = polygon_table[105] = p11; /* 1001 || 1221 */
polygon_table[160] = polygon_table[10] = p12; /* 2200 || 0022 */
polygon_table[130] = polygon_table[40] = p13; /* 2002 || 0220 */

/* square case */
polygon_table[85] = p14; /* 1111 */

/* pentagon cases */
polygon_table[101] = polygon_table[69] = p15; /* 1211 || 1011 */
polygon_table[149] = polygon_table[21] = p16; /* 2111 || 0111 */
polygon_table[86] = polygon_table[84] = p17; /* 1112 || 1110 */
polygon_table[89] = polygon_table[81] = p18; /* 1121 || 1101 */
polygon_table[96] = polygon_table[74] = p19; /* 1200 || 1022 */
polygon_table[24] = polygon_table[146] = p20; /* 0120 || 2102 */
polygon_table[6] = polygon_table[164] = p21; /* 0012 || 2210 */
polygon_table[129] = polygon_table[41] = p22; /* 2001 || 0221 */
polygon_table[66] = polygon_table[104] = p23; /* 1002 || 1220 */
polygon_table[144] = polygon_table[26] = p24; /* 2100 || 0122 */
polygon_table[36] = polygon_table[134] = p25; /* 0210 || 2012 */
polygon_table[9] = polygon_table[161] = p26; /* 0021 || 2201 */

/* hexagon cases */
polygon_table[37] = polygon_table[133] = p27; /* 0211 || 2011 */
polygon_table[148] = polygon_table[22] = p28; /* 2110 || 0112 */
polygon_table[82] = polygon_table[88] = p29; /* 1102 || 1120 */
polygon_table[73] = polygon_table[97] = p30; /* 1021 || 1201 */
polygon_table[145] = polygon_table[25] = p31; /* 2101 || 0121 */
polygon_table[70] = polygon_table[100] = p32; /* 1012 || 1210 */

/* 8-sided cases */
polygon_table[34] = function (c) { return [p07(c), p05(c)]; }; /* 0202 || 2020 with flipped == 0 */
polygon_table[35] = p33; /* flipped == 1 state for 0202 and 2020 */
polygon_table[136] = function (c) { return [p06(c), p04(c)]; }; /* 2020 || 0202 with flipped == 0 */

/* 6-sided cases */
polygon_table[153] = function (c) { return [p02(c), p00(c)]; }; /* 0101 with flipped == 0 || 2121 with flipped == 2 */
polygon_table[102] = function (c) { return [p01(c), p03(c)]; }; /* 1010 with flipped == 0 || 1212 with flipped == 2 */
polygon_table[155] = p34; /* 0101 with flipped == 1 || 2121 with flipped == 1 */
polygon_table[103] = p35; /* 1010 with flipped == 1 || 1212 with flipped == 1 */

/* 7-sided cases */
polygon_table[152] = function (c) { return [p02(c), p04(c)]; }; /* 2120 with flipped == 2 || 0102 with flipped == 0 */
polygon_table[156] = p36; /* 2120 with flipped == 1 || 0102 with flipped == 1 */
polygon_table[137] = function (c) { return [p06(c), p00(c)]; }; /* 2021 with flipped == 2 || 0201 with flipped == 0 */
polygon_table[139] = p37; /* 2021 with flipped == 1 || 0201 with flipped == 1 */
polygon_table[98] = function (c) { return [p05(c), p03(c)]; }; /* 1202 with flipped == 2 || 1020 with flipped == 0 */
polygon_table[99] = p38; /* 1202 with flipped == 1 || 1020 with flipped == 1 */
polygon_table[38] = function (c) { return [p01(c), p07(c)]; }; /* 0212 with flipped == 2 || 2010 with flipped == 0 */
polygon_table[39] = p39; /* 0212 with flipped == 1 || 2010 with flipped == 1 */


/*
####################################
Some small helper functions
####################################
*/

/* assume that x1 == 1 &&  x0 == 0 */
function interpolateX$1(y, y0, y1) {
    return (y - y0) / (y1 - y0);
}

function isArray(myArray) {
    return myArray.constructor.toString().indexOf('Array') > -1;
}

/*
####################################
Below is the actual Marching Squares implementation
####################################
*/

function computeBandGrid(data, minV, bandwidth) {
    var rows = data.length - 1;
    var cols = data[0].length - 1;
    var BandGrid = { rows: rows, cols: cols, cells: [] };

    var maxV = minV + Math.abs(bandwidth);

    for (var j = 0; j < rows; ++j) {
        BandGrid.cells[j] = [];
        for (var i = 0; i < cols; ++i) {
            /*  compose the 4-trit corner representation */
            var cval = 0;

            var tl = data[j + 1][i];
            var tr = data[j + 1][i + 1];
            var br = data[j][i + 1];
            var bl = data[j][i];

            if (isNaN(tl) || isNaN(tr) || isNaN(br) || isNaN(bl)) {
                continue;
            }

            cval |= (tl < minV) ? 0 : (tl > maxV) ? 128 : 64;
            cval |= (tr < minV) ? 0 : (tr > maxV) ? 32 : 16;
            cval |= (br < minV) ? 0 : (br > maxV) ? 8 : 4;
            cval |= (bl < minV) ? 0 : (bl > maxV) ? 2 : 1;

            var cval_real = +cval;

            /* resolve ambiguity via averaging */
            var flipped = 0;
            if ((cval === 17)  || /* 0101 */
          (cval === 18)  || /* 0102 */
          (cval === 33)  || /* 0201 */
          (cval === 34)  || /* 0202 */
          (cval === 38)  || /* 0212 */
          (cval === 68)  || /* 1010 */
          (cval === 72)  || /* 1020 */
          (cval === 98)  || /* 1202 */
          (cval === 102)  || /* 1212 */
          (cval === 132)  || /* 2010 */
          (cval === 136)  || /* 2020 */
          (cval === 137)  || /* 2021 */
          (cval === 152)  || /* 2120 */
          (cval === 153) /* 2121 */
            ) {
                var average = (tl + tr + br + bl) / 4;
                /* set flipped state */
                flipped = (average > maxV) ? 2 : (average < minV) ? 0 : 1;

                /* adjust cval for flipped cases */

                /* 8-sided cases */
                if (cval === 34) {
                    if (flipped === 1) {
                        cval = 35;
                    } else if (flipped === 0) {
                        cval = 136;
                    }
                } else if (cval === 136) {
                    if (flipped === 1) {
                        cval = 35;
                        flipped = 4;
                    } else if (flipped === 0) {
                        cval = 34;
                    }
                }

                /* 6-sided polygon cases */
                else if (cval === 17) {
                    if (flipped === 1) {
                        cval = 155;
                        flipped = 4;
                    } else if (flipped === 0) {
                        cval = 153;
                    }
                } else if (cval === 68) {
                    if (flipped === 1) {
                        cval = 103;
                        flipped = 4;
                    } else if (flipped === 0) {
                        cval = 102;
                    }
                } else if (cval === 153) {
                    if (flipped === 1)
                        cval = 155;
                } else if (cval === 102) {
                    if (flipped === 1)
                        cval = 103;
                }

                /* 7-sided polygon cases */
                else if (cval === 152) {
                    if (flipped < 2) {
                        cval    = 156;
                        flipped = 1;
                    }
                } else if (cval === 137) {
                    if (flipped < 2) {
                        cval = 139;
                        flipped = 1;
                    }
                } else if (cval === 98) {
                    if (flipped < 2) {
                        cval    = 99;
                        flipped = 1;
                    }
                } else if (cval === 38) {
                    if (flipped < 2) {
                        cval    = 39;
                        flipped = 1;
                    }
                } else if (cval === 18) {
                    if (flipped > 0) {
                        cval = 156;
                        flipped = 4;
                    } else {
                        cval = 152;
                    }
                } else if (cval === 33) {
                    if (flipped > 0) {
                        cval = 139;
                        flipped = 4;
                    } else {
                        cval = 137;
                    }
                } else if (cval === 72) {
                    if (flipped > 0) {
                        cval = 99;
                        flipped = 4;
                    } else {
                        cval = 98;
                    }
                } else if (cval === 132) {
                    if (flipped > 0) {
                        cval = 39;
                        flipped = 4;
                    } else {
                        cval = 38;
                    }
                }
            }

            /* add cell to BandGrid if it contains at least one polygon-side */
            if ((cval != 0) && (cval != 170)) {
                var topleft, topright, bottomleft, bottomright,
                    righttop, rightbottom, lefttop, leftbottom;

                topleft = topright = bottomleft = bottomright = righttop =
                rightbottom = lefttop = leftbottom = 0.5;

                var edges = [];

                /* do interpolation here */
                /* 1st Triangles */
                if (cval === 1) { /* 0001 */
                    bottomleft = 1 - interpolateX$1(minV, br, bl);
                    leftbottom = 1 - interpolateX$1(minV, tl, bl);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 169) { /* 2221 */
                    bottomleft = interpolateX$1(maxV, bl, br);
                    leftbottom = interpolateX$1(maxV, bl, tl);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 4) { /* 0010 */
                    rightbottom = 1 - interpolateX$1(minV, tr, br);
                    bottomright = interpolateX$1(minV, bl, br);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 166) { /* 2212 */
                    rightbottom = interpolateX$1(maxV, br, tr);
                    bottomright = 1 - interpolateX$1(maxV, br, bl);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 16) { /* 0100 */
                    righttop = interpolateX$1(minV, br, tr);
                    topright = interpolateX$1(minV, tl, tr);
                    edges.push(isoBandEdgeRT[cval]);
                } else if (cval === 154) { /* 2122 */
                    righttop = 1 - interpolateX$1(maxV, tr, br);
                    topright = 1 - interpolateX$1(maxV, tr, tl);
                    edges.push(isoBandEdgeRT[cval]);
                } else if (cval === 64) { /* 1000 */
                    lefttop = interpolateX$1(minV, bl, tl);
                    topleft = 1 - interpolateX$1(minV, tr, tl);
                    edges.push(isoBandEdgeLT[cval]);
                } else if (cval === 106) { /* 1222 */
                    lefttop = 1 - interpolateX$1(maxV, tl, bl);
                    topleft = interpolateX$1(maxV, tl, tr);
                    edges.push(isoBandEdgeLT[cval]);
                }
                /* 2nd Trapezoids */
                else if (cval === 168) { /* 2220 */
                    bottomright = interpolateX$1(maxV, bl, br);
                    bottomleft = interpolateX$1(minV, bl, br);
                    leftbottom = interpolateX$1(minV, bl, tl);
                    lefttop = interpolateX$1(maxV, bl, tl);
                    edges.push(isoBandEdgeBR[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 2) { /* 0002 */
                    bottomright = 1 - interpolateX$1(minV, br, bl);
                    bottomleft = 1 - interpolateX$1(maxV, br, bl);
                    leftbottom = 1 - interpolateX$1(maxV, tl, bl);
                    lefttop = 1 - interpolateX$1(minV, tl, bl);
                    edges.push(isoBandEdgeBR[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 162) { /* 2202 */
                    righttop = interpolateX$1(maxV, br, tr);
                    rightbottom = interpolateX$1(minV, br, tr);
                    bottomright = 1 - interpolateX$1(minV, br, bl);
                    bottomleft = 1 - interpolateX$1(maxV, br, bl);
                    edges.push(isoBandEdgeBR[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 8) { /* 0020 */
                    righttop = 1 - interpolateX$1(minV, tr, br);
                    rightbottom = 1 - interpolateX$1(maxV, tr, br);
                    bottomright = interpolateX$1(maxV, bl, br);
                    bottomleft = interpolateX$1(minV, bl, br);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 138) { /* 2022 */
                    righttop = 1 - interpolateX$1(minV, tr, br);
                    rightbottom = 1 - interpolateX$1(maxV, tr, br);
                    topleft = 1 - interpolateX$1(maxV, tr, tl);
                    topright = 1 - interpolateX$1(minV, tr, tl);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 32) { /* 0200 */
                    righttop = interpolateX$1(maxV, br, tr);
                    rightbottom = interpolateX$1(minV, br, tr);
                    topleft = interpolateX$1(minV, tl, tr);
                    topright = interpolateX$1(maxV, tl, tr);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 42) { /* 0222 */
                    leftbottom = 1 - interpolateX$1(maxV, tl, bl);
                    lefttop = 1 - interpolateX$1(minV, tl, bl);
                    topleft = interpolateX$1(minV, tl, tr);
                    topright = interpolateX$1(maxV, tl, tr);
                    edges.push(isoBandEdgeLB[cval]);
                    edges.push(isoBandEdgeLT[cval]);
                } else if (cval === 128) { /* 2000 */
                    leftbottom = interpolateX$1(minV, bl, tl);
                    lefttop = interpolateX$1(maxV, bl, tl);
                    topleft = 1 - interpolateX$1(maxV, tr, tl);
                    topright = 1 - interpolateX$1(minV, tr, tl);
                    edges.push(isoBandEdgeLB[cval]);
                    edges.push(isoBandEdgeLT[cval]);
                }

                /* 3rd rectangle cases */
                if (cval === 5) { /* 0011 */
                    rightbottom = 1 - interpolateX$1(minV, tr, br);
                    leftbottom = 1 - interpolateX$1(minV, tl, bl);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 165) { /* 2211 */
                    rightbottom = interpolateX$1(maxV, br, tr);
                    leftbottom = interpolateX$1(maxV, bl, tl);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 20) { /* 0110 */
                    bottomright = interpolateX$1(minV, bl, br);
                    topright = interpolateX$1(minV, tl, tr);
                    edges.push(isoBandEdgeBR[cval]);
                } else if (cval === 150) { /* 2112 */
                    bottomright = 1 - interpolateX$1(maxV, br, bl);
                    topright = 1 - interpolateX$1(maxV, tr, tl);
                    edges.push(isoBandEdgeBR[cval]);
                } else if (cval === 80) { /* 1100 */
                    righttop = interpolateX$1(minV, br, tr);
                    lefttop = interpolateX$1(minV, bl, tl);
                    edges.push(isoBandEdgeRT[cval]);
                } else if (cval === 90) { /* 1122 */
                    righttop = 1 - interpolateX$1(maxV, tr, br);
                    lefttop = 1 - interpolateX$1(maxV, tl, bl);
                    edges.push(isoBandEdgeRT[cval]);
                } else if (cval === 65) { /* 1001 */
                    bottomleft = 1 - interpolateX$1(minV, br, bl);
                    topleft = 1 - interpolateX$1(minV, tr, tl);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 105) { /* 1221 */
                    bottomleft = interpolateX$1(maxV, bl, br);
                    topleft = interpolateX$1(maxV, tl, tr);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 160) { /* 2200 */
                    righttop = interpolateX$1(maxV, br, tr);
                    rightbottom = interpolateX$1(minV, br, tr);
                    leftbottom = interpolateX$1(minV, bl, tl);
                    lefttop = interpolateX$1(maxV, bl, tl);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 10) { /* 0022 */
                    righttop = 1 - interpolateX$1(minV, tr, br);
                    rightbottom = 1 - interpolateX$1(maxV, tr, br);
                    leftbottom = 1 - interpolateX$1(maxV, tl, bl);
                    lefttop = 1 - interpolateX$1(minV, tl, bl);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 130) { /* 2002 */
                    bottomright = 1 - interpolateX$1(minV, br, bl);
                    bottomleft = 1 - interpolateX$1(maxV, br, bl);
                    topleft = 1 - interpolateX$1(maxV, tr, tl);
                    topright = 1 - interpolateX$1(minV, tr, tl);
                    edges.push(isoBandEdgeBR[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 40) { /* 0220 */
                    bottomright = interpolateX$1(maxV, bl, br);
                    bottomleft = interpolateX$1(minV, bl, br);
                    topleft = interpolateX$1(minV, tl, tr);
                    topright = interpolateX$1(maxV, tl, tr);
                    edges.push(isoBandEdgeBR[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                }

                /* 4th single pentagon cases */
                else if (cval === 101) { /* 1211 */
                    rightbottom = interpolateX$1(maxV, br, tr);
                    topleft = interpolateX$1(maxV, tl, tr);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 69) { /* 1011 */
                    rightbottom = 1 - interpolateX$1(minV, tr, br);
                    topleft = 1 - interpolateX$1(minV, tr, tl);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 149) { /* 2111 */
                    leftbottom = interpolateX$1(maxV, bl, tl);
                    topright = 1 - interpolateX$1(maxV, tr, tl);
                    edges.push(isoBandEdgeLB[cval]);
                } else if (cval === 21) { /* 0111 */
                    leftbottom = 1 - interpolateX$1(minV, tl, bl);
                    topright = interpolateX$1(minV, tl, tr);
                    edges.push(isoBandEdgeLB[cval]);
                } else if (cval === 86) { /* 1112 */
                    bottomright = 1 - interpolateX$1(maxV, br, bl);
                    lefttop = 1 - interpolateX$1(maxV, tl, bl);
                    edges.push(isoBandEdgeBR[cval]);
                } else if (cval === 84) { /* 1110 */
                    bottomright = interpolateX$1(minV, bl, br);
                    lefttop = interpolateX$1(minV, bl, tl);
                    edges.push(isoBandEdgeBR[cval]);
                } else if (cval === 89) { /* 1121 */
                    righttop = 1 - interpolateX$1(maxV, tr, br);
                    bottomleft = interpolateX$1(maxV, bl, br);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 81) { /* 1101 */
                    righttop = interpolateX$1(minV, br, tr);
                    bottomleft = 1 - interpolateX$1(minV, br, bl);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 96) { /* 1200 */
                    righttop = interpolateX$1(maxV, br, tr);
                    rightbottom = interpolateX$1(minV, br, tr);
                    lefttop = interpolateX$1(minV, bl, tl);
                    topleft = interpolateX$1(maxV, tl, tr);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 74) { /* 1022 */
                    righttop = 1 - interpolateX$1(minV, tr, br);
                    rightbottom = 1 - interpolateX$1(maxV, tr, br);
                    lefttop = 1 - interpolateX$1(maxV, tl, bl);
                    topleft = 1 - interpolateX$1(minV, tr, tl);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 24) { /* 0120 */
                    righttop = 1 - interpolateX$1(maxV, tr, br);
                    bottomright = interpolateX$1(maxV, bl, br);
                    bottomleft = interpolateX$1(minV, bl, br);
                    topright = interpolateX$1(minV, tl, tr);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 146) { /* 2102 */
                    righttop = interpolateX$1(minV, br, tr);
                    bottomright = 1 - interpolateX$1(minV, br, bl);
                    bottomleft = 1 - interpolateX$1(maxV, br, bl);
                    topright = 1 - interpolateX$1(maxV, tr, tl);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 6) { /* 0012 */
                    rightbottom = 1 - interpolateX$1(minV, tr, br);
                    bottomright = 1 - interpolateX$1(maxV, br, bl);
                    leftbottom = 1 - interpolateX$1(maxV, tl, bl);
                    lefttop = 1 - interpolateX$1(minV, tl, bl);
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeBR[cval]);
                } else if (cval === 164) { /* 2210 */
                    rightbottom = interpolateX$1(maxV, br, tr);
                    bottomright = interpolateX$1(minV, bl, br);
                    leftbottom = interpolateX$1(minV, bl, tl);
                    lefttop = interpolateX$1(maxV, bl, tl);
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeBR[cval]);
                } else if (cval === 129) { /* 2001 */
                    bottomleft = 1 - interpolateX$1(minV, br, bl);
                    leftbottom = interpolateX$1(maxV, bl, tl);
                    topleft = 1 - interpolateX$1(maxV, tr, tl);
                    topright = 1 - interpolateX$1(minV, tr, tl);
                    edges.push(isoBandEdgeBL[cval]);
                    edges.push(isoBandEdgeLB[cval]);
                } else if (cval === 41) { /* 0221 */
                    bottomleft = interpolateX$1(maxV, bl, br);
                    leftbottom = 1 - interpolateX$1(minV, tl, bl);
                    topleft = interpolateX$1(minV, tl, tr);
                    topright = interpolateX$1(maxV, tl, tr);
                    edges.push(isoBandEdgeBL[cval]);
                    edges.push(isoBandEdgeLB[cval]);
                } else if (cval === 66) { /* 1002 */
                    bottomright = 1 - interpolateX$1(minV, br, bl);
                    bottomleft = 1 - interpolateX$1(maxV, br, bl);
                    lefttop = 1 - interpolateX$1(maxV, tl, bl);
                    topleft = 1 - interpolateX$1(minV, tr, tl);
                    edges.push(isoBandEdgeBR[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 104) { /* 1220 */
                    bottomright = interpolateX$1(maxV, bl, br);
                    bottomleft = interpolateX$1(minV, bl, br);
                    lefttop = interpolateX$1(minV, bl, tl);
                    topleft = interpolateX$1(maxV, tl, tr);
                    edges.push(isoBandEdgeBL[cval]);
                    edges.push(isoBandEdgeTL[cval]);
                } else if (cval === 144) { /* 2100 */
                    righttop = interpolateX$1(minV, br, tr);
                    leftbottom = interpolateX$1(minV, bl, tl);
                    lefttop = interpolateX$1(maxV, bl, tl);
                    topright = 1 - interpolateX$1(maxV, tr, tl);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeLT[cval]);
                } else if (cval === 26) { /* 0122 */
                    righttop = 1 - interpolateX$1(maxV, tr, br);
                    leftbottom = 1 - interpolateX$1(maxV, tl, bl);
                    lefttop = 1 - interpolateX$1(minV, tl, bl);
                    topright = interpolateX$1(minV, tl, tr);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeLT[cval]);
                } else if (cval === 36) { /* 0210 */
                    rightbottom = interpolateX$1(maxV, br, tr);
                    bottomright = interpolateX$1(minV, bl, br);
                    topleft = interpolateX$1(minV, tl, tr);
                    topright = interpolateX$1(maxV, tl, tr);
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeBR[cval]);
                } else if (cval === 134) { /* 2012 */
                    rightbottom = 1 - interpolateX$1(minV, tr, br);
                    bottomright = 1 - interpolateX$1(maxV, br, bl);
                    topleft = 1 - interpolateX$1(maxV, tr, tl);
                    topright = 1 - interpolateX$1(minV, tr, tl);
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeBR[cval]);
                } else if (cval === 9) { /* 0021 */
                    righttop = 1 - interpolateX$1(minV, tr, br);
                    rightbottom = 1 - interpolateX$1(maxV, tr, br);
                    bottomleft = interpolateX$1(maxV, bl, br);
                    leftbottom = 1 - interpolateX$1(minV, tl, bl);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 161) { /* 2201 */
                    righttop = interpolateX$1(maxV, br, tr);
                    rightbottom = interpolateX$1(minV, br, tr);
                    bottomleft = 1 - interpolateX$1(minV, br, bl);
                    leftbottom = interpolateX$1(maxV, bl, tl);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                }

                /* 5th single hexagon cases */
                else if (cval === 37) { /* 0211 */
                    rightbottom = interpolateX$1(maxV, br, tr);
                    leftbottom = 1 - interpolateX$1(minV, tl, bl);
                    topleft = interpolateX$1(minV, tl, tr);
                    topright = interpolateX$1(maxV, tl, tr);
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeLB[cval]);
                } else if (cval === 133) { /* 2011 */
                    rightbottom = 1 - interpolateX$1(minV, tr, br);
                    leftbottom = interpolateX$1(maxV, bl, tl);
                    topleft = 1 - interpolateX$1(maxV, tr, tl);
                    topright = 1 - interpolateX$1(minV, tr, tl);
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeLB[cval]);
                } else if (cval === 148) { /* 2110 */
                    bottomright = interpolateX$1(minV, bl, br);
                    leftbottom = interpolateX$1(minV, bl, tl);
                    lefttop = interpolateX$1(maxV, bl, tl);
                    topright = 1 - interpolateX$1(maxV, tr, tl);
                    edges.push(isoBandEdgeBR[cval]);
                    edges.push(isoBandEdgeLT[cval]);
                } else if (cval === 22) { /* 0112 */
                    bottomright = 1 - interpolateX$1(maxV, br, bl);
                    leftbottom = 1 - interpolateX$1(maxV, tl, bl);
                    lefttop = 1 - interpolateX$1(minV, tl, bl);
                    topright = interpolateX$1(minV, tl, tr);
                    edges.push(isoBandEdgeBR[cval]);
                    edges.push(isoBandEdgeLT[cval]);
                } else if (cval === 82) { /* 1102 */
                    righttop = interpolateX$1(minV, br, tr);
                    bottomright = 1 - interpolateX$1(minV, br, bl);
                    bottomleft = 1 - interpolateX$1(maxV, br, bl);
                    lefttop = 1 - interpolateX$1(maxV, tl, bl);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 88) { /* 1120 */
                    righttop = 1 - interpolateX$1(maxV, tr, br);
                    bottomright = interpolateX$1(maxV, bl, br);
                    bottomleft = interpolateX$1(minV, bl, br);
                    lefttop = interpolateX$1(minV, bl, tl);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 73) { /* 1021 */
                    righttop = 1 - interpolateX$1(minV, tr, br);
                    rightbottom = 1 - interpolateX$1(maxV, tr, br);
                    bottomleft = interpolateX$1(maxV, bl, br);
                    topleft = 1 - interpolateX$1(minV, tr, tl);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 97) { /* 1201 */
                    righttop = interpolateX$1(maxV, br, tr);
                    rightbottom = interpolateX$1(minV, br, tr);
                    bottomleft = 1 - interpolateX$1(minV, br, bl);
                    topleft = interpolateX$1(maxV, tl, tr);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                } else if (cval === 145) { /* 2101 */
                    righttop = interpolateX$1(minV, br, tr);
                    bottomleft = 1 - interpolateX$1(minV, br, bl);
                    leftbottom = interpolateX$1(maxV, bl, tl);
                    topright = 1 - interpolateX$1(maxV, tr, tl);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeLB[cval]);
                } else if (cval === 25) { /* 0121 */
                    righttop = 1 - interpolateX$1(maxV, tr, br);
                    bottomleft = interpolateX$1(maxV, bl, br);
                    leftbottom = 1 - interpolateX$1(minV, tl, bl);
                    topright = interpolateX$1(minV, tl, tr);
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeLB[cval]);
                } else if (cval === 70) { /* 1012 */
                    rightbottom = 1 - interpolateX$1(minV, tr, br);
                    bottomright = 1 - interpolateX$1(maxV, br, bl);
                    lefttop = 1 - interpolateX$1(maxV, tl, bl);
                    topleft = 1 - interpolateX$1(minV, tr, tl);
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeBR[cval]);
                } else if (cval === 100) { /* 1210 */
                    rightbottom = interpolateX$1(maxV, br, tr);
                    bottomright = interpolateX$1(minV, bl, br);
                    lefttop = interpolateX$1(minV, bl, tl);
                    topleft = interpolateX$1(maxV, tl, tr);
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeBR[cval]);
                }

                /* 8-sided cases */
                else if (cval === 34) { /* 0202 || 2020 with flipped == 0 */
                    if (flipped === 0) {
                        righttop = 1 - interpolateX$1(minV, tr, br);
                        rightbottom = 1 - interpolateX$1(maxV, tr, br);
                        bottomright = interpolateX$1(maxV, bl, br);
                        bottomleft = interpolateX$1(minV, bl, br);
                        leftbottom = interpolateX$1(minV, bl, tl);
                        lefttop = interpolateX$1(maxV, bl, tl);
                        topleft = 1 - interpolateX$1(maxV, tr, tl);
                        topright = 1 - interpolateX$1(minV, tr, tl);
                    } else {
                        righttop = interpolateX$1(maxV, br, tr);
                        rightbottom = interpolateX$1(minV, br, tr);
                        bottomright = 1 - interpolateX$1(minV, br, bl);
                        bottomleft = 1 - interpolateX$1(maxV, br, bl);
                        leftbottom = 1 - interpolateX$1(maxV, tl, bl);
                        lefttop = 1 - interpolateX$1(minV, tl, bl);
                        topleft = interpolateX$1(minV, tl, tr);
                        topright = interpolateX$1(maxV, tl, tr);
                    }
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeLB[cval]);
                    edges.push(isoBandEdgeLT[cval]);
                } else if (cval === 35) { /* flipped == 1 state for 0202, and 2020 with flipped == 4*/
                    if (flipped === 4) {
                        righttop = 1 - interpolateX$1(minV, tr, br);
                        rightbottom = 1 - interpolateX$1(maxV, tr, br);
                        bottomright = interpolateX$1(maxV, bl, br);
                        bottomleft = interpolateX$1(minV, bl, br);
                        leftbottom = interpolateX$1(minV, bl, tl);
                        lefttop = interpolateX$1(maxV, bl, tl);
                        topleft = 1 - interpolateX$1(maxV, tr, tl);
                        topright = 1 - interpolateX$1(minV, tr, tl);
                    } else {
                        righttop = interpolateX$1(maxV, br, tr);
                        rightbottom = interpolateX$1(minV, br, tr);
                        bottomright = 1 - interpolateX$1(minV, br, bl);
                        bottomleft = 1 - interpolateX$1(maxV, br, bl);
                        leftbottom = 1 - interpolateX$1(maxV, tl, bl);
                        lefttop = 1 - interpolateX$1(minV, tl, bl);
                        topleft = interpolateX$1(minV, tl, tr);
                        topright = interpolateX$1(maxV, tl, tr);
                    }
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                    edges.push(isoBandEdgeLT[cval]);
                } else if (cval === 136) { /* 2020 || 0202 with flipped == 0 */
                    if (flipped === 0) {
                        righttop = interpolateX$1(maxV, br, tr);
                        rightbottom = interpolateX$1(minV, br, tr);
                        bottomright = 1 - interpolateX$1(minV, br, bl);
                        bottomleft = 1 - interpolateX$1(maxV, br, bl);
                        leftbottom = 1 - interpolateX$1(maxV, tl, bl);
                        lefttop = 1 - interpolateX$1(minV, tl, bl);
                        topleft = interpolateX$1(minV, tl, tr);
                        topright = interpolateX$1(maxV, tl, tr);
                    } else {
                        righttop = 1 - interpolateX$1(minV, tr, br);
                        rightbottom = 1 - interpolateX$1(maxV, tr, br);
                        bottomright = interpolateX$1(maxV, bl, br);
                        bottomleft = interpolateX$1(minV, bl, br);
                        leftbottom = interpolateX$1(minV, bl, tl);
                        lefttop = interpolateX$1(maxV, bl, tl);
                        topleft = 1 - interpolateX$1(maxV, tr, tl);
                        topright = 1 - interpolateX$1(minV, tr, tl);
                    }
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeLB[cval]);
                    edges.push(isoBandEdgeLT[cval]);
                }

                /* 6-sided polygon cases */
                else if (cval === 153) { /* 0101 with flipped == 0 || 2121 with flipped == 2 */
                    if (flipped === 0) {
                        righttop = interpolateX$1(minV, br, tr);
                        bottomleft = 1 - interpolateX$1(minV, br, bl);
                        leftbottom = 1 - interpolateX$1(minV, tl, bl);
                        topright = interpolateX$1(minV, tl, tr);
                    } else {
                        righttop = 1 - interpolateX$1(maxV, tr, br);
                        bottomleft = interpolateX$1(maxV, bl, br);
                        leftbottom = interpolateX$1(maxV, bl, tl);
                        topright = 1 - interpolateX$1(maxV, tr, tl);
                    }
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 102) { /* 1010 with flipped == 0 || 1212 with flipped == 2 */
                    if (flipped === 0) {
                        rightbottom = 1 - interpolateX$1(minV, tr, br);
                        bottomright = interpolateX$1(minV, bl, br);
                        lefttop = interpolateX$1(minV, bl, tl);
                        topleft = 1 - interpolateX$1(minV, tr, tl);
                    } else {
                        rightbottom = interpolateX$1(maxV, br, tr);
                        bottomright = 1 - interpolateX$1(maxV, br, bl);
                        lefttop = 1 - interpolateX$1(maxV, tl, bl);
                        topleft = interpolateX$1(maxV, tl, tr);
                    }
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeLT[cval]);
                } else if (cval === 155) { /* 0101 with flipped == 4 || 2121 with flipped == 1 */
                    if (flipped === 4) {
                        righttop = interpolateX$1(minV, br, tr);
                        bottomleft = 1 - interpolateX$1(minV, br, bl);
                        leftbottom = 1 - interpolateX$1(minV, tl, bl);
                        topright = interpolateX$1(minV, tl, tr);
                    } else {
                        righttop = 1 - interpolateX$1(maxV, tr, br);
                        bottomleft = interpolateX$1(maxV, bl, br);
                        leftbottom = interpolateX$1(maxV, bl, tl);
                        topright = 1 - interpolateX$1(maxV, tr, tl);
                    }
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeLB[cval]);
                } else if (cval === 103) { /* 1010 with flipped == 4 || 1212 with flipped == 1 */
                    if (flipped === 4) {
                        rightbottom = 1 - interpolateX$1(minV, tr, br);
                        bottomright = interpolateX$1(minV, bl, br);
                        lefttop = interpolateX$1(minV, bl, tl);
                        topleft = 1 - interpolateX$1(minV, tr, tl);
                    } else {
                        rightbottom = interpolateX$1(maxV, br, tr);
                        bottomright = 1 - interpolateX$1(maxV, br, bl);
                        lefttop = 1 - interpolateX$1(maxV, tl, bl);
                        topleft = interpolateX$1(maxV, tl, tr);
                    }
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeBR[cval]);
                }

                /* 7-sided polygon cases */
                else if (cval === 152) { /* 2120 with flipped == 2 || 0102 with flipped == 0 */
                    if (flipped === 0) {
                        righttop = interpolateX$1(minV, br, tr);
                        bottomright = 1 - interpolateX$1(minV, br, bl);
                        bottomleft = 1 - interpolateX$1(maxV, br, bl);
                        leftbottom = 1 - interpolateX$1(maxV, tl, bl);
                        lefttop = 1 - interpolateX$1(minV, tl, bl);
                        topright = interpolateX$1(minV, tl, tr);
                    } else {
                        righttop = 1 - interpolateX$1(maxV, tr, br);
                        bottomright = interpolateX$1(maxV, bl, br);
                        bottomleft = interpolateX$1(minV, bl, br);
                        leftbottom = interpolateX$1(minV, bl, tl);
                        lefttop = interpolateX$1(maxV, bl, tl);
                        topright = 1 - interpolateX$1(maxV, tr, tl);
                    }
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeBR[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 156) { /* 2120 with flipped == 1 || 0102 with flipped == 4 */
                    if (flipped === 4) {
                        righttop = interpolateX$1(minV, br, tr);
                        bottomright = 1 - interpolateX$1(minV, br, bl);
                        bottomleft = 1 - interpolateX$1(maxV, br, bl);
                        leftbottom = 1 - interpolateX$1(maxV, tl, bl);
                        lefttop = 1 - interpolateX$1(minV, tl, bl);
                        topright = interpolateX$1(minV, tl, tr);
                    } else {
                        righttop = 1 - interpolateX$1(maxV, tr, br);
                        bottomright = interpolateX$1(maxV, bl, br);
                        bottomleft = interpolateX$1(minV, bl, br);
                        leftbottom = interpolateX$1(minV, bl, tl);
                        lefttop = interpolateX$1(maxV, bl, tl);
                        topright = 1 - interpolateX$1(maxV, tr, tl);
                    }
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                    edges.push(isoBandEdgeLT[cval]);
                } else if (cval === 137) { /* 2021 with flipped == 2 || 0201 with flipped == 0 */
                    if (flipped === 0) {
                        righttop = interpolateX$1(maxV, br, tr);
                        rightbottom = interpolateX$1(minV, br, tr);
                        bottomleft = 1 - interpolateX$1(minV, br, bl);
                        leftbottom = 1 - interpolateX$1(minV, tl, bl);
                        topleft = interpolateX$1(minV, tl, tr);
                        topright = interpolateX$1(maxV, tl, tr);
                    } else {
                        righttop = 1 - interpolateX$1(minV, tr, br);
                        rightbottom = 1 - interpolateX$1(maxV, tr, br);
                        bottomleft = interpolateX$1(maxV, bl, br);
                        leftbottom = interpolateX$1(maxV, bl, tl);
                        topleft = 1 - interpolateX$1(maxV, tr, tl);
                        topright = 1 - interpolateX$1(minV, tr, tl);
                    }
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 139) { /* 2021 with flipped == 1 || 0201 with flipped == 4 */
                    if (flipped === 4) {
                        righttop = interpolateX$1(maxV, br, tr);
                        rightbottom = interpolateX$1(minV, br, tr);
                        bottomleft = 1 - interpolateX$1(minV, br, bl);
                        leftbottom = 1 - interpolateX$1(minV, tl, bl);
                        topleft = interpolateX$1(minV, tl, tr);
                        topright = interpolateX$1(maxV, tl, tr);
                    } else {
                        righttop = 1 - interpolateX$1(minV, tr, br);
                        rightbottom = 1 - interpolateX$1(maxV, tr, br);
                        bottomleft = interpolateX$1(maxV, bl, br);
                        leftbottom = interpolateX$1(maxV, bl, tl);
                        topleft = 1 - interpolateX$1(maxV, tr, tl);
                        topright = 1 - interpolateX$1(minV, tr, tl);
                    }
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeLB[cval]);
                } else if (cval === 98) { /* 1202 with flipped == 2 || 1020 with flipped == 0 */
                    if (flipped === 0) {
                        righttop = 1 - interpolateX$1(minV, tr, br);
                        rightbottom = 1 - interpolateX$1(maxV, tr, br);
                        bottomright = interpolateX$1(maxV, bl, br);
                        bottomleft = interpolateX$1(minV, bl, br);
                        lefttop = interpolateX$1(minV, bl, tl);
                        topleft = 1 - interpolateX$1(minV, tr, tl);
                    } else {
                        righttop = interpolateX$1(maxV, br, tr);
                        rightbottom = interpolateX$1(minV, br, tr);
                        bottomright = 1 - interpolateX$1(minV, br, bl);
                        bottomleft = 1 - interpolateX$1(maxV, br, bl);
                        lefttop = 1 - interpolateX$1(maxV, tl, bl);
                        topleft = interpolateX$1(maxV, tl, tr);
                    }
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeLT[cval]);
                } else if (cval === 99) { /* 1202 with flipped == 1 || 1020 with flipped == 4 */
                    if (flipped === 4) {
                        righttop = 1 - interpolateX$1(minV, tr, br);
                        rightbottom = 1 - interpolateX$1(maxV, tr, br);
                        bottomright = interpolateX$1(maxV, bl, br);
                        bottomleft = interpolateX$1(minV, bl, br);
                        lefttop = interpolateX$1(minV, bl, tl);
                        topleft = 1 - interpolateX$1(minV, tr, tl);
                    } else {
                        righttop = interpolateX$1(maxV, br, tr);
                        rightbottom = interpolateX$1(minV, br, tr);
                        bottomright = 1 - interpolateX$1(minV, br, bl);
                        bottomleft = 1 - interpolateX$1(maxV, br, bl);
                        lefttop = 1 - interpolateX$1(maxV, tl, bl);
                        topleft = interpolateX$1(maxV, tl, tr);
                    }
                    edges.push(isoBandEdgeRT[cval]);
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeBL[cval]);
                } else if (cval === 38) { /* 0212 with flipped == 2 || 2010 with flipped == 0 */
                    if (flipped === 0) {
                        rightbottom = 1 - interpolateX$1(minV, tr, br);
                        bottomright = interpolateX$1(minV, bl, br);
                        leftbottom = interpolateX$1(minV, bl, tl);
                        lefttop = interpolateX$1(maxV, bl, tl);
                        topleft = 1 - interpolateX$1(maxV, tr, tl);
                        topright = 1 - interpolateX$1(minV, tr, tl);
                    } else {
                        rightbottom = interpolateX$1(maxV, br, tr);
                        bottomright = 1 - interpolateX$1(maxV, br, bl);
                        leftbottom = 1 - interpolateX$1(maxV, tl, bl);
                        lefttop = 1 - interpolateX$1(minV, tl, bl);
                        topleft = interpolateX$1(minV, tl, tr);
                        topright = interpolateX$1(maxV, tl, tr);
                    }
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeLB[cval]);
                    edges.push(isoBandEdgeLT[cval]);
                } else if (cval === 39) { /* 0212 with flipped == 1 || 2010 with flipped == 4 */
                    if (flipped === 4) {
                        rightbottom = 1 - interpolateX$1(minV, tr, br);
                        bottomright = interpolateX$1(minV, bl, br);
                        leftbottom = interpolateX$1(minV, bl, tl);
                        lefttop = interpolateX$1(maxV, bl, tl);
                        topleft = 1 - interpolateX$1(maxV, tr, tl);
                        topright = 1 - interpolateX$1(minV, tr, tl);
                    } else {
                        rightbottom = interpolateX$1(maxV, br, tr);
                        bottomright = 1 - interpolateX$1(maxV, br, bl);
                        leftbottom = 1 - interpolateX$1(maxV, tl, bl);
                        lefttop = 1 - interpolateX$1(minV, tl, bl);
                        topleft = interpolateX$1(minV, tl, tr);
                        topright = interpolateX$1(maxV, tl, tr);
                    }
                    edges.push(isoBandEdgeRB[cval]);
                    edges.push(isoBandEdgeBR[cval]);
                    edges.push(isoBandEdgeLT[cval]);
                } else if (cval === 85) {
                    righttop = 1;
                    rightbottom = 0;
                    bottomright = 1;
                    bottomleft = 0;
                    leftbottom = 0;
                    lefttop = 1;
                    topleft = 0;
                    topright = 1;
                }

                if (topleft < 0 || topleft > 1 || topright < 0 || topright > 1 || righttop < 0 || righttop > 1 || bottomright < 0 || bottomright > 1 || leftbottom < 0 || leftbottom > 1 || lefttop < 0 || lefttop > 1) {
                    console.log('MarchingSquaresJS-isoBands: ' + cval + ' ' + cval_real + ' ' + tl + ',' + tr + ',' + br + ',' + bl + ' ' + flipped + ' ' + topleft + ' ' + topright + ' ' + righttop + ' ' + rightbottom + ' ' + bottomright + ' ' + bottomleft + ' ' + leftbottom + ' ' + lefttop);
                }

                BandGrid.cells[j][i] = {
                    cval: cval,
                    cval_real: cval_real,
                    flipped: flipped,
                    topleft: topleft,
                    topright: topright,
                    righttop: righttop,
                    rightbottom: rightbottom,
                    bottomright: bottomright,
                    bottomleft: bottomleft,
                    leftbottom: leftbottom,
                    lefttop: lefttop,
                    edges: edges
                };
            }
        }
    }

    return BandGrid;
}

function BandGrid2AreaPaths(grid) {
    var areas = [];
    var rows = grid.rows;
    var cols = grid.cols;
    var currentPolygon = [];

    for (var j = 0; j < rows; j++) {
        for (var i = 0; i < cols; i++) {
            if ((typeof grid.cells[j][i] !== 'undefined') && (grid.cells[j][i].edges.length > 0)) {
                /* trace back polygon path starting from this cell */

                var cell = grid.cells[j][i];

                /* get start coordinates */

                var prev  = getStartXY(cell),
                    next  = null,
                    p     = i,
                    q     = j;

                if (prev !== null) {
                    currentPolygon.push([prev.p[0] + p, prev.p[1] + q]);
                    //console.log(cell);
                    //console.log("coords: " + (prev.p[0] + p) + " " + (prev.p[1] + q));
                }

                do {
                    //console.log(p + "," + q);
                    //console.log(grid.cells[q][p]);
                    //console.log(grid.cells[q][p].edges);
                    //console.log("from : " + prev.x + " " + prev.y + " " + prev.o);

                    next = getExitXY(grid.cells[q][p], prev.x, prev.y, prev.o);
                    if (next !== null) {
                        //console.log("coords: " + (next.p[0] + p) + " " + (next.p[1] + q));
                        currentPolygon.push([next.p[0] + p, next.p[1] + q]);
                        p += next.x;
                        q += next.y;
                        prev = next;
                    } else {
                        //console.log("getExitXY() returned null!");
                        break;
                    }
                    //console.log("to : " + next.x + " " + next.y + " " + next.o);
                    /* special case, where we've reached the grid boundaries */
                    if ((q < 0) || (q >= rows) || (p < 0) || (p >= cols) || (typeof grid.cells[q][p] === 'undefined')) {
                        /* to create a closed path, we need to trace our way
                arround the missing data, until we find an entry
                point again
            */

                        /* set back coordinates of current cell */
                        p -= next.x;
                        q -= next.y;

                        //console.log("reached boundary at " + p + " " + q);

                        var missing = traceOutOfGridPath(grid, p, q, next.x, next.y, next.o);
                        if (missing !== null) {
                            missing.path.forEach(function (pp) {
                                //console.log("coords: " + (pp[0]) + " " + (pp[1]));
                                currentPolygon.push(pp);
                            });
                            p = missing.i;
                            q = missing.j;
                            prev = missing;
                        } else {
                            break;
                        }
                        //console.log(grid.cells[q][p]);
                    }
                } while ((typeof grid.cells[q][p] !== 'undefined') &&
                  (grid.cells[q][p].edges.length > 0));

                areas.push(currentPolygon);
                //console.log("next polygon");
                //console.log(currentPolygon);
                currentPolygon = [];
                if (grid.cells[j][i].edges.length > 0)
                    i--;
            }
        }
    }
    return areas;
}

function traceOutOfGridPath(grid, i, j, d_x, d_y, d_o) {
    var cell = grid.cells[j][i];
    var cval = cell.cval_real;
    var p = i + d_x,
        q = j + d_y;
    var path = [];
    var closed = false;

    while (!closed) {
    //console.log("processing cell " + p + "," + q + " " + d_x + " " + d_y + " " + d_o);
        if ((typeof grid.cells[q] === 'undefined') || (typeof grid.cells[q][p] === 'undefined')) {
            //console.log("which is undefined");
            /* we can't move on, so we have to change direction to proceed further */

            /* go back to previous cell */
            q -= d_y;
            p -= d_x;
            cell = grid.cells[q][p];
            cval = cell.cval_real;

            /* check where we've left defined cells of the grid... */
            if (d_y === -1) { /* we came from top */
                if (d_o === 0) {  /* exit left */
                    if (cval & Node3) { /* lower left node is within range, so we move left */
                        path.push([p, q]);
                        d_x = -1;
                        d_y = 0;
                        d_o = 0;
                    } else if (cval & Node2) { /* lower right node is within range, so we move right */
                        path.push([p + 1, q]);
                        d_x = 1;
                        d_y = 0;
                        d_o = 0;
                    } else { /* close the path */
                        path.push([p + cell.bottomright, q]);
                        d_x = 0;
                        d_y = 1;
                        d_o = 1;
                        closed = true;
                        break;
                    }
                } else if (cval & Node3) {
                    path.push([p, q]);
                    d_x = -1;
                    d_y = 0;
                    d_o = 0;
                } else if (cval & Node2) {
                    path.push([p + cell.bottomright, q]);
                    d_x = 0;
                    d_y = 1;
                    d_o = 1;
                    closed = true;
                    break;
                } else {
                    path.push([p + cell.bottomleft, q]);
                    d_x = 0;
                    d_y = 1;
                    d_o = 0;
                    closed = true;
                    break;
                }
            } else if (d_y === 1) { /* we came from bottom */
                //console.log("we came from bottom and hit a non-existing cell " + (p + d_x) + "," + (q + d_y) + "!");
                if (d_o === 0) { /* exit left */
                    if (cval & Node1) { /* top right node is within range, so we move right */
                        path.push([p + 1, q + 1]);
                        d_x = 1;
                        d_y = 0;
                        d_o = 1;
                    } else if (!(cval & Node0)) { /* found entry within same cell */
                        path.push([p + cell.topright, q + 1]);
                        d_x = 0;
                        d_y = -1;
                        d_o = 1;
                        closed = true;
                        //console.log("found entry from bottom at " + p + "," + q);
                        break;
                    } else {
                        path.push([p + cell.topleft, q + 1]);
                        d_x = 0;
                        d_y = -1;
                        d_o = 0;
                        closed = true;
                        break;
                    }
                } else if (cval & Node1) {
                    path.push([p + 1, q + 1]);
                    d_x = 1;
                    d_y = 0;
                    d_o = 1;
                } else { /* move right */
                    path.push([p + 1, q + 1]);
                    d_x = 1;
                    d_y = 0;
                    d_o = 1;
                    //console.log("wtf");
                    //break;
                }
            } else if (d_x === -1) { /* we came from right */
                //console.log("we came from right and hit a non-existing cell at " + (p + d_x) + "," + (q + d_y) + "!");
                if (d_o === 0) {
                    //console.log("continue at bottom");
                    if (cval & Node0) {
                        path.push([p, q + 1]);
                        d_x = 0;
                        d_y = 1;
                        d_o = 0;
                        //console.log("moving upwards to " + (p + d_x) + "," + (q + d_y) + "!");
                    } else if (!(cval & Node3)) { /* there has to be an entry into the regular grid again! */
                        //console.log("exiting top");
                        path.push([p, q + cell.lefttop]);
                        d_x = 1;
                        d_y = 0;
                        d_o = 1;
                        closed = true;
                        break;
                    } else {
                        //console.log("exiting bottom");
                        path.push([p, q + cell.leftbottom]);
                        d_x = 1;
                        d_y = 0;
                        d_o = 0;
                        closed = true;
                        break;
                    }
                } else {
                    //console.log("continue at top");
                    if (cval & Node0) {
                        path.push([p, q + 1]);
                        d_x = 0;
                        d_y = 1;
                        d_o = 0;
                        //console.log("moving upwards to " + (p + d_x) + "," + (q + d_y) + "!");
                    } else { /* */
                        console.log('MarchingSquaresJS-isoBands: wtf');
                        break;
                    }
                }
            } else if (d_x === 1) { /* we came from left */
                //console.log("we came from left and hit a non-existing cell " + (p + d_x) + "," + (q + d_y) + "!");
                if (d_o === 0) { /* exit bottom */
                    if (cval & Node2) {
                        path.push([p + 1, q]);
                        d_x = 0;
                        d_y = -1;
                        d_o = 1;
                    } else {
                        path.push([p + 1, q + cell.rightbottom]);
                        d_x = -1;
                        d_y = 0;
                        d_o = 0;
                        closed = true;
                        break;
                    }
                } else { /* exit top */
                    if (cval & Node2) {
                        path.push([p + 1, q]);
                        d_x = 0;
                        d_y = -1;
                        d_o = 1;
                    } else if (!(cval & Node1)) {
                        path.push([p + 1, q + cell.rightbottom]);
                        d_x = -1;
                        d_y = 0;
                        d_o = 0;
                        closed = true;
                        break;
                    } else {
                        path.push([p + 1, q + cell.righttop]);
                        d_x = -1;
                        d_y = 0;
                        d_o = 1;
                        break;
                    }
                }
            } else { /* we came from the same cell */
                console.log('MarchingSquaresJS-isoBands: we came from nowhere!');
                break;
            }

        } else { /* try to find an entry into the regular grid again! */
            cell = grid.cells[q][p];
            cval = cell.cval_real;
            //console.log("which is defined");

            if (d_x === -1) {
                if (d_o === 0) {
                    /* try to go downwards */
                    if ((typeof grid.cells[q - 1] !== 'undefined') && (typeof grid.cells[q - 1][p] !== 'undefined')) {
                        d_x = 0;
                        d_y = -1;
                        d_o = 1;
                    } else if (cval & Node3) { /* proceed searching in x-direction */
                        //console.log("proceeding in x-direction!");
                        path.push([p, q]);
                    } else { /* we must have found an entry into the regular grid */
                        path.push([p + cell.bottomright, q]);
                        d_x = 0;
                        d_y = 1;
                        d_o = 1;
                        closed = true;
                        //console.log("found entry from bottom at " + p + "," + q);
                        break;
                    }
                } else if (cval & Node0) { /* proceed searchin in x-direction */
                    console.log('MarchingSquaresJS-isoBands: proceeding in x-direction!');
                } else { /* we must have found an entry into the regular grid */
                    console.log('MarchingSquaresJS-isoBands: found entry from top at ' + p + ',' + q);
                    break;
                }
            } else if (d_x === 1) {
                if (d_o === 0) {
                    console.log('MarchingSquaresJS-isoBands: wtf');
                    break;
                } else {
                    /* try to go upwards */
                    if ((typeof grid.cells[q + 1] !== 'undefined') && (typeof grid.cells[q + 1][p] !== 'undefined')) {
                        d_x = 0;
                        d_y = 1;
                        d_o = 0;
                    } else if (cval & Node1) {
                        path.push([p + 1, q + 1]);
                        d_x = 1;
                        d_y = 0;
                        d_o = 1;
                    } else { /* found an entry point into regular grid! */
                        path.push([p + cell.topleft, q + 1]);
                        d_x = 0;
                        d_y = -1;
                        d_o = 0;
                        closed = true;
                        //console.log("found entry from bottom at " + p + "," + q);
                        break;
                    }
                }
            } else if (d_y === -1) {
                if (d_o === 1) {
                    /* try to go right */
                    if (typeof grid.cells[q][p + 1] !== 'undefined') {
                        d_x = 1;
                        d_y = 0;
                        d_o = 1;
                    } else if (cval & Node2) {
                        path.push([p + 1, q]);
                        d_x = 0;
                        d_y = -1;
                        d_o = 1;
                    } else { /* found entry into regular grid! */
                        path.push([p + 1, q + cell.righttop]);
                        d_x = -1;
                        d_y = 0;
                        d_o = 1;
                        closed = true;
                        //console.log("found entry from top at " + p + "," + q);
                        break;
                    }
                } else {
                    console.log('MarchingSquaresJS-isoBands: wtf');
                    break;
                }
            } else if (d_y === 1) {
                if (d_o === 0) {
                    //console.log("we came from bottom left and proceed to the left");
                    /* try to go left */
                    if (typeof grid.cells[q][p - 1] !== 'undefined') {
                        d_x = -1;
                        d_y = 0;
                        d_o = 0;
                    } else if (cval & Node0) {
                        path.push([p, q + 1]);
                        d_x = 0;
                        d_y = 1;
                        d_o = 0;
                    } else { /* found an entry point into regular grid! */
                        path.push([p, q + cell.leftbottom]);
                        d_x = 1;
                        d_y = 0;
                        d_o = 0;
                        closed = true;
                        //console.log("found entry from bottom at " + p + "," + q);
                        break;
                    }
                } else {
                    //console.log("we came from bottom right and proceed to the right");
                    console.log('MarchingSquaresJS-isoBands: wtf');
                    break;
                }
            } else {
                console.log('MarchingSquaresJS-isoBands: where did we came from???');
                break;
            }

        }

        p += d_x;
        q += d_y;
        //console.log("going on to  " + p + "," + q + " via " + d_x + " " + d_y + " " + d_o);

        if ((p === i) && (q === j)) { /* bail out, once we've closed a circle path */
            break;
        }

    }

    //console.log("exit with " + p + "," + q + " " + d_x + " " + d_y + " " + d_o);
    return { path: path, i: p, j: q, x: d_x, y: d_y, o: d_o };
}

function deleteEdge(cell, edgeIdx) {
    delete cell.edges[edgeIdx];
    for (var k = edgeIdx + 1; k < cell.edges.length; k++) {
        cell.edges[k - 1] = cell.edges[k];
    }
    cell.edges.pop();
}

function getStartXY(cell) {

    if (cell.edges.length > 0) {
        var e = cell.edges[cell.edges.length - 1];
        //console.log("starting with edge " + e);
        var cval = cell.cval_real;
        switch (e) {
        case 0:   if (cval & Node1) { /* node 1 within range */
            return {p: [1, cell.righttop], x: -1, y: 0, o: 1};
        } else { /* node 1 below or above threshold */
            return {p: [cell.topleft, 1], x: 0, y: -1, o: 0};
        }
        case 1:   if (cval & Node2) {
            return {p: [cell.topleft, 1], x: 0, y: -1, o: 0};
        } else {
            return {p: [1, cell.rightbottom], x: -1, y: 0, o: 0};
        }
        case 2:   if (cval & Node2) {
            return {p: [cell.bottomright, 0], x: 0, y: 1, o: 1};
        } else {
            return {p: [cell.topleft, 1], x: 0, y: -1, o: 0};
        }
        case 3:   if (cval & Node3) {
            return {p: [cell.topleft, 1], x: 0, y: -1, o: 0};
        } else {
            return {p: [cell.bottomleft, 0], x: 0, y: 1, o: 0};
        }
        case 4:   if (cval & Node1) {
            return {p: [1, cell.righttop], x: -1, y: 0, o: 1};
        } else {
            return {p: [cell.topright, 1], x: 0, y: -1, o: 1};
        }
        case 5:   if (cval & Node2) {
            return {p: [cell.topright, 1], x: 0, y: -1, o: 1};
        } else {
            return {p: [1, cell.rightbottom], x: -1, y: 0, o: 0};
        }
        case 6:   if (cval & Node2) {
            return {p: [cell.bottomright, 0], x: 0, y: 1, o: 1};
        } else {
            return {p: [cell.topright, 1], x: 0, y: -1, o: 1};
        }
        case 7:   if (cval & Node3) {
            return {p: [cell.topright, 1], x: 0, y: -1, o: 1};
        } else {
            return {p: [cell.bottomleft, 0], x: 0, y: 1, o: 0};
        }
        case 8:   if (cval & Node2) {
            return {p: [cell.bottomright, 0], x: 0, y: 1, o: 1};
        } else {
            return {p: [1, cell.righttop], x: -1, y: 0, o: 1};
        }
        case 9:   if (cval & Node3) {
            return {p: [1, cell.righttop], x: -1, y: 0, o: 1};
        } else {
            return {p: [cell.bottomleft, 0], x: 0, y: 1, o: 0};
        }
        case 10:  if (cval & Node3) {
            return {p: [0, cell.leftbottom], x: 1, y: 0, o: 0};
        } else {
            return {p: [1, cell.righttop], x: -1, y: 0, o: 1};
        }
        case 11:  if (cval & Node0) {
            return {p: [1, cell.righttop], x: -1, y: 0, o: 1};
        } else {
            return {p: [0, cell.lefttop], x: 1, y: 0, o: 1};
        }
        case 12:  if (cval & Node2) {
            return {p: [cell.bottomright, 0], x: 0, y: 1, o: 1};
        } else {
            return {p: [1, cell.rightbottom], x: -1, y: 0, o: 0};
        }
        case 13:  if (cval & Node3) {
            return {p: [1, cell.rightbottom], x: -1, y: 0, o: 0};
        } else {
            return {p: [cell.bottomleft, 0], x: 0, y: 1, o: 0};
        }
        case 14:  if (cval & Node3) {
            return {p: [0, cell.leftbottom], x: 1, y: 0, o: 0};
        } else {
            return {p: [1, cell.rightbottom], x: -1, y: 0, o: 0};
        }
        case 15:  if (cval & Node0) {
            return {p: [1, cell.rightbottom], x: -1, y: 0, o: 0};
        } else {
            return {p: [0, cell.lefttop], x: 1, y: 0, o: 1};
        }
        case 16:  if (cval & Node2) {
            return {p: [cell.bottomright, 0], x: 0, y: 1, o: 1};
        } else {
            return {p: [0, cell.leftbottom], x: 1, y: 0, o: 0};
        }
        case 17:  if (cval & Node0) {
            return {p: [cell.bottomright, 0], x: 0, y: 1, o: 1};
        } else {
            return {p: [0, cell.lefttop], x: 1, y: 0, o: 1};
        }
        case 18:  if (cval & Node3) {
            return {p: [0, cell.leftbottom], x: 1, y: 0, o: 0};
        } else {
            return {p: [cell.bottomleft, 0], x: 0, y: 1, o: 0};
        }
        case 19:  if (cval & Node0) {
            return {p: [cell.bottomleft, 0], x: 0, y: 1, o: 0};
        } else {
            return {p: [0, cell.lefttop], x: 1, y: 0, o: 1};
        }
        case 20:  if (cval & Node0) {
            return {p: [cell.topleft, 1], x: 0, y: -1, o: 0};
        } else {
            return {p: [0, cell.leftbottom], x: 1, y: 0, o: 0};
        }
        case 21:  if (cval & Node1) {
            return {p: [0, cell.leftbottom], x: 1, y: 0, o: 0};
        } else {
            return {p: [cell.topright, 1], x: 0, y: -1, o: 1};
        }
        case 22:  if (cval & Node0) {
            return {p: [cell.topleft, 1], x: 0, y: -1, o: 0};
        } else {
            return {p: [0, cell.lefttop], x: 1, y: 0, o: 1};
        }
        case 23:  if (cval & Node1) {
            return {p: [0, cell.lefttop], x: 1, y: 0, o: 1};
        } else {
            return {p: [cell.topright, 1], x: 0, y: -1, o: 1};
        }
        default:  console.log('MarchingSquaresJS-isoBands: edge index out of range!');
            console.log(cell);
            break;
        }
    }

    return null;
}

function getExitXY(cell, x, y, o) {

    var e, id_x, d_x, d_y, cval = cell.cval;
    var d_o;

    switch (x) {
    case -1:  switch (o) {
    case 0:   e = isoBandEdgeRB[cval];
        d_x = isoBandNextXRB[cval];
        d_y = isoBandNextYRB[cval];
        d_o = isoBandNextORB[cval];
        break;
    default:  e = isoBandEdgeRT[cval];
        d_x = isoBandNextXRT[cval];
        d_y = isoBandNextYRT[cval];
        d_o = isoBandNextORT[cval];
        break;
        }
        break;
    case 1:   switch (o) {
    case 0:   e = isoBandEdgeLB[cval];
        d_x = isoBandNextXLB[cval];
        d_y = isoBandNextYLB[cval];
        d_o = isoBandNextOLB[cval];
        break;
    default:  e = isoBandEdgeLT[cval];
        d_x = isoBandNextXLT[cval];
        d_y = isoBandNextYLT[cval];
        d_o = isoBandNextOLT[cval];
        break;
        }
        break;
    default:  switch (y) {
    case -1:  switch (o) {
    case 0:   e = isoBandEdgeTL[cval];
        d_x = isoBandNextXTL[cval];
        d_y = isoBandNextYTL[cval];
        d_o = isoBandNextOTL[cval];
        break;
    default:  e = isoBandEdgeTR[cval];
        d_x = isoBandNextXTR[cval];
        d_y = isoBandNextYTR[cval];
        d_o = isoBandNextOTR[cval];
        break;
        }
        break;
    case 1:   switch (o) {
    case 0:   e = isoBandEdgeBL[cval];
        d_x = isoBandNextXBL[cval];
        d_y = isoBandNextYBL[cval];
        d_o = isoBandNextOBL[cval];
        break;
    default:  e = isoBandEdgeBR[cval];
        d_x = isoBandNextXBR[cval];
        d_y = isoBandNextYBR[cval];
        d_o = isoBandNextOBR[cval];
        break;
        }
        break;
    default:  break;
        }
        break;
    }

    id_x = cell.edges.indexOf(e);
    if (typeof cell.edges[id_x] !== 'undefined') {
        deleteEdge(cell, id_x);
    } else {
    //console.log("wrong edges...");
    //console.log(x + " " + y + " " + o);
    //console.log(cell);
        return null;
    }

    cval = cell.cval_real;

    switch (e) {
    case 0:   if (cval & Node1) { /* node 1 within range */
        x = cell.topleft;
        y = 1;
    } else { /* node 1 below or above threshold */
        x = 1;
        y = cell.righttop;
    }
        break;
    case 1:   if (cval & Node2) {
        x = 1;
        y = cell.rightbottom;
    } else {
        x = cell.topleft;
        y = 1;
    }
        break;
    case 2:   if (cval & Node2) {
        x = cell.topleft;
        y = 1;
    } else {
        x = cell.bottomright;
        y = 0;
    }
        break;
    case 3:   if (cval & Node3) {
        x = cell.bottomleft;
        y = 0;
    } else {
        x = cell.topleft;
        y = 1;
    }
        break;
    case 4:   if (cval & Node1) {
        x = cell.topright;
        y = 1;
    } else {
        x = 1;
        y = cell.righttop;
    }
        break;
    case 5:   if (cval & Node2) {
        x = 1;
        y = cell.rightbottom;
    } else {
        x = cell.topright;
        y = 1;
    }
        break;
    case 6:   if (cval & Node2) {
        x = cell.topright;
        y = 1;
    } else {
        x = cell.bottomright;
        y = 0;
    }
        break;
    case 7:   if (cval & Node3) {
        x = cell.bottomleft;
        y = 0;
    } else {
        x = cell.topright;
        y = 1;
    }
        break;
    case 8:   if (cval & Node2) {
        x = 1;
        y = cell.righttop;
    } else {
        x = cell.bottomright;
        y = 0;
    }
        break;
    case 9:   if (cval & Node3) {
        x = cell.bottomleft;
        y = 0;
    } else {
        x = 1;
        y = cell.righttop;
    }
        break;
    case 10:  if (cval & Node3) {
        x = 1;
        y = cell.righttop;
    } else {
        x = 0;
        y = cell.leftbottom;
    }
        break;
    case 11:  if (cval & Node0) {
        x = 0;
        y = cell.lefttop;
    } else {
        x = 1;
        y = cell.righttop;
    }
        break;
    case 12:  if (cval & Node2) {
        x = 1;
        y = cell.rightbottom;
    } else {
        x = cell.bottomright;
        y = 0;
    }
        break;
    case 13:  if (cval & Node3) {
        x = cell.bottomleft;
        y = 0;
    } else {
        x = 1;
        y = cell.rightbottom;
    }
        break;
    case 14:  if (cval & Node3) {
        x = 1;
        y = cell.rightbottom;
    } else {
        x = 0;
        y = cell.leftbottom;
    }
        break;
    case 15:  if (cval & Node0) {
        x = 0;
        y = cell.lefttop;
    } else {
        x = 1;
        y = cell.rightbottom;
    }
        break;
    case 16:  if (cval & Node2) {
        x = 0;
        y = cell.leftbottom;
    } else {
        x = cell.bottomright;
        y = 0;
    }
        break;
    case 17:  if (cval & Node0) {
        x = 0;
        y = cell.lefttop;
    } else {
        x = cell.bottomright;
        y = 0;
    }
        break;
    case 18:  if (cval & Node3) {
        x = cell.bottomleft;
        y = 0;
    } else {
        x = 0;
        y = cell.leftbottom;
    }
        break;
    case 19:  if (cval & Node0) {
        x = 0;
        y = cell.lefttop;
    } else {
        x = cell.bottomleft;
        y = 0;
    }
        break;
    case 20:  if (cval & Node0) {
        x = 0;
        y = cell.leftbottom;
    } else {
        x = cell.topleft;
        y = 1;
    }
        break;
    case 21:  if (cval & Node1) {
        x = cell.topright;
        y = 1;
    } else {
        x = 0;
        y = cell.leftbottom;
    }
        break;
    case 22:  if (cval & Node0) {
        x = 0;
        y = cell.lefttop;
    } else {
        x = cell.topleft;
        y = 1;
    }
        break;
    case 23:  if (cval & Node1) {
        x = cell.topright;
        y = 1;
    } else {
        x = 0;
        y = cell.lefttop;
    }
        break;
    default:  console.log('MarchingSquaresJS-isoBands: edge index out of range!');
        console.log(cell);
        return null;
    }

    if ((typeof x === 'undefined') || (typeof y === 'undefined') ||
      (typeof d_x === 'undefined') || (typeof d_y === 'undefined') ||
      (typeof d_o === 'undefined')) {
        console.log('MarchingSquaresJS-isoBands: undefined value!');
        console.log(cell);
        console.log(x + ' ' + y + ' ' + d_x + ' ' + d_y + ' ' + d_o);
    }
    return {p: [x, y], x: d_x, y: d_y, o: d_o};
}

function BandGrid2Areas(grid) {
    var areas = [];
    var area_idx = 0;

    grid.cells.forEach(function (g, j) {
        g.forEach(function (gg, i) {
            if (typeof gg !== 'undefined') {
                var a = polygon_table[gg.cval](gg);
                if ((typeof a === 'object') && isArray(a)) {
                    if ((typeof a[0] === 'object') && isArray(a[0])) {
                        if ((typeof a[0][0] === 'object') && isArray(a[0][0])) {
                            a.forEach(function (aa) {
                                aa.forEach(function (aaa) {
                                    aaa[0] += i;
                                    aaa[1] += j;
                                });
                                areas[area_idx++] = aa;
                            });
                        } else {
                            a.forEach(function (aa) {
                                aa[0] += i;
                                aa[1] += j;
                            });
                            areas[area_idx++] = a;
                        }
                    } else {
                        console.log('MarchingSquaresJS-isoBands: bandcell polygon with malformed coordinates');
                    }
                } else {
                    console.log('MarchingSquaresJS-isoBands: bandcell polygon with null coordinates');
                }
            }
        });
    });

    return areas;
}

/**
 * Takes a grid {@link FeatureCollection} of {@link Point} features with z-values and an array of
 * value breaks and generates filled contour isobands.
 *
 * @name isobands
 * @param {FeatureCollection<Point>} pointGrid input points
 * @param {Array<number>} breaks where to draw contours
 * @param {Object} [options={}] options on output
 * @param {string} [options.zProperty='elevation'] the property name in `points` from which z-values will be pulled
 * @param {Object} [options.commonProperties={}] GeoJSON properties passed to ALL isobands
 * @param {Array<Object>} [options.breaksProperties=[]] GeoJSON properties passed, in order, to the correspondent isoband (order defined by breaks)
 * @returns {FeatureCollection<MultiPolygon>} a FeatureCollection of {@link MultiPolygon} features representing isobands
 */
function isobands(pointGrid, breaks, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var zProperty = options.zProperty || 'elevation';
    var commonProperties = options.commonProperties || {};
    var breaksProperties = options.breaksProperties || [];

    // Validation
    collectionOf(pointGrid, 'Point', 'Input must contain Points');
    if (!breaks) throw new Error('breaks is required');
    if (!Array.isArray(breaks)) throw new Error('breaks is not an Array');
    if (!isObject(commonProperties)) throw new Error('commonProperties is not an Object');
    if (!Array.isArray(breaksProperties)) throw new Error('breaksProperties is not an Array');

    // Isoband methods
    var matrix = gridToMatrix$1(pointGrid, {zProperty: zProperty, flip: true});
    var contours = createContourLines(matrix, breaks, zProperty);
    contours = rescaleContours(contours, matrix, pointGrid);

    var multipolygons = contours.map(function (contour, index) {
        if (breaksProperties[index] && !isObject(breaksProperties[index])) {
            throw new Error('Each mappedProperty is required to be an Object');
        }
        // collect all properties
        var contourProperties = Object.assign(
            {},
            commonProperties,
            breaksProperties[index]
        );
        contourProperties[zProperty] = contour[zProperty];
        var multiP = multiPolygon(contour.groupedRings, contourProperties);
        return multiP;
    });

    return featureCollection(multipolygons);
}

/**
 * Creates the contours lines (featuresCollection of polygon features) from the 2D data grid
 *
 * Marchingsquares process the grid data as a 3D representation of a function on a 2D plane, therefore it
 * assumes the points (x-y coordinates) are one 'unit' distance. The result of the IsoBands function needs to be
 * rescaled, with turfjs, to the original area and proportions on the map
 *
 * @private
 * @param {Array<Array<number>>} matrix Grid Data
 * @param {Array<number>} breaks Breaks
 * @param {string} [property='elevation'] Property
 * @returns {Array<any>} contours
 */
function createContourLines(matrix, breaks, property) {

    var contours = [];
    for (var i = 1; i < breaks.length; i++) {
        var lowerBand = +breaks[i - 1]; // make sure the breaks value is a number
        var upperBand = +breaks[i];

        var isobandsCoords = isoBands(matrix, lowerBand, upperBand - lowerBand);
        // as per GeoJson rules for creating a Polygon, make sure the first element
        // in the array of LinearRings represents the exterior ring (i.e. biggest area),
        // and any subsequent elements represent interior rings (i.e. smaller area);
        // this avoids rendering issues of the MultiPolygons on the map
        var nestedRings = orderByArea(isobandsCoords);
        var groupedRings = groupNestedRings(nestedRings);
        var obj = {};
        obj['groupedRings'] = groupedRings;
        obj[property] = lowerBand + '-' + upperBand;
        contours.push(obj);
    }
    return contours;
}

/**
 * Transform isobands of 2D grid to polygons for the map
 *
 * @private
 * @param {Array<any>} contours Contours
 * @param {Array<Array<number>>} matrix Grid Data
 * @param {Object} points Points by Latitude
 * @returns {Array<any>} contours
 */
function rescaleContours(contours, matrix, points$$1) {

    // get dimensions (on the map) of the original grid
    var gridBbox = bbox(points$$1); // [ minX, minY, maxX, maxY ]
    var originalWidth = gridBbox[2] - gridBbox[0];
    var originalHeigth = gridBbox[3] - gridBbox[1];

    // get origin, which is the first point of the last row on the rectangular data on the map
    var x0 = gridBbox[0];
    var y0 = gridBbox[1];
    // get number of cells per side
    var matrixWidth = matrix[0].length - 1;
    var matrixHeight = matrix.length - 1;
    // calculate the scaling factor between matrix and rectangular grid on the map
    var scaleX = originalWidth / matrixWidth;
    var scaleY = originalHeigth / matrixHeight;

    var resize = function (point$$1) {
        point$$1[0] = point$$1[0] * scaleX + x0;
        point$$1[1] = point$$1[1] * scaleY + y0;
    };

    // resize and shift each point/line of the isobands
    contours.forEach(function (contour) {
        contour.groupedRings.forEach(function (lineRingSet) {
            lineRingSet.forEach(function (lineRing) {
                lineRing.forEach(resize);
            });
        });
    });
    return contours;
}


/*  utility functions */


/**
 * Returns an array of coordinates (of LinearRings) in descending order by area
 *
 * @private
 * @param {Array<LineString>} ringsCoords array of closed LineString
 * @returns {Array} array of the input LineString ordered by area
 */
function orderByArea(ringsCoords) {
    var ringsWithArea = [];
    var areas = [];
    ringsCoords.forEach(function (coords) {
        // var poly = polygon([points]);
        var ringArea = area$1(polygon([coords]));
        // create an array of areas value
        areas.push(ringArea);
        // associate each lineRing with its area
        ringsWithArea.push({ring: coords, area: ringArea});
    });
    areas.sort(function (a, b) { // bigger --> smaller
        return b - a;
    });
    // create a new array of linearRings coordinates ordered by their area
    var orderedByArea = [];
    areas.forEach(function (area$$1) {
        for (var lr = 0; lr < ringsWithArea.length; lr++) {
            if (ringsWithArea[lr].area === area$$1) {
                orderedByArea.push(ringsWithArea[lr].ring);
                ringsWithArea.splice(lr, 1);
                break;
            }
        }
    });
    return orderedByArea;
}

/**
 * Returns an array of arrays of coordinates, each representing
 * a set of (coordinates of) nested LinearRings,
 * i.e. the first ring contains all the others
 *
 * @private
 * @param {Array} orderedLinearRings array of coordinates (of LinearRings) in descending order by area
 * @returns {Array<Array>} Array of coordinates of nested LinearRings
 */
function groupNestedRings(orderedLinearRings) {
    // create a list of the (coordinates of) LinearRings
    var lrList = orderedLinearRings.map(function (lr) {
        return {lrCoordinates: lr, grouped: false};
    });
    var groupedLinearRingsCoords = [];
    while (!allGrouped(lrList)) {
        for (var i = 0; i < lrList.length; i++) {
            if (!lrList[i].grouped) {
                // create new group starting with the larger not already grouped ring
                var group = [];
                group.push(lrList[i].lrCoordinates);
                lrList[i].grouped = true;
                var outerMostPoly = polygon([lrList[i].lrCoordinates]);
                // group all the rings contained by the outermost ring
                for (var j = i + 1; j < lrList.length; j++) {
                    if (!lrList[j].grouped) {
                        var lrPoly = polygon([lrList[j].lrCoordinates]);
                        if (isInside(lrPoly, outerMostPoly)) {
                            group.push(lrList[j].lrCoordinates);
                            lrList[j].grouped = true;
                        }
                    }
                }
                // insert the new group
                groupedLinearRingsCoords.push(group);
            }
        }
    }
    return groupedLinearRingsCoords;
}

/**
 * @private
 * @param {Polygon} testPolygon polygon of interest
 * @param {Polygon} targetPolygon polygon you want to compare with
 * @returns {boolean} true if test-Polygon is inside target-Polygon
 */
function isInside(testPolygon, targetPolygon) {
    var points$$1 = explode(testPolygon);
    for (var i = 0; i < points$$1.features.length; i++) {
        if (!booleanPointInPolygon(points$$1.features[i], targetPolygon)) {
            return false;
        }
    }
    return true;
}

/**
 * @private
 * @param {Array<Object>} list list of objects which might contain the 'group' attribute
 * @returns {boolean} true if all the objects in the list are marked as grouped
 */
function allGrouped(list) {
    for (var i = 0; i < list.length; i++) {
        if (list[i].grouped === false) {
            return false;
        }
    }
    return true;
}

/**
 * Rotates any geojson Feature or Geometry of a specified angle, around its `centroid` or a given `pivot` point;
 * all rotations follow the right-hand rule: https://en.wikipedia.org/wiki/Right-hand_rule
 *
 * @name transformRotate
 * @param {GeoJSON} geojson object to be rotated
 * @param {number} angle of rotation (along the vertical axis), from North in decimal degrees, negative clockwise
 * @param {Object} [options={}] Optional parameters
 * @param {Coord} [options.pivot='centroid'] point around which the rotation will be performed
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated (significant performance increase if true)
 * @returns {GeoJSON} the rotated GeoJSON feature
 * @example
 * var poly = turf.polygon([[[0,29],[3.5,29],[2.5,32],[0,29]]]);
 * var options = {pivot: [0, 25]};
 * var rotatedPoly = turf.transformRotate(poly, 10, options);
 *
 * //addToMap
 * var addToMap = [poly, rotatedPoly];
 * rotatedPoly.properties = {stroke: '#F00', 'stroke-width': 4};
 */
function transformRotate(geojson, angle, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var pivot = options.pivot;
    var mutate = options.mutate;

    // Input validation
    if (!geojson) throw new Error('geojson is required');
    if (angle === undefined || angle === null || isNaN(angle)) throw new Error('angle is required');

    // Shortcut no-rotation
    if (angle === 0) return geojson;

    // Use centroid of GeoJSON if pivot is not provided
    if (!pivot) pivot = centroid(geojson);

    // Clone geojson to avoid side effects
    if (mutate === false || mutate === undefined) geojson = clone(geojson);

    // Rotate each coordinate
    coordEach(geojson, function (pointCoords) {
        var initialAngle = rhumbBearing(pivot, pointCoords);
        var finalAngle = initialAngle + angle;
        var distance = rhumbDistance(pivot, pointCoords);
        var newCoords = getCoords(rhumbDestination(pivot, distance, finalAngle));
        pointCoords[0] = newCoords[0];
        pointCoords[1] = newCoords[1];
    });
    return geojson;
}

/**
 * Scale a GeoJSON from a given point by a factor of scaling (ex: factor=2 would make the GeoJSON 200% larger).
 * If a FeatureCollection is provided, the origin point will be calculated based on each individual Feature.
 *
 * @name transformScale
 * @param {GeoJSON} geojson GeoJSON to be scaled
 * @param {number} factor of scaling, positive or negative values greater than 0
 * @param {Object} [options={}] Optional parameters
 * @param {string|Coord} [options.origin='centroid'] Point from which the scaling will occur (string options: sw/se/nw/ne/center/centroid)
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated (significant performance increase if true)
 * @returns {GeoJSON} scaled GeoJSON
 * @example
 * var poly = turf.polygon([[[0,29],[3.5,29],[2.5,32],[0,29]]]);
 * var scaledPoly = turf.transformScale(poly, 3);
 *
 * //addToMap
 * var addToMap = [poly, scaledPoly];
 * scaledPoly.properties = {stroke: '#F00', 'stroke-width': 4};
 */
function transformScale(geojson, factor, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var origin = options.origin;
    var mutate = options.mutate;

    // Input validation
    if (!geojson) throw new Error('geojson required');
    if (typeof factor !== 'number' || factor === 0) throw new Error('invalid factor');
    var originIsPoint = Array.isArray(origin) || typeof origin === 'object';

    // Clone geojson to avoid side effects
    if (mutate !== true) geojson = clone(geojson);

    // Scale each Feature separately
    if (geojson.type === 'FeatureCollection' && !originIsPoint) {
        featureEach(geojson, function (feature$$1, index) {
            geojson.features[index] = scale(feature$$1, factor, origin);
        });
        return geojson;
    }
    // Scale Feature/Geometry
    return scale(geojson, factor, origin);
}

/**
 * Scale Feature/Geometry
 *
 * @private
 * @param {Feature|Geometry} feature GeoJSON Feature/Geometry
 * @param {number} factor of scaling, positive or negative values greater than 0
 * @param {string|Coord} [origin="centroid"] Point from which the scaling will occur (string options: sw/se/nw/ne/center/centroid)
 * @returns {Feature|Geometry} scaled GeoJSON Feature/Geometry
 */
function scale(feature$$1, factor, origin) {
    // Default params
    var isPoint = getType(feature$$1) === 'Point';
    origin = defineOrigin(feature$$1, origin);

    // Shortcut no-scaling
    if (factor === 1 || isPoint) return feature$$1;

    // Scale each coordinate
    coordEach(feature$$1, function (coord) {
        var originalDistance = rhumbDistance(origin, coord);
        var bearing = rhumbBearing(origin, coord);
        var newDistance = originalDistance * factor;
        var newCoord = getCoords(rhumbDestination(origin, newDistance, bearing));
        coord[0] = newCoord[0];
        coord[1] = newCoord[1];
        if (coord.length === 3) coord[2] *= factor;
    });

    return feature$$1;
}

/**
 * Define Origin
 *
 * @private
 * @param {GeoJSON} geojson GeoJSON
 * @param {string|Coord} origin sw/se/nw/ne/center/centroid
 * @returns {Feature<Point>} Point origin
 */
function defineOrigin(geojson, origin) {
    // Default params
    if (origin === undefined || origin === null) origin = 'centroid';

    // Input Coord
    if (Array.isArray(origin) || typeof origin === 'object') return getCoord(origin);

    // Define BBox
    var bbox$$1 = (geojson.bbox) ? geojson.bbox : bbox(geojson);
    var west = bbox$$1[0];
    var south = bbox$$1[1];
    var east = bbox$$1[2];
    var north = bbox$$1[3];

    switch (origin) {
    case 'sw':
    case 'southwest':
    case 'westsouth':
    case 'bottomleft':
        return point([west, south]);
    case 'se':
    case 'southeast':
    case 'eastsouth':
    case 'bottomright':
        return point([east, south]);
    case 'nw':
    case 'northwest':
    case 'westnorth':
    case 'topleft':
        return point([west, north]);
    case 'ne':
    case 'northeast':
    case 'eastnorth':
    case 'topright':
        return point([east, north]);
    case 'center':
        return center(geojson);
    case undefined:
    case null:
    case 'centroid':
        return centroid(geojson);
    default:
        throw new Error('invalid origin');
    }
}

/**
 * Moves any geojson Feature or Geometry of a specified distance along a Rhumb Line
 * on the provided direction angle.
 *
 * @name transformTranslate
 * @param {GeoJSON} geojson object to be translated
 * @param {number} distance length of the motion; negative values determine motion in opposite direction
 * @param {number} direction of the motion; angle from North in decimal degrees, positive clockwise
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units='kilometers'] in which `distance` will be express; miles, kilometers, degrees, or radians
 * @param {number} [options.zTranslation=0] length of the vertical motion, same unit of distance
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated (significant performance increase if true)
 * @returns {GeoJSON} the translated GeoJSON object
 * @example
 * var poly = turf.polygon([[[0,29],[3.5,29],[2.5,32],[0,29]]]);
 * var translatedPoly = turf.transformTranslate(poly, 100, 35);
 *
 * //addToMap
 * var addToMap = [poly, translatedPoly];
 * translatedPoly.properties = {stroke: '#F00', 'stroke-width': 4};
 */
function transformTranslate(geojson, distance, direction, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var units = options.units;
    var zTranslation = options.zTranslation;
    var mutate = options.mutate;

    // Input validation
    if (!geojson) throw new Error('geojson is required');
    if (distance === undefined || distance === null || isNaN(distance)) throw new Error('distance is required');
    if (zTranslation && typeof zTranslation !== 'number' && isNaN(zTranslation)) throw new Error('zTranslation is not a number');

    // Shortcut no-motion
    zTranslation = (zTranslation !== undefined) ? zTranslation : 0;
    if (distance === 0 && zTranslation === 0) return geojson;

    if (direction === undefined || direction === null || isNaN(direction)) throw new Error('direction is required');

    // Invert with negative distances
    if (distance < 0) {
        distance = -distance;
        direction = -direction;
    }

    // Clone geojson to avoid side effects
    if (mutate === false || mutate === undefined) geojson = clone(geojson);

    // Translate each coordinate
    coordEach(geojson, function (pointCoords) {
        var newCoords = getCoords(rhumbDestination(pointCoords, distance, direction, {units: units}));
        pointCoords[0] = newCoords[0];
        pointCoords[1] = newCoords[1];
        if (zTranslation && pointCoords.length === 3) pointCoords[2] += zTranslation;
    });
    return geojson;
}

/**
 * https://github.com/rook2pawn/node-intersection
 *
 * Author @rook2pawn
 */

/**
 * AB
 *
 * @private
 * @param {Array<Array<number>>} segment - 2 vertex line segment
 * @returns {Array<number>} coordinates [x, y]
 */
function ab(segment) {
    var start = segment[0];
    var end = segment[1];
    return [end[0] - start[0], end[1] - start[1]];
}

/**
 * Cross Product
 *
 * @private
 * @param {Array<number>} v1 coordinates [x, y]
 * @param {Array<number>} v2 coordinates [x, y]
 * @returns {Array<number>} Cross Product
 */
function crossProduct(v1, v2) {
    return (v1[0] * v2[1]) - (v2[0] * v1[1]);
}

/**
 * Add
 *
 * @private
 * @param {Array<number>} v1 coordinates [x, y]
 * @param {Array<number>} v2 coordinates [x, y]
 * @returns {Array<number>} Add
 */
function add(v1, v2) {
    return [v1[0] + v2[0], v1[1] + v2[1]];
}

/**
 * Sub
 *
 * @private
 * @param {Array<number>} v1 coordinates [x, y]
 * @param {Array<number>} v2 coordinates [x, y]
 * @returns {Array<number>} Sub
 */
function sub(v1, v2) {
    return [v1[0] - v2[0], v1[1] - v2[1]];
}

/**
 * scalarMult
 *
 * @private
 * @param {number} s scalar
 * @param {Array<number>} v coordinates [x, y]
 * @returns {Array<number>} scalarMult
 */
function scalarMult(s, v) {
    return [s * v[0], s * v[1]];
}

/**
 * Intersect Segments
 *
 * @private
 * @param {Array<number>} a coordinates [x, y]
 * @param {Array<number>} b coordinates [x, y]
 * @returns {Array<number>} intersection
 */
function intersectSegments(a, b) {
    var p = a[0];
    var r = ab(a);
    var q = b[0];
    var s = ab(b);

    var cross = crossProduct(r, s);
    var qmp = sub(q, p);
    var numerator = crossProduct(qmp, s);
    var t = numerator / cross;
    var intersection = add(p, scalarMult(t, r));
    return intersection;
}

/**
 * Is Parallel
 *
 * @private
 * @param {Array<number>} a coordinates [x, y]
 * @param {Array<number>} b coordinates [x, y]
 * @returns {boolean} true if a and b are parallel (or co-linear)
 */
function isParallel(a, b) {
    var r = ab(a);
    var s = ab(b);
    return (crossProduct(r, s) === 0);
}

/**
 * Intersection
 *
 * @private
 * @param {Array<number>} a coordinates [x, y]
 * @param {Array<number>} b coordinates [x, y]
 * @returns {Array<number>|boolean} true if a and b are parallel (or co-linear)
 */
function intersection(a, b) {
    if (isParallel(a, b)) return false;
    return intersectSegments(a, b);
}

/**
 * Takes a {@link LineString|line} and returns a {@link LineString|line} at offset by the specified distance.
 *
 * @name lineOffset
 * @param {Geometry|Feature<LineString|MultiLineString>} geojson input GeoJSON
 * @param {number} distance distance to offset the line (can be of negative value)
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units='kilometers'] can be degrees, radians, miles, kilometers, inches, yards, meters
 * @returns {Feature<LineString|MultiLineString>} Line offset from the input line
 * @example
 * var line = turf.lineString([[-83, 30], [-84, 36], [-78, 41]], { "stroke": "#F00" });
 *
 * var offsetLine = turf.lineOffset(line, 2, {units: 'miles'});
 *
 * //addToMap
 * var addToMap = [offsetLine, line]
 * offsetLine.properties.stroke = "#00F"
 */
function lineOffset(geojson, distance, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var units = options.units;

    // Valdiation
    if (!geojson) throw new Error('geojson is required');
    if (distance === undefined || distance === null || isNaN(distance)) throw new Error('distance is required');

    var type = getType(geojson);
    var properties = geojson.properties;

    switch (type) {
    case 'LineString':
        return lineOffsetFeature(geojson, distance, units);
    case 'MultiLineString':
        var coords = [];
        flattenEach(geojson, function (feature$$1) {
            coords.push(lineOffsetFeature(feature$$1, distance, units).geometry.coordinates);
        });
        return multiLineString(coords, properties);
    default:
        throw new Error('geometry ' + type + ' is not supported');
    }
}

/**
 * Line Offset
 *
 * @private
 * @param {Geometry|Feature<LineString>} line input line
 * @param {number} distance distance to offset the line (can be of negative value)
 * @param {string} [units=kilometers] units
 * @returns {Feature<LineString>} Line offset from the input line
 */
function lineOffsetFeature(line, distance, units) {
    var segments = [];
    var offsetDegrees = lengthToDegrees(distance, units);
    var coords = getCoords(line);
    var finalCoords = [];
    coords.forEach(function (currentCoords, index) {
        if (index !== coords.length - 1) {
            var segment = processSegment(currentCoords, coords[index + 1], offsetDegrees);
            segments.push(segment);
            if (index > 0) {
                var seg2Coords = segments[index - 1];
                var intersects = intersection(segment, seg2Coords);

                // Handling for line segments that aren't straight
                if (intersects !== false) {
                    seg2Coords[1] = intersects;
                    segment[0] = intersects;
                }

                finalCoords.push(seg2Coords[0]);
                if (index === coords.length - 2) {
                    finalCoords.push(segment[0]);
                    finalCoords.push(segment[1]);
                }
            }
            // Handling for lines that only have 1 segment
            if (coords.length === 2) {
                finalCoords.push(segment[0]);
                finalCoords.push(segment[1]);
            }
        }
    });
    return lineString(finalCoords, line.properties);
}

/**
 * Process Segment
 * Inspiration taken from http://stackoverflow.com/questions/2825412/draw-a-parallel-line
 *
 * @private
 * @param {Array<number>} point1 Point coordinates
 * @param {Array<number>} point2 Point coordinates
 * @param {number} offset Offset
 * @returns {Array<Array<number>>} offset points
 */
function processSegment(point1, point2, offset) {
    var L = Math.sqrt((point1[0] - point2[0]) * (point1[0] - point2[0]) + (point1[1] - point2[1]) * (point1[1] - point2[1]));

    var out1x = point1[0] + offset * (point2[1] - point1[1]) / L;
    var out2x = point2[0] + offset * (point2[1] - point1[1]) / L;
    var out1y = point1[1] + offset * (point1[0] - point2[0]) / L;
    var out2y = point2[1] + offset * (point1[0] - point2[0]) / L;
    return [[out1x, out1y], [out2x, out2y]];
}

/**
 * Returns the direction of the point q relative to the vector p1 -> p2.
 *
 * Implementation of geos::algorithm::CGAlgorithm::orientationIndex()
 * (same as geos::algorithm::CGAlgorithm::computeOrientation())
 *
 * @param {number[]} p1 - the origin point of the vector
 * @param {number[]} p2 - the final point of the vector
 * @param {number[]} q - the point to compute the direction to
 *
 * @returns {number} - 1 if q is ccw (left) from p1->p2,
 *    -1 if q is cw (right) from p1->p2,
 *     0 if q is colinear with p1->p2
 */
function orientationIndex(p1, p2, q) {
    var dx1 = p2[0] - p1[0],
        dy1 = p2[1] - p1[1],
        dx2 = q[0] - p2[0],
        dy2 = q[1] - p2[1];

    return Math.sign(dx1 * dy2 - dx2 * dy1);
}

/**
 * Checks if two envelopes are equal.
 *
 * The function assumes that the arguments are envelopes, i.e.: Rectangular polygon
 *
 * @param {Feature<Polygon>} env1 - Envelope
 * @param {Feature<Polygon>} env2 - Envelope
 * @returns {boolean} - True if the envelopes are equal
 */
function envelopeIsEqual(env1, env2) {
    var envX1 = env1.geometry.coordinates.map(function (c) { return c[0]; }),
        envY1 = env1.geometry.coordinates.map(function (c) { return c[1]; }),
        envX2 = env2.geometry.coordinates.map(function (c) { return c[0]; }),
        envY2 = env2.geometry.coordinates.map(function (c) { return c[1]; });

    return Math.max(null, envX1) === Math.max(null, envX2) &&
    Math.max(null, envY1) === Math.max(null, envY2) &&
    Math.min(null, envX1) === Math.min(null, envX2) &&
    Math.min(null, envY1) === Math.min(null, envY2);
}

/**
 * Check if a envelope is contained in other one.
 *
 * The function assumes that the arguments are envelopes, i.e.: Convex polygon
 * XXX: Envelopes are rectangular, checking if a point is inside a rectangule is something easy,
 * this could be further improved.
 *
 * @param {Feature<Polygon>} self - Envelope
 * @param {Feature<Polygon>} env - Envelope
 * @returns {boolean} - True if env is contained in self
 */
function envelopeContains(self, env) {
    return env.geometry.coordinates[0].every(function (c) { return booleanPointInPolygon(point(c), self); });
}

/**
 * Checks if two coordinates are equal.
 *
 * @param {number[]} coord1 - First coordinate
 * @param {number[]} coord2 - Second coordinate
 * @returns {boolean} - True if coordinates are equal
 */
function coordinatesEqual(coord1, coord2) {
    return coord1[0] === coord2[0] && coord1[1] === coord2[1];
}

/**
 * Node
 */
var Node$1 = function Node(coordinates) {
    this.id = Node.buildId(coordinates);
    this.coordinates = coordinates; //< {Number[]}
    this.innerEdges = []; //< {Edge[]}

    // We wil store to (out) edges in an CCW order as geos::planargraph::DirectedEdgeStar does
    this.outerEdges = []; //< {Edge[]}
    this.outerEdgesSorted = false; //< {Boolean} flag that stores if the outer Edges had been sorted
};

Node$1.buildId = function buildId (coordinates) {
    return coordinates.join(',');
};

Node$1.prototype.removeInnerEdge = function removeInnerEdge (edge) {
    this.innerEdges = this.innerEdges.filter(function (e) { return e.from.id !== edge.from.id; });
};

Node$1.prototype.removeOuterEdge = function removeOuterEdge (edge) {
    this.outerEdges = this.outerEdges.filter(function (e) { return e.to.id !== edge.to.id; });
};

/**
 * Outer edges are stored CCW order.
 *
 * @memberof Node
 * @param {Edge} edge - Edge to add as an outerEdge.
 */
Node$1.prototype.addOuterEdge = function addOuterEdge (edge) {
    this.outerEdges.push(edge);
    this.outerEdgesSorted = false;
};

/**
 * Sorts outer edges in CCW way.
 *
 * @memberof Node
 * @private
 */
Node$1.prototype.sortOuterEdges = function sortOuterEdges () {
        var this$1 = this;

    if (!this.outerEdgesSorted) {
        //this.outerEdges.sort((a, b) => a.compareTo(b));
        // Using this comparator in order to be deterministic
        this.outerEdges.sort(function (a, b) {
            var aNode = a.to,
                bNode = b.to;

            if (aNode.coordinates[0] - this$1.coordinates[0] >= 0 && bNode.coordinates[0] - this$1.coordinates[0] < 0)
                { return 1; }
            if (aNode.coordinates[0] - this$1.coordinates[0] < 0 && bNode.coordinates[0] - this$1.coordinates[0] >= 0)
                { return -1; }

            if (aNode.coordinates[0] - this$1.coordinates[0] === 0 && bNode.coordinates[0] - this$1.coordinates[0] === 0) {
                if (aNode.coordinates[1] - this$1.coordinates[1] >= 0 || bNode.coordinates[1] - this$1.coordinates[1] >= 0)
                    { return aNode.coordinates[1] - bNode.coordinates[1]; }
                return bNode.coordinates[1] - aNode.coordinates[1];
            }

            var det = orientationIndex(this$1.coordinates, aNode.coordinates, bNode.coordinates);
            if (det < 0)
                { return 1; }
            if (det > 0)
                { return -1; }

            var d1 = Math.pow(aNode.coordinates[0] - this$1.coordinates[0], 2) + Math.pow(aNode.coordinates[1] - this$1.coordinates[1], 2),
                d2 = Math.pow(bNode.coordinates[0] - this$1.coordinates[0], 2) + Math.pow(bNode.coordinates[1] - this$1.coordinates[1], 2);

            return d1 - d2;
        });
        this.outerEdgesSorted = true;
    }
};

/**
 * Retrieves outer edges.
 *
 * They are sorted if they aren't in the CCW order.
 *
 * @memberof Node
 * @returns {Edge[]} - List of outer edges sorted in a CCW order.
 */
Node$1.prototype.getOuterEdges = function getOuterEdges () {
    this.sortOuterEdges();
    return this.outerEdges;
};

Node$1.prototype.getOuterEdge = function getOuterEdge (i) {
    this.sortOuterEdges();
    return this.outerEdges[i];
};

Node$1.prototype.addInnerEdge = function addInnerEdge (edge) {
    this.innerEdges.push(edge);
};

/**
 * This class is inspired by GEOS's geos::operation::polygonize::PolygonizeDirectedEdge
 */
var Edge = function Edge(from, to) {
    this.from = from; //< start
    this.to = to; //< End

    this.next = undefined; //< The edge to be computed after
    this.label = undefined; //< Used in order to detect Cut Edges (Bridges)
    this.symetric = undefined; //< The symetric edge of this
    this.ring = undefined; //< EdgeRing in which the Edge is

    this.from.addOuterEdge(this);
    this.to.addInnerEdge(this);
};

/**
 * Removes edge from from and to nodes.
 */
Edge.prototype.getSymetric = function getSymetric () {
    if (!this.symetric) {
        this.symetric = new Edge(this.to, this.from);
        this.symetric.symetric = this;
    }

    return this.symetric;
};

Edge.prototype.deleteEdge = function deleteEdge () {
    this.from.removeOuterEdge(this);
    this.to.removeInnerEdge(this);
};

/**
 * Compares Edge equallity.
 *
 * An edge is equal to another, if the from and to nodes are the same.
 *
 * @param {Edge} edge - Another Edge
 * @returns {boolean} - True if Edges are equal, False otherwise
 */
Edge.prototype.isEqual = function isEqual (edge) {
    return this.from.id === edge.from.id && this.to.id === edge.to.id;
};

Edge.prototype.toString = function toString () {
    return ("Edge { " + (this.from.id) + " -> " + (this.to.id) + " }");
};

/**
 * Returns a LineString representation of the Edge
 *
 * @returns {Feature<LineString>} - LineString representation of the Edge
 */
Edge.prototype.toLineString = function toLineString () {
    return lineString([this.from.coordinates, this.to.coordinates]);
};

/**
 * Comparator of two edges.
 *
 * Implementation of geos::planargraph::DirectedEdge::compareTo.
 *
 * @param {Edge} edge - Another edge to compare with this one
 * @returns {number} -1 if this Edge has a greater angle with the positive x-axis than b,
 *      0 if the Edges are colinear,
 *      1 otherwise
 */
Edge.prototype.compareTo = function compareTo (edge) {
    return orientationIndex(edge.from.coordinates, edge.to.coordinates, this.to.coordinates);
};

/**
 * Ring of edges which form a polygon.
 *
 * The ring may be either an outer shell or a hole.
 *
 * This class is inspired in GEOS's geos::operation::polygonize::EdgeRing
 */
var EdgeRing = function EdgeRing() {
    this.edges = [];
    this.polygon = undefined; //< Caches Polygon representation
    this.envelope = undefined; //< Caches Envelope representation
};

var prototypeAccessors = { length: { configurable: true } };

/**
 * Add an edge to the ring, inserting it in the last position.
 *
 * @memberof EdgeRing
 * @param {Edge} edge - Edge to be inserted
 */
EdgeRing.prototype.push = function push (edge) {
// Emulate Array getter ([]) behaviour
    this[this.edges.length] = edge;
    this.edges.push(edge);
    this.polygon = this.envelope = undefined;
};

/**
 * Get Edge.
 *
 * @memberof EdgeRing
 * @param {number} i - Index
 * @returns {Edge} - Edge in the i position
 */
EdgeRing.prototype.get = function get (i) {
    return this.edges[i];
};

/**
 * Getter of length property.
 *
 * @memberof EdgeRing
 * @returns {number} - Length of the edge ring.
 */
prototypeAccessors.length.get = function () {
    return this.edges.length;
};

/**
 * Similar to Array.prototype.forEach for the list of Edges in the EdgeRing.
 *
 * @memberof EdgeRing
 * @param {Function} f - The same function to be passed to Array.prototype.forEach
 */
EdgeRing.prototype.forEach = function forEach (f) {
    this.edges.forEach(f);
};

/**
 * Similar to Array.prototype.map for the list of Edges in the EdgeRing.
 *
 * @memberof EdgeRing
 * @param {Function} f - The same function to be passed to Array.prototype.map
 * @returns {Array} - The mapped values in the function
 */
EdgeRing.prototype.map = function map (f) {
    return this.edges.map(f);
};

/**
 * Similar to Array.prototype.some for the list of Edges in the EdgeRing.
 *
 * @memberof EdgeRing
 * @param {Function} f - The same function to be passed to Array.prototype.some
 * @returns {boolean} - True if an Edge check the condition
 */
EdgeRing.prototype.some = function some (f) {
    return this.edges.some(f);
};

/**
 * Check if the ring is valid in geomtry terms.
 *
 * A ring must have either 0 or 4 or more points. The first and the last must be
 * equal (in 2D)
 * geos::geom::LinearRing::validateConstruction
 *
 * @memberof EdgeRing
 * @returns {boolean} - Validity of the EdgeRing
 */
EdgeRing.prototype.isValid = function isValid () {
// TODO: stub
    return true;
};

/**
 * Tests whether this ring is a hole.
 *
 * A ring is a hole if it is oriented counter-clockwise.
 * Similar implementation of geos::algorithm::CGAlgorithms::isCCW
 *
 * @memberof EdgeRing
 * @returns {boolean} - true: if it is a hole
 */
EdgeRing.prototype.isHole = function isHole () {
        var this$1 = this;

// XXX: Assuming Ring is valid
// Find highest point
    var hiIndex = this.edges.reduce(function (high, edge, i) {
            if (edge.from.coordinates[1] > this$1.edges[high].from.coordinates[1])
                { high = i; }
            return high;
        }, 0),
        iPrev = (hiIndex === 0 ? this.length : hiIndex) - 1,
        iNext = (hiIndex + 1) % this.length,
        disc = orientationIndex(this.edges[iPrev].from.coordinates, this.edges[hiIndex].from.coordinates, this.edges[iNext].from.coordinates);

    if (disc === 0)
        { return this.edges[iPrev].from.coordinates[0] > this.edges[iNext].from.coordinates[0]; }
    return disc > 0;
};

/**
 * Creates a MultiPoint representing the EdgeRing (discarts edges directions).
 *
 * @memberof EdgeRing
 * @returns {Feature<MultiPoint>} - Multipoint representation of the EdgeRing
 */
EdgeRing.prototype.toMultiPoint = function toMultiPoint () {
    return multiPoint(this.edges.map(function (edge) { return edge.from.coordinates; }));
};

/**
 * Creates a Polygon representing the EdgeRing.
 *
 * @memberof EdgeRing
 * @returns {Feature<Polygon>} - Polygon representation of the Edge Ring
 */
EdgeRing.prototype.toPolygon = function toPolygon () {
    if (this.polygon)
        { return this.polygon; }
    var coordinates = this.edges.map(function (edge) { return edge.from.coordinates; });
    coordinates.push(this.edges[0].from.coordinates);
    return (this.polygon = polygon([coordinates]));
};

/**
 * Calculates the envelope of the EdgeRing.
 *
 * @memberof EdgeRing
 * @returns {Feature<Polygon>} - envelope
 */
EdgeRing.prototype.getEnvelope = function getEnvelope () {
    if (this.envelope)
        { return this.envelope; }
    return (this.envelope = envelope(this.toPolygon()));
};

/**
 * `geos::operation::polygonize::EdgeRing::findEdgeRingContaining`
 *
 * @param {EdgeRing} testEdgeRing - EdgeRing to look in the list
 * @param {EdgeRing[]} shellList - List of EdgeRing in which to search
 *
 * @returns {EdgeRing} - EdgeRing which contains the testEdgeRing
 */
EdgeRing.findEdgeRingContaining = function findEdgeRingContaining (testEdgeRing, shellList) {
    var testEnvelope = testEdgeRing.getEnvelope();

    var minEnvelope,
        minShell;
    shellList.forEach(function (shell) {
        var tryEnvelope = shell.getEnvelope();

        if (minShell)
            { minEnvelope = minShell.getEnvelope(); }

        // the hole envelope cannot equal the shell envelope
        if (envelopeIsEqual(tryEnvelope, testEnvelope))
            { return; }

        if (envelopeContains(tryEnvelope, testEnvelope)) {
            var testPoint = testEdgeRing.map(function (edge) { return edge.from.coordinates; })
                .find(function (pt) { return !shell.some(function (edge) { return coordinatesEqual(pt, edge.from.coordinates); }); });

            if (testPoint && shell.inside(point(testPoint))) {
                if (!minShell || envelopeContains(minEnvelope, tryEnvelope))
                    { minShell = shell; }
            }
        }
    });

    return minShell;
};

/**
 * Checks if the point is inside the edgeRing
 *
 * @param {Feature<Point>} pt - Point to check if it is inside the edgeRing
 * @returns {boolean} - True if it is inside, False otherwise
 */
EdgeRing.prototype.inside = function inside (pt) {
    return booleanPointInPolygon(pt, this.toPolygon());
};

Object.defineProperties( EdgeRing.prototype, prototypeAccessors );

/**
 * Validates the geoJson.
 *
 * @param {GeoJSON} geoJson - input geoJson.
 * @throws {Error} if geoJson is invalid.
 */
function validateGeoJson(geoJson) {
    if (!geoJson)
        { throw new Error('No geojson passed'); }

    if (geoJson.type !== 'FeatureCollection' &&
    geoJson.type !== 'GeometryCollection' &&
    geoJson.type !== 'MultiLineString' &&
    geoJson.type !== 'LineString' &&
    geoJson.type !== 'Feature'
    )
        { throw new Error(("Invalid input type '" + (geoJson.type) + "'. Geojson must be FeatureCollection, GeometryCollection, LineString, MultiLineString or Feature")); }
}

/**
 * Represents a planar graph of edges and nodes that can be used to compute a polygonization.
 *
 * Although, this class is inspired by GEOS's `geos::operation::polygonize::PolygonizeGraph`,
 * it isn't a rewrite. As regards algorithm, this class implements the same logic, but it
 * isn't a javascript transcription of the C++ source.
 *
 * This graph is directed (both directions are created)
 */
var Graph = function Graph() {
    this.edges = []; //< {Edge[]} dirEdges

    // The key is the `id` of the Node (ie: coordinates.join(','))
    this.nodes = {};
};

/**
 * Removes Dangle Nodes (nodes with grade 1).
 */
Graph.fromGeoJson = function fromGeoJson (geoJson) {
    validateGeoJson(geoJson);

    var graph = new Graph();
    flattenEach(geoJson, function (feature$$1) {
        featureOf(feature$$1, 'LineString', 'Graph::fromGeoJson');
        // When a LineString if formed by many segments, split them
        coordReduce(feature$$1, function (prev, cur) {
            if (prev) {
                var start = graph.getNode(prev),
                    end = graph.getNode(cur);

                graph.addEdge(start, end);
            }
            return cur;
        });
    });

    return graph;
};

/**
 * Creates or get a Node.
 *
 * @param {number[]} coordinates - Coordinates of the node
 * @returns {Node} - The created or stored node
 */
Graph.prototype.getNode = function getNode (coordinates) {
    var id = Node$1.buildId(coordinates);
    var node = this.nodes[id];
    if (!node)
        { node = this.nodes[id] = new Node$1(coordinates); }

    return node;
};

/**
 * Adds an Edge and its symetricall.
 *
 * Edges are added symetrically, i.e.: we also add its symetric
 *
 * @param {Node} from - Node which starts the Edge
 * @param {Node} to - Node which ends the Edge
 */
Graph.prototype.addEdge = function addEdge (from, to) {
    var edge = new Edge(from, to),
        symetricEdge = edge.getSymetric();

    this.edges.push(edge);
    this.edges.push(symetricEdge);
};

Graph.prototype.deleteDangles = function deleteDangles () {
        var this$1 = this;

    Object.keys(this.nodes)
        .map(function (id) { return this$1.nodes[id]; })
        .forEach(function (node) { return this$1._removeIfDangle(node); });
};

/**
 * Check if node is dangle, if so, remove it.
 *
 * It calls itself recursively, removing a dangling node might cause another dangling node
 *
 * @param {Node} node - Node to check if it's a dangle
 */
Graph.prototype._removeIfDangle = function _removeIfDangle (node) {
        var this$1 = this;

// As edges are directed and symetrical, we count only innerEdges
    if (node.innerEdges.length <= 1) {
        var outerNodes = node.getOuterEdges().map(function (e) { return e.to; });
        this.removeNode(node);
        outerNodes.forEach(function (n) { return this$1._removeIfDangle(n); });
    }
};

/**
 * Delete cut-edges (bridge edges).
 *
 * The graph will be traversed, all the edges will be labeled according the ring
 * in which they are. (The label is a number incremented by 1). Edges with the same
 * label are cut-edges.
 */
Graph.prototype.deleteCutEdges = function deleteCutEdges () {
        var this$1 = this;

    this._computeNextCWEdges();
    this._findLabeledEdgeRings();

    // Cut-edges (bridges) are edges where both edges have the same label
    this.edges.forEach(function (edge) {
        if (edge.label === edge.symetric.label) {
            this$1.removeEdge(edge.symetric);
            this$1.removeEdge(edge);
        }
    });
};

/**
 * Set the `next` property of each Edge.
 *
 * The graph will be transversed in a CW form, so, we set the next of the symetrical edge as the previous one.
 * OuterEdges are sorted CCW.
 *
 * @param {Node} [node] - If no node is passed, the function calls itself for every node in the Graph
 */
Graph.prototype._computeNextCWEdges = function _computeNextCWEdges (node) {
        var this$1 = this;

    if (typeof node === 'undefined') {
        Object.keys(this.nodes)
            .forEach(function (id) { return this$1._computeNextCWEdges(this$1.nodes[id]); });
    } else {
        node.getOuterEdges().forEach(function (edge, i) {
            node.getOuterEdge((i === 0 ? node.getOuterEdges().length : i) - 1).symetric.next = edge;
        });
    }
};

/**
 * Computes the next edge pointers going CCW around the given node, for the given edgering label.
 *
 * This algorithm has the effect of converting maximal edgerings into minimal edgerings
 *
 * XXX: method literally transcribed from `geos::operation::polygonize::PolygonizeGraph::computeNextCCWEdges`,
 * could be written in a more javascript way.
 *
 * @param {Node} node - Node
 * @param {number} label - Ring's label
 */
Graph.prototype._computeNextCCWEdges = function _computeNextCCWEdges (node, label) {
    var edges = node.getOuterEdges();
    var firstOutDE,
        prevInDE;

    for (var i = edges.length - 1; i >= 0; --i) {
        var de = edges[i],
            sym = de.symetric,
            outDE = (void 0),
            inDE = (void 0);

        if (de.label === label)
            { outDE = de; }

        if (sym.label === label)
            { inDE = sym; }

        if (!outDE || !inDE) // This edge is not in edgering
            { continue; }

        if (inDE)
            { prevInDE = inDE; }

        if (outDE) {
            if (prevInDE) {
                prevInDE.next = outDE;
                prevInDE = undefined;
            }

            if (!firstOutDE)
                { firstOutDE = outDE; }
        }
    }

    if (prevInDE)
        { prevInDE.next = firstOutDE; }
};


/**
 * Finds rings and labels edges according to which rings are.
 *
 * The label is a number which is increased for each ring.
 *
 * @returns {Edge[]} edges that start rings
 */
Graph.prototype._findLabeledEdgeRings = function _findLabeledEdgeRings () {
    var edgeRingStarts = [];
    var label = 0;
    this.edges.forEach(function (edge) {
        if (edge.label >= 0)
            { return; }

        edgeRingStarts.push(edge);

        var e = edge;
        do {
            e.label = label;
            e = e.next;
        } while (!edge.isEqual(e));

        label++;
    });

    return edgeRingStarts;
};

/**
 * Computes the EdgeRings formed by the edges in this graph.
 *
 * @returns {EdgeRing[]} - A list of all the EdgeRings in the graph.
 */
Graph.prototype.getEdgeRings = function getEdgeRings () {
        var this$1 = this;

    this._computeNextCWEdges();

    // Clear labels
    this.edges.forEach(function (edge) {
        edge.label = undefined;
    });

    this._findLabeledEdgeRings().forEach(function (edge) {
        // convertMaximalToMinimalEdgeRings
        this$1._findIntersectionNodes(edge).forEach(function (node) {
            this$1._computeNextCCWEdges(node, edge.label);
        });
    });

    var edgeRingList = [];

    // find all edgerings
    this.edges.forEach(function (edge) {
        if (edge.ring)
            { return; }
        edgeRingList.push(this$1._findEdgeRing(edge));
    });

    return edgeRingList;
};

/**
 * Find all nodes in a Maxima EdgeRing which are self-intersection nodes.
 *
 * @param {Node} startEdge - Start Edge of the Ring
 * @returns {Node[]} - intersection nodes
 */
Graph.prototype._findIntersectionNodes = function _findIntersectionNodes (startEdge) {
    var intersectionNodes = [];
    var edge = startEdge;
    var loop = function () {
        // getDegree
        var degree = 0;
        edge.from.getOuterEdges().forEach(function (e) {
            if (e.label === startEdge.label)
                { ++degree; }
        });

        if (degree > 1)
            { intersectionNodes.push(edge.from); }

        edge = edge.next;
    };

        do {
            loop();
        } while (!startEdge.isEqual(edge));

    return intersectionNodes;
};

/**
 * Get the edge-ring which starts from the provided Edge.
 *
 * @param {Edge} startEdge - starting edge of the edge ring
 * @returns {EdgeRing} - EdgeRing which start Edge is the provided one.
 */
Graph.prototype._findEdgeRing = function _findEdgeRing (startEdge) {
    var edge = startEdge;
    var edgeRing = new EdgeRing();

    do {
        edgeRing.push(edge);
        edge.ring = edgeRing;
        edge = edge.next;
    } while (!startEdge.isEqual(edge));

    return edgeRing;
};

/**
 * Removes a node from the Graph.
 *
 * It also removes edges asociated to that node
 * @param {Node} node - Node to be removed
 */
Graph.prototype.removeNode = function removeNode (node) {
        var this$1 = this;

    node.getOuterEdges().forEach(function (edge) { return this$1.removeEdge(edge); });
    node.innerEdges.forEach(function (edge) { return this$1.removeEdge(edge); });
    delete this.nodes[node.id];
};

/**
 * Remove edge from the graph and deletes the edge.
 *
 * @param {Edge} edge - Edge to be removed
 */
Graph.prototype.removeEdge = function removeEdge (edge) {
    this.edges = this.edges.filter(function (e) { return !e.isEqual(edge); });
    edge.deleteEdge();
};

/**
 * Polygonizes {@link LineString|(Multi)LineString(s)} into {@link Polygons}.
 *
 * Implementation of GEOSPolygonize function (`geos::operation::polygonize::Polygonizer`).
 *
 * Polygonizes a set of lines that represents edges in a planar graph. Edges must be correctly
 * noded, i.e., they must only meet at their endpoints.
 *
 * The implementation correctly handles:
 *
 * - Dangles: edges which have one or both ends which are not incident on another edge endpoint.
 * - Cut Edges (bridges): edges that are connected at both ends but which do not form part of a polygon.
 *
 * @name polygonize
 * @param {FeatureCollection|Geometry|Feature<LineString|MultiLineString>} geoJson Lines in order to polygonize
 * @returns {FeatureCollection<Polygon>} Polygons created
 * @throws {Error} if geoJson is invalid.
 */
function polygonize$1(geoJson) {
    var graph = Graph.fromGeoJson(geoJson);

    // 1. Remove dangle node
    graph.deleteDangles();

    // 2. Remove cut-edges (bridge edges)
    graph.deleteCutEdges();

    // 3. Get all holes and shells
    var holes = [],
        shells = [];

    graph.getEdgeRings()
        .filter(function (edgeRing) { return edgeRing.isValid(); })
        .forEach(function (edgeRing) {
            if (edgeRing.isHole())
                { holes.push(edgeRing); }
            else
                { shells.push(edgeRing); }
        });

    // 4. Assign Holes to Shells
    holes.forEach(function (hole) {
        if (EdgeRing.findEdgeRingContaining(hole, shells))
            { shells.push(hole); }
    });

    // 5. EdgeRings to Polygons
    return featureCollection(shells.map(function (shell) { return shell.toPolygon(); }));
}

/**
 * Boolean-disjoint returns (TRUE) if the intersection of the two geometries is an empty set.
 *
 * @name booleanDisjoint
 * @param {Geometry|Feature<any>} feature1 GeoJSON Feature or Geometry
 * @param {Geometry|Feature<any>} feature2 GeoJSON Feature or Geometry
 * @returns {boolean} true/false
 * @example
 * var point = turf.point([2, 2]);
 * var line = turf.lineString([[1, 1], [1, 2], [1, 3], [1, 4]]);
 *
 * turf.booleanDisjoint(line, point);
 * //=true
 */
function booleanDisjoint(feature1, feature2) {
    var boolean;
    flattenEach(feature1, function (flatten1) {
        flattenEach(feature2, function (flatten2) {
            if (boolean === false) return false;
            boolean = disjoint(flatten1.geometry, flatten2.geometry);
        });
    });
    return boolean;
}

/**
 * Disjoint operation for simple Geometries (Point/LineString/Polygon)
 *
 * @private
 * @param {Geometry<any>} geom1 GeoJSON Geometry
 * @param {Geometry<any>} geom2 GeoJSON Geometry
 * @returns {boolean} true/false
 */
function disjoint(geom1, geom2) {
    switch (geom1.type) {
    case 'Point':
        switch (geom2.type) {
        case 'Point':
            return !compareCoords$1(geom1.coordinates, geom2.coordinates);
        case 'LineString':
            return !isPointOnLine$1(geom2, geom1);
        case 'Polygon':
            return !booleanPointInPolygon(geom1, geom2);
        }
        /* istanbul ignore next */
        break;
    case 'LineString':
        switch (geom2.type) {
        case 'Point':
            return !isPointOnLine$1(geom1, geom2);
        case 'LineString':
            return !isLineOnLine$1(geom1, geom2);
        case 'Polygon':
            return !isLineInPoly$1(geom2, geom1);
        }
        /* istanbul ignore next */
        break;
    case 'Polygon':
        switch (geom2.type) {
        case 'Point':
            return !booleanPointInPolygon(geom2, geom1);
        case 'LineString':
            return !isLineInPoly$1(geom1, geom2);
        case 'Polygon':
            return !isPolyInPoly$1(geom2, geom1);
        }
    }
}

// http://stackoverflow.com/a/11908158/1979085
function isPointOnLine$1(lineString, point) {
    for (var i = 0; i < lineString.coordinates.length - 1; i++) {
        if (isPointOnLineSegment$2(lineString.coordinates[i], lineString.coordinates[i + 1], point.coordinates)) {
            return true;
        }
    }
    return false;
}

function isLineOnLine$1(lineString1, lineString2) {
    var doLinesIntersect = lineIntersect(lineString1, lineString2);
    if (doLinesIntersect.features.length > 0) {
        return true;
    }
    return false;
}

function isLineInPoly$1(polygon, lineString) {
    var doLinesIntersect = lineIntersect(lineString, polygonToLine(polygon));
    if (doLinesIntersect.features.length > 0) {
        return true;
    }
    return false;
}

/**
 * Is Polygon (geom1) in Polygon (geom2)
 * Only takes into account outer rings
 * See http://stackoverflow.com/a/4833823/1979085
 *
 * @private
 * @param {Geometry|Feature<Polygon>} feature1 Polygon1
 * @param {Geometry|Feature<Polygon>} feature2 Polygon2
 * @returns {boolean} true/false
 */
function isPolyInPoly$1(feature1, feature2) {
    for (var i = 0; i < feature1.coordinates[0].length; i++) {
        if (booleanPointInPolygon(feature1.coordinates[0][i], feature2)) {
            return true;
        }
    }
    for (var i2 = 0; i2 < feature2.coordinates[0].length; i2++) {
        if (booleanPointInPolygon(feature2.coordinates[0][i2], feature1)) {
            return true;
        }
    }
    return false;
}

function isPointOnLineSegment$2(LineSegmentStart, LineSegmentEnd, Point) {
    var dxc = Point[0] - LineSegmentStart[0];
    var dyc = Point[1] - LineSegmentStart[1];
    var dxl = LineSegmentEnd[0] - LineSegmentStart[0];
    var dyl = LineSegmentEnd[1] - LineSegmentStart[1];
    var cross = dxc * dyl - dyc * dxl;
    if (cross !== 0) {
        return false;
    }
    if (Math.abs(dxl) >= Math.abs(dyl)) {
        if (dxl > 0) {
            return LineSegmentStart[0] <= Point[0] && Point[0] <= LineSegmentEnd[0];
        } else {
            return LineSegmentEnd[0] <= Point[0] && Point[0] <= LineSegmentStart[0];
        }
    } else if (dyl > 0) {
        return LineSegmentStart[1] <= Point[1] && Point[1] <= LineSegmentEnd[1];
    } else {
        return LineSegmentEnd[1] <= Point[1] && Point[1] <= LineSegmentStart[1];
    }
}

/**
 * compareCoords
 *
 * @private
 * @param {Position} pair1 point [x,y]
 * @param {Position} pair2 point [x,y]
 * @returns {boolean} true/false if coord pairs match
 */
function compareCoords$1(pair1, pair2) {
    return pair1[0] === pair2[0] && pair1[1] === pair2[1];
}

/**
 * Boolean-contains returns True if the second geometry is completely contained by the first geometry.
 * The interiors of both geometries must intersect and, the interior and boundary of the secondary (geometry b)
 * must not intersect the exterior of the primary (geometry a).
 * Boolean-contains returns the exact opposite result of the `@turf/boolean-within`.
 *
 * @name booleanContains
 * @param {Geometry|Feature<any>} feature1 GeoJSON Feature or Geometry
 * @param {Geometry|Feature<any>} feature2 GeoJSON Feature or Geometry
 * @returns {boolean} true/false
 * @example
 * var line = turf.lineString([[1, 1], [1, 2], [1, 3], [1, 4]]);
 * var point = turf.point([1, 2]);
 *
 * turf.booleanContains(line, point);
 * //=true
 */
function booleanContains(feature1, feature2) {
    var type1 = getType(feature1);
    var type2 = getType(feature2);
    var geom1 = getGeom(feature1);
    var geom2 = getGeom(feature2);
    var coords1 = getCoords(feature1);
    var coords2 = getCoords(feature2);

    switch (type1) {
    case 'Point':
        switch (type2) {
        case 'Point':
            return compareCoords$2(coords1, coords2);
        default:
            throw new Error('feature2 ' + type2 + ' geometry not supported');
        }
    case 'MultiPoint':
        switch (type2) {
        case 'Point':
            return isPointInMultiPoint$1(geom1, geom2);
        case 'MultiPoint':
            return isMultiPointInMultiPoint$1(geom1, geom2);
        default:
            throw new Error('feature2 ' + type2 + ' geometry not supported');
        }
    case 'LineString':
        switch (type2) {
        case 'Point':
            return booleanPointOnLine(geom2, geom1, {ignoreEndVertices: true});
        case 'LineString':
            return isLineOnLine$2(geom1, geom2);
        case 'MultiPoint':
            return isMultiPointOnLine$1(geom1, geom2);
        default:
            throw new Error('feature2 ' + type2 + ' geometry not supported');
        }
    case 'Polygon':
        switch (type2) {
        case 'Point':
            return booleanPointInPolygon(geom2, geom1, {ignoreBoundary: true});
        case 'LineString':
            return isLineInPoly$2(geom1, geom2);
        case 'Polygon':
            return isPolyInPoly$2(geom1, geom2);
        case 'MultiPoint':
            return isMultiPointInPoly$1(geom1, geom2);
        default:
            throw new Error('feature2 ' + type2 + ' geometry not supported');
        }
    default:
        throw new Error('feature1 ' + type1 + ' geometry not supported');
    }
}

function isPointInMultiPoint$1(multiPoint, point) {
    var i;
    var output = false;
    for (i = 0; i < multiPoint.coordinates.length; i++) {
        if (compareCoords$2(multiPoint.coordinates[i], point.coordinates)) {
            output = true;
            break;
        }
    }
    return output;
}

function isMultiPointInMultiPoint$1(multiPoint1, multiPoint2) {
    for (var i = 0; i < multiPoint2.coordinates.length; i++) {
        var matchFound = false;
        for (var i2 = 0; i2 < multiPoint1.coordinates.length; i2++) {
            if (compareCoords$2(multiPoint2.coordinates[i], multiPoint1.coordinates[i2])) {
                matchFound = true;
                break;
            }
        }
        if (!matchFound) {
            return false;
        }
    }
    return true;
}


function isMultiPointOnLine$1(lineString, multiPoint) {
    var haveFoundInteriorPoint = false;
    for (var i = 0; i < multiPoint.coordinates.length; i++) {
        if (booleanPointOnLine(multiPoint.coordinates[i], lineString, {ignoreEndVertices: true})) {
            haveFoundInteriorPoint = true;
        }
        if (!booleanPointOnLine(multiPoint.coordinates[i], lineString)) {
            return false;
        }
    }
    if (haveFoundInteriorPoint) {
        return true;
    }
    return false;
}

function isMultiPointInPoly$1(polygon, multiPoint) {
    for (var i = 0; i < multiPoint.coordinates.length; i++) {
        if (!booleanPointInPolygon(multiPoint.coordinates[i], polygon, {ignoreBoundary: true})) {
            return false;
        }
    }
    return true;
}

function isLineOnLine$2(lineString1, lineString2) {
    var haveFoundInteriorPoint = false;
    for (var i = 0; i < lineString2.coordinates.length; i++) {
        if (booleanPointOnLine({type: 'Point', coordinates: lineString2.coordinates[i]}, lineString1, { ignoreEndVertices: true })) {
            haveFoundInteriorPoint = true;
        }
        if (!booleanPointOnLine({type: 'Point', coordinates: lineString2.coordinates[i]}, lineString1, {ignoreEndVertices: false })) {
            return false;
        }
    }
    return haveFoundInteriorPoint;
}

function isLineInPoly$2(polygon, linestring) {
    var output = false;
    var i = 0;

    var polyBbox = bbox(polygon);
    var lineBbox = bbox(linestring);
    if (!doBBoxOverlap$1(polyBbox, lineBbox)) {
        return false;
    }
    for (i; i < linestring.coordinates.length - 1; i++) {
        var midPoint = getMidpoint$1(linestring.coordinates[i], linestring.coordinates[i + 1]);
        if (booleanPointInPolygon({type: 'Point', coordinates: midPoint}, polygon, { ignoreBoundary: true })) {
            output = true;
            break;
        }
    }
    return output;
}

/**
 * Is Polygon2 in Polygon1
 * Only takes into account outer rings
 *
 * @private
 * @param {Geometry|Feature<Polygon>} feature1 Polygon1
 * @param {Geometry|Feature<Polygon>} feature2 Polygon2
 * @returns {boolean} true/false
 */
function isPolyInPoly$2(feature1, feature2) {
    var poly1Bbox = bbox(feature1);
    var poly2Bbox = bbox(feature2);
    if (!doBBoxOverlap$1(poly1Bbox, poly2Bbox)) {
        return false;
    }
    for (var i = 0; i < feature2.coordinates[0].length; i++) {
        if (!booleanPointInPolygon(feature2.coordinates[0][i], feature1)) {
            return false;
        }
    }
    return true;
}

function doBBoxOverlap$1(bbox1, bbox2) {
    if (bbox1[0] > bbox2[0]) return false;
    if (bbox1[2] < bbox2[2]) return false;
    if (bbox1[1] > bbox2[1]) return false;
    if (bbox1[3] < bbox2[3]) return false;
    return true;
}

/**
 * compareCoords
 *
 * @private
 * @param {Position} pair1 point [x,y]
 * @param {Position} pair2 point [x,y]
 * @returns {boolean} true/false if coord pairs match
 */
function compareCoords$2(pair1, pair2) {
    return pair1[0] === pair2[0] && pair1[1] === pair2[1];
}

function getMidpoint$1(pair1, pair2) {
    return [(pair1[0] + pair2[0]) / 2, (pair1[1] + pair2[1]) / 2];
}

/**
 * Boolean-Crosses returns True if the intersection results in a geometry whose dimension is one less than
 * the maximum dimension of the two source geometries and the intersection set is interior to
 * both source geometries.
 *
 * Boolean-Crosses returns t (TRUE) for only multipoint/polygon, multipoint/linestring, linestring/linestring, linestring/polygon, and linestring/multipolygon comparisons.
 *
 * @name booleanCrosses
 * @param {Geometry|Feature<any>} feature1 GeoJSON Feature or Geometry
 * @param {Geometry|Feature<any>} feature2 GeoJSON Feature or Geometry
 * @returns {boolean} true/false
 * @example
 * var line1 = turf.lineString([[-2, 2], [4, 2]]);
 * var line2 = turf.lineString([[1, 1], [1, 2], [1, 3], [1, 4]]);
 *
 * var cross = turf.booleanCrosses(line1, line2);
 * //=true
 */
function booleanCrosses(feature1, feature2) {
    var type1 = getType(feature1);
    var type2 = getType(feature2);
    var geom1 = getGeom(feature1);
    var geom2 = getGeom(feature2);

    switch (type1) {
    case 'MultiPoint':
        switch (type2) {
        case 'LineString':
            return doMultiPointAndLineStringCross(geom1, geom2);
        case 'Polygon':
            return doesMultiPointCrossPoly(geom1, geom2);
        default:
            throw new Error('feature2 ' + type2 + ' geometry not supported');
        }
    case 'LineString':
        switch (type2) {
        case 'MultiPoint': // An inverse operation
            return doMultiPointAndLineStringCross(geom2, geom1);
        case 'LineString':
            return doLineStringsCross(geom1, geom2);
        case 'Polygon':
            return doLineStringAndPolygonCross(geom1, geom2);
        default:
            throw new Error('feature2 ' + type2 + ' geometry not supported');
        }
    case 'Polygon':
        switch (type2) {
        case 'MultiPoint': // An inverse operation
            return doesMultiPointCrossPoly(geom2, geom1);
        case 'LineString': // An inverse operation
            return doLineStringAndPolygonCross(geom2, geom1);
        default:
            throw new Error('feature2 ' + type2 + ' geometry not supported');
        }
    default:
        throw new Error('feature1 ' + type1 + ' geometry not supported');
    }
}

function doMultiPointAndLineStringCross(multiPoint$$1, lineString$$1) {
    var foundIntPoint = false;
    var foundExtPoint = false;
    var pointLength = multiPoint$$1.coordinates.length;
    var i = 0;
    while (i < pointLength && !foundIntPoint && !foundExtPoint) {
        for (var i2 = 0; i2 < lineString$$1.coordinates.length - 1; i2++) {
            var incEndVertices = true;
            if (i2 === 0 || i2 === lineString$$1.coordinates.length - 2) {
                incEndVertices = false;
            }
            if (isPointOnLineSegment$3(lineString$$1.coordinates[i2], lineString$$1.coordinates[i2 + 1], multiPoint$$1.coordinates[i], incEndVertices)) {
                foundIntPoint = true;
            } else {
                foundExtPoint = true;
            }
        }
        i++;
    }
    return foundIntPoint && foundExtPoint;
}

function doLineStringsCross(lineString1, lineString2) {
    var doLinesIntersect = lineIntersect(lineString1, lineString2);
    if (doLinesIntersect.features.length > 0) {
        for (var i = 0; i < lineString1.coordinates.length - 1; i++) {
            for (var i2 = 0; i2 < lineString2.coordinates.length - 1; i2++) {
                var incEndVertices = true;
                if (i2 === 0 || i2 === lineString2.coordinates.length - 2) {
                    incEndVertices = false;
                }
                if (isPointOnLineSegment$3(lineString1.coordinates[i], lineString1.coordinates[i + 1], lineString2.coordinates[i2], incEndVertices)) {
                    return true;
                }
            }
        }
    }
    return false;
}

function doLineStringAndPolygonCross(lineString$$1, polygon$$1) {
    var doLinesIntersect = lineIntersect(lineString$$1, polygonToLine(polygon$$1));
    if (doLinesIntersect.features.length > 0) {
        return true;
    }
    return false;
}

function doesMultiPointCrossPoly(multiPoint$$1, polygon$$1) {
    var foundIntPoint = false;
    var foundExtPoint = false;
    var pointLength = multiPoint$$1.coordinates[0].length;
    var i = 0;
    while (i < pointLength && foundIntPoint && foundExtPoint) {
        if (booleanPointInPolygon(point(multiPoint$$1.coordinates[0][i]), polygon$$1)) {
            foundIntPoint = true;
        } else {
            foundExtPoint = true;
        }
        i++;
    }

    return foundExtPoint && foundExtPoint;
}

/**
 * Is a point on a line segment
 * Only takes into account outer rings
 * See http://stackoverflow.com/a/4833823/1979085
 *
 * @private
 * @param {number[]} lineSegmentStart coord pair of start of line
 * @param {number[]} lineSegmentEnd coord pair of end of line
 * @param {number[]} pt coord pair of point to check
 * @param {boolean} incEnd whether the point is allowed to fall on the line ends
 * @returns {boolean} true/false
 */
function isPointOnLineSegment$3(lineSegmentStart, lineSegmentEnd, pt, incEnd) {
    var dxc = pt[0] - lineSegmentStart[0];
    var dyc = pt[1] - lineSegmentStart[1];
    var dxl = lineSegmentEnd[0] - lineSegmentStart[0];
    var dyl = lineSegmentEnd[1] - lineSegmentStart[1];
    var cross = dxc * dyl - dyc * dxl;
    if (cross !== 0) {
        return false;
    }
    if (incEnd) {
        if (Math.abs(dxl) >= Math.abs(dyl)) {
            return dxl > 0 ? lineSegmentStart[0] <= pt[0] && pt[0] <= lineSegmentEnd[0] : lineSegmentEnd[0] <= pt[0] && pt[0] <= lineSegmentStart[0];
        }
        return dyl > 0 ? lineSegmentStart[1] <= pt[1] && pt[1] <= lineSegmentEnd[1] : lineSegmentEnd[1] <= pt[1] && pt[1] <= lineSegmentStart[1];
    } else {
        if (Math.abs(dxl) >= Math.abs(dyl)) {
            return dxl > 0 ? lineSegmentStart[0] < pt[0] && pt[0] < lineSegmentEnd[0] : lineSegmentEnd[0] < pt[0] && pt[0] < lineSegmentStart[0];
        }
        return dyl > 0 ? lineSegmentStart[1] < pt[1] && pt[1] < lineSegmentEnd[1] : lineSegmentEnd[1] < pt[1] && pt[1] < lineSegmentStart[1];
    }
}

var keys = createCommonjsModule(function (module, exports) {
exports = module.exports = typeof Object.keys === 'function'
  ? Object.keys : shim;

exports.shim = shim;
function shim (obj) {
  var keys = [];
  for (var key in obj) keys.push(key);
  return keys;
}
});

var keys_1 = keys.shim;

var is_arguments = createCommonjsModule(function (module, exports) {
var supportsArgumentsClass = (function(){
  return Object.prototype.toString.call(arguments)
})() == '[object Arguments]';

exports = module.exports = supportsArgumentsClass ? supported : unsupported;

exports.supported = supported;
function supported(object) {
  return Object.prototype.toString.call(object) == '[object Arguments]';
}

exports.unsupported = unsupported;
function unsupported(object){
  return object &&
    typeof object == 'object' &&
    typeof object.length == 'number' &&
    Object.prototype.hasOwnProperty.call(object, 'callee') &&
    !Object.prototype.propertyIsEnumerable.call(object, 'callee') ||
    false;
}
});

var is_arguments_1 = is_arguments.supported;
var is_arguments_2 = is_arguments.unsupported;

var deepEqual_1 = createCommonjsModule(function (module) {
var pSlice = Array.prototype.slice;



var deepEqual = module.exports = function (actual, expected, opts) {
  if (!opts) opts = {};
  // 7.1. All identical values are equivalent, as determined by ===.
  if (actual === expected) {
    return true;

  } else if (actual instanceof Date && expected instanceof Date) {
    return actual.getTime() === expected.getTime();

  // 7.3. Other pairs that do not both pass typeof value == 'object',
  // equivalence is determined by ==.
  } else if (!actual || !expected || typeof actual != 'object' && typeof expected != 'object') {
    return opts.strict ? actual === expected : actual == expected;

  // 7.4. For all other Object pairs, including Array objects, equivalence is
  // determined by having the same number of owned properties (as verified
  // with Object.prototype.hasOwnProperty.call), the same set of keys
  // (although not necessarily the same order), equivalent values for every
  // corresponding key, and an identical 'prototype' property. Note: this
  // accounts for both named and indexed properties on Arrays.
  } else {
    return objEquiv(actual, expected, opts);
  }
};

function isUndefinedOrNull(value) {
  return value === null || value === undefined;
}

function isBuffer (x) {
  if (!x || typeof x !== 'object' || typeof x.length !== 'number') return false;
  if (typeof x.copy !== 'function' || typeof x.slice !== 'function') {
    return false;
  }
  if (x.length > 0 && typeof x[0] !== 'number') return false;
  return true;
}

function objEquiv(a, b, opts) {
  var i, key;
  if (isUndefinedOrNull(a) || isUndefinedOrNull(b))
    return false;
  // an identical 'prototype' property.
  if (a.prototype !== b.prototype) return false;
  //~~~I've managed to break Object.keys through screwy arguments passing.
  //   Converting to array solves the problem.
  if (is_arguments(a)) {
    if (!is_arguments(b)) {
      return false;
    }
    a = pSlice.call(a);
    b = pSlice.call(b);
    return deepEqual(a, b, opts);
  }
  if (isBuffer(a)) {
    if (!isBuffer(b)) {
      return false;
    }
    if (a.length !== b.length) return false;
    for (i = 0; i < a.length; i++) {
      if (a[i] !== b[i]) return false;
    }
    return true;
  }
  try {
    var ka = keys(a),
        kb = keys(b);
  } catch (e) {//happens when one is a string literal and the other isn't
    return false;
  }
  // having the same number of owned properties (keys incorporates
  // hasOwnProperty)
  if (ka.length != kb.length)
    return false;
  //the same set of keys (although not necessarily the same order),
  ka.sort();
  kb.sort();
  //~~~cheap key test
  for (i = ka.length - 1; i >= 0; i--) {
    if (ka[i] != kb[i])
      return false;
  }
  //equivalent values for every corresponding key, and
  //~~~possibly expensive deep test
  for (i = ka.length - 1; i >= 0; i--) {
    key = ka[i];
    if (!deepEqual(a[key], b[key], opts)) return false;
  }
  return typeof a === typeof b;
}
});

//index.js


var Equality = function(opt) {
  this.precision = opt && opt.precision ? opt.precision : 17;
  this.direction = opt && opt.direction ? opt.direction : false;
  this.pseudoNode = opt && opt.pseudoNode ? opt.pseudoNode : false;
  this.objectComparator = opt && opt.objectComparator ? opt.objectComparator : objectComparator;
};

Equality.prototype.compare = function(g1,g2) {
  if (g1.type !== g2.type || !sameLength(g1,g2)) return false;

  switch(g1.type) {
  case 'Point':
    return this.compareCoord(g1.coordinates, g2.coordinates);
    break;
  case 'LineString':
    return this.compareLine(g1.coordinates, g2.coordinates,0,false);
    break;
  case 'Polygon':
    return this.comparePolygon(g1,g2);
    break;
  case 'Feature':
    return this.compareFeature(g1, g2);
  default:
    if (g1.type.indexOf('Multi') === 0) {
      var context = this;
      var g1s = explode$2(g1);
      var g2s = explode$2(g2);
      return g1s.every(function(g1part) {
        return this.some(function(g2part) {
          return context.compare(g1part,g2part);
        });
      },g2s);
    }
  }
  return false;
};

function explode$2(g) {
  return g.coordinates.map(function(part) {
    return {
      type: g.type.replace('Multi', ''),
      coordinates: part}
  });
}
//compare length of coordinates/array
function sameLength(g1,g2) {
   return g1.hasOwnProperty('coordinates') ?
    g1.coordinates.length === g2.coordinates.length
    : g1.length === g2.length;
}

// compare the two coordinates [x,y]
Equality.prototype.compareCoord = function(c1,c2) {
  if (c1.length !== c2.length) {
    return false;
  }

  for (var i=0; i < c1.length; i++) {
    if (c1[i].toFixed(this.precision) !== c2[i].toFixed(this.precision)) {
      return false;
    }
  }
  return true;
};

Equality.prototype.compareLine = function(path1,path2,ind,isPoly) {
  if (!sameLength(path1,path2)) return false;
  var p1 = this.pseudoNode ? path1 : this.removePseudo(path1);
  var p2 = this.pseudoNode ? path2 : this.removePseudo(path2);
  if (isPoly && !this.compareCoord(p1[0],p2[0])) {
    // fix start index of both to same point
    p2 = this.fixStartIndex(p2,p1);
    if(!p2) return;
  }
  // for linestring ind =0 and for polygon ind =1
  var sameDirection = this.compareCoord(p1[ind],p2[ind]);
  if (this.direction || sameDirection
  ) {
    return this.comparePath(p1, p2);
  } else {
    if (this.compareCoord(p1[ind],p2[p2.length - (1+ind)])
    ) {
      return this.comparePath(p1.slice().reverse(), p2);
    }
    return false;
  }
};
Equality.prototype.fixStartIndex = function(sourcePath,targetPath) {
  //make sourcePath first point same as of targetPath
  var correctPath,ind = -1;
  for (var i=0; i< sourcePath.length; i++) {
    if(this.compareCoord(sourcePath[i],targetPath[0])) {
      ind = i;
      break;
    }
  }
  if (ind >= 0) {
    correctPath = [].concat(
      sourcePath.slice(ind,sourcePath.length),
      sourcePath.slice(1,ind+1));
  }
  return correctPath;
};
Equality.prototype.comparePath = function (p1,p2) {
  var cont = this;
  return p1.every(function(c,i) {
    return cont.compareCoord(c,this[i]);
  },p2);
};

Equality.prototype.comparePolygon = function(g1,g2) {
  if (this.compareLine(g1.coordinates[0],g2.coordinates[0],1,true)) {
    var holes1 = g1.coordinates.slice(1,g1.coordinates.length);
    var holes2 = g2.coordinates.slice(1,g2.coordinates.length);
    var cont = this;
    return holes1.every(function(h1) {
      return this.some(function(h2) {
        return cont.compareLine(h1,h2,1,true);
      });
    },holes2);
  } else {
    return false;
  }
};

Equality.prototype.compareFeature = function(g1,g2) {
  if (
    g1.id !== g2.id ||
    !this.objectComparator(g1.properties, g2.properties) ||
    !this.compareBBox(g1,g2)
  ) {
    return false;
  }
  return this.compare(g1.geometry, g2.geometry);
};

Equality.prototype.compareBBox = function(g1,g2) {
  if (
    (!g1.bbox && !g2.bbox) || 
    (
      g1.bbox && g2.bbox &&
      this.compareCoord(g1.bbox, g2.bbox)
    )
  )  {
    return true;
  }
  return false;
};
Equality.prototype.removePseudo = function(path) {
  //TODO to be implement
  return path;
};

function objectComparator(obj1, obj2) {
  return deepEqual_1(obj1, obj2, {strict: true});
}

var geojsonEquality = Equality;

/**
 * Compares two geometries of the same dimension and returns true if their intersection set results in a geometry
 * different from both but of the same dimension. It applies to Polygon/Polygon, LineString/LineString,
 * Multipoint/Multipoint, MultiLineString/MultiLineString and MultiPolygon/MultiPolygon.
 *
 * @name booleanOverlap
 * @param  {Geometry|Feature<LineString|MultiLineString|Polygon|MultiPolygon>} feature1 input
 * @param  {Geometry|Feature<LineString|MultiLineString|Polygon|MultiPolygon>} feature2 input
 * @returns {boolean} true/false
 * @example
 * var poly1 = turf.polygon([[[0,0],[0,5],[5,5],[5,0],[0,0]]]);
 * var poly2 = turf.polygon([[[1,1],[1,6],[6,6],[6,1],[1,1]]]);
 * var poly3 = turf.polygon([[[10,10],[10,15],[15,15],[15,10],[10,10]]]);
 *
 * turf.booleanOverlap(poly1, poly2)
 * //=true
 * turf.booleanOverlap(poly2, poly3)
 * //=false
 */
function booleanOverlap(feature1, feature2) {
    // validation
    if (!feature1) throw new Error('feature1 is required');
    if (!feature2) throw new Error('feature2 is required');
    var type1 = getType(feature1);
    var type2 = getType(feature2);
    if (type1 !== type2) throw new Error('features must be of the same type');
    if (type1 === 'Point') throw new Error('Point geometry not supported');

    // features must be not equal
    var equality = new geojsonEquality({precision: 6});
    if (equality.compare(feature1, feature2)) return false;

    var overlap = 0;

    switch (type1) {
    case 'MultiPoint':
        var coords1 = coordAll(feature1);
        var coords2 = coordAll(feature2);
        coords1.forEach(function (coord1) {
            coords2.forEach(function (coord2) {
                if (coord1[0] === coord2[0] && coord1[1] === coord2[1]) overlap++;
            });
        });
        break;

    case 'LineString':
    case 'MultiLineString':
        segmentEach(feature1, function (segment1) {
            segmentEach(feature2, function (segment2) {
                if (lineOverlap(segment1, segment2).features.length) overlap++;
            });
        });
        break;

    case 'Polygon':
    case 'MultiPolygon':
        segmentEach(feature1, function (segment1) {
            segmentEach(feature2, function (segment2) {
                if (lineIntersect(segment1, segment2).features.length) overlap++;
            });
        });
        break;
    }

    return overlap > 0;
}

/**
 * Determine whether two geometries of the same type have identical X,Y coordinate values.
 * See http://edndoc.esri.com/arcsde/9.0/general_topics/understand_spatial_relations.htm
 *
 * @name booleanEqual
 * @param {Geometry|Feature} feature1 GeoJSON input
 * @param {Geometry|Feature} feature2 GeoJSON input
 * @returns {boolean} true if the objects are equal, false otherwise
 * @example
 * var pt1 = turf.point([0, 0]);
 * var pt2 = turf.point([0, 0]);
 * var pt3 = turf.point([1, 1]);
 *
 * turf.booleanEqual(pt1, pt2);
 * //= true
 * turf.booleanEqual(pt2, pt3);
 * //= false
 */
function booleanEqual(feature1, feature2) {
    // validation
    if (!feature1) throw new Error('feature1 is required');
    if (!feature2) throw new Error('feature2 is required');
    var type1 = getType(feature1);
    var type2 = getType(feature2);
    if (type1 !== type2) return false;

    var equality = new geojsonEquality({precision: 6});
    return equality.compare(cleanCoords(feature1), cleanCoords(feature2));
}

var DBSCAN_1 = createCommonjsModule(function (module) {
/**
 * DBSCAN - Density based clustering
 *
 * @author Lukasz Krawczyk <contact@lukaszkrawczyk.eu>
 * @copyright MIT
 */

/**
 * DBSCAN class construcotr
 * @constructor
 *
 * @param {Array} dataset
 * @param {number} epsilon
 * @param {number} minPts
 * @param {function} distanceFunction
 * @returns {DBSCAN}
 */
function DBSCAN(dataset, epsilon, minPts, distanceFunction) {
  /** @type {Array} */
  this.dataset = [];
  /** @type {number} */
  this.epsilon = 1;
  /** @type {number} */
  this.minPts = 2;
  /** @type {function} */
  this.distance = this._euclideanDistance;
  /** @type {Array} */
  this.clusters = [];
  /** @type {Array} */
  this.noise = [];

  // temporary variables used during computation

  /** @type {Array} */
  this._visited = [];
  /** @type {Array} */
  this._assigned = [];
  /** @type {number} */
  this._datasetLength = 0;

  this._init(dataset, epsilon, minPts, distanceFunction);
}

/******************************************************************************/
// public functions

/**
 * Start clustering
 *
 * @param {Array} dataset
 * @param {number} epsilon
 * @param {number} minPts
 * @param {function} distanceFunction
 * @returns {undefined}
 * @access public
 */
DBSCAN.prototype.run = function(dataset, epsilon, minPts, distanceFunction) {
  this._init(dataset, epsilon, minPts, distanceFunction);

  for (var pointId = 0; pointId < this._datasetLength; pointId++) {
    // if point is not visited, check if it forms a cluster
    if (this._visited[pointId] !== 1) {
      this._visited[pointId] = 1;

      // if closest neighborhood is too small to form a cluster, mark as noise
      var neighbors = this._regionQuery(pointId);

      if (neighbors.length < this.minPts) {
        this.noise.push(pointId);
      } else {
        // create new cluster and add point
        var clusterId = this.clusters.length;
        this.clusters.push([]);
        this._addToCluster(pointId, clusterId);

        this._expandCluster(clusterId, neighbors);
      }
    }
  }

  return this.clusters;
};

/******************************************************************************/
// protected functions

/**
 * Set object properties
 *
 * @param {Array} dataset
 * @param {number} epsilon
 * @param {number} minPts
 * @param {function} distance
 * @returns {undefined}
 * @access protected
 */
DBSCAN.prototype._init = function(dataset, epsilon, minPts, distance) {

  if (dataset) {

    if (!(dataset instanceof Array)) {
      throw Error('Dataset must be of type array, ' +
        typeof dataset + ' given');
    }

    this.dataset = dataset;
    this.clusters = [];
    this.noise = [];

    this._datasetLength = dataset.length;
    this._visited = new Array(this._datasetLength);
    this._assigned = new Array(this._datasetLength);
  }

  if (epsilon) {
    this.epsilon = epsilon;
  }

  if (minPts) {
    this.minPts = minPts;
  }

  if (distance) {
    this.distance = distance;
  }
};

/**
 * Expand cluster to closest points of given neighborhood
 *
 * @param {number} clusterId
 * @param {Array} neighbors
 * @returns {undefined}
 * @access protected
 */
DBSCAN.prototype._expandCluster = function(clusterId, neighbors) {

  /**
   * It's very important to calculate length of neighbors array each time,
   * as the number of elements changes over time
   */
  for (var i = 0; i < neighbors.length; i++) {
    var pointId2 = neighbors[i];

    if (this._visited[pointId2] !== 1) {
      this._visited[pointId2] = 1;
      var neighbors2 = this._regionQuery(pointId2);

      if (neighbors2.length >= this.minPts) {
        neighbors = this._mergeArrays(neighbors, neighbors2);
      }
    }

    // add to cluster
    if (this._assigned[pointId2] !== 1) {
      this._addToCluster(pointId2, clusterId);
    }
  }
};

/**
 * Add new point to cluster
 *
 * @param {number} pointId
 * @param {number} clusterId
 */
DBSCAN.prototype._addToCluster = function(pointId, clusterId) {
  this.clusters[clusterId].push(pointId);
  this._assigned[pointId] = 1;
};

/**
 * Find all neighbors around given point
 *
 * @param {number} pointId,
 * @param {number} epsilon
 * @returns {Array}
 * @access protected
 */
DBSCAN.prototype._regionQuery = function(pointId) {
  var neighbors = [];

  for (var id = 0; id < this._datasetLength; id++) {
    var dist = this.distance(this.dataset[pointId], this.dataset[id]);
    if (dist < this.epsilon) {
      neighbors.push(id);
    }
  }

  return neighbors;
};

/******************************************************************************/
// helpers

/**
 * @param {Array} a
 * @param {Array} b
 * @returns {Array}
 * @access protected
 */
DBSCAN.prototype._mergeArrays = function(a, b) {
  var len = b.length;

  for (var i = 0; i < len; i++) {
    var P = b[i];
    if (a.indexOf(P) < 0) {
      a.push(P);
    }
  }

  return a;
};

/**
 * Calculate euclidean distance in multidimensional space
 *
 * @param {Array} p
 * @param {Array} q
 * @returns {number}
 * @access protected
 */
DBSCAN.prototype._euclideanDistance = function(p, q) {
  var sum = 0;
  var i = Math.min(p.length, q.length);

  while (i--) {
    sum += (p[i] - q[i]) * (p[i] - q[i]);
  }

  return Math.sqrt(sum);
};

if ('object' !== 'undefined' && module.exports) {
  module.exports = DBSCAN;
}
});

var KMEANS_1 = createCommonjsModule(function (module) {
/**
 * KMEANS clustering
 *
 * @author Lukasz Krawczyk <contact@lukaszkrawczyk.eu>
 * @copyright MIT
 */

/**
 * KMEANS class constructor
 * @constructor
 *
 * @param {Array} dataset
 * @param {number} k - number of clusters
 * @param {function} distance - distance function
 * @returns {KMEANS}
 */
 function KMEANS(dataset, k, distance) {
  this.k = 3; // number of clusters
  this.dataset = []; // set of feature vectors
  this.assignments = []; // set of associated clusters for each feature vector
  this.centroids = []; // vectors for our clusters

  this.init(dataset, k, distance);
}

/**
 * @returns {undefined}
 */
KMEANS.prototype.init = function(dataset, k, distance) {
  this.assignments = [];
  this.centroids = [];

  if (typeof dataset !== 'undefined') {
    this.dataset = dataset;
  }

  if (typeof k !== 'undefined') {
    this.k = k;
  }

  if (typeof distance !== 'undefined') {
    this.distance = distance;
  }
};

/**
 * @returns {undefined}
 */
KMEANS.prototype.run = function(dataset, k) {
  this.init(dataset, k);

  var len = this.dataset.length;

  // initialize centroids
  for (var i = 0; i < this.k; i++) {
    this.centroids[i] = this.randomCentroid();
	}

  var change = true;
  while(change) {

    // assign feature vectors to clusters
    change = this.assign();

    // adjust location of centroids
    for (var centroidId = 0; centroidId < this.k; centroidId++) {
      var mean = new Array(maxDim);
      var count = 0;

      // init mean vector
      for (var dim = 0; dim < maxDim; dim++) {
        mean[dim] = 0;
      }

      for (var j = 0; j < len; j++) {
        var maxDim = this.dataset[j].length;

        // if current cluster id is assigned to point
        if (centroidId === this.assignments[j]) {
          for (var dim = 0; dim < maxDim; dim++) {
            mean[dim] += this.dataset[j][dim];
          }
          count++;
        }
      }

      if (count > 0) {
        // if cluster contain points, adjust centroid position
        for (var dim = 0; dim < maxDim; dim++) {
          mean[dim] /= count;
        }
        this.centroids[centroidId] = mean;
      } else {
        // if cluster is empty, generate new random centroid
        this.centroids[centroidId] = this.randomCentroid();
        change = true;
      }
    }
  }

  return this.getClusters();
};

/**
 * Generate random centroid
 *
 * @returns {Array}
 */
KMEANS.prototype.randomCentroid = function() {
  var maxId = this.dataset.length -1;
  var centroid;
  var id;

  do {
    id = Math.round(Math.random() * maxId);
    centroid = this.dataset[id];
  } while (this.centroids.indexOf(centroid) >= 0);

  return centroid;
};

/**
 * Assign points to clusters
 *
 * @returns {boolean}
 */
KMEANS.prototype.assign = function() {
  var change = false;
  var len = this.dataset.length;
  var closestCentroid;

  for (var i = 0; i < len; i++) {
    closestCentroid = this.argmin(this.dataset[i], this.centroids, this.distance);

    if (closestCentroid != this.assignments[i]) {
      this.assignments[i] = closestCentroid;
      change = true;
    }
  }

  return change;
};

/**
 * Extract information about clusters
 *
 * @returns {undefined}
 */
KMEANS.prototype.getClusters = function() {
  var clusters = new Array(this.k);
  var centroidId;

  for (var pointId = 0; pointId < this.assignments.length; pointId++) {
    centroidId = this.assignments[pointId];

    // init empty cluster
    if (typeof clusters[centroidId] === 'undefined') {
      clusters[centroidId] = [];
    }

    clusters[centroidId].push(pointId);
  }

  return clusters;
};

// utils

/**
 * @params {Array} point
 * @params {Array.<Array>} set
 * @params {Function} f
 * @returns {number}
 */
KMEANS.prototype.argmin = function(point, set, f) {
  var min = Number.MAX_VALUE;
  var arg = 0;
  var len = set.length;
  var d;

  for (var i = 0; i < len; i++) {
    d = f(point, set[i]);
    if (d < min) {
      min = d;
      arg = i;
    }
  }

  return arg;
};

/**
 * Euclidean distance
 *
 * @params {number} p
 * @params {number} q
 * @returns {number}
 */
KMEANS.prototype.distance = function(p, q) {
  var sum = 0;
  var i = Math.min(p.length, q.length);

  while (i--) {
    var diff = p[i] - q[i];
    sum += diff * diff;
  }

  return Math.sqrt(sum);
};

if ('object' !== 'undefined' && module.exports) {
  module.exports = KMEANS;
}
});

var PriorityQueue_1 = createCommonjsModule(function (module) {
/**
 * PriorityQueue
 * Elements in this queue are sorted according to their value
 *
 * @author Lukasz Krawczyk <contact@lukaszkrawczyk.eu>
 * @copyright MIT
 */

/**
 * PriorityQueue class construcotr
 * @constructor
 *
 * @example
 * queue: [1,2,3,4]
 * priorities: [4,1,2,3]
 * > result = [1,4,2,3]
 *
 * @param {Array} elements
 * @param {Array} priorities
 * @param {string} sorting - asc / desc
 * @returns {PriorityQueue}
 */
function PriorityQueue(elements, priorities, sorting) {
  /** @type {Array} */
  this._queue = [];
  /** @type {Array} */
  this._priorities = [];
  /** @type {string} */
  this._sorting = 'desc';

  this._init(elements, priorities, sorting);
}

/**
 * Insert element
 *
 * @param {Object} ele
 * @param {Object} priority
 * @returns {undefined}
 * @access public
 */
PriorityQueue.prototype.insert = function(ele, priority) {
  var indexToInsert = this._queue.length;
  var index = indexToInsert;

  while (index--) {
    var priority2 = this._priorities[index];
    if (this._sorting === 'desc') {
      if (priority > priority2) {
        indexToInsert = index;
      }
    } else {
      if (priority < priority2) {
        indexToInsert = index;
      }
    }
  }

  this._insertAt(ele, priority, indexToInsert);
};

/**
 * Remove element
 *
 * @param {Object} ele
 * @returns {undefined}
 * @access public
 */
PriorityQueue.prototype.remove = function(ele) {
  var index = this._queue.length;

  while (index--) {
    var ele2 = this._queue[index];
    if (ele === ele2) {
      this._queue.splice(index, 1);
      this._priorities.splice(index, 1);
      break;
    }
  }
};

/**
 * For each loop wrapper
 *
 * @param {function} func
 * @returs {undefined}
 * @access public
 */
PriorityQueue.prototype.forEach = function(func) {
  this._queue.forEach(func);
};

/**
 * @returns {Array}
 * @access public
 */
PriorityQueue.prototype.getElements = function() {
  return this._queue;
};

/**
 * @param {number} index
 * @returns {Object}
 * @access public
 */
PriorityQueue.prototype.getElementPriority = function(index) {
  return this._priorities[index];
};

/**
 * @returns {Array}
 * @access public
 */
PriorityQueue.prototype.getPriorities = function() {
  return this._priorities;
};

/**
 * @returns {Array}
 * @access public
 */
PriorityQueue.prototype.getElementsWithPriorities = function() {
  var result = [];

  for (var i = 0, l = this._queue.length; i < l; i++) {
    result.push([this._queue[i], this._priorities[i]]);
  }

  return result;
};

/**
 * Set object properties
 *
 * @param {Array} elements
 * @param {Array} priorities
 * @returns {undefined}
 * @access protected
 */
PriorityQueue.prototype._init = function(elements, priorities, sorting) {

  if (elements && priorities) {
    this._queue = [];
    this._priorities = [];

    if (elements.length !== priorities.length) {
      throw new Error('Arrays must have the same length');
    }

    for (var i = 0; i < elements.length; i++) {
      this.insert(elements[i], priorities[i]);
    }
  }

  if (sorting) {
    this._sorting = sorting;
  }
};

/**
 * Insert element at given position
 *
 * @param {Object} ele
 * @param {number} index
 * @returns {undefined}
 * @access protected
 */
PriorityQueue.prototype._insertAt = function(ele, priority, index) {
  if (this._queue.length === index) {
    this._queue.push(ele);
    this._priorities.push(priority);
  } else {
    this._queue.splice(index, 0, ele);
    this._priorities.splice(index, 0, priority);
  }
};

if ('object' !== 'undefined' && module.exports) {
  module.exports = PriorityQueue;
}
});

var OPTICS_1 = createCommonjsModule(function (module) {
/**
 * @requires ./PriorityQueue.js
 */

if ('object' !== 'undefined' && module.exports) {
      var PriorityQueue = PriorityQueue_1;
}

/**
 * OPTICS - Ordering points to identify the clustering structure
 *
 * @author Lukasz Krawczyk <contact@lukaszkrawczyk.eu>
 * @copyright MIT
 */

/**
 * OPTICS class constructor
 * @constructor
 *
 * @param {Array} dataset
 * @param {number} epsilon
 * @param {number} minPts
 * @param {function} distanceFunction
 * @returns {OPTICS}
 */
function OPTICS(dataset, epsilon, minPts, distanceFunction) {
  /** @type {number} */
  this.epsilon = 1;
  /** @type {number} */
  this.minPts = 1;
  /** @type {function} */
  this.distance = this._euclideanDistance;

  // temporary variables used during computation

  /** @type {Array} */
  this._reachability = [];
  /** @type {Array} */
  this._processed = [];
  /** @type {number} */
  this._coreDistance = 0;
  /** @type {Array} */
  this._orderedList = [];

  this._init(dataset, epsilon, minPts, distanceFunction);
}

/******************************************************************************/
// pulic functions

/**
 * Start clustering
 *
 * @param {Array} dataset
 * @returns {undefined}
 * @access public
 */
OPTICS.prototype.run = function(dataset, epsilon, minPts, distanceFunction) {
  this._init(dataset, epsilon, minPts, distanceFunction);

  for (var pointId = 0, l = this.dataset.length; pointId < l; pointId++) {
    if (this._processed[pointId] !== 1) {
      this._processed[pointId] = 1;
      this.clusters.push([pointId]);
      var clusterId = this.clusters.length - 1;

      this._orderedList.push(pointId);
      var priorityQueue = new PriorityQueue(null, null, 'asc');
      var neighbors = this._regionQuery(pointId);

      // using priority queue assign elements to new cluster
      if (this._distanceToCore(pointId) !== undefined) {
        this._updateQueue(pointId, neighbors, priorityQueue);
        this._expandCluster(clusterId, priorityQueue);
      }
    }
  }

  return this.clusters;
};

/**
 * Generate reachability plot for all points
 *
 * @returns {array}
 * @access public
 */
OPTICS.prototype.getReachabilityPlot = function() {
  var reachabilityPlot = [];

  for (var i = 0, l = this._orderedList.length; i < l; i++) {
    var pointId = this._orderedList[i];
    var distance = this._reachability[pointId];

    reachabilityPlot.push([pointId, distance]);
  }

  return reachabilityPlot;
};

/******************************************************************************/
// protected functions

/**
 * Set object properties
 *
 * @param {Array} dataset
 * @param {number} epsilon
 * @param {number} minPts
 * @param {function} distance
 * @returns {undefined}
 * @access protected
 */
OPTICS.prototype._init = function(dataset, epsilon, minPts, distance) {

  if (dataset) {

    if (!(dataset instanceof Array)) {
      throw Error('Dataset must be of type array, ' +
        typeof dataset + ' given');
    }

    this.dataset = dataset;
    this.clusters = [];
    this._reachability = new Array(this.dataset.length);
    this._processed = new Array(this.dataset.length);
    this._coreDistance = 0;
    this._orderedList = [];
  }

  if (epsilon) {
    this.epsilon = epsilon;
  }

  if (minPts) {
    this.minPts = minPts;
  }

  if (distance) {
    this.distance = distance;
  }
};

/**
 * Update information in queue
 *
 * @param {number} pointId
 * @param {Array} neighbors
 * @param {PriorityQueue} queue
 * @returns {undefined}
 * @access protected
 */
OPTICS.prototype._updateQueue = function(pointId, neighbors, queue) {
  var self = this;

  this._coreDistance = this._distanceToCore(pointId);
  neighbors.forEach(function(pointId2) {
    if (self._processed[pointId2] === undefined) {
      var dist = self.distance(self.dataset[pointId], self.dataset[pointId2]);
      var newReachableDistance = Math.max(self._coreDistance, dist);

      if (self._reachability[pointId2] === undefined) {
        self._reachability[pointId2] = newReachableDistance;
        queue.insert(pointId2, newReachableDistance);
      } else {
        if (newReachableDistance < self._reachability[pointId2]) {
          self._reachability[pointId2] = newReachableDistance;
          queue.remove(pointId2);
          queue.insert(pointId2, newReachableDistance);
        }
      }
    }
  });
};

/**
 * Expand cluster
 *
 * @param {number} clusterId
 * @param {PriorityQueue} queue
 * @returns {undefined}
 * @access protected
 */
OPTICS.prototype._expandCluster = function(clusterId, queue) {
  var queueElements = queue.getElements();

  for (var p = 0, l = queueElements.length; p < l; p++) {
    var pointId = queueElements[p];
    if (this._processed[pointId] === undefined) {
      var neighbors = this._regionQuery(pointId);
      this._processed[pointId] = 1;

      this.clusters[clusterId].push(pointId);
      this._orderedList.push(pointId);

      if (this._distanceToCore(pointId) !== undefined) {
        this._updateQueue(pointId, neighbors, queue);
        this._expandCluster(clusterId, queue);
      }
    }
  }
};

/**
 * Calculating distance to cluster core
 *
 * @param {number} pointId
 * @returns {number}
 * @access protected
 */
OPTICS.prototype._distanceToCore = function(pointId) {
  var l = this.epsilon;
  for (var coreDistCand = 0; coreDistCand < l; coreDistCand++) {
    var neighbors = this._regionQuery(pointId, coreDistCand);
    if (neighbors.length >= this.minPts) {
      return coreDistCand;
    }
  }

  return;
};

/**
 * Find all neighbors around given point
 *
 * @param {number} pointId
 * @param {number} epsilon
 * @returns {Array}
 * @access protected
 */
OPTICS.prototype._regionQuery = function(pointId, epsilon) {
  epsilon = epsilon || this.epsilon;
  var neighbors = [];

  for (var id = 0, l = this.dataset.length; id < l; id++) {
    if (this.distance(this.dataset[pointId], this.dataset[id]) < epsilon) {
      neighbors.push(id);
    }
  }

  return neighbors;
};

/******************************************************************************/
// helpers

/**
 * Calculate euclidean distance in multidimensional space
 *
 * @param {Array} p
 * @param {Array} q
 * @returns {number}
 * @access protected
 */
OPTICS.prototype._euclideanDistance = function(p, q) {
  var sum = 0;
  var i = Math.min(p.length, q.length);

  while (i--) {
    sum += (p[i] - q[i]) * (p[i] - q[i]);
  }

  return Math.sqrt(sum);
};

if ('object' !== 'undefined' && module.exports) {
  module.exports = OPTICS;
}
});

var lib = createCommonjsModule(function (module) {
if ('object' !== 'undefined' && module.exports) {
    module.exports = {
      DBSCAN: DBSCAN_1,
      KMEANS: KMEANS_1,
      OPTICS: OPTICS_1,
      PriorityQueue: PriorityQueue_1
    };
}
});

var lib_1 = lib.DBSCAN;
var lib_2 = lib.KMEANS;
var lib_3 = lib.OPTICS;
var lib_4 = lib.PriorityQueue;

/**
 * Takes a set of {@link Point|points} and partition them into clusters according to {@link DBSCAN's|https://en.wikipedia.org/wiki/DBSCAN} data clustering algorithm.
 *
 * @name clustersDbscan
 * @param {FeatureCollection<Point>} points to be clustered
 * @param {number} maxDistance Maximum Distance between any point of the cluster to generate the clusters (kilometers only)
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.units=kilometers] in which `maxDistance` is expressed, can be degrees, radians, miles, or kilometers
 * @param {number} [options.minPoints=3] Minimum number of points to generate a single cluster,
 * points which do not meet this requirement will be classified as an 'edge' or 'noise'.
 * @returns {FeatureCollection<Point>} Clustered Points with an additional two properties associated to each Feature:
 * - {number} cluster - the associated clusterId
 * - {string} dbscan - type of point it has been classified as ('core'|'edge'|'noise')
 * @example
 * // create random points with random z-values in their properties
 * var points = turf.randomPoint(100, {bbox: [0, 30, 20, 50]});
 * var maxDistance = 100;
 * var clustered = turf.clustersDbscan(points, maxDistance);
 *
 * //addToMap
 * var addToMap = [clustered];
 */
function clustersDbscan(points$$1, maxDistance, options) {
    // Optional parameters
    options = options || {};
    if (typeof options !== 'object') throw new Error('options is invalid');
    var minPoints = options.minPoints;
    var units = options.units;

    // Input validation
    collectionOf(points$$1, 'Point', 'Input must contain Points');
    if (maxDistance === null || maxDistance === undefined) throw new Error('maxDistance is required');
    if (!(Math.sign(maxDistance) > 0)) throw new Error('Invalid maxDistance');
    if (!(minPoints === undefined || minPoints === null || Math.sign(minPoints) > 0)) throw new Error('Invalid minPoints');

    // Clone points to prevent any mutations
    points$$1 = clone(points$$1, true);

    // Defaults
    minPoints = minPoints || 3;

    // create clustered ids
    var dbscan = new lib.DBSCAN();
    var clusteredIds = dbscan.run(coordAll(points$$1), convertLength(maxDistance, units), minPoints, distance);

    // Tag points to Clusters ID
    var clusterId = -1;
    clusteredIds.forEach(function (clusterIds) {
        clusterId++;
        // assign cluster ids to input points
        clusterIds.forEach(function (idx) {
            var clusterPoint = points$$1.features[idx];
            if (!clusterPoint.properties) clusterPoint.properties = {};
            clusterPoint.properties.cluster = clusterId;
            clusterPoint.properties.dbscan = 'core';
        });
    });

    // handle noise points, if any
    // edges points are tagged by DBSCAN as both 'noise' and 'cluster' as they can "reach" less than 'minPoints' number of points
    dbscan.noise.forEach(function (noiseId) {
        var noisePoint = points$$1.features[noiseId];
        if (!noisePoint.properties) noisePoint.properties = {};
        if (noisePoint.properties.cluster) noisePoint.properties.dbscan = 'edge';
        else noisePoint.properties.dbscan = 'noise';
    });

    return points$$1;
}

var distance$2 = {
	/**
  * Euclidean distance
  */
	eudist: function eudist(v1, v2, sqrt) {
		var len = v1.length;
		var sum = 0;

		for (var i = 0; i < len; i++) {
			var d = (v1[i] || 0) - (v2[i] || 0);
			sum += d * d;
		}
		// Square root not really needed
		return sqrt ? Math.sqrt(sum) : sum;
	},
	mandist: function mandist(v1, v2, sqrt) {
		var len = v1.length;
		var sum = 0;

		for (var i = 0; i < len; i++) {
			sum += Math.abs((v1[i] || 0) - (v2[i] || 0));
		}

		// Square root not really needed
		return sqrt ? Math.sqrt(sum) : sum;
	},


	/**
  * Unidimensional distance
  */
	dist: function dist(v1, v2, sqrt) {
		var d = Math.abs(v1 - v2);
		return sqrt ? d : d * d;
	}
};

var eudist$1 = distance$2.eudist;
var dist$1 = distance$2.dist;

var kinit = {
	kmrand: function kmrand(data, k) {
		var map = {},
		    ks = [],
		    t = k << 2;
		var len = data.length;
		var multi = data[0].length > 0;

		while (ks.length < k && t-- > 0) {
			var d = data[Math.floor(Math.random() * len)];
			var key = multi ? d.join("_") : "" + d;
			if (!map[key]) {
				map[key] = true;
				ks.push(d);
			}
		}

		if (ks.length < k) throw new Error("Error initializating clusters");else return ks;
	},


	/**
  * K-means++ initial centroid selection
  */
	kmpp: function kmpp(data, k) {
		var distance = data[0].length ? eudist$1 : dist$1;
		var ks = [],
		    len = data.length;
		var multi = data[0].length > 0;
		var c = data[Math.floor(Math.random() * len)];
		var key = multi ? c.join("_") : "" + c;
		ks.push(c);
		while (ks.length < k) {
			// Min Distances between current centroids and data points
			var dists = [],
			    lk = ks.length;
			var dsum = 0,
			    prs = [];

			for (var i = 0; i < len; i++) {
				var min = Infinity;
				for (var j = 0; j < lk; j++) {
					var _dist = distance(data[i], ks[j]);
					if (_dist <= min) min = _dist;
				}
				dists[i] = min;
			}

			// Sum all min distances
			for (var _i = 0; _i < len; _i++) {
				dsum += dists[_i];
			}

			// Probabilities and cummulative prob (cumsum)
			for (var _i2 = 0; _i2 < len; _i2++) {
				prs[_i2] = { i: _i2, v: data[_i2], pr: dists[_i2] / dsum, cs: 0 };
			}

			// Sort Probabilities
			prs.sort(function (a, b) {
				return a.pr - b.pr;
			});

			// Cummulative Probabilities
			prs[0].cs = prs[0].pr;
			for (var _i3 = 1; _i3 < len; _i3++) {
				prs[_i3].cs = prs[_i3 - 1].cs + prs[_i3].pr;
			}

			// Randomize
			var rnd = Math.random();

			// Gets only the items whose cumsum >= rnd
			var idx = 0;
			while (idx < len - 1 && prs[idx++].cs < rnd) {}
			ks.push(prs[idx - 1].v);
			/*
   let done = false;
   while(!done) {
   	// this is our new centroid
   	c = prs[idx-1].v
   	key = multi? c.join("_") : `${c}`;
   	if(!map[key]) {
   		map[key] = true;
   		ks.push(c);
   		done = true;
   	}
   	else {
   		idx++;
   	}
   }
   */
		}

		return ks;
	}
};

/*jshint esversion: 6 */

var eudist = distance$2.eudist;
var kmrand = kinit.kmrand;
var kmpp = kinit.kmpp;

var MAX = 10000;

/**
 * Inits an array with values
 */
function init(len, val, v) {
	v = v || [];
	for (var i = 0; i < len; i++) {
		v[i] = val;
	}return v;
}

function skmeans(data, k, initial, maxit) {
	var ks = [],
	    old = [],
	    idxs = [],
	    dist = [];
	var conv = false,
	    it = maxit || MAX;
	var len = data.length,
	    vlen = data[0].length,
	    multi = vlen > 0;
	var count = [];

	if (!initial) {
		var _idxs = {};
		while (ks.length < k) {
			var idx = Math.floor(Math.random() * len);
			if (!_idxs[idx]) {
				_idxs[idx] = true;
				ks.push(data[idx]);
			}
		}
	} else if (initial == "kmrand") {
		ks = kmrand(data, k);
	} else if (initial == "kmpp") {
		ks = kmpp(data, k);
	} else {
		ks = initial;
	}

	do {
		// Reset k count
		init(k, 0, count);

		// For each value in data, find the nearest centroid
		for (var i = 0; i < len; i++) {
			var min = Infinity,
			    _idx = 0;
			for (var j = 0; j < k; j++) {
				// Multidimensional or unidimensional
				var dist = multi ? eudist(data[i], ks[j]) : Math.abs(data[i] - ks[j]);
				if (dist <= min) {
					min = dist;
					_idx = j;
				}
			}
			idxs[i] = _idx; // Index of the selected centroid for that value
			count[_idx]++; // Number of values for this centroid
		}

		// Recalculate centroids
		var sum = [],
		    old = [];
		for (var _j = 0; _j < k; _j++) {
			// Multidimensional or unidimensional
			sum[_j] = multi ? init(vlen, 0, sum[_j]) : 0;
			old[_j] = ks[_j];
		}

		// If multidimensional
		if (multi) {
			for (var _j2 = 0; _j2 < k; _j2++) {
				ks[_j2] = [];
			} // Sum values and count for each centroid
			for (var _i = 0; _i < len; _i++) {
				var _idx2 = idxs[_i],
				    // Centroid for that item
				vsum = sum[_idx2],
				    // Sum values for this centroid
				vect = data[_i]; // Current vector

				// Accumulate value on the centroid for current vector
				for (var h = 0; h < vlen; h++) {
					vsum[h] += vect[h];
				}
			}
			// Calculate the average for each centroid
			conv = true;
			for (var _j3 = 0; _j3 < k; _j3++) {
				var ksj = ks[_j3],
				    // Current centroid
				sumj = sum[_j3],
				    // Accumulated centroid values
				oldj = old[_j3],
				    // Old centroid value
				cj = count[_j3]; // Number of elements for this centroid

				// New average
				for (var _h = 0; _h < vlen; _h++) {
					ksj[_h] = sumj[_h] / cj || 0; // New centroid
				}

				// Find if centroids have moved
				if (conv) {
					for (var _h2 = 0; _h2 < vlen; _h2++) {
						if (oldj[_h2] != ksj[_h2]) {
							conv = false;
							break;
						}
					}
				}
			}
		}
		// If unidimensional
		else {
				// Sum values and count for each centroid
				for (var _i2 = 0; _i2 < len; _i2++) {
					var _idx3 = idxs[_i2];
					sum[_idx3] += data[_i2];
				}
				// Calculate the average for each centroid
				for (var _j4 = 0; _j4 < k; _j4++) {
					ks[_j4] = sum[_j4] / count[_j4] || 0; // New centroid
				}
				// Find if centroids have moved
				conv = true;
				for (var _j5 = 0; _j5 < k; _j5++) {
					if (old[_j5] != ks[_j5]) {
						conv = false;
						break;
					}
				}
			}

		conv = conv || --it <= 0;
	} while (!conv);

	return {
		it: MAX - it,
		k: k,
		idxs: idxs,
		centroids: ks
	};
}

var main = skmeans;

/**
 * Takes a set of {@link Point|points} and partition them into clusters using the k-mean .
 * It uses the [k-means algorithm](https://en.wikipedia.org/wiki/K-means_clustering)
 *
 * @name clustersKmeans
 * @param {FeatureCollection<Point>} points to be clustered
 * @param {Object} [options={}] Optional parameters
 * @param {number} [options.numberOfClusters=Math.sqrt(numberOfPoints/2)] numberOfClusters that will be generated
 * @param {boolean} [options.mutate=false] allows GeoJSON input to be mutated (significant performance increase if true)
 * @returns {FeatureCollection<Point>} Clustered Points with an additional two properties associated to each Feature:
 * - {number} cluster - the associated clusterId
 * - {[number, number]} centroid - Centroid of the cluster [Longitude, Latitude]
 * @example
 * // create random points with random z-values in their properties
 * var points = turf.randomPoint(100, {bbox: [0, 30, 20, 50]});
 * var options = {numberOfClusters: 7};
 * var clustered = turf.clustersKmeans(points, options);
 *
 * //addToMap
 * var addToMap = [clustered];
 */
function clustersKmeans(points, options) {
    // Optional parameters
    options = options || {};
    if (typeof options !== 'object') throw new Error('options is invalid');
    var numberOfClusters = options.numberOfClusters;
    var mutate = options.mutate;

    // Input validation
    collectionOf(points, 'Point', 'Input must contain Points');

    // Default Params
    var count = points.features.length;
    numberOfClusters = numberOfClusters || Math.round(Math.sqrt(count / 2));

    // numberOfClusters can't be greater than the number of points
    // fallbacks to count
    if (numberOfClusters > count) numberOfClusters = count;

    // Clone points to prevent any mutations (enabled by default)
    if (mutate === false || mutate === undefined) points = clone(points, true);

    // collect points coordinates
    var data = coordAll(points);

    // create seed to avoid skmeans to drift
    var initialCentroids = data.slice(0, numberOfClusters);

    // create skmeans clusters
    var skmeansResult = main(data, numberOfClusters, initialCentroids);

    // store centroids {clusterId: [number, number]}
    var centroids = {};
    skmeansResult.centroids.forEach(function (coord, idx) {
        centroids[idx] = coord;
    });

    // add associated cluster number
    featureEach(points, function (point, index) {
        var clusterId = skmeansResult.idxs[index];
        point.properties.cluster = clusterId;
        point.properties.centroid = centroids[clusterId];
    });

    return points;
}

/**
 * Boolean-Parallel returns True if each segment of `line1` is parallel to the correspondent segment of `line2`
 *
 * @name booleanParallel
 * @param {Geometry|Feature<LineString>} line1 GeoJSON Feature or Geometry
 * @param {Geometry|Feature<LineString>} line2 GeoJSON Feature or Geometry
 * @returns {boolean} true/false if the lines are parallel
 * @example
 * var line1 = turf.lineString([[0, 0], [0, 1]]);
 * var line2 = turf.lineString([[1, 0], [1, 1]]);
 *
 * turf.booleanParallel(line1, line2);
 * //=true
 */
function booleanParallel(line1, line2) {
    // validation
    if (!line1) throw new Error('line1 is required');
    if (!line2) throw new Error('line2 is required');
    var type1 = getType$1(line1, 'line1');
    if (type1 !== 'LineString') throw new Error('line1 must be a LineString');
    var type2 = getType$1(line2, 'line2');
    if (type2 !== 'LineString') throw new Error('line2 must be a LineString');

    var segments1 = lineSegment(cleanCoords(line1)).features;
    var segments2 = lineSegment(cleanCoords(line2)).features;

    for (var i = 0; i < segments1.length; i++) {
        var segment1 = segments1[i].geometry.coordinates;
        if (!segments2[i]) break;
        var segment2 = segments2[i].geometry.coordinates;
        if (!isParallel$1(segment1, segment2)) return false;
    }
    return true;
}


/**
 * Compares slopes and return result
 *
 * @private
 * @param {Geometry|Feature<LineString>} segment1 Geometry or Feature
 * @param {Geometry|Feature<LineString>} segment2 Geometry or Feature
 * @returns {boolean} if slopes are equal
 */
function isParallel$1(segment1, segment2) {
    var slope1 = bearingToAzimuth(rhumbBearing(segment1[0], segment1[1]));
    var slope2 = bearingToAzimuth(rhumbBearing(segment2[0], segment2[1]));
    return slope1 === slope2;
}


/**
 * Returns Feature's type
 *
 * @private
 * @param {Geometry|Feature<any>} geojson Geometry or Feature
 * @param {string} name of the variable
 * @returns {string} Feature's type
 */
function getType$1(geojson, name) {
    if (geojson.geometry && geojson.geometry.type) return geojson.geometry.type;
    if (geojson.type) return geojson.type; // if GeoJSON geometry
    throw new Error('Invalid GeoJSON object for ' + name);
}

// javascript-astar 0.4.1
// http://github.com/bgrins/javascript-astar
// Freely distributable under the MIT License.
// Implements the astar search algorithm in javascript using a Binary Heap.
// Includes Binary Heap (with modifications) from Marijn Haverbeke.
// http://eloquentjavascript.net/appendix2.html

function pathTo(node) {
    var curr = node,
        path = [];
    while (curr.parent) {
        path.unshift(curr);
        curr = curr.parent;
    }
    return path;
}

function getHeap() {
    return new BinaryHeap(function (node) {
        return node.f;
    });
}

/**
 * Astar
 * @private
 */
var astar = {
    /**
     * Perform an A* Search on a graph given a start and end node.
     *
     * @private
     * @memberof astar
     * @param {Graph} graph Graph
     * @param {GridNode} start Start
     * @param {GridNode} end End
     * @param {Object} [options] Options
     * @param {bool} [options.closest] Specifies whether to return the path to the closest node if the target is unreachable.
     * @param {Function} [options.heuristic] Heuristic function (see astar.heuristics).
     * @returns {Object} Search
     */
    search: function (graph, start, end, options) {
        graph.cleanDirty();
        options = options || {};
        var heuristic = options.heuristic || astar.heuristics.manhattan,
            closest = options.closest || false;

        var openHeap = getHeap(),
            closestNode = start; // set the start node to be the closest if required

        start.h = heuristic(start, end);

        openHeap.push(start);

        while (openHeap.size() > 0) {

            // Grab the lowest f(x) to process next.  Heap keeps this sorted for us.
            var currentNode = openHeap.pop();

            // End case -- result has been found, return the traced path.
            if (currentNode === end) {
                return pathTo(currentNode);
            }

            // Normal case -- move currentNode from open to closed, process each of its neighbors.
            currentNode.closed = true;

            // Find all neighbors for the current node.
            var neighbors = graph.neighbors(currentNode);

            for (var i = 0, il = neighbors.length; i < il; ++i) {
                var neighbor = neighbors[i];

                if (neighbor.closed || neighbor.isWall()) {
                    // Not a valid node to process, skip to next neighbor.
                    continue;
                }

                // The g score is the shortest distance from start to current node.
                // We need to check if the path we have arrived at this neighbor is the shortest one we have seen yet.
                var gScore = currentNode.g + neighbor.getCost(currentNode),
                    beenVisited = neighbor.visited;

                if (!beenVisited || gScore < neighbor.g) {

                    // Found an optimal (so far) path to this node.  Take score for node to see how good it is.
                    neighbor.visited = true;
                    neighbor.parent = currentNode;
                    neighbor.h = neighbor.h || heuristic(neighbor, end);
                    neighbor.g = gScore;
                    neighbor.f = neighbor.g + neighbor.h;
                    graph.markDirty(neighbor);
                    if (closest) {
                        // If the neighbour is closer than the current closestNode or if it's equally close but has
                        // a cheaper path than the current closest node then it becomes the closest node
                        if (neighbor.h < closestNode.h || (neighbor.h === closestNode.h && neighbor.g < closestNode.g)) {
                            closestNode = neighbor;
                        }
                    }

                    if (!beenVisited) {
                        // Pushing to heap will put it in proper place based on the 'f' value.
                        openHeap.push(neighbor);
                    } else {
                        // Already seen the node, but since it has been rescored we need to reorder it in the heap
                        openHeap.rescoreElement(neighbor);
                    }
                }
            }
        }

        if (closest) {
            return pathTo(closestNode);
        }

        // No result was found - empty array signifies failure to find path.
        return [];
    },
    // See list of heuristics: http://theory.stanford.edu/~amitp/GameProgramming/Heuristics.html
    heuristics: {
        manhattan: function (pos0, pos1) {
            var d1 = Math.abs(pos1.x - pos0.x);
            var d2 = Math.abs(pos1.y - pos0.y);
            return d1 + d2;
        },
        diagonal: function (pos0, pos1) {
            var D = 1;
            var D2 = Math.sqrt(2);
            var d1 = Math.abs(pos1.x - pos0.x);
            var d2 = Math.abs(pos1.y - pos0.y);
            return (D * (d1 + d2)) + ((D2 - (2 * D)) * Math.min(d1, d2));
        }
    },
    cleanNode: function (node) {
        node.f = 0;
        node.g = 0;
        node.h = 0;
        node.visited = false;
        node.closed = false;
        node.parent = null;
    }
};

/**
 * A graph memory structure
 *
 * @private
 * @param {Array} gridIn 2D array of input weights
 * @param {Object} [options] Options
 * @param {boolean} [options.diagonal] Specifies whether diagonal moves are allowed
 * @returns {void} Graph
 */
function Graph$1(gridIn, options) {
    options = options || {};
    this.nodes = [];
    this.diagonal = !!options.diagonal;
    this.grid = [];
    for (var x = 0; x < gridIn.length; x++) {
        this.grid[x] = [];

        for (var y = 0, row = gridIn[x]; y < row.length; y++) {
            var node = new GridNode(x, y, row[y]);
            this.grid[x][y] = node;
            this.nodes.push(node);
        }
    }
    this.init();
}

Graph$1.prototype.init = function () {
    this.dirtyNodes = [];
    for (var i = 0; i < this.nodes.length; i++) {
        astar.cleanNode(this.nodes[i]);
    }
};

Graph$1.prototype.cleanDirty = function () {
    for (var i = 0; i < this.dirtyNodes.length; i++) {
        astar.cleanNode(this.dirtyNodes[i]);
    }
    this.dirtyNodes = [];
};

Graph$1.prototype.markDirty = function (node) {
    this.dirtyNodes.push(node);
};

Graph$1.prototype.neighbors = function (node) {
    var ret = [],
        x = node.x,
        y = node.y,
        grid = this.grid;

    // West
    if (grid[x - 1] && grid[x - 1][y]) {
        ret.push(grid[x - 1][y]);
    }

    // East
    if (grid[x + 1] && grid[x + 1][y]) {
        ret.push(grid[x + 1][y]);
    }

    // South
    if (grid[x] && grid[x][y - 1]) {
        ret.push(grid[x][y - 1]);
    }

    // North
    if (grid[x] && grid[x][y + 1]) {
        ret.push(grid[x][y + 1]);
    }

    if (this.diagonal) {
        // Southwest
        if (grid[x - 1] && grid[x - 1][y - 1]) {
            ret.push(grid[x - 1][y - 1]);
        }

        // Southeast
        if (grid[x + 1] && grid[x + 1][y - 1]) {
            ret.push(grid[x + 1][y - 1]);
        }

        // Northwest
        if (grid[x - 1] && grid[x - 1][y + 1]) {
            ret.push(grid[x - 1][y + 1]);
        }

        // Northeast
        if (grid[x + 1] && grid[x + 1][y + 1]) {
            ret.push(grid[x + 1][y + 1]);
        }
    }

    return ret;
};

Graph$1.prototype.toString = function () {
    var graphString = [],
        nodes = this.grid, // when using grid
        rowDebug, row, y, l;
    for (var x = 0, len = nodes.length; x < len; x++) {
        rowDebug = [];
        row = nodes[x];
        for (y = 0, l = row.length; y < l; y++) {
            rowDebug.push(row[y].weight);
        }
        graphString.push(rowDebug.join(' '));
    }
    return graphString.join('\n');
};

function GridNode(x, y, weight) {
    this.x = x;
    this.y = y;
    this.weight = weight;
}

GridNode.prototype.toString = function () {
    return '[' + this.x + ' ' + this.y + ']';
};

GridNode.prototype.getCost = function (fromNeighbor) {
    // Take diagonal weight into consideration.
    if (fromNeighbor && fromNeighbor.x !== this.x && fromNeighbor.y !== this.y) {
        return this.weight * 1.41421;
    }
    return this.weight;
};

GridNode.prototype.isWall = function () {
    return this.weight === 0;
};

function BinaryHeap(scoreFunction) {
    this.content = [];
    this.scoreFunction = scoreFunction;
}

BinaryHeap.prototype = {
    push: function (element) {
        // Add the new element to the end of the array.
        this.content.push(element);

        // Allow it to sink down.
        this.sinkDown(this.content.length - 1);
    },
    pop: function () {
        // Store the first element so we can return it later.
        var result = this.content[0];
        // Get the element at the end of the array.
        var end = this.content.pop();
        // If there are any elements left, put the end element at the
        // start, and let it bubble up.
        if (this.content.length > 0) {
            this.content[0] = end;
            this.bubbleUp(0);
        }
        return result;
    },
    remove: function (node) {
        var i = this.content.indexOf(node);

        // When it is found, the process seen in 'pop' is repeated
        // to fill up the hole.
        var end = this.content.pop();

        if (i !== this.content.length - 1) {
            this.content[i] = end;

            if (this.scoreFunction(end) < this.scoreFunction(node)) {
                this.sinkDown(i);
            } else {
                this.bubbleUp(i);
            }
        }
    },
    size: function () {
        return this.content.length;
    },
    rescoreElement: function (node) {
        this.sinkDown(this.content.indexOf(node));
    },
    sinkDown: function (n) {
        // Fetch the element that has to be sunk.
        var element = this.content[n];

        // When at 0, an element can not sink any further.
        while (n > 0) {

            // Compute the parent element's index, and fetch it.
            var parentN = ((n + 1) >> 1) - 1,
                parent = this.content[parentN];
            // Swap the elements if the parent is greater.
            if (this.scoreFunction(element) < this.scoreFunction(parent)) {
                this.content[parentN] = element;
                this.content[n] = parent;
                // Update 'n' to continue at the new position.
                n = parentN;
            // Found a parent that is less, no need to sink any further.
            } else {
                break;
            }
        }
    },
    bubbleUp: function (n) {
        // Look up the target element and its score.
        var length = this.content.length,
            element = this.content[n],
            elemScore = this.scoreFunction(element);

        while (true) {
            // Compute the indices of the child elements.
            var child2N = (n + 1) << 1,
                child1N = child2N - 1;
            // This is used to store the new position of the element, if any.
            var swap = null,
                child1Score;
            // If the first child exists (is inside the array)...
            if (child1N < length) {
                // Look it up and compute its score.
                var child1 = this.content[child1N];
                child1Score = this.scoreFunction(child1);

                // If the score is less than our element's, we need to swap.
                if (child1Score < elemScore) {
                    swap = child1N;
                }
            }

            // Do the same checks for the other child.
            if (child2N < length) {
                var child2 = this.content[child2N],
                    child2Score = this.scoreFunction(child2);
                if (child2Score < (swap === null ? elemScore : child1Score)) {
                    swap = child2N;
                }
            }

            // If the element needs to be moved, swap it, and continue.
            if (swap !== null) {
                this.content[n] = this.content[swap];
                this.content[swap] = element;
                n = swap;
            // Otherwise, we are done.
            } else {
                break;
            }
        }
    }
};

/**
 * Returns the shortest {@link LineString|path} from {@link Point|start} to {@link Point|end} without colliding with
 * any {@link Feature} in {@link FeatureCollection<Polygon>| obstacles}
 *
 * @name shortestPath
 * @param {Coord} start point
 * @param {Coord} end point
 * @param {Object} [options={}] optional parameters
 * @param {Geometry|Feature|FeatureCollection<Polygon>} [options.obstacles] areas which path cannot travel
 * @param {number} [options.minDistance] minimum distance between shortest path and obstacles
 * @param {string} [options.units='kilometers'] unit in which resolution & minimum distance will be expressed in; it can be degrees, radians, miles, kilometers, ...
 * @param {number} [options.resolution=100] distance between matrix points on which the path will be calculated
 * @returns {Feature<LineString>} shortest path between start and end
 * @example
 * var start = [-5, -6];
 * var end = [9, -6];
 * var options = {
 *   obstacles: turf.polygon([[[0, -7], [5, -7], [5, -3], [0, -3], [0, -7]]])
 * };
 *
 * var path = turf.shortestPath(start, end, options);
 *
 * //addToMap
 * var addToMap = [start, end, options.obstacles, path];
 */
function shortestPath(start, end, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var resolution = options.resolution;
    var minDistance = options.minDistance;
    var obstacles = options.obstacles || featureCollection([]);

    // validation
    if (!start) throw new Error('start is required');
    if (!end) throw new Error('end is required');
    if (resolution && !isNumber(resolution) || resolution <= 0) throw new Error('options.resolution must be a number, greater than 0');
    if (minDistance) throw new Error('options.minDistance is not yet implemented');

    // Normalize Inputs
    var startCoord = getCoord(start);
    var endCoord = getCoord(end);
    start = point(startCoord);
    end = point(endCoord);

    // Handle obstacles
    switch (getType(obstacles)) {
    case 'FeatureCollection':
        if (obstacles.features.length === 0) return lineString([startCoord, endCoord]);
        break;
    case 'Polygon':
        obstacles = featureCollection([feature(getGeom(obstacles))]);
        break;
    default:
        throw new Error('invalid obstacles');
    }

    // define path grid area
    var collection = obstacles;
    collection.features.push(start);
    collection.features.push(end);
    var box = bbox(transformScale(bboxPolygon(bbox(collection)), 1.15)); // extend 15%
    if (!resolution) {
        var width = distance([box[0], box[1]], [box[2], box[1]], options);
        resolution = width / 100;
    }
    collection.features.pop();
    collection.features.pop();

    var west = box[0];
    var south = box[1];
    var east = box[2];
    var north = box[3];

    var xFraction = resolution / (distance([west, south], [east, south], options));
    var cellWidth = xFraction * (east - west);
    var yFraction = resolution / (distance([west, south], [west, north], options));
    var cellHeight = yFraction * (north - south);

    var bboxHorizontalSide = (east - west);
    var bboxVerticalSide = (north - south);
    var columns = Math.floor(bboxHorizontalSide / cellWidth);
    var rows = Math.floor(bboxVerticalSide / cellHeight);
    // adjust origin of the grid
    var deltaX = (bboxHorizontalSide - columns * cellWidth) / 2;
    var deltaY = (bboxVerticalSide - rows * cellHeight) / 2;

    // loop through points only once to speed up process
    // define matrix grid for A-star algorithm
    var pointMatrix = [];
    var matrix = [];

    var closestToStart = [];
    var closestToEnd = [];
    var minDistStart = Infinity;
    var minDistEnd = Infinity;
    var currentY = north - deltaY;
    var r = 0;
    while (currentY >= south) {
        // var currentY = south + deltaY;
        var matrixRow = [];
        var pointMatrixRow = [];
        var currentX = west + deltaX;
        var c = 0;
        while (currentX <= east) {
            var pt = point([currentX, currentY]);
            var isInsideObstacle = isInside$1(pt, obstacles);
            // feed obstacles matrix
            matrixRow.push(isInsideObstacle ? 0 : 1); // with javascript-astar
            // matrixRow.push(isInsideObstacle ? 1 : 0); // with astar-andrea
            // map point's coords
            pointMatrixRow.push(currentX + '|' + currentY);
            // set closest points
            var distStart = distance(pt, start);
            // if (distStart < minDistStart) {
            if (!isInsideObstacle && distStart < minDistStart) {
                minDistStart = distStart;
                closestToStart = {x: c, y: r};
            }
            var distEnd = distance(pt, end);
            // if (distEnd < minDistEnd) {
            if (!isInsideObstacle && distEnd < minDistEnd) {
                minDistEnd = distEnd;
                closestToEnd = {x: c, y: r};
            }
            currentX += cellWidth;
            c++;
        }
        matrix.push(matrixRow);
        pointMatrix.push(pointMatrixRow);
        currentY -= cellHeight;
        r++;
    }

    // find path on matrix grid

    // javascript-astar ----------------------
    var graph = new Graph$1(matrix, {diagonal: true});
    var startOnMatrix = graph.grid[closestToStart.y][closestToStart.x];
    var endOnMatrix = graph.grid[closestToEnd.y][closestToEnd.x];
    var result = astar.search(graph, startOnMatrix, endOnMatrix);

    var path = [startCoord];
    result.forEach(function (coord) {
        var coords = pointMatrix[coord.x][coord.y].split('|');
        path.push([+coords[0], +coords[1]]); // make sure coords are numbers
    });
    path.push(endCoord);
    // ---------------------------------------


    // astar-andrea ------------------------
    // var result = aStar(matrix, [closestToStart.x, closestToStart.y], [closestToEnd.x, closestToEnd.y], 'DiagonalFree');
    // var path = [start.geometry.coordinates];
    // result.forEach(function (coord) {
    //     var coords = pointMatrix[coord[1]][coord[0]].split('|');
    //     path.push([+coords[0], +coords[1]]); // make sure coords are numbers
    // });
    // path.push(end.geometry.coordinates);
    // ---------------------------------------


    return cleanCoords(lineString(path));
}

/**
 * Checks if Point is inside any of the Polygons
 *
 * @private
 * @param {Feature<Point>} pt to check
 * @param {FeatureCollection<Polygon>} polygons features
 * @returns {boolean} if inside or not
 */
function isInside$1(pt, polygons$$1) {
    for (var i = 0; i < polygons$$1.features.length; i++) {
        if (booleanPointInPolygon(pt, polygons$$1.features[i])) {
            return true;
        }
    }
    return false;
}

var constant = function(x) {
  return function() {
    return x;
  };
};

function x(d) {
  return d[0];
}

function y(d) {
  return d[1];
}

function RedBlackTree() {
  this._ = null; // root node
}

function RedBlackNode(node) {
  node.U = // parent node
  node.C = // color - true for red, false for black
  node.L = // left node
  node.R = // right node
  node.P = // previous node
  node.N = null; // next node
}

RedBlackTree.prototype = {
  constructor: RedBlackTree,

  insert: function(after, node) {
    var parent, grandpa, uncle;

    if (after) {
      node.P = after;
      node.N = after.N;
      if (after.N) after.N.P = node;
      after.N = node;
      if (after.R) {
        after = after.R;
        while (after.L) after = after.L;
        after.L = node;
      } else {
        after.R = node;
      }
      parent = after;
    } else if (this._) {
      after = RedBlackFirst(this._);
      node.P = null;
      node.N = after;
      after.P = after.L = node;
      parent = after;
    } else {
      node.P = node.N = null;
      this._ = node;
      parent = null;
    }
    node.L = node.R = null;
    node.U = parent;
    node.C = true;

    after = node;
    while (parent && parent.C) {
      grandpa = parent.U;
      if (parent === grandpa.L) {
        uncle = grandpa.R;
        if (uncle && uncle.C) {
          parent.C = uncle.C = false;
          grandpa.C = true;
          after = grandpa;
        } else {
          if (after === parent.R) {
            RedBlackRotateLeft(this, parent);
            after = parent;
            parent = after.U;
          }
          parent.C = false;
          grandpa.C = true;
          RedBlackRotateRight(this, grandpa);
        }
      } else {
        uncle = grandpa.L;
        if (uncle && uncle.C) {
          parent.C = uncle.C = false;
          grandpa.C = true;
          after = grandpa;
        } else {
          if (after === parent.L) {
            RedBlackRotateRight(this, parent);
            after = parent;
            parent = after.U;
          }
          parent.C = false;
          grandpa.C = true;
          RedBlackRotateLeft(this, grandpa);
        }
      }
      parent = after.U;
    }
    this._.C = false;
  },

  remove: function(node) {
    if (node.N) node.N.P = node.P;
    if (node.P) node.P.N = node.N;
    node.N = node.P = null;

    var parent = node.U,
        sibling,
        left = node.L,
        right = node.R,
        next,
        red;

    if (!left) next = right;
    else if (!right) next = left;
    else next = RedBlackFirst(right);

    if (parent) {
      if (parent.L === node) parent.L = next;
      else parent.R = next;
    } else {
      this._ = next;
    }

    if (left && right) {
      red = next.C;
      next.C = node.C;
      next.L = left;
      left.U = next;
      if (next !== right) {
        parent = next.U;
        next.U = node.U;
        node = next.R;
        parent.L = node;
        next.R = right;
        right.U = next;
      } else {
        next.U = parent;
        parent = next;
        node = next.R;
      }
    } else {
      red = node.C;
      node = next;
    }

    if (node) node.U = parent;
    if (red) return;
    if (node && node.C) { node.C = false; return; }

    do {
      if (node === this._) break;
      if (node === parent.L) {
        sibling = parent.R;
        if (sibling.C) {
          sibling.C = false;
          parent.C = true;
          RedBlackRotateLeft(this, parent);
          sibling = parent.R;
        }
        if ((sibling.L && sibling.L.C)
            || (sibling.R && sibling.R.C)) {
          if (!sibling.R || !sibling.R.C) {
            sibling.L.C = false;
            sibling.C = true;
            RedBlackRotateRight(this, sibling);
            sibling = parent.R;
          }
          sibling.C = parent.C;
          parent.C = sibling.R.C = false;
          RedBlackRotateLeft(this, parent);
          node = this._;
          break;
        }
      } else {
        sibling = parent.L;
        if (sibling.C) {
          sibling.C = false;
          parent.C = true;
          RedBlackRotateRight(this, parent);
          sibling = parent.L;
        }
        if ((sibling.L && sibling.L.C)
          || (sibling.R && sibling.R.C)) {
          if (!sibling.L || !sibling.L.C) {
            sibling.R.C = false;
            sibling.C = true;
            RedBlackRotateLeft(this, sibling);
            sibling = parent.L;
          }
          sibling.C = parent.C;
          parent.C = sibling.L.C = false;
          RedBlackRotateRight(this, parent);
          node = this._;
          break;
        }
      }
      sibling.C = true;
      node = parent;
      parent = parent.U;
    } while (!node.C);

    if (node) node.C = false;
  }
};

function RedBlackRotateLeft(tree, node) {
  var p = node,
      q = node.R,
      parent = p.U;

  if (parent) {
    if (parent.L === p) parent.L = q;
    else parent.R = q;
  } else {
    tree._ = q;
  }

  q.U = parent;
  p.U = q;
  p.R = q.L;
  if (p.R) p.R.U = p;
  q.L = p;
}

function RedBlackRotateRight(tree, node) {
  var p = node,
      q = node.L,
      parent = p.U;

  if (parent) {
    if (parent.L === p) parent.L = q;
    else parent.R = q;
  } else {
    tree._ = q;
  }

  q.U = parent;
  p.U = q;
  p.L = q.R;
  if (p.L) p.L.U = p;
  q.R = p;
}

function RedBlackFirst(node) {
  while (node.L) node = node.L;
  return node;
}

function createEdge(left, right, v0, v1) {
  var edge = [null, null],
      index = edges.push(edge) - 1;
  edge.left = left;
  edge.right = right;
  if (v0) setEdgeEnd(edge, left, right, v0);
  if (v1) setEdgeEnd(edge, right, left, v1);
  cells[left.index].halfedges.push(index);
  cells[right.index].halfedges.push(index);
  return edge;
}

function createBorderEdge(left, v0, v1) {
  var edge = [v0, v1];
  edge.left = left;
  return edge;
}

function setEdgeEnd(edge, left, right, vertex) {
  if (!edge[0] && !edge[1]) {
    edge[0] = vertex;
    edge.left = left;
    edge.right = right;
  } else if (edge.left === right) {
    edge[1] = vertex;
  } else {
    edge[0] = vertex;
  }
}

// Liang–Barsky line clipping.
function clipEdge(edge, x0, y0, x1, y1) {
  var a = edge[0],
      b = edge[1],
      ax = a[0],
      ay = a[1],
      bx = b[0],
      by = b[1],
      t0 = 0,
      t1 = 1,
      dx = bx - ax,
      dy = by - ay,
      r;

  r = x0 - ax;
  if (!dx && r > 0) return;
  r /= dx;
  if (dx < 0) {
    if (r < t0) return;
    if (r < t1) t1 = r;
  } else if (dx > 0) {
    if (r > t1) return;
    if (r > t0) t0 = r;
  }

  r = x1 - ax;
  if (!dx && r < 0) return;
  r /= dx;
  if (dx < 0) {
    if (r > t1) return;
    if (r > t0) t0 = r;
  } else if (dx > 0) {
    if (r < t0) return;
    if (r < t1) t1 = r;
  }

  r = y0 - ay;
  if (!dy && r > 0) return;
  r /= dy;
  if (dy < 0) {
    if (r < t0) return;
    if (r < t1) t1 = r;
  } else if (dy > 0) {
    if (r > t1) return;
    if (r > t0) t0 = r;
  }

  r = y1 - ay;
  if (!dy && r < 0) return;
  r /= dy;
  if (dy < 0) {
    if (r > t1) return;
    if (r > t0) t0 = r;
  } else if (dy > 0) {
    if (r < t0) return;
    if (r < t1) t1 = r;
  }

  if (!(t0 > 0) && !(t1 < 1)) return true; // TODO Better check?

  if (t0 > 0) edge[0] = [ax + t0 * dx, ay + t0 * dy];
  if (t1 < 1) edge[1] = [ax + t1 * dx, ay + t1 * dy];
  return true;
}

function connectEdge(edge, x0, y0, x1, y1) {
  var v1 = edge[1];
  if (v1) return true;

  var v0 = edge[0],
      left = edge.left,
      right = edge.right,
      lx = left[0],
      ly = left[1],
      rx = right[0],
      ry = right[1],
      fx = (lx + rx) / 2,
      fy = (ly + ry) / 2,
      fm,
      fb;

  if (ry === ly) {
    if (fx < x0 || fx >= x1) return;
    if (lx > rx) {
      if (!v0) v0 = [fx, y0];
      else if (v0[1] >= y1) return;
      v1 = [fx, y1];
    } else {
      if (!v0) v0 = [fx, y1];
      else if (v0[1] < y0) return;
      v1 = [fx, y0];
    }
  } else {
    fm = (lx - rx) / (ry - ly);
    fb = fy - fm * fx;
    if (fm < -1 || fm > 1) {
      if (lx > rx) {
        if (!v0) v0 = [(y0 - fb) / fm, y0];
        else if (v0[1] >= y1) return;
        v1 = [(y1 - fb) / fm, y1];
      } else {
        if (!v0) v0 = [(y1 - fb) / fm, y1];
        else if (v0[1] < y0) return;
        v1 = [(y0 - fb) / fm, y0];
      }
    } else {
      if (ly < ry) {
        if (!v0) v0 = [x0, fm * x0 + fb];
        else if (v0[0] >= x1) return;
        v1 = [x1, fm * x1 + fb];
      } else {
        if (!v0) v0 = [x1, fm * x1 + fb];
        else if (v0[0] < x0) return;
        v1 = [x0, fm * x0 + fb];
      }
    }
  }

  edge[0] = v0;
  edge[1] = v1;
  return true;
}

function clipEdges(x0, y0, x1, y1) {
  var i = edges.length,
      edge;

  while (i--) {
    if (!connectEdge(edge = edges[i], x0, y0, x1, y1)
        || !clipEdge(edge, x0, y0, x1, y1)
        || !(Math.abs(edge[0][0] - edge[1][0]) > epsilon
            || Math.abs(edge[0][1] - edge[1][1]) > epsilon)) {
      delete edges[i];
    }
  }
}

function createCell(site) {
  return cells[site.index] = {
    site: site,
    halfedges: []
  };
}

function cellHalfedgeAngle(cell, edge) {
  var site = cell.site,
      va = edge.left,
      vb = edge.right;
  if (site === vb) vb = va, va = site;
  if (vb) return Math.atan2(vb[1] - va[1], vb[0] - va[0]);
  if (site === va) va = edge[1], vb = edge[0];
  else va = edge[0], vb = edge[1];
  return Math.atan2(va[0] - vb[0], vb[1] - va[1]);
}

function cellHalfedgeStart(cell, edge) {
  return edge[+(edge.left !== cell.site)];
}

function cellHalfedgeEnd(cell, edge) {
  return edge[+(edge.left === cell.site)];
}

function sortCellHalfedges() {
  for (var i = 0, n = cells.length, cell, halfedges, j, m; i < n; ++i) {
    if ((cell = cells[i]) && (m = (halfedges = cell.halfedges).length)) {
      var index = new Array(m),
          array = new Array(m);
      for (j = 0; j < m; ++j) index[j] = j, array[j] = cellHalfedgeAngle(cell, edges[halfedges[j]]);
      index.sort(function(i, j) { return array[j] - array[i]; });
      for (j = 0; j < m; ++j) array[j] = halfedges[index[j]];
      for (j = 0; j < m; ++j) halfedges[j] = array[j];
    }
  }
}

function clipCells(x0, y0, x1, y1) {
  var nCells = cells.length,
      iCell,
      cell,
      site,
      iHalfedge,
      halfedges,
      nHalfedges,
      start,
      startX,
      startY,
      end,
      endX,
      endY,
      cover = true;

  for (iCell = 0; iCell < nCells; ++iCell) {
    if (cell = cells[iCell]) {
      site = cell.site;
      halfedges = cell.halfedges;
      iHalfedge = halfedges.length;

      // Remove any dangling clipped edges.
      while (iHalfedge--) {
        if (!edges[halfedges[iHalfedge]]) {
          halfedges.splice(iHalfedge, 1);
        }
      }

      // Insert any border edges as necessary.
      iHalfedge = 0, nHalfedges = halfedges.length;
      while (iHalfedge < nHalfedges) {
        end = cellHalfedgeEnd(cell, edges[halfedges[iHalfedge]]), endX = end[0], endY = end[1];
        start = cellHalfedgeStart(cell, edges[halfedges[++iHalfedge % nHalfedges]]), startX = start[0], startY = start[1];
        if (Math.abs(endX - startX) > epsilon || Math.abs(endY - startY) > epsilon) {
          halfedges.splice(iHalfedge, 0, edges.push(createBorderEdge(site, end,
              Math.abs(endX - x0) < epsilon && y1 - endY > epsilon ? [x0, Math.abs(startX - x0) < epsilon ? startY : y1]
              : Math.abs(endY - y1) < epsilon && x1 - endX > epsilon ? [Math.abs(startY - y1) < epsilon ? startX : x1, y1]
              : Math.abs(endX - x1) < epsilon && endY - y0 > epsilon ? [x1, Math.abs(startX - x1) < epsilon ? startY : y0]
              : Math.abs(endY - y0) < epsilon && endX - x0 > epsilon ? [Math.abs(startY - y0) < epsilon ? startX : x0, y0]
              : null)) - 1);
          ++nHalfedges;
        }
      }

      if (nHalfedges) cover = false;
    }
  }

  // If there weren’t any edges, have the closest site cover the extent.
  // It doesn’t matter which corner of the extent we measure!
  if (cover) {
    var dx, dy, d2, dc = Infinity;

    for (iCell = 0, cover = null; iCell < nCells; ++iCell) {
      if (cell = cells[iCell]) {
        site = cell.site;
        dx = site[0] - x0;
        dy = site[1] - y0;
        d2 = dx * dx + dy * dy;
        if (d2 < dc) dc = d2, cover = cell;
      }
    }

    if (cover) {
      var v00 = [x0, y0], v01 = [x0, y1], v11 = [x1, y1], v10 = [x1, y0];
      cover.halfedges.push(
        edges.push(createBorderEdge(site = cover.site, v00, v01)) - 1,
        edges.push(createBorderEdge(site, v01, v11)) - 1,
        edges.push(createBorderEdge(site, v11, v10)) - 1,
        edges.push(createBorderEdge(site, v10, v00)) - 1
      );
    }
  }

  // Lastly delete any cells with no edges; these were entirely clipped.
  for (iCell = 0; iCell < nCells; ++iCell) {
    if (cell = cells[iCell]) {
      if (!cell.halfedges.length) {
        delete cells[iCell];
      }
    }
  }
}

var circlePool = [];

var firstCircle;

function Circle() {
  RedBlackNode(this);
  this.x =
  this.y =
  this.arc =
  this.site =
  this.cy = null;
}

function attachCircle(arc) {
  var lArc = arc.P,
      rArc = arc.N;

  if (!lArc || !rArc) return;

  var lSite = lArc.site,
      cSite = arc.site,
      rSite = rArc.site;

  if (lSite === rSite) return;

  var bx = cSite[0],
      by = cSite[1],
      ax = lSite[0] - bx,
      ay = lSite[1] - by,
      cx = rSite[0] - bx,
      cy = rSite[1] - by;

  var d = 2 * (ax * cy - ay * cx);
  if (d >= -epsilon2) return;

  var ha = ax * ax + ay * ay,
      hc = cx * cx + cy * cy,
      x = (cy * ha - ay * hc) / d,
      y = (ax * hc - cx * ha) / d;

  var circle = circlePool.pop() || new Circle;
  circle.arc = arc;
  circle.site = cSite;
  circle.x = x + bx;
  circle.y = (circle.cy = y + by) + Math.sqrt(x * x + y * y); // y bottom

  arc.circle = circle;

  var before = null,
      node = circles._;

  while (node) {
    if (circle.y < node.y || (circle.y === node.y && circle.x <= node.x)) {
      if (node.L) node = node.L;
      else { before = node.P; break; }
    } else {
      if (node.R) node = node.R;
      else { before = node; break; }
    }
  }

  circles.insert(before, circle);
  if (!before) firstCircle = circle;
}

function detachCircle(arc) {
  var circle = arc.circle;
  if (circle) {
    if (!circle.P) firstCircle = circle.N;
    circles.remove(circle);
    circlePool.push(circle);
    RedBlackNode(circle);
    arc.circle = null;
  }
}

var beachPool = [];

function Beach() {
  RedBlackNode(this);
  this.edge =
  this.site =
  this.circle = null;
}

function createBeach(site) {
  var beach = beachPool.pop() || new Beach;
  beach.site = site;
  return beach;
}

function detachBeach(beach) {
  detachCircle(beach);
  beaches.remove(beach);
  beachPool.push(beach);
  RedBlackNode(beach);
}

function removeBeach(beach) {
  var circle = beach.circle,
      x = circle.x,
      y = circle.cy,
      vertex = [x, y],
      previous = beach.P,
      next = beach.N,
      disappearing = [beach];

  detachBeach(beach);

  var lArc = previous;
  while (lArc.circle
      && Math.abs(x - lArc.circle.x) < epsilon
      && Math.abs(y - lArc.circle.cy) < epsilon) {
    previous = lArc.P;
    disappearing.unshift(lArc);
    detachBeach(lArc);
    lArc = previous;
  }

  disappearing.unshift(lArc);
  detachCircle(lArc);

  var rArc = next;
  while (rArc.circle
      && Math.abs(x - rArc.circle.x) < epsilon
      && Math.abs(y - rArc.circle.cy) < epsilon) {
    next = rArc.N;
    disappearing.push(rArc);
    detachBeach(rArc);
    rArc = next;
  }

  disappearing.push(rArc);
  detachCircle(rArc);

  var nArcs = disappearing.length,
      iArc;
  for (iArc = 1; iArc < nArcs; ++iArc) {
    rArc = disappearing[iArc];
    lArc = disappearing[iArc - 1];
    setEdgeEnd(rArc.edge, lArc.site, rArc.site, vertex);
  }

  lArc = disappearing[0];
  rArc = disappearing[nArcs - 1];
  rArc.edge = createEdge(lArc.site, rArc.site, null, vertex);

  attachCircle(lArc);
  attachCircle(rArc);
}

function addBeach(site) {
  var x = site[0],
      directrix = site[1],
      lArc,
      rArc,
      dxl,
      dxr,
      node = beaches._;

  while (node) {
    dxl = leftBreakPoint(node, directrix) - x;
    if (dxl > epsilon) node = node.L; else {
      dxr = x - rightBreakPoint(node, directrix);
      if (dxr > epsilon) {
        if (!node.R) {
          lArc = node;
          break;
        }
        node = node.R;
      } else {
        if (dxl > -epsilon) {
          lArc = node.P;
          rArc = node;
        } else if (dxr > -epsilon) {
          lArc = node;
          rArc = node.N;
        } else {
          lArc = rArc = node;
        }
        break;
      }
    }
  }

  createCell(site);
  var newArc = createBeach(site);
  beaches.insert(lArc, newArc);

  if (!lArc && !rArc) return;

  if (lArc === rArc) {
    detachCircle(lArc);
    rArc = createBeach(lArc.site);
    beaches.insert(newArc, rArc);
    newArc.edge = rArc.edge = createEdge(lArc.site, newArc.site);
    attachCircle(lArc);
    attachCircle(rArc);
    return;
  }

  if (!rArc) { // && lArc
    newArc.edge = createEdge(lArc.site, newArc.site);
    return;
  }

  // else lArc !== rArc
  detachCircle(lArc);
  detachCircle(rArc);

  var lSite = lArc.site,
      ax = lSite[0],
      ay = lSite[1],
      bx = site[0] - ax,
      by = site[1] - ay,
      rSite = rArc.site,
      cx = rSite[0] - ax,
      cy = rSite[1] - ay,
      d = 2 * (bx * cy - by * cx),
      hb = bx * bx + by * by,
      hc = cx * cx + cy * cy,
      vertex = [(cy * hb - by * hc) / d + ax, (bx * hc - cx * hb) / d + ay];

  setEdgeEnd(rArc.edge, lSite, rSite, vertex);
  newArc.edge = createEdge(lSite, site, null, vertex);
  rArc.edge = createEdge(site, rSite, null, vertex);
  attachCircle(lArc);
  attachCircle(rArc);
}

function leftBreakPoint(arc, directrix) {
  var site = arc.site,
      rfocx = site[0],
      rfocy = site[1],
      pby2 = rfocy - directrix;

  if (!pby2) return rfocx;

  var lArc = arc.P;
  if (!lArc) return -Infinity;

  site = lArc.site;
  var lfocx = site[0],
      lfocy = site[1],
      plby2 = lfocy - directrix;

  if (!plby2) return lfocx;

  var hl = lfocx - rfocx,
      aby2 = 1 / pby2 - 1 / plby2,
      b = hl / plby2;

  if (aby2) return (-b + Math.sqrt(b * b - 2 * aby2 * (hl * hl / (-2 * plby2) - lfocy + plby2 / 2 + rfocy - pby2 / 2))) / aby2 + rfocx;

  return (rfocx + lfocx) / 2;
}

function rightBreakPoint(arc, directrix) {
  var rArc = arc.N;
  if (rArc) return leftBreakPoint(rArc, directrix);
  var site = arc.site;
  return site[1] === directrix ? site[0] : Infinity;
}

var epsilon = 1e-6;
var epsilon2 = 1e-12;
var beaches;
var cells;
var circles;
var edges;

function triangleArea(a, b, c) {
  return (a[0] - c[0]) * (b[1] - a[1]) - (a[0] - b[0]) * (c[1] - a[1]);
}

function lexicographic(a, b) {
  return b[1] - a[1]
      || b[0] - a[0];
}

function Diagram(sites, extent) {
  var site = sites.sort(lexicographic).pop(),
      x,
      y,
      circle;

  edges = [];
  cells = new Array(sites.length);
  beaches = new RedBlackTree;
  circles = new RedBlackTree;

  while (true) {
    circle = firstCircle;
    if (site && (!circle || site[1] < circle.y || (site[1] === circle.y && site[0] < circle.x))) {
      if (site[0] !== x || site[1] !== y) {
        addBeach(site);
        x = site[0], y = site[1];
      }
      site = sites.pop();
    } else if (circle) {
      removeBeach(circle.arc);
    } else {
      break;
    }
  }

  sortCellHalfedges();

  if (extent) {
    var x0 = +extent[0][0],
        y0 = +extent[0][1],
        x1 = +extent[1][0],
        y1 = +extent[1][1];
    clipEdges(x0, y0, x1, y1);
    clipCells(x0, y0, x1, y1);
  }

  this.edges = edges;
  this.cells = cells;

  beaches =
  circles =
  edges =
  cells = null;
}

Diagram.prototype = {
  constructor: Diagram,

  polygons: function() {
    var edges = this.edges;

    return this.cells.map(function(cell) {
      var polygon = cell.halfedges.map(function(i) { return cellHalfedgeStart(cell, edges[i]); });
      polygon.data = cell.site.data;
      return polygon;
    });
  },

  triangles: function() {
    var triangles = [],
        edges = this.edges;

    this.cells.forEach(function(cell, i) {
      if (!(m = (halfedges = cell.halfedges).length)) return;
      var site = cell.site,
          halfedges,
          j = -1,
          m,
          s0,
          e1 = edges[halfedges[m - 1]],
          s1 = e1.left === site ? e1.right : e1.left;

      while (++j < m) {
        s0 = s1;
        e1 = edges[halfedges[j]];
        s1 = e1.left === site ? e1.right : e1.left;
        if (s0 && s1 && i < s0.index && i < s1.index && triangleArea(site, s0, s1) < 0) {
          triangles.push([site.data, s0.data, s1.data]);
        }
      }
    });

    return triangles;
  },

  links: function() {
    return this.edges.filter(function(edge) {
      return edge.right;
    }).map(function(edge) {
      return {
        source: edge.left.data,
        target: edge.right.data
      };
    });
  },

  find: function(x, y, radius) {
    var that = this, i0, i1 = that._found || 0, n = that.cells.length, cell;

    // Use the previously-found cell, or start with an arbitrary one.
    while (!(cell = that.cells[i1])) if (++i1 >= n) return null;
    var dx = x - cell.site[0], dy = y - cell.site[1], d2 = dx * dx + dy * dy;

    // Traverse the half-edges to find a closer cell, if any.
    do {
      cell = that.cells[i0 = i1], i1 = null;
      cell.halfedges.forEach(function(e) {
        var edge = that.edges[e], v = edge.left;
        if ((v === cell.site || !v) && !(v = edge.right)) return;
        var vx = x - v[0], vy = y - v[1], v2 = vx * vx + vy * vy;
        if (v2 < d2) d2 = v2, i1 = v.index;
      });
    } while (i1 !== null);

    that._found = i0;

    return radius == null || d2 <= radius * radius ? cell.site : null;
  }
};

var voronoi = function() {
  var x$$1 = x,
      y$$1 = y,
      extent = null;

  function voronoi(data) {
    return new Diagram(data.map(function(d, i) {
      var s = [Math.round(x$$1(d, i, data) / epsilon) * epsilon, Math.round(y$$1(d, i, data) / epsilon) * epsilon];
      s.index = i;
      s.data = d;
      return s;
    }), extent);
  }

  voronoi.polygons = function(data) {
    return voronoi(data).polygons();
  };

  voronoi.links = function(data) {
    return voronoi(data).links();
  };

  voronoi.triangles = function(data) {
    return voronoi(data).triangles();
  };

  voronoi.x = function(_) {
    return arguments.length ? (x$$1 = typeof _ === "function" ? _ : constant(+_), voronoi) : x$$1;
  };

  voronoi.y = function(_) {
    return arguments.length ? (y$$1 = typeof _ === "function" ? _ : constant(+_), voronoi) : y$$1;
  };

  voronoi.extent = function(_) {
    return arguments.length ? (extent = _ == null ? null : [[+_[0][0], +_[0][1]], [+_[1][0], +_[1][1]]], voronoi) : extent && [[extent[0][0], extent[0][1]], [extent[1][0], extent[1][1]]];
  };

  voronoi.size = function(_) {
    return arguments.length ? (extent = _ == null ? null : [[0, 0], [+_[0], +_[1]]], voronoi) : extent && [extent[1][0] - extent[0][0], extent[1][1] - extent[0][1]];
  };

  return voronoi;
};

/**
 * @private
 * @param {Array<Array<number>>} coords representing a polygon
 * @returns {Feature<Polygon>} polygon
 */
function coordsToPolygon(coords) {
    coords = coords.slice();
    coords.push(coords[0]);
    return polygon([coords]);
}

/**
 * Takes a FeatureCollection of points, and a bounding box, and returns a FeatureCollection
 * of Voronoi polygons.
 *
 * The Voronoi algorithim used comes from the d3-voronoi package.
 *
 * @name voronoi
 * @param {FeatureCollection<Point>} points to find the Voronoi polygons around.
 * @param {Object} [options={}] Optional parameters
 * @param {number[]} [options.bbox=[-180, -85, 180, -85]] clipping rectangle, in [minX, minY, maxX, MaxY] order.
 * @returns {FeatureCollection<Polygon>} a set of polygons, one per input point.
 * @example
 * var options = {
 *   bbox: [-70, 40, -60, 60]
 * };
 * var points = turf.randomPoint(100, options);
 * var voronoiPolygons = turf.voronoi(points, options);
 *
 * //addToMap
 * var addToMap = [voronoiPolygons, points];
 */
function voronoi$1(points$$1, options) {
    // Optional params
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var bbox = options.bbox || [-180, -85, 180, 85];

    // Input Validation
    if (!points$$1) throw new Error('points is required');
    if (!Array.isArray(bbox)) throw new Error('bbox is invalid');
    collectionOf(points$$1, 'Point', 'points');

    // Main
    return featureCollection(
        voronoi()
            .x(function (feature$$1) { return feature$$1.geometry.coordinates[0]; })
            .y(function (feature$$1) { return feature$$1.geometry.coordinates[1]; })
            .extent([[bbox[0], bbox[1]], [bbox[2], bbox[3]]])
            .polygons(points$$1.features)
            .map(coordsToPolygon)
    );
}

/**
 * Takes a {@link Point} and calculates the ellipse polygon given two semi-axes expressed in variable units and steps for precision.
 *
 * @param {Coord} center center point
 * @param {number} xSemiAxis semi (major) axis of the ellipse along the x-axis
 * @param {number} ySemiAxis semi (minor) axis of the ellipse along the y-axis
 * @param {Object} [options={}] Optional parameters
 * @param {number} [options.angle=0] angle of rotation (along the vertical axis), from North in decimal degrees, negative clockwise
 * @param {Coord} [options.pivot='origin'] point around which the rotation will be performed
 * @param {number} [options.steps=64] number of steps
 * @param {string} [options.units='kilometers'] unit of measurement for axes
 * @param {Object} [options.properties={}] properties
 * @returns {Feature<Polygon>} ellipse polygon
 * @example
 * var center = [-75, 40];
 * var xSemiAxis = 5;
 * var ySemiAxis = 2;
 * var ellipse = turf.ellipse(center, xSemiAxis, ySemiAxis);
 *
 * //addToMap
 * var addToMap = [turf.point(center), ellipse]
 */
function ellipse(center, xSemiAxis, ySemiAxis, options) {
    // Optional params
    options = options || {};
    var steps = options.steps || 64;
    var units = options.units || 'kilometers';
    var angle = options.angle || 0;
    var pivot = options.pivot || center;
    var properties = options.properties || center.properties || {};

    // validation
    if (!center) throw new Error('center is required');
    if (!xSemiAxis) throw new Error('xSemiAxis is required');
    if (!ySemiAxis) throw new Error('ySemiAxis is required');
    if (!isObject(options)) throw new Error('options must be an object');
    if (!isNumber(steps)) throw new Error('steps must be a number');
    if (!isNumber(angle)) throw new Error('angle must be a number');

    var centerCoords = getCoord(center);
    if (units === 'degrees') {
        var angleRad = degreesToRadians(angle);
    } else {
        xSemiAxis = rhumbDestination(center, xSemiAxis, 90, {units: units});
        ySemiAxis = rhumbDestination(center, ySemiAxis, 0, {units: units});
        xSemiAxis = getCoord(xSemiAxis)[0] - centerCoords[0];
        ySemiAxis = getCoord(ySemiAxis)[1] - centerCoords[1];
    }

    var coordinates = [];
    for (var i = 0; i < steps; i += 1) {
        var stepAngle = i * -360 / steps;
        var x = ((xSemiAxis * ySemiAxis) / Math.sqrt(Math.pow(ySemiAxis, 2) + (Math.pow(xSemiAxis, 2) * Math.pow(getTanDeg(stepAngle), 2))));
        var y = ((xSemiAxis * ySemiAxis) / Math.sqrt(Math.pow(xSemiAxis, 2) + (Math.pow(ySemiAxis, 2) / Math.pow(getTanDeg(stepAngle), 2))));

        if (stepAngle < -90 && stepAngle >= -270) x = -x;
        if (stepAngle < -180 && stepAngle >= -360) y = -y;
        if (units === 'degrees') {
            var newx = x * Math.cos(angleRad) + y * Math.sin(angleRad);
            var newy = y * Math.cos(angleRad) - x * Math.sin(angleRad);
            x = newx;
            y = newy;
        }

        coordinates.push([x + centerCoords[0], y + centerCoords[1]]);
    }
    coordinates.push(coordinates[0]);
    if (units === 'degrees') {
        return polygon([coordinates], properties);
    } else {
        return transformRotate(polygon([coordinates], properties), angle, { pivot: pivot });
    }
}

/**
 * Get Tan Degrees
 *
 * @private
 * @param {number} deg Degrees
 * @returns {number} Tan Degrees
 */
function getTanDeg(deg) {
    var rad = deg * Math.PI / 180;
    return Math.tan(rad);
}

/**
 * Takes a {@link Feature} or {@link FeatureCollection} and returns the mean center. Can be weighted.
 *
 * @name centerMean
 * @param {GeoJSON} geojson GeoJSON to be centered
 * @param {Object} [options={}] Optional parameters
 * @param {Object} [options.properties={}] an Object that is used as the {@link Feature}'s properties
 * @param {number} [options.weight] the property name used to weight the center
 * @returns {Feature<Point>} a Point feature at the mean center point of all input features
 * @example
 * var features = turf.featureCollection([
 *   turf.point([-97.522259, 35.4691], {weight: 10}),
 *   turf.point([-97.502754, 35.463455], {weight: 3}),
 *   turf.point([-97.508269, 35.463245], {weight: 5})
 * ]);
 *
 * var options = {weight: "weight"}
 * var mean = turf.centerMean(features, options);
 *
 * //addToMap
 * var addToMap = [features, mean]
 * mean.properties['marker-size'] = 'large';
 * mean.properties['marker-color'] = '#000';
 */
function centerMean(geojson, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var properties = options.properties;
    var weightTerm = options.weight;

    // Input validation
    if (!geojson) throw new Error('geojson is required');

    var sumXs = 0;
    var sumYs = 0;
    var sumNs = 0;
    geomEach(geojson, function (geom, featureIndex, properties) {
        var weight = properties[weightTerm];
        weight = (weight === undefined || weight === null) ? 1 : weight;
        if (!isNumber(weight)) throw new Error('weight value must be a number for feature index ' + featureIndex);
        weight = Number(weight);
        if (weight > 0) {
            coordEach(geom, function (coord) {
                sumXs += coord[0] * weight;
                sumYs += coord[1] * weight;
                sumNs += weight;
            });
        }
    });
    return point([sumXs / sumNs, sumYs / sumNs], properties);
}

/**
 * Takes a {@link FeatureCollection} of points and calculates the median center,
 * algorithimically. The median center is understood as the point that is
 * requires the least total travel from all other points.
 *
 * Turfjs has four different functions for calculating the center of a set of
 * data. Each is useful depending on circumstance.
 *
 * `@turf/center` finds the simple center of a dataset, by finding the
 * midpoint between the extents of the data. That is, it divides in half the
 * farthest east and farthest west point as well as the farthest north and
 * farthest south.
 *
 * `@turf/center-of-mass` imagines that the dataset is a sheet of paper.
 * The center of mass is where the sheet would balance on a fingertip.
 *
 * `@turf/center-mean` takes the averages of all the coordinates and
 * produces a value that respects that. Unlike `@turf/center`, it is
 * sensitive to clusters and outliers. It lands in the statistical middle of a
 * dataset, not the geographical. It can also be weighted, meaning certain
 * points are more important than others.
 *
 * `@turf/center-median` takes the mean center and tries to find, iteratively,
 * a new point that requires the least amount of travel from all the points in
 * the dataset. It is not as sensitive to outliers as `@turf/center`, but it is
 * attracted to clustered data. It, too, can be weighted.
 *
 * **Bibliography**
 *
 * Harold W. Kuhn and Robert E. Kuenne, “An Efficient Algorithm for the
 * Numerical Solution of the Generalized Weber Problem in Spatial
 * Economics,” _Journal of Regional Science_ 4, no. 2 (1962): 21–33,
 * doi:{@link https://doi.org/10.1111/j.1467-9787.1962.tb00902.x}.
 *
 * James E. Burt, Gerald M. Barber, and David L. Rigby, _Elementary
 * Statistics for Geographers_, 3rd ed., New York: The Guilford
 * Press, 2009, 150–151.
 *
 * @name centerMedian
 * @param {FeatureCollection<any>} features Any GeoJSON Feature Collection
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.weight] the property name used to weight the center
 * @param {number} [options.tolerance=0.001] the difference in distance between candidate medians at which point the algorighim stops iterating.
 * @param {number} [options.counter=10] how many attempts to find the median, should the tolerance be insufficient.
 * @returns {Feature<Point>} The median center of the collection
 * @example
 * var points = turf.points([[0, 0], [1, 0], [0, 1], [5, 8]]);
 * var medianCenter = turf.centerMedian(points);
 *
 * //addToMap
 * var addToMap = [points, medianCenter]
 */
function centerMedian(features, options) {
    // Optional params
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var counter = options.counter || 10;
    if (!isNumber(counter)) throw new Error('counter must be a number');
    var weightTerm = options.weight;

    // Calculate mean center:
    var meanCenter = centerMean(features, {weight: options.weight});

    // Calculate center of every feature:
    var centroids = featureCollection([]);
    featureEach(features, function (feature$$1) {
        centroids.features.push(centroid(feature$$1, {weight: feature$$1.properties[weightTerm]}));
    });

    centroids.properties = {
        tolerance: options.tolerance,
        medianCandidates: []
    };
    return findMedian(meanCenter.geometry.coordinates, [0, 0], centroids, counter);
}

/**
 * Recursive function to find new candidate medians.
 *
 * @private
 * @param {Position} candidateMedian current candidate median
 * @param {Position} previousCandidate the previous candidate median
 * @param {FeatureCollection<Point>} centroids the collection of centroids whose median we are determining
 * @param {number} counter how many attempts to try before quitting.
 * @returns {Feature<Point>} the median center of the dataset.
 */
function findMedian(candidateMedian, previousCandidate, centroids, counter) {
    var tolerance = centroids.properties.tolerance || 0.001;
    var candidateXsum = 0;
    var candidateYsum = 0;
    var kSum = 0;
    var centroidCount = 0;
    featureEach(centroids, function (theCentroid) {
        var weightValue = theCentroid.properties.weight;
        var weight = (weightValue === undefined || weightValue === null) ? 1 : weightValue;
        weight = Number(weight);
        if (!isNumber(weight)) throw new Error('weight value must be a number');
        if (weight > 0) {
            centroidCount += 1;
            var distanceFromCandidate = weight * distance(theCentroid, candidateMedian);
            if (distanceFromCandidate === 0) distanceFromCandidate = 1;
            var k = weight / distanceFromCandidate;
            candidateXsum += theCentroid.geometry.coordinates[0] * k;
            candidateYsum += theCentroid.geometry.coordinates[1] * k;
            kSum += k;
        }
    });
    if (centroidCount < 1) throw new Error('no features to measure');
    var candidateX = candidateXsum / kSum;
    var candidateY = candidateYsum / kSum;
    if (centroidCount === 1 || counter === 0 || (Math.abs(candidateX - previousCandidate[0]) < tolerance && Math.abs(candidateY - previousCandidate[1]) < tolerance)) {
        return point([candidateX, candidateY], {medianCandidates: centroids.properties.medianCandidates});
    } else {
        centroids.properties.medianCandidates.push([candidateX, candidateY]);
        return findMedian([candidateX, candidateY], candidateMedian, centroids, counter - 1);
    }
}

/**
 * Takes a {@link FeatureCollection} and returns a standard deviational ellipse,
 * also known as a “directional distribution.” The standard deviational ellipse
 * aims to show the direction and the distribution of a dataset by drawing
 * an ellipse that contains about one standard deviation’s worth (~ 70%) of the
 * data.
 *
 * This module mirrors the functionality of [Directional Distribution](http://desktop.arcgis.com/en/arcmap/10.3/tools/spatial-statistics-toolbox/directional-distribution.htm)
 * in ArcGIS and the [QGIS Standard Deviational Ellipse Plugin](http://arken.nmbu.no/~havatv/gis/qgisplugins/SDEllipse/)
 *
 * **Bibliography**
 *
 * • Robert S. Yuill, “The Standard Deviational Ellipse; An Updated Tool for
 * Spatial Description,” _Geografiska Annaler_ 53, no. 1 (1971): 28–39,
 * doi:{@link https://doi.org/10.2307/490885|10.2307/490885}.
 *
 * • Paul Hanly Furfey, “A Note on Lefever’s “Standard Deviational Ellipse,”
 * _American Journal of Sociology_ 33, no. 1 (1927): 94—98,
 * doi:{@link https://doi.org/10.1086/214336|10.1086/214336}.
 *
 *
 * @name standardDeviationalEllipse
 * @param {FeatureCollection<Point>} points GeoJSON points
 * @param {Object} [options={}] Optional parameters
 * @param {string} [options.weight] the property name used to weight the center
 * @param {number} [options.steps=64] number of steps for the polygon
 * @param {Object} [options.properties={}] properties to pass to the resulting ellipse
 * @returns {Feature<Polygon>} an elliptical Polygon that includes approximately 1 SD of the dataset within it.
 * @example
 *
 * var bbox = [-74, 40.72, -73.98, 40.74];
 * var points = turf.randomPoint(400, {bbox: bbox});
 * var sdEllipse = turf.standardDeviationalEllipse(points);
 *
 * //addToMap
 * var addToMap = [points, sdEllipse];
 *
 */
function standardDeviationalEllipse(points$$1, options) {
    // Optional params
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var steps = options.steps || 64;
    var weightTerm = options.weight;
    var properties = options.properties || {};

    // Validation:
    if (!isNumber(steps)) throw new Error('steps must be a number');
    if (!isObject(properties)) throw new Error('properties must be a number');

    // Calculate mean center & number of features:
    var numberOfFeatures = coordAll(points$$1).length;
    var meanCenter = centerMean(points$$1, {weight: weightTerm});

    // Calculate angle of rotation:
    // [X, Y] = mean center of all [x, y].
    // theta = arctan( (A + B) / C )
    // A = sum((x - X)^2) - sum((y - Y)^2)
    // B = sqrt(A^2 + 4(sum((x - X)(y - Y))^2))
    // C = 2(sum((x - X)(y - Y)))

    var xDeviationSquaredSum = 0;
    var yDeviationSquaredSum = 0;
    var xyDeviationSum = 0;

    featureEach(points$$1, function (point$$1) {
        var weight = point$$1.properties[weightTerm] || 1;
        var deviation = getDeviations(getCoords(point$$1), getCoords(meanCenter));
        xDeviationSquaredSum += Math.pow(deviation.x, 2) * weight;
        yDeviationSquaredSum += Math.pow(deviation.y, 2) * weight;
        xyDeviationSum += deviation.x * deviation.y * weight;
    });

    var bigA = xDeviationSquaredSum - yDeviationSquaredSum;
    var bigB = Math.sqrt(Math.pow(bigA, 2) + 4 * Math.pow(xyDeviationSum, 2));
    var bigC = 2 * xyDeviationSum;
    var theta = Math.atan((bigA + bigB) / bigC);
    var thetaDeg = theta * 180 / Math.PI;

    // Calculate axes:
    // sigmaX = sqrt((1 / n - 2) * sum((((x - X) * cos(theta)) - ((y - Y) * sin(theta)))^2))
    // sigmaY = sqrt((1 / n - 2) * sum((((x - X) * sin(theta)) - ((y - Y) * cos(theta)))^2))
    var sigmaXsum = 0;
    var sigmaYsum = 0;
    var weightsum = 0;
    featureEach(points$$1, function (point$$1) {
        var weight = point$$1.properties[weightTerm] || 1;
        var deviation = getDeviations(getCoords(point$$1), getCoords(meanCenter));
        sigmaXsum += Math.pow((deviation.x * Math.cos(theta)) - (deviation.y * Math.sin(theta)), 2) * weight;
        sigmaYsum += Math.pow((deviation.x * Math.sin(theta)) + (deviation.y * Math.cos(theta)), 2) * weight;
        weightsum += weight;
    });

    var sigmaX = Math.sqrt(2 * sigmaXsum / weightsum);
    var sigmaY = Math.sqrt(2 * sigmaYsum / weightsum);

    var theEllipse = ellipse(meanCenter, sigmaX, sigmaY, {units: 'degrees', angle: thetaDeg, steps: steps, properties: properties});
    var pointsWithinEllipse = pointsWithinPolygon(points$$1, featureCollection([theEllipse]));
    var standardDeviationalEllipseProperties = {
        meanCenterCoordinates: getCoords(meanCenter),
        semiMajorAxis: sigmaX,
        semiMinorAxis: sigmaY,
        numberOfFeatures: numberOfFeatures,
        angle: thetaDeg,
        percentageWithinEllipse: 100 * coordAll(pointsWithinEllipse).length / numberOfFeatures
    };
    theEllipse.properties.standardDeviationalEllipse = standardDeviationalEllipseProperties;

    return theEllipse;
}

/**
 * Get x_i - X and y_i - Y
 *
 * @private
 * @param {Array} coordinates Array of [x_i, y_i]
 * @param {Array} center Array of [X, Y]
 * @returns {Object} { x: n, y: m }
 */
function getDeviations(coordinates, center) {
    return {
        x: coordinates[0] - center[0],
        y: coordinates[1] - center[1]
    };
}

/**
 * Returns a random position within a {@link bounding box}.
 *
 * @name randomPosition
 * @param {Array<number>} [bbox=[-180, -90, 180, 90]] a bounding box inside of which positions are placed.
 * @returns {Array<number>} Position [longitude, latitude]
 * @example
 * var position = turf.randomPosition([-180, -90, 180, 90])
 * //=position
 */
function randomPosition(bbox) {
    if (isObject(bbox)) bbox = bbox.bbox;
    if (bbox && !Array.isArray(bbox)) throw new Error('bbox is invalid');
    if (bbox) return coordInBBox(bbox);
    else return [lon(), lat()];
}

/**
 * Returns a random {@link point}.
 *
 * @name randomPoint
 * @param {number} [count=1] how many geometries will be generated
 * @param {Object} [options={}] Optional parameters
 * @param {Array<number>} [options.bbox=[-180, -90, 180, 90]] a bounding box inside of which geometries are placed.
 * @returns {FeatureCollection<Point>} GeoJSON FeatureCollection of points
 * @example
 * var points = turf.randomPoint(25, {bbox: [-180, -90, 180, 90]})
 * //=points
 */
function randomPoint(count, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var bbox = options.bbox;
    if (count === undefined || count === null) count = 1;

    var features = [];
    for (var i = 0; i < count; i++) {
        features.push(point(randomPosition(bbox)));
    }
    return featureCollection(features);
}

/**
 * Returns a random {@link polygon}.
 *
 * @name randomPolygon
 * @param {number} [count=1] how many geometries will be generated
 * @param {Object} [options={}] Optional parameters
 * @param {Array<number>} [options.bbox=[-180, -90, 180, 90]] a bounding box inside of which geometries are placed.
 * @param {number} [options.num_vertices=10] is how many coordinates each LineString will contain.
 * @param {number} [options.max_radial_length=10] is the maximum number of decimal degrees latitude or longitude that a vertex can reach out of the center of the Polygon.
 * @returns {FeatureCollection<Point>} GeoJSON FeatureCollection of points
 * @example
 * var polygons = turf.randomPolygon(25, {bbox: [-180, -90, 180, 90]})
 * //=polygons
 */
function randomPolygon(count, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var bbox = options.bbox;
    var num_vertices = options.num_vertices;
    var max_radial_length = options.max_radial_length;
    if (count === undefined || count === null) count = 1;

    // Validation
    if (!isNumber(num_vertices)) num_vertices = 10;
    if (!isNumber(max_radial_length)) max_radial_length = 10;

    var features = [];
    for (var i = 0; i < count; i++) {
        var vertices = [],
            circle_offsets = Array.apply(null,
                new Array(num_vertices + 1)).map(Math.random);

        circle_offsets.forEach(sumOffsets);
        circle_offsets.forEach(scaleOffsets);
        vertices[vertices.length - 1] = vertices[0]; // close the ring

        // center the polygon around something
        vertices = vertices.map(vertexToCoordinate(randomPosition(bbox)));
        features.push(polygon([vertices]));
    }

    function sumOffsets(cur, index, arr) {
        arr[index] = (index > 0) ? cur + arr[index - 1] : cur;
    }

    function scaleOffsets(cur) {
        cur = cur * 2 * Math.PI / circle_offsets[circle_offsets.length - 1];
        var radial_scaler = Math.random();
        vertices.push([
            radial_scaler * max_radial_length * Math.sin(cur),
            radial_scaler * max_radial_length * Math.cos(cur)
        ]);
    }

    return featureCollection(features);
}

/**
 * Returns a random {@link linestring}.
 *
 * @name randomLineString
 * @param {number} [count=1] how many geometries will be generated
 * @param {Object} [options={}] Optional parameters
 * @param {Array<number>} [options.bbox=[-180, -90, 180, 90]] a bounding box inside of which geometries are placed.
 * @param {number} [options.num_vertices=10] is how many coordinates each LineString will contain.
 * @param {number} [options.max_length=0.0001] is the maximum number of decimal degrees that a vertex can be from its predecessor
 * @param {number} [options.max_rotation=Math.PI / 8] is the maximum number of radians that a line segment can turn from the previous segment.
 * @returns {FeatureCollection<Point>} GeoJSON FeatureCollection of points
 * @example
 * var lineStrings = turf.randomLineString(25, {bbox: [-180, -90, 180, 90]})
 * //=lineStrings
 */
function randomLineString(count, options) {
    // Optional parameters
    options = options || {};
    if (!isObject(options)) throw new Error('options is invalid');
    var bbox = options.bbox;
    var num_vertices = options.num_vertices;
    var max_length = options.max_length;
    var max_rotation = options.max_rotation;
    if (count === undefined || count === null) count = 1;

    // Default parameters
    if (!isNumber(num_vertices) || num_vertices < 2) num_vertices = 10;
    if (!isNumber(max_length)) max_length = 0.0001;
    if (!isNumber(max_rotation)) max_rotation = Math.PI / 8;

    var features = [];
    for (var i = 0; i < count; i++) {
        var startingPoint = randomPosition(bbox);
        var vertices = [startingPoint];
        for (var j = 0; j < num_vertices - 1; j++) {
            var priorAngle = (j === 0) ?
                Math.random() * 2 * Math.PI :
                Math.tan(
                    (vertices[j][1] - vertices[j - 1][1]) /
              (vertices[j][0] - vertices[j - 1][0])
                );
            var angle = priorAngle + (Math.random() - 0.5) * max_rotation * 2;
            var distance = Math.random() * max_length;
            vertices.push([
                vertices[j][0] + distance * Math.cos(angle),
                vertices[j][1] + distance * Math.sin(angle)
            ]);
        }
        features.push(lineString(vertices));
    }

    return featureCollection(features);
}

function vertexToCoordinate(hub) {
    return function (cur) { return [cur[0] + hub[0], cur[1] + hub[1]]; };
}

function rnd() { return Math.random() - 0.5; }
function lon() { return rnd() * 360; }
function lat() { return rnd() * 180; }

function coordInBBox(bbox) {
    return [
        (Math.random() * (bbox[2] - bbox[0])) + bbox[0],
        (Math.random() * (bbox[3] - bbox[1])) + bbox[1]];
}




var main_es$4 = Object.freeze({
	randomPosition: randomPosition,
	randomPoint: randomPoint,
	randomPolygon: randomPolygon,
	randomLineString: randomLineString
});

/**
 * Get Cluster
 *
 * @name getCluster
 * @param {FeatureCollection} geojson GeoJSON Features
 * @param {*} filter Filter used on GeoJSON properties to get Cluster
 * @returns {FeatureCollection} Single Cluster filtered by GeoJSON Properties
 * @example
 * var geojson = turf.featureCollection([
 *     turf.point([0, 0], {'marker-symbol': 'circle'}),
 *     turf.point([2, 4], {'marker-symbol': 'star'}),
 *     turf.point([3, 6], {'marker-symbol': 'star'}),
 *     turf.point([5, 1], {'marker-symbol': 'square'}),
 *     turf.point([4, 2], {'marker-symbol': 'circle'})
 * ]);
 *
 * // Create a cluster using K-Means (adds `cluster` to GeoJSON properties)
 * var clustered = turf.clustersKmeans(geojson);
 *
 * // Retrieve first cluster (0)
 * var cluster = turf.getCluster(clustered, {cluster: 0});
 * //= cluster
 *
 * // Retrieve cluster based on custom properties
 * turf.getCluster(clustered, {'marker-symbol': 'circle'}).length;
 * //= 2
 * turf.getCluster(clustered, {'marker-symbol': 'square'}).length;
 * //= 1
 */
function getCluster(geojson, filter) {
    // Validation
    if (!geojson) throw new Error('geojson is required');
    if (geojson.type !== 'FeatureCollection') throw new Error('geojson must be a FeatureCollection');
    if (filter === undefined || filter === null) throw new Error('filter is required');

    // Filter Features
    var features = [];
    featureEach(geojson, function (feature$$1) {
        if (applyFilter(feature$$1.properties, filter)) features.push(feature$$1);
    });
    return featureCollection(features);
}

/**
 * Callback for clusterEach
 *
 * @callback clusterEachCallback
 * @param {FeatureCollection} [cluster] The current cluster being processed.
 * @param {*} [clusterValue] Value used to create cluster being processed.
 * @param {number} [currentIndex] The index of the current element being processed in the array.Starts at index 0
 * @returns {void}
 */

/**
 * clusterEach
 *
 * @name clusterEach
 * @param {FeatureCollection} geojson GeoJSON Features
 * @param {string|number} property GeoJSON property key/value used to create clusters
 * @param {Function} callback a method that takes (cluster, clusterValue, currentIndex)
 * @returns {void}
 * @example
 * var geojson = turf.featureCollection([
 *     turf.point([0, 0]),
 *     turf.point([2, 4]),
 *     turf.point([3, 6]),
 *     turf.point([5, 1]),
 *     turf.point([4, 2])
 * ]);
 *
 * // Create a cluster using K-Means (adds `cluster` to GeoJSON properties)
 * var clustered = turf.clustersKmeans(geojson);
 *
 * // Iterate over each cluster
 * turf.clusterEach(clustered, 'cluster', function (cluster, clusterValue, currentIndex) {
 *     //= cluster
 *     //= clusterValue
 *     //= currentIndex
 * })
 *
 * // Calculate the total number of clusters
 * var total = 0
 * turf.clusterEach(clustered, 'cluster', function () {
 *     total++;
 * });
 *
 * // Create an Array of all the values retrieved from the 'cluster' property
 * var values = []
 * turf.clusterEach(clustered, 'cluster', function (cluster, clusterValue) {
 *     values.push(clusterValue);
 * });
 */
function clusterEach(geojson, property, callback) {
    // Validation
    if (!geojson) throw new Error('geojson is required');
    if (geojson.type !== 'FeatureCollection') throw new Error('geojson must be a FeatureCollection');
    if (property === undefined || property === null) throw new Error('property is required');

    // Create clusters based on property values
    var bins = createBins(geojson, property);
    var values = Object.keys(bins);
    for (var index = 0; index < values.length; index++) {
        var value = values[index];
        var bin = bins[value];
        var features = [];
        for (var i = 0; i < bin.length; i++) {
            features.push(geojson.features[bin[i]]);
        }
        callback(featureCollection(features), value, index);
    }
}

/**
 * Callback for clusterReduce
 *
 * The first time the callback function is called, the values provided as arguments depend
 * on whether the reduce method has an initialValue argument.
 *
 * If an initialValue is provided to the reduce method:
 *  - The previousValue argument is initialValue.
 *  - The currentValue argument is the value of the first element present in the array.
 *
 * If an initialValue is not provided:
 *  - The previousValue argument is the value of the first element present in the array.
 *  - The currentValue argument is the value of the second element present in the array.
 *
 * @callback clusterReduceCallback
 * @param {*} [previousValue] The accumulated value previously returned in the last invocation
 * of the callback, or initialValue, if supplied.
 * @param {FeatureCollection} [cluster] The current cluster being processed.
 * @param {*} [clusterValue] Value used to create cluster being processed.
 * @param {number} [currentIndex] The index of the current element being processed in the
 * array. Starts at index 0, if an initialValue is provided, and at index 1 otherwise.
 */

/**
 * Reduce clusters in GeoJSON Features, similar to Array.reduce()
 *
 * @name clusterReduce
 * @param {FeatureCollection} geojson GeoJSON Features
 * @param {string|number} property GeoJSON property key/value used to create clusters
 * @param {Function} callback a method that takes (previousValue, cluster, clusterValue, currentIndex)
 * @param {*} [initialValue] Value to use as the first argument to the first call of the callback.
 * @returns {*} The value that results from the reduction.
 * @example
 * var geojson = turf.featureCollection([
 *     turf.point([0, 0]),
 *     turf.point([2, 4]),
 *     turf.point([3, 6]),
 *     turf.point([5, 1]),
 *     turf.point([4, 2])
 * ]);
 *
 * // Create a cluster using K-Means (adds `cluster` to GeoJSON properties)
 * var clustered = turf.clustersKmeans(geojson);
 *
 * // Iterate over each cluster and perform a calculation
 * var initialValue = 0
 * turf.clusterReduce(clustered, 'cluster', function (previousValue, cluster, clusterValue, currentIndex) {
 *     //=previousValue
 *     //=cluster
 *     //=clusterValue
 *     //=currentIndex
 *     return previousValue++;
 * }, initialValue);
 *
 * // Calculate the total number of clusters
 * var total = turf.clusterReduce(clustered, 'cluster', function (previousValue) {
 *     return previousValue++;
 * }, 0);
 *
 * // Create an Array of all the values retrieved from the 'cluster' property
 * var values = turf.clusterReduce(clustered, 'cluster', function (previousValue, cluster, clusterValue) {
 *     return previousValue.concat(clusterValue);
 * }, []);
 */
function clusterReduce(geojson, property, callback, initialValue) {
    var previousValue = initialValue;
    clusterEach(geojson, property, function (cluster, clusterValue, currentIndex) {
        if (currentIndex === 0 && initialValue === undefined) previousValue = cluster;
        else previousValue = callback(previousValue, cluster, clusterValue, currentIndex);
    });
    return previousValue;
}

/**
 * Create Bins
 *
 * @private
 * @param {FeatureCollection} geojson GeoJSON Features
 * @param {string|number} property Property values are used to create bins
 * @returns {Object} bins with Feature IDs
 * @example
 * var geojson = turf.featureCollection([
 *     turf.point([0, 0], {cluster: 0, foo: 'null'}),
 *     turf.point([2, 4], {cluster: 1, foo: 'bar'}),
 *     turf.point([5, 1], {0: 'foo'}),
 *     turf.point([3, 6], {cluster: 1}),
 * ]);
 * createBins(geojson, 'cluster');
 * //= { '0': [ 0 ], '1': [ 1, 3 ] }
 */
function createBins(geojson, property) {
    var bins = {};

    featureEach(geojson, function (feature$$1, i) {
        var properties = feature$$1.properties || {};
        if (properties.hasOwnProperty(property)) {
            var value = properties[property];
            if (bins.hasOwnProperty(value)) bins[value].push(i);
            else bins[value] = [i];
        }
    });
    return bins;
}

/**
 * Apply Filter
 *
 * @private
 * @param {*} properties Properties
 * @param {*} filter Filter
 * @returns {boolean} applied Filter to properties
 */
function applyFilter(properties, filter) {
    if (properties === undefined) return false;
    var filterType = typeof filter;

    // String & Number
    if (filterType === 'number' || filterType === 'string') return properties.hasOwnProperty(filter);
    // Array
    else if (Array.isArray(filter)) {
        for (var i = 0; i < filter.length; i++) {
            if (!applyFilter(properties, filter[i])) return false;
        }
        return true;
    // Object
    } else {
        return propertiesContainsFilter(properties, filter);
    }
}

/**
 * Properties contains filter (does not apply deepEqual operations)
 *
 * @private
 * @param {*} properties Properties
 * @param {Object} filter Filter
 * @returns {boolean} does filter equal Properties
 * @example
 * propertiesContainsFilter({foo: 'bar', cluster: 0}, {cluster: 0})
 * //= true
 * propertiesContainsFilter({foo: 'bar', cluster: 0}, {cluster: 1})
 * //= false
 */
function propertiesContainsFilter(properties, filter) {
    var keys = Object.keys(filter);
    for (var i = 0; i < keys.length; i++) {
        var key = keys[i];
        if (properties[key] !== filter[key]) return false;
    }
    return true;
}

/**
 * Filter Properties
 *
 * @private
 * @param {*} properties Properties
 * @param {Array<string>} keys Used to filter Properties
 * @returns {*} filtered Properties
 * @example
 * filterProperties({foo: 'bar', cluster: 0}, ['cluster'])
 * //= {cluster: 0}
 */
function filterProperties(properties, keys) {
    if (!keys) return {};
    if (!keys.length) return {};

    var newProperties = {};
    for (var i = 0; i < keys.length; i++) {
        var key = keys[i];
        if (properties.hasOwnProperty(key)) newProperties[key] = properties[key];
    }
    return newProperties;
}




var main_es$5 = Object.freeze({
	getCluster: getCluster,
	clusterEach: clusterEach,
	clusterReduce: clusterReduce,
	createBins: createBins,
	applyFilter: applyFilter,
	propertiesContainsFilter: propertiesContainsFilter,
	filterProperties: filterProperties
});

/* Polyfill service v3.13.0
 * For detailed credits and licence information see http://github.com/financial-times/polyfill-service
 *
 * - Array.prototype.fill, License: CC0 */

if (!('fill' in Array.prototype)) {
  Object.defineProperty(Array.prototype, 'fill', {
    configurable: true,
    value: function fill (value) {
      if (this === undefined || this === null) {
        throw new TypeError(this + ' is not an object')
      }

      var arrayLike = Object(this);

      var length = Math.max(Math.min(arrayLike.length, 9007199254740991), 0) || 0;

      var relativeStart = 1 in arguments ? parseInt(Number(arguments[1]), 10) || 0 : 0;

      relativeStart = relativeStart < 0 ? Math.max(length + relativeStart, 0) : Math.min(relativeStart, length);

      var relativeEnd = 2 in arguments && arguments[2] !== undefined ? parseInt(Number(arguments[2]), 10) || 0 : length;

      relativeEnd = relativeEnd < 0 ? Math.max(length + arguments[2], 0) : Math.min(relativeEnd, length);

      while (relativeStart < relativeEnd) {
        arrayLike[relativeStart] = value;

        ++relativeStart;
      }

      return arrayLike
    },
    writable: true
  });
}

/**
 * Polyfill for IE support
 */
Number.isFinite = Number.isFinite || function (value) {
  return typeof value === 'number' && isFinite(value)
};

Number.isInteger = Number.isInteger || function (val) {
  return typeof val === 'number' &&
  isFinite(val) &&
  Math.floor(val) === val
};

Number.parseFloat = Number.parseFloat || parseFloat;

Number.isNaN = Number.isNaN || function (value) {
  return value !== value // eslint-disable-line
};

/**
 * Polyfill for IE support
 */
Math.trunc = Math.trunc || function (x) {
  return x < 0 ? Math.ceil(x) : Math.floor(x)
};

var NumberUtil = function NumberUtil () {};

NumberUtil.prototype.interfaces_ = function interfaces_ () {
  return []
};
NumberUtil.prototype.getClass = function getClass () {
  return NumberUtil
};
NumberUtil.prototype.equalsWithTolerance = function equalsWithTolerance (x1, x2, tolerance) {
  return Math.abs(x1 - x2) <= tolerance
};

var IllegalArgumentException = function IllegalArgumentException () {};

var Double = function Double () {};

var staticAccessors$1 = { MAX_VALUE: { configurable: true } };

Double.isNaN = function isNaN (n) { return Number.isNaN(n) };
Double.doubleToLongBits = function doubleToLongBits (n) { return n };
Double.longBitsToDouble = function longBitsToDouble (n) { return n };
Double.isInfinite = function isInfinite (n) { return !Number.isFinite(n) };
staticAccessors$1.MAX_VALUE.get = function () { return Number.MAX_VALUE };

Object.defineProperties( Double, staticAccessors$1 );

var Comparable = function Comparable () {};

var Clonable = function Clonable () {};

var Comparator = function Comparator () {};

function Serializable () {}

// import Assert from '../util/Assert'

var Coordinate = function Coordinate () {
  this.x = null;
  this.y = null;
  this.z = null;
  if (arguments.length === 0) {
    this.x = 0.0;
    this.y = 0.0;
    this.z = Coordinate.NULL_ORDINATE;
  } else if (arguments.length === 1) {
    var c = arguments[0];
    this.x = c.x;
    this.y = c.y;
    this.z = c.z;
  } else if (arguments.length === 2) {
    this.x = arguments[0];
    this.y = arguments[1];
    this.z = Coordinate.NULL_ORDINATE;
  } else if (arguments.length === 3) {
    this.x = arguments[0];
    this.y = arguments[1];
    this.z = arguments[2];
  }
};

var staticAccessors = { DimensionalComparator: { configurable: true },serialVersionUID: { configurable: true },NULL_ORDINATE: { configurable: true },X: { configurable: true },Y: { configurable: true },Z: { configurable: true } };
Coordinate.prototype.setOrdinate = function setOrdinate (ordinateIndex, value) {
  switch (ordinateIndex) {
    case Coordinate.X:
      this.x = value;
      break
    case Coordinate.Y:
      this.y = value;
      break
    case Coordinate.Z:
      this.z = value;
      break
    default:
      throw new IllegalArgumentException('Invalid ordinate index: ' + ordinateIndex)
  }
};
Coordinate.prototype.equals2D = function equals2D () {
  if (arguments.length === 1) {
    var other = arguments[0];
    if (this.x !== other.x) {
      return false
    }
    if (this.y !== other.y) {
      return false
    }
    return true
  } else if (arguments.length === 2) {
    var c = arguments[0];
    var tolerance = arguments[1];
    if (!NumberUtil.equalsWithTolerance(this.x, c.x, tolerance)) {
      return false
    }
    if (!NumberUtil.equalsWithTolerance(this.y, c.y, tolerance)) {
      return false
    }
    return true
  }
};
Coordinate.prototype.getOrdinate = function getOrdinate (ordinateIndex) {
  switch (ordinateIndex) {
    case Coordinate.X:
      return this.x
    case Coordinate.Y:
      return this.y
    case Coordinate.Z:
      return this.z
    default:
  }
  throw new IllegalArgumentException('Invalid ordinate index: ' + ordinateIndex)
};
Coordinate.prototype.equals3D = function equals3D (other) {
  return this.x === other.x &&
         this.y === other.y &&
         ((this.z === other.z || Double.isNaN(this.z)) &&
         Double.isNaN(other.z))
};
Coordinate.prototype.equals = function equals (other) {
  if (!(other instanceof Coordinate)) {
    return false
  }
  return this.equals2D(other)
};
Coordinate.prototype.equalInZ = function equalInZ (c, tolerance) {
  return NumberUtil.equalsWithTolerance(this.z, c.z, tolerance)
};
Coordinate.prototype.compareTo = function compareTo (o) {
  var other = o;
  if (this.x < other.x) { return -1 }
  if (this.x > other.x) { return 1 }
  if (this.y < other.y) { return -1 }
  if (this.y > other.y) { return 1 }
  return 0
};
Coordinate.prototype.clone = function clone () {
  // try {
  // var coord = null
  // return coord
  // } catch (e) {
  // if (e instanceof CloneNotSupportedException) {
  //   Assert.shouldNeverReachHere("this shouldn't happen because this class is Cloneable")
  //   return null
  // } else throw e
  // } finally {}
};
Coordinate.prototype.copy = function copy () {
  return new Coordinate(this)
};
Coordinate.prototype.toString = function toString () {
  return '(' + this.x + ', ' + this.y + ', ' + this.z + ')'
};
Coordinate.prototype.distance3D = function distance3D (c) {
  var dx = this.x - c.x;
  var dy = this.y - c.y;
  var dz = this.z - c.z;
  return Math.sqrt(dx * dx + dy * dy + dz * dz)
};
Coordinate.prototype.distance = function distance (c) {
  var dx = this.x - c.x;
  var dy = this.y - c.y;
  return Math.sqrt(dx * dx + dy * dy)
};
Coordinate.prototype.hashCode = function hashCode () {
  var result = 17;
  result = 37 * result + Coordinate.hashCode(this.x);
  result = 37 * result + Coordinate.hashCode(this.y);
  return result
};
Coordinate.prototype.setCoordinate = function setCoordinate (other) {
  this.x = other.x;
  this.y = other.y;
  this.z = other.z;
};
Coordinate.prototype.interfaces_ = function interfaces_ () {
  return [Comparable, Clonable, Serializable]
};
Coordinate.prototype.getClass = function getClass () {
  return Coordinate
};
Coordinate.hashCode = function hashCode () {
  if (arguments.length === 1) {
    var x = arguments[0];
    var f = Double.doubleToLongBits(x);
    return Math.trunc((f ^ f) >>> 32)
  }
};
staticAccessors.DimensionalComparator.get = function () { return DimensionalComparator };
staticAccessors.serialVersionUID.get = function () { return 6683108902428366910 };
staticAccessors.NULL_ORDINATE.get = function () { return Double.NaN };
staticAccessors.X.get = function () { return 0 };
staticAccessors.Y.get = function () { return 1 };
staticAccessors.Z.get = function () { return 2 };

Object.defineProperties( Coordinate, staticAccessors );

var DimensionalComparator = function DimensionalComparator (dimensionsToTest) {
  this._dimensionsToTest = 2;
  if (arguments.length === 0) {} else if (arguments.length === 1) {
    var dimensionsToTest$1 = arguments[0];
    if (dimensionsToTest$1 !== 2 && dimensionsToTest$1 !== 3) { throw new IllegalArgumentException('only 2 or 3 dimensions may be specified') }
    this._dimensionsToTest = dimensionsToTest$1;
  }
};
DimensionalComparator.prototype.compare = function compare (o1, o2) {
  var c1 = o1;
  var c2 = o2;
  var compX = DimensionalComparator.compare(c1.x, c2.x);
  if (compX !== 0) { return compX }
  var compY = DimensionalComparator.compare(c1.y, c2.y);
  if (compY !== 0) { return compY }
  if (this._dimensionsToTest <= 2) { return 0 }
  var compZ = DimensionalComparator.compare(c1.z, c2.z);
  return compZ
};
DimensionalComparator.prototype.interfaces_ = function interfaces_ () {
  return [Comparator]
};
DimensionalComparator.prototype.getClass = function getClass () {
  return DimensionalComparator
};
DimensionalComparator.compare = function compare (a, b) {
  if (a < b) { return -1 }
  if (a > b) { return 1 }
  if (Double.isNaN(a)) {
    if (Double.isNaN(b)) { return 0 }
    return -1
  }
  if (Double.isNaN(b)) { return 1 }
  return 0
};

// import hasInterface from '../../../../hasInterface'
// import CoordinateSequence from './CoordinateSequence'

var CoordinateSequenceFactory = function CoordinateSequenceFactory () {};

CoordinateSequenceFactory.prototype.create = function create () {
  // if (arguments.length === 1) {
  // if (arguments[0] instanceof Array) {
  //   let coordinates = arguments[0]
  // } else if (hasInterface(arguments[0], CoordinateSequence)) {
  //   let coordSeq = arguments[0]
  // }
  // } else if (arguments.length === 2) {
  // let size = arguments[0]
  // let dimension = arguments[1]
  // }
};
CoordinateSequenceFactory.prototype.interfaces_ = function interfaces_ () {
  return []
};
CoordinateSequenceFactory.prototype.getClass = function getClass () {
  return CoordinateSequenceFactory
};

var Location = function Location () {};

var staticAccessors$4 = { INTERIOR: { configurable: true },BOUNDARY: { configurable: true },EXTERIOR: { configurable: true },NONE: { configurable: true } };

Location.prototype.interfaces_ = function interfaces_ () {
  return []
};
Location.prototype.getClass = function getClass () {
  return Location
};
Location.toLocationSymbol = function toLocationSymbol (locationValue) {
  switch (locationValue) {
    case Location.EXTERIOR:
      return 'e'
    case Location.BOUNDARY:
      return 'b'
    case Location.INTERIOR:
      return 'i'
    case Location.NONE:
      return '-'
    default:
  }
  throw new IllegalArgumentException('Unknown location value: ' + locationValue)
};
staticAccessors$4.INTERIOR.get = function () { return 0 };
staticAccessors$4.BOUNDARY.get = function () { return 1 };
staticAccessors$4.EXTERIOR.get = function () { return 2 };
staticAccessors$4.NONE.get = function () { return -1 };

Object.defineProperties( Location, staticAccessors$4 );

var hasInterface = function (o, i) {
  return o.interfaces_ && o.interfaces_().indexOf(i) > -1
};

var MathUtil = function MathUtil () {};

var staticAccessors$5 = { LOG_10: { configurable: true } };

MathUtil.prototype.interfaces_ = function interfaces_ () {
  return []
};
MathUtil.prototype.getClass = function getClass () {
  return MathUtil
};
MathUtil.log10 = function log10 (x) {
  var ln = Math.log(x);
  if (Double.isInfinite(ln)) { return ln }
  if (Double.isNaN(ln)) { return ln }
  return ln / MathUtil.LOG_10
};
MathUtil.min = function min (v1, v2, v3, v4) {
  var min = v1;
  if (v2 < min) { min = v2; }
  if (v3 < min) { min = v3; }
  if (v4 < min) { min = v4; }
  return min
};
MathUtil.clamp = function clamp () {
  if (typeof arguments[2] === 'number' && (typeof arguments[0] === 'number' && typeof arguments[1] === 'number')) {
    var x = arguments[0];
    var min = arguments[1];
    var max = arguments[2];
    if (x < min) { return min }
    if (x > max) { return max }
    return x
  } else if (Number.isInteger(arguments[2]) && (Number.isInteger(arguments[0]) && Number.isInteger(arguments[1]))) {
    var x$1 = arguments[0];
    var min$1 = arguments[1];
    var max$1 = arguments[2];
    if (x$1 < min$1) { return min$1 }
    if (x$1 > max$1) { return max$1 }
    return x$1
  }
};
MathUtil.wrap = function wrap (index, max) {
  if (index < 0) {
    return max - -index % max
  }
  return index % max
};
MathUtil.max = function max () {
  if (arguments.length === 3) {
    var v1 = arguments[0];
    var v2 = arguments[1];
    var v3 = arguments[2];
    var max = v1;
    if (v2 > max) { max = v2; }
    if (v3 > max) { max = v3; }
    return max
  } else if (arguments.length === 4) {
    var v1$1 = arguments[0];
    var v2$1 = arguments[1];
    var v3$1 = arguments[2];
    var v4 = arguments[3];
    var max$1 = v1$1;
    if (v2$1 > max$1) { max$1 = v2$1; }
    if (v3$1 > max$1) { max$1 = v3$1; }
    if (v4 > max$1) { max$1 = v4; }
    return max$1
  }
};
MathUtil.average = function average (x1, x2) {
  return (x1 + x2) / 2.0
};
staticAccessors$5.LOG_10.get = function () { return Math.log(10) };

Object.defineProperties( MathUtil, staticAccessors$5 );

var StringBuffer = function StringBuffer (str) {
  this.str = str;
};
StringBuffer.prototype.append = function append (e) {
  this.str += e;
};

StringBuffer.prototype.setCharAt = function setCharAt (i, c) {
  this.str = this.str.substr(0, i) + c + this.str.substr(i + 1);
};

StringBuffer.prototype.toString = function toString (e) {
  return this.str
};

var Integer = function Integer (value) {
  this.value = value;
};
Integer.prototype.intValue = function intValue () {
  return this.value
};
Integer.prototype.compareTo = function compareTo (o) {
  if (this.value < o) { return -1 }
  if (this.value > o) { return 1 }
  return 0
};
Integer.isNaN = function isNaN (n) { return Number.isNaN(n) };

var Character = function Character () {};

Character.isWhitespace = function isWhitespace (c) { return ((c <= 32 && c >= 0) || c === 127) };
Character.toUpperCase = function toUpperCase (c) { return c.toUpperCase() };

var DD = function DD () {
  this._hi = 0.0;
  this._lo = 0.0;
  if (arguments.length === 0) {
    this.init(0.0);
  } else if (arguments.length === 1) {
    if (typeof arguments[0] === 'number') {
      var x = arguments[0];
      this.init(x);
    } else if (arguments[0] instanceof DD) {
      var dd = arguments[0];
      this.init(dd);
    } else if (typeof arguments[0] === 'string') {
      var str = arguments[0];
      DD.call(this, DD.parse(str));
    }
  } else if (arguments.length === 2) {
    var hi = arguments[0];
    var lo = arguments[1];
    this.init(hi, lo);
  }
};

var staticAccessors$7 = { PI: { configurable: true },TWO_PI: { configurable: true },PI_2: { configurable: true },E: { configurable: true },NaN: { configurable: true },EPS: { configurable: true },SPLIT: { configurable: true },MAX_PRINT_DIGITS: { configurable: true },TEN: { configurable: true },ONE: { configurable: true },SCI_NOT_EXPONENT_CHAR: { configurable: true },SCI_NOT_ZERO: { configurable: true } };
DD.prototype.le = function le (y) {
  return (this._hi < y._hi || this._hi === y._hi) && this._lo <= y._lo
};
DD.prototype.extractSignificantDigits = function extractSignificantDigits (insertDecimalPoint, magnitude) {
  var y = this.abs();
  var mag = DD.magnitude(y._hi);
  var scale = DD.TEN.pow(mag);
  y = y.divide(scale);
  if (y.gt(DD.TEN)) {
    y = y.divide(DD.TEN);
    mag += 1;
  } else if (y.lt(DD.ONE)) {
    y = y.multiply(DD.TEN);
    mag -= 1;
  }
  var decimalPointPos = mag + 1;
  var buf = new StringBuffer();
  var numDigits = DD.MAX_PRINT_DIGITS - 1;
  for (var i = 0; i <= numDigits; i++) {
    if (insertDecimalPoint && i === decimalPointPos) {
      buf.append('.');
    }
    var digit = Math.trunc(y._hi);
    if (digit < 0) {
      break
    }
    var rebiasBy10 = false;
    var digitChar = 0;
    if (digit > 9) {
      rebiasBy10 = true;
      digitChar = '9';
    } else {
      digitChar = '0' + digit;
    }
    buf.append(digitChar);
    y = y.subtract(DD.valueOf(digit)).multiply(DD.TEN);
    if (rebiasBy10) { y.selfAdd(DD.TEN); }
    var continueExtractingDigits = true;
    var remMag = DD.magnitude(y._hi);
    if (remMag < 0 && Math.abs(remMag) >= numDigits - i) { continueExtractingDigits = false; }
    if (!continueExtractingDigits) { break }
  }
  magnitude[0] = mag;
  return buf.toString()
};
DD.prototype.sqr = function sqr () {
  return this.multiply(this)
};
DD.prototype.doubleValue = function doubleValue () {
  return this._hi + this._lo
};
DD.prototype.subtract = function subtract () {
  if (arguments[0] instanceof DD) {
    var y = arguments[0];
    return this.add(y.negate())
  } else if (typeof arguments[0] === 'number') {
    var y$1 = arguments[0];
    return this.add(-y$1)
  }
};
DD.prototype.equals = function equals () {
  if (arguments.length === 1) {
    var y = arguments[0];
    return this._hi === y._hi && this._lo === y._lo
  }
};
DD.prototype.isZero = function isZero () {
  return this._hi === 0.0 && this._lo === 0.0
};
DD.prototype.selfSubtract = function selfSubtract () {
  if (arguments[0] instanceof DD) {
    var y = arguments[0];
    if (this.isNaN()) { return this }
    return this.selfAdd(-y._hi, -y._lo)
  } else if (typeof arguments[0] === 'number') {
    var y$1 = arguments[0];
    if (this.isNaN()) { return this }
    return this.selfAdd(-y$1, 0.0)
  }
};
DD.prototype.getSpecialNumberString = function getSpecialNumberString () {
  if (this.isZero()) { return '0.0' }
  if (this.isNaN()) { return 'NaN ' }
  return null
};
DD.prototype.min = function min (x) {
  if (this.le(x)) {
    return this
  } else {
    return x
  }
};
DD.prototype.selfDivide = function selfDivide () {
  if (arguments.length === 1) {
    if (arguments[0] instanceof DD) {
      var y = arguments[0];
      return this.selfDivide(y._hi, y._lo)
    } else if (typeof arguments[0] === 'number') {
      var y$1 = arguments[0];
      return this.selfDivide(y$1, 0.0)
    }
  } else if (arguments.length === 2) {
    var yhi = arguments[0];
    var ylo = arguments[1];
    var hc = null;
    var tc = null;
    var hy = null;
    var ty = null;
    var C = null;
    var c = null;
    var U = null;
    var u = null;
    C = this._hi / yhi;
    c = DD.SPLIT * C;
    hc = c - C;
    u = DD.SPLIT * yhi;
    hc = c - hc;
    tc = C - hc;
    hy = u - yhi;
    U = C * yhi;
    hy = u - hy;
    ty = yhi - hy;
    u = hc * hy - U + hc * ty + tc * hy + tc * ty;
    c = (this._hi - U - u + this._lo - C * ylo) / yhi;
    u = C + c;
    this._hi = u;
    this._lo = C - u + c;
    return this
  }
};
DD.prototype.dump = function dump () {
  return 'DD<' + this._hi + ', ' + this._lo + '>'
};
DD.prototype.divide = function divide () {
  if (arguments[0] instanceof DD) {
    var y = arguments[0];
    var hc = null;
    var tc = null;
    var hy = null;
    var ty = null;
    var C = null;
    var c = null;
    var U = null;
    var u = null;
    C = this._hi / y._hi;
    c = DD.SPLIT * C;
    hc = c - C;
    u = DD.SPLIT * y._hi;
    hc = c - hc;
    tc = C - hc;
    hy = u - y._hi;
    U = C * y._hi;
    hy = u - hy;
    ty = y._hi - hy;
    u = hc * hy - U + hc * ty + tc * hy + tc * ty;
    c = (this._hi - U - u + this._lo - C * y._lo) / y._hi;
    u = C + c;
    var zhi = u;
    var zlo = C - u + c;
    return new DD(zhi, zlo)
  } else if (typeof arguments[0] === 'number') {
    var y$1 = arguments[0];
    if (Double.isNaN(y$1)) { return DD.createNaN() }
    return DD.copy(this).selfDivide(y$1, 0.0)
  }
};
DD.prototype.ge = function ge (y) {
  return (this._hi > y._hi || this._hi === y._hi) && this._lo >= y._lo
};
DD.prototype.pow = function pow (exp) {
  if (exp === 0.0) { return DD.valueOf(1.0) }
  var r = new DD(this);
  var s = DD.valueOf(1.0);
  var n = Math.abs(exp);
  if (n > 1) {
    while (n > 0) {
      if (n % 2 === 1) {
        s.selfMultiply(r);
      }
      n /= 2;
      if (n > 0) { r = r.sqr(); }
    }
  } else {
    s = r;
  }
  if (exp < 0) { return s.reciprocal() }
  return s
};
DD.prototype.ceil = function ceil () {
  if (this.isNaN()) { return DD.NaN }
  var fhi = Math.ceil(this._hi);
  var flo = 0.0;
  if (fhi === this._hi) {
    flo = Math.ceil(this._lo);
  }
  return new DD(fhi, flo)
};
DD.prototype.compareTo = function compareTo (o) {
  var other = o;
  if (this._hi < other._hi) { return -1 }
  if (this._hi > other._hi) { return 1 }
  if (this._lo < other._lo) { return -1 }
  if (this._lo > other._lo) { return 1 }
  return 0
};
DD.prototype.rint = function rint () {
  if (this.isNaN()) { return this }
  var plus5 = this.add(0.5);
  return plus5.floor()
};
DD.prototype.setValue = function setValue () {
  if (arguments[0] instanceof DD) {
    var value = arguments[0];
    this.init(value);
    return this
  } else if (typeof arguments[0] === 'number') {
    var value$1 = arguments[0];
    this.init(value$1);
    return this
  }
};
DD.prototype.max = function max (x) {
  if (this.ge(x)) {
    return this
  } else {
    return x
  }
};
DD.prototype.sqrt = function sqrt () {
  if (this.isZero()) { return DD.valueOf(0.0) }
  if (this.isNegative()) {
    return DD.NaN
  }
  var x = 1.0 / Math.sqrt(this._hi);
  var ax = this._hi * x;
  var axdd = DD.valueOf(ax);
  var diffSq = this.subtract(axdd.sqr());
  var d2 = diffSq._hi * (x * 0.5);
  return axdd.add(d2)
};
DD.prototype.selfAdd = function selfAdd () {
  if (arguments.length === 1) {
    if (arguments[0] instanceof DD) {
      var y = arguments[0];
      return this.selfAdd(y._hi, y._lo)
    } else if (typeof arguments[0] === 'number') {
      var y$1 = arguments[0];
      var H = null;
      var h = null;
      var S = null;
      var s = null;
      var e = null;
      var f = null;
      S = this._hi + y$1;
      e = S - this._hi;
      s = S - e;
      s = y$1 - e + (this._hi - s);
      f = s + this._lo;
      H = S + f;
      h = f + (S - H);
      this._hi = H + h;
      this._lo = h + (H - this._hi);
      return this
    }
  } else if (arguments.length === 2) {
    var yhi = arguments[0];
    var ylo = arguments[1];
    var H$1 = null;
    var h$1 = null;
    var T = null;
    var t = null;
    var S$1 = null;
    var s$1 = null;
    var e$1 = null;
    var f$1 = null;
    S$1 = this._hi + yhi;
    T = this._lo + ylo;
    e$1 = S$1 - this._hi;
    f$1 = T - this._lo;
    s$1 = S$1 - e$1;
    t = T - f$1;
    s$1 = yhi - e$1 + (this._hi - s$1);
    t = ylo - f$1 + (this._lo - t);
    e$1 = s$1 + T;
    H$1 = S$1 + e$1;
    h$1 = e$1 + (S$1 - H$1);
    e$1 = t + h$1;
    var zhi = H$1 + e$1;
    var zlo = e$1 + (H$1 - zhi);
    this._hi = zhi;
    this._lo = zlo;
    return this
  }
};
DD.prototype.selfMultiply = function selfMultiply () {
  if (arguments.length === 1) {
    if (arguments[0] instanceof DD) {
      var y = arguments[0];
      return this.selfMultiply(y._hi, y._lo)
    } else if (typeof arguments[0] === 'number') {
      var y$1 = arguments[0];
      return this.selfMultiply(y$1, 0.0)
    }
  } else if (arguments.length === 2) {
    var yhi = arguments[0];
    var ylo = arguments[1];
    var hx = null;
    var tx = null;
    var hy = null;
    var ty = null;
    var C = null;
    var c = null;
    C = DD.SPLIT * this._hi;
    hx = C - this._hi;
    c = DD.SPLIT * yhi;
    hx = C - hx;
    tx = this._hi - hx;
    hy = c - yhi;
    C = this._hi * yhi;
    hy = c - hy;
    ty = yhi - hy;
    c = hx * hy - C + hx * ty + tx * hy + tx * ty + (this._hi * ylo + this._lo * yhi);
    var zhi = C + c;
    hx = C - zhi;
    var zlo = c + hx;
    this._hi = zhi;
    this._lo = zlo;
    return this
  }
};
DD.prototype.selfSqr = function selfSqr () {
  return this.selfMultiply(this)
};
DD.prototype.floor = function floor () {
  if (this.isNaN()) { return DD.NaN }
  var fhi = Math.floor(this._hi);
  var flo = 0.0;
  if (fhi === this._hi) {
    flo = Math.floor(this._lo);
  }
  return new DD(fhi, flo)
};
DD.prototype.negate = function negate () {
  if (this.isNaN()) { return this }
  return new DD(-this._hi, -this._lo)
};
DD.prototype.clone = function clone () {
  // try {
  // return null
  // } catch (ex) {
  // if (ex instanceof CloneNotSupportedException) {
  //   return null
  // } else throw ex
  // } finally {}
};
DD.prototype.multiply = function multiply () {
  if (arguments[0] instanceof DD) {
    var y = arguments[0];
    if (y.isNaN()) { return DD.createNaN() }
    return DD.copy(this).selfMultiply(y)
  } else if (typeof arguments[0] === 'number') {
    var y$1 = arguments[0];
    if (Double.isNaN(y$1)) { return DD.createNaN() }
    return DD.copy(this).selfMultiply(y$1, 0.0)
  }
};
DD.prototype.isNaN = function isNaN () {
  return Double.isNaN(this._hi)
};
DD.prototype.intValue = function intValue () {
  return Math.trunc(this._hi)
};
DD.prototype.toString = function toString () {
  var mag = DD.magnitude(this._hi);
  if (mag >= -3 && mag <= 20) { return this.toStandardNotation() }
  return this.toSciNotation()
};
DD.prototype.toStandardNotation = function toStandardNotation () {
  var specialStr = this.getSpecialNumberString();
  if (specialStr !== null) { return specialStr }
  var magnitude = new Array(1).fill(null);
  var sigDigits = this.extractSignificantDigits(true, magnitude);
  var decimalPointPos = magnitude[0] + 1;
  var num = sigDigits;
  if (sigDigits.charAt(0) === '.') {
    num = '0' + sigDigits;
  } else if (decimalPointPos < 0) {
    num = '0.' + DD.stringOfChar('0', -decimalPointPos) + sigDigits;
  } else if (sigDigits.indexOf('.') === -1) {
    var numZeroes = decimalPointPos - sigDigits.length;
    var zeroes = DD.stringOfChar('0', numZeroes);
    num = sigDigits + zeroes + '.0';
  }
  if (this.isNegative()) { return '-' + num }
  return num
};
DD.prototype.reciprocal = function reciprocal () {
  var hc = null;
  var tc = null;
  var hy = null;
  var ty = null;
  var C = null;
  var c = null;
  var U = null;
  var u = null;
  C = 1.0 / this._hi;
  c = DD.SPLIT * C;
  hc = c - C;
  u = DD.SPLIT * this._hi;
  hc = c - hc;
  tc = C - hc;
  hy = u - this._hi;
  U = C * this._hi;
  hy = u - hy;
  ty = this._hi - hy;
  u = hc * hy - U + hc * ty + tc * hy + tc * ty;
  c = (1.0 - U - u - C * this._lo) / this._hi;
  var zhi = C + c;
  var zlo = C - zhi + c;
  return new DD(zhi, zlo)
};
DD.prototype.toSciNotation = function toSciNotation () {
  if (this.isZero()) { return DD.SCI_NOT_ZERO }
  var specialStr = this.getSpecialNumberString();
  if (specialStr !== null) { return specialStr }
  var magnitude = new Array(1).fill(null);
  var digits = this.extractSignificantDigits(false, magnitude);
  var expStr = DD.SCI_NOT_EXPONENT_CHAR + magnitude[0];
  if (digits.charAt(0) === '0') {
    throw new Error('Found leading zero: ' + digits)
  }
  var trailingDigits = '';
  if (digits.length > 1) { trailingDigits = digits.substring(1); }
  var digitsWithDecimal = digits.charAt(0) + '.' + trailingDigits;
  if (this.isNegative()) { return '-' + digitsWithDecimal + expStr }
  return digitsWithDecimal + expStr
};
DD.prototype.abs = function abs () {
  if (this.isNaN()) { return DD.NaN }
  if (this.isNegative()) { return this.negate() }
  return new DD(this)
};
DD.prototype.isPositive = function isPositive () {
  return (this._hi > 0.0 || this._hi === 0.0) && this._lo > 0.0
};
DD.prototype.lt = function lt (y) {
  return (this._hi < y._hi || this._hi === y._hi) && this._lo < y._lo
};
DD.prototype.add = function add () {
  if (arguments[0] instanceof DD) {
    var y = arguments[0];
    return DD.copy(this).selfAdd(y)
  } else if (typeof arguments[0] === 'number') {
    var y$1 = arguments[0];
    return DD.copy(this).selfAdd(y$1)
  }
};
DD.prototype.init = function init () {
  if (arguments.length === 1) {
    if (typeof arguments[0] === 'number') {
      var x = arguments[0];
      this._hi = x;
      this._lo = 0.0;
    } else if (arguments[0] instanceof DD) {
      var dd = arguments[0];
      this._hi = dd._hi;
      this._lo = dd._lo;
    }
  } else if (arguments.length === 2) {
    var hi = arguments[0];
    var lo = arguments[1];
    this._hi = hi;
    this._lo = lo;
  }
};
DD.prototype.gt = function gt (y) {
  return (this._hi > y._hi || this._hi === y._hi) && this._lo > y._lo
};
DD.prototype.isNegative = function isNegative () {
  return (this._hi < 0.0 || this._hi === 0.0) && this._lo < 0.0
};
DD.prototype.trunc = function trunc () {
  if (this.isNaN()) { return DD.NaN }
  if (this.isPositive()) { return this.floor(); } else { return this.ceil() }
};
DD.prototype.signum = function signum () {
  if (this._hi > 0) { return 1 }
  if (this._hi < 0) { return -1 }
  if (this._lo > 0) { return 1 }
  if (this._lo < 0) { return -1 }
  return 0
};
DD.prototype.interfaces_ = function interfaces_ () {
  return [Serializable, Comparable, Clonable]
};
DD.prototype.getClass = function getClass () {
  return DD
};
DD.sqr = function sqr (x) {
  return DD.valueOf(x).selfMultiply(x)
};
DD.valueOf = function valueOf () {
  if (typeof arguments[0] === 'string') {
    var str = arguments[0];
    return DD.parse(str)
  } else if (typeof arguments[0] === 'number') {
    var x = arguments[0];
    return new DD(x)
  }
};
DD.sqrt = function sqrt (x) {
  return DD.valueOf(x).sqrt()
};
DD.parse = function parse (str) {
  var i = 0;
  var strlen = str.length;
  while (Character.isWhitespace(str.charAt(i))) { i++; }
  var isNegative = false;
  if (i < strlen) {
    var signCh = str.charAt(i);
    if (signCh === '-' || signCh === '+') {
      i++;
      if (signCh === '-') { isNegative = true; }
    }
  }
  var val = new DD();
  var numDigits = 0;
  var numBeforeDec = 0;
  var exp = 0;
  while (true) {
    if (i >= strlen) { break }
    var ch = str.charAt(i);
    i++;
    if (Character.isDigit(ch)) {
      var d = ch - '0';
      val.selfMultiply(DD.TEN);
      val.selfAdd(d);
      numDigits++;
      continue
    }
    if (ch === '.') {
      numBeforeDec = numDigits;
      continue
    }
    if (ch === 'e' || ch === 'E') {
      var expStr = str.substring(i);
      try {
        exp = Integer.parseInt(expStr);
      } catch (ex) {
        if (ex instanceof Error) {
          throw new Error('Invalid exponent ' + expStr + ' in string ' + str)
        } else { throw ex }
      } finally {}
      break
    }
    throw new Error("Unexpected character '" + ch + "' at position " + i + ' in string ' + str)
  }
  var val2 = val;
  var numDecPlaces = numDigits - numBeforeDec - exp;
  if (numDecPlaces === 0) {
    val2 = val;
  } else if (numDecPlaces > 0) {
    var scale = DD.TEN.pow(numDecPlaces);
    val2 = val.divide(scale);
  } else if (numDecPlaces < 0) {
    var scale$1 = DD.TEN.pow(-numDecPlaces);
    val2 = val.multiply(scale$1);
  }
  if (isNegative) {
    return val2.negate()
  }
  return val2
};
DD.createNaN = function createNaN () {
  return new DD(Double.NaN, Double.NaN)
};
DD.copy = function copy (dd) {
  return new DD(dd)
};
DD.magnitude = function magnitude (x) {
  var xAbs = Math.abs(x);
  var xLog10 = Math.log(xAbs) / Math.log(10);
  var xMag = Math.trunc(Math.floor(xLog10));
  var xApprox = Math.pow(10, xMag);
  if (xApprox * 10 <= xAbs) { xMag += 1; }
  return xMag
};
DD.stringOfChar = function stringOfChar (ch, len) {
  var buf = new StringBuffer();
  for (var i = 0; i < len; i++) {
    buf.append(ch);
  }
  return buf.toString()
};
staticAccessors$7.PI.get = function () { return new DD(3.141592653589793116e+00, 1.224646799147353207e-16) };
staticAccessors$7.TWO_PI.get = function () { return new DD(6.283185307179586232e+00, 2.449293598294706414e-16) };
staticAccessors$7.PI_2.get = function () { return new DD(1.570796326794896558e+00, 6.123233995736766036e-17) };
staticAccessors$7.E.get = function () { return new DD(2.718281828459045091e+00, 1.445646891729250158e-16) };
staticAccessors$7.NaN.get = function () { return new DD(Double.NaN, Double.NaN) };
staticAccessors$7.EPS.get = function () { return 1.23259516440783e-32 };
staticAccessors$7.SPLIT.get = function () { return 134217729.0 };
staticAccessors$7.MAX_PRINT_DIGITS.get = function () { return 32 };
staticAccessors$7.TEN.get = function () { return DD.valueOf(10.0) };
staticAccessors$7.ONE.get = function () { return DD.valueOf(1.0) };
staticAccessors$7.SCI_NOT_EXPONENT_CHAR.get = function () { return 'E' };
staticAccessors$7.SCI_NOT_ZERO.get = function () { return '0.0E0' };

Object.defineProperties( DD, staticAccessors$7 );

var CGAlgorithmsDD = function CGAlgorithmsDD () {};

var staticAccessors$6 = { DP_SAFE_EPSILON: { configurable: true } };

CGAlgorithmsDD.prototype.interfaces_ = function interfaces_ () {
  return []
};
CGAlgorithmsDD.prototype.getClass = function getClass () {
  return CGAlgorithmsDD
};
CGAlgorithmsDD.orientationIndex = function orientationIndex (p1, p2, q) {
  var index = CGAlgorithmsDD.orientationIndexFilter(p1, p2, q);
  if (index <= 1) { return index }
  var dx1 = DD.valueOf(p2.x).selfAdd(-p1.x);
  var dy1 = DD.valueOf(p2.y).selfAdd(-p1.y);
  var dx2 = DD.valueOf(q.x).selfAdd(-p2.x);
  var dy2 = DD.valueOf(q.y).selfAdd(-p2.y);
  return dx1.selfMultiply(dy2).selfSubtract(dy1.selfMultiply(dx2)).signum()
};
CGAlgorithmsDD.signOfDet2x2 = function signOfDet2x2 (x1, y1, x2, y2) {
  var det = x1.multiply(y2).selfSubtract(y1.multiply(x2));
  return det.signum()
};
CGAlgorithmsDD.intersection = function intersection (p1, p2, q1, q2) {
  var denom1 = DD.valueOf(q2.y).selfSubtract(q1.y).selfMultiply(DD.valueOf(p2.x).selfSubtract(p1.x));
  var denom2 = DD.valueOf(q2.x).selfSubtract(q1.x).selfMultiply(DD.valueOf(p2.y).selfSubtract(p1.y));
  var denom = denom1.subtract(denom2);
  var numx1 = DD.valueOf(q2.x).selfSubtract(q1.x).selfMultiply(DD.valueOf(p1.y).selfSubtract(q1.y));
  var numx2 = DD.valueOf(q2.y).selfSubtract(q1.y).selfMultiply(DD.valueOf(p1.x).selfSubtract(q1.x));
  var numx = numx1.subtract(numx2);
  var fracP = numx.selfDivide(denom).doubleValue();
  var x = DD.valueOf(p1.x).selfAdd(DD.valueOf(p2.x).selfSubtract(p1.x).selfMultiply(fracP)).doubleValue();
  var numy1 = DD.valueOf(p2.x).selfSubtract(p1.x).selfMultiply(DD.valueOf(p1.y).selfSubtract(q1.y));
  var numy2 = DD.valueOf(p2.y).selfSubtract(p1.y).selfMultiply(DD.valueOf(p1.x).selfSubtract(q1.x));
  var numy = numy1.subtract(numy2);
  var fracQ = numy.selfDivide(denom).doubleValue();
  var y = DD.valueOf(q1.y).selfAdd(DD.valueOf(q2.y).selfSubtract(q1.y).selfMultiply(fracQ)).doubleValue();
  return new Coordinate(x, y)
};
CGAlgorithmsDD.orientationIndexFilter = function orientationIndexFilter (pa, pb, pc) {
  var detsum = null;
  var detleft = (pa.x - pc.x) * (pb.y - pc.y);
  var detright = (pa.y - pc.y) * (pb.x - pc.x);
  var det = detleft - detright;
  if (detleft > 0.0) {
    if (detright <= 0.0) {
      return CGAlgorithmsDD.signum(det)
    } else {
      detsum = detleft + detright;
    }
  } else if (detleft < 0.0) {
    if (detright >= 0.0) {
      return CGAlgorithmsDD.signum(det)
    } else {
      detsum = -detleft - detright;
    }
  } else {
    return CGAlgorithmsDD.signum(det)
  }
  var errbound = CGAlgorithmsDD.DP_SAFE_EPSILON * detsum;
  if (det >= errbound || -det >= errbound) {
    return CGAlgorithmsDD.signum(det)
  }
  return 2
};
CGAlgorithmsDD.signum = function signum (x) {
  if (x > 0) { return 1 }
  if (x < 0) { return -1 }
  return 0
};
staticAccessors$6.DP_SAFE_EPSILON.get = function () { return 1e-15 };

Object.defineProperties( CGAlgorithmsDD, staticAccessors$6 );

var CoordinateSequence = function CoordinateSequence () {};

var staticAccessors$8 = { X: { configurable: true },Y: { configurable: true },Z: { configurable: true },M: { configurable: true } };

staticAccessors$8.X.get = function () { return 0 };
staticAccessors$8.Y.get = function () { return 1 };
staticAccessors$8.Z.get = function () { return 2 };
staticAccessors$8.M.get = function () { return 3 };
CoordinateSequence.prototype.setOrdinate = function setOrdinate (index, ordinateIndex, value) {};
CoordinateSequence.prototype.size = function size () {};
CoordinateSequence.prototype.getOrdinate = function getOrdinate (index, ordinateIndex) {};
CoordinateSequence.prototype.getCoordinate = function getCoordinate () {};
CoordinateSequence.prototype.getCoordinateCopy = function getCoordinateCopy (i) {};
CoordinateSequence.prototype.getDimension = function getDimension () {};
CoordinateSequence.prototype.getX = function getX (index) {};
CoordinateSequence.prototype.clone = function clone () {};
CoordinateSequence.prototype.expandEnvelope = function expandEnvelope (env) {};
CoordinateSequence.prototype.copy = function copy () {};
CoordinateSequence.prototype.getY = function getY (index) {};
CoordinateSequence.prototype.toCoordinateArray = function toCoordinateArray () {};
CoordinateSequence.prototype.interfaces_ = function interfaces_ () {
  return [Clonable]
};
CoordinateSequence.prototype.getClass = function getClass () {
  return CoordinateSequence
};

Object.defineProperties( CoordinateSequence, staticAccessors$8 );

var Exception = function Exception () {};

var NotRepresentableException = (function (Exception$$1) {
  function NotRepresentableException () {
    Exception$$1.call(this, 'Projective point not representable on the Cartesian plane.');
  }

  if ( Exception$$1 ) NotRepresentableException.__proto__ = Exception$$1;
  NotRepresentableException.prototype = Object.create( Exception$$1 && Exception$$1.prototype );
  NotRepresentableException.prototype.constructor = NotRepresentableException;
  NotRepresentableException.prototype.interfaces_ = function interfaces_ () {
    return []
  };
  NotRepresentableException.prototype.getClass = function getClass () {
    return NotRepresentableException
  };

  return NotRepresentableException;
}(Exception));

var System = function System () {};

System.arraycopy = function arraycopy (src, srcPos, dest, destPos, len) {
  var c = 0;
  for (var i = srcPos; i < srcPos + len; i++) {
    dest[destPos + c] = src[i];
    c++;
  }
};

System.getProperty = function getProperty (name) {
  return {
    'line.separator': '\n'
  }[name]
};

var HCoordinate = function HCoordinate () {
  this.x = null;
  this.y = null;
  this.w = null;
  if (arguments.length === 0) {
    this.x = 0.0;
    this.y = 0.0;
    this.w = 1.0;
  } else if (arguments.length === 1) {
    var p = arguments[0];
    this.x = p.x;
    this.y = p.y;
    this.w = 1.0;
  } else if (arguments.length === 2) {
    if (typeof arguments[0] === 'number' && typeof arguments[1] === 'number') {
      var _x = arguments[0];
      var _y = arguments[1];
      this.x = _x;
      this.y = _y;
      this.w = 1.0;
    } else if (arguments[0] instanceof HCoordinate && arguments[1] instanceof HCoordinate) {
      var p1 = arguments[0];
      var p2 = arguments[1];
      this.x = p1.y * p2.w - p2.y * p1.w;
      this.y = p2.x * p1.w - p1.x * p2.w;
      this.w = p1.x * p2.y - p2.x * p1.y;
    } else if (arguments[0] instanceof Coordinate && arguments[1] instanceof Coordinate) {
      var p1$1 = arguments[0];
      var p2$1 = arguments[1];
      this.x = p1$1.y - p2$1.y;
      this.y = p2$1.x - p1$1.x;
      this.w = p1$1.x * p2$1.y - p2$1.x * p1$1.y;
    }
  } else if (arguments.length === 3) {
    var _x$1 = arguments[0];
    var _y$1 = arguments[1];
    var _w = arguments[2];
    this.x = _x$1;
    this.y = _y$1;
    this.w = _w;
  } else if (arguments.length === 4) {
    var p1$2 = arguments[0];
    var p2$2 = arguments[1];
    var q1 = arguments[2];
    var q2 = arguments[3];
    var px = p1$2.y - p2$2.y;
    var py = p2$2.x - p1$2.x;
    var pw = p1$2.x * p2$2.y - p2$2.x * p1$2.y;
    var qx = q1.y - q2.y;
    var qy = q2.x - q1.x;
    var qw = q1.x * q2.y - q2.x * q1.y;
    this.x = py * qw - qy * pw;
    this.y = qx * pw - px * qw;
    this.w = px * qy - qx * py;
  }
};
HCoordinate.prototype.getY = function getY () {
  var a = this.y / this.w;
  if (Double.isNaN(a) || Double.isInfinite(a)) {
    throw new NotRepresentableException()
  }
  return a
};
HCoordinate.prototype.getX = function getX () {
  var a = this.x / this.w;
  if (Double.isNaN(a) || Double.isInfinite(a)) {
    throw new NotRepresentableException()
  }
  return a
};
HCoordinate.prototype.getCoordinate = function getCoordinate () {
  var p = new Coordinate();
  p.x = this.getX();
  p.y = this.getY();
  return p
};
HCoordinate.prototype.interfaces_ = function interfaces_ () {
  return []
};
HCoordinate.prototype.getClass = function getClass () {
  return HCoordinate
};
HCoordinate.intersection = function intersection (p1, p2, q1, q2) {
  var px = p1.y - p2.y;
  var py = p2.x - p1.x;
  var pw = p1.x * p2.y - p2.x * p1.y;
  var qx = q1.y - q2.y;
  var qy = q2.x - q1.x;
  var qw = q1.x * q2.y - q2.x * q1.y;
  var x = py * qw - qy * pw;
  var y = qx * pw - px * qw;
  var w = px * qy - qx * py;
  var xInt = x / w;
  var yInt = y / w;
  if (Double.isNaN(xInt) || (Double.isInfinite(xInt) || Double.isNaN(yInt)) || Double.isInfinite(yInt)) {
    throw new NotRepresentableException()
  }
  return new Coordinate(xInt, yInt)
};

var Envelope = function Envelope () {
  this._minx = null;
  this._maxx = null;
  this._miny = null;
  this._maxy = null;
  if (arguments.length === 0) {
    this.init();
  } else if (arguments.length === 1) {
    if (arguments[0] instanceof Coordinate) {
      var p = arguments[0];
      this.init(p.x, p.x, p.y, p.y);
    } else if (arguments[0] instanceof Envelope) {
      var env = arguments[0];
      this.init(env);
    }
  } else if (arguments.length === 2) {
    var p1 = arguments[0];
    var p2 = arguments[1];
    this.init(p1.x, p2.x, p1.y, p2.y);
  } else if (arguments.length === 4) {
    var x1 = arguments[0];
    var x2 = arguments[1];
    var y1 = arguments[2];
    var y2 = arguments[3];
    this.init(x1, x2, y1, y2);
  }
};

var staticAccessors$9 = { serialVersionUID: { configurable: true } };
Envelope.prototype.getArea = function getArea () {
  return this.getWidth() * this.getHeight()
};
Envelope.prototype.equals = function equals (other) {
  if (!(other instanceof Envelope)) {
    return false
  }
  var otherEnvelope = other;
  if (this.isNull()) {
    return otherEnvelope.isNull()
  }
  return this._maxx === otherEnvelope.getMaxX() && this._maxy === otherEnvelope.getMaxY() && this._minx === otherEnvelope.getMinX() && this._miny === otherEnvelope.getMinY()
};
Envelope.prototype.intersection = function intersection (env) {
  if (this.isNull() || env.isNull() || !this.intersects(env)) { return new Envelope() }
  var intMinX = this._minx > env._minx ? this._minx : env._minx;
  var intMinY = this._miny > env._miny ? this._miny : env._miny;
  var intMaxX = this._maxx < env._maxx ? this._maxx : env._maxx;
  var intMaxY = this._maxy < env._maxy ? this._maxy : env._maxy;
  return new Envelope(intMinX, intMaxX, intMinY, intMaxY)
};
Envelope.prototype.isNull = function isNull () {
  return this._maxx < this._minx
};
Envelope.prototype.getMaxX = function getMaxX () {
  return this._maxx
};
Envelope.prototype.covers = function covers () {
  if (arguments.length === 1) {
    if (arguments[0] instanceof Coordinate) {
      var p = arguments[0];
      return this.covers(p.x, p.y)
    } else if (arguments[0] instanceof Envelope) {
      var other = arguments[0];
      if (this.isNull() || other.isNull()) {
        return false
      }
      return other.getMinX() >= this._minx && other.getMaxX() <= this._maxx && other.getMinY() >= this._miny && other.getMaxY() <= this._maxy
    }
  } else if (arguments.length === 2) {
    var x = arguments[0];
    var y = arguments[1];
    if (this.isNull()) { return false }
    return x >= this._minx && x <= this._maxx && y >= this._miny && y <= this._maxy
  }
};
Envelope.prototype.intersects = function intersects () {
  if (arguments.length === 1) {
    if (arguments[0] instanceof Envelope) {
      var other = arguments[0];
      if (this.isNull() || other.isNull()) {
        return false
      }
      return !(other._minx > this._maxx || other._maxx < this._minx || other._miny > this._maxy || other._maxy < this._miny)
    } else if (arguments[0] instanceof Coordinate) {
      var p = arguments[0];
      return this.intersects(p.x, p.y)
    }
  } else if (arguments.length === 2) {
    var x = arguments[0];
    var y = arguments[1];
    if (this.isNull()) { return false }
    return !(x > this._maxx || x < this._minx || y > this._maxy || y < this._miny)
  }
};
Envelope.prototype.getMinY = function getMinY () {
  return this._miny
};
Envelope.prototype.getMinX = function getMinX () {
  return this._minx
};
Envelope.prototype.expandToInclude = function expandToInclude () {
  if (arguments.length === 1) {
    if (arguments[0] instanceof Coordinate) {
      var p = arguments[0];
      this.expandToInclude(p.x, p.y);
    } else if (arguments[0] instanceof Envelope) {
      var other = arguments[0];
      if (other.isNull()) {
        return null
      }
      if (this.isNull()) {
        this._minx = other.getMinX();
        this._maxx = other.getMaxX();
        this._miny = other.getMinY();
        this._maxy = other.getMaxY();
      } else {
        if (other._minx < this._minx) {
          this._minx = other._minx;
        }
        if (other._maxx > this._maxx) {
          this._maxx = other._maxx;
        }
        if (other._miny < this._miny) {
          this._miny = other._miny;
        }
        if (other._maxy > this._maxy) {
          this._maxy = other._maxy;
        }
      }
    }
  } else if (arguments.length === 2) {
    var x = arguments[0];
    var y = arguments[1];
    if (this.isNull()) {
      this._minx = x;
      this._maxx = x;
      this._miny = y;
      this._maxy = y;
    } else {
      if (x < this._minx) {
        this._minx = x;
      }
      if (x > this._maxx) {
        this._maxx = x;
      }
      if (y < this._miny) {
        this._miny = y;
      }
      if (y > this._maxy) {
        this._maxy = y;
      }
    }
  }
};
Envelope.prototype.minExtent = function minExtent () {
  if (this.isNull()) { return 0.0 }
  var w = this.getWidth();
  var h = this.getHeight();
  if (w < h) { return w }
  return h
};
Envelope.prototype.getWidth = function getWidth () {
  if (this.isNull()) {
    return 0
  }
  return this._maxx - this._minx
};
Envelope.prototype.compareTo = function compareTo (o) {
  var env = o;
  if (this.isNull()) {
    if (env.isNull()) { return 0 }
    return -1
  } else {
    if (env.isNull()) { return 1 }
  }
  if (this._minx < env._minx) { return -1 }
  if (this._minx > env._minx) { return 1 }
  if (this._miny < env._miny) { return -1 }
  if (this._miny > env._miny) { return 1 }
  if (this._maxx < env._maxx) { return -1 }
  if (this._maxx > env._maxx) { return 1 }
  if (this._maxy < env._maxy) { return -1 }
  if (this._maxy > env._maxy) { return 1 }
  return 0
};
Envelope.prototype.translate = function translate (transX, transY) {
  if (this.isNull()) {
    return null
  }
  this.init(this.getMinX() + transX, this.getMaxX() + transX, this.getMinY() + transY, this.getMaxY() + transY);
};
Envelope.prototype.toString = function toString () {
  return 'Env[' + this._minx + ' : ' + this._maxx + ', ' + this._miny + ' : ' + this._maxy + ']'
};
Envelope.prototype.setToNull = function setToNull () {
  this._minx = 0;
  this._maxx = -1;
  this._miny = 0;
  this._maxy = -1;
};
Envelope.prototype.getHeight = function getHeight () {
  if (this.isNull()) {
    return 0
  }
  return this._maxy - this._miny
};
Envelope.prototype.maxExtent = function maxExtent () {
  if (this.isNull()) { return 0.0 }
  var w = this.getWidth();
  var h = this.getHeight();
  if (w > h) { return w }
  return h
};
Envelope.prototype.expandBy = function expandBy () {
  if (arguments.length === 1) {
    var distance = arguments[0];
    this.expandBy(distance, distance);
  } else if (arguments.length === 2) {
    var deltaX = arguments[0];
    var deltaY = arguments[1];
    if (this.isNull()) { return null }
    this._minx -= deltaX;
    this._maxx += deltaX;
    this._miny -= deltaY;
    this._maxy += deltaY;
    if (this._minx > this._maxx || this._miny > this._maxy) { this.setToNull(); }
  }
};
Envelope.prototype.contains = function contains () {
  if (arguments.length === 1) {
    if (arguments[0] instanceof Envelope) {
      var other = arguments[0];
      return this.covers(other)
    } else if (arguments[0] instanceof Coordinate) {
      var p = arguments[0];
      return this.covers(p)
    }
  } else if (arguments.length === 2) {
    var x = arguments[0];
    var y = arguments[1];
    return this.covers(x, y)
  }
};
Envelope.prototype.centre = function centre () {
  if (this.isNull()) { return null }
  return new Coordinate((this.getMinX() + this.getMaxX()) / 2.0, (this.getMinY() + this.getMaxY()) / 2.0)
};
Envelope.prototype.init = function init () {
  if (arguments.length === 0) {
    this.setToNull();
  } else if (arguments.length === 1) {
    if (arguments[0] instanceof Coordinate) {
      var p = arguments[0];
      this.init(p.x, p.x, p.y, p.y);
    } else if (arguments[0] instanceof Envelope) {
      var env = arguments[0];
      this._minx = env._minx;
      this._maxx = env._maxx;
      this._miny = env._miny;
      this._maxy = env._maxy;
    }
  } else if (arguments.length === 2) {
    var p1 = arguments[0];
    var p2 = arguments[1];
    this.init(p1.x, p2.x, p1.y, p2.y);
  } else if (arguments.length === 4) {
    var x1 = arguments[0];
    var x2 = arguments[1];
    var y1 = arguments[2];
    var y2 = arguments[3];
    if (x1 < x2) {
      this._minx = x1;
      this._maxx = x2;
    } else {
      this._minx = x2;
      this._maxx = x1;
    }
    if (y1 < y2) {
      this._miny = y1;
      this._maxy = y2;
    } else {
      this._miny = y2;
      this._maxy = y1;
    }
  }
};
Envelope.prototype.getMaxY = function getMaxY () {
  return this._maxy
};
Envelope.prototype.distance = function distance (env) {
  if (this.intersects(env)) { return 0 }
  var dx = 0.0;
  if (this._maxx < env._minx) { dx = env._minx - this._maxx; } else if (this._minx > env._maxx) { dx = this._minx - env._maxx; }
  var dy = 0.0;
  if (this._maxy < env._miny) { dy = env._miny - this._maxy; } else if (this._miny > env._maxy) { dy = this._miny - env._maxy; }
  if (dx === 0.0) { return dy }
  if (dy === 0.0) { return dx }
  return Math.sqrt(dx * dx + dy * dy)
};
Envelope.prototype.hashCode = function hashCode () {
  var result = 17;
  result = 37 * result + Coordinate.hashCode(this._minx);
  result = 37 * result + Coordinate.hashCode(this._maxx);
  result = 37 * result + Coordinate.hashCode(this._miny);
  result = 37 * result + Coordinate.hashCode(this._maxy);
  return result
};
Envelope.prototype.interfaces_ = function interfaces_ () {
  return [Comparable, Serializable]
};
Envelope.prototype.getClass = function getClass () {
  return Envelope
};
Envelope.intersects = function intersects () {
  if (arguments.length === 3) {
    var p1 = arguments[0];
    var p2 = arguments[1];
    var q = arguments[2];
    if (q.x >= (p1.x < p2.x ? p1.x : p2.x) && q.x <= (p1.x > p2.x ? p1.x : p2.x) && (q.y >= (p1.y < p2.y ? p1.y : p2.y) && q.y <= (p1.y > p2.y ? p1.y : p2.y))) {
      return true
    }
    return false
  } else if (arguments.length === 4) {
    var p1$1 = arguments[0];
    var p2$1 = arguments[1];
    var q1 = arguments[2];
    var q2 = arguments[3];
    var minq = Math.min(q1.x, q2.x);
    var maxq = Math.max(q1.x, q2.x);
    var minp = Math.min(p1$1.x, p2$1.x);
    var maxp = Math.max(p1$1.x, p2$1.x);
    if (minp > maxq) { return false }
    if (maxp < minq) { return false }
    minq = Math.min(q1.y, q2.y);
    maxq = Math.max(q1.y, q2.y);
    minp = Math.min(p1$1.y, p2$1.y);
    maxp = Math.max(p1$1.y, p2$1.y);
    if (minp > maxq) { return false }
    if (maxp < minq) { return false }
    return true
  }
};
staticAccessors$9.serialVersionUID.get = function () { return 5873921885273102420 };

Object.defineProperties( Envelope, staticAccessors$9 );

var regExes = {
  'typeStr': /^\s*(\w+)\s*\(\s*(.*)\s*\)\s*$/,
  'emptyTypeStr': /^\s*(\w+)\s*EMPTY\s*$/,
  'spaces': /\s+/,
  'parenComma': /\)\s*,\s*\(/,
  'doubleParenComma': /\)\s*\)\s*,\s*\(\s*\(/, // can't use {2} here
  'trimParens': /^\s*\(?(.*?)\)?\s*$/
};

/**
 * Class for reading and writing Well-Known Text.
 *
 * NOTE: Adapted from OpenLayers 2.11 implementation.
 */

/** Create a new parser for WKT
 *
 * @param {GeometryFactory} geometryFactory
 * @return An instance of WKTParser.
 * @constructor
 * @private
 */
var WKTParser = function WKTParser (geometryFactory) {
  this.geometryFactory = geometryFactory || new GeometryFactory();
};
/**
 * Deserialize a WKT string and return a geometry. Supports WKT for POINT,
 * MULTIPOINT, LINESTRING, LINEARRING, MULTILINESTRING, POLYGON, MULTIPOLYGON,
 * and GEOMETRYCOLLECTION.
 *
 * @param {String} wkt A WKT string.
 * @return {Geometry} A geometry instance.
 * @private
 */
WKTParser.prototype.read = function read (wkt) {
  var geometry, type, str;
  wkt = wkt.replace(/[\n\r]/g, ' ');
  var matches = regExes.typeStr.exec(wkt);
  if (wkt.search('EMPTY') !== -1) {
    matches = regExes.emptyTypeStr.exec(wkt);
    matches[2] = undefined;
  }
  if (matches) {
    type = matches[1].toLowerCase();
    str = matches[2];
    if (parse$1[type]) {
      geometry = parse$1[type].apply(this, [str]);
    }
  }

  if (geometry === undefined) { throw new Error('Could not parse WKT ' + wkt) }

  return geometry
};

/**
 * Serialize a geometry into a WKT string.
 *
 * @param {Geometry} geometry A feature or array of features.
 * @return {String} The WKT string representation of the input geometries.
 * @private
 */
WKTParser.prototype.write = function write (geometry) {
  return this.extractGeometry(geometry)
};

/**
 * Entry point to construct the WKT for a single Geometry object.
 *
 * @param {Geometry} geometry
 * @return {String} A WKT string of representing the geometry.
 * @private
 */
WKTParser.prototype.extractGeometry = function extractGeometry (geometry) {
  var type = geometry.getGeometryType().toLowerCase();
  if (!extract$1[type]) {
    return null
  }
  var wktType = type.toUpperCase();
  var data;
  if (geometry.isEmpty()) {
    data = wktType + ' EMPTY';
  } else {
    data = wktType + '(' + extract$1[type].apply(this, [geometry]) + ')';
  }
  return data
};

/**
 * Object with properties corresponding to the geometry types. Property values
 * are functions that do the actual data extraction.
 * @private
 */
var extract$1 = {
  coordinate: function coordinate (coordinate$1) {
    return coordinate$1.x + ' ' + coordinate$1.y
  },

  /**
   * Return a space delimited string of point coordinates.
   *
   * @param {Point}
   *          point
   * @return {String} A string of coordinates representing the point.
   */
  point: function point (point$1) {
    return extract$1.coordinate.call(this, point$1._coordinates._coordinates[0])
  },

  /**
   * Return a comma delimited string of point coordinates from a multipoint.
   *
   * @param {MultiPoint}
   *          multipoint
   * @return {String} A string of point coordinate strings representing the
   *         multipoint.
   */
  multipoint: function multipoint (multipoint$1) {
    var this$1 = this;

    var array = [];
    for (var i = 0, len = multipoint$1._geometries.length; i < len; ++i) {
      array.push('(' + extract$1.point.apply(this$1, [multipoint$1._geometries[i]]) + ')');
    }
    return array.join(',')
  },

  /**
   * Return a comma delimited string of point coordinates from a line.
   *
   * @param {LineString} linestring
   * @return {String} A string of point coordinate strings representing the linestring.
   */
  linestring: function linestring (linestring$1) {
    var this$1 = this;

    var array = [];
    for (var i = 0, len = linestring$1._points._coordinates.length; i < len; ++i) {
      array.push(extract$1.coordinate.apply(this$1, [linestring$1._points._coordinates[i]]));
    }
    return array.join(',')
  },

  linearring: function linearring (linearring$1) {
    var this$1 = this;

    var array = [];
    for (var i = 0, len = linearring$1._points._coordinates.length; i < len; ++i) {
      array.push(extract$1.coordinate.apply(this$1, [linearring$1._points._coordinates[i]]));
    }
    return array.join(',')
  },

  /**
   * Return a comma delimited string of linestring strings from a
   * multilinestring.
   *
   * @param {MultiLineString} multilinestring
   * @return {String} A string of of linestring strings representing the multilinestring.
   */
  multilinestring: function multilinestring (multilinestring$1) {
    var this$1 = this;

    var array = [];
    for (var i = 0, len = multilinestring$1._geometries.length; i < len; ++i) {
      array.push('(' +
        extract$1.linestring.apply(this$1, [multilinestring$1._geometries[i]]) +
        ')');
    }
    return array.join(',')
  },

  /**
   * Return a comma delimited string of linear ring arrays from a polygon.
   *
   * @param {Polygon} polygon
   * @return {String} An array of linear ring arrays representing the polygon.
   */
  polygon: function polygon (polygon$1) {
    var this$1 = this;

    var array = [];
    array.push('(' + extract$1.linestring.apply(this, [polygon$1._shell]) + ')');
    for (var i = 0, len = polygon$1._holes.length; i < len; ++i) {
      array.push('(' + extract$1.linestring.apply(this$1, [polygon$1._holes[i]]) + ')');
    }
    return array.join(',')
  },

  /**
   * Return an array of polygon arrays from a multipolygon.
   *
   * @param {MultiPolygon} multipolygon
   * @return {String} An array of polygon arrays representing the multipolygon.
   */
  multipolygon: function multipolygon (multipolygon$1) {
    var this$1 = this;

    var array = [];
    for (var i = 0, len = multipolygon$1._geometries.length; i < len; ++i) {
      array.push('(' + extract$1.polygon.apply(this$1, [multipolygon$1._geometries[i]]) + ')');
    }
    return array.join(',')
  },

  /**
   * Return the WKT portion between 'GEOMETRYCOLLECTION(' and ')' for an
   * geometrycollection.
   *
   * @param {GeometryCollection} collection
   * @return {String} internal WKT representation of the collection.
   */
  geometrycollection: function geometrycollection (collection) {
    var this$1 = this;

    var array = [];
    for (var i = 0, len = collection._geometries.length; i < len; ++i) {
      array.push(this$1.extractGeometry(collection._geometries[i]));
    }
    return array.join(',')
  }
};

/**
 * Object with properties corresponding to the geometry types. Property values
 * are functions that do the actual parsing.
 * @private
 */
var parse$1 = {
  /**
   * Return point geometry given a point WKT fragment.
   *
   * @param {String} str A WKT fragment representing the point.
   * @return {Point} A point geometry.
   * @private
   */
  point: function point (str) {
    if (str === undefined) {
      return this.geometryFactory.createPoint()
    }

    var coords = str.trim().split(regExes.spaces);
    return this.geometryFactory.createPoint(new Coordinate(Number.parseFloat(coords[0]),
      Number.parseFloat(coords[1])))
  },

  /**
   * Return a multipoint geometry given a multipoint WKT fragment.
   *
   * @param {String} str A WKT fragment representing the multipoint.
   * @return {Point} A multipoint feature.
   * @private
   */
  multipoint: function multipoint (str) {
    var this$1 = this;

    if (str === undefined) {
      return this.geometryFactory.createMultiPoint()
    }

    var point;
    var points = str.trim().split(',');
    var components = [];
    for (var i = 0, len = points.length; i < len; ++i) {
      point = points[i].replace(regExes.trimParens, '$1');
      components.push(parse$1.point.apply(this$1, [point]));
    }
    return this.geometryFactory.createMultiPoint(components)
  },

  /**
   * Return a linestring geometry given a linestring WKT fragment.
   *
   * @param {String} str A WKT fragment representing the linestring.
   * @return {LineString} A linestring geometry.
   * @private
   */
  linestring: function linestring (str) {
    if (str === undefined) {
      return this.geometryFactory.createLineString()
    }

    var points = str.trim().split(',');
    var components = [];
    var coords;
    for (var i = 0, len = points.length; i < len; ++i) {
      coords = points[i].trim().split(regExes.spaces);
      components.push(new Coordinate(Number.parseFloat(coords[0]), Number.parseFloat(coords[1])));
    }
    return this.geometryFactory.createLineString(components)
  },

  /**
   * Return a linearring geometry given a linearring WKT fragment.
   *
   * @param {String} str A WKT fragment representing the linearring.
   * @return {LinearRing} A linearring geometry.
   * @private
   */
  linearring: function linearring (str) {
    if (str === undefined) {
      return this.geometryFactory.createLinearRing()
    }

    var points = str.trim().split(',');
    var components = [];
    var coords;
    for (var i = 0, len = points.length; i < len; ++i) {
      coords = points[i].trim().split(regExes.spaces);
      components.push(new Coordinate(Number.parseFloat(coords[0]), Number.parseFloat(coords[1])));
    }
    return this.geometryFactory.createLinearRing(components)
  },

  /**
   * Return a multilinestring geometry given a multilinestring WKT fragment.
   *
   * @param {String} str A WKT fragment representing the multilinestring.
   * @return {MultiLineString} A multilinestring geometry.
   * @private
   */
  multilinestring: function multilinestring (str) {
    var this$1 = this;

    if (str === undefined) {
      return this.geometryFactory.createMultiLineString()
    }

    var line;
    var lines = str.trim().split(regExes.parenComma);
    var components = [];
    for (var i = 0, len = lines.length; i < len; ++i) {
      line = lines[i].replace(regExes.trimParens, '$1');
      components.push(parse$1.linestring.apply(this$1, [line]));
    }
    return this.geometryFactory.createMultiLineString(components)
  },

  /**
   * Return a polygon geometry given a polygon WKT fragment.
   *
   * @param {String} str A WKT fragment representing the polygon.
   * @return {Polygon} A polygon geometry.
   * @private
   */
  polygon: function polygon (str) {
    var this$1 = this;

    if (str === undefined) {
      return this.geometryFactory.createPolygon()
    }

    var ring, linestring, linearring;
    var rings = str.trim().split(regExes.parenComma);
    var shell;
    var holes = [];
    for (var i = 0, len = rings.length; i < len; ++i) {
      ring = rings[i].replace(regExes.trimParens, '$1');
      linestring = parse$1.linestring.apply(this$1, [ring]);
      linearring = this$1.geometryFactory.createLinearRing(linestring._points);
      if (i === 0) {
        shell = linearring;
      } else {
        holes.push(linearring);
      }
    }
    return this.geometryFactory.createPolygon(shell, holes)
  },

  /**
   * Return a multipolygon geometry given a multipolygon WKT fragment.
   *
   * @param {String} str A WKT fragment representing the multipolygon.
   * @return {MultiPolygon} A multipolygon geometry.
   * @private
   */
  multipolygon: function multipolygon (str) {
    var this$1 = this;

    if (str === undefined) {
      return this.geometryFactory.createMultiPolygon()
    }

    var polygon;
    var polygons = str.trim().split(regExes.doubleParenComma);
    var components = [];
    for (var i = 0, len = polygons.length; i < len; ++i) {
      polygon = polygons[i].replace(regExes.trimParens, '$1');
      components.push(parse$1.polygon.apply(this$1, [polygon]));
    }
    return this.geometryFactory.createMultiPolygon(components)
  },

  /**
   * Return a geometrycollection given a geometrycollection WKT fragment.
   *
   * @param {String} str A WKT fragment representing the geometrycollection.
   * @return {GeometryCollection}
   * @private
   */
  geometrycollection: function geometrycollection (str) {
    var this$1 = this;

    if (str === undefined) {
      return this.geometryFactory.createGeometryCollection()
    }

    // separate components of the collection with |
    str = str.replace(/,\s*([A-Za-z])/g, '|$1');
    var wktArray = str.trim().split('|');
    var components = [];
    for (var i = 0, len = wktArray.length; i < len; ++i) {
      components.push(this$1.read(wktArray[i]));
    }
    return this.geometryFactory.createGeometryCollection(components)
  }
};

/**
 * Writes the Well-Known Text representation of a {@link Geometry}. The
 * Well-Known Text format is defined in the <A
 * HREF="http://www.opengis.org/techno/specs.htm"> OGC Simple Features
 * Specification for SQL</A>.
 * <p>
 * The <code>WKTWriter</code> outputs coordinates rounded to the precision
 * model. Only the maximum number of decimal places necessary to represent the
 * ordinates to the required precision will be output.
 * <p>
 * The SFS WKT spec does not define a special tag for {@link LinearRing}s.
 * Under the spec, rings are output as <code>LINESTRING</code>s.
 */

/**
 * @param {GeometryFactory} geometryFactory
 * @constructor
 */
var WKTWriter = function WKTWriter (geometryFactory) {
  this.parser = new WKTParser(geometryFactory);
};

/**
 * Converts a <code>Geometry</code> to its Well-known Text representation.
 *
 * @param {Geometry} geometry a <code>Geometry</code> to process.
 * @return {string} a <Geometry Tagged Text> string (see the OpenGIS Simple
 *       Features Specification).
 * @memberof WKTWriter
 */
WKTWriter.prototype.write = function write (geometry) {
  return this.parser.write(geometry)
};
/**
 * Generates the WKT for a <tt>LINESTRING</tt> specified by two
 * {@link Coordinate}s.
 *
 * @param p0 the first coordinate.
 * @param p1 the second coordinate.
 *
 * @return the WKT.
 * @private
 */
WKTWriter.toLineString = function toLineString (p0, p1) {
  if (arguments.length !== 2) {
    throw new Error('Not implemented')
  }
  return 'LINESTRING ( ' + p0.x + ' ' + p0.y + ', ' + p1.x + ' ' + p1.y + ' )'
};

var RuntimeException = (function (Error) {
  function RuntimeException (message) {
    Error.call(this, message);
    this.name = 'RuntimeException';
    this.message = message;
    this.stack = (new Error()).stack;
  }

  if ( Error ) RuntimeException.__proto__ = Error;
  RuntimeException.prototype = Object.create( Error && Error.prototype );
  RuntimeException.prototype.constructor = RuntimeException;

  return RuntimeException;
}(Error));

var AssertionFailedException = (function (RuntimeException$$1) {
  function AssertionFailedException () {
    RuntimeException$$1.call(this);
    if (arguments.length === 0) {
      RuntimeException$$1.call(this);
    } else if (arguments.length === 1) {
      var message = arguments[0];
      RuntimeException$$1.call(this, message);
    }
  }

  if ( RuntimeException$$1 ) AssertionFailedException.__proto__ = RuntimeException$$1;
  AssertionFailedException.prototype = Object.create( RuntimeException$$1 && RuntimeException$$1.prototype );
  AssertionFailedException.prototype.constructor = AssertionFailedException;
  AssertionFailedException.prototype.interfaces_ = function interfaces_ () {
    return []
  };
  AssertionFailedException.prototype.getClass = function getClass () {
    return AssertionFailedException
  };

  return AssertionFailedException;
}(RuntimeException));

var Assert = function Assert () {};

Assert.prototype.interfaces_ = function interfaces_ () {
  return []
};
Assert.prototype.getClass = function getClass () {
  return Assert
};
Assert.shouldNeverReachHere = function shouldNeverReachHere () {
  if (arguments.length === 0) {
    Assert.shouldNeverReachHere(null);
  } else if (arguments.length === 1) {
    var message = arguments[0];
    throw new AssertionFailedException('Should never reach here' + (message !== null ? ': ' + message : ''))
  }
};
Assert.isTrue = function isTrue () {
  var assertion;
  var message;
  if (arguments.length === 1) {
    assertion = arguments[0];
    Assert.isTrue(assertion, null);
  } else if (arguments.length === 2) {
    assertion = arguments[0];
    message = arguments[1];
    if (!assertion) {
      if (message === null) {
        throw new AssertionFailedException()
      } else {
        throw new AssertionFailedException(message)
      }
    }
  }
};
Assert.equals = function equals () {
  var expectedValue;
  var actualValue;
  var message;
  if (arguments.length === 2) {
    expectedValue = arguments[0];
    actualValue = arguments[1];
    Assert.equals(expectedValue, actualValue, null);
  } else if (arguments.length === 3) {
    expectedValue = arguments[0];
    actualValue = arguments[1];
    message = arguments[2];
    if (!actualValue.equals(expectedValue)) {
      throw new AssertionFailedException('Expected ' + expectedValue + ' but encountered ' + actualValue + (message !== null ? ': ' + message : ''))
    }
  }
};

var LineIntersector = function LineIntersector () {
  this._result = null;
  this._inputLines = Array(2).fill().map(function () { return Array(2); });
  this._intPt = new Array(2).fill(null);
  this._intLineIndex = null;
  this._isProper = null;
  this._pa = null;
  this._pb = null;
  this._precisionModel = null;
  this._intPt[0] = new Coordinate();
  this._intPt[1] = new Coordinate();
  this._pa = this._intPt[0];
  this._pb = this._intPt[1];
  this._result = 0;
};

var staticAccessors$10 = { DONT_INTERSECT: { configurable: true },DO_INTERSECT: { configurable: true },COLLINEAR: { configurable: true },NO_INTERSECTION: { configurable: true },POINT_INTERSECTION: { configurable: true },COLLINEAR_INTERSECTION: { configurable: true } };
LineIntersector.prototype.getIndexAlongSegment = function getIndexAlongSegment (segmentIndex, intIndex) {
  this.computeIntLineIndex();
  return this._intLineIndex[segmentIndex][intIndex]
};
LineIntersector.prototype.getTopologySummary = function getTopologySummary () {
  var catBuf = new StringBuffer();
  if (this.isEndPoint()) { catBuf.append(' endpoint'); }
  if (this._isProper) { catBuf.append(' proper'); }
  if (this.isCollinear()) { catBuf.append(' collinear'); }
  return catBuf.toString()
};
LineIntersector.prototype.computeIntersection = function computeIntersection (p1, p2, p3, p4) {
  this._inputLines[0][0] = p1;
  this._inputLines[0][1] = p2;
  this._inputLines[1][0] = p3;
  this._inputLines[1][1] = p4;
  this._result = this.computeIntersect(p1, p2, p3, p4);
};
LineIntersector.prototype.getIntersectionNum = function getIntersectionNum () {
  return this._result
};
LineIntersector.prototype.computeIntLineIndex = function computeIntLineIndex () {
  if (arguments.length === 0) {
    if (this._intLineIndex === null) {
      this._intLineIndex = Array(2).fill().map(function () { return Array(2); });
      this.computeIntLineIndex(0);
      this.computeIntLineIndex(1);
    }
  } else if (arguments.length === 1) {
    var segmentIndex = arguments[0];
    var dist0 = this.getEdgeDistance(segmentIndex, 0);
    var dist1 = this.getEdgeDistance(segmentIndex, 1);
    if (dist0 > dist1) {
      this._intLineIndex[segmentIndex][0] = 0;
      this._intLineIndex[segmentIndex][1] = 1;
    } else {
      this._intLineIndex[segmentIndex][0] = 1;
      this._intLineIndex[segmentIndex][1] = 0;
    }
  }
};
LineIntersector.prototype.isProper = function isProper () {
  return this.hasIntersection() && this._isProper
};
LineIntersector.prototype.setPrecisionModel = function setPrecisionModel (precisionModel) {
  this._precisionModel = precisionModel;
};
LineIntersector.prototype.isInteriorIntersection = function isInteriorIntersection () {
    var this$1 = this;

  if (arguments.length === 0) {
    if (this.isInteriorIntersection(0)) { return true }
    if (this.isInteriorIntersection(1)) { return true }
    return false
  } else if (arguments.length === 1) {
    var inputLineIndex = arguments[0];
    for (var i = 0; i < this._result; i++) {
      if (!(this$1._intPt[i].equals2D(this$1._inputLines[inputLineIndex][0]) || this$1._intPt[i].equals2D(this$1._inputLines[inputLineIndex][1]))) {
        return true
      }
    }
    return false
  }
};
LineIntersector.prototype.getIntersection = function getIntersection (intIndex) {
  return this._intPt[intIndex]
};
LineIntersector.prototype.isEndPoint = function isEndPoint () {
  return this.hasIntersection() && !this._isProper
};
LineIntersector.prototype.hasIntersection = function hasIntersection () {
  return this._result !== LineIntersector.NO_INTERSECTION
};
LineIntersector.prototype.getEdgeDistance = function getEdgeDistance (segmentIndex, intIndex) {
  var dist = LineIntersector.computeEdgeDistance(this._intPt[intIndex], this._inputLines[segmentIndex][0], this._inputLines[segmentIndex][1]);
  return dist
};
LineIntersector.prototype.isCollinear = function isCollinear () {
  return this._result === LineIntersector.COLLINEAR_INTERSECTION
};
LineIntersector.prototype.toString = function toString () {
  return WKTWriter.toLineString(this._inputLines[0][0], this._inputLines[0][1]) + ' - ' + WKTWriter.toLineString(this._inputLines[1][0], this._inputLines[1][1]) + this.getTopologySummary()
};
LineIntersector.prototype.getEndpoint = function getEndpoint (segmentIndex, ptIndex) {
  return this._inputLines[segmentIndex][ptIndex]
};
LineIntersector.prototype.isIntersection = function isIntersection (pt) {
    var this$1 = this;

  for (var i = 0; i < this._result; i++) {
    if (this$1._intPt[i].equals2D(pt)) {
      return true
    }
  }
  return false
};
LineIntersector.prototype.getIntersectionAlongSegment = function getIntersectionAlongSegment (segmentIndex, intIndex) {
  this.computeIntLineIndex();
  return this._intPt[this._intLineIndex[segmentIndex][intIndex]]
};
LineIntersector.prototype.interfaces_ = function interfaces_ () {
  return []
};
LineIntersector.prototype.getClass = function getClass () {
  return LineIntersector
};
LineIntersector.computeEdgeDistance = function computeEdgeDistance (p, p0, p1) {
  var dx = Math.abs(p1.x - p0.x);
  var dy = Math.abs(p1.y - p0.y);
  var dist = -1.0;
  if (p.equals(p0)) {
    dist = 0.0;
  } else if (p.equals(p1)) {
    if (dx > dy) { dist = dx; } else { dist = dy; }
  } else {
    var pdx = Math.abs(p.x - p0.x);
    var pdy = Math.abs(p.y - p0.y);
    if (dx > dy) { dist = pdx; } else { dist = pdy; }
    if (dist === 0.0 && !p.equals(p0)) {
      dist = Math.max(pdx, pdy);
    }
  }
  Assert.isTrue(!(dist === 0.0 && !p.equals(p0)), 'Bad distance calculation');
  return dist
};
LineIntersector.nonRobustComputeEdgeDistance = function nonRobustComputeEdgeDistance (p, p1, p2) {
  var dx = p.x - p1.x;
  var dy = p.y - p1.y;
  var dist = Math.sqrt(dx * dx + dy * dy);
  Assert.isTrue(!(dist === 0.0 && !p.equals(p1)), 'Invalid distance calculation');
  return dist
};
staticAccessors$10.DONT_INTERSECT.get = function () { return 0 };
staticAccessors$10.DO_INTERSECT.get = function () { return 1 };
staticAccessors$10.COLLINEAR.get = function () { return 2 };
staticAccessors$10.NO_INTERSECTION.get = function () { return 0 };
staticAccessors$10.POINT_INTERSECTION.get = function () { return 1 };
staticAccessors$10.COLLINEAR_INTERSECTION.get = function () { return 2 };

Object.defineProperties( LineIntersector, staticAccessors$10 );

var RobustLineIntersector = (function (LineIntersector$$1) {
  function RobustLineIntersector () {
    LineIntersector$$1.apply(this, arguments);
  }

  if ( LineIntersector$$1 ) RobustLineIntersector.__proto__ = LineIntersector$$1;
  RobustLineIntersector.prototype = Object.create( LineIntersector$$1 && LineIntersector$$1.prototype );
  RobustLineIntersector.prototype.constructor = RobustLineIntersector;

  RobustLineIntersector.prototype.isInSegmentEnvelopes = function isInSegmentEnvelopes (intPt) {
    var env0 = new Envelope(this._inputLines[0][0], this._inputLines[0][1]);
    var env1 = new Envelope(this._inputLines[1][0], this._inputLines[1][1]);
    return env0.contains(intPt) && env1.contains(intPt)
  };
  RobustLineIntersector.prototype.computeIntersection = function computeIntersection () {
    if (arguments.length === 3) {
      var p = arguments[0];
      var p1 = arguments[1];
      var p2 = arguments[2];
      this._isProper = false;
      if (Envelope.intersects(p1, p2, p)) {
        if (CGAlgorithms.orientationIndex(p1, p2, p) === 0 && CGAlgorithms.orientationIndex(p2, p1, p) === 0) {
          this._isProper = true;
          if (p.equals(p1) || p.equals(p2)) {
            this._isProper = false;
          }
          this._result = LineIntersector$$1.POINT_INTERSECTION;
          return null
        }
      }
      this._result = LineIntersector$$1.NO_INTERSECTION;
    } else { return LineIntersector$$1.prototype.computeIntersection.apply(this, arguments) }
  };
  RobustLineIntersector.prototype.normalizeToMinimum = function normalizeToMinimum (n1, n2, n3, n4, normPt) {
    normPt.x = this.smallestInAbsValue(n1.x, n2.x, n3.x, n4.x);
    normPt.y = this.smallestInAbsValue(n1.y, n2.y, n3.y, n4.y);
    n1.x -= normPt.x;
    n1.y -= normPt.y;
    n2.x -= normPt.x;
    n2.y -= normPt.y;
    n3.x -= normPt.x;
    n3.y -= normPt.y;
    n4.x -= normPt.x;
    n4.y -= normPt.y;
  };
  RobustLineIntersector.prototype.safeHCoordinateIntersection = function safeHCoordinateIntersection (p1, p2, q1, q2) {
    var intPt = null;
    try {
      intPt = HCoordinate.intersection(p1, p2, q1, q2);
    } catch (e) {
      if (e instanceof NotRepresentableException) {
        intPt = RobustLineIntersector.nearestEndpoint(p1, p2, q1, q2);
      } else { throw e }
    } finally {}
    return intPt
  };
  RobustLineIntersector.prototype.intersection = function intersection (p1, p2, q1, q2) {
    var intPt = this.intersectionWithNormalization(p1, p2, q1, q2);
    if (!this.isInSegmentEnvelopes(intPt)) {
      intPt = new Coordinate(RobustLineIntersector.nearestEndpoint(p1, p2, q1, q2));
    }
    if (this._precisionModel !== null) {
      this._precisionModel.makePrecise(intPt);
    }
    return intPt
  };
  RobustLineIntersector.prototype.smallestInAbsValue = function smallestInAbsValue (x1, x2, x3, x4) {
    var x = x1;
    var xabs = Math.abs(x);
    if (Math.abs(x2) < xabs) {
      x = x2;
      xabs = Math.abs(x2);
    }
    if (Math.abs(x3) < xabs) {
      x = x3;
      xabs = Math.abs(x3);
    }
    if (Math.abs(x4) < xabs) {
      x = x4;
    }
    return x
  };
  RobustLineIntersector.prototype.checkDD = function checkDD (p1, p2, q1, q2, intPt) {
    var intPtDD = CGAlgorithmsDD.intersection(p1, p2, q1, q2);
    var isIn = this.isInSegmentEnvelopes(intPtDD);
    System.out.println('DD in env = ' + isIn + '  --------------------- ' + intPtDD);
    if (intPt.distance(intPtDD) > 0.0001) {
      System.out.println('Distance = ' + intPt.distance(intPtDD));
    }
  };
  RobustLineIntersector.prototype.intersectionWithNormalization = function intersectionWithNormalization (p1, p2, q1, q2) {
    var n1 = new Coordinate(p1);
    var n2 = new Coordinate(p2);
    var n3 = new Coordinate(q1);
    var n4 = new Coordinate(q2);
    var normPt = new Coordinate();
    this.normalizeToEnvCentre(n1, n2, n3, n4, normPt);
    var intPt = this.safeHCoordinateIntersection(n1, n2, n3, n4);
    intPt.x += normPt.x;
    intPt.y += normPt.y;
    return intPt
  };
  RobustLineIntersector.prototype.computeCollinearIntersection = function computeCollinearIntersection (p1, p2, q1, q2) {
    var p1q1p2 = Envelope.intersects(p1, p2, q1);
    var p1q2p2 = Envelope.intersects(p1, p2, q2);
    var q1p1q2 = Envelope.intersects(q1, q2, p1);
    var q1p2q2 = Envelope.intersects(q1, q2, p2);
    if (p1q1p2 && p1q2p2) {
      this._intPt[0] = q1;
      this._intPt[1] = q2;
      return LineIntersector$$1.COLLINEAR_INTERSECTION
    }
    if (q1p1q2 && q1p2q2) {
      this._intPt[0] = p1;
      this._intPt[1] = p2;
      return LineIntersector$$1.COLLINEAR_INTERSECTION
    }
    if (p1q1p2 && q1p1q2) {
      this._intPt[0] = q1;
      this._intPt[1] = p1;
      return q1.equals(p1) && !p1q2p2 && !q1p2q2 ? LineIntersector$$1.POINT_INTERSECTION : LineIntersector$$1.COLLINEAR_INTERSECTION
    }
    if (p1q1p2 && q1p2q2) {
      this._intPt[0] = q1;
      this._intPt[1] = p2;
      return q1.equals(p2) && !p1q2p2 && !q1p1q2 ? LineIntersector$$1.POINT_INTERSECTION : LineIntersector$$1.COLLINEAR_INTERSECTION
    }
    if (p1q2p2 && q1p1q2) {
      this._intPt[0] = q2;
      this._intPt[1] = p1;
      return q2.equals(p1) && !p1q1p2 && !q1p2q2 ? LineIntersector$$1.POINT_INTERSECTION : LineIntersector$$1.COLLINEAR_INTERSECTION
    }
    if (p1q2p2 && q1p2q2) {
      this._intPt[0] = q2;
      this._intPt[1] = p2;
      return q2.equals(p2) && !p1q1p2 && !q1p1q2 ? LineIntersector$$1.POINT_INTERSECTION : LineIntersector$$1.COLLINEAR_INTERSECTION
    }
    return LineIntersector$$1.NO_INTERSECTION
  };
  RobustLineIntersector.prototype.normalizeToEnvCentre = function normalizeToEnvCentre (n00, n01, n10, n11, normPt) {
    var minX0 = n00.x < n01.x ? n00.x : n01.x;
    var minY0 = n00.y < n01.y ? n00.y : n01.y;
    var maxX0 = n00.x > n01.x ? n00.x : n01.x;
    var maxY0 = n00.y > n01.y ? n00.y : n01.y;
    var minX1 = n10.x < n11.x ? n10.x : n11.x;
    var minY1 = n10.y < n11.y ? n10.y : n11.y;
    var maxX1 = n10.x > n11.x ? n10.x : n11.x;
    var maxY1 = n10.y > n11.y ? n10.y : n11.y;
    var intMinX = minX0 > minX1 ? minX0 : minX1;
    var intMaxX = maxX0 < maxX1 ? maxX0 : maxX1;
    var intMinY = minY0 > minY1 ? minY0 : minY1;
    var intMaxY = maxY0 < maxY1 ? maxY0 : maxY1;
    var intMidX = (intMinX + intMaxX) / 2.0;
    var intMidY = (intMinY + intMaxY) / 2.0;
    normPt.x = intMidX;
    normPt.y = intMidY;
    n00.x -= normPt.x;
    n00.y -= normPt.y;
    n01.x -= normPt.x;
    n01.y -= normPt.y;
    n10.x -= normPt.x;
    n10.y -= normPt.y;
    n11.x -= normPt.x;
    n11.y -= normPt.y;
  };
  RobustLineIntersector.prototype.computeIntersect = function computeIntersect (p1, p2, q1, q2) {
    this._isProper = false;
    if (!Envelope.intersects(p1, p2, q1, q2)) { return LineIntersector$$1.NO_INTERSECTION }
    var Pq1 = CGAlgorithms.orientationIndex(p1, p2, q1);
    var Pq2 = CGAlgorithms.orientationIndex(p1, p2, q2);
    if ((Pq1 > 0 && Pq2 > 0) || (Pq1 < 0 && Pq2 < 0)) {
      return LineIntersector$$1.NO_INTERSECTION
    }
    var Qp1 = CGAlgorithms.orientationIndex(q1, q2, p1);
    var Qp2 = CGAlgorithms.orientationIndex(q1, q2, p2);
    if ((Qp1 > 0 && Qp2 > 0) || (Qp1 < 0 && Qp2 < 0)) {
      return LineIntersector$$1.NO_INTERSECTION
    }
    var collinear = Pq1 === 0 && Pq2 === 0 && Qp1 === 0 && Qp2 === 0;
    if (collinear) {
      return this.computeCollinearIntersection(p1, p2, q1, q2)
    }
    if (Pq1 === 0 || Pq2 === 0 || Qp1 === 0 || Qp2 === 0) {
      this._isProper = false;
      if (p1.equals2D(q1) || p1.equals2D(q2)) {
        this._intPt[0] = p1;
      } else if (p2.equals2D(q1) || p2.equals2D(q2)) {
        this._intPt[0] = p2;
      } else if (Pq1 === 0) {
        this._intPt[0] = new Coordinate(q1);
      } else if (Pq2 === 0) {
        this._intPt[0] = new Coordinate(q2);
      } else if (Qp1 === 0) {
        this._intPt[0] = new Coordinate(p1);
      } else if (Qp2 === 0) {
        this._intPt[0] = new Coordinate(p2);
      }
    } else {
      this._isProper = true;
      this._intPt[0] = this.intersection(p1, p2, q1, q2);
    }
    return LineIntersector$$1.POINT_INTERSECTION
  };
  RobustLineIntersector.prototype.interfaces_ = function interfaces_ () {
    return []
  };
  RobustLineIntersector.prototype.getClass = function getClass () {
    return RobustLineIntersector
  };
  RobustLineIntersector.nearestEndpoint = function nearestEndpoint (p1, p2, q1, q2) {
    var nearestPt = p1;
    var minDist = CGAlgorithms.distancePointLine(p1, q1, q2);
    var dist = CGAlgorithms.distancePointLine(p2, q1, q2);
    if (dist < minDist) {
      minDist = dist;
      nearestPt = p2;
    }
    dist = CGAlgorithms.distancePointLine(q1, p1, p2);
    if (dist < minDist) {
      minDist = dist;
      nearestPt = q1;
    }
    dist = CGAlgorithms.distancePointLine(q2, p1, p2);
    if (dist < minDist) {
      minDist = dist;
      nearestPt = q2;
    }
    return nearestPt
  };

  return RobustLineIntersector;
}(LineIntersector));

var RobustDeterminant = function RobustDeterminant () {};

RobustDeterminant.prototype.interfaces_ = function interfaces_ () {
  return []
};
RobustDeterminant.prototype.getClass = function getClass () {
  return RobustDeterminant
};
RobustDeterminant.orientationIndex = function orientationIndex (p1, p2, q) {
  var dx1 = p2.x - p1.x;
  var dy1 = p2.y - p1.y;
  var dx2 = q.x - p2.x;
  var dy2 = q.y - p2.y;
  return RobustDeterminant.signOfDet2x2(dx1, dy1, dx2, dy2)
};
RobustDeterminant.signOfDet2x2 = function signOfDet2x2 (x1, y1, x2, y2) {
  var sign = null;
  var swap = null;
  var k = null;
  sign = 1;
  if (x1 === 0.0 || y2 === 0.0) {
    if (y1 === 0.0 || x2 === 0.0) {
      return 0
    } else if (y1 > 0) {
      if (x2 > 0) {
        return -sign
      } else {
        return sign
      }
    } else {
      if (x2 > 0) {
        return sign
      } else {
        return -sign
      }
    }
  }
  if (y1 === 0.0 || x2 === 0.0) {
    if (y2 > 0) {
      if (x1 > 0) {
        return sign
      } else {
        return -sign
      }
    } else {
      if (x1 > 0) {
        return -sign
      } else {
        return sign
      }
    }
  }
  if (y1 > 0.0) {
    if (y2 > 0.0) {
      if (y1 <= y2) {
        
      } else {
        sign = -sign;
        swap = x1;
        x1 = x2;
        x2 = swap;
        swap = y1;
        y1 = y2;
        y2 = swap;
      }
    } else {
      if (y1 <= -y2) {
        sign = -sign;
        x2 = -x2;
        y2 = -y2;
      } else {
        swap = x1;
        x1 = -x2;
        x2 = swap;
        swap = y1;
        y1 = -y2;
        y2 = swap;
      }
    }
  } else {
    if (y2 > 0.0) {
      if (-y1 <= y2) {
        sign = -sign;
        x1 = -x1;
        y1 = -y1;
      } else {
        swap = -x1;
        x1 = x2;
        x2 = swap;
        swap = -y1;
        y1 = y2;
        y2 = swap;
      }
    } else {
      if (y1 >= y2) {
        x1 = -x1;
        y1 = -y1;
        x2 = -x2;
        y2 = -y2;
      } else {
        sign = -sign;
        swap = -x1;
        x1 = -x2;
        x2 = swap;
        swap = -y1;
        y1 = -y2;
        y2 = swap;
      }
    }
  }
  if (x1 > 0.0) {
    if (x2 > 0.0) {
      if (x1 <= x2) {
        
      } else {
        return sign
      }
    } else {
      return sign
    }
  } else {
    if (x2 > 0.0) {
      return -sign
    } else {
      if (x1 >= x2) {
        sign = -sign;
        x1 = -x1;
        x2 = -x2;
      } else {
        return -sign
      }
    }
  }
  while (true) {
    k = Math.floor(x2 / x1);
    x2 = x2 - k * x1;
    y2 = y2 - k * y1;
    if (y2 < 0.0) {
      return -sign
    }
    if (y2 > y1) {
      return sign
    }
    if (x1 > x2 + x2) {
      if (y1 < y2 + y2) {
        return sign
      }
    } else {
      if (y1 > y2 + y2) {
        return -sign
      } else {
        x2 = x1 - x2;
        y2 = y1 - y2;
        sign = -sign;
      }
    }
    if (y2 === 0.0) {
      if (x2 === 0.0) {
        return 0
      } else {
        return -sign
      }
    }
    if (x2 === 0.0) {
      return sign
    }
    k = Math.floor(x1 / x2);
    x1 = x1 - k * x2;
    y1 = y1 - k * y2;
    if (y1 < 0.0) {
      return sign
    }
    if (y1 > y2) {
      return -sign
    }
    if (x2 > x1 + x1) {
      if (y2 < y1 + y1) {
        return -sign
      }
    } else {
      if (y2 > y1 + y1) {
        return sign
      } else {
        x1 = x2 - x1;
        y1 = y2 - y1;
        sign = -sign;
      }
    }
    if (y1 === 0.0) {
      if (x1 === 0.0) {
        return 0
      } else {
        return sign
      }
    }
    if (x1 === 0.0) {
      return -sign
    }
  }
};

var RayCrossingCounter = function RayCrossingCounter () {
  this._p = null;
  this._crossingCount = 0;
  this._isPointOnSegment = false;
  var p = arguments[0];
  this._p = p;
};
RayCrossingCounter.prototype.countSegment = function countSegment (p1, p2) {
  if (p1.x < this._p.x && p2.x < this._p.x) { return null }
  if (this._p.x === p2.x && this._p.y === p2.y) {
    this._isPointOnSegment = true;
    return null
  }
  if (p1.y === this._p.y && p2.y === this._p.y) {
    var minx = p1.x;
    var maxx = p2.x;
    if (minx > maxx) {
      minx = p2.x;
      maxx = p1.x;
    }
    if (this._p.x >= minx && this._p.x <= maxx) {
      this._isPointOnSegment = true;
    }
    return null
  }
  if ((p1.y > this._p.y && p2.y <= this._p.y) || (p2.y > this._p.y && p1.y <= this._p.y)) {
    var x1 = p1.x - this._p.x;
    var y1 = p1.y - this._p.y;
    var x2 = p2.x - this._p.x;
    var y2 = p2.y - this._p.y;
    var xIntSign = RobustDeterminant.signOfDet2x2(x1, y1, x2, y2);
    if (xIntSign === 0.0) {
      this._isPointOnSegment = true;
      return null
    }
    if (y2 < y1) { xIntSign = -xIntSign; }
    if (xIntSign > 0.0) {
      this._crossingCount++;
    }
  }
};
RayCrossingCounter.prototype.isPointInPolygon = function isPointInPolygon () {
  return this.getLocation() !== Location.EXTERIOR
};
RayCrossingCounter.prototype.getLocation = function getLocation () {
  if (this._isPointOnSegment) { return Location.BOUNDARY }
  if (this._crossingCount % 2 === 1) {
    return Location.INTERIOR
  }
  return Location.EXTERIOR
};
RayCrossingCounter.prototype.isOnSegment = function isOnSegment () {
  return this._isPointOnSegment
};
RayCrossingCounter.prototype.interfaces_ = function interfaces_ () {
  return []
};
RayCrossingCounter.prototype.getClass = function getClass () {
  return RayCrossingCounter
};
RayCrossingCounter.locatePointInRing = function locatePointInRing () {
  if (arguments[0] instanceof Coordinate && hasInterface(arguments[1], CoordinateSequence)) {
    var p = arguments[0];
    var ring = arguments[1];
    var counter = new RayCrossingCounter(p);
    var p1 = new Coordinate();
    var p2 = new Coordinate();
    for (var i = 1; i < ring.size(); i++) {
      ring.getCoordinate(i, p1);
      ring.getCoordinate(i - 1, p2);
      counter.countSegment(p1, p2);
      if (counter.isOnSegment()) { return counter.getLocation() }
    }
    return counter.getLocation()
  } else if (arguments[0] instanceof Coordinate && arguments[1] instanceof Array) {
    var p$1 = arguments[0];
    var ring$1 = arguments[1];
    var counter$1 = new RayCrossingCounter(p$1);
    for (var i$1 = 1; i$1 < ring$1.length; i$1++) {
      var p1$1 = ring$1[i$1];
      var p2$1 = ring$1[i$1 - 1];
      counter$1.countSegment(p1$1, p2$1);
      if (counter$1.isOnSegment()) { return counter$1.getLocation() }
    }
    return counter$1.getLocation()
  }
};

var CGAlgorithms = function CGAlgorithms () {};

var staticAccessors$3 = { CLOCKWISE: { configurable: true },RIGHT: { configurable: true },COUNTERCLOCKWISE: { configurable: true },LEFT: { configurable: true },COLLINEAR: { configurable: true },STRAIGHT: { configurable: true } };

CGAlgorithms.prototype.interfaces_ = function interfaces_ () {
  return []
};
CGAlgorithms.prototype.getClass = function getClass () {
  return CGAlgorithms
};
CGAlgorithms.orientationIndex = function orientationIndex (p1, p2, q) {
  return CGAlgorithmsDD.orientationIndex(p1, p2, q)
};
CGAlgorithms.signedArea = function signedArea () {
  if (arguments[0] instanceof Array) {
    var ring = arguments[0];
    if (ring.length < 3) { return 0.0 }
    var sum = 0.0;
    var x0 = ring[0].x;
    for (var i = 1; i < ring.length - 1; i++) {
      var x = ring[i].x - x0;
      var y1 = ring[i + 1].y;
      var y2 = ring[i - 1].y;
      sum += x * (y2 - y1);
    }
    return sum / 2.0
  } else if (hasInterface(arguments[0], CoordinateSequence)) {
    var ring$1 = arguments[0];
    var n = ring$1.size();
    if (n < 3) { return 0.0 }
    var p0 = new Coordinate();
    var p1 = new Coordinate();
    var p2 = new Coordinate();
    ring$1.getCoordinate(0, p1);
    ring$1.getCoordinate(1, p2);
    var x0$1 = p1.x;
    p2.x -= x0$1;
    var sum$1 = 0.0;
    for (var i$1 = 1; i$1 < n - 1; i$1++) {
      p0.y = p1.y;
      p1.x = p2.x;
      p1.y = p2.y;
      ring$1.getCoordinate(i$1 + 1, p2);
      p2.x -= x0$1;
      sum$1 += p1.x * (p0.y - p2.y);
    }
    return sum$1 / 2.0
  }
};
CGAlgorithms.distanceLineLine = function distanceLineLine (A, B, C, D) {
  if (A.equals(B)) { return CGAlgorithms.distancePointLine(A, C, D) }
  if (C.equals(D)) { return CGAlgorithms.distancePointLine(D, A, B) }
  var noIntersection = false;
  if (!Envelope.intersects(A, B, C, D)) {
    noIntersection = true;
  } else {
    var denom = (B.x - A.x) * (D.y - C.y) - (B.y - A.y) * (D.x - C.x);
    if (denom === 0) {
      noIntersection = true;
    } else {
      var rNumb = (A.y - C.y) * (D.x - C.x) - (A.x - C.x) * (D.y - C.y);
      var sNum = (A.y - C.y) * (B.x - A.x) - (A.x - C.x) * (B.y - A.y);
      var s = sNum / denom;
      var r = rNumb / denom;
      if (r < 0 || r > 1 || s < 0 || s > 1) {
        noIntersection = true;
      }
    }
  }
  if (noIntersection) {
    return MathUtil.min(CGAlgorithms.distancePointLine(A, C, D), CGAlgorithms.distancePointLine(B, C, D), CGAlgorithms.distancePointLine(C, A, B), CGAlgorithms.distancePointLine(D, A, B))
  }
  return 0.0
};
CGAlgorithms.isPointInRing = function isPointInRing (p, ring) {
  return CGAlgorithms.locatePointInRing(p, ring) !== Location.EXTERIOR
};
CGAlgorithms.computeLength = function computeLength (pts) {
  var n = pts.size();
  if (n <= 1) { return 0.0 }
  var len = 0.0;
  var p = new Coordinate();
  pts.getCoordinate(0, p);
  var x0 = p.x;
  var y0 = p.y;
  for (var i = 1; i < n; i++) {
    pts.getCoordinate(i, p);
    var x1 = p.x;
    var y1 = p.y;
    var dx = x1 - x0;
    var dy = y1 - y0;
    len += Math.sqrt(dx * dx + dy * dy);
    x0 = x1;
    y0 = y1;
  }
  return len
};
CGAlgorithms.isCCW = function isCCW (ring) {
  var nPts = ring.length - 1;
  if (nPts < 3) { throw new IllegalArgumentException('Ring has fewer than 4 points, so orientation cannot be determined') }
  var hiPt = ring[0];
  var hiIndex = 0;
  for (var i = 1; i <= nPts; i++) {
    var p = ring[i];
    if (p.y > hiPt.y) {
      hiPt = p;
      hiIndex = i;
    }
  }
  var iPrev = hiIndex;
  do {
    iPrev = iPrev - 1;
    if (iPrev < 0) { iPrev = nPts; }
  } while (ring[iPrev].equals2D(hiPt) && iPrev !== hiIndex)
  var iNext = hiIndex;
  do {
    iNext = (iNext + 1) % nPts;
  } while (ring[iNext].equals2D(hiPt) && iNext !== hiIndex)
  var prev = ring[iPrev];
  var next = ring[iNext];
  if (prev.equals2D(hiPt) || next.equals2D(hiPt) || prev.equals2D(next)) { return false }
  var disc = CGAlgorithms.computeOrientation(prev, hiPt, next);
  var isCCW = false;
  if (disc === 0) {
    isCCW = prev.x > next.x;
  } else {
    isCCW = disc > 0;
  }
  return isCCW
};
CGAlgorithms.locatePointInRing = function locatePointInRing (p, ring) {
  return RayCrossingCounter.locatePointInRing(p, ring)
};
CGAlgorithms.distancePointLinePerpendicular = function distancePointLinePerpendicular (p, A, B) {
  var len2 = (B.x - A.x) * (B.x - A.x) + (B.y - A.y) * (B.y - A.y);
  var s = ((A.y - p.y) * (B.x - A.x) - (A.x - p.x) * (B.y - A.y)) / len2;
  return Math.abs(s) * Math.sqrt(len2)
};
CGAlgorithms.computeOrientation = function computeOrientation (p1, p2, q) {
  return CGAlgorithms.orientationIndex(p1, p2, q)
};
CGAlgorithms.distancePointLine = function distancePointLine () {
  if (arguments.length === 2) {
    var p = arguments[0];
    var line = arguments[1];
    if (line.length === 0) { throw new IllegalArgumentException('Line array must contain at least one vertex') }
    var minDistance = p.distance(line[0]);
    for (var i = 0; i < line.length - 1; i++) {
      var dist = CGAlgorithms.distancePointLine(p, line[i], line[i + 1]);
      if (dist < minDistance) {
        minDistance = dist;
      }
    }
    return minDistance
  } else if (arguments.length === 3) {
    var p$1 = arguments[0];
    var A = arguments[1];
    var B = arguments[2];
    if (A.x === B.x && A.y === B.y) { return p$1.distance(A) }
    var len2 = (B.x - A.x) * (B.x - A.x) + (B.y - A.y) * (B.y - A.y);
    var r = ((p$1.x - A.x) * (B.x - A.x) + (p$1.y - A.y) * (B.y - A.y)) / len2;
    if (r <= 0.0) { return p$1.distance(A) }
    if (r >= 1.0) { return p$1.distance(B) }
    var s = ((A.y - p$1.y) * (B.x - A.x) - (A.x - p$1.x) * (B.y - A.y)) / len2;
    return Math.abs(s) * Math.sqrt(len2)
  }
};
CGAlgorithms.isOnLine = function isOnLine (p, pt) {
  var lineIntersector = new RobustLineIntersector();
  for (var i = 1; i < pt.length; i++) {
    var p0 = pt[i - 1];
    var p1 = pt[i];
    lineIntersector.computeIntersection(p, p0, p1);
    if (lineIntersector.hasIntersection()) {
      return true
    }
  }
  return false
};
staticAccessors$3.CLOCKWISE.get = function () { return -1 };
staticAccessors$3.RIGHT.get = function () { return CGAlgorithms.CLOCKWISE };
staticAccessors$3.COUNTERCLOCKWISE.get = function () { return 1 };
staticAccessors$3.LEFT.get = function () { return CGAlgorithms.COUNTERCLOCKWISE };
staticAccessors$3.COLLINEAR.get = function () { return 0 };
staticAccessors$3.STRAIGHT.get = function () { return CGAlgorithms.COLLINEAR };

Object.defineProperties( CGAlgorithms, staticAccessors$3 );

var GeometryComponentFilter = function GeometryComponentFilter () {};

GeometryComponentFilter.prototype.filter = function filter (geom) {};
GeometryComponentFilter.prototype.interfaces_ = function interfaces_ () {
  return []
};
GeometryComponentFilter.prototype.getClass = function getClass () {
  return GeometryComponentFilter
};

var Geometry = function Geometry () {
  var factory = arguments[0];

  this._envelope = null;
  this._factory = null;
  this._SRID = null;
  this._userData = null;
  this._factory = factory;
  this._SRID = factory.getSRID();
};

var staticAccessors$11 = { serialVersionUID: { configurable: true },SORTINDEX_POINT: { configurable: true },SORTINDEX_MULTIPOINT: { configurable: true },SORTINDEX_LINESTRING: { configurable: true },SORTINDEX_LINEARRING: { configurable: true },SORTINDEX_MULTILINESTRING: { configurable: true },SORTINDEX_POLYGON: { configurable: true },SORTINDEX_MULTIPOLYGON: { configurable: true },SORTINDEX_GEOMETRYCOLLECTION: { configurable: true },geometryChangedFilter: { configurable: true } };
Geometry.prototype.isGeometryCollection = function isGeometryCollection () {
  return this.getSortIndex() === Geometry.SORTINDEX_GEOMETRYCOLLECTION
};
Geometry.prototype.getFactory = function getFactory () {
  return this._factory
};
Geometry.prototype.getGeometryN = function getGeometryN (n) {
  return this
};
Geometry.prototype.getArea = function getArea () {
  return 0.0
};
Geometry.prototype.isRectangle = function isRectangle () {
  return false
};
Geometry.prototype.equals = function equals () {
  if (arguments[0] instanceof Geometry) {
    var g$1 = arguments[0];
    if (g$1 === null) { return false }
    return this.equalsTopo(g$1)
  } else if (arguments[0] instanceof Object) {
    var o = arguments[0];
    if (!(o instanceof Geometry)) { return false }
    var g = o;
    return this.equalsExact(g)
  }
};
Geometry.prototype.equalsExact = function equalsExact (other) {
  return this === other || this.equalsExact(other, 0)
};
Geometry.prototype.geometryChanged = function geometryChanged () {
  this.apply(Geometry.geometryChangedFilter);
};
Geometry.prototype.geometryChangedAction = function geometryChangedAction () {
  this._envelope = null;
};
Geometry.prototype.equalsNorm = function equalsNorm (g) {
  if (g === null) { return false }
  return this.norm().equalsExact(g.norm())
};
Geometry.prototype.getLength = function getLength () {
  return 0.0
};
Geometry.prototype.getNumGeometries = function getNumGeometries () {
  return 1
};
Geometry.prototype.compareTo = function compareTo () {
  if (arguments.length === 1) {
    var o = arguments[0];
    var other = o;
    if (this.getSortIndex() !== other.getSortIndex()) {
      return this.getSortIndex() - other.getSortIndex()
    }
    if (this.isEmpty() && other.isEmpty()) {
      return 0
    }
    if (this.isEmpty()) {
      return -1
    }
    if (other.isEmpty()) {
      return 1
    }
    return this.compareToSameClass(o)
  } else if (arguments.length === 2) {
    var other$1 = arguments[0];
    var comp = arguments[1];
    if (this.getSortIndex() !== other$1.getSortIndex()) {
      return this.getSortIndex() - other$1.getSortIndex()
    }
    if (this.isEmpty() && other$1.isEmpty()) {
      return 0
    }
    if (this.isEmpty()) {
      return -1
    }
    if (other$1.isEmpty()) {
      return 1
    }
    return this.compareToSameClass(other$1, comp)
  }
};
Geometry.prototype.getUserData = function getUserData () {
  return this._userData
};
Geometry.prototype.getSRID = function getSRID () {
  return this._SRID
};
Geometry.prototype.getEnvelope = function getEnvelope () {
  return this.getFactory().toGeometry(this.getEnvelopeInternal())
};
Geometry.prototype.checkNotGeometryCollection = function checkNotGeometryCollection (g) {
  if (g.getSortIndex() === Geometry.SORTINDEX_GEOMETRYCOLLECTION) {
    throw new IllegalArgumentException('This method does not support GeometryCollection arguments')
  }
};
Geometry.prototype.equal = function equal (a, b, tolerance) {
  if (tolerance === 0) {
    return a.equals(b)
  }
  return a.distance(b) <= tolerance
};
Geometry.prototype.norm = function norm () {
  var copy = this.copy();
  copy.normalize();
  return copy
};
Geometry.prototype.getPrecisionModel = function getPrecisionModel () {
  return this._factory.getPrecisionModel()
};
Geometry.prototype.getEnvelopeInternal = function getEnvelopeInternal () {
  if (this._envelope === null) {
    this._envelope = this.computeEnvelopeInternal();
  }
  return new Envelope(this._envelope)
};
Geometry.prototype.setSRID = function setSRID (SRID) {
  this._SRID = SRID;
};
Geometry.prototype.setUserData = function setUserData (userData) {
  this._userData = userData;
};
Geometry.prototype.compare = function compare (a, b) {
  var i = a.iterator();
  var j = b.iterator();
  while (i.hasNext() && j.hasNext()) {
    var aElement = i.next();
    var bElement = j.next();
    var comparison = aElement.compareTo(bElement);
    if (comparison !== 0) {
      return comparison
    }
  }
  if (i.hasNext()) {
    return 1
  }
  if (j.hasNext()) {
    return -1
  }
  return 0
};
Geometry.prototype.hashCode = function hashCode () {
  return this.getEnvelopeInternal().hashCode()
};
Geometry.prototype.isGeometryCollectionOrDerived = function isGeometryCollectionOrDerived () {
  if (this.getSortIndex() === Geometry.SORTINDEX_GEOMETRYCOLLECTION || this.getSortIndex() === Geometry.SORTINDEX_MULTIPOINT || this.getSortIndex() === Geometry.SORTINDEX_MULTILINESTRING || this.getSortIndex() === Geometry.SORTINDEX_MULTIPOLYGON) {
    return true
  }
  return false
};
Geometry.prototype.interfaces_ = function interfaces_ () {
  return [Clonable, Comparable, Serializable]
};
Geometry.prototype.getClass = function getClass () {
  return Geometry
};
Geometry.hasNonEmptyElements = function hasNonEmptyElements (geometries) {
  for (var i = 0; i < geometries.length; i++) {
    if (!geometries[i].isEmpty()) {
      return true
    }
  }
  return false
};
Geometry.hasNullElements = function hasNullElements (array) {
  for (var i = 0; i < array.length; i++) {
    if (array[i] === null) {
      return true
    }
  }
  return false
};
staticAccessors$11.serialVersionUID.get = function () { return 8763622679187376702 };
staticAccessors$11.SORTINDEX_POINT.get = function () { return 0 };
staticAccessors$11.SORTINDEX_MULTIPOINT.get = function () { return 1 };
staticAccessors$11.SORTINDEX_LINESTRING.get = function () { return 2 };
staticAccessors$11.SORTINDEX_LINEARRING.get = function () { return 3 };
staticAccessors$11.SORTINDEX_MULTILINESTRING.get = function () { return 4 };
staticAccessors$11.SORTINDEX_POLYGON.get = function () { return 5 };
staticAccessors$11.SORTINDEX_MULTIPOLYGON.get = function () { return 6 };
staticAccessors$11.SORTINDEX_GEOMETRYCOLLECTION.get = function () { return 7 };
staticAccessors$11.geometryChangedFilter.get = function () { return geometryChangedFilter };

Object.defineProperties( Geometry, staticAccessors$11 );

var geometryChangedFilter = function geometryChangedFilter () {};

geometryChangedFilter.interfaces_ = function interfaces_ () {
  return [GeometryComponentFilter]
};
geometryChangedFilter.filter = function filter (geom) {
  geom.geometryChangedAction();
};

var CoordinateFilter = function CoordinateFilter () {};

CoordinateFilter.prototype.filter = function filter (coord) {};
CoordinateFilter.prototype.interfaces_ = function interfaces_ () {
  return []
};
CoordinateFilter.prototype.getClass = function getClass () {
  return CoordinateFilter
};

var BoundaryNodeRule = function BoundaryNodeRule () {};

var staticAccessors$12 = { Mod2BoundaryNodeRule: { configurable: true },EndPointBoundaryNodeRule: { configurable: true },MultiValentEndPointBoundaryNodeRule: { configurable: true },MonoValentEndPointBoundaryNodeRule: { configurable: true },MOD2_BOUNDARY_RULE: { configurable: true },ENDPOINT_BOUNDARY_RULE: { configurable: true },MULTIVALENT_ENDPOINT_BOUNDARY_RULE: { configurable: true },MONOVALENT_ENDPOINT_BOUNDARY_RULE: { configurable: true },OGC_SFS_BOUNDARY_RULE: { configurable: true } };

BoundaryNodeRule.prototype.isInBoundary = function isInBoundary (boundaryCount) {};
BoundaryNodeRule.prototype.interfaces_ = function interfaces_ () {
  return []
};
BoundaryNodeRule.prototype.getClass = function getClass () {
  return BoundaryNodeRule
};
staticAccessors$12.Mod2BoundaryNodeRule.get = function () { return Mod2BoundaryNodeRule };
staticAccessors$12.EndPointBoundaryNodeRule.get = function () { return EndPointBoundaryNodeRule };
staticAccessors$12.MultiValentEndPointBoundaryNodeRule.get = function () { return MultiValentEndPointBoundaryNodeRule };
staticAccessors$12.MonoValentEndPointBoundaryNodeRule.get = function () { return MonoValentEndPointBoundaryNodeRule };
staticAccessors$12.MOD2_BOUNDARY_RULE.get = function () { return new Mod2BoundaryNodeRule() };
staticAccessors$12.ENDPOINT_BOUNDARY_RULE.get = function () { return new EndPointBoundaryNodeRule() };
staticAccessors$12.MULTIVALENT_ENDPOINT_BOUNDARY_RULE.get = function () { return new MultiValentEndPointBoundaryNodeRule() };
staticAccessors$12.MONOVALENT_ENDPOINT_BOUNDARY_RULE.get = function () { return new MonoValentEndPointBoundaryNodeRule() };
staticAccessors$12.OGC_SFS_BOUNDARY_RULE.get = function () { return BoundaryNodeRule.MOD2_BOUNDARY_RULE };

Object.defineProperties( BoundaryNodeRule, staticAccessors$12 );

var Mod2BoundaryNodeRule = function Mod2BoundaryNodeRule () {};

Mod2BoundaryNodeRule.prototype.isInBoundary = function isInBoundary (boundaryCount) {
  return boundaryCount % 2 === 1
};
Mod2BoundaryNodeRule.prototype.interfaces_ = function interfaces_ () {
  return [BoundaryNodeRule]
};
Mod2BoundaryNodeRule.prototype.getClass = function getClass () {
  return Mod2BoundaryNodeRule
};

var EndPointBoundaryNodeRule = function EndPointBoundaryNodeRule () {};

EndPointBoundaryNodeRule.prototype.isInBoundary = function isInBoundary (boundaryCount) {
  return boundaryCount > 0
};
EndPointBoundaryNodeRule.prototype.interfaces_ = function interfaces_ () {
  return [BoundaryNodeRule]
};
EndPointBoundaryNodeRule.prototype.getClass = function getClass () {
  return EndPointBoundaryNodeRule
};

var MultiValentEndPointBoundaryNodeRule = function MultiValentEndPointBoundaryNodeRule () {};

MultiValentEndPointBoundaryNodeRule.prototype.isInBoundary = function isInBoundary (boundaryCount) {
  return boundaryCount > 1
};
MultiValentEndPointBoundaryNodeRule.prototype.interfaces_ = function interfaces_ () {
  return [BoundaryNodeRule]
};
MultiValentEndPointBoundaryNodeRule.prototype.getClass = function getClass () {
  return MultiValentEndPointBoundaryNodeRule
};

var MonoValentEndPointBoundaryNodeRule = function MonoValentEndPointBoundaryNodeRule () {};

MonoValentEndPointBoundaryNodeRule.prototype.isInBoundary = function isInBoundary (boundaryCount) {
  return boundaryCount === 1
};
MonoValentEndPointBoundaryNodeRule.prototype.interfaces_ = function interfaces_ () {
  return [BoundaryNodeRule]
};
MonoValentEndPointBoundaryNodeRule.prototype.getClass = function getClass () {
  return MonoValentEndPointBoundaryNodeRule
};

// import Iterator from './Iterator'

/**
 * @see http://download.oracle.com/javase/6/docs/api/java/util/Collection.html
 *
 * @constructor
 * @private
 */
var Collection = function Collection () {};

Collection.prototype.add = function add () {};

/**
 * Appends all of the elements in the specified collection to the end of this
 * list, in the order that they are returned by the specified collection's
 * iterator (optional operation).
 * @param {javascript.util.Collection} c
 * @return {boolean}
 */
Collection.prototype.addAll = function addAll () {};

/**
 * Returns true if this collection contains no elements.
 * @return {boolean}
 */
Collection.prototype.isEmpty = function isEmpty () {};

/**
 * Returns an iterator over the elements in this collection.
 * @return {javascript.util.Iterator}
 */
Collection.prototype.iterator = function iterator () {};

/**
 * Returns an iterator over the elements in this collection.
 * @return {number}
 */
Collection.prototype.size = function size () {};

/**
 * Returns an array containing all of the elements in this collection.
 * @return {Array}
 */
Collection.prototype.toArray = function toArray () {};

/**
 * Removes a single instance of the specified element from this collection if it
 * is present. (optional)
 * @param {Object} e
 * @return {boolean}
 */
Collection.prototype.remove = function remove () {};

/**
 * @param {string} [message] Optional message
 * @extends {Error}
 * @constructor
 * @private
 */
var IndexOutOfBoundsException = (function (Error) {
  function IndexOutOfBoundsException (message) {
    Error.call(this);
    this.message = message || '';
  }

  if ( Error ) IndexOutOfBoundsException.__proto__ = Error;
  IndexOutOfBoundsException.prototype = Object.create( Error && Error.prototype );
  IndexOutOfBoundsException.prototype.constructor = IndexOutOfBoundsException;

  var staticAccessors = { name: { configurable: true } };

  /**
   * @type {string}
   */
  staticAccessors.name.get = function () { return 'IndexOutOfBoundsException' };

  Object.defineProperties( IndexOutOfBoundsException, staticAccessors );

  return IndexOutOfBoundsException;
}(Error));

/**
 * @see http://download.oracle.com/javase/6/docs/api/java/util/Iterator.html
 * @constructor
 * @private
 */
var Iterator = function Iterator () {};

Iterator.prototype.hasNext = function hasNext () {};

/**
 * Returns the next element in the iteration.
 * @return {Object}
 */
Iterator.prototype.next = function next () {};

/**
 * Removes from the underlying collection the last element returned by the
 * iterator (optional operation).
 */
Iterator.prototype.remove = function remove () {};

/**
 * @see http://download.oracle.com/javase/6/docs/api/java/util/List.html
 *
 * @extends {javascript.util.Collection}
 * @constructor
 * @private
 */
var List = (function (Collection$$1) {
  function List () {
    Collection$$1.apply(this, arguments);
  }

  if ( Collection$$1 ) List.__proto__ = Collection$$1;
  List.prototype = Object.create( Collection$$1 && Collection$$1.prototype );
  List.prototype.constructor = List;

  List.prototype.get = function get () { };

  /**
   * Replaces the element at the specified position in this list with the
   * specified element (optional operation).
   * @param {number} index
   * @param {Object} e
   * @return {Object}
   */
  List.prototype.set = function set () { };

  /**
   * Returns true if this collection contains no elements.
   * @return {boolean}
   */
  List.prototype.isEmpty = function isEmpty () { };

  return List;
}(Collection));

/**
 * @param {string=} message Optional message
 * @extends {Error}
 * @constructor
 * @private
 */
function NoSuchElementException (message) {
  this.message = message || '';
}
NoSuchElementException.prototype = new Error();

/**
 * @type {string}
 */
NoSuchElementException.prototype.name = 'NoSuchElementException';

// import OperationNotSupported from './OperationNotSupported'

/**
 * @see http://download.oracle.com/javase/6/docs/api/java/util/ArrayList.html
 *
 * @extends List
 * @private
 */
var ArrayList = (function (List$$1) {
  function ArrayList () {
    List$$1.call(this);
    this.array_ = [];

    if (arguments[0] instanceof Collection) {
      this.addAll(arguments[0]);
    }
  }

  if ( List$$1 ) ArrayList.__proto__ = List$$1;
  ArrayList.prototype = Object.create( List$$1 && List$$1.prototype );
  ArrayList.prototype.constructor = ArrayList;

  ArrayList.prototype.ensureCapacity = function ensureCapacity () {};
  ArrayList.prototype.interfaces_ = function interfaces_ () { return [List$$1, Collection] };

  /**
   * @override
   */
  ArrayList.prototype.add = function add (e) {
    if (arguments.length === 1) {
      this.array_.push(e);
    } else {
      this.array_.splice(arguments[0], arguments[1]);
    }
    return true
  };

  ArrayList.prototype.clear = function clear () {
    this.array_ = [];
  };

  /**
   * @override
   */
  ArrayList.prototype.addAll = function addAll (c) {
    var this$1 = this;

    for (var i = c.iterator(); i.hasNext();) {
      this$1.add(i.next());
    }
    return true
  };

  /**
   * @override
   */
  ArrayList.prototype.set = function set (index, element) {
    var oldElement = this.array_[index];
    this.array_[index] = element;
    return oldElement
  };

  /**
   * @override
   */
  ArrayList.prototype.iterator = function iterator () {
    return new Iterator_(this)
  };

  /**
   * @override
   */
  ArrayList.prototype.get = function get (index) {
    if (index < 0 || index >= this.size()) {
      throw new IndexOutOfBoundsException()
    }

    return this.array_[index]
  };

  /**
   * @override
   */
  ArrayList.prototype.isEmpty = function isEmpty () {
    return this.array_.length === 0
  };

  /**
   * @override
   */
  ArrayList.prototype.size = function size () {
    return this.array_.length
  };

  /**
   * @override
   */
  ArrayList.prototype.toArray = function toArray () {
    var this$1 = this;

    var array = [];

    for (var i = 0, len = this.array_.length; i < len; i++) {
      array.push(this$1.array_[i]);
    }

    return array
  };

  /**
   * @override
   */
  ArrayList.prototype.remove = function remove (o) {
    var this$1 = this;

    var found = false;

    for (var i = 0, len = this.array_.length; i < len; i++) {
      if (this$1.array_[i] === o) {
        this$1.array_.splice(i, 1);
        found = true;
        break
      }
    }

    return found
  };

  return ArrayList;
}(List));

/**
 * @extends {Iterator}
 * @param {ArrayList} arrayList
 * @constructor
 * @private
 */
var Iterator_ = (function (Iterator$$1) {
  function Iterator_ (arrayList) {
    Iterator$$1.call(this);
    /**
     * @type {ArrayList}
     * @private
    */
    this.arrayList_ = arrayList;
    /**
     * @type {number}
     * @private
    */
    this.position_ = 0;
  }

  if ( Iterator$$1 ) Iterator_.__proto__ = Iterator$$1;
  Iterator_.prototype = Object.create( Iterator$$1 && Iterator$$1.prototype );
  Iterator_.prototype.constructor = Iterator_;

  /**
   * @override
   */
  Iterator_.prototype.next = function next () {
    if (this.position_ === this.arrayList_.size()) {
      throw new NoSuchElementException()
    }
    return this.arrayList_.get(this.position_++)
  };

  /**
   * @override
   */
  Iterator_.prototype.hasNext = function hasNext () {
    if (this.position_ < this.arrayList_.size()) {
      return true
    } else {
      return false
    }
  };

  /**
   * TODO: should be in ListIterator
   * @override
   */
  Iterator_.prototype.set = function set (element) {
    return this.arrayList_.set(this.position_ - 1, element)
  };

  /**
   * @override
   */
  Iterator_.prototype.remove = function remove () {
    this.arrayList_.remove(this.arrayList_.get(this.position_));
  };

  return Iterator_;
}(Iterator));

var CoordinateList = (function (ArrayList$$1) {
  function CoordinateList () {
    ArrayList$$1.call(this);
    if (arguments.length === 0) {
    } else if (arguments.length === 1) {
      var coord = arguments[0];
      this.ensureCapacity(coord.length);
      this.add(coord, true);
    } else if (arguments.length === 2) {
      var coord$1 = arguments[0];
      var allowRepeated = arguments[1];
      this.ensureCapacity(coord$1.length);
      this.add(coord$1, allowRepeated);
    }
  }

  if ( ArrayList$$1 ) CoordinateList.__proto__ = ArrayList$$1;
  CoordinateList.prototype = Object.create( ArrayList$$1 && ArrayList$$1.prototype );
  CoordinateList.prototype.constructor = CoordinateList;

  var staticAccessors = { coordArrayType: { configurable: true } };
  staticAccessors.coordArrayType.get = function () { return new Array(0).fill(null) };
  CoordinateList.prototype.getCoordinate = function getCoordinate (i) {
    return this.get(i)
  };
  CoordinateList.prototype.addAll = function addAll () {
    var this$1 = this;

    if (arguments.length === 2) {
      var coll = arguments[0];
      var allowRepeated = arguments[1];
      var isChanged = false;
      for (var i = coll.iterator(); i.hasNext();) {
        this$1.add(i.next(), allowRepeated);
        isChanged = true;
      }
      return isChanged
    } else { return ArrayList$$1.prototype.addAll.apply(this, arguments) }
  };
  CoordinateList.prototype.clone = function clone () {
    var this$1 = this;

    var clone = ArrayList$$1.prototype.clone.call(this);
    for (var i = 0; i < this.size(); i++) {
      clone.add(i, this$1.get(i).copy());
    }
    return clone
  };
  CoordinateList.prototype.toCoordinateArray = function toCoordinateArray () {
    return this.toArray(CoordinateList.coordArrayType)
  };
  CoordinateList.prototype.add = function add () {
    var this$1 = this;

    if (arguments.length === 1) {
      var coord = arguments[0];
      ArrayList$$1.prototype.add.call(this, coord);
    } else if (arguments.length === 2) {
      if (arguments[0] instanceof Array && typeof arguments[1] === 'boolean') {
        var coord$1 = arguments[0];
        var allowRepeated = arguments[1];
        this.add(coord$1, allowRepeated, true);
        return true
      } else if (arguments[0] instanceof Coordinate && typeof arguments[1] === 'boolean') {
        var coord$2 = arguments[0];
        var allowRepeated$1 = arguments[1];
        if (!allowRepeated$1) {
          if (this.size() >= 1) {
            var last = this.get(this.size() - 1);
            if (last.equals2D(coord$2)) { return null }
          }
        }
        ArrayList$$1.prototype.add.call(this, coord$2);
      } else if (arguments[0] instanceof Object && typeof arguments[1] === 'boolean') {
        var obj = arguments[0];
        var allowRepeated$2 = arguments[1];
        this.add(obj, allowRepeated$2);
        return true
      }
    } else if (arguments.length === 3) {
      if (typeof arguments[2] === 'boolean' && (arguments[0] instanceof Array && typeof arguments[1] === 'boolean')) {
        var coord$3 = arguments[0];
        var allowRepeated$3 = arguments[1];
        var direction = arguments[2];
        if (direction) {
          for (var i$1 = 0; i$1 < coord$3.length; i$1++) {
            this$1.add(coord$3[i$1], allowRepeated$3);
          }
        } else {
          for (var i$2 = coord$3.length - 1; i$2 >= 0; i$2--) {
            this$1.add(coord$3[i$2], allowRepeated$3);
          }
        }
        return true
      } else if (typeof arguments[2] === 'boolean' && (Number.isInteger(arguments[0]) && arguments[1] instanceof Coordinate)) {
        var i$3 = arguments[0];
        var coord$4 = arguments[1];
        var allowRepeated$4 = arguments[2];
        if (!allowRepeated$4) {
          var size = this.size();
          if (size > 0) {
            if (i$3 > 0) {
              var prev = this.get(i$3 - 1);
              if (prev.equals2D(coord$4)) { return null }
            }
            if (i$3 < size) {
              var next = this.get(i$3);
              if (next.equals2D(coord$4)) { return null }
            }
          }
        }
        ArrayList$$1.prototype.add.call(this, i$3, coord$4);
      }
    } else if (arguments.length === 4) {
      var coord$5 = arguments[0];
      var allowRepeated$5 = arguments[1];
      var start = arguments[2];
      var end = arguments[3];
      var inc = 1;
      if (start > end) { inc = -1; }
      for (var i = start; i !== end; i += inc) {
        this$1.add(coord$5[i], allowRepeated$5);
      }
      return true
    }
  };
  CoordinateList.prototype.closeRing = function closeRing () {
    if (this.size() > 0) { this.add(new Coordinate(this.get(0)), false); }
  };
  CoordinateList.prototype.interfaces_ = function interfaces_ () {
    return []
  };
  CoordinateList.prototype.getClass = function getClass () {
    return CoordinateList
  };

  Object.defineProperties( CoordinateList, staticAccessors );

  return CoordinateList;
}(ArrayList));

var CoordinateArrays = function CoordinateArrays () {};

var staticAccessors$13 = { ForwardComparator: { configurable: true },BidirectionalComparator: { configurable: true },coordArrayType: { configurable: true } };

staticAccessors$13.ForwardComparator.get = function () { return ForwardComparator };
staticAccessors$13.BidirectionalComparator.get = function () { return BidirectionalComparator };
staticAccessors$13.coordArrayType.get = function () { return new Array(0).fill(null) };

CoordinateArrays.prototype.interfaces_ = function interfaces_ () {
  return []
};
CoordinateArrays.prototype.getClass = function getClass () {
  return CoordinateArrays
};
CoordinateArrays.isRing = function isRing (pts) {
  if (pts.length < 4) { return false }
  if (!pts[0].equals2D(pts[pts.length - 1])) { return false }
  return true
};
CoordinateArrays.ptNotInList = function ptNotInList (testPts, pts) {
  for (var i = 0; i < testPts.length; i++) {
    var testPt = testPts[i];
    if (CoordinateArrays.indexOf(testPt, pts) < 0) { return testPt }
  }
  return null
};
CoordinateArrays.scroll = function scroll (coordinates, firstCoordinate) {
  var i = CoordinateArrays.indexOf(firstCoordinate, coordinates);
  if (i < 0) { return null }
  var newCoordinates = new Array(coordinates.length).fill(null);
  System.arraycopy(coordinates, i, newCoordinates, 0, coordinates.length - i);
  System.arraycopy(coordinates, 0, newCoordinates, coordinates.length - i, i);
  System.arraycopy(newCoordinates, 0, coordinates, 0, coordinates.length);
};
CoordinateArrays.equals = function equals () {
  if (arguments.length === 2) {
    var coord1 = arguments[0];
    var coord2 = arguments[1];
    if (coord1 === coord2) { return true }
    if (coord1 === null || coord2 === null) { return false }
    if (coord1.length !== coord2.length) { return false }
    for (var i = 0; i < coord1.length; i++) {
      if (!coord1[i].equals(coord2[i])) { return false }
    }
    return true
  } else if (arguments.length === 3) {
    var coord1$1 = arguments[0];
    var coord2$1 = arguments[1];
    var coordinateComparator = arguments[2];
    if (coord1$1 === coord2$1) { return true }
    if (coord1$1 === null || coord2$1 === null) { return false }
    if (coord1$1.length !== coord2$1.length) { return false }
    for (var i$1 = 0; i$1 < coord1$1.length; i$1++) {
      if (coordinateComparator.compare(coord1$1[i$1], coord2$1[i$1]) !== 0) { return false }
    }
    return true
  }
};
CoordinateArrays.intersection = function intersection (coordinates, env) {
  var coordList = new CoordinateList();
  for (var i = 0; i < coordinates.length; i++) {
    if (env.intersects(coordinates[i])) { coordList.add(coordinates[i], true); }
  }
  return coordList.toCoordinateArray()
};
CoordinateArrays.hasRepeatedPoints = function hasRepeatedPoints (coord) {
  for (var i = 1; i < coord.length; i++) {
    if (coord[i - 1].equals(coord[i])) {
      return true
    }
  }
  return false
};
CoordinateArrays.removeRepeatedPoints = function removeRepeatedPoints (coord) {
  if (!CoordinateArrays.hasRepeatedPoints(coord)) { return coord }
  var coordList = new CoordinateList(coord, false);
  return coordList.toCoordinateArray()
};
CoordinateArrays.reverse = function reverse (coord) {
  var last = coord.length - 1;
  var mid = Math.trunc(last / 2);
  for (var i = 0; i <= mid; i++) {
    var tmp = coord[i];
    coord[i] = coord[last - i];
    coord[last - i] = tmp;
  }
};
CoordinateArrays.removeNull = function removeNull (coord) {
  var nonNull = 0;
  for (var i = 0; i < coord.length; i++) {
    if (coord[i] !== null) { nonNull++; }
  }
  var newCoord = new Array(nonNull).fill(null);
  if (nonNull === 0) { return newCoord }
  var j = 0;
  for (var i$1 = 0; i$1 < coord.length; i$1++) {
    if (coord[i$1] !== null) { newCoord[j++] = coord[i$1]; }
  }
  return newCoord
};
CoordinateArrays.copyDeep = function copyDeep () {
  if (arguments.length === 1) {
    var coordinates = arguments[0];
    var copy = new Array(coordinates.length).fill(null);
    for (var i = 0; i < coordinates.length; i++) {
      copy[i] = new Coordinate(coordinates[i]);
    }
    return copy
  } else if (arguments.length === 5) {
    var src = arguments[0];
    var srcStart = arguments[1];
    var dest = arguments[2];
    var destStart = arguments[3];
    var length = arguments[4];
    for (var i$1 = 0; i$1 < length; i$1++) {
      dest[destStart + i$1] = new Coordinate(src[srcStart + i$1]);
    }
  }
};
CoordinateArrays.isEqualReversed = function isEqualReversed (pts1, pts2) {
  for (var i = 0; i < pts1.length; i++) {
    var p1 = pts1[i];
    var p2 = pts2[pts1.length - i - 1];
    if (p1.compareTo(p2) !== 0) { return false }
  }
  return true
};
CoordinateArrays.envelope = function envelope (coordinates) {
  var env = new Envelope();
  for (var i = 0; i < coordinates.length; i++) {
    env.expandToInclude(coordinates[i]);
  }
  return env
};
CoordinateArrays.toCoordinateArray = function toCoordinateArray (coordList) {
  return coordList.toArray(CoordinateArrays.coordArrayType)
};
CoordinateArrays.atLeastNCoordinatesOrNothing = function atLeastNCoordinatesOrNothing (n, c) {
  return c.length >= n ? c : []
};
CoordinateArrays.indexOf = function indexOf (coordinate, coordinates) {
  for (var i = 0; i < coordinates.length; i++) {
    if (coordinate.equals(coordinates[i])) {
      return i
    }
  }
  return -1
};
CoordinateArrays.increasingDirection = function increasingDirection (pts) {
  for (var i = 0; i < Math.trunc(pts.length / 2); i++) {
    var j = pts.length - 1 - i;
    var comp = pts[i].compareTo(pts[j]);
    if (comp !== 0) { return comp }
  }
  return 1
};
CoordinateArrays.compare = function compare (pts1, pts2) {
  var i = 0;
  while (i < pts1.length && i < pts2.length) {
    var compare = pts1[i].compareTo(pts2[i]);
    if (compare !== 0) { return compare }
    i++;
  }
  if (i < pts2.length) { return -1 }
  if (i < pts1.length) { return 1 }
  return 0
};
CoordinateArrays.minCoordinate = function minCoordinate (coordinates) {
  var minCoord = null;
  for (var i = 0; i < coordinates.length; i++) {
    if (minCoord === null || minCoord.compareTo(coordinates[i]) > 0) {
      minCoord = coordinates[i];
    }
  }
  return minCoord
};
CoordinateArrays.extract = function extract (pts, start, end) {
  start = MathUtil.clamp(start, 0, pts.length);
  end = MathUtil.clamp(end, -1, pts.length);
  var npts = end - start + 1;
  if (end < 0) { npts = 0; }
  if (start >= pts.length) { npts = 0; }
  if (end < start) { npts = 0; }
  var extractPts = new Array(npts).fill(null);
  if (npts === 0) { return extractPts }
  var iPts = 0;
  for (var i = start; i <= end; i++) {
    extractPts[iPts++] = pts[i];
  }
  return extractPts
};

Object.defineProperties( CoordinateArrays, staticAccessors$13 );

var ForwardComparator = function ForwardComparator () {};

ForwardComparator.prototype.compare = function compare (o1, o2) {
  var pts1 = o1;
  var pts2 = o2;
  return CoordinateArrays.compare(pts1, pts2)
};
ForwardComparator.prototype.interfaces_ = function interfaces_ () {
  return [Comparator]
};
ForwardComparator.prototype.getClass = function getClass () {
  return ForwardComparator
};

var BidirectionalComparator = function BidirectionalComparator () {};

BidirectionalComparator.prototype.compare = function compare (o1, o2) {
  var pts1 = o1;
  var pts2 = o2;
  if (pts1.length < pts2.length) { return -1 }
  if (pts1.length > pts2.length) { return 1 }
  if (pts1.length === 0) { return 0 }
  var forwardComp = CoordinateArrays.compare(pts1, pts2);
  var isEqualRev = CoordinateArrays.isEqualReversed(pts1, pts2);
  if (isEqualRev) { return 0 }
  return forwardComp
};
BidirectionalComparator.prototype.OLDcompare = function OLDcompare (o1, o2) {
  var pts1 = o1;
  var pts2 = o2;
  if (pts1.length < pts2.length) { return -1 }
  if (pts1.length > pts2.length) { return 1 }
  if (pts1.length === 0) { return 0 }
  var dir1 = CoordinateArrays.increasingDirection(pts1);
  var dir2 = CoordinateArrays.increasingDirection(pts2);
  var i1 = dir1 > 0 ? 0 : pts1.length - 1;
  var i2 = dir2 > 0 ? 0 : pts1.length - 1;
  for (var i = 0; i < pts1.length; i++) {
    var comparePt = pts1[i1].compareTo(pts2[i2]);
    if (comparePt !== 0) { return comparePt }
    i1 += dir1;
    i2 += dir2;
  }
  return 0
};
BidirectionalComparator.prototype.interfaces_ = function interfaces_ () {
  return [Comparator]
};
BidirectionalComparator.prototype.getClass = function getClass () {
  return BidirectionalComparator
};

/**
 * @see http://download.oracle.com/javase/6/docs/api/java/util/Map.html
 *
 * @constructor
 * @private
 */
var Map$1 = function Map () {};

Map$1.prototype.get = function get () {};
/**
 * Associates the specified value with the specified key in this map (optional
 * operation).
 * @param {Object} key
 * @param {Object} value
 * @return {Object}
 */
Map$1.prototype.put = function put () {};

/**
 * Returns the number of key-value mappings in this map.
 * @return {number}
 */
Map$1.prototype.size = function size () {};

/**
 * Returns a Collection view of the values contained in this map.
 * @return {javascript.util.Collection}
 */
Map$1.prototype.values = function values () {};

/**
 * Returns a {@link Set} view of the mappings contained in this map.
 * The set is backed by the map, so changes to the map are
 * reflected in the set, and vice-versa.If the map is modified
 * while an iteration over the set is in progress (except through
 * the iterator's own <tt>remove</tt> operation, or through the
 * <tt>setValue</tt> operation on a map entry returned by the
 * iterator) the results of the iteration are undefined.The set
 * supports element removal, which removes the corresponding
 * mapping from the map, via the <tt>Iterator.remove</tt>,
 * <tt>Set.remove</tt>, <tt>removeAll</tt>, <tt>retainAll</tt> and
 * <tt>clear</tt> operations.It does not support the
 * <tt>add</tt> or <tt>addAll</tt> operations.
 *
 * @return {Set} a set view of the mappings contained in this map
 */
Map$1.prototype.entrySet = function entrySet () {};

/**
 * @see http://download.oracle.com/javase/6/docs/api/java/util/SortedMap.html
 *
 * @extends {Map}
 * @constructor
 * @private
 */
var SortedMap = (function (Map) {
	function SortedMap () {
		Map.apply(this, arguments);
	}if ( Map ) SortedMap.__proto__ = Map;
	SortedMap.prototype = Object.create( Map && Map.prototype );
	SortedMap.prototype.constructor = SortedMap;

	

	return SortedMap;
}(Map$1));

/**
 * @param {string=} message Optional message
 * @extends {Error}
 * @constructor
 * @private
 */
function OperationNotSupported (message) {
  this.message = message || '';
}
OperationNotSupported.prototype = new Error();

/**
 * @type {string}
 */
OperationNotSupported.prototype.name = 'OperationNotSupported';

/**
 * @see http://download.oracle.com/javase/6/docs/api/java/util/Set.html
 *
 * @extends {Collection}
 * @constructor
 * @private
 */
function Set() {}
Set.prototype = new Collection();


/**
 * Returns true if this set contains the specified element. More formally,
 * returns true if and only if this set contains an element e such that (o==null ?
 * e==null : o.equals(e)).
 * @param {Object} e
 * @return {boolean}
 */
Set.prototype.contains = function() {};

/**
 * @see http://docs.oracle.com/javase/6/docs/api/java/util/HashSet.html
 *
 * @extends {javascript.util.Set}
 * @constructor
 * @private
 */
var HashSet = (function (Set$$1) {
  function HashSet () {
    Set$$1.call(this);
    this.array_ = [];

    if (arguments[0] instanceof Collection) {
      this.addAll(arguments[0]);
    }
  }

  if ( Set$$1 ) HashSet.__proto__ = Set$$1;
  HashSet.prototype = Object.create( Set$$1 && Set$$1.prototype );
  HashSet.prototype.constructor = HashSet;

  /**
   * @override
   */
  HashSet.prototype.contains = function contains (o) {
    var this$1 = this;

    for (var i = 0, len = this.array_.length; i < len; i++) {
      var e = this$1.array_[i];
      if (e === o) {
        return true
      }
    }
    return false
  };

  /**
   * @override
   */
  HashSet.prototype.add = function add (o) {
    if (this.contains(o)) {
      return false
    }

    this.array_.push(o);

    return true
  };

  /**
   * @override
   */
  HashSet.prototype.addAll = function addAll (c) {
    var this$1 = this;

    for (var i = c.iterator(); i.hasNext();) {
      this$1.add(i.next());
    }
    return true
  };

  /**
   * @override
   */
  HashSet.prototype.remove = function remove (o) {
    // throw new javascript.util.OperationNotSupported()
    throw new Error()
  };

  /**
   * @override
   */
  HashSet.prototype.size = function size () {
    return this.array_.length
  };

  /**
   * @override
   */
  HashSet.prototype.isEmpty = function isEmpty () {
    return this.array_.length === 0
  };

  /**
   * @override
   */
  HashSet.prototype.toArray = function toArray () {
    var this$1 = this;

    var array = [];

    for (var i = 0, len = this.array_.length; i < len; i++) {
      array.push(this$1.array_[i]);
    }

    return array
  };

  /**
   * @override
   */
  HashSet.prototype.iterator = function iterator () {
    return new Iterator_$1(this)
  };

  return HashSet;
}(Set));

/**
   * @extends {Iterator}
   * @param {HashSet} hashSet
   * @constructor
   * @private
   */
var Iterator_$1 = (function (Iterator$$1) {
  function Iterator_ (hashSet) {
    Iterator$$1.call(this);
    /**
     * @type {HashSet}
     * @private
     */
    this.hashSet_ = hashSet;
    /**
     * @type {number}
     * @private
     */
    this.position_ = 0;
  }

  if ( Iterator$$1 ) Iterator_.__proto__ = Iterator$$1;
  Iterator_.prototype = Object.create( Iterator$$1 && Iterator$$1.prototype );
  Iterator_.prototype.constructor = Iterator_;

  /**
   * @override
   */
  Iterator_.prototype.next = function next () {
    if (this.position_ === this.hashSet_.size()) {
      throw new NoSuchElementException()
    }
    return this.hashSet_.array_[this.position_++]
  };

  /**
   * @override
   */
  Iterator_.prototype.hasNext = function hasNext () {
    if (this.position_ < this.hashSet_.size()) {
      return true
    } else {
      return false
    }
  };

  /**
   * @override
   */
  Iterator_.prototype.remove = function remove () {
    throw new OperationNotSupported()
  };

  return Iterator_;
}(Iterator));

var BLACK = 0;
var RED = 1;
function colorOf (p) { return (p === null ? BLACK : p.color) }
function parentOf (p) { return (p === null ? null : p.parent) }
function setColor (p, c) { if (p !== null) { p.color = c; } }
function leftOf (p) { return (p === null ? null : p.left) }
function rightOf (p) { return (p === null ? null : p.right) }

/**
 * @see http://download.oracle.com/javase/6/docs/api/java/util/TreeMap.html
 *
 * @extends {SortedMap}
 * @constructor
 * @private
 */
function TreeMap () {
  /**
   * @type {Object}
   * @private
   */
  this.root_ = null;
  /**
   * @type {number}
   * @private
  */
  this.size_ = 0;
}
TreeMap.prototype = new SortedMap();

/**
 * @override
 */
TreeMap.prototype.get = function (key) {
  var p = this.root_;
  while (p !== null) {
    var cmp = key['compareTo'](p.key);
    if (cmp < 0) { p = p.left; }
    else if (cmp > 0) { p = p.right; }
    else { return p.value }
  }
  return null
};

/**
 * @override
 */
TreeMap.prototype.put = function (key, value) {
  if (this.root_ === null) {
    this.root_ = {
      key: key,
      value: value,
      left: null,
      right: null,
      parent: null,
      color: BLACK,
      getValue: function getValue () { return this.value },
      getKey: function getKey () { return this.key }
    };
    this.size_ = 1;
    return null
  }
  var t = this.root_;
  var parent;
  var cmp;
  do {
    parent = t;
    cmp = key['compareTo'](t.key);
    if (cmp < 0) {
      t = t.left;
    } else if (cmp > 0) {
      t = t.right;
    } else {
      var oldValue = t.value;
      t.value = value;
      return oldValue
    }
  } while (t !== null)
  var e = {
    key: key,
    left: null,
    right: null,
    value: value,
    parent: parent,
    color: BLACK,
    getValue: function getValue () { return this.value },
    getKey: function getKey () { return this.key }
  };
  if (cmp < 0) {
    parent.left = e;
  } else {
    parent.right = e;
  }
  this.fixAfterInsertion(e);
  this.size_++;
  return null
};

/**
 * @param {Object} x
 */
TreeMap.prototype.fixAfterInsertion = function (x) {
  var this$1 = this;

  x.color = RED;
  while (x != null && x !== this.root_ && x.parent.color === RED) {
    if (parentOf(x) === leftOf(parentOf(parentOf(x)))) {
      var y = rightOf(parentOf(parentOf(x)));
      if (colorOf(y) === RED) {
        setColor(parentOf(x), BLACK);
        setColor(y, BLACK);
        setColor(parentOf(parentOf(x)), RED);
        x = parentOf(parentOf(x));
      } else {
        if (x === rightOf(parentOf(x))) {
          x = parentOf(x);
          this$1.rotateLeft(x);
        }
        setColor(parentOf(x), BLACK);
        setColor(parentOf(parentOf(x)), RED);
        this$1.rotateRight(parentOf(parentOf(x)));
      }
    } else {
      var y$1 = leftOf(parentOf(parentOf(x)));
      if (colorOf(y$1) === RED) {
        setColor(parentOf(x), BLACK);
        setColor(y$1, BLACK);
        setColor(parentOf(parentOf(x)), RED);
        x = parentOf(parentOf(x));
      } else {
        if (x === leftOf(parentOf(x))) {
          x = parentOf(x);
          this$1.rotateRight(x);
        }
        setColor(parentOf(x), BLACK);
        setColor(parentOf(parentOf(x)), RED);
        this$1.rotateLeft(parentOf(parentOf(x)));
      }
    }
  }
  this.root_.color = BLACK;
};

/**
 * @override
 */
TreeMap.prototype.values = function () {
  var arrayList = new ArrayList();
  var p = this.getFirstEntry();
  if (p !== null) {
    arrayList.add(p.value);
    while ((p = TreeMap.successor(p)) !== null) {
      arrayList.add(p.value);
    }
  }
  return arrayList
};

/**
 * @override
 */
TreeMap.prototype.entrySet = function () {
  var hashSet = new HashSet();
  var p = this.getFirstEntry();
  if (p !== null) {
    hashSet.add(p);
    while ((p = TreeMap.successor(p)) !== null) {
      hashSet.add(p);
    }
  }
  return hashSet
};

/**
 * @param {Object} p
 */
TreeMap.prototype.rotateLeft = function (p) {
  if (p != null) {
    var r = p.right;
    p.right = r.left;
    if (r.left != null) { r.left.parent = p; }
    r.parent = p.parent;
    if (p.parent === null) { this.root_ = r; } else if (p.parent.left === p) { p.parent.left = r; } else { p.parent.right = r; }
    r.left = p;
    p.parent = r;
  }
};

/**
 * @param {Object} p
 */
TreeMap.prototype.rotateRight = function (p) {
  if (p != null) {
    var l = p.left;
    p.left = l.right;
    if (l.right != null) { l.right.parent = p; }
    l.parent = p.parent;
    if (p.parent === null) { this.root_ = l; } else if (p.parent.right === p) { p.parent.right = l; } else { p.parent.left = l; }
    l.right = p;
    p.parent = l;
  }
};

/**
 * @return {Object}
 */
TreeMap.prototype.getFirstEntry = function () {
  var p = this.root_;
  if (p != null) {
    while (p.left != null) {
      p = p.left;
    }
  }
  return p
};

/**
 * @param {Object} t
 * @return {Object}
 * @private
 */
TreeMap.successor = function (t) {
  if (t === null) { return null } else if (t.right !== null) {
    var p = t.right;
    while (p.left !== null) {
      p = p.left;
    }
    return p
  } else {
    var p$1 = t.parent;
    var ch = t;
    while (p$1 !== null && ch === p$1.right) {
      ch = p$1;
      p$1 = p$1.parent;
    }
    return p$1
  }
};

/**
 * @override
 */
TreeMap.prototype.size = function () {
  return this.size_
};

var Lineal = function Lineal () {};

Lineal.prototype.interfaces_ = function interfaces_ () {
  return []
};
Lineal.prototype.getClass = function getClass () {
  return Lineal
};

/**
 * @see http://download.oracle.com/javase/6/docs/api/java/util/SortedSet.html
 *
 * @extends {Set}
 * @constructor
 * @private
 */
function SortedSet () {}
SortedSet.prototype = new Set();

// import Iterator from './Iterator'
/**
 * @see http://download.oracle.com/javase/6/docs/api/java/util/TreeSet.html
 *
 * @extends {SortedSet}
 * @constructor
 * @private
 */
function TreeSet () {
  /**
   * @type {Array}
   * @private
  */
  this.array_ = [];

  if (arguments[0] instanceof Collection) {
    this.addAll(arguments[0]);
  }
}
TreeSet.prototype = new SortedSet();

/**
 * @override
 */
TreeSet.prototype.contains = function (o) {
  var this$1 = this;

  for (var i = 0, len = this.array_.length; i < len; i++) {
    var e = this$1.array_[i];
    if (e['compareTo'](o) === 0) {
      return true
    }
  }
  return false
};

/**
 * @override
 */
TreeSet.prototype.add = function (o) {
  var this$1 = this;

  if (this.contains(o)) {
    return false
  }

  for (var i = 0, len = this.array_.length; i < len; i++) {
    var e = this$1.array_[i];
    if (e['compareTo'](o) === 1) {
      this$1.array_.splice(i, 0, o);
      return true
    }
  }

  this.array_.push(o);

  return true
};

/**
 * @override
 */
TreeSet.prototype.addAll = function (c) {
  var this$1 = this;

  for (var i = c.iterator(); i.hasNext();) {
    this$1.add(i.next());
  }
  return true
};

/**
 * @override
 */
TreeSet.prototype.remove = function (e) {
  throw new OperationNotSupported()
};

/**
 * @override
 */
TreeSet.prototype.size = function () {
  return this.array_.length
};

/**
 * @override
 */
TreeSet.prototype.isEmpty = function () {
  return this.array_.length === 0
};

/**
 * @override
 */
TreeSet.prototype.toArray = function () {
  var this$1 = this;

  var array = [];

  for (var i = 0, len = this.array_.length; i < len; i++) {
    array.push(this$1.array_[i]);
  }

  return array
};

/**
 * @override
 */
TreeSet.prototype.iterator = function () {
  return new Iterator_$2(this)
};

/**
 * @extends {javascript.util.Iterator}
 * @param {javascript.util.TreeSet} treeSet
 * @constructor
 * @private
 */
var Iterator_$2 = function (treeSet) {
  /**
   * @type {javascript.util.TreeSet}
   * @private
   */
  this.treeSet_ = treeSet;
  /**
   * @type {number}
   * @private
   */
  this.position_ = 0;
};

/**
 * @override
 */
Iterator_$2.prototype.next = function () {
  if (this.position_ === this.treeSet_.size()) {
    throw new NoSuchElementException()
  }
  return this.treeSet_.array_[this.position_++]
};

/**
 * @override
 */
Iterator_$2.prototype.hasNext = function () {
  if (this.position_ < this.treeSet_.size()) {
    return true
  } else {
    return false
  }
};

/**
 * @override
 */
Iterator_$2.prototype.remove = function () {
  throw new OperationNotSupported()
};

/**
 * @see http://download.oracle.com/javase/6/docs/api/java/util/Arrays.html
 *
 * @constructor
 * @private
 */
var Arrays = function Arrays () {};

Arrays.sort = function sort () {
  var a = arguments[0];
  var i;
  var t;
  var comparator;
  var compare;
  if (arguments.length === 1) {
    compare = function (a, b) {
      return a.compareTo(b)
    };
    a.sort(compare);
  } else if (arguments.length === 2) {
    comparator = arguments[1];
    compare = function (a, b) {
      return comparator['compare'](a, b)
    };
    a.sort(compare);
  } else if (arguments.length === 3) {
    t = a.slice(arguments[1], arguments[2]);
    t.sort();
    var r = a.slice(0, arguments[1]).concat(t, a.slice(arguments[2], a.length));
    a.splice(0, a.length);
    for (i = 0; i < r.length; i++) {
      a.push(r[i]);
    }
  } else if (arguments.length === 4) {
    t = a.slice(arguments[1], arguments[2]);
    comparator = arguments[3];
    compare = function (a, b) {
      return comparator['compare'](a, b)
    };
    t.sort(compare);
    r = a.slice(0, arguments[1]).concat(t, a.slice(arguments[2], a.length));
    a.splice(0, a.length);
    for (i = 0; i < r.length; i++) {
      a.push(r[i]);
    }
  }
};
/**
 * @param {Array} array
 * @return {ArrayList}
 */
Arrays.asList = function asList (array) {
  var arrayList = new ArrayList();
  for (var i = 0, len = array.length; i < len; i++) {
    arrayList.add(array[i]);
  }
  return arrayList
};

var Dimension = function Dimension () {};

var staticAccessors$14 = { P: { configurable: true },L: { configurable: true },A: { configurable: true },FALSE: { configurable: true },TRUE: { configurable: true },DONTCARE: { configurable: true },SYM_FALSE: { configurable: true },SYM_TRUE: { configurable: true },SYM_DONTCARE: { configurable: true },SYM_P: { configurable: true },SYM_L: { configurable: true },SYM_A: { configurable: true } };

staticAccessors$14.P.get = function () { return 0 };
staticAccessors$14.L.get = function () { return 1 };
staticAccessors$14.A.get = function () { return 2 };
staticAccessors$14.FALSE.get = function () { return -1 };
staticAccessors$14.TRUE.get = function () { return -2 };
staticAccessors$14.DONTCARE.get = function () { return -3 };
staticAccessors$14.SYM_FALSE.get = function () { return 'F' };
staticAccessors$14.SYM_TRUE.get = function () { return 'T' };
staticAccessors$14.SYM_DONTCARE.get = function () { return '*' };
staticAccessors$14.SYM_P.get = function () { return '0' };
staticAccessors$14.SYM_L.get = function () { return '1' };
staticAccessors$14.SYM_A.get = function () { return '2' };

Dimension.prototype.interfaces_ = function interfaces_ () {
  return []
};
Dimension.prototype.getClass = function getClass () {
  return Dimension
};
Dimension.toDimensionSymbol = function toDimensionSymbol (dimensionValue) {
  switch (dimensionValue) {
    case Dimension.FALSE:
      return Dimension.SYM_FALSE
    case Dimension.TRUE:
      return Dimension.SYM_TRUE
    case Dimension.DONTCARE:
      return Dimension.SYM_DONTCARE
    case Dimension.P:
      return Dimension.SYM_P
    case Dimension.L:
      return Dimension.SYM_L
    case Dimension.A:
      return Dimension.SYM_A
    default:
  }
  throw new IllegalArgumentException('Unknown dimension value: ' + dimensionValue)
};
Dimension.toDimensionValue = function toDimensionValue (dimensionSymbol) {
  switch (Character.toUpperCase(dimensionSymbol)) {
    case Dimension.SYM_FALSE:
      return Dimension.FALSE
    case Dimension.SYM_TRUE:
      return Dimension.TRUE
    case Dimension.SYM_DONTCARE:
      return Dimension.DONTCARE
    case Dimension.SYM_P:
      return Dimension.P
    case Dimension.SYM_L:
      return Dimension.L
    case Dimension.SYM_A:
      return Dimension.A
    default:
  }
  throw new IllegalArgumentException('Unknown dimension symbol: ' + dimensionSymbol)
};

Object.defineProperties( Dimension, staticAccessors$14 );

var GeometryFilter = function GeometryFilter () {};

GeometryFilter.prototype.filter = function filter (geom) {};
GeometryFilter.prototype.interfaces_ = function interfaces_ () {
  return []
};
GeometryFilter.prototype.getClass = function getClass () {
  return GeometryFilter
};

var CoordinateSequenceFilter = function CoordinateSequenceFilter () {};

CoordinateSequenceFilter.prototype.filter = function filter (seq, i) {};
CoordinateSequenceFilter.prototype.isDone = function isDone () {};
CoordinateSequenceFilter.prototype.isGeometryChanged = function isGeometryChanged () {};
CoordinateSequenceFilter.prototype.interfaces_ = function interfaces_ () {
  return []
};
CoordinateSequenceFilter.prototype.getClass = function getClass () {
  return CoordinateSequenceFilter
};

var GeometryCollection = (function (Geometry$$1) {
  function GeometryCollection (geometries, factory) {
    Geometry$$1.call(this, factory);
    this._geometries = geometries || [];

    if (Geometry$$1.hasNullElements(this._geometries)) {
      throw new IllegalArgumentException('geometries must not contain null elements')
    }
  }

  if ( Geometry$$1 ) GeometryCollection.__proto__ = Geometry$$1;
  GeometryCollection.prototype = Object.create( Geometry$$1 && Geometry$$1.prototype );
  GeometryCollection.prototype.constructor = GeometryCollection;

  var staticAccessors = { serialVersionUID: { configurable: true } };
  GeometryCollection.prototype.computeEnvelopeInternal = function computeEnvelopeInternal () {
    var this$1 = this;

    var envelope = new Envelope();
    for (var i = 0; i < this._geometries.length; i++) {
      envelope.expandToInclude(this$1._geometries[i].getEnvelopeInternal());
    }
    return envelope
  };
  GeometryCollection.prototype.getGeometryN = function getGeometryN (n) {
    return this._geometries[n]
  };
  GeometryCollection.prototype.getSortIndex = function getSortIndex () {
    return Geometry$$1.SORTINDEX_GEOMETRYCOLLECTION
  };
  GeometryCollection.prototype.getCoordinates = function getCoordinates () {
    var this$1 = this;

    var coordinates = new Array(this.getNumPoints()).fill(null);
    var k = -1;
    for (var i = 0; i < this._geometries.length; i++) {
      var childCoordinates = this$1._geometries[i].getCoordinates();
      for (var j = 0; j < childCoordinates.length; j++) {
        k++;
        coordinates[k] = childCoordinates[j];
      }
    }
    return coordinates
  };
  GeometryCollection.prototype.getArea = function getArea () {
    var this$1 = this;

    var area = 0.0;
    for (var i = 0; i < this._geometries.length; i++) {
      area += this$1._geometries[i].getArea();
    }
    return area
  };
  GeometryCollection.prototype.equalsExact = function equalsExact () {
    var this$1 = this;

    if (arguments.length === 2) {
      var other = arguments[0];
      var tolerance = arguments[1];
      if (!this.isEquivalentClass(other)) {
        return false
      }
      var otherCollection = other;
      if (this._geometries.length !== otherCollection._geometries.length) {
        return false
      }
      for (var i = 0; i < this._geometries.length; i++) {
        if (!this$1._geometries[i].equalsExact(otherCollection._geometries[i], tolerance)) {
          return false
        }
      }
      return true
    } else { return Geometry$$1.prototype.equalsExact.apply(this, arguments) }
  };
  GeometryCollection.prototype.normalize = function normalize () {
    var this$1 = this;

    for (var i = 0; i < this._geometries.length; i++) {
      this$1._geometries[i].normalize();
    }
    Arrays.sort(this._geometries);
  };
  GeometryCollection.prototype.getCoordinate = function getCoordinate () {
    if (this.isEmpty()) { return null }
    return this._geometries[0].getCoordinate()
  };
  GeometryCollection.prototype.getBoundaryDimension = function getBoundaryDimension () {
    var this$1 = this;

    var dimension = Dimension.FALSE;
    for (var i = 0; i < this._geometries.length; i++) {
      dimension = Math.max(dimension, this$1._geometries[i].getBoundaryDimension());
    }
    return dimension
  };
  GeometryCollection.prototype.getDimension = function getDimension () {
    var this$1 = this;

    var dimension = Dimension.FALSE;
    for (var i = 0; i < this._geometries.length; i++) {
      dimension = Math.max(dimension, this$1._geometries[i].getDimension());
    }
    return dimension
  };
  GeometryCollection.prototype.getLength = function getLength () {
    var this$1 = this;

    var sum = 0.0;
    for (var i = 0; i < this._geometries.length; i++) {
      sum += this$1._geometries[i].getLength();
    }
    return sum
  };
  GeometryCollection.prototype.getNumPoints = function getNumPoints () {
    var this$1 = this;

    var numPoints = 0;
    for (var i = 0; i < this._geometries.length; i++) {
      numPoints += this$1._geometries[i].getNumPoints();
    }
    return numPoints
  };
  GeometryCollection.prototype.getNumGeometries = function getNumGeometries () {
    return this._geometries.length
  };
  GeometryCollection.prototype.reverse = function reverse () {
    var this$1 = this;

    var n = this._geometries.length;
    var revGeoms = new Array(n).fill(null);
    for (var i = 0; i < this._geometries.length; i++) {
      revGeoms[i] = this$1._geometries[i].reverse();
    }
    return this.getFactory().createGeometryCollection(revGeoms)
  };
  GeometryCollection.prototype.compareToSameClass = function compareToSameClass () {
    var this$1 = this;

    if (arguments.length === 1) {
      var o = arguments[0];
      var theseElements = new TreeSet(Arrays.asList(this._geometries));
      var otherElements = new TreeSet(Arrays.asList(o._geometries));
      return this.compare(theseElements, otherElements)
    } else if (arguments.length === 2) {
      var o$1 = arguments[0];
      var comp = arguments[1];
      var gc = o$1;
      var n1 = this.getNumGeometries();
      var n2 = gc.getNumGeometries();
      var i = 0;
      while (i < n1 && i < n2) {
        var thisGeom = this$1.getGeometryN(i);
        var otherGeom = gc.getGeometryN(i);
        var holeComp = thisGeom.compareToSameClass(otherGeom, comp);
        if (holeComp !== 0) { return holeComp }
        i++;
      }
      if (i < n1) { return 1 }
      if (i < n2) { return -1 }
      return 0
    }
  };
  GeometryCollection.prototype.apply = function apply () {
    var this$1 = this;

    if (hasInterface(arguments[0], CoordinateFilter)) {
      var filter = arguments[0];
      for (var i = 0; i < this._geometries.length; i++) {
        this$1._geometries[i].apply(filter);
      }
    } else if (hasInterface(arguments[0], CoordinateSequenceFilter)) {
      var filter$1 = arguments[0];
      if (this._geometries.length === 0) { return null }
      for (var i$1 = 0; i$1 < this._geometries.length; i$1++) {
        this$1._geometries[i$1].apply(filter$1);
        if (filter$1.isDone()) {
          break
        }
      }
      if (filter$1.isGeometryChanged()) { this.geometryChanged(); }
    } else if (hasInterface(arguments[0], GeometryFilter)) {
      var filter$2 = arguments[0];
      filter$2.filter(this);
      for (var i$2 = 0; i$2 < this._geometries.length; i$2++) {
        this$1._geometries[i$2].apply(filter$2);
      }
    } else if (hasInterface(arguments[0], GeometryComponentFilter)) {
      var filter$3 = arguments[0];
      filter$3.filter(this);
      for (var i$3 = 0; i$3 < this._geometries.length; i$3++) {
        this$1._geometries[i$3].apply(filter$3);
      }
    }
  };
  GeometryCollection.prototype.getBoundary = function getBoundary () {
    this.checkNotGeometryCollection(this);
    Assert.shouldNeverReachHere();
    return null
  };
  GeometryCollection.prototype.clone = function clone () {
    var this$1 = this;

    var gc = Geometry$$1.prototype.clone.call(this);
    gc._geometries = new Array(this._geometries.length).fill(null);
    for (var i = 0; i < this._geometries.length; i++) {
      gc._geometries[i] = this$1._geometries[i].clone();
    }
    return gc
  };
  GeometryCollection.prototype.getGeometryType = function getGeometryType () {
    return 'GeometryCollection'
  };
  GeometryCollection.prototype.copy = function copy () {
    var this$1 = this;

    var geometries = new Array(this._geometries.length).fill(null);
    for (var i = 0; i < geometries.length; i++) {
      geometries[i] = this$1._geometries[i].copy();
    }
    return new GeometryCollection(geometries, this._factory)
  };
  GeometryCollection.prototype.isEmpty = function isEmpty () {
    var this$1 = this;

    for (var i = 0; i < this._geometries.length; i++) {
      if (!this$1._geometries[i].isEmpty()) {
        return false
      }
    }
    return true
  };
  GeometryCollection.prototype.interfaces_ = function interfaces_ () {
    return []
  };
  GeometryCollection.prototype.getClass = function getClass () {
    return GeometryCollection
  };
  staticAccessors.serialVersionUID.get = function () { return -5694727726395021467 };

  Object.defineProperties( GeometryCollection, staticAccessors );

  return GeometryCollection;
}(Geometry));

var MultiLineString = (function (GeometryCollection$$1) {
  function MultiLineString () {
    GeometryCollection$$1.apply(this, arguments);
  }

  if ( GeometryCollection$$1 ) MultiLineString.__proto__ = GeometryCollection$$1;
  MultiLineString.prototype = Object.create( GeometryCollection$$1 && GeometryCollection$$1.prototype );
  MultiLineString.prototype.constructor = MultiLineString;

  var staticAccessors = { serialVersionUID: { configurable: true } };

  MultiLineString.prototype.getSortIndex = function getSortIndex () {
    return Geometry.SORTINDEX_MULTILINESTRING
  };
  MultiLineString.prototype.equalsExact = function equalsExact () {
    if (arguments.length === 2) {
      var other = arguments[0];
      var tolerance = arguments[1];
      if (!this.isEquivalentClass(other)) {
        return false
      }
      return GeometryCollection$$1.prototype.equalsExact.call(this, other, tolerance)
    } else { return GeometryCollection$$1.prototype.equalsExact.apply(this, arguments) }
  };
  MultiLineString.prototype.getBoundaryDimension = function getBoundaryDimension () {
    if (this.isClosed()) {
      return Dimension.FALSE
    }
    return 0
  };
  MultiLineString.prototype.isClosed = function isClosed () {
    var this$1 = this;

    if (this.isEmpty()) {
      return false
    }
    for (var i = 0; i < this._geometries.length; i++) {
      if (!this$1._geometries[i].isClosed()) {
        return false
      }
    }
    return true
  };
  MultiLineString.prototype.getDimension = function getDimension () {
    return 1
  };
  MultiLineString.prototype.reverse = function reverse () {
    var this$1 = this;

    var nLines = this._geometries.length;
    var revLines = new Array(nLines).fill(null);
    for (var i = 0; i < this._geometries.length; i++) {
      revLines[nLines - 1 - i] = this$1._geometries[i].reverse();
    }
    return this.getFactory().createMultiLineString(revLines)
  };
  MultiLineString.prototype.getBoundary = function getBoundary () {
    return new BoundaryOp(this).getBoundary()
  };
  MultiLineString.prototype.getGeometryType = function getGeometryType () {
    return 'MultiLineString'
  };
  MultiLineString.prototype.copy = function copy () {
    var this$1 = this;

    var lineStrings = new Array(this._geometries.length).fill(null);
    for (var i = 0; i < lineStrings.length; i++) {
      lineStrings[i] = this$1._geometries[i].copy();
    }
    return new MultiLineString(lineStrings, this._factory)
  };
  MultiLineString.prototype.interfaces_ = function interfaces_ () {
    return [Lineal]
  };
  MultiLineString.prototype.getClass = function getClass () {
    return MultiLineString
  };
  staticAccessors.serialVersionUID.get = function () { return 8166665132445433741 };

  Object.defineProperties( MultiLineString, staticAccessors );

  return MultiLineString;
}(GeometryCollection));

var BoundaryOp = function BoundaryOp () {
  this._geom = null;
  this._geomFact = null;
  this._bnRule = null;
  this._endpointMap = null;
  if (arguments.length === 1) {
    var geom = arguments[0];
    var bnRule = BoundaryNodeRule.MOD2_BOUNDARY_RULE;
    this._geom = geom;
    this._geomFact = geom.getFactory();
    this._bnRule = bnRule;
  } else if (arguments.length === 2) {
    var geom$1 = arguments[0];
    var bnRule$1 = arguments[1];
    this._geom = geom$1;
    this._geomFact = geom$1.getFactory();
    this._bnRule = bnRule$1;
  }
};
BoundaryOp.prototype.boundaryMultiLineString = function boundaryMultiLineString (mLine) {
  if (this._geom.isEmpty()) {
    return this.getEmptyMultiPoint()
  }
  var bdyPts = this.computeBoundaryCoordinates(mLine);
  if (bdyPts.length === 1) {
    return this._geomFact.createPoint(bdyPts[0])
  }
  return this._geomFact.createMultiPointFromCoords(bdyPts)
};
BoundaryOp.prototype.getBoundary = function getBoundary () {
  if (this._geom instanceof LineString$1) { return this.boundaryLineString(this._geom) }
  if (this._geom instanceof MultiLineString) { return this.boundaryMultiLineString(this._geom) }
  return this._geom.getBoundary()
};
BoundaryOp.prototype.boundaryLineString = function boundaryLineString (line) {
  if (this._geom.isEmpty()) {
    return this.getEmptyMultiPoint()
  }
  if (line.isClosed()) {
    var closedEndpointOnBoundary = this._bnRule.isInBoundary(2);
    if (closedEndpointOnBoundary) {
      return line.getStartPoint()
    } else {
      return this._geomFact.createMultiPoint()
    }
  }
  return this._geomFact.createMultiPoint([line.getStartPoint(), line.getEndPoint()])
};
BoundaryOp.prototype.getEmptyMultiPoint = function getEmptyMultiPoint () {
  return this._geomFact.createMultiPoint()
};
BoundaryOp.prototype.computeBoundaryCoordinates = function computeBoundaryCoordinates (mLine) {
    var this$1 = this;

  var bdyPts = new ArrayList();
  this._endpointMap = new TreeMap();
  for (var i = 0; i < mLine.getNumGeometries(); i++) {
    var line = mLine.getGeometryN(i);
    if (line.getNumPoints() === 0) { continue }
    this$1.addEndpoint(line.getCoordinateN(0));
    this$1.addEndpoint(line.getCoordinateN(line.getNumPoints() - 1));
  }
  for (var it = this._endpointMap.entrySet().iterator(); it.hasNext();) {
    var entry = it.next();
    var counter = entry.getValue();
    var valence = counter.count;
    if (this$1._bnRule.isInBoundary(valence)) {
      bdyPts.add(entry.getKey());
    }
  }
  return CoordinateArrays.toCoordinateArray(bdyPts)
};
BoundaryOp.prototype.addEndpoint = function addEndpoint (pt) {
  var counter = this._endpointMap.get(pt);
  if (counter === null) {
    counter = new Counter();
    this._endpointMap.put(pt, counter);
  }
  counter.count++;
};
BoundaryOp.prototype.interfaces_ = function interfaces_ () {
  return []
};
BoundaryOp.prototype.getClass = function getClass () {
  return BoundaryOp
};
BoundaryOp.getBoundary = function getBoundary () {
  if (arguments.length === 1) {
    var g = arguments[0];
    var bop = new BoundaryOp(g);
    return bop.getBoundary()
  } else if (arguments.length === 2) {
    var g$1 = arguments[0];
    var bnRule = arguments[1];
    var bop$1 = new BoundaryOp(g$1, bnRule);
    return bop$1.getBoundary()
  }
};

var Counter = function Counter () {
  this.count = null;
};
Counter.prototype.interfaces_ = function interfaces_ () {
  return []
};
Counter.prototype.getClass = function getClass () {
  return Counter
};

// boundary

function PrintStream () {}

function StringReader () {}

var DecimalFormat = function DecimalFormat () {};

function ByteArrayOutputStream () {}

function IOException () {}

function LineNumberReader () {}

var StringUtil = function StringUtil () {};

var staticAccessors$15 = { NEWLINE: { configurable: true },SIMPLE_ORDINATE_FORMAT: { configurable: true } };

StringUtil.prototype.interfaces_ = function interfaces_ () {
  return []
};
StringUtil.prototype.getClass = function getClass () {
  return StringUtil
};
StringUtil.chars = function chars (c, n) {
  var ch = new Array(n).fill(null);
  for (var i = 0; i < n; i++) {
    ch[i] = c;
  }
  return String(ch)
};
StringUtil.getStackTrace = function getStackTrace () {
  if (arguments.length === 1) {
    var t = arguments[0];
    var os = new ByteArrayOutputStream();
    var ps = new PrintStream(os);
    t.printStackTrace(ps);
    return os.toString()
  } else if (arguments.length === 2) {
    var t$1 = arguments[0];
    var depth = arguments[1];
    var stackTrace = '';
    var stringReader = new StringReader(StringUtil.getStackTrace(t$1));
    var lineNumberReader = new LineNumberReader(stringReader);
    for (var i = 0; i < depth; i++) {
      try {
        stackTrace += lineNumberReader.readLine() + StringUtil.NEWLINE;
      } catch (e) {
        if (e instanceof IOException) {
          Assert.shouldNeverReachHere();
        } else { throw e }
      } finally {}
    }
    return stackTrace
  }
};
StringUtil.split = function split (s, separator) {
  var separatorlen = separator.length;
  var tokenList = new ArrayList();
  var tmpString = '' + s;
  var pos = tmpString.indexOf(separator);
  while (pos >= 0) {
    var token = tmpString.substring(0, pos);
    tokenList.add(token);
    tmpString = tmpString.substring(pos + separatorlen);
    pos = tmpString.indexOf(separator);
  }
  if (tmpString.length > 0) { tokenList.add(tmpString); }
  var res = new Array(tokenList.size()).fill(null);
  for (var i = 0; i < res.length; i++) {
    res[i] = tokenList.get(i);
  }
  return res
};
StringUtil.toString = function toString () {
  if (arguments.length === 1) {
    var d = arguments[0];
    return StringUtil.SIMPLE_ORDINATE_FORMAT.format(d)
  }
};
StringUtil.spaces = function spaces (n) {
  return StringUtil.chars(' ', n)
};
staticAccessors$15.NEWLINE.get = function () { return System.getProperty('line.separator') };
staticAccessors$15.SIMPLE_ORDINATE_FORMAT.get = function () { return new DecimalFormat('0.#') };

Object.defineProperties( StringUtil, staticAccessors$15 );

var CoordinateSequences = function CoordinateSequences () {};

CoordinateSequences.prototype.interfaces_ = function interfaces_ () {
  return []
};
CoordinateSequences.prototype.getClass = function getClass () {
  return CoordinateSequences
};
CoordinateSequences.copyCoord = function copyCoord (src, srcPos, dest, destPos) {
  var minDim = Math.min(src.getDimension(), dest.getDimension());
  for (var dim = 0; dim < minDim; dim++) {
    dest.setOrdinate(destPos, dim, src.getOrdinate(srcPos, dim));
  }
};
CoordinateSequences.isRing = function isRing (seq) {
  var n = seq.size();
  if (n === 0) { return true }
  if (n <= 3) { return false }
  return seq.getOrdinate(0, CoordinateSequence.X) === seq.getOrdinate(n - 1, CoordinateSequence.X) && seq.getOrdinate(0, CoordinateSequence.Y) === seq.getOrdinate(n - 1, CoordinateSequence.Y)
};
CoordinateSequences.isEqual = function isEqual (cs1, cs2) {
  var cs1Size = cs1.size();
  var cs2Size = cs2.size();
  if (cs1Size !== cs2Size) { return false }
  var dim = Math.min(cs1.getDimension(), cs2.getDimension());
  for (var i = 0; i < cs1Size; i++) {
    for (var d = 0; d < dim; d++) {
      var v1 = cs1.getOrdinate(i, d);
      var v2 = cs2.getOrdinate(i, d);
      if (cs1.getOrdinate(i, d) === cs2.getOrdinate(i, d)) { continue }
      if (Double.isNaN(v1) && Double.isNaN(v2)) { continue }
      return false
    }
  }
  return true
};
CoordinateSequences.extend = function extend (fact, seq, size) {
  var newseq = fact.create(size, seq.getDimension());
  var n = seq.size();
  CoordinateSequences.copy(seq, 0, newseq, 0, n);
  if (n > 0) {
    for (var i = n; i < size; i++) { CoordinateSequences.copy(seq, n - 1, newseq, i, 1); }
  }
  return newseq
};
CoordinateSequences.reverse = function reverse (seq) {
  var last = seq.size() - 1;
  var mid = Math.trunc(last / 2);
  for (var i = 0; i <= mid; i++) {
    CoordinateSequences.swap(seq, i, last - i);
  }
};
CoordinateSequences.swap = function swap (seq, i, j) {
  if (i === j) { return null }
  for (var dim = 0; dim < seq.getDimension(); dim++) {
    var tmp = seq.getOrdinate(i, dim);
    seq.setOrdinate(i, dim, seq.getOrdinate(j, dim));
    seq.setOrdinate(j, dim, tmp);
  }
};
CoordinateSequences.copy = function copy (src, srcPos, dest, destPos, length) {
  for (var i = 0; i < length; i++) {
    CoordinateSequences.copyCoord(src, srcPos + i, dest, destPos + i);
  }
};
CoordinateSequences.toString = function toString () {
  if (arguments.length === 1) {
    var cs = arguments[0];
    var size = cs.size();
    if (size === 0) { return '()' }
    var dim = cs.getDimension();
    var buf = new StringBuffer();
    buf.append('(');
    for (var i = 0; i < size; i++) {
      if (i > 0) { buf.append(' '); }
      for (var d = 0; d < dim; d++) {
        if (d > 0) { buf.append(','); }
        buf.append(StringUtil.toString(cs.getOrdinate(i, d)));
      }
    }
    buf.append(')');
    return buf.toString()
  }
};
CoordinateSequences.ensureValidRing = function ensureValidRing (fact, seq) {
  var n = seq.size();
  if (n === 0) { return seq }
  if (n <= 3) { return CoordinateSequences.createClosedRing(fact, seq, 4) }
  var isClosed = seq.getOrdinate(0, CoordinateSequence.X) === seq.getOrdinate(n - 1, CoordinateSequence.X) && seq.getOrdinate(0, CoordinateSequence.Y) === seq.getOrdinate(n - 1, CoordinateSequence.Y);
  if (isClosed) { return seq }
  return CoordinateSequences.createClosedRing(fact, seq, n + 1)
};
CoordinateSequences.createClosedRing = function createClosedRing (fact, seq, size) {
  var newseq = fact.create(size, seq.getDimension());
  var n = seq.size();
  CoordinateSequences.copy(seq, 0, newseq, 0, n);
  for (var i = n; i < size; i++) { CoordinateSequences.copy(seq, 0, newseq, i, 1); }
  return newseq
};

var LineString$1 = (function (Geometry$$1) {
  function LineString (points, factory) {
    Geometry$$1.call(this, factory);
    this._points = null;
    this.init(points);
  }

  if ( Geometry$$1 ) LineString.__proto__ = Geometry$$1;
  LineString.prototype = Object.create( Geometry$$1 && Geometry$$1.prototype );
  LineString.prototype.constructor = LineString;

  var staticAccessors = { serialVersionUID: { configurable: true } };
  LineString.prototype.computeEnvelopeInternal = function computeEnvelopeInternal () {
    if (this.isEmpty()) {
      return new Envelope()
    }
    return this._points.expandEnvelope(new Envelope())
  };
  LineString.prototype.isRing = function isRing () {
    return this.isClosed() && this.isSimple()
  };
  LineString.prototype.getSortIndex = function getSortIndex () {
    return Geometry$$1.SORTINDEX_LINESTRING
  };
  LineString.prototype.getCoordinates = function getCoordinates () {
    return this._points.toCoordinateArray()
  };
  LineString.prototype.equalsExact = function equalsExact () {
    var this$1 = this;

    if (arguments.length === 2) {
      var other = arguments[0];
      var tolerance = arguments[1];
      if (!this.isEquivalentClass(other)) {
        return false
      }
      var otherLineString = other;
      if (this._points.size() !== otherLineString._points.size()) {
        return false
      }
      for (var i = 0; i < this._points.size(); i++) {
        if (!this$1.equal(this$1._points.getCoordinate(i), otherLineString._points.getCoordinate(i), tolerance)) {
          return false
        }
      }
      return true
    } else { return Geometry$$1.prototype.equalsExact.apply(this, arguments) }
  };
  LineString.prototype.normalize = function normalize () {
    var this$1 = this;

    for (var i = 0; i < Math.trunc(this._points.size() / 2); i++) {
      var j = this$1._points.size() - 1 - i;
      if (!this$1._points.getCoordinate(i).equals(this$1._points.getCoordinate(j))) {
        if (this$1._points.getCoordinate(i).compareTo(this$1._points.getCoordinate(j)) > 0) {
          CoordinateSequences.reverse(this$1._points);
        }
        return null
      }
    }
  };
  LineString.prototype.getCoordinate = function getCoordinate () {
    if (this.isEmpty()) { return null }
    return this._points.getCoordinate(0)
  };
  LineString.prototype.getBoundaryDimension = function getBoundaryDimension () {
    if (this.isClosed()) {
      return Dimension.FALSE
    }
    return 0
  };
  LineString.prototype.isClosed = function isClosed () {
    if (this.isEmpty()) {
      return false
    }
    return this.getCoordinateN(0).equals2D(this.getCoordinateN(this.getNumPoints() - 1))
  };
  LineString.prototype.getEndPoint = function getEndPoint () {
    if (this.isEmpty()) {
      return null
    }
    return this.getPointN(this.getNumPoints() - 1)
  };
  LineString.prototype.getDimension = function getDimension () {
    return 1
  };
  LineString.prototype.getLength = function getLength () {
    return CGAlgorithms.computeLength(this._points)
  };
  LineString.prototype.getNumPoints = function getNumPoints () {
    return this._points.size()
  };
  LineString.prototype.reverse = function reverse () {
    var seq = this._points.copy();
    CoordinateSequences.reverse(seq);
    var revLine = this.getFactory().createLineString(seq);
    return revLine
  };
  LineString.prototype.compareToSameClass = function compareToSameClass () {
    var this$1 = this;

    if (arguments.length === 1) {
      var o = arguments[0];
      var line = o;
      var i = 0;
      var j = 0;
      while (i < this._points.size() && j < line._points.size()) {
        var comparison = this$1._points.getCoordinate(i).compareTo(line._points.getCoordinate(j));
        if (comparison !== 0) {
          return comparison
        }
        i++;
        j++;
      }
      if (i < this._points.size()) {
        return 1
      }
      if (j < line._points.size()) {
        return -1
      }
      return 0
    } else if (arguments.length === 2) {
      var o$1 = arguments[0];
      var comp = arguments[1];
      var line$1 = o$1;
      return comp.compare(this._points, line$1._points)
    }
  };
  LineString.prototype.apply = function apply () {
    var this$1 = this;

    if (hasInterface(arguments[0], CoordinateFilter)) {
      var filter = arguments[0];
      for (var i = 0; i < this._points.size(); i++) {
        filter.filter(this$1._points.getCoordinate(i));
      }
    } else if (hasInterface(arguments[0], CoordinateSequenceFilter)) {
      var filter$1 = arguments[0];
      if (this._points.size() === 0) { return null }
      for (var i$1 = 0; i$1 < this._points.size(); i$1++) {
        filter$1.filter(this$1._points, i$1);
        if (filter$1.isDone()) { break }
      }
      if (filter$1.isGeometryChanged()) { this.geometryChanged(); }
    } else if (hasInterface(arguments[0], GeometryFilter)) {
      var filter$2 = arguments[0];
      filter$2.filter(this);
    } else if (hasInterface(arguments[0], GeometryComponentFilter)) {
      var filter$3 = arguments[0];
      filter$3.filter(this);
    }
  };
  LineString.prototype.getBoundary = function getBoundary () {
    return new BoundaryOp(this).getBoundary()
  };
  LineString.prototype.isEquivalentClass = function isEquivalentClass (other) {
    return other instanceof LineString
  };
  LineString.prototype.clone = function clone () {
    var ls = Geometry$$1.prototype.clone.call(this);
    ls._points = this._points.clone();
    return ls
  };
  LineString.prototype.getCoordinateN = function getCoordinateN (n) {
    return this._points.getCoordinate(n)
  };
  LineString.prototype.getGeometryType = function getGeometryType () {
    return 'LineString'
  };
  LineString.prototype.copy = function copy () {
    return new LineString(this._points.copy(), this._factory)
  };
  LineString.prototype.getCoordinateSequence = function getCoordinateSequence () {
    return this._points
  };
  LineString.prototype.isEmpty = function isEmpty () {
    return this._points.size() === 0
  };
  LineString.prototype.init = function init (points) {
    if (points === null) {
      points = this.getFactory().getCoordinateSequenceFactory().create([]);
    }
    if (points.size() === 1) {
      throw new IllegalArgumentException('Invalid number of points in LineString (found ' + points.size() + ' - must be 0 or >= 2)')
    }
    this._points = points;
  };
  LineString.prototype.isCoordinate = function isCoordinate (pt) {
    var this$1 = this;

    for (var i = 0; i < this._points.size(); i++) {
      if (this$1._points.getCoordinate(i).equals(pt)) {
        return true
      }
    }
    return false
  };
  LineString.prototype.getStartPoint = function getStartPoint () {
    if (this.isEmpty()) {
      return null
    }
    return this.getPointN(0)
  };
  LineString.prototype.getPointN = function getPointN (n) {
    return this.getFactory().createPoint(this._points.getCoordinate(n))
  };
  LineString.prototype.interfaces_ = function interfaces_ () {
    return [Lineal]
  };
  LineString.prototype.getClass = function getClass () {
    return LineString
  };
  staticAccessors.serialVersionUID.get = function () { return 3110669828065365560 };

  Object.defineProperties( LineString, staticAccessors );

  return LineString;
}(Geometry));

var Puntal = function Puntal () {};

Puntal.prototype.interfaces_ = function interfaces_ () {
  return []
};
Puntal.prototype.getClass = function getClass () {
  return Puntal
};

var Point = (function (Geometry$$1) {
  function Point (coordinates, factory) {
    Geometry$$1.call(this, factory);
    this._coordinates = coordinates || null;
    this.init(this._coordinates);
  }

  if ( Geometry$$1 ) Point.__proto__ = Geometry$$1;
  Point.prototype = Object.create( Geometry$$1 && Geometry$$1.prototype );
  Point.prototype.constructor = Point;

  var staticAccessors = { serialVersionUID: { configurable: true } };
  Point.prototype.computeEnvelopeInternal = function computeEnvelopeInternal () {
    if (this.isEmpty()) {
      return new Envelope()
    }
    var env = new Envelope();
    env.expandToInclude(this._coordinates.getX(0), this._coordinates.getY(0));
    return env
  };
  Point.prototype.getSortIndex = function getSortIndex () {
    return Geometry$$1.SORTINDEX_POINT
  };
  Point.prototype.getCoordinates = function getCoordinates () {
    return this.isEmpty() ? [] : [this.getCoordinate()]
  };
  Point.prototype.equalsExact = function equalsExact () {
    if (arguments.length === 2) {
      var other = arguments[0];
      var tolerance = arguments[1];
      if (!this.isEquivalentClass(other)) {
        return false
      }
      if (this.isEmpty() && other.isEmpty()) {
        return true
      }
      if (this.isEmpty() !== other.isEmpty()) {
        return false
      }
      return this.equal(other.getCoordinate(), this.getCoordinate(), tolerance)
    } else { return Geometry$$1.prototype.equalsExact.apply(this, arguments) }
  };
  Point.prototype.normalize = function normalize () {};
  Point.prototype.getCoordinate = function getCoordinate () {
    return this._coordinates.size() !== 0 ? this._coordinates.getCoordinate(0) : null
  };
  Point.prototype.getBoundaryDimension = function getBoundaryDimension () {
    return Dimension.FALSE
  };
  Point.prototype.getDimension = function getDimension () {
    return 0
  };
  Point.prototype.getNumPoints = function getNumPoints () {
    return this.isEmpty() ? 0 : 1
  };
  Point.prototype.reverse = function reverse () {
    return this.copy()
  };
  Point.prototype.getX = function getX () {
    if (this.getCoordinate() === null) {
      throw new Error('getX called on empty Point')
    }
    return this.getCoordinate().x
  };
  Point.prototype.compareToSameClass = function compareToSameClass () {
    if (arguments.length === 1) {
      var other = arguments[0];
      var point$1 = other;
      return this.getCoordinate().compareTo(point$1.getCoordinate())
    } else if (arguments.length === 2) {
      var other$1 = arguments[0];
      var comp = arguments[1];
      var point = other$1;
      return comp.compare(this._coordinates, point._coordinates)
    }
  };
  Point.prototype.apply = function apply () {
    if (hasInterface(arguments[0], CoordinateFilter)) {
      var filter = arguments[0];
      if (this.isEmpty()) {
        return null
      }
      filter.filter(this.getCoordinate());
    } else if (hasInterface(arguments[0], CoordinateSequenceFilter)) {
      var filter$1 = arguments[0];
      if (this.isEmpty()) { return null }
      filter$1.filter(this._coordinates, 0);
      if (filter$1.isGeometryChanged()) { this.geometryChanged(); }
    } else if (hasInterface(arguments[0], GeometryFilter)) {
      var filter$2 = arguments[0];
      filter$2.filter(this);
    } else if (hasInterface(arguments[0], GeometryComponentFilter)) {
      var filter$3 = arguments[0];
      filter$3.filter(this);
    }
  };
  Point.prototype.getBoundary = function getBoundary () {
    return this.getFactory().createGeometryCollection(null)
  };
  Point.prototype.clone = function clone () {
    var p = Geometry$$1.prototype.clone.call(this);
    p._coordinates = this._coordinates.clone();
    return p
  };
  Point.prototype.getGeometryType = function getGeometryType () {
    return 'Point'
  };
  Point.prototype.copy = function copy () {
    return new Point(this._coordinates.copy(), this._factory)
  };
  Point.prototype.getCoordinateSequence = function getCoordinateSequence () {
    return this._coordinates
  };
  Point.prototype.getY = function getY () {
    if (this.getCoordinate() === null) {
      throw new Error('getY called on empty Point')
    }
    return this.getCoordinate().y
  };
  Point.prototype.isEmpty = function isEmpty () {
    return this._coordinates.size() === 0
  };
  Point.prototype.init = function init (coordinates) {
    if (coordinates === null) {
      coordinates = this.getFactory().getCoordinateSequenceFactory().create([]);
    }
    Assert.isTrue(coordinates.size() <= 1);
    this._coordinates = coordinates;
  };
  Point.prototype.isSimple = function isSimple () {
    return true
  };
  Point.prototype.interfaces_ = function interfaces_ () {
    return [Puntal]
  };
  Point.prototype.getClass = function getClass () {
    return Point
  };
  staticAccessors.serialVersionUID.get = function () { return 4902022702746614570 };

  Object.defineProperties( Point, staticAccessors );

  return Point;
}(Geometry));

var Polygonal = function Polygonal () {};

Polygonal.prototype.interfaces_ = function interfaces_ () {
  return []
};
Polygonal.prototype.getClass = function getClass () {
  return Polygonal
};

var Polygon = (function (Geometry$$1) {
  function Polygon (shell, holes, factory) {
    Geometry$$1.call(this, factory);
    this._shell = null;
    this._holes = null;
    if (shell === null) {
      shell = this.getFactory().createLinearRing();
    }
    if (holes === null) {
      holes = [];
    }
    if (Geometry$$1.hasNullElements(holes)) {
      throw new IllegalArgumentException('holes must not contain null elements')
    }
    if (shell.isEmpty() && Geometry$$1.hasNonEmptyElements(holes)) {
      throw new IllegalArgumentException('shell is empty but holes are not')
    }
    this._shell = shell;
    this._holes = holes;
  }

  if ( Geometry$$1 ) Polygon.__proto__ = Geometry$$1;
  Polygon.prototype = Object.create( Geometry$$1 && Geometry$$1.prototype );
  Polygon.prototype.constructor = Polygon;

  var staticAccessors = { serialVersionUID: { configurable: true } };
  Polygon.prototype.computeEnvelopeInternal = function computeEnvelopeInternal () {
    return this._shell.getEnvelopeInternal()
  };
  Polygon.prototype.getSortIndex = function getSortIndex () {
    return Geometry$$1.SORTINDEX_POLYGON
  };
  Polygon.prototype.getCoordinates = function getCoordinates () {
    var this$1 = this;

    if (this.isEmpty()) {
      return []
    }
    var coordinates = new Array(this.getNumPoints()).fill(null);
    var k = -1;
    var shellCoordinates = this._shell.getCoordinates();
    for (var x = 0; x < shellCoordinates.length; x++) {
      k++;
      coordinates[k] = shellCoordinates[x];
    }
    for (var i = 0; i < this._holes.length; i++) {
      var childCoordinates = this$1._holes[i].getCoordinates();
      for (var j = 0; j < childCoordinates.length; j++) {
        k++;
        coordinates[k] = childCoordinates[j];
      }
    }
    return coordinates
  };
  Polygon.prototype.getArea = function getArea () {
    var this$1 = this;

    var area = 0.0;
    area += Math.abs(CGAlgorithms.signedArea(this._shell.getCoordinateSequence()));
    for (var i = 0; i < this._holes.length; i++) {
      area -= Math.abs(CGAlgorithms.signedArea(this$1._holes[i].getCoordinateSequence()));
    }
    return area
  };
  Polygon.prototype.isRectangle = function isRectangle () {
    if (this.getNumInteriorRing() !== 0) { return false }
    if (this._shell === null) { return false }
    if (this._shell.getNumPoints() !== 5) { return false }
    var seq = this._shell.getCoordinateSequence();
    var env = this.getEnvelopeInternal();
    for (var i = 0; i < 5; i++) {
      var x = seq.getX(i);
      if (!(x === env.getMinX() || x === env.getMaxX())) { return false }
      var y = seq.getY(i);
      if (!(y === env.getMinY() || y === env.getMaxY())) { return false }
    }
    var prevX = seq.getX(0);
    var prevY = seq.getY(0);
    for (var i$1 = 1; i$1 <= 4; i$1++) {
      var x$1 = seq.getX(i$1);
      var y$1 = seq.getY(i$1);
      var xChanged = x$1 !== prevX;
      var yChanged = y$1 !== prevY;
      if (xChanged === yChanged) { return false }
      prevX = x$1;
      prevY = y$1;
    }
    return true
  };
  Polygon.prototype.equalsExact = function equalsExact () {
    var this$1 = this;

    if (arguments.length === 2) {
      var other = arguments[0];
      var tolerance = arguments[1];
      if (!this.isEquivalentClass(other)) {
        return false
      }
      var otherPolygon = other;
      var thisShell = this._shell;
      var otherPolygonShell = otherPolygon._shell;
      if (!thisShell.equalsExact(otherPolygonShell, tolerance)) {
        return false
      }
      if (this._holes.length !== otherPolygon._holes.length) {
        return false
      }
      for (var i = 0; i < this._holes.length; i++) {
        if (!this$1._holes[i].equalsExact(otherPolygon._holes[i], tolerance)) {
          return false
        }
      }
      return true
    } else { return Geometry$$1.prototype.equalsExact.apply(this, arguments) }
  };
  Polygon.prototype.normalize = function normalize () {
    var this$1 = this;

    if (arguments.length === 0) {
      this.normalize(this._shell, true);
      for (var i = 0; i < this._holes.length; i++) {
        this$1.normalize(this$1._holes[i], false);
      }
      Arrays.sort(this._holes);
    } else if (arguments.length === 2) {
      var ring = arguments[0];
      var clockwise = arguments[1];
      if (ring.isEmpty()) {
        return null
      }
      var uniqueCoordinates = new Array(ring.getCoordinates().length - 1).fill(null);
      System.arraycopy(ring.getCoordinates(), 0, uniqueCoordinates, 0, uniqueCoordinates.length);
      var minCoordinate = CoordinateArrays.minCoordinate(ring.getCoordinates());
      CoordinateArrays.scroll(uniqueCoordinates, minCoordinate);
      System.arraycopy(uniqueCoordinates, 0, ring.getCoordinates(), 0, uniqueCoordinates.length);
      ring.getCoordinates()[uniqueCoordinates.length] = uniqueCoordinates[0];
      if (CGAlgorithms.isCCW(ring.getCoordinates()) === clockwise) {
        CoordinateArrays.reverse(ring.getCoordinates());
      }
    }
  };
  Polygon.prototype.getCoordinate = function getCoordinate () {
    return this._shell.getCoordinate()
  };
  Polygon.prototype.getNumInteriorRing = function getNumInteriorRing () {
    return this._holes.length
  };
  Polygon.prototype.getBoundaryDimension = function getBoundaryDimension () {
    return 1
  };
  Polygon.prototype.getDimension = function getDimension () {
    return 2
  };
  Polygon.prototype.getLength = function getLength () {
    var this$1 = this;

    var len = 0.0;
    len += this._shell.getLength();
    for (var i = 0; i < this._holes.length; i++) {
      len += this$1._holes[i].getLength();
    }
    return len
  };
  Polygon.prototype.getNumPoints = function getNumPoints () {
    var this$1 = this;

    var numPoints = this._shell.getNumPoints();
    for (var i = 0; i < this._holes.length; i++) {
      numPoints += this$1._holes[i].getNumPoints();
    }
    return numPoints
  };
  Polygon.prototype.reverse = function reverse () {
    var this$1 = this;

    var poly = this.copy();
    poly._shell = this._shell.copy().reverse();
    poly._holes = new Array(this._holes.length).fill(null);
    for (var i = 0; i < this._holes.length; i++) {
      poly._holes[i] = this$1._holes[i].copy().reverse();
    }
    return poly
  };
  Polygon.prototype.convexHull = function convexHull () {
    return this.getExteriorRing().convexHull()
  };
  Polygon.prototype.compareToSameClass = function compareToSameClass () {
    var this$1 = this;

    if (arguments.length === 1) {
      var o = arguments[0];
      var thisShell = this._shell;
      var otherShell = o._shell;
      return thisShell.compareToSameClass(otherShell)
    } else if (arguments.length === 2) {
      var o$1 = arguments[0];
      var comp = arguments[1];
      var poly = o$1;
      var thisShell$1 = this._shell;
      var otherShell$1 = poly._shell;
      var shellComp = thisShell$1.compareToSameClass(otherShell$1, comp);
      if (shellComp !== 0) { return shellComp }
      var nHole1 = this.getNumInteriorRing();
      var nHole2 = poly.getNumInteriorRing();
      var i = 0;
      while (i < nHole1 && i < nHole2) {
        var thisHole = this$1.getInteriorRingN(i);
        var otherHole = poly.getInteriorRingN(i);
        var holeComp = thisHole.compareToSameClass(otherHole, comp);
        if (holeComp !== 0) { return holeComp }
        i++;
      }
      if (i < nHole1) { return 1 }
      if (i < nHole2) { return -1 }
      return 0
    }
  };
  Polygon.prototype.apply = function apply (filter) {
    var this$1 = this;

    if (hasInterface(filter, CoordinateFilter)) {
      this._shell.apply(filter);
      for (var i$1 = 0; i$1 < this._holes.length; i$1++) {
        this$1._holes[i$1].apply(filter);
      }
    } else if (hasInterface(filter, CoordinateSequenceFilter)) {
      this._shell.apply(filter);
      if (!filter.isDone()) {
        for (var i$2 = 0; i$2 < this._holes.length; i$2++) {
          this$1._holes[i$2].apply(filter);
          if (filter.isDone()) { break }
        }
      }
      if (filter.isGeometryChanged()) { this.geometryChanged(); }
    } else if (hasInterface(filter, GeometryFilter)) {
      filter.filter(this);
    } else if (hasInterface(filter, GeometryComponentFilter)) {
      filter.filter(this);
      this._shell.apply(filter);
      for (var i = 0; i < this._holes.length; i++) {
        this$1._holes[i].apply(filter);
      }
    }
  };
  Polygon.prototype.getBoundary = function getBoundary () {
    var this$1 = this;

    if (this.isEmpty()) {
      return this.getFactory().createMultiLineString()
    }
    var rings = new Array(this._holes.length + 1).fill(null);
    rings[0] = this._shell;
    for (var i = 0; i < this._holes.length; i++) {
      rings[i + 1] = this$1._holes[i];
    }
    if (rings.length <= 1) { return this.getFactory().createLinearRing(rings[0].getCoordinateSequence()) }
    return this.getFactory().createMultiLineString(rings)
  };
  Polygon.prototype.clone = function clone () {
    var this$1 = this;

    var poly = Geometry$$1.prototype.clone.call(this);
    poly._shell = this._shell.clone();
    poly._holes = new Array(this._holes.length).fill(null);
    for (var i = 0; i < this._holes.length; i++) {
      poly._holes[i] = this$1._holes[i].clone();
    }
    return poly
  };
  Polygon.prototype.getGeometryType = function getGeometryType () {
    return 'Polygon'
  };
  Polygon.prototype.copy = function copy () {
    var this$1 = this;

    var shell = this._shell.copy();
    var holes = new Array(this._holes.length).fill(null);
    for (var i = 0; i < holes.length; i++) {
      holes[i] = this$1._holes[i].copy();
    }
    return new Polygon(shell, holes, this._factory)
  };
  Polygon.prototype.getExteriorRing = function getExteriorRing () {
    return this._shell
  };
  Polygon.prototype.isEmpty = function isEmpty () {
    return this._shell.isEmpty()
  };
  Polygon.prototype.getInteriorRingN = function getInteriorRingN (n) {
    return this._holes[n]
  };
  Polygon.prototype.interfaces_ = function interfaces_ () {
    return [Polygonal]
  };
  Polygon.prototype.getClass = function getClass () {
    return Polygon
  };
  staticAccessors.serialVersionUID.get = function () { return -3494792200821764533 };

  Object.defineProperties( Polygon, staticAccessors );

  return Polygon;
}(Geometry));

var MultiPoint = (function (GeometryCollection$$1) {
  function MultiPoint () {
    GeometryCollection$$1.apply(this, arguments);
  }

  if ( GeometryCollection$$1 ) MultiPoint.__proto__ = GeometryCollection$$1;
  MultiPoint.prototype = Object.create( GeometryCollection$$1 && GeometryCollection$$1.prototype );
  MultiPoint.prototype.constructor = MultiPoint;

  var staticAccessors = { serialVersionUID: { configurable: true } };

  MultiPoint.prototype.getSortIndex = function getSortIndex () {
    return Geometry.SORTINDEX_MULTIPOINT
  };
  MultiPoint.prototype.isValid = function isValid () {
    return true
  };
  MultiPoint.prototype.equalsExact = function equalsExact () {
    if (arguments.length === 2) {
      var other = arguments[0];
      var tolerance = arguments[1];
      if (!this.isEquivalentClass(other)) {
        return false
      }
      return GeometryCollection$$1.prototype.equalsExact.call(this, other, tolerance)
    } else { return GeometryCollection$$1.prototype.equalsExact.apply(this, arguments) }
  };
  MultiPoint.prototype.getCoordinate = function getCoordinate () {
    if (arguments.length === 1) {
      var n = arguments[0];
      return this._geometries[n].getCoordinate()
    } else { return GeometryCollection$$1.prototype.getCoordinate.apply(this, arguments) }
  };
  MultiPoint.prototype.getBoundaryDimension = function getBoundaryDimension () {
    return Dimension.FALSE
  };
  MultiPoint.prototype.getDimension = function getDimension () {
    return 0
  };
  MultiPoint.prototype.getBoundary = function getBoundary () {
    return this.getFactory().createGeometryCollection(null)
  };
  MultiPoint.prototype.getGeometryType = function getGeometryType () {
    return 'MultiPoint'
  };
  MultiPoint.prototype.copy = function copy () {
    var this$1 = this;

    var points = new Array(this._geometries.length).fill(null);
    for (var i = 0; i < points.length; i++) {
      points[i] = this$1._geometries[i].copy();
    }
    return new MultiPoint(points, this._factory)
  };
  MultiPoint.prototype.interfaces_ = function interfaces_ () {
    return [Puntal]
  };
  MultiPoint.prototype.getClass = function getClass () {
    return MultiPoint
  };
  staticAccessors.serialVersionUID.get = function () { return -8048474874175355449 };

  Object.defineProperties( MultiPoint, staticAccessors );

  return MultiPoint;
}(GeometryCollection));

var LinearRing = (function (LineString$$1) {
  function LinearRing (points, factory) {
    if (points instanceof Coordinate && factory instanceof GeometryFactory) {
      points = factory.getCoordinateSequenceFactory().create(points);
    }
    LineString$$1.call(this, points, factory);
    this.validateConstruction();
  }

  if ( LineString$$1 ) LinearRing.__proto__ = LineString$$1;
  LinearRing.prototype = Object.create( LineString$$1 && LineString$$1.prototype );
  LinearRing.prototype.constructor = LinearRing;

  var staticAccessors = { MINIMUM_VALID_SIZE: { configurable: true },serialVersionUID: { configurable: true } };
  LinearRing.prototype.getSortIndex = function getSortIndex () {
    return Geometry.SORTINDEX_LINEARRING
  };
  LinearRing.prototype.getBoundaryDimension = function getBoundaryDimension () {
    return Dimension.FALSE
  };
  LinearRing.prototype.isClosed = function isClosed () {
    if (this.isEmpty()) {
      return true
    }
    return LineString$$1.prototype.isClosed.call(this)
  };
  LinearRing.prototype.reverse = function reverse () {
    var seq = this._points.copy();
    CoordinateSequences.reverse(seq);
    var rev = this.getFactory().createLinearRing(seq);
    return rev
  };
  LinearRing.prototype.validateConstruction = function validateConstruction () {
    if (!this.isEmpty() && !LineString$$1.prototype.isClosed.call(this)) {
      throw new IllegalArgumentException('Points of LinearRing do not form a closed linestring')
    }
    if (this.getCoordinateSequence().size() >= 1 && this.getCoordinateSequence().size() < LinearRing.MINIMUM_VALID_SIZE) {
      throw new IllegalArgumentException('Invalid number of points in LinearRing (found ' + this.getCoordinateSequence().size() + ' - must be 0 or >= 4)')
    }
  };
  LinearRing.prototype.getGeometryType = function getGeometryType () {
    return 'LinearRing'
  };
  LinearRing.prototype.copy = function copy () {
    return new LinearRing(this._points.copy(), this._factory)
  };
  LinearRing.prototype.interfaces_ = function interfaces_ () {
    return []
  };
  LinearRing.prototype.getClass = function getClass () {
    return LinearRing
  };
  staticAccessors.MINIMUM_VALID_SIZE.get = function () { return 4 };
  staticAccessors.serialVersionUID.get = function () { return -4261142084085851829 };

  Object.defineProperties( LinearRing, staticAccessors );

  return LinearRing;
}(LineString$1));

var MultiPolygon = (function (GeometryCollection$$1) {
  function MultiPolygon () {
    GeometryCollection$$1.apply(this, arguments);
  }

  if ( GeometryCollection$$1 ) MultiPolygon.__proto__ = GeometryCollection$$1;
  MultiPolygon.prototype = Object.create( GeometryCollection$$1 && GeometryCollection$$1.prototype );
  MultiPolygon.prototype.constructor = MultiPolygon;

  var staticAccessors = { serialVersionUID: { configurable: true } };

  MultiPolygon.prototype.getSortIndex = function getSortIndex () {
    return Geometry.SORTINDEX_MULTIPOLYGON
  };
  MultiPolygon.prototype.equalsExact = function equalsExact () {
    if (arguments.length === 2) {
      var other = arguments[0];
      var tolerance = arguments[1];
      if (!this.isEquivalentClass(other)) {
        return false
      }
      return GeometryCollection$$1.prototype.equalsExact.call(this, other, tolerance)
    } else { return GeometryCollection$$1.prototype.equalsExact.apply(this, arguments) }
  };
  MultiPolygon.prototype.getBoundaryDimension = function getBoundaryDimension () {
    return 1
  };
  MultiPolygon.prototype.getDimension = function getDimension () {
    return 2
  };
  MultiPolygon.prototype.reverse = function reverse () {
    var this$1 = this;

    var n = this._geometries.length;
    var revGeoms = new Array(n).fill(null);
    for (var i = 0; i < this._geometries.length; i++) {
      revGeoms[i] = this$1._geometries[i].reverse();
    }
    return this.getFactory().createMultiPolygon(revGeoms)
  };
  MultiPolygon.prototype.getBoundary = function getBoundary () {
    var this$1 = this;

    if (this.isEmpty()) {
      return this.getFactory().createMultiLineString()
    }
    var allRings = new ArrayList();
    for (var i = 0; i < this._geometries.length; i++) {
      var polygon = this$1._geometries[i];
      var rings = polygon.getBoundary();
      for (var j = 0; j < rings.getNumGeometries(); j++) {
        allRings.add(rings.getGeometryN(j));
      }
    }
    var allRingsArray = new Array(allRings.size()).fill(null);
    return this.getFactory().createMultiLineString(allRings.toArray(allRingsArray))
  };
  MultiPolygon.prototype.getGeometryType = function getGeometryType () {
    return 'MultiPolygon'
  };
  MultiPolygon.prototype.copy = function copy () {
    var this$1 = this;

    var polygons = new Array(this._geometries.length).fill(null);
    for (var i = 0; i < polygons.length; i++) {
      polygons[i] = this$1._geometries[i].copy();
    }
    return new MultiPolygon(polygons, this._factory)
  };
  MultiPolygon.prototype.interfaces_ = function interfaces_ () {
    return [Polygonal]
  };
  MultiPolygon.prototype.getClass = function getClass () {
    return MultiPolygon
  };
  staticAccessors.serialVersionUID.get = function () { return -551033529766975875 };

  Object.defineProperties( MultiPolygon, staticAccessors );

  return MultiPolygon;
}(GeometryCollection));

var GeometryEditor = function GeometryEditor (factory) {
  this._factory = factory || null;
  this._isUserDataCopied = false;
};

var staticAccessors$16 = { NoOpGeometryOperation: { configurable: true },CoordinateOperation: { configurable: true },CoordinateSequenceOperation: { configurable: true } };
GeometryEditor.prototype.setCopyUserData = function setCopyUserData (isUserDataCopied) {
  this._isUserDataCopied = isUserDataCopied;
};
GeometryEditor.prototype.edit = function edit (geometry, operation) {
  if (geometry === null) { return null }
  var result = this.editInternal(geometry, operation);
  if (this._isUserDataCopied) {
    result.setUserData(geometry.getUserData());
  }
  return result
};
GeometryEditor.prototype.editInternal = function editInternal (geometry, operation) {
  if (this._factory === null) { this._factory = geometry.getFactory(); }
  if (geometry instanceof GeometryCollection) {
    return this.editGeometryCollection(geometry, operation)
  }
  if (geometry instanceof Polygon) {
    return this.editPolygon(geometry, operation)
  }
  if (geometry instanceof Point) {
    return operation.edit(geometry, this._factory)
  }
  if (geometry instanceof LineString$1) {
    return operation.edit(geometry, this._factory)
  }
  Assert.shouldNeverReachHere('Unsupported Geometry class: ' + geometry.getClass().getName());
  return null
};
GeometryEditor.prototype.editGeometryCollection = function editGeometryCollection (collection, operation) {
    var this$1 = this;

  var collectionForType = operation.edit(collection, this._factory);
  var geometries = new ArrayList();
  for (var i = 0; i < collectionForType.getNumGeometries(); i++) {
    var geometry = this$1.edit(collectionForType.getGeometryN(i), operation);
    if (geometry === null || geometry.isEmpty()) {
      continue
    }
    geometries.add(geometry);
  }
  if (collectionForType.getClass() === MultiPoint) {
    return this._factory.createMultiPoint(geometries.toArray([]))
  }
  if (collectionForType.getClass() === MultiLineString) {
    return this._factory.createMultiLineString(geometries.toArray([]))
  }
  if (collectionForType.getClass() === MultiPolygon) {
    return this._factory.createMultiPolygon(geometries.toArray([]))
  }
  return this._factory.createGeometryCollection(geometries.toArray([]))
};
GeometryEditor.prototype.editPolygon = function editPolygon (polygon, operation) {
    var this$1 = this;

  var newPolygon = operation.edit(polygon, this._factory);
  if (newPolygon === null) { newPolygon = this._factory.createPolygon(null); }
  if (newPolygon.isEmpty()) {
    return newPolygon
  }
  var shell = this.edit(newPolygon.getExteriorRing(), operation);
  if (shell === null || shell.isEmpty()) {
    return this._factory.createPolygon()
  }
  var holes = new ArrayList();
  for (var i = 0; i < newPolygon.getNumInteriorRing(); i++) {
    var hole = this$1.edit(newPolygon.getInteriorRingN(i), operation);
    if (hole === null || hole.isEmpty()) {
      continue
    }
    holes.add(hole);
  }
  return this._factory.createPolygon(shell, holes.toArray([]))
};
GeometryEditor.prototype.interfaces_ = function interfaces_ () {
  return []
};
GeometryEditor.prototype.getClass = function getClass () {
  return GeometryEditor
};
GeometryEditor.GeometryEditorOperation = function GeometryEditorOperation () {};
staticAccessors$16.NoOpGeometryOperation.get = function () { return NoOpGeometryOperation };
staticAccessors$16.CoordinateOperation.get = function () { return CoordinateOperation };
staticAccessors$16.CoordinateSequenceOperation.get = function () { return CoordinateSequenceOperation };

Object.defineProperties( GeometryEditor, staticAccessors$16 );

var NoOpGeometryOperation = function NoOpGeometryOperation () {};

NoOpGeometryOperation.prototype.edit = function edit (geometry, factory) {
  return geometry
};
NoOpGeometryOperation.prototype.interfaces_ = function interfaces_ () {
  return [GeometryEditor.GeometryEditorOperation]
};
NoOpGeometryOperation.prototype.getClass = function getClass () {
  return NoOpGeometryOperation
};

var CoordinateOperation = function CoordinateOperation () {};

CoordinateOperation.prototype.edit = function edit (geometry, factory) {
  var coords = this.editCoordinates(geometry.getCoordinates(), geometry);
  if (coords === null) { return geometry }
  if (geometry instanceof LinearRing) {
    return factory.createLinearRing(coords)
  }
  if (geometry instanceof LineString$1) {
    return factory.createLineString(coords)
  }
  if (geometry instanceof Point) {
    if (coords.length > 0) {
      return factory.createPoint(coords[0])
    } else {
      return factory.createPoint()
    }
  }
  return geometry
};
CoordinateOperation.prototype.interfaces_ = function interfaces_ () {
  return [GeometryEditor.GeometryEditorOperation]
};
CoordinateOperation.prototype.getClass = function getClass () {
  return CoordinateOperation
};

var CoordinateSequenceOperation = function CoordinateSequenceOperation () {};

CoordinateSequenceOperation.prototype.edit = function edit (geometry, factory) {
  if (geometry instanceof LinearRing) {
    return factory.createLinearRing(this.edit(geometry.getCoordinateSequence(), geometry))
  }
  if (geometry instanceof LineString$1) {
    return factory.createLineString(this.edit(geometry.getCoordinateSequence(), geometry))
  }
  if (geometry instanceof Point) {
    return factory.createPoint(this.edit(geometry.getCoordinateSequence(), geometry))
  }
  return geometry
};
CoordinateSequenceOperation.prototype.interfaces_ = function interfaces_ () {
  return [GeometryEditor.GeometryEditorOperation]
};
CoordinateSequenceOperation.prototype.getClass = function getClass () {
  return CoordinateSequenceOperation
};

var CoordinateArraySequence = function CoordinateArraySequence () {
  var this$1 = this;

  this._dimension = 3;
  this._coordinates = null;
  if (arguments.length === 1) {
    if (arguments[0] instanceof Array) {
      this._coordinates = arguments[0];
      this._dimension = 3;
    } else if (Number.isInteger(arguments[0])) {
      var size = arguments[0];
      this._coordinates = new Array(size).fill(null);
      for (var i = 0; i < size; i++) {
        this$1._coordinates[i] = new Coordinate();
      }
    } else if (hasInterface(arguments[0], CoordinateSequence)) {
      var coordSeq = arguments[0];
      if (coordSeq === null) {
        this._coordinates = new Array(0).fill(null);
        return null
      }
      this._dimension = coordSeq.getDimension();
      this._coordinates = new Array(coordSeq.size()).fill(null);
      for (var i$1 = 0; i$1 < this._coordinates.length; i$1++) {
        this$1._coordinates[i$1] = coordSeq.getCoordinateCopy(i$1);
      }
    }
  } else if (arguments.length === 2) {
    if (arguments[0] instanceof Array && Number.isInteger(arguments[1])) {
      var coordinates = arguments[0];
      var dimension = arguments[1];
      this._coordinates = coordinates;
      this._dimension = dimension;
      if (coordinates === null) { this._coordinates = new Array(0).fill(null); }
    } else if (Number.isInteger(arguments[0]) && Number.isInteger(arguments[1])) {
      var size$1 = arguments[0];
      var dimension$1 = arguments[1];
      this._coordinates = new Array(size$1).fill(null);
      this._dimension = dimension$1;
      for (var i$2 = 0; i$2 < size$1; i$2++) {
        this$1._coordinates[i$2] = new Coordinate();
      }
    }
  }
};

var staticAccessors$18 = { serialVersionUID: { configurable: true } };
CoordinateArraySequence.prototype.setOrdinate = function setOrdinate (index, ordinateIndex, value) {
  switch (ordinateIndex) {
    case CoordinateSequence.X:
      this._coordinates[index].x = value;
      break
    case CoordinateSequence.Y:
      this._coordinates[index].y = value;
      break
    case CoordinateSequence.Z:
      this._coordinates[index].z = value;
      break
    default:
      throw new IllegalArgumentException('invalid ordinateIndex')
  }
};
CoordinateArraySequence.prototype.size = function size () {
  return this._coordinates.length
};
CoordinateArraySequence.prototype.getOrdinate = function getOrdinate (index, ordinateIndex) {
  switch (ordinateIndex) {
    case CoordinateSequence.X:
      return this._coordinates[index].x
    case CoordinateSequence.Y:
      return this._coordinates[index].y
    case CoordinateSequence.Z:
      return this._coordinates[index].z
    default:
  }
  return Double.NaN
};
CoordinateArraySequence.prototype.getCoordinate = function getCoordinate () {
  if (arguments.length === 1) {
    var i = arguments[0];
    return this._coordinates[i]
  } else if (arguments.length === 2) {
    var index = arguments[0];
    var coord = arguments[1];
    coord.x = this._coordinates[index].x;
    coord.y = this._coordinates[index].y;
    coord.z = this._coordinates[index].z;
  }
};
CoordinateArraySequence.prototype.getCoordinateCopy = function getCoordinateCopy (i) {
  return new Coordinate(this._coordinates[i])
};
CoordinateArraySequence.prototype.getDimension = function getDimension () {
  return this._dimension
};
CoordinateArraySequence.prototype.getX = function getX (index) {
  return this._coordinates[index].x
};
CoordinateArraySequence.prototype.clone = function clone () {
    var this$1 = this;

  var cloneCoordinates = new Array(this.size()).fill(null);
  for (var i = 0; i < this._coordinates.length; i++) {
    cloneCoordinates[i] = this$1._coordinates[i].clone();
  }
  return new CoordinateArraySequence(cloneCoordinates, this._dimension)
};
CoordinateArraySequence.prototype.expandEnvelope = function expandEnvelope (env) {
    var this$1 = this;

  for (var i = 0; i < this._coordinates.length; i++) {
    env.expandToInclude(this$1._coordinates[i]);
  }
  return env
};
CoordinateArraySequence.prototype.copy = function copy () {
    var this$1 = this;

  var cloneCoordinates = new Array(this.size()).fill(null);
  for (var i = 0; i < this._coordinates.length; i++) {
    cloneCoordinates[i] = this$1._coordinates[i].copy();
  }
  return new CoordinateArraySequence(cloneCoordinates, this._dimension)
};
CoordinateArraySequence.prototype.toString = function toString () {
    var this$1 = this;

  if (this._coordinates.length > 0) {
    var strBuf = new StringBuffer(17 * this._coordinates.length);
    strBuf.append('(');
    strBuf.append(this._coordinates[0]);
    for (var i = 1; i < this._coordinates.length; i++) {
      strBuf.append(', ');
      strBuf.append(this$1._coordinates[i]);
    }
    strBuf.append(')');
    return strBuf.toString()
  } else {
    return '()'
  }
};
CoordinateArraySequence.prototype.getY = function getY (index) {
  return this._coordinates[index].y
};
CoordinateArraySequence.prototype.toCoordinateArray = function toCoordinateArray () {
  return this._coordinates
};
CoordinateArraySequence.prototype.interfaces_ = function interfaces_ () {
  return [CoordinateSequence, Serializable]
};
CoordinateArraySequence.prototype.getClass = function getClass () {
  return CoordinateArraySequence
};
staticAccessors$18.serialVersionUID.get = function () { return -915438501601840650 };

Object.defineProperties( CoordinateArraySequence, staticAccessors$18 );

var CoordinateArraySequenceFactory = function CoordinateArraySequenceFactory () {};

var staticAccessors$17 = { serialVersionUID: { configurable: true },instanceObject: { configurable: true } };

CoordinateArraySequenceFactory.prototype.readResolve = function readResolve () {
  return CoordinateArraySequenceFactory.instance()
};
CoordinateArraySequenceFactory.prototype.create = function create () {
  if (arguments.length === 1) {
    if (arguments[0] instanceof Array) {
      var coordinates = arguments[0];
      return new CoordinateArraySequence(coordinates)
    } else if (hasInterface(arguments[0], CoordinateSequence)) {
      var coordSeq = arguments[0];
      return new CoordinateArraySequence(coordSeq)
    }
  } else if (arguments.length === 2) {
    var size = arguments[0];
    var dimension = arguments[1];
    if (dimension > 3) { dimension = 3; }
    if (dimension < 2) { return new CoordinateArraySequence(size) }
    return new CoordinateArraySequence(size, dimension)
  }
};
CoordinateArraySequenceFactory.prototype.interfaces_ = function interfaces_ () {
  return [CoordinateSequenceFactory, Serializable]
};
CoordinateArraySequenceFactory.prototype.getClass = function getClass () {
  return CoordinateArraySequenceFactory
};
CoordinateArraySequenceFactory.instance = function instance () {
  return CoordinateArraySequenceFactory.instanceObject
};

staticAccessors$17.serialVersionUID.get = function () { return -4099577099607551657 };
staticAccessors$17.instanceObject.get = function () { return new CoordinateArraySequenceFactory() };

Object.defineProperties( CoordinateArraySequenceFactory, staticAccessors$17 );

/**
 * @see http://download.oracle.com/javase/6/docs/api/java/util/HashMap.html
 *
 * @extends {javascript.util.Map}
 * @constructor
 * @private
 */
var HashMap = (function (MapInterface) {
  function HashMap () {
    MapInterface.call(this);
    this.map_ = new Map();
  }

  if ( MapInterface ) HashMap.__proto__ = MapInterface;
  HashMap.prototype = Object.create( MapInterface && MapInterface.prototype );
  HashMap.prototype.constructor = HashMap;
  /**
   * @override
   */
  HashMap.prototype.get = function get (key) {
    return this.map_.get(key) || null
  };

  /**
   * @override
   */
  HashMap.prototype.put = function put (key, value) {
    this.map_.set(key, value);
    return value
  };

  /**
   * @override
   */
  HashMap.prototype.values = function values () {
    var arrayList = new ArrayList();
    var it = this.map_.values();
    var o = it.next();
    while (!o.done) {
      arrayList.add(o.value);
      o = it.next();
    }
    return arrayList
  };

  /**
   * @override
   */
  HashMap.prototype.entrySet = function entrySet () {
    var hashSet = new HashSet();
    this.map_.entries().forEach(function (entry) { return hashSet.add(entry); });
    return hashSet
  };

  /**
   * @override
   */
  HashMap.prototype.size = function size () {
    return this.map_.size()
  };

  return HashMap;
}(Map$1));

var PrecisionModel = function PrecisionModel () {
  this._modelType = null;
  this._scale = null;
  if (arguments.length === 0) {
    this._modelType = PrecisionModel.FLOATING;
  } else if (arguments.length === 1) {
    if (arguments[0] instanceof Type) {
      var modelType = arguments[0];
      this._modelType = modelType;
      if (modelType === PrecisionModel.FIXED) {
        this.setScale(1.0);
      }
    } else if (typeof arguments[0] === 'number') {
      var scale = arguments[0];
      this._modelType = PrecisionModel.FIXED;
      this.setScale(scale);
    } else if (arguments[0] instanceof PrecisionModel) {
      var pm = arguments[0];
      this._modelType = pm._modelType;
      this._scale = pm._scale;
    }
  }
};

var staticAccessors$19 = { serialVersionUID: { configurable: true },maximumPreciseValue: { configurable: true } };
PrecisionModel.prototype.equals = function equals (other) {
  if (!(other instanceof PrecisionModel)) {
    return false
  }
  var otherPrecisionModel = other;
  return this._modelType === otherPrecisionModel._modelType && this._scale === otherPrecisionModel._scale
};
PrecisionModel.prototype.compareTo = function compareTo (o) {
  var other = o;
  var sigDigits = this.getMaximumSignificantDigits();
  var otherSigDigits = other.getMaximumSignificantDigits();
  return new Integer(sigDigits).compareTo(new Integer(otherSigDigits))
};
PrecisionModel.prototype.getScale = function getScale () {
  return this._scale
};
PrecisionModel.prototype.isFloating = function isFloating () {
  return this._modelType === PrecisionModel.FLOATING || this._modelType === PrecisionModel.FLOATING_SINGLE
};
PrecisionModel.prototype.getType = function getType () {
  return this._modelType
};
PrecisionModel.prototype.toString = function toString () {
  var description = 'UNKNOWN';
  if (this._modelType === PrecisionModel.FLOATING) {
    description = 'Floating';
  } else if (this._modelType === PrecisionModel.FLOATING_SINGLE) {
    description = 'Floating-Single';
  } else if (this._modelType === PrecisionModel.FIXED) {
    description = 'Fixed (Scale=' + this.getScale() + ')';
  }
  return description
};
PrecisionModel.prototype.makePrecise = function makePrecise () {
  if (typeof arguments[0] === 'number') {
    var val = arguments[0];
    if (Double.isNaN(val)) { return val }
    if (this._modelType === PrecisionModel.FLOATING_SINGLE) {
      var floatSingleVal = val;
      return floatSingleVal
    }
    if (this._modelType === PrecisionModel.FIXED) {
      return Math.round(val * this._scale) / this._scale
    }
    return val
  } else if (arguments[0] instanceof Coordinate) {
    var coord = arguments[0];
    if (this._modelType === PrecisionModel.FLOATING) { return null }
    coord.x = this.makePrecise(coord.x);
    coord.y = this.makePrecise(coord.y);
  }
};
PrecisionModel.prototype.getMaximumSignificantDigits = function getMaximumSignificantDigits () {
  var maxSigDigits = 16;
  if (this._modelType === PrecisionModel.FLOATING) {
    maxSigDigits = 16;
  } else if (this._modelType === PrecisionModel.FLOATING_SINGLE) {
    maxSigDigits = 6;
  } else if (this._modelType === PrecisionModel.FIXED) {
    maxSigDigits = 1 + Math.trunc(Math.ceil(Math.log(this.getScale()) / Math.log(10)));
  }
  return maxSigDigits
};
PrecisionModel.prototype.setScale = function setScale (scale) {
  this._scale = Math.abs(scale);
};
PrecisionModel.prototype.interfaces_ = function interfaces_ () {
  return [Serializable, Comparable]
};
PrecisionModel.prototype.getClass = function getClass () {
  return PrecisionModel
};
PrecisionModel.mostPrecise = function mostPrecise (pm1, pm2) {
  if (pm1.compareTo(pm2) >= 0) { return pm1 }
  return pm2
};
staticAccessors$19.serialVersionUID.get = function () { return 7777263578777803835 };
staticAccessors$19.maximumPreciseValue.get = function () { return 9007199254740992.0 };

Object.defineProperties( PrecisionModel, staticAccessors$19 );

var Type = function Type (name) {
  this._name = name || null;
  Type.nameToTypeMap.put(name, this);
};

var staticAccessors$1$1 = { serialVersionUID: { configurable: true },nameToTypeMap: { configurable: true } };
Type.prototype.readResolve = function readResolve () {
  return Type.nameToTypeMap.get(this._name)
};
Type.prototype.toString = function toString () {
  return this._name
};
Type.prototype.interfaces_ = function interfaces_ () {
  return [Serializable]
};
Type.prototype.getClass = function getClass () {
  return Type
};
staticAccessors$1$1.serialVersionUID.get = function () { return -5528602631731589822 };
staticAccessors$1$1.nameToTypeMap.get = function () { return new HashMap() };

Object.defineProperties( Type, staticAccessors$1$1 );

PrecisionModel.Type = Type;
PrecisionModel.FIXED = new Type('FIXED');
PrecisionModel.FLOATING = new Type('FLOATING');
PrecisionModel.FLOATING_SINGLE = new Type('FLOATING SINGLE');

var GeometryFactory = function GeometryFactory () {
  this._precisionModel = new PrecisionModel();
  this._SRID = 0;
  this._coordinateSequenceFactory = GeometryFactory.getDefaultCoordinateSequenceFactory();

  if (arguments.length === 0) {
  } else if (arguments.length === 1) {
    if (hasInterface(arguments[0], CoordinateSequenceFactory)) {
      this._coordinateSequenceFactory = arguments[0];
    } else if (arguments[0] instanceof PrecisionModel) {
      this._precisionModel = arguments[0];
    }
  } else if (arguments.length === 2) {
    this._precisionModel = arguments[0];
    this._SRID = arguments[1];
  } else if (arguments.length === 3) {
    this._precisionModel = arguments[0];
    this._SRID = arguments[1];
    this._coordinateSequenceFactory = arguments[2];
  }
};

var staticAccessors$2 = { serialVersionUID: { configurable: true } };
GeometryFactory.prototype.toGeometry = function toGeometry (envelope) {
  if (envelope.isNull()) {
    return this.createPoint(null)
  }
  if (envelope.getMinX() === envelope.getMaxX() && envelope.getMinY() === envelope.getMaxY()) {
    return this.createPoint(new Coordinate(envelope.getMinX(), envelope.getMinY()))
  }
  if (envelope.getMinX() === envelope.getMaxX() || envelope.getMinY() === envelope.getMaxY()) {
    return this.createLineString([new Coordinate(envelope.getMinX(), envelope.getMinY()), new Coordinate(envelope.getMaxX(), envelope.getMaxY())])
  }
  return this.createPolygon(this.createLinearRing([new Coordinate(envelope.getMinX(), envelope.getMinY()), new Coordinate(envelope.getMinX(), envelope.getMaxY()), new Coordinate(envelope.getMaxX(), envelope.getMaxY()), new Coordinate(envelope.getMaxX(), envelope.getMinY()), new Coordinate(envelope.getMinX(), envelope.getMinY())]), null)
};
GeometryFactory.prototype.createLineString = function createLineString (coordinates) {
  if (!coordinates) { return new LineString$1(this.getCoordinateSequenceFactory().create([]), this) }
  else if (coordinates instanceof Array) { return new LineString$1(this.getCoordinateSequenceFactory().create(coordinates), this) }
  else if (hasInterface(coordinates, CoordinateSequence)) { return new LineString$1(coordinates, this) }
};
GeometryFactory.prototype.createMultiLineString = function createMultiLineString () {
  if (arguments.length === 0) {
    return new MultiLineString(null, this)
  } else if (arguments.length === 1) {
    var lineStrings = arguments[0];
    return new MultiLineString(lineStrings, this)
  }
};
GeometryFactory.prototype.buildGeometry = function buildGeometry (geomList) {
  var geomClass = null;
  var isHeterogeneous = false;
  var hasGeometryCollection = false;
  for (var i = geomList.iterator(); i.hasNext();) {
    var geom = i.next();
    var partClass = geom.getClass();
    if (geomClass === null) {
      geomClass = partClass;
    }
    if (partClass !== geomClass) {
      isHeterogeneous = true;
    }
    if (geom.isGeometryCollectionOrDerived()) { hasGeometryCollection = true; }
  }
  if (geomClass === null) {
    return this.createGeometryCollection()
  }
  if (isHeterogeneous || hasGeometryCollection) {
    return this.createGeometryCollection(GeometryFactory.toGeometryArray(geomList))
  }
  var geom0 = geomList.iterator().next();
  var isCollection = geomList.size() > 1;
  if (isCollection) {
    if (geom0 instanceof Polygon) {
      return this.createMultiPolygon(GeometryFactory.toPolygonArray(geomList))
    } else if (geom0 instanceof LineString$1) {
      return this.createMultiLineString(GeometryFactory.toLineStringArray(geomList))
    } else if (geom0 instanceof Point) {
      return this.createMultiPoint(GeometryFactory.toPointArray(geomList))
    }
    Assert.shouldNeverReachHere('Unhandled class: ' + geom0.getClass().getName());
  }
  return geom0
};
GeometryFactory.prototype.createMultiPointFromCoords = function createMultiPointFromCoords (coordinates) {
  return this.createMultiPoint(coordinates !== null ? this.getCoordinateSequenceFactory().create(coordinates) : null)
};
GeometryFactory.prototype.createPoint = function createPoint () {
  if (arguments.length === 0) {
    return this.createPoint(this.getCoordinateSequenceFactory().create([]))
  } else if (arguments.length === 1) {
    if (arguments[0] instanceof Coordinate) {
      var coordinate = arguments[0];
      return this.createPoint(coordinate !== null ? this.getCoordinateSequenceFactory().create([coordinate]) : null)
    } else if (hasInterface(arguments[0], CoordinateSequence)) {
      var coordinates = arguments[0];
      return new Point(coordinates, this)
    }
  }
};
GeometryFactory.prototype.getCoordinateSequenceFactory = function getCoordinateSequenceFactory () {
  return this._coordinateSequenceFactory
};
GeometryFactory.prototype.createPolygon = function createPolygon () {
  if (arguments.length === 0) {
    return new Polygon(null, null, this)
  } else if (arguments.length === 1) {
    if (hasInterface(arguments[0], CoordinateSequence)) {
      var coordinates = arguments[0];
      return this.createPolygon(this.createLinearRing(coordinates))
    } else if (arguments[0] instanceof Array) {
      var coordinates$1 = arguments[0];
      return this.createPolygon(this.createLinearRing(coordinates$1))
    } else if (arguments[0] instanceof LinearRing) {
      var shell = arguments[0];
      return this.createPolygon(shell, null)
    }
  } else if (arguments.length === 2) {
    var shell$1 = arguments[0];
    var holes = arguments[1];
    return new Polygon(shell$1, holes, this)
  }
};
GeometryFactory.prototype.getSRID = function getSRID () {
  return this._SRID
};
GeometryFactory.prototype.createGeometryCollection = function createGeometryCollection () {
  if (arguments.length === 0) {
    return new GeometryCollection(null, this)
  } else if (arguments.length === 1) {
    var geometries = arguments[0];
    return new GeometryCollection(geometries, this)
  }
};
GeometryFactory.prototype.createGeometry = function createGeometry (g) {
  var editor = new GeometryEditor(this);
  return editor.edit(g, {
    edit: function () {
      if (arguments.length === 2) {
        var coordSeq = arguments[0];
        // const geometry = arguments[1]
        return this._coordinateSequenceFactory.create(coordSeq)
      }
    }
  })
};
GeometryFactory.prototype.getPrecisionModel = function getPrecisionModel () {
  return this._precisionModel
};
GeometryFactory.prototype.createLinearRing = function createLinearRing () {
  if (arguments.length === 0) {
    return this.createLinearRing(this.getCoordinateSequenceFactory().create([]))
  } else if (arguments.length === 1) {
    if (arguments[0] instanceof Array) {
      var coordinates = arguments[0];
      return this.createLinearRing(coordinates !== null ? this.getCoordinateSequenceFactory().create(coordinates) : null)
    } else if (hasInterface(arguments[0], CoordinateSequence)) {
      var coordinates$1 = arguments[0];
      return new LinearRing(coordinates$1, this)
    }
  }
};
GeometryFactory.prototype.createMultiPolygon = function createMultiPolygon () {
  if (arguments.length === 0) {
    return new MultiPolygon(null, this)
  } else if (arguments.length === 1) {
    var polygons = arguments[0];
    return new MultiPolygon(polygons, this)
  }
};
GeometryFactory.prototype.createMultiPoint = function createMultiPoint () {
    var this$1 = this;

  if (arguments.length === 0) {
    return new MultiPoint(null, this)
  } else if (arguments.length === 1) {
    if (arguments[0] instanceof Array) {
      var point = arguments[0];
      return new MultiPoint(point, this)
    } else if (arguments[0] instanceof Array) {
      var coordinates = arguments[0];
      return this.createMultiPoint(coordinates !== null ? this.getCoordinateSequenceFactory().create(coordinates) : null)
    } else if (hasInterface(arguments[0], CoordinateSequence)) {
      var coordinates$1 = arguments[0];
      if (coordinates$1 === null) {
        return this.createMultiPoint(new Array(0).fill(null))
      }
      var points = new Array(coordinates$1.size()).fill(null);
      for (var i = 0; i < coordinates$1.size(); i++) {
        var ptSeq = this$1.getCoordinateSequenceFactory().create(1, coordinates$1.getDimension());
        CoordinateSequences.copy(coordinates$1, i, ptSeq, 0, 1);
        points[i] = this$1.createPoint(ptSeq);
      }
      return this.createMultiPoint(points)
    }
  }
};
GeometryFactory.prototype.interfaces_ = function interfaces_ () {
  return [Serializable]
};
GeometryFactory.prototype.getClass = function getClass () {
  return GeometryFactory
};
GeometryFactory.toMultiPolygonArray = function toMultiPolygonArray (multiPolygons) {
  var multiPolygonArray = new Array(multiPolygons.size()).fill(null);
  return multiPolygons.toArray(multiPolygonArray)
};
GeometryFactory.toGeometryArray = function toGeometryArray (geometries) {
  if (geometries === null) { return null }
  var geometryArray = new Array(geometries.size()).fill(null);
  return geometries.toArray(geometryArray)
};
GeometryFactory.getDefaultCoordinateSequenceFactory = function getDefaultCoordinateSequenceFactory () {
  return CoordinateArraySequenceFactory.instance()
};
GeometryFactory.toMultiLineStringArray = function toMultiLineStringArray (multiLineStrings) {
  var multiLineStringArray = new Array(multiLineStrings.size()).fill(null);
  return multiLineStrings.toArray(multiLineStringArray)
};
GeometryFactory.toLineStringArray = function toLineStringArray (lineStrings) {
  var lineStringArray = new Array(lineStrings.size()).fill(null);
  return lineStrings.toArray(lineStringArray)
};
GeometryFactory.toMultiPointArray = function toMultiPointArray (multiPoints) {
  var multiPointArray = new Array(multiPoints.size()).fill(null);
  return multiPoints.toArray(multiPointArray)
};
GeometryFactory.toLinearRingArray = function toLinearRingArray (linearRings) {
  var linearRingArray = new Array(linearRings.size()).fill(null);
  return linearRings.toArray(linearRingArray)
};
GeometryFactory.toPointArray = function toPointArray (points) {
  var pointArray = new Array(points.size()).fill(null);
  return points.toArray(pointArray)
};
GeometryFactory.toPolygonArray = function toPolygonArray (polygons) {
  var polygonArray = new Array(polygons.size()).fill(null);
  return polygons.toArray(polygonArray)
};
GeometryFactory.createPointFromInternalCoord = function createPointFromInternalCoord (coord, exemplar) {
  exemplar.getPrecisionModel().makePrecise(coord);
  return exemplar.getFactory().createPoint(coord)
};
staticAccessors$2.serialVersionUID.get = function () { return -6820524753094095635 };

Object.defineProperties( GeometryFactory, staticAccessors$2 );

var geometryTypes = ['Point', 'MultiPoint', 'LineString', 'MultiLineString', 'Polygon', 'MultiPolygon'];

/**
 * Class for reading and writing Well-Known Text.Create a new parser for GeoJSON
 * NOTE: Adapted from OpenLayers 2.11 implementation.
 */

/**
 * Create a new parser for GeoJSON
 *
 * @param {GeometryFactory} geometryFactory
 * @return An instance of GeoJsonParser.
 * @constructor
 * @private
 */
var GeoJSONParser = function GeoJSONParser (geometryFactory) {
  this.geometryFactory = geometryFactory || new GeometryFactory();
};
/**
 * Deserialize a GeoJSON object and return the Geometry or Feature(Collection) with JSTS Geometries
 *
 * @param {}
 *        A GeoJSON object.
 * @return {} A Geometry instance or object representing a Feature(Collection) with Geometry instances.
 * @private
 */
GeoJSONParser.prototype.read = function read (json) {
  var obj;
  if (typeof json === 'string') {
    obj = JSON.parse(json);
  } else {
    obj = json;
  }

  var type = obj.type;

  if (!parse[type]) {
    throw new Error('Unknown GeoJSON type: ' + obj.type)
  }

  if (geometryTypes.indexOf(type) !== -1) {
    return parse[type].apply(this, [obj.coordinates])
  } else if (type === 'GeometryCollection') {
    return parse[type].apply(this, [obj.geometries])
  }

  // feature or feature collection
  return parse[type].apply(this, [obj])
};

/**
 * Serialize a Geometry object into GeoJSON
 *
 * @param {Geometry}
 *        geometry A Geometry or array of Geometries.
 * @return {Object} A GeoJSON object represting the input Geometry/Geometries.
 * @private
 */
GeoJSONParser.prototype.write = function write (geometry) {
  var type = geometry.getGeometryType();

  if (!extract$1$1[type]) {
    throw new Error('Geometry is not supported')
  }

  return extract$1$1[type].apply(this, [geometry])
};

var parse = {
  /**
   * Parse a GeoJSON Feature object
   *
   * @param {Object}
   *          obj Object to parse.
   *
   * @return {Object} Feature with geometry/bbox converted to JSTS Geometries.
   */
  Feature: function (obj) {
    var feature = {};

    // copy features
    for (var key in obj) {
      feature[key] = obj[key];
    }

    // parse geometry
    if (obj.geometry) {
      var type = obj.geometry.type;
      if (!parse[type]) {
        throw new Error('Unknown GeoJSON type: ' + obj.type)
      }
      feature.geometry = this.read(obj.geometry);
    }

    // bbox
    if (obj.bbox) {
      feature.bbox = parse.bbox.apply(this, [obj.bbox]);
    }

    return feature
  },

  /**
   * Parse a GeoJSON FeatureCollection object
   *
   * @param {Object}
   *          obj Object to parse.
   *
   * @return {Object} FeatureCollection with geometry/bbox converted to JSTS Geometries.
   */
  FeatureCollection: function (obj) {
    var this$1 = this;

    var featureCollection = {};

    if (obj.features) {
      featureCollection.features = [];

      for (var i = 0; i < obj.features.length; ++i) {
        featureCollection.features.push(this$1.read(obj.features[i]));
      }
    }

    if (obj.bbox) {
      featureCollection.bbox = this.parse.bbox.apply(this, [obj.bbox]);
    }

    return featureCollection
  },

  /**
   * Convert the ordinates in an array to an array of Coordinates
   *
   * @param {Array}
   *          array Array with {Number}s.
   *
   * @return {Array} Array with Coordinates.
   */
  coordinates: function (array) {
    var coordinates = [];
    for (var i = 0; i < array.length; ++i) {
      var sub = array[i];
      coordinates.push(new Coordinate(sub[0], sub[1]));
    }
    return coordinates
  },

  /**
   * Convert the bbox to a LinearRing
   *
   * @param {Array}
   *          array Array with [xMin, yMin, xMax, yMax].
   *
   * @return {Array} Array with Coordinates.
   */
  bbox: function (array) {
    return this.geometryFactory.createLinearRing([
      new Coordinate(array[0], array[1]),
      new Coordinate(array[2], array[1]),
      new Coordinate(array[2], array[3]),
      new Coordinate(array[0], array[3]),
      new Coordinate(array[0], array[1])
    ])
  },

  /**
   * Convert an Array with ordinates to a Point
   *
   * @param {Array}
   *          array Array with ordinates.
   *
   * @return {Point} Point.
   */
  Point: function (array) {
    var coordinate = new Coordinate(array[0], array[1]);
    return this.geometryFactory.createPoint(coordinate)
  },

  /**
   * Convert an Array with coordinates to a MultiPoint
   *
   * @param {Array}
   *          array Array with coordinates.
   *
   * @return {MultiPoint} MultiPoint.
   */
  MultiPoint: function (array) {
    var this$1 = this;

    var points = [];
    for (var i = 0; i < array.length; ++i) {
      points.push(parse.Point.apply(this$1, [array[i]]));
    }
    return this.geometryFactory.createMultiPoint(points)
  },

  /**
   * Convert an Array with coordinates to a LineString
   *
   * @param {Array}
   *          array Array with coordinates.
   *
   * @return {LineString} LineString.
   */
  LineString: function (array) {
    var coordinates = parse.coordinates.apply(this, [array]);
    return this.geometryFactory.createLineString(coordinates)
  },

  /**
   * Convert an Array with coordinates to a MultiLineString
   *
   * @param {Array}
   *          array Array with coordinates.
   *
   * @return {MultiLineString} MultiLineString.
   */
  MultiLineString: function (array) {
    var this$1 = this;

    var lineStrings = [];
    for (var i = 0; i < array.length; ++i) {
      lineStrings.push(parse.LineString.apply(this$1, [array[i]]));
    }
    return this.geometryFactory.createMultiLineString(lineStrings)
  },

  /**
   * Convert an Array to a Polygon
   *
   * @param {Array}
   *          array Array with shell and holes.
   *
   * @return {Polygon} Polygon.
   */
  Polygon: function (array) {
    var this$1 = this;

    var shellCoordinates = parse.coordinates.apply(this, [array[0]]);
    var shell = this.geometryFactory.createLinearRing(shellCoordinates);
    var holes = [];
    for (var i = 1; i < array.length; ++i) {
      var hole = array[i];
      var coordinates = parse.coordinates.apply(this$1, [hole]);
      var linearRing = this$1.geometryFactory.createLinearRing(coordinates);
      holes.push(linearRing);
    }
    return this.geometryFactory.createPolygon(shell, holes)
  },

  /**
   * Convert an Array to a MultiPolygon
   *
   * @param {Array}
   *          array Array of arrays with shell and rings.
   *
   * @return {MultiPolygon} MultiPolygon.
   */
  MultiPolygon: function (array) {
    var this$1 = this;

    var polygons = [];
    for (var i = 0; i < array.length; ++i) {
      var polygon = array[i];
      polygons.push(parse.Polygon.apply(this$1, [polygon]));
    