'use strict';
const core = require('@twind/core'), // eslint-disable-next-line no-var
/**
 * Whether the current browser supports `adoptedStyleSheets`.
 * @link https://github.com/lit/lit/blob/393e30cf7c7f97712e524df34e7343147055fc5d/packages/reactive-element/src/css-tag.ts#L13
 */ supportsAdoptingStyleSheets = 'undefined' != typeof ShadowRoot && ('undefined' == typeof ShadyCSS || ShadyCSS.nativeShadow) && 'adoptedStyleSheets' in Document.prototype && 'replace' in CSSStyleSheet.prototype;
function getShadowRoot(element) {
    return element.shadowRoot || element.attachShadow({
        mode: 'open'
    });
}
module.exports = function(config, isProduction = false) {
    // 1. Create separate adoptable sheet
    let sheet = function() {
        // TODO: search for SSR stylesheet
        if (supportsAdoptingStyleSheets) try {
            // Try using adoptedStyleSheets — not supported in all browsers
            let sheet = core.cssom(new CSSStyleSheet());
            sheet.connect = (element)=>{
                let shadowRoot = getShadowRoot(element);
                shadowRoot.adoptedStyleSheets = [
                    ...shadowRoot.adoptedStyleSheets,
                    sheet.target
                ];
            };
            sheet.disconnect = core.noop;
            return sheet;
        } catch  {}
        // ignore — continue with fallback
        // Fallback: create a dedicated stylesheet for each element instance
        // Create our base stylesheet — its css rules will be copied to each element style
        let style = document.createElement('style');
        // Prevent this style sheet from being applied anywhere
        style.media = 'not all';
        document.head.prepend(style);
        let sheets = [
            core.cssom(style)
        ], /** @type {WeakMap<HTMLElement, import('@twind/core').Sheet>} */ sheetsByElement = new WeakMap();
        return {
            get target () {
                return sheets[0].target;
            },
            snapshot () {
                let restores = sheets.map((sheet)=>sheet.snapshot());
                return ()=>restores.forEach((restore)=>restore());
            },
            clear () {
                sheets.forEach((sheets)=>sheets.clear());
            },
            destroy () {
                sheets.forEach((sheets)=>sheets.destroy());
            },
            insert (css, index, rule) {
                // We first insert the rule into our base sheet
                // This call will check (try-catch) and warn if necessary
                sheets[0].insert(css, index, rule);
                // For all connected sheets we insert the resulting rule directly
                // by-passing the try-catch block
                let cssRule = this.target.cssRules[index];
                sheets.forEach((sheets, notFirst)=>notFirst && sheets.target.insertRule(cssRule.cssText, index));
            },
            resume (addClassName, insert) {
                return sheets[0].resume(addClassName, insert);
            },
            connect (element) {
                let style = document.createElement('style');
                getShadowRoot(element).appendChild(style);
                let sheet = core.cssom(style), // Directly copy all rules from our base sheet
                { cssRules  } = this.target;
                for(let i = 0; i < cssRules.length; i++)sheet.target.insertRule(cssRules[i].cssText, i);
                sheets.push(sheet);
                sheetsByElement.set(element, sheet);
            },
            disconnect (element) {
                let index = sheets.indexOf(sheetsByElement.get(element));
                index >= 0 && sheets.splice(index, 1);
            }
        };
    }(), // 2. Use that to create an own twind instance
    tw = core.twind({
        ...config,
        // in production use short hashed class names
        hash: config.hash ?? isProduction
    }, sheet), // 3. One MutationObserver for all element instances
    observer = core.mo(tw);
    return function install$(BaseElement) {
        return class WithTwindElement extends BaseElement {
            connectedCallback() {
                super.connectedCallback?.();
                sheet.connect(this);
                observer.observe(getShadowRoot(this));
            }
            disconnectedCallback() {
                sheet.disconnect(this);
                super.disconnectedCallback?.();
            }
            constructor(...args){
                super(...args);
                this.tw = tw;
            }
        };
    };
};
//# sourceMappingURL=with-web-components.dev.cjs.map
