#!/usr/bin/env node
"use strict";
var minimist = require('minimist');
var extend = require('xtend');
var chalk = require('chalk');
var Promise = require('any-promise');
var listify = require('listify');
var typings_1 = require('../typings');
var cli_1 = require('../utils/cli');
var config_1 = require('../utils/config');
var parse_1 = require('../utils/parse');
var registry_1 = require('../lib/registry');
var error_1 = require('../lib/error');
var args = minimist(process.argv.slice(2), {
    boolean: ['save', 'saveDev', 'ambient', 'verbose', 'help', 'production'],
    string: ['name', 'source'],
    alias: {
        save: ['S'],
        saveDev: ['save-dev', 'D'],
        name: ['n'],
        ambient: ['A'],
        verbose: ['v'],
        help: ['h'],
        source: ['s'],
        production: ['p']
    }
});
if (args.help) {
    console.log("\n" + config_1.PROJECT_NAME + " install (with no arguments, in package directory)\n" + config_1.PROJECT_NAME + " install <pkg>[@<version>] [ --source [" + Object.keys(registry_1.VALID_SOURCES).join(' | ') + "] ]\n" + config_1.PROJECT_NAME + " install file:<path>\n" + config_1.PROJECT_NAME + " install github:<github username>/<github project>[/<path>][#<commit>]\n" + config_1.PROJECT_NAME + " install bitbucket:<bitbucket username>/<bitbucket project>[/<path>][#<commit>]\n" + config_1.PROJECT_NAME + " install npm:<package>\n" + config_1.PROJECT_NAME + " install bower:<package>\n" + config_1.PROJECT_NAME + " install <https?:// url>\n\nAliases: i, in\nOptions: [--name] [--save|--save-dev] [--ambient] [--production]\n");
    process.exit(0);
}
function printResult(output, args) {
    if (output.references) {
        var references = Object.keys(output.references);
        if (references.length) {
            console.log(chalk.bold("Unused References:"));
            console.log();
            console.log('Typings has stripped some references during installation. You can install the stripped references by executing the following commands:');
            console.log();
            var flags = '';
            if (args.save) {
                flags += ' --save';
            }
            else if (args.saveDev) {
                flags += ' --save-dev';
            }
            if (args.ambient) {
                flags += ' --ambient';
            }
            for (var _i = 0, references_1 = references; _i < references_1.length; _i++) {
                var path = references_1[_i];
                console.log("typings install '" + chalk.bold(path) + "'" + flags + " " + chalk.dim("# From " + listify(output.references[path].map(function (x) { return JSON.stringify(x.name); }))));
            }
            console.log('');
        }
    }
    console.log(cli_1.archifyDependencyTree(output.tree, args));
}
function installer(args) {
    var cwd = process.cwd();
    var verbose = args.verbose, save = args.save, saveDev = args.saveDev, name = args.name, ambient = args.ambient, source = args.source, production = args.production;
    var options = { save: save, saveDev: saveDev, name: name, ambient: ambient, cwd: cwd, production: production };
    if (!args._.length) {
        return cli_1.loader(typings_1.install(options), args)
            .then(function (output) { return printResult(output, args); });
    }
    function installLocation(location, options) {
        function handle(options) {
            return cli_1.loader(typings_1.installDependency(location, options), args)
                .then(function (output) { return printResult(output, args); });
        }
        if (typeof options.name === 'string') {
            return handle(options);
        }
        return cli_1.inquire([{
                name: 'name',
                type: 'input',
                message: 'What is the dependency name?',
                default: parse_1.inferDependencyName(location)
            }])
            .then(function (answers) { return handle(extend(options, { name: answers.name })); });
    }
    var location = args._[0];
    if (!registry_1.isRegistryPath(location)) {
        return installLocation(location, options);
    }
    var _a = registry_1.parseRegistryPath(location), dependencyName = _a.name, version = _a.version;
    function installFrom(source) {
        var saveName = name || dependencyName;
        var sourceName = registry_1.VALID_SOURCES[source];
        return registry_1.getVersions(source, dependencyName, version)
            .then(function (project) {
            var versions = project.versions;
            var version = versions[0];
            console.log("Installing " + dependencyName + "@" + version.version + " (" + sourceName + ")...");
            if (name != null && name !== saveName) {
                console.log("Writing dependency as \"" + saveName + "\"...");
            }
            console.log('');
            return installLocation(version.location, extend(options, { name: saveName }));
        })
            .catch(function (err) { return cli_1.handleError(err, { verbose: verbose }); });
    }
    if (args.source) {
        return installFrom(args.source);
    }
    return cli_1.loader(registry_1.search({ name: dependencyName, ambient: ambient }), { verbose: verbose })
        .then(function (result) {
        var results = result.results;
        if (results.length === 0) {
            var prompt_1 = !ambient ? 'Did you want to install an ambient typing? Try using "--ambient". ' : '';
            return Promise.reject(new error_1.default(("Unable to find \"" + dependencyName + "\" in the registry. ") + prompt_1 +
                "If you can contribute this type definition, please help us: " +
                "https://github.com/typings/registry"));
        }
        if (results.length === 1) {
            var item_1 = results[0];
            var source_1 = registry_1.VALID_SOURCES[item_1.source];
            return cli_1.inquire([{
                    type: 'confirm',
                    name: 'ok',
                    message: "Found " + dependencyName + " typings for " + source_1 + ". Continue?"
                }])
                .then(function (answers) {
                if (answers.ok) {
                    return installFrom(item_1.source);
                }
            });
        }
        return cli_1.inquire([{
                type: 'list',
                name: 'source',
                message: "Found " + dependencyName + " typings for multiple registries",
                choices: results.map(function (result) {
                    return {
                        name: registry_1.VALID_SOURCES[result.source],
                        value: result.source
                    };
                })
            }])
            .then(function (answers) { return installFrom(answers.source); });
    })
        .catch(function (err) { return cli_1.handleError(err, { verbose: verbose }); });
}
installer(args);
//# sourceMappingURL=typings-install.js.map