"use strict";
var ts = require('typescript');
var extend = require('xtend');
var has = require('has');
var Promise = require('any-promise');
var os_1 = require('os');
var path_1 = require('path');
var fs_1 = require('../utils/fs');
var path_2 = require('../utils/path');
var references_1 = require('../utils/references');
var config_1 = require('../utils/config');
var parse_1 = require('../utils/parse');
var typings_1 = require('../typings');
var error_1 = require('./error');
function compile(tree, options) {
    var files = {};
    var moduleName = options.name;
    return Promise.all([
        compileDependencyTree(tree, extend(options, { browser: false, moduleName: moduleName, files: files })),
        compileDependencyTree(tree, extend(options, { browser: true, moduleName: moduleName, files: files }))
    ])
        .then(function (_a) {
        var main = _a[0], browser = _a[1];
        return {
            tree: tree,
            main: main.contents,
            browser: browser.contents,
            references: mergeReferences(main.references, browser.references)
        };
    });
}
Object.defineProperty(exports, "__esModule", { value: true });
exports.default = compile;
function mergeReferences(main, browser) {
    var map = {};
    function addEntry(entry, browser) {
        var path = entry.path, raw = entry.raw, src = entry.src, name = entry.name;
        var location = parse_1.resolveDependency(raw, path_2.relativeTo(src, path));
        var values = map[location] || (map[location] = []);
        for (var _i = 0, values_1 = values; _i < values_1.length; _i++) {
            var value = values_1[_i];
            if (value.name === name) {
                if (browser) {
                    value.browser = true;
                }
                else {
                    value.main = true;
                }
                return;
            }
        }
        values.push({
            name: name,
            main: !browser,
            browser: browser
        });
    }
    for (var _i = 0, main_1 = main; _i < main_1.length; _i++) {
        var entry = main_1[_i];
        addEntry(entry, false);
    }
    for (var _a = 0, browser_1 = browser; _a < browser_1.length; _a++) {
        var entry = browser_1[_a];
        addEntry(entry, true);
    }
    return map;
}
function resolveFromWithModuleName(src, to) {
    if (to == null || typeof to === 'boolean' || path_2.isModuleName(to)) {
        return to;
    }
    return path_2.resolveFrom(src, path_2.normalizeToDefinition(to));
}
function getStringifyOptions(tree, options, parent) {
    var overrides = {};
    var isTypings = typeof tree.typings === 'string';
    var main = isTypings ? tree.typings : tree.main;
    var browser = isTypings ? tree.browserTypings : tree.browser;
    if (options.browser && browser) {
        if (typeof browser === 'string') {
            var mainDefinition = path_2.resolveFrom(tree.src, path_2.normalizeToDefinition(main));
            var browserDefinition = path_2.resolveFrom(tree.src, path_2.normalizeToDefinition(browser));
            overrides[mainDefinition] = browserDefinition;
        }
        else {
            for (var _i = 0, _a = Object.keys(browser); _i < _a.length; _i++) {
                var key = _a[_i];
                var from = resolveFromWithModuleName(tree.src, key);
                var to = resolveFromWithModuleName(tree.src, browser[key]);
                overrides[from] = to;
            }
        }
    }
    var imported = {};
    var referenced = {};
    var dependencies = {};
    var entry = main == null ? main : path_2.resolveFrom(tree.src, path_2.normalizeToDefinition(main));
    return extend(options, {
        tree: tree,
        entry: entry,
        isTypings: isTypings,
        overrides: overrides,
        imported: imported,
        referenced: referenced,
        dependencies: dependencies,
        parent: parent
    });
}
function compileDependencyTree(tree, options) {
    return compileDependencyPath(null, getStringifyOptions(tree, options, undefined));
}
function compileDependencyPath(path, options) {
    var tree = options.tree, entry = options.entry;
    if (path == null) {
        if (entry == null) {
            return Promise.reject(new error_1.default(("Unable to resolve entry \".d.ts\" file for \"" + options.name + "\", ") +
                'please make sure the module has a main or typings field'));
        }
        return stringifyDependencyPath(path_2.resolveFrom(tree.src, entry), options);
    }
    return stringifyDependencyPath(path_2.resolveFrom(tree.src, path), options);
}
function cachedReadFileFrom(path, options) {
    if (!has(options.files, path)) {
        options.files[path] = fs_1.readFileFrom(path);
    }
    return options.files[path];
}
function cachedStringifyOptions(name, compileOptions, options) {
    var tree = getDependency(name, options);
    if (!has(options.dependencies, name)) {
        if (tree) {
            options.dependencies[name] = getStringifyOptions(tree, compileOptions, options);
        }
        else {
            options.dependencies[name] = null;
        }
    }
    return options.dependencies[name];
}
function getPath(path, options) {
    if (has(options.overrides, path)) {
        return options.overrides[path];
    }
    return path;
}
function getDependency(name, options) {
    var tree = options.tree, overrides = options.overrides;
    if (has(options.overrides, name)) {
        return tree.dependencies[overrides[name]];
    }
    if (has(tree.dependencies, name)) {
        return tree.dependencies[name];
    }
}
function stringifyDependencyPath(path, options) {
    var resolved = getPath(path_2.normalizeToDefinition(path), options);
    var tree = options.tree, ambient = options.ambient, cwd = options.cwd, browser = options.browser, name = options.name, files = options.files, meta = options.meta, entry = options.entry;
    var raw = tree.raw, src = tree.src;
    function loadByModuleName(path) {
        var _a = getModuleNameParts(path), moduleName = _a[0], modulePath = _a[1];
        var dependencyName = ambient ? moduleName : "" + name + config_1.DEPENDENCY_SEPARATOR + moduleName;
        var compileOptions = { cwd: cwd, browser: browser, moduleName: moduleName, files: files, name: dependencyName, ambient: false, meta: meta };
        var stringifyOptions = cachedStringifyOptions(moduleName, compileOptions, options);
        if (!stringifyOptions) {
            return Promise.resolve({
                contents: null,
                references: []
            });
        }
        return compileDependencyPath(modulePath, stringifyOptions);
    }
    if (path_2.isModuleName(resolved)) {
        return loadByModuleName(resolved);
    }
    return cachedReadFileFrom(resolved, options)
        .then(function (rawContents) {
        var info = ts.preProcessFile(rawContents);
        if (info.isLibFile) {
            return;
        }
        var importedFiles = info.importedFiles.map(function (x) { return resolveFromWithModuleName(resolved, x.fileName); });
        var referencedFiles = info.referencedFiles.map(function (x) { return path_2.resolveFrom(resolved, x.fileName); });
        var moduleAugmentations = (info.ambientExternalModules || []).map(function (x) { return resolveFromWithModuleName(resolved, x); });
        var ambientModules = moduleAugmentations.filter(function (x) { return importedFiles.indexOf(x) === -1; });
        if (ambientModules.length && !ambient) {
            return Promise.reject(new error_1.default(("Attempted to compile \"" + options.name + "\" as a dependency, but ") +
                "it contains some ambient module declarations " +
                ("(" + ambientModules.map(JSON.stringify).join(', ') + ").")));
        }
        if (ambient) {
            Object.keys(tree.dependencies).forEach(function (x) { return importedFiles.push(x); });
        }
        var imports = importedFiles.map(function (importedFile) {
            var path = getPath(importedFile, options);
            if (has(options.imported, path)) {
                return;
            }
            if (ambientModules.indexOf(path) > -1) {
                return;
            }
            options.imported[path] = true;
            if (path_2.isModuleName(path)) {
                return loadByModuleName(path);
            }
            return stringifyDependencyPath(path, options);
        });
        return Promise.all(imports)
            .then(function (imports) {
            var stringifyOptions = extend(options, { originalPath: path });
            var stringified = stringifyFile(resolved, rawContents, stringifyOptions);
            var references = referencedFiles.map(function (path) { return ({ name: name, path: path, raw: raw, src: src }); });
            var contents = [];
            for (var _i = 0, imports_1 = imports; _i < imports_1.length; _i++) {
                var imported = imports_1[_i];
                if (imported) {
                    references = references.concat(imported.references);
                    if (imported.contents != null) {
                        contents.push(imported.contents);
                    }
                }
            }
            contents.push(stringified);
            return {
                contents: contents.join(os_1.EOL + os_1.EOL),
                references: references
            };
        });
    }, function (cause) {
        var authorPhrase = options.parent ? "The author of \"" + options.parent.name + "\" needs to" : 'You should';
        var relativePath = path_2.relativeTo(tree.src, resolved);
        if (path === entry) {
            return Promise.reject(new error_1.default(("Unable to read typings for \"" + options.name + "\". ") +
                (authorPhrase + " check the path is correct"), cause));
        }
        return Promise.reject(new error_1.default(("Unable to read \"" + relativePath + "\" from \"" + options.name + "\". ") +
            (authorPhrase + " check the entry in \"" + config_1.CONFIG_FILE + "\" is correct"), cause));
    });
}
function getModuleNameParts(name) {
    var parts = name.split(/[\\\/]/);
    var moduleName = parts.shift();
    var modulePath = parts.length === 0 ? null : parts.join('/');
    return [moduleName, modulePath];
}
function stringifyFile(path, rawContents, options) {
    var contents = rawContents.replace(references_1.REFERENCE_REGEXP, '');
    var sourceFile = ts.createSourceFile(path, contents, ts.ScriptTarget.Latest, true);
    var tree = options.tree, name = options.name, originalPath = options.originalPath;
    var source = path_2.isHttp(path) ? path : path_1.relative(options.cwd, path);
    var prefix = options.meta ? "// Compiled using " + config_1.PROJECT_NAME + "@" + typings_1.VERSION + os_1.EOL + "// Source: " + source + os_1.EOL : '';
    if (options.ambient) {
        if (sourceFile.externalModuleIndicator) {
            throw new error_1.default(("Attempted to compile \"" + options.name + "\" as an ambient ") +
                "module, but it looks like an external module.");
        }
        return "" + prefix + contents.trim();
    }
    var wasDeclared = false;
    var hasExports = false;
    var hasDefaultExport = false;
    var hasExportEquals = false;
    function importPath(name) {
        var resolved = getPath(resolveFromWithModuleName(path, name), options);
        if (path_2.isModuleName(resolved)) {
            var moduleName = getModuleNameParts(resolved)[0];
            if (options.dependencies[moduleName] == null) {
                return name;
            }
            return "" + options.name + config_1.DEPENDENCY_SEPARATOR + resolved;
        }
        var relativePath = path_2.relativeTo(tree.src, path_2.fromDefinition(resolved));
        return path_2.normalizeSlashes(path_1.join(options.name, relativePath));
    }
    function replacer(node) {
        if (node.kind === ts.SyntaxKind.ExportAssignment) {
            hasDefaultExport = !node.isExportEquals;
            hasExportEquals = !hasDefaultExport;
        }
        else if (node.kind === ts.SyntaxKind.ExportDeclaration) {
            hasExports = true;
        }
        else {
            hasExports = hasExports || !!(node.flags & ts.NodeFlags.Export);
            hasDefaultExport = hasDefaultExport || !!(node.flags & ts.NodeFlags.Default);
        }
        if (node.kind === ts.SyntaxKind.StringLiteral &&
            (node.parent.kind === ts.SyntaxKind.ExportDeclaration ||
                node.parent.kind === ts.SyntaxKind.ImportDeclaration ||
                node.parent.kind === ts.SyntaxKind.ModuleDeclaration)) {
            return " '" + importPath(node.text) + "'";
        }
        if (node.kind === ts.SyntaxKind.DeclareKeyword) {
            wasDeclared = true;
            return sourceFile.text.slice(node.getFullStart(), node.getStart());
        }
        if (node.kind === ts.SyntaxKind.ExternalModuleReference) {
            var path_3 = importPath(node.expression.text);
            return " require('" + path_3 + "')";
        }
    }
    function read(start, end) {
        var text = sourceFile.text.slice(start, end);
        if (start === 0) {
            return text.replace(/^\s+$/, '');
        }
        if (end == null) {
            return text.replace(/\s+$/, '');
        }
        if (wasDeclared) {
            wasDeclared = false;
            return text.replace(/^\s+/, '');
        }
        return text;
    }
    var isEntry = originalPath === options.entry;
    var moduleText = processTree(sourceFile, replacer, read);
    if (isEntry && options.isTypings) {
        return prefix + declareText(name, moduleText);
    }
    var modulePath = importPath(path);
    var declared = declareText(modulePath, moduleText);
    if (!isEntry) {
        return prefix + declared;
    }
    var importParts = [];
    if (hasExportEquals) {
        importParts.push("import main = require('" + modulePath + "');");
        importParts.push("export = main;");
    }
    else {
        if (hasExports) {
            importParts.push("export * from '" + modulePath + "';");
        }
        if (hasDefaultExport) {
            importParts.push("export { default } from '" + modulePath + "';");
        }
    }
    return prefix + declared + os_1.EOL + declareText(name, importParts.join(os_1.EOL));
}
function declareText(name, text) {
    return "declare module '" + name + "' {" + (text ? os_1.EOL + text + os_1.EOL : '') + "}";
}
function processTree(sourceFile, replacer, reader) {
    var code = '';
    var position = 0;
    function skip(node) {
        position = node.end;
    }
    function readThrough(node) {
        if (node.pos > position) {
            code += reader(position, node.pos);
            position = node.pos;
        }
    }
    function visit(node) {
        readThrough(node);
        var replacement = replacer(node);
        if (replacement != null) {
            code += replacement;
            skip(node);
        }
        else {
            ts.forEachChild(node, visit);
        }
    }
    visit(sourceFile);
    code += reader(position);
    return code;
}
//# sourceMappingURL=compile.js.map