"use strict";
var extend = require('xtend');
var invariant = require('invariant');
var zipObject = require('zip-object');
var Promise = require('any-promise');
var path_1 = require('path');
var url_1 = require('url');
var fs_1 = require('../utils/fs');
var parse_1 = require('../utils/parse');
var find_1 = require('../utils/find');
var path_2 = require('../utils/path');
var config_1 = require('../utils/config');
var error_1 = require('./error');
var DEFAULT_DEPENDENCY = {
    src: undefined,
    raw: undefined,
    dependencies: {},
    devDependencies: {},
    ambientDependencies: {},
    ambientDevDependencies: {}
};
function resolveAllDependencies(options) {
    return Promise.all([
        resolveBowerDependencies(options).catch(function () { return extend(DEFAULT_DEPENDENCY); }),
        resolveNpmDependencies(options).catch(function () { return extend(DEFAULT_DEPENDENCY); }),
        resolveTypeDependencies(options).catch(function () { return extend(DEFAULT_DEPENDENCY); })
    ])
        .then(mergeDependencies);
}
exports.resolveAllDependencies = resolveAllDependencies;
function resolveDependency(dependency, options, parent) {
    if (dependency.type === 'npm') {
        return resolveNpmDependency(dependency.location, dependency.raw, options, parent);
    }
    if (dependency.type === 'bower') {
        return resolveBowerDependency(dependency.location, dependency.raw, options, parent);
    }
    return resolveFileDependency(dependency.location, dependency.raw, options, parent);
}
exports.resolveDependency = resolveDependency;
function resolveNpmDependency(name, raw, options, parent) {
    return find_1.findUp(options.cwd, path_1.join('node_modules', name))
        .then(function (modulePath) {
        if (path_2.isDefinition(modulePath)) {
            return resolveFileDependency(modulePath, raw, options, parent);
        }
        return resolveNpmDependencyFrom(modulePath, raw, options, parent);
    }, function (error) {
        return Promise.reject(resolveError(raw, error, parent));
    });
}
function resolveBowerDependency(name, raw, options, parent) {
    return resolveBowerComponentPath(options.cwd)
        .then(function (componentPath) {
        var modulePath = path_1.resolve(componentPath, name);
        if (path_2.isDefinition(modulePath)) {
            return resolveFileDependency(modulePath, raw, options, parent);
        }
        return resolveBowerDependencyFrom(modulePath, raw, componentPath, options, parent);
    }, function (error) {
        return Promise.reject(resolveError(raw, error, parent));
    });
}
function resolveFileDependency(location, raw, options, parent) {
    var src;
    if (path_2.isHttp(location)) {
        src = location;
    }
    else if (parent && path_2.isHttp(parent.src)) {
        src = url_1.resolve(parent.src, location);
    }
    else {
        src = path_1.resolve(options.cwd, location);
    }
    if (!path_2.isDefinition(src)) {
        return resolveTypeDependencyFrom(src, raw, options, parent);
    }
    return Promise.resolve(extend(DEFAULT_DEPENDENCY, {
        typings: src,
        src: src,
        raw: raw,
        parent: parent
    }));
}
function resolveBowerDependencies(options) {
    return find_1.findUp(options.cwd, 'bower.json')
        .then(function (bowerJsonPath) {
        return resolveBowerComponentPath(path_1.dirname(bowerJsonPath))
            .then(function (componentPath) {
            return resolveBowerDependencyFrom(bowerJsonPath, undefined, componentPath, options);
        });
    }, function (cause) {
        return Promise.reject(new error_1.default("Unable to resolve Bower dependencies", cause));
    });
}
exports.resolveBowerDependencies = resolveBowerDependencies;
function resolveBowerDependencyFrom(src, raw, componentPath, options, parent) {
    checkCircularDependency(parent, src);
    return fs_1.readJson(src)
        .then(function (bowerJson) {
        if (bowerJson === void 0) { bowerJson = {}; }
        var tree = extend(DEFAULT_DEPENDENCY, {
            name: bowerJson.name,
            version: bowerJson.version,
            main: bowerJson.main,
            browser: bowerJson.browser,
            typings: bowerJson.typings,
            browserTypings: bowerJson.browserTypings,
            src: src,
            raw: raw,
            parent: parent
        });
        var dependencyMap = extend(bowerJson.dependencies);
        var devDependencyMap = extend(options.dev ? bowerJson.devDependencies : {});
        return Promise.all([
            resolveBowerDependencyMap(componentPath, dependencyMap, options, tree),
            resolveBowerDependencyMap(componentPath, devDependencyMap, options, tree),
            maybeResolveTypeDependencyFrom(path_1.join(src, '..', config_1.CONFIG_FILE), raw, options, tree)
        ])
            .then(function (_a) {
            var dependencies = _a[0], devDependencies = _a[1], typedPackage = _a[2];
            tree.dependencies = extend(dependencies, typedPackage.dependencies);
            tree.devDependencies = extend(devDependencies, typedPackage.devDependencies);
            return tree;
        });
    }, function (error) {
        return Promise.reject(resolveError(raw, error, parent));
    });
}
function resolveBowerComponentPath(path) {
    return fs_1.readJson(path_1.resolve(path, '.bowerrc'))
        .then(function (bowerrc) {
        if (bowerrc === void 0) { bowerrc = {}; }
        return path_1.resolve(path, bowerrc.directory || 'bower_components');
    }, function () {
        return path_1.resolve(path, 'bower_components');
    });
}
function resolveBowerDependencyMap(componentPath, dependencies, options, parent) {
    var keys = Object.keys(dependencies);
    return Promise.all(keys.map(function (name) {
        var modulePath = path_1.resolve(componentPath, name, 'bower.json');
        var resolveOptions = extend(options, { dev: false, ambient: false });
        return resolveBowerDependencyFrom(modulePath, "bower:" + name, componentPath, resolveOptions, parent);
    }))
        .then(function (results) { return zipObject(keys, results); });
}
function resolveNpmDependencies(options) {
    return find_1.findUp(options.cwd, 'package.json')
        .then(function (packgeJsonPath) {
        return resolveNpmDependencyFrom(packgeJsonPath, undefined, options);
    }, function (cause) {
        return Promise.reject(new error_1.default("Unable to resolve NPM dependencies", cause));
    });
}
exports.resolveNpmDependencies = resolveNpmDependencies;
function resolveNpmDependencyFrom(src, raw, options, parent) {
    checkCircularDependency(parent, src);
    return fs_1.readJson(src)
        .then(function (packageJson) {
        if (packageJson === void 0) { packageJson = {}; }
        var tree = extend(DEFAULT_DEPENDENCY, {
            name: packageJson.name,
            version: packageJson.version,
            main: packageJson.main || 'index.js',
            browser: packageJson.browser,
            typings: packageJson.typings,
            browserTypings: packageJson.browserTypings,
            src: src,
            raw: raw,
            parent: parent
        });
        var dependencyMap = extend(packageJson.dependencies, packageJson.peerDependencies);
        var devDependencyMap = extend(options.dev ? packageJson.devDependencies : {});
        return Promise.all([
            resolveNpmDependencyMap(src, dependencyMap, options, tree),
            resolveNpmDependencyMap(src, devDependencyMap, options, tree),
            maybeResolveTypeDependencyFrom(path_1.join(src, '..', config_1.CONFIG_FILE), raw, options, tree)
        ])
            .then(function (_a) {
            var dependencies = _a[0], devDependencies = _a[1], typedPackage = _a[2];
            tree.dependencies = extend(dependencies, typedPackage.dependencies);
            tree.devDependencies = extend(devDependencies, typedPackage.devDependencies);
            return tree;
        });
    }, function (error) {
        return Promise.reject(resolveError(raw, error, parent));
    });
}
function resolveNpmDependencyMap(src, dependencies, options, parent) {
    var cwd = path_1.dirname(src);
    var keys = Object.keys(dependencies);
    return Promise.all(keys.map(function (name) {
        var resolveOptions = extend(options, { dev: false, ambient: false, cwd: cwd });
        return resolveNpmDependency(path_1.join(name, 'package.json'), "npm:" + name, resolveOptions, parent);
    }))
        .then(function (results) { return zipObject(keys, results); });
}
function resolveTypeDependencies(options) {
    return find_1.findConfigFile(options.cwd)
        .then(function (path) {
        return resolveTypeDependencyFrom(path, undefined, options);
    }, function (cause) {
        return Promise.reject(new error_1.default("Unable to resolve Typings dependencies", cause));
    });
}
exports.resolveTypeDependencies = resolveTypeDependencies;
function resolveTypeDependencyFrom(src, raw, options, parent) {
    checkCircularDependency(parent, src);
    return fs_1.readConfigFrom(src)
        .then(function (config) {
        var tree = extend(DEFAULT_DEPENDENCY, {
            name: config.name,
            main: config.main,
            browser: config.browser,
            typings: config.typings,
            browserTypings: config.browserTypings,
            type: config_1.PROJECT_NAME,
            src: src,
            raw: raw,
            parent: parent
        });
        var ambient = options.ambient, dev = options.dev;
        var dependencyMap = extend(config.dependencies);
        var devDependencyMap = extend(dev ? config.devDependencies : {});
        var ambientDependencyMap = extend(ambient ? config.ambientDependencies : {});
        var ambientDevDependencyMap = extend(ambient && dev ? config.ambientDevDependencies : {});
        return Promise.all([
            resolveTypeDependencyMap(src, dependencyMap, options, tree),
            resolveTypeDependencyMap(src, devDependencyMap, options, tree),
            resolveTypeDependencyMap(src, ambientDependencyMap, options, tree),
            resolveTypeDependencyMap(src, ambientDevDependencyMap, options, tree),
        ])
            .then(function (_a) {
            var dependencies = _a[0], devDependencies = _a[1], ambientDependencies = _a[2], ambientDevDependencies = _a[3];
            tree.dependencies = dependencies;
            tree.devDependencies = devDependencies;
            tree.ambientDependencies = ambientDependencies;
            tree.ambientDevDependencies = ambientDevDependencies;
            return tree;
        });
    }, function (error) {
        return Promise.reject(resolveError(raw, error, parent));
    });
}
function maybeResolveTypeDependencyFrom(src, raw, options, parent) {
    return resolveTypeDependencyFrom(src, raw, options, parent).catch(function () { return extend(DEFAULT_DEPENDENCY); });
}
function resolveTypeDependencyMap(src, dependencies, options, parent) {
    var cwd = path_1.dirname(src);
    var keys = Object.keys(dependencies);
    return Promise.all(keys.map(function (name) {
        var resolveOptions = extend(options, { dev: false, ambient: false, cwd: cwd });
        return resolveDependency(parse_1.parseDependency(dependencies[name]), resolveOptions, parent);
    }))
        .then(function (results) { return zipObject(keys, results); });
}
function checkCircularDependency(tree, filename) {
    if (tree) {
        var currentSrc = tree.src;
        do {
            invariant(tree.src !== filename, "Circular dependency detected using \"" + currentSrc + "\"");
        } while (tree = tree.parent);
    }
}
function resolveError(raw, cause, parent) {
    var message = "Unable to resolve " + (raw == null ? 'typings' : "\"" + raw + "\"");
    if (parent != null && parent.raw != null) {
        message += " from \"" + parent.raw + "\"";
    }
    return new error_1.default(message, cause);
}
function mergeDependencies(trees) {
    var dependency = extend(DEFAULT_DEPENDENCY);
    trees.forEach(function (dependencyTree) {
        if (dependencyTree == null) {
            return;
        }
        var name = dependencyTree.name, src = dependencyTree.src, main = dependencyTree.main, browser = dependencyTree.browser, typings = dependencyTree.typings, browserTypings = dependencyTree.browserTypings;
        if (typeof main === 'string' ||
            typeof browser === 'string' ||
            typeof typings === 'string' ||
            typeof browserTypings === 'string') {
            dependency.name = name;
            dependency.src = src;
            dependency.main = main;
            dependency.browser = browser;
            dependency.typings = typings;
            dependency.browserTypings = browserTypings;
        }
        dependency.dependencies = extend(dependency.dependencies, dependencyTree.dependencies);
        dependency.devDependencies = extend(dependency.devDependencies, dependencyTree.devDependencies);
        dependency.ambientDependencies = extend(dependency.ambientDependencies, dependencyTree.ambientDependencies);
        dependency.ambientDevDependencies = extend(dependency.ambientDevDependencies, dependencyTree.ambientDevDependencies);
    });
    return dependency;
}
//# sourceMappingURL=dependencies.js.map