"use strict";
var interface_1 = require("./interface");
var uiRouterNgModule_1 = require("./uiRouterNgModule");
var core_1 = require("@angular/core");
var common_1 = require("../common/common");
/**
 * Returns a function which lazy loads a nested module
 *
 * Use this function as a [[StateDeclaration.lazyLoad]] property to lazy load a state tree (an NgModule).
 *
 * @param path the path to the module source code.
 * @returns A function which takes a transition, then:
 *
 * - Gets the Injector (scoped properly for the destination state)
 * - Loads and creates the NgModule
 * - Finds the "replacement state" for the target state, and adds the new NgModule Injector to it (as a resolve)
 *
 * returns the new states array
 */
function loadNgModule(path) {
    /** Get the parent NgModule Injector (from resolves) */
    var getNg2Injector = function (transition) {
        return transition.injector().getAsync(interface_1.NG2_INJECTOR_TOKEN);
    };
    /**
     * Lazy loads the NgModule using the NgModuleFactoryLoader
     *
     * Use the parent NgModule's Injector to:
     * - Find the correct NgModuleFactoryLoader
     * - Load the new NgModuleFactory from the path string (async)
     * - Create the new NgModule
     */
    var createNg2Module = function (path, ng2Injector) {
        return ng2Injector.get(core_1.NgModuleFactoryLoader).load(path).then(function (factory) {
            return factory.create(ng2Injector);
        });
    };
    /**
     * Apply the Lazy Loaded NgModule's Injector to the newly loaded state tree.
     *
     * Lazy loading uses a placeholder state which is removed (and replaced) after the module is loaded.
     * The NgModule should include a state with the same name as the placeholder.
     *
     * Find the *newly loaded state* with the same name as the *placeholder state*.
     * The NgModule's Injector (and ComponentFactoryResolver) will be added to that state.
     * The Injector/Factory are used when creating Components for the `replacement` state and all its children.
     */
    function applyNgModuleToNewStates(transition, ng2Module) {
        var targetName = transition.to().name;
        var newStates = ng2Module.injector.get(uiRouterNgModule_1.UIROUTER_STATES_TOKEN).reduce(common_1.unnestR, []);
        var replacementState = newStates.find(function (state) { return state.name === targetName; });
        if (!replacementState) {
            throw new Error(("The module that was loaded from " + path + " should have a state named '" + targetName + "'") +
                (", but it only had: " + (newStates || []).map(function (s) { return s.name; }).join(', ')));
        }
        // Add the injector as a resolve.
        replacementState['_ngModuleInjector'] = ng2Module.injector;
        // Return states to be registered by the lazyLoadHook
        return { states: newStates };
    }
    return function (transition) { return getNg2Injector(transition)
        .then(function (injector) { return createNg2Module(path, injector); })
        .then(function (moduleRef) { return applyNgModuleToNewStates(transition, moduleRef); }); };
}
exports.loadNgModule = loadNgModule;
//# sourceMappingURL=lazyLoadNgModule.js.map