"use strict";
/** @module ng2 */ /** */
var core_1 = require("@angular/core");
var uiRouterNgModule_1 = require("./uiRouterNgModule");
var uiRouterConfig_1 = require("./uiRouterConfig");
var router_1 = require("../router");
var resolvable_1 = require("../resolve/resolvable");
var resolveContext_1 = require("../resolve/resolveContext");
/**
 * Returns a function which lazy loads a nested module
 *
 * Use this function as a [[StateDeclaration.lazyLoad]] property to lazy load a state tree (an NgModule).
 *
 * @param path the path to the module source code.
 * @returns A function which takes a transition, then:
 *
 * - Gets the Injector (scoped properly for the destination state)
 * - Loads and creates the NgModule
 * - Finds the "replacement state" for the target state, and adds the new NgModule Injector to it (as a resolve)
 *
 * returns the new states array
 */
function loadNgModule(path) {
    /** Get the parent NgModule Injector (from resolves) */
    var getNg2Injector = function (transition) {
        return transition.injector().getAsync(resolveContext_1.NATIVE_INJECTOR_TOKEN);
    };
    /**
     * Lazy loads the NgModule using the NgModuleFactoryLoader
     *
     * Use the parent NgModule's Injector to:
     * - Find the correct NgModuleFactoryLoader
     * - Load the new NgModuleFactory from the path string (async)
     * - Create the new NgModule
     */
    var createNg2Module = function (path, ng2Injector) {
        return ng2Injector.get(core_1.NgModuleFactoryLoader).load(path).then(function (factory) {
            return factory.create(ng2Injector);
        });
    };
    /**
     * Apply the UI-Router Modules found in the lazy loaded module.
     *
     * Apply the Lazy Loaded NgModule's newly created Injector to the right state in the state tree.
     *
     * Lazy loading uses a placeholder state which is removed (and replaced) after the module is loaded.
     * The NgModule should include a state with the same name as the placeholder.
     *
     * Find the *newly loaded state* with the same name as the *placeholder state*.
     * The NgModule's Injector (and ComponentFactoryResolver) will be added to that state.
     * The Injector/Factory are used when creating Components for the `replacement` state and all its children.
     */
    function loadUIRouterModules(transition, ng2Module) {
        var injector = ng2Module.injector;
        var parentInjector = ng2Module.injector['parent'];
        var uiRouter = injector.get(router_1.UIRouter);
        var originalName = transition.to().name;
        var originalState = uiRouter.stateRegistry.get(originalName);
        var rootModules = injector.get(uiRouterNgModule_1.UIROUTER_ROOT_MODULE);
        var parentRootModules = parentInjector.get(uiRouterNgModule_1.UIROUTER_ROOT_MODULE);
        var newRootModules = rootModules.filter(function (module) { return parentRootModules.indexOf(module) === -1; });
        if (newRootModules.length) {
            console.log(rootModules);
            throw new Error('Lazy loaded modules should not contain a UIRouterModule.forRoot() module');
        }
        var modules = injector.get(uiRouterNgModule_1.UIROUTER_MODULE_TOKEN);
        modules.forEach(function (module) { return uiRouterConfig_1.applyModuleConfig(uiRouter, injector, module); });
        var replacementState = uiRouter.stateRegistry.get(originalName);
        if (replacementState === originalState) {
            throw new Error("The module that was loaded from " + path + " should have a ui-router state named '" + originalName + "'");
        }
        // Supply the newly loaded states with the Injector from the lazy loaded NgModule
        replacementState.$$state().resolvables.push(resolvable_1.Resolvable.fromData(resolveContext_1.NATIVE_INJECTOR_TOKEN, injector));
        return {};
    }
    return function (transition) { return getNg2Injector(transition)
        .then(function (injector) { return createNg2Module(path, injector); })
        .then(function (moduleRef) { return loadUIRouterModules(transition, moduleRef); }); };
}
exports.loadNgModule = loadNgModule;
//# sourceMappingURL=lazyLoadNgModule.js.map