import Configuration from '../Configuration.js';
import TimezoneUtil from '../format/TimezoneUtil.js';
var aAllParts = [
        'year',
        'month',
        'day',
        'hour',
        'minute',
        'second',
        'fractionalSecond'
    ], rIsUTCString = /Z|GMT|:.*[\+|\-]|^([\+|\-]\d{2})?\d{4}(-\d{2}){0,2}$/, aWeekday = [
        'Sun',
        'Mon',
        'Tue',
        'Wed',
        'Thu',
        'Fri',
        'Sat'
    ], aMonths = [
        'Jan',
        'Feb',
        'Mar',
        'Apr',
        'May',
        'Jun',
        'Jul',
        'Aug',
        'Sep',
        'Oct',
        'Nov',
        'Dec'
    ], mWeekdayToDay = {
        Sun: 0,
        Mon: 1,
        Tue: 2,
        Wed: 3,
        Thu: 4,
        Fri: 5,
        Sat: 6
    };
function addLeadingZeros(iValue, iLength) {
    return (iValue < 0 ? '-' : '') + Math.abs(iValue).toString().padStart(iLength, '0');
}
function UI5Date(vDateParts, sTimezoneID) {
    var oDateInstance = UI5Date._createDateInstance(vDateParts);
    Object.defineProperties(this, {
        sTimezoneID: { value: sTimezoneID },
        oDate: {
            value: oDateInstance,
            writable: true
        },
        oDateParts: {
            value: undefined,
            writable: true
        }
    });
    if (isNaN(oDateInstance)) {
        return;
    }
    if (vDateParts.length > 1 || vDateParts.length === 1 && typeof vDateParts[0] === 'string' && !rIsUTCString.test(vDateParts[0])) {
        this._setParts(aAllParts, [
            oDateInstance.getFullYear(),
            oDateInstance.getMonth(),
            oDateInstance.getDate(),
            oDateInstance.getHours(),
            oDateInstance.getMinutes(),
            oDateInstance.getSeconds(),
            oDateInstance.getMilliseconds()
        ]);
    }
}
UI5Date.prototype = Object.create(Date.prototype, { constructor: { value: Date } });
UI5Date.prototype[Symbol.toStringTag] = 'Date';
UI5Date.prototype._getPart = function (sPart) {
    var iResult;
    if (isNaN(this.oDate)) {
        return NaN;
    }
    this.oDateParts = this.oDateParts || TimezoneUtil._getParts(this.oDate, this.sTimezoneID);
    if (sPart === 'weekday') {
        return mWeekdayToDay[this.oDateParts.weekday];
    }
    iResult = parseInt(this.oDateParts[sPart]);
    if (sPart === 'month') {
        iResult -= 1;
    } else if (sPart === 'year') {
        if (this.oDateParts.era === 'B') {
            iResult = 1 - iResult;
        }
    }
    return iResult;
};
UI5Date.prototype._setParts = function (aParts, aValues) {
    var i, oCurrentDateParts, oNewDateAsUTCTimestamp, iNewTimestamp, sPart, vValue, oDateParts = {}, iMaxLength = Math.min(aParts.length, aValues.length);
    if (iMaxLength === 0) {
        return this.setTime(NaN);
    }
    for (i = 0; i < iMaxLength; i += 1) {
        vValue = parseInt(+aValues[i]);
        sPart = aParts[i];
        if (isNaN(vValue)) {
            return this.setTime(NaN);
        }
        if (sPart === 'month') {
            vValue += 1;
        } else if (sPart === 'year') {
            if (vValue <= 0) {
                vValue = 1 - vValue;
                oDateParts.era = 'B';
            } else {
                oDateParts.era = 'A';
            }
        }
        oDateParts[sPart] = vValue.toString();
    }
    if (this.oDateParts) {
        oCurrentDateParts = this.oDateParts;
    } else if (isNaN(this.oDate)) {
        oCurrentDateParts = {
            day: '1',
            fractionalSecond: '0',
            hour: '0',
            minute: '0',
            month: '1',
            second: '0'
        };
    } else {
        oCurrentDateParts = TimezoneUtil._getParts(this.oDate, this.sTimezoneID);
    }
    oDateParts = Object.assign({}, oCurrentDateParts, oDateParts);
    oNewDateAsUTCTimestamp = TimezoneUtil._getDateFromParts(oDateParts);
    if (isNaN(oNewDateAsUTCTimestamp)) {
        return this.setTime(NaN);
    }
    iNewTimestamp = oNewDateAsUTCTimestamp.getTime() + TimezoneUtil.calculateOffset(oNewDateAsUTCTimestamp, this.sTimezoneID) * 1000;
    return this.setTime(iNewTimestamp);
};
UI5Date.prototype.clone = function () {
    return UI5Date.getInstance(this);
};
UI5Date.prototype.getDate = function () {
    return this._getPart('day');
};
UI5Date.prototype.getDay = function () {
    return this._getPart('weekday');
};
UI5Date.prototype.getFullYear = function () {
    return this._getPart('year');
};
UI5Date.prototype.getHours = function () {
    return this._getPart('hour');
};
UI5Date.prototype.getMilliseconds = function () {
    return this._getPart('fractionalSecond');
};
UI5Date.prototype.getMinutes = function () {
    return this._getPart('minute');
};
UI5Date.prototype.getMonth = function () {
    return this._getPart('month');
};
UI5Date.prototype.getSeconds = function () {
    return this._getPart('second');
};
UI5Date.prototype.getTimezoneOffset = function () {
    return TimezoneUtil.calculateOffset(this.oDate, this.sTimezoneID) / 60;
};
UI5Date.prototype.getYear = function () {
    return this._getPart('year') - 1900;
};
UI5Date.prototype.setDate = function (iDay) {
    return this._setParts(['day'], arguments);
};
UI5Date.prototype.setFullYear = function (iYear, iMonth, iDay) {
    return this._setParts([
        'year',
        'month',
        'day'
    ], arguments);
};
UI5Date.prototype.setHours = function (iHours, iMinutes, iSeconds, iMilliseconds) {
    return this._setParts([
        'hour',
        'minute',
        'second',
        'fractionalSecond'
    ], arguments);
};
UI5Date.prototype.setMilliseconds = function (iMilliseconds) {
    return this._setParts(['fractionalSecond'], arguments);
};
UI5Date.prototype.setMinutes = function (iMinutes, iSeconds, iMilliseconds) {
    return this._setParts([
        'minute',
        'second',
        'fractionalSecond'
    ], arguments);
};
UI5Date.prototype.setMonth = function (iMonth, iDay) {
    return this._setParts([
        'month',
        'day'
    ], arguments);
};
UI5Date.prototype.setSeconds = function (iSeconds, iMilliseconds) {
    return this._setParts([
        'second',
        'fractionalSecond'
    ], arguments);
};
UI5Date.prototype.setTime = function (iTime) {
    this.oDateParts = undefined;
    return this.oDate.setTime(iTime);
};
UI5Date.prototype.setYear = function (iYear) {
    return this._setParts(['year'], [parseInt(iYear) + 1900]);
};
UI5Date.prototype.toDateString = function () {
    if (isNaN(this.oDate)) {
        return this.oDate.toDateString();
    }
    return aWeekday[this.getDay()] + ' ' + aMonths[this.getMonth()] + ' ' + addLeadingZeros(this.getDate(), 2) + ' ' + addLeadingZeros(this.getFullYear(), 4);
};
UI5Date.prototype.toString = function () {
    if (isNaN(this.oDate)) {
        return this.oDate.toString();
    }
    return this.toDateString() + ' ' + this.toTimeString();
};
UI5Date.prototype.toTimeString = function () {
    var iHours, iMinutes, sSign, iTimeZoneOffset;
    if (isNaN(this.oDate)) {
        return this.oDate.toTimeString();
    }
    iTimeZoneOffset = this.getTimezoneOffset();
    sSign = iTimeZoneOffset > 0 ? '-' : '+';
    iHours = Math.floor(Math.abs(iTimeZoneOffset) / 60);
    iMinutes = Math.abs(iTimeZoneOffset) % 60;
    return addLeadingZeros(this.getHours(), 2) + ':' + addLeadingZeros(this.getMinutes(), 2) + ':' + addLeadingZeros(this.getSeconds(), 2) + ' GMT' + sSign + addLeadingZeros(iHours, 2) + addLeadingZeros(iMinutes, 2);
};
[
    'getTime',
    'getUTCDate',
    'getUTCDay',
    'getUTCFullYear',
    'getUTCHours',
    'getUTCMilliseconds',
    'getUTCMinutes',
    'getUTCMonth',
    'getUTCSeconds',
    'toGMTString',
    'toISOString',
    'toJSON',
    'toUTCString',
    'valueOf'
].forEach(function (sMethod) {
    UI5Date.prototype[sMethod] = function () {
        return this.oDate[sMethod].apply(this.oDate, arguments);
    };
});
[
    'toLocaleDateString',
    'toLocaleString',
    'toLocaleTimeString'
].forEach(function (sMethod) {
    UI5Date.prototype[sMethod] = function (sLocale, oOptions) {
        return this.oDate[sMethod](sLocale || Configuration.getLanguageTag(), Object.assign({ timeZone: this.sTimezoneID }, oOptions));
    };
});
[
    'setUTCDate',
    'setUTCFullYear',
    'setUTCHours',
    'setUTCMilliseconds',
    'setUTCMinutes',
    'setUTCMonth',
    'setUTCSeconds'
].forEach(function (sMethod) {
    UI5Date.prototype[sMethod] = function () {
        this.oDateParts = undefined;
        return this.oDate[sMethod].apply(this.oDate, arguments);
    };
});
UI5Date._createDateInstance = function (vParts) {
    if (vParts[0] instanceof Date) {
        vParts[0] = vParts[0].valueOf();
    }
    return new (Function.prototype.bind.apply(Date, [].concat.apply([null], vParts)))();
};
UI5Date.getInstance = function () {
    var sTimezone = Configuration.getTimezone();
    if (sTimezone !== TimezoneUtil.getLocalTimezone()) {
        return new UI5Date(arguments, sTimezone);
    }
    return UI5Date._createDateInstance(arguments);
};
UI5Date.checkDate = function (oDate) {
    if (isNaN(oDate.getTime())) {
        throw new Error('The given Date is not valid');
    }
    if (!(oDate instanceof UI5Date) && Configuration.getTimezone() !== TimezoneUtil.getLocalTimezone()) {
        throw new Error('Configured time zone requires the parameter \'oDate\' to be an instance of' + ' sap.ui.core.date.UI5Date');
    }
};
export default UI5Date;