import CalendarType from '../CalendarType.js';
import Locale from '../Locale.js';
import LocaleData from '../LocaleData.js';
import UI5Date from '../date/UI5Date.js';
import UniversalDate from '../date/UniversalDate.js';
import CalendarUtils from '../date/CalendarUtils.js';
import CalendarWeekNumbering from '../date/CalendarWeekNumbering.js';
import TimezoneUtil from './TimezoneUtil.js';
import deepEqual from '../../../base/util/deepEqual.js';
import formatMessage from '../../../base/strings/formatMessage.js';
import Log from '../../../base/Log.js';
import extend from '../../../base/util/extend.js';
import Configuration from '../Configuration.js';
var DateFormat = function () {
    throw new Error();
};
var mDateFormatTypes = {
    DATE: 'date',
    TIME: 'time',
    DATETIME: 'datetime',
    DATETIME_WITH_TIMEZONE: 'datetimeWithTimezone'
};
var mCldrDatePattern = {};
var checkTimezoneParameterType = function (sTimezone) {
    if (typeof sTimezone !== 'string' && !(sTimezone instanceof String) && sTimezone != null) {
        throw new TypeError('The given timezone must be a string.');
    }
};
DateFormat.oDateInfo = {
    type: mDateFormatTypes.DATE,
    oDefaultFormatOptions: {
        style: 'medium',
        relativeScale: 'day',
        relativeStyle: 'wide'
    },
    aFallbackFormatOptions: [
        { style: 'short' },
        { style: 'medium' },
        { pattern: 'yyyy-MM-dd' },
        {
            pattern: 'yyyyMMdd',
            strictParsing: true
        }
    ],
    bShortFallbackFormatOptions: true,
    bPatternFallbackWithoutDelimiter: true,
    getPattern: function (oLocaleData, sStyle, sCalendarType) {
        return oLocaleData.getDatePattern(sStyle, sCalendarType);
    },
    oRequiredParts: {
        'text': true,
        'year': true,
        'weekYear': true,
        'month': true,
        'day': true
    },
    aRelativeScales: [
        'year',
        'month',
        'week',
        'day'
    ],
    aRelativeParseScales: [
        'year',
        'quarter',
        'month',
        'week',
        'day',
        'hour',
        'minute',
        'second'
    ],
    aIntervalCompareFields: [
        'Era',
        'FullYear',
        'Quarter',
        'Month',
        'Week',
        'Date'
    ]
};
DateFormat.oDateTimeInfo = {
    type: mDateFormatTypes.DATETIME,
    oDefaultFormatOptions: {
        style: 'medium',
        relativeScale: 'auto',
        relativeStyle: 'wide'
    },
    aFallbackFormatOptions: [
        { style: 'short' },
        { style: 'medium' },
        { pattern: 'yyyy-MM-dd\'T\'HH:mm:ss' },
        { pattern: 'yyyyMMdd HHmmss' }
    ],
    getPattern: function (oLocaleData, sStyle, sCalendarType) {
        var iSlashIndex = sStyle.indexOf('/');
        if (iSlashIndex > 0) {
            return oLocaleData.getCombinedDateTimePattern(sStyle.substr(0, iSlashIndex), sStyle.substr(iSlashIndex + 1), sCalendarType);
        } else {
            return oLocaleData.getCombinedDateTimePattern(sStyle, sStyle, sCalendarType);
        }
    },
    oRequiredParts: {
        'text': true,
        'year': true,
        'weekYear': true,
        'month': true,
        'day': true,
        'hour0_23': true,
        'hour1_24': true,
        'hour0_11': true,
        'hour1_12': true
    },
    aRelativeScales: [
        'year',
        'month',
        'week',
        'day',
        'hour',
        'minute',
        'second'
    ],
    aRelativeParseScales: [
        'year',
        'quarter',
        'month',
        'week',
        'day',
        'hour',
        'minute',
        'second'
    ],
    aIntervalCompareFields: [
        'Era',
        'FullYear',
        'Quarter',
        'Month',
        'Week',
        'Date',
        'DayPeriod',
        'Hours',
        'Minutes',
        'Seconds'
    ]
};
DateFormat._getDateTimeWithTimezoneInfo = function (oFormatOptions) {
    var bShowDate = oFormatOptions.showDate === undefined || oFormatOptions.showDate;
    var bShowTime = oFormatOptions.showTime === undefined || oFormatOptions.showTime;
    var bShowTimezone = oFormatOptions.showTimezone === undefined || oFormatOptions.showTimezone;
    var oBaselineType = DateFormat.oDateTimeInfo;
    if (bShowDate && !bShowTime) {
        oBaselineType = DateFormat.oDateInfo;
    } else if (!bShowDate && bShowTime) {
        oBaselineType = DateFormat.oTimeInfo;
    }
    return Object.assign({}, oBaselineType, {
        type: mDateFormatTypes.DATETIME_WITH_TIMEZONE,
        getTimezonePattern: function (sPattern) {
            if (!bShowDate && !bShowTime && bShowTimezone) {
                return 'VV';
            } else if (!bShowTimezone) {
                return sPattern;
            } else {
                return sPattern + ' VV';
            }
        },
        getPattern: function (oLocaleData, sStyle, sCalendarType) {
            if (!bShowDate && !bShowTime && bShowTimezone) {
                return 'VV';
            }
            if (!bShowTimezone) {
                return oBaselineType.getPattern(oLocaleData, sStyle, sCalendarType);
            }
            var sPattern = oBaselineType.getPattern(oLocaleData, sStyle, sCalendarType);
            return oLocaleData.applyTimezonePattern(sPattern);
        }
    });
};
DateFormat.oTimeInfo = {
    type: mDateFormatTypes.TIME,
    oDefaultFormatOptions: {
        style: 'medium',
        relativeScale: 'auto',
        relativeStyle: 'wide'
    },
    aFallbackFormatOptions: [
        { style: 'short' },
        { style: 'medium' },
        { pattern: 'HH:mm:ss' },
        { pattern: 'HHmmss' }
    ],
    getPattern: function (oLocaleData, sStyle, sCalendarType) {
        return oLocaleData.getTimePattern(sStyle, sCalendarType);
    },
    oRequiredParts: {
        'text': true,
        'hour0_23': true,
        'hour1_24': true,
        'hour0_11': true,
        'hour1_12': true
    },
    aRelativeScales: [
        'hour',
        'minute',
        'second'
    ],
    aRelativeParseScales: [
        'year',
        'quarter',
        'month',
        'week',
        'day',
        'hour',
        'minute',
        'second'
    ],
    aIntervalCompareFields: [
        'DayPeriod',
        'Hours',
        'Minutes',
        'Seconds'
    ]
};
DateFormat.getInstance = function (oFormatOptions, oLocale) {
    return this.getDateInstance(oFormatOptions, oLocale);
};
DateFormat.getDateInstance = function (oFormatOptions, oLocale) {
    return this.createInstance(oFormatOptions, oLocale, this.oDateInfo);
};
DateFormat.getDateTimeInstance = function (oFormatOptions, oLocale) {
    return this.createInstance(oFormatOptions, oLocale, this.oDateTimeInfo);
};
DateFormat.getDateTimeWithTimezoneInstance = function (oFormatOptions, oLocale) {
    if (oFormatOptions && !(oFormatOptions instanceof Locale)) {
        oFormatOptions = Object.assign({}, oFormatOptions);
        if (typeof oFormatOptions.showTimezone === 'string') {
            var sShowTimezone = oFormatOptions.showTimezone;
            if (oFormatOptions.showDate === undefined && oFormatOptions.showTime === undefined) {
                if (sShowTimezone === 'Hide') {
                    oFormatOptions.showTimezone = false;
                } else if (sShowTimezone === 'Only') {
                    oFormatOptions.showDate = false;
                    oFormatOptions.showTime = false;
                }
            }
            oFormatOptions.showTimezone = sShowTimezone !== 'Hide';
        }
        if (oFormatOptions.showDate === false && oFormatOptions.showTime === false && oFormatOptions.showTimezone === false) {
            throw new TypeError('Invalid Configuration. One of the following format options must be true: showDate, showTime or showTimezone.');
        }
    }
    return this.createInstance(oFormatOptions, oLocale, DateFormat._getDateTimeWithTimezoneInfo(oFormatOptions || {}));
};
DateFormat.getTimeInstance = function (oFormatOptions, oLocale) {
    return this.createInstance(oFormatOptions, oLocale, this.oTimeInfo);
};
function createIntervalPatternWithNormalConnector(oFormat) {
    var sPattern = oFormat.oLocaleData.getIntervalPattern('', oFormat.oFormatOptions.calendarType);
    sPattern = sPattern.replace(/[^\{\}01 ]/, '-');
    return sPattern.replace(/\{(0|1)\}/g, oFormat.oFormatOptions.pattern);
}
DateFormat.createInstance = function (oFormatOptions, oLocale, oInfo, bIsFallback) {
    var aFallbackFormatOptions, oFormat, sPattern;
    oFormat = Object.create(this.prototype);
    if (oFormatOptions instanceof Locale) {
        oLocale = oFormatOptions;
        oFormatOptions = undefined;
    }
    if (!oLocale) {
        oLocale = Configuration.getFormatSettings().getFormatLocale();
    }
    oFormat.oLocale = oLocale;
    oFormat.oLocaleData = LocaleData.getInstance(oLocale);
    oFormat.oFormatOptions = extend({}, oInfo.oDefaultFormatOptions, oFormatOptions);
    if (oInfo.type === mDateFormatTypes.DATETIME_WITH_TIMEZONE) {
        oFormat.oFormatOptions.interval = false;
        oFormat.oFormatOptions.singleIntervalValue = false;
        oFormat.oFormatOptions.UTC = false;
    } else {
        oFormat.oFormatOptions.showTimezone = undefined;
        oFormat.oFormatOptions.showDate = undefined;
        oFormat.oFormatOptions.showTime = undefined;
    }
    oFormat.type = oInfo.type;
    if (!oFormat.oFormatOptions.calendarType) {
        oFormat.oFormatOptions.calendarType = Configuration.getCalendarType();
    }
    if (oFormat.oFormatOptions.firstDayOfWeek === undefined && oFormat.oFormatOptions.minimalDaysInFirstWeek !== undefined || oFormat.oFormatOptions.firstDayOfWeek !== undefined && oFormat.oFormatOptions.minimalDaysInFirstWeek === undefined) {
        throw new TypeError('Format options firstDayOfWeek and minimalDaysInFirstWeek need both to be set, but only one was provided.');
    }
    if (oFormat.oFormatOptions.calendarWeekNumbering && !Object.values(CalendarWeekNumbering).includes(oFormat.oFormatOptions.calendarWeekNumbering)) {
        throw new TypeError('Illegal format option calendarWeekNumbering: \'' + oFormat.oFormatOptions.calendarWeekNumbering + '\'');
    }
    if (!oFormat.oFormatOptions.pattern) {
        if (oFormat.oFormatOptions.format) {
            oFormat.oFormatOptions.pattern = oFormat.oLocaleData.getCustomDateTimePattern(oFormat.oFormatOptions.format, oFormat.oFormatOptions.calendarType);
        } else {
            oFormat.oFormatOptions.pattern = oInfo.getPattern(oFormat.oLocaleData, oFormat.oFormatOptions.style, oFormat.oFormatOptions.calendarType);
        }
    }
    if (oFormat.oFormatOptions.interval) {
        if (oFormat.oFormatOptions.format) {
            oFormat.intervalPatterns = oFormat.oLocaleData.getCustomIntervalPattern(oFormat.oFormatOptions.format, null, oFormat.oFormatOptions.calendarType);
            if (typeof oFormat.intervalPatterns === 'string') {
                oFormat.intervalPatterns = [oFormat.intervalPatterns];
            }
            oFormat.intervalPatterns.push(oFormat.oLocaleData.getCustomDateTimePattern(oFormat.oFormatOptions.format, oFormat.oFormatOptions.calendarType));
        } else {
            oFormat.intervalPatterns = [
                oFormat.oLocaleData.getCombinedIntervalPattern(oFormat.oFormatOptions.pattern, oFormat.oFormatOptions.calendarType),
                oFormat.oFormatOptions.pattern
            ];
        }
        var sCommonConnectorPattern = createIntervalPatternWithNormalConnector(oFormat);
        oFormat.intervalPatterns.push(sCommonConnectorPattern);
    }
    if (!bIsFallback) {
        aFallbackFormatOptions = oInfo.aFallbackFormatOptions;
        if (oInfo.bShortFallbackFormatOptions) {
            sPattern = oInfo.getPattern(oFormat.oLocaleData, 'short');
            aFallbackFormatOptions = aFallbackFormatOptions.concat(DateFormat._createFallbackOptionsWithoutDelimiter(sPattern));
        }
        if (oFormat.oFormatOptions.pattern && oInfo.bPatternFallbackWithoutDelimiter) {
            aFallbackFormatOptions = DateFormat._createFallbackOptionsWithoutDelimiter(oFormat.oFormatOptions.pattern).concat(aFallbackFormatOptions);
        }
        aFallbackFormatOptions = aFallbackFormatOptions.reduce(function (aFallbacks, oOptions) {
            var aKeys = Object.keys(oOptions), bDuplicate = aFallbacks.some(function (oOptions0) {
                    return Object.keys(oOptions0).length === aKeys.length && aKeys.every(function (sKey) {
                        return oOptions0[sKey] === oOptions[sKey];
                    });
                });
            if (!bDuplicate) {
                aFallbacks.push(oOptions);
            }
            return aFallbacks;
        }, []);
        oFormat.aFallbackFormats = DateFormat._createFallbackFormat(aFallbackFormatOptions, oFormat.oFormatOptions.calendarType, oLocale, oInfo, oFormat.oFormatOptions);
    }
    oFormat.oRequiredParts = oInfo.oRequiredParts;
    oFormat.aRelativeScales = oInfo.aRelativeScales;
    oFormat.aRelativeParseScales = oInfo.aRelativeParseScales;
    oFormat.aIntervalCompareFields = oInfo.aIntervalCompareFields;
    oFormat.init();
    return oFormat;
};
DateFormat.prototype.init = function () {
    var sCalendarType = this.oFormatOptions.calendarType;
    this.aMonthsAbbrev = this.oLocaleData.getMonths('abbreviated', sCalendarType);
    this.aMonthsWide = this.oLocaleData.getMonths('wide', sCalendarType);
    this.aMonthsNarrow = this.oLocaleData.getMonths('narrow', sCalendarType);
    this.aMonthsAbbrevSt = this.oLocaleData.getMonthsStandAlone('abbreviated', sCalendarType);
    this.aMonthsWideSt = this.oLocaleData.getMonthsStandAlone('wide', sCalendarType);
    this.aMonthsNarrowSt = this.oLocaleData.getMonthsStandAlone('narrow', sCalendarType);
    this.aDaysAbbrev = this.oLocaleData.getDays('abbreviated', sCalendarType);
    this.aDaysWide = this.oLocaleData.getDays('wide', sCalendarType);
    this.aDaysNarrow = this.oLocaleData.getDays('narrow', sCalendarType);
    this.aDaysShort = this.oLocaleData.getDays('short', sCalendarType);
    this.aDaysAbbrevSt = this.oLocaleData.getDaysStandAlone('abbreviated', sCalendarType);
    this.aDaysWideSt = this.oLocaleData.getDaysStandAlone('wide', sCalendarType);
    this.aDaysNarrowSt = this.oLocaleData.getDaysStandAlone('narrow', sCalendarType);
    this.aDaysShortSt = this.oLocaleData.getDaysStandAlone('short', sCalendarType);
    this.aQuartersAbbrev = this.oLocaleData.getQuarters('abbreviated', sCalendarType);
    this.aQuartersWide = this.oLocaleData.getQuarters('wide', sCalendarType);
    this.aQuartersNarrow = this.oLocaleData.getQuarters('narrow', sCalendarType);
    this.aQuartersAbbrevSt = this.oLocaleData.getQuartersStandAlone('abbreviated', sCalendarType);
    this.aQuartersWideSt = this.oLocaleData.getQuartersStandAlone('wide', sCalendarType);
    this.aQuartersNarrowSt = this.oLocaleData.getQuartersStandAlone('narrow', sCalendarType);
    this.aErasNarrow = this.oLocaleData.getEras('narrow', sCalendarType);
    this.aErasAbbrev = this.oLocaleData.getEras('abbreviated', sCalendarType);
    this.aErasWide = this.oLocaleData.getEras('wide', sCalendarType);
    this.aDayPeriodsAbbrev = this.oLocaleData.getDayPeriods('abbreviated', sCalendarType);
    this.aDayPeriodsNarrow = this.oLocaleData.getDayPeriods('narrow', sCalendarType);
    this.aDayPeriodsWide = this.oLocaleData.getDayPeriods('wide', sCalendarType);
    this.oFlexibleDayPeriodsAbbrev = this.oLocaleData.getFlexibleDayPeriods('abbreviated', sCalendarType);
    this.oFlexibleDayPeriodsNarrow = this.oLocaleData.getFlexibleDayPeriods('narrow', sCalendarType);
    this.oFlexibleDayPeriodsWide = this.oLocaleData.getFlexibleDayPeriods('wide', sCalendarType);
    this.oFlexibleDayPeriodsAbbrevSt = this.oLocaleData.getFlexibleDayPeriodsStandAlone('abbreviated', sCalendarType);
    this.oFlexibleDayPeriodsNarrowSt = this.oLocaleData.getFlexibleDayPeriodsStandAlone('narrow', sCalendarType);
    this.oFlexibleDayPeriodsWideSt = this.oLocaleData.getFlexibleDayPeriodsStandAlone('wide', sCalendarType);
    this.aFormatArray = this.parseCldrDatePattern(this.oFormatOptions.pattern);
    this.sAllowedCharacters = this.getAllowedCharacters(this.aFormatArray);
};
DateFormat._createFallbackFormat = function (aFallbackFormatOptions, sCalendarType, oLocale, oInfo, oParentFormatOptions) {
    return aFallbackFormatOptions.map(function (oOptions) {
        var oFormatOptions = Object.assign({}, oOptions);
        oFormatOptions.showDate = oParentFormatOptions.showDate;
        oFormatOptions.showTime = oParentFormatOptions.showTime;
        oFormatOptions.showTimezone = oParentFormatOptions.showTimezone;
        if (typeof oInfo.getTimezonePattern === 'function' && oFormatOptions.pattern) {
            oFormatOptions.pattern = oInfo.getTimezonePattern(oFormatOptions.pattern);
        }
        if (oParentFormatOptions.interval) {
            oFormatOptions.interval = true;
        }
        oFormatOptions.calendarType = sCalendarType;
        return DateFormat.createInstance(oFormatOptions, oLocale, oInfo, true);
    });
};
DateFormat._createFallbackOptionsWithoutDelimiter = function (sBasePattern) {
    var rNonDateFields = /[^dMyGU]/g, oDayReplace = {
            regex: /d+/g,
            replace: 'dd'
        }, oMonthReplace = {
            regex: /M+/g,
            replace: 'MM'
        }, oYearReplace = {
            regex: /[yU]+/g,
            replace: [
                'yyyy',
                'yy'
            ]
        };
    sBasePattern = sBasePattern.replace(rNonDateFields, '');
    sBasePattern = sBasePattern.replace(oDayReplace.regex, oDayReplace.replace);
    sBasePattern = sBasePattern.replace(oMonthReplace.regex, oMonthReplace.replace);
    return oYearReplace.replace.map(function (sReplace) {
        return {
            pattern: sBasePattern.replace(oYearReplace.regex, sReplace),
            strictParsing: true
        };
    });
};
var oParseHelper = {
    isNumber: function (iCharCode) {
        return iCharCode >= 48 && iCharCode <= 57;
    },
    findNumbers: function (sValue, iMaxLength) {
        var iLength = 0;
        while (iLength < iMaxLength && this.isNumber(sValue.charCodeAt(iLength))) {
            iLength++;
        }
        return sValue.substr(0, iLength);
    },
    startsWithIgnoreCase: function (sValue, sSubstring, sLocale) {
        if (sValue.startsWith(sSubstring)) {
            return true;
        }
        try {
            var sSubToLocaleUpperCase = sSubstring.toLocaleUpperCase(sLocale);
            var sValueUpperCase = sValue.toLocaleUpperCase(sLocale);
            if (sSubToLocaleUpperCase.length !== sSubstring.length || sValueUpperCase.length !== sValue.length) {
                return false;
            }
            return sValueUpperCase.startsWith(sSubToLocaleUpperCase);
        } catch (e) {
            return false;
        }
    },
    findEntry: function (sValue, aList, sLocale) {
        var iFoundIndex = -1, iMatchedLength = 0;
        for (var j = 0; j < aList.length; j++) {
            if (aList[j] && aList[j].length > iMatchedLength && this.startsWithIgnoreCase(sValue, aList[j], sLocale)) {
                iFoundIndex = j;
                iMatchedLength = aList[j].length;
            }
        }
        return {
            index: iFoundIndex,
            length: iMatchedLength
        };
    },
    parseTZ: function (sValue, bColonSeparated) {
        var iLength = 0;
        var iTZFactor = sValue.charAt(0) === '+' ? -1 : 1;
        var sPart;
        iLength++;
        sPart = this.findNumbers(sValue.substr(iLength), 2);
        var iTZDiffHour = parseInt(sPart);
        iLength += 2;
        if (bColonSeparated) {
            iLength++;
        }
        sPart = this.findNumbers(sValue.substr(iLength), 2);
        var iTZDiff = 0;
        if (sPart) {
            iLength += 2;
            iTZDiff = parseInt(sPart);
        }
        return {
            length: iLength,
            tzDiff: (iTZDiff + 60 * iTZDiffHour) * 60 * iTZFactor
        };
    },
    checkValid: function (sSymbolName, bPartInvalid, oFormat) {
        if (sSymbolName in oFormat.oRequiredParts && bPartInvalid) {
            return false;
        }
        return true;
    }
};
DateFormat._createPatternSymbol = function (mParameters) {
    var fnIsNumeric = typeof mParameters.isNumeric === 'function' && mParameters.isNumeric || function () {
        return mParameters.isNumeric || false;
    };
    return {
        name: mParameters.name,
        format: mParameters.format || function () {
            return '';
        },
        parse: mParameters.parse || function () {
            return {};
        },
        isNumeric: fnIsNumeric
    };
};
DateFormat.prototype.oSymbols = {
    '': DateFormat._createPatternSymbol({
        name: 'text',
        format: function (oField, oDate) {
            return oField.value;
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var sChar;
            var bValid = true;
            var iValueIndex = 0;
            var iPatternIndex = 0;
            var sDelimiter = '-~\u2010\u2011\u2012\u2013\u2014\uFE58\uFE63\uFF0D\uFF5E';
            for (; iPatternIndex < oPart.value.length; iPatternIndex++) {
                sChar = oPart.value.charAt(iPatternIndex);
                if (sChar === ' ') {
                    while (sValue.charAt(iValueIndex) === ' ') {
                        iValueIndex++;
                    }
                } else if (sDelimiter.includes(sChar)) {
                    if (!sDelimiter.includes(sValue.charAt(iValueIndex))) {
                        bValid = false;
                    }
                    iValueIndex++;
                } else {
                    if (sValue.charAt(iValueIndex) !== sChar) {
                        bValid = false;
                    }
                    iValueIndex++;
                }
                if (!bValid) {
                    break;
                }
            }
            if (bValid) {
                return { length: iValueIndex };
            } else {
                var bPartInvalid = false;
                if (oConfig.index < oConfig.formatArray.length - 1) {
                    bPartInvalid = oConfig.formatArray[oConfig.index + 1].type in oFormat.oRequiredParts;
                }
                return { valid: oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat) };
            }
        }
    }),
    'G': DateFormat._createPatternSymbol({
        name: 'era',
        format: function (oField, oDate, bUTC, oFormat) {
            var iEra = oDate.getUTCEra();
            if (oField.digits <= 3) {
                return oFormat.aErasAbbrev[iEra];
            } else if (oField.digits === 4) {
                return oFormat.aErasWide[iEra];
            } else {
                return oFormat.aErasNarrow[iEra];
            }
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var aErasVariants = [
                oFormat.aErasWide,
                oFormat.aErasAbbrev,
                oFormat.aErasNarrow
            ];
            for (var i = 0; i < aErasVariants.length; i++) {
                var aVariants = aErasVariants[i];
                var oFound = oParseHelper.findEntry(sValue, aVariants, oFormat.oLocaleData.sCLDRLocaleId);
                if (oFound.index !== -1) {
                    return {
                        era: oFound.index,
                        length: oFound.length
                    };
                }
            }
            return {
                era: oFormat.aErasWide.length - 1,
                valid: oParseHelper.checkValid(oPart.type, true, oFormat)
            };
        }
    }),
    'y': DateFormat._createPatternSymbol({
        name: 'year',
        format: function (oField, oDate, bUTC, oFormat) {
            var iYear = oDate.getUTCFullYear();
            var sYear = String(iYear);
            var sCalendarType = oFormat.oFormatOptions.calendarType;
            if (oField.digits === 2 && sYear.length > 2) {
                sYear = sYear.substr(sYear.length - 2);
            }
            if (sCalendarType !== CalendarType.Japanese && oField.digits === 1 && iYear < 100) {
                sYear = sYear.padStart(4, '0');
            }
            return sYear.padStart(oField.digits, '0');
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var iExpectedDigits, sPart, bPartInvalid, sCalendarType = oFormat.oFormatOptions.calendarType;
            if (oPart.digits === 1) {
                iExpectedDigits = 4;
            } else if (oPart.digits === 2) {
                iExpectedDigits = 2;
            } else {
                iExpectedDigits = oPart.digits;
            }
            sPart = oParseHelper.findNumbers(sValue, iExpectedDigits);
            bPartInvalid = sPart === '' || oConfig.exactLength && sPart.length !== iExpectedDigits;
            var iYear = parseInt(sPart);
            if (sCalendarType !== CalendarType.Japanese && sPart.length <= 2) {
                var oCurrentDate = UniversalDate.getInstance(UI5Date.getInstance(), sCalendarType), iCurrentYear = oCurrentDate.getUTCFullYear(), iCurrentCentury = Math.floor(iCurrentYear / 100), iYearDiff = iCurrentCentury * 100 + iYear - iCurrentYear;
                if (iYearDiff < -70) {
                    iYear += (iCurrentCentury + 1) * 100;
                } else if (iYearDiff < 30) {
                    iYear += iCurrentCentury * 100;
                } else {
                    iYear += (iCurrentCentury - 1) * 100;
                }
            }
            return {
                length: sPart.length,
                valid: oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat),
                year: iYear
            };
        },
        isNumeric: true
    }),
    'Y': DateFormat._createPatternSymbol({
        name: 'weekYear',
        format: function (oField, oDate, bUTC, oFormat) {
            var oWeek = oDate.getUTCWeek(oFormat.oLocale, getCalendarWeekParameter(oFormat.oFormatOptions));
            var iWeekYear = oWeek.year;
            var sWeekYear = String(iWeekYear);
            var sCalendarType = oFormat.oFormatOptions.calendarType;
            if (oField.digits === 2 && sWeekYear.length > 2) {
                sWeekYear = sWeekYear.substr(sWeekYear.length - 2);
            }
            if (sCalendarType !== CalendarType.Japanese && oField.digits === 1 && iWeekYear < 100) {
                sWeekYear = sWeekYear.padStart(4, '0');
            }
            return sWeekYear.padStart(oField.digits, '0');
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var iExpectedDigits, sPart, bPartInvalid, sCalendarType = oFormat.oFormatOptions.calendarType;
            if (oPart.digits === 1) {
                iExpectedDigits = 4;
            } else if (oPart.digits === 2) {
                iExpectedDigits = 2;
            } else {
                iExpectedDigits = oPart.digits;
            }
            sPart = oParseHelper.findNumbers(sValue, iExpectedDigits);
            bPartInvalid = sPart === '' || oConfig.exactLength && sPart.length !== iExpectedDigits;
            var iYear = parseInt(sPart);
            var iWeekYear = iYear;
            if (sCalendarType !== CalendarType.Japanese && sPart.length <= 2) {
                var oCurrentDate = UniversalDate.getInstance(UI5Date.getInstance(), sCalendarType), iCurrentYear = oCurrentDate.getUTCFullYear(), iCurrentCentury = Math.floor(iCurrentYear / 100), iYearDiff = iCurrentCentury * 100 + iWeekYear - iCurrentYear;
                if (iYearDiff < -70) {
                    iWeekYear += (iCurrentCentury + 1) * 100;
                } else if (iYearDiff < 30) {
                    iWeekYear += iCurrentCentury * 100;
                } else {
                    iWeekYear += (iCurrentCentury - 1) * 100;
                }
            }
            return {
                length: sPart.length,
                valid: oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat),
                year: iYear,
                weekYear: iWeekYear
            };
        },
        isNumeric: true
    }),
    'M': DateFormat._createPatternSymbol({
        name: 'month',
        format: function (oField, oDate, bUTC, oFormat) {
            var iMonth = oDate.getUTCMonth();
            if (oField.digits === 3) {
                return oFormat.aMonthsAbbrev[iMonth];
            } else if (oField.digits === 4) {
                return oFormat.aMonthsWide[iMonth];
            } else if (oField.digits > 4) {
                return oFormat.aMonthsNarrow[iMonth];
            } else {
                return String(iMonth + 1).padStart(oField.digits, '0');
            }
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var iMonth, sPart, bPartInvalid, bValid, aMonthsVariants = [
                    oFormat.aMonthsWide,
                    oFormat.aMonthsWideSt,
                    oFormat.aMonthsAbbrev,
                    oFormat.aMonthsAbbrevSt,
                    oFormat.aMonthsNarrow,
                    oFormat.aMonthsNarrowSt
                ];
            if (oPart.digits < 3) {
                sPart = oParseHelper.findNumbers(sValue, Math.max(oPart.digits, 2));
                bPartInvalid = sPart === '' || oConfig.exactLength && sPart.length < 2;
                bValid = oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat);
                iMonth = parseInt(sPart) - 1;
                if (oConfig.strict && (iMonth > 11 || iMonth < 0)) {
                    bValid = false;
                }
            } else {
                for (var i = 0; i < aMonthsVariants.length; i++) {
                    var aVariants = aMonthsVariants[i];
                    var oFound = oParseHelper.findEntry(sValue, aVariants, oFormat.oLocaleData.sCLDRLocaleId);
                    if (oFound.index !== -1) {
                        return {
                            month: oFound.index,
                            length: oFound.length
                        };
                    }
                }
                bValid = oParseHelper.checkValid(oPart.type, true, oFormat);
            }
            return {
                month: iMonth,
                length: sPart ? sPart.length : 0,
                valid: bValid
            };
        },
        isNumeric: function (iDigits) {
            return iDigits < 3;
        }
    }),
    'L': DateFormat._createPatternSymbol({
        name: 'monthStandalone',
        format: function (oField, oDate, bUTC, oFormat) {
            var iMonth = oDate.getUTCMonth();
            if (oField.digits === 3) {
                return oFormat.aMonthsAbbrevSt[iMonth];
            } else if (oField.digits === 4) {
                return oFormat.aMonthsWideSt[iMonth];
            } else if (oField.digits > 4) {
                return oFormat.aMonthsNarrowSt[iMonth];
            } else {
                return String(iMonth + 1).padStart(oField.digits, '0');
            }
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var iMonth, sPart, bPartInvalid, bValid, aMonthsVariants = [
                    oFormat.aMonthsWide,
                    oFormat.aMonthsWideSt,
                    oFormat.aMonthsAbbrev,
                    oFormat.aMonthsAbbrevSt,
                    oFormat.aMonthsNarrow,
                    oFormat.aMonthsNarrowSt
                ];
            if (oPart.digits < 3) {
                sPart = oParseHelper.findNumbers(sValue, Math.max(oPart.digits, 2));
                bPartInvalid = sPart === '' || oConfig.exactLength && sPart.length < 2;
                bValid = oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat);
                iMonth = parseInt(sPart) - 1;
                if (oConfig.strict && (iMonth > 11 || iMonth < 0)) {
                    bValid = false;
                }
            } else {
                for (var i = 0; i < aMonthsVariants.length; i++) {
                    var aVariants = aMonthsVariants[i];
                    var oFound = oParseHelper.findEntry(sValue, aVariants, oFormat.oLocaleData.sCLDRLocaleId);
                    if (oFound.index !== -1) {
                        return {
                            month: oFound.index,
                            length: oFound.length
                        };
                    }
                }
                bValid = oParseHelper.checkValid(oPart.type, true, oFormat);
            }
            return {
                month: iMonth,
                length: sPart ? sPart.length : 0,
                valid: bValid
            };
        },
        isNumeric: function (iDigits) {
            return iDigits < 3;
        }
    }),
    'w': DateFormat._createPatternSymbol({
        name: 'weekInYear',
        format: function (oField, oDate, bUTC, oFormat) {
            var oWeek = oDate.getUTCWeek(oFormat.oLocale, getCalendarWeekParameter(oFormat.oFormatOptions));
            var iWeek = oWeek.week;
            var sWeek = String(iWeek + 1);
            if (oField.digits < 3) {
                sWeek = sWeek.padStart(oField.digits, '0');
            } else {
                sWeek = oFormat.oLocaleData.getCalendarWeek(oField.digits === 3 ? 'narrow' : 'wide', sWeek.padStart(2, '0'));
            }
            return sWeek;
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var sPart, bPartInvalid, bValid, iWeek, iLength = 0;
            if (oPart.digits < 3) {
                sPart = oParseHelper.findNumbers(sValue, 2);
                iLength = sPart.length;
                iWeek = parseInt(sPart) - 1;
                bPartInvalid = !sPart || oConfig.exactLength && iLength < 2;
                bValid = oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat);
            } else {
                sPart = oFormat.oLocaleData.getCalendarWeek(oPart.digits === 3 ? 'narrow' : 'wide');
                sPart = sPart.replace('{0}', '([0-9]+)');
                var rWeekNumber = new RegExp(sPart), oResult = rWeekNumber.exec(sValue);
                if (oResult) {
                    iLength = oResult[0].length;
                    iWeek = parseInt(oResult[oResult.length - 1]) - 1;
                } else {
                    bValid = oParseHelper.checkValid(oPart.type, true, oFormat);
                }
            }
            return {
                length: iLength,
                valid: bValid,
                week: iWeek
            };
        },
        isNumeric: function (iDigits) {
            return iDigits < 3;
        }
    }),
    'W': DateFormat._createPatternSymbol({ name: 'weekInMonth' }),
    'D': DateFormat._createPatternSymbol({ name: 'dayInYear' }),
    'd': DateFormat._createPatternSymbol({
        name: 'day',
        format: function (oField, oDate) {
            var iDate = oDate.getUTCDate();
            return String(iDate).padStart(oField.digits, '0');
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var sPart = oParseHelper.findNumbers(sValue, Math.max(oPart.digits, 2)), bPartInvalid = sPart === '' || oConfig.exactLength && sPart.length < 2, bValid = oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat), iDay = parseInt(sPart);
            if (oConfig.strict && (iDay > 31 || iDay < 1)) {
                bValid = false;
            }
            return {
                day: iDay,
                length: sPart.length,
                valid: bValid
            };
        },
        isNumeric: true
    }),
    'Q': DateFormat._createPatternSymbol({
        name: 'quarter',
        format: function (oField, oDate, bUTC, oFormat) {
            var iQuarter = oDate.getUTCQuarter();
            if (oField.digits === 3) {
                return oFormat.aQuartersAbbrev[iQuarter];
            } else if (oField.digits === 4) {
                return oFormat.aQuartersWide[iQuarter];
            } else if (oField.digits > 4) {
                return oFormat.aQuartersNarrow[iQuarter];
            } else {
                return String(iQuarter + 1).padStart(oField.digits, '0');
            }
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var sPart, bPartInvalid, iQuarter, bValid;
            var aQuartersVariants = [
                oFormat.aQuartersWide,
                oFormat.aQuartersWideSt,
                oFormat.aQuartersAbbrev,
                oFormat.aQuartersAbbrevSt,
                oFormat.aQuartersNarrow,
                oFormat.aQuartersNarrowSt
            ];
            if (oPart.digits < 3) {
                sPart = oParseHelper.findNumbers(sValue, Math.max(oPart.digits, 2));
                bPartInvalid = sPart === '' || oConfig.exactLength && sPart.length < 2;
                bValid = oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat);
                iQuarter = parseInt(sPart) - 1;
                if (oConfig.strict && iQuarter > 3) {
                    bValid = false;
                }
            } else {
                for (var i = 0; i < aQuartersVariants.length; i++) {
                    var aVariants = aQuartersVariants[i];
                    var oFound = oParseHelper.findEntry(sValue, aVariants, oFormat.oLocaleData.sCLDRLocaleId);
                    if (oFound.index !== -1) {
                        return {
                            quarter: oFound.index,
                            length: oFound.length
                        };
                    }
                }
                bValid = oParseHelper.checkValid(oPart.type, true, oFormat);
            }
            return {
                length: sPart ? sPart.length : 0,
                quarter: iQuarter,
                valid: bValid
            };
        },
        isNumeric: function (iDigits) {
            return iDigits < 3;
        }
    }),
    'q': DateFormat._createPatternSymbol({
        name: 'quarterStandalone',
        format: function (oField, oDate, bUTC, oFormat) {
            var iQuarter = oDate.getUTCQuarter();
            if (oField.digits === 3) {
                return oFormat.aQuartersAbbrevSt[iQuarter];
            } else if (oField.digits === 4) {
                return oFormat.aQuartersWideSt[iQuarter];
            } else if (oField.digits > 4) {
                return oFormat.aQuartersNarrowSt[iQuarter];
            } else {
                return String(iQuarter + 1).padStart(oField.digits, '0');
            }
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var sPart, bPartInvalid, iQuarter, bValid;
            var aQuartersVariants = [
                oFormat.aQuartersWide,
                oFormat.aQuartersWideSt,
                oFormat.aQuartersAbbrev,
                oFormat.aQuartersAbbrevSt,
                oFormat.aQuartersNarrow,
                oFormat.aQuartersNarrowSt
            ];
            if (oPart.digits < 3) {
                sPart = oParseHelper.findNumbers(sValue, Math.max(oPart.digits, 2));
                bPartInvalid = sPart === '' || oConfig.exactLength && sPart.length < 2;
                bValid = oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat);
                iQuarter = parseInt(sPart) - 1;
                if (oConfig.strict && iQuarter > 3) {
                    bValid = false;
                }
            } else {
                for (var i = 0; i < aQuartersVariants.length; i++) {
                    var aVariants = aQuartersVariants[i];
                    var oFound = oParseHelper.findEntry(sValue, aVariants, oFormat.oLocaleData.sCLDRLocaleId);
                    if (oFound.index !== -1) {
                        return {
                            quarter: oFound.index,
                            length: oFound.length
                        };
                    }
                }
                bValid = oParseHelper.checkValid(oPart.type, true, oFormat);
            }
            return {
                length: sPart ? sPart.length : 0,
                quarter: iQuarter,
                valid: bValid
            };
        },
        isNumeric: function (iDigits) {
            return iDigits < 3;
        }
    }),
    'F': DateFormat._createPatternSymbol({ name: 'dayOfWeekInMonth' }),
    'E': DateFormat._createPatternSymbol({
        name: 'dayNameInWeek',
        format: function (oField, oDate, bUTC, oFormat) {
            var iDay = oDate.getUTCDay();
            if (oField.digits < 4) {
                return oFormat.aDaysAbbrev[iDay];
            } else if (oField.digits === 4) {
                return oFormat.aDaysWide[iDay];
            } else if (oField.digits === 5) {
                return oFormat.aDaysNarrow[iDay];
            } else {
                return oFormat.aDaysShort[iDay];
            }
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var aDaysVariants = [
                oFormat.aDaysWide,
                oFormat.aDaysWideSt,
                oFormat.aDaysAbbrev,
                oFormat.aDaysAbbrevSt,
                oFormat.aDaysShort,
                oFormat.aDaysShortSt,
                oFormat.aDaysNarrow,
                oFormat.aDaysNarrowSt
            ];
            for (var i = 0; i < aDaysVariants.length; i++) {
                var aVariants = aDaysVariants[i];
                var oFound = oParseHelper.findEntry(sValue, aVariants, oFormat.oLocaleData.sCLDRLocaleId);
                if (oFound.index !== -1) {
                    return {
                        dayOfWeek: oFound.index,
                        length: oFound.length
                    };
                }
            }
        }
    }),
    'c': DateFormat._createPatternSymbol({
        name: 'dayNameInWeekStandalone',
        format: function (oField, oDate, bUTC, oFormat) {
            var iDay = oDate.getUTCDay();
            if (oField.digits < 4) {
                return oFormat.aDaysAbbrevSt[iDay];
            } else if (oField.digits === 4) {
                return oFormat.aDaysWideSt[iDay];
            } else if (oField.digits === 5) {
                return oFormat.aDaysNarrowSt[iDay];
            } else {
                return oFormat.aDaysShortSt[iDay];
            }
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var aDaysVariants = [
                oFormat.aDaysWide,
                oFormat.aDaysWideSt,
                oFormat.aDaysAbbrev,
                oFormat.aDaysAbbrevSt,
                oFormat.aDaysShort,
                oFormat.aDaysShortSt,
                oFormat.aDaysNarrow,
                oFormat.aDaysNarrowSt
            ];
            for (var i = 0; i < aDaysVariants.length; i++) {
                var aVariants = aDaysVariants[i];
                var oFound = oParseHelper.findEntry(sValue, aVariants, oFormat.oLocaleData.sCLDRLocaleId);
                if (oFound.index !== -1) {
                    return {
                        day: oFound.index,
                        length: oFound.length
                    };
                }
            }
        }
    }),
    'u': DateFormat._createPatternSymbol({
        name: 'dayNumberOfWeek',
        format: function (oField, oDate, bUTC, oFormat) {
            var iDay = oDate.getUTCDay();
            return oFormat._adaptDayOfWeek(iDay);
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var sPart = oParseHelper.findNumbers(sValue, oPart.digits), bPartInvalid = oConfig.exactLength && sPart.length !== oPart.digits;
            return {
                dayNumberOfWeek: parseInt(sPart),
                length: sPart.length,
                valid: oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat)
            };
        },
        isNumeric: true
    }),
    'a': DateFormat._createPatternSymbol({
        name: 'amPmMarker',
        format: function (oField, oDate, bUTC, oFormat) {
            var iDayPeriod = oDate.getUTCDayPeriod();
            if (oField.digits <= 3) {
                return oFormat.aDayPeriodsAbbrev[iDayPeriod];
            } else if (oField.digits === 4) {
                return oFormat.aDayPeriodsWide[iDayPeriod];
            } else {
                return oFormat.aDayPeriodsNarrow[iDayPeriod];
            }
        },
        parse: function (sValue, oPart, oFormat, oConfig, sTimezone) {
            var rAMPM, bAMPMAlternativeCase, oEntry, i, aMatch, normalize, aVariants, aDayPeriodsVariants = [
                    oFormat.aDayPeriodsWide,
                    oFormat.aDayPeriodsAbbrev,
                    oFormat.aDayPeriodsNarrow
                ];
            rAMPM = /[aApP](?:\.)?[\x20\xA0]?[mM](?:\.)?/;
            aMatch = sValue.match(rAMPM);
            bAMPMAlternativeCase = aMatch && aMatch.index === 0;
            function normalize(sValue) {
                return sValue.replace(/[\x20\xA0]/g, '').replace(/\./g, '');
            }
            if (bAMPMAlternativeCase) {
                sValue = normalize(sValue);
            }
            for (i = 0; i < aDayPeriodsVariants.length; i += 1) {
                aVariants = aDayPeriodsVariants[i];
                if (bAMPMAlternativeCase) {
                    aVariants = aVariants.map(normalize);
                }
                oEntry = oParseHelper.findEntry(sValue, aVariants, oFormat.oLocaleData.sCLDRLocaleId);
                if (oEntry.index !== -1) {
                    return {
                        pm: oEntry.index === 1,
                        length: bAMPMAlternativeCase ? aMatch[0].length : oEntry.length
                    };
                }
            }
            return { valid: false };
        }
    }),
    'B': DateFormat._createPatternSymbol({
        name: 'flexibleDayPeriod',
        format: function (oField, oDate, bUTC, oFormat) {
            var bContainsHour = oFormat.aFormatArray.some(function (oFormatElement) {
                    return 'hHKk'.includes(oFormatElement.symbol);
                }), sFlexibleDayPeriod = oFormat.oLocaleData.getFlexibleDayPeriodOfTime(oDate.getUTCHours(), oDate.getUTCMinutes());
            if (bContainsHour) {
                if (oField.digits <= 3) {
                    return oFormat.oFlexibleDayPeriodsAbbrev[sFlexibleDayPeriod];
                }
                if (oField.digits === 4) {
                    return oFormat.oFlexibleDayPeriodsWide[sFlexibleDayPeriod];
                }
                return oFormat.oFlexibleDayPeriodsNarrow[sFlexibleDayPeriod];
            }
            if (oField.digits <= 3) {
                return oFormat.oFlexibleDayPeriodsAbbrevSt[sFlexibleDayPeriod];
            }
            if (oField.digits === 4) {
                return oFormat.oFlexibleDayPeriodsWideSt[sFlexibleDayPeriod];
            }
            return oFormat.oFlexibleDayPeriodsNarrowSt[sFlexibleDayPeriod];
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var i, oFound, oVariant, bContainsHour = oFormat.aFormatArray.some(function (oFormatElement) {
                    return 'hHKk'.includes(oFormatElement.symbol);
                }), aFlexibleDayPeriodVariants = [
                    oFormat.oFlexibleDayPeriodsWide,
                    oFormat.oFlexibleDayPeriodsAbbrev,
                    oFormat.oFlexibleDayPeriodsNarrow
                ];
            if (bContainsHour) {
                for (i = 0; i < aFlexibleDayPeriodVariants.length; i++) {
                    oVariant = aFlexibleDayPeriodVariants[i];
                    oFound = oParseHelper.findEntry(sValue, Object.values(oVariant), oFormat.oLocaleData.sCLDRLocaleId);
                    if (oFound.index !== -1) {
                        return {
                            flexDayPeriod: Object.keys(oVariant)[oFound.index],
                            length: oFound.length
                        };
                    }
                }
            }
            return { valid: false };
        }
    }),
    'H': DateFormat._createPatternSymbol({
        name: 'hour0_23',
        format: function (oField, oDate) {
            var iHours = oDate.getUTCHours();
            return String(iHours).padStart(oField.digits, '0');
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var sPart = oParseHelper.findNumbers(sValue, Math.max(oPart.digits, 2)), iHours = parseInt(sPart), bPartInvalid = sPart === '' || oConfig.exactLength && sPart.length < 2, bValid = oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat);
            if (oConfig.strict && iHours > 23) {
                bValid = false;
            }
            return {
                hour: iHours,
                length: sPart.length,
                valid: bValid
            };
        },
        isNumeric: true
    }),
    'k': DateFormat._createPatternSymbol({
        name: 'hour1_24',
        format: function (oField, oDate) {
            var iHours = oDate.getUTCHours();
            var sHours = iHours === 0 ? '24' : String(iHours);
            return sHours.padStart(oField.digits, '0');
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var sPart = oParseHelper.findNumbers(sValue, Math.max(oPart.digits, 2)), iHours = parseInt(sPart), bPartInvalid = sPart === '' || oConfig.exactLength && sPart.length < 2, bValid = oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat);
            if (iHours === 24) {
                iHours = 0;
            }
            if (oConfig.strict && iHours > 23) {
                bValid = false;
            }
            return {
                hour: iHours,
                length: sPart.length,
                valid: bValid
            };
        },
        isNumeric: true
    }),
    'K': DateFormat._createPatternSymbol({
        name: 'hour0_11',
        format: function (oField, oDate) {
            var iHours = oDate.getUTCHours();
            var sHours = String(iHours > 11 ? iHours - 12 : iHours);
            return sHours.padStart(oField.digits, '0');
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var sPart = oParseHelper.findNumbers(sValue, Math.max(oPart.digits, 2)), iHours = parseInt(sPart), bPartInvalid = sPart === '' || oConfig.exactLength && sPart.length < 2, bValid = oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat);
            if (oConfig.strict && iHours > 11) {
                bValid = false;
            }
            return {
                hour: iHours,
                length: sPart.length,
                valid: bValid
            };
        },
        isNumeric: true
    }),
    'h': DateFormat._createPatternSymbol({
        name: 'hour1_12',
        format: function (oField, oDate) {
            var iHours = oDate.getUTCHours();
            var sHours;
            if (iHours > 12) {
                sHours = String(iHours - 12);
            } else if (iHours === 0) {
                sHours = '12';
            } else {
                sHours = String(iHours);
            }
            return sHours.padStart(oField.digits, '0');
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var bPM = oConfig.dateValue.pm, sPart = oParseHelper.findNumbers(sValue, Math.max(oPart.digits, 2)), iHours = parseInt(sPart), bPartInvalid = sPart === '' || oConfig.exactLength && sPart.length < 2, bValid = oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat);
            if (iHours === 12) {
                iHours = 0;
                bPM = bPM === undefined ? true : bPM;
            }
            if (oConfig.strict && iHours > 11) {
                bValid = false;
            }
            return {
                hour: iHours,
                length: sPart.length,
                pm: bPM,
                valid: bValid
            };
        },
        isNumeric: true
    }),
    'm': DateFormat._createPatternSymbol({
        name: 'minute',
        format: function (oField, oDate) {
            var iMinutes = oDate.getUTCMinutes();
            return String(iMinutes).padStart(oField.digits, '0');
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var sPart = oParseHelper.findNumbers(sValue, Math.max(oPart.digits, 2)), iMinutes = parseInt(sPart), bPartInvalid = sPart === '' || oConfig.exactLength && sPart.length < 2, bValid = oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat);
            if (oConfig.strict && iMinutes > 59) {
                bValid = false;
            }
            return {
                length: sPart.length,
                minute: iMinutes,
                valid: bValid
            };
        },
        isNumeric: true
    }),
    's': DateFormat._createPatternSymbol({
        name: 'second',
        format: function (oField, oDate) {
            var iSeconds = oDate.getUTCSeconds();
            return String(iSeconds).padStart(oField.digits, '0');
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var iExpectedDigits = Math.max(oPart.digits, 2), sPart = oParseHelper.findNumbers(sValue, iExpectedDigits), bPartInvalid = sPart === '' || oConfig.exactLength && sPart.length < iExpectedDigits, iSeconds = parseInt(sPart), bValid = oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat);
            if (oConfig.strict && iSeconds > 59) {
                bValid = false;
            }
            return {
                length: sPart.length,
                second: iSeconds,
                valid: bValid
            };
        },
        isNumeric: true
    }),
    'S': DateFormat._createPatternSymbol({
        name: 'fractionalsecond',
        format: function (oField, oDate) {
            var iMilliseconds = oDate.getUTCMilliseconds();
            var sMilliseconds = String(iMilliseconds);
            var sFractionalseconds = sMilliseconds.padStart(3, '0');
            sFractionalseconds = sFractionalseconds.substr(0, oField.digits);
            sFractionalseconds = sFractionalseconds.padEnd(oField.digits, '0');
            return sFractionalseconds;
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var sPart = oParseHelper.findNumbers(sValue, oPart.digits), iLength = sPart.length, bPartInvalid = oConfig.exactLength && iLength < oPart.digits;
            sPart = sPart.substr(0, 3);
            sPart = sPart.padEnd(3, '0');
            var iMilliseconds = parseInt(sPart);
            return {
                length: iLength,
                millisecond: iMilliseconds,
                valid: oParseHelper.checkValid(oPart.type, bPartInvalid, oFormat)
            };
        },
        isNumeric: true
    }),
    'z': DateFormat._createPatternSymbol({
        name: 'timezoneGeneral',
        format: function (oField, oDate, bUTC, oFormat, sTimezone) {
            if (oField.digits > 3 && oDate.getTimezoneLong && oDate.getTimezoneLong()) {
                return oDate.getTimezoneLong();
            } else if (oDate.getTimezoneShort && oDate.getTimezoneShort()) {
                return oDate.getTimezoneShort();
            }
            var iTimezoneOffset = TimezoneUtil.calculateOffset(oDate, sTimezone);
            var sTimeZone = 'GMT';
            var iTZOffset = Math.abs(iTimezoneOffset / 60);
            var bPositiveOffset = iTimezoneOffset > 0;
            var iHourOffset = Math.floor(iTZOffset / 60);
            var iMinuteOffset = Math.floor(iTZOffset % 60);
            if (!bUTC && iTZOffset !== 0) {
                sTimeZone += bPositiveOffset ? '-' : '+';
                sTimeZone += String(iHourOffset).padStart(2, '0');
                sTimeZone += ':';
                sTimeZone += String(iMinuteOffset).padStart(2, '0');
            } else {
                sTimeZone += 'Z';
            }
            return sTimeZone;
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            var iLength = 0;
            var iTZDiff;
            var oTZ = sValue.substring(0, 3);
            if (oTZ === 'GMT' || oTZ === 'UTC') {
                iLength = 3;
            } else if (sValue.substring(0, 2) === 'UT') {
                iLength = 2;
            } else if (sValue.charAt(0) === 'Z') {
                iLength = 1;
                iTZDiff = 0;
            } else {
                return { error: 'cannot be parsed correctly by sap.ui.core.format.DateFormat: The given timezone is not supported!' };
            }
            if (sValue.charAt(0) !== 'Z') {
                var oParsedTZ = oParseHelper.parseTZ(sValue.substr(iLength), true);
                iLength += oParsedTZ.length;
                iTZDiff = oParsedTZ.tzDiff;
            }
            return {
                length: iLength,
                tzDiff: iTZDiff
            };
        }
    }),
    'Z': DateFormat._createPatternSymbol({
        name: 'timezoneRFC822',
        format: function (oField, oDate, bUTC, oFormat, sTimezone) {
            var iTimezoneOffset = TimezoneUtil.calculateOffset(oDate, sTimezone);
            var iTZOffset = Math.abs(iTimezoneOffset / 60);
            var bPositiveOffset = iTimezoneOffset > 0;
            var iHourOffset = Math.floor(iTZOffset / 60);
            var iMinuteOffset = Math.floor(iTZOffset % 60);
            var sTimeZone = '';
            if (!bUTC) {
                sTimeZone += bPositiveOffset ? '-' : '+';
                sTimeZone += String(iHourOffset).padStart(2, '0');
                sTimeZone += String(iMinuteOffset).padStart(2, '0');
            }
            return sTimeZone;
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            return oParseHelper.parseTZ(sValue, false);
        }
    }),
    'X': DateFormat._createPatternSymbol({
        name: 'timezoneISO8601',
        format: function (oField, oDate, bUTC, oFormat, sTimezone) {
            var iTimezoneOffset = TimezoneUtil.calculateOffset(oDate, sTimezone);
            var iTZOffset = Math.abs(iTimezoneOffset / 60);
            var bPositiveOffset = iTimezoneOffset > 0;
            var iHourOffset = Math.floor(iTZOffset / 60);
            var iMinuteOffset = Math.floor(iTZOffset % 60);
            var sTimeZone = '';
            if (!bUTC && iTZOffset !== 0) {
                sTimeZone += bPositiveOffset ? '-' : '+';
                sTimeZone += String(iHourOffset).padStart(2, '0');
                if (oField.digits > 1 || iMinuteOffset > 0) {
                    if (oField.digits === 3 || oField.digits === 5) {
                        sTimeZone += ':';
                    }
                    sTimeZone += String(iMinuteOffset).padStart(2, '0');
                }
            } else {
                sTimeZone += 'Z';
            }
            return sTimeZone;
        },
        parse: function (sValue, oPart, oFormat, oConfig) {
            if (sValue.charAt(0) === 'Z') {
                return {
                    length: 1,
                    tzDiff: 0
                };
            } else {
                return oParseHelper.parseTZ(sValue, oPart.digits === 3 || oPart.digits === 5);
            }
        }
    }),
    'V': DateFormat._createPatternSymbol({
        name: 'timezoneID',
        format: function (oField, oDate, bUTC, oFormat, sTimezone) {
            if (!bUTC && oField.digits === 2) {
                return oFormat.oLocaleData.getTimezoneTranslations()[sTimezone] || sTimezone;
            }
            return '';
        },
        parse: function (sValue, oPart, oFormat, oConfig, sTimezone) {
            var oTimezoneParsed = {
                timezone: '',
                length: 0
            };
            if (oPart.digits === 2) {
                var mTimezoneTranslations = oFormat.oLocaleData.getTimezoneTranslations();
                if (sValue === mTimezoneTranslations[sTimezone]) {
                    return {
                        timezone: sTimezone,
                        length: sValue.length
                    };
                }
                var aTimezoneTranslations = Object.values(mTimezoneTranslations);
                var oTimezoneResult = oParseHelper.findEntry(sValue, aTimezoneTranslations, oFormat.oLocaleData.sCLDRLocaleId);
                if (oTimezoneResult.index !== -1) {
                    return {
                        timezone: Object.keys(mTimezoneTranslations)[oTimezoneResult.index],
                        length: oTimezoneResult.length
                    };
                }
                var sCurrentValue = '';
                for (var i = 0; i < sValue.length; i++) {
                    sCurrentValue += sValue[i];
                    if (TimezoneUtil.isValidTimezone(sCurrentValue)) {
                        oTimezoneParsed.timezone = sCurrentValue;
                        oTimezoneParsed.length = sCurrentValue.length;
                    }
                }
            }
            return oTimezoneParsed;
        }
    })
};
DateFormat.prototype._format = function (oJSDate, bUTC, sTimezone) {
    if (this.oFormatOptions.relative) {
        var sRes = this.formatRelative(oJSDate, bUTC, this.oFormatOptions.relativeRange, sTimezone);
        if (sRes) {
            return sRes;
        }
    }
    var sCalendarType = this.oFormatOptions.calendarType;
    var oDate = UniversalDate.getInstance(oJSDate, sCalendarType);
    var aBuffer = [], oPart, sResult, sSymbol;
    for (var i = 0; i < this.aFormatArray.length; i++) {
        oPart = this.aFormatArray[i];
        sSymbol = oPart.symbol || '';
        aBuffer.push(this.oSymbols[sSymbol].format(oPart, oDate, bUTC, this, sTimezone));
    }
    sResult = aBuffer.join('');
    if (Configuration.getOriginInfo()) {
        sResult = new String(sResult);
        sResult.originInfo = {
            source: 'Common Locale Data Repository',
            locale: this.oLocale.toString(),
            style: this.oFormatOptions.style,
            pattern: this.oFormatOptions.pattern
        };
    }
    return sResult;
};
DateFormat.prototype.format = function (vJSDate, bUTC) {
    var sTimezone;
    if (this.type === mDateFormatTypes.DATETIME_WITH_TIMEZONE) {
        sTimezone = bUTC;
        bUTC = false;
        checkTimezoneParameterType(sTimezone);
        if (sTimezone && !TimezoneUtil.isValidTimezone(sTimezone)) {
            Log.error('The given timezone isn\'t valid.');
            return '';
        }
    }
    var sCalendarType = this.oFormatOptions.calendarType, sResult;
    if (bUTC === undefined) {
        bUTC = this.oFormatOptions.UTC;
    }
    sTimezone = sTimezone || Configuration.getTimezone();
    if (Array.isArray(vJSDate)) {
        if (!this.oFormatOptions.interval) {
            Log.error('Non-interval DateFormat can\'t format more than one date instance.');
            return '';
        }
        if (vJSDate.length !== 2) {
            Log.error('Interval DateFormat can only format with 2 date instances but ' + vJSDate.length + ' is given.');
            return '';
        }
        vJSDate = vJSDate.map(function (oJSDate) {
            return convertToTimezone(oJSDate, sTimezone, bUTC);
        });
        if (this.oFormatOptions.singleIntervalValue) {
            if (vJSDate[0] === null) {
                Log.error('First date instance which is passed to the interval DateFormat shouldn\'t be null.');
                return '';
            }
            if (vJSDate[1] === null) {
                sResult = this._format(vJSDate[0], bUTC, sTimezone);
            }
        }
        if (sResult === undefined) {
            if (!vJSDate.every(isValidDateObject)) {
                Log.error('At least one date instance which is passed to the interval DateFormat isn\'t valid.');
                return '';
            }
            sResult = this._formatInterval(vJSDate, bUTC);
        }
    } else {
        if (!isValidDateObject(vJSDate)) {
            if (this.type === mDateFormatTypes.DATETIME_WITH_TIMEZONE && this.oFormatOptions.pattern.includes('VV')) {
                return this.oLocaleData.getTimezoneTranslations()[sTimezone] || sTimezone;
            }
            Log.error('The given date instance isn\'t valid.');
            return '';
        }
        if (this.oFormatOptions.interval) {
            Log.error('Interval DateFormat expects an array with two dates for the first argument but only one date is given.');
            return '';
        }
        vJSDate = convertToTimezone(vJSDate, sTimezone, bUTC);
        sResult = this._format(vJSDate, bUTC, sTimezone);
    }
    if (sCalendarType === CalendarType.Japanese && this.oLocale.getLanguage() === 'ja') {
        sResult = sResult.replace(/(^|[^\d])1年/g, '$1元年');
    }
    return sResult;
};
DateFormat.prototype._formatInterval = function (aJSDates, bUTC) {
    var sCalendarType = this.oFormatOptions.calendarType;
    var oFromDate = UniversalDate.getInstance(aJSDates[0], sCalendarType);
    var oToDate = UniversalDate.getInstance(aJSDates[1], sCalendarType);
    var oDate;
    var oPart;
    var sSymbol;
    var aBuffer = [];
    var sPattern;
    var aFormatArray = [];
    var oDiffField = this._getGreatestDiffField([
        oFromDate,
        oToDate
    ]);
    if (!oDiffField) {
        return this._format(aJSDates[0], bUTC);
    }
    if (this.oFormatOptions.format) {
        sPattern = this.oLocaleData.getCustomIntervalPattern(this.oFormatOptions.format, oDiffField, sCalendarType);
    } else {
        sPattern = this.oLocaleData.getCombinedIntervalPattern(this.oFormatOptions.pattern, sCalendarType);
    }
    aFormatArray = this.parseCldrDatePattern(sPattern);
    oDate = oFromDate;
    for (var i = 0; i < aFormatArray.length; i++) {
        oPart = aFormatArray[i];
        sSymbol = oPart.symbol || '';
        if (oPart.repeat) {
            oDate = oToDate;
        }
        aBuffer.push(this.oSymbols[sSymbol].format(oPart, oDate, bUTC, this));
    }
    return aBuffer.join('');
};
var mFieldToGroup = {
    Era: 'Era',
    FullYear: 'Year',
    Quarter: 'Quarter',
    Month: 'Month',
    Week: 'Week',
    Date: 'Day',
    DayPeriod: 'DayPeriod',
    Hours: 'Hour',
    Minutes: 'Minute',
    Seconds: 'Second'
};
DateFormat.prototype._getGreatestDiffField = function (aDates) {
    var bDiffFound = false, mDiff = {};
    this.aIntervalCompareFields.forEach(function (sField) {
        var sGetterPrefix = 'getUTC', sMethodName = sGetterPrefix + sField, sFieldGroup = mFieldToGroup[sField], vFromValue = aDates[0][sMethodName].apply(aDates[0]), vToValue = aDates[1][sMethodName].apply(aDates[1]);
        if (!deepEqual(vFromValue, vToValue)) {
            bDiffFound = true;
            mDiff[sFieldGroup] = true;
        }
    });
    if (bDiffFound) {
        return mDiff;
    }
    return null;
};
DateFormat.prototype._parse = function (sValue, aFormatArray, bUTC, bStrict, sTimezone) {
    var sFlexibleDayPeriod, oNextPart, oPart, bPM, oPrevPart, oResult, sSubValue, oDateValue = {
            valid: true,
            lastTimezonePatternSymbol: ''
        }, iIndex = 0, oParseConf = {
            formatArray: aFormatArray,
            dateValue: oDateValue,
            strict: bStrict
        }, that = this;
    function getSymbol(oPart0) {
        return that.oSymbols[oPart0.symbol || ''];
    }
    function isNumeric(oPart0) {
        return !!oPart0 && getSymbol(oPart0).isNumeric(oPart0.digits);
    }
    for (var i = 0; i < aFormatArray.length; i++) {
        sSubValue = sValue.substr(iIndex);
        oPart = aFormatArray[i];
        oPrevPart = aFormatArray[i - 1];
        oNextPart = aFormatArray[i + 1];
        oParseConf.index = i;
        oParseConf.exactLength = isNumeric(oPart) && (isNumeric(oPrevPart) || isNumeric(oNextPart));
        oResult = getSymbol(oPart).parse(sSubValue, oPart, this, oParseConf, sTimezone) || {};
        if (oResult.tzDiff !== undefined || oResult.timezone) {
            oResult.lastTimezonePatternSymbol = oPart.symbol;
        }
        oDateValue = extend(oDateValue, oResult);
        if (oResult.valid === false) {
            break;
        }
        iIndex += oResult.length || 0;
    }
    oDateValue.index = iIndex;
    bPM = oDateValue.pm;
    if (oDateValue.flexDayPeriod && oDateValue.hour * 60 + (oDateValue.minute || 0) < 720) {
        sFlexibleDayPeriod = this.oLocaleData.getFlexibleDayPeriodOfTime(oDateValue.hour + 12, oDateValue.minute || 0);
        bPM = oDateValue.flexDayPeriod === sFlexibleDayPeriod;
    }
    if (bPM) {
        oDateValue.hour += 12;
    }
    if (oDateValue.dayNumberOfWeek === undefined && oDateValue.dayOfWeek !== undefined) {
        oDateValue.dayNumberOfWeek = this._adaptDayOfWeek(oDateValue.dayOfWeek);
    }
    if (oDateValue.quarter !== undefined && oDateValue.month === undefined && oDateValue.day === undefined) {
        oDateValue.month = 3 * oDateValue.quarter;
        oDateValue.day = 1;
    }
    return oDateValue;
};
DateFormat.prototype._parseInterval = function (sValue, sCalendarType, bUTC, bStrict, sTimezone) {
    var aDateValues, iRepeat, oDateValue;
    this.intervalPatterns.some(function (sPattern) {
        var aFormatArray = this.parseCldrDatePattern(sPattern);
        iRepeat = undefined;
        for (var i = 0; i < aFormatArray.length; i++) {
            if (aFormatArray[i].repeat) {
                iRepeat = i;
                break;
            }
        }
        if (iRepeat === undefined) {
            oDateValue = this._parse(sValue, aFormatArray, bUTC, bStrict, sTimezone);
            if (oDateValue.index === 0 || oDateValue.index < sValue.length) {
                oDateValue.valid = false;
            }
            if (oDateValue.valid === false) {
                return;
            }
            aDateValues = [
                oDateValue,
                oDateValue
            ];
            return true;
        } else {
            aDateValues = [];
            oDateValue = this._parse(sValue, aFormatArray.slice(0, iRepeat), bUTC, bStrict, sTimezone);
            if (oDateValue.valid === false) {
                return;
            }
            aDateValues.push(oDateValue);
            var iLength = oDateValue.index;
            oDateValue = this._parse(sValue.substring(iLength), aFormatArray.slice(iRepeat), bUTC, bStrict, sTimezone);
            if (oDateValue.index === 0 || oDateValue.index + iLength < sValue.length) {
                oDateValue.valid = false;
            }
            if (oDateValue.valid === false) {
                return;
            }
            aDateValues.push(oDateValue);
            return true;
        }
    }.bind(this));
    return aDateValues;
};
function getCalendarWeekParameter(oFormatOptions) {
    if (oFormatOptions.calendarWeekNumbering) {
        return oFormatOptions.calendarWeekNumbering;
    } else if (oFormatOptions.firstDayOfWeek !== undefined && oFormatOptions.minimalDaysInFirstWeek !== undefined) {
        return {
            firstDayOfWeek: oFormatOptions.firstDayOfWeek,
            minimalDaysInFirstWeek: oFormatOptions.minimalDaysInFirstWeek
        };
    }
    return undefined;
}
var convertToTimezone = function (oJSDate, sTimezone, bUTC) {
    if (!bUTC && isValidDateObject(oJSDate)) {
        return TimezoneUtil.convertToTimezone(oJSDate, sTimezone);
    }
    return oJSDate;
};
var fnCreateDate = function (oDateValue, sCalendarType, bUTC, bStrict, sTimezone, oFormatOptions, oLocale) {
    if (!oDateValue.valid) {
        return null;
    }
    var oDate, iYear = typeof oDateValue.year === 'number' ? oDateValue.year : 1970;
    oDate = UniversalDate.getInstance(new Date(0), sCalendarType);
    oDate.setUTCEra(oDateValue.era || UniversalDate.getCurrentEra(sCalendarType));
    oDate.setUTCFullYear(iYear);
    oDate.setUTCMonth(oDateValue.month || 0);
    oDate.setUTCDate(oDateValue.day || 1);
    oDate.setUTCHours(oDateValue.hour || 0);
    oDate.setUTCMinutes(oDateValue.minute || 0);
    oDate.setUTCSeconds(oDateValue.second || 0);
    oDate.setUTCMilliseconds(oDateValue.millisecond || 0);
    if (bStrict && (oDateValue.day || 1) !== oDate.getUTCDate()) {
        return null;
    }
    if (oDateValue.week !== undefined && (oDateValue.month === undefined || oDateValue.day === undefined)) {
        oDate.setUTCWeek({
            year: oDateValue.weekYear || oDateValue.year,
            week: oDateValue.week
        }, oLocale, getCalendarWeekParameter(oFormatOptions));
        if (oDateValue.dayNumberOfWeek !== undefined) {
            oDate.setUTCDate(oDate.getUTCDate() + oDateValue.dayNumberOfWeek - 1);
        }
    }
    oDate = oDate.getJSDate();
    if (!bUTC && (oDateValue.lastTimezonePatternSymbol === 'V' && oDateValue.timezone || oDateValue.tzDiff === undefined)) {
        if (oDateValue.timezone) {
            sTimezone = oDateValue.timezone;
        }
        if (sTimezone) {
            oDateValue.tzDiff = TimezoneUtil.calculateOffset(oDate, sTimezone);
        }
    }
    if (oDateValue.tzDiff) {
        oDate.setUTCSeconds(oDate.getUTCSeconds() + oDateValue.tzDiff);
    }
    return oDate;
};
function mergeWithoutOverwrite(object1, object2) {
    if (object1 === object2) {
        return object1;
    }
    var oMergedObject = {};
    Object.keys(object1).forEach(function (sKey) {
        oMergedObject[sKey] = object1[sKey];
    });
    Object.keys(object2).forEach(function (sKey) {
        if (!oMergedObject.hasOwnProperty(sKey)) {
            oMergedObject[sKey] = object2[sKey];
        }
    });
    return oMergedObject;
}
function isValidDateRange(oStartDate, oEndDate) {
    if (oStartDate.getTime() > oEndDate.getTime()) {
        return false;
    }
    return true;
}
function isValidDateObject(oDate) {
    return oDate && typeof oDate.getTime === 'function' && !isNaN(oDate.getTime());
}
DateFormat.prototype.parse = function (sValue, bUTC, bStrict) {
    var bShowDate = this.oFormatOptions.showDate === undefined || this.oFormatOptions.showDate;
    var bShowTime = this.oFormatOptions.showTime === undefined || this.oFormatOptions.showTime;
    if (this.type === mDateFormatTypes.DATETIME_WITH_TIMEZONE && (bShowDate && !bShowTime || !bShowDate && bShowTime)) {
        throw new TypeError('The input can only be parsed back to date if both date and time are supplied.');
    }
    var sTimezone;
    if (bUTC === undefined && this.type !== mDateFormatTypes.DATETIME_WITH_TIMEZONE) {
        bUTC = this.oFormatOptions.UTC;
    }
    var bUTCInputParameter = bUTC;
    if (this.type === mDateFormatTypes.DATETIME_WITH_TIMEZONE) {
        sTimezone = bUTC;
        bUTC = false;
        checkTimezoneParameterType(sTimezone);
        if (sTimezone && !TimezoneUtil.isValidTimezone(sTimezone)) {
            Log.error('The given timezone isn\'t valid.');
            return null;
        }
    }
    sValue = sValue == null ? '' : String(sValue).trim();
    var oDateValue;
    var sCalendarType = this.oFormatOptions.calendarType;
    sTimezone = sTimezone || Configuration.getTimezone();
    if (bStrict === undefined) {
        bStrict = this.oFormatOptions.strictParsing;
    }
    if (sCalendarType === CalendarType.Japanese && this.oLocale.getLanguage() === 'ja') {
        sValue = sValue.replace(/元年/g, '1年');
    }
    if (!this.oFormatOptions.interval) {
        var oJSDate = this.parseRelative(sValue, bUTC);
        if (oJSDate) {
            return oJSDate;
        }
        oDateValue = this._parse(sValue, this.aFormatArray, bUTC, bStrict, sTimezone);
        if (oDateValue.index === 0 || oDateValue.index < sValue.length) {
            oDateValue.valid = false;
        }
        oJSDate = fnCreateDate(oDateValue, sCalendarType, bUTC, bStrict, sTimezone, this.oFormatOptions, this.oLocale);
        if (oJSDate) {
            if (this.type === mDateFormatTypes.DATETIME_WITH_TIMEZONE) {
                var bShowTimezone = this.oFormatOptions.showTimezone === undefined || this.oFormatOptions.showTimezone;
                if (!bShowTimezone && bShowDate && bShowTime) {
                    return [
                        oJSDate,
                        undefined
                    ];
                } else if (bShowTimezone && !bShowDate && !bShowTime) {
                    return [
                        undefined,
                        oDateValue.timezone
                    ];
                }
                return [
                    oJSDate,
                    oDateValue.timezone || undefined
                ];
            }
            return oJSDate;
        }
    } else {
        var aDateValues = this._parseInterval(sValue, sCalendarType, bUTC, bStrict, sTimezone);
        var oJSDate1, oJSDate2;
        if (aDateValues && aDateValues.length === 2) {
            var oDateValue1 = mergeWithoutOverwrite(aDateValues[0], aDateValues[1]);
            var oDateValue2 = mergeWithoutOverwrite(aDateValues[1], aDateValues[0]);
            oJSDate1 = fnCreateDate(oDateValue1, sCalendarType, bUTC, bStrict, sTimezone, this.oFormatOptions, this.oLocale);
            oJSDate2 = fnCreateDate(oDateValue2, sCalendarType, bUTC, bStrict, sTimezone, this.oFormatOptions, this.oLocale);
            if (oJSDate1 && oJSDate2) {
                if (this.oFormatOptions.singleIntervalValue && oJSDate1.getTime() === oJSDate2.getTime()) {
                    return [
                        oJSDate1,
                        null
                    ];
                }
                var bValid = isValidDateRange(oJSDate1, oJSDate2);
                if (bStrict && !bValid) {
                    Log.error('StrictParsing: Invalid date range. The given end date is before the start date.');
                    return [
                        null,
                        null
                    ];
                }
                return [
                    oJSDate1,
                    oJSDate2
                ];
            }
        }
    }
    if (this.aFallbackFormats) {
        var vDate;
        this.aFallbackFormats.every(function (oFallbackFormat) {
            vDate = oFallbackFormat.parse(sValue, bUTCInputParameter, bStrict);
            if (Array.isArray(vDate)) {
                if (oFallbackFormat.type === mDateFormatTypes.DATETIME_WITH_TIMEZONE) {
                    return false;
                }
                return !(vDate[0] && vDate[1]);
            } else {
                return !vDate;
            }
        });
        return vDate;
    }
    if (!this.oFormatOptions.interval) {
        return null;
    } else {
        return [
            null,
            null
        ];
    }
};
DateFormat.prototype.parseCldrDatePattern = function (sPattern) {
    if (mCldrDatePattern[sPattern]) {
        return mCldrDatePattern[sPattern];
    }
    var aFormatArray = [], i, bQuoted = false, oCurrentObject = null, sState = '', sNewState = '', mAppeared = {}, bIntervalStartFound = false;
    for (i = 0; i < sPattern.length; i++) {
        var sCurChar = sPattern.charAt(i), sNextChar, sPrevChar, sPrevPrevChar;
        if (bQuoted) {
            if (sCurChar === '\'') {
                sPrevChar = sPattern.charAt(i - 1);
                sPrevPrevChar = sPattern.charAt(i - 2);
                sNextChar = sPattern.charAt(i + 1);
                if (sPrevChar === '\'' && sPrevPrevChar !== '\'') {
                    bQuoted = false;
                } else if (sNextChar === '\'') {
                    i += 1;
                } else {
                    bQuoted = false;
                    continue;
                }
            }
            if (sState === 'text') {
                oCurrentObject.value += sCurChar;
            } else {
                oCurrentObject = {
                    type: 'text',
                    value: sCurChar
                };
                aFormatArray.push(oCurrentObject);
                sState = 'text';
            }
        } else {
            if (sCurChar === '\'') {
                bQuoted = true;
            } else if (this.oSymbols[sCurChar]) {
                sNewState = this.oSymbols[sCurChar].name;
                if (sState === sNewState) {
                    oCurrentObject.digits++;
                } else {
                    oCurrentObject = {
                        type: sNewState,
                        symbol: sCurChar,
                        digits: 1
                    };
                    aFormatArray.push(oCurrentObject);
                    sState = sNewState;
                    if (!bIntervalStartFound) {
                        if (mAppeared[sNewState]) {
                            oCurrentObject.repeat = true;
                            bIntervalStartFound = true;
                        } else {
                            mAppeared[sNewState] = true;
                        }
                    }
                }
            } else {
                if (sState === 'text') {
                    oCurrentObject.value += sCurChar;
                } else {
                    oCurrentObject = {
                        type: 'text',
                        value: sCurChar
                    };
                    aFormatArray.push(oCurrentObject);
                    sState = 'text';
                }
            }
        }
    }
    mCldrDatePattern[sPattern] = aFormatArray;
    return aFormatArray;
};
DateFormat.prototype.parseRelative = function (sValue, bUTC) {
    var aPatterns, oEntry, rPattern, oResult, iValue;
    if (!sValue) {
        return null;
    }
    aPatterns = this.oLocaleData.getRelativePatterns(this.aRelativeParseScales, this.oFormatOptions.relativeStyle);
    for (var i = 0; i < aPatterns.length; i++) {
        oEntry = aPatterns[i];
        rPattern = new RegExp('^\\s*' + oEntry.pattern.replace(/\{0\}/, '(\\d+)') + '\\s*$', 'i');
        oResult = rPattern.exec(sValue);
        if (oResult) {
            if (oEntry.value !== undefined) {
                return computeRelativeDate(oEntry.value, oEntry.scale);
            } else {
                iValue = parseInt(oResult[1]);
                return computeRelativeDate(iValue * oEntry.sign, oEntry.scale);
            }
        }
    }
    function computeRelativeDate(iDiff, sScale) {
        var oResult = UI5Date.getInstance();
        if (bUTC) {
            oResult.setUTCFullYear(oResult.getFullYear(), oResult.getMonth(), oResult.getDate());
            oResult.setUTCHours(oResult.getHours(), oResult.getMinutes(), oResult.getSeconds(), oResult.getMilliseconds());
            switch (sScale) {
            case 'second':
                oResult.setUTCSeconds(oResult.getUTCSeconds() + iDiff);
                break;
            case 'minute':
                oResult.setUTCMinutes(oResult.getUTCMinutes() + iDiff);
                break;
            case 'hour':
                oResult.setUTCHours(oResult.getUTCHours() + iDiff);
                break;
            case 'day':
                oResult.setUTCDate(oResult.getUTCDate() + iDiff);
                break;
            case 'week':
                oResult.setUTCDate(oResult.getUTCDate() + iDiff * 7);
                break;
            case 'month':
                oResult.setUTCMonth(oResult.getUTCMonth() + iDiff);
                break;
            case 'quarter':
                oResult.setUTCMonth(oResult.getUTCMonth() + iDiff * 3);
                break;
            case 'year':
                oResult.setUTCFullYear(oResult.getUTCFullYear() + iDiff);
                break;
            }
        } else {
            switch (sScale) {
            case 'second':
                oResult.setSeconds(oResult.getSeconds() + iDiff);
                break;
            case 'minute':
                oResult.setMinutes(oResult.getMinutes() + iDiff);
                break;
            case 'hour':
                oResult.setHours(oResult.getHours() + iDiff);
                break;
            case 'day':
                oResult.setDate(oResult.getDate() + iDiff);
                break;
            case 'week':
                oResult.setDate(oResult.getDate() + iDiff * 7);
                break;
            case 'month':
                oResult.setMonth(oResult.getMonth() + iDiff);
                break;
            case 'quarter':
                oResult.setMonth(oResult.getMonth() + iDiff * 3);
                break;
            case 'year':
                oResult.setFullYear(oResult.getFullYear() + iDiff);
                break;
            }
        }
        return oResult;
    }
};
DateFormat.prototype.formatRelative = function (oJSDate, bUTC, aRange, sTimezone) {
    var oDateUTC, iDiff, iDiffSeconds, sPattern, oToday = convertToTimezone(new Date(), sTimezone), sScale = this.oFormatOptions.relativeScale || 'day';
    iDiffSeconds = (oJSDate.getTime() - oToday.getTime()) / 1000;
    if (this.oFormatOptions.relativeScale === 'auto') {
        sScale = this._getScale(iDiffSeconds, this.aRelativeScales);
        sScale = fixScaleForMonths(oJSDate, oToday, sScale, iDiffSeconds);
    }
    if (!aRange) {
        aRange = this._mRanges[sScale];
    }
    if (sScale === 'year' || sScale === 'month' || sScale === 'day') {
        oToday = new Date(Date.UTC(oToday.getUTCFullYear(), oToday.getUTCMonth(), oToday.getUTCDate()));
        oDateUTC = new Date(0);
        oDateUTC.setUTCFullYear(oJSDate.getUTCFullYear(), oJSDate.getUTCMonth(), oJSDate.getUTCDate());
        oJSDate = oDateUTC;
    }
    iDiff = this._getDifference(sScale, [
        oToday,
        oJSDate
    ]);
    if (this.oFormatOptions.relativeScale !== 'auto' && (iDiff < aRange[0] || iDiff > aRange[1])) {
        return null;
    }
    sPattern = this.oLocaleData.getRelativePattern(sScale, iDiff, iDiffSeconds > 0, this.oFormatOptions.relativeStyle);
    return formatMessage(sPattern, [Math.abs(iDiff)]);
};
DateFormat.prototype._mRanges = {
    second: [
        -60,
        60
    ],
    minute: [
        -60,
        60
    ],
    hour: [
        -24,
        24
    ],
    day: [
        -6,
        6
    ],
    week: [
        -4,
        4
    ],
    month: [
        -12,
        12
    ],
    year: [
        -10,
        10
    ]
};
DateFormat.prototype._mScales = {
    second: 1,
    minute: 60,
    hour: 3600,
    day: 86400,
    week: 604800,
    month: 2592000,
    quarter: 7776000,
    year: 31536000
};
DateFormat.prototype._getScale = function (iDiffSeconds, aScales) {
    var sScale, sTestScale;
    iDiffSeconds = Math.abs(iDiffSeconds);
    for (var i = 0; i < aScales.length; i++) {
        sTestScale = aScales[i];
        if (iDiffSeconds >= this._mScales[sTestScale]) {
            sScale = sTestScale;
            break;
        }
    }
    if (!sScale) {
        sScale = aScales[aScales.length - 1];
    }
    return sScale;
};
function fixScaleForMonths(oJSDate, oToday, sScale, iDiffSeconds) {
    var iMonthDiff = Math.abs(oJSDate.getUTCMonth() - oToday.getUTCMonth());
    if (sScale === 'week' && iMonthDiff === 2) {
        return 'month';
    } else if (sScale === 'week' && iMonthDiff === 1) {
        if (oJSDate.getUTCDate() === oToday.getUTCDate() || iDiffSeconds < 0 && oJSDate.getUTCDate() < oToday.getUTCDate() || iDiffSeconds > 0 && oJSDate.getUTCDate() > oToday.getUTCDate()) {
            return 'month';
        }
    } else if (sScale === 'month' && iMonthDiff === 1) {
        if (iDiffSeconds > 0 && oJSDate.getUTCDate() < oToday.getUTCDate() || iDiffSeconds < 0 && oJSDate.getUTCDate() > oToday.getUTCDate()) {
            return 'week';
        }
    }
    return sScale;
}
function cutDateFields(oDate, iStartIndex) {
    var sMethodName, aFields = [
            'FullYear',
            'Month',
            'Date',
            'Hours',
            'Minutes',
            'Seconds',
            'Milliseconds'
        ], oDateCopy = new Date(oDate.getTime());
    for (var i = iStartIndex; i < aFields.length; i++) {
        sMethodName = 'setUTC' + aFields[iStartIndex];
        oDateCopy[sMethodName].apply(oDateCopy, [0]);
    }
    return oDateCopy;
}
var mRelativeDiffs = {
    year: function (oFromDate, oToDate) {
        return oToDate.getUTCFullYear() - oFromDate.getUTCFullYear();
    },
    month: function (oFromDate, oToDate) {
        return oToDate.getUTCMonth() - oFromDate.getUTCMonth() + this.year(oFromDate, oToDate) * 12;
    },
    week: function (oFromDate, oToDate, oFormat) {
        var iFromDay = oFormat._adaptDayOfWeek(oFromDate.getUTCDay());
        var iToDay = oFormat._adaptDayOfWeek(oToDate.getUTCDay());
        oFromDate = cutDateFields(oFromDate, 3);
        oToDate = cutDateFields(oToDate, 3);
        return (oToDate.getTime() - oFromDate.getTime() - (iToDay - iFromDay) * oFormat._mScales.day * 1000) / (oFormat._mScales.week * 1000);
    },
    day: function (oFromDate, oToDate, oFormat) {
        oFromDate = cutDateFields(oFromDate, 3);
        oToDate = cutDateFields(oToDate, 3);
        return (oToDate.getTime() - oFromDate.getTime()) / (oFormat._mScales.day * 1000);
    },
    hour: function (oFromDate, oToDate, oFormat) {
        oFromDate = cutDateFields(oFromDate, 4);
        oToDate = cutDateFields(oToDate, 4);
        return (oToDate.getTime() - oFromDate.getTime()) / (oFormat._mScales.hour * 1000);
    },
    minute: function (oFromDate, oToDate, oFormat) {
        oFromDate = cutDateFields(oFromDate, 5);
        oToDate = cutDateFields(oToDate, 5);
        return (oToDate.getTime() - oFromDate.getTime()) / (oFormat._mScales.minute * 1000);
    },
    second: function (oFromDate, oToDate, oFormat) {
        oFromDate = cutDateFields(oFromDate, 6);
        oToDate = cutDateFields(oToDate, 6);
        return (oToDate.getTime() - oFromDate.getTime()) / (oFormat._mScales.second * 1000);
    }
};
DateFormat.prototype._adaptDayOfWeek = function (iDayOfWeek) {
    var vCalendarWeekParameter = getCalendarWeekParameter(this.oFormatOptions), iFirstDayOfWeek;
    if (typeof vCalendarWeekParameter === 'object') {
        iFirstDayOfWeek = vCalendarWeekParameter.firstDayOfWeek;
    } else {
        iFirstDayOfWeek = CalendarUtils.getWeekConfigurationValues(vCalendarWeekParameter, this.oLocale).firstDayOfWeek;
    }
    var iDayNumberOfWeek = iDayOfWeek - (iFirstDayOfWeek - 1);
    if (iDayNumberOfWeek <= 0) {
        iDayNumberOfWeek += 7;
    }
    return iDayNumberOfWeek;
};
DateFormat.prototype._getDifference = function (sScale, aDates) {
    var oFromDate = aDates[0];
    var oToDate = aDates[1];
    return Math.round(mRelativeDiffs[sScale](oFromDate, oToDate, this));
};
DateFormat.prototype.getAllowedCharacters = function (aFormatArray) {
    if (this.oFormatOptions.relative) {
        return '';
    }
    var sAllowedCharacters = '';
    var bNumbers = false;
    var bAll = false;
    var oPart;
    for (var i = 0; i < aFormatArray.length; i++) {
        oPart = aFormatArray[i];
        switch (oPart.type) {
        case 'text':
            if (sAllowedCharacters.indexOf(oPart.value) < 0) {
                sAllowedCharacters += oPart.value;
            }
            break;
        case 'day':
        case 'year':
        case 'weekYear':
        case 'dayNumberOfWeek':
        case 'weekInYear':
        case 'hour0_23':
        case 'hour1_24':
        case 'hour0_11':
        case 'hour1_12':
        case 'minute':
        case 'second':
        case 'fractionalsecond':
            if (!bNumbers) {
                sAllowedCharacters += '0123456789';
                bNumbers = true;
            }
            break;
        case 'month':
        case 'monthStandalone':
            if (oPart.digits < 3) {
                if (!bNumbers) {
                    sAllowedCharacters += '0123456789';
                    bNumbers = true;
                }
            } else {
                bAll = true;
            }
            break;
        default:
            bAll = true;
            break;
        }
    }
    if (bAll) {
        sAllowedCharacters = '';
    }
    return sAllowedCharacters;
};
export default DateFormat;