/**
 * # Angular 1 injectable services
 *
 * This is a list of the objects which can be injected using angular's injector.
 *
 * There are three different kind of injectable objects:
 *
 * ## **Provider** objects
 * #### injectable into a `.config()` block during configtime
 *
 * - [[$uiRouterProvider]]: The UI-Router instance
 * - [[$stateProvider]]: State registration
 * - [[$transitionsProvider]]: Transition hooks
 * - [[$urlServiceProvider]]: All URL related public APIs
 *
 * - [[$uiViewScrollProvider]]: Disable ui-router view scrolling
 * - [[$urlRouterProvider]]: (deprecated) Url matching rules
 * - [[$urlMatcherFactoryProvider]]: (deprecated) Url parsing config
 *
 * ## **Service** objects
 * #### injectable globally during runtime
 *
 * - [[$uiRouter]]: The UI-Router instance
 * - [[$trace]]: Enable transition trace/debug
 * - [[$transitions]]: Transition hooks
 * - [[$state]]: Imperative state related APIs
 * - [[$stateRegistry]]: State registration
 * - [[$urlService]]: All URL related public APIs
 * - [[$uiRouterGlobals]]: Global variables
 * - [[$uiViewScroll]]: Scroll an element into view
 *
 * - [[$stateParams]]: (deprecated) Global state param values
 * - [[$urlRouter]]: (deprecated) URL synchronization
 * - [[$urlMatcherFactory]]: (deprecated) URL parsing config
 *
 * ## **Per-Transition** objects
 *
 * - These kind of objects are injectable into:
 *   - Resolves ([[Ng1StateDeclaration.resolve]]),
 *   - Transition Hooks ([[TransitionService.onStart]], etc),
 *   - Routed Controllers ([[Ng1ViewDeclaration.controller]])
 *
 * #### Different instances are injected based on the [[Transition]]
 *
 * - [[$transition$]]: The current Transition object
 * - [[$stateParams]]: State param values for pending Transition (deprecated)
 * - Any resolve data defined using [[Ng1StateDeclaration.resolve]]
 *
 * @ng1api
 * @preferred
 * @module injectables
 */ /** */
/**
 * The current (or pending) State Parameters
 *
 * An injectable global **Service Object** which holds the state parameters for the latest **SUCCESSFUL** transition.
 *
 * The values are not updated until *after* a `Transition` successfully completes.
 *
 * **Also:** an injectable **Per-Transition Object** object which holds the pending state parameters for the pending `Transition` currently running.
 *
 * ### Deprecation warning:
 *
 * The value injected for `$stateParams` is different depending on where it is injected.
 *
 * - When injected into an angular service, the object injected is the global **Service Object** with the parameter values for the latest successful `Transition`.
 * - When injected into transition hooks, resolves, or view controllers, the object is the **Per-Transition Object** with the parameter values for the running `Transition`.
 *
 * Because of these confusing details, this service is deprecated.
 *
 * ### Instead of using the global `$stateParams` service object,
 * inject [[$uiRouterGlobals]] and use [[UIRouterGlobals.params]]
 *
 * ```js
 * MyService.$inject = ['$uiRouterGlobals'];
 * function MyService($uiRouterGlobals) {
 *   return {
 *     paramValues: function () {
 *       return $uiRouterGlobals.params;
 *     }
 *   }
 * }
 * ```
 *
 * ### Instead of using the per-transition `$stateParams` object,
 * inject the current `Transition` (as [[$transition$]]) and use [[Transition.params]]
 *
 * ```js
 * MyController.$inject = ['$transition$'];
 * function MyController($transition$) {
 *   var username = $transition$.params().username;
 *   // .. do something with username
 * }
 * ```
 *
 * ---
 *
 * This object can be injected into other services.
 *
 * #### Deprecated Example:
 * ```js
 * SomeService.$inject = ['$http', '$stateParams'];
 * function SomeService($http, $stateParams) {
 *   return {
 *     getUser: function() {
 *       return $http.get('/api/users/' + $stateParams.username);
 *     }
 *   }
 * };
 * angular.service('SomeService', SomeService);
 * ```
 * @deprecated
 */
var $stateParams;
/**
 * Global UI-Router variables
 *
 * The router global state as a **Service Object** (injectable during runtime).
 *
 * This object contains globals such as the current state and current parameter values.
 */
var $uiRouterGlobals;
/**
 * The UI-Router instance
 *
 * The [[UIRouter]] singleton (the router instance) as a **Service Object** (injectable during runtime).
 *
 * This object is the UI-Router singleton instance, created by angular dependency injection during application bootstrap.
 * It has references to the other UI-Router services
 *
 * #### Note: This object is also exposed as [[$uiRouterProvider]] for injection during angular config time.
 */
var $uiRouter;
/**
 * The UI-Router instance
 *
 * The [[UIRouter]] singleton (the router instance) as a **Provider Object** (injectable during config phase).
 *
 * This object is the UI-Router singleton instance, created by angular dependency injection during application bootstrap.
 * It has references to the other UI-Router services
 *
 * #### Note: This object is also exposed as [[$uiRouter]] for injection during runtime.
 */
var $uiRouterProvider;
/**
 * Transition debug/tracing
 *
 * The [[Trace]] singleton as a **Service Object** (injectable during runtime).
 *
 * Enables or disables Transition tracing which can help to debug issues.
 */
var $trace;
/**
 * The Transition Service
 *
 * The [[TransitionService]] singleton as a **Service Object** (injectable during runtime).
 *
 * This angular service exposes the [[TransitionService]] singleton, which is primarily
 * used to register global transition hooks.
 *
 * #### Note: This object is also exposed as [[$transitionsProvider]] for injection during the config phase.
 */
var $transitions;
/**
 * The Transition Service
 *
 * The [[TransitionService]] singleton as a **Provider Object** (injectable during config phase)
 *
 * This angular service exposes the [[TransitionService]] singleton, which is primarily
 * used to register global transition hooks.
 *
 * #### Note: This object is also exposed as [[$transitions]] for injection during runtime.
 */
var $transitionsProvider;
/**
 * The current [[Transition]] object
 *
 * The current [[Transition]] object as a **Per-Transition Object** (injectable into Resolve, Hooks, Controllers)
 *
 * This object returns information about the current transition, including:
 *
 * - To/from states
 * - To/from parameters
 * - Transition options
 * - States being entered, exited, and retained
 * - Resolve data
 * - A Promise for the transition
 * - Any transition failure information
 * - An injector for both Service and Per-Transition Objects
 */
var $transition$;
/**
 * The State Service
 *
 * The [[StateService]] singleton as a **Service Object** (injectable during runtime).
 *
 * This service used to manage and query information on registered states.
 * It exposes state related APIs including:
 *
 * - Start a [[Transition]]
 * - Imperatively lazy load states
 * - Check if a state is currently active
 * - Look up states by name
 * - Build URLs for a state+parameters
 * - Configure the global Transition error handler
 *
 * This angular service exposes the [[StateService]] singleton.
 */
var $state;
/**
 * The State Registry
 *
 * The [[StateRegistry]] singleton as a **Service Object** (injectable during runtime).
 *
 * This service is used to register/deregister states.
 * It has state registration related APIs including:
 *
 * - Register/deregister states
 * - Listen for state registration/deregistration
 * - Get states by name
 * - Add state decorators (to customize the state creation process)
 *
 * #### Note: This object is also exposed as [[$stateRegistryProvider]] for injection during the config phase.
 */
var $stateRegistry;
/**
 * The State Registry
 *
 * The [[StateRegistry]] singleton as a **Provider Object** (injectable during config time).
 *
 * This service is used to register/deregister states.
 * It has state registration related APIs including:
 *
 * - Register/deregister states
 * - Listen for state registration/deregistration
 * - Get states by name
 * - Add state decorators (to customize the state creation process)
 *
 * #### Note: This object is also exposed as [[$stateRegistry]] for injection during runtime.
 */
var $stateRegistryProvider;
/**
 * The View Scroll provider
 *
 * The [[UIViewScrollProvider]] as a **Provider Object** (injectable during config time).
 *
 * This angular service exposes the [[UIViewScrollProvider]] singleton and is
 * used to disable UI-Router's scroll behavior.
 */
var $uiViewScrollProvider;
/**
 * The View Scroll function
 *
 * The View Scroll function as a **Service Object** (injectable during runtime).
 *
 * This is a function that scrolls an element into view.
 * The element is scrolled after a `$timeout` so the DOM has time to refresh.
 *
 * If you prefer to rely on `$anchorScroll` to scroll the view to the anchor,
 * this can be enabled by calling [[UIViewScrollProvider.useAnchorScroll]].
 *
 * Note: this function is used by the [[directives.uiView]] when the `autoscroll` expression evaluates to true.
 */
var $uiViewScroll;
/**
 * The StateProvider
 *
 * An angular1-only [[StateProvider]] as a **Provider Object** (injectable during config time).
 *
 * This angular service exposes the [[StateProvider]] singleton.
 *
 * The `StateProvider` is primarily used to register states or add custom state decorators.
 *
 * ##### Note: This provider is a ng1 vestige.
 * It is a passthrough to [[$stateRegistry]] and [[$state]].
 */
var $stateProvider;
/**
 * The URL Service Provider
 *
 * The [[UrlService]] singleton as a **Provider Object** (injectable during the angular config phase).
 *
 * A service used to configure and interact with the URL.
 * It has URL related APIs including:
 *
 * - register custom Parameter types `UrlService.config.type` ([[UrlConfigApi.type]])
 * - add URL rules: `UrlService.rules.when` ([[UrlRulesApi.when]])
 * - configure behavior when no url matches: `UrlService.rules.otherwise` ([[UrlRulesApi.otherwise]])
 * - delay initial URL synchronization [[UrlService.deferIntercept]].
 * - get or set the current url: [[UrlService.url]]
 *
 * ##### Note: This service can also be injected during runtime as [[$urlService]].
 */
var $urlServiceProvider;
/**
 * The URL Service
 *
 * The [[UrlService]] singleton as a **Service Object** (injectable during runtime).
 *
 * Note: This service can also be injected during the config phase as [[$urlServiceProvider]].
 *
 * Used to configure the URL.
 * It has URL related APIs including:
 *
 * - register custom Parameter types `UrlService.config.type` ([[UrlConfigApi.type]])
 * - add URL rules: `UrlService.rules.when` ([[UrlRulesApi.when]])
 * - configure behavior when no url matches: `UrlService.rules.otherwise` ([[UrlRulesApi.otherwise]])
 * - delay initial URL synchronization [[UrlService.deferIntercept]].
 * - get or set the current url: [[UrlService.url]]
 *
 * ##### Note: This service can also be injected during the config phase as [[$urlServiceProvider]].
 */
var $urlService;
/**
 * The URL Router Provider
 *
 * ### Deprecation warning: This object is now considered internal. Use [[$urlServiceProvider]] instead.
 *
 * The [[UrlRouter]] singleton as a **Provider Object** (injectable during config time).
 *
 * #### Note: This object is also exposed as [[$urlRouter]] for injection during runtime.
 *
 * @deprecated
 */
var $urlRouterProvider;
/**
 * The Url Router
 *
 * ### Deprecation warning: This object is now considered internal. Use [[$urlService]] instead.
 *
 * The [[UrlRouter]] singleton as a **Service Object** (injectable during runtime).
 *
 * #### Note: This object is also exposed as [[$urlRouterProvider]] for injection during angular config time.
 *
 * @deprecated
 */
var $urlRouter;
/**
 * The URL Matcher Factory
 *
 * ### Deprecation warning: This object is now considered internal. Use [[$urlService]] instead.
 *
 * The [[UrlMatcherFactory]] singleton as a **Service Object** (injectable during runtime).
 *
 * This service is used to set url mapping options, define custom parameter types, and create [[UrlMatcher]] objects.
 *
 * #### Note: This object is also exposed as [[$urlMatcherFactoryProvider]] for injection during angular config time.
 *
 * @deprecated
 */
var $urlMatcherFactory;
/**
 * The URL Matcher Factory
 *
 * ### Deprecation warning: This object is now considered internal. Use [[$urlService]] instead.
 *
 * The [[UrlMatcherFactory]] singleton as a **Provider Object** (injectable during config time).
 *
 * This service is used to set url mapping options, define custom parameter types, and create [[UrlMatcher]] objects.
 *
 * #### Note: This object is also exposed as [[$urlMatcherFactory]] for injection during runtime.
 *
 * @deprecated
 */
var $urlMatcherFactoryProvider;
//# sourceMappingURL=data:application/json;base64,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