/** @module view */
/** for typedoc */
import { ng as angular } from "./angular";
import { isArray, isDefined, isFunction, isObject, services, tail, kebobString, unnestR, Resolvable } from "@uirouter/core";
/**
 * Service which manages loading of templates from a ViewConfig.
 */
var TemplateFactory = (function () {
    function TemplateFactory() {
        var _this = this;
        /** @hidden */ this._useHttp = angular.version.minor < 3;
        /** @hidden */ this.$get = ['$http', '$templateCache', '$injector', function ($http, $templateCache, $injector) {
                _this.$templateRequest = $injector.has && $injector.has('$templateRequest') && $injector.get('$templateRequest');
                _this.$http = $http;
                _this.$templateCache = $templateCache;
                return _this;
            }];
    }
    /** @hidden */
    TemplateFactory.prototype.useHttpService = function (value) {
        this._useHttp = value;
    };
    ;
    /**
     * Creates a template from a configuration object.
     *
     * @param config Configuration object for which to load a template.
     * The following properties are search in the specified order, and the first one
     * that is defined is used to create the template:
     *
     * @param params  Parameters to pass to the template function.
     * @param context The resolve context associated with the template's view
     *
     * @return {string|object}  The template html as a string, or a promise for
     * that string,or `null` if no template is configured.
     */
    TemplateFactory.prototype.fromConfig = function (config, params, context) {
        var defaultTemplate = "<ui-view></ui-view>";
        var asTemplate = function (result) { return services.$q.when(result).then(function (str) { return ({ template: str }); }); };
        var asComponent = function (result) { return services.$q.when(result).then(function (str) { return ({ component: str }); }); };
        return (isDefined(config.template) ? asTemplate(this.fromString(config.template, params)) :
            isDefined(config.templateUrl) ? asTemplate(this.fromUrl(config.templateUrl, params)) :
                isDefined(config.templateProvider) ? asTemplate(this.fromProvider(config.templateProvider, params, context)) :
                    isDefined(config.component) ? asComponent(config.component) :
                        isDefined(config.componentProvider) ? asComponent(this.fromComponentProvider(config.componentProvider, params, context)) :
                            asTemplate(defaultTemplate));
    };
    ;
    /**
     * Creates a template from a string or a function returning a string.
     *
     * @param template html template as a string or function that returns an html template as a string.
     * @param params Parameters to pass to the template function.
     *
     * @return {string|object} The template html as a string, or a promise for that
     * string.
     */
    TemplateFactory.prototype.fromString = function (template, params) {
        return isFunction(template) ? template(params) : template;
    };
    ;
    /**
     * Loads a template from the a URL via `$http` and `$templateCache`.
     *
     * @param {string|Function} url url of the template to load, or a function
     * that returns a url.
     * @param {Object} params Parameters to pass to the url function.
     * @return {string|Promise.<string>} The template html as a string, or a promise
     * for that string.
     */
    TemplateFactory.prototype.fromUrl = function (url, params) {
        if (isFunction(url))
            url = url(params);
        if (url == null)
            return null;
        if (this._useHttp) {
            return this.$http.get(url, { cache: this.$templateCache, headers: { Accept: 'text/html' } })
                .then(function (response) {
                return response.data;
            });
        }
        return this.$templateRequest(url);
    };
    ;
    /**
     * Creates a template by invoking an injectable provider function.
     *
     * @param provider Function to invoke via `locals`
     * @param {Function} injectFn a function used to invoke the template provider
     * @return {string|Promise.<string>} The template html as a string, or a promise
     * for that string.
     */
    TemplateFactory.prototype.fromProvider = function (provider, params, context) {
        var deps = services.$injector.annotate(provider);
        var providerFn = isArray(provider) ? tail(provider) : provider;
        var resolvable = new Resolvable("", providerFn, deps);
        return resolvable.get(context);
    };
    ;
    /**
     * Creates a component's template by invoking an injectable provider function.
     *
     * @param provider Function to invoke via `locals`
     * @param {Function} injectFn a function used to invoke the template provider
     * @return {string} The template html as a string: "<component-name input1='::$resolve.foo'></component-name>".
     */
    TemplateFactory.prototype.fromComponentProvider = function (provider, params, context) {
        var deps = services.$injector.annotate(provider);
        var providerFn = isArray(provider) ? tail(provider) : provider;
        var resolvable = new Resolvable("", providerFn, deps);
        return resolvable.get(context);
    };
    ;
    /**
     * Creates a template from a component's name
     *
     * This implements route-to-component.
     * It works by retrieving the component (directive) metadata from the injector.
     * It analyses the component's bindings, then constructs a template that instantiates the component.
     * The template wires input and output bindings to resolves or from the parent component.
     *
     * @param uiView {object} The parent ui-view (for binding outputs to callbacks)
     * @param context The ResolveContext (for binding outputs to callbacks returned from resolves)
     * @param component {string} Component's name in camel case.
     * @param bindings An object defining the component's bindings: {foo: '<'}
     * @return {string} The template as a string: "<component-name input1='::$resolve.foo'></component-name>".
     */
    TemplateFactory.prototype.makeComponentTemplate = function (uiView, context, component, bindings) {
        bindings = bindings || {};
        // Bind once prefix
        var prefix = angular.version.minor >= 3 ? "::" : "";
        // Convert to kebob name. Add x- prefix if the string starts with `x-` or `data-`
        var kebob = function (camelCase) {
            var kebobed = kebobString(camelCase);
            return /^(x|data)-/.exec(kebobed) ? "x-" + kebobed : kebobed;
        };
        var attributeTpl = function (input) {
            var name = input.name, type = input.type;
            var attrName = kebob(name);
            // If the ui-view has an attribute which matches a binding on the routed component
            // then pass that attribute through to the routed component template.
            // Prefer ui-view wired mappings to resolve data, unless the resolve was explicitly bound using `bindings:`
            if (uiView.attr(attrName) && !bindings[name])
                return attrName + "='" + uiView.attr(attrName) + "'";
            var resolveName = bindings[name] || name;
            // Pre-evaluate the expression for "@" bindings by enclosing in {{ }}
            // some-attr="{{ ::$resolve.someResolveName }}"
            if (type === '@')
                return attrName + "='{{" + prefix + "$resolve." + resolveName + "}}'";
            // Wire "&" callbacks to resolves that return a callback function
            // Get the result of the resolve (should be a function) and annotate it to get its arguments.
            // some-attr="$resolve.someResolveResultName(foo, bar)"
            if (type === '&') {
                var res = context.getResolvable(resolveName);
                var fn = res && res.data;
                var args = fn && services.$injector.annotate(fn) || [];
                // account for array style injection, i.e., ['foo', function(foo) {}]
                var arrayIdxStr = isArray(fn) ? "[" + (fn.length - 1) + "]" : '';
                return attrName + "='$resolve." + resolveName + arrayIdxStr + "(" + args.join(",") + ")'";
            }
            // some-attr="::$resolve.someResolveName"
            return attrName + "='" + prefix + "$resolve." + resolveName + "'";
        };
        var attrs = getComponentBindings(component).map(attributeTpl).join(" ");
        var kebobName = kebob(component);
        return "<" + kebobName + " " + attrs + "></" + kebobName + ">";
    };
    ;
    return TemplateFactory;
}());
export { TemplateFactory };
// Gets all the directive(s)' inputs ('@', '=', and '<') and outputs ('&')
function getComponentBindings(name) {
    var cmpDefs = services.$injector.get(name + "Directive"); // could be multiple
    if (!cmpDefs || !cmpDefs.length)
        throw new Error("Unable to find component named '" + name + "'");
    return cmpDefs.map(getBindings).reduce(unnestR, []);
}
// Given a directive definition, find its object input attributes
// Use different properties, depending on the type of directive (component, bindToController, normal)
var getBindings = function (def) {
    if (isObject(def.bindToController))
        return scopeBindings(def.bindToController);
    return scopeBindings(def.scope);
};
// for ng 1.2 style, process the scope: { input: "=foo" }
// for ng 1.3 through ng 1.5, process the component's bindToController: { input: "=foo" } object
var scopeBindings = function (bindingsObj) { return Object.keys(bindingsObj || {})
    .map(function (key) { return [key, /^([=<@&])[?]?(.*)/.exec(bindingsObj[key])]; })
    .filter(function (tuple) { return isDefined(tuple) && isArray(tuple[1]); })
    .map(function (tuple) { return ({ name: tuple[1][2] || tuple[0], type: tuple[1][1] }); }); };
//# sourceMappingURL=data:application/json;base64,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