/** @module url */ /** */
import { BaseUrlRule } from "@uirouter/core";
import { services, isString, isFunction, isArray, identity } from "@uirouter/core";
/**
 * Manages rules for client-side URL
 *
 * ### Deprecation warning:
 * This class is now considered to be an internal API
 * Use the [[UrlService]] instead.
 * For configuring URL rules, use the [[UrlRulesApi]] which can be found as [[UrlService.rules]].
 *
 * This class manages the router rules for what to do when the URL changes.
 *
 * This provider remains for backwards compatibility.
 *
 * @deprecated
 */
var UrlRouterProvider = (function () {
    /** @hidden */
    function UrlRouterProvider(router) {
        this._router = router;
        this._urlRouter = router.urlRouter;
    }
    /** @hidden */
    UrlRouterProvider.prototype.$get = function () {
        var urlRouter = this._urlRouter;
        urlRouter.update(true);
        if (!urlRouter.interceptDeferred)
            urlRouter.listen();
        return urlRouter;
    };
    /**
     * Registers a url handler function.
     *
     * Registers a low level url handler (a `rule`).
     * A rule detects specific URL patterns and returns a redirect, or performs some action.
     *
     * If a rule returns a string, the URL is replaced with the string, and all rules are fired again.
     *
     * #### Example:
     * ```js
     * var app = angular.module('app', ['ui.router.router']);
     *
     * app.config(function ($urlRouterProvider) {
     *   // Here's an example of how you might allow case insensitive urls
     *   $urlRouterProvider.rule(function ($injector, $location) {
     *     var path = $location.path(),
     *         normalized = path.toLowerCase();
     *
     *     if (path !== normalized) {
     *       return normalized;
     *     }
     *   });
     * });
     * ```
     *
     * @param ruleFn
     * Handler function that takes `$injector` and `$location` services as arguments.
     * You can use them to detect a url and return a different url as a string.
     *
     * @return [[UrlRouterProvider]] (`this`)
     */
    UrlRouterProvider.prototype.rule = function (ruleFn) {
        var _this = this;
        if (!isFunction(ruleFn))
            throw new Error("'rule' must be a function");
        var match = function () {
            return ruleFn(services.$injector, _this._router.locationService);
        };
        var rule = new BaseUrlRule(match, identity);
        this._urlRouter.rule(rule);
        return this;
    };
    ;
    /**
     * Defines the path or behavior to use when no url can be matched.
     *
     * #### Example:
     * ```js
     * var app = angular.module('app', ['ui.router.router']);
     *
     * app.config(function ($urlRouterProvider) {
     *   // if the path doesn't match any of the urls you configured
     *   // otherwise will take care of routing the user to the
     *   // specified url
     *   $urlRouterProvider.otherwise('/index');
     *
     *   // Example of using function rule as param
     *   $urlRouterProvider.otherwise(function ($injector, $location) {
     *     return '/a/valid/url';
     *   });
     * });
     * ```
     *
     * @param rule
     * The url path you want to redirect to or a function rule that returns the url path or performs a `$state.go()`.
     * The function version is passed two params: `$injector` and `$location` services, and should return a url string.
     *
     * @return {object} `$urlRouterProvider` - `$urlRouterProvider` instance
     */
    UrlRouterProvider.prototype.otherwise = function (rule) {
        var _this = this;
        var urlRouter = this._urlRouter;
        if (isString(rule)) {
            urlRouter.otherwise(rule);
        }
        else if (isFunction(rule)) {
            urlRouter.otherwise(function () { return rule(services.$injector, _this._router.locationService); });
        }
        else {
            throw new Error("'rule' must be a string or function");
        }
        return this;
    };
    ;
    /**
     * Registers a handler for a given url matching.
     *
     * If the handler is a string, it is
     * treated as a redirect, and is interpolated according to the syntax of match
     * (i.e. like `String.replace()` for `RegExp`, or like a `UrlMatcher` pattern otherwise).
     *
     * If the handler is a function, it is injectable.
     * It gets invoked if `$location` matches.
     * You have the option of inject the match object as `$match`.
     *
     * The handler can return
     *
     * - **falsy** to indicate that the rule didn't match after all, then `$urlRouter`
     *   will continue trying to find another one that matches.
     * - **string** which is treated as a redirect and passed to `$location.url()`
     * - **void** or any **truthy** value tells `$urlRouter` that the url was handled.
     *
     * #### Example:
     * ```js
     * var app = angular.module('app', ['ui.router.router']);
     *
     * app.config(function ($urlRouterProvider) {
     *   $urlRouterProvider.when($state.url, function ($match, $stateParams) {
     *     if ($state.$current.navigable !== state ||
     *         !equalForKeys($match, $stateParams) {
     *      $state.transitionTo(state, $match, false);
     *     }
     *   });
     * });
     * ```
     *
     * @param what A pattern string to match, compiled as a [[UrlMatcher]].
     * @param handler The path (or function that returns a path) that you want to redirect your user to.
     * @param ruleCallback [optional] A callback that receives the `rule` registered with [[UrlMatcher.rule]]
     *
     * Note: the handler may also invoke arbitrary code, such as `$state.go()`
     */
    UrlRouterProvider.prototype.when = function (what, handler) {
        if (isArray(handler) || isFunction(handler)) {
            handler = UrlRouterProvider.injectableHandler(this._router, handler);
        }
        this._urlRouter.when(what, handler);
        return this;
    };
    ;
    UrlRouterProvider.injectableHandler = function (router, handler) {
        return function (match) {
            return services.$injector.invoke(handler, null, { $match: match, $stateParams: router.globals.params });
        };
    };
    /**
     * Disables monitoring of the URL.
     *
     * Call this method before UI-Router has bootstrapped.
     * It will stop UI-Router from performing the initial url sync.
     *
     * This can be useful to perform some asynchronous initialization before the router starts.
     * Once the initialization is complete, call [[listen]] to tell UI-Router to start watching and synchronizing the URL.
     *
     * #### Example:
     * ```js
     * var app = angular.module('app', ['ui.router']);
     *
     * app.config(function ($urlRouterProvider) {
     *   // Prevent $urlRouter from automatically intercepting URL changes;
     *   $urlRouterProvider.deferIntercept();
     * })
     *
     * app.run(function (MyService, $urlRouter, $http) {
     *   $http.get("/stuff").then(function(resp) {
     *     MyService.doStuff(resp.data);
     *     $urlRouter.listen();
     *     $urlRouter.sync();
     *   });
     * });
     * ```
     *
     * @param defer Indicates whether to defer location change interception.
     *        Passing no parameter is equivalent to `true`.
     */
    UrlRouterProvider.prototype.deferIntercept = function (defer) {
        this._urlRouter.deferIntercept(defer);
    };
    ;
    return UrlRouterProvider;
}());
export { UrlRouterProvider };
//# sourceMappingURL=data:application/json;base64,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