"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * # Angular 1 Directives
 *
 * These are the directives included in UI-Router for Angular 1.
 * These directives are used in templates to create viewports and link/navigate to states.
 *
 * @ng1api
 * @preferred
 * @module directives
 */ /** for typedoc */
var angular_1 = require("../angular");
var core_1 = require("@uirouter/core");
/** @hidden */
function parseStateRef(ref) {
    var paramsOnly = ref.match(/^\s*({[^}]*})\s*$/), parsed;
    if (paramsOnly)
        ref = '(' + paramsOnly[1] + ')';
    parsed = ref.replace(/\n/g, " ").match(/^\s*([^(]*?)\s*(\((.*)\))?\s*$/);
    if (!parsed || parsed.length !== 4)
        throw new Error("Invalid state ref '" + ref + "'");
    return { state: parsed[1] || null, paramExpr: parsed[3] || null };
}
/** @hidden */
function stateContext(el) {
    var $uiView = el.parent().inheritedData('$uiView');
    var path = core_1.parse('$cfg.path')($uiView);
    return path ? core_1.tail(path).state.name : undefined;
}
/** @hidden */
function processedDef($state, $element, def) {
    var uiState = def.uiState || $state.current.name;
    var uiStateOpts = core_1.extend(defaultOpts($element, $state), def.uiStateOpts || {});
    var href = $state.href(uiState, def.uiStateParams, uiStateOpts);
    return { uiState: uiState, uiStateParams: def.uiStateParams, uiStateOpts: uiStateOpts, href: href };
}
/** @hidden */
function getTypeInfo(el) {
    // SVGAElement does not use the href attribute, but rather the 'xlinkHref' attribute.
    var isSvg = Object.prototype.toString.call(el.prop('href')) === '[object SVGAnimatedString]';
    var isForm = el[0].nodeName === "FORM";
    return {
        attr: isForm ? "action" : (isSvg ? 'xlink:href' : 'href'),
        isAnchor: el.prop("tagName").toUpperCase() === "A",
        clickable: !isForm
    };
}
/** @hidden */
function clickHook(el, $state, $timeout, type, getDef) {
    return function (e) {
        var button = e.which || e.button, target = getDef();
        if (!(button > 1 || e.ctrlKey || e.metaKey || e.shiftKey || el.attr('target'))) {
            // HACK: This is to allow ng-clicks to be processed before the transition is initiated:
            var transition = $timeout(function () {
                $state.go(target.uiState, target.uiStateParams, target.uiStateOpts);
            });
            e.preventDefault();
            // if the state has no URL, ignore one preventDefault from the <a> directive.
            var ignorePreventDefaultCount = type.isAnchor && !target.href ? 1 : 0;
            e.preventDefault = function () {
                if (ignorePreventDefaultCount-- <= 0)
                    $timeout.cancel(transition);
            };
        }
    };
}
/** @hidden */
function defaultOpts(el, $state) {
    return {
        relative: stateContext(el) || $state.$current,
        inherit: true,
        source: "sref"
    };
}
/** @hidden */
function bindEvents(element, scope, hookFn, uiStateOpts) {
    var events;
    if (uiStateOpts) {
        events = uiStateOpts.events;
    }
    if (!core_1.isArray(events)) {
        events = ['click'];
    }
    var on = element.on ? 'on' : 'bind';
    for (var _i = 0, events_1 = events; _i < events_1.length; _i++) {
        var event_1 = events_1[_i];
        element[on](event_1, hookFn);
    }
    scope.$on('$destroy', function () {
        var off = element.off ? 'off' : 'unbind';
        for (var _i = 0, events_2 = events; _i < events_2.length; _i++) {
            var event_2 = events_2[_i];
            element[off](event_2, hookFn);
        }
    });
}
/**
 * `ui-sref`: A directive for linking to a state
 *
 * A directive which links to a state (and optionally, parameters).
 * When clicked, this directive activates the linked state with the supplied parameter values.
 *
 * ### Linked State
 * The attribute value of the `ui-sref` is the name of the state to link to.
 *
 * #### Example:
 * This will activate the `home` state when the link is clicked.
 * ```html
 * <a ui-sref="home">Home</a>
 * ```
 *
 * ### Relative Links
 * You can also use relative state paths within `ui-sref`, just like a relative path passed to `$state.go()` ([[StateService.go]]).
 * You just need to be aware that the path is relative to the state that *created* the link.
 * This allows a state to create a relative `ui-sref` which always targets the same destination.
 *
 * #### Example:
 * Both these links are relative to the parent state, even when a child state is currently active.
 * ```html
 * <a ui-sref=".child1">child 1 state</a>
 * <a ui-sref=".child2">child 2 state</a>
 * ```
 *
 * This link activates the parent state.
 * ```html
 * <a ui-sref="^">Return</a>
 * ```
 *
 * ### hrefs
 * If the linked state has a URL, the directive will automatically generate and
 * update the `href` attribute (using the [[StateService.href]]  method).
 *
 * #### Example:
 * Assuming the `users` state has a url of `/users/`
 * ```html
 * <a ui-sref="users" href="/users/">Users</a>
 * ```
 *
 * ### Parameter Values
 * In addition to the state name, a `ui-sref` can include parameter values which are applied when activating the state.
 * Param values can be provided in the `ui-sref` value after the state name, enclosed by parentheses.
 * The content inside the parentheses is an expression, evaluated to the parameter values.
 *
 * #### Example:
 * This example renders a list of links to users.
 * The state's `userId` parameter value comes from each user's `user.id` property.
 * ```html
 * <li ng-repeat="user in users">
 *   <a ui-sref="users.detail({ userId: user.id })">{{ user.displayName }}</a>
 * </li>
 * ```
 *
 * Note:
 * The parameter values expression is `$watch`ed for updates.
 *
 * ### Transition Options
 * You can specify [[TransitionOptions]] to pass to [[StateService.go]] by using the `ui-sref-opts` attribute.
 * Options are restricted to `location`, `inherit`, and `reload`.
 *
 * #### Example:
 * ```html
 * <a ui-sref="home" ui-sref-opts="{ reload: true }">Home</a>
 * ```
 *
 * ### Other DOM Events
 *
 * You can also customize which DOM events to respond to (instead of `click`) by
 * providing an `events` array in the `ui-sref-opts` attribute.
 *
 * #### Example:
 * ```html
 * <input type="text" ui-sref="contacts" ui-sref-opts="{ events: ['change', 'blur'] }">
 * ```
 *
 * ### Highlighting the active link
 * This directive can be used in conjunction with [[uiSrefActive]] to highlight the active link.
 *
 * ### Examples
 * If you have the following template:
 *
 * ```html
 * <a ui-sref="home">Home</a>
 * <a ui-sref="about">About</a>
 * <a ui-sref="{page: 2}">Next page</a>
 *
 * <ul>
 *     <li ng-repeat="contact in contacts">
 *         <a ui-sref="contacts.detail({ id: contact.id })">{{ contact.name }}</a>
 *     </li>
 * </ul>
 * ```
 *
 * Then (assuming the current state is `contacts`) the rendered html including hrefs would be:
 *
 * ```html
 * <a href="#/home" ui-sref="home">Home</a>
 * <a href="#/about" ui-sref="about">About</a>
 * <a href="#/contacts?page=2" ui-sref="{page: 2}">Next page</a>
 *
 * <ul>
 *     <li ng-repeat="contact in contacts">
 *         <a href="#/contacts/1" ui-sref="contacts.detail({ id: contact.id })">Joe</a>
 *     </li>
 *     <li ng-repeat="contact in contacts">
 *         <a href="#/contacts/2" ui-sref="contacts.detail({ id: contact.id })">Alice</a>
 *     </li>
 *     <li ng-repeat="contact in contacts">
 *         <a href="#/contacts/3" ui-sref="contacts.detail({ id: contact.id })">Bob</a>
 *     </li>
 * </ul>
 *
 * <a href="#/home" ui-sref="home" ui-sref-opts="{reload: true}">Home</a>
 * ```
 *
 * ### Notes
 *
 * - You can use `ui-sref` to change **only the parameter values** by omitting the state name and parentheses.
 * #### Example:
 * Sets the `lang` parameter to `en` and remains on the same state.
 *
 * ```html
 * <a ui-sref="{ lang: 'en' }">English</a>
 * ```
 *
 * - A middle-click, right-click, or ctrl-click is handled (natively) by the browser to open the href in a new window, for example.
 *
 * - Unlike the parameter values expression, the state name is not `$watch`ed (for performance reasons).
 * If you need to dynamically update the state being linked to, use the fully dynamic [[uiState]] directive.
 */
var uiSref;
uiSref = ['$uiRouter', '$timeout',
    function $StateRefDirective($uiRouter, $timeout) {
        var $state = $uiRouter.stateService;
        return {
            restrict: 'A',
            require: ['?^uiSrefActive', '?^uiSrefActiveEq'],
            link: function (scope, element, attrs, uiSrefActive) {
                var type = getTypeInfo(element);
                var active = uiSrefActive[1] || uiSrefActive[0];
                var unlinkInfoFn = null;
                var hookFn;
                var rawDef = {};
                var getDef = function () { return processedDef($state, element, rawDef); };
                var ref = parseStateRef(attrs.uiSref);
                rawDef.uiState = ref.state;
                rawDef.uiStateOpts = attrs.uiSrefOpts ? scope.$eval(attrs.uiSrefOpts) : {};
                function update() {
                    var def = getDef();
                    if (unlinkInfoFn)
                        unlinkInfoFn();
                    if (active)
                        unlinkInfoFn = active.$$addStateInfo(def.uiState, def.uiStateParams);
                    if (def.href != null)
                        attrs.$set(type.attr, def.href);
                }
                if (ref.paramExpr) {
                    scope.$watch(ref.paramExpr, function (val) {
                        rawDef.uiStateParams = core_1.extend({}, val);
                        update();
                    }, true);
                    rawDef.uiStateParams = core_1.extend({}, scope.$eval(ref.paramExpr));
                }
                update();
                scope.$on('$destroy', $uiRouter.stateRegistry.onStatesChanged(update));
                scope.$on('$destroy', $uiRouter.transitionService.onSuccess({}, update));
                if (!type.clickable)
                    return;
                hookFn = clickHook(element, $state, $timeout, type, getDef);
                bindEvents(element, scope, hookFn, rawDef.uiStateOpts);
            }
        };
    }];
/**
 * `ui-state`: A fully dynamic directive for linking to a state
 *
 * A directive which links to a state (and optionally, parameters).
 * When clicked, this directive activates the linked state with the supplied parameter values.
 *
 * **This directive is very similar to [[uiSref]], but it `$observe`s and `$watch`es/evaluates all its inputs.**
 *
 * A directive which links to a state (and optionally, parameters).
 * When clicked, this directive activates the linked state with the supplied parameter values.
 *
 * ### Linked State
 * The attribute value of `ui-state` is an expression which is `$watch`ed and evaluated as the state to link to.
 * **This is in contrast with `ui-sref`, which takes a state name as a string literal.**
 *
 * #### Example:
 * Create a list of links.
 * ```html
 * <li ng-repeat="link in navlinks">
 *   <a ui-state="link.state">{{ link.displayName }}</a>
 * </li>
 * ```
 *
 * ### Relative Links
 * If the expression evaluates to a relative path, it is processed like [[uiSref]].
 * You just need to be aware that the path is relative to the state that *created* the link.
 * This allows a state to create relative `ui-state` which always targets the same destination.
 *
 * ### hrefs
 * If the linked state has a URL, the directive will automatically generate and
 * update the `href` attribute (using the [[StateService.href]]  method).
 *
 * ### Parameter Values
 * In addition to the state name expression, a `ui-state` can include parameter values which are applied when activating the state.
 * Param values should be provided using the `ui-state-params` attribute.
 * The `ui-state-params` attribute value is `$watch`ed and evaluated as an expression.
 *
 * #### Example:
 * This example renders a list of links with param values.
 * The state's `userId` parameter value comes from each user's `user.id` property.
 * ```html
 * <li ng-repeat="link in navlinks">
 *   <a ui-state="link.state" ui-state-params="link.params">{{ link.displayName }}</a>
 * </li>
 * ```
 *
 * ### Transition Options
 * You can specify [[TransitionOptions]] to pass to [[StateService.go]] by using the `ui-state-opts` attribute.
 * Options are restricted to `location`, `inherit`, and `reload`.
 * The value of the `ui-state-opts` is `$watch`ed and evaluated as an expression.
 *
 * #### Example:
 * ```html
 * <a ui-state="returnto.state" ui-state-opts="{ reload: true }">Home</a>
 * ```
 *
 * ### Other DOM Events
 *
 * You can also customize which DOM events to respond to (instead of `click`) by
 * providing an `events` array in the `ui-state-opts` attribute.
 *
 * #### Example:
 * ```html
 * <input type="text" ui-state="contacts" ui-state-opts="{ events: ['change', 'blur'] }">
 * ```
 *
 * ### Highlighting the active link
 * This directive can be used in conjunction with [[uiSrefActive]] to highlight the active link.
 *
 * ### Notes
 *
 * - You can use `ui-params` to change **only the parameter values** by omitting the state name and supplying only `ui-state-params`.
 *   However, it might be simpler to use [[uiSref]] parameter-only links.
 *
 * #### Example:
 * Sets the `lang` parameter to `en` and remains on the same state.
 *
 * ```html
 * <a ui-state="" ui-state-params="{ lang: 'en' }">English</a>
 * ```
 *
 * - A middle-click, right-click, or ctrl-click is handled (natively) by the browser to open the href in a new window, for example.
 * ```
 */
var uiState;
uiState = ['$uiRouter', '$timeout',
    function $StateRefDynamicDirective($uiRouter, $timeout) {
        var $state = $uiRouter.stateService;
        return {
            restrict: 'A',
            require: ['?^uiSrefActive', '?^uiSrefActiveEq'],
            link: function (scope, element, attrs, uiSrefActive) {
                var type = getTypeInfo(element);
                var active = uiSrefActive[1] || uiSrefActive[0];
                var unlinkInfoFn = null;
                var hookFn;
                var rawDef = {};
                var getDef = function () { return processedDef($state, element, rawDef); };
                var inputAttrs = ['uiState', 'uiStateParams', 'uiStateOpts'];
                var watchDeregFns = inputAttrs.reduce(function (acc, attr) { return (acc[attr] = core_1.noop, acc); }, {});
                function update() {
                    var def = getDef();
                    if (unlinkInfoFn)
                        unlinkInfoFn();
                    if (active)
                        unlinkInfoFn = active.$$addStateInfo(def.uiState, def.uiStateParams);
                    if (def.href != null)
                        attrs.$set(type.attr, def.href);
                }
                inputAttrs.forEach(function (field) {
                    rawDef[field] = attrs[field] ? scope.$eval(attrs[field]) : null;
                    attrs.$observe(field, function (expr) {
                        watchDeregFns[field]();
                        watchDeregFns[field] = scope.$watch(expr, function (newval) {
                            rawDef[field] = newval;
                            update();
                        }, true);
                    });
                });
                update();
                scope.$on('$destroy', $uiRouter.stateRegistry.onStatesChanged(update));
                scope.$on('$destroy', $uiRouter.transitionService.onSuccess({}, update));
                if (!type.clickable)
                    return;
                hookFn = clickHook(element, $state, $timeout, type, getDef);
                bindEvents(element, scope, hookFn, rawDef.uiStateOpts);
            }
        };
    }];
/**
 * `ui-sref-active` and `ui-sref-active-eq`: A directive that adds a CSS class when a `ui-sref` is active
 *
 * A directive working alongside [[uiSref]] and [[uiState]] to add classes to an element when the
 * related directive's state is active (and remove them when it is inactive).
 *
 * The primary use-case is to highlight the active link in navigation menus,
 * distinguishing it from the inactive menu items.
 *
 * ### Linking to a `ui-sref` or `ui-state`
 * `ui-sref-active` can live on the same element as `ui-sref`/`ui-state`, or it can be on a parent element.
 * If a `ui-sref-active` is a parent to more than one `ui-sref`/`ui-state`, it will apply the CSS class when **any of the links are active**.
 *
 * ### Matching
 *
 * The `ui-sref-active` directive applies the CSS class when the `ui-sref`/`ui-state`'s target state **or any child state is active**.
 * This is a "fuzzy match" which uses [[StateService.includes]].
 *
 * The `ui-sref-active-eq` directive applies the CSS class when the `ui-sref`/`ui-state`'s target state is directly active (not when child states are active).
 * This is an "exact match" which uses [[StateService.is]].
 *
 * ### Parameter values
 * If the `ui-sref`/`ui-state` includes parameter values, the current parameter values must match the link's values for the link to be highlighted.
 * This allows a list of links to the same state with different parameters to be rendered, and the correct one highlighted.
 *
 * #### Example:
 * ```html
 * <li ng-repeat="user in users" ui-sref-active="active">
 *   <a ui-sref="user.details({ userId: user.id })">{{ user.lastName }}</a>
 * </li>
 * ```
 *
 * ### Examples
 *
 * Given the following template:
 * #### Example:
 * ```html
 * <ul>
 *   <li ui-sref-active="active" class="item">
 *     <a href ui-sref="app.user({user: 'bilbobaggins'})">@bilbobaggins</a>
 *   </li>
 * </ul>
 * ```
 *
 * When the app state is `app.user` (or any child state),
 * and contains the state parameter "user" with value "bilbobaggins",
 * the resulting HTML will appear as (note the 'active' class):
 *
 * ```html
 * <ul>
 *   <li ui-sref-active="active" class="item active">
 *     <a ui-sref="app.user({user: 'bilbobaggins'})" href="/users/bilbobaggins">@bilbobaggins</a>
 *   </li>
 * </ul>
 * ```
 *
 * ### Glob mode
 *
 * It is possible to pass `ui-sref-active` an expression that evaluates to an object.
 * The objects keys represent active class names and values represent the respective state names/globs.
 * `ui-sref-active` will match if the current active state **includes** any of
 * the specified state names/globs, even the abstract ones.
 *
 * #### Example:
 * Given the following template, with "admin" being an abstract state:
 * ```html
 * <div ui-sref-active="{'active': 'admin.**'}">
 *   <a ui-sref-active="active" ui-sref="admin.roles">Roles</a>
 * </div>
 * ```
 *
 * When the current state is "admin.roles" the "active" class will be applied to both the <div> and <a> elements.
 * It is important to note that the state names/globs passed to `ui-sref-active` override any state provided by a linked `ui-sref`.
 *
 * ### Notes:
 *
 * - The class name is interpolated **once** during the directives link time (any further changes to the
 * interpolated value are ignored).
 *
 * - Multiple classes may be specified in a space-separated format: `ui-sref-active='class1 class2 class3'`
 */
var uiSrefActive;
uiSrefActive = ['$state', '$stateParams', '$interpolate', '$uiRouter',
    function $StateRefActiveDirective($state, $stateParams, $interpolate, $uiRouter) {
        return {
            restrict: "A",
            controller: ['$scope', '$element', '$attrs',
                function ($scope, $element, $attrs) {
                    var states = [], activeEqClass, uiSrefActive;
                    // There probably isn't much point in $observing this
                    // uiSrefActive and uiSrefActiveEq share the same directive object with some
                    // slight difference in logic routing
                    activeEqClass = $interpolate($attrs.uiSrefActiveEq || '', false)($scope);
                    try {
                        uiSrefActive = $scope.$eval($attrs.uiSrefActive);
                    }
                    catch (e) {
                        // Do nothing. uiSrefActive is not a valid expression.
                        // Fall back to using $interpolate below
                    }
                    uiSrefActive = uiSrefActive || $interpolate($attrs.uiSrefActive || '', false)($scope);
                    if (core_1.isObject(uiSrefActive)) {
                        core_1.forEach(uiSrefActive, function (stateOrName, activeClass) {
                            if (core_1.isString(stateOrName)) {
                                var ref = parseStateRef(stateOrName);
                                addState(ref.state, $scope.$eval(ref.paramExpr), activeClass);
                            }
                        });
                    }
                    // Allow uiSref to communicate with uiSrefActive[Equals]
                    this.$$addStateInfo = function (newState, newParams) {
                        // we already got an explicit state provided by ui-sref-active, so we
                        // shadow the one that comes from ui-sref
                        if (core_1.isObject(uiSrefActive) && states.length > 0) {
                            return;
                        }
                        var deregister = addState(newState, newParams, uiSrefActive);
                        update();
                        return deregister;
                    };
                    function updateAfterTransition(trans) {
                        trans.promise.then(update, core_1.noop);
                    }
                    $scope.$on('$stateChangeSuccess', update);
                    $scope.$on('$destroy', $uiRouter.transitionService.onStart({}, updateAfterTransition));
                    if ($uiRouter.globals.transition) {
                        updateAfterTransition($uiRouter.globals.transition);
                    }
                    function addState(stateName, stateParams, activeClass) {
                        var state = $state.get(stateName, stateContext($element));
                        var stateInfo = {
                            state: state || { name: stateName },
                            params: stateParams,
                            activeClass: activeClass
                        };
                        states.push(stateInfo);
                        return function removeState() {
                            core_1.removeFrom(states)(stateInfo);
                        };
                    }
                    // Update route state
                    function update() {
                        var splitClasses = function (str) {
                            return str.split(/\s/).filter(core_1.identity);
                        };
                        var getClasses = function (stateList) {
                            return stateList.map(function (x) { return x.activeClass; }).map(splitClasses).reduce(core_1.unnestR, []);
                        };
                        var allClasses = getClasses(states).concat(splitClasses(activeEqClass)).reduce(core_1.uniqR, []);
                        var fuzzyClasses = getClasses(states.filter(function (x) { return $state.includes(x.state.name, x.params); }));
                        var exactlyMatchesAny = !!states.filter(function (x) { return $state.is(x.state.name, x.params); }).length;
                        var exactClasses = exactlyMatchesAny ? splitClasses(activeEqClass) : [];
                        var addClasses = fuzzyClasses.concat(exactClasses).reduce(core_1.uniqR, []);
                        var removeClasses = allClasses.filter(function (cls) { return !core_1.inArray(addClasses, cls); });
                        $scope.$evalAsync(function () {
                            addClasses.forEach(function (className) { return $element.addClass(className); });
                            removeClasses.forEach(function (className) { return $element.removeClass(className); });
                        });
                    }
                    update();
                }]
        };
    }];
angular_1.ng.module('ui.router.state')
    .directive('uiSref', uiSref)
    .directive('uiSrefActive', uiSrefActive)
    .directive('uiSrefActiveEq', uiSrefActive)
    .directive('uiState', uiState);
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RhdGVEaXJlY3RpdmVzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2RpcmVjdGl2ZXMvc3RhdGVEaXJlY3RpdmVzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUE7Ozs7Ozs7OztHQVNHLENBQUMsa0JBQWtCO0FBQ3RCLHNDQUEyQztBQUczQyx1Q0FHd0I7QUFNeEIsY0FBYztBQUNkLHVCQUF1QixHQUFXO0lBQ2hDLElBQUksVUFBVSxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsbUJBQW1CLENBQUMsRUFBRSxNQUFNLENBQUM7SUFDeEQsRUFBRSxDQUFDLENBQUMsVUFBVSxDQUFDO1FBQUMsR0FBRyxHQUFHLEdBQUcsR0FBRyxVQUFVLENBQUMsQ0FBQyxDQUFDLEdBQUcsR0FBRyxDQUFDO0lBRWhELE1BQU0sR0FBRyxHQUFHLENBQUMsT0FBTyxDQUFDLEtBQUssRUFBRSxHQUFHLENBQUMsQ0FBQyxLQUFLLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztJQUN6RSxFQUFFLENBQUMsQ0FBQyxDQUFDLE1BQU0sSUFBSSxNQUFNLENBQUMsTUFBTSxLQUFLLENBQUMsQ0FBQztRQUFDLE1BQU0sSUFBSSxLQUFLLENBQUMscUJBQXFCLEdBQUcsR0FBRyxHQUFHLEdBQUcsQ0FBQyxDQUFDO0lBQ3ZGLE1BQU0sQ0FBQyxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUksSUFBSSxFQUFFLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUksSUFBSSxFQUFFLENBQUM7QUFDcEUsQ0FBQztBQUVELGNBQWM7QUFDZCxzQkFBc0IsRUFBb0I7SUFDeEMsSUFBSSxPQUFPLEdBQWdCLEVBQUUsQ0FBQyxNQUFNLEVBQXVCLENBQUMsYUFBYSxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQ3JGLElBQUksSUFBSSxHQUFlLFlBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUNuRCxNQUFNLENBQUMsSUFBSSxHQUFHLFdBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxLQUFLLENBQUMsSUFBSSxHQUFHLFNBQVMsQ0FBQztBQUNsRCxDQUFDO0FBRUQsY0FBYztBQUNkLHNCQUFzQixNQUFvQixFQUFFLFFBQTBCLEVBQUUsR0FBUTtJQUM5RSxJQUFJLE9BQU8sR0FBRyxHQUFHLENBQUMsT0FBTyxJQUFJLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDO0lBQ2pELElBQUksV0FBVyxHQUFHLGFBQU0sQ0FBQyxXQUFXLENBQUMsUUFBUSxFQUFFLE1BQU0sQ0FBQyxFQUFFLEdBQUcsQ0FBQyxXQUFXLElBQUksRUFBRSxDQUFDLENBQUM7SUFDL0UsSUFBSSxJQUFJLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLGFBQWEsRUFBRSxXQUFXLENBQUMsQ0FBQztJQUNoRSxNQUFNLENBQUMsRUFBRSxPQUFPLFNBQUEsRUFBRSxhQUFhLEVBQUUsR0FBRyxDQUFDLGFBQWEsRUFBRSxXQUFXLGFBQUEsRUFBRSxJQUFJLE1BQUEsRUFBRSxDQUFDO0FBQzFFLENBQUM7QUFTRCxjQUFjO0FBQ2QscUJBQXFCLEVBQW9CO0lBQ3ZDLHFGQUFxRjtJQUNyRixJQUFJLEtBQUssR0FBRyxNQUFNLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxLQUFLLDRCQUE0QixDQUFDO0lBQzdGLElBQUksTUFBTSxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxRQUFRLEtBQUssTUFBTSxDQUFDO0lBRXZDLE1BQU0sQ0FBQztRQUNMLElBQUksRUFBRSxNQUFNLEdBQUcsUUFBUSxHQUFHLENBQUMsS0FBSyxHQUFHLFlBQVksR0FBRyxNQUFNLENBQUM7UUFDekQsUUFBUSxFQUFFLEVBQUUsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsV0FBVyxFQUFFLEtBQUssR0FBRztRQUNsRCxTQUFTLEVBQUUsQ0FBQyxNQUFNO0tBQ25CLENBQUM7QUFDSixDQUFDO0FBRUQsY0FBYztBQUNkLG1CQUFtQixFQUFvQixFQUFFLE1BQW9CLEVBQUUsUUFBeUIsRUFBRSxJQUFjLEVBQUUsTUFBaUI7SUFDekgsTUFBTSxDQUFDLFVBQVUsQ0FBeUI7UUFDeEMsSUFBSSxNQUFNLEdBQUcsQ0FBQyxDQUFDLEtBQUssSUFBSSxDQUFDLENBQUMsTUFBTSxFQUFFLE1BQU0sR0FBRyxNQUFNLEVBQUUsQ0FBQztRQUVwRCxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxJQUFJLENBQUMsQ0FBQyxPQUFPLElBQUksQ0FBQyxDQUFDLFFBQVEsSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQy9FLHVGQUF1RjtZQUN2RixJQUFJLFVBQVUsR0FBRyxRQUFRLENBQUM7Z0JBQ3hCLE1BQU0sQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLE9BQU8sRUFBRSxNQUFNLENBQUMsYUFBYSxFQUFFLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUN0RSxDQUFDLENBQUMsQ0FBQztZQUNILENBQUMsQ0FBQyxjQUFjLEVBQUUsQ0FBQztZQUVuQiw2RUFBNkU7WUFDN0UsSUFBSSx5QkFBeUIsR0FBRyxJQUFJLENBQUMsUUFBUSxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBRXRFLENBQUMsQ0FBQyxjQUFjLEdBQUc7Z0JBQ2pCLEVBQUUsQ0FBQyxDQUFDLHlCQUF5QixFQUFFLElBQUksQ0FBQyxDQUFDO29CQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLENBQUM7WUFDcEUsQ0FBQyxDQUFDO1FBQ0osQ0FBQztJQUNILENBQUMsQ0FBQztBQUNKLENBQUM7QUFFRCxjQUFjO0FBQ2QscUJBQXFCLEVBQW9CLEVBQUUsTUFBb0I7SUFDN0QsTUFBTSxDQUFDO1FBQ0wsUUFBUSxFQUFFLFlBQVksQ0FBQyxFQUFFLENBQUMsSUFBSSxNQUFNLENBQUMsUUFBUTtRQUM3QyxPQUFPLEVBQUUsSUFBSTtRQUNiLE1BQU0sRUFBRSxNQUFNO0tBQ2YsQ0FBQztBQUNKLENBQUM7QUFFRCxjQUFjO0FBQ2Qsb0JBQW9CLE9BQXlCLEVBQUUsS0FBYSxFQUFFLE1BQTJDLEVBQUUsV0FBZ0I7SUFDekgsSUFBSSxNQUFNLENBQUM7SUFFWCxFQUFFLENBQUMsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDO1FBQ2hCLE1BQU0sR0FBRyxXQUFXLENBQUMsTUFBTSxDQUFDO0lBQzlCLENBQUM7SUFFRCxFQUFFLENBQUMsQ0FBQyxDQUFDLGNBQU8sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDckIsTUFBTSxHQUFHLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDckIsQ0FBQztJQUVELElBQUksRUFBRSxHQUFHLE9BQU8sQ0FBQyxFQUFFLEdBQUcsSUFBSSxHQUFHLE1BQU0sQ0FBQztJQUNwQyxHQUFHLENBQUMsQ0FBYyxVQUFNLEVBQU4saUJBQU0sRUFBTixvQkFBTSxFQUFOLElBQU07UUFBbkIsSUFBSSxPQUFLLGVBQUE7UUFDWixPQUFPLENBQUMsRUFBRSxDQUFDLENBQUMsT0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0tBQzVCO0lBRUQsS0FBSyxDQUFDLEdBQUcsQ0FBQyxVQUFVLEVBQUU7UUFDcEIsSUFBSSxHQUFHLEdBQUcsT0FBTyxDQUFDLEdBQUcsR0FBRyxLQUFLLEdBQUcsUUFBUSxDQUFDO1FBQ3pDLEdBQUcsQ0FBQyxDQUFjLFVBQU0sRUFBTixpQkFBTSxFQUFOLG9CQUFNLEVBQU4sSUFBTTtZQUFuQixJQUFJLE9BQUssZUFBQTtZQUNaLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxPQUFLLEVBQUUsTUFBTSxDQUFDLENBQUM7U0FDN0I7SUFDSCxDQUFDLENBQUMsQ0FBQztBQUNMLENBQUM7QUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBb0lHO0FBQ0gsSUFBSSxNQUFxQixDQUFDO0FBQzFCLE1BQU0sR0FBRyxDQUFDLFdBQVcsRUFBRSxVQUFVO0lBQy9CLDRCQUE0QixTQUFtQixFQUFFLFFBQXlCO1FBQ3hFLElBQUksTUFBTSxHQUFHLFNBQVMsQ0FBQyxZQUFZLENBQUM7UUFFcEMsTUFBTSxDQUFDO1lBQ0wsUUFBUSxFQUFFLEdBQUc7WUFDYixPQUFPLEVBQUUsQ0FBQyxnQkFBZ0IsRUFBRSxrQkFBa0IsQ0FBQztZQUMvQyxJQUFJLEVBQUUsVUFBVSxLQUFhLEVBQUUsT0FBeUIsRUFBRSxLQUFVLEVBQUUsWUFBaUI7Z0JBQ3JGLElBQUksSUFBSSxHQUFHLFdBQVcsQ0FBQyxPQUFPLENBQUMsQ0FBQztnQkFDaEMsSUFBSSxNQUFNLEdBQUcsWUFBWSxDQUFDLENBQUMsQ0FBQyxJQUFJLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDaEQsSUFBSSxZQUFZLEdBQWEsSUFBSSxDQUFDO2dCQUNsQyxJQUFJLE1BQU0sQ0FBQztnQkFFWCxJQUFJLE1BQU0sR0FBRyxFQUFTLENBQUM7Z0JBQ3ZCLElBQUksTUFBTSxHQUFHLGNBQU0sT0FBQSxZQUFZLENBQUMsTUFBTSxFQUFFLE9BQU8sRUFBRSxNQUFNLENBQUMsRUFBckMsQ0FBcUMsQ0FBQztnQkFFekQsSUFBSSxHQUFHLEdBQUcsYUFBYSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQztnQkFDdEMsTUFBTSxDQUFDLE9BQU8sR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDO2dCQUMzQixNQUFNLENBQUMsV0FBVyxHQUFHLEtBQUssQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLEdBQUcsRUFBRSxDQUFDO2dCQUUzRTtvQkFDRSxJQUFJLEdBQUcsR0FBRyxNQUFNLEVBQUUsQ0FBQztvQkFDbkIsRUFBRSxDQUFDLENBQUMsWUFBWSxDQUFDO3dCQUFDLFlBQVksRUFBRSxDQUFDO29CQUNqQyxFQUFFLENBQUMsQ0FBQyxNQUFNLENBQUM7d0JBQUMsWUFBWSxHQUFHLE1BQU0sQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLE9BQU8sRUFBRSxHQUFHLENBQUMsYUFBYSxDQUFDLENBQUM7b0JBQ2pGLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLElBQUksSUFBSSxDQUFDO3dCQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ3hELENBQUM7Z0JBRUQsRUFBRSxDQUFDLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7b0JBQ2xCLEtBQUssQ0FBQyxNQUFNLENBQUMsR0FBRyxDQUFDLFNBQVMsRUFBRSxVQUFVLEdBQUc7d0JBQ3ZDLE1BQU0sQ0FBQyxhQUFhLEdBQUcsYUFBTSxDQUFDLEVBQUUsRUFBRSxHQUFHLENBQUMsQ0FBQzt3QkFDdkMsTUFBTSxFQUFFLENBQUM7b0JBQ1gsQ0FBQyxFQUFFLElBQUksQ0FBQyxDQUFDO29CQUNULE1BQU0sQ0FBQyxhQUFhLEdBQUcsYUFBTSxDQUFDLEVBQUUsRUFBRSxLQUFLLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO2dCQUNoRSxDQUFDO2dCQUVELE1BQU0sRUFBRSxDQUFDO2dCQUVULEtBQUssQ0FBQyxHQUFHLENBQUMsVUFBVSxFQUFRLFNBQVMsQ0FBQyxhQUFhLENBQUMsZUFBZSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7Z0JBQzdFLEtBQUssQ0FBQyxHQUFHLENBQUMsVUFBVSxFQUFRLFNBQVMsQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLENBQUMsRUFBRSxFQUFFLE1BQU0sQ0FBQyxDQUFDLENBQUM7Z0JBRS9FLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQztvQkFBQyxNQUFNLENBQUM7Z0JBQzVCLE1BQU0sR0FBRyxTQUFTLENBQUMsT0FBTyxFQUFFLE1BQU0sRUFBRSxRQUFRLEVBQUUsSUFBSSxFQUFFLE1BQU0sQ0FBQyxDQUFDO2dCQUM1RCxVQUFVLENBQUMsT0FBTyxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFDO1lBQ3pELENBQUM7U0FDRixDQUFDO0lBQ0osQ0FBQyxDQUFDLENBQUM7QUFFTDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0FtRkc7QUFDSCxJQUFJLE9BQXNCLENBQUM7QUFDM0IsT0FBTyxHQUFHLENBQUMsV0FBVyxFQUFFLFVBQVU7SUFDaEMsbUNBQW1DLFNBQW1CLEVBQUUsUUFBeUI7UUFDL0UsSUFBSSxNQUFNLEdBQUcsU0FBUyxDQUFDLFlBQVksQ0FBQztRQUVwQyxNQUFNLENBQUM7WUFDTCxRQUFRLEVBQUUsR0FBRztZQUNiLE9BQU8sRUFBRSxDQUFDLGdCQUFnQixFQUFFLGtCQUFrQixDQUFDO1lBQy9DLElBQUksRUFBRSxVQUFVLEtBQWEsRUFBRSxPQUF5QixFQUFFLEtBQVUsRUFBRSxZQUFpQjtnQkFDckYsSUFBSSxJQUFJLEdBQUcsV0FBVyxDQUFDLE9BQU8sQ0FBQyxDQUFDO2dCQUNoQyxJQUFJLE1BQU0sR0FBRyxZQUFZLENBQUMsQ0FBQyxDQUFDLElBQUksWUFBWSxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNoRCxJQUFJLFlBQVksR0FBYSxJQUFJLENBQUM7Z0JBQ2xDLElBQUksTUFBTSxDQUFDO2dCQUVYLElBQUksTUFBTSxHQUFHLEVBQVMsQ0FBQztnQkFDdkIsSUFBSSxNQUFNLEdBQUcsY0FBTSxPQUFBLFlBQVksQ0FBQyxNQUFNLEVBQUUsT0FBTyxFQUFFLE1BQU0sQ0FBQyxFQUFyQyxDQUFxQyxDQUFDO2dCQUV6RCxJQUFJLFVBQVUsR0FBRyxDQUFDLFNBQVMsRUFBRSxlQUFlLEVBQUUsYUFBYSxDQUFDLENBQUM7Z0JBQzdELElBQUksYUFBYSxHQUFHLFVBQVUsQ0FBQyxNQUFNLENBQUMsVUFBQyxHQUFHLEVBQUUsSUFBSSxJQUFLLE9BQUEsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsV0FBSSxFQUFFLEdBQUcsQ0FBQyxFQUF2QixDQUF1QixFQUFFLEVBQUUsQ0FBQyxDQUFDO2dCQUVsRjtvQkFDRSxJQUFJLEdBQUcsR0FBRyxNQUFNLEVBQUUsQ0FBQztvQkFDbkIsRUFBRSxDQUFDLENBQUMsWUFBWSxDQUFDO3dCQUFDLFlBQVksRUFBRSxDQUFDO29CQUNqQyxFQUFFLENBQUMsQ0FBQyxNQUFNLENBQUM7d0JBQUMsWUFBWSxHQUFHLE1BQU0sQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLE9BQU8sRUFBRSxHQUFHLENBQUMsYUFBYSxDQUFDLENBQUM7b0JBQ2pGLEVBQUUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLElBQUksSUFBSSxDQUFDO3dCQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ3hELENBQUM7Z0JBRUQsVUFBVSxDQUFDLE9BQU8sQ0FBQyxVQUFDLEtBQUs7b0JBQ3ZCLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxLQUFLLENBQUMsS0FBSyxDQUFDLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsR0FBRyxJQUFJLENBQUM7b0JBRWhFLEtBQUssQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQUMsSUFBSTt3QkFDekIsYUFBYSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUM7d0JBQ3ZCLGFBQWEsQ0FBQyxLQUFLLENBQUMsR0FBRyxLQUFLLENBQUMsTUFBTSxDQUFDLElBQUksRUFBRSxVQUFDLE1BQU07NEJBQy9DLE1BQU0sQ0FBQyxLQUFLLENBQUMsR0FBRyxNQUFNLENBQUM7NEJBQ3ZCLE1BQU0sRUFBRSxDQUFDO3dCQUNYLENBQUMsRUFBRSxJQUFJLENBQUMsQ0FBQztvQkFDWCxDQUFDLENBQUMsQ0FBQTtnQkFDSixDQUFDLENBQUMsQ0FBQztnQkFFSCxNQUFNLEVBQUUsQ0FBQztnQkFFVCxLQUFLLENBQUMsR0FBRyxDQUFDLFVBQVUsRUFBUSxTQUFTLENBQUMsYUFBYSxDQUFDLGVBQWUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO2dCQUM3RSxLQUFLLENBQUMsR0FBRyxDQUFDLFVBQVUsRUFBUSxTQUFTLENBQUMsaUJBQWlCLENBQUMsU0FBUyxDQUFDLEVBQUUsRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDO2dCQUUvRSxFQUFFLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUM7b0JBQUMsTUFBTSxDQUFDO2dCQUM1QixNQUFNLEdBQUcsU0FBUyxDQUFDLE9BQU8sRUFBRSxNQUFNLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxNQUFNLENBQUMsQ0FBQztnQkFDNUQsVUFBVSxDQUFDLE9BQU8sRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUN6RCxDQUFDO1NBQ0YsQ0FBQztJQUNKLENBQUMsQ0FBQyxDQUFDO0FBR0w7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBZ0ZHO0FBQ0gsSUFBSSxZQUEyQixDQUFDO0FBQ2hDLFlBQVksR0FBRyxDQUFDLFFBQVEsRUFBRSxjQUFjLEVBQUUsY0FBYyxFQUFFLFdBQVc7SUFDbkUsa0NBQWtDLE1BQW9CLEVBQUUsWUFBaUIsRUFBRSxZQUFpQyxFQUFFLFNBQW1CO1FBQy9ILE1BQU0sQ0FBQztZQUNMLFFBQVEsRUFBRSxHQUFHO1lBQ2IsVUFBVSxFQUFFLENBQUMsUUFBUSxFQUFFLFVBQVUsRUFBRSxRQUFRO2dCQUN6QyxVQUFVLE1BQWMsRUFBRSxRQUEwQixFQUFFLE1BQVc7b0JBQy9ELElBQUksTUFBTSxHQUFnQixFQUFFLEVBQ3hCLGFBQXFCLEVBQ3JCLFlBQWlCLENBQUM7b0JBRXRCLHFEQUFxRDtvQkFDckQsNEVBQTRFO29CQUM1RSxxQ0FBcUM7b0JBQ3JDLGFBQWEsR0FBRyxZQUFZLENBQUMsTUFBTSxDQUFDLGNBQWMsSUFBSSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUM7b0JBRXpFLElBQUksQ0FBQzt3QkFDSCxZQUFZLEdBQUcsTUFBTSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsWUFBWSxDQUFDLENBQUM7b0JBQ25ELENBQUM7b0JBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQzt3QkFDWCxzREFBc0Q7d0JBQ3RELHdDQUF3QztvQkFDMUMsQ0FBQztvQkFDRCxZQUFZLEdBQUcsWUFBWSxJQUFJLFlBQVksQ0FBQyxNQUFNLENBQUMsWUFBWSxJQUFJLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQyxNQUFNLENBQUMsQ0FBQztvQkFDdEYsRUFBRSxDQUFDLENBQUMsZUFBUSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQzt3QkFDM0IsY0FBTyxDQUFDLFlBQVksRUFBRSxVQUFVLFdBQXdCLEVBQUUsV0FBbUI7NEJBQzNFLEVBQUUsQ0FBQyxDQUFDLGVBQVEsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0NBQzFCLElBQUksR0FBRyxHQUFHLGFBQWEsQ0FBQyxXQUFXLENBQUMsQ0FBQztnQ0FDckMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDLEVBQUUsV0FBVyxDQUFDLENBQUM7NEJBQ2hFLENBQUM7d0JBQ0gsQ0FBQyxDQUFDLENBQUM7b0JBQ0wsQ0FBQztvQkFFRCx3REFBd0Q7b0JBQ3hELElBQUksQ0FBQyxjQUFjLEdBQUcsVUFBVSxRQUFnQixFQUFFLFNBQWM7d0JBQzlELHFFQUFxRTt3QkFDckUseUNBQXlDO3dCQUN6QyxFQUFFLENBQUMsQ0FBQyxlQUFRLENBQUMsWUFBWSxDQUFDLElBQUksTUFBTSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDOzRCQUNoRCxNQUFNLENBQUM7d0JBQ1QsQ0FBQzt3QkFDRCxJQUFJLFVBQVUsR0FBRyxRQUFRLENBQUMsUUFBUSxFQUFFLFNBQVMsRUFBRSxZQUFZLENBQUMsQ0FBQzt3QkFDN0QsTUFBTSxFQUFFLENBQUM7d0JBQ1QsTUFBTSxDQUFDLFVBQVUsQ0FBQztvQkFDcEIsQ0FBQyxDQUFDO29CQUVGLCtCQUErQixLQUFLO3dCQUNsQyxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsV0FBSSxDQUFDLENBQUM7b0JBQ25DLENBQUM7b0JBRUQsTUFBTSxDQUFDLEdBQUcsQ0FBQyxxQkFBcUIsRUFBRSxNQUFNLENBQUMsQ0FBQztvQkFDMUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxVQUFVLEVBQVEsU0FBUyxDQUFDLGlCQUFpQixDQUFDLE9BQU8sQ0FBQyxFQUFFLEVBQUUscUJBQXFCLENBQUMsQ0FBQyxDQUFDO29CQUM3RixFQUFFLENBQUMsQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUM7d0JBQ2pDLHFCQUFxQixDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLENBQUM7b0JBQ3RELENBQUM7b0JBRUQsa0JBQWtCLFNBQWlCLEVBQUUsV0FBZ0IsRUFBRSxXQUFtQjt3QkFDeEUsSUFBSSxLQUFLLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyxTQUFTLEVBQUUsWUFBWSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7d0JBRTFELElBQUksU0FBUyxHQUFHOzRCQUNkLEtBQUssRUFBRSxLQUFLLElBQUksRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFOzRCQUNuQyxNQUFNLEVBQUUsV0FBVzs0QkFDbkIsV0FBVyxFQUFFLFdBQVc7eUJBQ3pCLENBQUM7d0JBRUYsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQzt3QkFFdkIsTUFBTSxDQUFDOzRCQUNMLGlCQUFVLENBQUMsTUFBTSxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUM7d0JBQ2hDLENBQUMsQ0FBQTtvQkFDSCxDQUFDO29CQUVELHFCQUFxQjtvQkFDckI7d0JBQ0UsSUFBTSxZQUFZLEdBQUcsVUFBQSxHQUFHOzRCQUNwQixPQUFBLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxDQUFDLGVBQVEsQ0FBQzt3QkFBaEMsQ0FBZ0MsQ0FBQzt3QkFDckMsSUFBTSxVQUFVLEdBQUcsVUFBQyxTQUFzQjs0QkFDdEMsT0FBQSxTQUFTLENBQUMsR0FBRyxDQUFDLFVBQUEsQ0FBQyxJQUFJLE9BQUEsQ0FBQyxDQUFDLFdBQVcsRUFBYixDQUFhLENBQUMsQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLENBQUMsTUFBTSxDQUFDLGNBQU8sRUFBRSxFQUFFLENBQUM7d0JBQXZFLENBQXVFLENBQUM7d0JBRTVFLElBQUksVUFBVSxHQUFHLFVBQVUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxNQUFNLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLFlBQUssRUFBRSxFQUFFLENBQUMsQ0FBQzt3QkFDMUYsSUFBSSxZQUFZLEdBQUcsVUFBVSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsVUFBQSxDQUFDLElBQUksT0FBQSxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxNQUFNLENBQUMsRUFBdkMsQ0FBdUMsQ0FBQyxDQUFDLENBQUM7d0JBQzNGLElBQUksaUJBQWlCLEdBQUcsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsVUFBQSxDQUFDLElBQUksT0FBQSxNQUFNLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLENBQUMsQ0FBQyxNQUFNLENBQUMsRUFBakMsQ0FBaUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQzt3QkFDdkYsSUFBSSxZQUFZLEdBQUcsaUJBQWlCLEdBQUcsWUFBWSxDQUFDLGFBQWEsQ0FBQyxHQUFHLEVBQUUsQ0FBQzt3QkFFeEUsSUFBSSxVQUFVLEdBQUcsWUFBWSxDQUFDLE1BQU0sQ0FBQyxZQUFZLENBQUMsQ0FBQyxNQUFNLENBQUMsWUFBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO3dCQUNyRSxJQUFJLGFBQWEsR0FBRyxVQUFVLENBQUMsTUFBTSxDQUFDLFVBQUEsR0FBRyxJQUFJLE9BQUEsQ0FBQyxjQUFPLENBQUMsVUFBVSxFQUFFLEdBQUcsQ0FBQyxFQUF6QixDQUF5QixDQUFDLENBQUM7d0JBRXhFLE1BQU0sQ0FBQyxVQUFVLENBQUM7NEJBQ2hCLFVBQVUsQ0FBQyxPQUFPLENBQUMsVUFBQSxTQUFTLElBQUksT0FBQSxRQUFRLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxFQUE1QixDQUE0QixDQUFDLENBQUM7NEJBQzlELGFBQWEsQ0FBQyxPQUFPLENBQUMsVUFBQSxTQUFTLElBQUksT0FBQSxRQUFRLENBQUMsV0FBVyxDQUFDLFNBQVMsQ0FBQyxFQUEvQixDQUErQixDQUFDLENBQUM7d0JBQ3RFLENBQUMsQ0FBQyxDQUFDO29CQUNMLENBQUM7b0JBRUQsTUFBTSxFQUFFLENBQUM7Z0JBQ1gsQ0FBQyxDQUFDO1NBQ0wsQ0FBQztJQUNKLENBQUMsQ0FBQyxDQUFDO0FBT0wsWUFBTyxDQUFDLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQztLQUM1QixTQUFTLENBQUMsUUFBUSxFQUFFLE1BQU0sQ0FBQztLQUMzQixTQUFTLENBQUMsY0FBYyxFQUFFLFlBQVksQ0FBQztLQUN2QyxTQUFTLENBQUMsZ0JBQWdCLEVBQUUsWUFBWSxDQUFDO0tBQ3pDLFNBQVMsQ0FBQyxTQUFTLEVBQUUsT0FBTyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICMgQW5ndWxhciAxIERpcmVjdGl2ZXNcbiAqXG4gKiBUaGVzZSBhcmUgdGhlIGRpcmVjdGl2ZXMgaW5jbHVkZWQgaW4gVUktUm91dGVyIGZvciBBbmd1bGFyIDEuXG4gKiBUaGVzZSBkaXJlY3RpdmVzIGFyZSB1c2VkIGluIHRlbXBsYXRlcyB0byBjcmVhdGUgdmlld3BvcnRzIGFuZCBsaW5rL25hdmlnYXRlIHRvIHN0YXRlcy5cbiAqXG4gKiBAbmcxYXBpXG4gKiBAcHJlZmVycmVkXG4gKiBAbW9kdWxlIGRpcmVjdGl2ZXNcbiAqLyAvKiogZm9yIHR5cGVkb2MgKi9cbmltcG9ydCB7IG5nIGFzIGFuZ3VsYXIgfSBmcm9tIFwiLi4vYW5ndWxhclwiO1xuaW1wb3J0IHsgSUF1Z21lbnRlZEpRdWVyeSwgSVRpbWVvdXRTZXJ2aWNlLCBJU2NvcGUsIElJbnRlcnBvbGF0ZVNlcnZpY2UgfSBmcm9tIFwiYW5ndWxhclwiO1xuXG5pbXBvcnQge1xuICAgIE9iaiwgZXh0ZW5kLCBmb3JFYWNoLCB0YWlsLCBpc1N0cmluZywgaXNPYmplY3QsIGlzQXJyYXksIHBhcnNlLCBub29wLCB1bm5lc3RSLCBpZGVudGl0eSwgdW5pcVIsIGluQXJyYXksIHJlbW92ZUZyb20sXG4gICAgUmF3UGFyYW1zLCBQYXRoTm9kZSwgU3RhdGVPck5hbWUsIFN0YXRlU2VydmljZSwgU3RhdGVEZWNsYXJhdGlvbiwgVUlSb3V0ZXJcbn0gZnJvbSBcIkB1aXJvdXRlci9jb3JlXCI7XG5pbXBvcnQgeyBVSVZpZXdEYXRhIH0gZnJvbSBcIi4vdmlld0RpcmVjdGl2ZVwiO1xuXG4vKiogQGhpZGRlbiBVc2VkIGZvciB0eXBlZG9jICovXG5leHBvcnQgaW50ZXJmYWNlIG5nMV9kaXJlY3RpdmUge31cblxuLyoqIEBoaWRkZW4gKi9cbmZ1bmN0aW9uIHBhcnNlU3RhdGVSZWYocmVmOiBzdHJpbmcpIHtcbiAgbGV0IHBhcmFtc09ubHkgPSByZWYubWF0Y2goL15cXHMqKHtbXn1dKn0pXFxzKiQvKSwgcGFyc2VkO1xuICBpZiAocGFyYW1zT25seSkgcmVmID0gJygnICsgcGFyYW1zT25seVsxXSArICcpJztcblxuICBwYXJzZWQgPSByZWYucmVwbGFjZSgvXFxuL2csIFwiIFwiKS5tYXRjaCgvXlxccyooW14oXSo/KVxccyooXFwoKC4qKVxcKSk/XFxzKiQvKTtcbiAgaWYgKCFwYXJzZWQgfHwgcGFyc2VkLmxlbmd0aCAhPT0gNCkgdGhyb3cgbmV3IEVycm9yKFwiSW52YWxpZCBzdGF0ZSByZWYgJ1wiICsgcmVmICsgXCInXCIpO1xuICByZXR1cm4geyBzdGF0ZTogcGFyc2VkWzFdIHx8IG51bGwsIHBhcmFtRXhwcjogcGFyc2VkWzNdIHx8IG51bGwgfTtcbn1cblxuLyoqIEBoaWRkZW4gKi9cbmZ1bmN0aW9uIHN0YXRlQ29udGV4dChlbDogSUF1Z21lbnRlZEpRdWVyeSkge1xuICBsZXQgJHVpVmlldzogVUlWaWV3RGF0YSA9IChlbC5wYXJlbnQoKSBhcyBJQXVnbWVudGVkSlF1ZXJ5KS5pbmhlcml0ZWREYXRhKCckdWlWaWV3Jyk7XG4gIGxldCBwYXRoOiBQYXRoTm9kZVtdID0gcGFyc2UoJyRjZmcucGF0aCcpKCR1aVZpZXcpO1xuICByZXR1cm4gcGF0aCA/IHRhaWwocGF0aCkuc3RhdGUubmFtZSA6IHVuZGVmaW5lZDtcbn1cblxuLyoqIEBoaWRkZW4gKi9cbmZ1bmN0aW9uIHByb2Nlc3NlZERlZigkc3RhdGU6IFN0YXRlU2VydmljZSwgJGVsZW1lbnQ6IElBdWdtZW50ZWRKUXVlcnksIGRlZjogRGVmKTogRGVmIHtcbiAgbGV0IHVpU3RhdGUgPSBkZWYudWlTdGF0ZSB8fCAkc3RhdGUuY3VycmVudC5uYW1lO1xuICBsZXQgdWlTdGF0ZU9wdHMgPSBleHRlbmQoZGVmYXVsdE9wdHMoJGVsZW1lbnQsICRzdGF0ZSksIGRlZi51aVN0YXRlT3B0cyB8fCB7fSk7XG4gIGxldCBocmVmID0gJHN0YXRlLmhyZWYodWlTdGF0ZSwgZGVmLnVpU3RhdGVQYXJhbXMsIHVpU3RhdGVPcHRzKTtcbiAgcmV0dXJuIHsgdWlTdGF0ZSwgdWlTdGF0ZVBhcmFtczogZGVmLnVpU3RhdGVQYXJhbXMsIHVpU3RhdGVPcHRzLCBocmVmIH07XG59XG5cbi8qKiBAaGlkZGVuICovXG5pbnRlcmZhY2UgVHlwZUluZm8ge1xuICBhdHRyOiBzdHJpbmc7XG4gIGlzQW5jaG9yOiBib29sZWFuO1xuICBjbGlja2FibGU6IGJvb2xlYW47XG59XG5cbi8qKiBAaGlkZGVuICovXG5mdW5jdGlvbiBnZXRUeXBlSW5mbyhlbDogSUF1Z21lbnRlZEpRdWVyeSk6IFR5cGVJbmZvIHtcbiAgLy8gU1ZHQUVsZW1lbnQgZG9lcyBub3QgdXNlIHRoZSBocmVmIGF0dHJpYnV0ZSwgYnV0IHJhdGhlciB0aGUgJ3hsaW5rSHJlZicgYXR0cmlidXRlLlxuICB2YXIgaXNTdmcgPSBPYmplY3QucHJvdG90eXBlLnRvU3RyaW5nLmNhbGwoZWwucHJvcCgnaHJlZicpKSA9PT0gJ1tvYmplY3QgU1ZHQW5pbWF0ZWRTdHJpbmddJztcbiAgdmFyIGlzRm9ybSA9IGVsWzBdLm5vZGVOYW1lID09PSBcIkZPUk1cIjtcblxuICByZXR1cm4ge1xuICAgIGF0dHI6IGlzRm9ybSA/IFwiYWN0aW9uXCIgOiAoaXNTdmcgPyAneGxpbms6aHJlZicgOiAnaHJlZicpLFxuICAgIGlzQW5jaG9yOiBlbC5wcm9wKFwidGFnTmFtZVwiKS50b1VwcGVyQ2FzZSgpID09PSBcIkFcIixcbiAgICBjbGlja2FibGU6ICFpc0Zvcm1cbiAgfTtcbn1cblxuLyoqIEBoaWRkZW4gKi9cbmZ1bmN0aW9uIGNsaWNrSG9vayhlbDogSUF1Z21lbnRlZEpRdWVyeSwgJHN0YXRlOiBTdGF0ZVNlcnZpY2UsICR0aW1lb3V0OiBJVGltZW91dFNlcnZpY2UsIHR5cGU6IFR5cGVJbmZvLCBnZXREZWY6ICgpID0+IERlZikge1xuICByZXR1cm4gZnVuY3Rpb24gKGU6IEpRdWVyeU1vdXNlRXZlbnRPYmplY3QpIHtcbiAgICB2YXIgYnV0dG9uID0gZS53aGljaCB8fCBlLmJ1dHRvbiwgdGFyZ2V0ID0gZ2V0RGVmKCk7XG5cbiAgICBpZiAoIShidXR0b24gPiAxIHx8IGUuY3RybEtleSB8fCBlLm1ldGFLZXkgfHwgZS5zaGlmdEtleSB8fCBlbC5hdHRyKCd0YXJnZXQnKSkpIHtcbiAgICAgIC8vIEhBQ0s6IFRoaXMgaXMgdG8gYWxsb3cgbmctY2xpY2tzIHRvIGJlIHByb2Nlc3NlZCBiZWZvcmUgdGhlIHRyYW5zaXRpb24gaXMgaW5pdGlhdGVkOlxuICAgICAgdmFyIHRyYW5zaXRpb24gPSAkdGltZW91dChmdW5jdGlvbiAoKSB7XG4gICAgICAgICRzdGF0ZS5nbyh0YXJnZXQudWlTdGF0ZSwgdGFyZ2V0LnVpU3RhdGVQYXJhbXMsIHRhcmdldC51aVN0YXRlT3B0cyk7XG4gICAgICB9KTtcbiAgICAgIGUucHJldmVudERlZmF1bHQoKTtcblxuICAgICAgLy8gaWYgdGhlIHN0YXRlIGhhcyBubyBVUkwsIGlnbm9yZSBvbmUgcHJldmVudERlZmF1bHQgZnJvbSB0aGUgPGE+IGRpcmVjdGl2ZS5cbiAgICAgIHZhciBpZ25vcmVQcmV2ZW50RGVmYXVsdENvdW50ID0gdHlwZS5pc0FuY2hvciAmJiAhdGFyZ2V0LmhyZWYgPyAxIDogMDtcblxuICAgICAgZS5wcmV2ZW50RGVmYXVsdCA9IGZ1bmN0aW9uICgpIHtcbiAgICAgICAgaWYgKGlnbm9yZVByZXZlbnREZWZhdWx0Q291bnQtLSA8PSAwKSAkdGltZW91dC5jYW5jZWwodHJhbnNpdGlvbik7XG4gICAgICB9O1xuICAgIH1cbiAgfTtcbn1cblxuLyoqIEBoaWRkZW4gKi9cbmZ1bmN0aW9uIGRlZmF1bHRPcHRzKGVsOiBJQXVnbWVudGVkSlF1ZXJ5LCAkc3RhdGU6IFN0YXRlU2VydmljZSkge1xuICByZXR1cm4ge1xuICAgIHJlbGF0aXZlOiBzdGF0ZUNvbnRleHQoZWwpIHx8ICRzdGF0ZS4kY3VycmVudCxcbiAgICBpbmhlcml0OiB0cnVlLFxuICAgIHNvdXJjZTogXCJzcmVmXCJcbiAgfTtcbn1cblxuLyoqIEBoaWRkZW4gKi9cbmZ1bmN0aW9uIGJpbmRFdmVudHMoZWxlbWVudDogSUF1Z21lbnRlZEpRdWVyeSwgc2NvcGU6IElTY29wZSwgaG9va0ZuOiAoZTogSlF1ZXJ5TW91c2VFdmVudE9iamVjdCkgPT4gdm9pZCwgdWlTdGF0ZU9wdHM6IGFueSk6IHZvaWQge1xuICBsZXQgZXZlbnRzO1xuXG4gIGlmICh1aVN0YXRlT3B0cykge1xuICAgIGV2ZW50cyA9IHVpU3RhdGVPcHRzLmV2ZW50cztcbiAgfVxuXG4gIGlmICghaXNBcnJheShldmVudHMpKSB7XG4gICAgZXZlbnRzID0gWydjbGljayddO1xuICB9XG5cbiAgbGV0IG9uID0gZWxlbWVudC5vbiA/ICdvbicgOiAnYmluZCc7XG4gIGZvciAobGV0IGV2ZW50IG9mIGV2ZW50cykge1xuICAgIGVsZW1lbnRbb25dKGV2ZW50LCBob29rRm4pO1xuICB9XG5cbiAgc2NvcGUuJG9uKCckZGVzdHJveScsIGZ1bmN0aW9uKCkge1xuICAgIGxldCBvZmYgPSBlbGVtZW50Lm9mZiA/ICdvZmYnIDogJ3VuYmluZCc7XG4gICAgZm9yIChsZXQgZXZlbnQgb2YgZXZlbnRzKSB7XG4gICAgICBlbGVtZW50W29mZl0oZXZlbnQsIGhvb2tGbik7XG4gICAgfVxuICB9KTtcbn1cblxuLyoqXG4gKiBgdWktc3JlZmA6IEEgZGlyZWN0aXZlIGZvciBsaW5raW5nIHRvIGEgc3RhdGVcbiAqXG4gKiBBIGRpcmVjdGl2ZSB3aGljaCBsaW5rcyB0byBhIHN0YXRlIChhbmQgb3B0aW9uYWxseSwgcGFyYW1ldGVycykuXG4gKiBXaGVuIGNsaWNrZWQsIHRoaXMgZGlyZWN0aXZlIGFjdGl2YXRlcyB0aGUgbGlua2VkIHN0YXRlIHdpdGggdGhlIHN1cHBsaWVkIHBhcmFtZXRlciB2YWx1ZXMuXG4gKlxuICogIyMjIExpbmtlZCBTdGF0ZVxuICogVGhlIGF0dHJpYnV0ZSB2YWx1ZSBvZiB0aGUgYHVpLXNyZWZgIGlzIHRoZSBuYW1lIG9mIHRoZSBzdGF0ZSB0byBsaW5rIHRvLlxuICpcbiAqICMjIyMgRXhhbXBsZTpcbiAqIFRoaXMgd2lsbCBhY3RpdmF0ZSB0aGUgYGhvbWVgIHN0YXRlIHdoZW4gdGhlIGxpbmsgaXMgY2xpY2tlZC5cbiAqIGBgYGh0bWxcbiAqIDxhIHVpLXNyZWY9XCJob21lXCI+SG9tZTwvYT5cbiAqIGBgYFxuICpcbiAqICMjIyBSZWxhdGl2ZSBMaW5rc1xuICogWW91IGNhbiBhbHNvIHVzZSByZWxhdGl2ZSBzdGF0ZSBwYXRocyB3aXRoaW4gYHVpLXNyZWZgLCBqdXN0IGxpa2UgYSByZWxhdGl2ZSBwYXRoIHBhc3NlZCB0byBgJHN0YXRlLmdvKClgIChbW1N0YXRlU2VydmljZS5nb11dKS5cbiAqIFlvdSBqdXN0IG5lZWQgdG8gYmUgYXdhcmUgdGhhdCB0aGUgcGF0aCBpcyByZWxhdGl2ZSB0byB0aGUgc3RhdGUgdGhhdCAqY3JlYXRlZCogdGhlIGxpbmsuXG4gKiBUaGlzIGFsbG93cyBhIHN0YXRlIHRvIGNyZWF0ZSBhIHJlbGF0aXZlIGB1aS1zcmVmYCB3aGljaCBhbHdheXMgdGFyZ2V0cyB0aGUgc2FtZSBkZXN0aW5hdGlvbi5cbiAqXG4gKiAjIyMjIEV4YW1wbGU6XG4gKiBCb3RoIHRoZXNlIGxpbmtzIGFyZSByZWxhdGl2ZSB0byB0aGUgcGFyZW50IHN0YXRlLCBldmVuIHdoZW4gYSBjaGlsZCBzdGF0ZSBpcyBjdXJyZW50bHkgYWN0aXZlLlxuICogYGBgaHRtbFxuICogPGEgdWktc3JlZj1cIi5jaGlsZDFcIj5jaGlsZCAxIHN0YXRlPC9hPlxuICogPGEgdWktc3JlZj1cIi5jaGlsZDJcIj5jaGlsZCAyIHN0YXRlPC9hPlxuICogYGBgXG4gKlxuICogVGhpcyBsaW5rIGFjdGl2YXRlcyB0aGUgcGFyZW50IHN0YXRlLlxuICogYGBgaHRtbFxuICogPGEgdWktc3JlZj1cIl5cIj5SZXR1cm48L2E+XG4gKiBgYGBcbiAqXG4gKiAjIyMgaHJlZnNcbiAqIElmIHRoZSBsaW5rZWQgc3RhdGUgaGFzIGEgVVJMLCB0aGUgZGlyZWN0aXZlIHdpbGwgYXV0b21hdGljYWxseSBnZW5lcmF0ZSBhbmRcbiAqIHVwZGF0ZSB0aGUgYGhyZWZgIGF0dHJpYnV0ZSAodXNpbmcgdGhlIFtbU3RhdGVTZXJ2aWNlLmhyZWZdXSAgbWV0aG9kKS5cbiAqXG4gKiAjIyMjIEV4YW1wbGU6XG4gKiBBc3N1bWluZyB0aGUgYHVzZXJzYCBzdGF0ZSBoYXMgYSB1cmwgb2YgYC91c2Vycy9gXG4gKiBgYGBodG1sXG4gKiA8YSB1aS1zcmVmPVwidXNlcnNcIiBocmVmPVwiL3VzZXJzL1wiPlVzZXJzPC9hPlxuICogYGBgXG4gKlxuICogIyMjIFBhcmFtZXRlciBWYWx1ZXNcbiAqIEluIGFkZGl0aW9uIHRvIHRoZSBzdGF0ZSBuYW1lLCBhIGB1aS1zcmVmYCBjYW4gaW5jbHVkZSBwYXJhbWV0ZXIgdmFsdWVzIHdoaWNoIGFyZSBhcHBsaWVkIHdoZW4gYWN0aXZhdGluZyB0aGUgc3RhdGUuXG4gKiBQYXJhbSB2YWx1ZXMgY2FuIGJlIHByb3ZpZGVkIGluIHRoZSBgdWktc3JlZmAgdmFsdWUgYWZ0ZXIgdGhlIHN0YXRlIG5hbWUsIGVuY2xvc2VkIGJ5IHBhcmVudGhlc2VzLlxuICogVGhlIGNvbnRlbnQgaW5zaWRlIHRoZSBwYXJlbnRoZXNlcyBpcyBhbiBleHByZXNzaW9uLCBldmFsdWF0ZWQgdG8gdGhlIHBhcmFtZXRlciB2YWx1ZXMuXG4gKlxuICogIyMjIyBFeGFtcGxlOlxuICogVGhpcyBleGFtcGxlIHJlbmRlcnMgYSBsaXN0IG9mIGxpbmtzIHRvIHVzZXJzLlxuICogVGhlIHN0YXRlJ3MgYHVzZXJJZGAgcGFyYW1ldGVyIHZhbHVlIGNvbWVzIGZyb20gZWFjaCB1c2VyJ3MgYHVzZXIuaWRgIHByb3BlcnR5LlxuICogYGBgaHRtbFxuICogPGxpIG5nLXJlcGVhdD1cInVzZXIgaW4gdXNlcnNcIj5cbiAqICAgPGEgdWktc3JlZj1cInVzZXJzLmRldGFpbCh7IHVzZXJJZDogdXNlci5pZCB9KVwiPnt7IHVzZXIuZGlzcGxheU5hbWUgfX08L2E+XG4gKiA8L2xpPlxuICogYGBgXG4gKlxuICogTm90ZTpcbiAqIFRoZSBwYXJhbWV0ZXIgdmFsdWVzIGV4cHJlc3Npb24gaXMgYCR3YXRjaGBlZCBmb3IgdXBkYXRlcy5cbiAqXG4gKiAjIyMgVHJhbnNpdGlvbiBPcHRpb25zXG4gKiBZb3UgY2FuIHNwZWNpZnkgW1tUcmFuc2l0aW9uT3B0aW9uc11dIHRvIHBhc3MgdG8gW1tTdGF0ZVNlcnZpY2UuZ29dXSBieSB1c2luZyB0aGUgYHVpLXNyZWYtb3B0c2AgYXR0cmlidXRlLlxuICogT3B0aW9ucyBhcmUgcmVzdHJpY3RlZCB0byBgbG9jYXRpb25gLCBgaW5oZXJpdGAsIGFuZCBgcmVsb2FkYC5cbiAqXG4gKiAjIyMjIEV4YW1wbGU6XG4gKiBgYGBodG1sXG4gKiA8YSB1aS1zcmVmPVwiaG9tZVwiIHVpLXNyZWYtb3B0cz1cInsgcmVsb2FkOiB0cnVlIH1cIj5Ib21lPC9hPlxuICogYGBgXG4gKlxuICogIyMjIE90aGVyIERPTSBFdmVudHNcbiAqXG4gKiBZb3UgY2FuIGFsc28gY3VzdG9taXplIHdoaWNoIERPTSBldmVudHMgdG8gcmVzcG9uZCB0byAoaW5zdGVhZCBvZiBgY2xpY2tgKSBieVxuICogcHJvdmlkaW5nIGFuIGBldmVudHNgIGFycmF5IGluIHRoZSBgdWktc3JlZi1vcHRzYCBhdHRyaWJ1dGUuXG4gKlxuICogIyMjIyBFeGFtcGxlOlxuICogYGBgaHRtbFxuICogPGlucHV0IHR5cGU9XCJ0ZXh0XCIgdWktc3JlZj1cImNvbnRhY3RzXCIgdWktc3JlZi1vcHRzPVwieyBldmVudHM6IFsnY2hhbmdlJywgJ2JsdXInXSB9XCI+XG4gKiBgYGBcbiAqXG4gKiAjIyMgSGlnaGxpZ2h0aW5nIHRoZSBhY3RpdmUgbGlua1xuICogVGhpcyBkaXJlY3RpdmUgY2FuIGJlIHVzZWQgaW4gY29uanVuY3Rpb24gd2l0aCBbW3VpU3JlZkFjdGl2ZV1dIHRvIGhpZ2hsaWdodCB0aGUgYWN0aXZlIGxpbmsuXG4gKlxuICogIyMjIEV4YW1wbGVzXG4gKiBJZiB5b3UgaGF2ZSB0aGUgZm9sbG93aW5nIHRlbXBsYXRlOlxuICpcbiAqIGBgYGh0bWxcbiAqIDxhIHVpLXNyZWY9XCJob21lXCI+SG9tZTwvYT5cbiAqIDxhIHVpLXNyZWY9XCJhYm91dFwiPkFib3V0PC9hPlxuICogPGEgdWktc3JlZj1cIntwYWdlOiAyfVwiPk5leHQgcGFnZTwvYT5cbiAqXG4gKiA8dWw+XG4gKiAgICAgPGxpIG5nLXJlcGVhdD1cImNvbnRhY3QgaW4gY29udGFjdHNcIj5cbiAqICAgICAgICAgPGEgdWktc3JlZj1cImNvbnRhY3RzLmRldGFpbCh7IGlkOiBjb250YWN0LmlkIH0pXCI+e3sgY29udGFjdC5uYW1lIH19PC9hPlxuICogICAgIDwvbGk+XG4gKiA8L3VsPlxuICogYGBgXG4gKlxuICogVGhlbiAoYXNzdW1pbmcgdGhlIGN1cnJlbnQgc3RhdGUgaXMgYGNvbnRhY3RzYCkgdGhlIHJlbmRlcmVkIGh0bWwgaW5jbHVkaW5nIGhyZWZzIHdvdWxkIGJlOlxuICpcbiAqIGBgYGh0bWxcbiAqIDxhIGhyZWY9XCIjL2hvbWVcIiB1aS1zcmVmPVwiaG9tZVwiPkhvbWU8L2E+XG4gKiA8YSBocmVmPVwiIy9hYm91dFwiIHVpLXNyZWY9XCJhYm91dFwiPkFib3V0PC9hPlxuICogPGEgaHJlZj1cIiMvY29udGFjdHM/cGFnZT0yXCIgdWktc3JlZj1cIntwYWdlOiAyfVwiPk5leHQgcGFnZTwvYT5cbiAqXG4gKiA8dWw+XG4gKiAgICAgPGxpIG5nLXJlcGVhdD1cImNvbnRhY3QgaW4gY29udGFjdHNcIj5cbiAqICAgICAgICAgPGEgaHJlZj1cIiMvY29udGFjdHMvMVwiIHVpLXNyZWY9XCJjb250YWN0cy5kZXRhaWwoeyBpZDogY29udGFjdC5pZCB9KVwiPkpvZTwvYT5cbiAqICAgICA8L2xpPlxuICogICAgIDxsaSBuZy1yZXBlYXQ9XCJjb250YWN0IGluIGNvbnRhY3RzXCI+XG4gKiAgICAgICAgIDxhIGhyZWY9XCIjL2NvbnRhY3RzLzJcIiB1aS1zcmVmPVwiY29udGFjdHMuZGV0YWlsKHsgaWQ6IGNvbnRhY3QuaWQgfSlcIj5BbGljZTwvYT5cbiAqICAgICA8L2xpPlxuICogICAgIDxsaSBuZy1yZXBlYXQ9XCJjb250YWN0IGluIGNvbnRhY3RzXCI+XG4gKiAgICAgICAgIDxhIGhyZWY9XCIjL2NvbnRhY3RzLzNcIiB1aS1zcmVmPVwiY29udGFjdHMuZGV0YWlsKHsgaWQ6IGNvbnRhY3QuaWQgfSlcIj5Cb2I8L2E+XG4gKiAgICAgPC9saT5cbiAqIDwvdWw+XG4gKlxuICogPGEgaHJlZj1cIiMvaG9tZVwiIHVpLXNyZWY9XCJob21lXCIgdWktc3JlZi1vcHRzPVwie3JlbG9hZDogdHJ1ZX1cIj5Ib21lPC9hPlxuICogYGBgXG4gKlxuICogIyMjIE5vdGVzXG4gKlxuICogLSBZb3UgY2FuIHVzZSBgdWktc3JlZmAgdG8gY2hhbmdlICoqb25seSB0aGUgcGFyYW1ldGVyIHZhbHVlcyoqIGJ5IG9taXR0aW5nIHRoZSBzdGF0ZSBuYW1lIGFuZCBwYXJlbnRoZXNlcy5cbiAqICMjIyMgRXhhbXBsZTpcbiAqIFNldHMgdGhlIGBsYW5nYCBwYXJhbWV0ZXIgdG8gYGVuYCBhbmQgcmVtYWlucyBvbiB0aGUgc2FtZSBzdGF0ZS5cbiAqXG4gKiBgYGBodG1sXG4gKiA8YSB1aS1zcmVmPVwieyBsYW5nOiAnZW4nIH1cIj5FbmdsaXNoPC9hPlxuICogYGBgXG4gKlxuICogLSBBIG1pZGRsZS1jbGljaywgcmlnaHQtY2xpY2ssIG9yIGN0cmwtY2xpY2sgaXMgaGFuZGxlZCAobmF0aXZlbHkpIGJ5IHRoZSBicm93c2VyIHRvIG9wZW4gdGhlIGhyZWYgaW4gYSBuZXcgd2luZG93LCBmb3IgZXhhbXBsZS5cbiAqXG4gKiAtIFVubGlrZSB0aGUgcGFyYW1ldGVyIHZhbHVlcyBleHByZXNzaW9uLCB0aGUgc3RhdGUgbmFtZSBpcyBub3QgYCR3YXRjaGBlZCAoZm9yIHBlcmZvcm1hbmNlIHJlYXNvbnMpLlxuICogSWYgeW91IG5lZWQgdG8gZHluYW1pY2FsbHkgdXBkYXRlIHRoZSBzdGF0ZSBiZWluZyBsaW5rZWQgdG8sIHVzZSB0aGUgZnVsbHkgZHluYW1pYyBbW3VpU3RhdGVdXSBkaXJlY3RpdmUuXG4gKi9cbmxldCB1aVNyZWY6IG5nMV9kaXJlY3RpdmU7XG51aVNyZWYgPSBbJyR1aVJvdXRlcicsICckdGltZW91dCcsXG4gIGZ1bmN0aW9uICRTdGF0ZVJlZkRpcmVjdGl2ZSgkdWlSb3V0ZXI6IFVJUm91dGVyLCAkdGltZW91dDogSVRpbWVvdXRTZXJ2aWNlKSB7XG4gICAgbGV0ICRzdGF0ZSA9ICR1aVJvdXRlci5zdGF0ZVNlcnZpY2U7XG5cbiAgICByZXR1cm4ge1xuICAgICAgcmVzdHJpY3Q6ICdBJyxcbiAgICAgIHJlcXVpcmU6IFsnP151aVNyZWZBY3RpdmUnLCAnP151aVNyZWZBY3RpdmVFcSddLFxuICAgICAgbGluazogZnVuY3Rpb24gKHNjb3BlOiBJU2NvcGUsIGVsZW1lbnQ6IElBdWdtZW50ZWRKUXVlcnksIGF0dHJzOiBhbnksIHVpU3JlZkFjdGl2ZTogYW55KSB7XG4gICAgICAgIGxldCB0eXBlID0gZ2V0VHlwZUluZm8oZWxlbWVudCk7XG4gICAgICAgIGxldCBhY3RpdmUgPSB1aVNyZWZBY3RpdmVbMV0gfHwgdWlTcmVmQWN0aXZlWzBdO1xuICAgICAgICBsZXQgdW5saW5rSW5mb0ZuOiBGdW5jdGlvbiA9IG51bGw7XG4gICAgICAgIGxldCBob29rRm47XG5cbiAgICAgICAgbGV0IHJhd0RlZiA9IHt9IGFzIERlZjtcbiAgICAgICAgbGV0IGdldERlZiA9ICgpID0+IHByb2Nlc3NlZERlZigkc3RhdGUsIGVsZW1lbnQsIHJhd0RlZik7XG5cbiAgICAgICAgbGV0IHJlZiA9IHBhcnNlU3RhdGVSZWYoYXR0cnMudWlTcmVmKTtcbiAgICAgICAgcmF3RGVmLnVpU3RhdGUgPSByZWYuc3RhdGU7XG4gICAgICAgIHJhd0RlZi51aVN0YXRlT3B0cyA9IGF0dHJzLnVpU3JlZk9wdHMgPyBzY29wZS4kZXZhbChhdHRycy51aVNyZWZPcHRzKSA6IHt9O1xuXG4gICAgICAgIGZ1bmN0aW9uIHVwZGF0ZSgpIHtcbiAgICAgICAgICBsZXQgZGVmID0gZ2V0RGVmKCk7XG4gICAgICAgICAgaWYgKHVubGlua0luZm9GbikgdW5saW5rSW5mb0ZuKCk7XG4gICAgICAgICAgaWYgKGFjdGl2ZSkgdW5saW5rSW5mb0ZuID0gYWN0aXZlLiQkYWRkU3RhdGVJbmZvKGRlZi51aVN0YXRlLCBkZWYudWlTdGF0ZVBhcmFtcyk7XG4gICAgICAgICAgaWYgKGRlZi5ocmVmICE9IG51bGwpIGF0dHJzLiRzZXQodHlwZS5hdHRyLCBkZWYuaHJlZik7XG4gICAgICAgIH1cblxuICAgICAgICBpZiAocmVmLnBhcmFtRXhwcikge1xuICAgICAgICAgIHNjb3BlLiR3YXRjaChyZWYucGFyYW1FeHByLCBmdW5jdGlvbiAodmFsKSB7XG4gICAgICAgICAgICByYXdEZWYudWlTdGF0ZVBhcmFtcyA9IGV4dGVuZCh7fSwgdmFsKTtcbiAgICAgICAgICAgIHVwZGF0ZSgpO1xuICAgICAgICAgIH0sIHRydWUpO1xuICAgICAgICAgIHJhd0RlZi51aVN0YXRlUGFyYW1zID0gZXh0ZW5kKHt9LCBzY29wZS4kZXZhbChyZWYucGFyYW1FeHByKSk7XG4gICAgICAgIH1cblxuICAgICAgICB1cGRhdGUoKTtcblxuICAgICAgICBzY29wZS4kb24oJyRkZXN0cm95JywgPGFueT4gJHVpUm91dGVyLnN0YXRlUmVnaXN0cnkub25TdGF0ZXNDaGFuZ2VkKHVwZGF0ZSkpO1xuICAgICAgICBzY29wZS4kb24oJyRkZXN0cm95JywgPGFueT4gJHVpUm91dGVyLnRyYW5zaXRpb25TZXJ2aWNlLm9uU3VjY2Vzcyh7fSwgdXBkYXRlKSk7XG5cbiAgICAgICAgaWYgKCF0eXBlLmNsaWNrYWJsZSkgcmV0dXJuO1xuICAgICAgICBob29rRm4gPSBjbGlja0hvb2soZWxlbWVudCwgJHN0YXRlLCAkdGltZW91dCwgdHlwZSwgZ2V0RGVmKTtcbiAgICAgICAgYmluZEV2ZW50cyhlbGVtZW50LCBzY29wZSwgaG9va0ZuLCByYXdEZWYudWlTdGF0ZU9wdHMpO1xuICAgICAgfVxuICAgIH07XG4gIH1dO1xuXG4vKipcbiAqIGB1aS1zdGF0ZWA6IEEgZnVsbHkgZHluYW1pYyBkaXJlY3RpdmUgZm9yIGxpbmtpbmcgdG8gYSBzdGF0ZVxuICpcbiAqIEEgZGlyZWN0aXZlIHdoaWNoIGxpbmtzIHRvIGEgc3RhdGUgKGFuZCBvcHRpb25hbGx5LCBwYXJhbWV0ZXJzKS5cbiAqIFdoZW4gY2xpY2tlZCwgdGhpcyBkaXJlY3RpdmUgYWN0aXZhdGVzIHRoZSBsaW5rZWQgc3RhdGUgd2l0aCB0aGUgc3VwcGxpZWQgcGFyYW1ldGVyIHZhbHVlcy5cbiAqXG4gKiAqKlRoaXMgZGlyZWN0aXZlIGlzIHZlcnkgc2ltaWxhciB0byBbW3VpU3JlZl1dLCBidXQgaXQgYCRvYnNlcnZlYHMgYW5kIGAkd2F0Y2hgZXMvZXZhbHVhdGVzIGFsbCBpdHMgaW5wdXRzLioqXG4gKlxuICogQSBkaXJlY3RpdmUgd2hpY2ggbGlua3MgdG8gYSBzdGF0ZSAoYW5kIG9wdGlvbmFsbHksIHBhcmFtZXRlcnMpLlxuICogV2hlbiBjbGlja2VkLCB0aGlzIGRpcmVjdGl2ZSBhY3RpdmF0ZXMgdGhlIGxpbmtlZCBzdGF0ZSB3aXRoIHRoZSBzdXBwbGllZCBwYXJhbWV0ZXIgdmFsdWVzLlxuICpcbiAqICMjIyBMaW5rZWQgU3RhdGVcbiAqIFRoZSBhdHRyaWJ1dGUgdmFsdWUgb2YgYHVpLXN0YXRlYCBpcyBhbiBleHByZXNzaW9uIHdoaWNoIGlzIGAkd2F0Y2hgZWQgYW5kIGV2YWx1YXRlZCBhcyB0aGUgc3RhdGUgdG8gbGluayB0by5cbiAqICoqVGhpcyBpcyBpbiBjb250cmFzdCB3aXRoIGB1aS1zcmVmYCwgd2hpY2ggdGFrZXMgYSBzdGF0ZSBuYW1lIGFzIGEgc3RyaW5nIGxpdGVyYWwuKipcbiAqXG4gKiAjIyMjIEV4YW1wbGU6XG4gKiBDcmVhdGUgYSBsaXN0IG9mIGxpbmtzLlxuICogYGBgaHRtbFxuICogPGxpIG5nLXJlcGVhdD1cImxpbmsgaW4gbmF2bGlua3NcIj5cbiAqICAgPGEgdWktc3RhdGU9XCJsaW5rLnN0YXRlXCI+e3sgbGluay5kaXNwbGF5TmFtZSB9fTwvYT5cbiAqIDwvbGk+XG4gKiBgYGBcbiAqXG4gKiAjIyMgUmVsYXRpdmUgTGlua3NcbiAqIElmIHRoZSBleHByZXNzaW9uIGV2YWx1YXRlcyB0byBhIHJlbGF0aXZlIHBhdGgsIGl0IGlzIHByb2Nlc3NlZCBsaWtlIFtbdWlTcmVmXV0uXG4gKiBZb3UganVzdCBuZWVkIHRvIGJlIGF3YXJlIHRoYXQgdGhlIHBhdGggaXMgcmVsYXRpdmUgdG8gdGhlIHN0YXRlIHRoYXQgKmNyZWF0ZWQqIHRoZSBsaW5rLlxuICogVGhpcyBhbGxvd3MgYSBzdGF0ZSB0byBjcmVhdGUgcmVsYXRpdmUgYHVpLXN0YXRlYCB3aGljaCBhbHdheXMgdGFyZ2V0cyB0aGUgc2FtZSBkZXN0aW5hdGlvbi5cbiAqXG4gKiAjIyMgaHJlZnNcbiAqIElmIHRoZSBsaW5rZWQgc3RhdGUgaGFzIGEgVVJMLCB0aGUgZGlyZWN0aXZlIHdpbGwgYXV0b21hdGljYWxseSBnZW5lcmF0ZSBhbmRcbiAqIHVwZGF0ZSB0aGUgYGhyZWZgIGF0dHJpYnV0ZSAodXNpbmcgdGhlIFtbU3RhdGVTZXJ2aWNlLmhyZWZdXSAgbWV0aG9kKS5cbiAqXG4gKiAjIyMgUGFyYW1ldGVyIFZhbHVlc1xuICogSW4gYWRkaXRpb24gdG8gdGhlIHN0YXRlIG5hbWUgZXhwcmVzc2lvbiwgYSBgdWktc3RhdGVgIGNhbiBpbmNsdWRlIHBhcmFtZXRlciB2YWx1ZXMgd2hpY2ggYXJlIGFwcGxpZWQgd2hlbiBhY3RpdmF0aW5nIHRoZSBzdGF0ZS5cbiAqIFBhcmFtIHZhbHVlcyBzaG91bGQgYmUgcHJvdmlkZWQgdXNpbmcgdGhlIGB1aS1zdGF0ZS1wYXJhbXNgIGF0dHJpYnV0ZS5cbiAqIFRoZSBgdWktc3RhdGUtcGFyYW1zYCBhdHRyaWJ1dGUgdmFsdWUgaXMgYCR3YXRjaGBlZCBhbmQgZXZhbHVhdGVkIGFzIGFuIGV4cHJlc3Npb24uXG4gKlxuICogIyMjIyBFeGFtcGxlOlxuICogVGhpcyBleGFtcGxlIHJlbmRlcnMgYSBsaXN0IG9mIGxpbmtzIHdpdGggcGFyYW0gdmFsdWVzLlxuICogVGhlIHN0YXRlJ3MgYHVzZXJJZGAgcGFyYW1ldGVyIHZhbHVlIGNvbWVzIGZyb20gZWFjaCB1c2VyJ3MgYHVzZXIuaWRgIHByb3BlcnR5LlxuICogYGBgaHRtbFxuICogPGxpIG5nLXJlcGVhdD1cImxpbmsgaW4gbmF2bGlua3NcIj5cbiAqICAgPGEgdWktc3RhdGU9XCJsaW5rLnN0YXRlXCIgdWktc3RhdGUtcGFyYW1zPVwibGluay5wYXJhbXNcIj57eyBsaW5rLmRpc3BsYXlOYW1lIH19PC9hPlxuICogPC9saT5cbiAqIGBgYFxuICpcbiAqICMjIyBUcmFuc2l0aW9uIE9wdGlvbnNcbiAqIFlvdSBjYW4gc3BlY2lmeSBbW1RyYW5zaXRpb25PcHRpb25zXV0gdG8gcGFzcyB0byBbW1N0YXRlU2VydmljZS5nb11dIGJ5IHVzaW5nIHRoZSBgdWktc3RhdGUtb3B0c2AgYXR0cmlidXRlLlxuICogT3B0aW9ucyBhcmUgcmVzdHJpY3RlZCB0byBgbG9jYXRpb25gLCBgaW5oZXJpdGAsIGFuZCBgcmVsb2FkYC5cbiAqIFRoZSB2YWx1ZSBvZiB0aGUgYHVpLXN0YXRlLW9wdHNgIGlzIGAkd2F0Y2hgZWQgYW5kIGV2YWx1YXRlZCBhcyBhbiBleHByZXNzaW9uLlxuICpcbiAqICMjIyMgRXhhbXBsZTpcbiAqIGBgYGh0bWxcbiAqIDxhIHVpLXN0YXRlPVwicmV0dXJudG8uc3RhdGVcIiB1aS1zdGF0ZS1vcHRzPVwieyByZWxvYWQ6IHRydWUgfVwiPkhvbWU8L2E+XG4gKiBgYGBcbiAqXG4gKiAjIyMgT3RoZXIgRE9NIEV2ZW50c1xuICpcbiAqIFlvdSBjYW4gYWxzbyBjdXN0b21pemUgd2hpY2ggRE9NIGV2ZW50cyB0byByZXNwb25kIHRvIChpbnN0ZWFkIG9mIGBjbGlja2ApIGJ5XG4gKiBwcm92aWRpbmcgYW4gYGV2ZW50c2AgYXJyYXkgaW4gdGhlIGB1aS1zdGF0ZS1vcHRzYCBhdHRyaWJ1dGUuXG4gKlxuICogIyMjIyBFeGFtcGxlOlxuICogYGBgaHRtbFxuICogPGlucHV0IHR5cGU9XCJ0ZXh0XCIgdWktc3RhdGU9XCJjb250YWN0c1wiIHVpLXN0YXRlLW9wdHM9XCJ7IGV2ZW50czogWydjaGFuZ2UnLCAnYmx1ciddIH1cIj5cbiAqIGBgYFxuICpcbiAqICMjIyBIaWdobGlnaHRpbmcgdGhlIGFjdGl2ZSBsaW5rXG4gKiBUaGlzIGRpcmVjdGl2ZSBjYW4gYmUgdXNlZCBpbiBjb25qdW5jdGlvbiB3aXRoIFtbdWlTcmVmQWN0aXZlXV0gdG8gaGlnaGxpZ2h0IHRoZSBhY3RpdmUgbGluay5cbiAqXG4gKiAjIyMgTm90ZXNcbiAqXG4gKiAtIFlvdSBjYW4gdXNlIGB1aS1wYXJhbXNgIHRvIGNoYW5nZSAqKm9ubHkgdGhlIHBhcmFtZXRlciB2YWx1ZXMqKiBieSBvbWl0dGluZyB0aGUgc3RhdGUgbmFtZSBhbmQgc3VwcGx5aW5nIG9ubHkgYHVpLXN0YXRlLXBhcmFtc2AuXG4gKiAgIEhvd2V2ZXIsIGl0IG1pZ2h0IGJlIHNpbXBsZXIgdG8gdXNlIFtbdWlTcmVmXV0gcGFyYW1ldGVyLW9ubHkgbGlua3MuXG4gKlxuICogIyMjIyBFeGFtcGxlOlxuICogU2V0cyB0aGUgYGxhbmdgIHBhcmFtZXRlciB0byBgZW5gIGFuZCByZW1haW5zIG9uIHRoZSBzYW1lIHN0YXRlLlxuICpcbiAqIGBgYGh0bWxcbiAqIDxhIHVpLXN0YXRlPVwiXCIgdWktc3RhdGUtcGFyYW1zPVwieyBsYW5nOiAnZW4nIH1cIj5FbmdsaXNoPC9hPlxuICogYGBgXG4gKlxuICogLSBBIG1pZGRsZS1jbGljaywgcmlnaHQtY2xpY2ssIG9yIGN0cmwtY2xpY2sgaXMgaGFuZGxlZCAobmF0aXZlbHkpIGJ5IHRoZSBicm93c2VyIHRvIG9wZW4gdGhlIGhyZWYgaW4gYSBuZXcgd2luZG93LCBmb3IgZXhhbXBsZS5cbiAqIGBgYFxuICovXG5sZXQgdWlTdGF0ZTogbmcxX2RpcmVjdGl2ZTtcbnVpU3RhdGUgPSBbJyR1aVJvdXRlcicsICckdGltZW91dCcsXG4gIGZ1bmN0aW9uICRTdGF0ZVJlZkR5bmFtaWNEaXJlY3RpdmUoJHVpUm91dGVyOiBVSVJvdXRlciwgJHRpbWVvdXQ6IElUaW1lb3V0U2VydmljZSkge1xuICAgIGxldCAkc3RhdGUgPSAkdWlSb3V0ZXIuc3RhdGVTZXJ2aWNlO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIHJlc3RyaWN0OiAnQScsXG4gICAgICByZXF1aXJlOiBbJz9edWlTcmVmQWN0aXZlJywgJz9edWlTcmVmQWN0aXZlRXEnXSxcbiAgICAgIGxpbms6IGZ1bmN0aW9uIChzY29wZTogSVNjb3BlLCBlbGVtZW50OiBJQXVnbWVudGVkSlF1ZXJ5LCBhdHRyczogYW55LCB1aVNyZWZBY3RpdmU6IGFueSkge1xuICAgICAgICBsZXQgdHlwZSA9IGdldFR5cGVJbmZvKGVsZW1lbnQpO1xuICAgICAgICBsZXQgYWN0aXZlID0gdWlTcmVmQWN0aXZlWzFdIHx8IHVpU3JlZkFjdGl2ZVswXTtcbiAgICAgICAgbGV0IHVubGlua0luZm9GbjogRnVuY3Rpb24gPSBudWxsO1xuICAgICAgICBsZXQgaG9va0ZuO1xuXG4gICAgICAgIGxldCByYXdEZWYgPSB7fSBhcyBEZWY7XG4gICAgICAgIGxldCBnZXREZWYgPSAoKSA9PiBwcm9jZXNzZWREZWYoJHN0YXRlLCBlbGVtZW50LCByYXdEZWYpO1xuXG4gICAgICAgIGxldCBpbnB1dEF0dHJzID0gWyd1aVN0YXRlJywgJ3VpU3RhdGVQYXJhbXMnLCAndWlTdGF0ZU9wdHMnXTtcbiAgICAgICAgbGV0IHdhdGNoRGVyZWdGbnMgPSBpbnB1dEF0dHJzLnJlZHVjZSgoYWNjLCBhdHRyKSA9PiAoYWNjW2F0dHJdID0gbm9vcCwgYWNjKSwge30pO1xuXG4gICAgICAgIGZ1bmN0aW9uIHVwZGF0ZSgpIHtcbiAgICAgICAgICBsZXQgZGVmID0gZ2V0RGVmKCk7XG4gICAgICAgICAgaWYgKHVubGlua0luZm9GbikgdW5saW5rSW5mb0ZuKCk7XG4gICAgICAgICAgaWYgKGFjdGl2ZSkgdW5saW5rSW5mb0ZuID0gYWN0aXZlLiQkYWRkU3RhdGVJbmZvKGRlZi51aVN0YXRlLCBkZWYudWlTdGF0ZVBhcmFtcyk7XG4gICAgICAgICAgaWYgKGRlZi5ocmVmICE9IG51bGwpIGF0dHJzLiRzZXQodHlwZS5hdHRyLCBkZWYuaHJlZik7XG4gICAgICAgIH1cblxuICAgICAgICBpbnB1dEF0dHJzLmZvckVhY2goKGZpZWxkKSA9PiB7XG4gICAgICAgICAgcmF3RGVmW2ZpZWxkXSA9IGF0dHJzW2ZpZWxkXSA/IHNjb3BlLiRldmFsKGF0dHJzW2ZpZWxkXSkgOiBudWxsO1xuXG4gICAgICAgICAgYXR0cnMuJG9ic2VydmUoZmllbGQsIChleHByKSA9PiB7XG4gICAgICAgICAgICB3YXRjaERlcmVnRm5zW2ZpZWxkXSgpO1xuICAgICAgICAgICAgd2F0Y2hEZXJlZ0Zuc1tmaWVsZF0gPSBzY29wZS4kd2F0Y2goZXhwciwgKG5ld3ZhbCkgPT4ge1xuICAgICAgICAgICAgICByYXdEZWZbZmllbGRdID0gbmV3dmFsO1xuICAgICAgICAgICAgICB1cGRhdGUoKTtcbiAgICAgICAgICAgIH0sIHRydWUpO1xuICAgICAgICAgIH0pXG4gICAgICAgIH0pO1xuXG4gICAgICAgIHVwZGF0ZSgpO1xuXG4gICAgICAgIHNjb3BlLiRvbignJGRlc3Ryb3knLCA8YW55PiAkdWlSb3V0ZXIuc3RhdGVSZWdpc3RyeS5vblN0YXRlc0NoYW5nZWQodXBkYXRlKSk7XG4gICAgICAgIHNjb3BlLiRvbignJGRlc3Ryb3knLCA8YW55PiAkdWlSb3V0ZXIudHJhbnNpdGlvblNlcnZpY2Uub25TdWNjZXNzKHt9LCB1cGRhdGUpKTtcblxuICAgICAgICBpZiAoIXR5cGUuY2xpY2thYmxlKSByZXR1cm47XG4gICAgICAgIGhvb2tGbiA9IGNsaWNrSG9vayhlbGVtZW50LCAkc3RhdGUsICR0aW1lb3V0LCB0eXBlLCBnZXREZWYpO1xuICAgICAgICBiaW5kRXZlbnRzKGVsZW1lbnQsIHNjb3BlLCBob29rRm4sIHJhd0RlZi51aVN0YXRlT3B0cyk7XG4gICAgICB9XG4gICAgfTtcbiAgfV07XG5cblxuLyoqXG4gKiBgdWktc3JlZi1hY3RpdmVgIGFuZCBgdWktc3JlZi1hY3RpdmUtZXFgOiBBIGRpcmVjdGl2ZSB0aGF0IGFkZHMgYSBDU1MgY2xhc3Mgd2hlbiBhIGB1aS1zcmVmYCBpcyBhY3RpdmVcbiAqXG4gKiBBIGRpcmVjdGl2ZSB3b3JraW5nIGFsb25nc2lkZSBbW3VpU3JlZl1dIGFuZCBbW3VpU3RhdGVdXSB0byBhZGQgY2xhc3NlcyB0byBhbiBlbGVtZW50IHdoZW4gdGhlXG4gKiByZWxhdGVkIGRpcmVjdGl2ZSdzIHN0YXRlIGlzIGFjdGl2ZSAoYW5kIHJlbW92ZSB0aGVtIHdoZW4gaXQgaXMgaW5hY3RpdmUpLlxuICpcbiAqIFRoZSBwcmltYXJ5IHVzZS1jYXNlIGlzIHRvIGhpZ2hsaWdodCB0aGUgYWN0aXZlIGxpbmsgaW4gbmF2aWdhdGlvbiBtZW51cyxcbiAqIGRpc3Rpbmd1aXNoaW5nIGl0IGZyb20gdGhlIGluYWN0aXZlIG1lbnUgaXRlbXMuXG4gKlxuICogIyMjIExpbmtpbmcgdG8gYSBgdWktc3JlZmAgb3IgYHVpLXN0YXRlYFxuICogYHVpLXNyZWYtYWN0aXZlYCBjYW4gbGl2ZSBvbiB0aGUgc2FtZSBlbGVtZW50IGFzIGB1aS1zcmVmYC9gdWktc3RhdGVgLCBvciBpdCBjYW4gYmUgb24gYSBwYXJlbnQgZWxlbWVudC5cbiAqIElmIGEgYHVpLXNyZWYtYWN0aXZlYCBpcyBhIHBhcmVudCB0byBtb3JlIHRoYW4gb25lIGB1aS1zcmVmYC9gdWktc3RhdGVgLCBpdCB3aWxsIGFwcGx5IHRoZSBDU1MgY2xhc3Mgd2hlbiAqKmFueSBvZiB0aGUgbGlua3MgYXJlIGFjdGl2ZSoqLlxuICpcbiAqICMjIyBNYXRjaGluZ1xuICpcbiAqIFRoZSBgdWktc3JlZi1hY3RpdmVgIGRpcmVjdGl2ZSBhcHBsaWVzIHRoZSBDU1MgY2xhc3Mgd2hlbiB0aGUgYHVpLXNyZWZgL2B1aS1zdGF0ZWAncyB0YXJnZXQgc3RhdGUgKipvciBhbnkgY2hpbGQgc3RhdGUgaXMgYWN0aXZlKiouXG4gKiBUaGlzIGlzIGEgXCJmdXp6eSBtYXRjaFwiIHdoaWNoIHVzZXMgW1tTdGF0ZVNlcnZpY2UuaW5jbHVkZXNdXS5cbiAqXG4gKiBUaGUgYHVpLXNyZWYtYWN0aXZlLWVxYCBkaXJlY3RpdmUgYXBwbGllcyB0aGUgQ1NTIGNsYXNzIHdoZW4gdGhlIGB1aS1zcmVmYC9gdWktc3RhdGVgJ3MgdGFyZ2V0IHN0YXRlIGlzIGRpcmVjdGx5IGFjdGl2ZSAobm90IHdoZW4gY2hpbGQgc3RhdGVzIGFyZSBhY3RpdmUpLlxuICogVGhpcyBpcyBhbiBcImV4YWN0IG1hdGNoXCIgd2hpY2ggdXNlcyBbW1N0YXRlU2VydmljZS5pc11dLlxuICpcbiAqICMjIyBQYXJhbWV0ZXIgdmFsdWVzXG4gKiBJZiB0aGUgYHVpLXNyZWZgL2B1aS1zdGF0ZWAgaW5jbHVkZXMgcGFyYW1ldGVyIHZhbHVlcywgdGhlIGN1cnJlbnQgcGFyYW1ldGVyIHZhbHVlcyBtdXN0IG1hdGNoIHRoZSBsaW5rJ3MgdmFsdWVzIGZvciB0aGUgbGluayB0byBiZSBoaWdobGlnaHRlZC5cbiAqIFRoaXMgYWxsb3dzIGEgbGlzdCBvZiBsaW5rcyB0byB0aGUgc2FtZSBzdGF0ZSB3aXRoIGRpZmZlcmVudCBwYXJhbWV0ZXJzIHRvIGJlIHJlbmRlcmVkLCBhbmQgdGhlIGNvcnJlY3Qgb25lIGhpZ2hsaWdodGVkLlxuICpcbiAqICMjIyMgRXhhbXBsZTpcbiAqIGBgYGh0bWxcbiAqIDxsaSBuZy1yZXBlYXQ9XCJ1c2VyIGluIHVzZXJzXCIgdWktc3JlZi1hY3RpdmU9XCJhY3RpdmVcIj5cbiAqICAgPGEgdWktc3JlZj1cInVzZXIuZGV0YWlscyh7IHVzZXJJZDogdXNlci5pZCB9KVwiPnt7IHVzZXIubGFzdE5hbWUgfX08L2E+XG4gKiA8L2xpPlxuICogYGBgXG4gKlxuICogIyMjIEV4YW1wbGVzXG4gKlxuICogR2l2ZW4gdGhlIGZvbGxvd2luZyB0ZW1wbGF0ZTpcbiAqICMjIyMgRXhhbXBsZTpcbiAqIGBgYGh0bWxcbiAqIDx1bD5cbiAqICAgPGxpIHVpLXNyZWYtYWN0aXZlPVwiYWN0aXZlXCIgY2xhc3M9XCJpdGVtXCI+XG4gKiAgICAgPGEgaHJlZiB1aS1zcmVmPVwiYXBwLnVzZXIoe3VzZXI6ICdiaWxib2JhZ2dpbnMnfSlcIj5AYmlsYm9iYWdnaW5zPC9hPlxuICogICA8L2xpPlxuICogPC91bD5cbiAqIGBgYFxuICpcbiAqIFdoZW4gdGhlIGFwcCBzdGF0ZSBpcyBgYXBwLnVzZXJgIChvciBhbnkgY2hpbGQgc3RhdGUpLFxuICogYW5kIGNvbnRhaW5zIHRoZSBzdGF0ZSBwYXJhbWV0ZXIgXCJ1c2VyXCIgd2l0aCB2YWx1ZSBcImJpbGJvYmFnZ2luc1wiLFxuICogdGhlIHJlc3VsdGluZyBIVE1MIHdpbGwgYXBwZWFyIGFzIChub3RlIHRoZSAnYWN0aXZlJyBjbGFzcyk6XG4gKlxuICogYGBgaHRtbFxuICogPHVsPlxuICogICA8bGkgdWktc3JlZi1hY3RpdmU9XCJhY3RpdmVcIiBjbGFzcz1cIml0ZW0gYWN0aXZlXCI+XG4gKiAgICAgPGEgdWktc3JlZj1cImFwcC51c2VyKHt1c2VyOiAnYmlsYm9iYWdnaW5zJ30pXCIgaHJlZj1cIi91c2Vycy9iaWxib2JhZ2dpbnNcIj5AYmlsYm9iYWdnaW5zPC9hPlxuICogICA8L2xpPlxuICogPC91bD5cbiAqIGBgYFxuICpcbiAqICMjIyBHbG9iIG1vZGVcbiAqXG4gKiBJdCBpcyBwb3NzaWJsZSB0byBwYXNzIGB1aS1zcmVmLWFjdGl2ZWAgYW4gZXhwcmVzc2lvbiB0aGF0IGV2YWx1YXRlcyB0byBhbiBvYmplY3QuXG4gKiBUaGUgb2JqZWN0cyBrZXlzIHJlcHJlc2VudCBhY3RpdmUgY2xhc3MgbmFtZXMgYW5kIHZhbHVlcyByZXByZXNlbnQgdGhlIHJlc3BlY3RpdmUgc3RhdGUgbmFtZXMvZ2xvYnMuXG4gKiBgdWktc3JlZi1hY3RpdmVgIHdpbGwgbWF0Y2ggaWYgdGhlIGN1cnJlbnQgYWN0aXZlIHN0YXRlICoqaW5jbHVkZXMqKiBhbnkgb2ZcbiAqIHRoZSBzcGVjaWZpZWQgc3RhdGUgbmFtZXMvZ2xvYnMsIGV2ZW4gdGhlIGFic3RyYWN0IG9uZXMuXG4gKlxuICogIyMjIyBFeGFtcGxlOlxuICogR2l2ZW4gdGhlIGZvbGxvd2luZyB0ZW1wbGF0ZSwgd2l0aCBcImFkbWluXCIgYmVpbmcgYW4gYWJzdHJhY3Qgc3RhdGU6XG4gKiBgYGBodG1sXG4gKiA8ZGl2IHVpLXNyZWYtYWN0aXZlPVwieydhY3RpdmUnOiAnYWRtaW4uKionfVwiPlxuICogICA8YSB1aS1zcmVmLWFjdGl2ZT1cImFjdGl2ZVwiIHVpLXNyZWY9XCJhZG1pbi5yb2xlc1wiPlJvbGVzPC9hPlxuICogPC9kaXY+XG4gKiBgYGBcbiAqXG4gKiBXaGVuIHRoZSBjdXJyZW50IHN0YXRlIGlzIFwiYWRtaW4ucm9sZXNcIiB0aGUgXCJhY3RpdmVcIiBjbGFzcyB3aWxsIGJlIGFwcGxpZWQgdG8gYm90aCB0aGUgPGRpdj4gYW5kIDxhPiBlbGVtZW50cy5cbiAqIEl0IGlzIGltcG9ydGFudCB0byBub3RlIHRoYXQgdGhlIHN0YXRlIG5hbWVzL2dsb2JzIHBhc3NlZCB0byBgdWktc3JlZi1hY3RpdmVgIG92ZXJyaWRlIGFueSBzdGF0ZSBwcm92aWRlZCBieSBhIGxpbmtlZCBgdWktc3JlZmAuXG4gKlxuICogIyMjIE5vdGVzOlxuICpcbiAqIC0gVGhlIGNsYXNzIG5hbWUgaXMgaW50ZXJwb2xhdGVkICoqb25jZSoqIGR1cmluZyB0aGUgZGlyZWN0aXZlcyBsaW5rIHRpbWUgKGFueSBmdXJ0aGVyIGNoYW5nZXMgdG8gdGhlXG4gKiBpbnRlcnBvbGF0ZWQgdmFsdWUgYXJlIGlnbm9yZWQpLlxuICpcbiAqIC0gTXVsdGlwbGUgY2xhc3NlcyBtYXkgYmUgc3BlY2lmaWVkIGluIGEgc3BhY2Utc2VwYXJhdGVkIGZvcm1hdDogYHVpLXNyZWYtYWN0aXZlPSdjbGFzczEgY2xhc3MyIGNsYXNzMydgXG4gKi9cbmxldCB1aVNyZWZBY3RpdmU6IG5nMV9kaXJlY3RpdmU7XG51aVNyZWZBY3RpdmUgPSBbJyRzdGF0ZScsICckc3RhdGVQYXJhbXMnLCAnJGludGVycG9sYXRlJywgJyR1aVJvdXRlcicsXG4gIGZ1bmN0aW9uICRTdGF0ZVJlZkFjdGl2ZURpcmVjdGl2ZSgkc3RhdGU6IFN0YXRlU2VydmljZSwgJHN0YXRlUGFyYW1zOiBPYmosICRpbnRlcnBvbGF0ZTogSUludGVycG9sYXRlU2VydmljZSwgJHVpUm91dGVyOiBVSVJvdXRlcikge1xuICAgIHJldHVybiB7XG4gICAgICByZXN0cmljdDogXCJBXCIsXG4gICAgICBjb250cm9sbGVyOiBbJyRzY29wZScsICckZWxlbWVudCcsICckYXR0cnMnLFxuICAgICAgICBmdW5jdGlvbiAoJHNjb3BlOiBJU2NvcGUsICRlbGVtZW50OiBJQXVnbWVudGVkSlF1ZXJ5LCAkYXR0cnM6IGFueSkge1xuICAgICAgICAgIGxldCBzdGF0ZXM6IFN0YXRlRGF0YVtdID0gW10sXG4gICAgICAgICAgICAgIGFjdGl2ZUVxQ2xhc3M6IHN0cmluZyxcbiAgICAgICAgICAgICAgdWlTcmVmQWN0aXZlOiBhbnk7XG5cbiAgICAgICAgICAvLyBUaGVyZSBwcm9iYWJseSBpc24ndCBtdWNoIHBvaW50IGluICRvYnNlcnZpbmcgdGhpc1xuICAgICAgICAgIC8vIHVpU3JlZkFjdGl2ZSBhbmQgdWlTcmVmQWN0aXZlRXEgc2hhcmUgdGhlIHNhbWUgZGlyZWN0aXZlIG9iamVjdCB3aXRoIHNvbWVcbiAgICAgICAgICAvLyBzbGlnaHQgZGlmZmVyZW5jZSBpbiBsb2dpYyByb3V0aW5nXG4gICAgICAgICAgYWN0aXZlRXFDbGFzcyA9ICRpbnRlcnBvbGF0ZSgkYXR0cnMudWlTcmVmQWN0aXZlRXEgfHwgJycsIGZhbHNlKSgkc2NvcGUpO1xuXG4gICAgICAgICAgdHJ5IHtcbiAgICAgICAgICAgIHVpU3JlZkFjdGl2ZSA9ICRzY29wZS4kZXZhbCgkYXR0cnMudWlTcmVmQWN0aXZlKTtcbiAgICAgICAgICB9IGNhdGNoIChlKSB7XG4gICAgICAgICAgICAvLyBEbyBub3RoaW5nLiB1aVNyZWZBY3RpdmUgaXMgbm90IGEgdmFsaWQgZXhwcmVzc2lvbi5cbiAgICAgICAgICAgIC8vIEZhbGwgYmFjayB0byB1c2luZyAkaW50ZXJwb2xhdGUgYmVsb3dcbiAgICAgICAgICB9XG4gICAgICAgICAgdWlTcmVmQWN0aXZlID0gdWlTcmVmQWN0aXZlIHx8ICRpbnRlcnBvbGF0ZSgkYXR0cnMudWlTcmVmQWN0aXZlIHx8ICcnLCBmYWxzZSkoJHNjb3BlKTtcbiAgICAgICAgICBpZiAoaXNPYmplY3QodWlTcmVmQWN0aXZlKSkge1xuICAgICAgICAgICAgZm9yRWFjaCh1aVNyZWZBY3RpdmUsIGZ1bmN0aW9uIChzdGF0ZU9yTmFtZTogU3RhdGVPck5hbWUsIGFjdGl2ZUNsYXNzOiBzdHJpbmcpIHtcbiAgICAgICAgICAgICAgaWYgKGlzU3RyaW5nKHN0YXRlT3JOYW1lKSkge1xuICAgICAgICAgICAgICAgIGxldCByZWYgPSBwYXJzZVN0YXRlUmVmKHN0YXRlT3JOYW1lKTtcbiAgICAgICAgICAgICAgICBhZGRTdGF0ZShyZWYuc3RhdGUsICRzY29wZS4kZXZhbChyZWYucGFyYW1FeHByKSwgYWN0aXZlQ2xhc3MpO1xuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9KTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICAvLyBBbGxvdyB1aVNyZWYgdG8gY29tbXVuaWNhdGUgd2l0aCB1aVNyZWZBY3RpdmVbRXF1YWxzXVxuICAgICAgICAgIHRoaXMuJCRhZGRTdGF0ZUluZm8gPSBmdW5jdGlvbiAobmV3U3RhdGU6IHN0cmluZywgbmV3UGFyYW1zOiBPYmopIHtcbiAgICAgICAgICAgIC8vIHdlIGFscmVhZHkgZ290IGFuIGV4cGxpY2l0IHN0YXRlIHByb3ZpZGVkIGJ5IHVpLXNyZWYtYWN0aXZlLCBzbyB3ZVxuICAgICAgICAgICAgLy8gc2hhZG93IHRoZSBvbmUgdGhhdCBjb21lcyBmcm9tIHVpLXNyZWZcbiAgICAgICAgICAgIGlmIChpc09iamVjdCh1aVNyZWZBY3RpdmUpICYmIHN0YXRlcy5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGxldCBkZXJlZ2lzdGVyID0gYWRkU3RhdGUobmV3U3RhdGUsIG5ld1BhcmFtcywgdWlTcmVmQWN0aXZlKTtcbiAgICAgICAgICAgIHVwZGF0ZSgpO1xuICAgICAgICAgICAgcmV0dXJuIGRlcmVnaXN0ZXI7XG4gICAgICAgICAgfTtcblxuICAgICAgICAgIGZ1bmN0aW9uIHVwZGF0ZUFmdGVyVHJhbnNpdGlvbih0cmFucykge1xuICAgICAgICAgICAgdHJhbnMucHJvbWlzZS50aGVuKHVwZGF0ZSwgbm9vcCk7XG4gICAgICAgICAgfVxuXG4gICAgICAgICAgJHNjb3BlLiRvbignJHN0YXRlQ2hhbmdlU3VjY2VzcycsIHVwZGF0ZSk7XG4gICAgICAgICAgJHNjb3BlLiRvbignJGRlc3Ryb3knLCA8YW55PiAkdWlSb3V0ZXIudHJhbnNpdGlvblNlcnZpY2Uub25TdGFydCh7fSwgdXBkYXRlQWZ0ZXJUcmFuc2l0aW9uKSk7XG4gICAgICAgICAgaWYgKCR1aVJvdXRlci5nbG9iYWxzLnRyYW5zaXRpb24pIHtcbiAgICAgICAgICAgIHVwZGF0ZUFmdGVyVHJhbnNpdGlvbigkdWlSb3V0ZXIuZ2xvYmFscy50cmFuc2l0aW9uKTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICBmdW5jdGlvbiBhZGRTdGF0ZShzdGF0ZU5hbWU6IHN0cmluZywgc3RhdGVQYXJhbXM6IE9iaiwgYWN0aXZlQ2xhc3M6IHN0cmluZykge1xuICAgICAgICAgICAgdmFyIHN0YXRlID0gJHN0YXRlLmdldChzdGF0ZU5hbWUsIHN0YXRlQ29udGV4dCgkZWxlbWVudCkpO1xuXG4gICAgICAgICAgICB2YXIgc3RhdGVJbmZvID0ge1xuICAgICAgICAgICAgICBzdGF0ZTogc3RhdGUgfHwgeyBuYW1lOiBzdGF0ZU5hbWUgfSxcbiAgICAgICAgICAgICAgcGFyYW1zOiBzdGF0ZVBhcmFtcyxcbiAgICAgICAgICAgICAgYWN0aXZlQ2xhc3M6IGFjdGl2ZUNsYXNzXG4gICAgICAgICAgICB9O1xuXG4gICAgICAgICAgICBzdGF0ZXMucHVzaChzdGF0ZUluZm8pO1xuXG4gICAgICAgICAgICByZXR1cm4gZnVuY3Rpb24gcmVtb3ZlU3RhdGUoKSB7XG4gICAgICAgICAgICAgIHJlbW92ZUZyb20oc3RhdGVzKShzdGF0ZUluZm8pO1xuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cblxuICAgICAgICAgIC8vIFVwZGF0ZSByb3V0ZSBzdGF0ZVxuICAgICAgICAgIGZ1bmN0aW9uIHVwZGF0ZSgpIHtcbiAgICAgICAgICAgIGNvbnN0IHNwbGl0Q2xhc3NlcyA9IHN0ciA9PlxuICAgICAgICAgICAgICAgIHN0ci5zcGxpdCgvXFxzLykuZmlsdGVyKGlkZW50aXR5KTtcbiAgICAgICAgICAgIGNvbnN0IGdldENsYXNzZXMgPSAoc3RhdGVMaXN0OiBTdGF0ZURhdGFbXSkgPT5cbiAgICAgICAgICAgICAgICBzdGF0ZUxpc3QubWFwKHggPT4geC5hY3RpdmVDbGFzcykubWFwKHNwbGl0Q2xhc3NlcykucmVkdWNlKHVubmVzdFIsIFtdKTtcblxuICAgICAgICAgICAgbGV0IGFsbENsYXNzZXMgPSBnZXRDbGFzc2VzKHN0YXRlcykuY29uY2F0KHNwbGl0Q2xhc3NlcyhhY3RpdmVFcUNsYXNzKSkucmVkdWNlKHVuaXFSLCBbXSk7XG4gICAgICAgICAgICBsZXQgZnV6enlDbGFzc2VzID0gZ2V0Q2xhc3NlcyhzdGF0ZXMuZmlsdGVyKHggPT4gJHN0YXRlLmluY2x1ZGVzKHguc3RhdGUubmFtZSwgeC5wYXJhbXMpKSk7XG4gICAgICAgICAgICBsZXQgZXhhY3RseU1hdGNoZXNBbnkgPSAhIXN0YXRlcy5maWx0ZXIoeCA9PiAkc3RhdGUuaXMoeC5zdGF0ZS5uYW1lLCB4LnBhcmFtcykpLmxlbmd0aDtcbiAgICAgICAgICAgIGxldCBleGFjdENsYXNzZXMgPSBleGFjdGx5TWF0Y2hlc0FueSA/IHNwbGl0Q2xhc3NlcyhhY3RpdmVFcUNsYXNzKSA6IFtdO1xuXG4gICAgICAgICAgICBsZXQgYWRkQ2xhc3NlcyA9IGZ1enp5Q2xhc3Nlcy5jb25jYXQoZXhhY3RDbGFzc2VzKS5yZWR1Y2UodW5pcVIsIFtdKTtcbiAgICAgICAgICAgIGxldCByZW1vdmVDbGFzc2VzID0gYWxsQ2xhc3Nlcy5maWx0ZXIoY2xzID0+ICFpbkFycmF5KGFkZENsYXNzZXMsIGNscykpO1xuXG4gICAgICAgICAgICAkc2NvcGUuJGV2YWxBc3luYygoKSA9PiB7XG4gICAgICAgICAgICAgIGFkZENsYXNzZXMuZm9yRWFjaChjbGFzc05hbWUgPT4gJGVsZW1lbnQuYWRkQ2xhc3MoY2xhc3NOYW1lKSk7XG4gICAgICAgICAgICAgIHJlbW92ZUNsYXNzZXMuZm9yRWFjaChjbGFzc05hbWUgPT4gJGVsZW1lbnQucmVtb3ZlQ2xhc3MoY2xhc3NOYW1lKSk7XG4gICAgICAgICAgICB9KTtcbiAgICAgICAgICB9XG5cbiAgICAgICAgICB1cGRhdGUoKTtcbiAgICAgICAgfV1cbiAgICB9O1xuICB9XTtcblxuLyoqIEBoaWRkZW4gKi9cbmludGVyZmFjZSBEZWYgeyB1aVN0YXRlOiBzdHJpbmc7IGhyZWY6IHN0cmluZzsgdWlTdGF0ZVBhcmFtczogT2JqOyB1aVN0YXRlT3B0czogYW55OyB9XG4vKiogQGhpZGRlbiAqL1xuaW50ZXJmYWNlIFN0YXRlRGF0YSB7IHN0YXRlOiBTdGF0ZURlY2xhcmF0aW9uOyBwYXJhbXM6IFJhd1BhcmFtczsgYWN0aXZlQ2xhc3M6IHN0cmluZzsgfVxuXG5hbmd1bGFyLm1vZHVsZSgndWkucm91dGVyLnN0YXRlJylcbiAgICAuZGlyZWN0aXZlKCd1aVNyZWYnLCB1aVNyZWYpXG4gICAgLmRpcmVjdGl2ZSgndWlTcmVmQWN0aXZlJywgdWlTcmVmQWN0aXZlKVxuICAgIC5kaXJlY3RpdmUoJ3VpU3JlZkFjdGl2ZUVxJywgdWlTcmVmQWN0aXZlKVxuICAgIC5kaXJlY3RpdmUoJ3VpU3RhdGUnLCB1aVN0YXRlKTtcbiJdfQ==