"use strict";
/**
 * # Angular 1 injectable services
 *
 * This is a list of the objects which can be injected using angular's injector.
 *
 * There are three different kind of injectable objects:
 *
 * ## **Provider** objects
 * #### injectable into a `.config()` block during configtime
 *
 * - [[$uiRouterProvider]]: The UI-Router instance
 * - [[$stateProvider]]: State registration
 * - [[$transitionsProvider]]: Transition hooks
 * - [[$urlServiceProvider]]: All URL related public APIs
 *
 * - [[$uiViewScrollProvider]]: Disable ui-router view scrolling
 * - [[$urlRouterProvider]]: (deprecated) Url matching rules
 * - [[$urlMatcherFactoryProvider]]: (deprecated) Url parsing config
 *
 * ## **Service** objects
 * #### injectable globally during runtime
 *
 * - [[$uiRouter]]: The UI-Router instance
 * - [[$trace]]: Enable transition trace/debug
 * - [[$transitions]]: Transition hooks
 * - [[$state]]: Imperative state related APIs
 * - [[$stateRegistry]]: State registration
 * - [[$urlService]]: All URL related public APIs
 * - [[$uiRouterGlobals]]: Global variables
 * - [[$uiViewScroll]]: Scroll an element into view
 *
 * - [[$stateParams]]: (deprecated) Global state param values
 * - [[$urlRouter]]: (deprecated) URL synchronization
 * - [[$urlMatcherFactory]]: (deprecated) URL parsing config
 *
 * ## **Per-Transition** objects
 *
 * - These kind of objects are injectable into:
 *   - Resolves ([[Ng1StateDeclaration.resolve]]),
 *   - Transition Hooks ([[TransitionService.onStart]], etc),
 *   - Routed Controllers ([[Ng1ViewDeclaration.controller]])
 *
 * #### Different instances are injected based on the [[Transition]]
 *
 * - [[$transition$]]: The current Transition object
 * - [[$stateParams]]: State param values for pending Transition (deprecated)
 * - Any resolve data defined using [[Ng1StateDeclaration.resolve]]
 *
 * @ng1api
 * @preferred
 * @module injectables
 */ /** */
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * The current (or pending) State Parameters
 *
 * An injectable global **Service Object** which holds the state parameters for the latest **SUCCESSFUL** transition.
 *
 * The values are not updated until *after* a `Transition` successfully completes.
 *
 * **Also:** an injectable **Per-Transition Object** object which holds the pending state parameters for the pending `Transition` currently running.
 *
 * ### Deprecation warning:
 *
 * The value injected for `$stateParams` is different depending on where it is injected.
 *
 * - When injected into an angular service, the object injected is the global **Service Object** with the parameter values for the latest successful `Transition`.
 * - When injected into transition hooks, resolves, or view controllers, the object is the **Per-Transition Object** with the parameter values for the running `Transition`.
 *
 * Because of these confusing details, this service is deprecated.
 *
 * ### Instead of using the global `$stateParams` service object,
 * inject [[$uiRouterGlobals]] and use [[UIRouterGlobals.params]]
 *
 * ```js
 * MyService.$inject = ['$uiRouterGlobals'];
 * function MyService($uiRouterGlobals) {
 *   return {
 *     paramValues: function () {
 *       return $uiRouterGlobals.params;
 *     }
 *   }
 * }
 * ```
 *
 * ### Instead of using the per-transition `$stateParams` object,
 * inject the current `Transition` (as [[$transition$]]) and use [[Transition.params]]
 *
 * ```js
 * MyController.$inject = ['$transition$'];
 * function MyController($transition$) {
 *   var username = $transition$.params().username;
 *   // .. do something with username
 * }
 * ```
 *
 * ---
 *
 * This object can be injected into other services.
 *
 * #### Deprecated Example:
 * ```js
 * SomeService.$inject = ['$http', '$stateParams'];
 * function SomeService($http, $stateParams) {
 *   return {
 *     getUser: function() {
 *       return $http.get('/api/users/' + $stateParams.username);
 *     }
 *   }
 * };
 * angular.service('SomeService', SomeService);
 * ```
 * @deprecated
 */
var $stateParams;
/**
 * Global UI-Router variables
 *
 * The router global state as a **Service Object** (injectable during runtime).
 *
 * This object contains globals such as the current state and current parameter values.
 */
var $uiRouterGlobals;
/**
 * The UI-Router instance
 *
 * The [[UIRouter]] singleton (the router instance) as a **Service Object** (injectable during runtime).
 *
 * This object is the UI-Router singleton instance, created by angular dependency injection during application bootstrap.
 * It has references to the other UI-Router services
 *
 * #### Note: This object is also exposed as [[$uiRouterProvider]] for injection during angular config time.
 */
var $uiRouter;
/**
 * The UI-Router instance
 *
 * The [[UIRouter]] singleton (the router instance) as a **Provider Object** (injectable during config phase).
 *
 * This object is the UI-Router singleton instance, created by angular dependency injection during application bootstrap.
 * It has references to the other UI-Router services
 *
 * #### Note: This object is also exposed as [[$uiRouter]] for injection during runtime.
 */
var $uiRouterProvider;
/**
 * Transition debug/tracing
 *
 * The [[Trace]] singleton as a **Service Object** (injectable during runtime).
 *
 * Enables or disables Transition tracing which can help to debug issues.
 */
var $trace;
/**
 * The Transition Service
 *
 * The [[TransitionService]] singleton as a **Service Object** (injectable during runtime).
 *
 * This angular service exposes the [[TransitionService]] singleton, which is primarily
 * used to register global transition hooks.
 *
 * #### Note: This object is also exposed as [[$transitionsProvider]] for injection during the config phase.
 */
var $transitions;
/**
 * The Transition Service
 *
 * The [[TransitionService]] singleton as a **Provider Object** (injectable during config phase)
 *
 * This angular service exposes the [[TransitionService]] singleton, which is primarily
 * used to register global transition hooks.
 *
 * #### Note: This object is also exposed as [[$transitions]] for injection during runtime.
 */
var $transitionsProvider;
/**
 * The current [[Transition]] object
 *
 * The current [[Transition]] object as a **Per-Transition Object** (injectable into Resolve, Hooks, Controllers)
 *
 * This object returns information about the current transition, including:
 *
 * - To/from states
 * - To/from parameters
 * - Transition options
 * - States being entered, exited, and retained
 * - Resolve data
 * - A Promise for the transition
 * - Any transition failure information
 * - An injector for both Service and Per-Transition Objects
 */
var $transition$;
/**
 * The State Service
 *
 * The [[StateService]] singleton as a **Service Object** (injectable during runtime).
 *
 * This service used to manage and query information on registered states.
 * It exposes state related APIs including:
 *
 * - Start a [[Transition]]
 * - Imperatively lazy load states
 * - Check if a state is currently active
 * - Look up states by name
 * - Build URLs for a state+parameters
 * - Configure the global Transition error handler
 *
 * This angular service exposes the [[StateService]] singleton.
 */
var $state;
/**
 * The State Registry
 *
 * The [[StateRegistry]] singleton as a **Service Object** (injectable during runtime).
 *
 * This service is used to register/deregister states.
 * It has state registration related APIs including:
 *
 * - Register/deregister states
 * - Listen for state registration/deregistration
 * - Get states by name
 * - Add state decorators (to customize the state creation process)
 *
 * #### Note: This object is also exposed as [[$stateRegistryProvider]] for injection during the config phase.
 */
var $stateRegistry;
/**
 * The State Registry
 *
 * The [[StateRegistry]] singleton as a **Provider Object** (injectable during config time).
 *
 * This service is used to register/deregister states.
 * It has state registration related APIs including:
 *
 * - Register/deregister states
 * - Listen for state registration/deregistration
 * - Get states by name
 * - Add state decorators (to customize the state creation process)
 *
 * #### Note: This object is also exposed as [[$stateRegistry]] for injection during runtime.
 */
var $stateRegistryProvider;
/**
 * The View Scroll provider
 *
 * The [[UIViewScrollProvider]] as a **Provider Object** (injectable during config time).
 *
 * This angular service exposes the [[UIViewScrollProvider]] singleton and is
 * used to disable UI-Router's scroll behavior.
 */
var $uiViewScrollProvider;
/**
 * The View Scroll function
 *
 * The View Scroll function as a **Service Object** (injectable during runtime).
 *
 * This is a function that scrolls an element into view.
 * The element is scrolled after a `$timeout` so the DOM has time to refresh.
 *
 * If you prefer to rely on `$anchorScroll` to scroll the view to the anchor,
 * this can be enabled by calling [[UIViewScrollProvider.useAnchorScroll]].
 *
 * Note: this function is used by the [[directives.uiView]] when the `autoscroll` expression evaluates to true.
 */
var $uiViewScroll;
/**
 * The StateProvider
 *
 * An angular1-only [[StateProvider]] as a **Provider Object** (injectable during config time).
 *
 * This angular service exposes the [[StateProvider]] singleton.
 *
 * The `StateProvider` is primarily used to register states or add custom state decorators.
 *
 * ##### Note: This provider is a ng1 vestige.
 * It is a passthrough to [[$stateRegistry]] and [[$state]].
 */
var $stateProvider;
/**
 * The URL Service Provider
 *
 * The [[UrlService]] singleton as a **Provider Object** (injectable during the angular config phase).
 *
 * A service used to configure and interact with the URL.
 * It has URL related APIs including:
 *
 * - register custom Parameter types `UrlService.config.type` ([[UrlConfigApi.type]])
 * - add URL rules: `UrlService.rules.when` ([[UrlRulesApi.when]])
 * - configure behavior when no url matches: `UrlService.rules.otherwise` ([[UrlRulesApi.otherwise]])
 * - delay initial URL synchronization [[UrlService.deferIntercept]].
 * - get or set the current url: [[UrlService.url]]
 *
 * ##### Note: This service can also be injected during runtime as [[$urlService]].
 */
var $urlServiceProvider;
/**
 * The URL Service
 *
 * The [[UrlService]] singleton as a **Service Object** (injectable during runtime).
 *
 * Note: This service can also be injected during the config phase as [[$urlServiceProvider]].
 *
 * Used to configure the URL.
 * It has URL related APIs including:
 *
 * - register custom Parameter types `UrlService.config.type` ([[UrlConfigApi.type]])
 * - add URL rules: `UrlService.rules.when` ([[UrlRulesApi.when]])
 * - configure behavior when no url matches: `UrlService.rules.otherwise` ([[UrlRulesApi.otherwise]])
 * - delay initial URL synchronization [[UrlService.deferIntercept]].
 * - get or set the current url: [[UrlService.url]]
 *
 * ##### Note: This service can also be injected during the config phase as [[$urlServiceProvider]].
 */
var $urlService;
/**
 * The URL Router Provider
 *
 * ### Deprecation warning: This object is now considered internal. Use [[$urlServiceProvider]] instead.
 *
 * The [[UrlRouter]] singleton as a **Provider Object** (injectable during config time).
 *
 * #### Note: This object is also exposed as [[$urlRouter]] for injection during runtime.
 *
 * @deprecated
 */
var $urlRouterProvider;
/**
 * The Url Router
 *
 * ### Deprecation warning: This object is now considered internal. Use [[$urlService]] instead.
 *
 * The [[UrlRouter]] singleton as a **Service Object** (injectable during runtime).
 *
 * #### Note: This object is also exposed as [[$urlRouterProvider]] for injection during angular config time.
 *
 * @deprecated
 */
var $urlRouter;
/**
 * The URL Matcher Factory
 *
 * ### Deprecation warning: This object is now considered internal. Use [[$urlService]] instead.
 *
 * The [[UrlMatcherFactory]] singleton as a **Service Object** (injectable during runtime).
 *
 * This service is used to set url mapping options, define custom parameter types, and create [[UrlMatcher]] objects.
 *
 * #### Note: This object is also exposed as [[$urlMatcherFactoryProvider]] for injection during angular config time.
 *
 * @deprecated
 */
var $urlMatcherFactory;
/**
 * The URL Matcher Factory
 *
 * ### Deprecation warning: This object is now considered internal. Use [[$urlService]] instead.
 *
 * The [[UrlMatcherFactory]] singleton as a **Provider Object** (injectable during config time).
 *
 * This service is used to set url mapping options, define custom parameter types, and create [[UrlMatcher]] objects.
 *
 * #### Note: This object is also exposed as [[$urlMatcherFactory]] for injection during runtime.
 *
 * @deprecated
 */
var $urlMatcherFactoryProvider;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5qZWN0YWJsZXMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvaW5qZWN0YWJsZXMudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0FtREcsQ0FBQyxNQUFNOztBQVVWOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0E0REc7QUFDSCxJQUFJLFlBQXlCLENBQUM7QUFFOUI7Ozs7OztHQU1HO0FBQ0gsSUFBSSxnQkFBaUMsQ0FBQztBQUV0Qzs7Ozs7Ozs7O0dBU0c7QUFDSCxJQUFJLFNBQW1CLENBQUU7QUFFekI7Ozs7Ozs7OztHQVNHO0FBQ0gsSUFBSSxpQkFBMkIsQ0FBQztBQUVoQzs7Ozs7O0dBTUc7QUFDSCxJQUFJLE1BQWEsQ0FBQztBQUVsQjs7Ozs7Ozs7O0dBU0c7QUFDSCxJQUFJLFlBQStCLENBQUM7QUFFcEM7Ozs7Ozs7OztHQVNHO0FBQ0gsSUFBSSxvQkFBdUMsQ0FBQztBQUU1Qzs7Ozs7Ozs7Ozs7Ozs7O0dBZUc7QUFDSCxJQUFJLFlBQXdCLENBQUM7QUFFN0I7Ozs7Ozs7Ozs7Ozs7Ozs7R0FnQkc7QUFDSCxJQUFJLE1BQW9CLENBQUM7QUFFekI7Ozs7Ozs7Ozs7Ozs7O0dBY0c7QUFDSCxJQUFJLGNBQTZCLENBQUM7QUFFbEM7Ozs7Ozs7Ozs7Ozs7O0dBY0c7QUFDSCxJQUFJLHNCQUFxQyxDQUFDO0FBRTFDOzs7Ozs7O0dBT0c7QUFDSCxJQUFJLHFCQUEyQyxDQUFDO0FBRWhEOzs7Ozs7Ozs7Ozs7R0FZRztBQUNILElBQUksYUFBeUMsQ0FBQztBQUU5Qzs7Ozs7Ozs7Ozs7R0FXRztBQUNILElBQUksY0FBNkIsQ0FBQztBQUVsQzs7Ozs7Ozs7Ozs7Ozs7O0dBZUc7QUFDSCxJQUFJLG1CQUErQixDQUFDO0FBRXBDOzs7Ozs7Ozs7Ozs7Ozs7OztHQWlCRztBQUNILElBQUksV0FBdUIsQ0FBQztBQUU1Qjs7Ozs7Ozs7OztHQVVHO0FBQ0gsSUFBSSxrQkFBcUMsQ0FBQztBQUUxQzs7Ozs7Ozs7OztHQVVHO0FBQ0gsSUFBSSxVQUFxQixDQUFDO0FBRTFCOzs7Ozs7Ozs7Ozs7R0FZRztBQUNILElBQUksa0JBQXFDLENBQUM7QUFFMUM7Ozs7Ozs7Ozs7OztHQVlHO0FBQ0gsSUFBSSwwQkFBNkMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIyBBbmd1bGFyIDEgaW5qZWN0YWJsZSBzZXJ2aWNlc1xuICpcbiAqIFRoaXMgaXMgYSBsaXN0IG9mIHRoZSBvYmplY3RzIHdoaWNoIGNhbiBiZSBpbmplY3RlZCB1c2luZyBhbmd1bGFyJ3MgaW5qZWN0b3IuXG4gKlxuICogVGhlcmUgYXJlIHRocmVlIGRpZmZlcmVudCBraW5kIG9mIGluamVjdGFibGUgb2JqZWN0czpcbiAqXG4gKiAjIyAqKlByb3ZpZGVyKiogb2JqZWN0c1xuICogIyMjIyBpbmplY3RhYmxlIGludG8gYSBgLmNvbmZpZygpYCBibG9jayBkdXJpbmcgY29uZmlndGltZVxuICpcbiAqIC0gW1skdWlSb3V0ZXJQcm92aWRlcl1dOiBUaGUgVUktUm91dGVyIGluc3RhbmNlXG4gKiAtIFtbJHN0YXRlUHJvdmlkZXJdXTogU3RhdGUgcmVnaXN0cmF0aW9uXG4gKiAtIFtbJHRyYW5zaXRpb25zUHJvdmlkZXJdXTogVHJhbnNpdGlvbiBob29rc1xuICogLSBbWyR1cmxTZXJ2aWNlUHJvdmlkZXJdXTogQWxsIFVSTCByZWxhdGVkIHB1YmxpYyBBUElzXG4gKlxuICogLSBbWyR1aVZpZXdTY3JvbGxQcm92aWRlcl1dOiBEaXNhYmxlIHVpLXJvdXRlciB2aWV3IHNjcm9sbGluZ1xuICogLSBbWyR1cmxSb3V0ZXJQcm92aWRlcl1dOiAoZGVwcmVjYXRlZCkgVXJsIG1hdGNoaW5nIHJ1bGVzXG4gKiAtIFtbJHVybE1hdGNoZXJGYWN0b3J5UHJvdmlkZXJdXTogKGRlcHJlY2F0ZWQpIFVybCBwYXJzaW5nIGNvbmZpZ1xuICpcbiAqICMjICoqU2VydmljZSoqIG9iamVjdHNcbiAqICMjIyMgaW5qZWN0YWJsZSBnbG9iYWxseSBkdXJpbmcgcnVudGltZVxuICpcbiAqIC0gW1skdWlSb3V0ZXJdXTogVGhlIFVJLVJvdXRlciBpbnN0YW5jZVxuICogLSBbWyR0cmFjZV1dOiBFbmFibGUgdHJhbnNpdGlvbiB0cmFjZS9kZWJ1Z1xuICogLSBbWyR0cmFuc2l0aW9uc11dOiBUcmFuc2l0aW9uIGhvb2tzXG4gKiAtIFtbJHN0YXRlXV06IEltcGVyYXRpdmUgc3RhdGUgcmVsYXRlZCBBUElzXG4gKiAtIFtbJHN0YXRlUmVnaXN0cnldXTogU3RhdGUgcmVnaXN0cmF0aW9uXG4gKiAtIFtbJHVybFNlcnZpY2VdXTogQWxsIFVSTCByZWxhdGVkIHB1YmxpYyBBUElzXG4gKiAtIFtbJHVpUm91dGVyR2xvYmFsc11dOiBHbG9iYWwgdmFyaWFibGVzXG4gKiAtIFtbJHVpVmlld1Njcm9sbF1dOiBTY3JvbGwgYW4gZWxlbWVudCBpbnRvIHZpZXdcbiAqXG4gKiAtIFtbJHN0YXRlUGFyYW1zXV06IChkZXByZWNhdGVkKSBHbG9iYWwgc3RhdGUgcGFyYW0gdmFsdWVzXG4gKiAtIFtbJHVybFJvdXRlcl1dOiAoZGVwcmVjYXRlZCkgVVJMIHN5bmNocm9uaXphdGlvblxuICogLSBbWyR1cmxNYXRjaGVyRmFjdG9yeV1dOiAoZGVwcmVjYXRlZCkgVVJMIHBhcnNpbmcgY29uZmlnXG4gKlxuICogIyMgKipQZXItVHJhbnNpdGlvbioqIG9iamVjdHNcbiAqXG4gKiAtIFRoZXNlIGtpbmQgb2Ygb2JqZWN0cyBhcmUgaW5qZWN0YWJsZSBpbnRvOlxuICogICAtIFJlc29sdmVzIChbW05nMVN0YXRlRGVjbGFyYXRpb24ucmVzb2x2ZV1dKSxcbiAqICAgLSBUcmFuc2l0aW9uIEhvb2tzIChbW1RyYW5zaXRpb25TZXJ2aWNlLm9uU3RhcnRdXSwgZXRjKSxcbiAqICAgLSBSb3V0ZWQgQ29udHJvbGxlcnMgKFtbTmcxVmlld0RlY2xhcmF0aW9uLmNvbnRyb2xsZXJdXSlcbiAqXG4gKiAjIyMjIERpZmZlcmVudCBpbnN0YW5jZXMgYXJlIGluamVjdGVkIGJhc2VkIG9uIHRoZSBbW1RyYW5zaXRpb25dXVxuICpcbiAqIC0gW1skdHJhbnNpdGlvbiRdXTogVGhlIGN1cnJlbnQgVHJhbnNpdGlvbiBvYmplY3RcbiAqIC0gW1skc3RhdGVQYXJhbXNdXTogU3RhdGUgcGFyYW0gdmFsdWVzIGZvciBwZW5kaW5nIFRyYW5zaXRpb24gKGRlcHJlY2F0ZWQpXG4gKiAtIEFueSByZXNvbHZlIGRhdGEgZGVmaW5lZCB1c2luZyBbW05nMVN0YXRlRGVjbGFyYXRpb24ucmVzb2x2ZV1dXG4gKlxuICogQG5nMWFwaVxuICogQHByZWZlcnJlZFxuICogQG1vZHVsZSBpbmplY3RhYmxlc1xuICovIC8qKiAqL1xuXG5pbXBvcnQgeyBTdGF0ZVByb3ZpZGVyIH0gZnJvbSBcIi4vc3RhdGVQcm92aWRlclwiO1xuaW1wb3J0IHtcbiAgICBTdGF0ZVNlcnZpY2UsIFRyYW5zaXRpb25TZXJ2aWNlLCBUcmFuc2l0aW9uLCBVcmxSb3V0ZXIsIFVybE1hdGNoZXJGYWN0b3J5LFxuICAgIFN0YXRlUGFyYW1zLCBTdGF0ZVJlZ2lzdHJ5LCBVSVJvdXRlckdsb2JhbHMsIFVJUm91dGVyLCBUcmFjZSwgVXJsU2VydmljZVxufSBmcm9tIFwiQHVpcm91dGVyL2NvcmVcIjtcbmltcG9ydCB7IFVJVmlld1Njcm9sbFByb3ZpZGVyIH0gZnJvbSBcIi4vdmlld1Njcm9sbFwiO1xuaW1wb3J0IHsgVXJsUm91dGVyUHJvdmlkZXIgfSBmcm9tIFwiLi91cmxSb3V0ZXJQcm92aWRlclwiO1xuXG4vKipcbiAqIFRoZSBjdXJyZW50IChvciBwZW5kaW5nKSBTdGF0ZSBQYXJhbWV0ZXJzXG4gKlxuICogQW4gaW5qZWN0YWJsZSBnbG9iYWwgKipTZXJ2aWNlIE9iamVjdCoqIHdoaWNoIGhvbGRzIHRoZSBzdGF0ZSBwYXJhbWV0ZXJzIGZvciB0aGUgbGF0ZXN0ICoqU1VDQ0VTU0ZVTCoqIHRyYW5zaXRpb24uXG4gKlxuICogVGhlIHZhbHVlcyBhcmUgbm90IHVwZGF0ZWQgdW50aWwgKmFmdGVyKiBhIGBUcmFuc2l0aW9uYCBzdWNjZXNzZnVsbHkgY29tcGxldGVzLlxuICpcbiAqICoqQWxzbzoqKiBhbiBpbmplY3RhYmxlICoqUGVyLVRyYW5zaXRpb24gT2JqZWN0Kiogb2JqZWN0IHdoaWNoIGhvbGRzIHRoZSBwZW5kaW5nIHN0YXRlIHBhcmFtZXRlcnMgZm9yIHRoZSBwZW5kaW5nIGBUcmFuc2l0aW9uYCBjdXJyZW50bHkgcnVubmluZy5cbiAqXG4gKiAjIyMgRGVwcmVjYXRpb24gd2FybmluZzpcbiAqXG4gKiBUaGUgdmFsdWUgaW5qZWN0ZWQgZm9yIGAkc3RhdGVQYXJhbXNgIGlzIGRpZmZlcmVudCBkZXBlbmRpbmcgb24gd2hlcmUgaXQgaXMgaW5qZWN0ZWQuXG4gKlxuICogLSBXaGVuIGluamVjdGVkIGludG8gYW4gYW5ndWxhciBzZXJ2aWNlLCB0aGUgb2JqZWN0IGluamVjdGVkIGlzIHRoZSBnbG9iYWwgKipTZXJ2aWNlIE9iamVjdCoqIHdpdGggdGhlIHBhcmFtZXRlciB2YWx1ZXMgZm9yIHRoZSBsYXRlc3Qgc3VjY2Vzc2Z1bCBgVHJhbnNpdGlvbmAuXG4gKiAtIFdoZW4gaW5qZWN0ZWQgaW50byB0cmFuc2l0aW9uIGhvb2tzLCByZXNvbHZlcywgb3IgdmlldyBjb250cm9sbGVycywgdGhlIG9iamVjdCBpcyB0aGUgKipQZXItVHJhbnNpdGlvbiBPYmplY3QqKiB3aXRoIHRoZSBwYXJhbWV0ZXIgdmFsdWVzIGZvciB0aGUgcnVubmluZyBgVHJhbnNpdGlvbmAuXG4gKlxuICogQmVjYXVzZSBvZiB0aGVzZSBjb25mdXNpbmcgZGV0YWlscywgdGhpcyBzZXJ2aWNlIGlzIGRlcHJlY2F0ZWQuXG4gKlxuICogIyMjIEluc3RlYWQgb2YgdXNpbmcgdGhlIGdsb2JhbCBgJHN0YXRlUGFyYW1zYCBzZXJ2aWNlIG9iamVjdCxcbiAqIGluamVjdCBbWyR1aVJvdXRlckdsb2JhbHNdXSBhbmQgdXNlIFtbVUlSb3V0ZXJHbG9iYWxzLnBhcmFtc11dXG4gKlxuICogYGBganNcbiAqIE15U2VydmljZS4kaW5qZWN0ID0gWyckdWlSb3V0ZXJHbG9iYWxzJ107XG4gKiBmdW5jdGlvbiBNeVNlcnZpY2UoJHVpUm91dGVyR2xvYmFscykge1xuICogICByZXR1cm4ge1xuICogICAgIHBhcmFtVmFsdWVzOiBmdW5jdGlvbiAoKSB7XG4gKiAgICAgICByZXR1cm4gJHVpUm91dGVyR2xvYmFscy5wYXJhbXM7XG4gKiAgICAgfVxuICogICB9XG4gKiB9XG4gKiBgYGBcbiAqXG4gKiAjIyMgSW5zdGVhZCBvZiB1c2luZyB0aGUgcGVyLXRyYW5zaXRpb24gYCRzdGF0ZVBhcmFtc2Agb2JqZWN0LFxuICogaW5qZWN0IHRoZSBjdXJyZW50IGBUcmFuc2l0aW9uYCAoYXMgW1skdHJhbnNpdGlvbiRdXSkgYW5kIHVzZSBbW1RyYW5zaXRpb24ucGFyYW1zXV1cbiAqXG4gKiBgYGBqc1xuICogTXlDb250cm9sbGVyLiRpbmplY3QgPSBbJyR0cmFuc2l0aW9uJCddO1xuICogZnVuY3Rpb24gTXlDb250cm9sbGVyKCR0cmFuc2l0aW9uJCkge1xuICogICB2YXIgdXNlcm5hbWUgPSAkdHJhbnNpdGlvbiQucGFyYW1zKCkudXNlcm5hbWU7XG4gKiAgIC8vIC4uIGRvIHNvbWV0aGluZyB3aXRoIHVzZXJuYW1lXG4gKiB9XG4gKiBgYGBcbiAqXG4gKiAtLS1cbiAqXG4gKiBUaGlzIG9iamVjdCBjYW4gYmUgaW5qZWN0ZWQgaW50byBvdGhlciBzZXJ2aWNlcy5cbiAqXG4gKiAjIyMjIERlcHJlY2F0ZWQgRXhhbXBsZTpcbiAqIGBgYGpzXG4gKiBTb21lU2VydmljZS4kaW5qZWN0ID0gWyckaHR0cCcsICckc3RhdGVQYXJhbXMnXTtcbiAqIGZ1bmN0aW9uIFNvbWVTZXJ2aWNlKCRodHRwLCAkc3RhdGVQYXJhbXMpIHtcbiAqICAgcmV0dXJuIHtcbiAqICAgICBnZXRVc2VyOiBmdW5jdGlvbigpIHtcbiAqICAgICAgIHJldHVybiAkaHR0cC5nZXQoJy9hcGkvdXNlcnMvJyArICRzdGF0ZVBhcmFtcy51c2VybmFtZSk7XG4gKiAgICAgfVxuICogICB9XG4gKiB9O1xuICogYW5ndWxhci5zZXJ2aWNlKCdTb21lU2VydmljZScsIFNvbWVTZXJ2aWNlKTtcbiAqIGBgYFxuICogQGRlcHJlY2F0ZWRcbiAqL1xudmFyICRzdGF0ZVBhcmFtczogU3RhdGVQYXJhbXM7XG5cbi8qKlxuICogR2xvYmFsIFVJLVJvdXRlciB2YXJpYWJsZXNcbiAqXG4gKiBUaGUgcm91dGVyIGdsb2JhbCBzdGF0ZSBhcyBhICoqU2VydmljZSBPYmplY3QqKiAoaW5qZWN0YWJsZSBkdXJpbmcgcnVudGltZSkuXG4gKlxuICogVGhpcyBvYmplY3QgY29udGFpbnMgZ2xvYmFscyBzdWNoIGFzIHRoZSBjdXJyZW50IHN0YXRlIGFuZCBjdXJyZW50IHBhcmFtZXRlciB2YWx1ZXMuXG4gKi9cbnZhciAkdWlSb3V0ZXJHbG9iYWxzOiBVSVJvdXRlckdsb2JhbHM7XG5cbi8qKlxuICogVGhlIFVJLVJvdXRlciBpbnN0YW5jZVxuICpcbiAqIFRoZSBbW1VJUm91dGVyXV0gc2luZ2xldG9uICh0aGUgcm91dGVyIGluc3RhbmNlKSBhcyBhICoqU2VydmljZSBPYmplY3QqKiAoaW5qZWN0YWJsZSBkdXJpbmcgcnVudGltZSkuXG4gKlxuICogVGhpcyBvYmplY3QgaXMgdGhlIFVJLVJvdXRlciBzaW5nbGV0b24gaW5zdGFuY2UsIGNyZWF0ZWQgYnkgYW5ndWxhciBkZXBlbmRlbmN5IGluamVjdGlvbiBkdXJpbmcgYXBwbGljYXRpb24gYm9vdHN0cmFwLlxuICogSXQgaGFzIHJlZmVyZW5jZXMgdG8gdGhlIG90aGVyIFVJLVJvdXRlciBzZXJ2aWNlc1xuICpcbiAqICMjIyMgTm90ZTogVGhpcyBvYmplY3QgaXMgYWxzbyBleHBvc2VkIGFzIFtbJHVpUm91dGVyUHJvdmlkZXJdXSBmb3IgaW5qZWN0aW9uIGR1cmluZyBhbmd1bGFyIGNvbmZpZyB0aW1lLlxuICovXG5sZXQgJHVpUm91dGVyOiBVSVJvdXRlciA7XG5cbi8qKlxuICogVGhlIFVJLVJvdXRlciBpbnN0YW5jZVxuICpcbiAqIFRoZSBbW1VJUm91dGVyXV0gc2luZ2xldG9uICh0aGUgcm91dGVyIGluc3RhbmNlKSBhcyBhICoqUHJvdmlkZXIgT2JqZWN0KiogKGluamVjdGFibGUgZHVyaW5nIGNvbmZpZyBwaGFzZSkuXG4gKlxuICogVGhpcyBvYmplY3QgaXMgdGhlIFVJLVJvdXRlciBzaW5nbGV0b24gaW5zdGFuY2UsIGNyZWF0ZWQgYnkgYW5ndWxhciBkZXBlbmRlbmN5IGluamVjdGlvbiBkdXJpbmcgYXBwbGljYXRpb24gYm9vdHN0cmFwLlxuICogSXQgaGFzIHJlZmVyZW5jZXMgdG8gdGhlIG90aGVyIFVJLVJvdXRlciBzZXJ2aWNlc1xuICpcbiAqICMjIyMgTm90ZTogVGhpcyBvYmplY3QgaXMgYWxzbyBleHBvc2VkIGFzIFtbJHVpUm91dGVyXV0gZm9yIGluamVjdGlvbiBkdXJpbmcgcnVudGltZS5cbiAqL1xudmFyICR1aVJvdXRlclByb3ZpZGVyOiBVSVJvdXRlcjtcblxuLyoqXG4gKiBUcmFuc2l0aW9uIGRlYnVnL3RyYWNpbmdcbiAqXG4gKiBUaGUgW1tUcmFjZV1dIHNpbmdsZXRvbiBhcyBhICoqU2VydmljZSBPYmplY3QqKiAoaW5qZWN0YWJsZSBkdXJpbmcgcnVudGltZSkuXG4gKlxuICogRW5hYmxlcyBvciBkaXNhYmxlcyBUcmFuc2l0aW9uIHRyYWNpbmcgd2hpY2ggY2FuIGhlbHAgdG8gZGVidWcgaXNzdWVzLlxuICovXG52YXIgJHRyYWNlOiBUcmFjZTtcblxuLyoqXG4gKiBUaGUgVHJhbnNpdGlvbiBTZXJ2aWNlXG4gKlxuICogVGhlIFtbVHJhbnNpdGlvblNlcnZpY2VdXSBzaW5nbGV0b24gYXMgYSAqKlNlcnZpY2UgT2JqZWN0KiogKGluamVjdGFibGUgZHVyaW5nIHJ1bnRpbWUpLlxuICpcbiAqIFRoaXMgYW5ndWxhciBzZXJ2aWNlIGV4cG9zZXMgdGhlIFtbVHJhbnNpdGlvblNlcnZpY2VdXSBzaW5nbGV0b24sIHdoaWNoIGlzIHByaW1hcmlseVxuICogdXNlZCB0byByZWdpc3RlciBnbG9iYWwgdHJhbnNpdGlvbiBob29rcy5cbiAqXG4gKiAjIyMjIE5vdGU6IFRoaXMgb2JqZWN0IGlzIGFsc28gZXhwb3NlZCBhcyBbWyR0cmFuc2l0aW9uc1Byb3ZpZGVyXV0gZm9yIGluamVjdGlvbiBkdXJpbmcgdGhlIGNvbmZpZyBwaGFzZS5cbiAqL1xudmFyICR0cmFuc2l0aW9uczogVHJhbnNpdGlvblNlcnZpY2U7XG5cbi8qKlxuICogVGhlIFRyYW5zaXRpb24gU2VydmljZVxuICpcbiAqIFRoZSBbW1RyYW5zaXRpb25TZXJ2aWNlXV0gc2luZ2xldG9uIGFzIGEgKipQcm92aWRlciBPYmplY3QqKiAoaW5qZWN0YWJsZSBkdXJpbmcgY29uZmlnIHBoYXNlKVxuICpcbiAqIFRoaXMgYW5ndWxhciBzZXJ2aWNlIGV4cG9zZXMgdGhlIFtbVHJhbnNpdGlvblNlcnZpY2VdXSBzaW5nbGV0b24sIHdoaWNoIGlzIHByaW1hcmlseVxuICogdXNlZCB0byByZWdpc3RlciBnbG9iYWwgdHJhbnNpdGlvbiBob29rcy5cbiAqXG4gKiAjIyMjIE5vdGU6IFRoaXMgb2JqZWN0IGlzIGFsc28gZXhwb3NlZCBhcyBbWyR0cmFuc2l0aW9uc11dIGZvciBpbmplY3Rpb24gZHVyaW5nIHJ1bnRpbWUuXG4gKi9cbnZhciAkdHJhbnNpdGlvbnNQcm92aWRlcjogVHJhbnNpdGlvblNlcnZpY2U7XG5cbi8qKlxuICogVGhlIGN1cnJlbnQgW1tUcmFuc2l0aW9uXV0gb2JqZWN0XG4gKlxuICogVGhlIGN1cnJlbnQgW1tUcmFuc2l0aW9uXV0gb2JqZWN0IGFzIGEgKipQZXItVHJhbnNpdGlvbiBPYmplY3QqKiAoaW5qZWN0YWJsZSBpbnRvIFJlc29sdmUsIEhvb2tzLCBDb250cm9sbGVycylcbiAqXG4gKiBUaGlzIG9iamVjdCByZXR1cm5zIGluZm9ybWF0aW9uIGFib3V0IHRoZSBjdXJyZW50IHRyYW5zaXRpb24sIGluY2x1ZGluZzpcbiAqXG4gKiAtIFRvL2Zyb20gc3RhdGVzXG4gKiAtIFRvL2Zyb20gcGFyYW1ldGVyc1xuICogLSBUcmFuc2l0aW9uIG9wdGlvbnNcbiAqIC0gU3RhdGVzIGJlaW5nIGVudGVyZWQsIGV4aXRlZCwgYW5kIHJldGFpbmVkXG4gKiAtIFJlc29sdmUgZGF0YVxuICogLSBBIFByb21pc2UgZm9yIHRoZSB0cmFuc2l0aW9uXG4gKiAtIEFueSB0cmFuc2l0aW9uIGZhaWx1cmUgaW5mb3JtYXRpb25cbiAqIC0gQW4gaW5qZWN0b3IgZm9yIGJvdGggU2VydmljZSBhbmQgUGVyLVRyYW5zaXRpb24gT2JqZWN0c1xuICovXG52YXIgJHRyYW5zaXRpb24kOiBUcmFuc2l0aW9uO1xuXG4vKipcbiAqIFRoZSBTdGF0ZSBTZXJ2aWNlXG4gKlxuICogVGhlIFtbU3RhdGVTZXJ2aWNlXV0gc2luZ2xldG9uIGFzIGEgKipTZXJ2aWNlIE9iamVjdCoqIChpbmplY3RhYmxlIGR1cmluZyBydW50aW1lKS5cbiAqXG4gKiBUaGlzIHNlcnZpY2UgdXNlZCB0byBtYW5hZ2UgYW5kIHF1ZXJ5IGluZm9ybWF0aW9uIG9uIHJlZ2lzdGVyZWQgc3RhdGVzLlxuICogSXQgZXhwb3NlcyBzdGF0ZSByZWxhdGVkIEFQSXMgaW5jbHVkaW5nOlxuICpcbiAqIC0gU3RhcnQgYSBbW1RyYW5zaXRpb25dXVxuICogLSBJbXBlcmF0aXZlbHkgbGF6eSBsb2FkIHN0YXRlc1xuICogLSBDaGVjayBpZiBhIHN0YXRlIGlzIGN1cnJlbnRseSBhY3RpdmVcbiAqIC0gTG9vayB1cCBzdGF0ZXMgYnkgbmFtZVxuICogLSBCdWlsZCBVUkxzIGZvciBhIHN0YXRlK3BhcmFtZXRlcnNcbiAqIC0gQ29uZmlndXJlIHRoZSBnbG9iYWwgVHJhbnNpdGlvbiBlcnJvciBoYW5kbGVyXG4gKlxuICogVGhpcyBhbmd1bGFyIHNlcnZpY2UgZXhwb3NlcyB0aGUgW1tTdGF0ZVNlcnZpY2VdXSBzaW5nbGV0b24uXG4gKi9cbnZhciAkc3RhdGU6IFN0YXRlU2VydmljZTtcblxuLyoqXG4gKiBUaGUgU3RhdGUgUmVnaXN0cnlcbiAqXG4gKiBUaGUgW1tTdGF0ZVJlZ2lzdHJ5XV0gc2luZ2xldG9uIGFzIGEgKipTZXJ2aWNlIE9iamVjdCoqIChpbmplY3RhYmxlIGR1cmluZyBydW50aW1lKS5cbiAqXG4gKiBUaGlzIHNlcnZpY2UgaXMgdXNlZCB0byByZWdpc3Rlci9kZXJlZ2lzdGVyIHN0YXRlcy5cbiAqIEl0IGhhcyBzdGF0ZSByZWdpc3RyYXRpb24gcmVsYXRlZCBBUElzIGluY2x1ZGluZzpcbiAqXG4gKiAtIFJlZ2lzdGVyL2RlcmVnaXN0ZXIgc3RhdGVzXG4gKiAtIExpc3RlbiBmb3Igc3RhdGUgcmVnaXN0cmF0aW9uL2RlcmVnaXN0cmF0aW9uXG4gKiAtIEdldCBzdGF0ZXMgYnkgbmFtZVxuICogLSBBZGQgc3RhdGUgZGVjb3JhdG9ycyAodG8gY3VzdG9taXplIHRoZSBzdGF0ZSBjcmVhdGlvbiBwcm9jZXNzKVxuICpcbiAqICMjIyMgTm90ZTogVGhpcyBvYmplY3QgaXMgYWxzbyBleHBvc2VkIGFzIFtbJHN0YXRlUmVnaXN0cnlQcm92aWRlcl1dIGZvciBpbmplY3Rpb24gZHVyaW5nIHRoZSBjb25maWcgcGhhc2UuXG4gKi9cbnZhciAkc3RhdGVSZWdpc3RyeTogU3RhdGVSZWdpc3RyeTtcblxuLyoqXG4gKiBUaGUgU3RhdGUgUmVnaXN0cnlcbiAqXG4gKiBUaGUgW1tTdGF0ZVJlZ2lzdHJ5XV0gc2luZ2xldG9uIGFzIGEgKipQcm92aWRlciBPYmplY3QqKiAoaW5qZWN0YWJsZSBkdXJpbmcgY29uZmlnIHRpbWUpLlxuICpcbiAqIFRoaXMgc2VydmljZSBpcyB1c2VkIHRvIHJlZ2lzdGVyL2RlcmVnaXN0ZXIgc3RhdGVzLlxuICogSXQgaGFzIHN0YXRlIHJlZ2lzdHJhdGlvbiByZWxhdGVkIEFQSXMgaW5jbHVkaW5nOlxuICpcbiAqIC0gUmVnaXN0ZXIvZGVyZWdpc3RlciBzdGF0ZXNcbiAqIC0gTGlzdGVuIGZvciBzdGF0ZSByZWdpc3RyYXRpb24vZGVyZWdpc3RyYXRpb25cbiAqIC0gR2V0IHN0YXRlcyBieSBuYW1lXG4gKiAtIEFkZCBzdGF0ZSBkZWNvcmF0b3JzICh0byBjdXN0b21pemUgdGhlIHN0YXRlIGNyZWF0aW9uIHByb2Nlc3MpXG4gKlxuICogIyMjIyBOb3RlOiBUaGlzIG9iamVjdCBpcyBhbHNvIGV4cG9zZWQgYXMgW1skc3RhdGVSZWdpc3RyeV1dIGZvciBpbmplY3Rpb24gZHVyaW5nIHJ1bnRpbWUuXG4gKi9cbnZhciAkc3RhdGVSZWdpc3RyeVByb3ZpZGVyOiBTdGF0ZVJlZ2lzdHJ5O1xuXG4vKipcbiAqIFRoZSBWaWV3IFNjcm9sbCBwcm92aWRlclxuICpcbiAqIFRoZSBbW1VJVmlld1Njcm9sbFByb3ZpZGVyXV0gYXMgYSAqKlByb3ZpZGVyIE9iamVjdCoqIChpbmplY3RhYmxlIGR1cmluZyBjb25maWcgdGltZSkuXG4gKlxuICogVGhpcyBhbmd1bGFyIHNlcnZpY2UgZXhwb3NlcyB0aGUgW1tVSVZpZXdTY3JvbGxQcm92aWRlcl1dIHNpbmdsZXRvbiBhbmQgaXNcbiAqIHVzZWQgdG8gZGlzYWJsZSBVSS1Sb3V0ZXIncyBzY3JvbGwgYmVoYXZpb3IuXG4gKi9cbnZhciAkdWlWaWV3U2Nyb2xsUHJvdmlkZXI6IFVJVmlld1Njcm9sbFByb3ZpZGVyO1xuXG4vKipcbiAqIFRoZSBWaWV3IFNjcm9sbCBmdW5jdGlvblxuICpcbiAqIFRoZSBWaWV3IFNjcm9sbCBmdW5jdGlvbiBhcyBhICoqU2VydmljZSBPYmplY3QqKiAoaW5qZWN0YWJsZSBkdXJpbmcgcnVudGltZSkuXG4gKlxuICogVGhpcyBpcyBhIGZ1bmN0aW9uIHRoYXQgc2Nyb2xscyBhbiBlbGVtZW50IGludG8gdmlldy5cbiAqIFRoZSBlbGVtZW50IGlzIHNjcm9sbGVkIGFmdGVyIGEgYCR0aW1lb3V0YCBzbyB0aGUgRE9NIGhhcyB0aW1lIHRvIHJlZnJlc2guXG4gKlxuICogSWYgeW91IHByZWZlciB0byByZWx5IG9uIGAkYW5jaG9yU2Nyb2xsYCB0byBzY3JvbGwgdGhlIHZpZXcgdG8gdGhlIGFuY2hvcixcbiAqIHRoaXMgY2FuIGJlIGVuYWJsZWQgYnkgY2FsbGluZyBbW1VJVmlld1Njcm9sbFByb3ZpZGVyLnVzZUFuY2hvclNjcm9sbF1dLlxuICpcbiAqIE5vdGU6IHRoaXMgZnVuY3Rpb24gaXMgdXNlZCBieSB0aGUgW1tkaXJlY3RpdmVzLnVpVmlld11dIHdoZW4gdGhlIGBhdXRvc2Nyb2xsYCBleHByZXNzaW9uIGV2YWx1YXRlcyB0byB0cnVlLlxuICovXG52YXIgJHVpVmlld1Njcm9sbDogKCRlbGVtZW50OiBKUXVlcnkpID0+IHZvaWQ7XG5cbi8qKlxuICogVGhlIFN0YXRlUHJvdmlkZXJcbiAqXG4gKiBBbiBhbmd1bGFyMS1vbmx5IFtbU3RhdGVQcm92aWRlcl1dIGFzIGEgKipQcm92aWRlciBPYmplY3QqKiAoaW5qZWN0YWJsZSBkdXJpbmcgY29uZmlnIHRpbWUpLlxuICpcbiAqIFRoaXMgYW5ndWxhciBzZXJ2aWNlIGV4cG9zZXMgdGhlIFtbU3RhdGVQcm92aWRlcl1dIHNpbmdsZXRvbi5cbiAqXG4gKiBUaGUgYFN0YXRlUHJvdmlkZXJgIGlzIHByaW1hcmlseSB1c2VkIHRvIHJlZ2lzdGVyIHN0YXRlcyBvciBhZGQgY3VzdG9tIHN0YXRlIGRlY29yYXRvcnMuXG4gKlxuICogIyMjIyMgTm90ZTogVGhpcyBwcm92aWRlciBpcyBhIG5nMSB2ZXN0aWdlLlxuICogSXQgaXMgYSBwYXNzdGhyb3VnaCB0byBbWyRzdGF0ZVJlZ2lzdHJ5XV0gYW5kIFtbJHN0YXRlXV0uXG4gKi9cbnZhciAkc3RhdGVQcm92aWRlcjogU3RhdGVQcm92aWRlcjtcblxuLyoqXG4gKiBUaGUgVVJMIFNlcnZpY2UgUHJvdmlkZXJcbiAqXG4gKiBUaGUgW1tVcmxTZXJ2aWNlXV0gc2luZ2xldG9uIGFzIGEgKipQcm92aWRlciBPYmplY3QqKiAoaW5qZWN0YWJsZSBkdXJpbmcgdGhlIGFuZ3VsYXIgY29uZmlnIHBoYXNlKS5cbiAqXG4gKiBBIHNlcnZpY2UgdXNlZCB0byBjb25maWd1cmUgYW5kIGludGVyYWN0IHdpdGggdGhlIFVSTC5cbiAqIEl0IGhhcyBVUkwgcmVsYXRlZCBBUElzIGluY2x1ZGluZzpcbiAqXG4gKiAtIHJlZ2lzdGVyIGN1c3RvbSBQYXJhbWV0ZXIgdHlwZXMgYFVybFNlcnZpY2UuY29uZmlnLnR5cGVgIChbW1VybENvbmZpZ0FwaS50eXBlXV0pXG4gKiAtIGFkZCBVUkwgcnVsZXM6IGBVcmxTZXJ2aWNlLnJ1bGVzLndoZW5gIChbW1VybFJ1bGVzQXBpLndoZW5dXSlcbiAqIC0gY29uZmlndXJlIGJlaGF2aW9yIHdoZW4gbm8gdXJsIG1hdGNoZXM6IGBVcmxTZXJ2aWNlLnJ1bGVzLm90aGVyd2lzZWAgKFtbVXJsUnVsZXNBcGkub3RoZXJ3aXNlXV0pXG4gKiAtIGRlbGF5IGluaXRpYWwgVVJMIHN5bmNocm9uaXphdGlvbiBbW1VybFNlcnZpY2UuZGVmZXJJbnRlcmNlcHRdXS5cbiAqIC0gZ2V0IG9yIHNldCB0aGUgY3VycmVudCB1cmw6IFtbVXJsU2VydmljZS51cmxdXVxuICpcbiAqICMjIyMjIE5vdGU6IFRoaXMgc2VydmljZSBjYW4gYWxzbyBiZSBpbmplY3RlZCBkdXJpbmcgcnVudGltZSBhcyBbWyR1cmxTZXJ2aWNlXV0uXG4gKi9cbnZhciAkdXJsU2VydmljZVByb3ZpZGVyOiBVcmxTZXJ2aWNlO1xuXG4vKipcbiAqIFRoZSBVUkwgU2VydmljZVxuICpcbiAqIFRoZSBbW1VybFNlcnZpY2VdXSBzaW5nbGV0b24gYXMgYSAqKlNlcnZpY2UgT2JqZWN0KiogKGluamVjdGFibGUgZHVyaW5nIHJ1bnRpbWUpLlxuICpcbiAqIE5vdGU6IFRoaXMgc2VydmljZSBjYW4gYWxzbyBiZSBpbmplY3RlZCBkdXJpbmcgdGhlIGNvbmZpZyBwaGFzZSBhcyBbWyR1cmxTZXJ2aWNlUHJvdmlkZXJdXS5cbiAqXG4gKiBVc2VkIHRvIGNvbmZpZ3VyZSB0aGUgVVJMLlxuICogSXQgaGFzIFVSTCByZWxhdGVkIEFQSXMgaW5jbHVkaW5nOlxuICpcbiAqIC0gcmVnaXN0ZXIgY3VzdG9tIFBhcmFtZXRlciB0eXBlcyBgVXJsU2VydmljZS5jb25maWcudHlwZWAgKFtbVXJsQ29uZmlnQXBpLnR5cGVdXSlcbiAqIC0gYWRkIFVSTCBydWxlczogYFVybFNlcnZpY2UucnVsZXMud2hlbmAgKFtbVXJsUnVsZXNBcGkud2hlbl1dKVxuICogLSBjb25maWd1cmUgYmVoYXZpb3Igd2hlbiBubyB1cmwgbWF0Y2hlczogYFVybFNlcnZpY2UucnVsZXMub3RoZXJ3aXNlYCAoW1tVcmxSdWxlc0FwaS5vdGhlcndpc2VdXSlcbiAqIC0gZGVsYXkgaW5pdGlhbCBVUkwgc3luY2hyb25pemF0aW9uIFtbVXJsU2VydmljZS5kZWZlckludGVyY2VwdF1dLlxuICogLSBnZXQgb3Igc2V0IHRoZSBjdXJyZW50IHVybDogW1tVcmxTZXJ2aWNlLnVybF1dXG4gKlxuICogIyMjIyMgTm90ZTogVGhpcyBzZXJ2aWNlIGNhbiBhbHNvIGJlIGluamVjdGVkIGR1cmluZyB0aGUgY29uZmlnIHBoYXNlIGFzIFtbJHVybFNlcnZpY2VQcm92aWRlcl1dLlxuICovXG52YXIgJHVybFNlcnZpY2U6IFVybFNlcnZpY2U7XG5cbi8qKlxuICogVGhlIFVSTCBSb3V0ZXIgUHJvdmlkZXJcbiAqXG4gKiAjIyMgRGVwcmVjYXRpb24gd2FybmluZzogVGhpcyBvYmplY3QgaXMgbm93IGNvbnNpZGVyZWQgaW50ZXJuYWwuIFVzZSBbWyR1cmxTZXJ2aWNlUHJvdmlkZXJdXSBpbnN0ZWFkLlxuICpcbiAqIFRoZSBbW1VybFJvdXRlcl1dIHNpbmdsZXRvbiBhcyBhICoqUHJvdmlkZXIgT2JqZWN0KiogKGluamVjdGFibGUgZHVyaW5nIGNvbmZpZyB0aW1lKS5cbiAqXG4gKiAjIyMjIE5vdGU6IFRoaXMgb2JqZWN0IGlzIGFsc28gZXhwb3NlZCBhcyBbWyR1cmxSb3V0ZXJdXSBmb3IgaW5qZWN0aW9uIGR1cmluZyBydW50aW1lLlxuICpcbiAqIEBkZXByZWNhdGVkXG4gKi9cbnZhciAkdXJsUm91dGVyUHJvdmlkZXI6IFVybFJvdXRlclByb3ZpZGVyO1xuXG4vKipcbiAqIFRoZSBVcmwgUm91dGVyXG4gKlxuICogIyMjIERlcHJlY2F0aW9uIHdhcm5pbmc6IFRoaXMgb2JqZWN0IGlzIG5vdyBjb25zaWRlcmVkIGludGVybmFsLiBVc2UgW1skdXJsU2VydmljZV1dIGluc3RlYWQuXG4gKlxuICogVGhlIFtbVXJsUm91dGVyXV0gc2luZ2xldG9uIGFzIGEgKipTZXJ2aWNlIE9iamVjdCoqIChpbmplY3RhYmxlIGR1cmluZyBydW50aW1lKS5cbiAqXG4gKiAjIyMjIE5vdGU6IFRoaXMgb2JqZWN0IGlzIGFsc28gZXhwb3NlZCBhcyBbWyR1cmxSb3V0ZXJQcm92aWRlcl1dIGZvciBpbmplY3Rpb24gZHVyaW5nIGFuZ3VsYXIgY29uZmlnIHRpbWUuXG4gKlxuICogQGRlcHJlY2F0ZWRcbiAqL1xudmFyICR1cmxSb3V0ZXI6IFVybFJvdXRlcjtcblxuLyoqXG4gKiBUaGUgVVJMIE1hdGNoZXIgRmFjdG9yeVxuICpcbiAqICMjIyBEZXByZWNhdGlvbiB3YXJuaW5nOiBUaGlzIG9iamVjdCBpcyBub3cgY29uc2lkZXJlZCBpbnRlcm5hbC4gVXNlIFtbJHVybFNlcnZpY2VdXSBpbnN0ZWFkLlxuICpcbiAqIFRoZSBbW1VybE1hdGNoZXJGYWN0b3J5XV0gc2luZ2xldG9uIGFzIGEgKipTZXJ2aWNlIE9iamVjdCoqIChpbmplY3RhYmxlIGR1cmluZyBydW50aW1lKS5cbiAqXG4gKiBUaGlzIHNlcnZpY2UgaXMgdXNlZCB0byBzZXQgdXJsIG1hcHBpbmcgb3B0aW9ucywgZGVmaW5lIGN1c3RvbSBwYXJhbWV0ZXIgdHlwZXMsIGFuZCBjcmVhdGUgW1tVcmxNYXRjaGVyXV0gb2JqZWN0cy5cbiAqXG4gKiAjIyMjIE5vdGU6IFRoaXMgb2JqZWN0IGlzIGFsc28gZXhwb3NlZCBhcyBbWyR1cmxNYXRjaGVyRmFjdG9yeVByb3ZpZGVyXV0gZm9yIGluamVjdGlvbiBkdXJpbmcgYW5ndWxhciBjb25maWcgdGltZS5cbiAqXG4gKiBAZGVwcmVjYXRlZFxuICovXG52YXIgJHVybE1hdGNoZXJGYWN0b3J5OiBVcmxNYXRjaGVyRmFjdG9yeTtcblxuLyoqXG4gKiBUaGUgVVJMIE1hdGNoZXIgRmFjdG9yeVxuICpcbiAqICMjIyBEZXByZWNhdGlvbiB3YXJuaW5nOiBUaGlzIG9iamVjdCBpcyBub3cgY29uc2lkZXJlZCBpbnRlcm5hbC4gVXNlIFtbJHVybFNlcnZpY2VdXSBpbnN0ZWFkLlxuICpcbiAqIFRoZSBbW1VybE1hdGNoZXJGYWN0b3J5XV0gc2luZ2xldG9uIGFzIGEgKipQcm92aWRlciBPYmplY3QqKiAoaW5qZWN0YWJsZSBkdXJpbmcgY29uZmlnIHRpbWUpLlxuICpcbiAqIFRoaXMgc2VydmljZSBpcyB1c2VkIHRvIHNldCB1cmwgbWFwcGluZyBvcHRpb25zLCBkZWZpbmUgY3VzdG9tIHBhcmFtZXRlciB0eXBlcywgYW5kIGNyZWF0ZSBbW1VybE1hdGNoZXJdXSBvYmplY3RzLlxuICpcbiAqICMjIyMgTm90ZTogVGhpcyBvYmplY3QgaXMgYWxzbyBleHBvc2VkIGFzIFtbJHVybE1hdGNoZXJGYWN0b3J5XV0gZm9yIGluamVjdGlvbiBkdXJpbmcgcnVudGltZS5cbiAqXG4gKiBAZGVwcmVjYXRlZFxuICovXG52YXIgJHVybE1hdGNoZXJGYWN0b3J5UHJvdmlkZXI6IFVybE1hdGNoZXJGYWN0b3J5O1xuXG5cblxuIl19