"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/** @module view */
/** for typedoc */
var angular_1 = require("./angular");
var core_1 = require("@uirouter/core");
/**
 * Service which manages loading of templates from a ViewConfig.
 */
var TemplateFactory = (function () {
    function TemplateFactory() {
        var _this = this;
        /** @hidden */ this._useHttp = angular_1.ng.version.minor < 3;
        /** @hidden */ this.$get = ['$http', '$templateCache', '$injector', function ($http, $templateCache, $injector) {
                _this.$templateRequest = $injector.has && $injector.has('$templateRequest') && $injector.get('$templateRequest');
                _this.$http = $http;
                _this.$templateCache = $templateCache;
                return _this;
            }];
    }
    /** @hidden */
    TemplateFactory.prototype.useHttpService = function (value) {
        this._useHttp = value;
    };
    ;
    /**
     * Creates a template from a configuration object.
     *
     * @param config Configuration object for which to load a template.
     * The following properties are search in the specified order, and the first one
     * that is defined is used to create the template:
     *
     * @param params  Parameters to pass to the template function.
     * @param context The resolve context associated with the template's view
     *
     * @return {string|object}  The template html as a string, or a promise for
     * that string,or `null` if no template is configured.
     */
    TemplateFactory.prototype.fromConfig = function (config, params, context) {
        var defaultTemplate = "<ui-view></ui-view>";
        var asTemplate = function (result) { return core_1.services.$q.when(result).then(function (str) { return ({ template: str }); }); };
        var asComponent = function (result) { return core_1.services.$q.when(result).then(function (str) { return ({ component: str }); }); };
        return (core_1.isDefined(config.template) ? asTemplate(this.fromString(config.template, params)) :
            core_1.isDefined(config.templateUrl) ? asTemplate(this.fromUrl(config.templateUrl, params)) :
                core_1.isDefined(config.templateProvider) ? asTemplate(this.fromProvider(config.templateProvider, params, context)) :
                    core_1.isDefined(config.component) ? asComponent(config.component) :
                        core_1.isDefined(config.componentProvider) ? asComponent(this.fromComponentProvider(config.componentProvider, params, context)) :
                            asTemplate(defaultTemplate));
    };
    ;
    /**
     * Creates a template from a string or a function returning a string.
     *
     * @param template html template as a string or function that returns an html template as a string.
     * @param params Parameters to pass to the template function.
     *
     * @return {string|object} The template html as a string, or a promise for that
     * string.
     */
    TemplateFactory.prototype.fromString = function (template, params) {
        return core_1.isFunction(template) ? template(params) : template;
    };
    ;
    /**
     * Loads a template from the a URL via `$http` and `$templateCache`.
     *
     * @param {string|Function} url url of the template to load, or a function
     * that returns a url.
     * @param {Object} params Parameters to pass to the url function.
     * @return {string|Promise.<string>} The template html as a string, or a promise
     * for that string.
     */
    TemplateFactory.prototype.fromUrl = function (url, params) {
        if (core_1.isFunction(url))
            url = url(params);
        if (url == null)
            return null;
        if (this._useHttp) {
            return this.$http.get(url, { cache: this.$templateCache, headers: { Accept: 'text/html' } })
                .then(function (response) {
                return response.data;
            });
        }
        return this.$templateRequest(url);
    };
    ;
    /**
     * Creates a template by invoking an injectable provider function.
     *
     * @param provider Function to invoke via `locals`
     * @param {Function} injectFn a function used to invoke the template provider
     * @return {string|Promise.<string>} The template html as a string, or a promise
     * for that string.
     */
    TemplateFactory.prototype.fromProvider = function (provider, params, context) {
        var deps = core_1.services.$injector.annotate(provider);
        var providerFn = core_1.isArray(provider) ? core_1.tail(provider) : provider;
        var resolvable = new core_1.Resolvable("", providerFn, deps);
        return resolvable.get(context);
    };
    ;
    /**
     * Creates a component's template by invoking an injectable provider function.
     *
     * @param provider Function to invoke via `locals`
     * @param {Function} injectFn a function used to invoke the template provider
     * @return {string} The template html as a string: "<component-name input1='::$resolve.foo'></component-name>".
     */
    TemplateFactory.prototype.fromComponentProvider = function (provider, params, context) {
        var deps = core_1.services.$injector.annotate(provider);
        var providerFn = core_1.isArray(provider) ? core_1.tail(provider) : provider;
        var resolvable = new core_1.Resolvable("", providerFn, deps);
        return resolvable.get(context);
    };
    ;
    /**
     * Creates a template from a component's name
     *
     * This implements route-to-component.
     * It works by retrieving the component (directive) metadata from the injector.
     * It analyses the component's bindings, then constructs a template that instantiates the component.
     * The template wires input and output bindings to resolves or from the parent component.
     *
     * @param uiView {object} The parent ui-view (for binding outputs to callbacks)
     * @param context The ResolveContext (for binding outputs to callbacks returned from resolves)
     * @param component {string} Component's name in camel case.
     * @param bindings An object defining the component's bindings: {foo: '<'}
     * @return {string} The template as a string: "<component-name input1='::$resolve.foo'></component-name>".
     */
    TemplateFactory.prototype.makeComponentTemplate = function (uiView, context, component, bindings) {
        bindings = bindings || {};
        // Bind once prefix
        var prefix = angular_1.ng.version.minor >= 3 ? "::" : "";
        // Convert to kebob name. Add x- prefix if the string starts with `x-` or `data-`
        var kebob = function (camelCase) {
            var kebobed = core_1.kebobString(camelCase);
            return /^(x|data)-/.exec(kebobed) ? "x-" + kebobed : kebobed;
        };
        var attributeTpl = function (input) {
            var name = input.name, type = input.type;
            var attrName = kebob(name);
            // If the ui-view has an attribute which matches a binding on the routed component
            // then pass that attribute through to the routed component template.
            // Prefer ui-view wired mappings to resolve data, unless the resolve was explicitly bound using `bindings:`
            if (uiView.attr(attrName) && !bindings[name])
                return attrName + "='" + uiView.attr(attrName) + "'";
            var resolveName = bindings[name] || name;
            // Pre-evaluate the expression for "@" bindings by enclosing in {{ }}
            // some-attr="{{ ::$resolve.someResolveName }}"
            if (type === '@')
                return attrName + "='{{" + prefix + "$resolve." + resolveName + "}}'";
            // Wire "&" callbacks to resolves that return a callback function
            // Get the result of the resolve (should be a function) and annotate it to get its arguments.
            // some-attr="$resolve.someResolveResultName(foo, bar)"
            if (type === '&') {
                var res = context.getResolvable(resolveName);
                var fn = res && res.data;
                var args = fn && core_1.services.$injector.annotate(fn) || [];
                // account for array style injection, i.e., ['foo', function(foo) {}]
                var arrayIdxStr = core_1.isArray(fn) ? "[" + (fn.length - 1) + "]" : '';
                return attrName + "='$resolve." + resolveName + arrayIdxStr + "(" + args.join(",") + ")'";
            }
            // some-attr="::$resolve.someResolveName"
            return attrName + "='" + prefix + "$resolve." + resolveName + "'";
        };
        var attrs = getComponentBindings(component).map(attributeTpl).join(" ");
        var kebobName = kebob(component);
        return "<" + kebobName + " " + attrs + "></" + kebobName + ">";
    };
    ;
    return TemplateFactory;
}());
exports.TemplateFactory = TemplateFactory;
// Gets all the directive(s)' inputs ('@', '=', and '<') and outputs ('&')
function getComponentBindings(name) {
    var cmpDefs = core_1.services.$injector.get(name + "Directive"); // could be multiple
    if (!cmpDefs || !cmpDefs.length)
        throw new Error("Unable to find component named '" + name + "'");
    return cmpDefs.map(getBindings).reduce(core_1.unnestR, []);
}
// Given a directive definition, find its object input attributes
// Use different properties, depending on the type of directive (component, bindToController, normal)
var getBindings = function (def) {
    if (core_1.isObject(def.bindToController))
        return scopeBindings(def.bindToController);
    return scopeBindings(def.scope);
};
// for ng 1.2 style, process the scope: { input: "=foo" }
// for ng 1.3 through ng 1.5, process the component's bindToController: { input: "=foo" } object
var scopeBindings = function (bindingsObj) { return Object.keys(bindingsObj || {})
    .map(function (key) { return [key, /^([=<@&])[?]?(.*)/.exec(bindingsObj[key])]; })
    .filter(function (tuple) { return core_1.isDefined(tuple) && core_1.isArray(tuple[1]); })
    .map(function (tuple) { return ({ name: tuple[1][2] || tuple[0], type: tuple[1][1] }); }); };
//# sourceMappingURL=data:application/json;base64,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