import { Param, } from '@uirouter/core';
var DSRPlugin = /** @class */ (function () {
    function DSRPlugin($uiRouter) {
        var _this = this;
        this.name = 'deep-state-redirect';
        this.hookDeregFns = [];
        this.$transitions = $uiRouter.transitionService;
        this.$state = $uiRouter.stateService;
        this.hookDeregFns.push(this.$transitions.onRetain({ retained: function (state) { return !!_this.getDsrProp(state.self); } }, this.recordDeepState.bind(this)));
        this.hookDeregFns.push(this.$transitions.onEnter({ entering: function (state) { return !!_this.getDsrProp(state.self); } }, this.recordDeepState.bind(this)));
        this.hookDeregFns.push(this.$transitions.onBefore({ to: function (state) { return !!_this.getDsrProp(state.self); } }, this.deepStateRedirect.bind(this)));
    }
    DSRPlugin.prototype.dispose = function (router) {
        this.hookDeregFns.forEach(function (fn) { return fn(); });
    };
    /**
     * Resets deep state redirect
     *
     * A deep state is recorded for each DSR state.
     * This function resets recorded deep state redirect(s) to the initial value.
     *
     * If called with no parameters, the redirects for all states are reset.
     *
     * If called with a `state` parameter, the redirect for that state is reset.
     *
     * If called with `state` and `params` parameters, the redirect for that state and set of parameter values is reset.
     *
     * @param state (optional) the redirect for this state will be reset
     * @param params (optional) the redirect for the state and parameters will be reset
     */
    DSRPlugin.prototype.reset = function (state, params) {
        var $state = this.$state;
        if (!state) {
            $state.get().forEach(function (_state) { return delete _state.$$state().$dsr; });
        }
        else if (!params) {
            delete $state.get(state).$$state().$dsr;
        }
        else {
            var $$state = $state.get(state).$$state();
            $$state.$dsr = $$state.$dsr.filter(this.paramsEqual($$state, params, undefined, true));
        }
    };
    /**
     * Returns the recorded redirect
     *
     * Returns the recorded redirect for a given DSR `state` (and optionally `params`).
     *
     * @param state the DSR state
     * @param params (optional) the parameter values
     *
     * @returns the recorded redirect `TargetState`
     */
    DSRPlugin.prototype.getRedirect = function (state, params) {
        return this.getDeepStateRedirect(state, params);
    };
    DSRPlugin.prototype.getDsrProp = function (state) {
        return state.deepStateRedirect || state.dsr;
    };
    DSRPlugin.prototype.getConfig = function (state) {
        var $state = this.$state;
        var dsrProp = this.getDsrProp(state);
        if (typeof dsrProp === 'undefined')
            return;
        var params;
        var defaultTarget = typeof dsrProp === 'string' ? $state.target(dsrProp) : undefined;
        var fn = typeof dsrProp === 'function' ? dsrProp : undefined;
        if (typeof dsrProp === 'object') {
            fn = dsrProp.fn;
            if (typeof dsrProp.default === 'object') {
                defaultTarget = $state.target(dsrProp.default.state, dsrProp.default.params, dsrProp.default.options);
            }
            else if (typeof dsrProp.default === 'string') {
                defaultTarget = $state.target(dsrProp.default);
            }
            var paramsProp_1 = dsrProp.params;
            if (paramsProp_1 === true) {
                params = function () { return true; };
            }
            else if (Array.isArray(paramsProp_1)) {
                params = function (param) { return paramsProp_1.indexOf(param.id) !== -1; };
            }
        }
        fn = fn || (function (transition, target) { return target; });
        return { default: defaultTarget, params: params, fn: fn };
    };
    DSRPlugin.prototype.paramsEqual = function (state, transParams, paramPredicate, negate) {
        if (paramPredicate === void 0) { paramPredicate = function () { return true; }; }
        if (negate === void 0) { negate = false; }
        var schema = state.parameters({ inherit: true }).filter(paramPredicate);
        return function (redirect) {
            var equals = Param.equals(schema, redirect.triggerParams, transParams);
            return negate ? !equals : equals;
        };
    };
    DSRPlugin.prototype.recordDeepState = function (transition, state) {
        var _this = this;
        var $state = this.$state;
        var paramsConfig = this.getConfig(state).params;
        var _state = state.$$state();
        transition.promise.then(function () {
            var transTo = transition.to();
            var transParams = transition.params();
            var recordedDsrTarget = $state.target(transTo, transParams);
            if (paramsConfig) {
                var recordedDSR = _state.$dsr || [];
                var predicate = _this.paramsEqual(transTo.$$state(), transParams, undefined, true);
                _state.$dsr = recordedDSR.filter(predicate);
                _state.$dsr.push({ triggerParams: transParams, target: recordedDsrTarget });
            }
            else {
                _state.$dsr = recordedDsrTarget;
            }
        });
    };
    DSRPlugin.prototype.deepStateRedirect = function (transition) {
        var opts = transition.options();
        if (opts['ignoreDsr'] || (opts.custom && opts.custom.ignoreDsr))
            return;
        var config = this.getConfig(transition.to());
        var redirect = this.getDeepStateRedirect(transition.to(), transition.params());
        redirect = config.fn(transition, redirect);
        if (redirect && redirect.state() === transition.to())
            return;
        return redirect;
    };
    DSRPlugin.prototype.getDeepStateRedirect = function (stateOrName, params) {
        var $state = this.$state;
        var _state = $state.get(stateOrName);
        var state = _state && _state.$$state();
        var config = this.getConfig(_state);
        var dsrTarget;
        if (config.params) {
            var predicate = this.paramsEqual(state, params, config.params, false);
            var match = state.$dsr && state.$dsr.filter(predicate)[0];
            dsrTarget = match && match.target;
        }
        else {
            dsrTarget = state.$dsr;
        }
        dsrTarget = dsrTarget || config.default;
        if (dsrTarget) {
            // merge original params with deep state redirect params
            var targetParams = Object.assign({}, params, dsrTarget.params());
            dsrTarget = $state.target(dsrTarget.state(), targetParams, dsrTarget.options());
        }
        return dsrTarget;
    };
    return DSRPlugin;
}());
export { DSRPlugin };
//# sourceMappingURL=dsr.js.map