/** @packageDocumentation @reactapi @module react_hooks */
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
import { useCallback, useContext, useEffect, useMemo, useState } from 'react';
import { isString } from '@uirouter/core';
import { UISrefActiveContext } from '../components';
import { useDeepObjectDiff } from './useDeepObjectDiff';
import { useParentView } from './useParentView';
import { useRouter } from './useRouter';
/** @hidden */
export var IncorrectStateNameTypeError = "The state name passed to useSref must be a string.";
/** @hidden Gets all StateDeclarations that are registered in the StateRegistry. */
function useListOfAllStates(router) {
    var initial = useMemo(function () { return router.stateRegistry.get(); }, []);
    var _a = useState(initial), states = _a[0], setStates = _a[1];
    useEffect(function () { return router.stateRegistry.onStatesChanged(function () { return setStates(router.stateRegistry.get()); }); }, []);
    return states;
}
/** @hidden Gets the StateDeclaration that this sref targets */
function useTargetState(router, stateName, relative) {
    // Whenever any states are added/removed from the registry, get the target state again
    var allStates = useListOfAllStates(router);
    return useMemo(function () {
        return router.stateRegistry.get(stateName, relative);
    }, [router, stateName, relative, allStates]);
}
/**
 * A hook to create a link to a state.
 *
 * This hook returns link (anchor tag) props for a given state reference.
 * The resulting props can be spread onto an anchor tag.
 *
 * The props returned from this hook are:
 *
 * - `href`: the browser URL of the referenced state
 * - `onClick`: a mouse event handler that will active the referenced state
 *
 * Example:
 * ```jsx
 * function HomeLink() {
 *   const sref = useSref('home');
 *   return <a {...sref}>Home</a>
 * }
 * ```
 *
 * Example:
 * ```jsx
 * function UserLink({ userId, username }) {
 *   const sref = useSref('users.user', { userId: userId });
 *   return <a {...sref}>{username}</a>
 * }
 * ```
 *
 * @param stateName The name of the state to link to
 * @param params Any parameter values
 * @param options Transition options used when the onClick handler fires.
 */
export function useSref(stateName, params, options) {
    if (params === void 0) { params = {}; }
    if (options === void 0) { options = {}; }
    if (!isString(stateName)) {
        throw new Error(IncorrectStateNameTypeError);
    }
    var router = useRouter();
    // memoize the params object until the nested values actually change so they can be used as deps
    var paramsMemo = useMemo(function () { return params; }, [useDeepObjectDiff(params)]);
    var relative = useParentView().context.name;
    var optionsMemo = useMemo(function () { return (__assign({ relative: relative, inherit: true }, options)); }, [relative, options]);
    var targetState = useTargetState(router, stateName, relative);
    // Update href when the target StateDeclaration changes (in case the the state definition itself changes)
    // This is necessary to handle things like future states
    var href = useMemo(function () {
        return router.stateService.href(stateName, paramsMemo, optionsMemo);
    }, [router, stateName, paramsMemo, optionsMemo, targetState]);
    var onClick = useCallback(function (e) {
        if (!e.defaultPrevented && !(e.button == 1 || e.metaKey || e.ctrlKey)) {
            e.preventDefault();
            router.stateService.go(stateName, paramsMemo, optionsMemo);
        }
    }, [router, stateName, paramsMemo, optionsMemo]);
    // Participate in any parent UISrefActive
    var parentUISrefActiveAddStateInfo = useContext(UISrefActiveContext);
    useEffect(function () {
        return parentUISrefActiveAddStateInfo(targetState && targetState.name, paramsMemo);
    }, [targetState, paramsMemo]);
    return { onClick: onClick, href: href };
}
//# sourceMappingURL=useSref.js.map