"use strict";
/** @packageDocumentation @reactapi @module react_hooks */
Object.defineProperty(exports, "__esModule", { value: true });
var react_1 = require("react");
var useDeepObjectDiff_1 = require("./useDeepObjectDiff");
var useOnStateChanged_1 = require("./useOnStateChanged");
var useParentView_1 = require("./useParentView");
var useRouter_1 = require("./useRouter");
/** @hidden */
function checkIfActive(router, stateName, params, relative, exact) {
    return exact
        ? router.stateService.is(stateName, params, { relative: relative })
        : router.stateService.includes(stateName, params, { relative: relative });
}
/**
 * A hook that returns true if a given state is active.
 *
 * Example:
 * ```jsx
 * function ContactsLabel() {
 *  const isActive = useIsActive('contacts');
 *  return <span className={isActive ? 'active' : 'inactive'}>Contacts></span>
 * }
 * ```
 *
 * Example:
 * ```jsx
 * function JoeLabel() {
 *  const isActive = useIsActive('contacts.contact', { contactId: 'joe' });
 *  return <span className={isActive ? 'active' : 'inactive'}>Joe></span>
 * }
 * ```
 *
 * @param stateName the name of the state to check.
 *        Relative state names such as '.child' are supported.
 *        Relative states are resolved relative to the state that rendered the hook.
 * @param params if present, the hook will only return true if all the provided parameter values match.
 * @param exact when true, the hook returns true only when the state matches exactly.
 *        when false, returns true if the state matches, or any child state matches.
 */
function useIsActive(stateName, params, exact) {
    if (params === void 0) { params = null; }
    if (exact === void 0) { exact = false; }
    var router = useRouter_1.useRouter();
    var relative = useParentView_1.useParentView().context.name;
    // Don't re-compute initialIsActive on every render
    var initialIsActive = react_1.useMemo(function () { return checkIfActive(router, stateName, params, relative, exact); }, []);
    var _a = react_1.useState(initialIsActive), isActive = _a[0], setIsActive = _a[1];
    var checkIfActiveChanged = function () {
        var newIsActive = checkIfActive(router, stateName, params, relative, exact);
        if (newIsActive !== isActive) {
            setIsActive(newIsActive);
        }
    };
    useOnStateChanged_1.useOnStateChanged(checkIfActiveChanged);
    react_1.useEffect(checkIfActiveChanged, [router, stateName, useDeepObjectDiff_1.useDeepObjectDiff(params), exact]);
    return isActive;
}
exports.useIsActive = useIsActive;
//# sourceMappingURL=useIsActive.js.map