'use strict';

const types = require('./types-dc22e301.js');

const errorIcon = `<svg aria-labelledby="error-title" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="#000000">
    <title id="error-title">Error</title>
    <g id="close-octagon">
        <path id="Vector" d="M8.77 3L3.5 8.27V15.73L8.77 21H16.23L21.5 15.73V8.27L16.23 3M8.91 7L12.5 10.59L16.09 7L17.5 8.41L13.91 12L17.5 15.59L16.09 17L12.5 13.41L8.91 17L7.5 15.59L11.09 12L7.5 8.41" />
    </g>
</svg>
`;

const infoIcon = `<svg aria-labelledby="info-title" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="#000000">
    <title id="info-title">For your information</title>
    <g id="info">
        <path id="Vector" d="M12 2C6.48 2 2 6.48 2 12C2 17.52 6.48 22 12 22C17.52 22 22 17.52 22 12C22 6.48 17.52 2 12 2ZM12 17C11.45 17 11 16.55 11 16V12C11 11.45 11.45 11 12 11C12.55 11 13 11.45 13 12V16C13 16.55 12.55 17 12 17ZM13 9H11V7H13V9Z" />
    </g>
</svg>
`;

const neutralIcon = `<svg aria-labelledby="neutral-title" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="#000000">
<title id="neutral-title">Neutral</title>
    <g id="Icon">
        <path id="Vector" d="M19.5 3H5.5C4.4 3 3.5 3.9 3.5 5V19C3.5 20.1 4.4 21 5.5 21H19.5C20.6 21 21.5 20.1 21.5 19V5C21.5 3.9 20.6 3 19.5 3ZM12.51 18C11.81 18 11.25 17.44 11.25 16.74C11.25 16.03 11.81 15.49 12.51 15.49C13.22 15.49 13.76 16.03 13.76 16.74C13.75 17.43 13.22 18 12.51 18ZM15.52 10.6C14.76 11.71 14.04 12.06 13.65 12.77C13.55 12.95 13.49 13.09 13.46 13.4C13.41 13.85 13.01 14.18 12.56 14.18H12.5C11.98 14.18 11.57 13.74 11.62 13.22C11.65 12.88 11.73 12.53 11.92 12.19C12.41 11.32 13.34 10.8 13.88 10.03C14.45 9.22 14.13 7.7 12.51 7.7C11.8 7.7 11.33 8.06 11.04 8.49C10.79 8.85 10.35 9.02 9.94 8.85C9.41 8.64 9.22 8 9.54 7.54C10.15 6.65 11.17 6 12.49 6C13.97 6 14.98 6.67 15.5 7.52C15.94 8.24 16.2 9.59 15.52 10.6Z" />
    </g>
</svg>
`;

const successIcon = `<svg aria-labelledby="success-title" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="#000000">
    <title id="success-title">Success</title>
    <g id="check_circle">
        <path id="Vector" d="M12.5 2C6.98 2 2.5 6.48 2.5 12C2.5 17.52 6.98 22 12.5 22C18.02 22 22.5 17.52 22.5 12C22.5 6.48 18.02 2 12.5 2ZM9.79 16.29L6.2 12.7C5.81 12.31 5.81 11.68 6.2 11.29C6.59 10.9 7.22 10.9 7.61 11.29L10.5 14.17L17.38 7.29C17.77 6.9 18.4 6.9 18.79 7.29C19.18 7.68 19.18 8.31 18.79 8.7L11.2 16.29C10.82 16.68 10.18 16.68 9.79 16.29Z" />
    </g>
</svg>
`;

const warningIcon = `<svg aria-labelledby="warning-title" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="#000000">
    <title id="warning-title">Warning</title>
    <g id="warning">
        <path id="Vector" d="M4.47 20.5037H19.53C21.07 20.5037 22.03 18.8337 21.26 17.5037L13.73 4.49375C12.96 3.16375 11.04 3.16375 10.27 4.49375L2.74 17.5037C1.97 18.8337 2.93 20.5037 4.47 20.5037ZM12 13.5037C11.45 13.5037 11 13.0537 11 12.5037V10.5037C11 9.95375 11.45 9.50375 12 9.50375C12.55 9.50375 13 9.95375 13 10.5037V12.5037C13 13.0537 12.55 13.5037 12 13.5037ZM13 17.5037H11V15.5037H13V17.5037Z" />
    </g>
</svg>
`;

// Global status icons
const VARIANT_ICONS = {
    neutral: {
        icon: neutralIcon,
        ariaLabel: "Neutral",
    },
    info: {
        icon: infoIcon,
        ariaLabel: "For your information",
    },
    warning: {
        icon: warningIcon,
        ariaLabel: "Warning",
    },
    error: {
        icon: errorIcon,
        ariaLabel: "Error",
    },
    success: {
        icon: successIcon,
        ariaLabel: "Success",
    },
};
// Global ARIA attributes
const IC_INHERITED_ARIA = [
    "aria-atomic",
    "aria-autocomplete",
    "aria-busy",
    "aria-controls",
    "aria-current",
    "aria-describedby",
    "aria-description",
    "aria-details",
    "aria-disabled",
    "aria-dropeffect",
    "aria-errormessage",
    "aria-expanded",
    "aria-flowto",
    "aria-grabbed",
    "aria-haspopup",
    "aria-hidden",
    "aria-invalid",
    "aria-keyshortcuts",
    "aria-label",
    "aria-labelledby",
    "aria-live",
    "aria-owns",
    "aria-relevant",
    "aria-roledescription",
];
const IC_DEVICE_SIZES = {
    XS: 576,
    S: 768,
    M: 992,
    L: 1200,
    XL: 99999,
};
const IC_BLOCK_COLOR_COMPONENTS = [
    "ic-hero",
    "ic-top-navigation",
    "ic-footer",
    "ic-side-navigation",
    "ic-alert",
];
const IC_FIXED_COLOR_COMPONENTS = ["ic-alert"];
const IC_BLOCK_COLOR_EXCEPTIONS = {
    "ic-alert": ["ic-link"],
};
/* Range within which the chosen theme colour would not have a sufficient brightness difference with either of the black or white foreground colours
 * The brightness difference must be greater than 125 to provide good colour visibility
 * Calculated by:
 * - Using the brightness formula for both colours: https://www.w3.org/TR/AERT/#color-contrast
 * - Adding 125 to the brightness of the black foreground colour - RGB(11, 12, 12)
 * - Subtracting 125 from the brightness of the white foreground colour - RGB(255, 255, 255)
 */
const BLACK_MIN_COLOR_BRIGHTNESS = 136.701;
const WHITE_MAX_COLOR_BRIGHTNESS = 130;

const DARK_MODE_THRESHOLD = 133.3505;
const icInput = "ic-input";
const linkIcInput = "input.ic-input";
/**
 * Used to inherit global attributes set on the host. Called in componentWillLoad and assigned
 * to a variable that is later used in the render function.
 *
 * This does not need to be reactive as changing attributes on the host element
 * does not trigger a re-render.
 */
const inheritAttributes = (element, attributes = []) => {
    const attributeObject = {};
    attributes.forEach((attr) => {
        if (element.hasAttribute(attr)) {
            const value = element.getAttribute(attr);
            if (value !== null) {
                attributeObject[attr] = value;
            }
            element.removeAttribute(attr);
        }
    });
    return attributeObject;
};
const debounceEvent = (event, wait) => {
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    const original = event._original || event;
    return {
        _original: event,
        emit: debounce(original.emit.bind(original), wait),
    };
};
const debounce = (func, wait = 0) => {
    let timer;
    return (...args) => {
        clearTimeout(timer);
        timer = setTimeout(func, wait, ...args);
    };
};
/**
 * This method is used to add a hidden input to a host element that contains
 * a Shadow DOM. It does not add the input inside of the Shadow root which
 * allows it to be picked up inside of forms. It should contain the same
 * values as the host element.
 *
 * @param always Add a hidden input even if the container does not use Shadow
 * @param container The element where the input will be added
 * @param name The name of the input
 * @param value The value of the input
 * @param disabled If true, the input is disabled
 */
const renderHiddenInput = (always, container, name, value, disabled) => {
    if (name !== undefined && (always || hasShadowDom(container))) {
        const inputs = container.querySelectorAll(linkIcInput);
        const inputEls = Array.from(inputs);
        const filtered = inputEls.filter((el) => container === el.parentElement);
        let input = filtered[0];
        if (input === null || input === undefined) {
            input = container.ownerDocument.createElement("input");
            input.type = "hidden";
            input.classList.add(icInput);
            container.appendChild(input);
        }
        input.disabled = disabled;
        input.name = name;
        if (value instanceof Date) {
            input.value = value ? value.toISOString() : null;
        }
        else {
            input.value = value || "";
        }
    }
};
/**
 * This method is used to add a hidden file input to a host element that contains
 * a Shadow DOM. It does not add the input inside of the Shadow root which
 * allows it to be picked up inside of forms. It should contain the same
 * values as the host element.
 *
 * @param event: The event that is emitted once a file is selected.
 * @param container The element where the input will be added
 * @param multiple If true, multiple files can be selected
 * @param name The name of the input
 * @param value The value of the input
 * @param disabled If true, the input is disabled
 * @param accept A string of the accepted files
 */
const renderFileHiddenInput = (event, container, multiple, name, value, disabled, accept) => {
    if (name !== undefined && hasShadowDom(container)) {
        const inputs = container.querySelectorAll(linkIcInput);
        const inputEls = Array.from(inputs);
        const filtered = inputEls.filter((el) => container === el.parentElement);
        let input = filtered[0];
        if (input === null || input === undefined) {
            input = container.ownerDocument.createElement("input");
            input.classList.add(icInput);
            container.appendChild(input);
        }
        input.type = "file";
        input.hidden = true;
        input.multiple = multiple;
        input.name = name;
        input.files = value;
        input.disabled = disabled;
        input.accept = accept;
        input.onchange = () => {
            event.emit(input.files);
        };
        input.click();
    }
};
const removeHiddenInput = (container) => {
    const inputs = container.querySelectorAll("input.ic-input");
    const inputEls = Array.from(inputs);
    const filtered = inputEls.filter((el) => container === el.parentElement);
    const input = filtered[0];
    input === null || input === void 0 ? void 0 : input.remove();
};
const hasShadowDom = (el) => !!el.shadowRoot && !!el.attachShadow;
const getInputHelperTextID = (id) => id + "-helper-text";
const getInputValidationTextID = (id) => id + "-validation-text";
const getInputDescribedByText = (inputId, helperText, validationText) => `${helperText ? getInputHelperTextID(inputId) : ""} ${validationText ? getInputValidationTextID(inputId) : ""}`.trim();
/**
 * This method helps to understand the context in which a component exists,
 * to assist with choosing appropriate foreground colours to use. For example,
 * this method will help you use the 'white' version of a button if it's within
 * a block colour element using white foreground text.
 *
 * This only works for components/elements passed via <slot> and not if they
 * are part of an IC component.
 *
 * ""
 * @returns IcThemeForeground depending on the context
 */
const getThemeFromContext = (el, themeFromEvent = null) => {
    var _a;
    const parentElement = el.parentElement || el.getRootNode().host.parentElement;
    const blockColorParent = parentElement.closest(IC_BLOCK_COLOR_COMPONENTS.join(","));
    // If within a block color component
    if (blockColorParent !== null) {
        const parentTag = blockColorParent.tagName.toLowerCase();
        const currentTag = el.tagName.toLowerCase();
        if ((_a = IC_BLOCK_COLOR_EXCEPTIONS[parentTag]) === null || _a === void 0 ? void 0 : _a.includes(currentTag)) {
            return types.IcThemeForegroundEnum.Default;
        }
        else if (themeFromEvent !== null &&
            !IC_FIXED_COLOR_COMPONENTS.includes(parentTag)) {
            return themeFromEvent;
        }
        else if (blockColorParent.classList.contains(types.IcThemeForegroundEnum.Dark)) {
            return types.IcThemeForegroundEnum.Dark;
        }
        return types.IcThemeForegroundEnum.Light;
    }
    return types.IcThemeForegroundEnum.Default;
};
/**
 * Checks if the current device is a mobile or tablet device.
 * @returns {boolean} Returns true if the device is a mobile or tablet device, otherwise returns false.
 */
const isMobileOrTablet = () => "maxTouchPoints" in navigator && "userAgent" in navigator
    ? navigator.maxTouchPoints > 0 &&
        /iPad|iPhone|iPod|Android|webOS|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent)
    : false;
/**
 * Will create a button within the lightDOM which interacts with the parent form.
 * This is required as buttons within the shadowDOM will not invoke a submit or reset
 *
 * @param form - parent form element which contains shadowDom button
 * @param button - shadowDOM button
 */
const handleHiddenFormButtonClick = (form, button) => {
    const hiddenFormButton = document.createElement("button");
    hiddenFormButton.setAttribute("type", button.type);
    hiddenFormButton.style.display = "none";
    form.appendChild(hiddenFormButton);
    hiddenFormButton.click();
    hiddenFormButton.remove();
};
const isEmptyString = (value) => value ? value.trim().length === 0 : true;
// A helper function that checks if a prop has been defined
const isPropDefined = (prop) => prop !== undefined ? prop : null;
/**
 * Extracts the label using the value from an object. Requires the object to have a label and value property.
 * @param value - value from object
 * @param options - list of menu items
 * @returns - label corresponding to value
 */
const getLabelFromValue = (value, options, valueField = "value", labelField = "label") => {
    const ungroupedOptions = [];
    if (options.length > 0 && options.map) {
        options.map((option) => {
            if (option.children) {
                option.children.map((option) => ungroupedOptions.push(option));
            }
            else {
                ungroupedOptions.push(option);
            }
        });
        const matchingValue = ungroupedOptions.find((option) => option[valueField] === value);
        if (matchingValue !== undefined)
            return matchingValue[labelField];
    }
    return undefined;
};
/**
 * Filters the options based on the search string.
 * @param options - array of options
 * @param includeDescriptions - determines whether option descriptions are included when filtering options
 * @param searchString - string used to filter the options
 * @param position - whether the search string matches the start of or anywhere in the options
 * @returns filtered array of options
 */
const getFilteredMenuOptions = (options, includeDescriptions, searchString, position, labelField = "label") => options.filter((option) => {
    var _a;
    const label = option[labelField].toLowerCase();
    const description = (_a = option.description) === null || _a === void 0 ? void 0 : _a.toLowerCase();
    const lowerSearchString = searchString.toLowerCase();
    return position === "anywhere"
        ? includeDescriptions
            ? label.includes(lowerSearchString) ||
                (description === null || description === void 0 ? void 0 : description.includes(lowerSearchString))
            : label.includes(lowerSearchString)
        : includeDescriptions
            ? label.startsWith(lowerSearchString) ||
                (description === null || description === void 0 ? void 0 : description.startsWith(lowerSearchString))
            : label.startsWith(lowerSearchString);
});
const deviceSizeMatches = (size) => window.matchMedia(`(max-width: ${size}px)`).matches;
const getCurrentDeviceSize = () => {
    if (deviceSizeMatches(DEVICE_SIZES.S)) {
        return DEVICE_SIZES.S;
    }
    if (deviceSizeMatches(DEVICE_SIZES.M)) {
        return DEVICE_SIZES.M;
    }
    if (deviceSizeMatches(DEVICE_SIZES.L)) {
        return DEVICE_SIZES.L;
    }
    if (deviceSizeMatches(DEVICE_SIZES.XL)) {
        return DEVICE_SIZES.XL;
    }
    //fallback needed as all of above get initialised to 0 in jest tests
    return DEVICE_SIZES.UNDEFINED;
};
const getCssProperty = (cssVar) => getComputedStyle(document.documentElement).getPropertyValue(cssVar);
/**
 * Returns the brightness of the theme colour, calculated by using the theme RGB CSS values by:
 * - Multiplying each RGB value by a set number: https://www.w3.org/TR/AERT/#color-contrast
 * - Adding them together and dividing by 1000
 * This is a similar calculation to its CSS counterpart: "--ic-theme-text"
 * @returns number representing the brightness of the theme colour
 */
const getThemeColorBrightness = () => {
    const themeRed = getCssProperty("--ic-theme-primary-r");
    const themeGreen = getCssProperty("--ic-theme-primary-g");
    const themeBlue = getCssProperty("--ic-theme-primary-b");
    return ((parseInt(themeRed) * 299 +
        parseInt(themeGreen) * 587 +
        parseInt(themeBlue) * 114) /
        1000);
};
/**
 * Returns if dark or light foreground colors should be used for color contrast reasons
 * @returns "dark" or "light"
 * @param brightness - Optional custom brightness value. Defaults to `getThemeColorBrightness`
 */
const getThemeForegroundColor = (brightness = getThemeColorBrightness()) => brightness > DARK_MODE_THRESHOLD
    ? types.IcThemeForegroundEnum.Dark
    : types.IcThemeForegroundEnum.Light;
const getSlot = (element, name) => {
    if (element && element.querySelector) {
        return element.querySelector(`[slot="${name}"]`);
    }
    return null;
};
const slotHasContent = (element, name) => getSlot(element, name) !== null;
const getSlotContent = (element, name) => {
    const slot = getSlot(element, name);
    if (slot) {
        return getSlotElements(slot);
    }
    return null;
};
const getSlotElements = (slot) => {
    const slotContent = slot.firstElementChild;
    if (slotContent !== null) {
        const elements = slotContent.assignedElements
            ? slotContent.assignedElements()
            : slotContent.childNodes;
        return elements.length ? elements : slot.tagName ? [slot] : null;
    }
    else {
        //check for single element
        return slot === null ? null : [slot];
    }
};
const getNavItemParentDetails = ({ parentElement, }) => {
    let navType = { navType: "", parent: null };
    switch (parentElement.tagName) {
        case "IC-NAVIGATION-GROUP":
            navType = getNavItemParentDetails(parentElement);
            break;
        case "IC-TOP-NAVIGATION":
            navType = { navType: "top", parent: parentElement };
            break;
        case "IC-SIDE-NAVIGATION":
            navType = { navType: "side", parent: parentElement };
            break;
        case "IC-PAGE-HEADER":
            navType = { navType: "page-header", parent: null };
            break;
    }
    return navType;
};
const DEVICE_SIZES = {
    XS: Number(getCssProperty("--ic-breakpoint-xs").replace("px", "")), // 0
    S: Number(getCssProperty("--ic-breakpoint-sm").replace("px", "")), // 576
    M: Number(getCssProperty("--ic-breakpoint-md").replace("px", "")), // 768
    L: Number(getCssProperty("--ic-breakpoint-lg").replace("px", "")), // 992
    XL: Number(getCssProperty("--ic-breakpoint-xl").replace("px", "")), // 1200
    UNDEFINED: 1200,
};
const hasValidationStatus = (status, disabled) => !!status && !disabled;
const isSlotUsed = ({ children }, slotName) => Array.from(children).some((child) => child.getAttribute("slot") === slotName);
// added as a common method to allow detection of gatsby hydration issue, where (camelCase) props are initially undefined & then update
// with a value. Allows a callback function to be executed when this is the case
const onComponentPropUndefinedChange = (oldValue, newValue, callback) => {
    if (oldValue === undefined && newValue !== oldValue) {
        callback();
    }
};
const onComponentRequiredPropUndefined = (props, component) => {
    props.forEach(({ prop, propName }) => {
        if (prop === null || prop === undefined) {
            console.error(`No ${propName} specified for ${component} component - prop '${propName}' (web components) / '${kebabToCamelCase(propName)}' (react) required`);
        }
    });
};
const kebabToCamelCase = (kebabCase) => kebabCase
    .toLowerCase()
    .split("-")
    .map((word, index) => index === 0
    ? word
    : `${word.substring(0, 1).toUpperCase()}${word.substring(1)}`)
    .join("");
const checkResizeObserver = (callbackFn) => {
    if (typeof window !== "undefined" &&
        typeof window.ResizeObserver !== "undefined") {
        callbackFn();
    }
};
const hex2dec = (v) => parseInt(v, 16);
const hexToRgba = (hex) => {
    const hexChars = hex
        .replace("#", "")
        .split("")
        .map((char) => char.repeat(2));
    return {
        r: hex2dec(hex.length === 4 ? hexChars[0] : hex.slice(1, 3)),
        g: hex2dec(hex.length === 4 ? hexChars[1] : hex.slice(3, 5)),
        b: hex2dec(hex.length === 4 ? hexChars[2] : hex.slice(5)),
        a: 1,
    };
};
const rgbaStrToObj = (rgbaStr) => {
    const isRGBA = rgbaStr.slice(3, 4).toLowerCase() === "a";
    const rgbValues = rgbaStr
        .substring(isRGBA ? 5 : 4, rgbaStr.length - 1)
        .replace(/ /g, "")
        .split(",")
        .map(Number);
    return {
        r: rgbValues[0],
        g: rgbValues[1],
        b: rgbValues[2],
        a: isRGBA ? rgbValues[3] : 1,
    };
};
const elementOverflowsX = ({ scrollWidth, clientWidth, }) => scrollWidth > clientWidth;
const hasClassificationBanner = () => !!document.querySelector("ic-classification-banner:not([inline='true'])");
const addFormResetListener = (el, callbackFn) => {
    var _a;
    (_a = el.closest("FORM")) === null || _a === void 0 ? void 0 : _a.addEventListener("reset", callbackFn);
};
const removeFormResetListener = (el, callbackFn) => {
    var _a;
    (_a = el.closest("FORM")) === null || _a === void 0 ? void 0 : _a.removeEventListener("reset", callbackFn);
};
const pxToRem = (px, base = 16) => `${(1 / base) * parseInt(px)}rem`;
const removeDisabledFalse = (disabled, element) => {
    if (!disabled) {
        element.removeAttribute("disabled");
    }
};
const convertToRGBA = (color) => {
    const firstChar = color === null || color === void 0 ? void 0 : color.slice(0, 1).toLowerCase();
    return firstChar === "#"
        ? hexToRgba(color)
        : firstChar === "r"
            ? rgbaStrToObj(color)
            : null;
};
const capitalize = (text) => text.charAt(0).toUpperCase() + text.slice(1);
const checkSlotInChildMutations = (addedNodes, removedNodes, slotName) => {
    const hasSlot = (nodeList) => Array.from(nodeList).some((node) => Array.isArray(slotName)
        ? slotName.some((name) => node.slot === name)
        : node.slot === slotName);
    return hasSlot(addedNodes) || hasSlot(removedNodes);
};
const isElInAGGrid = (el) => !!el.closest(".ag-cell") && !!el.closest(".ag-root");

exports.BLACK_MIN_COLOR_BRIGHTNESS = BLACK_MIN_COLOR_BRIGHTNESS;
exports.DEVICE_SIZES = DEVICE_SIZES;
exports.IC_DEVICE_SIZES = IC_DEVICE_SIZES;
exports.IC_INHERITED_ARIA = IC_INHERITED_ARIA;
exports.VARIANT_ICONS = VARIANT_ICONS;
exports.WHITE_MAX_COLOR_BRIGHTNESS = WHITE_MAX_COLOR_BRIGHTNESS;
exports.addFormResetListener = addFormResetListener;
exports.capitalize = capitalize;
exports.checkResizeObserver = checkResizeObserver;
exports.checkSlotInChildMutations = checkSlotInChildMutations;
exports.convertToRGBA = convertToRGBA;
exports.debounceEvent = debounceEvent;
exports.elementOverflowsX = elementOverflowsX;
exports.errorIcon = errorIcon;
exports.getCssProperty = getCssProperty;
exports.getCurrentDeviceSize = getCurrentDeviceSize;
exports.getFilteredMenuOptions = getFilteredMenuOptions;
exports.getInputDescribedByText = getInputDescribedByText;
exports.getInputHelperTextID = getInputHelperTextID;
exports.getInputValidationTextID = getInputValidationTextID;
exports.getLabelFromValue = getLabelFromValue;
exports.getNavItemParentDetails = getNavItemParentDetails;
exports.getSlot = getSlot;
exports.getSlotContent = getSlotContent;
exports.getSlotElements = getSlotElements;
exports.getThemeColorBrightness = getThemeColorBrightness;
exports.getThemeForegroundColor = getThemeForegroundColor;
exports.getThemeFromContext = getThemeFromContext;
exports.handleHiddenFormButtonClick = handleHiddenFormButtonClick;
exports.hasClassificationBanner = hasClassificationBanner;
exports.hasValidationStatus = hasValidationStatus;
exports.hexToRgba = hexToRgba;
exports.inheritAttributes = inheritAttributes;
exports.isElInAGGrid = isElInAGGrid;
exports.isEmptyString = isEmptyString;
exports.isMobileOrTablet = isMobileOrTablet;
exports.isPropDefined = isPropDefined;
exports.isSlotUsed = isSlotUsed;
exports.onComponentPropUndefinedChange = onComponentPropUndefinedChange;
exports.onComponentRequiredPropUndefined = onComponentRequiredPropUndefined;
exports.pxToRem = pxToRem;
exports.removeDisabledFalse = removeDisabledFalse;
exports.removeFormResetListener = removeFormResetListener;
exports.removeHiddenInput = removeHiddenInput;
exports.renderFileHiddenInput = renderFileHiddenInput;
exports.renderHiddenInput = renderHiddenInput;
exports.rgbaStrToObj = rgbaStrToObj;
exports.slotHasContent = slotHasContent;
exports.successIcon = successIcon;
exports.warningIcon = warningIcon;

//# sourceMappingURL=helpers-94b33ab0.js.map