import { h, Host, } from "@stencil/core";
import { isSlotUsed } from "../../utils/helpers";
import chevronIcon from "../../assets/chevron-icon.svg";
let accordionIds = 0;
/**
 * @slot heading - Content is placed as the accordion heading.
 * @slot icon - Content is placed to the left of the heading.
 */
export class Accordion {
    constructor() {
        this.accordionId = `ic-accordion-${accordionIds++}`;
        this.CONTENT_VISIBILITY_PROPERTY = "--ic-expanded-content-visibility";
        this.toggleExpanded = () => {
            this.expanded = !this.expanded;
            this.accordionClicked.emit({ id: this.accordionId });
        };
        // Set accordion animation
        this.setAccordionAnimation = (el, duration, property, delay) => {
            el.style.transitionDuration = `${duration}ms`;
            el.style.transitionProperty = property;
            el.style.transitionDelay = delay;
        };
        this.setExpandedContentStyle = (ev, expandedContent) => {
            if (ev.propertyName === "height" && expandedContent.clientHeight > 0) {
                expandedContent.classList.add("expanded-content-opened");
                expandedContent.style.height = "auto";
            }
        };
        this.hideExpandedContent = (ev, expandedContent) => {
            if (ev.propertyName === "height" && expandedContent.clientHeight === 0) {
                expandedContent.style.setProperty(this.CONTENT_VISIBILITY_PROPERTY, "hidden");
            }
        };
        this.animateExpandedContent = () => {
            const elementHeight = this.expandedContentEl.scrollHeight;
            if (elementHeight > 0 && this.expanded) {
                this.expandedContentEl.style.setProperty(this.CONTENT_VISIBILITY_PROPERTY, "visible");
                this.expandedContentEl.style.height = `${elementHeight}px`;
                this.setAccordionAnimation(this.expandedContentEl, "300", "height", "ease-out");
                this.expandedContentEl.addEventListener("transitionend", (e) => {
                    this.setExpandedContentStyle(e, this.expandedContentEl);
                });
            }
            else if (!this.expanded) {
                this.expandedContentEl.style.height = `${this.expandedContentEl.scrollHeight}px`;
                if (this.expandedContentEl.scrollHeight > 0 && !this.expanded) {
                    this.expandedContentEl.style.height = "0";
                    this.setAccordionAnimation(this.expandedContentEl, "300", "height", "ease-in");
                    this.expandedContentEl.classList.remove("expanded-content-opened");
                }
                this.expandedContentEl.addEventListener("transitionend", (e) => {
                    this.hideExpandedContent(e, this.expandedContentEl);
                });
            }
        };
        this.appearance = "default";
        this.disabled = false;
        this.expanded = false;
        this.heading = "";
        this.message = "";
        this.size = "default";
    }
    // Every time expanded is set via toggleExpanded or accordionGroup, animate to open or close
    handleExpandedWatch() {
        this.animateExpandedContent();
    }
    /**
     * Sets focus on accordion heading.
     */
    async setFocus() {
        if (this.accordionBtnHeading) {
            this.accordionBtnHeading.focus();
        }
    }
    disconnectedCallback() {
        if (this.expandedContentEl) {
            this.expandedContentEl.removeEventListener("transitionend", (e) => this.setExpandedContentStyle(e, this.expandedContentEl), true);
            this.expandedContentEl.removeEventListener("transitionend", (e) => this.hideExpandedContent(e, this.expandedContentEl), true);
        }
    }
    componentDidLoad() {
        // So accordion opens by default if expanded set to true
        if (this.expanded) {
            this.expandedContentEl.style.height = "auto";
            this.expandedContentEl.style.setProperty(this.CONTENT_VISIBILITY_PROPERTY, "visible");
        }
    }
    render() {
        const { appearance, size, disabled, expanded } = this;
        return (h(Host, { id: this.accordionId, class: {
                [`${appearance}`]: true,
                ["disabled"]: disabled,
            }, "aria-disabled": disabled ? "true" : "false" }, h("button", { ref: (el) => (this.accordionBtnHeading = el), id: `${this.accordionId}-button`, disabled: disabled, tabindex: disabled ? -1 : 0, class: {
                [`${size}`]: true,
                ["section-button"]: true,
                ["section-button-open"]: expanded && !disabled,
            }, "aria-expanded": `${expanded}`, "aria-controls": "expanded-content-area", onClick: this.toggleExpanded }, isSlotUsed(this.el, "icon") && (h("div", { class: "icon-container" }, h("slot", { name: "icon" }))), h("ic-typography", { variant: "subtitle-large", class: "section-header" }, isSlotUsed(this.el, "heading") ? (h("slot", { name: "heading" })) : (this.heading)), h("span", { class: {
                ["expand-chevron"]: true,
                ["content-expanded-chevron"]: expanded && !disabled,
            }, "aria-hidden": "true", innerHTML: chevronIcon })), h("div", { class: {
                ["expanded-content"]: true,
            }, "aria-labelledby": `${this.accordionId}-button`, role: "region", "aria-hidden": `${!expanded}`, id: "expanded-content-area", ref: (el) => (this.expandedContentEl = el) }, h("div", { class: "expanded-content-inner" }, this.message ? (h("ic-typography", { variant: "body" }, this.message)) : (h("slot", null))))));
    }
    static get is() { return "ic-accordion"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["ic-accordion.css"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["ic-accordion.css"]
        };
    }
    static get properties() {
        return {
            "appearance": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcThemeForeground",
                    "resolved": "\"dark\" | \"default\" | \"light\"",
                    "references": {
                        "IcThemeForeground": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcThemeForeground"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": "Determines whether the light or dark variant of the accordion should be displayed."
                        }],
                    "text": ""
                },
                "attribute": "appearance",
                "reflect": false,
                "defaultValue": "\"default\""
            },
            "disabled": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the accordion will be disabled."
                },
                "attribute": "disabled",
                "reflect": false,
                "defaultValue": "false"
            },
            "expanded": {
                "type": "boolean",
                "mutable": true,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the accordion appears expanded."
                },
                "attribute": "expanded",
                "reflect": false,
                "defaultValue": "false"
            },
            "heading": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The section header outlining section content."
                },
                "attribute": "heading",
                "reflect": false,
                "defaultValue": "\"\""
            },
            "message": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The main body message of the accordion."
                },
                "attribute": "message",
                "reflect": false,
                "defaultValue": "\"\""
            },
            "size": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcSizes",
                    "resolved": "\"default\" | \"large\" | \"small\"",
                    "references": {
                        "IcSizes": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcSizes"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The size of the accordion."
                },
                "attribute": "size",
                "reflect": false,
                "defaultValue": "\"default\""
            }
        };
    }
    static get events() {
        return [{
                "method": "accordionClicked",
                "name": "accordionClicked",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": "Emitted when accordion is clicked."
                        }],
                    "text": ""
                },
                "complexType": {
                    "original": "{ id: string }",
                    "resolved": "{ id: string; }",
                    "references": {}
                }
            }];
    }
    static get methods() {
        return {
            "setFocus": {
                "complexType": {
                    "signature": "() => Promise<void>",
                    "parameters": [],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "Sets focus on accordion heading.",
                    "tags": []
                }
            }
        };
    }
    static get elementRef() { return "el"; }
    static get watchers() {
        return [{
                "propName": "expanded",
                "methodName": "handleExpandedWatch"
            }];
    }
}
//# sourceMappingURL=ic-accordion.js.map
