import { Host, h, } from "@stencil/core";
import { convertToRGBA, getCssProperty, getThemeForegroundColor, hexToRgba, isPropDefined, onComponentRequiredPropUndefined, rgbaStrToObj, } from "../../utils/helpers";
/**
 * @slot badge-icon - Icon will be rendered inside the badge if type is set to icon.
 */
export class Badge {
    constructor() {
        this.ariaLabel = null;
        this.setBadgeColour = () => {
            const colorRGBA = convertToRGBA(this.customColor);
            if (colorRGBA) {
                this.customColorRGBA = colorRGBA;
                const { r, g, b, a } = colorRGBA;
                this.el.style.backgroundColor = `rgba(${r}, ${g}, ${b}, ${a})`;
            }
        };
        this.getBadgeRGB = () => {
            switch (this.variant) {
                case "custom":
                    return this.customColorRGBA;
                case "error":
                case "success":
                case "warning":
                case "info": {
                    return rgbaStrToObj(getCssProperty(`--ic-status-${this.variant}`));
                }
                case "neutral":
                case "light":
                    return hexToRgba(getCssProperty(`--ic-architectural-${this.variant === "neutral" ? 500 : 40}`));
            }
        };
        this.getBadgeForeground = () => {
            const { r, g, b } = this.getBadgeRGB();
            this.foregroundColour = getThemeForegroundColor((r * 299 + g * 587 + b * 114) / 1000);
        };
        this.getTextLabel = () => this.maxNumber && Number(this.textLabel) > this.maxNumber
            ? `${this.maxNumber}+`
            : this.textLabel;
        // Set aria-label on badge and / or parent element
        // Aria-describedby seems to not work, probably due to shadow DOM
        this.setAccessibleLabel = () => {
            const parentEl = this.el.parentElement;
            const defaultAriaLabel = this.isAccessibleLabelDefined()
                ? this.accessibleLabel
                : this.textLabel || "with badge being displayed";
            if (parentEl) {
                const { tagName } = parentEl;
                if (tagName !== "IC-CARD" &&
                    (tagName !== "IC-TAB" || (tagName === "IC-TAB" && this.parentAriaLabel))) {
                    const ariaLabelPrefix = this.parentAriaLabel
                        ? `${this.parentAriaLabel} ,`
                        : "";
                    parentEl.ariaLabel = this.visible
                        ? `${ariaLabelPrefix} ${defaultAriaLabel}`
                        : undefined;
                }
                else {
                    this.ariaLabel = `, ${defaultAriaLabel}`;
                }
            }
        };
        this.isAccessibleLabelDefined = () => isPropDefined(this.accessibleLabel) && this.accessibleLabel !== null;
        this.accessibleLabel = undefined;
        this.customColor = null;
        this.maxNumber = undefined;
        this.position = "far";
        this.size = "default";
        this.textLabel = undefined;
        this.type = "text";
        this.variant = "neutral";
        this.visible = true;
    }
    accessibleLabelHandler() {
        this.setAccessibleLabel();
    }
    customColorHandler() {
        this.variant === "custom" && this.setBadgeColour();
    }
    variantHandler() {
        this.getBadgeForeground();
    }
    visibleHandler() {
        this.setAccessibleLabel();
    }
    componentWillLoad() {
        var _a;
        this.variant === "custom" && this.setBadgeColour();
        this.getBadgeForeground();
        const ariaLabel = (_a = this.el.parentElement) === null || _a === void 0 ? void 0 : _a.ariaLabel;
        if (ariaLabel)
            this.parentAriaLabel = ariaLabel;
        this.setAccessibleLabel();
    }
    componentDidLoad() {
        this.type === "text" &&
            onComponentRequiredPropUndefined([{ prop: this.textLabel, propName: "text-label" }], "Badge");
    }
    /**
     * @deprecated This method should not be used anymore. Use visible prop to set badge visibility.
     */
    async showBadge() {
        this.visible = true;
    }
    /**
     * @deprecated This method should not be used anymore. Use visible prop to set badge visibility.
     */
    async hideBadge() {
        this.visible = false;
    }
    render() {
        const { ariaLabel, el, foregroundColour, getTextLabel, position, size, textLabel, type, variant, visible, } = this;
        return (h(Host, { class: {
                [`${position}`]: true,
                [`${size}`]: true,
                [`${variant}`]: true,
                [`${type}`]: true,
                [`foreground-${foregroundColour}`]: foregroundColour !== null,
                [`${visible ? "show" : "hide"}`]: true,
            }, id: el.id || null, "aria-label": ariaLabel, role: "status" }, type === "icon" && h("slot", { name: "badge-icon" }), type === "text" && textLabel && (h("ic-typography", { variant: size === "small" ? "badge-small" : "badge" }, getTextLabel())), type === "dot" && h("span", { class: "sr-only" }, "badge")));
    }
    static get is() { return "ic-badge"; }
    static get encapsulation() { return "shadow"; }
    static get delegatesFocus() { return true; }
    static get originalStyleUrls() {
        return {
            "$": ["ic-badge.css"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["ic-badge.css"]
        };
    }
    static get properties() {
        return {
            "accessibleLabel": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The accessible label of the badge component to provide context for screen reader users."
                },
                "attribute": "accessible-label",
                "reflect": false
            },
            "customColor": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcColor",
                    "resolved": "`#${string}` | `rgb(${string})` | `rgba(${string})`",
                    "references": {
                        "IcColor": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcColor"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The custom badge colour. This will only style the badge component if variant=\"custom\".\nCan be a hex value e.g. \"#ff0000\", RGB e.g. \"rgb(255, 0, 0)\", or RGBA e.g. \"rgba(255, 0, 0, 1)\"."
                },
                "attribute": "custom-color",
                "reflect": false,
                "defaultValue": "null"
            },
            "maxNumber": {
                "type": "number",
                "mutable": false,
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The maximum number shown on the badge appended with a +.\nThis will only be displayed if type=\"text\" and textLabel is not empty."
                },
                "attribute": "max-number",
                "reflect": false
            },
            "position": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcBadgePositions",
                    "resolved": "\"far\" | \"inline\" | \"near\"",
                    "references": {
                        "IcBadgePositions": {
                            "location": "import",
                            "path": "./ic-badge.types",
                            "id": "src/components/ic-badge/ic-badge.types.ts::IcBadgePositions"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The positioning of the badge in reference to the parent element."
                },
                "attribute": "position",
                "reflect": false,
                "defaultValue": "\"far\""
            },
            "size": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcSizes",
                    "resolved": "\"default\" | \"large\" | \"small\"",
                    "references": {
                        "IcSizes": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcSizes"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The size of the badge to be displayed."
                },
                "attribute": "size",
                "reflect": false,
                "defaultValue": "\"default\""
            },
            "textLabel": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The text displayed in the badge. This will only be displayed if type=\"text\"."
                },
                "attribute": "text-label",
                "reflect": false
            },
            "type": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcBadgeTypes",
                    "resolved": "\"dot\" | \"icon\" | \"text\"",
                    "references": {
                        "IcBadgeTypes": {
                            "location": "import",
                            "path": "./ic-badge.types",
                            "id": "src/components/ic-badge/ic-badge.types.ts::IcBadgeTypes"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The type of badge to be displayed."
                },
                "attribute": "type",
                "reflect": false,
                "defaultValue": "\"text\""
            },
            "variant": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcBadgeVariants",
                    "resolved": "\"custom\" | \"error\" | \"info\" | \"light\" | \"neutral\" | \"success\" | \"warning\"",
                    "references": {
                        "IcBadgeVariants": {
                            "location": "import",
                            "path": "./ic-badge.types",
                            "id": "src/components/ic-badge/ic-badge.types.ts::IcBadgeVariants"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The variant of the badge to be displayed."
                },
                "attribute": "variant",
                "reflect": false,
                "defaultValue": "\"neutral\""
            },
            "visible": {
                "type": "boolean",
                "mutable": true,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the badge will be displayed."
                },
                "attribute": "visible",
                "reflect": false,
                "defaultValue": "true"
            }
        };
    }
    static get methods() {
        return {
            "showBadge": {
                "complexType": {
                    "signature": "() => Promise<void>",
                    "parameters": [],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "",
                    "tags": [{
                            "name": "deprecated",
                            "text": "This method should not be used anymore. Use visible prop to set badge visibility."
                        }]
                }
            },
            "hideBadge": {
                "complexType": {
                    "signature": "() => Promise<void>",
                    "parameters": [],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "",
                    "tags": [{
                            "name": "deprecated",
                            "text": "This method should not be used anymore. Use visible prop to set badge visibility."
                        }]
                }
            }
        };
    }
    static get elementRef() { return "el"; }
    static get watchers() {
        return [{
                "propName": "accessibleLabel",
                "methodName": "accessibleLabelHandler"
            }, {
                "propName": "customColor",
                "methodName": "customColorHandler"
            }, {
                "propName": "variant",
                "methodName": "variantHandler"
            }, {
                "propName": "visible",
                "methodName": "visibleHandler"
            }];
    }
}
//# sourceMappingURL=ic-badge.js.map
