import { h, forceUpdate, } from "@stencil/core";
import { onComponentRequiredPropUndefined, isSlotUsed, getThemeFromContext, removeDisabledFalse, checkSlotInChildMutations, } from "../../utils/helpers";
import { IcThemeForegroundEnum, } from "../../utils/types";
import chevronIcon from "../../assets/chevron-icon.svg";
/**
 * @slot heading - Content will be placed at the top of the card to the right of the icon.
 * @slot message - Content will be placed in the main body of the card.
 * @slot subheading - Content will be placed below the card heading.
 * @slot adornment - Content will be placed below the card subheading.
 * @slot image-top - Content will be placed at the top of the card above all other content.
 * @slot image-mid - Content will be placed below the card heading section.
 * @slot icon - Content will be placed to the left of the card heading.
 * @slot interaction-button - Content will be placed in the top right corner of the heading section.
 * @slot interaction-controls - Content will be placed below the card message.
 * @slot expanded-content - Content will be placed below the interaction controls but will not initially be rendered.
 * @slot badge - Badge component overlaying the top right of the card.
 */
export class Card {
    constructor() {
        this.hostMutationObserver = null;
        this.parentFocussed = () => {
            this.isFocussed = true;
        };
        this.parentBlurred = () => {
            this.isFocussed = false;
        };
        this.toggleExpanded = () => {
            this.areaExpanded = !this.areaExpanded;
        };
        this.hostMutationCallback = (mutationList) => {
            if (mutationList.some(({ type, addedNodes, removedNodes }) => type === "childList"
                ? checkSlotInChildMutations(addedNodes, removedNodes, [
                    "message",
                    "adornment",
                    "expanded-content",
                    "image-top",
                    "image-mid",
                    "icon",
                    "interaction-button",
                    "badge",
                    "interaction-controls",
                ])
                : false)) {
                forceUpdate(this);
            }
        };
        this.appearance = "default";
        this.areaExpanded = false;
        this.isFocussed = false;
        this.parentEl = null;
        this.parentIsAnchorTag = false;
        this.clickable = false;
        this.disabled = false;
        this.expandable = false;
        this.fullWidth = false;
        this.heading = undefined;
        this.href = undefined;
        this.hreflang = "";
        this.message = "";
        this.referrerpolicy = undefined;
        this.rel = undefined;
        this.subheading = undefined;
        this.target = undefined;
    }
    disconnectedCallback() {
        var _a;
        if (this.parentIsAnchorTag) {
            this.parentEl.removeEventListener("focus", this.parentFocussed);
            this.parentEl.removeEventListener("blur", this.parentBlurred);
        }
        (_a = this.hostMutationObserver) === null || _a === void 0 ? void 0 : _a.disconnect();
    }
    componentWillLoad() {
        this.parentEl = this.el.parentElement;
        if (this.parentEl.tagName === "A") {
            this.clickable = true;
            this.parentIsAnchorTag = true;
            this.parentEl.classList.add("ic-card-wrapper-link");
            this.parentEl.addEventListener("focus", this.parentFocussed);
            this.parentEl.addEventListener("blur", this.parentBlurred);
        }
        removeDisabledFalse(this.disabled, this.el);
    }
    componentDidLoad() {
        !isSlotUsed(this.el, "heading") &&
            onComponentRequiredPropUndefined([{ prop: this.heading, propName: "heading" }], "Card");
        this.updateTheme();
        this.hostMutationObserver = new MutationObserver(this.hostMutationCallback);
        this.hostMutationObserver.observe(this.el, {
            childList: true,
        });
    }
    handleHostClick(event) {
        if (this.disabled) {
            event.stopImmediatePropagation();
        }
    }
    themeChangeHandler(ev) {
        const theme = ev.detail;
        this.updateTheme(theme.mode);
    }
    /**
     * Sets focus on the card.
     */
    async setFocus() {
        if (this.el.shadowRoot.querySelector("a")) {
            this.el.shadowRoot.querySelector("a").focus();
        }
        else if (this.el.shadowRoot.querySelector("button")) {
            this.el.shadowRoot.querySelector("button").focus();
        }
    }
    updateTheme(newTheme = null) {
        const foregroundColor = getThemeFromContext(this.el, newTheme || null);
        if (foregroundColor !== IcThemeForegroundEnum.Default) {
            this.appearance = foregroundColor;
        }
    }
    render() {
        const { clickable, disabled, expandable, heading, message, href, hreflang, referrerpolicy, rel, subheading, target, fullWidth, parentIsAnchorTag, isFocussed, } = this;
        const Component = parentIsAnchorTag
            ? "div"
            : clickable
                ? this.href === undefined
                    ? "button"
                    : "a"
                : "div";
        const attrs = Component == "a" && {
            href: href,
            hrefLang: hreflang,
            referrerPolicy: referrerpolicy,
            rel: rel,
            target: target,
        };
        return (h(Component, Object.assign({ class: {
                ["card"]: true,
                ["clickable"]: clickable && !disabled,
                ["disabled"]: disabled,
                ["fullwidth"]: fullWidth,
                ["focussed"]: isFocussed,
                ["dark"]: this.appearance === IcThemeForegroundEnum.Dark,
            }, tabindex: clickable && !parentIsAnchorTag ? 0 : null, "aria-disabled": disabled ? "true" : null, disabled: disabled ? true : null }, attrs), isSlotUsed(this.el, "image-top") && (h("div", { class: "image-top" }, h("slot", { name: "image-top" }))), h("div", { class: "card-header" }, isSlotUsed(this.el, "icon") && (h("div", { class: "icon" }, h("slot", { name: "icon" }))), h("div", { class: "card-title" }, h("slot", { name: "heading" }, h("ic-typography", { variant: "h4" }, h("p", null, heading)))), isSlotUsed(this.el, "interaction-button") && (h("div", { class: "interaction-button" }, h("slot", { name: "interaction-button" })))), (subheading || isSlotUsed(this.el, "subheading")) && (h("div", { class: "subheading" }, h("slot", { name: "subheading" }, h("ic-typography", { variant: "subtitle-small" }, subheading)))), isSlotUsed(this.el, "adornment") && (h("div", { class: "adornment" }, h("slot", { name: "adornment" }))), isSlotUsed(this.el, "image-mid") && (h("div", { class: "image-mid" }, h("slot", { name: "image-mid" }))), (message || isSlotUsed(this.el, "message")) && (h("div", { class: {
                ["card-message"]: true,
            } }, message && h("ic-typography", { variant: "body" }, message), isSlotUsed(this.el, "message") && h("slot", { name: "message" }))), (isSlotUsed(this.el, "interaction-controls") || expandable) && (h("div", { class: "interaction-area" }, h("div", { class: "interaction-controls" }, h("slot", { name: "interaction-controls" })), expandable && (h("ic-tooltip", { id: "ic-tooltip-expand-button", label: "Toggle expandable area", silent: true }, h("button", { class: {
                ["toggle-button"]: true,
                [`toggle-button-${this.areaExpanded ? "expanded" : "closed"}`]: true,
            }, "aria-label": "Toggle expandable area", "aria-expanded": `${this.areaExpanded}`, "aria-controls": this.areaExpanded ? "expanded-content-area" : null, onClick: this.toggleExpanded, innerHTML: chevronIcon }))))), isSlotUsed(this.el, "expanded-content") && this.areaExpanded && (h("div", { class: "expanded-content", id: "expanded-content-area" }, h("slot", { name: "expanded-content" }))), isSlotUsed(this.el, "badge") && h("slot", { name: "badge" })));
    }
    static get is() { return "ic-card"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["ic-card.css"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["ic-card.css"]
        };
    }
    static get properties() {
        return {
            "clickable": {
                "type": "boolean",
                "mutable": true,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the card will be a clickable variant, instead of static."
                },
                "attribute": "clickable",
                "reflect": false,
                "defaultValue": "false"
            },
            "disabled": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the card will be disabled if it is clickable."
                },
                "attribute": "disabled",
                "reflect": false,
                "defaultValue": "false"
            },
            "expandable": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the card will have an expandable area and expansion toggle button."
                },
                "attribute": "expandable",
                "reflect": false,
                "defaultValue": "false"
            },
            "fullWidth": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the card will fill the width of the container."
                },
                "attribute": "full-width",
                "reflect": false,
                "defaultValue": "false"
            },
            "heading": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The heading for the card. This is required, unless a slotted heading is used."
                },
                "attribute": "heading",
                "reflect": false
            },
            "href": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string | undefined",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The URL that the clickable card link points to. If set, the clickable card will render as an \"a\" tag, otherwise it will render as a button."
                },
                "attribute": "href",
                "reflect": false
            },
            "hreflang": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The human language of the linked URL."
                },
                "attribute": "hreflang",
                "reflect": false,
                "defaultValue": "\"\""
            },
            "message": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The main body message of the card."
                },
                "attribute": "message",
                "reflect": false,
                "defaultValue": "\"\""
            },
            "referrerpolicy": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "ReferrerPolicy",
                    "resolved": "\"\" | \"no-referrer\" | \"no-referrer-when-downgrade\" | \"origin\" | \"origin-when-cross-origin\" | \"same-origin\" | \"strict-origin\" | \"strict-origin-when-cross-origin\" | \"unsafe-url\"",
                    "references": {
                        "ReferrerPolicy": {
                            "location": "global",
                            "id": "global::ReferrerPolicy"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "How much of the referrer to send when following the link."
                },
                "attribute": "referrerpolicy",
                "reflect": false
            },
            "rel": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The relationship of the linked URL as space-separated link types."
                },
                "attribute": "rel",
                "reflect": false
            },
            "subheading": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The subheading for the card."
                },
                "attribute": "subheading",
                "reflect": false
            },
            "target": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The place to display the linked URL, as the name for a browsing context (a tab, window, or iframe)."
                },
                "attribute": "target",
                "reflect": false
            }
        };
    }
    static get states() {
        return {
            "appearance": {},
            "areaExpanded": {},
            "isFocussed": {},
            "parentEl": {},
            "parentIsAnchorTag": {}
        };
    }
    static get methods() {
        return {
            "setFocus": {
                "complexType": {
                    "signature": "() => Promise<void>",
                    "parameters": [],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "Sets focus on the card.",
                    "tags": []
                }
            }
        };
    }
    static get elementRef() { return "el"; }
    static get listeners() {
        return [{
                "name": "click",
                "method": "handleHostClick",
                "target": undefined,
                "capture": true,
                "passive": false
            }, {
                "name": "themeChange",
                "method": "themeChangeHandler",
                "target": "document",
                "capture": false,
                "passive": false
            }];
    }
}
//# sourceMappingURL=ic-card.js.map
