import { h, Host, } from "@stencil/core";
import LeftArrow from "./assets/left-arrow.svg";
import RightArrow from "./assets/right-arrow.svg";
import { getSlotElements } from "../../utils/helpers";
import { checkResizeObserver, elementOverflowsX } from "../../utils/helpers";
import { IcThemeForegroundEnum } from "../../utils/types";
const SCROLL_DELAY_MS = 200;
export class HorizontalScroll {
    constructor() {
        this.buttonStateSet = false;
        this.focusHandler = (event) => {
            this.itemFocusHandler(Array.from(this.items).indexOf(event.target));
        };
        this.resizeObserverCallback = () => {
            if (this.el.clientWidth >= this.itemsContainerEl.scrollWidth) {
                this.itemOverflow = false;
            }
            else {
                this.itemOverflow = elementOverflowsX(this.itemsContainerEl);
            }
            if (this.itemOverflow) {
                this.lastItemVisible =
                    this.itemsContainerEl.offsetWidth + this.itemsContainerEl.scrollLeft >=
                        this.itemsContainerEl.scrollWidth;
            }
        };
        this.runResizeObserver = () => {
            this.resizeObserver = new ResizeObserver(() => {
                this.resizeObserverCallback();
            });
            this.resizeObserver.observe(this.itemsContainerEl);
        };
        this.scrollLeft = () => {
            this.scrollItemIntoView(this.getCurrentLeftItem() - 1);
        };
        this.scrollRight = () => {
            this.scrollItemIntoView(this.getCurrentLeftItem() + 1);
        };
        this.longScrollRight = () => {
            this.scrollRight();
            this.scrollDelay = window.setTimeout(this.longScrollRight, SCROLL_DELAY_MS);
        };
        this.longScrollLeft = () => {
            this.scrollLeft();
            this.scrollDelay = window.setTimeout(this.longScrollLeft, SCROLL_DELAY_MS);
        };
        this.leftArrowMouseDownHandler = (e) => {
            e.preventDefault();
            this.scrollDelay = window.setTimeout(this.longScrollLeft, SCROLL_DELAY_MS);
        };
        this.rightArrowMouseDownHandler = (e) => {
            e.preventDefault();
            this.scrollDelay = window.setTimeout(this.longScrollRight, SCROLL_DELAY_MS);
        };
        this.arrowMouseUpHandler = () => {
            window.clearTimeout(this.scrollDelay);
        };
        this.getCurrentLeftItem = () => {
            const index = this.itemOffsets.findIndex((el) => el > Math.round(this.itemsContainerEl.scrollLeft));
            return index < 0 ? 0 : index;
        };
        this.scrollHandler = () => {
            window.clearTimeout(this.isScrolling);
            // Set a timeout to run after scrolling ends
            this.isScrolling = window.setTimeout(this.scrollStopped, 50);
        };
        this.scrollStopped = () => {
            // If scrollItemIntoView has been called, the button states will already be set
            // Can't just handle it here as it causes strange jumping behaviour in positioning
            const scrollLeft = Math.round(this.itemsContainerEl.scrollLeft);
            if (this.buttonStateSet === false) {
                this.firstItemVisible = scrollLeft === 0;
                this.lastItemVisible =
                    this.itemsContainerEl.offsetWidth + scrollLeft >=
                        this.itemsContainerEl.scrollWidth;
            }
            this.buttonStateSet = false;
        };
        this.firstItemVisible = true;
        this.itemOverflow = false;
        this.lastItemVisible = false;
        this.appearance = "default";
        this.focusTrigger = "focus";
    }
    componentWillLoad() {
        this.itemsContainerEl = this.el.children[0];
        this.itemsContainerEl.addEventListener("scroll", this.scrollHandler);
        this.items = getSlotElements(this.itemsContainerEl);
        this.items.forEach((item) => {
            if (item.addEventListener) {
                item.addEventListener(this.focusTrigger, this.focusHandler);
            }
        });
    }
    componentDidLoad() {
        let runningTotal = 0;
        this.itemOffsets = this.items.map((item) => {
            runningTotal += item.offsetWidth;
            return runningTotal;
        });
        checkResizeObserver(this.runResizeObserver);
        // Add event listener to scroll containers as mouse events are not fired on disabled elements (ic-button's <button>)
        // 'mouseleave' needed in case the user moves their mouse while holding the arrow buttons
        // - 'mouseup' otherwise not detected and scrolling not stopped
        const scrollArrows = Array.from(this.el.shadowRoot.querySelectorAll("div"));
        ["mouseup", "mouseleave"].forEach((event) => {
            scrollArrows.forEach((arrow) => arrow.addEventListener(event, this.arrowMouseUpHandler));
        });
    }
    disconnectedCallback() {
        if (this.resizeObserver !== undefined) {
            this.resizeObserver.disconnect();
        }
        const scrollArrows = Array.from(this.el.shadowRoot.querySelectorAll("div"));
        ["mouseup", "mouseleave"].forEach((event) => {
            scrollArrows.forEach((arrow) => arrow.removeEventListener(event, this.arrowMouseUpHandler));
        });
        this.items.forEach((item) => {
            if (item.removeEventListener) {
                item.removeEventListener(this.focusTrigger, this.focusHandler);
            }
        });
        this.itemsContainerEl.removeEventListener("scroll", this.scrollHandler);
    }
    /**
     * @internal if side scrolling enabled, scrolls the specified item into view.
     */
    async scrollItemIntoView(itemPosition) {
        this.firstItemVisible = itemPosition <= 0;
        const newScrollPos = itemPosition <= 0 ? 0 : this.itemOffsets[itemPosition - 1];
        this.lastItemVisible =
            this.itemsContainerEl.offsetWidth + newScrollPos >=
                this.itemsContainerEl.scrollWidth;
        this.buttonStateSet = true;
        this.itemsContainerEl.scrollLeft = newScrollPos;
    }
    itemFocusHandler(itemPosition) {
        if (this.itemOverflow) {
            this.scrollItemIntoView(itemPosition);
        }
    }
    render() {
        const { appearance, firstItemVisible, lastItemVisible, itemOverflow } = this;
        return (h(Host, { class: {
                ["visible"]: itemOverflow,
                ["dark"]: this.appearance === IcThemeForegroundEnum.Dark,
                ["light"]: this.appearance === IcThemeForegroundEnum.Light,
            } }, h("div", { "aria-hidden": "true", class: {
                ["scroll-container-left"]: true,
                ["hidden"]: !itemOverflow,
                ["disabled"]: firstItemVisible,
            }, role: "tab" }, h("ic-button", { class: "scroll-arrow", variant: "icon", "aria-label": "Scroll left", appearance: appearance, innerHTML: LeftArrow, disabled: firstItemVisible, tabindex: "-1", onClick: this.scrollLeft, onMouseDown: this.leftArrowMouseDownHandler }), h("span", { class: "scroll-splitter-left" })), h("slot", null), h("div", { "aria-hidden": "true", class: {
                ["scroll-container-right"]: true,
                ["hidden"]: !itemOverflow,
                ["disabled"]: lastItemVisible,
            }, role: "tab" }, h("span", { class: "scroll-splitter-right" }), h("ic-button", { class: "scroll-arrow", variant: "icon", "aria-label": "Scroll right", appearance: appearance, innerHTML: RightArrow, disabled: lastItemVisible, tabindex: "-1", onClick: this.scrollRight, onMouseDown: this.rightArrowMouseDownHandler }))));
    }
    static get is() { return "ic-horizontal-scroll"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["./ic-horizontal-scroll.css"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["ic-horizontal-scroll.css"]
        };
    }
    static get properties() {
        return {
            "appearance": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcThemeForeground",
                    "resolved": "\"dark\" | \"default\" | \"light\"",
                    "references": {
                        "IcThemeForeground": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcThemeForeground"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The appearance of the horizontal scroll, e.g. dark, light or the default."
                },
                "attribute": "appearance",
                "reflect": false,
                "defaultValue": "\"default\""
            },
            "focusTrigger": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": "The name of the event that triggers focus handler logic."
                        }],
                    "text": ""
                },
                "attribute": "focus-trigger",
                "reflect": false,
                "defaultValue": "\"focus\""
            }
        };
    }
    static get states() {
        return {
            "firstItemVisible": {},
            "itemOverflow": {},
            "lastItemVisible": {}
        };
    }
    static get methods() {
        return {
            "scrollItemIntoView": {
                "complexType": {
                    "signature": "(itemPosition: number) => Promise<void>",
                    "parameters": [{
                            "name": "itemPosition",
                            "type": "number",
                            "docs": ""
                        }],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "",
                    "tags": [{
                            "name": "internal",
                            "text": "if side scrolling enabled, scrolls the specified item into view."
                        }]
                }
            }
        };
    }
    static get elementRef() { return "el"; }
}
//# sourceMappingURL=ic-horizontal-scroll.js.map
