import { Host, h } from "@stencil/core";
import { isSlotUsed, DEVICE_SIZES, getCurrentDeviceSize, checkResizeObserver, } from "../../utils/helpers";
/**
 * @slot breadcrumbs - Content will be rendered in the breadcrumb area, above the title and action areas.
 * @slot heading - Content will be rendered in the title area, in place of the heading.
 * @slot subheading - Content will be rendered in the title area, in place of the subheading.
 * @slot heading-adornment - Content will be rendered in the title area, adjacent to the heading.
 * @slot actions - Content will be rendered in the action area, adjacent to the title area.
 * @slot input - Content will be rendered in the input area, below the title area and adjacent to the action area.
 * @slot stepper - Content will be rendered in the navigation area, below the title and action areas. Note: stepper slot cannot be used when the tabs slot is being used.
 * @slot tabs - Content will be rendered in the navigation area, below the title and action areas. Note: the stepper slot cannot be used when the tabs slot is being used.
 */
export class PageHeader {
    constructor() {
        this.resizeObserver = null;
        this.resizeObserverCallback = () => {
            if (this.reverseOrder && isSlotUsed(this.el, "actions")) {
                this.applyReverseOrder();
            }
        };
        this.runResizeObserver = () => {
            this.resizeObserver = new ResizeObserver(() => {
                this.resizeObserverCallback();
            });
            this.resizeObserver.observe(this.el);
        };
        this.applyReverseOrder = () => {
            const currSize = getCurrentDeviceSize();
            if (currSize !== this.deviceSize) {
                this.deviceSize = currSize;
            }
            const actionArea = this.el.shadowRoot.querySelector("div.action-area");
            const actionHeights = [];
            /* For each button in the actions slot, check the height of the button and push this number into the actionHeights array and then find the greatest number in this array (i.e. find the height of the tallest button) */
            for (let i = 0; i < this.actionContent.length; i++) {
                const action = this.actionContent[i];
                let actionHeight = action.offsetHeight;
                if (actionHeight === undefined) {
                    actionHeight = 0;
                }
                actionHeights.push(actionHeight);
            }
            const max = Math.max(...actionHeights);
            /* The smallest breakpoint displays buttons in the same order the user has slotted them in. When the device size is larger than the small breakpoint then the order of the button reverses due to best practice for reading patterns. If the device size is larger than the small breakpoint and the buttons have started to wrap then they follow the order the user has slotted them in. This logic is programmed based on the height of the action-area div that wraps the actions slot. If the height of the action-area div is taller than the height of the tallest button, then it is assumed that the taller height is caused by the buttons wrapping */
            let actionAreaHeight = actionArea.offsetHeight;
            if (actionAreaHeight === undefined) {
                actionAreaHeight = 0;
            }
            const appendActionContent = () => {
                this.actionContent = this.actionContent.reverse();
                this.actionContent.forEach((btn) => {
                    this.el.append(btn);
                });
                this.areButtonsReversed = !this.areButtonsReversed;
            };
            if ((this.deviceSize > DEVICE_SIZES.S &&
                actionAreaHeight <= max &&
                !this.areButtonsReversed) ||
                (((this.deviceSize > DEVICE_SIZES.S && actionAreaHeight > max) ||
                    this.deviceSize <= DEVICE_SIZES.S) &&
                    this.areButtonsReversed)) {
                appendActionContent();
            }
        };
        this.actionContent = undefined;
        this.areButtonsReversed = false;
        this.deviceSize = DEVICE_SIZES.XL;
        this.aligned = "left";
        this.border = true;
        this.heading = undefined;
        this.reverseOrder = false;
        this.size = "default";
        this.small = false;
        this.sticky = false;
        this.stickyDesktopOnly = false;
        this.subheading = undefined;
    }
    disconnectedCallback() {
        if (this.resizeObserver !== null) {
            this.resizeObserver.disconnect();
        }
    }
    componentWillLoad() {
        this.actionContent = Array.from(this.el.querySelectorAll(`[slot="actions"]`));
    }
    componentDidLoad() {
        checkResizeObserver(this.runResizeObserver);
    }
    render() {
        const { small, size, border, heading, subheading, aligned, sticky, stickyDesktopOnly, } = this;
        const navAriaLabel = heading
            ? `${heading} page sections`
            : "navigation-landmark-page-header";
        return (h(Host, { class: {
                ["sticky"]: sticky,
                ["sticky-desktop"]: !sticky && stickyDesktopOnly,
            }, "aria-label": this.el.ariaLabel || "page header" }, h("header", { class: {
                ["border-bottom"]: border,
                ["tabs"]: isSlotUsed(this.el, "tabs"),
            }, role: "presentation" }, h("ic-section-container", { aligned: aligned, fullHeight: isSlotUsed(this.el, "tabs") }, isSlotUsed(this.el, "breadcrumbs") && (h("div", { class: "breadcrumb-area" }, h("slot", { name: "breadcrumbs" }))), h("div", { class: "main-content" }, h("div", { class: "title-area" }, h("div", { class: "header-content" }, h("slot", { name: "heading" }, h("ic-typography", { variant: small || size === "small" ? "h4" : "h2", class: "heading" }, h("h2", null, heading))), h("slot", { name: "heading-adornment" })), h("div", null, h("slot", { name: "subheading" }, subheading && (h("ic-typography", { variant: "body", class: {
                ["subheading"]: true,
                ["small"]: small || size === "small",
            } }, subheading))))), isSlotUsed(this.el, "actions") && (h("div", { class: "action-area" }, h("slot", { name: "actions" }))), isSlotUsed(this.el, "input") && (h("div", { class: "input-area" }, h("slot", { name: "input" })))), (isSlotUsed(this.el, "stepper") ||
            isSlotUsed(this.el, "tabs")) && (h("div", { class: "navigation-area" }, isSlotUsed(this.el, "stepper") &&
            !isSlotUsed(this.el, "tabs") && h("slot", { name: "stepper" }), isSlotUsed(this.el, "tabs") && (h("nav", { "aria-label": navAriaLabel }, h("ic-horizontal-scroll", null, h("ul", { class: "tabs-slot" }, h("slot", { name: "tabs" })))))))))));
    }
    static get is() { return "ic-page-header"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["ic-page-header.css"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["ic-page-header.css"]
        };
    }
    static get properties() {
        return {
            "aligned": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcAlignment",
                    "resolved": "\"center\" | \"full-width\" | \"left\"",
                    "references": {
                        "IcAlignment": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcAlignment"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The alignment of the page header."
                },
                "attribute": "aligned",
                "reflect": false,
                "defaultValue": "\"left\""
            },
            "border": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, a border will be displayed along the bottom of the page header."
                },
                "attribute": "border",
                "reflect": false,
                "defaultValue": "true"
            },
            "heading": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The title to render on the page header."
                },
                "attribute": "heading",
                "reflect": false
            },
            "reverseOrder": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the reading pattern and tab order will change in the action area for viewport widths of above 576px and when actions have not wrapped."
                },
                "attribute": "reverse-order",
                "reflect": false,
                "defaultValue": "false"
            },
            "size": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcSizesNoLarge",
                    "resolved": "\"default\" | \"small\"",
                    "references": {
                        "IcSizesNoLarge": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcSizesNoLarge"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The size of the page header component."
                },
                "attribute": "size",
                "reflect": false,
                "defaultValue": "\"default\""
            },
            "small": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [{
                            "name": "deprecated",
                            "text": "This prop should not be used anymore. Set prop `size` to \"small\" instead."
                        }],
                    "text": ""
                },
                "attribute": "small",
                "reflect": false,
                "defaultValue": "false"
            },
            "sticky": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the page header will be sticky at all breakpoints."
                },
                "attribute": "sticky",
                "reflect": false,
                "defaultValue": "false"
            },
            "stickyDesktopOnly": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the page header will only be sticky for viewport widths of 992px and above."
                },
                "attribute": "sticky-desktop-only",
                "reflect": false,
                "defaultValue": "false"
            },
            "subheading": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The subtitle to render on the page header."
                },
                "attribute": "subheading",
                "reflect": false
            }
        };
    }
    static get states() {
        return {
            "actionContent": {},
            "areButtonsReversed": {},
            "deviceSize": {}
        };
    }
    static get elementRef() { return "el"; }
}
//# sourceMappingURL=ic-page-header.js.map
