import { Host, h, } from "@stencil/core";
import { getSlotElements, isPropDefined } from "../../utils/helpers";
import { createPopper } from "@popperjs/core";
export class PopoverMenu {
    constructor() {
        this.ARIA_LABEL = "aria-label";
        this.popoverMenuEls = [];
        this.setButtonFocus = () => {
            var _a;
            (_a = this.popoverMenuEls[this.currentFocus]) === null || _a === void 0 ? void 0 : _a.focus();
        };
        // Checks that the popover menu has an anchor
        this.findAnchorEl = (anchor) => {
            let anchorElement = null;
            if (!anchor) {
                this.submenuId === undefined &&
                    console.error("No anchor specified for popover component");
            }
            else {
                anchorElement = document.querySelector(anchor.indexOf("#") === 0 ? anchor : "#" + anchor);
                if (anchorElement === null) {
                    console.error(`Popover anchor element '${anchor}' not found`);
                }
            }
            return anchorElement;
        };
        this.isNotPopoverMenuEl = (ev) => {
            const { id, tagName } = ev.target;
            return (id !== this.anchor &&
                tagName !== "IC-MENU-ITEM" &&
                tagName !== "IC-MENU-GROUP" &&
                tagName !== "IC-POPOVER-MENU");
        };
        this.getNextItemToSelect = (currentItem, movingDown) => {
            const numButtons = this.popoverMenuEls.length - 1;
            if (currentItem < 1) {
                currentItem = 0;
            }
            let nextItem = movingDown ? currentItem + 1 : currentItem - 1;
            // Check if wrap around necessary
            if (nextItem < 0) {
                nextItem = numButtons;
            }
            else if (nextItem > numButtons) {
                nextItem = 0;
            }
            return nextItem;
        };
        this.addMenuItems = (elements) => {
            for (let i = 0; i < elements.length; i++) {
                const el = elements[i];
                if (el.tagName === "IC-MENU-ITEM") {
                    this.popoverMenuEls.push(el);
                }
                else if (el.tagName === "IC-MENU-GROUP") {
                    const groupSlotWrapper = el.shadowRoot.querySelector("ul");
                    const menuGroupElements = getSlotElements(groupSlotWrapper);
                    this.addMenuItems(menuGroupElements);
                }
            }
        };
        this.getMenuAriaLabel = () => {
            const ariaLabel = this.el.getAttribute(this.ARIA_LABEL);
            if (isPropDefined(this.submenuId)) {
                return `${ariaLabel}, within nested level ${this.submenuLevel} ${this.parentLabel} submenu,`;
            }
            else {
                return ariaLabel;
            }
        };
        this.handleBackButtonClick = () => {
            this.parentPopover.openFromChild();
            this.open = false;
        };
        this.initPopperJS = () => {
            this.popperInstance = createPopper(this.anchorEl, this.el, {
                placement: "bottom-start",
                modifiers: [
                    {
                        name: "offset",
                        options: {
                            offset: [0, 4],
                        },
                    },
                    {
                        name: "flip",
                        options: {
                            fallbackPlacements: ["top-start", "top-end", "bottom-end"],
                            rootBoundary: "viewport",
                        },
                    },
                ],
            });
        };
        this.openingFromChild = false;
        this.openingFromParent = false;
        this.anchor = undefined;
        this.parentLabel = undefined;
        this.parentPopover = undefined;
        this.submenuId = undefined;
        this.submenuLevel = 1;
        this.open = undefined;
    }
    watchOpenHandler() {
        if (this.open) {
            if (this.parentPopover !== undefined &&
                !this.popoverMenuEls.some((menuItem) => menuItem.id)) {
                this.popoverMenuEls.unshift(this.backButton);
            }
            this.currentFocus = isPropDefined(this.submenuId) ? 1 : 0;
            // Needed so that anchorEl isn't always focused
            setTimeout(this.setButtonFocus, 50);
        }
        else if (this.popperInstance) {
            this.popperInstance.destroy();
            this.popperInstance = null;
        }
    }
    disconnectedCallback() {
        if (this.popperInstance) {
            this.popperInstance.destroy();
            this.popperInstance = null;
        }
    }
    componentDidLoad() {
        const slotWrapper = this.el.shadowRoot.querySelector("ul.button");
        const popoverMenuElements = getSlotElements(slotWrapper);
        if (popoverMenuElements !== null) {
            this.addMenuItems(popoverMenuElements);
        }
        if (this.submenuId === undefined &&
            this.el.getAttribute(this.ARIA_LABEL) === null) {
            console.error(`No aria-label specified for popover menu component - aria-label required`);
        }
    }
    componentWillRender() {
        this.anchorEl = this.findAnchorEl(this.anchor);
    }
    componentDidRender() {
        if (this.open && !this.popperInstance) {
            this.initPopperJS();
        }
    }
    handleMenuItemClick(ev) {
        if (!ev.detail.hasSubMenu && ev.detail.label !== "Back") {
            this.closeMenu();
        }
    }
    // This is listening for the event emitted when a menu item is acting as a trigger button
    handleSubmenuChange(ev) {
        // Finds the trigger menu item that has emitted the event
        const target = ev.target;
        this.open = false;
        // Find the popover menu that the menu item triggers (i.e. submenu-trigger-for === submenu-id).
        const childEl = document.querySelector(`ic-popover-menu[submenu-id=${target.submenuTriggerFor}]`);
        // Set the parent popover menu of the submenu and open the submenu
        childEl.parentPopover = this.el;
        childEl.anchor = this.anchor;
        childEl.ariaLabel = this.el.getAttribute(this.ARIA_LABEL);
        childEl.openFromParent();
        childEl.submenuLevel = this.submenuLevel + 1;
        // Set the label in the submenu using the label of the menu item that has emitted the event
        childEl.parentLabel = target.label;
    }
    handleClick(ev) {
        if (this.open && this.isNotPopoverMenuEl(ev)) {
            // If menu is open and the next click on the document is not a popover El, close the popover
            this.closeMenu();
        }
    }
    // Manages the keyboard navigation in the popover menu
    handleKeyDown(ev) {
        switch (ev.key) {
            case "ArrowDown":
            case "ArrowUp":
                ev.preventDefault();
                this.currentFocus = this.getNextItemToSelect(this.currentFocus, ev.key === "ArrowDown");
                this.setButtonFocus();
                break;
            case "Home":
                //Sets home focus as first element, or back button
                this.currentFocus = 0;
                this.setButtonFocus();
                break;
            case "End":
                //Sets end focus as last element
                this.currentFocus = this.popoverMenuEls.length - 1;
                this.setButtonFocus();
                break;
            case "Escape":
            case "Tab":
                ev.preventDefault();
                if (this.open) {
                    this.closeMenu(true);
                    this.el.blur();
                }
                break;
        }
    }
    /**
     * @internal Opens the menu from the child menu.
     */
    async openFromChild() {
        this.open = true;
        this.openingFromChild = true;
        setTimeout(() => (this.openingFromChild = false), 1000);
    }
    /**
     * @internal Opens the menu from the parent menu.
     */
    async openFromParent() {
        this.open = true;
        this.openingFromParent = true;
        setTimeout(() => (this.openingFromParent = false), 1000);
    }
    /**
     * @internal Close the menu, emit icPopoverClosed of the root popover
     * @param setFocusToAnchor when true return focus to anchor element when menu is closed
     */
    async closeMenu(setFocusToAnchor = false) {
        var _a;
        this.open = false;
        if (this.parentPopover) {
            this.parentPopover.closeMenu(setFocusToAnchor);
        }
        else {
            if (setFocusToAnchor) {
                (_a = this.anchorEl) === null || _a === void 0 ? void 0 : _a.focus();
            }
            this.icPopoverClosed.emit();
        }
    }
    render() {
        return (h(Host, { class: { open: this.open } }, h("div", { id: this.parentPopover === undefined
                ? `ic-popover-submenu-${this.submenuId}`
                : "", class: {
                menu: true,
            }, tabindex: open ? "0" : "-1" }, h("div", { class: {
                "opening-from-parent": this.openingFromParent,
                "opening-from-child": this.openingFromChild,
            } }, isPropDefined(this.submenuId) && (h("div", null, h("ic-menu-item", { class: "ic-popover-submenu-back-button", ref: (el) => (this.backButton = el), label: "Back", onClick: this.handleBackButtonClick, id: `ic-popover-submenu-back-button-${this.submenuLevel}` }, h("svg", { slot: "icon", viewBox: "0 0 24 24", fill: "none", xmlns: "http://www.w3.org/2000/svg", class: "submenu-back-icon" }, h("path", { d: "M20 11H7.83L13.42 5.41L12 4L4 12L12 20L13.41 18.59L7.83 13H20V11Z", fill: "currentColor" }))), h("ic-typography", { variant: "subtitle-small", class: "parent-label" }, this.parentLabel))), h("ul", { class: "button", "aria-label": this.getMenuAriaLabel(), role: "menu", "aria-owns": isPropDefined(this.submenuId)
                ? `ic-popover-submenu-back-button-${this.submenuLevel}`
                : false, "aria-controls": isPropDefined(this.submenuId)
                ? `ic-popover-submenu-back-button-${this.submenuLevel}`
                : false }, h("slot", null))))));
    }
    static get is() { return "ic-popover-menu"; }
    static get encapsulation() { return "shadow"; }
    static get delegatesFocus() { return true; }
    static get originalStyleUrls() {
        return {
            "$": ["ic-popover-menu.css"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["ic-popover-menu.css"]
        };
    }
    static get properties() {
        return {
            "anchor": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The ID of the element the popover menu will anchor itself to. This is required unless the popover is a submenu."
                },
                "attribute": "anchor",
                "reflect": false
            },
            "parentLabel": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": "The parent popover menu of a child popover menu."
                        }],
                    "text": ""
                },
                "attribute": "parent-label",
                "reflect": false
            },
            "parentPopover": {
                "type": "unknown",
                "mutable": false,
                "complexType": {
                    "original": "HTMLIcPopoverMenuElement",
                    "resolved": "HTMLIcPopoverMenuElement",
                    "references": {
                        "HTMLIcPopoverMenuElement": {
                            "location": "global",
                            "id": "global::HTMLIcPopoverMenuElement"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": "The parent popover menu of a child popover menu."
                        }],
                    "text": ""
                }
            },
            "submenuId": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The unique identifier for a popover submenu."
                },
                "attribute": "submenu-id",
                "reflect": false
            },
            "submenuLevel": {
                "type": "number",
                "mutable": false,
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": "The level of menu being displayed."
                        }],
                    "text": ""
                },
                "attribute": "submenu-level",
                "reflect": false,
                "defaultValue": "1"
            },
            "open": {
                "type": "boolean",
                "mutable": true,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the popover menu will be displayed."
                },
                "attribute": "open",
                "reflect": true,
                "defaultValue": "undefined"
            }
        };
    }
    static get states() {
        return {
            "openingFromChild": {},
            "openingFromParent": {}
        };
    }
    static get events() {
        return [{
                "method": "icPopoverClosed",
                "name": "icPopoverClosed",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when the popover menu is closed."
                },
                "complexType": {
                    "original": "void",
                    "resolved": "void",
                    "references": {}
                }
            }];
    }
    static get methods() {
        return {
            "openFromChild": {
                "complexType": {
                    "signature": "() => Promise<void>",
                    "parameters": [],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "",
                    "tags": [{
                            "name": "internal",
                            "text": "Opens the menu from the child menu."
                        }]
                }
            },
            "openFromParent": {
                "complexType": {
                    "signature": "() => Promise<void>",
                    "parameters": [],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "",
                    "tags": [{
                            "name": "internal",
                            "text": "Opens the menu from the parent menu."
                        }]
                }
            },
            "closeMenu": {
                "complexType": {
                    "signature": "(setFocusToAnchor?: boolean) => Promise<void>",
                    "parameters": [{
                            "name": "setFocusToAnchor",
                            "type": "boolean",
                            "docs": "when true return focus to anchor element when menu is closed"
                        }],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "",
                    "tags": [{
                            "name": "internal",
                            "text": "Close the menu, emit icPopoverClosed of the root popover"
                        }, {
                            "name": "param",
                            "text": "setFocusToAnchor when true return focus to anchor element when menu is closed"
                        }]
                }
            }
        };
    }
    static get elementRef() { return "el"; }
    static get watchers() {
        return [{
                "propName": "open",
                "methodName": "watchOpenHandler"
            }];
    }
    static get listeners() {
        return [{
                "name": "handleMenuItemClick",
                "method": "handleMenuItemClick",
                "target": undefined,
                "capture": false,
                "passive": false
            }, {
                "name": "triggerPopoverMenuInstance",
                "method": "handleSubmenuChange",
                "target": undefined,
                "capture": true,
                "passive": false
            }, {
                "name": "click",
                "method": "handleClick",
                "target": "document",
                "capture": false,
                "passive": false
            }, {
                "name": "keydown",
                "method": "handleKeyDown",
                "target": undefined,
                "capture": false,
                "passive": false
            }];
    }
}
//# sourceMappingURL=ic-popover-menu.js.map
