import { Host, h, } from "@stencil/core";
import { IcThemeForegroundEnum, } from "../../utils/types";
import { getInputDescribedByText, renderHiddenInput, handleHiddenFormButtonClick, getLabelFromValue, onComponentRequiredPropUndefined, getFilteredMenuOptions, removeDisabledFalse, } from "../../utils/helpers";
import clearIcon from "../../assets/clear-icon.svg";
import searchIcon from "../../assets/search-icon.svg";
let inputIds = 0;
// eslint-disable-next-line @typescript-eslint/no-unused-vars
export class SearchBar {
    constructor() {
        this.assistiveHintEl = null;
        this.hasTimedOut = false;
        this.inputId = `ic-search-bar-input-${inputIds++}`;
        this.menuCloseFromMenuChangeEvent = false;
        this.menuId = `${this.inputId}-menu`;
        this.preLoad = true;
        this.preventSubmit = false;
        this.prevNoOption = false;
        this.retryButtonClick = false;
        this.truncateValue = false;
        this.handleClear = (ev) => {
            const keyboardEvent = ev;
            const mouseEvent = ev;
            if (mouseEvent.type === "click" ||
                keyboardEvent.code === "Enter" ||
                keyboardEvent.code === "Space") {
                this.value = "";
                this.inputEl.value = "";
                this.loading = false;
                clearTimeout(this.timeoutTimer);
                this.filteredOptions = this.options;
                this.el.setFocus();
                this.icClear.emit();
                ev.preventDefault();
                this.preventSubmit = true;
            }
        };
        this.onInput = (ev) => {
            this.value = ev.target.value;
            const noOptions = [
                { [this.labelField]: this.emptyOptionListText, [this.valueField]: "" },
            ];
            if (this.options.length > 0) {
                this.setMenuChange(true);
                this.preLoad = false;
                if (this.disableFilter === false) {
                    const rawFilteredOptions = getFilteredMenuOptions(this.options, false, this.value, "anywhere", this.labelField);
                    this.filteredOptions =
                        rawFilteredOptions.length > 0 ? rawFilteredOptions : noOptions;
                }
            }
            if (!this.showClearButton) {
                this.handleShowClearButton(true);
            }
            this.debounceAriaLiveUpdate();
        };
        this.onInputBlur = (ev) => {
            const value = ev.target.value;
            const nextFocus = ev.relatedTarget;
            this.icInputBlur.emit({ value: value, relatedTarget: nextFocus });
        };
        this.onInputFocus = (ev) => {
            const value = ev.target.value;
            this.icInputFocus.emit({ value: value });
            this.handleShowClearButton(true);
        };
        this.handleClearBlur = (ev) => {
            const nextFocus = ev.relatedTarget;
            this.icClearBlur.emit({ relatedTarget: nextFocus });
            this.clearButtonFocused = false;
        };
        this.handleSubmitSearchBlur = (ev) => {
            const nextFocus = ev.relatedTarget;
            this.icSubmitSearchBlur.emit({ relatedTarget: nextFocus });
            this.searchSubmitFocused = false;
        };
        this.handleMouseDown = (ev) => {
            ev.preventDefault();
        };
        this.handleSubmitSearchFocus = () => {
            this.searchSubmitFocused = true;
        };
        this.handleSubmitSearch = () => {
            this.highlightedValue && (this.value = this.highlightedValue);
            this.highlightedValue = undefined;
            this.icSubmitSearch.emit({ value: this.value });
            const form = this.el.closest("FORM");
            if (this.searchSubmitButton && !!form && !this.preventSubmit) {
                handleHiddenFormButtonClick(form, this.searchSubmitButton);
            }
        };
        this.handleSubmitSearchKeyDown = (ev) => {
            if (ev.key === " ") {
                ev.preventDefault();
                this.handleSubmitSearch();
            }
        };
        this.handleRetry = (ev) => {
            this.retryViaKeyPress = ev.detail.keyPressed === "Enter";
            this.icRetryLoad.emit({ value: ev.detail.value });
            this.triggerLoading();
            this.retryButtonClick = true;
        };
        this.triggerLoading = () => {
            const loadingOption = [
                {
                    [this.labelField]: this.loadingLabel,
                    [this.valueField]: "",
                    loading: true,
                },
            ];
            if (this.filteredOptions !== loadingOption)
                this.filteredOptions = loadingOption;
            if (this.timeout) {
                this.timeoutTimer = window.setTimeout(() => {
                    this.filteredOptions = [
                        {
                            [this.labelField]: this.loadingErrorLabel,
                            [this.valueField]: "",
                            timedOut: true,
                        },
                    ];
                }, this.timeout);
            }
        };
        this.handleOptionSelect = (ev) => {
            if (ev.detail.label === this.emptyOptionListText) {
                this.el.setFocus();
                return;
            }
            this.value = ev.detail.value;
            this.icOptionSelect.emit({ value: this.value });
        };
        this.handleMenuOptionHighlight = (ev) => {
            var _a;
            const optionValue = (_a = ev.detail.optionId) === null || _a === void 0 ? void 0 : _a.replace(`${this.menuId}-`, "");
            optionValue && (this.highlightedValue = optionValue);
            if (ev.detail.optionId) {
                this.ariaActiveDescendant = ev.detail.optionId;
            }
            else {
                this.ariaActiveDescendant = undefined;
            }
        };
        this.handleMenuChange = (ev) => {
            this.setMenuChange(ev.detail.open);
            if (!ev.detail.open) {
                this.handleMenuCloseFromMenuChange(true);
                if (ev.detail.focusInput === undefined || ev.detail.focusInput) {
                    this.el.setFocus();
                }
            }
        };
        this.setMenuChange = (open) => {
            if (this.open !== open) {
                this.open = open;
                this.icMenuChange.emit({ open });
            }
        };
        this.handleHostFocus = () => {
            if (this.options && this.value && !this.menuCloseFromMenuChangeEvent) {
                this.setMenuChange(true);
            }
            this.handleTruncateValue(false);
            this.icSearchBarFocus.emit();
        };
        this.handleHostBlur = (ev) => {
            const nextFocus = ev.relatedTarget;
            if (this.open &&
                this.options &&
                nextFocus !== this.menu &&
                !this.retryViaKeyPress &&
                !this.retryButtonClick) {
                this.setMenuChange(false);
            }
            if (this.retryButtonClick || this.retryViaKeyPress) {
                this.inputEl.setFocus();
            }
            this.handleShowClearButton(false);
            this.handleMenuCloseFromMenuChange(false);
            this.handleTruncateValue(true);
            this.icSearchBarBlur.emit({ relatedTarget: nextFocus, value: this.value });
            this.retryViaKeyPress = false;
            this.retryButtonClick = false;
        };
        this.handleShowClearButton = (visible) => {
            this.showClearButton = visible;
        };
        this.handleFocusClearButton = () => {
            this.clearButtonFocused = true;
        };
        this.handleMenuCloseFromMenuChange = (fromEvent) => {
            this.menuCloseFromMenuChangeEvent = fromEvent;
        };
        this.handleTruncateValue = (truncate) => {
            this.truncateValue = truncate;
        };
        this.renderAssistiveHintEl = () => {
            var _a, _b;
            const input = (_b = (_a = this.el.shadowRoot
                .querySelector("ic-text-field")) === null || _a === void 0 ? void 0 : _a.shadowRoot) === null || _b === void 0 ? void 0 : _b.querySelector(`#${this.inputId}`);
            if (input &&
                Object.keys(input).length > 0 &&
                this.hasOptionsOrFilterDisabled()) {
                this.assistiveHintEl = document.createElement("span");
                this.assistiveHintEl.innerText = this.hintText;
                this.assistiveHintEl.id = `${this.inputId}-assistive-hint`;
                this.assistiveHintEl.style.display = "none";
                if (input.after !== undefined) {
                    input.after(this.assistiveHintEl);
                }
            }
        };
        this.updateSearchResultAriaLive = () => {
            const searchResultsStatusEl = this.el.shadowRoot.querySelector(".search-results-status");
            if (searchResultsStatusEl) {
                if (!this.open ||
                    this.value === "" ||
                    this.value.length < this.charactersUntilSuggestion) {
                    searchResultsStatusEl.innerText = "";
                }
                else if (this.hasOptionsOrFilterDisabled() &&
                    this.filteredOptions.length > 0 &&
                    this.open &&
                    !this.filteredOptions[0].loading) {
                    if (this.hadNoOptions()) {
                        searchResultsStatusEl.innerText = this.emptyOptionListText;
                    }
                    else {
                        searchResultsStatusEl.innerText = `${this.filteredOptions.length} result${this.filteredOptions.length > 1 ? "s" : ""} available`;
                    }
                }
            }
        };
        this.hasOptionsOrFilterDisabled = () => this.options.length > 0 || this.disableFilter;
        this.hadNoOptions = () => this.filteredOptions.length === 1 &&
            this.filteredOptions[0][this.labelField] === this.emptyOptionListText &&
            this.searchMode === "navigation";
        this.isSubmitDisabled = () => {
            const valueNotSet = this.value === undefined || this.value === null || this.value === "";
            const valueLengthLess = this.value.length < this.charactersUntilSuggestion;
            return (valueNotSet ||
                valueLengthLess ||
                this.disabled ||
                this.hadNoOptions() ||
                this.hasTimedOut ||
                this.loading);
        };
        this.highlightFirstOptionAfterNoResults = () => {
            if (this.prevNoOption && this.menu && !this.hasTimedOut) {
                this.menu.handleSetFirstOption();
                this.prevNoOption = false;
            }
            const prevNoOptionsList = this.filteredOptions.find((filteredOption) => filteredOption[this.labelField] === this.emptyOptionListText ||
                filteredOption[this.labelField] === this.loadingErrorLabel ||
                filteredOption[this.labelField] === this.loadingLabel);
            if (prevNoOptionsList) {
                this.prevNoOption = true;
            }
        };
        this.ariaActiveDescendant = undefined;
        this.clearButtonFocused = false;
        this.highlightedValue = undefined;
        this.open = false;
        this.searchSubmitFocused = false;
        this.showClearButton = false;
        this.autocapitalize = "off";
        this.autocomplete = "off";
        this.autocorrect = "off";
        this.autofocus = false;
        this.charactersUntilSuggestion = 2;
        this.disabled = false;
        this.disableFilter = false;
        this.debounce = 0;
        this.emptyOptionListText = "No results found";
        this.focusOnLoad = false;
        this.fullWidth = false;
        this.helperText = "";
        this.hideLabel = false;
        this.hintText = "When autocomplete results are available use the up and down arrows to choose and press enter to select";
        this.label = undefined;
        this.labelField = "label";
        this.loading = false;
        this.loadingErrorLabel = "Loading Error";
        this.loadingLabel = "Loading...";
        this.name = this.inputId;
        this.placeholder = "Search";
        this.readonly = false;
        this.required = false;
        this.searchMode = "navigation";
        this.size = "default";
        this.small = false;
        this.spellcheck = false;
        this.timeout = undefined;
        this.valueField = "value";
        this.filteredOptions = [];
        this.options = [];
        this.value = "";
    }
    loadingHandler(newValue) {
        if (newValue && !this.hasTimedOut) {
            this.preLoad = false;
            this.triggerLoading();
        }
    }
    filteredOptionsHandler(newOptions) {
        this.hasTimedOut = newOptions.some((opt) => opt.timedOut);
    }
    watchOptionsHandler(newOptions) {
        if (this.disableFilter && !this.hasTimedOut) {
            this.loading = false;
            clearTimeout(this.timeoutTimer);
            if (newOptions.length > 0) {
                this.filteredOptions = newOptions;
            }
            else {
                if (this.hadNoOptions()) {
                    return;
                }
                this.setMenuChange(true);
                !this.preLoad &&
                    (this.filteredOptions = [
                        {
                            [this.labelField]: this.emptyOptionListText,
                            [this.valueField]: "",
                        },
                    ]);
                this.preLoad = true;
            }
        }
        this.debounceAriaLiveUpdate();
    }
    watchValueHandler(newValue) {
        if (this.inputEl &&
            this.options &&
            !!getLabelFromValue(newValue, this.options, this.valueField, this.labelField)) {
            this.inputEl.value = getLabelFromValue(newValue, this.options, this.valueField, this.labelField);
        }
        else if (this.inputEl && this.inputEl.value !== newValue) {
            this.inputEl.value = newValue;
        }
    }
    disconnectedCallback() {
        if (this.assistiveHintEl) {
            this.assistiveHintEl.remove();
        }
    }
    componentWillLoad() {
        this.watchValueHandler(this.value);
        removeDisabledFalse(this.disabled, this.el);
        if (this.small) {
            this.size = "small";
        }
    }
    componentDidLoad() {
        if (this.focusOnLoad) {
            this.el.setFocus();
        }
        if (this.hasOptionsOrFilterDisabled()) {
            this.renderAssistiveHintEl();
            if (this.disableFilter) {
                this.filteredOptions = this.options;
            }
        }
        onComponentRequiredPropUndefined([{ prop: this.label, propName: "label" }], "Search Bar");
        if (this.inputEl !== undefined) {
            this.anchorEl = this.inputEl.shadowRoot.querySelector("ic-input-component-container");
        }
    }
    componentWillRender() {
        this.highlightFirstOptionAfterNoResults();
    }
    handleKeyDown(ev) {
        const keyEv = ev.detail.event;
        if (this.menu && this.open) {
            this.menu.handleKeyboardOpen(keyEv);
        }
    }
    handleKeyUp(ev) {
        if (ev.key === "Enter") {
            if (this.preventSubmit || this.isSubmitDisabled()) {
                return;
            }
            this.handleSubmitSearch();
            this.setMenuChange(false);
        }
        if (ev.key === "Escape") {
            this.setMenuChange(false);
        }
        if (this.preventSubmit) {
            this.preventSubmit = false;
        }
    }
    /**
     * Sets focus on the native `input`.
     */
    async setFocus() {
        this.retryViaKeyPress = false;
        this.retryButtonClick = false;
        if (this.inputEl) {
            this.inputEl.setFocus();
        }
    }
    debounceAriaLiveUpdate() {
        clearTimeout(this.debounceAriaLive);
        this.debounceAriaLive = window.setTimeout(() => {
            this.updateSearchResultAriaLive();
        }, 500);
    }
    render() {
        const { inputId, name, label, required, small, size, placeholder, helperText, disabled, value, readonly, spellcheck, fullWidth, options, open, hideLabel, menuId, ariaActiveDescendant, truncateValue, autofocus, autocapitalize, autocomplete, filteredOptions, } = this;
        const disabledMode = readonly || disabled;
        const describedBy = getInputDescribedByText(inputId, helperText !== "", false).trim();
        let describedById;
        if (describedBy !== "" && this.hasOptionsOrFilterDisabled()) {
            describedById = `${describedBy} ${this.inputId}-assistive-hint`;
        }
        else if (this.hasOptionsOrFilterDisabled()) {
            describedById = `${this.inputId}-assistive-hint`;
        }
        else if (describedBy !== "") {
            describedById = describedBy;
        }
        else {
            describedById = undefined;
        }
        const hasSuggestedSearch = !!value && this.hasOptionsOrFilterDisabled();
        const menuOpen = hasSuggestedSearch && open && filteredOptions.length > 0;
        const menuRendered = menuOpen && value.length >= this.charactersUntilSuggestion;
        const labelValue = getLabelFromValue(value, options, this.valueField, this.labelField);
        renderHiddenInput(true, this.el, name, value, disabledMode);
        return (h(Host, { class: {
                search: true,
                fullwidth: fullWidth,
                disabled,
                small: size === "small",
            }, onFocus: this.handleHostFocus, onBlur: this.handleHostBlur }, h("ic-text-field", { ref: (el) => (this.inputEl = el), inputId: inputId, label: label, helperText: helperText, required: required, disabled: disabledMode && !readonly, readonly: readonly, size: size, hideLabel: hideLabel, fullWidth: fullWidth, name: name, truncateValue: truncateValue, value: options && !!labelValue ? labelValue : value, placeholder: placeholder, onInput: this.onInput, onBlur: this.onInputBlur, onFocus: this.onInputFocus, "aria-label": hideLabel ? label : "", "aria-describedby": describedById, "aria-owns": menuRendered ? menuId : undefined, "aria-controls": menuRendered ? menuId : undefined, "aria-haspopup": options.length > 0 ? "listbox" : undefined, ariaExpanded: options.length > 0 ? `${menuOpen}` : undefined, ariaActiveDescendant: ariaActiveDescendant, "aria-autocomplete": hasSuggestedSearch ? "list" : undefined, role: options.length > 0 ? "combobox" : undefined, autocomplete: autocomplete, autocapitalize: autocapitalize, autoFocus: autofocus, spellcheck: spellcheck, inputmode: "search", debounce: this.debounce }, h("div", { class: {
                "clear-button-container": true,
                "clear-button-visible": value && !disabledMode && this.showClearButton,
            }, slot: "clear-button" }, h("ic-button", { id: "clear-button", class: "clear-button", "aria-label": "Clear", innerHTML: clearIcon, onClick: this.handleClear, onMouseDown: this.handleMouseDown, size: size, onFocus: this.handleFocusClearButton, onBlur: this.handleClearBlur, onKeyDown: this.handleClear, type: "submit", variant: "icon", appearance: this.clearButtonFocused
                ? IcThemeForegroundEnum.Light
                : IcThemeForegroundEnum.Dark }), h("div", { class: "divider" })), h("div", { class: {
                "search-submit-button-container": true,
                "search-submit-button-disabled": this.isSubmitDisabled(),
            }, slot: "search-submit-button" }, h("ic-button", { id: "search-submit-button", "aria-label": "Search", ref: (el) => (this.searchSubmitButton = el), class: {
                ["search-submit-button"]: true,
                ["search-submit-button-small"]: !!small,
            }, disabled: this.isSubmitDisabled(), innerHTML: searchIcon, size: size, onClick: this.handleSubmitSearch, onMouseDown: this.handleMouseDown, onBlur: this.handleSubmitSearchBlur, onFocus: this.handleSubmitSearchFocus, onKeyDown: this.handleSubmitSearchKeyDown, type: "submit", variant: "icon", appearance: this.searchSubmitFocused
                ? IcThemeForegroundEnum.Light
                : IcThemeForegroundEnum.Default })), h("div", { class: {
                "menu-container": true,
                fullwidth: fullWidth,
            }, slot: "menu" }, menuRendered && (h("ic-menu", { class: {
                "no-results": this.hadNoOptions() ||
                    (filteredOptions.length === 1 &&
                        (filteredOptions[0][this.labelField] ===
                            this.loadingLabel ||
                            filteredOptions[0][this.labelField] ===
                                this.loadingErrorLabel)),
            }, activationType: "manual", anchorEl: this.anchorEl, autoFocusOnSelected: false, searchMode: this.searchMode, inputEl: this.inputEl, inputLabel: label, ref: (el) => (this.menu = el), small: size === "small", fullWidth: fullWidth, menuId: menuId, open: !!menuRendered, options: filteredOptions, onMenuOptionSelect: this.handleOptionSelect, onMenuStateChange: this.handleMenuChange, onMenuOptionId: this.handleMenuOptionHighlight, onRetryButtonClicked: this.handleRetry, parentEl: this.el, value: value, labelField: this.labelField, valueField: this.valueField })))), h("div", { "aria-live": "polite", role: "status", class: "search-results-status" })));
    }
    static get is() { return "ic-search-bar"; }
    static get encapsulation() { return "shadow"; }
    static get delegatesFocus() { return true; }
    static get originalStyleUrls() {
        return {
            "$": ["ic-search-bar.css"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["ic-search-bar.css"]
        };
    }
    static get properties() {
        return {
            "autocapitalize": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The automatic capitalisation of the text value as it is entered/edited by the user.\nAvailable options: \"off\", \"none\", \"on\", \"sentences\", \"words\", \"characters\"."
                },
                "attribute": "autocapitalize",
                "reflect": false,
                "defaultValue": "\"off\""
            },
            "autocomplete": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcAutocompleteTypes",
                    "resolved": "\"name\" | \"on\" | \"off\" | \"honorific-prefix\" | \"given-name\" | \"additional-name\" | \"family-name\" | \"honorific-suffix\" | \"nickname\" | \"email\" | \"username\" | \"new-password\" | \"current-password\" | \"one-time-code\" | \"organization-title\" | \"organization\" | \"street-address\" | \"address-line1\" | \"address-line2\" | \"address-line3\" | \"address-level4\" | \"address-level3\" | \"address-level2\" | \"address-level1\" | \"country\" | \"country-name\" | \"postal-code\" | \"cc-name\" | \"cc-given-name\" | \"cc-additional-name\" | \"cc-family-name\" | \"cc-number\" | \"cc-exp\" | \"cc-exp-month\" | \"cc-exp-year\" | \"cc-csc\" | \"cc-type\" | \"transaction-currency\" | \"transaction-amount\" | \"language\" | \"bday\" | \"bday-day\" | \"bday-month\" | \"bday-year\" | \"sex\" | \"tel\" | \"tel-country-code\" | \"tel-national\" | \"tel-area-code\" | \"tel-local\" | \"tel-extension\" | \"impp\" | \"url\" | \"photo\"",
                    "references": {
                        "IcAutocompleteTypes": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcAutocompleteTypes"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The state of autocompletion the browser can apply on the text value."
                },
                "attribute": "autocomplete",
                "reflect": false,
                "defaultValue": "\"off\""
            },
            "autocorrect": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcAutocorrectStates",
                    "resolved": "\"off\" | \"on\"",
                    "references": {
                        "IcAutocorrectStates": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcAutocorrectStates"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The state of autocorrection the browser can apply when the user is entering/editing the text value."
                },
                "attribute": "autocorrect",
                "reflect": false,
                "defaultValue": "\"off\""
            },
            "autofocus": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the form control will have input focus when the page loads."
                },
                "attribute": "autofocus",
                "reflect": false,
                "defaultValue": "false"
            },
            "charactersUntilSuggestion": {
                "type": "number",
                "mutable": false,
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The number of characters until suggestions appear. The submit button will be disabled until the inputted value is equal to or greater than this number."
                },
                "attribute": "characters-until-suggestion",
                "reflect": false,
                "defaultValue": "2"
            },
            "disabled": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the disabled state will be set."
                },
                "attribute": "disabled",
                "reflect": false,
                "defaultValue": "false"
            },
            "disableFilter": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Specify whether to disable the built in filtering. For example, if options will already be filtered from external source.\nIf `true`, all options provided will be displayed."
                },
                "attribute": "disable-filter",
                "reflect": false,
                "defaultValue": "false"
            },
            "debounce": {
                "type": "number",
                "mutable": false,
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The amount of time, in milliseconds, to wait to trigger the `icChange` event after each keystroke."
                },
                "attribute": "debounce",
                "reflect": false,
                "defaultValue": "0"
            },
            "emptyOptionListText": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The text displayed when there are no options in the option list."
                },
                "attribute": "empty-option-list-text",
                "reflect": false,
                "defaultValue": "\"No results found\""
            },
            "focusOnLoad": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the search bar will be focused when component loaded."
                },
                "attribute": "focus-on-load",
                "reflect": false,
                "defaultValue": "false"
            },
            "fullWidth": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Specify whether the search bar fills the full width of the container.\nIf `true`, this overrides the --input-width CSS variable."
                },
                "attribute": "full-width",
                "reflect": false,
                "defaultValue": "false"
            },
            "helperText": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The helper text that will be displayed for additional field guidance."
                },
                "attribute": "helper-text",
                "reflect": false,
                "defaultValue": "\"\""
            },
            "hideLabel": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the label will be hidden and the required label value will be applied as an aria-label."
                },
                "attribute": "hide-label",
                "reflect": false,
                "defaultValue": "false"
            },
            "hintText": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The hint text for the hidden assistive description element."
                },
                "attribute": "hint-text",
                "reflect": false,
                "defaultValue": "\"When autocomplete results are available use the up and down arrows to choose and press enter to select\""
            },
            "label": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": true,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The label for the search bar."
                },
                "attribute": "label",
                "reflect": false
            },
            "labelField": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The custom name for the label field to correspond with the IcMenuOption type."
                },
                "attribute": "label-field",
                "reflect": false,
                "defaultValue": "\"label\""
            },
            "loading": {
                "type": "boolean",
                "mutable": true,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Trigger loading state when fetching options asynchronously"
                },
                "attribute": "loading",
                "reflect": false,
                "defaultValue": "false"
            },
            "loadingErrorLabel": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Change the message displayed when external loading times out."
                },
                "attribute": "loading-error-label",
                "reflect": false,
                "defaultValue": "\"Loading Error\""
            },
            "loadingLabel": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Change the message displayed whilst the options are being loaded externally."
                },
                "attribute": "loading-label",
                "reflect": false,
                "defaultValue": "\"Loading...\""
            },
            "name": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The name of the control, which is submitted with the form data."
                },
                "attribute": "name",
                "reflect": false,
                "defaultValue": "this.inputId"
            },
            "placeholder": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The placeholder value to display."
                },
                "attribute": "placeholder",
                "reflect": false,
                "defaultValue": "\"Search\""
            },
            "readonly": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the readonly state will be set."
                },
                "attribute": "readonly",
                "reflect": false,
                "defaultValue": "false"
            },
            "required": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the search bar will require a value."
                },
                "attribute": "required",
                "reflect": false,
                "defaultValue": "false"
            },
            "searchMode": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcSearchBarSearchModes",
                    "resolved": "\"navigation\" | \"query\"",
                    "references": {
                        "IcSearchBarSearchModes": {
                            "location": "import",
                            "path": "./ic-search-bar.types",
                            "id": "src/components/ic-search-bar/ic-search-bar.types.ts::IcSearchBarSearchModes"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Specify the mode search bar uses to search. `navigation` allows for quick lookups of a set of values, `query` allows for more general searches."
                },
                "attribute": "search-mode",
                "reflect": false,
                "defaultValue": "\"navigation\""
            },
            "size": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcSizesNoLarge",
                    "resolved": "\"default\" | \"small\"",
                    "references": {
                        "IcSizesNoLarge": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcSizesNoLarge"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The size of the search bar component."
                },
                "attribute": "size",
                "reflect": false,
                "defaultValue": "\"default\""
            },
            "small": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [{
                            "name": "deprecated",
                            "text": "This prop should not be used anymore. Set prop `size` to \"small\" instead."
                        }],
                    "text": ""
                },
                "attribute": "small",
                "reflect": false,
                "defaultValue": "false"
            },
            "spellcheck": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the value of the search will have its spelling and grammar checked."
                },
                "attribute": "spellcheck",
                "reflect": false,
                "defaultValue": "false"
            },
            "timeout": {
                "type": "number",
                "mutable": false,
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If using external filtering, set a timeout for when loading takes too long."
                },
                "attribute": "timeout",
                "reflect": false
            },
            "valueField": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The custom name for the value field to correspond with the IcMenuOption type."
                },
                "attribute": "value-field",
                "reflect": false,
                "defaultValue": "\"value\""
            },
            "options": {
                "type": "unknown",
                "mutable": false,
                "complexType": {
                    "original": "IcMenuOption[]",
                    "resolved": "IcMenuOption[]",
                    "references": {
                        "IcMenuOption": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcMenuOption"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The suggested search options."
                },
                "defaultValue": "[]"
            },
            "value": {
                "type": "string",
                "mutable": true,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The value of the search input."
                },
                "attribute": "value",
                "reflect": true,
                "defaultValue": "\"\""
            }
        };
    }
    static get states() {
        return {
            "ariaActiveDescendant": {},
            "clearButtonFocused": {},
            "highlightedValue": {},
            "open": {},
            "searchSubmitFocused": {},
            "showClearButton": {},
            "filteredOptions": {}
        };
    }
    static get events() {
        return [{
                "method": "icChange",
                "name": "icChange",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when the value has changed."
                },
                "complexType": {
                    "original": "IcValueEventDetail",
                    "resolved": "IcValueEventDetail",
                    "references": {
                        "IcValueEventDetail": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcValueEventDetail"
                        }
                    }
                }
            }, {
                "method": "icClear",
                "name": "icClear",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when value is cleared with clear button"
                },
                "complexType": {
                    "original": "void",
                    "resolved": "void",
                    "references": {}
                }
            }, {
                "method": "icInput",
                "name": "icInput",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when a keyboard input occurred."
                },
                "complexType": {
                    "original": "IcValueEventDetail",
                    "resolved": "IcValueEventDetail",
                    "references": {
                        "IcValueEventDetail": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcValueEventDetail"
                        }
                    }
                }
            }, {
                "method": "icInputBlur",
                "name": "icInputBlur",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [{
                            "name": "deprecated",
                            "text": "This event should not be used anymore. Use icSearchBarBlur instead."
                        }],
                    "text": ""
                },
                "complexType": {
                    "original": "IcSearchBarBlurEventDetail",
                    "resolved": "IcSearchBarBlurEventDetail",
                    "references": {
                        "IcSearchBarBlurEventDetail": {
                            "location": "import",
                            "path": "./ic-search-bar.types",
                            "id": "src/components/ic-search-bar/ic-search-bar.types.ts::IcSearchBarBlurEventDetail"
                        }
                    }
                }
            }, {
                "method": "icInputFocus",
                "name": "icInputFocus",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [{
                            "name": "deprecated",
                            "text": "This event should not be used anymore. Use icSearchBarFocus instead."
                        }],
                    "text": ""
                },
                "complexType": {
                    "original": "IcValueEventDetail",
                    "resolved": "IcValueEventDetail",
                    "references": {
                        "IcValueEventDetail": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcValueEventDetail"
                        }
                    }
                }
            }, {
                "method": "icOptionSelect",
                "name": "icOptionSelect",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when option is highlighted within the menu"
                },
                "complexType": {
                    "original": "IcOptionSelectEventDetail",
                    "resolved": "IcOptionSelectEventDetail",
                    "references": {
                        "IcOptionSelectEventDetail": {
                            "location": "import",
                            "path": "../ic-menu/ic-menu.types",
                            "id": "src/components/ic-menu/ic-menu.types.ts::IcOptionSelectEventDetail"
                        }
                    }
                }
            }, {
                "method": "icMenuChange",
                "name": "icMenuChange",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when the state of the menu changes (i.e. open or close)"
                },
                "complexType": {
                    "original": "IcMenuChangeEventDetail",
                    "resolved": "IcMenuChangeEventDetail",
                    "references": {
                        "IcMenuChangeEventDetail": {
                            "location": "import",
                            "path": "../ic-menu/ic-menu.types",
                            "id": "src/components/ic-menu/ic-menu.types.ts::IcMenuChangeEventDetail"
                        }
                    }
                }
            }, {
                "method": "icClearBlur",
                "name": "icClearBlur",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": "- Emitted when blur is invoked from clear button"
                        }],
                    "text": ""
                },
                "complexType": {
                    "original": "IcBlurEventDetail",
                    "resolved": "IcBlurEventDetail",
                    "references": {
                        "IcBlurEventDetail": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcBlurEventDetail"
                        }
                    }
                }
            }, {
                "method": "icRetryLoad",
                "name": "icRetryLoad",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when the 'retry loading' button is clicked"
                },
                "complexType": {
                    "original": "IcValueEventDetail",
                    "resolved": "IcValueEventDetail",
                    "references": {
                        "IcValueEventDetail": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcValueEventDetail"
                        }
                    }
                }
            }, {
                "method": "icSubmitSearch",
                "name": "icSubmitSearch",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when the search value has been submitted"
                },
                "complexType": {
                    "original": "IcValueEventDetail",
                    "resolved": "IcValueEventDetail",
                    "references": {
                        "IcValueEventDetail": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcValueEventDetail"
                        }
                    }
                }
            }, {
                "method": "icSubmitSearchBlur",
                "name": "icSubmitSearchBlur",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": "- Emitted when blur is invoked from search submit button"
                        }],
                    "text": ""
                },
                "complexType": {
                    "original": "IcBlurEventDetail",
                    "resolved": "IcBlurEventDetail",
                    "references": {
                        "IcBlurEventDetail": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcBlurEventDetail"
                        }
                    }
                }
            }, {
                "method": "icSearchBarBlur",
                "name": "icSearchBarBlur",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when blur is invoked from ic-search-bar"
                },
                "complexType": {
                    "original": "IcSearchBarBlurEventDetail",
                    "resolved": "IcSearchBarBlurEventDetail",
                    "references": {
                        "IcSearchBarBlurEventDetail": {
                            "location": "import",
                            "path": "./ic-search-bar.types",
                            "id": "src/components/ic-search-bar/ic-search-bar.types.ts::IcSearchBarBlurEventDetail"
                        }
                    }
                }
            }, {
                "method": "icSearchBarFocus",
                "name": "icSearchBarFocus",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when focus is invoked from ic-search-bar"
                },
                "complexType": {
                    "original": "void",
                    "resolved": "void",
                    "references": {}
                }
            }];
    }
    static get methods() {
        return {
            "setFocus": {
                "complexType": {
                    "signature": "() => Promise<void>",
                    "parameters": [],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "Sets focus on the native `input`.",
                    "tags": []
                }
            }
        };
    }
    static get elementRef() { return "el"; }
    static get watchers() {
        return [{
                "propName": "loading",
                "methodName": "loadingHandler"
            }, {
                "propName": "filteredOptions",
                "methodName": "filteredOptionsHandler"
            }, {
                "propName": "options",
                "methodName": "watchOptionsHandler"
            }, {
                "propName": "value",
                "methodName": "watchValueHandler"
            }];
    }
    static get listeners() {
        return [{
                "name": "icKeydown",
                "method": "handleKeyDown",
                "target": undefined,
                "capture": false,
                "passive": false
            }, {
                "name": "keyup",
                "method": "handleKeyUp",
                "target": undefined,
                "capture": false,
                "passive": false
            }];
    }
}
//# sourceMappingURL=ic-search-bar.js.map
