import { Host, h, forceUpdate, } from "@stencil/core";
import { getInputDescribedByText, getLabelFromValue, hasValidationStatus, inheritAttributes, onComponentRequiredPropUndefined, renderHiddenInput, isMobileOrTablet, getFilteredMenuOptions, addFormResetListener, removeFormResetListener, removeDisabledFalse, checkSlotInChildMutations, } from "../../utils/helpers";
import { IC_INHERITED_ARIA } from "../../utils/constants";
import { IcInformationStatus, IcThemeForegroundEnum, } from "../../utils/types";
import Expand from "./assets/Expand.svg";
import Clear from "./assets/Clear.svg";
let inputIds = 0;
const MUTABLE_ATTRIBUTES = [...IC_INHERITED_ARIA, "tabindex", "title"];
/**
 * @slot icon - Content will be placed to the left of the select text input.
 */
export class Select {
    constructor() {
        this.hasSetDefaultValue = false;
        this.inheritedAttributes = {};
        this.initialOptionsEmpty = false;
        this.inputId = `ic-select-input-${inputIds++}`;
        this.menuId = `${this.inputId}-menu`;
        this.searchableMenuItemSelected = false;
        this.ungroupedOptions = [];
        this.hostMutationObserver = null;
        // triggered when attributes of host element change
        this.hostMutationCallback = (mutationList) => {
            let forceComponentUpdate = false;
            mutationList.forEach(({ attributeName, type, addedNodes, removedNodes }) => {
                if (MUTABLE_ATTRIBUTES.includes(attributeName)) {
                    this.inheritedAttributes[attributeName] =
                        this.el.getAttribute(attributeName);
                    forceComponentUpdate = true;
                }
                else if (type === "childList") {
                    forceComponentUpdate = checkSlotInChildMutations(addedNodes, removedNodes, "icon");
                }
            });
            if (forceComponentUpdate) {
                forceUpdate(this);
            }
        };
        this.handleRetry = (ev) => {
            var _a;
            if (ev.detail.keyPressed)
                (_a = this.searchableSelectElement) === null || _a === void 0 ? void 0 : _a.focus();
            this.blurredBecauseButtonPressed = true;
            this.retryButtonClick = true;
            this.hasSetDefaultValue = true;
            this.icRetryLoad.emit({ value: this.hiddenInputValue });
        };
        this.emitIcChange = (value) => {
            if (!this.searchable) {
                this.value = value;
            }
            clearTimeout(this.debounceIcChange);
            this.debounceIcChange = window.setTimeout(() => {
                this.icChange.emit({ value: value });
            }, this.currDebounce);
        };
        this.emitImmediateIcChange = (value) => {
            this.value = value;
            clearTimeout(this.debounceIcChange);
            this.icChange.emit({ value: value });
        };
        /**
         * Processes the provided array of IcMenuOptions, removing duplicates and reporting them with a console.warn
         * @param options array of IcMenuOptions
         * @returns a new options object, with all entries possessing a duplicate 'value' field removed
         */
        this.deduplicateOptions = (options) => {
            const uniqueValues = [];
            const dedupedOptions = [];
            let dedupedChildren;
            options.forEach((option) => {
                if (option.children) {
                    //If an option has children, we will loop through them
                    dedupedChildren = [];
                    option.children.forEach((child) => {
                        if (uniqueValues.includes(child.value)) {
                            console.warn(`ic-select with label ${this.label} was populated with duplicate option (value: ${child.value}) which has been removed.`);
                        }
                        else {
                            uniqueValues.push(child.value);
                            dedupedChildren.push(child);
                        }
                    });
                    // construct a modified option, inserting the deduplicated children alongside the original information
                    const modifiedParent = Object.assign(Object.assign({}, option), { children: dedupedChildren });
                    dedupedOptions.push(modifiedParent);
                }
                else {
                    // If an option does not have children, assess to see if it's value has been included already
                    if (uniqueValues.includes(option.value)) {
                        console.warn(`ic-select with label ${this.label} was populated with duplicate option (value: ${option.value}) which has been removed.`);
                    }
                    else {
                        uniqueValues.push(option.value);
                        dedupedOptions.push(option);
                    }
                }
            });
            return dedupedOptions;
        };
        /**
         * Loop through options array and for all options with no value, infer it from the label
         */
        this.setOptionsValuesFromLabels = () => {
            var _a;
            if (((_a = this.options) === null || _a === void 0 ? void 0 : _a.length) > 0 && this.options.map) {
                this.options.map((option) => {
                    if (!option.value) {
                        option.value = option.label;
                    }
                });
            }
        };
        this.setUngroupedOptions = (event) => {
            this.ungroupedOptions = event.detail.options;
        };
        this.setTextColor = () => {
            if (this.nativeSelectElement.selectedIndex === 0) {
                this.nativeSelectElement.className = "placeholder";
            }
            else {
                this.nativeSelectElement.className = "select-option-selected";
            }
        };
        this.setMenuChange = (open) => {
            if (this.open !== open) {
                this.open = open;
            }
        };
        this.getLabelFromValue = (value) => {
            return getLabelFromValue(value, this.uniqueOptions);
        };
        this.getFilteredChildMenuOptions = (option) => {
            let children = option.children;
            if (this.searchable) {
                children = getFilteredMenuOptions(option.children, this.includeDescriptionsInSearch, this.searchableSelectInputValue, this.searchMatchPosition);
            }
            else {
                children = getFilteredMenuOptions(option.children, false, this.pressedCharacters, "start");
            }
            const newOption = Object.assign({}, option);
            newOption.children = children;
            return newOption;
        };
        this.handleNativeSelectChange = () => {
            this.icOptionSelect.emit({ value: this.nativeSelectElement.value });
            this.emitImmediateIcChange(this.nativeSelectElement.value);
            this.setTextColor();
        };
        this.handleCustomSelectChange = (event) => {
            if (this.searchable && event.detail.label === this.emptyOptionListText) {
                this.searchableSelectElement.focus();
                return;
            }
            if (this.searchable) {
                this.value = event.detail.value;
                this.searchableMenuItemSelected = true;
                // After editing the input, if selecting the same option as before, set the input value to label again
                if (this.value === this.currValue) {
                    this.searchableSelectInputValue = this.getLabelFromValue(this.value);
                }
                this.inputValueToFilter = null;
                this.hiddenInputValue = this.getValueFromLabel(this.searchableSelectInputValue);
            }
            this.ariaActiveDescendant = event.detail.optionId;
            this.icOptionSelect.emit({ value: event.detail.value });
            this.emitIcChange(event.detail.value);
        };
        this.handleMenuChange = (event) => {
            this.open = event.detail.open;
            this.pressedCharacters = "";
            this.searchable && this.handleFocusIndicatorDisplay();
        };
        // clears the debounce delay when navigating the menu with arrow keys etc
        // to prevent delay in change event, which should only occur when typing in input
        this.handleMenuKeyPress = (ev) => {
            ev.cancelBubble = true;
            this.handleCharacterKeyDown(ev.detail.key);
        };
        this.handleMenuValueChange = (ev) => {
            this.value = ev.detail.value;
        };
        this.handleFocusIndicatorDisplay = () => {
            const focusIndicator = this.el.shadowRoot.querySelector(".focus-indicator");
            if (this.open) {
                focusIndicator.classList.add("focus-indicator-enabled");
            }
            else {
                focusIndicator.classList.remove("focus-indicator-enabled");
            }
        };
        this.handleMouseDown = (event) => {
            if (!this.open) {
                event.preventDefault();
            }
        };
        this.isExternalFiltering = () => this.searchable && this.disableFilter;
        this.handleClick = (event) => {
            var _a;
            if (!this.open) {
                if (this.isExternalFiltering()) {
                    this.menu.options = this.filteredOptions;
                }
                else if (!this.hasTimedOut &&
                    !this.loading &&
                    !((_a = this.noOptions) === null || _a === void 0 ? void 0 : _a.length) &&
                    (!this.searchable || this.searchableMenuItemSelected)) {
                    this.noOptions = null;
                    this.menu.options = this.uniqueOptions;
                }
            }
            if (event.detail !== 0) {
                this.menu.handleClickOpen();
            }
        };
        this.handleExpandIconMouseDown = (event) => {
            if (!this.disabled) {
                event.preventDefault();
                this.searchableSelectElement.focus();
                this.handleClick(event);
            }
        };
        this.handleClear = (event) => {
            event.stopPropagation();
            this.hasTimedOut = false;
            clearTimeout(this.timeoutTimer);
            this.noOptions = null;
            this.emitImmediateIcChange(null);
            this.icClear.emit();
            if (this.searchable) {
                this.searchableSelectElement.value = null;
                this.searchableSelectInputValue = null;
                this.filteredOptions = this.uniqueOptions;
                this.hiddenInputValue = null;
                this.searchableSelectElement.focus();
            }
            else {
                this.customSelectElement.focus();
            }
        };
        this.handleCharacterKeyDown = (key) => {
            // Only close menu when space is pressed if not being used alongside character keys to quickly select options
            if (this.open &&
                key === " " &&
                this.pressedCharacters.length === 0 &&
                !this.hasTimedOut &&
                !this.loading) {
                this.setMenuChange(false);
            }
            if (key.length === 1 && !this.searchable) {
                window.clearTimeout(this.characterKeyPressTimer);
                this.characterKeyPressTimer = window.setTimeout(() => (this.pressedCharacters = ""), 1000);
                this.pressedCharacters += key;
                this.handleFilter();
                if (!this.noOptions) {
                    this.emitImmediateIcChange(this.filteredOptions[0].value);
                }
            }
            else {
                this.pressedCharacters = "";
            }
        };
        this.handleNativeSelectKeyDown = (event) => {
            if ((event.key !== "Escape" && event.key !== "Tab") || this.open) {
                event.cancelBubble = true;
            }
            this.handleCharacterKeyDown(event.key);
        };
        this.handleKeyDown = (event) => {
            if ((event.key !== "Escape" && event.key !== "Tab") || this.open) {
                event.cancelBubble = true;
            }
            const isArrowKey = event.key === "ArrowDown" || event.key === "ArrowUp";
            if (!this.open) {
                if (this.isExternalFiltering() && (event.key === "Enter" || isArrowKey)) {
                    this.menu.options = this.filteredOptions;
                }
                else {
                    if (!this.hasTimedOut) {
                        this.noOptions = null;
                        this.menu.options = this.uniqueOptions;
                    }
                }
            }
            if (this.open && event.key === "Enter") {
                this.setMenuChange(false);
            }
            else {
                if (!(isArrowKey && this.noOptions !== null)) {
                    if (!(event.key === " " && this.pressedCharacters.length > 0)) {
                        // Keyboard events get passed onto ic-menu
                        this.menu.handleKeyboardOpen(event);
                    }
                    this.handleCharacterKeyDown(event.key);
                }
            }
        };
        this.handleClearButtonFocus = () => {
            this.clearButtonFocused = true;
        };
        this.handleClearButtonBlur = (ev) => {
            var _a;
            const retryButton = (_a = this.menu) === null || _a === void 0 ? void 0 : _a.querySelector("#retry-button");
            if (!(this.searchableSelectElement &&
                ev.relatedTarget === this.searchableSelectElement) &&
                !(retryButton && ev.relatedTarget === retryButton)) {
                this.setMenuChange(false);
                this.handleFocusIndicatorDisplay();
            }
            this.clearButtonFocused = false;
        };
        this.handleFilter = () => {
            var _a;
            const options = this.deduplicateOptions(this.searchable ? [...this.uniqueOptions] : this.ungroupedOptions);
            let isGrouped = false;
            let newFilteredOptions = [];
            options.map((option) => {
                if (option.children)
                    isGrouped = true;
            });
            let menuOptionsFiltered;
            if (this.searchable) {
                menuOptionsFiltered = getFilteredMenuOptions(options, this.includeDescriptionsInSearch, this.inputValueToFilter, this.searchMatchPosition);
                this.searchableMenuItemSelected = false;
            }
            else {
                menuOptionsFiltered = getFilteredMenuOptions(options, false, this.pressedCharacters, "start");
            }
            if (!isGrouped &&
                ((_a = menuOptionsFiltered[0]) === null || _a === void 0 ? void 0 : _a.label) !== this.emptyOptionListText) {
                newFilteredOptions = menuOptionsFiltered;
            }
            else if (isGrouped) {
                options.map((option) => {
                    if (this.includeGroupTitlesInSearch) {
                        if (menuOptionsFiltered.indexOf(option) !== -1) {
                            newFilteredOptions.push(option);
                        }
                        else {
                            newFilteredOptions.push(this.getFilteredChildMenuOptions(option));
                        }
                    }
                    else {
                        newFilteredOptions.push(this.getFilteredChildMenuOptions(option));
                    }
                });
            }
            let noChildOptionsWhenFiltered = false;
            if (isGrouped) {
                noChildOptionsWhenFiltered = true;
                newFilteredOptions.map((option) => {
                    if (option.children.length > 0) {
                        noChildOptionsWhenFiltered = false;
                    }
                });
            }
            if (newFilteredOptions.length > 0 && !noChildOptionsWhenFiltered) {
                this.noOptions = null;
                this.filteredOptions = newFilteredOptions;
            }
            else {
                this.noOptions = [{ label: this.emptyOptionListText, value: "" }];
                this.filteredOptions = this.noOptions;
            }
        };
        /**
         * Put the select component into loading state.
         * Replace options with the loading message. If timeout is enabled, set the timeout and once passed, replace options with the loading error message
         */
        this.triggerLoading = () => {
            this.hasTimedOut = false;
            this.noOptions = [{ label: this.loadingLabel, value: "", loading: true }];
            if (this.filteredOptions !== this.noOptions && this.searchable) {
                this.filteredOptions = this.noOptions;
            }
            else if (this.uniqueOptions !== this.noOptions && !this.searchable) {
                this.uniqueOptions = this.noOptions;
            }
            if (this.timeout) {
                this.timeoutTimer = window.setTimeout(() => {
                    this.loading = false;
                    this.hasTimedOut = true;
                    this.noOptions = [
                        { label: this.loadingErrorLabel, value: "", timedOut: true },
                    ];
                    this.filteredOptions = this.noOptions;
                    if (!this.searchable)
                        this.uniqueOptions = this.noOptions;
                }, this.timeout);
            }
        };
        this.getValueFromLabel = (label) => {
            var _a;
            return (_a = this.uniqueOptions.find((option) => option.label === label)) === null || _a === void 0 ? void 0 : _a.value;
        };
        this.handleSearchableSelectInput = (event) => {
            this.searchableSelectInputValue = event.target.value;
            this.icInput.emit({ value: this.searchableSelectInputValue });
            this.emitIcChange(this.searchableSelectInputValue);
            this.hiddenInputValue = this.searchableSelectInputValue;
            this.inputValueToFilter = this.searchableSelectInputValue;
            this.setMenuChange(true);
            if (!this.disableFilter) {
                this.handleFilter();
                this.debounceAriaLiveUpdate();
            }
        };
        this.updateSearchableSelectResultAriaLive = () => {
            const searchableSelectResultsStatusEl = this.el.shadowRoot.querySelector(".searchable-select-results-status");
            if (searchableSelectResultsStatusEl) {
                if (this.noOptions !== null) {
                    searchableSelectResultsStatusEl.innerText = this.emptyOptionListText;
                }
                else {
                    searchableSelectResultsStatusEl.innerText = "";
                }
            }
        };
        this.getDefaultValue = (value) => this.getLabelFromValue(value) || value || null;
        this.onFocus = () => {
            this.icFocus.emit();
        };
        this.onBlur = ({ relatedTarget }) => {
            var _a;
            const target = relatedTarget;
            if (target !== null &&
                ((target.tagName === "UL" && target.className.includes("menu")) ||
                    (target.tagName === "LI" && target.className.includes("option")))) {
                return;
            }
            const retryButton = (_a = this.menu) === null || _a === void 0 ? void 0 : _a.querySelector("#retry-button");
            const isSearchableAndNoFocusedInternalElements = this.searchable &&
                !!this.menu &&
                target !== this.menu &&
                !Array.from(this.menu.querySelectorAll("[role='option']")).includes(target) &&
                !(this.clearButton && target === this.clearButton) &&
                !(retryButton && target === retryButton);
            if (isSearchableAndNoFocusedInternalElements) {
                if (!this.retryButtonClick) {
                    this.setMenuChange(false);
                }
                this.handleFocusIndicatorDisplay();
            }
            this.retryButtonClick = false;
            this.icBlur.emit();
        };
        this.onTimeoutBlur = (ev) => {
            if (ev.detail.ev.relatedTarget !==
                this.searchableSelectElement &&
                !this.blurredBecauseButtonPressed) {
                this.setMenuChange(false);
                this.handleFocusIndicatorDisplay();
                this.icBlur.emit();
            }
            this.blurredBecauseButtonPressed = false;
        };
        this.handleFormReset = () => {
            this.value = this.initialValue;
            if (this.searchable) {
                this.searchableSelectInputValue = this.getDefaultValue(this.value);
                this.hiddenInputValue = this.value;
            }
        };
        this.ariaActiveDescendant = undefined;
        this.clearButtonFocused = false;
        this.debounceIcChange = undefined;
        this.hiddenInputValue = undefined;
        this.noOptions = null;
        this.open = false;
        this.pressedCharacters = "";
        this.searchableSelectInputValue = null;
        this.charactersUntilSuggestions = 0;
        this.disabled = false;
        this.disableFilter = false;
        this.emptyOptionListText = "No results found";
        this.form = undefined;
        this.formaction = undefined;
        this.formenctype = undefined;
        this.formmethod = undefined;
        this.formnovalidate = undefined;
        this.formtarget = undefined;
        this.fullWidth = false;
        this.helperText = "";
        this.hideLabel = false;
        this.includeDescriptionsInSearch = false;
        this.includeGroupTitlesInSearch = false;
        this.label = undefined;
        this.loadingErrorLabel = "Loading Error";
        this.loadingLabel = "Loading...";
        this.name = this.inputId;
        this.placeholder = "Select an option";
        this.readonly = false;
        this.required = false;
        this.searchable = false;
        this.searchMatchPosition = "anywhere";
        this.selectOnEnter = false;
        this.showClearButton = false;
        this.size = "default";
        this.small = false;
        this.timeout = undefined;
        this.validationStatus = "";
        this.validationText = "";
        this.loading = false;
        this.options = [];
        this.filteredOptions = this.options;
        this.uniqueOptions = this.options;
        this.debounce = 0;
        this.currDebounce = this.debounce;
        this.value = undefined;
        this.initialValue = this.value;
        this.inputValueToFilter = this.value;
        this.currValue = this.value;
    }
    loadingHandler(newValue) {
        newValue && this.triggerLoading();
    }
    watchOptionsHandler() {
        var _a;
        if (!this.hasTimedOut) {
            this.loading = false;
            clearTimeout(this.timeoutTimer);
            if (this.isExternalFiltering()) {
                if (((_a = this.options) === null || _a === void 0 ? void 0 : _a.length) > 0) {
                    this.setOptionsValuesFromLabels();
                    this.noOptions = null;
                    this.uniqueOptions = this.deduplicateOptions(this.options);
                    this.filteredOptions = this.uniqueOptions;
                }
                else {
                    this.noOptions = [{ label: this.emptyOptionListText, value: "" }];
                    this.uniqueOptions = this.noOptions;
                    this.filteredOptions = this.noOptions;
                }
                this.updateSearchableSelectResultAriaLive();
                this.setDefaultValue();
            }
            else {
                this.setOptionsValuesFromLabels();
                this.uniqueOptions = this.deduplicateOptions(this.options);
                this.filteredOptions = this.uniqueOptions;
                if (this.initialOptionsEmpty) {
                    this.setDefaultValue();
                    this.initialOptionsEmpty = false;
                }
            }
        }
        else {
            if (!this.searchable) {
                this.options = this.noOptions;
            }
        }
    }
    debounceChangedHandler(newValue) {
        this.updateOnChangeDebounce(newValue);
    }
    valueChangedHandler() {
        if (this.value !== this.currValue) {
            this.currValue = this.value;
        }
        if (this.searchable) {
            this.searchableSelectInputValue =
                this.getLabelFromValue(this.currValue) || this.currValue;
        }
    }
    openChangedHandler() {
        this.open ? this.icOpen.emit() : this.icClose.emit();
    }
    disconnectedCallback() {
        var _a;
        removeFormResetListener(this.el, this.handleFormReset);
        (_a = this.hostMutationObserver) === null || _a === void 0 ? void 0 : _a.disconnect();
    }
    componentWillLoad() {
        var _a;
        this.inheritedAttributes = inheritAttributes(this.el, MUTABLE_ATTRIBUTES);
        removeDisabledFalse(this.disabled, this.el);
        this.setOptionsValuesFromLabels();
        addFormResetListener(this.el, this.handleFormReset);
        if (!((_a = this.options) === null || _a === void 0 ? void 0 : _a.length)) {
            this.initialOptionsEmpty = true;
            this.noOptions = [{ label: this.emptyOptionListText, value: "" }];
            this.uniqueOptions = this.noOptions;
            this.filteredOptions = this.noOptions;
        }
        else {
            this.setDefaultValue();
            this.uniqueOptions = this.deduplicateOptions(this.options);
        }
    }
    componentDidLoad() {
        onComponentRequiredPropUndefined([{ prop: this.label, propName: "label" }], "Select");
        if (this.loading) {
            this.triggerLoading();
        }
        this.hiddenInputValue = this.searchable && this.currValue;
        this.hostMutationObserver = new MutationObserver(this.hostMutationCallback);
        this.hostMutationObserver.observe(this.el, {
            attributes: true,
            childList: true,
        });
    }
    componentDidRender() {
        if (this.nativeSelectElement && !this.disabled) {
            this.setTextColor();
        }
    }
    /**
     * Sets focus on the input box.
     */
    async setFocus() {
        if (this.nativeSelectElement) {
            this.nativeSelectElement.focus();
        }
        else if (this.customSelectElement) {
            this.customSelectElement.focus();
        }
        else if (this.searchableSelectElement) {
            this.searchableSelectElement.focus();
        }
    }
    updateOnChangeDebounce(newValue) {
        if (this.currDebounce !== newValue) {
            this.currDebounce = newValue;
        }
    }
    debounceAriaLiveUpdate() {
        clearTimeout(this.debounceAria);
        window.setTimeout(() => {
            this.updateSearchableSelectResultAriaLive();
        }, 800);
    }
    setDefaultValue() {
        if (!this.hasSetDefaultValue && this.currValue) {
            this.searchableSelectInputValue = this.getDefaultValue(this.currValue);
            this.initialValue = this.currValue;
            this.hasSetDefaultValue = true;
        }
    }
    render() {
        const { small, size, disabled, fullWidth, helperText, hideLabel, label, menuId, name, options, placeholder, readonly, required, searchable, showClearButton, validationStatus, validationText, currValue, } = this;
        renderHiddenInput(true, this.el, name, this.searchable ? this.hiddenInputValue : currValue, disabled);
        const invalid = `${validationStatus === IcInformationStatus.Error}`;
        const describedBy = getInputDescribedByText(this.inputId, helperText !== "", hasValidationStatus(this.validationStatus, this.disabled)).trim();
        let showLeftIcon = !!this.el.querySelector(`[slot="icon"]`);
        if (showLeftIcon && (disabled || (readonly && !this.value))) {
            showLeftIcon = false;
        }
        return (h(Host, { class: {
                disabled,
                searchable,
                small,
                [size]: size !== "default",
                "full-width": fullWidth,
            }, onBlur: this.onBlur }, h("ic-input-container", { readonly: readonly }, !hideLabel && (h("ic-input-label", { for: this.inputId, label: label, helperText: helperText, required: required, disabled: disabled, readonly: readonly })), h("ic-input-component-container", { ref: (el) => (this.anchorEl = el), class: { "menu-open": this.open }, small: small, size: size, fullWidth: fullWidth, disabled: disabled, readonly: readonly, validationStatus: validationStatus }, showLeftIcon && (h("span", { slot: "left-icon", class: {
                ["readonly"]: readonly,
                ["has-value"]: !!this.value,
            } }, h("slot", { name: "icon" }))), readonly ? (h("ic-typography", null, h("p", null, this.getLabelFromValue(currValue)))) : isMobileOrTablet() ? (h("select", Object.assign({ ref: (el) => (this.nativeSelectElement = el), disabled: disabled, onChange: this.handleNativeSelectChange, required: required, id: this.inputId, "aria-label": label, "aria-describedby": describedBy, "aria-invalid": invalid, onBlur: this.onBlur, onFocus: this.onFocus, onKeyDown: this.handleNativeSelectKeyDown, form: this.form }, this.inheritedAttributes), h("option", { value: "", selected: true, disabled: !showClearButton }, placeholder), options.map((option) => {
            if (option.children) {
                return (h("optgroup", { label: option.label }, option.children.map((option) => (h("option", { value: option.value, disabled: option.disabled, selected: option.value === currValue }, option.label)))));
            }
            else {
                return (h("option", { value: option.value, disabled: option.disabled, selected: option.value === currValue }, option.label));
            }
        }))) : searchable ? (h("div", { class: "searchable-select-container" }, h("input", { class: "select-input", role: "combobox", autocomplete: "off", "aria-label": label, "aria-describedby": describedBy, "aria-activedescendant": this.ariaActiveDescendant, "aria-autocomplete": "list", "aria-expanded": `${this.open}`, "aria-invalid": invalid, "aria-required": `${required}`, "aria-controls": menuId, ref: (el) => (this.searchableSelectElement = el), id: this.inputId, value: this.searchableSelectInputValue, placeholder: placeholder, disabled: disabled, onInput: this.handleSearchableSelectInput, onClick: this.handleClick, onKeyDown: this.handleKeyDown, onFocus: this.onFocus, onBlur: this.onBlur, form: this.form, formaction: this.formaction, formenctype: this.formenctype, formmethod: this.formmethod, formnovalidate: this.formnovalidate, formtarget: this.formtarget }), this.searchableSelectInputValue &&
            (showClearButton || searchable) && (h("div", { class: "clear-button-container" }, h("ic-button", { id: "clear-button", ref: (el) => (this.clearButton = el), "aria-label": this.searchableSelectInputValue && currValue === null
                ? "Clear input"
                : "Clear selection", class: "clear-button", innerHTML: Clear, onClick: this.handleClear, onFocus: this.handleClearButtonFocus, onBlur: this.handleClearButtonBlur, size: size, variant: "icon", appearance: this.clearButtonFocused
                ? IcThemeForegroundEnum.Light
                : IcThemeForegroundEnum.Dark }), h("div", { class: "divider" }))), h("span", { onMouseDown: this.handleExpandIconMouseDown, class: {
                "expand-icon": true,
                "expand-icon-open": this.open,
                "expand-icon-filled": !(currValue == null || currValue === ""),
            }, innerHTML: Expand, "aria-hidden": "true" }), h("div", { "aria-live": "polite", role: "status", class: "searchable-select-results-status" }))) : (h("div", { class: "select-container" }, h("button", { class: "select-input", ref: (el) => (this.customSelectElement = el), id: this.inputId, "aria-label": `${label}, ${this.getLabelFromValue(currValue) || placeholder}${required ? ", required" : ""}`, "aria-describedby": describedBy, "aria-invalid": invalid, "aria-haspopup": "listbox", "aria-expanded": this.open ? "true" : "false", "aria-owns": menuId, "aria-controls": menuId, disabled: disabled, onBlur: this.onBlur, onFocus: this.onFocus, onClick: this.handleClick, onMouseDown: this.handleMouseDown, onKeyDown: this.handleKeyDown }, h("ic-typography", { variant: "body", class: {
                "value-text": true,
                placeholder: this.getLabelFromValue(currValue) === undefined,
            } }, this.getLabelFromValue(currValue) || placeholder), h("div", { class: "select-input-end" }, currValue && showClearButton && (h("div", { class: "divider" })), h("span", { class: {
                "expand-icon": true,
                "expand-icon-open": this.open,
                "expand-icon-filled": !(currValue == null || currValue === ""),
            }, innerHTML: Expand, "aria-hidden": "true" }))), currValue && showClearButton && (h("ic-button", { id: "clear-button", "aria-label": "Clear selection", class: "clear-button", innerHTML: Clear, onClick: this.handleClear, onFocus: this.handleClearButtonFocus, onBlur: this.handleClearButtonBlur, size: size, variant: "icon", appearance: this.clearButtonFocused
                ? IcThemeForegroundEnum.Light
                : IcThemeForegroundEnum.Dark }))))), !isMobileOrTablet() && (h("ic-menu", { class: {
                "no-results": this.loading ||
                    this.hasTimedOut ||
                    (this.noOptions !== null &&
                        this.noOptions[0] &&
                        this.noOptions[0].label === this.emptyOptionListText),
            }, ref: (el) => (this.menu = el), inputEl: searchable
                ? this.searchableSelectElement
                : this.customSelectElement, inputLabel: label, anchorEl: this.anchorEl, small: small, size: size, menuId: menuId, open: this.open, options: searchable ? this.filteredOptions : this.uniqueOptions, value: currValue, fullWidth: fullWidth, selectOnEnter: this.selectOnEnter, onMenuStateChange: this.handleMenuChange, onMenuOptionSelect: this.handleCustomSelectChange, onMenuKeyPress: this.handleMenuKeyPress, onMenuValueChange: this.handleMenuValueChange, onUngroupedOptionsSet: this.setUngroupedOptions, onRetryButtonClicked: this.handleRetry, parentEl: this.el, onTimeoutBlur: this.onTimeoutBlur, activationType: this.searchable || this.selectOnEnter ? "manual" : "automatic" })), hasValidationStatus(this.validationStatus, this.disabled) && (h("ic-input-validation", { class: { "menu-open": this.open }, ariaLiveMode: "polite", status: validationStatus, message: validationText, for: this.inputId })))));
    }
    static get is() { return "ic-select"; }
    static get encapsulation() { return "shadow"; }
    static get delegatesFocus() { return true; }
    static get originalStyleUrls() {
        return {
            "$": ["ic-select.css"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["ic-select.css"]
        };
    }
    static get properties() {
        return {
            "charactersUntilSuggestions": {
                "type": "number",
                "mutable": false,
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [{
                            "name": "deprecated",
                            "text": "This prop should not be used anymore."
                        }],
                    "text": ""
                },
                "attribute": "characters-until-suggestions",
                "reflect": false,
                "defaultValue": "0"
            },
            "disabled": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the disabled state will be set."
                },
                "attribute": "disabled",
                "reflect": true,
                "defaultValue": "false"
            },
            "disableFilter": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the built in filtering will be disabled for a searchable variant. For example, if options will already be filtered from external source."
                },
                "attribute": "disable-filter",
                "reflect": false,
                "defaultValue": "false"
            },
            "emptyOptionListText": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The text displayed when there are no options in the option list."
                },
                "attribute": "empty-option-list-text",
                "reflect": false,
                "defaultValue": "\"No results found\""
            },
            "form": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The <form> element to associate the select with."
                },
                "attribute": "form",
                "reflect": false
            },
            "formaction": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The URL that processes the information submitted by the select. It overrides the action attribute of the select's form owner. Does nothing if there is no form owner.\nThis prop should only be used with searchable select and will only be applied if searchable is true."
                },
                "attribute": "formaction",
                "reflect": false
            },
            "formenctype": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The way the submitted form data is encoded. This prop should only be used with searchable select and will only be applied if searchable is true."
                },
                "attribute": "formenctype",
                "reflect": false
            },
            "formmethod": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The HTTP method used to submit the form. This prop should only be used with searchable select and will only be applied if searchable is true."
                },
                "attribute": "formmethod",
                "reflect": false
            },
            "formnovalidate": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the form will not be validated when submitted. This prop should only be used with searchable select and will only be applied if searchable is true."
                },
                "attribute": "formnovalidate",
                "reflect": false
            },
            "formtarget": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The place to display the response from submitting the form. It overrides the target attribute of the select's form owner.\nThis prop should only be used with searchable select and will only be applied if searchable is true."
                },
                "attribute": "formtarget",
                "reflect": false
            },
            "fullWidth": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the select element will fill the width of the container. This prop should only be used with searchable select and will only be applied if searchable is true."
                },
                "attribute": "full-width",
                "reflect": false,
                "defaultValue": "false"
            },
            "helperText": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The helper text that will be displayed for additional field guidance."
                },
                "attribute": "helper-text",
                "reflect": false,
                "defaultValue": "\"\""
            },
            "hideLabel": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the label will be hidden and the required label value will be applied as an aria-label."
                },
                "attribute": "hide-label",
                "reflect": false,
                "defaultValue": "false"
            },
            "includeDescriptionsInSearch": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, descriptions of options will be included when filtering options in a searchable select. Only applies to built in filtering."
                },
                "attribute": "include-descriptions-in-search",
                "reflect": false,
                "defaultValue": "false"
            },
            "includeGroupTitlesInSearch": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, group titles of grouped options will be included when filtering options in a searchable select. Only applies to built in filtering."
                },
                "attribute": "include-group-titles-in-search",
                "reflect": false,
                "defaultValue": "false"
            },
            "label": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": true,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The label for the select."
                },
                "attribute": "label",
                "reflect": false
            },
            "loadingErrorLabel": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The message displayed when external loading times out."
                },
                "attribute": "loading-error-label",
                "reflect": false,
                "defaultValue": "\"Loading Error\""
            },
            "loadingLabel": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The message displayed whilst the options are being loaded externally."
                },
                "attribute": "loading-label",
                "reflect": false,
                "defaultValue": "\"Loading...\""
            },
            "name": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The name of the control, which is submitted with the form data."
                },
                "attribute": "name",
                "reflect": false,
                "defaultValue": "this.inputId"
            },
            "placeholder": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The placeholder value to be displayed."
                },
                "attribute": "placeholder",
                "reflect": false,
                "defaultValue": "\"Select an option\""
            },
            "readonly": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the readonly state will be set."
                },
                "attribute": "readonly",
                "reflect": false,
                "defaultValue": "false"
            },
            "required": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the select will require a value."
                },
                "attribute": "required",
                "reflect": false,
                "defaultValue": "false"
            },
            "searchable": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, a searchable variant of the select will be displayed which can be typed in to filter options."
                },
                "attribute": "searchable",
                "reflect": false,
                "defaultValue": "false"
            },
            "searchMatchPosition": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcSearchMatchPositions",
                    "resolved": "\"anywhere\" | \"start\"",
                    "references": {
                        "IcSearchMatchPositions": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcSearchMatchPositions"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Whether the search string of the searchable select should match the start of or anywhere in the options. Only applies to built in filtering."
                },
                "attribute": "search-match-position",
                "reflect": false,
                "defaultValue": "\"anywhere\""
            },
            "selectOnEnter": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the icOptionSelect event will be fired on enter instead of ArrowUp and ArrowDown."
                },
                "attribute": "select-on-enter",
                "reflect": false,
                "defaultValue": "false"
            },
            "showClearButton": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, a button which clears the select input when clicked will be displayed. The button will always appear on the searchable select."
                },
                "attribute": "show-clear-button",
                "reflect": false,
                "defaultValue": "false"
            },
            "size": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcSizes",
                    "resolved": "\"default\" | \"large\" | \"small\"",
                    "references": {
                        "IcSizes": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcSizes"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The size of the select component."
                },
                "attribute": "size",
                "reflect": false,
                "defaultValue": "\"default\""
            },
            "small": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [{
                            "name": "deprecated",
                            "text": "This prop should not be used anymore. Set prop `size` to \"small\" instead."
                        }],
                    "text": ""
                },
                "attribute": "small",
                "reflect": false,
                "defaultValue": "false"
            },
            "timeout": {
                "type": "number",
                "mutable": false,
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If using external filtering, set a timeout for when loading takes too long."
                },
                "attribute": "timeout",
                "reflect": false
            },
            "validationStatus": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcInformationStatusOrEmpty",
                    "resolved": "\"\" | \"error\" | \"success\" | \"warning\"",
                    "references": {
                        "IcInformationStatusOrEmpty": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcInformationStatusOrEmpty"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The validation status - e.g. 'error' | 'warning' | 'success'."
                },
                "attribute": "validation-status",
                "reflect": false,
                "defaultValue": "\"\""
            },
            "validationText": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The text to display as the validation message."
                },
                "attribute": "validation-text",
                "reflect": false,
                "defaultValue": "\"\""
            },
            "loading": {
                "type": "boolean",
                "mutable": true,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the loading state will be triggered when fetching options asynchronously."
                },
                "attribute": "loading",
                "reflect": false,
                "defaultValue": "false"
            },
            "options": {
                "type": "unknown",
                "mutable": false,
                "complexType": {
                    "original": "IcMenuOption[]",
                    "resolved": "IcMenuOption[]",
                    "references": {
                        "IcMenuOption": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcMenuOption"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The possible selection options."
                },
                "defaultValue": "[]"
            },
            "debounce": {
                "type": "number",
                "mutable": false,
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The amount of time, in milliseconds, to wait to trigger the `icChange` event after each keystroke."
                },
                "attribute": "debounce",
                "reflect": false,
                "defaultValue": "0"
            },
            "value": {
                "type": "string",
                "mutable": true,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The value of the select, reflected by the value of the currently selected option. For the searchable variant, the value is also reflected by the user input."
                },
                "attribute": "value",
                "reflect": false
            }
        };
    }
    static get states() {
        return {
            "ariaActiveDescendant": {},
            "clearButtonFocused": {},
            "debounceIcChange": {},
            "hiddenInputValue": {},
            "noOptions": {},
            "open": {},
            "pressedCharacters": {},
            "searchableSelectInputValue": {},
            "filteredOptions": {},
            "uniqueOptions": {},
            "currDebounce": {},
            "initialValue": {},
            "inputValueToFilter": {},
            "currValue": {}
        };
    }
    static get events() {
        return [{
                "method": "icBlur",
                "name": "icBlur",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when the select loses focus."
                },
                "complexType": {
                    "original": "void",
                    "resolved": "void",
                    "references": {}
                }
            }, {
                "method": "icChange",
                "name": "icChange",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when the value changes."
                },
                "complexType": {
                    "original": "IcValueEventDetail",
                    "resolved": "IcValueEventDetail",
                    "references": {
                        "IcValueEventDetail": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcValueEventDetail"
                        }
                    }
                }
            }, {
                "method": "icClear",
                "name": "icClear",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when the clear button is clicked."
                },
                "complexType": {
                    "original": "void",
                    "resolved": "void",
                    "references": {}
                }
            }, {
                "method": "icClose",
                "name": "icClose",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when the select options menu is closed."
                },
                "complexType": {
                    "original": "void",
                    "resolved": "void",
                    "references": {}
                }
            }, {
                "method": "icFocus",
                "name": "icFocus",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when the select gains focus."
                },
                "complexType": {
                    "original": "void",
                    "resolved": "void",
                    "references": {}
                }
            }, {
                "method": "icInput",
                "name": "icInput",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when a keyboard input occurred."
                },
                "complexType": {
                    "original": "IcValueEventDetail",
                    "resolved": "IcValueEventDetail",
                    "references": {
                        "IcValueEventDetail": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcValueEventDetail"
                        }
                    }
                }
            }, {
                "method": "icOpen",
                "name": "icOpen",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when the select options menu is opened."
                },
                "complexType": {
                    "original": "void",
                    "resolved": "void",
                    "references": {}
                }
            }, {
                "method": "icOptionSelect",
                "name": "icOptionSelect",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when an option is highlighted within the menu.\nHighlighting a menu item will also trigger an `icChange/onIcChange` due to the value being updated."
                },
                "complexType": {
                    "original": "IcOptionSelectEventDetail",
                    "resolved": "IcOptionSelectEventDetail",
                    "references": {
                        "IcOptionSelectEventDetail": {
                            "location": "import",
                            "path": "../ic-menu/ic-menu.types",
                            "id": "src/components/ic-menu/ic-menu.types.ts::IcOptionSelectEventDetail"
                        }
                    }
                }
            }, {
                "method": "icRetryLoad",
                "name": "icRetryLoad",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when the 'retry loading' button is clicked for a searchable variant."
                },
                "complexType": {
                    "original": "IcValueEventDetail",
                    "resolved": "IcValueEventDetail",
                    "references": {
                        "IcValueEventDetail": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcValueEventDetail"
                        }
                    }
                }
            }];
    }
    static get methods() {
        return {
            "setFocus": {
                "complexType": {
                    "signature": "() => Promise<void>",
                    "parameters": [],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "Sets focus on the input box.",
                    "tags": []
                }
            }
        };
    }
    static get elementRef() { return "el"; }
    static get watchers() {
        return [{
                "propName": "loading",
                "methodName": "loadingHandler"
            }, {
                "propName": "options",
                "methodName": "watchOptionsHandler"
            }, {
                "propName": "debounce",
                "methodName": "debounceChangedHandler"
            }, {
                "propName": "value",
                "methodName": "valueChangedHandler"
            }, {
                "propName": "open",
                "methodName": "openChangedHandler"
            }];
    }
}
//# sourceMappingURL=ic-select.js.map
