import { Host, h, } from "@stencil/core";
import menuIcon from "../../assets/hamburger-menu-icon.svg";
import closeIcon from "../../assets/close-icon.svg";
import chevronIcon from "../../assets/chevron-icon.svg";
import { getCurrentDeviceSize, DEVICE_SIZES, checkResizeObserver, isSlotUsed, getThemeForegroundColor, getCssProperty, hasClassificationBanner, onComponentRequiredPropUndefined, isEmptyString, isPropDefined, } from "../../utils/helpers";
import { IcThemeForegroundEnum, } from "../../utils/types";
/**
 * @slot app-icon - Content will be rendered adjacent to the app title at the very top of the side navigation.
 * @slot app-title - Handle routing by nesting a route in the app title.
 * @slot primary-navigation - Content will be rendered at the top of the side navigation.
 * @slot secondary-navigation - Content will be rendered at the bottom of the side navigation.
 */
export class SideNavigation {
    constructor() {
        this.ANIMATION_DURATION = parseInt(getCssProperty("--ic-transition-duration-slow")) || 0;
        this.IC_NAVIGATION_ITEM = "ic-navigation-item";
        this.resizeObserver = null;
        this.COLLAPSED_ICON_LABELS_END = "collapsed-icon-labels-end";
        this.COLLAPSED_ICON_LABELS_START = "collapsed-icon-labels-start";
        this.menuButton = null;
        this.emitSideNavigationExpanded = (objDetails) => {
            this.sideNavExpanded.emit({
                sideNavExpanded: objDetails.sideNavExpanded,
                sideNavMobile: objDetails.sideNavMobile,
            });
        };
        this.toggleMenu = () => {
            this.menuOpen = !this.menuOpen;
            this.setMobileMenuAriaAttributes(this.menuOpen);
            this.arrangeSlottedNavigationItem(this.menuOpen);
            this.setToggleMenuFlyoutMenuVisibility(this.menuOpen);
            this.emitSideNavigationExpanded({
                sideNavExpanded: this.menuOpen,
                sideNavMobile: true,
            });
        };
        this.setToggleMenuFlyoutMenuVisibility = (menuOpen) => {
            const sideNav = this.el.shadowRoot.querySelector("#side-navigation");
            const sideNavInner = sideNav.querySelector(".side-navigation-inner");
            const bottomWrapper = sideNav.querySelector(".bottom-wrapper");
            const menuVisibilityVisible = "menu-visibility-visible";
            if (menuOpen) {
                bottomWrapper.classList.add(menuVisibilityVisible);
                sideNavInner.classList.add(menuVisibilityVisible);
            }
            else {
                setTimeout(() => {
                    sideNavInner.classList.remove(menuVisibilityVisible);
                    bottomWrapper.classList.remove(menuVisibilityVisible);
                }, this.ANIMATION_DURATION);
            }
        };
        this.setMobileMenuAriaAttributes = (menuOpen) => {
            if (this.menuButton !== null) {
                this.menuButton.setAttribute("aria-expanded", `${menuOpen}`);
                this.menuButton.setAttribute("aria-label", `${menuOpen ? "Close" : "Open"} navigation menu`);
            }
        };
        this.setAndRemoveNoWrapAfterMenuExpanded = () => {
            const appTitle = this.el.shadowRoot.querySelector(".title-link ic-typography") ||
                this.el.querySelector("[slot='app-title']");
            appTitle.classList.add("ic-typography-no-wrap");
            setTimeout(() => {
                appTitle.classList.remove("ic-typography-no-wrap");
            }, this.ANIMATION_DURATION);
        };
        this.toggleMenuExpanded = () => {
            this.menuExpanded = !this.menuExpanded;
            if (this.menuExpanded) {
                this.setAndRemoveNoWrapAfterMenuExpanded();
                this.el.shadowRoot
                    .querySelector(".app-title-inner-wrapper")
                    .classList.add("app-title-show");
            }
            else {
                this.el.style.setProperty("--navigation-item-width", "320px");
                this.el.shadowRoot
                    .querySelector(".app-title-inner-wrapper")
                    .classList.remove("app-title-show");
                this.el.addEventListener("transitionend", (e) => {
                    if (e.propertyName === "width") {
                        this.el.style.setProperty("--navigation-item-width", null);
                    }
                });
            }
            this.arrangeSlottedNavigationItem(this.menuExpanded);
            this.displayTooltipWithExpandedLongLabel(this.menuExpanded);
            if (this.collapsedIconLabels) {
                this.animateCollapsedIconLabels();
            }
            this.setExpandedButtonHeight();
            this.emitSideNavigationExpanded({ sideNavExpanded: this.menuExpanded });
        };
        /**
         * In order to style nested slotted elements (e.g. using React Router components), this method
         * rearranges the a tag and labels and adds inline styling expand/collapsed animations as external CSS classes are not
         * do not take affect.
         * @param menuExpanded boolean - true or false depending on side navigation state
         */
        this.arrangeSlottedNavigationItem = (menuExpanded) => {
            const navItems = this.el.querySelectorAll("ic-navigation-item");
            navItems.forEach((navItem) => {
                const isNamedSlot = isSlotUsed(navItem, "navigation-item");
                const isUnnamedSlot = navItem.children[0] && !navItem.children[0].getAttribute("slot");
                if (isNamedSlot || isUnnamedSlot) {
                    let navItemSlot;
                    if (isNamedSlot) {
                        navItemSlot = navItem.querySelector("[slot='navigation-item']");
                    }
                    else {
                        navItemSlot = navItem.children[0];
                    }
                    const iconWrapper = document.createElement("div");
                    const icon = navItemSlot.querySelector("svg");
                    const label = navItem.textContent.trim();
                    const icTypography = document.createElement("ic-typography");
                    icTypography.classList.add("ic-typography-label", "hydrated", "navigation-item-side-nav-slotted-text");
                    iconWrapper.style.height = "var(--ic-space-lg)";
                    iconWrapper.append(icon);
                    navItemSlot.textContent = "";
                    icTypography.textContent = label;
                    navItemSlot.append(iconWrapper);
                    navItemSlot.append(icTypography);
                    if (this.collapsedIconLabels) {
                        this.styleSlottedCollapsedIconLabels(menuExpanded, icTypography);
                    }
                    else {
                        this.styleSlottedIconLabels(menuExpanded, icTypography);
                    }
                }
            });
        };
        this.styleSlottedCollapsedIconLabels = (menuExpanded, icTypography) => {
            if (menuExpanded) {
                icTypography.style.marginTop = "0";
                icTypography.style.whiteSpace = "nowrap";
                icTypography.style.overflow = "hidden";
                icTypography.style.textOverflow = "ellipsis";
            }
            else {
                icTypography.style.marginTop = "10px";
                icTypography.style.whiteSpace = "nowrap";
                icTypography.style.overflow = "hidden";
                icTypography.style.textOverflow = "ellipsis";
            }
        };
        this.setExpandedButtonHeight = () => {
            const appStatusWrapper = this.el.shadowRoot.querySelector("#side-navigation > .bottom-wrapper > .bottom-side-nav > .app-status-wrapper");
            if (appStatusWrapper.offsetHeight !== 0) {
                this.el.style.setProperty("--sm-side-navigation-bottom-bar-height", `${appStatusWrapper.offsetHeight}px`);
            }
        };
        this.styleSlottedIconLabels = (menuExpanded, icTypography) => {
            if (menuExpanded) {
                icTypography.style.opacity = "1";
                icTypography.style.visibility = "visible";
                icTypography.style.transition =
                    "visibility 0s, opacity var(--ic-easing-transition-slow)";
            }
            else {
                icTypography.style.opacity = "0";
                icTypography.style.visibility = "hidden";
                icTypography.style.transition =
                    "visibility 0s, opacity var(--ic-easing-transition-slow)";
            }
        };
        this.transitionHandler = (type) => {
            const primaryNavigationWrapper = this.el.shadowRoot.querySelector(".primary-navigation");
            const secondaryNavigationWrapper = this.el.shadowRoot.querySelector(".bottom-wrapper > .secondary-navigation");
            const classToRemove = type === "start"
                ? this.COLLAPSED_ICON_LABELS_END
                : this.COLLAPSED_ICON_LABELS_START;
            const classToAdd = type === "start"
                ? this.COLLAPSED_ICON_LABELS_START
                : this.COLLAPSED_ICON_LABELS_END;
            if (primaryNavigationWrapper) {
                primaryNavigationWrapper.classList.remove(classToRemove);
                primaryNavigationWrapper.classList.add(classToAdd);
            }
            if (secondaryNavigationWrapper) {
                secondaryNavigationWrapper.classList.remove(classToRemove);
                secondaryNavigationWrapper.classList.add(classToAdd);
            }
        };
        this.transitionEndHandler = () => {
            this.transitionHandler("end");
        };
        this.animateCollapsedIconLabels = () => {
            this.transitionHandler("start");
            this.transitionEndHandler();
            this.el.addEventListener("transitionend", this.transitionEndHandler);
        };
        this.paddingIconWidth = (navItems) => {
            const navItemLink = (navItems[0].shadowRoot &&
                (navItems[0].shadowRoot.querySelector("ic-tooltip a") ||
                    navItems[0].shadowRoot.querySelector("ic-tooltip div"))) ||
                navItems[0].querySelector("a") ||
                navItems[0].querySelector("div");
            const navItemSVG = navItems[0].querySelector("svg");
            const navStyles = {
                gap: window.getComputedStyle(navItemLink).gap,
                iconWidth: window.getComputedStyle(navItemSVG).width,
                paddingLeft: window.getComputedStyle(navItemLink).paddingLeft,
            };
            return Object.values(navStyles).reduce((prev, curr) => {
                return (prev += parseInt(curr));
            }, 0);
        };
        this.displayTooltipWithExpandedLongLabel = (menuExpanded) => {
            let timer;
            if (menuExpanded) {
                timer = setTimeout(() => {
                    const sideNavWidth = this.el.clientWidth;
                    const navigationItems = Array.from(this.el.querySelectorAll(this.IC_NAVIGATION_ITEM));
                    const paddingIconDelta = navigationItems.length
                        ? this.paddingIconWidth(navigationItems)
                        : 0;
                    navigationItems.forEach((navigationItem) => {
                        var _a, _b;
                        const icTypographyScrollWidth = ((_a = (navigationItem.shadowRoot &&
                            navigationItem.shadowRoot.querySelector("ic-tooltip .link ic-typography.ic-typography-label"))) === null || _a === void 0 ? void 0 : _a.scrollWidth) ||
                            ((_b = navigationItem.querySelector("ic-typography.ic-typography-label")) === null || _b === void 0 ? void 0 : _b.scrollWidth);
                        if (icTypographyScrollWidth > sideNavWidth - paddingIconDelta) {
                            navigationItem.setAttribute("display-navigation-tooltip", "true");
                        }
                    });
                }, this.ANIMATION_DURATION);
            }
            else {
                clearTimeout(timer);
                const navigationItems = Array.from(this.el.querySelectorAll(this.IC_NAVIGATION_ITEM));
                navigationItems.forEach((navigationItem) => {
                    navigationItem.setAttribute("display-navigation-tooltip", "false");
                });
            }
        };
        this.setMenuExpanded = (expanded) => {
            this.menuExpanded = expanded;
        };
        /**
         * As the mobile top bar is fixed, a padding top is required
         * to push main content down the height of the mobile top bar
         * @param value - padding-top css value
         */
        this.setParentPaddingTop = (value) => {
            this.el.parentElement.style.setProperty("padding-top", value);
        };
        this.setParentPaddingLeft = (value) => {
            this.el.parentElement.style.setProperty("padding-left", value);
        };
        this.renderAppTitle = (isAppNameSubtitleVariant) => {
            const displayShortAppTitle = this.deviceSize <= DEVICE_SIZES.S && !isEmptyString(this.shortAppTitle);
            return (h("ic-typography", { variant: displayShortAppTitle || isAppNameSubtitleVariant
                    ? "subtitle-small"
                    : "h3", "aria-label": displayShortAppTitle
                    ? `${this.appTitle} (${this.shortAppTitle})`
                    : undefined }, h("h1", null, displayShortAppTitle ? this.shortAppTitle : this.appTitle)));
        };
        this.resizeObserverCallback = (currSize) => {
            var _a;
            this.deviceSize = currSize;
            const isSmallAndDisableTopBar = currSize === DEVICE_SIZES.S && !this.disableTopBarBehaviour;
            if (!this.disableAutoParentStyling) {
                const topBarHeight = (_a = this.el.shadowRoot.querySelector(".top-bar")) === null || _a === void 0 ? void 0 : _a.scrollHeight;
                this.setParentPaddingTop(isSmallAndDisableTopBar ? `${topBarHeight}px` : "0");
                if (isSmallAndDisableTopBar)
                    this.setParentPaddingLeft("0");
                if (isSmallAndDisableTopBar && this.inline) {
                    this.el.parentElement.style.setProperty("height", `calc(100% - ${topBarHeight}px)`);
                }
                else if (!isSmallAndDisableTopBar) {
                    this.el.parentElement.style.setProperty("height", "100%");
                }
            }
            if (!this.disableAutoParentStyling) {
                const paddingLeft = `calc(var(--ic-space-xxl) ${this.collapsedIconLabels ? "* 2" : "+ var(--ic-space-xs)"})`;
                if (currSize > DEVICE_SIZES.L) {
                    this.setParentPaddingTop("0");
                    this.setParentPaddingLeft("0");
                }
                else if ((currSize > DEVICE_SIZES.S || this.disableTopBarBehaviour) &&
                    currSize <= DEVICE_SIZES.M &&
                    this.static) {
                    this.setParentPaddingLeft(paddingLeft);
                }
                else if ((currSize > DEVICE_SIZES.S || this.disableTopBarBehaviour) &&
                    currSize <= DEVICE_SIZES.L) {
                    this.setParentPaddingLeft(this.static && this.menuExpanded
                        ? "calc(var(--ic-space-xl) * 10)"
                        : paddingLeft);
                }
            }
        };
        this.runResizeObserver = () => {
            this.resizeObserver = new ResizeObserver(() => {
                const currSize = getCurrentDeviceSize();
                this.deviceSizeAppTitle = currSize;
                this.resizeObserverCallback(currSize);
            });
            this.resizeObserver.observe(document.body, { box: "content-box" });
        };
        this.setCollapsedIconLabels = () => {
            const navigationItems = Array.from(this.el.querySelectorAll(this.IC_NAVIGATION_ITEM));
            navigationItems.forEach((navigationItem) => {
                navigationItem.setAttribute("collapsed-icon-label", "true");
            });
        };
        this.styleSlottedCollapsedIconLabel = () => {
            const dynamicSlottedIcTypographyComps = Array.from(this.el.querySelectorAll(".navigation-item-side-nav-slotted-text"));
            dynamicSlottedIcTypographyComps.forEach((icTypography) => {
                var _a, _b;
                if ((_b = (_a = icTypography === null || icTypography === void 0 ? void 0 : icTypography.parentElement) === null || _a === void 0 ? void 0 : _a.parentElement) === null || _b === void 0 ? void 0 : _b.classList.contains("navigation-item-side-nav-collapsed-with-label")) {
                    icTypography.style.whiteSpace = "nowrap";
                    icTypography.style.overflow = "hidden";
                    icTypography.style.textOverflow = "ellipsis";
                    icTypography.style.marginTop = "10px";
                }
            });
        };
        this.renderTopBar = ({ isSDevice, foregroundColor, menuOpen, href, isAppNameSubtitleVariant, }) => {
            const hasTitle = this.appTitle !== "" && isPropDefined(this.appTitle);
            const Component = isSlotUsed(this.el, "app-title") ? "div" : "a";
            const attrs = Component == "a" && {
                href: href,
            };
            return (h("div", { class: {
                    "top-bar": true,
                    [this.foregroundColor]: true,
                } }, isSDevice && (h("nav", { "aria-labelledby": "menu-navigation-toggle-button-landmark", "aria-hidden": "false" }, h("ic-button", { "aria-label": "Open navigation menu", class: "menu-button", id: "menu-button", variant: "secondary", size: "small", "full-width": "true", appearance: foregroundColor, onClick: this.toggleMenu, ariaOwnsId: "side-navigation", "aria-haspopup": "true", "aria-expanded": "false", ref: (el) => (this.menuButton = el) }, h("span", { class: "mobile-top-bar-menu-icon", slot: "icon", innerHTML: menuOpen ? closeIcon : menuIcon }), menuOpen ? "Close" : "Menu"), h("span", { id: "menu-navigation-toggle-button-landmark", class: "navigation-landmark-title", "aria-hidden": "true" }, "Navigation menu toggle button"))), h("div", { class: "app-title-wrapper" }, (hasTitle || isSlotUsed(this.el, "app-title")) && (h(Component, Object.assign({}, attrs, { class: "title-link" }), h("div", { class: "app-icon-container", "aria-hidden": "true" }, h("slot", { name: "app-icon" })), h("div", { class: "app-title-inner-wrapper" }, isSlotUsed(this.el, "app-title") ? (h("slot", { name: "app-title" })) : (this.renderAppTitle(isAppNameSubtitleVariant))))))));
        };
        this.deviceSize = getCurrentDeviceSize();
        this.deviceSizeAppTitle = DEVICE_SIZES.S;
        this.foregroundColor = getThemeForegroundColor();
        this.hasSecondaryNavigation = false;
        this.menuExpanded = false;
        this.menuOpen = false;
        this.appTitle = undefined;
        this.collapsedIconLabels = false;
        this.disableAutoParentStyling = false;
        this.disableTopBarBehaviour = false;
        this.expanded = false;
        this.href = "/";
        this.inline = false;
        this.shortAppTitle = "";
        this.static = false;
        this.status = undefined;
        this.version = undefined;
    }
    watchExpandedHandler() {
        this.setMenuExpanded(this.expanded);
    }
    componentWillLoad() {
        this.setMenuExpanded(this.expanded);
        if (this.collapsedIconLabels) {
            this.setCollapsedIconLabels();
        }
        this.hasSecondaryNavigation = isSlotUsed(this.el, "secondary-navigation");
    }
    componentDidLoad() {
        this.emitSideNavigationExpanded({
            sideNavExpanded: this.menuExpanded,
            sideNavMobile: this.deviceSize === DEVICE_SIZES.S && !this.disableTopBarBehaviour,
        });
        checkResizeObserver(this.runResizeObserver);
        this.styleSlottedCollapsedIconLabel();
        this.arrangeSlottedNavigationItem(this.menuExpanded);
        this.displayTooltipWithExpandedLongLabel(this.menuExpanded);
        this.setExpandedButtonHeight();
        !isSlotUsed(this.el, "app-title") &&
            onComponentRequiredPropUndefined([{ prop: this.appTitle, propName: "app-title" }], "Side Navigation");
    }
    disconnectedCallback() {
        if (this.resizeObserver !== null) {
            this.resizeObserver.disconnect();
        }
        this.el.removeEventListener("transitionend", this.transitionEndHandler);
    }
    themeChangeHandler({ detail }) {
        this.foregroundColor = detail.mode;
    }
    render() {
        const { appTitle, menuOpen, foregroundColor, menuExpanded, href, status, version, collapsedIconLabels, inline, } = this;
        const isSDevice = !this.disableTopBarBehaviour && this.deviceSize === DEVICE_SIZES.S;
        const isMdDevice = this.deviceSize === DEVICE_SIZES.M;
        const isLgDevice = this.deviceSize >= DEVICE_SIZES.L;
        const isAppNameSubtitleVariant = this.deviceSizeAppTitle === DEVICE_SIZES.S;
        const displayExpandBtn = isMdDevice || this.disableTopBarBehaviour || (isLgDevice && !this.static);
        const topBarProps = {
            isSDevice,
            foregroundColor,
            menuOpen,
            href,
            isAppNameSubtitleVariant,
            appTitle,
        };
        return (h(Host, { class: {
                "xs-menu-open": menuOpen && isSDevice,
                "xs-menu-close": !menuOpen && isSDevice,
                "sm-collapsed": !isSDevice && !menuExpanded,
                "sm-expanded": !isSDevice && menuExpanded,
                "side-display": this.deviceSize > DEVICE_SIZES.S || this.disableTopBarBehaviour,
                [IcThemeForegroundEnum.Dark]: foregroundColor === IcThemeForegroundEnum.Dark,
                ["collapsed-labels"]: !isSDevice && !menuExpanded && collapsedIconLabels,
                ["inline"]: inline,
            } }, isSDevice && this.renderTopBar(Object.assign({}, topBarProps)), h("div", { class: "side-navigation", id: "side-navigation" }, !isSDevice && this.renderTopBar(Object.assign({}, topBarProps)), h("div", { class: "side-navigation-inner" }, isSlotUsed(this.el, "primary-navigation") && (h("nav", { class: "primary-navigation", "aria-labelledby": "primary-navigation-landmark" }, h("span", { "aria-hidden": "true", class: "navigation-landmark-title", id: "primary-navigation-landmark" }, "Primary"), h("ul", { class: "navigation-list" }, h("slot", { name: "primary-navigation" }))))), h("div", { class: {
                ["bottom-wrapper"]: true,
                ["classification-spacing"]: hasClassificationBanner(),
            } }, isSlotUsed(this.el, "secondary-navigation") && (h("nav", { class: "secondary-navigation", "aria-labelledby": "secondary-navigation-landmark" }, h("span", { "aria-hidden": "true", class: "navigation-landmark-title", id: "secondary-navigation-landmark" }, "Secondary"), h("ul", { class: "navigation-list" }, h("slot", { name: "secondary-navigation" })))), h("div", { class: "bottom-side-nav" }, this.hasSecondaryNavigation && h("ic-divider", null), displayExpandBtn && (h("button", { class: "menu-expand-button", innerHTML: chevronIcon, onClick: this.toggleMenuExpanded, "aria-label": `${menuExpanded ? "Collapse" : "Expand"} side navigation` })), h("div", { class: "app-status-wrapper" }, status !== "" && (h("div", { class: {
                ["app-status"]: true,
            } }, h("ic-typography", { "aria-label": "app tag", variant: "label-uppercase", class: "app-status-text" }, status))), version !== "" && (h("ic-typography", { variant: "label", class: "app-version", "aria-label": "app version" }, version))))))));
    }
    static get is() { return "ic-side-navigation"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["ic-side-navigation.css"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["ic-side-navigation.css"]
        };
    }
    static get properties() {
        return {
            "appTitle": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The app title to be displayed. This is required, unless a slotted app title link is used."
                },
                "attribute": "app-title",
                "reflect": false
            },
            "collapsedIconLabels": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the icon and label will appear when side navigation is collapsed."
                },
                "attribute": "collapsed-icon-labels",
                "reflect": false,
                "defaultValue": "false"
            },
            "disableAutoParentStyling": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If `true`, automatic parent wrapper styling will be disabled."
                },
                "attribute": "disable-auto-parent-styling",
                "reflect": false,
                "defaultValue": "false"
            },
            "disableTopBarBehaviour": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the side navigation will not display as a top bar on small devices."
                },
                "attribute": "disable-top-bar-behaviour",
                "reflect": false,
                "defaultValue": "false"
            },
            "expanded": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the side navigation will display in an expanded state."
                },
                "attribute": "expanded",
                "reflect": false,
                "defaultValue": "false"
            },
            "href": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The URL that the app title link points to."
                },
                "attribute": "href",
                "reflect": false,
                "defaultValue": "\"/\""
            },
            "inline": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": "If `true`, side navigation will be contained by its parent element."
                        }],
                    "text": ""
                },
                "attribute": "inline",
                "reflect": false,
                "defaultValue": "false"
            },
            "shortAppTitle": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The short title of the app to be displayed at small screen sizes in place of the app title."
                },
                "attribute": "short-app-title",
                "reflect": false,
                "defaultValue": "\"\""
            },
            "static": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the menu expand button will be removed (PLEASE NOTE: This takes effect on screen sizes 992px and above)."
                },
                "attribute": "static",
                "reflect": false,
                "defaultValue": "false"
            },
            "status": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The status of the app to be displayed."
                },
                "attribute": "status",
                "reflect": false
            },
            "version": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The version of the app to be displayed."
                },
                "attribute": "version",
                "reflect": false
            }
        };
    }
    static get states() {
        return {
            "deviceSize": {},
            "deviceSizeAppTitle": {},
            "foregroundColor": {},
            "hasSecondaryNavigation": {},
            "menuExpanded": {},
            "menuOpen": {}
        };
    }
    static get events() {
        return [{
                "method": "sideNavExpanded",
                "name": "sideNavExpanded",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when the side navigation is collapsed and expanded."
                },
                "complexType": {
                    "original": "IcExpandedDetail",
                    "resolved": "IcExpandedDetail",
                    "references": {
                        "IcExpandedDetail": {
                            "location": "import",
                            "path": "./ic-side-navigation.types",
                            "id": "src/components/ic-side-navigation/ic-side-navigation.types.ts::IcExpandedDetail"
                        }
                    }
                }
            }];
    }
    static get elementRef() { return "el"; }
    static get watchers() {
        return [{
                "propName": "expanded",
                "methodName": "watchExpandedHandler"
            }];
    }
    static get listeners() {
        return [{
                "name": "themeChange",
                "method": "themeChangeHandler",
                "target": "document",
                "capture": false,
                "passive": false
            }];
    }
}
//# sourceMappingURL=ic-side-navigation.js.map
