import { Host, h } from "@stencil/core";
import { checkResizeObserver, pxToRem } from "../../utils/helpers";
export class Stepper {
    constructor() {
        this.resizeObserver = null;
        this.visuallyHidden = "visually-hidden";
        // Get all steps currently within this stepper
        this.getChildren = () => {
            this.steps = Array.from(this.el.querySelectorAll("ic-step"));
            this.stepsWithStepTitles = Array.from(this.el.querySelectorAll("ic-step[step-title]"));
        };
        // Inform the user that stepTitles are required on all steps in a compact stepper
        this.checkStepTitles = () => {
            if (this.stepsWithStepTitles.length < this.steps.length &&
                this.variantOverride &&
                this.variant === "compact") {
                this.noOfResizes = this.noOfResizes + 1;
                if (this.noOfResizes === 1) {
                    console.error(`The prop 'step-title' (web components) / 'stepTitle' (react) is required for all steps of the Stepper component (compact variant)`);
                }
            }
        };
        this.setStepTypes = () => {
            this.getChildren();
            this.stepTypes = [];
            for (let i = 0; i < this.steps.length; i++) {
                this.stepTypes.push("active");
            }
        };
        this.setStepperWidth = () => {
            this.alignedFullWidth =
                this.variant === "default" && this.aligned === "full-width";
            const lastStep = this.steps[this.steps.length - 1];
            lastStep.style.maxWidth = "none";
            if (this.alignedFullWidth) {
                this.stepperWidth = this.el.offsetWidth;
                lastStep.style.maxWidth = pxToRem(`${this.stepperWidth / this.steps.length}px`);
                this.lastStepWidth = lastStep.offsetWidth;
            }
        };
        this.initialiseStepStates = () => {
            this.steps.forEach((step, index) => {
                // Set variant
                step.variant = this.variant;
                // Assign stepNum to each step
                step.stepNum = index + 1;
                // Assign lastStep to final step
                step.lastStep = index === this.steps.length - 1;
                step.lastStepNum = this.steps.length;
                if (step.stepType !== "current") {
                    step.current = false;
                    this.stepTypes[index] = step.stepType;
                }
                else {
                    step.current = true;
                }
                const stepTitleArea = step.shadowRoot &&
                    step.shadowRoot.querySelector(".step > .step-title-area");
                if (this.stepsWithStepTitles.length == 0 && this.variantOverride) {
                    if (this.variant === "compact") {
                        this.autoSetStepTitles = true;
                        if (this.autoSetStepTitles) {
                            step.stepTitle = "Step " + step.stepNum;
                            stepTitleArea &&
                                stepTitleArea
                                    .querySelector(".step-title")
                                    .setAttribute("aria-hidden", "true");
                        }
                    }
                    if (this.variant === "default") {
                        this.autoSetStepTitles = false;
                        if (!this.autoSetStepTitles) {
                            step.stepTitle = undefined;
                            stepTitleArea &&
                                stepTitleArea
                                    .querySelector(".step-title")
                                    .removeAttribute("aria-hidden");
                        }
                    }
                }
                if (this.variant === "compact") {
                    if (step.current === true || step.stepType === "current") {
                        step.classList.remove("hide");
                        step.classList.add("show");
                    }
                    else if (step.classList.contains("show")) {
                        step.classList.remove("show");
                        step.classList.add("hide");
                    }
                    if (!step.lastStep) {
                        step.nextStepTitle = this.steps[index + 1].stepTitle;
                        step.progress = (step.stepNum / this.steps.length) * 100;
                    }
                    else if (step.lastStep && this.stepTypes[index] !== "completed") {
                        step.progress = 95;
                    }
                    else {
                        step.progress = 100;
                    }
                    if (this.hideStepInfo && stepTitleArea !== null) {
                        stepTitleArea.classList.remove(this.visuallyHidden);
                    }
                    step.compactStepStyling = this.stepTypes[index];
                }
                if (this.variant === "default") {
                    if (!step.lastStep) {
                        if (this.alignedFullWidth) {
                            step.style.width = pxToRem(`${(this.stepperWidth - this.lastStepWidth) /
                                (this.steps.length - 1)}px`);
                            step.style.minWidth = pxToRem("148px");
                        }
                    }
                    else if (step.lastStep) {
                        step.classList.add("last-step");
                        if (this.alignedFullWidth) {
                            step.style.maxWidth = `${this.lastStepWidth}px`;
                        }
                        else {
                            step.style.maxWidth =
                                this.connectorWidth > 100
                                    ? pxToRem(`${this.connectorWidth + 48}px`)
                                    : pxToRem("148px");
                        }
                    }
                    if (this.aligned === "left") {
                        step.style.width =
                            this.connectorWidth > 100
                                ? pxToRem(`${this.connectorWidth + 48}px`)
                                : pxToRem("148px");
                        const stepConnect = step.shadowRoot.querySelector(".step > .step-top > .step-connect");
                        if (stepConnect) {
                            stepConnect.style.width =
                                this.connectorWidth > 100
                                    ? pxToRem(`${this.connectorWidth}px`)
                                    : pxToRem("100px");
                        }
                    }
                    if (this.hideStepInfo && stepTitleArea !== null) {
                        stepTitleArea.classList.add(this.visuallyHidden);
                    }
                }
            });
        };
        this.setHideStepInfo = () => {
            this.steps.forEach((step) => {
                var _a, _b;
                const stepTitleArea = step.shadowRoot.querySelector(".step > .step-title-area");
                if (stepTitleArea !== null) {
                    if (this.hideStepInfo) {
                        (_a = stepTitleArea.classList) === null || _a === void 0 ? void 0 : _a.add(this.visuallyHidden);
                    }
                    else {
                        (_b = stepTitleArea.classList) === null || _b === void 0 ? void 0 : _b.remove(this.visuallyHidden);
                    }
                }
            });
        };
        this.overrideVariant = () => {
            if (this.variantOverride) {
                let minDefaultStepperWidth = 148 * this.steps.length;
                if (this.aligned === "left" && this.connectorWidth > 100) {
                    minDefaultStepperWidth = (this.connectorWidth + 48) * this.steps.length;
                }
                if (this.el.clientWidth < minDefaultStepperWidth) {
                    this.variant = "compact";
                }
                else {
                    this.variant = "default";
                }
            }
        };
        this.resizeObserverCallback = () => {
            this.getChildren();
            this.checkStepTitles();
            this.overrideVariant();
            this.setStepperWidth();
            this.initialiseStepStates();
        };
        this.runResizeObserver = () => {
            this.resizeObserver = new ResizeObserver(() => {
                this.resizeObserverCallback();
            });
            this.resizeObserver.observe(this.el);
        };
        this.alignedFullWidth = true;
        this.autoSetStepTitles = true;
        this.lastStepWidth = 0;
        this.noOfResizes = 0;
        this.stepperWidth = document
            .querySelector("ic-stepper")
            .getBoundingClientRect().width;
        this.stepTypes = [];
        this.variantOverride = this.variant !== "compact";
        this.aligned = "full-width";
        this.appearance = "default";
        this.connectorWidth = 100;
        this.hideStepInfo = false;
        this.variant = "default";
    }
    handlePropChange() {
        this.setHideStepInfo();
        this.getChildren();
    }
    disconnectedCallback() {
        if (this.resizeObserver !== null) {
            this.resizeObserver.disconnect();
        }
    }
    componentWillLoad() {
        this.setStepTypes();
        if (this.variant === "compact") {
            this.variantOverride = false;
        }
    }
    componentWillRender() {
        this.initialiseStepStates();
    }
    componentDidLoad() {
        checkResizeObserver(this.runResizeObserver);
    }
    render() {
        return (h(Host, { class: {
                [`${this.variant}`]: true,
                ["aligned-left"]: this.variant === "default" && this.aligned === "left",
            } }, h("ul", { class: "step-item-list" }, h("slot", null))));
    }
    static get is() { return "ic-stepper"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["ic-stepper.css"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["ic-stepper.css"]
        };
    }
    static get properties() {
        return {
            "aligned": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcStepperAlignment",
                    "resolved": "\"full-width\" | \"left\"",
                    "references": {
                        "IcStepperAlignment": {
                            "location": "import",
                            "path": "./ic-stepper.types",
                            "id": "src/components/ic-stepper/ic-stepper.types.ts::IcStepperAlignment"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The alignment of the default stepper within its container."
                },
                "attribute": "aligned",
                "reflect": false,
                "defaultValue": "\"full-width\""
            },
            "appearance": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "\"light\" | \"default\"",
                    "resolved": "\"default\" | \"light\"",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The appearance of the stepper."
                },
                "attribute": "appearance",
                "reflect": false,
                "defaultValue": "\"default\""
            },
            "connectorWidth": {
                "type": "number",
                "mutable": false,
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The length of the connector between each step in pixels. Minimum length is 100px."
                },
                "attribute": "connector-width",
                "reflect": false,
                "defaultValue": "100"
            },
            "hideStepInfo": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the information about each step, i.e. step title, step subtitle and step status, will be hidden on all default steps. The information about each step will still be visible in the compact variant of the stepper."
                },
                "attribute": "hide-step-info",
                "reflect": false,
                "defaultValue": "false"
            },
            "variant": {
                "type": "string",
                "mutable": true,
                "complexType": {
                    "original": "IcStepVariants",
                    "resolved": "\"compact\" | \"default\"",
                    "references": {
                        "IcStepVariants": {
                            "location": "import",
                            "path": "../ic-step/ic-step.types",
                            "id": "src/components/ic-step/ic-step.types.ts::IcStepVariants"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The variant of the stepper."
                },
                "attribute": "variant",
                "reflect": false,
                "defaultValue": "\"default\""
            }
        };
    }
    static get states() {
        return {
            "alignedFullWidth": {},
            "autoSetStepTitles": {},
            "lastStepWidth": {},
            "noOfResizes": {},
            "stepperWidth": {},
            "stepTypes": {},
            "variantOverride": {}
        };
    }
    static get elementRef() { return "el"; }
    static get watchers() {
        return [{
                "propName": "hideStepInfo",
                "methodName": "handlePropChange"
            }, {
                "propName": "variant",
                "methodName": "handlePropChange"
            }];
    }
}
//# sourceMappingURL=ic-stepper.js.map
