import { h, } from "@stencil/core";
import { IcThemeForegroundEnum, } from "../../utils/types";
const CONTEXT_ID_ATTR = "context-id";
export class TabContext {
    constructor() {
        this.newTabPanels = [];
        this.newTabs = [];
        this.emitEvents = (tabIndex) => {
            const tabLabel = this.el
                .querySelectorAll("ic-tab")
            // eslint-disable-next-line no-unexpected-multiline
            [tabIndex].textContent.trim();
            this.icTabSelect.emit({ tabIndex, tabLabel });
            this.tabSelect.emit({ tabIndex, tabLabel });
        };
        /** Sets attributes to link tab-group, tabs and tab-panels */
        this.linkTabs = () => {
            this.tabs.forEach((tab, index) => {
                const tabId = `ic-tab-${index}-context-${this.contextId}`;
                const tabPanelId = `ic-tab-panel-${index}-context-${this.contextId}`;
                tab.setAttribute("id", tabId);
                tab.tabId = `ic-tab--${index}-context-${this.contextId}`;
                tab.tabPosition = index;
                tab.setAttribute("aria-controls", tabPanelId);
                tab.setAttribute(CONTEXT_ID_ATTR, this.contextId);
                this.tabGroup.setAttribute(CONTEXT_ID_ATTR, this.contextId);
                this.tabPanels[index].setAttribute("id", tabPanelId);
                this.tabPanels[index].setAttribute("aria-labelledby", tabId);
                this.tabPanels[index].setAttribute(CONTEXT_ID_ATTR, this.contextId);
                if (this.appearance === IcThemeForegroundEnum.Light) {
                    tab.appearance = this.appearance;
                    this.tabPanels[index].appearance = this.appearance;
                }
            });
            if (this.appearance === IcThemeForegroundEnum.Light) {
                this.tabGroup.appearance = this.appearance;
            }
        };
        /**
         * Gets tabs and tabpanels with the same context ID using querySelector to selector the children in relation to the host
         */
        this.getChildren = () => {
            this.tabGroup = this.el.querySelector("ic-tab-group");
            this.tabs = Array.from(this.tabGroup.querySelectorAll("ic-tab"));
            this.tabPanels = Array.from(this.el.children).filter((child) => child.tagName === "IC-TAB-PANEL");
            this.enabledTabs = this.getEnabledTabs();
        };
        this.keydownHandler = (event) => {
            const isManual = this.activationType === "manual";
            const enabledTabIndex = this.enabledTabs.findIndex((tab) => tab.tabId ===
                this.tabs[isManual ? this.focusedTabIndex : this.selectedTab].tabId);
            const keyboardFunction = isManual
                ? this.keyboardFocusTab
                : this.keyboardSelectTab;
            let preventDefault = true;
            switch (event.key) {
                case "Home":
                    keyboardFunction(0);
                    break;
                case "End":
                    keyboardFunction(this.enabledTabs.length - 1);
                    break;
                case "ArrowRight":
                    keyboardFunction(enabledTabIndex < this.enabledTabs.length - 1
                        ? enabledTabIndex + 1
                        : 0);
                    break;
                case "ArrowLeft":
                    keyboardFunction((enabledTabIndex > 0 ? enabledTabIndex : this.enabledTabs.length) - 1);
                    break;
                default:
                    if (isManual && (event.key === "Enter" || event.key === " ")) {
                        this.keyboardSelectTab(this.focusedTabIndex);
                    }
                    else {
                        preventDefault = false;
                    }
            }
            if (preventDefault)
                event.preventDefault();
        };
        /** Sets the tab that is selected on initial render */
        this.setInitialTab = () => {
            if (this.selectedTabIndex !== undefined) {
                this.selectedTab = this.selectedTabIndex;
                this.focusedTabIndex = this.selectedTabIndex;
            }
            else {
                const firstEnabledTabIndex = this.tabs.findIndex((tab) => tab.tabId === this.enabledTabs[0].tabId);
                this.selectedTab = firstEnabledTabIndex;
                this.focusedTabIndex = firstEnabledTabIndex;
            }
        };
        /** Passes the selected tab to the tab and tab panel components */
        this.configureTabs = () => {
            this.enabledTabs.forEach((tab) => {
                tab.selected = tab.tabPosition === this.selectedTab;
            });
            this.tabPanels.forEach((tabPanel, index) => {
                tabPanel.hidden = index !== this.selectedTab;
            });
        };
        this.getEnabledTabs = () => Array.from(this.tabs).filter((child) => !child.disabled);
        /** Sets focus on tab and selects it */
        this.keyboardSelectTab = (enabledTabIndex) => {
            const newIndex = this.tabs.findIndex((tab) => tab.tabId === this.enabledTabs[enabledTabIndex].tabId);
            this.enabledTabs[enabledTabIndex].focus();
            if (this.selectedTabIndex === undefined) {
                this.selectedTab = newIndex;
            }
            else {
                this.emitEvents(newIndex);
            }
        };
        /** Sets focus on tab without selecting it (for manual activation) */
        this.keyboardFocusTab = (enabledTabIndex) => {
            this.enabledTabs[enabledTabIndex].focus();
            this.focusedTabIndex = this.tabs.findIndex((tab) => tab.tabId === this.enabledTabs[enabledTabIndex].tabId);
        };
        this.selectedTab = undefined;
        this.activationType = "automatic";
        this.appearance = "dark";
        this.contextId = "default";
        this.selectedTabIndex = undefined;
    }
    watchAppearanceHandler() {
        this.tabs.forEach((tab, index) => {
            tab.appearance = this.appearance;
            this.tabPanels[index].appearance = this.appearance;
        });
        this.tabGroup.appearance = this.appearance;
    }
    updateSelectedTab(newValue) {
        this.selectedTab = newValue;
    }
    componentDidLoad() {
        if (this.selectedTabIndex !== undefined) {
            this.selectedTab = this.selectedTabIndex;
        }
        this.getChildren();
        this.linkTabs();
        this.tabGroup.addEventListener("keydown", this.keydownHandler);
        this.setInitialTab();
        this.configureTabs();
    }
    componentWillUpdate() {
        this.configureTabs();
    }
    disconnectedCallback() {
        this.tabGroup.removeEventListener("keydown", this.keydownHandler);
    }
    tabClickHandler(event) {
        if (this.selectedTabIndex === undefined &&
            event.detail.contextId === this.contextId) {
            this.selectedTab = event.detail.position;
        }
        this.emitEvents(event.detail.position);
        event.stopImmediatePropagation();
    }
    tabCreatedHandler(ev) {
        if (this.tabs && this.tabPanels) {
            (ev.detail.setFocus ? this.newTabs : this.newTabPanels).push(ev.detail);
            if (this.newTabs.length === this.newTabPanels.length) {
                this.tabs.push(...this.newTabs);
                this.tabPanels.push(...this.newTabPanels);
                this.enabledTabs = this.getEnabledTabs();
                this.linkTabs();
                if (!this.tabs[this.selectedTab] || !this.tabPanels[this.selectedTab])
                    this.setInitialTab();
                this.configureTabs();
                this.newTabs = [];
                this.newTabPanels = [];
            }
        }
    }
    tabEnabledHandler() {
        this.enabledTabs = this.getEnabledTabs();
    }
    /**
     * @internal Used to set tab/tab panel IDs when a tab/tab panel has been removed
     */
    async tabRemovedHandler(hadFocus) {
        this.getChildren();
        this.linkTabs();
        if (this.tabs[this.selectedTab] && this.tabPanels[this.selectedTab]) {
            this.tabs[this.selectedTab].selected = true;
            this.tabPanels[this.selectedTab].hidden = false;
        }
        else {
            this.setInitialTab();
        }
        if (hadFocus) {
            this.tabs[this.selectedTab].setFocus();
        }
    }
    render() {
        return h("slot", null);
    }
    static get is() { return "ic-tab-context"; }
    static get properties() {
        return {
            "activationType": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcActivationTypes",
                    "resolved": "\"automatic\" | \"manual\"",
                    "references": {
                        "IcActivationTypes": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcActivationTypes"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Determines whether tabs have to be manually activated (by pressing 'Enter' or 'Space') when they receive focus using keyboard navigation."
                },
                "attribute": "activation-type",
                "reflect": false,
                "defaultValue": "\"automatic\""
            },
            "appearance": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcThemeForegroundNoDefault",
                    "resolved": "\"dark\" | \"light\"",
                    "references": {
                        "IcThemeForegroundNoDefault": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcThemeForegroundNoDefault"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The appearance of the tab context, e.g dark, or light."
                },
                "attribute": "appearance",
                "reflect": false,
                "defaultValue": "\"dark\""
            },
            "contextId": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The unique context needed if using multiple tabs inside one another i.e. rendering another set of tabs inside a tab panel."
                },
                "attribute": "context-id",
                "reflect": true,
                "defaultValue": "\"default\""
            },
            "selectedTabIndex": {
                "type": "number",
                "mutable": false,
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The selected tab to be controlled by the user. Must be used alongside the icTabSelect event to manage tab selection."
                },
                "attribute": "selected-tab-index",
                "reflect": false
            }
        };
    }
    static get states() {
        return {
            "selectedTab": {}
        };
    }
    static get events() {
        return [{
                "method": "icTabSelect",
                "name": "icTabSelect",
                "bubbles": false,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when a user selects a tab."
                },
                "complexType": {
                    "original": "IcTabSelectEventDetail",
                    "resolved": "IcTabSelectEventDetail",
                    "references": {
                        "IcTabSelectEventDetail": {
                            "location": "import",
                            "path": "../ic-tab/ic-tab.types",
                            "id": "src/components/ic-tab/ic-tab.types.ts::IcTabSelectEventDetail"
                        }
                    }
                }
            }, {
                "method": "tabSelect",
                "name": "tabSelect",
                "bubbles": false,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [{
                            "name": "deprecated",
                            "text": "This event should not be used anymore. Use icTabSelect instead."
                        }],
                    "text": ""
                },
                "complexType": {
                    "original": "IcTabSelectEventDetail",
                    "resolved": "IcTabSelectEventDetail",
                    "references": {
                        "IcTabSelectEventDetail": {
                            "location": "import",
                            "path": "../ic-tab/ic-tab.types",
                            "id": "src/components/ic-tab/ic-tab.types.ts::IcTabSelectEventDetail"
                        }
                    }
                }
            }];
    }
    static get methods() {
        return {
            "tabRemovedHandler": {
                "complexType": {
                    "signature": "(hadFocus?: boolean) => Promise<void>",
                    "parameters": [{
                            "name": "hadFocus",
                            "type": "boolean",
                            "docs": ""
                        }],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        }
                    },
                    "return": "Promise<void>"
                },
                "docs": {
                    "text": "",
                    "tags": [{
                            "name": "internal",
                            "text": "Used to set tab/tab panel IDs when a tab/tab panel has been removed"
                        }]
                }
            }
        };
    }
    static get elementRef() { return "el"; }
    static get watchers() {
        return [{
                "propName": "appearance",
                "methodName": "watchAppearanceHandler"
            }, {
                "propName": "selectedTabIndex",
                "methodName": "updateSelectedTab"
            }];
    }
    static get listeners() {
        return [{
                "name": "tabClick",
                "method": "tabClickHandler",
                "target": undefined,
                "capture": false,
                "passive": false
            }, {
                "name": "tabCreated",
                "method": "tabCreatedHandler",
                "target": undefined,
                "capture": false,
                "passive": false
            }, {
                "name": "tabPanelCreated",
                "method": "tabCreatedHandler",
                "target": undefined,
                "capture": false,
                "passive": false
            }, {
                "name": "tabEnabled",
                "method": "tabEnabledHandler",
                "target": undefined,
                "capture": false,
                "passive": false
            }];
    }
}
//# sourceMappingURL=ic-tab-context.js.map
