import { h, Host, } from "@stencil/core";
import closeIcon from "../../assets/close-icon.svg";
import { VARIANT_ICONS } from "../../utils/constants";
import { getSlot, isPropDefined, isSlotUsed, onComponentRequiredPropUndefined, } from "../../utils/helpers";
import { IcThemeForegroundEnum, } from "../../utils/types";
const AUTO_DISMISS_TIMER_REFRESH_RATE_MS = 1000;
const TOAST_HEADING_CHAR_LIMIT = 70;
const TOAST_MESSAGE_CHAR_LIMIT = 140;
/**
 * @slot action - IcButton or IcLink is placed below header and message. If used will default toast to manual `dismiss` type.
 * @slot neutral-icon - A custom neutral icon is placed on the left side of the component. If used will default toast to `neutral` variant.
 */
export class Toast {
    constructor() {
        this.interactiveElements = [];
        this.dismissAction = () => {
            this.icDismiss.emit();
        };
        this.handleProgressChange = () => {
            this.timerProgress -=
                (AUTO_DISMISS_TIMER_REFRESH_RATE_MS / this.autoDismissTimeout) * 100;
        };
        this.timerProgress = 100;
        this.visible = false;
        this.autoDismissTimeout = 5000;
        this.dismissButtonAriaLabel = "dismiss";
        this.dismissMode = "manual";
        this.heading = undefined;
        this.message = undefined;
        this.neutralIconAriaLabel = undefined;
        this.variant = undefined;
    }
    dismissModeChangeHandler(newValue) {
        this.isManual = newValue === "manual";
    }
    disconnectedCallback() {
        window.clearTimeout(this.dismissTimeout);
        window.clearInterval(this.timerRefreshInterval);
    }
    componentWillLoad() {
        var _a, _b;
        this.handleLongText(this.heading.length > TOAST_HEADING_CHAR_LIMIT, ((_a = this.message) === null || _a === void 0 ? void 0 : _a.length) > TOAST_MESSAGE_CHAR_LIMIT);
        if (this.autoDismissTimeout < 5000)
            this.autoDismissTimeout = 5000;
        if (isSlotUsed(this.el, "action"))
            this.dismissMode = "manual";
        this.isManual = this.dismissMode === "manual";
        if (isSlotUsed(this.el, "neutral-icon"))
            this.variant = "neutral";
        if (this.variant === "neutral") {
            this.neutralVariantLabel =
                (_b = this.neutralIconAriaLabel) !== null && _b !== void 0 ? _b : VARIANT_ICONS[this.variant].ariaLabel;
        }
        if (this.isManual) {
            const toastMessage = isPropDefined(this.message)
                ? `. ${this.message}`
                : "";
            this.el.setAttribute("aria-label", this.variant
                ? this.neutralVariantLabel || VARIANT_ICONS[this.variant].ariaLabel
                : this.heading);
            (this.variant || this.message) &&
                this.el.setAttribute("aria-description", this.variant ? `${this.heading}${toastMessage}` : this.message);
        }
    }
    componentDidLoad() {
        onComponentRequiredPropUndefined([{ prop: this.heading, propName: "heading" }], "Toast");
        const actionContent = getSlot(this.el, "action");
        const dismissButton = this.el.shadowRoot.querySelector("ic-button");
        if (actionContent)
            this.interactiveElements.push(actionContent);
        if (dismissButton)
            this.interactiveElements.push(dismissButton);
    }
    handleDismiss() {
        this.visible = false;
        clearInterval(this.timerRefreshInterval);
        this.timerProgress = 100;
    }
    handleKeyboard(ev) {
        if (this.isManual && this.visible) {
            switch (ev.key) {
                case "Tab":
                    ev.preventDefault();
                    this.findNextInteractiveElement(ev.shiftKey).setFocus();
                    break;
                case "Escape":
                    !ev.repeat && this.dismissAction();
                    ev.stopImmediatePropagation();
                    break;
            }
        }
    }
    handleTimer(ev) {
        if (!this.isManual) {
            switch (ev.type) {
                case "mouseenter":
                    window.clearTimeout(this.dismissTimeout);
                    window.clearInterval(this.timerRefreshInterval);
                    this.timerProgress = 100;
                    break;
                case "mouseleave":
                    this.dismissTimeout = window.setTimeout(this.dismissAction, this.autoDismissTimeout);
                    this.timerRefreshInterval = window.setInterval(this.handleProgressChange, AUTO_DISMISS_TIMER_REFRESH_RATE_MS);
                    break;
            }
        }
    }
    /**
     * @internal Used to display the individual toast.
     * @returns The element that previously had focus before the toast appeared
     */
    async setVisible() {
        if (!this.visible)
            this.visible = true;
        if (!this.isManual) {
            this.dismissTimeout = window.setTimeout(this.dismissAction, this.autoDismissTimeout);
            this.timerRefreshInterval = window.setInterval(this.handleProgressChange, AUTO_DISMISS_TIMER_REFRESH_RATE_MS);
            return null;
        }
        else {
            window.setTimeout(() => this.interactiveElements[0].setFocus(), 200);
            return document.activeElement;
        }
    }
    handleLongText(headingTooLong, messageTooLong) {
        if (messageTooLong || headingTooLong) {
            console.error(`Too many characters in toast ${headingTooLong ? "heading" : ""}${headingTooLong && messageTooLong ? " and " : ""}${messageTooLong ? "message" : ""}. Refer to character limits specified in the prop description`);
        }
    }
    findNextInteractiveElement(isBackwards) {
        const firstEl = this.interactiveElements[0];
        const lastEl = this.interactiveElements[this.interactiveElements.length - 1];
        if (this.isActive(isBackwards ? firstEl : lastEl))
            return isBackwards ? lastEl : firstEl;
        let currentIndex;
        return this.interactiveElements.some((el, index) => {
            if (!this.isActive(el))
                return false;
            currentIndex = index;
            return true;
        })
            ? this.interactiveElements[currentIndex + (isBackwards ? -1 : 1)]
            : firstEl;
    }
    isActive(targetEl) {
        return targetEl === this.el
            ? !!this.el.shadowRoot.activeElement
            : document.activeElement === targetEl;
    }
    render() {
        const { variant, heading, message, visible, isManual, dismissButtonAriaLabel, } = this;
        return (h(Host, { class: { ["hidden"]: !visible }, role: isManual ? "dialog" : "status" }, h("div", { class: "container" }, variant && visible && (h("div", { class: "toast-icon-container" }, h("div", { class: {
                ["divider"]: true,
                [`divider-${variant}`]: true,
            } }), variant === "neutral" ? (h("slot", { name: "neutral-icon" })) : (h("span", { class: "toast-icon", innerHTML: VARIANT_ICONS[variant].icon })))), h("div", { class: {
                ["toast-content"]: true,
                ["no-icon"]: variant === "neutral" && !isSlotUsed(this.el, "neutral-icon"),
            } }, h("div", { class: "toast-message" }, h("ic-typography", { variant: "subtitle-large" }, visible && h("h5", null, heading)), message && (h("ic-typography", { variant: "body" }, visible && h("p", null, message)))), isSlotUsed(this.el, "action") && (h("div", { class: "toast-action-container" }, h("slot", { name: "action" })))), !isManual ? (h("ic-loading-indicator", { class: "toast-dismiss-timer", appearance: "light", size: "icon", progress: this.timerProgress })) : (h("ic-button", { id: "dismiss-button", innerHTML: closeIcon, onClick: this.dismissAction, variant: "icon", appearance: IcThemeForegroundEnum.Light, "aria-label": dismissButtonAriaLabel })))));
    }
    static get is() { return "ic-toast"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["ic-toast.css"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["ic-toast.css"]
        };
    }
    static get properties() {
        return {
            "autoDismissTimeout": {
                "type": "number",
                "mutable": true,
                "complexType": {
                    "original": "number",
                    "resolved": "number",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If toast dismissMode is set to `automatic`, use this prop to define the time before the toast dismisses (in MILLISECONDS)\n(NOTE: Has a minimum value of `5000ms`)"
                },
                "attribute": "auto-dismiss-timeout",
                "reflect": false,
                "defaultValue": "5000"
            },
            "dismissButtonAriaLabel": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "If toast can be manually dismissed, this prop sets a custom aria-label for the ic-button component"
                },
                "attribute": "dismiss-button-aria-label",
                "reflect": false,
                "defaultValue": "\"dismiss\""
            },
            "dismissMode": {
                "type": "string",
                "mutable": true,
                "complexType": {
                    "original": "IcActivationTypes",
                    "resolved": "\"automatic\" | \"manual\"",
                    "references": {
                        "IcActivationTypes": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcActivationTypes"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "How the toast will be dismissed. If manual will display a dismiss button."
                },
                "attribute": "dismiss-mode",
                "reflect": false,
                "defaultValue": "\"manual\""
            },
            "heading": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": true,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The title to display at the start of the toast. (NOTE: Should be no more than `70` characters)"
                },
                "attribute": "heading",
                "reflect": false
            },
            "message": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The main body message of the toast. (NOTE: Should be no more than `140` characters)"
                },
                "attribute": "message",
                "reflect": false
            },
            "neutralIconAriaLabel": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "Provides a custom alt-text to be announced to screen readers, if slotting a custom neutral icon"
                },
                "attribute": "neutral-icon-aria-label",
                "reflect": false
            },
            "variant": {
                "type": "string",
                "mutable": true,
                "complexType": {
                    "original": "IcStatusVariants",
                    "resolved": "\"error\" | \"info\" | \"neutral\" | \"success\" | \"warning\"",
                    "references": {
                        "IcStatusVariants": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcStatusVariants"
                        }
                    }
                },
                "required": false,
                "optional": true,
                "docs": {
                    "tags": [],
                    "text": "The variant of the toast being rendered"
                },
                "attribute": "variant",
                "reflect": false
            }
        };
    }
    static get states() {
        return {
            "timerProgress": {},
            "visible": {}
        };
    }
    static get events() {
        return [{
                "method": "icDismiss",
                "name": "icDismiss",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Is emitted when the user dismisses the toast"
                },
                "complexType": {
                    "original": "void",
                    "resolved": "void",
                    "references": {}
                }
            }];
    }
    static get methods() {
        return {
            "setVisible": {
                "complexType": {
                    "signature": "() => Promise<HTMLElement>",
                    "parameters": [],
                    "references": {
                        "Promise": {
                            "location": "global",
                            "id": "global::Promise"
                        },
                        "HTMLElement": {
                            "location": "global",
                            "id": "global::HTMLElement"
                        }
                    },
                    "return": "Promise<HTMLElement>"
                },
                "docs": {
                    "text": "",
                    "tags": [{
                            "name": "internal",
                            "text": "Used to display the individual toast."
                        }, {
                            "name": "returns",
                            "text": "The element that previously had focus before the toast appeared"
                        }]
                }
            }
        };
    }
    static get elementRef() { return "el"; }
    static get watchers() {
        return [{
                "propName": "dismissMode",
                "methodName": "dismissModeChangeHandler"
            }];
    }
    static get listeners() {
        return [{
                "name": "icDismiss",
                "method": "handleDismiss",
                "target": undefined,
                "capture": true,
                "passive": false
            }, {
                "name": "keydown",
                "method": "handleKeyboard",
                "target": "document",
                "capture": false,
                "passive": false
            }, {
                "name": "mouseenter",
                "method": "handleTimer",
                "target": undefined,
                "capture": false,
                "passive": true
            }, {
                "name": "mouseleave",
                "method": "handleTimer",
                "target": undefined,
                "capture": false,
                "passive": true
            }];
    }
}
//# sourceMappingURL=ic-toast.js.map
