import { Host, h, } from "@stencil/core";
import { IcThemeForegroundEnum, } from "../../utils/types";
import { checkResizeObserver, DEVICE_SIZES, getCurrentDeviceSize, getThemeForegroundColor, getSlot, onComponentPropUndefinedChange, onComponentRequiredPropUndefined, isSlotUsed, isEmptyString, isPropDefined, } from "../../utils/helpers";
/**
 * @slot app-icon - Content will be rendered to left of app title. Anything that is slotted here will be hidden from screen readers.
 * @slot app-title - Handle routing by nesting a route in the app title.
 * @slot short-app-title - Handle routing by nesting a route in the short app title (to be displayed in place of app title on small screen sizes).
 * @slot search - Content will be rendered in search area to left of buttons.
 * @slot toggle-icon - Icon to be displayed on the button to toggle search slot content on smaller devices
 * @slot navigation - Content will be rendered in navigation panel.
 * @slot buttons - Content will be rendered to right of search bar.
 */
export class TopNavigation {
    constructor() {
        this.hasAppIcon = false;
        this.hasIconButtons = false;
        this.hasNavigation = false;
        this.hasSearchSlotContent = false;
        this.resizeObserver = null;
        this.searchBar = null;
        this.checkSlots = () => {
            this.hasAppIcon = isSlotUsed(this.el, "app-icon");
            this.hasNavigation = isSlotUsed(this.el, "navigation");
            this.hasIconButtons = isSlotUsed(this.el, "buttons");
            this.hasSearchSlotContent = isSlotUsed(this.el, "search");
        };
        this.initialiseSearchBar = () => {
            if (this.hasSearchSlotContent) {
                const slot = getSlot(this.el, "search");
                if ((slot === null || slot === void 0 ? void 0 : slot.tagName) === "IC-SEARCH-BAR") {
                    this.searchBar = slot;
                }
                else if ((slot === null || slot === void 0 ? void 0 : slot.tagName) === "FORM") {
                    this.searchBar = slot.querySelector("ic-search-bar");
                }
                if (this.searchBar !== null) {
                    this.searchBar.hideLabel = true;
                }
            }
        };
        this.menuButtonClick = () => {
            this.showNavMenu(true);
        };
        this.searchButtonMouseDownHandler = () => {
            this.searchButtonClick = true;
        };
        this.searchButtonClickHandler = () => {
            this.toggleSearchBar();
            this.searchButtonClick = false;
        };
        this.resizeObserverCallback = (currSize) => {
            if (currSize !== this.deviceSize) {
                this.deviceSize = currSize;
                if (currSize > this.customMobileBreakpoint) {
                    this.showNavMenu(false);
                    if (this.mobileSearchBarVisible) {
                        this.toggleSearchBar();
                    }
                }
                this.topNavResized.emit({
                    size: currSize,
                });
                if (this.searchBar && document.activeElement === this.searchBar) {
                    this.searchBar.setAttribute("hidden", "true");
                    //remove attribute again as this trigger a redraw & applies css
                    this.searchBar.removeAttribute("hidden");
                    setTimeout(() => {
                        this.searchBar.focus();
                    }, 100);
                }
            }
        };
        this.runResizeObserver = () => {
            this.resizeObserver = new ResizeObserver(() => {
                this.resizeObserverCallback(getCurrentDeviceSize());
            });
            this.resizeObserver.observe(this.el);
        };
        this.deviceSize = DEVICE_SIZES.XL;
        this.foregroundColor = getThemeForegroundColor();
        this.hasFullWidthSearchBar = false;
        this.mobileSearchBarVisible = false;
        this.mobileSearchHiddenOnBlur = false;
        this.navMenuVisible = false;
        this.searchButtonClick = false;
        this.searchValue = "";
        this.contentAligned = "full-width";
        this.customMobileBreakpoint = DEVICE_SIZES.L;
        this.href = "/";
        this.inline = false;
        this.shortAppTitle = "";
        this.status = "";
        this.version = "";
        this.appTitle = undefined;
    }
    watchPropHandler(newValue, oldValue) {
        //added for Gatsby rehydration issue where prop is initially undefined but then changes to actual value
        //this watch can be removed once Gatsby hydration issue is resolved
        onComponentPropUndefinedChange(oldValue, newValue, this.initialiseSearchBar);
    }
    disconnectedCallback() {
        var _a;
        (_a = this.resizeObserver) === null || _a === void 0 ? void 0 : _a.disconnect();
    }
    componentWillLoad() {
        this.checkSlots();
        this.deviceSize = getCurrentDeviceSize();
        this.initialiseSearchBar();
    }
    componentDidLoad() {
        checkResizeObserver(this.runResizeObserver);
        !isSlotUsed(this.el, "app-title") &&
            onComponentRequiredPropUndefined([{ prop: this.appTitle, propName: "app-title" }], "Top Navigation");
    }
    componentWillRender() {
        this.checkSlots();
    }
    navBarMenuCloseHandler() {
        this.showNavMenu(false);
        this.el.shadowRoot.querySelector("#menu-button").focus();
    }
    searchInputBlurHandler({ detail, }) {
        if (detail !== null) {
            if (this.mobileSearchBarVisible && !this.searchButtonClick) {
                //don't hide if blur was triggered by click on search button - let the click handler toggle the state
                this.toggleSearchBar();
            }
            this.searchValue = detail.value;
        }
    }
    searchValueChangeHandler({ detail }) {
        this.searchValue = detail.value;
    }
    themeChangeHandler({ detail }) {
        this.foregroundColor = detail.mode;
    }
    toggleSearchBar() {
        this.mobileSearchBarVisible = !this.mobileSearchBarVisible;
        if (this.searchBar !== null) {
            this.mobileSearchButtonEl.setAttribute("aria-label", `${this.mobileSearchBarVisible ? "Hide" : "Show"} search`);
            this.hasFullWidthSearchBar = this.mobileSearchBarVisible;
            this.searchBar.fullWidth = this.mobileSearchBarVisible;
            if (this.mobileSearchBarVisible) {
                setTimeout(() => {
                    this.searchBar.focus();
                }, 100);
            }
        }
    }
    showNavMenu(show) {
        this.navMenuVisible = show;
        (show ? this.icNavigationMenuOpened : this.icNavigationMenuClosed).emit();
        document.body.style.height = show ? "100%" : "auto";
        document.body.style.overflow = show ? "hidden" : "auto";
    }
    render() {
        const { appTitle, contentAligned, customMobileBreakpoint, deviceSize, el, foregroundColor, hasAppIcon, hasFullWidthSearchBar, hasIconButtons, hasNavigation, hasSearchSlotContent, href, inline, menuButtonClick, mobileSearchBarVisible, navMenuVisible, searchButtonClickHandler, searchButtonMouseDownHandler, shortAppTitle, status, version, } = this;
        const hasStatus = status !== "";
        const hasVersion = version !== "";
        const hasMenuContent = hasNavigation || hasIconButtons || hasStatus || hasVersion;
        const searchButtonSize = deviceSize <= DEVICE_SIZES.S ? "default" : "large";
        const hasTitle = appTitle !== "" && isPropDefined(appTitle);
        const overMobileBreakpoint = deviceSize <= customMobileBreakpoint;
        const appTitleVariant = overMobileBreakpoint
            ? deviceSize <= DEVICE_SIZES.S
                ? "subtitle-small"
                : "h4"
            : "h3";
        const mobileSearchButtonTitle = `${mobileSearchBarVisible ? "Hide" : "Show"} search`;
        const menuSize = deviceSize <= DEVICE_SIZES.S ? "small" : "default";
        const shortAppTitleSlot = isSlotUsed(el, "short-app-title");
        const hasAppTitleSlot = isSlotUsed(el, "app-title");
        const Component = hasAppTitleSlot ? "div" : "a";
        const attrs = Component == "a" && {
            href: href,
        };
        return (h(Host, { class: {
                "fullwidth-searchbar": hasFullWidthSearchBar,
                "mobile-mode": overMobileBreakpoint,
                [IcThemeForegroundEnum.Dark]: foregroundColor === IcThemeForegroundEnum.Dark,
            } }, h("div", { class: "top-navigation" }, h("ic-section-container", { aligned: contentAligned, "full-height": true }, h("header", { role: "banner" }, h("div", { class: "top-panel-container" }, h("div", { class: "app-details-container" }, (hasTitle || hasAppTitleSlot) && (h(Component, Object.assign({ class: "title-link" }, attrs), hasAppIcon && (h("div", { class: "app-icon-container" }, h("slot", { name: "app-icon" }))), deviceSize <= DEVICE_SIZES.S &&
            (!isEmptyString(shortAppTitle) || shortAppTitleSlot) ? (h("ic-typography", { variant: "subtitle-small", "aria-label": (!hasAppTitleSlot || !shortAppTitleSlot) &&
                `${appTitle} (${shortAppTitle})` }, h("h1", null, shortAppTitleSlot ? (h("slot", { name: "short-app-title" })) : (shortAppTitle)))) : (h("ic-typography", { variant: appTitleVariant }, h("h1", { class: "title-wrap" }, hasAppTitleSlot ? (h("slot", { name: "app-title" })) : (appTitle)))))), hasStatus && (h("div", { class: "app-status" }, h("ic-typography", { "aria-label": "app tag", variant: "label-uppercase", class: "app-status-text" }, status))), hasVersion && (h("div", { class: "app-version" }, h("ic-typography", { variant: "label", class: "app-version-text", "aria-label": "app version" }, version)))), (hasSearchSlotContent || hasMenuContent) && (h("div", { class: "search-menu-container" }, h("div", { class: "search-actions-container" }, !overMobileBreakpoint && h("slot", { name: "search" }), hasSearchSlotContent && overMobileBreakpoint && (h("ic-button", { id: "search-toggle-button", ref: (el) => (this.mobileSearchButtonEl = el), onMouseDown: searchButtonMouseDownHandler, variant: "icon", size: searchButtonSize, "aria-label": mobileSearchButtonTitle, appearance: foregroundColor, onClick: searchButtonClickHandler }, h("slot", { name: "toggle-icon" }, h("svg", { xmlns: "http://www.w3.org/2000/svg", viewBox: "0 0 24 24", fill: "#ffffff" }, h("path", { d: "M0 0h24v24H0V0z", fill: "none" }), h("path", { d: "M15.5 14h-.79l-.28-.27C15.41 12.59 16 11.11 16 9.5 16 5.91 13.09 3 9.5 3S3 5.91 3 9.5 5.91 16 9.5 16c1.61 0 3.09-.59 4.23-1.57l.27.28v.79l5 4.99L20.49 19l-4.99-5zm-6 0C7.01 14 5 11.99 5 9.5S7.01 5 9.5 5 14 7.01 14 9.5 11.99 14 9.5 14z" }))))), hasIconButtons && !overMobileBreakpoint && (h("div", { class: "icon-buttons-container" }, h("slot", { name: "buttons" }))), hasMenuContent && overMobileBreakpoint && (h("div", { class: "menu-button-container" }, h("span", { id: "navigation-landmark-button-text", class: "navigation-landmark-button-text", "aria-hidden": "true" }, "Main navigation button"), h("nav", { "aria-labelledby": "navigation-landmark-button-text", "aria-hidden": `${navMenuVisible}` }, h("ic-button", { id: "menu-button", appearance: foregroundColor, variant: "secondary", "aria-expanded": "false", "aria-haspopup": "true", "aria-label": `Open ${hasNavigation ? "navigation" : "app"} menu`, size: menuSize, onClick: menuButtonClick }, "Menu", h("svg", { xmlns: "http://www.w3.org/2000/svg", height: "24px", viewBox: "0 0 24 24", width: "24px", fill: "#ffffff", slot: "icon" }, h("path", { d: "M0 0h24v24H0V0z", fill: "none" }), h("path", { d: "M3 18h18v-2H3v2zm0-5h18v-2H3v2zm0-7v2h18V6H3z" })))))))))), mobileSearchBarVisible && (h("div", { class: "search-bar-container" }, h("slot", { name: "search" }))), hasNavigation && !overMobileBreakpoint && (h("div", { class: "navigation-tabs" }, h("span", { id: "navigation-landmark-text", class: "navigation-landmark-text", "aria-hidden": "true" }, "Main pages"), h("nav", { "aria-labelledby": "navigation-landmark-text", class: "nav-panel-container" }, h("ic-horizontal-scroll", { appearance: foregroundColor }, h("ul", { class: "navigation-item-list", tabindex: "-1" }, h("slot", { name: "navigation" }))))))))), navMenuVisible && (h("ic-navigation-menu", { version: version, status: status, class: {
                ["inline"]: inline,
            } }, hasIconButtons && (h("div", { class: "menu-buttons-slot", slot: "buttons" }, h("slot", { name: "buttons" }))), h("ul", { slot: "navigation" }, h("slot", { name: "navigation" }))))));
    }
    static get is() { return "ic-top-navigation"; }
    static get encapsulation() { return "shadow"; }
    static get originalStyleUrls() {
        return {
            "$": ["ic-top-navigation.css"]
        };
    }
    static get styleUrls() {
        return {
            "$": ["ic-top-navigation.css"]
        };
    }
    static get properties() {
        return {
            "contentAligned": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "IcAlignment",
                    "resolved": "\"center\" | \"full-width\" | \"left\"",
                    "references": {
                        "IcAlignment": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcAlignment"
                        }
                    }
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The alignment of the top navigation content."
                },
                "attribute": "content-aligned",
                "reflect": false,
                "defaultValue": "\"full-width\""
            },
            "customMobileBreakpoint": {
                "type": "number",
                "mutable": false,
                "complexType": {
                    "original": "IcDeviceSizes",
                    "resolved": "0 | 1200 | 576 | 768 | 992",
                    "references": {
                        "IcDeviceSizes": {
                            "location": "import",
                            "path": "../../utils/types",
                            "id": "src/utils/types.ts::IcDeviceSizes"
                        }
                    }
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "Can set a custom breakpoint for the top navigation to switch to mobile mode.\nMust be one of our specified breakpoints in px: `0`, `576`, `768`, `992`, `1200`."
                },
                "attribute": "custom-mobile-breakpoint",
                "reflect": false,
                "defaultValue": "DEVICE_SIZES.L"
            },
            "href": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The URL to navigate to when the app title is clicked."
                },
                "attribute": "href",
                "reflect": false,
                "defaultValue": "\"/\""
            },
            "inline": {
                "type": "boolean",
                "mutable": false,
                "complexType": {
                    "original": "boolean",
                    "resolved": "boolean",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "If `true`, the flyout navigation menu on small devices will be contained by the parent element."
                },
                "attribute": "inline",
                "reflect": false,
                "defaultValue": "false"
            },
            "shortAppTitle": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The short title of the app to be displayed at small screen sizes in place of the app title."
                },
                "attribute": "short-app-title",
                "reflect": false,
                "defaultValue": "\"\""
            },
            "status": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The status info to be displayed."
                },
                "attribute": "status",
                "reflect": false,
                "defaultValue": "\"\""
            },
            "version": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The version info to be displayed."
                },
                "attribute": "version",
                "reflect": false,
                "defaultValue": "\"\""
            },
            "appTitle": {
                "type": "string",
                "mutable": false,
                "complexType": {
                    "original": "string",
                    "resolved": "string",
                    "references": {}
                },
                "required": false,
                "optional": false,
                "docs": {
                    "tags": [],
                    "text": "The app title to be displayed. This is required, unless a slotted app title link is used."
                },
                "attribute": "app-title",
                "reflect": false
            }
        };
    }
    static get states() {
        return {
            "deviceSize": {},
            "foregroundColor": {},
            "hasFullWidthSearchBar": {},
            "mobileSearchBarVisible": {},
            "mobileSearchHiddenOnBlur": {},
            "navMenuVisible": {},
            "searchButtonClick": {},
            "searchValue": {}
        };
    }
    static get events() {
        return [{
                "method": "icNavigationMenuClosed",
                "name": "icNavigationMenuClosed",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": "- Emitted when the menu is closed."
                        }],
                    "text": ""
                },
                "complexType": {
                    "original": "void",
                    "resolved": "void",
                    "references": {}
                }
            }, {
                "method": "icNavigationMenuOpened",
                "name": "icNavigationMenuOpened",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [{
                            "name": "internal",
                            "text": "- Emitted when the menu is opened."
                        }],
                    "text": ""
                },
                "complexType": {
                    "original": "void",
                    "resolved": "void",
                    "references": {}
                }
            }, {
                "method": "topNavResized",
                "name": "topNavResized",
                "bubbles": true,
                "cancelable": true,
                "composed": true,
                "docs": {
                    "tags": [],
                    "text": "Emitted when the top navigation is resized."
                },
                "complexType": {
                    "original": "{ size: number }",
                    "resolved": "{ size: number; }",
                    "references": {}
                }
            }];
    }
    static get elementRef() { return "el"; }
    static get watchers() {
        return [{
                "propName": "appTitle",
                "methodName": "watchPropHandler"
            }];
    }
    static get listeners() {
        return [{
                "name": "icNavigationMenuClose",
                "method": "navBarMenuCloseHandler",
                "target": undefined,
                "capture": false,
                "passive": false
            }, {
                "name": "icSearchBarBlur",
                "method": "searchInputBlurHandler",
                "target": undefined,
                "capture": false,
                "passive": false
            }, {
                "name": "icChange",
                "method": "searchValueChangeHandler",
                "target": undefined,
                "capture": false,
                "passive": false
            }, {
                "name": "themeChange",
                "method": "themeChangeHandler",
                "target": "document",
                "capture": false,
                "passive": false
            }];
    }
}
//# sourceMappingURL=ic-top-navigation.js.map
