import { proxyCustomElement, HTMLElement, createEvent, h } from '@stencil/core/internal/client';
import { I as IcThemeForegroundEnum } from './types.js';

const CONTEXT_ID_ATTR = "context-id";
const TabContext = /*@__PURE__*/ proxyCustomElement(class TabContext extends HTMLElement {
    constructor() {
        super();
        this.__registerHost();
        this.icTabSelect = createEvent(this, "icTabSelect", 3);
        this.tabSelect = createEvent(this, "tabSelect", 3);
        this.newTabPanels = [];
        this.newTabs = [];
        this.emitEvents = (tabIndex) => {
            const tabLabel = this.el
                .querySelectorAll("ic-tab")
            // eslint-disable-next-line no-unexpected-multiline
            [tabIndex].textContent.trim();
            this.icTabSelect.emit({ tabIndex, tabLabel });
            this.tabSelect.emit({ tabIndex, tabLabel });
        };
        /** Sets attributes to link tab-group, tabs and tab-panels */
        this.linkTabs = () => {
            this.tabs.forEach((tab, index) => {
                const tabId = `ic-tab-${index}-context-${this.contextId}`;
                const tabPanelId = `ic-tab-panel-${index}-context-${this.contextId}`;
                tab.setAttribute("id", tabId);
                tab.tabId = `ic-tab--${index}-context-${this.contextId}`;
                tab.tabPosition = index;
                tab.setAttribute("aria-controls", tabPanelId);
                tab.setAttribute(CONTEXT_ID_ATTR, this.contextId);
                this.tabGroup.setAttribute(CONTEXT_ID_ATTR, this.contextId);
                this.tabPanels[index].setAttribute("id", tabPanelId);
                this.tabPanels[index].setAttribute("aria-labelledby", tabId);
                this.tabPanels[index].setAttribute(CONTEXT_ID_ATTR, this.contextId);
                if (this.appearance === IcThemeForegroundEnum.Light) {
                    tab.appearance = this.appearance;
                    this.tabPanels[index].appearance = this.appearance;
                }
            });
            if (this.appearance === IcThemeForegroundEnum.Light) {
                this.tabGroup.appearance = this.appearance;
            }
        };
        /**
         * Gets tabs and tabpanels with the same context ID using querySelector to selector the children in relation to the host
         */
        this.getChildren = () => {
            this.tabGroup = this.el.querySelector("ic-tab-group");
            this.tabs = Array.from(this.tabGroup.querySelectorAll("ic-tab"));
            this.tabPanels = Array.from(this.el.children).filter((child) => child.tagName === "IC-TAB-PANEL");
            this.enabledTabs = this.getEnabledTabs();
        };
        this.keydownHandler = (event) => {
            const isManual = this.activationType === "manual";
            const enabledTabIndex = this.enabledTabs.findIndex((tab) => tab.tabId ===
                this.tabs[isManual ? this.focusedTabIndex : this.selectedTab].tabId);
            const keyboardFunction = isManual
                ? this.keyboardFocusTab
                : this.keyboardSelectTab;
            let preventDefault = true;
            switch (event.key) {
                case "Home":
                    keyboardFunction(0);
                    break;
                case "End":
                    keyboardFunction(this.enabledTabs.length - 1);
                    break;
                case "ArrowRight":
                    keyboardFunction(enabledTabIndex < this.enabledTabs.length - 1
                        ? enabledTabIndex + 1
                        : 0);
                    break;
                case "ArrowLeft":
                    keyboardFunction((enabledTabIndex > 0 ? enabledTabIndex : this.enabledTabs.length) - 1);
                    break;
                default:
                    if (isManual && (event.key === "Enter" || event.key === " ")) {
                        this.keyboardSelectTab(this.focusedTabIndex);
                    }
                    else {
                        preventDefault = false;
                    }
            }
            if (preventDefault)
                event.preventDefault();
        };
        /** Sets the tab that is selected on initial render */
        this.setInitialTab = () => {
            if (this.selectedTabIndex !== undefined) {
                this.selectedTab = this.selectedTabIndex;
                this.focusedTabIndex = this.selectedTabIndex;
            }
            else {
                const firstEnabledTabIndex = this.tabs.findIndex((tab) => tab.tabId === this.enabledTabs[0].tabId);
                this.selectedTab = firstEnabledTabIndex;
                this.focusedTabIndex = firstEnabledTabIndex;
            }
        };
        /** Passes the selected tab to the tab and tab panel components */
        this.configureTabs = () => {
            this.enabledTabs.forEach((tab) => {
                tab.selected = tab.tabPosition === this.selectedTab;
            });
            this.tabPanels.forEach((tabPanel, index) => {
                tabPanel.hidden = index !== this.selectedTab;
            });
        };
        this.getEnabledTabs = () => Array.from(this.tabs).filter((child) => !child.disabled);
        /** Sets focus on tab and selects it */
        this.keyboardSelectTab = (enabledTabIndex) => {
            const newIndex = this.tabs.findIndex((tab) => tab.tabId === this.enabledTabs[enabledTabIndex].tabId);
            this.enabledTabs[enabledTabIndex].focus();
            if (this.selectedTabIndex === undefined) {
                this.selectedTab = newIndex;
            }
            else {
                this.emitEvents(newIndex);
            }
        };
        /** Sets focus on tab without selecting it (for manual activation) */
        this.keyboardFocusTab = (enabledTabIndex) => {
            this.enabledTabs[enabledTabIndex].focus();
            this.focusedTabIndex = this.tabs.findIndex((tab) => tab.tabId === this.enabledTabs[enabledTabIndex].tabId);
        };
        this.selectedTab = undefined;
        this.activationType = "automatic";
        this.appearance = "dark";
        this.contextId = "default";
        this.selectedTabIndex = undefined;
    }
    watchAppearanceHandler() {
        this.tabs.forEach((tab, index) => {
            tab.appearance = this.appearance;
            this.tabPanels[index].appearance = this.appearance;
        });
        this.tabGroup.appearance = this.appearance;
    }
    updateSelectedTab(newValue) {
        this.selectedTab = newValue;
    }
    componentDidLoad() {
        if (this.selectedTabIndex !== undefined) {
            this.selectedTab = this.selectedTabIndex;
        }
        this.getChildren();
        this.linkTabs();
        this.tabGroup.addEventListener("keydown", this.keydownHandler);
        this.setInitialTab();
        this.configureTabs();
    }
    componentWillUpdate() {
        this.configureTabs();
    }
    disconnectedCallback() {
        this.tabGroup.removeEventListener("keydown", this.keydownHandler);
    }
    tabClickHandler(event) {
        if (this.selectedTabIndex === undefined &&
            event.detail.contextId === this.contextId) {
            this.selectedTab = event.detail.position;
        }
        this.emitEvents(event.detail.position);
        event.stopImmediatePropagation();
    }
    tabCreatedHandler(ev) {
        if (this.tabs && this.tabPanels) {
            (ev.detail.setFocus ? this.newTabs : this.newTabPanels).push(ev.detail);
            if (this.newTabs.length === this.newTabPanels.length) {
                this.tabs.push(...this.newTabs);
                this.tabPanels.push(...this.newTabPanels);
                this.enabledTabs = this.getEnabledTabs();
                this.linkTabs();
                if (!this.tabs[this.selectedTab] || !this.tabPanels[this.selectedTab])
                    this.setInitialTab();
                this.configureTabs();
                this.newTabs = [];
                this.newTabPanels = [];
            }
        }
    }
    tabEnabledHandler() {
        this.enabledTabs = this.getEnabledTabs();
    }
    /**
     * @internal Used to set tab/tab panel IDs when a tab/tab panel has been removed
     */
    async tabRemovedHandler(hadFocus) {
        this.getChildren();
        this.linkTabs();
        if (this.tabs[this.selectedTab] && this.tabPanels[this.selectedTab]) {
            this.tabs[this.selectedTab].selected = true;
            this.tabPanels[this.selectedTab].hidden = false;
        }
        else {
            this.setInitialTab();
        }
        if (hadFocus) {
            this.tabs[this.selectedTab].setFocus();
        }
    }
    render() {
        return h("slot", null);
    }
    get el() { return this; }
    static get watchers() { return {
        "appearance": ["watchAppearanceHandler"],
        "selectedTabIndex": ["updateSelectedTab"]
    }; }
}, [4, "ic-tab-context", {
        "activationType": [1, "activation-type"],
        "appearance": [1],
        "contextId": [513, "context-id"],
        "selectedTabIndex": [2, "selected-tab-index"],
        "selectedTab": [32],
        "tabRemovedHandler": [64]
    }, [[0, "tabClick", "tabClickHandler"], [0, "tabCreated", "tabCreatedHandler"], [0, "tabPanelCreated", "tabCreatedHandler"], [0, "tabEnabled", "tabEnabledHandler"]], {
        "appearance": ["watchAppearanceHandler"],
        "selectedTabIndex": ["updateSelectedTab"]
    }]);
function defineCustomElement$1() {
    if (typeof customElements === "undefined") {
        return;
    }
    const components = ["ic-tab-context"];
    components.forEach(tagName => { switch (tagName) {
        case "ic-tab-context":
            if (!customElements.get(tagName)) {
                customElements.define(tagName, TabContext);
            }
            break;
    } });
}

const IcTabContext = TabContext;
const defineCustomElement = defineCustomElement$1;

export { IcTabContext, defineCustomElement };

//# sourceMappingURL=ic-tab-context.js.map