/* 
 * Copyright 2019, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Add an event listener to `element` that only fires when the target of the
 * event matches `selector`.
 * 
 * @param {Element} element
 * @param {String} eventName
 * @param {String} selector
 * @param {Function} handler
 */
export function addEventDelegate(element, eventName, selector, handler) {
	element.addEventListener(eventName, event => {
		if (event.target.matches(selector)) {
			handler(event);
		}
	});
}

/**
 * Removes all of an element's child nodes.
 * 
 * @param {Element} element
 */
export function clearChildren(element) {
	while (element.firstChild) {
		element.removeChild(element.firstChild);
	}
}

/**
 * Deserialize an HTML string into a document fragment.
 * 
 * @param {String} htmlString
 * @return {DocumentFragment}
 */
export function deserialize(htmlString) {
	return document.createRange().createContextualFragment(htmlString);
}

/**
 * Serialize a document fragment into an HTML string.
 * 
 * @param {DocumentFragment} documentFragment
 * @return {String}
 */
export function serialize(documentFragment) {
	return new XMLSerializer().serializeToString(documentFragment);
}
