"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addEventDelegate = addEventDelegate;
exports.clearChildren = clearChildren;
exports.deserialize = deserialize;
exports.parseJsonFromElement = parseJsonFromElement;
exports.serialize = serialize;

/* 
 * Copyright 2019, Emanuel Rabina (http://www.ultraq.net.nz/)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Add an event listener to `element` that only fires when the target of the
 * event matches `selector`.
 * 
 * @param {Element} element
 * @param {String} eventName
 * @param {String} selector
 * @param {Function} handler
 */
function addEventDelegate(element, eventName, selector, handler) {
  element.addEventListener(eventName, function (event) {
    if (event.target.matches(selector)) {
      handler(event);
    }
  });
}
/**
 * Removes all of an element's child nodes.
 * 
 * @param {Element} element
 */


function clearChildren(element) {
  while (element.firstChild) {
    element.removeChild(element.firstChild);
  }
}
/**
 * Deserialize an HTML string into a document fragment.
 * 
 * @param {String} htmlString
 * @return {Document}
 */


function deserialize(htmlString) {
  return new DOMParser().parseFromString(htmlString, 'text/html');
}
/**
 * Parse the text content of the element picked out by the given selector as
 * JSON data, returning it as an object.
 * 
 * @param {String} selector
 *   A CSS selector for picking out the HTML element that contains the JSON data
 *   to load.
 * @return {Object}
 *   The JSON data converted to an object, or `null` if no data could be read.
 */


function parseJsonFromElement(selector) {
  var _document$querySelect, _document$querySelect2;

  return JSON.parse(((_document$querySelect = document.querySelector(selector)) === null || _document$querySelect === void 0 ? void 0 : (_document$querySelect2 = _document$querySelect.textContent) === null || _document$querySelect2 === void 0 ? void 0 : _document$querySelect2.trim()) || null);
}
/**
 * Serialize a document or document fragment into an HTML string.
 * 
 * @param {Document|DocumentFragment} documentOrFragment
 * @return {String}
 */


function serialize(documentOrFragment) {
  if (documentOrFragment.nodeType === Node.DOCUMENT_NODE) {
    var result = '';
    var contentType = documentOrFragment.contentType,
        docType = documentOrFragment.docType,
        firstElementChild = documentOrFragment.firstElementChild;

    if (docType) {
      result += "<!DOCTYPE ".concat(docType.name, ">");
    } else if (firstElementChild.tagName === 'HTML' || contentType === 'text/html') {
      result += "<!DOCTYPE html>";
    }

    result += firstElementChild.outerHTML;
    return result;
  }

  return documentOrFragment.outerHTML;
}

//# sourceMappingURL=dom-utils.cjs.js.map