import { existsSync, promises } from 'fs';
import { resolve, dirname, relative, basename } from 'pathe';
import fg from 'fast-glob';
import consola from 'consola';
import { cyan, green, dim } from 'colorette';
import { debounce } from 'perfect-debounce';
import { createGenerator, toArray } from '@unocss/core';
import { loadConfig } from '@unocss/config';
import presetUno from '@unocss/preset-uno';

const version = "0.38.1";

class PrettyError extends Error {
  constructor(message) {
    super(message);
    this.name = this.constructor.name;
    if (typeof Error.captureStackTrace === "function")
      Error.captureStackTrace(this, this.constructor);
    else
      this.stack = new Error(message).stack;
  }
}
function handleError(error) {
  if (error instanceof PrettyError)
    consola.error(error.message);
  process.exitCode = 1;
}

const defaultConfig = {
  envMode: "build",
  presets: [
    presetUno()
  ]
};

const name = "unocss";
let uno;
const fileCache = /* @__PURE__ */ new Map();
const getAbsolutePath = (file) => resolve(process.cwd(), file);
async function generate(options) {
  const outFile = options.outFile ?? getAbsolutePath("uno.css");
  const { css, matched } = await uno.generate([...fileCache].join("\n"));
  const dir = dirname(outFile);
  if (!existsSync(dir))
    await promises.mkdir(dir, { recursive: true });
  await promises.writeFile(outFile, css, "utf-8");
  if (!options.watch) {
    consola.success(`${[...matched].length} utilities generated to ${cyan(relative(process.cwd(), outFile))}
`);
  }
}
async function resolveOptions(options) {
  if (!options.patterns?.length) {
    throw new PrettyError(`No glob patterns, try ${cyan(`${name} <path/to/**/*>`)}`);
  }
  return options;
}
async function build(_options) {
  const options = await resolveOptions(_options);
  const { config, sources: configSources } = await loadConfig(process.cwd(), options.config);
  uno = createGenerator(config, defaultConfig);
  const files = await fg(options.patterns);
  await Promise.all(files.map(async (file) => {
    const absolutePath = getAbsolutePath(file);
    fileCache.set(absolutePath, await promises.readFile(absolutePath, "utf8"));
  }));
  consola.log(green(`${name} v${version}`));
  consola.start(`UnoCSS ${options.watch ? "in watch mode..." : "for production..."}`);
  const debouncedBuild = debounce(async () => {
    generate(options).catch(handleError);
  }, 100);
  const startWatcher = async () => {
    if (!options.watch)
      return;
    const { watch } = await import('chokidar');
    const { patterns } = options;
    const ignored = ["**/{.git,node_modules}/**"];
    const watcher = watch(patterns, {
      ignoreInitial: true,
      ignorePermissionErrors: true,
      ignored
    });
    if (configSources.length)
      watcher.add(configSources);
    watcher.on("all", async (type, file) => {
      if (configSources.includes(file)) {
        uno.setConfig((await loadConfig()).config);
        consola.info(`${cyan(basename(file))} changed, setting new config`);
      } else {
        consola.log(`${green(type)} ${dim(file)}`);
        const absolutePath = getAbsolutePath(file);
        if (type.startsWith("unlink"))
          fileCache.delete(absolutePath);
        else
          fileCache.set(absolutePath, await promises.readFile(absolutePath, "utf8"));
      }
      debouncedBuild();
    });
    consola.info(`Watching for changes in ${toArray(patterns).map((i) => cyan(i)).join(", ")}`);
  };
  await generate(options);
  startWatcher();
}

export { build as b, generate as g, handleError as h, resolveOptions as r, version as v };
