// src/index.ts
import { createGenerator, isString, toArray } from "@unocss/core";

// src/utils.ts
function camelize(str) {
  return str.replace(/-(\w)/g, (_, c) => c ? c.toUpperCase() : "");
}
function capitalize(str) {
  return str.charAt(0).toUpperCase() + str.slice(1);
}
function hyphenate(str) {
  return str.replace(/(?:^|\B)([A-Z])/g, "-$1").toLowerCase();
}
var prefixes = ["Webkit", "Moz", "ms"];
function autoPrefixer(style) {
  const prefixCache = {};
  function autoPrefix(rawName) {
    const cached = prefixCache[rawName];
    if (cached)
      return cached;
    let name = camelize(rawName);
    if (name !== "filter" && name in style)
      return prefixCache[rawName] = hyphenate(name);
    name = capitalize(name);
    for (let i = 0; i < prefixes.length; i++) {
      const prefixed = `${prefixes[i]}${name}`;
      if (prefixed in style)
        return prefixCache[rawName] = hyphenate(capitalize(prefixed));
    }
    return rawName;
  }
  return ({ entries }) => entries.forEach((e) => {
    if (!e[0].startsWith("--"))
      e[0] = autoPrefix(e[0]);
  });
}
function decodeHtml(html) {
  return html.replace(/&amp;/g, "&").replace(/&gt;/g, ">").replace(/&lt;/g, "<");
}

// src/index.ts
function init(inlineConfig = {}) {
  if (typeof window == "undefined") {
    console.warn("@unocss/runtime been used in non-browser environment, skipped.");
    return;
  }
  const defaultWindow = window;
  const defaultDocument = window.document;
  const html = () => defaultDocument.documentElement;
  const userConfig = defaultWindow.__unocss || {};
  const runtimeOptions = Object.assign({}, inlineConfig, userConfig.runtime);
  const userConfigDefaults = runtimeOptions.defaults || {};
  const cloakAttribute = runtimeOptions.cloakAttribute ?? "un-cloak";
  if (runtimeOptions.autoPrefix) {
    const postprocessors = userConfigDefaults.postprocess = toArray(userConfigDefaults.postprocess);
    postprocessors.unshift(autoPrefixer(defaultDocument.createElement("div").style));
  }
  runtimeOptions.configResolved?.(userConfig, userConfigDefaults);
  const uno = createGenerator(userConfig, userConfigDefaults);
  const styleElements = /* @__PURE__ */ new Map();
  let paused = true;
  let tokens = /* @__PURE__ */ new Set();
  let inspector;
  let _timer;
  let _resolvers = [];
  const scheduleUpdate = () => new Promise((resolve) => {
    _resolvers.push(resolve);
    if (_timer != null)
      clearTimeout(_timer);
    _timer = setTimeout(() => updateStyle().then(() => {
      const resolvers = _resolvers;
      _resolvers = [];
      resolvers.forEach((r) => r());
    }), 0);
  });
  function removeCloak(node) {
    if (node.nodeType !== 1)
      return;
    const el = node;
    if (el.hasAttribute(cloakAttribute))
      el.removeAttribute(cloakAttribute);
    el.querySelectorAll(`[${cloakAttribute}]`).forEach((n) => {
      n.removeAttribute(cloakAttribute);
    });
  }
  function getStyleElement(layer, previousLayer) {
    let styleElement = styleElements.get(layer);
    if (!styleElement) {
      styleElement = defaultDocument.createElement("style");
      styleElement.setAttribute("data-unocss-runtime-layer", layer);
      styleElements.set(layer, styleElement);
      if (previousLayer == null) {
        html().prepend(styleElement);
      } else {
        const previousStyle = getStyleElement(previousLayer);
        const parentNode = previousStyle.parentNode;
        if (parentNode)
          parentNode.insertBefore(styleElement, previousStyle.nextSibling);
        else
          html().prepend(styleElement);
      }
    }
    return styleElement;
  }
  async function updateStyle() {
    const result = await uno.generate(tokens);
    result.layers.reduce((previous, current) => {
      getStyleElement(current, previous).innerHTML = result.getLayer(current) ?? "";
      return current;
    }, void 0);
    tokens = result.matched;
    return {
      ...result,
      getStyleElement: (layer) => styleElements.get(layer),
      getStyleElements: () => styleElements
    };
  }
  async function extract(str) {
    const tokenSize = tokens.size;
    await uno.applyExtractors(str, void 0, tokens);
    if (tokenSize !== tokens.size)
      await scheduleUpdate();
  }
  async function extractAll() {
    const body = defaultDocument.body;
    const outerHTML = body && body.outerHTML;
    if (outerHTML) {
      await extract(`${outerHTML} ${decodeHtml(outerHTML)}`);
      removeCloak(html());
      removeCloak(body);
    }
  }
  const mutationObserver = new MutationObserver((mutations) => {
    if (paused)
      return;
    mutations.forEach(async (mutation) => {
      if (mutation.target.nodeType !== 1)
        return;
      const target = mutation.target;
      for (const item of styleElements) {
        if (target === item[1])
          return;
      }
      if (mutation.type === "childList") {
        mutation.addedNodes.forEach(async (node) => {
          if (node.nodeType !== 1)
            return;
          const el = node;
          if (inspector && !inspector(el))
            return;
          await extract(el.outerHTML);
          removeCloak(el);
        });
      } else {
        if (inspector && !inspector(target))
          return;
        if (mutation.attributeName !== cloakAttribute) {
          const attrs = Array.from(target.attributes).map((i) => i.value ? `${i.name}="${i.value}"` : i.name).join(" ");
          const tag = `<${target.tagName.toLowerCase()} ${attrs}>`;
          await extract(tag);
        }
        if (target.hasAttribute(cloakAttribute))
          target.removeAttribute(cloakAttribute);
      }
    });
  });
  let observing = false;
  function observe() {
    if (observing)
      return;
    const target = html() || defaultDocument.body;
    if (!target)
      return;
    mutationObserver.observe(target, {
      childList: true,
      subtree: true,
      attributes: true
    });
    observing = true;
  }
  function execute() {
    if (runtimeOptions.bypassDefined)
      getDefinedCssSelectors(uno.blocked);
    extractAll();
    observe();
  }
  function ready() {
    if (defaultDocument.readyState === "loading")
      defaultWindow.addEventListener("DOMContentLoaded", execute);
    else
      execute();
  }
  const unoCssRuntime = defaultWindow.__unocss_runtime = defaultWindow.__unocss_runtime = {
    version: uno.version,
    uno,
    async extract(userTokens) {
      if (!isString(userTokens)) {
        userTokens.forEach((t) => tokens.add(t));
        userTokens = "";
      }
      await extract(userTokens);
    },
    extractAll,
    inspect(callback) {
      inspector = callback;
    },
    toggleObserver(set) {
      if (set === void 0)
        paused = !paused;
      else
        paused = !!set;
      if (!observing && !paused)
        ready();
    },
    update: updateStyle
  };
  if (runtimeOptions.ready?.(unoCssRuntime) !== false) {
    paused = false;
    ready();
  }
}
function getDefinedCssSelectors(selectors = /* @__PURE__ */ new Set()) {
  for (let i = 0; i < document.styleSheets.length; i++) {
    const sheet = document.styleSheets[i];
    let list;
    try {
      list = sheet.cssRules || sheet.rules;
      if (!list)
        continue;
      Array.from(list).flatMap((r) => r.selectorText?.split(/,/g) || []).forEach((s) => {
        if (!s)
          return;
        s = s.trim();
        if (s.startsWith("."))
          s = s.slice(1);
        selectors.add(s);
      });
    } catch (e) {
      continue;
    }
  }
  return selectors;
}
export {
  init as default
};
