'use strict';
const behavior = require('../utils/behavior');
const forEach = require('array-foreach');
const ignore = require('receptor/ignore');
const select = require('../utils/select');

const CLICK = require('../events').CLICK;
const PREFIX = require('../config').prefix;

const BUTTON = '.js-search-button';
const FORM = '.js-search-form';
const INPUT = '[type=search]';
const CONTEXT = 'header'; // XXX
const VISUALLY_HIDDEN = `${PREFIX}-sr-only`;

let lastButton;

const showSearch = function (event) {
  toggleSearch(this, true);
  lastButton = this;
};

const hideSearch = function (event) {
  toggleSearch(this, false);
  lastButton = undefined;
};

const getForm = button => {
  const context = button.closest(CONTEXT);
  return context
    ? context.querySelector(FORM)
    : document.querySelector(FORM);
};

const toggleSearch = (button, active) => {
  const form = getForm(button);
  if (!form) {
    throw new Error(`No ${FORM} found for search toggle in ${CONTEXT}!`);
  }

  button.hidden = active;
  form.classList.toggle(VISUALLY_HIDDEN, !active);

  if (active) {
    const input = form.querySelector(INPUT);
    if (input) {
      input.focus();
    }
    // when the user clicks _outside_ of the form w/ignore(): hide the
    // search, then remove the listener
    const listener = ignore(form, e => {
      if (lastButton) {
        hideSearch.call(lastButton);
      }
      document.body.removeEventListener(CLICK, listener);
    });

    // Normally we would just run this code without a timeout, but
    // IE11 and Edge will actually call the listener *immediately* because
    // they are currently handling this exact type of event, so we'll
    // make sure the browser is done handling the current click event,
    // if any, before we attach the listener.
    setTimeout(() => {
      document.body.addEventListener(CLICK, listener);
    }, 0);
  }
};

const search = behavior({
  [ CLICK ]: {
    [ BUTTON ]: showSearch,
  },
}, {
  init: (target) => {
    forEach(select(BUTTON, target), button => {
      toggleSearch(button, false);
    });
  },
  teardown: (target) => {
    // forget the last button clicked
    lastButton = undefined;
  },
});

/**
 * TODO for 2.0, remove this statement and export `navigation` directly:
 *
 * module.exports = behavior({...});
 */
const assign = require('object-assign');
module.exports = assign(
  el => search.on(el),
  search
);
