'use strict';

/** Magic bytes 'ARROW1' indicating the Arrow 'file' format. */
const MAGIC = Uint8Array.of(65, 82, 82, 79, 87, 49);

/**
 * Apache Arrow version.
 */
const Version = /** @type {const} */ ({
  /** 0.1.0 (October 2016). */
  V1: 0,
  /** 0.2.0 (February 2017). Non-backwards compatible with V1. */
  V2: 1,
  /** 0.3.0 -> 0.7.1 (May - December 2017). Non-backwards compatible with V2. */
  V3: 2,
  /** >= 0.8.0 (December 2017). Non-backwards compatible with V3. */
  V4: 3,
  /**
   * >= 1.0.0 (July 2020). Backwards compatible with V4 (V5 readers can read V4
   * metadata and IPC messages). Implementations are recommended to provide a
   * V4 compatibility mode with V5 format changes disabled.
   *
   * Incompatible changes between V4 and V5:
   * - Union buffer layout has changed.
   *   In V5, Unions don't have a validity bitmap buffer.
   */
  V5: 4
});

/**
 * Endianness of Arrow-encoded data.
 */
const Endianness = /** @type {const} */ ({
  Little: 0,
  Big: 1
});

/**
 * Message header type codes.
 */
const MessageHeader = /** @type {const} */ ({
  NONE: 0,
  /**
   * A Schema describes the columns in a record batch.
   */
  Schema: 1,
  /**
   * For sending dictionary encoding information. Any Field can be
   * dictionary-encoded, but in this case none of its children may be
   * dictionary-encoded.
   * There is one vector / column per dictionary, but that vector / column
   * may be spread across multiple dictionary batches by using the isDelta
   * flag.
   */
  DictionaryBatch: 2,
  /**
   * A data header describing the shared memory layout of a "record" or "row"
   * batch. Some systems call this a "row batch" internally and others a "record
   * batch".
   */
  RecordBatch: 3,
  /**
   * EXPERIMENTAL: Metadata for n-dimensional arrays, aka "tensors" or
   * "ndarrays". Arrow implementations in general are not required to implement
   * this type.
   *
   * Not currently supported by Flechette.
   */
  Tensor: 4,
  /**
   * EXPERIMENTAL: Metadata for n-dimensional sparse arrays, aka "sparse
   * tensors". Arrow implementations in general are not required to implement
   * this type.
   *
   * Not currently supported by Flechette.
   */
  SparseTensor: 5
});

/**
 * Field data type ids.
 * Only non-negative values ever occur in IPC flatbuffer binary data.
 */
const Type = /** @type {const} */ ({
  /**
   * Dictionary types compress data by using a set of integer indices to
   * lookup potentially repeated vales in a separate dictionary of values.
   *
   * This type entry is provided for API convenience, it does not occur
   * in actual Arrow IPC binary data.
   */
  Dictionary: -1,
  /** No data type. Included for flatbuffer compatibility. */
  NONE: 0,
  /** Null values only. */
  Null: 1,
  /** Integers, either signed or unsigned, with 8, 16, 32, or 64 bit widths. */
  Int: 2,
  /** Floating point numbers with 16, 32, or 64 bit precision. */
  Float: 3,
  /** Opaque binary data. */
  Binary: 4,
  /** Unicode with UTF-8 encoding. */
  Utf8: 5,
  /** Booleans represented as 8 bit bytes. */
  Bool: 6,
  /**
   * Exact decimal value represented as an integer value in two's complement.
   * Currently only 128-bit (16-byte) and 256-bit (32-byte) integers are used.
   * The representation uses the endianness indicated in the schema.
   */
  Decimal: 7,
  /**
   * Date is either a 32-bit or 64-bit signed integer type representing an
   * elapsed time since UNIX epoch (1970-01-01), stored in either of two units:
   * - Milliseconds (64 bits) indicating UNIX time elapsed since the epoch (no
   * leap seconds), where the values are evenly divisible by 86400000
   * - Days (32 bits) since the UNIX epoch
   */
  Date: 8,
  /**
   * Time is either a 32-bit or 64-bit signed integer type representing an
   * elapsed time since midnight, stored in either of four units: seconds,
   * milliseconds, microseconds or nanoseconds.
   *
   * The integer `bitWidth` depends on the `unit` and must be one of the following:
   * - SECOND and MILLISECOND: 32 bits
   * - MICROSECOND and NANOSECOND: 64 bits
   *
   * The allowed values are between 0 (inclusive) and 86400 (=24*60*60) seconds
   * (exclusive), adjusted for the time unit (for example, up to 86400000
   * exclusive for the MILLISECOND unit).
   * This definition doesn't allow for leap seconds. Time values from
   * measurements with leap seconds will need to be corrected when ingesting
   * into Arrow (for example by replacing the value 86400 with 86399).
   */
  Time: 9,
  /**
   * Timestamp is a 64-bit signed integer representing an elapsed time since a
   * fixed epoch, stored in either of four units: seconds, milliseconds,
   * microseconds or nanoseconds, and is optionally annotated with a timezone.
   *
   * Timestamp values do not include any leap seconds (in other words, all
   * days are considered 86400 seconds long).
   *
   * The timezone is an optional string for the name of a timezone, one of:
   *
   *  - As used in the Olson timezone database (the "tz database" or
   *    "tzdata"), such as "America/New_York".
   *  - An absolute timezone offset of the form "+XX:XX" or "-XX:XX",
   *    such as "+07:30".
   *
   * Whether a timezone string is present indicates different semantics about
   * the data.
   */
  Timestamp: 10,
  /**
   * A "calendar" interval which models types that don't necessarily
   * have a precise duration without the context of a base timestamp (e.g.
   * days can differ in length during day light savings time transitions).
   * All integers in the units below are stored in the endianness indicated
   * by the schema.
   *
   *  - YEAR_MONTH - Indicates the number of elapsed whole months, stored as
   *    4-byte signed integers.
   *  - DAY_TIME - Indicates the number of elapsed days and milliseconds (no
   *    leap seconds), stored as 2 contiguous 32-bit signed integers (8-bytes
   *    in total). Support of this IntervalUnit is not required for full arrow
   *    compatibility.
   *  - MONTH_DAY_NANO - A triple of the number of elapsed months, days, and
   *    nanoseconds. The values are stored contiguously in 16-byte blocks.
   *    Months and days are encoded as 32-bit signed integers and nanoseconds
   *    is encoded as a 64-bit signed integer. Nanoseconds does not allow for
   *    leap seconds. Each field is independent (e.g. there is no constraint
   *    that nanoseconds have the same sign as days or that the quantity of
   *    nanoseconds represents less than a day's worth of time).
   */
  Interval: 11,
  /**
   * List (vector) data supporting variably-sized lists.
   * A list has a single child data type for list entries.
   */
  List: 12,
  /**
   * A struct consisting of multiple named child data types.
   */
  Struct: 13,
  /**
   * A union is a complex type with parallel child data types. By default ids
   * in the type vector refer to the offsets in the children. Optionally
   * typeIds provides an indirection between the child offset and the type id.
   * For each child `typeIds[offset]` is the id used in the type vector.
   */
  Union: 14,
  /**
   * Binary data where each entry has the same fixed size.
   */
  FixedSizeBinary: 15,
  /**
   * List (vector) data where every list has the same fixed size.
   * A list has a single child data type for list entries.
   */
  FixedSizeList: 16,
  /**
   * A Map is a logical nested type that is represented as
   * List<entries: Struct<key: K, value: V>>
   *
   * In this layout, the keys and values are each respectively contiguous. We do
   * not constrain the key and value types, so the application is responsible
   * for ensuring that the keys are hashable and unique. Whether the keys are sorted
   * may be set in the metadata for this field.
   *
   * In a field with Map type, the field has a child Struct field, which then
   * has two children: key type and the second the value type. The names of the
   * child fields may be respectively "entries", "key", and "value", but this is
   * not enforced.
   *
   * Map
   * ```text
   *   - child[0] entries: Struct
   *   - child[0] key: K
   *   - child[1] value: V
   *  ```
   * Neither the "entries" field nor the "key" field may be nullable.
   *
   * The metadata is structured so that Arrow systems without special handling
   * for Map can make Map an alias for List. The "layout" attribute for the Map
   * field must have the same contents as a List.
   */
  Map: 17,
  /**
   * An absolute length of time unrelated to any calendar artifacts. For the
   * purposes of Arrow implementations, adding this value to a Timestamp
   * ("t1") naively (i.e. simply summing the two numbers) is acceptable even
   * though in some cases the resulting Timestamp (t2) would not account for
   * leap-seconds during the elapsed time between "t1" and "t2". Similarly,
   * representing the difference between two Unix timestamp is acceptable, but
   * would yield a value that is possibly a few seconds off from the true
   * elapsed time.
   *
   * The resolution defaults to millisecond, but can be any of the other
   * supported TimeUnit values as with Timestamp and Time types. This type is
   * always represented as an 8-byte integer.
   */
  Duration: 18,
  /**
   * Same as Binary, but with 64-bit offsets, allowing representation of
   * extremely large data values.
   */
  LargeBinary: 19,
  /**
   * Same as Utf8, but with 64-bit offsets, allowing representation of
   * extremely large data values.
   */
  LargeUtf8: 20,
  /**
   * Same as List, but with 64-bit offsets, allowing representation of
   * extremely large data values.
   */
  LargeList: 21,
  /**
   * Contains two child arrays, run_ends and values. The run_ends child array
   * must be a 16/32/64-bit integer array which encodes the indices at which
   * the run with the value in each corresponding index in the values child
   * array ends. Like list/struct types, the value array can be of any type.
   */
  RunEndEncoded: 22,
  /**
   * Logically the same as Binary, but the internal representation uses a view
   * struct that contains the string length and either the string's entire data
   * inline (for small strings) or an inlined prefix, an index of another buffer,
   * and an offset pointing to a slice in that buffer (for non-small strings).
   *
   * Since it uses a variable number of data buffers, each Field with this type
   * must have a corresponding entry in `variadicBufferCounts`.
   */
  BinaryView: 23,
  /**
   * Logically the same as Utf8, but the internal representation uses a view
   * struct that contains the string length and either the string's entire data
   * inline (for small strings) or an inlined prefix, an index of another buffer,
   * and an offset pointing to a slice in that buffer (for non-small strings).
   *
   * Since it uses a variable number of data buffers, each Field with this type
   * must have a corresponding entry in `variadicBufferCounts`.
   */
  Utf8View: 24,
  /**
   * Represents the same logical types that List can, but contains offsets and
   * sizes allowing for writes in any order and sharing of child values among
   * list values.
   */
  ListView: 25,
  /**
   * Same as ListView, but with 64-bit offsets and sizes, allowing to represent
   * extremely large data values.
   */
  LargeListView: 26
});

/**
 * Floating point number precision.
 */
const Precision = /** @type {const} */ ({
  /** 16-bit floating point number. */
  HALF: 0,
  /** 32-bit floating point number. */
  SINGLE: 1,
  /** 64-bit floating point number. */
  DOUBLE: 2
});

/**
 * Date units.
 */
const DateUnit = /** @type {const} */ ({
  /* Days (as 32 bit int) since the UNIX epoch. */
  DAY: 0,
  /**
   * Milliseconds (as 64 bit int) indicating UNIX time elapsed since the epoch
   * (no leap seconds), with values evenly divisible by 86400000.
   */
  MILLISECOND: 1
});

/**
 * Time units.
 */
const TimeUnit = /** @type {const} */ ({
  /** Seconds. */
  SECOND: 0,
  /** Milliseconds. */
  MILLISECOND: 1,
  /** Microseconds. */
  MICROSECOND: 2,
  /** Nanoseconds. */
  NANOSECOND: 3
});

/**
 * Date/time interval units.
 */
const IntervalUnit = /** @type {const} */ ({
  /**
   * Indicates the number of elapsed whole months, stored as 4-byte signed
   * integers.
   */
  YEAR_MONTH: 0,
  /**
   * Indicates the number of elapsed days and milliseconds (no leap seconds),
   * stored as 2 contiguous 32-bit signed integers (8-bytes in total). Support
   * of this IntervalUnit is not required for full arrow compatibility.
   */
  DAY_TIME: 1,
  /**
   * A triple of the number of elapsed months, days, and nanoseconds.
   * The values are stored contiguously in 16-byte blocks. Months and days are
   * encoded as 32-bit signed integers and nanoseconds is encoded as a 64-bit
   * signed integer. Nanoseconds does not allow for leap seconds. Each field is
   * independent (e.g. there is no constraint that nanoseconds have the same
   * sign as days or that the quantity of nanoseconds represents less than a
   * day's worth of time).
   */
  MONTH_DAY_NANO: 2
});

/**
 * Union type modes.
 */
const UnionMode = /** @type {const} */ ({
  /** Sparse union layout with full arrays for each sub-type. */
  Sparse: 0,
  /** Dense union layout with offsets into value arrays. */
  Dense: 1
});

const uint8Array = Uint8Array;
const uint16Array = Uint16Array;
const uint32Array = Uint32Array;
const uint64Array = BigUint64Array;
const int8Array = Int8Array;
const int16Array = Int16Array;
const int32Array = Int32Array;
const int64Array = BigInt64Array;
const float32Array = Float32Array;
const float64Array = Float64Array;

/**
 * Return the appropriate typed array constructor for the given
 * integer type metadata.
 * @param {number} bitWidth The integer size in bits.
 * @param {boolean} signed Flag indicating if the integer is signed.
 * @returns {import('../types.js').IntArrayConstructor}
 */
function intArrayType(bitWidth, signed) {
  const i = Math.log2(bitWidth) - 3;
  return (
    signed
      ? [int8Array, int16Array, int32Array, int64Array]
      : [uint8Array, uint16Array, uint32Array, uint64Array]
  )[i];
}

/** Shared prototype for typed arrays. */
const TypedArray = Object.getPrototypeOf(Int8Array);

/**
 * Check if a value is a typed array.
 * @param {*} value The value to check.
 * @returns {value is import('../types.js').TypedArray}
 *  True if value is a typed array, false otherwise.
 */
function isTypedArray(value) {
  return value instanceof TypedArray;
}

/**
 * Check if a value is either a standard array or typed array.
 * @param {*} value The value to check.
 * @returns {value is (Array | import('../types.js').TypedArray)}
 *  True if value is an array, false otherwise.
 */
function isArray(value) {
  return Array.isArray(value) || isTypedArray(value);
}

/**
 * Check if a value is an array type (constructor) for 64-bit integers,
 * one of BigInt64Array or BigUint64Array.
 * @param {*} value The value to check.
 * @returns {value is import('../types.js').Int64ArrayConstructor}
 *  True if value is a 64-bit array type, false otherwise.
 */
function isInt64ArrayType(value) {
  return value === int64Array || value === uint64Array;
}

/**
 * Determine the correct index into an offset array for a given
 * full column row index. Assumes offset indices can be manipulated
 * as 32-bit signed integers.
 * @param {import('../types.js').IntegerArray} offsets The offsets array.
 * @param {number} index The full column row index.
 */
function bisect(offsets, index) {
  let a = 0;
  let b = offsets.length;
  if (b <= 2147483648) { // 2 ** 31
    // fast version, use unsigned bit shift
    // array length fits within 32-bit signed integer
    do {
      const mid = (a + b) >>> 1;
      if (offsets[mid] <= index) a = mid + 1;
      else b = mid;
    } while (a < b);
  } else {
    // slow version, use division and truncate
    // array length exceeds 32-bit signed integer
    do {
      const mid = Math.trunc((a + b) / 2);
      if (offsets[mid] <= index) a = mid + 1;
      else b = mid;
    } while (a < b);
  }
  return a;
}

/**
 * Compute a 64-bit aligned buffer size.
 * @param {number} length The starting size.
 * @param {number} bpe Bytes per element.
 * @returns {number} The aligned size.
 */
function align64(length, bpe = 1) {
  return (((length * bpe) + 7) & ~7) / bpe;
}

/**
 * Return a 64-bit aligned version of the array.
 * @template {import('../types.js').TypedArray} T
 * @param {T} array The array.
 * @param {number} length The current array length.
 * @returns {T} The aligned array.
 */
function align(array, length = array.length) {
  const alignedLength = align64(length, array.BYTES_PER_ELEMENT);
  return array.length > alignedLength ? /** @type {T} */ (array.subarray(0, alignedLength))
    : array.length < alignedLength ? resize(array, alignedLength)
    : array;
}

/**
 * Resize a typed array to exactly the specified length.
 * @template {import('../types.js').TypedArray} T
 * @param {T} array The array.
 * @param {number} newLength The new length.
 * @param {number} [offset] The offset at which to copy the old array.
 * @returns {T} The resized array.
 */
function resize(array, newLength, offset = 0) {
  // @ts-ignore
  const newArray = new array.constructor(newLength);
  newArray.set(array, offset);
  return newArray;
}

/**
 * Grow a typed array to accommdate a minimum index. The array size is
 * doubled until it exceeds the minimum index.
 * @template {import('../types.js').TypedArray} T
 * @param {T} array The array.
 * @param {number} index The minimum index.
 * @param {boolean} [shift] Flag to shift copied bytes to back of array.
 * @returns {T} The resized array.
 */
function grow(array, index, shift) {
  while (array.length <= index) {
    array = resize(array, array.length << 1, shift ? array.length : 0);
  }
  return array;
}

/**
 * Check if a value is a Date instance
 * @param {*} value The value to check.
 * @returns {value is Date} True if value is a Date, false otherwise.
 */
function isDate(value) {
  return value instanceof Date;
}

/**
 * Check if a value is iterable.
 * @param {*} value The value to check.
 * @returns {value is Iterable} True if value is iterable, false otherwise.
 */
function isIterable(value) {
  return typeof value[Symbol.iterator] === 'function';
}

/**
 * Return the input value if it passes a test.
 * Otherwise throw an error using the given message generator.
 * @template T
 * @param {T} value he value to check.
 * @param {(value: T) => boolean} test The test function.
 * @param {(value: *) => string} message Message generator.
 * @returns {T} The input value.
 * @throws if the value does not pass the test
 */
function check(value, test, message) {
  if (test(value)) return value;
  throw new Error(message(value));
}

/**
 * Return the input value if it exists in the provided set.
 * Otherwise throw an error using the given message generator.
 * @template T
 * @param {T} value The value to check.
 * @param {T[] | Record<string,T>} set The set of valid values.
 * @param {(value: *) => string} [message] Message generator.
 * @returns {T} The input value.
 * @throws if the value is not included in the set
 */
function checkOneOf(value, set, message) {
  set = Array.isArray(set) ? set : Object.values(set);
  return check(
    value,
    (value) => set.includes(value),
    message ?? (() => `${value} must be one of ${set}`)
  );
}

/**
 * Return the first object key that pairs with the given value.
 * @param {Record<string,any>} object The object to search.
 * @param {any} value The value to lookup.
 * @returns {string} The first matching key, or '<Unknown>' if not found.
 */
function keyFor(object, value) {
  for (const [key, val] of Object.entries(object)) {
    if (val === value) return key;
  }
  return '<Unknown>';
}

/**
 * @typedef {import('./types.js').Field | import('./types.js').DataType} FieldInput
 */

const invalidDataType = (typeId) =>
  `Unsupported data type: "${keyFor(Type, typeId)}" (id ${typeId})`;

/**
 * Return a new field instance for use in a schema or type definition. A field
 * represents a field name, data type, and additional metadata. Fields are used
 * to represent child types within nested types like List, Struct, and Union.
 * @param {string} name The field name.
 * @param {import('./types.js').DataType} type The field data type.
 * @param {boolean} [nullable=true] Flag indicating if the field is nullable
 *  (default `true`).
 * @param {Map<string,string>|null} [metadata=null] Custom field metadata
 *  annotations (default `null`).
 * @returns {import('./types.js').Field} The field instance.
 */
const field = (name, type, nullable = true, metadata = null) => ({
  name,
  type,
  nullable,
  metadata
});

/**
 * Checks if a value is a field instance.
 * @param {any} value
 * @returns {value is import('./types.js').Field}
 */
function isField(value) {
  return Object.hasOwn(value, 'name') && isDataType(value.type)
}

/**
 * Checks if a value is a data type instance.
 * @param {any} value
 * @returns {value is import('./types.js').DataType}
 */
function isDataType(value) {
  return typeof value?.typeId === 'number';
}

/**
 * Return a field instance from a field or data type input.
 * @param {FieldInput} value
 *  The value to map to a field.
 * @param {string} [defaultName] The default field name.
 * @param {boolean} [defaultNullable=true] The default nullable value.
 * @returns {import('./types.js').Field} The field instance.
 */
function asField(value, defaultName = '', defaultNullable = true) {
  return isField(value)
    ? value
    : field(
        defaultName,
        check(value, isDataType, () => `Data type expected.`),
        defaultNullable
      );
}

/////

/**
 * Return a basic type with only a type id.
 * @template {typeof Type[keyof typeof Type]} T
 * @param {T} typeId The type id.
 */
const basicType = (typeId) => ({ typeId });

/**
 * Return a Dictionary data type instance.  A dictionary type consists of a
 * dictionary of values (which may be of any type) and corresponding integer
 * indices that reference those values. If values are repeated, a dictionary
 * encoding can provide substantial space savings. In the IPC format,
 * dictionary indices reside alongside other columns in a record batch, while
 * dictionary values are written to special dictionary batches, linked by a
 * unique dictionary *id*.
 * @param {import('./types.js').DataType} type The data type of dictionary
 *  values.
 * @param {import('./types.js').IntType} [indexType] The data type of
 *  dictionary indices. Must be an integer type (default `int32`).
 * @param {boolean} [ordered=false] Indicates if dictionary values are
 *  ordered (default `false`).
 * @param {number} [id=-1] The dictionary id. The default value (-1) indicates
 *  the dictionary applies to a single column only. Provide an explicit id in
 *  order to reuse a dictionary across columns when building, in which case
 *  different dictionaries *must* have different unique ids. All dictionary
 *  ids are later resolved (possibly to new values) upon IPC encoding.
 * @returns {import('./types.js').DictionaryType}
 */
const dictionary = (type, indexType, ordered = false, id = -1) => ({
  typeId: Type.Dictionary,
  id,
  dictionary: type,
  indices: indexType || int32(),
  ordered
});

/**
 * Return a Null data type instance. Null data requires no storage and all
 * extracted values are `null`.
 * @returns {import('./types.js').NullType} The null data type.
 */
const nullType = () => basicType(Type.Null);

/**
 * Return an Int data type instance.
 * @param {import('./types.js').IntBitWidth} [bitWidth=32] The integer bit width.
 *  One of `8`, `16`, `32` (default), or `64`.
 * @param {boolean} [signed=true] Flag for signed or unsigned integers
 *  (default `true`).
 * @returns {import('./types.js').IntType} The integer data type.
 */
const int = (bitWidth = 32, signed = true) => ({
  typeId: Type.Int,
  bitWidth: checkOneOf(bitWidth, [8, 16, 32, 64]),
  signed,
  values: intArrayType(bitWidth, signed)
});
/**
 * Return an Int data type instance for 8 bit signed integers.
 * @returns {import('./types.js').IntType} The integer data type.
 */
const int8 = () => int(8);
/**
 * Return an Int data type instance for 16 bit signed integers.
 * @returns {import('./types.js').IntType} The integer data type.
 */
const int16 = () => int(16);
/**
 * Return an Int data type instance for 32 bit signed integers.
 * @returns {import('./types.js').IntType} The integer data type.
 */
const int32 = () => int(32);
/**
 * Return an Int data type instance for 64 bit signed integers.
 * @returns {import('./types.js').IntType} The integer data type.
 */
const int64 = () => int(64);
/**
 * Return an Int data type instance for 8 bit unsigned integers.
 * @returns {import('./types.js').IntType} The integer data type.
 */
const uint8 = () => int(8, false);
/**
 * Return an Int data type instance for 16 bit unsigned integers.
 * @returns {import('./types.js').IntType} The integer data type.
 */
const uint16 = () => int(16, false);
/**
 * Return an Int data type instance for 32 bit unsigned integers.
 * @returns {import('./types.js').IntType} The integer data type.
 */
const uint32 = () => int(32, false);
/**
 * Return an Int data type instance for 64 bit unsigned integers.
 * @returns {import('./types.js').IntType} The integer data type.
 */
const uint64 = () => int(64, false);

/**
 * Return a Float data type instance for floating point numbers.
 * @param {import('./types.js').Precision_} [precision=2] The floating point
 *  precision. One of `Precision.HALF` (16-bit), `Precision.SINGLE` (32-bit)
 *  or `Precision.DOUBLE` (64-bit, default).
 * @returns {import('./types.js').FloatType} The floating point data type.
 */
const float = (precision = 2) => ({
  typeId: Type.Float,
  precision: checkOneOf(precision, Precision),
  values: [uint16Array, float32Array, float64Array][precision]
});
/**
 * Return a Float data type instance for half-precision (16 bit) numbers.
 * @returns {import('./types.js').FloatType} The floating point data type.
 */
const float16 = () => float(Precision.HALF);
/**
 * Return a Float data type instance for single-precision (32 bit) numbers.
 * @returns {import('./types.js').FloatType} The floating point data type.
 */
const float32 = () => float(Precision.SINGLE);
/**
 * Return a Float data type instance for double-precision (64 bit) numbers.
 * @returns {import('./types.js').FloatType} The floating point data type.
 */
const float64 = () => float(Precision.DOUBLE);

/**
 * Return a Binary data type instance for variably-sized opaque binary data
 * with 32-bit offsets.
 * @returns {import('./types.js').BinaryType} The binary data type.
 */
const binary = () => ({
  typeId: Type.Binary,
  offsets: int32Array
});

/**
 * Return a Utf8 data type instance for Unicode string data.
 * [UTF-8](https://en.wikipedia.org/wiki/UTF-8) code points are stored as
 * binary data.
 * @returns {import('./types.js').Utf8Type} The utf8 data type.
 */
const utf8 = () => ({
  typeId: Type.Utf8,
  offsets: int32Array
});

/**
 * Return a Bool data type instance. Bool values are stored compactly in
 * bitmaps with eight values per byte.
 * @returns {import('./types.js').BoolType} The bool data type.
 */
const bool = () => basicType(Type.Bool);

/**
 * Return a Decimal data type instance. Decimal values are represented as 128
 * or 256 bit integers in two's complement. Decimals are fixed point numbers
 * with a set *precision* (total number of decimal digits) and *scale*
 * (number of fractional digits). For example, the number `35.42` can be
 * represented as `3542` with *precision* ≥ 4 and *scale* = 2.
 * @param {number} precision The decimal precision: the total number of
 *  decimal digits that can be represented.
 * @param {number} scale The number of fractional digits, beyond the
 *  decimal point.
 * @param {128 | 256} [bitWidth] The decimal bit width.
 *  One of 128 (default) or 256.
 * @returns {import('./types.js').DecimalType} The decimal data type.
 */
const decimal = (precision, scale, bitWidth = 128) => ({
  typeId: Type.Decimal,
  precision,
  scale,
  bitWidth: checkOneOf(bitWidth, [128, 256]),
  values: uint64Array
});

/**
 * Return a Date data type instance. Date values are 32-bit or 64-bit signed
 * integers representing elapsed time since the UNIX epoch (Jan 1, 1970 UTC),
 * either in units of days (32 bits) or milliseconds (64 bits, with values
 * evenly divisible by 86400000).
 * @param {import('./types.js').DateUnit_} unit The date unit.
 *  One of `DateUnit.DAY` or `DateUnit.MILLISECOND`.
 * @returns {import('./types.js').DateType} The date data type.
 */
const date = (unit) => ({
  typeId: Type.Date,
  unit: checkOneOf(unit, DateUnit),
  values: unit === DateUnit.DAY ? int32Array : int64Array
});
/**
 * Return a Date data type instance with units of days.
 * @returns {import('./types.js').DateType} The date data type.
 */
const dateDay = () => date(DateUnit.DAY);
/**
 * Return a Date data type instance with units of milliseconds.
 * @returns {import('./types.js').DateType} The date data type.
 */
const dateMillisecond = () => date(DateUnit.MILLISECOND);

/**
 * Return a Time data type instance, stored in one of four *unit*s: seconds,
 * milliseconds, microseconds or nanoseconds. The integer *bitWidth* depends
 * on the *unit* and must be 32 bits for seconds and milliseconds or 64 bits
 * for microseconds and nanoseconds. The allowed values are between 0
 * (inclusive) and 86400 (=24*60*60) seconds (exclusive), adjusted for the
 * time unit (for example, up to 86400000 exclusive for the
 * `DateUnit.MILLISECOND` unit.
 *
 * This definition doesn't allow for leap seconds. Time values from
 * measurements with leap seconds will need to be corrected when ingesting
 * into Arrow (for example by replacing the value 86400 with 86399).
 * @param {import('./types.js').TimeUnit_} unit The time unit.
 *  One of `TimeUnit.SECOND`, `TimeUnit.MILLISECOND` (default),
 *  `TimeUnit.MICROSECOND`, or `TimeUnit.NANOSECOND`.
 * @param {32 | 64} bitWidth The time bit width. One of `32` (for seconds
 *  and milliseconds) or `64` (for microseconds and nanoseconds).
 * @returns {import('./types.js').TimeType} The time data type.
 */
const time = (unit = TimeUnit.MILLISECOND, bitWidth = 32) => ({
  typeId: Type.Time,
  unit: checkOneOf(unit, TimeUnit),
  bitWidth: checkOneOf(bitWidth, [32, 64]),
  values: bitWidth === 32 ? int32Array : int64Array
});
/**
 * Return a Time data type instance, represented as seconds.
 * @returns {import('./types.js').TimeType} The time data type.
 */
const timeSecond = () => time(TimeUnit.SECOND, 32);
/**
 * Return a Time data type instance, represented as milliseconds.
 * @returns {import('./types.js').TimeType} The time data type.
 */
const timeMillisecond = () => time(TimeUnit.MILLISECOND, 32);
/**
 * Return a Time data type instance, represented as microseconds.
 * @returns {import('./types.js').TimeType} The time data type.
 */
const timeMicrosecond = () => time(TimeUnit.MICROSECOND, 64);
/**
 * Return a Time data type instance, represented as nanoseconds.
 * @returns {import('./types.js').TimeType} The time data type.
 */
const timeNanosecond = () => time(TimeUnit.NANOSECOND, 64);

/**
 * Return a Timestamp data type instance. Timestamp values are 64-bit signed
 * integers representing an elapsed time since a fixed epoch, stored in either
 * of four units: seconds, milliseconds, microseconds or nanoseconds, and are
 * optionally annotated with a timezone. Timestamp values do not include any
 * leap seconds (in other words, all days are considered 86400 seconds long).
 * @param {import('./types.js').TimeUnit_} [unit] The time unit.
 *  One of `TimeUnit.SECOND`, `TimeUnit.MILLISECOND` (default),
 *  `TimeUnit.MICROSECOND`, or `TimeUnit.NANOSECOND`.
 * @param {string|null} [timezone=null] An optional string for the name of a
 *  timezone. If provided, the value should either be a string as used in the
 *  Olson timezone database (the "tz database" or "tzdata"), such as
 *  "America/New_York", or an absolute timezone offset of the form "+XX:XX" or
 *  "-XX:XX", such as "+07:30".Whether a timezone string is present indicates
 *  different semantics about the data.
 * @returns {import('./types.js').TimestampType} The time data type.
 */
const timestamp = (unit = TimeUnit.MILLISECOND, timezone = null) => ({
  typeId: Type.Timestamp,
  unit: checkOneOf(unit, TimeUnit),
  timezone,
  values: int64Array
});

/**
 * Return an Interval type instance. Values represent calendar intervals stored
 * as integers for each date part. The supported *unit*s are year/moth,
 * day/time, and month/day/nanosecond intervals.
 *
 * `IntervalUnit.YEAR_MONTH` indicates the number of elapsed whole months,
 * stored as 32-bit signed integers.
 *
 * `IntervalUnit.DAY_TIME` indicates the number of elapsed days and
 * milliseconds (no leap seconds), stored as 2 contiguous 32-bit signed
 * integers (8-bytes in total).
 *
 * `IntervalUnit.MONTH_DAY_NANO` is a triple of the number of elapsed months,
 * days, and nanoseconds. The values are stored contiguously in 16-byte blocks.
 * Months and days are encoded as 32-bit signed integers and nanoseconds is
 * encoded as a 64-bit signed integer. Nanoseconds does not allow for leap
 * seconds. Each field is independent (e.g. there is no constraint that
 * nanoseconds have the same sign as days or that the quantity of nanoseconds
 * represents less than a day's worth of time).
 * @param {import('./types.js').IntervalUnit_} unit  The interval unit.
 *  One of `IntervalUnit.YEAR_MONTH`, `IntervalUnit.DAY_TIME`, or
 *  `IntervalUnit.MONTH_DAY_NANO` (default).
 * @returns {import('./types.js').IntervalType} The interval data type.
 */
const interval = (unit = IntervalUnit.MONTH_DAY_NANO) => ({
  typeId: Type.Interval,
  unit: checkOneOf(unit, IntervalUnit),
  values: unit === IntervalUnit.MONTH_DAY_NANO ? undefined : int32Array
});

/**
 * Return a List data type instance, representing variably-sized lists
 * (arrays) with 32-bit offsets. A list has a single child data type for
 * list entries. Lists are represented using integer offsets that indicate
 * list extents within a single child array containing all list values.
 * @param {FieldInput} child The child (list item) field or data type.
 * @returns {import('./types.js').ListType} The list data type.
 */
const list = (child) => ({
  typeId: Type.List,
  children: [ asField(child) ],
  offsets: int32Array
});

/**
 * Return a Struct data type instance. A struct consists of multiple named
 * child data types. Struct values are stored as parallel child batches, one
 * per child type, and extracted to standard JavaScript objects.
 * @param {import('./types.js').Field[] | Record<string, import('./types.js').DataType>} children
 *  An array of property fields, or an object mapping property names to data
 *  types. If an object, the instantiated fields are assumed to be nullable
 *  and have no metadata.
 * @returns {import('./types.js').StructType} The struct data type.
 */
const struct = (children) => ({
  typeId: Type.Struct,
  children: Array.isArray(children) && isField(children[0])
    ? /** @type {import('./types.js').Field[]} */ (children)
    : Object.entries(children).map(([name, type]) => field(name, type))
});

/**
 * Return a Union type instance. A union is a complex type with parallel
 * *children* data types. Union values are stored in either a sparse
 * (`UnionMode.Sparse`) or dense (`UnionMode.Dense`) layout *mode*. In a
 * sparse layout, child types are stored in parallel arrays with the same
 * lengths, resulting in many unused, empty values. In a dense layout, child
 * types have variable lengths and an offsets array is used to index the
 * appropriate value.
 *
 * By default, ids in the type vector refer to the index in the children
 * array. Optionally, *typeIds* provide an indirection between the child
 * index and the type id. For each child, `typeIds[index]` is the id used
 * in the type vector. The *typeIdForValue* argument provides a lookup
 * function for mapping input data to the proper child type id, and is
 * required if using builder methods.
 * @param {import('./types.js').UnionMode_} mode The union mode.
 *  One of `UnionMode.Sparse` or `UnionMode.Dense`.
 * @param {FieldInput[]} children The children fields or data types.
 *  Types are mapped to nullable fields with no metadata.
 * @param {number[]} [typeIds]  Children type ids, in the same order as the
 *  children types. Type ids provide a level of indirection over children
 *  types. If not provided, the children indices are used as the type ids.
 * @param {(value: any, index: number) => number} [typeIdForValue]
 *  A function that takes an arbitrary value and a row index and returns a
 *  correponding union type id. Required by builder methods.
 * @returns {import('./types.js').UnionType} The union data type.
 */
const union = (mode, children, typeIds, typeIdForValue) => {
  typeIds ??= children.map((v, i) => i);
  return {
    typeId: Type.Union,
    mode: checkOneOf(mode, UnionMode),
    typeIds,
    typeMap: typeIds.reduce((m, id, i) => ((m[id] = i), m), {}),
    children: children.map((v, i) => asField(v, `_${i}`)),
    typeIdForValue,
    offsets: int32Array,
  };
};

/**
 * Create a FixedSizeBinary data type instance for opaque binary data where
 * each entry has the same fixed size.
 * @param {number} stride The fixed size in bytes.
 * @returns {import('./types.js').FixedSizeBinaryType} The fixed size binary data type.
 */
const fixedSizeBinary = (stride) => ({
  typeId: Type.FixedSizeBinary,
  stride
});

/**
 * Return a FixedSizeList type instance for list (array) data where every list
 * has the same fixed size. A list has a single child data type for list
 * entries. Fixed size lists are represented as a single child array containing
 * all list values, indexed using the known stride.
 * @param {FieldInput} child The list item data type.
 * @param {number} stride The fixed list size.
 * @returns {import('./types.js').FixedSizeListType} The fixed size list data type.
 */
const fixedSizeList = (child, stride) => ({
  typeId: Type.FixedSizeList,
  stride,
  children: [ asField(child) ]
});

/**
 * Internal method to create a Map type instance.
 * @param {boolean} keysSorted Flag indicating if the map keys are sorted.
 * @param {import('./types.js').Field} child The child fields.
 * @returns {import('./types.js').MapType} The map data type.
 */
const mapType = (keysSorted, child) => ({
  typeId: Type.Map,
  keysSorted,
  children: [child],
  offsets: int32Array
});

/**
 * Return a Map data type instance representing collections of key-value pairs.
 * A Map is a logical nested type that is represented as a list of key-value
 * structs. The key and value types are not constrained, so the application is
 * responsible for ensuring that the keys are hashable and unique, and that
 * keys are properly sorted if *keysSorted* is `true`.
 * @param {FieldInput} keyField The map key field or data type.
 * @param {FieldInput} valueField The map value field or data type.
 * @param {boolean} [keysSorted=false] Flag indicating if the map keys are
 *  sorted (default `false`).
 * @returns {import('./types.js').MapType} The map data type.
 */
const map = (keyField, valueField, keysSorted = false) => mapType(
  keysSorted,
  field(
    'entries',
    struct([ asField(keyField, 'key', false), asField(valueField, 'value') ]),
    false
  )
);

/**
 * Return a Duration data type instance. Durations represent an absolute length
 * of time unrelated to any calendar artifacts. The resolution defaults to
 * millisecond, but can be any of the other `TimeUnit` values. This type is
 * always represented as a 64-bit integer.
 * @param {import('./types.js').TimeUnit_} unit
 * @returns {import('./types.js').DurationType} The duration data type.
 */
const duration = (unit = TimeUnit.MILLISECOND) => ({
  typeId: Type.Duration,
  unit: checkOneOf(unit, TimeUnit),
  values: int64Array
});

/**
 * Return a LargeBinary data type instance for variably-sized opaque binary
 * data with 64-bit offsets, allowing representation of extremely large data
 * values.
 * @returns {import('./types.js').LargeBinaryType} The large binary data type.
 */
const largeBinary = () => ({
  typeId: Type.LargeBinary,
  offsets: int64Array
});

/**
 * Return a LargeUtf8 data type instance for Unicode string data of variable
 * length with 64-bit offsets, allowing representation of extremely large data
 * values. [UTF-8](https://en.wikipedia.org/wiki/UTF-8) code points are stored
 * as binary data.
 * @returns {import('./types.js').LargeUtf8Type} The large utf8 data type.
 */
const largeUtf8 = () => ({
  typeId: Type.LargeUtf8,
  offsets: int64Array
});

/**
 * Return a LargeList data type instance, representing variably-sized lists
 * (arrays) with 64-bit offsets, allowing representation of extremely large
 * data values. A list has a single child data type for list entries. Lists
 * are represented using integer offsets that indicate list extents within a
 * single child array containing all list values.
 * @param {FieldInput} child The child (list item) field or data type.
 * @returns {import('./types.js').LargeListType} The large list data type.
 */
const largeList = (child) => ({
  typeId: Type.LargeList,
  children: [ asField(child) ],
  offsets: int64Array
});

/**
 * Return a RunEndEncoded data type instance, which compresses data by
 * representing consecutive repeated values as a run. This data type uses two
 * child arrays, `run_ends` and `values`. The `run_ends` child array must be
 * a 16, 32, or 64 bit integer array which encodes the indices at which the
 * run with the value in each corresponding index in the values child array
 * ends. Like list and struct types, the `values` array can be of any type.
 * @param {FieldInput} runsField The run-ends field or data type.
 * @param {FieldInput} valuesField The values field or data type.
 * @returns {import('./types.js').RunEndEncodedType} The large list data type.
 */
const runEndEncoded = (runsField, valuesField) => ({
  typeId: Type.RunEndEncoded,
  children: [
    check(
      asField(runsField, 'run_ends'),
      (field) => field.type.typeId === Type.Int,
      () => 'Run-ends must have an integer type.'
    ),
    asField(valuesField, 'values')
  ]
});

/**
 * Return a BinaryView data type instance. BinaryView data is logically the
 * same as the Binary type, but the internal representation uses a view struct
 * that contains the string length and either the string's entire data inline
 * (for small strings) or an inlined prefix, an index of another buffer, and an
 * offset pointing to a slice in that buffer (for non-small strings).
 *
 * Flechette can encode and decode BinaryView data; however, Flechette does
 * not currently support building BinaryView columns from JavaScript values.
 * @returns {import('./types.js').BinaryViewType} The binary view data type.
 */
const binaryView = () => /** @type{import('./types.js').BinaryViewType} */
  (basicType(Type.BinaryView));

/**
 * Return a Utf8View data type instance. Utf8View data is logically the same as
 * the Utf8 type, but the internal representation uses a view struct that
 * contains the string length and either the string's entire data inline (for
 * small strings) or an inlined prefix, an index of another buffer, and an
 * offset pointing to a slice in that buffer (for non-small strings).
 *
 * Flechette can encode and decode Utf8View data; however, Flechette does
 * not currently support building Utf8View columns from JavaScript values.
 * @returns {import('./types.js').Utf8ViewType} The utf8 view data type.
 */
const utf8View = () => /** @type{import('./types.js').Utf8ViewType} */
  (basicType(Type.Utf8View));

/**
 * Return a ListView data type instance, representing variably-sized lists
 * (arrays) with 32-bit offsets. ListView data represents the same logical
 * types that List can, but contains both offsets and sizes allowing for
 * writes in any order and sharing of child values among list values.
 *
 * Flechette can encode and decode ListView data; however, Flechette does not
 * currently support building ListView columns from JavaScript values.
 * @param {FieldInput} child The child (list item) field or data type.
 * @returns {import('./types.js').ListViewType} The list view data type.
 */
const listView = (child) => ({
  typeId: Type.ListView,
  children: [ asField(child, 'value') ],
  offsets: int32Array
});

/**
 * Return a LargeListView data type instance, representing variably-sized lists
 * (arrays) with 64-bit offsets, allowing representation of extremely large
 * data values. LargeListView data represents the same logical types that
 * LargeList can, but contains both offsets and sizes allowing for writes
 * in any order and sharing of child values among list values.
 *
 * Flechette can encode and decode LargeListView data; however, Flechette does
 * not currently support building LargeListView columns from JavaScript values.
 * @param {FieldInput} child The child (list item) field or data type.
 * @returns {import('./types.js').LargeListViewType} The large list view data type.
 */
const largeListView = (child) => ({
  typeId: Type.LargeListView,
  children: [ asField(child, 'value') ],
  offsets: int64Array
});

// typed arrays over a shared buffer to aid binary conversion
const f64 = new float64Array(2);
const buf = f64.buffer;
const i64 = new int64Array(buf);
const u32 = new uint32Array(buf);
const i32 = new int32Array(buf);
const u8 = new uint8Array(buf);

/**
 * Return a value unchanged.
 * @template T
 * @param {T} value The value.
 * @returns {T} The value.
 */
function identity(value) {
  return value;
}

/**
 * Return a value coerced to a BigInt.
 * @param {*} value The value.
 * @returns {bigint} The BigInt value.
 */
function toBigInt(value) {
  return BigInt(value);
}

/**
 * Return an offset conversion method for the given data type.
 * @param {{ offsets: import('../types.js').TypedArray}} type The array type.
 */
function toOffset(type) {
  return isInt64ArrayType(type) ? toBigInt : identity;
}

/**
 * Return the number of days from a millisecond timestamp.
 * @param {number} value The millisecond timestamp.
 * @returns {number} The number of days.
 */
function toDateDay(value) {
  return (value / 864e5) | 0;
}

/**
 * Return a timestamp conversion method for the given time unit.
 * @param {import('../types.js').TimeUnit_} unit The time unit.
 * @returns {(value: number) => bigint} The conversion method.
 */
function toTimestamp(unit) {
  return unit === TimeUnit.SECOND ? value => toBigInt(value / 1e3)
    : unit === TimeUnit.MILLISECOND ? toBigInt
    : unit === TimeUnit.MICROSECOND ? value => toBigInt(value * 1e3)
    : value => toBigInt(value * 1e6);
}

/**
 * Write month/day/nanosecond interval to a byte buffer.
 * @param {Array | Float64Array} interval The interval data.
 * @returns {Uint8Array} A byte buffer with the interval data.
 *  The returned buffer is reused across calls, and so should be
 *  copied to a target buffer immediately.
 */
function toMonthDayNanoBytes([m, d, n]) {
  i32[0] = m;
  i32[1] = d;
  i64[1] = toBigInt(n);
  return u8;
}

/**
 * Coerce a bigint value to a number. Throws an error if the bigint value
 * lies outside the range of what a number can precisely represent.
 * @param {bigint} value The value to check and possibly convert.
 * @returns {number} The converted number value.
 */
function toNumber(value) {
  if (value > Number.MAX_SAFE_INTEGER || value < Number.MIN_SAFE_INTEGER) {
    throw Error(`BigInt exceeds integer number representation: ${value}`);
  }
  return Number(value);
}

/**
 * Divide one BigInt value by another, and return the result as a number.
 * The division may involve unsafe integers and a loss of precision.
 * @param {bigint} num The numerator.
 * @param {bigint} div The divisor.
 * @returns {number} The result of the division as a floating point number.
 */
function divide(num, div) {
  return Number(num / div) + Number(num % div) / Number(div);
}

/**
 * Convert a floating point number or bigint to decimal bytes.
 * @param {number|bigint} value The number to encode. If a bigint, we assume
 *  it already represents the decimal in integer form with the correct scale.
 *  Otherwise, we assume a float that requires scaled integer conversion.
 * @param {BigUint64Array} buf The uint64 array to write to.
 * @param {number} offset The starting index offset into the array.
 * @param {number} stride The stride of an encoded decimal, in 64-bit steps.
 * @param {number} scale The scale mapping fractional digits to integers.
 */
function toDecimal(value, buf, offset, stride, scale) {
  const v = typeof value === 'bigint'
    ? value
    : toBigInt(Math.trunc(value * scale));
  // assignment into uint64array performs needed truncation for us
  buf[offset] = v;
  buf[offset + 1] = (v >> 64n);
  if (stride > 2) {
    buf[offset + 2] = (v >> 128n);
    buf[offset + 3] = (v >> 192n);
  }
}

// helper method to extract uint64 values from bigints
const asUint64 = v => BigInt.asUintN(64, v);

/**
 * Convert a 128-bit decimal value to a bigint.
 * @param {BigUint64Array} buf The uint64 array containing the decimal bytes.
 * @param {number} offset The starting index offset into the array.
 * @returns {bigint} The converted decimal as a bigint, such that all
 *  fractional digits are scaled up to integers (for example, 1.12 -> 112).
 */
function fromDecimal128(buf, offset) {
  const i = offset << 1;
  let x;
  if (BigInt.asIntN(64, buf[i + 1]) < 0) {
    x = asUint64(~buf[i]) | (asUint64(~buf[i + 1]) << 64n);
    x = -(x + 1n);
  } else {
    x = buf[i] | (buf[i + 1] << 64n);
  }
  return x;
}

/**
 * Convert a 256-bit decimal value to a bigint.
 * @param {BigUint64Array} buf The uint64 array containing the decimal bytes.
 * @param {number} offset The starting index offset into the array.
 * @returns {bigint} The converted decimal as a bigint, such that all
 *  fractional digits are scaled up to integers (for example, 1.12 -> 112).
 */
function fromDecimal256(buf, offset) {
  const i = offset << 2;
  let x;
  if (BigInt.asIntN(64, buf[i + 3]) < 0) {
    x = asUint64(~buf[i])
      | (asUint64(~buf[i + 1]) << 64n)
      | (asUint64(~buf[i + 2]) << 128n)
      | (asUint64(~buf[i + 3]) << 192n);
    x = -(x + 1n);
  } else {
    x = buf[i]
      | (buf[i + 1] << 64n)
      | (buf[i + 2] << 128n)
      | (buf[i + 3] << 192n);
  }
  return x;
}

/**
 * Convert a number to a 16-bit float as integer bytes..
 * Inspired by numpy's `npy_double_to_half`:
 * https://github.com/numpy/numpy/blob/5a5987291dc95376bb098be8d8e5391e89e77a2c/numpy/core/src/npymath/halffloat.c#L43
 * Adapted from https://github.com/apache/arrow/blob/main/js/src/util/math.ts
 * @param {number} value The 64-bit floating point number to convert.
 * @returns {number} The converted 16-bit integer.
 */
function toFloat16(value) {
  if (value !== value) return 0x7E00; // NaN
  f64[0] = value;

  // Magic numbers:
  // 0x80000000 = 10000000 00000000 00000000 00000000 -- masks the 32nd bit
  // 0x7ff00000 = 01111111 11110000 00000000 00000000 -- masks the 21st-31st bits
  // 0x000fffff = 00000000 00001111 11111111 11111111 -- masks the 1st-20th bit
  const sign = (u32[1] & 0x80000000) >> 16 & 0xFFFF;
  let expo = (u32[1] & 0x7FF00000), sigf = 0x0000;

  if (expo >= 0x40F00000) {
    //
    // If exponent overflowed, the float16 is either NaN or Infinity.
    // Rules to propagate the sign bit: mantissa > 0 ? NaN : +/-Infinity
    //
    // Magic numbers:
    // 0x40F00000 = 01000000 11110000 00000000 00000000 -- 6-bit exponent overflow
    // 0x7C000000 = 01111100 00000000 00000000 00000000 -- masks the 27th-31st bits
    //
    // returns:
    // qNaN, aka 32256 decimal, 0x7E00 hex, or 01111110 00000000 binary
    // sNaN, aka 32000 decimal, 0x7D00 hex, or 01111101 00000000 binary
    // +inf, aka 31744 decimal, 0x7C00 hex, or 01111100 00000000 binary
    // -inf, aka 64512 decimal, 0xFC00 hex, or 11111100 00000000 binary
    //
    // If mantissa is greater than 23 bits, set to +Infinity like numpy
    if (u32[0] > 0) {
      expo = 0x7C00;
    } else {
      expo = (expo & 0x7C000000) >> 16;
      sigf = (u32[1] & 0x000FFFFF) >> 10;
    }
  } else if (expo <= 0x3F000000) {
    //
    // If exponent underflowed, the float is either signed zero or subnormal.
    //
    // Magic numbers:
    // 0x3F000000 = 00111111 00000000 00000000 00000000 -- 6-bit exponent underflow
    //
    sigf = 0x100000 + (u32[1] & 0x000FFFFF);
    sigf = 0x100000 + (sigf << ((expo >> 20) - 998)) >> 21;
    expo = 0;
  } else {
    //
    // No overflow or underflow, rebase the exponent and round the mantissa
    // Magic numbers:
    // 0x200 = 00000010 00000000 -- masks off the 10th bit
    //
    // Ensure the first mantissa bit (the 10th one) is 1 and round
    expo = (expo - 0x3F000000) >> 10;
    sigf = ((u32[1] & 0x000FFFFF) + 0x200) >> 10;
  }
  return sign | expo | sigf & 0xFFFF;
}

const textDecoder = new TextDecoder('utf-8');
const textEncoder = new TextEncoder();

/**
 * Return a UTF-8 string decoded from a byte buffer.
 * @param {Uint8Array} buf The byte buffer.
 * @returns {string} The decoded string.
 */
function decodeUtf8(buf) {
  return textDecoder.decode(buf);
}

/**
 * Return a byte buffer encoded from a UTF-8 string.
 * @param {string } str The string to encode.
 * @returns {Uint8Array} The encoded byte buffer.
 */
function encodeUtf8(str) {
  return textEncoder.encode(str);
}

/**
 * Return a string-coercible key value that uniquely identifies a value.
 * @param {*} value The input value.
 * @returns {string} The key string.
 */
function keyString(value) {
  const val = typeof value !== 'object' || !value ? (value ?? null)
    : isDate(value) ? +value
    // @ts-ignore
    : isArray(value) ? `[${value.map(keyString)}]`
    : objectKey(value);
  return `${val}`;
}

function objectKey(value) {
  let s = '';
  let i = -1;
  for (const k in value) {
    if (++i > 0) s += ',';
    s += `"${k}":${keyString(value[k])}`;
  }
  return `{${s}}`;
}

/** The size in bytes of a 32-bit integer. */
const SIZEOF_INT = 4;

/** The size in bytes of a 16-bit integer. */
const SIZEOF_SHORT = 2;

/**
 * Return a boolean for a single bit in a bitmap.
 * @param {Uint8Array} bitmap The bitmap.
 * @param {number} index The bit index to read.
 * @returns {boolean} The boolean bitmap value.
 */
function decodeBit(bitmap, index) {
  return (bitmap[index >> 3] & 1 << (index % 8)) !== 0;
}

/**
 * Lookup helper for flatbuffer object (table) entries.
 * @param {Uint8Array} buf The byte buffer.
 * @param {number} index The base index of the object.
 */
function readObject(buf, index) {
  const pos = index + readInt32(buf, index);
  const vtable = pos - readInt32(buf, pos);
  const size = readInt16(buf, vtable);
  /**
   * Retrieve a value from a flatbuffer table layout.
   * @template T
   * @param {number} index The table entry index.
   * @param {(buf: Uint8Array, offset: number) => T} read Read function to invoke.
   * @param {T} [fallback=null] The default fallback value.
   * @returns {T}
   */
  return (index, read, fallback = null) => {
    if (index < size) {
      const off = readInt16(buf, vtable + index);
      if (off) return read(buf, pos + off);
    }
    return fallback;
  };
}

/**
 * Return a buffer offset value.
 * @param {Uint8Array} buf
 * @param {number} offset
 * @returns {number}
 */
function readOffset(buf, offset) {
  return offset;
}

/**
 * Return a boolean value.
 * @param {Uint8Array} buf
 * @param {number} offset
 * @returns {boolean}
 */
function readBoolean(buf, offset) {
  return !!readInt8(buf, offset);
}

/**
 * Return a signed 8-bit integer value.
 * @param {Uint8Array} buf
 * @param {number} offset
 * @returns {number}
 */
function readInt8(buf, offset) {
  return readUint8(buf, offset) << 24 >> 24;
}

/**
 * Return an unsigned 8-bit integer value.
 * @param {Uint8Array} buf
 * @param {number} offset
 * @returns {number}
 */
function readUint8(buf, offset) {
  return buf[offset];
}

/**
 * Return a signed 16-bit integer value.
 * @param {Uint8Array} buf
 * @param {number} offset
 * @returns {number}
 */
function readInt16(buf, offset) {
  return readUint16(buf, offset) << 16 >> 16;
}

/**
 * Return an unsigned 16-bit integer value.
 * @param {Uint8Array} buf
 * @param {number} offset
 * @returns {number}
 */
function readUint16(buf, offset) {
  return buf[offset] | buf[offset + 1] << 8;
}

/**
 * Return a signed 32-bit integer value.
 * @param {Uint8Array} buf
 * @param {number} offset
 * @returns {number}
 */
function readInt32(buf, offset) {
  return buf[offset]
    | buf[offset + 1] << 8
    | buf[offset + 2] << 16
    | buf[offset + 3] << 24;
}

/**
 * Return an unsigned 32-bit integer value.
 * @param {Uint8Array} buf
 * @param {number} offset
 * @returns {number}
 */
function readUint32(buf, offset) {
  return readInt32(buf, offset) >>> 0;
}

/**
 * Return a signed 64-bit integer value coerced to a JS number.
 * Throws an error if the value exceeds what a JS number can represent.
 * @param {Uint8Array} buf
 * @param {number} offset
 * @returns {number}
 */
function readInt64(buf, offset) {
  return toNumber(BigInt.asIntN(
    64,
    BigInt(readUint32(buf, offset)) +
      (BigInt(readUint32(buf, offset + SIZEOF_INT)) << 32n)
  ));
}

/**
 * Create a JavaScript string from UTF-8 data stored inside the FlatBuffer.
 * This allocates a new string and converts to wide chars upon each access.
 * @param {Uint8Array} buf The byte buffer.
 * @param {number} index The index of the string entry.
 * @returns {string} The decoded string.
 */
function readString(buf, index) {
  let offset = index + readInt32(buf, index); // get the string offset
  const length = readInt32(buf, offset);  // get the string length
  offset += SIZEOF_INT; // skip length value
  return decodeUtf8(buf.subarray(offset, offset + length));
}

/**
 * Extract a flatbuffer vector to an array.
 * @template T
 * @param {Uint8Array} buf The byte buffer.
 * @param {number} offset The offset location of the vector.
 * @param {number} stride The stride between vector entries.
 * @param {(buf: Uint8Array, pos: number) => T} extract Vector entry extraction function.
 * @returns {T[]} The extracted vector entries.
 */
function readVector(buf, offset, stride, extract) {
  if (!offset) return [];

  // get base position by adding offset delta
  const base = offset + readInt32(buf, offset);

  // read vector size, extract entries
  return Array.from(
    { length: readInt32(buf, base) },
    (_, i) => extract(buf, base + SIZEOF_INT + i * stride)
  );
}

const RowIndex = Symbol('rowIndex');

/**
 * Returns a row proxy object factory. The resulting method takes a
 * batch-level row index as input and returns an object that proxies
 * access to underlying batches.
 * @param {string[]} names The column (property) names
 * @param {import('../batch.js').Batch[]} batches The value batches.
 * @returns {(index: number) => Record<string, any>}
 */
function proxyFactory(names, batches) {
  class RowObject {
    /**
     * Create a new proxy row object representing a struct or table row.
     * @param {number} index The record batch row index.
     */
    constructor(index) {
      this[RowIndex] = index;
    }

    /**
     * Return a JSON-compatible object representation.
     */
    toJSON() {
      return structObject(names, batches, this[RowIndex]);
    }
  }
  // prototype for row proxy objects
  const proto = RowObject.prototype;

  for (let i = 0; i < names.length; ++i) {
    // skip duplicated column names
    if (Object.hasOwn(proto, names[i])) continue;

    // add a getter method for the current batch
    const batch = batches[i];
    Object.defineProperty(proto, names[i], {
      get() { return batch.at(this[RowIndex]); },
      enumerable: true
    });
  }

  return index => new RowObject(index);
}

/**
 * Returns a row object factory. The resulting method takes a
 * batch-level row index as input and returns an object whose property
 * values have been extracted from the batches.
 * @param {string[]} names The column (property) names
 * @param {import('../batch.js').Batch[]} batches The value batches.
 * @returns {(index: number) => Record<string, any>}
 */
function objectFactory(names, batches) {
  return index => structObject(names, batches, index);
}

/**
 * Return a vanilla object representing a struct (row object) type.
 * @param {string[]} names The column (property) names
 * @param {import('../batch.js').Batch[]} batches The value batches.
 * @param {number} index The record batch row index.
 * @returns {Record<string, any>}
 */
function structObject(names, batches, index) {
  const obj = {};
  for (let i = 0; i < names.length; ++i) {
    obj[names[i]] = batches[i].at(index);
  }
  return obj;
}

/**
 * Check if the input is a batch that supports direct access to
 * binary data in the form of typed arrays.
 * @param {Batch<any>?} batch The data batch to check.
 * @returns {boolean} True if a direct batch, false otherwise.
 */
function isDirectBatch(batch) {
  return batch instanceof DirectBatch;
}

/**
 * Column values from a single record batch.
 * A column may contain multiple batches.
 * @template T
 */
class Batch {
  /**
   * The array type to use when extracting data from the batch.
   * A null value indicates that the array type should match
   * the type of the batch's values array.
   * @type {ArrayConstructor | import('./types.js').TypedArrayConstructor | null}
   */
  static ArrayType = null;

  /**
   * Create a new column batch.
   * @param {object} options
   * @param {number} options.length The length of the batch
   * @param {number} options.nullCount The null value count
   * @param {import('./types.js').DataType} options.type The data type.
   * @param {Uint8Array} [options.validity] Validity bitmap buffer
   * @param {import('./types.js').TypedArray} [options.values] Values buffer
   * @param {import('./types.js').OffsetArray} [options.offsets] Offsets buffer
   * @param {import('./types.js').OffsetArray} [options.sizes] Sizes buffer
   * @param {Batch[]} [options.children] Children batches
   */
  constructor({
    length,
    nullCount,
    type,
    validity,
    values,
    offsets,
    sizes,
    children
  }) {
    this.length = length;
    this.nullCount = nullCount;
    this.type = type;
    this.validity = validity;
    this.values = values;
    this.offsets = offsets;
    this.sizes = sizes;
    this.children = children;

    // optimize access if this batch has no null values
    // some types (like union) may have null values in
    // child batches, but no top-level validity buffer
    if (!nullCount || !this.validity) {
      /** @type {(index: number) => T | null} */
      this.at = index => this.value(index);
    }
  }

  /**
   * Provide an informative object string tag.
   */
  get [Symbol.toStringTag]() {
    return 'Batch';
  }

  /**
   * Return the value at the given index.
   * @param {number} index The value index.
   * @returns {T | null} The value.
   */
  at(index) {
    return this.isValid(index) ? this.value(index) : null;
  }

  /**
   * Check if a value at the given index is valid (non-null).
   * @param {number} index The value index.
   * @returns {boolean} True if valid, false otherwise.
   */
  isValid(index) {
    return decodeBit(this.validity, index);
  }

  /**
   * Return the value at the given index. This method does not check the
   * validity bitmap and is intended primarily for internal use. In most
   * cases, callers should use the `at()` method instead.
   * @param {number} index The value index
   * @returns {T} The value, ignoring the validity bitmap.
   */
  value(index) {
    return /** @type {T} */ (this.values[index]);
  }

  /**
   * Extract an array of values within the given index range. Unlike
   * Array.slice, all arguments are required and may not be negative indices.
   * @param {number} start The starting index, inclusive
   * @param {number} end The ending index, exclusive
   * @returns {import('./types.js').ValueArray<T?>} The slice of values
   */
  slice(start, end) {
    const n = end - start;
    const values = Array(n);
    for (let i = 0; i < n; ++i) {
      values[i] = this.at(start + i);
    }
    return values;
  }

  /**
   * Return an iterator over the values in this batch.
   * @returns {Iterator<T?>}
   */
  *[Symbol.iterator]() {
    for (let i = 0; i < this.length; ++i) {
      yield this.at(i);
    }
  }
}

/**
 * A batch whose value buffer can be used directly, without transformation.
 * @template T
 * @extends {Batch<T>}
 */
class DirectBatch extends Batch {
  /**
   * Create a new column batch with direct value array access.
   * @param {object} options
   * @param {number} options.length The length of the batch
   * @param {number} options.nullCount The null value count
   * @param {import('./types.js').DataType} options.type The data type.
   * @param {Uint8Array} [options.validity] Validity bitmap buffer
   * @param {import('./types.js').TypedArray} options.values Values buffer
   */
  constructor(options) {
    super(options);
    // underlying buffers may be padded, exceeding the logical batch length
    // we trim the values array so we can safely access it directly
    const { length, values } = this;
    this.values = values.subarray(0, length);
  }

  /**
   * Extract an array of values within the given index range. Unlike
   * Array.slice, all arguments are required and may not be negative indices.
   * When feasible, a zero-copy subarray of a typed array is returned.
   * @param {number} start The starting index, inclusive
   * @param {number} end The ending index, exclusive
   * @returns {import('./types.js').ValueArray<T?>} The slice of values
   */
  slice(start, end) {
    // @ts-ignore
    return this.nullCount
      ? super.slice(start, end)
      : this.values.subarray(start, end);
  }

  /**
   * Return an iterator over the values in this batch.
   * @returns {Iterator<T?>}
   */
  [Symbol.iterator]() {
    return this.nullCount
      ? super[Symbol.iterator]()
      : /** @type {Iterator<T?>} */ (this.values[Symbol.iterator]());
  }
}

/**
 * A batch whose values are transformed to 64-bit numbers.
 * @extends {Batch<number>}
 */
class NumberBatch extends Batch {
  static ArrayType = float64Array;
}

/**
 * A batch whose values should be returned in a standard array.
 * @template T
 * @extends {Batch<T>}
 */
class ArrayBatch extends Batch {
  static ArrayType = Array;
}

/**
 * A batch of null values only.
 * @extends {ArrayBatch<null>}
 */
class NullBatch extends ArrayBatch {
  /**
   * @param {number} index The value index
   * @returns {null}
   */
  value(index) { // eslint-disable-line no-unused-vars
    return null;
  }
}

/**
 * A batch that coerces BigInt values to 64-bit numbers.
 * * @extends {NumberBatch}
 */
class Int64Batch extends NumberBatch {
  /**
   * @param {number} index The value index
   */
  value(index) {
    return toNumber(/** @type {bigint} */ (this.values[index]));
  }
}

/**
 * A batch of 16-bit floating point numbers, accessed as unsigned
 * 16-bit ints and transformed to 64-bit numbers.
 */
class Float16Batch extends NumberBatch {
  /**
   * @param {number} index The value index
   */
  value(index) {
    const v = /** @type {number} */ (this.values[index]);
    const expo = (v & 0x7C00) >> 10;
    const sigf = (v & 0x03FF) / 1024;
    const sign = (-1) ** ((v & 0x8000) >> 15);
    switch (expo) {
      case 0x1F: return sign * (sigf ? Number.NaN : 1 / 0);
      case 0x00: return sign * (sigf ? 6.103515625e-5 * sigf : 0);
    }
    return sign * (2 ** (expo - 15)) * (1 + sigf);
  }
}

/**
 * A batch of boolean values stored as a bitmap.
 * @extends {ArrayBatch<boolean>}
 */
class BoolBatch extends ArrayBatch {
  /**
   * @param {number} index The value index
   */
  value(index) {
    return decodeBit(/** @type {Uint8Array} */ (this.values), index);
  }
}

/**
 * An abstract class for a batch of 128- or 256-bit decimal numbers,
 * accessed in strided BigUint64Arrays.
 * @template T
 * @extends {Batch<T>}
 */
class DecimalBatch extends Batch {
  constructor(options) {
    super(options);
    const { bitWidth, scale } = /** @type {import('./types.js').DecimalType} */ (this.type);
    this.decimal = bitWidth === 128 ? fromDecimal128 : fromDecimal256;
    this.scale = 10n ** BigInt(scale);
  }
}

/**
 * A batch of 128- or 256-bit decimal numbers, returned as converted
 * 64-bit numbers. The number coercion may be lossy if the decimal
 * precision can not be represented in a 64-bit floating point format.
 * @extends {DecimalBatch<number>}
 */
class DecimalNumberBatch extends DecimalBatch {
  static ArrayType = float64Array;
  /**
   * @param {number} index The value index
   */
  value(index) {
    return divide(
      this.decimal(/** @type {BigUint64Array} */ (this.values), index),
      this.scale
    );
  }
}

/**
 * A batch of 128- or 256-bit decimal numbers, returned as scaled
 * bigint values, such that all fractional digits have been shifted
 * to integer places by the decimal type scale factor.
 * @extends {DecimalBatch<bigint>}
 */
class DecimalBigIntBatch extends DecimalBatch {
  static ArrayType = Array;
  /**
   * @param {number} index The value index
   */
  value(index) {
    return this.decimal(/** @type {BigUint64Array} */ (this.values), index);
  }
}

/**
 * A batch of date or timestamp values that are coerced to UNIX epoch timestamps
 * and returned as JS Date objects. This batch wraps a source batch that provides
 * timestamp values.
 * @extends {ArrayBatch<Date>}
 */
class DateBatch extends ArrayBatch {
  /**
   * Create a new date batch.
   * @param {Batch<number>} batch A batch of timestamp values.
   */
  constructor(batch) {
    super(batch);
    this.source = batch;
  }

  /**
   * @param {number} index The value index
   */
  value(index) {
    return new Date(this.source.value(index));
  }
}

/**
 * A batch of dates as day counts, coerced to timestamp numbers.
 */
class DateDayBatch extends NumberBatch {
  /**
   * @param {number} index The value index
   * @returns {number}
   */
  value(index) {
    // epoch days to milliseconds
    return 86400000 * /** @type {number} */ (this.values[index]);
  }
}

/**
 * A batch of dates as millisecond timestamps, coerced to numbers.
 */
const DateDayMillisecondBatch = Int64Batch;

/**
 * A batch of timestaps in seconds, coerced to millisecond numbers.
 */
class TimestampSecondBatch extends Int64Batch {
  /**
   * @param {number} index The value index
   */
  value(index) {
    return super.value(index) * 1e3; // seconds to milliseconds
  }
}

/**
 * A batch of timestaps in milliseconds, coerced to numbers.
 */
const TimestampMillisecondBatch = Int64Batch;

/**
 * A batch of timestaps in microseconds, coerced to millisecond numbers.
 */
class TimestampMicrosecondBatch extends Int64Batch {
  /**
   * @param {number} index The value index
   */
  value(index) {
    // microseconds to milliseconds
    return divide(/** @type {bigint} */ (this.values[index]), 1000n);
  }
}

/**
 * A batch of timestaps in nanoseconds, coerced to millisecond numbers.
 */
class TimestampNanosecondBatch extends Int64Batch {
  /**
   * @param {number} index The value index
   */
  value(index) {
    // nanoseconds to milliseconds
    return divide(/** @type {bigint} */ (this.values[index]), 1000000n);
  }
}

/**
 * A batch of day/time intervals, returned as two-element 32-bit int arrays.
 * @extends {ArrayBatch<Int32Array>}
 */
class IntervalDayTimeBatch extends ArrayBatch {
  /**
   * @param {number} index The value index
   * @returns {Int32Array}
   */
  value(index) {
    const values = /** @type {Int32Array} */ (this.values);
    return values.subarray(index << 1, (index + 1) << 1);
  }
}

/**
 * A batch of month/day/nanosecond intervals, returned as three-element arrays.
 * @extends {ArrayBatch<Float64Array>}
 */
class IntervalMonthDayNanoBatch extends ArrayBatch {
  /**
   * @param {number} index The value index
   */
  value(index) {
    const values = /** @type {Uint8Array} */ (this.values);
    const base = index << 4;
    return Float64Array.of(
      readInt32(values, base),
      readInt32(values, base + 4),
      readInt64(values, base + 8)
    );
  }
}

const offset32 = ({values, offsets}, index) => values.subarray(offsets[index], offsets[index + 1]);
const offset64 = ({values, offsets}, index) => values.subarray(toNumber(offsets[index]), toNumber(offsets[index + 1]));

/**
 * A batch of binary blobs with variable offsets, returned as byte buffers of
 * unsigned 8-bit integers. The offsets are 32-bit ints.
 * @extends {ArrayBatch<Uint8Array>}
 */
class BinaryBatch extends ArrayBatch {
  /**
   * @param {number} index
   * @returns {Uint8Array}
   */
  value(index) {
    return offset32(this, index);
  }
}

/**
 * A batch of binary blobs with variable offsets, returned as byte buffers of
 * unsigned 8-bit integers. The offsets are 64-bit ints. Value extraction will
 * fail if an offset exceeds `Number.MAX_SAFE_INTEGER`.
 * @extends {ArrayBatch<Uint8Array>}
 */
class LargeBinaryBatch extends ArrayBatch {
  /**
   * @param {number} index
   * @returns {Uint8Array}
   */
  value(index) {
    return offset64(this, index);
  }
}

/**
 * A batch of UTF-8 strings with variable offsets. The offsets are 32-bit ints.
 * @extends {ArrayBatch<string>}
 */
class Utf8Batch extends ArrayBatch {
  /**
   * @param {number} index
   */
  value(index) {
    return decodeUtf8(offset32(this, index));
  }
}

/**
 * A batch of UTF-8 strings with variable offsets. The offsets are 64-bit ints.
 * Value extraction will fail if an offset exceeds `Number.MAX_SAFE_INTEGER`.
 * @extends {ArrayBatch<string>}
 */
class LargeUtf8Batch extends ArrayBatch {
  /**
   * @param {number} index
   */
  value(index) {
    return decodeUtf8(offset64(this, index));
  }
}

/**
 * A batch of list (array) values of variable length. The list offsets are
 * 32-bit ints.
 * @template V
 * @extends {ArrayBatch<import('./types.js').ValueArray<V>>}
 */
class ListBatch extends ArrayBatch {
  /**
   * @param {number} index
   * @returns {import('./types.js').ValueArray<V>}
   */
  value(index) {
    const offsets = /** @type {Int32Array} */ (this.offsets);
    return this.children[0].slice(offsets[index], offsets[index + 1]);
  }
}

/**
 * A batch of list (array) values of variable length. The list offsets are
 * 64-bit ints. Value extraction will fail if an offset exceeds
 * `Number.MAX_SAFE_INTEGER`.
 * @template V
 * @extends {ArrayBatch<import('./types.js').ValueArray<V>>}
 */
class LargeListBatch extends ArrayBatch {
  /**
   * @param {number} index
   * @returns {import('./types.js').ValueArray<V>}
   */
  value(index) {
    const offsets = /** @type {BigInt64Array} */ (this.offsets);
    return this.children[0].slice(toNumber(offsets[index]), toNumber(offsets[index + 1]));
  }
}

/**
 * A batch of list (array) values of variable length. The list offsets and
 * sizes are 32-bit ints.
 * @template V
 * @extends {ArrayBatch<import('./types.js').ValueArray<V>>}
 */
class ListViewBatch extends ArrayBatch {
  /**
   * @param {number} index
   * @returns {import('./types.js').ValueArray<V>}
   */
  value(index) {
    const a = /** @type {number} */ (this.offsets[index]);
    const b = a + /** @type {number} */ (this.sizes[index]);
    return this.children[0].slice(a, b);
  }
}

/**
 * A batch of list (array) values of variable length. The list offsets and
 * sizes are 64-bit ints. Value extraction will fail if an offset or size
 * exceeds `Number.MAX_SAFE_INTEGER`.
 * @template V
 * @extends {ArrayBatch<import('./types.js').ValueArray<V>>}
 */
class LargeListViewBatch extends ArrayBatch {
  /**
   * @param {number} index
   * @returns {import('./types.js').ValueArray<V>}
   */
  value(index) {
    const a = /** @type {bigint} */ (this.offsets[index]);
    const b = a + /** @type {bigint} */ (this.sizes[index]);
    return this.children[0].slice(toNumber(a), toNumber(b));
  }
}

/**
 * A batch with a fixed stride.
 * @template T
 * @extends {ArrayBatch<T>}
 */
class FixedBatch extends ArrayBatch {
  constructor(options) {
    super(options);
    /** @type {number} */
    // @ts-ignore
    this.stride = this.type.stride;
  }
}

/**
 * A batch of binary blobs of fixed size, returned as byte buffers of unsigned
 * 8-bit integers.
 * @extends {FixedBatch<Uint8Array>}
 */
class FixedBinaryBatch extends FixedBatch {
  /**
   * @param {number} index
   * @returns {Uint8Array}
   */
  value(index) {
    const { stride, values } = this;
    return /** @type {Uint8Array} */ (values)
      .subarray(index * stride, (index + 1) * stride);
  }
}

/**
 * A batch of list (array) values of fixed length.
 * @template V
 * @extends {FixedBatch<import('./types.js').ValueArray<V>>}
 */
class FixedListBatch extends FixedBatch {
  /**
   * @param {number} index
   * @returns {import('./types.js').ValueArray<V>}
   */
  value(index) {
    const { children, stride } = this;
    return children[0].slice(index * stride, (index + 1) * stride);
  }
}

/**
 * Extract Map key-value pairs from parallel child batches.
 */
function pairs({ children, offsets }, index) {
  const [ keys, vals ] = children[0].children;
  const start = offsets[index];
  const end = offsets[index + 1];
  const entries = [];
  for (let i = start; i < end; ++i) {
    entries.push([keys.at(i), vals.at(i)]);
  }
  return entries;
}

/**
 * A batch of map (key, value) values. The map is represented as a list of
 * key-value structs.
 * @template K, V
 * @extends {ArrayBatch<[K, V][]>}
 */
class MapEntryBatch extends ArrayBatch {
  /**
   * Return the value at the given index.
   * @param {number} index The value index.
   * @returns {[K, V][]} The map entries as an array of [key, value] arrays.
   */
  value(index) {
    return /** @type {[K, V][]} */ (pairs(this, index));
  }
}

/**
 * A batch of map (key, value) values. The map is represented as a list of
 * key-value structs.
 * @template K, V
 * @extends {ArrayBatch<Map<K, V>>}
 */
class MapBatch extends ArrayBatch {
  /**
   * Return the value at the given index.
   * @param {number} index The value index.
   * @returns {Map<K, V>} The map value.
   */
  value(index) {
    return new Map(/** @type {[K, V][]} */ (pairs(this, index)));
  }
}

/**
 * A batch of union-type values with a sparse layout, enabling direct
 * lookup from the child value batches.
 * @template T
 * @extends {ArrayBatch<T>}
 */
class SparseUnionBatch extends ArrayBatch {
  /**
   * Create a new column batch.
   * @param {object} options
   * @param {number} options.length The length of the batch
   * @param {number} options.nullCount The null value count
   * @param {import('./types.js').DataType} options.type The data type.
   * @param {Uint8Array} [options.validity] Validity bitmap buffer
   * @param {Int32Array} [options.offsets] Offsets buffer
   * @param {Batch[]} options.children Children batches
   * @param {Int8Array} options.typeIds Union type ids buffer
   * @param {Record<string, number>} options.map A typeId to children index map
   */
  constructor({ typeIds, ...options }) {
    super(options);
    /** @type {Int8Array} */
    this.typeIds = typeIds;
    /** @type {Record<string, number>} */
    // @ts-ignore
    this.typeMap = this.type.typeMap;
  }

  /**
   * @param {number} index The value index.
   */
  value(index, offset = index) {
    const { typeIds, children, typeMap } = this;
    return children[typeMap[typeIds[index]]].at(offset);
  }
}

/**
 * A batch of union-type values with a dense layout, reqiring offset
 * lookups from the child value batches.
 * @template T
 * @extends {SparseUnionBatch<T>}
 */
class DenseUnionBatch extends SparseUnionBatch {
  /**
   * @param {number} index The value index.
   */
  value(index) {
    return super.value(index, /** @type {number} */ (this.offsets[index]));
  }
}

/**
 * A batch of struct values, containing a set of named properties.
 * Struct property values are extracted and returned as JS objects.
 * @extends {ArrayBatch<Record<string, any>>}
 */
class StructBatch extends ArrayBatch {
  constructor(options, factory = objectFactory) {
    super(options);
    /** @type {string[]} */
    // @ts-ignore
    this.names = this.type.children.map(child => child.name);
    this.factory = factory(this.names, this.children);
  }

  /**
   * @param {number} index The value index.
   * @returns {Record<string, any>}
   */
  value(index) {
    return this.factory(index);
  }
}

/**
 * A batch of struct values, containing a set of named properties.
 * Structs are returned as proxy objects that extract data directly
 * from underlying Arrow batches.
 * @extends {StructBatch}
 */
class StructProxyBatch extends StructBatch {
  constructor(options) {
    super(options, proxyFactory);
  }
}

/**
 * A batch of run-end-encoded values.
 * @template T
 * @extends {ArrayBatch<T>}
 */
class RunEndEncodedBatch extends ArrayBatch {
  /**
   * @param {number} index The value index.
   */
  value(index) {
    const [ { values: runs }, vals ] = this.children;
    return vals.at(
      bisect(/** @type {import('./types.js').IntegerArray} */(runs), index)
    );
  }
}

/**
 * A batch of dictionary-encoded values.
 * @template T
 * @extends {ArrayBatch<T>}
 */
class DictionaryBatch extends ArrayBatch {
  /**
   * Register the backing dictionary. Dictionaries are added
   * after batch creation as the complete dictionary may not
   * be finished across multiple record batches.
   * @param {import('./column.js').Column<T>} dictionary
   * The dictionary of column values.
   */
  setDictionary(dictionary) {
    this.dictionary = dictionary;
    this.cache = dictionary.cache();
    return this;
  }

  /**
   * @param {number} index The value index.
   */
  value(index) {
    return this.cache[this.key(index)];
  }

  /**
   * @param {number} index The value index.
   * @returns {number} The dictionary key
   */
  key(index) {
    return /** @type {number} */ (this.values[index]);
  }
}

/**
 * @template T
 * @extends {ArrayBatch<T>}
 */
class ViewBatch extends ArrayBatch {
  /**
   * Create a new view batch.
   * @param {object} options Batch options.
   * @param {number} options.length The length of the batch
   * @param {number} options.nullCount The null value count
   * @param {import('./types.js').DataType} options.type The data type.
   * @param {Uint8Array} [options.validity] Validity bitmap buffer
   * @param {Uint8Array} options.values Values buffer
   * @param {Uint8Array[]} options.data View data buffers
   */
  constructor({ data, ...options }) {
    super(options);
    this.data = data;
  }

  /**
   * Get the binary data at the provided index.
   * @param {number} index The value index.
   * @returns {Uint8Array}
   */
  view(index) {
    const { values, data } = this;
    const offset = index << 4; // each entry is 16 bytes
    let start = offset + 4;
    let buf = /** @type {Uint8Array} */ (values);
    const length = readInt32(buf, offset);
    if (length > 12) {
      // longer strings are in a data buffer
      start = readInt32(buf, offset + 12);
      buf = data[readInt32(buf, offset + 8)];
    }
    return buf.subarray(start, start + length);
  }
}

/**
 * A batch of binary blobs from variable data buffers, returned as byte
 * buffers of unsigned 8-bit integers.
 * @extends {ViewBatch<Uint8Array>}
 */
class BinaryViewBatch extends ViewBatch {
  /**
   * @param {number} index The value index.
   */
  value(index) {
    return this.view(index);
  }
}

/**
 * A batch of UTF-8 strings from variable data buffers.
 * @extends {ViewBatch<string>}
 */
class Utf8ViewBatch extends ViewBatch {
  /**
   * @param {number} index The value index.
   */
  value(index) {
    return decodeUtf8(this.view(index));
  }
}

/**
 * Build up a column from batches.
 */
function columnBuilder(type) {
  let data = [];
  return {
    add(batch) { data.push(batch); return this; },
    clear: () => data = [],
    done: () => new Column(data, type)
  };
}

/**
 * A data column. A column provides a view over one or more value batches,
 * each drawn from an Arrow record batch. While this class supports random
 * access to column values by integer index; however, extracting arrays using
 * `toArray()` or iterating over values (`for (const value of column) {...}`)
 * provide more efficient ways for bulk access or scanning.
 * @template T
 */
class Column {
  /**
   * Create a new column instance.
   * @param {import('./batch.js').Batch<T>[]} data The value batches.
   * @param {import('./types.js').DataType} [type] The column data type.
   *  If not specified, the type is extracted from the batches.
   */
  constructor(data, type = data[0]?.type) {
    /**
     * The column data type.
     * @type {import('./types.js').DataType}
     * @readonly
     */
    this.type = type;
    /**
     * The column length.
     * @type {number}
     * @readonly
     */
    this.length = data.reduce((m, c) => m + c.length, 0);
    /**
     * The count of null values in the column.
     * @type {number}
     * @readonly
     */
    this.nullCount = data.reduce((m, c) => m + c.nullCount, 0);
    /**
     * An array of column data batches.
     * @type {readonly import('./batch.js').Batch<T>[]}
     * @readonly
     */
    this.data = data;

    const n = data.length;
    const offsets = new Int32Array(n + 1);
    if (n === 1) {
      const [ batch ] = data;
      offsets[1] = batch.length;
      // optimize access to single batch
      this.at = index => batch.at(index);
    } else {
      for (let i = 0, s = 0; i < n; ++i) {
        offsets[i + 1] = (s += data[i].length);
      }
    }

    /**
     * Index offsets for data batches.
     * Used to map a column row index to a batch-specific index.
     * @type {Int32Array}
     * @readonly
     */
    this.offsets = offsets;
  }

  /**
   * Provide an informative object string tag.
   */
  get [Symbol.toStringTag]() {
    return 'Column';
  }

  /**
   * Return an iterator over the values in this column.
   * @returns {Iterator<T?>}
   */
  [Symbol.iterator]() {
    const data = this.data;
    return data.length === 1
      ? data[0][Symbol.iterator]()
      : batchedIterator(data);
  }

  /**
   * Return the column value at the given index. If a column has multiple
   * batches, this method performs binary search over the batch lengths to
   * determine the batch from which to retrieve the value. The search makes
   * lookup less efficient than a standard array access. If making a full
   * scan of a column, consider extracting arrays via `toArray()` or using an
   * iterator (`for (const value of column) {...}`).
   * @param {number} index The row index.
   * @returns {T | null} The value.
   */
  at(index) {
    // NOTE: if there is only one batch, this method is replaced with an
    // optimized version in the Column constructor.
    const { data, offsets } = this;
    const i = bisect(offsets, index) - 1;
    return data[i]?.at(index - offsets[i]); // undefined if out of range
  }

  /**
   * Return the column value at the given index. This method is the same as
   * `at()` and is provided for better compatibility with Apache Arrow JS.
   * @param {number} index The row index.
   * @returns {T | null} The value.
   */
  get(index) {
    return this.at(index);
  }

  /**
   * Extract column values into a single array instance. When possible,
   * a zero-copy subarray of the input Arrow data is returned.
   * @returns {import('./types.js').ValueArray<T?>}
   */
  toArray() {
    const { length, nullCount, data } = this;
    const copy = !nullCount && isDirectBatch(data[0]);
    const n = data.length;

    if (copy && n === 1) {
      // use batch array directly
      // @ts-ignore
      return data[0].values;
    }

    // determine output array type
    const ArrayType = !n || nullCount > 0 ? Array
      // @ts-ignore
      : (data[0].constructor.ArrayType ?? data[0].values.constructor);

    const array = new ArrayType(length);
    return copy ? copyArray(array, data) : extractArray(array, data);
  }

  /**
   * Return an array of cached column values.
   * Used internally to accelerate dictionary types.
   */
  cache() {
    return this._cache ?? (this._cache = this.toArray());
  }
}

function *batchedIterator(data) {
  for (let i = 0; i < data.length; ++i) {
    const iter = data[i][Symbol.iterator]();
    for (let next = iter.next(); !next.done; next = iter.next()) {
      yield next.value;
    }
  }
}

function copyArray(array, data) {
  for (let i = 0, offset = 0; i < data.length; ++i) {
    const { values } = data[i];
    array.set(values, offset);
    offset += values.length;
  }
  return array;
}

function extractArray(array, data) {
  let index = -1;
  for (let i = 0; i < data.length; ++i) {
    const batch = data[i];
    for (let j = 0; j < batch.length; ++j) {
      array[++index] = batch.at(j);
    }
  }
  return array;
}

/**
 * A table consists of a collection of named columns (or 'children').
 * To work with table data directly in JavaScript, use `toColumns()`
 * to extract an object that maps column names to extracted value arrays,
 * or `toArray()` to extract an array of row objects. For random access
 * by row index, use `getChild()` to access data for a specific column.
 */
class Table {
  /**
   * Create a new table with the given schema and columns (children).
   * @param {import('./types.js').Schema} schema The table schema.
   * @param {import('./column.js').Column[]} children The table columns.
   * @param {boolean} [useProxy=false] Flag indicating if row proxy
   *  objects should be used to represent table rows (default `false`).
   */
  constructor(schema, children, useProxy = false) {
    const names = schema.fields.map(f => f.name);

    /** @readonly */
    this.schema = schema;
    /** @readonly */
    this.names = names;
    /**
     * @type {import('./column.js').Column[]}
     * @readonly
     */
    this.children = children;
    /**
     * @type {import('./types.js').StructFactory}
     * @readonly
     */
    this.factory = useProxy ? proxyFactory : objectFactory;

    // lazily created row object generators
    const gen = [];

    /**
     * Returns a row object generator for the given batch index.
     * @private
     * @readonly
     * @param {number} b The batch index.
     * @returns {(index: number) => Record<string,any>}
     */
    this.getFactory = b => gen[b]
      ?? (gen[b] = this.factory(names, children.map(c => c.data[b])));
  }

  /**
   * Provide an informative object string tag.
   */
  get [Symbol.toStringTag]() {
    return 'Table';
  }

  /**
   * The number of columns in this table.
   * @return {number} The number of columns.
   */
  get numCols() {
    return this.names.length;
  }

  /**
   * The number of rows in this table.
   * @return {number} The number of rows.
   */
  get numRows() {
    return this.children[0]?.length ?? 0;
  }

  /**
   * Return the child column at the given index position.
   * @param {number} index The column index.
   * @returns {import('./column.js').Column<any>}
   */
  getChildAt(index) {
    return this.children[index];
  }

  /**
   * Return the first child column with the given name.
   * @param {string} name The column name.
   * @returns {import('./column.js').Column<any>}
   */
  getChild(name) {
    const i = this.names.findIndex(x => x === name);
    return i > -1 ? this.children[i] : undefined;
  }

  /**
   * Construct a new table containing only columns at the specified indices.
   * The order of columns in the new table matches the order of input indices.
   * @param {number[]} indices The indices of columns to keep.
   * @param {string[]} [as] Optional new names for selected columns.
   * @returns {Table} A new table with columns at the specified indices.
   */
  selectAt(indices, as = []) {
    const { children, factory, schema } = this;
    const { fields } = schema;
    return new Table(
      {
        ...schema,
        fields: indices.map((i, j) => renameField(fields[i], as[j]))
      },
      indices.map(i => children[i]),
      factory === proxyFactory
    );
  }

  /**
   * Construct a new table containing only columns with the specified names.
   * If columns have duplicate names, the first (with lowest index) is used.
   * The order of columns in the new table matches the order of input names.
   * @param {string[]} names Names of columns to keep.
   * @param {string[]} [as] Optional new names for selected columns.
   * @returns {Table} A new table with columns matching the specified names.
   */
  select(names, as) {
    const all = this.names;
    const indices = names.map(name => all.indexOf(name));
    return this.selectAt(indices, as);
  }

  /**
   * Return an object mapping column names to extracted value arrays.
   * @returns {Record<string, import('./types.js').ValueArray<any>>}
   */
  toColumns() {
    const { children, names } = this;
    /** @type {Record<string, import('./types.js').ValueArray<any>>} */
    const cols = {};
    names.forEach((name, i) => cols[name] = children[i]?.toArray() ?? [] );
    return cols;
  }

  /**
   * Return an array of objects representing the rows of this table.
   * @returns {Record<string, any>[]}
   */
  toArray() {
    const { children, getFactory, numRows } = this;
    const data = children[0]?.data ?? [];
    const output = Array(numRows);
    for (let b = 0, row = -1; b < data.length; ++b) {
      const f = getFactory(b);
      for (let i = 0; i < data[b].length; ++i) {
        output[++row] = f(i);
      }
    }
    return output;
  }

  /**
   * Return an iterator over objects representing the rows of this table.
   * @returns {Generator<Record<string, any>, any, null>}
   */
  *[Symbol.iterator]() {
    const { children, getFactory } = this;
    const data = children[0]?.data ?? [];
    for (let b = 0; b < data.length; ++b) {
      const f = getFactory(b);
      for (let i = 0; i < data[b].length; ++i) {
        yield f(i);
      }
    }
  }

  /**
   * Return a row object for the given index.
   * @param {number} index The row index.
   * @returns {Record<string, any>} The row object.
   */
  at(index) {
    const { children, getFactory, numRows } = this;
    if (index < 0 || index >= numRows) return null;
    const [{ offsets }] = children;
    const b = bisect(offsets, index) - 1;
    return getFactory(b)(index - offsets[b]);
  }

  /**
   * Return a row object for the given index. This method is the same as
   * `at()` and is provided for better compatibility with Apache Arrow JS.
   * @param {number} index The row index.
   * @returns {Record<string, any>} The row object.
   */
  get(index) {
    return this.at(index);
  }
}

function renameField(field, name) {
  return (name != null && name !== field.name)
    ? { ...field, name }
    : field;
}

function batchType(type, options = {}) {
  const { typeId, bitWidth, precision, unit } = type;
  const { useBigInt, useDate, useDecimalBigInt, useMap, useProxy } = options;

  switch (typeId) {
    case Type.Null: return NullBatch;
    case Type.Bool: return BoolBatch;
    case Type.Int:
    case Type.Time:
    case Type.Duration:
      return useBigInt || bitWidth < 64 ? DirectBatch : Int64Batch;
    case Type.Float:
      return precision ? DirectBatch : Float16Batch;
    case Type.Date:
      return wrap(
        unit === DateUnit.DAY ? DateDayBatch : DateDayMillisecondBatch,
        useDate && DateBatch
      );
    case Type.Timestamp:
      return wrap(
        unit === TimeUnit.SECOND ? TimestampSecondBatch
          : unit === TimeUnit.MILLISECOND ? TimestampMillisecondBatch
          : unit === TimeUnit.MICROSECOND ? TimestampMicrosecondBatch
          : TimestampNanosecondBatch,
        useDate && DateBatch
      );
    case Type.Decimal:
      return useDecimalBigInt ? DecimalBigIntBatch : DecimalNumberBatch;
    case Type.Interval:
      return unit === IntervalUnit.DAY_TIME ? IntervalDayTimeBatch
        : unit === IntervalUnit.YEAR_MONTH ? DirectBatch
        : IntervalMonthDayNanoBatch;
    case Type.FixedSizeBinary: return FixedBinaryBatch;
    case Type.Utf8: return Utf8Batch;
    case Type.LargeUtf8: return LargeUtf8Batch;
    case Type.Binary: return BinaryBatch;
    case Type.LargeBinary: return LargeBinaryBatch;
    case Type.BinaryView: return BinaryViewBatch;
    case Type.Utf8View: return Utf8ViewBatch;
    case Type.List: return ListBatch;
    case Type.LargeList: return LargeListBatch;
    case Type.Map: return useMap ? MapBatch : MapEntryBatch;
    case Type.ListView: return ListViewBatch;
    case Type.LargeListView: return LargeListViewBatch;
    case Type.FixedSizeList: return FixedListBatch;
    case Type.Struct: return useProxy ? StructProxyBatch : StructBatch;
    case Type.RunEndEncoded: return RunEndEncodedBatch;
    case Type.Dictionary: return DictionaryBatch;
    case Type.Union: return type.mode ? DenseUnionBatch : SparseUnionBatch;
  }
  throw new Error(invalidDataType(typeId));
}

function wrap(BaseClass, WrapperClass) {
  return WrapperClass
    ? class WrapBatch extends WrapperClass {
        constructor(options) {
          super(new BaseClass(options));
        }
      }
    : BaseClass;
}

/**
 * Decode a block that points to messages within an Arrow 'file' format.
 * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data
 * @param {number} index The starting index in the byte buffer
 * @returns The file block.
 */
function decodeBlock(buf, index) {
  //  0: offset
  //  8: metadataLength
  // 16: bodyLength
  return {
    offset: readInt64(buf, index),
    metadataLength: readInt32(buf, index + 8),
    bodyLength: readInt64(buf, index + 16)
  }
}

/**
 * Decode a vector of blocks.
 * @param {Uint8Array} buf
 * @param {number} index
 * @returns An array of file blocks.
 */
function decodeBlocks(buf, index) {
  return readVector(buf, index, 24, decodeBlock);
}

/**
 * Decode a record batch.
 * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data
 * @param {number} index The starting index in the byte buffer
 * @param {import('../types.js').Version_} version Arrow version value
 * @returns {import('../types.js').RecordBatch} The record batch
 */
function decodeRecordBatch(buf, index, version) {
  //  4: length
  //  6: nodes
  //  8: buffers
  // 10: compression (not supported)
  // 12: variadicBuffers (buffer counts for view-typed fields)
  const get = readObject(buf, index);
  if (get(10, readOffset, 0)) {
    throw new Error('Record batch compression not implemented');
  }

  // If an Arrow buffer was written before version 4,
  // advance 8 bytes to skip the now-removed page_id field
  const offset = version < Version.V4 ? 8 : 0;

  return {
    length: get(4, readInt64, 0),
    nodes: readVector(buf, get(6, readOffset), 16, (buf, pos) => ({
      length: readInt64(buf, pos),
      nullCount: readInt64(buf, pos + 8)
    })),
    regions: readVector(buf, get(8, readOffset), 16 + offset, (buf, pos) => ({
      offset: readInt64(buf, pos + offset),
      length: readInt64(buf, pos + offset + 8)
    })),
    variadic: readVector(buf, get(12, readOffset), 8, readInt64)
  };
}

/**
 * Decode a dictionary batch.
 * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data
 * @param {number} index The starting index in the byte buffer
 * @param {import('../types.js').Version_} version Arrow version value
 * @returns {import('../types.js').DictionaryBatch} The dictionary batch
 */
function decodeDictionaryBatch(buf, index, version) {
  //  4: id
  //  6: data
  //  8: isDelta
  const get = readObject(buf, index);
  return {
    id: get(4, readInt64, 0),
    data: get(6, (buf, off) => decodeRecordBatch(buf, off, version)),
    /**
     * If isDelta is true the values in the dictionary are to be appended to a
     * dictionary with the indicated id. If isDelta is false this dictionary
     * should replace the existing dictionary.
     */
    isDelta: get(8, readBoolean, false)
  };
}

/**
 * Decode a data type definition for a field.
 * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data.
 * @param {number} index The starting index in the byte buffer.
 * @param {number} typeId The data type id.
 * @param {import('../types.js').Field[]} [children] A list of parsed child fields.
 * @returns {import('../types.js').DataType} The data type.
 */
function decodeDataType(buf, index, typeId, children) {
  checkOneOf(typeId, Type, invalidDataType);
  const get = readObject(buf, index);

  switch (typeId) {
    // types without flatbuffer objects
    case Type.Binary: return binary();
    case Type.Utf8: return utf8();
    case Type.LargeBinary: return largeBinary();
    case Type.LargeUtf8: return largeUtf8();
    case Type.List: return list(children[0]);
    case Type.ListView: return listView(children[0]);
    case Type.LargeList: return largeList(children[0]);
    case Type.LargeListView: return largeListView(children[0]);
    case Type.Struct: return struct(children);
    case Type.RunEndEncoded: return runEndEncoded(children[0], children[1]);

    // types with flatbuffer objects
    case Type.Int: return int(
      // @ts-ignore
      get(4, readInt32, 0), // bitwidth
      get(6, readBoolean, false) // signed
    );
    case Type.Float: return float(
      // @ts-ignore
      get(4, readInt16, Precision.HALF) // precision
    );
    case Type.Decimal: return decimal(
      get(4, readInt32, 0), // precision
      get(6, readInt32, 0), // scale
      // @ts-ignore
      get(8, readInt32, 128) // bitwidth
    );
    case Type.Date: return date(
      // @ts-ignore
      get(4, readInt16, DateUnit.MILLISECOND) // unit
    );
    case Type.Time: return time(
      // @ts-ignore
      get(4, readInt16, TimeUnit.MILLISECOND), // unit
      get(6, readInt32, 32) // bitWidth
    );
    case Type.Timestamp: return timestamp(
      // @ts-ignore
      get(4, readInt16, TimeUnit.SECOND), // unit
      get(6, readString) // timezone
    );
    case Type.Interval: return interval(
      // @ts-ignore
      get(4, readInt16, IntervalUnit.YEAR_MONTH) // unit
    );
    case Type.Duration: return duration(
      // @ts-ignore
      get(4, readInt16, TimeUnit.MILLISECOND) // unit
    );

    case Type.FixedSizeBinary: return fixedSizeBinary(
      get(4, readInt32, 0) // stride
    );
    case Type.FixedSizeList: return fixedSizeList(
      children[0],
      get(4, readInt32, 0), // stride
    );
    case Type.Map: return mapType(
      get(4, readBoolean, false), // keysSorted
      children[0]
    );

    case Type.Union: return union(
      // @ts-ignore
      get(4, readInt16, UnionMode.Sparse), // mode
      children,
      readVector(buf, get(6, readOffset), 4, readInt32) // type ids
    );
  }
  // case Type.NONE:
  // case Type.Null:
  // case Type.Bool:
  // case Type.BinaryView:
  // case Type.Utf8View:
  // @ts-ignore
  return { typeId };
}

/**
 * Decode custom metadata consisting of key-value string pairs.
 * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data
 * @param {number} index The starting index in the byte buffer
 * @returns {import('../types.js').Metadata | null} The custom metadata map
 */
function decodeMetadata(buf, index) {
  const entries = readVector(buf, index, 4, (buf, pos) => {
    const get = readObject(buf, pos);
    return /** @type {[string, string]} */ ([
      get(4, readString), // 4: key (string)
      get(6, readString)  // 6: key (string)
    ]);
  });
  return entries.length ? new Map(entries) : null;
}

/**
 * Decode a table schema describing the fields and their data types.
 * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data
 * @param {number} index The starting index in the byte buffer
 * @param {import('../types.js').Version_} version Arrow version value
 * @returns {import('../types.js').Schema} The schema
 */
function decodeSchema(buf, index, version) {
  //  4: endianness (int16)
  //  6: fields (vector)
  //  8: metadata (vector)
  // 10: features (int64[])
  const get = readObject(buf, index);
  return {
    version,
    endianness: /** @type {import('../types.js').Endianness_} */ (get(4, readInt16, 0)),
    fields: get(6, decodeSchemaFields, []),
    metadata: get(8, decodeMetadata)
  };
}

/**
 * @returns {import('../types.js').Field[] | null}
 */
function decodeSchemaFields(buf, fieldsOffset) {
  return readVector(buf, fieldsOffset, 4, decodeField);
}

/**
 * @returns {import('../types.js').Field}
 */
function decodeField(buf, index) {
  //  4: name (string)
  //  6: nullable (bool)
  //  8: type id (uint8)
  // 10: type (union)
  // 12: dictionary (table)
  // 14: children (vector)
  // 16: metadata (vector)
  const get = readObject(buf, index);
  const typeId = get(8, readUint8, Type.NONE);
  const typeOffset = get(10, readOffset, 0);
  const dict = get(12, decodeDictionary);
  const children = get(14, (buf, off) => decodeFieldChildren(buf, off));

  let type = decodeDataType(buf, typeOffset, typeId, children);
  if (dict) {
    dict.dictionary = type;
    type = dict;
  }

  return {
    name: get(4, readString),
    type,
    nullable: get(6, readBoolean, false),
    metadata: get(16, decodeMetadata)
  };
}

/**
 * @returns {import('../types.js').Field[] | null}
 */
function decodeFieldChildren(buf, fieldOffset) {
  const children = readVector(buf, fieldOffset, 4, decodeField);
  return children.length ? children : null;
}

/**
 * @param {Uint8Array} buf
 * @param {number} index
 * @returns {import('../types.js').DictionaryType}
 */
function decodeDictionary(buf, index) {
  if (!index) return null;
  //  4: id (int64)
  //  6: indexType (Int type)
  //  8: isOrdered (boolean)
  // 10: kind (int16) currently only dense array is supported
  const get = readObject(buf, index);
  return dictionary(
    null, // data type will be populated by caller
    get(6, decodeInt, int32()), // index type
    get(8, readBoolean, false), // ordered
    get(4, readInt64, 0), // id
  );
}

/**
 * Decode an integer data type.
 * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data.
 * @param {number} index The starting index in the byte buffer.
 * @returns {import('../types.js').IntType}
 */
function decodeInt(buf, index) {
  return /** @type {import('../types.js').IntType} */ (
    decodeDataType(buf, index, Type.Int)
  );
}

const invalidMessageMetadata = (expected, actual) =>
  `Expected to read ${expected} metadata bytes, but only read ${actual}.`;

const invalidMessageBodyLength = (expected, actual) =>
  `Expected to read ${expected} bytes for message body, but only read ${actual}.`;

const invalidMessageType = (type) =>
  `Unsupported message type: ${type} (${keyFor(MessageHeader, type)})`;

/**
 * A "message" contains a block of Apache Arrow data, such as a schema,
 * record batch, or dictionary batch. This message decodes a single
 * message, returning its associated metadata and content.
 * @param {Uint8Array} buf A byte buffer of binary Arrow IPC data
 * @param {number} index The starting index in the byte buffer
 * @returns {import('../types.js').Message} The decoded message.
 */
function decodeMessage(buf, index) {
  // get message start
  let metadataLength = readInt32(buf, index) || 0;
  index += SIZEOF_INT;

  // ARROW-6313: If the first 4 bytes are continuation indicator (-1), read
  // the next 4 for the 32-bit metadata length. Otherwise, assume this is a
  // pre-v0.15 message, where the first 4 bytes are the metadata length.
  if (metadataLength === -1) {
    metadataLength = readInt32(buf, index) || 0;
    index += SIZEOF_INT;
  }
  if (metadataLength === 0) return null;

  const head = buf.subarray(index, index += metadataLength);
  if (head.byteLength < metadataLength) {
    throw new Error(invalidMessageMetadata(metadataLength, head.byteLength));
  }

  // decode message metadata
  //  4: version
  //  6: headerType
  //  8: headerIndex
  // 10: bodyLength
  const get = readObject(head, 0);
  const version = /** @type {import('../types.js').Version_} */
    (get(4, readInt16, Version.V1));
  const type = /** @type {import('../types.js').MessageHeader_} */
    (get(6, readUint8, MessageHeader.NONE));
  const offset = get(8, readOffset, 0);
  const bodyLength = get(10, readInt64, 0);
  let content;

  if (offset) {
    // decode message header
    const decoder = type === MessageHeader.Schema ? decodeSchema
      : type === MessageHeader.DictionaryBatch ? decodeDictionaryBatch
      : type === MessageHeader.RecordBatch ? decodeRecordBatch
      : null;
    if (!decoder) throw new Error(invalidMessageType(type));
    content = decoder(head, offset, version);

    // extract message body
    if (bodyLength > 0) {
      const body = buf.subarray(index, index += bodyLength);
      if (body.byteLength < bodyLength) {
        throw new Error(invalidMessageBodyLength(bodyLength, body.byteLength));
      }
      // @ts-ignore
      content.body = body;
    }
  }

  return { version, type, index, content };
}

/**
 * Decode [Apache Arrow IPC data][1] and return parsed schema, record batch,
 * and dictionary batch definitions. The input binary data may be either
 * an `ArrayBuffer` or `Uint8Array`. For Arrow data in the IPC 'stream' format,
 * an array of `Uint8Array` instances is also supported.
 *
 * This method stops short of generating views over field buffers. Use the
 * `createData()` method on the result to enable column data access.
 *
 * [1]: https://arrow.apache.org/docs/format/Columnar.html#serialization-and-interprocess-communication-ipc
 * @param {ArrayBuffer | Uint8Array | Uint8Array[]} data
 *  The source byte buffer, or an array of buffers. If an array, each byte
 *  array may contain one or more self-contained messages. Messages may NOT
 *  span multiple byte arrays.
 * @returns {import('../types.js').ArrowData}
 */
function decodeIPC(data) {
  const source = data instanceof ArrayBuffer
    ? new Uint8Array(data)
    : data;
  return source instanceof Uint8Array && isArrowFileFormat(source)
    ? decodeIPCFile(source)
    : decodeIPCStream(source);
}

/**
 * @param {Uint8Array} buf
 * @returns {boolean}
 */
function isArrowFileFormat(buf) {
  if (!buf || buf.length < 4) return false;
  for (let i = 0; i < 6; ++i) {
    if (MAGIC[i] !== buf[i]) return false;
  }
  return true;
}

/**
 * Decode data in the [Arrow IPC 'stream' format][1].
 *
 * [1]: https://arrow.apache.org/docs/format/Columnar.html#ipc-streaming-format
 * @param {Uint8Array | Uint8Array[]} data The source byte buffer, or an
 *  array of buffers. If an array, each byte array may contain one or more
 *  self-contained messages. Messages may NOT span multiple byte arrays.
 * @returns {import('../types.js').ArrowData}
 */
function decodeIPCStream(data) {
  const stream = [data].flat();

  let schema;
  const records = [];
  const dictionaries = [];

  // consume each message in the stream
  for (const buf of stream) {
    if (!(buf instanceof Uint8Array)) {
      throw new Error(`IPC data batch was not a Uint8Array.`);
    }
    let offset = 0;

    // decode all messages in current buffer
    while (true) {
      const m = decodeMessage(buf, offset);
      if (m === null) break; // end of messages
      offset = m.index;
      if (!m.content) continue;
      switch (m.type) {
        case MessageHeader.Schema:
          // ignore repeated schema messages
          if (!schema) schema = m.content;
          break;
        case MessageHeader.RecordBatch:
          records.push(m.content);
          break;
        case MessageHeader.DictionaryBatch:
          dictionaries.push(m.content);
          break;
      }
    }
  }

  return /** @type {import('../types.js').ArrowData} */ (
    { schema, dictionaries, records, metadata: null }
  );
}

/**
 * Decode data in the [Arrow IPC 'file' format][1].
 *
 * [1]: https://arrow.apache.org/docs/format/Columnar.html#ipc-file-format
 * @param {Uint8Array} data The source byte buffer.
 * @returns {import('../types.js').ArrowData}
 */
function decodeIPCFile(data) {
  // find footer location
  const offset = data.byteLength - (MAGIC.length + 4);
  const length = readInt32(data, offset);

  // decode file footer
  //  4: version
  //  6: schema
  //  8: dictionaries (vector)
  // 10: batches (vector)
  // 12: metadata
  const get = readObject(data, offset - length);
  const version = /** @type {import('../types.js').Version_} */
    (get(4, readInt16, Version.V1));
  const dicts = get(8, decodeBlocks, []);
  const recs = get(10, decodeBlocks, []);

  return /** @type {import('../types.js').ArrowData} */ ({
    schema: get(6, (buf, index) => decodeSchema(buf, index, version)),
    dictionaries: dicts.map(({ offset }) => decodeMessage(data, offset).content),
    records: recs.map(({ offset }) => decodeMessage(data, offset).content),
    metadata: get(12, decodeMetadata)
  });
}

/**
 * Decode [Apache Arrow IPC data][1] and return a new Table. The input binary
 * data may be either an `ArrayBuffer` or `Uint8Array`. For Arrow data in the
 * [IPC 'stream' format][2], an array of `Uint8Array` values is also supported.
 *
 * [1]: https://arrow.apache.org/docs/format/Columnar.html#serialization-and-interprocess-communication-ipc
 * [2]: https://arrow.apache.org/docs/format/Columnar.html#ipc-streaming-format
 * @param {ArrayBuffer | Uint8Array | Uint8Array[]} data
 *  The source byte buffer, or an array of buffers. If an array, each byte
 *  array may contain one or more self-contained messages. Messages may NOT
 *  span multiple byte arrays.
 * @param {import('../types.js').ExtractionOptions} [options]
 *  Options for controlling how values are transformed when extracted
 *  from an Arrow binary representation.
 * @returns {Table} A Table instance.
 */
function tableFromIPC(data, options) {
  return createTable(decodeIPC(data), options);
}

/**
 * Create a table from parsed IPC data.
 * @param {import('../types.js').ArrowData} data
 *  The IPC data, as returned by parseIPC.
 * @param {import('../types.js').ExtractionOptions} [options]
 *  Options for controlling how values are transformed when extracted
 *  from am Arrow binary representation.
 * @returns {Table} A Table instance.
 */
function createTable(data, options = {}) {
  const { schema = { fields: [] }, dictionaries, records } = data;
  const { version, fields } = schema;
  const dictionaryMap = new Map;
  const context = contextGenerator(options, version, dictionaryMap);

  // build dictionary type map
  const dictionaryTypes = new Map;
  visitSchemaFields(schema, field => {
    const type = field.type;
    if (type.typeId === Type.Dictionary) {
      dictionaryTypes.set(type.id, type.dictionary);
    }
  });

  // decode dictionaries, build dictionary column map
  const dicts = new Map;
  for (const dict of dictionaries) {
    const { id, data, isDelta, body } = dict;
    const type = dictionaryTypes.get(id);
    const batch = visit$1(type, context({ ...data, body }));
    if (!dicts.has(id)) {
      if (isDelta) {
        throw new Error('Delta update can not be first dictionary batch.');
      }
      dicts.set(id, columnBuilder(type).add(batch));
    } else {
      const dict = dicts.get(id);
      if (!isDelta) dict.clear();
      dict.add(batch);
    }
  }
  dicts.forEach((value, key) => dictionaryMap.set(key, value.done()));

  // decode column fields
  const cols = fields.map(f => columnBuilder(f.type));
  for (const batch of records) {
    const ctx = context(batch);
    fields.forEach((f, i) => cols[i].add(visit$1(f.type, ctx)));
  }

  return new Table(schema, cols.map(c => c.done()), options.useProxy);
}

/**
 * Visit all fields within a schema.
 * @param {import('../types.js').Schema} schema
 * @param {(field: import('../types.js').Field) => void} visitor
 */
function visitSchemaFields(schema, visitor) {
  schema.fields.forEach(function visitField(field) {
    visitor(field);
    // @ts-ignore
    field.type.dictionary?.children?.forEach(visitField);
    // @ts-ignore
    field.type.children?.forEach(visitField);
  });
}

/**
 * Context object generator for field visitation and buffer definition.
 */
function contextGenerator(options, version, dictionaryMap) {
  const base = {
    version,
    options,
    dictionary: id => dictionaryMap.get(id),
  };

  /**
   * Return a context generator.
   * @param {import('../types.js').RecordBatch} batch
   */
  return batch => {
    const { length, nodes, regions, variadic, body } = batch;
    let nodeIndex = -1;
    let bufferIndex = -1;
    let variadicIndex = -1;
    return {
      ...base,
      length,
      node: () => nodes[++nodeIndex],
      buffer: (ArrayType) => {
        const { length, offset } = regions[++bufferIndex];
        return ArrayType
          ? new ArrayType(body.buffer, body.byteOffset + offset, length / ArrayType.BYTES_PER_ELEMENT)
          : body.subarray(offset, offset + length)
      },
      variadic: () => variadic[++variadicIndex],
      visit(children) { return children.map(f => visit$1(f.type, this)); }
    };
  };
}

/**
 * Visit a field, instantiating views of buffer regions.
 */
function visit$1(type, ctx) {
  const { typeId } = type;
  const { length, options, node, buffer, variadic, version } = ctx;
  const BatchType = batchType(type, options);

  if (typeId === Type.Null) {
    // no field node, no buffers
    return new BatchType({ length, nullCount: length, type });
  }

  // extract the next { length, nullCount } field node
  const base = { ...node(), type };

  switch (typeId) {
    // validity and data value buffers
    case Type.Bool:
    case Type.Int:
    case Type.Time:
    case Type.Duration:
    case Type.Float:
    case Type.Decimal:
    case Type.Date:
    case Type.Timestamp:
    case Type.Interval:
    case Type.FixedSizeBinary:
      return new BatchType({
        ...base,
        validity: buffer(),
        values: buffer(type.values)
      });

    // validity, offset, and value buffers
    case Type.Utf8:
    case Type.LargeUtf8:
    case Type.Binary:
    case Type.LargeBinary:
      return new BatchType({
        ...base,
        validity: buffer(),
        offsets: buffer(type.offsets),
        values: buffer()
      });

    // views with variadic buffers
    case Type.BinaryView:
    case Type.Utf8View:
      return new BatchType({
        ...base,
        validity: buffer(),
        values: buffer(), // views buffer
        data: Array.from({ length: variadic() }, () => buffer()) // data buffers
      });

    // validity, offset, and list child
    case Type.List:
    case Type.LargeList:
    case Type.Map:
      return new BatchType({
        ...base,
        validity: buffer(),
        offsets: buffer(type.offsets),
        children: ctx.visit(type.children)
      });

    // validity, offset, size, and list child
    case Type.ListView:
    case Type.LargeListView:
      return new BatchType({
        ...base,
        validity: buffer(),
        offsets: buffer(type.offsets),
        sizes: buffer(type.offsets),
        children: ctx.visit(type.children)
      });

    // validity and children
    case Type.FixedSizeList:
    case Type.Struct:
      return new BatchType({
        ...base,
        validity: buffer(),
        children: ctx.visit(type.children)
      });

    // children only
    case Type.RunEndEncoded:
      return new BatchType({
        ...base,
        children: ctx.visit(type.children)
      });

    // dictionary
    case Type.Dictionary: {
      const { id, indices } = type;
      return new BatchType({
        ...base,
        validity: buffer(),
        values: buffer(indices.values),
      }).setDictionary(ctx.dictionary(id));
    }

    // union
    case Type.Union: {
      if (version < Version.V5) {
        buffer(); // skip unused null bitmap
      }
      return new BatchType({
        ...base,
        typeIds: buffer(int8Array),
        offsets: type.mode === UnionMode.Sparse ? null : buffer(type.offsets),
        children: ctx.visit(type.children)
      });
    }

    // unsupported type
    default:
      throw new Error(invalidDataType(typeId));
  }
}

function writeInt32(buf, index, value) {
  buf[index] = value;
  buf[index + 1] = value >> 8;
  buf[index + 2] = value >> 16;
  buf[index + 3] = value >> 24;
}

const INIT_SIZE = 1024;

/** Flatbuffer binary builder. */
class Builder {
  /**
   * Create a new builder instance.
   * @param {import('./sink.js').Sink} sink The byte consumer.
   */
  constructor(sink) {
    /**
     * Sink that consumes built byte buffers;
     * @type {import('./sink.js').Sink}
     */
    this.sink = sink;
    /**
     * Minimum alignment encountered so far.
     * @type {number}
     */
    this.minalign = 1;
    /**
     * Current byte buffer.
     * @type {Uint8Array}
     */
    this.buf = new Uint8Array(INIT_SIZE);
    /**
     * Remaining space in the current buffer.
     * @type {number}
     */
    this.space = INIT_SIZE;
    /**
     * List of offsets of all vtables. Used to find and
     * reuse tables upon duplicated table field schemas.
     * @type {number[]}
     */
    this.vtables = [];
    /**
     * Total bytes written to sink thus far.
     */
    this.outputBytes = 0;
  }

  /**
   * Returns the flatbuffer offset, relative to the end of the current buffer.
   * @returns {number} Offset relative to the end of the buffer.
   */
  offset() {
    return this.buf.length - this.space;
  }

  /**
   * Write a flatbuffer int8 value at the current buffer position
   * and advance the internal cursor.
   * @param {number} value
   */
  writeInt8(value) {
    this.buf[this.space -= 1] = value;
  }

  /**
   * Write a flatbuffer int16 value at the current buffer position
   * and advance the internal cursor.
   * @param {number} value
   */
  writeInt16(value) {
    this.buf[this.space -= 2] = value;
    this.buf[this.space + 1] = value >> 8;
  }

  /**
   * Write a flatbuffer int32 value at the current buffer position
   * and advance the internal cursor.
   * @param {number} value
   */
  writeInt32(value) {
    writeInt32(this.buf, this.space -= 4, value);
  }

  /**
   * Write a flatbuffer int64 value at the current buffer position
   * and advance the internal cursor.
   * @param {number} value
   */
  writeInt64(value) {
    const v = BigInt(value);
    this.writeInt32(Number(BigInt.asIntN(32, v >> BigInt(32))));
    this.writeInt32(Number(BigInt.asIntN(32, v)));
  }

  /**
   * Add a flatbuffer int8 value, properly aligned,
   * @param value The int8 value to add the buffer.
   */
  addInt8(value) {
    prep(this, 1, 0);
    this.writeInt8(value);
  }

  /**
   * Add a flatbuffer int16 value, properly aligned,
   * @param value The int16 value to add the buffer.
   */
  addInt16(value) {
    prep(this, 2, 0);
    this.writeInt16(value);
  }

  /**
   * Add a flatbuffer int32 value, properly aligned,
   * @param value The int32 value to add the buffer.
   */
  addInt32(value) {
    prep(this, 4, 0);
    this.writeInt32(value);
  }

  /**
   * Add a flatbuffer int64 values, properly aligned.
   * @param value The int64 value to add the buffer.
   */
  addInt64(value) {
    prep(this, 8, 0);
    this.writeInt64(value);
  }

  /**
   * Add a flatbuffer offset, relative to where it will be written.
   * @param {number} offset The offset to add.
   */
  addOffset(offset) {
    prep(this, SIZEOF_INT, 0); // Ensure alignment is already done.
    this.writeInt32(this.offset() - offset + SIZEOF_INT);
  }

  /**
   * Add a flatbuffer object (vtable).
   * @param {number} numFields The maximum number of fields
   *  this object may include.
   * @param {(tableBuilder: ReturnType<objectBuilder>) => void} [addFields]
   *  A callback function that writes all fields using an object builder.
   * @returns {number} The object offset.
   */
  addObject(numFields, addFields) {
    const b = objectBuilder(this, numFields);
    addFields?.(b);
    return b.finish();
  }

  /**
   * Add a flatbuffer vector (list).
   * @template T
   * @param {T[]} items An array of items to write.
   * @param {number} itemSize The size in bytes of a serialized item.
   * @param {number} alignment The desired byte alignment value.
   * @param {(builder: this, item: T) => void} writeItem A callback
   *  function that writes a vector item to this builder.
   * @returns {number} The vector offset.
   */
  addVector(items, itemSize, alignment, writeItem) {
    const n = items?.length;
    if (!n) return 0;
    prep(this, SIZEOF_INT, itemSize * n);
    prep(this, alignment, itemSize * n); // Just in case alignment > int.
    for (let i = n; --i >= 0;) {
      writeItem(this, items[i]);
    }
    this.writeInt32(n);
    return this.offset();
  }

  /**
   * Convenience method for writing a vector of byte buffer offsets.
   * @param {number[]} offsets
   * @returns {number} The vector offset.
   */
  addOffsetVector(offsets) {
    return this.addVector(offsets, 4, 4, (b, off) => b.addOffset(off));
  }

  /**
   * Add a flatbuffer UTF-8 string.
   * @param {string} s The string to encode.
   * @return {number} The string offset.
   */
  addString(s) {
    if (s == null) return 0;
    const utf8 = encodeUtf8(s);
    const n = utf8.length;
    this.addInt8(0); // string null terminator
    prep(this, SIZEOF_INT, n);
    this.buf.set(utf8, this.space -= n);
    this.writeInt32(n);
    return this.offset();
  }

  /**
   * Finish the current flatbuffer by adding a root offset.
   * @param {number} rootOffset The root offset.
   */
  finish(rootOffset) {
    prep(this, this.minalign, SIZEOF_INT);
    this.addOffset(rootOffset);
  }

  /**
   * Flush the current flatbuffer byte buffer content to the sink,
   * and reset the flatbuffer builder state.
   */
  flush() {
    const { buf, sink } = this;
    const bytes = buf.subarray(this.space, buf.length);
    sink.write(bytes);
    this.outputBytes += bytes.byteLength;
    this.minalign = 1;
    this.vtables = [];
    this.buf = new Uint8Array(INIT_SIZE);
    this.space = INIT_SIZE;
  }

  /**
   * Add a byte buffer directly to the builder sink. This method bypasses
   * any unflushed flatbuffer state and leaves it unchanged, writing the
   * buffer to the sink *before* the flatbuffer.
   * The buffer will be padded for 64-bit (8-byte) alignment as needed.
   * @param {Uint8Array} buffer The buffer to add.
   * @returns {number} The total byte count of the buffer and padding.
   */
  addBuffer(buffer) {
    const size = buffer.byteLength;
    if (!size) return 0;
    this.sink.write(buffer);
    this.outputBytes += size;
    const pad = ((size + 7) & ~7) - size;
    this.addPadding(pad);
    return size + pad;
  }

  /**
   * Write padding bytes directly to the builder sink. This method bypasses
   * any unflushed flatbuffer state and leaves it unchanged, writing the
   * padding bytes to the sink *before* the flatbuffer.
   * @param {number} byteCount The number of padding bytes.
   */
  addPadding(byteCount) {
    if (byteCount > 0) {
      this.sink.write(new Uint8Array(byteCount));
      this.outputBytes += byteCount;
    }
  }
}

/**
 * Prepare to write an element of `size` after `additionalBytes` have been
 * written, e.g. if we write a string, we need to align such the int length
 * field is aligned to 4 bytes, and the string data follows it directly. If all
 * we need to do is alignment, `additionalBytes` will be 0.
 * @param {Builder} builder The builder to prep.
 * @param {number} size The size of the new element to write.
 * @param {number} additionalBytes Additional padding size.
 */
function prep(builder, size, additionalBytes) {
  let { buf, space, minalign } = builder;

  // track the biggest thing we've ever aligned to
  if (size > minalign) {
    builder.minalign = size;
  }

  // find alignment needed so that `size` aligns after `additionalBytes`
  const bufSize = buf.length;
  const used = bufSize - space + additionalBytes;
  const alignSize = (~used + 1) & (size - 1);

  // reallocate the buffer if needed
  buf = grow(buf, used + alignSize + size - 1, true);
  space += buf.length - bufSize;

  // add padding
  for (let i = 0; i < alignSize; ++i) {
    buf[--space] = 0;
  }

  // update builder state
  builder.buf = buf;
  builder.space = space;
}

/**
 * Returns a builder object for flatbuffer objects (vtables).
 * @param {Builder} builder The underlying flatbuffer builder.
 * @param {number} numFields The expected number of fields, not
 *  including the standard size fields.
 */
function objectBuilder(builder, numFields) {
  /** @type {number[]} */
  const vtable = Array(numFields).fill(0);
  const startOffset = builder.offset();

  function slot(index) {
    vtable[index] = builder.offset();
  }

  return {
    /**
     * Add an int8-valued table field.
     * @param {number} index
     * @param {number} value
     * @param {number} defaultValue
     */
    addInt8(index, value, defaultValue) {
      if (value != defaultValue) {
        builder.addInt8(value);
        slot(index);
      }
    },

    /**
     * Add an int16-valued table field.
     * @param {number} index
     * @param {number} value
     * @param {number} defaultValue
     */
    addInt16(index, value, defaultValue) {
      if (value != defaultValue) {
        builder.addInt16(value);
        slot(index);
      }
    },

    /**
     * Add an int32-valued table field.
     * @param {number} index
     * @param {number} value
     * @param {number} defaultValue
     */
    addInt32(index, value, defaultValue) {
      if (value != defaultValue) {
        builder.addInt32(value);
        slot(index);
      }
    },

    /**
     * Add an int64-valued table field.
     * @param {number} index
     * @param {number} value
     * @param {number} defaultValue
     */
    addInt64(index, value, defaultValue) {
      if (value != defaultValue) {
        builder.addInt64(value);
        slot(index);
      }
    },

    /**
     * Add a buffer offset-valued table field.
     * @param {number} index
     * @param {number} value
     * @param {number} defaultValue
     */
    addOffset(index, value, defaultValue) {
      if (value != defaultValue) {
        builder.addOffset(value);
        slot(index);
      }
    },

    /**
     * Write the vtable to the buffer and return the table offset.
     * @returns {number} The buffer offset to the vtable.
     */
    finish() {
      // add offset entry, will overwrite later with actual offset
      builder.addInt32(0);
      const vtableOffset = builder.offset();

      // trim zero-valued fields (indicating default value)
      let i = numFields;
      while (--i >= 0 && vtable[i] === 0) {} // eslint-disable-line no-empty
      const size = i + 1;

      // Write out the current vtable.
      for (; i >= 0; --i) {
        // Offset relative to the start of the table.
        builder.addInt16(vtable[i] ? (vtableOffset - vtable[i]) : 0);
      }

      const standardFields = 2; // size fields
      builder.addInt16(vtableOffset - startOffset);
      const len = (size + standardFields) * SIZEOF_SHORT;
      builder.addInt16(len);

      // Search for an existing vtable that matches the current one.
      let existingTable = 0;
      const { buf, vtables, space: vt1 } = builder;
    outer_loop:
      for (i = 0; i < vtables.length; ++i) {
        const vt2 = buf.length - vtables[i];
        if (len == readInt16(buf, vt2)) {
          for (let j = SIZEOF_SHORT; j < len; j += SIZEOF_SHORT) {
            if (readInt16(buf, vt1 + j) != readInt16(buf, vt2 + j)) {
              continue outer_loop;
            }
          }
          existingTable = vtables[i];
          break;
        }
      }

      if (existingTable) {
        // Found a match: remove the current vtable.
        // Point table to existing vtable.
        builder.space = buf.length - vtableOffset;
        writeInt32(buf, builder.space, existingTable - vtableOffset);
      } else {
        // No match: add the location of the current vtable to the vtables list.
        // Point table to current vtable.
        const off = builder.offset();
        vtables.push(off);
        writeInt32(buf, buf.length - vtableOffset, off - vtableOffset);
      }

      return vtableOffset;
    }
  }
}

/**
 * @param {import('./builder.js').Builder} builder
 * @param {import('../types.js').RecordBatch} batch
 * @returns {number}
 */
function encodeRecordBatch(builder, batch) {
  const { nodes, regions, variadic } = batch;
  const nodeVector = builder.addVector(nodes, 16, 8,
    (builder, node) => {
      builder.writeInt64(node.nullCount);
      builder.writeInt64(node.length);
      return builder.offset();
    }
  );
  const regionVector = builder.addVector(regions, 16, 8,
    (builder, region) => {
      builder.writeInt64(region.length);
      builder.writeInt64(region.offset);
      return builder.offset();
    }
  );
  const variadicVector = builder.addVector(variadic, 8, 8,
    (builder, count) => builder.addInt64(count)
  );
  return builder.addObject(5, b => {
    b.addInt64(0, nodes[0].length, 0);
    b.addOffset(1, nodeVector, 0);
    b.addOffset(2, regionVector, 0);
    // NOT SUPPORTED: 3, compression offset
    b.addOffset(4, variadicVector, 0);
  });
}

/**
 * @param {import('./builder.js').Builder} builder
 * @param {import('../types.js').DictionaryBatch} dictionaryBatch
 * @returns {number}
 */
function encodeDictionaryBatch(builder, dictionaryBatch) {
  const dataOffset = encodeRecordBatch(builder, dictionaryBatch.data);
  return builder.addObject(3, b => {
    b.addInt64(0, dictionaryBatch.id, 0);
    b.addOffset(1, dataOffset, 0);
    b.addInt8(2, +dictionaryBatch.isDelta, 0);
  });
}

/**
 * @param {import('./builder.js').Builder} builder
 * @param {Map<string, string>} metadata
 * @returns {number}
 */
function encodeMetadata(builder, metadata) {
  return metadata?.size > 0
     ? builder.addOffsetVector(Array.from(metadata, ([k, v]) => {
        const key = builder.addString(`${k}`);
        const val = builder.addString(`${v}`);
        return builder.addObject(2, b => {
          b.addOffset(0, key, 0);
          b.addOffset(1, val, 0);
        });
      }))
    : 0;
}

/**
 * Encode a data type into a flatbuffer.
 * @param {import('./builder.js').Builder} builder
 * @param {import('../types.js').DataType} type
 * @returns {number} The offset at which the data type is written.
 */
function encodeDataType(builder, type) {
  const typeId = checkOneOf(type.typeId, Type, invalidDataType);

  switch (typeId) {
    case Type.Dictionary:
      return encodeDictionary(builder, type);
    case Type.Int:
      return encodeInt(builder, type);
    case Type.Float:
      return encodeFloat(builder, type);
    case Type.Decimal:
      return encodeDecimal(builder, type);
    case Type.Date:
      return encodeDate(builder, type);
    case Type.Time:
      return encodeTime(builder, type);
    case Type.Timestamp:
      return encodeTimestamp(builder, type);
    case Type.Interval:
      return encodeInterval(builder, type);
    case Type.Duration:
      return encodeDuration(builder, type);
    case Type.FixedSizeBinary:
    case Type.FixedSizeList:
      return encodeFixedSize(builder, type);
    case Type.Map:
      return encodeMap(builder, type);
    case Type.Union:
      return encodeUnion(builder, type);
  }
  // case Type.Null:
  // case Type.Binary:
  // case Type.LargeBinary:
  // case Type.BinaryView:
  // case Type.Bool:
  // case Type.Utf8:
  // case Type.Utf8View:
  // case Type.LargeUtf8:
  // case Type.List:
  // case Type.ListView:
  // case Type.LargeList:
  // case Type.LargeListView:
  // case Type.RunEndEncoded:
  // case Type.Struct:
  return builder.addObject(0);
}

function encodeDate(builder, type) {
  return builder.addObject(1, b => {
    b.addInt16(0, type.unit, DateUnit.MILLISECOND);
  });
}

function encodeDecimal(builder, type) {
  return builder.addObject(3, b => {
    b.addInt32(0, type.precision, 0);
    b.addInt32(1, type.scale, 0);
    b.addInt32(2, type.bitWidth, 128);
  });
}

function encodeDuration(builder, type) {
  return builder.addObject(1, b => {
    b.addInt16(0, type.unit, TimeUnit.MILLISECOND);
  });
}

function encodeFixedSize(builder, type) {
  return builder.addObject(1, b => {
    b.addInt32(0, type.stride, 0);
  });
}

function encodeFloat(builder, type) {
  return builder.addObject(1, b => {
    b.addInt16(0, type.precision, Precision.HALF);
  });
}

function encodeInt(builder, type) {
  return builder.addObject(2, b => {
    b.addInt32(0, type.bitWidth, 0);
    b.addInt8(1, +type.signed, 0);
  });
}

function encodeInterval(builder, type) {
  return builder.addObject(1, b => {
    b.addInt16(0, type.unit, IntervalUnit.YEAR_MONTH);
  });
}

function encodeMap(builder, type) {
  return builder.addObject(1, b => {
    b.addInt8(0, +type.keysSorted, 0);
  });
}

function encodeTime(builder, type) {
  return builder.addObject(2, b => {
    b.addInt16(0, type.unit, TimeUnit.MILLISECOND);
    b.addInt32(1, type.bitWidth, 32);
  });
}

function encodeTimestamp(builder, type) {
  const timezoneOffset = builder.addString(type.timezone);
  return builder.addObject(2, b => {
    b.addInt16(0, type.unit, TimeUnit.SECOND);
    b.addOffset(1, timezoneOffset, 0);
  });
}

function encodeUnion(builder, type) {
  const typeIdsOffset = builder.addVector(
    type.typeIds, 4, 4,
    (builder, value) => builder.addInt32(value)
  );
  return builder.addObject(2, b => {
    b.addInt16(0, type.mode, UnionMode.Sparse);
    b.addOffset(1, typeIdsOffset, 0);
  });
}

function encodeDictionary(builder, type) {
  const keyTypeOffset = isInt32(type.indices)
    ? 0
    : encodeDataType(builder, type.indices);
  return builder.addObject(4, b => {
    b.addInt64(0, type.id, 0);
    b.addOffset(1, keyTypeOffset, 0);
    b.addInt8(2, +type.ordered, 0);
    // NOT SUPPORTED: 3, dictionaryKind (defaults to dense array)
  });
}

function isInt32(type) {
  return type.typeId === Type.Int && type.bitWidth === 32 && type.signed;
}

const isLittleEndian = new Uint16Array(new Uint8Array([1, 0]).buffer)[0] === 1;

/**
 * @param {import('./builder.js').Builder} builder
 * @param {import('../types.js').Schema} schema
 * @returns {number}
 */
function encodeSchema(builder, schema) {
  const { fields, metadata } = schema;
  const fieldOffsets = fields.map(f => encodeField(builder, f));
  const fieldsVectorOffset = builder.addOffsetVector(fieldOffsets);
  const metadataOffset = encodeMetadata(builder, metadata);
  return builder.addObject(4, b => {
    b.addInt16(0, +(!isLittleEndian), 0);
    b.addOffset(1, fieldsVectorOffset, 0);
    b.addOffset(2, metadataOffset, 0);
    // NOT SUPPORTED: 3, features
  });
}

/**
 * @param {import('./builder.js').Builder} builder
 * @param {import('../types.js').Field} field
 * @returns {number}
 */
function encodeField(builder, field) {
  const { name, nullable, type, metadata } = field;
  let { typeId } = type;

  // encode field data type
  let typeOffset = 0;
  let dictionaryOffset = 0;
  if (typeId !== Type.Dictionary) {
    typeOffset = encodeDataType(builder, type);
  } else {
    const dict = /** @type {import('../types.js').DictionaryType} */ (type).dictionary;
    typeId = dict.typeId;
    dictionaryOffset = encodeDataType(builder, type);
    typeOffset = encodeDataType(builder, dict);
  }

  // encode children, metadata, name, and field object
  // @ts-ignore
  const childOffsets = (type.children || []).map(f => encodeField(builder, f));
  const childrenVectorOffset = builder.addOffsetVector(childOffsets);
  const metadataOffset = encodeMetadata(builder, metadata);
  const nameOffset = builder.addString(name);
  return builder.addObject(7, b => {
    b.addOffset(0, nameOffset, 0);
    b.addInt8(1, +nullable, +false);
    b.addInt8(2, typeId, Type.NONE);
    b.addOffset(3, typeOffset, 0);
    b.addOffset(4, dictionaryOffset, 0);
    b.addOffset(5, childrenVectorOffset, 0);
    b.addOffset(6, metadataOffset, 0);
  });
}

/**
 * Write a file footer.
 * @param {import('./builder.js').Builder} builder The binary builder.
 * @param {import('../types.js').Schema} schema The table schema.
 * @param {import('../types.js').Block[]} dictBlocks Dictionary batch file blocks.
 * @param {import('../types.js').Block[]} recordBlocks Record batch file blocks.
 * @param {Map<string,string> | null} metadata File-level metadata.
 */
function writeFooter(builder, schema, dictBlocks, recordBlocks, metadata) {
  // encode footer flatbuffer
  const metadataOffset = encodeMetadata(builder, metadata);
  const recsOffset = builder.addVector(recordBlocks, 24, 8, encodeBlock);
  const dictsOffset = builder.addVector(dictBlocks, 24, 8, encodeBlock);
  const schemaOffset = encodeSchema(builder, schema);
  builder.finish(
    builder.addObject(5, b => {
      b.addInt16(0, Version.V5, Version.V1);
      b.addOffset(1, schemaOffset, 0);
      b.addOffset(2, dictsOffset, 0);
      b.addOffset(3, recsOffset, 0);
      b.addOffset(4, metadataOffset, 0);
    })
  );
  const size = builder.offset();

  // add eos with continuation indicator
  builder.addInt32(0);
  builder.addInt32(-1);

  // write builder contents
  builder.flush();

  // write file tail
  builder.sink.write(new Uint8Array(Int32Array.of(size).buffer));
  builder.sink.write(MAGIC);
}

/**
 * Encode a file pointer block.
 * @param {import('./builder.js').Builder} builder
 * @param {import('../types.js').Block} block
 * @returns {number} the current block offset
 */
function encodeBlock(builder, { offset, metadataLength, bodyLength }) {
  builder.writeInt64(bodyLength);
  builder.writeInt32(0);
  builder.writeInt32(metadataLength);
  builder.writeInt64(offset);
  return builder.offset();
}

/**
 * Write an IPC message to the builder sink.
 * @param {import('./builder.js').Builder} builder
 * @param {import('../types.js').MessageHeader_} headerType
 * @param {number} headerOffset
 * @param {number} bodyLength
 * @param {import('../types.js').Block[]} [blocks]
 */
function writeMessage(builder, headerType, headerOffset, bodyLength, blocks) {
  builder.finish(
    builder.addObject(5, b => {
      b.addInt16(0, Version.V5, Version.V1);
      b.addInt8(1, headerType, MessageHeader.NONE);
      b.addOffset(2, headerOffset, 0);
      b.addInt64(3, bodyLength, 0);
      // NOT SUPPORTED: 4, message-level metadata
    })
  );

  const prefixSize = 8; // continuation indicator + message size
  const messageSize = builder.offset();
  const alignedSize = (messageSize + prefixSize + 7) & ~7;

  // track blocks for file footer
  blocks?.push({
    offset: builder.outputBytes,
    metadataLength: alignedSize,
    bodyLength
  });

  // write size prefix (including padding)
  builder.addInt32(alignedSize - prefixSize);

  // write the stream continuation indicator
  builder.addInt32(-1);

  // flush the builder content
  builder.flush();

  // add alignment padding as needed
  builder.addPadding(alignedSize - messageSize - prefixSize);
}

class Sink {
  /**
   * Write bytes to this sink.
   * @param {Uint8Array} bytes The byte buffer to write.
   */
  write(bytes) { // eslint-disable-line no-unused-vars
  }

  /**
   * Write padding bytes (zeroes) to this sink.
   * @param {number} byteCount The number of padding bytes.
   */
  pad(byteCount) {
    this.write(new Uint8Array(byteCount));
  }

  /**
   * @returns {Uint8Array | null}
   */
  finish() {
    return null;
  }
}

class MemorySink extends Sink {
  /**
   * A sink that collects bytes in memory.
   */
  constructor() {
    super();
    this.buffers = [];
  }

  /**
   * Write bytes
   * @param {Uint8Array} bytes
   */
  write(bytes) {
    this.buffers.push(bytes);
  }

  /**
   * @returns {Uint8Array}
   */
  finish() {
    const bufs = this.buffers;
    const size = bufs.reduce((sum, b) => sum + b.byteLength, 0);
    const buf = new Uint8Array(size);
    for (let i = 0, off = 0; i < bufs.length; ++i) {
      buf.set(bufs[i], off);
      off += bufs[i].byteLength;
    }
    return buf;
  }
}

const STREAM = 'stream';
const FILE = 'file';

/**
 * Encode assembled data into Arrow IPC binary format.
 * @param {any} data Assembled table data.
 * @param {object} options Encoding options.
 * @param {import('./sink.js').Sink} [options.sink] IPC byte consumer.
 * @param {'stream' | 'file'} [options.format] Arrow stream or file format.
 * @returns {import('./sink.js').Sink} The sink that was passed in.
 */
function encodeIPC(data, { sink, format = STREAM } = {}) {
  if (format !== STREAM && format !== FILE) {
    throw new Error(`Unrecognized Arrow IPC format: ${format}`);
  }
  const { schema, dictionaries = [], records = [], metadata } = data;
  const builder = new Builder(sink || new MemorySink());
  const file = format === FILE;
  const dictBlocks = [];
  const recordBlocks = [];

  if (file) {
    builder.addBuffer(MAGIC);
  } else if (schema) {
    writeMessage(
      builder,
      MessageHeader.Schema,
      encodeSchema(builder, schema),
      0
    );
  }

  for (const dict of dictionaries) {
    const { data } = dict;
    writeMessage(
      builder,
      MessageHeader.DictionaryBatch,
      encodeDictionaryBatch(builder, dict),
      data.byteLength,
      dictBlocks
    );
    writeBuffers(builder, data.buffers);
  }

  for (const batch of records) {
    writeMessage(
      builder,
      MessageHeader.RecordBatch,
      encodeRecordBatch(builder, batch),
      batch.byteLength,
      recordBlocks
    );
    writeBuffers(builder, batch.buffers);
  }

  if (file) {
    writeFooter(builder, schema, dictBlocks, recordBlocks, metadata);
  }

  return builder.sink;
}

/**
 * Write byte buffers to the builder sink.
 * Buffers are aligned to 64 bits (8 bytes) as needed.
 * @param {import('./builder.js').Builder} builder
 * @param {Uint8Array[]} buffers
 */
function writeBuffers(builder, buffers) {
  for (let i = 0; i < buffers.length; ++i) {
    builder.addBuffer(buffers[i]); // handles alignment for us
  }
}

/**
 * Encode an Arrow table into Arrow IPC binary format.
 * @param {import('../table.js').Table} table The Arrow table to encode.
 * @param {object} options Encoding options.
 * @param {import('./sink.js').Sink} [options.sink] IPC byte consumer.
 * @param {'stream' | 'file'} [options.format] Arrow stream or file format.
 * @returns {Uint8Array | null} The generated bytes (for an in-memory sink)
 *  or null (if using a sink that writes bytes elsewhere).
 */
function tableToIPC(table, options) {
  // accept a format string option for Arrow-JS compatibility
  if (typeof options === 'string') {
    options = { format: options };
  }

  const columns = table.children;
  checkBatchLengths(columns);

  const { dictionaries, idMap } = assembleDictionaryBatches(columns);
  const records = assembleRecordBatches(columns);
  const schema = assembleSchema(table.schema, idMap);
  const data = { schema, dictionaries, records };
  return encodeIPC(data, options).finish();
}

function checkBatchLengths(columns) {
  const n = columns[0]?.data.map(d => d.length);
  columns.forEach(({ data }) => {
    if (data.length !== n.length || data.some((b, i) => b.length !== n[i])) {
      throw new Error('Columns have inconsistent batch sizes.');
    }
  });
}

/**
 * Create a new assembly context.
 */
function assembleContext() {
  let byteLength = 0;
  const nodes = [];
  const regions = [];
  const buffers = [];
  const variadic = [];
  return {
    /**
     * @param {number} length
     * @param {number} nullCount
     */
    node(length, nullCount) {
      nodes.push({ length, nullCount });
    },
    /**
     * @param {import('../types.js').TypedArray} b
     */
    buffer(b) {
      const size = b.byteLength;
      const length = ((size + 7) & ~7);
      regions.push({ offset: byteLength, length });
      byteLength += length;
      buffers.push(new Uint8Array(b.buffer, b.byteOffset, size));
    },
    /**
     * @param {number} length
     */
    variadic(length) {
      variadic.push(length);
    },
    /**
     * @param {import('../types.js').DataType} type
     * @param {import('../batch.js').Batch} batch
     */
    children(type, batch) {
      // @ts-ignore
      type.children.forEach((field, index) => {
        visit(field.type, batch.children[index], this);
      });
    },
    /**
     * @returns {import('../types.js').RecordBatch}
     */
    done() {
      return { byteLength, nodes, regions, variadic, buffers };
    }
  };
}

/**
 * Assemble dictionary batches and their unique ids.
 * @param {import('../column.js').Column[]} columns The table columns.
 * @returns {{
 *    dictionaries: import('../types.js').DictionaryBatch[],
 *    idMap: Map<import('../types.js').DataType, number>
 *  }}
 *  The assembled dictionary batches and a map from dictionary column
 *  instances to dictionary ids.
 */
function assembleDictionaryBatches(columns) {
  const dictionaries = [];
  const dictMap = new Map;
  const idMap = new Map;
  let id = -1;

  // track dictionaries, key by dictionary column, assign ids
  const visitor = dictionaryColumn => {
    if (!dictMap.has(dictionaryColumn)) {
      dictMap.set(dictionaryColumn, ++id);
      for (let i = 0; i < dictionaryColumn.data.length; ++i) {
        dictionaries.push({
          id,
          isDelta: i > 0,
          data: assembleRecordBatch([dictionaryColumn], i)
        });
      }
      idMap.set(dictionaryColumn.type, id);
    } else {
      idMap.set(dictionaryColumn.type, dictMap.get(dictionaryColumn));
    }
  };

  // recurse through column batches to find dictionaries
  // it is sufficient to visit the first batch only,
  // as all batches have the same dictionary column
  columns.forEach(col => visitDictionaries(col.data[0], visitor));

  return { dictionaries, idMap };
}

/**
 * Traverse column batches to visit dictionary columns.
 * @param {import('../batch.js').Batch} batch
 * @param {(column: import('../column.js').Column) => void} visitor
 */
function visitDictionaries(batch, visitor) {
  if (batch?.type.typeId === Type.Dictionary) {
    // @ts-ignore - batch has type DictionaryBatch
    const dictionary = batch.dictionary;
    visitor(dictionary);
    visitDictionaries(dictionary.data[0], visitor);
  }
  batch?.children?.forEach(child => visitDictionaries(child, visitor));
}

/**
 * Assemble a schema with resolved dictionary ids.
 * @param {import('../types.js').Schema} schema The schema.
 * @param {Map<import('../types.js').DataType, number>} idMap A map
 *  from dictionary value types to dictionary ids.
 * @returns {import('../types.js').Schema} A new schema with resolved
 *  dictionary ids. If there are no dictionaries, the input schema is
 *  returned unchanged.
 */
function assembleSchema(schema, idMap) {
  // early exit if no dictionaries
  if (!idMap.size) return schema;

  const visit = type => {
    if (type.typeId === Type.Dictionary) {
      type.id = idMap.get(type.dictionary); // lookup and set id
      visitDictType(type);
    }
    if (type.children) {
      (type.children = type.children.slice()).forEach(visitFields);
    }
  };

  // visit a field in a field array
  const visitFields = (field, index, array) => {
    const type = { ...field.type };
    array[index] = { ...field, type };
    visit(type);
  };

  // visit a dictionary values type
  const visitDictType = (parentType) => {
    const type = { ...parentType.dictionary };
    parentType.dictionary = type;
    visit(type);
  };

  schema = { ...schema, fields: schema.fields.slice() };
  schema.fields.forEach(visitFields);
  return schema;
}

/**
 * Assemble record batches with marshalled buffers.
 * @param {import('../column.js').Column[]} columns The table columns.
 * @returns {import('../types.js').RecordBatch[]} The assembled record batches.
 */
function assembleRecordBatches(columns) {
  return (columns[0]?.data || [])
    .map((_, index) => assembleRecordBatch(columns, index));
}

/**
 * Assemble a record batch with marshalled buffers.
 * @param {import('../column.js').Column[]} columns The table columns.
 * @param {number} batchIndex The batch index.
 * @returns {import('../types.js').RecordBatch} The assembled record batch.
 */
function assembleRecordBatch(columns, batchIndex = 0) {
  const ctx = assembleContext();
  columns.forEach(column => {
    visit(column.type, column.data[batchIndex], ctx);
  });
  return ctx.done();
}

/**
 * Visit a column batch, assembling buffer data.
 * @param {import('../types.js').DataType} type The data type.
 * @param {import('../batch.js').Batch} batch The column batch.
 * @param {ReturnType<assembleContext>} ctx The assembly context.
 */
function visit(type, batch, ctx) {
  const { typeId } = type;

  // no field node, no buffers
  if (typeId === Type.Null) return;

  // record field node info
  ctx.node(batch.length, batch.nullCount);

  switch (typeId) {
    // validity and value buffers
    // backing dictionaries handled elsewhere
    case Type.Bool:
    case Type.Int:
    case Type.Time:
    case Type.Duration:
    case Type.Float:
    case Type.Date:
    case Type.Timestamp:
    case Type.Decimal:
    case Type.Interval:
    case Type.FixedSizeBinary:
    case Type.Dictionary: // dict key values
      ctx.buffer(batch.validity);
      ctx.buffer(batch.values);
      return;

    // validity, offset, and value buffers
    case Type.Utf8:
    case Type.LargeUtf8:
    case Type.Binary:
    case Type.LargeBinary:
      ctx.buffer(batch.validity);
      ctx.buffer(batch.offsets);
      ctx.buffer(batch.values);
      return;

    // views with variadic buffers
    case Type.BinaryView:
    case Type.Utf8View:
      ctx.buffer(batch.validity);
      ctx.buffer(batch.values);
      // @ts-ignore
      ctx.variadic(batch.data.length);
      // @ts-ignore
      batch.data.forEach(b => ctx.buffer(b));
      return;

    // validity, offset, and list child
    case Type.List:
    case Type.LargeList:
    case Type.Map:
      ctx.buffer(batch.validity);
      ctx.buffer(batch.offsets);
      ctx.children(type, batch);
      return;

    // validity, offset, size, and list child
    case Type.ListView:
    case Type.LargeListView:
      ctx.buffer(batch.validity);
      ctx.buffer(batch.offsets);
      ctx.buffer(batch.sizes);
      ctx.children(type, batch);
      return;

    // validity and children
    case Type.FixedSizeList:
    case Type.Struct:
      ctx.buffer(batch.validity);
      ctx.children(type, batch);
      return;

    // children only
    case Type.RunEndEncoded:
      ctx.children(type, batch);
      return;

    // union
    case Type.Union: {
      // @ts-ignore
      ctx.buffer(batch.typeIds);
      if (type.mode === UnionMode.Dense) {
        ctx.buffer(batch.offsets);
      }
      ctx.children(type, batch);
      return;
    }

    // unsupported type
    default:
      throw new Error(invalidDataType(typeId));
  }
}

/**
 * Create a new resizable buffer instance.
 * @param {import('../types.js').TypedArrayConstructor} [arrayType]
 *  The array type.
 * @returns {Buffer} The buffer.
 */
function buffer(arrayType) {
  return new Buffer(arrayType);
}

/**
 * Resizable byte buffer.
 */
class Buffer {
  /**
   * Create a new resizable buffer instance.
   * @param {import('../types.js').TypedArrayConstructor} arrayType
   */
  constructor(arrayType = uint8Array) {
    this.buf = new arrayType(512);
  }
  /**
   * Return the underlying data as a 64-bit aligned array of minimum size.
   * @param {number} size The desired minimum array size.
   * @returns {import('../types.js').TypedArray} The 64-bit aligned array.
   */
  array(size) {
    return align(this.buf, size);
  }
  /**
   * Prepare for writes to the given index, resizing as necessary.
   * @param {number} index The array index to prepare to write to.
   */
  prep(index) {
    if (index >= this.buf.length) {
      this.buf = grow(this.buf, index);
    }
  }
  /**
   * Return the value at the given index.
   * @param {number} index The array index.
   */
  get(index) {
    return this.buf[index];
  }
  /**
   * Set a value at the given index.
   * @param {number | bigint} value The value to set.
   * @param {number} index The index to write to.
   */
  set(value, index) {
    this.prep(index);
    this.buf[index] = value;
  }
  /**
   * Write a byte array at the given index. The method should be called
   * only when the underlying buffer is of type Uint8Array.
   * @param {Uint8Array} bytes The byte array.
   * @param {number} index The starting index to write to.
   */
  write(bytes, index) {
    this.prep(index + bytes.length);
    /** @type {Uint8Array} */ (this.buf).set(bytes, index);
  }
}

/**
 * Create a new resizable bitmap instance.
 * @returns {Bitmap} The bitmap buffer.
 */
function bitmap() {
  return new Bitmap();
}

/**
 * Resizable bitmap buffer.
 */
class Bitmap extends Buffer {
  /**
   * Set a bit to true at the given bitmap index.
   * @param {number} index The index to write to.
   */
  set(index) {
    const i = index >> 3;
    this.prep(i);
    /** @type {Uint8Array} */ (this.buf)[i] |= (1 << (index % 8));
  }
}

/**
 * Abstract class for building a column data batch.
 */
class BatchBuilder {
  constructor(type, ctx) {
    this.type = type;
    this.ctx = ctx;
    this.batchClass = ctx.batchType(type);
  }

  /**
   * Initialize the builder state.
   * @returns {this} This builder.
   */
  init() {
    this.index = -1;
    return this;
  }

  /**
   * Write a value to the builder.
   * @param {*} value
   * @param {number} index
   * @returns {boolean | void}
   */
  set(value, index) {
    this.index = index;
    return false;
  }

  /**
   * Returns a batch constructor options object.
   * Used internally to marshal batch data.
   * @returns {Record<string, any>}
   */
  done() {
    return null;
  }

  /**
   * Returns a completed batch and reinitializes the builder state.
   * @returns {import('../../batch.js').Batch}
   */
  batch() {
    const b = new this.batchClass(this.done());
    this.init();
    return b;
  }
}

/**
 * Builder for validity bitmaps within batches.
 */
class ValidityBuilder extends BatchBuilder {
  constructor(type, ctx) {
    super(type, ctx);
  }

  init() {
    this.nullCount = 0;
    this.validity = bitmap();
    return super.init();
  }

  /**
   * @param {*} value
   * @param {number} index
   * @returns {boolean | void}
   */
  set(value, index) {
    this.index = index;
    const isValid = value != null;
    if (isValid) {
      this.validity.set(index);
    } else {
      this.nullCount++;
    }
    return isValid;
  }

  done() {
    const { index, nullCount, type, validity } = this;
    return {
      length: index + 1,
      nullCount,
      type,
      validity: nullCount
        ? validity.array((index >> 3) + 1)
        : new uint8Array(0)
    };
  }
}

/**
 * Create a context object for managing dictionary builders.
 */
function dictionaryContext() {
  const idMap = new Map;
  const dicts = new Set;
  return {
    /**
     * Get a dictionary values builder for the given dictionary type.
     * @param {import('../../types.js').DictionaryType} type
     *  The dictionary type.
     * @param {*} ctx The builder context.
     * @returns {ReturnType<dictionaryValues>}
     */
    get(type, ctx) {
      // if a dictionary has a non-negative id, assume it was set
      // intentionally and track it for potential reuse across columns
      // otherwise the dictionary is used for a single column only
      const id = type.id;
      if (id >= 0 && idMap.has(id)) {
        return idMap.get(id);
      } else {
        const dict = dictionaryValues(type, ctx);
        if (id >= 0) idMap.set(id, dict);
        dicts.add(dict);
        return dict;
      }
    },
    /**
     * Finish building dictionary values columns and assign them to
     * their corresponding dictionary batches.
     * @param {import('../../types.js').ExtractionOptions} options
     */
    finish(options) {
      dicts.forEach(dict => dict.finish(options));
    }
  };
}

/**
 * Builder helper for creating dictionary values.
 * @param {import('../../types.js').DictionaryType} type
 *  The dictionary data type.
 * @param {ReturnType<import('../builder.js').builderContext>} ctx
 *  The builder context.
 */
function dictionaryValues(type, ctx) {
  const keys = Object.create(null);
  const values = ctx.builder(type.dictionary);
  const batches = [];

  values.init();
  let index = -1;

  return {
    type,
    values,

    add(batch) {
      batches.push(batch);
      return batch;
    },

    key(value) {
      const v = keyString(value);
      let k = keys[v];
      if (k === undefined) {
        keys[v] = k = ++index;
        values.set(value, k);
      }
      return k;
    },

    finish(options) {
      const valueType = type.dictionary;
      const batch = new (batchType(valueType, options))(values.done());
      const dictionary = new Column([batch]);
      batches.forEach(batch => batch.setDictionary(dictionary));
    }
  };
}

/**
 * Builder for dictionary-typed data batches.
 */
class DictionaryBuilder extends ValidityBuilder {
  constructor(type, ctx) {
    super(type, ctx);
    this.dict = ctx.dictionary(type);
  }

  init() {
    this.values = buffer(this.type.indices.values);
    return super.init();
  }

  set(value, index) {
    if (super.set(value, index)) {
      this.values.set(this.dict.key(value), index);
    }
  }

  done() {
    return {
      ...super.done(),
      values: this.values.array(this.index + 1)
    };
  }

  batch() {
    // register batch with dictionary
    // batch will be updated when the dictionary is finished
    return this.dict.add(super.batch());
  }
}

/**
 * Infer the data type for a given input array.
 * @param {(visitor: (value: any) => void) => void} visit
 *  A function that applies a callback to successive data values.
 * @returns {import('../types.js').DataType} The data type.
 */
function inferType(visit) {
  const profile = profiler();
  visit(value => profile.add(value));
  return profile.type();
}

function profiler() {
  let length = 0;
  let nullCount = 0;
  let boolCount = 0;
  let numberCount = 0;
  let intCount = 0;
  let bigintCount = 0;
  let dateCount = 0;
  let dayCount = 0;
  let stringCount = 0;
  let arrayCount = 0;
  let structCount = 0;
  let min = Infinity;
  let max = -Infinity;
  let minLength = Infinity;
  let maxLength = -Infinity;
  let minBigInt;
  let maxBigInt;
  let arrayProfile;
  let structProfiles = {};

  return {
    add(value) {
      length++;
      if (value == null) {
        nullCount++;
        return;
      }
      switch (typeof value) {
        case 'string':
          stringCount++;
          break;
        case 'number':
          numberCount++;
          if (value < min) min = value;
          if (value > max) max = value;
          if (Number.isInteger(value)) intCount++;
          break;
        case 'bigint':
          bigintCount++;
          if (minBigInt === undefined) {
            minBigInt = maxBigInt = value;
          } else {
            if (value < minBigInt) minBigInt = value;
            if (value > maxBigInt) maxBigInt = value;
          }
          break;
        case 'boolean':
          boolCount++;
          break;
        case 'object':
          if (value instanceof Date) {
            dateCount++;
            // 1 day = 1000ms * 60s * 60min * 24hr = 86400000
            if ((+value % 864e5) === 0) dayCount++;
          } else if (isArray(value)) {
            arrayCount++;
            const len = value.length;
            if (len < minLength) minLength = len;
            if (len > maxLength) maxLength = len;
            arrayProfile ??= profiler();
            value.forEach(arrayProfile.add);
          } else {
            structCount++;
            for (const key in value) {
              const fieldProfiler = structProfiles[key]
                ?? (structProfiles[key] = profiler());
              fieldProfiler.add(value[key]);
            }
          }
      }
    },
    type() {
      const valid = length - nullCount;
      return valid === 0 ? nullType()
        : intCount === valid ? intType(min, max)
        : numberCount === valid ? float64()
        : bigintCount === valid ? bigintType(minBigInt, maxBigInt)
        : boolCount === valid ? bool()
        : dayCount === valid ? dateDay()
        : dateCount === valid ? timestamp()
        : stringCount === valid ? dictionary(utf8())
        : arrayCount === valid ? arrayType(arrayProfile.type(), minLength, maxLength)
        : structCount === valid ? struct(
            Object.entries(structProfiles).map(_ => field(_[0], _[1].type()))
          )
        : unionType();
    }
  };
}

/**
 * Return a list or fixed list type.
 * @param {import('../types.js').DataType} type The child data type.
 * @param {number} minLength The minumum list length.
 * @param {number} maxLength The maximum list length.
 * @returns {import('../types.js').DataType} The data type.
 */
function arrayType(type, minLength, maxLength) {
  return maxLength === minLength
    ? fixedSizeList(type, minLength)
    : list(type);
}

/**
 * @param {number} min
 * @param {number} max
 * @returns {import('../types.js').DataType}
 */
function intType(min, max) {
  const v = Math.max(Math.abs(min) - 1, max);
  return v < (1 << 7) ? int8()
    : v < (1 << 15) ? int16()
    : v < (2 ** 31) ? int32()
    : float64();
}

/**
 * @param {bigint} min
 * @param {bigint} max
 * @returns {import('../types.js').IntType}
 */
function bigintType(min, max) {
  const v = -min > max ? -min - 1n : max;
  if (v >= 2 ** 63) {
    throw new Error(`BigInt exceeds 64 bits: ${v}`);
  }
  return int64();
}

/**
 * @returns {import('../types.js').UnionType}
 */
function unionType() {
  throw new Error('Mixed types detected, please define a union type.');
}

/**
 * Builder for batches of binary-typed data.
 */
class BinaryBuilder extends ValidityBuilder {
  constructor(type, ctx) {
    super(type, ctx);
    this.toOffset = toOffset(type.offsets);
  }

  init() {
    this.offsets = buffer(this.type.offsets);
    this.values = buffer();
    this.pos = 0;
    return super.init();
  }

  set(value, index) {
    const { offsets, values, toOffset } = this;
    if (super.set(value, index)) {
      values.write(value, this.pos);
      this.pos += value.length;
    }
    offsets.set(toOffset(this.pos), index + 1);
  }

  done() {
    return {
      ...super.done(),
      offsets: this.offsets.array(this.index + 2),
      values: this.values.array(this.pos + 1)
    };
  }
}

/**
 * Builder for batches of bool-typed data.
 */
class BoolBuilder extends ValidityBuilder {
  constructor(type, ctx) {
    super(type, ctx);
  }

  init() {
    this.values = bitmap();
    return super.init();
  }

  set(value, index) {
    super.set(value, index);
    if (value) this.values.set(index);
  }

  done() {
    return {
      ...super.done(),
      values: this.values.array((this.index >> 3) + 1)
    }
  }
}

/**
 * Builder for batches of decimal-typed data.
 */
class DecimalBuilder extends ValidityBuilder {
  constructor(type, ctx) {
    super(type, ctx);
    this.scale = 10 ** type.scale;
    this.stride = type.bitWidth >> 6;
  }

  init() {
    this.values = buffer(this.type.values);
    return super.init();
  }

  set(value, index) {
    const { scale, stride, values } = this;
    if (super.set(value, index)) {
      values.prep((index + 1) * stride);
      // @ts-ignore
      toDecimal(value, values.buf, index * stride, stride, scale);
    }
  }

  done() {
    const { index, stride, values } = this;
    return {
      ...super.done(),
      values: values.array((index + 1) * stride)
    };
  }
}

/**
 * Builder for fixed-size-binary-typed data batches.
 */
class FixedSizeBinaryBuilder extends ValidityBuilder {
  constructor(type, ctx) {
    super(type, ctx);
    this.stride = type.stride;
  }

  init() {
    this.values = buffer();
    return super.init();
  }

  set(value, index) {
    if (super.set(value, index)) {
      this.values.write(value, index * this.stride);
    }
  }

  done() {
    const { stride, values } = this;
    return {
      ...super.done(),
      values: values.array(stride * (this.index + 1))
    };
  }
}

/**
 * Builder for fixed-size-list-typed data batches.
 */
class FixedSizeListBuilder extends ValidityBuilder {
  constructor(type, ctx) {
    super(type, ctx);
    this.child = ctx.builder(this.type.children[0].type);
    this.stride = type.stride;
  }

  init() {
    this.child.init();
    return super.init();
  }

  set(value, index) {
    const { child, stride } = this;
    const base = index * stride;
    if (super.set(value, index)) {
      for (let i = 0; i < stride; ++i) {
        child.set(value[i], base + i);
      }
    } else {
      child.index = base + stride;
    }
  }

  done() {
    const { child } = this;
    return {
      ...super.done(),
      children: [ child.batch() ]
    };
  }
}

/**
 * Builder for day/time interval-typed data batches.
 */
class IntervalDayTimeBuilder extends ValidityBuilder {
  init() {
    this.values = buffer(this.type.values);
    return super.init();
  }

  set(value, index) {
    if (super.set(value, index)) {
      const i = index << 1;
      this.values.set(value[0], i);
      this.values.set(value[1], i + 1);
    }
  }

  done() {
    return {
      ...super.done(),
      values: this.values.array((this.index + 1) << 1)
    }
  }
}

/**
 * Builder for month/day/nano interval-typed data batches.
 */
class IntervalMonthDayNanoBuilder extends ValidityBuilder {
  init() {
    this.values = buffer();
    return super.init();
  }

  set(value, index) {
    if (super.set(value, index)) {
      this.values.write(toMonthDayNanoBytes(value), index << 4);
    }
  }

  done() {
    return {
      ...super.done(),
      values: this.values.array((this.index + 1) << 4)
    }
  }
}

/**
 * Abstract class for building list data batches.
 */
class AbstractListBuilder extends ValidityBuilder {
  constructor(type, ctx, child) {
    super(type, ctx);
    this.child = child;
  }

  init() {
    this.child.init();
    const offsetType = this.type.offsets;
    this.offsets = buffer(offsetType);
    this.toOffset = toOffset(offsetType);
    this.pos = 0;
    return super.init();
  }

  done() {
    return {
      ...super.done(),
      offsets: this.offsets.array(this.index + 2),
      children: [ this.child.batch() ]
    };
  }
}

/**
 * Builder for list-typed data batches.
 */
class ListBuilder extends AbstractListBuilder {
  constructor(type, ctx) {
    super(type, ctx, ctx.builder(type.children[0].type));
  }

  set(value, index) {
    const { child, offsets, toOffset } = this;
    if (super.set(value, index)) {
      value.forEach(v => child.set(v, this.pos++));
    }
    offsets.set(toOffset(this.pos), index + 1);
  }
}

/**
 * Abstract class for building list-typed data batches.
 */
class AbstractStructBuilder extends ValidityBuilder {
  constructor(type, ctx) {
    super(type, ctx);
    this.children = type.children.map(c => ctx.builder(c.type));
  }

  init() {
    this.children.forEach(c => c.init());
    return super.init();
  }

  done() {
    const { children } = this;
    children.forEach(c => c.index = this.index);
    return {
      ...super.done(),
      children: children.map(c => c.batch())
    };
  }
}

/**
 * Builder for struct-typed data batches.
 */
class StructBuilder extends AbstractStructBuilder {
  constructor(type, ctx) {
    super(type, ctx);
    this.setters = this.children.map((child, i) => {
      const name = type.children[i].name;
      return (value, index) => child.set(value?.[name], index);
    });
  }

  set(value, index) {
    super.set(value, index);
    const setters = this.setters;
    for (let i = 0; i < setters.length; ++i) {
      setters[i](value, index);
    }
  }
}

/**
 * Builder for map-typed data batches.
 */
class MapBuilder extends AbstractListBuilder {
  constructor(type, ctx) {
    super(type, ctx, new MapStructBuilder(type.children[0].type, ctx));
  }

  set(value, index) {
    const { child, offsets, toOffset } = this;
    if (super.set(value, index)) {
      for (const keyValuePair of value) {
        child.set(keyValuePair, this.pos++);
      }
    }
    offsets.set(toOffset(this.pos), index + 1);
  }
}

/**
 * Builder for key-value struct batches within a map.
 */
class MapStructBuilder extends AbstractStructBuilder {
  set(value, index) {
    super.set(value, index);
    const [key, val] = this.children;
    key.set(value[0], index);
    val.set(value[1], index);
  }
}

const NO_VALUE = {}; // empty object that fails strict equality

/**
 * Builder for run-end-encoded-typed data batches.
 */
class RunEndEncodedBuilder extends BatchBuilder {
  constructor(type, ctx) {
    super(type, ctx);
    this.children = type.children.map(c => ctx.builder(c.type));
  }

  init() {
    this.pos = 0;
    this.key = null;
    this.value = NO_VALUE;
    this.children.forEach(c => c.init());
    return super.init();
  }

  next() {
    const [runs, vals] = this.children;
    runs.set(this.index + 1, this.pos);
    vals.set(this.value, this.pos++);
  }

  set(value, index) {
    // perform fast strict equality test
    if (value !== this.value) {
      // if no match, fallback to key string test
      const key = keyString(value);
      if (key !== this.key) {
        // if key doesn't match, write prior run and update
        if (this.key) this.next();
        this.key = key;
        this.value = value;
      }
    }
    this.index = index;
  }

  done() {
    this.next();
    const { children, index, type } = this;
    return {
      length: index + 1,
      nullCount: 0,
      type,
      children: children.map(c => c.batch())
    };
  }
}

/**
 * Abstract class for building union-typed data batches.
 */
class AbstractUnionBuilder extends BatchBuilder {
  constructor(type, ctx) {
    super(type, ctx);
    this.children = type.children.map(c => ctx.builder(c.type));
    this.typeMap = type.typeMap;
    this.lookup = type.typeIdForValue;
  }

  init() {
    this.nullCount = 0;
    this.typeIds = buffer(int8Array);
    this.children.forEach(c => c.init());
    return super.init();
  }

  set(value, index) {
    const { children, lookup, typeMap, typeIds } = this;
    this.index = index;
    const typeId = lookup(value, index);
    const child = children[typeMap[typeId]];
    typeIds.set(typeId, index);
    if (value == null) ++this.nullCount;
    // @ts-ignore
    this.update(value, index, child);
  }

  done() {
    const { children, nullCount, type, typeIds } = this;
    const length = this.index + 1;
    return {
      length,
      nullCount,
      type,
      typeIds: typeIds.array(length),
      children: children.map(c => c.batch())
    };
  }
}

/**
 * Builder for sparse union-typed data batches.
 */
class SparseUnionBuilder extends AbstractUnionBuilder {
  update(value, index, child) {
    // update selected child with value
    // then set all other children to null
    child.set(value, index);
    this.children.forEach(c => { if (c !== child) c.set(null, index); });
  }
}

/**
 * Builder for dense union-typed data batches.
 */
class DenseUnionBuilder extends AbstractUnionBuilder {
  init() {
    this.offsets = buffer(this.type.offsets);
    return super.init();
  }

  update(value, index, child) {
    const offset = child.index + 1;
    child.set(value, offset);
    this.offsets.set(offset, index);
  }

  done() {
    return {
      ...super.done(),
      offsets: this.offsets.array(this.index + 1)
    };
  }
}

/**
 * Builder for utf8-typed data batches.
 */
class Utf8Builder extends BinaryBuilder {
  set(value, index) {
    super.set(value && encodeUtf8(value), index);
  }
}

/**
 * Builder for data batches that can be accessed directly as typed arrays.
 */
class DirectBuilder extends ValidityBuilder {
  constructor(type, ctx) {
    super(type, ctx);
    this.values = buffer(type.values);
  }

  init() {
    this.values = buffer(this.type.values);
    return super.init();
  }

  /**
   * @param {*} value
   * @param {number} index
   * @returns {boolean | void}
   */
  set(value, index) {
    if (super.set(value, index)) {
      this.values.set(value, index);
    }
  }
  done() {
    return {
      ...super.done(),
      values: this.values.array(this.index + 1)
    };
  }
}

/**
 * Builder for int64/uint64 data batches written as bigints.
 */
class Int64Builder extends DirectBuilder {
  set(value, index) {
    super.set(value == null ? value : toBigInt(value), index);
  }
}

/**
 * Builder for data batches whose values must pass through a transform
 * function prior to be written to a backing buffer.
 */
class TransformBuilder extends DirectBuilder {
  constructor(type, ctx, transform) {
    super(type, ctx);
    this.transform = transform;
  }
  set(value, index) {
    super.set(value == null ? value : this.transform(value), index);
  }
}

/**
 * Create a context object for shared builder state.
 * @param {import('../types.js').ExtractionOptions} [options]
 *  Batch extraction options.
* @param {ReturnType<dictionaryContext>} [dictionaries]
 *  Context object for tracking dictionaries.
 */
function builderContext(
  options = {},
  dictionaries = dictionaryContext()
) {
  return {
    batchType: type => batchType(type, options),
    builder(type) { return builder(type, this); },
    dictionary(type) { return dictionaries.get(type, this); },
    finish: () => dictionaries.finish(options)
  };
}

/**
 * Returns a batch builder for the given type and builder context.
 * @param {import('../types.js').DataType} type A data type.
 * @param {ReturnType<builderContext>} [ctx] A builder context.
 * @returns {import('./builders/batch.js').BatchBuilder}
 */
function builder(type, ctx = builderContext()) {
  const { typeId } = type;
  switch (typeId) {
    case Type.Int:
    case Type.Time:
    case Type.Duration:
      return isInt64ArrayType(type.values)
        ? new Int64Builder(type, ctx)
        : new DirectBuilder(type, ctx);
    case Type.Float:
      return type.precision
        ? new DirectBuilder(type, ctx)
        : new TransformBuilder(type, ctx, toFloat16)
    case Type.Binary:
    case Type.LargeBinary:
      return new BinaryBuilder(type, ctx);
    case Type.Utf8:
    case Type.LargeUtf8:
      return new Utf8Builder(type, ctx);
    case Type.Bool:
      return new BoolBuilder(type, ctx);
    case Type.Decimal:
      return new DecimalBuilder(type, ctx);
    case Type.Date:
      return new TransformBuilder(type, ctx, type.unit ? toBigInt : toDateDay);
    case Type.Timestamp:
      return new TransformBuilder(type, ctx, toTimestamp(type.unit));
    case Type.Interval:
      switch (type.unit) {
        case IntervalUnit.DAY_TIME:
          return new IntervalDayTimeBuilder(type, ctx);
        case IntervalUnit.MONTH_DAY_NANO:
          return new IntervalMonthDayNanoBuilder(type, ctx);
      }
      // IntervalUnit.YEAR_MONTH:
      return new DirectBuilder(type, ctx);
    case Type.List:
    case Type.LargeList:
      return new ListBuilder(type, ctx);
    case Type.Struct:
      return new StructBuilder(type, ctx);
    case Type.Union:
      return type.mode
        ? new DenseUnionBuilder(type, ctx)
        : new SparseUnionBuilder(type, ctx);
    case Type.FixedSizeBinary:
      return new FixedSizeBinaryBuilder(type, ctx);
    case Type.FixedSizeList:
      return new FixedSizeListBuilder(type, ctx);
    case Type.Map:
      return new MapBuilder(type, ctx);
    case Type.RunEndEncoded:
      return new RunEndEncodedBuilder(type, ctx);

    case Type.Dictionary:
      return new DictionaryBuilder(type, ctx);
  }
  // case Type.BinaryView:
  // case Type.Utf8View:
  // case Type.ListView:
  // case Type.LargeListView:
  throw new Error(invalidDataType(typeId));
}

/**
 * Create a new column by iterating over provided values.
 * @template T
 * @param {Iterable | ((callback: (value: any) => void) => void)} values
 *  Either an iterable object or a visitor function that applies a callback
 *  to successive data values (akin to Array.forEach).
 * @param {import('../types.js').DataType} [type] The data type.
 * @param {import('../types.js').ColumnBuilderOptions} [options]
 *  Builder options for the generated column.
 * @param {ReturnType<
 *    import('./builders/dictionary.js').dictionaryContext
 *  >} [dicts] Dictionary context object, for internal use only.
 * @returns {Column<T>} The generated column.
 */
function columnFromValues(values, type, options = {}, dicts) {
  const visit = isIterable(values)
    ? callback => { for (const value of values) callback(value); }
    : values;

  type ??= inferType(visit);
  const { maxBatchRows = Infinity, ...opt } = options;
  let data;

  if (type.typeId === Type.Null) {
    let length = 0;
    visit(() => ++length);
    data = nullBatches(type, length, maxBatchRows);
  } else {
    const ctx = builderContext(opt, dicts);
    const b = builder(type, ctx).init();
    const next = b => data.push(b.batch());
    data = [];

    let row = 0;
    visit(value => {
      b.set(value, row++);
      if (row >= maxBatchRows) {
        next(b);
        row = 0;
      }
    });
    if (row) next(b);

    // resolve dictionaries
    ctx.finish();
  }

  return new Column(data, type);
}

/**
 * Create null batches with the given batch size limit.
 * @param {import('../types.js').NullType} type The null data type.
 * @param {number} length The total column length.
 * @param {number} limit The maximum batch size.
 * @returns {import('../batch.js').NullBatch[]} The null batches.
 */
function nullBatches(type, length, limit) {
  const data = [];
  const batch = length => new NullBatch({ length, nullCount: length, type });
  const numBatches = Math.floor(length / limit);
  for (let i = 0; i < numBatches; ++i) {
    data.push(batch(limit));
  }
  const rem = length % limit;
  if (rem) data.push(batch(rem));
  return data;
}

/**
 * Create a new column from a provided data array.
 * @template T
 * @param {Array | import('../types.js').TypedArray} array The input data.
 * @param {import('../types.js').DataType} [type] The data type.
 *  If not specified, type inference is attempted.
 * @param {import('../types.js').ColumnBuilderOptions} [options]
 *  Builder options for the generated column.
 * @param {ReturnType<import('./builders/dictionary.js').dictionaryContext>} [dicts]
 *  Builder context object, for internal use only.
 * @returns {Column<T>} The generated column.
 */
function columnFromArray(array, type, options = {}, dicts) {
  return !type && isTypedArray(array)
    ? columnFromTypedArray(array, options)
    : columnFromValues(v => array.forEach(v), type, options, dicts);
}

/**
 * Create a new column from a typed array input.
 * @template T
 * @param {import('../types.js').TypedArray} values The input data.
 * @param {import('../types.js').ColumnBuilderOptions} options
 *  Builder options for the generated column.
 * @returns {Column<T>} The generated column.
 */
function columnFromTypedArray(values, { maxBatchRows, useBigInt }) {
  const arrayType = /** @type {import('../types.js').TypedArrayConstructor} */ (
    values.constructor
  );
  const type = typeForTypedArray(arrayType);
  const length = values.length;
  const limit = Math.min(maxBatchRows || Infinity, length);
  const numBatches = Math.floor(length / limit);

  const batches = [];
  const batchType = isInt64ArrayType(arrayType) && !useBigInt ? Int64Batch : DirectBatch;
  const add = (start, end) => batches.push(new batchType({
    length: end - start,
    nullCount: 0,
    type,
    validity: new uint8Array(0),
    values: values.subarray(start, end)
  }));

  let idx = 0;
  for (let i = 0; i < numBatches; ++i) add(idx, idx += limit);
  if (idx < length) add(idx, length);

  return new Column(batches);
}

/**
 * Return an Arrow data type for a given typed array type.
 * @param {import('../types.js').TypedArrayConstructor} arrayType
 *  The typed array type.
 * @returns {import('../types.js').DataType} The data type.
 */
function typeForTypedArray(arrayType) {
  switch (arrayType) {
    case float32Array: return float32();
    case float64Array: return float64();
    case int8Array: return int8();
    case int16Array: return int16();
    case int32Array: return int32();
    case int64Array: return int64();
    case uint8Array: return uint8();
    case uint16Array: return uint16();
    case uint32Array: return uint32();
    case uint64Array: return uint64();
  }
}

/**
 * Create a new table from a collection of columns. Columns are assumed
 * to have the same record batch sizes.
 * @param {[string, import('../column.js').Column][]
 *  | Record<string, import('../column.js').Column>} data The columns,
 *  as an object with name keys, or an array of [name, column] pairs.
 * @param {boolean} [useProxy] Flag indicating if row proxy
 *  objects should be used to represent table rows (default `false`).
 * @returns {Table} The new table.
 */
function tableFromColumns(data, useProxy) {
  const fields = [];
  const entries = Array.isArray(data) ? data : Object.entries(data);
  const length = entries[0]?.[1].length;

  const columns = entries.map(([name, col]) => {
    if (col.length !== length) {
      throw new Error('All columns must have the same length.');
    }
    fields.push(field(name, col.type));
    return col;
  });

  const schema = {
    version: Version.V5,
    endianness: Endianness.Little,
    fields,
    metadata: null
  };

  return new Table(schema, columns, useProxy);
}

/**
 * Create a new table from the provided arrays.
 * @param {[string, Array | import('../types.js').TypedArray][]
 *  | Record<string, Array | import('../types.js').TypedArray>} data
 *  The input data as a collection of named arrays.
 * @param {import('../types.js').TableBuilderOptions} options
 *  Table builder options, including an optional type map.
 * @returns {import('../table.js').Table} The new table.
 */
function tableFromArrays(data, options = {}) {
  const { types = {}, ...opt } = options;
  const dicts = dictionaryContext();
  const entries = Array.isArray(data) ? data : Object.entries(data);
  const columns = entries.map(([name, array]) =>
    /** @type {[string, import('../column.js').Column]} */ (
    [ name, columnFromArray(array, types[name], opt, dicts)]
  ));
  return tableFromColumns(columns, options.useProxy);
}

exports.Batch = Batch;
exports.Column = Column;
exports.DateUnit = DateUnit;
exports.Endianness = Endianness;
exports.IntervalUnit = IntervalUnit;
exports.Precision = Precision;
exports.Table = Table;
exports.TimeUnit = TimeUnit;
exports.Type = Type;
exports.UnionMode = UnionMode;
exports.Version = Version;
exports.batchType = batchType;
exports.binary = binary;
exports.binaryView = binaryView;
exports.bool = bool;
exports.columnFromArray = columnFromArray;
exports.columnFromValues = columnFromValues;
exports.date = date;
exports.dateDay = dateDay;
exports.dateMillisecond = dateMillisecond;
exports.decimal = decimal;
exports.dictionary = dictionary;
exports.dictionaryContext = dictionaryContext;
exports.duration = duration;
exports.field = field;
exports.fixedSizeBinary = fixedSizeBinary;
exports.fixedSizeList = fixedSizeList;
exports.float = float;
exports.float16 = float16;
exports.float32 = float32;
exports.float64 = float64;
exports.int = int;
exports.int16 = int16;
exports.int32 = int32;
exports.int64 = int64;
exports.int8 = int8;
exports.interval = interval;
exports.largeBinary = largeBinary;
exports.largeList = largeList;
exports.largeListView = largeListView;
exports.largeUtf8 = largeUtf8;
exports.list = list;
exports.listView = listView;
exports.map = map;
exports.nullType = nullType;
exports.runEndEncoded = runEndEncoded;
exports.struct = struct;
exports.tableFromArrays = tableFromArrays;
exports.tableFromColumns = tableFromColumns;
exports.tableFromIPC = tableFromIPC;
exports.tableToIPC = tableToIPC;
exports.time = time;
exports.timeMicrosecond = timeMicrosecond;
exports.timeMillisecond = timeMillisecond;
exports.timeNanosecond = timeNanosecond;
exports.timeSecond = timeSecond;
exports.timestamp = timestamp;
exports.uint16 = uint16;
exports.uint32 = uint32;
exports.uint64 = uint64;
exports.uint8 = uint8;
exports.union = union;
exports.utf8 = utf8;
exports.utf8View = utf8View;
