/**
 * A table consists of a collection of named columns (or 'children').
 * To work with table data directly in JavaScript, use `toColumns()`
 * to extract an object that maps column names to extracted value arrays,
 * or `toArray()` to extract an array of row objects. For random access
 * by row index, use `getChild()` to access data for a specific column.
 */
export class Table {
    /**
     * Create a new table with the given schema and columns (children).
     * @param {import('./types.js').Schema} schema The table schema.
     * @param {import('./column.js').Column[]} children The table columns.
     * @param {boolean} [useProxy=false] Flag indicating if row proxy
     *  objects should be used to represent table rows (default `false`).
     */
    constructor(schema: import("./types.js").Schema, children: import("./column.js").Column<any>[], useProxy?: boolean);
    /** @readonly */
    readonly schema: import("./types.js").Schema;
    /** @readonly */
    readonly names: string[];
    /**
     * @type {import('./column.js').Column[]}
     * @readonly
     */
    readonly children: import("./column.js").Column<any>[];
    /**
     * @type {import('./types.js').StructFactory}
     * @readonly
     */
    readonly factory: import("./types.js").StructFactory;
    /**
     * Returns a row object generator for the given batch index.
     * @private
     * @readonly
     * @param {number} b The batch index.
     * @returns {(index: number) => Record<string,any>}
     */
    private readonly getFactory;
    /**
     * The number of columns in this table.
     * @return {number} The number of columns.
     */
    get numCols(): number;
    /**
     * The number of rows in this table.
     * @return {number} The number of rows.
     */
    get numRows(): number;
    /**
     * Return the child column at the given index position.
     * @param {number} index The column index.
     * @returns {import('./column.js').Column<any>}
     */
    getChildAt(index: number): import("./column.js").Column<any>;
    /**
     * Return the first child column with the given name.
     * @param {string} name The column name.
     * @returns {import('./column.js').Column<any>}
     */
    getChild(name: string): import("./column.js").Column<any>;
    /**
     * Construct a new table containing only columns at the specified indices.
     * The order of columns in the new table matches the order of input indices.
     * @param {number[]} indices The indices of columns to keep.
     * @param {string[]} [as] Optional new names for selected columns.
     * @returns {Table} A new table with columns at the specified indices.
     */
    selectAt(indices: number[], as?: string[]): Table;
    /**
     * Construct a new table containing only columns with the specified names.
     * If columns have duplicate names, the first (with lowest index) is used.
     * The order of columns in the new table matches the order of input names.
     * @param {string[]} names Names of columns to keep.
     * @param {string[]} [as] Optional new names for selected columns.
     * @returns {Table} A new table with columns matching the specified names.
     */
    select(names: string[], as?: string[]): Table;
    /**
     * Return an object mapping column names to extracted value arrays.
     * @returns {Record<string, import('./types.js').ValueArray<any>>}
     */
    toColumns(): Record<string, import("./types.js").ValueArray<any>>;
    /**
     * Return an array of objects representing the rows of this table.
     * @returns {Record<string, any>[]}
     */
    toArray(): Record<string, any>[];
    /**
     * Return a row object for the given index.
     * @param {number} index The row index.
     * @returns {Record<string, any>} The row object.
     */
    at(index: number): Record<string, any>;
    /**
     * Return a row object for the given index. This method is the same as
     * `at()` and is provided for better compatibility with Apache Arrow JS.
     * @param {number} index The row index.
     * @returns {Record<string, any>} The row object.
     */
    get(index: number): Record<string, any>;
    /**
     * Provide an informative object string tag.
     */
    get [Symbol.toStringTag](): string;
    /**
     * Return an iterator over objects representing the rows of this table.
     * @returns {Generator<Record<string, any>, any, null>}
     */
    [Symbol.iterator](): Generator<Record<string, any>, any, null>;
}
