/**
 * Return a value unchanged.
 * @template T
 * @param {T} value The value.
 * @returns {T} The value.
 */
export function identity<T>(value: T): T;
/**
 * Return a value coerced to a BigInt.
 * @param {*} value The value.
 * @returns {bigint} The BigInt value.
 */
export function toBigInt(value: any): bigint;
/**
 * Return an offset conversion method for the given data type.
 * @param {{ offsets: import('../types.js').TypedArray}} type The array type.
 */
export function toOffset(type: {
    offsets: import("../types.js").TypedArray;
}): typeof toBigInt | typeof identity;
/**
 * Return the number of days from a millisecond timestamp.
 * @param {number} value The millisecond timestamp.
 * @returns {number} The number of days.
 */
export function toDateDay(value: number): number;
/**
 * Return a timestamp conversion method for the given time unit.
 * @param {import('../types.js').TimeUnit_} unit The time unit.
 * @returns {(value: number) => bigint} The conversion method.
 */
export function toTimestamp(unit: import("../types.js").TimeUnit_): (value: number) => bigint;
/**
 * Write month/day/nanosecond interval to a byte buffer.
 * @param {Array | Float64Array} interval The interval data.
 * @returns {Uint8Array} A byte buffer with the interval data.
 *  The returned buffer is reused across calls, and so should be
 *  copied to a target buffer immediately.
 */
export function toMonthDayNanoBytes([m, d, n]: any[] | Float64Array): Uint8Array;
/**
 * Coerce a bigint value to a number. Throws an error if the bigint value
 * lies outside the range of what a number can precisely represent.
 * @param {bigint} value The value to check and possibly convert.
 * @returns {number} The converted number value.
 */
export function toNumber(value: bigint): number;
/**
 * Divide one BigInt value by another, and return the result as a number.
 * The division may involve unsafe integers and a loss of precision.
 * @param {bigint} num The numerator.
 * @param {bigint} div The divisor.
 * @returns {number} The result of the division as a floating point number.
 */
export function divide(num: bigint, div: bigint): number;
/**
 * Convert a floating point number or bigint to decimal bytes.
 * @param {number|bigint} value The number to encode. If a bigint, we assume
 *  it already represents the decimal in integer form with the correct scale.
 *  Otherwise, we assume a float that requires scaled integer conversion.
 * @param {BigUint64Array} buf The uint64 array to write to.
 * @param {number} offset The starting index offset into the array.
 * @param {number} stride The stride of an encoded decimal, in 64-bit steps.
 * @param {number} scale The scale mapping fractional digits to integers.
 */
export function toDecimal(value: number | bigint, buf: BigUint64Array, offset: number, stride: number, scale: number): void;
/**
 * Convert a 128-bit decimal value to a bigint.
 * @param {BigUint64Array} buf The uint64 array containing the decimal bytes.
 * @param {number} offset The starting index offset into the array.
 * @returns {bigint} The converted decimal as a bigint, such that all
 *  fractional digits are scaled up to integers (for example, 1.12 -> 112).
 */
export function fromDecimal128(buf: BigUint64Array, offset: number): bigint;
/**
 * Convert a 256-bit decimal value to a bigint.
 * @param {BigUint64Array} buf The uint64 array containing the decimal bytes.
 * @param {number} offset The starting index offset into the array.
 * @returns {bigint} The converted decimal as a bigint, such that all
 *  fractional digits are scaled up to integers (for example, 1.12 -> 112).
 */
export function fromDecimal256(buf: BigUint64Array, offset: number): bigint;
/**
 * Convert a 16-bit float from integer bytes to a number.
 * Adapted from https://github.com/apache/arrow/blob/main/js/src/util/math.ts
 * @param {number} value The float as a 16-bit integer.
 * @returns {number} The converted 64-bit floating point number.
 */
export function fromFloat16(value: number): number;
/**
 * Convert a number to a 16-bit float as integer bytes..
 * Inspired by numpy's `npy_double_to_half`:
 * https://github.com/numpy/numpy/blob/5a5987291dc95376bb098be8d8e5391e89e77a2c/numpy/core/src/npymath/halffloat.c#L43
 * Adapted from https://github.com/apache/arrow/blob/main/js/src/util/math.ts
 * @param {number} value The 64-bit floating point number to convert.
 * @returns {number} The converted 16-bit integer.
 */
export function toFloat16(value: number): number;
