/**
@license
Copyright (c) 2017 Vaadin Ltd.
This program is available under Apache License Version 2.0, available at https://vaadin.com/license/
*/
import { microTask } from '@polymer/polymer/lib/utils/async.js';

import { PolymerElement } from '@polymer/polymer/polymer-element.js';

/**
 * @polymerMixin
 */
export const SortMixin = superClass => class SortMixin extends superClass {
  static get properties() {
    return {
      /**
       * When `true`, all `<vaadin-grid-sorter>` are applied for sorting.
       * @attr {boolean} multi-sort
       * @type {boolean}
       */
      multiSort: {
        type: Boolean,
        value: false
      },

      /**
       * @type {!Array<!GridSorter>}
       * @protected
       */
      _sorters: {
        type: Array,
        value: function() {
          return [];
        }
      },

      /** @private */
      _previousSorters: {
        type: Array,
        value: function() {
          return [];
        }
      }
    };
  }

  /** @protected */
  ready() {
    super.ready();
    this.addEventListener('sorter-changed', this._onSorterChanged);

    // With Polymer 2 & shady the 'sorter-changed' listener isn't guaranteed to be registered
    // before child <vaadin-grid-sorter>'s upgrade and fire the events. The following
    // makes sure that 'sorter-changed' is fired for all <vaadin-grid-sorter> elements
    // after this (<vaadin-grid>) is ready (and the listeners are active).
    if (window.ShadyDOM) {
      microTask.run(() => {
        const sorters = this.querySelectorAll('vaadin-grid-sorter');
        Array.from(sorters).forEach((sorter) => {
          // Don't try to fire if the sorter hasn't been upgraded yet
          if (sorter instanceof PolymerElement) {
            sorter.dispatchEvent(new CustomEvent('sorter-changed', {bubbles: true, composed: true}));
          }
        });
      });
    }
  }

  /** @private */
  _onSorterChanged(e) {
    const sorter = e.target;
    e.stopPropagation();
    this.__updateSorter(sorter);
    this.__applySorters();
  }

  /** @private */
  __removeSorters(sortersToRemove) {
    if (sortersToRemove.length == 0) {
      return;
    }

    this._sorters = this._sorters.filter((sorter) => sortersToRemove.indexOf(sorter) < 0);
    if (this.multiSort) {
      this.__updateSortOrders();
    }
    this.__applySorters();
  }

  /** @private */
  __updateSortOrders() {
    this._sorters.forEach((sorter, index) => (sorter._order = this._sorters.length > 1 ? index : null), this);
  }

  /** @private */
  __updateSorter(sorter) {
    if (!sorter.direction && this._sorters.indexOf(sorter) === -1) {
      return;
    }
    sorter._order = null;

    if (this.multiSort) {
      this._removeArrayItem(this._sorters, sorter);
      if (sorter.direction) {
        this._sorters.unshift(sorter);
      }
      this.__updateSortOrders();
      this._sorters.forEach((sorter, index) => sorter._order = this._sorters.length > 1 ? index : null, this);
    } else {
      if (sorter.direction) {
        const otherSorters = this._sorters.filter((s) => s != sorter);
        this._sorters = [sorter];
        otherSorters.forEach((sorter) => {
          sorter._order = null;
          sorter.direction = null;
        });
      }
    }
  }

  /** @private */
  __applySorters() {
    if (this.dataProvider &&
      // No need to clear cache if sorters didn't change and grid is attached
      this.isAttached &&
      JSON.stringify(this._previousSorters) !== JSON.stringify(this._mapSorters())) {
      this.clearCache();
    }

    this._a11yUpdateSorters();

    this._previousSorters = this._mapSorters();
  }

  /**
   * @return {!Array<!GridSorter>}
   * @protected
   */
  _mapSorters() {
    return this._sorters.map(sorter => {
      return {
        path: sorter.path,
        direction: sorter.direction
      };
    });
  }

  /** @private */
  _removeArrayItem(array, item) {
    const index = array.indexOf(item);
    if (index > -1) {
      array.splice(index, 1);
    }
  }
};
